/******************************************************************************/
/*                     Copyright (c) Sirius XM Radio Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio Inc.           */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/
/*******************************************************************************
*
*DESCRIPTION
*
*       This module will contain all the OSAL debug support.
*
*******************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <math.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"
#include "osal_core.h"
#include "osal_sem.h"
#include "osal_queue.h"
#include "osal_task.h"
#include "osal_time.h"
#include "osal_debughandler.h"
#include "_osal_debug.h"

#if OSAL_BUFFER_TEST == 1
#include "osal_dbuf_test.h"
#endif /* OSAL_BUFFER_TEST == 1 */

#if OSAL_DEBUG == 1

/*****************************************************************************
*
*       OSALD_eDebugMainMenuHandler
**
*       Inputs:
*               *pcTaskParameters	Command line entry string before CR/LF
*
*       Outputs:
*               NONE
*
*****************************************************************************/
static OSAL_DEBUG_STATE_ENUM OSALD_eDebugMainMenuHandler(
        char *pcCmdLine, const void *pvArg )
{
    OSAL_DEBUG_STATE_ENUM eState = OSAL_DEBUG_MAIN;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Decode the command line
    if (strcmp(pcCmdLine, "tasks") == 0)
    {
        eReturnCode = OSAL.eTaskList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "queues") == 0)
    {
        eReturnCode = OSAL.eQueueList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "sems") == 0)
    {
        eReturnCode = OSAL.eSemList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "crcs") == 0)
    {
        eReturnCode = OSAL.eCRCList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "timers") == 0)
    {
        eReturnCode = OSAL.eTimerList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "memory") == 0)
    {
       char *pcFilter, *acTokens = " \n";

       // Get filter string
       pcFilter = strtok( NULL, acTokens );

       eReturnCode = OSAL.eMemoryList(pcFilter);
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }

#if OSAL_INTERRUPT == 1
    else if (strcmp(pcCmdLine, "interrupts") == 0)
    {
        eReturnCode = OSAL.eIntList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
#endif /* OSAL_INTERRUPT == 1 */

#if OSAL_LOG == 1
    else if (strcmp(pcCmdLine, "log") == 0)
    {
        OSAL_OBJECT_HDL hLog;
        char *pacName, *acTokens = " \n";

        // Get name parameter
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            pcCmdLine = "list";
        }

        if (strcmp(pcCmdLine, "list") == 0)
        {
            eReturnCode = OSAL.eLogList(FALSE);
            if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
            {
                printf("Error! Could not get log list: %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else if (strcmp(pcCmdLine, "clear") == 0)
        {
            // Get name parameter
            pacName = strtok( NULL, acTokens );

            // Find the object this name belongs to
            hLog = OSAL.hLogGetHandleByName(pacName);
            if(hLog != OSAL_INVALID_OBJECT_HDL)
            {
                // We now have a log handle
                eReturnCode = OSAL.eLogClear(hLog);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to clear log! %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Cleared log, hdl = 0x%p\n", hLog);
                }
            }
            else
            {
                printf("Error! Cannot find object named %s\n",
                       pacName);
            }
        }
        else if (strcmp(pcCmdLine, "print") == 0)
        {
            // Get name parameter
            pacName = strtok( NULL, acTokens );

            // Find the object this name belongs to
            hLog = OSAL.hLogGetHandleByName(pacName);
            if(hLog != OSAL_INVALID_OBJECT_HDL)
            {
                char acData[33]; // arbitrary frame length
                N32 n32Num, n32Pos = 0, n32Total = 0;

                printf("--- START OF LOG ---\n");

                // We now have a log handle
                do
                {
                    // Grab a frame at a time...
                    n32Num = sizeof(acData) - 1;
                    n32Num = OSAL.n32LogRead(hLog, &acData[0], n32Pos, n32Num);

                    // Print if we have something at all
                    if(n32Num > 0)
                    {
                        // NULL terminate frame
                        acData[n32Num] = '\0';

                        // Increment the next position
                        n32Pos += n32Num;

                        // Increment the total read
                        n32Total += n32Num;

                        // print it
                        printf("%s", acData);

                        // Go get some more
                        continue;
                    }
                    else if(n32Num < 0)
                    {
                        // We already know the handle is valid
                        // so the log must be locked instead
                        // wait and try again...
                        OSAL.eTaskDelay(1);
                        continue;
                    }

                } while(n32Num != 0);

                printf("\n--- END OF LOG ---\n");
                printf("%d bytes\n", n32Total);
            }
            else
            {
                printf("Error! Cannot find object named %s\n",
                       pacName);
            }
        }
    }
#endif /* OSAL_LOG == 1 */

#if OSAL_BUFFER == 1
    else if (strcmp(pcCmdLine, "pools") == 0)
    {
        eReturnCode = OSAL.eBlockPoolList();
        if( (eReturnCode != OSAL_SUCCESS) && (eReturnCode != OSAL_NO_OBJECTS) )
        {
            printf("Error! Could not get object list: %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
#if OSAL_BUFFER_TEST == 1
    else if (strcmp(pcCmdLine, "pool_test") == 0)
    {
        // Test OSAL Dynamic Buffers...
        OSAL_vBufferTest();
    }
#endif /* OSAL_BUFFER_TEST == 1 */
#endif /* OSAL_BUFFER == 1 */
    else if (strcmp(pcCmdLine, "time") == 0)
    {
        UN32 un32CurrentTime;
        UN32 un32DST_Minutes = 0;
        N32 n32GMT_Minutes = 0;

        OSAL.eTimeGetGMToffset(&n32GMT_Minutes);
        OSAL.eTimeGetDSTadjustment(&un32DST_Minutes);

        /* Get the number of seconds since 'epoch' from the system */
        eReturnCode = OSAL.eTimeGet(&un32CurrentTime);
        if((eReturnCode == OSAL_SUCCESS) || (eReturnCode == OSAL_ERROR_INVALID_TIME))
        {
            TIME_T tTod;
            struct tm tmbuf;
            char cBuf[OSAL_ASCBUFSIZE];

            printf("%u seconds have elapsed since epoch\n",
                un32CurrentTime );

            tTod = un32CurrentTime;
            printf("%s (LOCAL)\n", OSAL.ctime_r(&tTod, cBuf));
            printf("%s (GMT)\n",
                OSAL.asctime_r(
                    OSAL.gmtime_r(&tTod, &tmbuf), cBuf));
            printf("[GMT:%d, DST:%u] minutes\n",
                n32GMT_Minutes,
                un32DST_Minutes
                    );

            printf("Result: %s\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "uptime") == 0)
    {
        UN32 un32Seconds;
        UN16 un16Msecs;

        /* Get the 'uptime' from the system */
        OSAL.vTimeUp(&un32Seconds, &un16Msecs);
        OSALD_vPrintTime(un32Seconds, un16Msecs);
    }
    else if (strcmp(pcCmdLine, "time_set") == 0)
    {
        struct tm tTime;
        TIME_T tTod;

        OSAL_vParseTime(pcCmdLine, &tTod, &tTime);

        /* Set the number of seconds since 'epoch' in the system */
        eReturnCode = OSAL.eTimeSet((UN32)tTod);
        if(eReturnCode == OSAL_SUCCESS)
        {
            UN32 un32DST_Minutes = 0;
            N32 n32GMT_Minutes = 0;
            char cBuf[OSAL_ASCBUFSIZE];
            OSAL.eTimeGetGMToffset(&n32GMT_Minutes);
            OSAL.eTimeGetDSTadjustment(&un32DST_Minutes);
            printf("GMT time has been set to %s\n",
                OSAL.ctime_r(&tTod, cBuf)
                    );
            printf("Local time has been set to %s [GMT:%d, DST:%u]\n",
                OSAL.asctime_r(
                    OSAL.localtime_r(&tTod, &tTime), cBuf),
                n32GMT_Minutes,
                un32DST_Minutes
                    );
        }
        else
        {
            printf("Error! Unable to set time\n");
        }
    }
    else if (strcmp(pcCmdLine, "gmt_set") == 0)
    {
        const char *acTokens = " \n";
        N32 n32GMT_Minutes = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Get next tokens...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            n32GMT_Minutes = atoi(pcCmdLine);

            eReturnCode = OSAL.eTimeSetGMToffset(n32GMT_Minutes);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("GMT Offset: %d minutes.\n",
                    n32GMT_Minutes);
            }
            else
            {
                printf("Error! Returned %s.\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
    }
    else if (strcmp(pcCmdLine, "dst_set") == 0)
    {
        const char *acTokens = " \n";
        UN32 un32DST_Minutes = 0;
        OSAL_RETURN_CODE_ENUM eReturnCode;

        // Get next tokens...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            un32DST_Minutes = atoi(pcCmdLine);

            eReturnCode = OSAL.eTimeSetDSTadjustment(un32DST_Minutes);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("DST Offset: %d minutes.\n",
                    un32DST_Minutes);
            }
            else
            {
                printf("Error! Returned %s.\n",
                    OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
    }
    else if (strcmp(pcCmdLine, "cpu") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        UN8 un8Current, un8Max, un8Min;

        eReturnCode = OSAL.eGetUtilizationCPU(&un8Current, &un8Max, &un8Min);

        if(eReturnCode == OSAL_SUCCESS)
        {
            printf("CPU Utilization: %u%% current, %u%% max, %u%% min\n",
                   un8Current, un8Max, un8Min);
        }
        else
        {
            printf("Error! Returned %s.\n",
                OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if (strcmp(pcCmdLine, "cpu_reset") == 0)
    {
        OSAL.vResetUtilizationCPU();
    }
    else if (strcmp(pcCmdLine, "cpu_load") == 0)
    {
        // cpu_load: artificially load the cpu
        // Usage:
        // cpu_load tasks load_pct {priority {period]}}

        char *acTokens = " \n",
             *pcNumTasks,
             *pcLoadPct,
             *pcPriority,
             *pcPeriod;

        // Get tokens...
        pcNumTasks = strtok( NULL, acTokens );
        pcLoadPct  = strtok( NULL, acTokens );
        pcPriority = strtok( NULL, acTokens );
        pcPeriod   = strtok( NULL, acTokens );

        // Default to 2 tasks, 50% load, med priority, 100 ms period if not supplied.
        gsCpuLoadInfo.un8NumTasks  = (pcNumTasks != NULL) ? atoi(pcNumTasks) : 2;
        gsCpuLoadInfo.un8LoadPct   = (pcLoadPct  != NULL) ? atoi(pcLoadPct)  : 50;
        gsCpuLoadInfo.un8Priority  = (pcPriority != NULL) ? atoi(pcPriority) : 2;
        gsCpuLoadInfo.un16PeriodMs = (pcPeriod   != NULL) ? atoi(pcPeriod)   : 100;

        if (gsCpuLoadInfo.un8NumTasks > MAX_CPU_LOAD_TASKS)
        {
            gsCpuLoadInfo.un8NumTasks = MAX_CPU_LOAD_TASKS;
            printf("You can only spawn a maximum of %d load tasks.\n", MAX_CPU_LOAD_TASKS);
        }

        if (gsCpuLoadInfo.un16PeriodMs > MAX_CPU_LOAD_PERIOD_MS)
        {
            gsCpuLoadInfo.un16PeriodMs = MAX_CPU_LOAD_PERIOD_MS;
            printf("The maximum period is %ums in length, clipping.\n", MAX_CPU_LOAD_PERIOD_MS);
        }

        if ((gsCpuLoadInfo.un8Priority > OSAL_TASK_PRIORITY_HIGH) ||
            (gsCpuLoadInfo.un8Priority < OSAL_TASK_PRIORITY_LOW))
        {
            gsCpuLoadInfo.un8Priority = OSAL_TASK_PRIORITY_MEDIUM;
            printf("Task priority must be between LOW (1) and HIGH (3). Setting to default. (MEDIUM)\n");
        }

        //
        // First, clear our the old tasks (if any)
        //

        OSALD_vUnLoadCpu();

        //
        // Wait as long as the max period so the old CPU
        // load tasks can stop (if necessary.)
        //

        OSAL.eTaskDelay(MAX_CPU_LOAD_PERIOD_MS);

        //
        // Now load the CPU up!
        //

        OSALD_vLoadCpu();
    }
    else if (strcmp(pcCmdLine, "cpu_unload") == 0)
    {
        //
        // Unburden the CPU
        //

        OSALD_vUnLoadCpu();
    }
    else if (strcmp(pcCmdLine, "mem") == 0)
    {
        BOOLEAN bOk;
        UN32 un32CurrentAllocatedBlocks, un32CurrentActualAllocatedBlocks,
             un32CurrentUserBytes, un32CurrentActualBytes,
             un32MaxAllocatedBlocks, un32MaxActualAllocatedBlocks,
             un32MaxUserBytes, un32MaxActualBytes, un32TotalSystemBytes;

        bOk = OSAL.bMemoryUsage( &un32CurrentAllocatedBlocks,
                                 &un32CurrentActualAllocatedBlocks,
                                 &un32CurrentUserBytes,
                                 &un32CurrentActualBytes,
                                 &un32MaxAllocatedBlocks,
                                 &un32MaxActualAllocatedBlocks,
                                 &un32MaxUserBytes,
                                 &un32MaxActualBytes,
                                 &un32TotalSystemBytes
                               );
        if(bOk == TRUE)
        {
            printf("Memory Usage:\n");

            printf("\tCurrent Memory Usage:\n");
            printf("\t\t%u blocks\n", un32CurrentAllocatedBlocks );
            printf("\t\t%u actual blocks\n", un32CurrentActualAllocatedBlocks);
            printf("\t\t%u user bytes\n", un32CurrentUserBytes);
            printf("\t\t%u actual bytes\n", un32CurrentActualBytes);

            printf("\tMaximum Memory Usage:\n");
            printf("\t\t%u blocks\n", un32MaxAllocatedBlocks );
            printf("\t\t%u actual blocks\n", un32MaxActualAllocatedBlocks);
            printf("\t\t%u user bytes\n", un32MaxUserBytes);
            printf("\t\t%u actual bytes\n", un32MaxActualBytes);

            printf("\tTotal System Memory: %u\n", un32TotalSystemBytes );

            printf("\tOS-Overhead:  %.1f%%\n",
                   un32CurrentUserBytes != 0 ?
                       (((double)(un32CurrentActualBytes - un32CurrentUserBytes))
                            / un32CurrentUserBytes) * 100  : 100);
        }
        else
        {
            printf("ERROR: Cannot print information!\n");
        }
    }
    else if (strcmp(pcCmdLine, "md") == 0)
    {
        // md - memory display
        // Usage:
        // md {address {# of objects]}}

        static UN8 *pun8Address = NULL; // Remember this
        static UN32 un32Length = 0; // Remember this too

        char *acTokens = " \n", *pcAddress, *pcNumObjects;

        // Get tokens...
        pcAddress = strtok( NULL, acTokens );
        pcNumObjects = strtok( NULL, acTokens );

        // Was an address provided?
        if(pcAddress != NULL)
        {
            unsigned long ulAddress;

            // Extract new address
            ulAddress = strtoul(pcAddress, NULL, 16);
            pun8Address = (UN8 *)(size_t)ulAddress;

            // Was number of objects provided?
            if(pcNumObjects != NULL)
            {
                un32Length = atoi(pcNumObjects);
            }
            else
            {
                un32Length = 0;
            }
        }
        else // No address, No num objects
        {
            // Add number of objects to previous address
            pun8Address += un32Length;
        }

        // Dump it!
        OSAL.vDump(stdout, NULL, pun8Address, un32Length);
    }
    else if (strcmp(pcCmdLine, "sleep") == 0)
    {
        BOOLEAN bOk;
        bOk = OSAL.bSleep();
        if(bOk == TRUE)
        {
            printf("OSAL: Sleep Mode\n");
        }
        else
        {
            printf("Error! Could not enter sleep mode\n");
        }
    }
    else if (strcmp(pcCmdLine, "wakeup") == 0)
    {
        BOOLEAN bOk;
        bOk = OSAL.bWakeup();
        if(bOk == TRUE)
        {
            printf("OSAL: Wakeup Mode\n");
        }
        else
        {
            printf("Error! Could not enter wakeup mode\n");
        }
    }
    else if (strcmp(pcCmdLine, "shutdown") == 0)
    {
        BOOLEAN bOk;
        bOk = OSAL.bShutdown();
        if(bOk == TRUE)
        {
            printf("OSAL: Shutdown Mode\n");
        }
        else
        {
            printf("Error! Could not enter shutdown mode\n");
        }
    }
    else if (strcmp(pcCmdLine, "version") == 0)
    {
        const char *pacVersion;

        pacVersion = OSAL.pacGetVersion(NULL, NULL, NULL);
        if(pacVersion != NULL)
        {
            printf("%s\n", pacVersion);
        }
    }

#if OSAL_TEST == 1
    else if (strcmp(pcCmdLine, "test") == 0)
    {
        eState = OSAL_DEBUG_TEST;
    }
#endif /* OSAL_TEST == 1 */

#if OSAL_FILE_SYSTEM == 1
    else if (strcmp(pcCmdLine, "fs") == 0)
    {
        eState = OSAL_DEBUG_FS;
    }
#endif /* OSAL_FILE_SYSTEM == 1 */

#if OSAL_FIXED_MATH == 1
    else if (strcmp(pcCmdLine, "fixed_math") == 0)
    {
        eState = OSAL_DEBUG_FIXED;
    }
#endif /* OSAL_FIXED_MATH == 1 */

    else if (strcmp(pcCmdLine, "e") == 0)
    {
       OSAL.vControlOutputThisTask(TRUE);
       printf("Task output enabled\n");
    }
    else if (strcmp(pcCmdLine, "d") == 0)
    {
       printf("Task output disabled\n");
       OSAL.vControlOutputThisTask(FALSE);
    }
    else if ((strcmp(pcCmdLine, "help") == 0) ||
            (strcmp(pcCmdLine, "HELP") == 0))
    {
        printf("\nAvailable commands:\n");
        printf("\ttasks                 -\tlist tasks\n");
        printf("\tqueues                -\tlist queues\n");
        printf("\tsems                  -\tlist semaphores\n");
        printf("\tcrcs                  -\tlist CRCs\n");
        printf("\ttimers                -\tlist timers\n");
        printf("\tmemory [filter]       -\tlist memory blocks, with filtering (if provided)\n");

#if OSAL_INTERRUPT == 1
        printf("\tinterrupts            -\tlist interrupts\n");
#endif /* OSAL_INTERRUPT == 1 */

#if OSAL_LOG == 1
        printf("\tlog list              -\tList logs\n");
        printf("\tlog print name        -\tPrint a log\n");
        printf("\tlog clear name        -\tClear(reset) a log\n");
#endif /* OSAL_LOG */

#if OSAL_BUFFER == 1
        printf("\tpools                 -\tlist pools\n");
#if OSAL_BUFFER_TEST == 1
        printf("\tpool_test             -\ttest pools & buffers\n");
#endif /* OSAL_BUFFER_TEST */
#endif /* OSAL_BUFFER */

        printf("\ttime                  -\tshow local time\n");
        printf("\tuptime                -\tshow system uptime\n");
        printf("\ttime_set              -\tset local time\n"
               "\t\tmm/dd/yyyy hh:mm:ss\n");
        printf("\tgmt_set               -\tset GMT offset (minutes)\n");
        printf("\tdst_set               -\tset DST offset (minutes)\n");
        printf("\tcpu                   -\tshow CPU utilization\n");
        printf("\tcpu_reset             -\treset CPU utilization\n");
        printf("\tcpu_load              -\tartificially utilize CPU\n");
        printf("\t      tasks load_pct {priority {period]}}\n");
        printf("\tcpu_unload            -\treset artificial CPU utilization\n");
        printf("\tmem                   -\tshow memory usage\n");
        printf("\tmd md {address {# of objects]}}\n"
               "\t                      -\tmemory display\n");
        printf("\tversion               -\tdisplay version\n");
        printf("\tsleep                 -\tenter sleep mode\n");
        printf("\twakeup                -\tenter wakeup mode\n");
        printf("\tshutdown              -\tenter shutdown mode\n");

#if OSAL_TEST == 1
        printf("\ttest                  -\tdisplay osal test menu\n");
#endif /* OSAL_TEST == 1 */

#if OSAL_FILE_SYSTEM == 1
        printf("\tfs                    -\tfile system menu\n");
#endif /* OSAL_FILE_SYSTEM == 1 */

#if OSAL_FIXED_MATH == 1
        printf("\tfixed_math            -\tfixed math menu\n");
#endif /* OSAL_FIXED_MATH == 1 */
        printf("\te                     -\tenable output\n");
        printf("\td                     -\tdisable output\n");
        printf("\thelp                  -\tdisplay this message\n");
        printf("\tquit                  -\tquit from app with status 0\n");
        printf("\texit                  -\texit this menu\n");
    }
    else if (strcmp(pcCmdLine, "quit") == 0)
    {
        exit(0);
    }
    else if (strcmp(pcCmdLine, "exit") == 0)
    {
        eState = OSAL_DEBUG_EXIT;
    }
    else
    {
        printf("\nInvalid command.  Type 'help' for a list of commands.\n");
    }

    return eState;
}

/*****************************************************************************
*
*       OSALD_vLoadCpu
*
*****************************************************************************/
static void OSALD_vLoadCpu()
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 un8Idx;
    UN8 un8DebugTaskPriority;

    //
    // The loop in the load task will only continue so long as
    // .bConsumeCPU is true, so we need to set this.
    //

    gsCpuLoadInfo.bConsumeCPU = TRUE;

    //
    // The user must provide the number of tasks (usually
    // corresponding to the number of cores.)
    //

    for ( un8Idx = 0;
        un8Idx < gsCpuLoadInfo.un8NumTasks;
        ++un8Idx)
    {
        //
        // Here we generate the name for our new task automatically.
        // Any old tasks will have been cleaned up by this
        // point, so name re-use should not be a problem.
        //

        OSAL_OBJECT_HDL hTask;
        char pacTaskName[MAX_CPU_LOAD_TASK_NAME_SIZE];

        //
        // Generate the name and try to get the handle
        // (to double-check that the named task doesn't
        // exist.
        //

        sprintf(pacTaskName, LOAD_TASK_PREFIX"%02d", un8Idx);
        hTask = OSAL.hTaskGetHandleByName ( pacTaskName );

        if( hTask == OSAL_INVALID_OBJECT_HDL )
        {
            //
            // Assuming the task didn't exist, it will now.
            //

            eReturnCode = OSAL.eTaskCreate( &hTask,
                pacTaskName,
                OSALD_n32CpuLoadTask,
                NULL,
                512, // 512-byte stack
                (OSAL_TASK_PRIORITY_ENUM)
                gsCpuLoadInfo.un8Priority,
                OSAL_TASK_OPTION_NONE
                );

            if ( eReturnCode != OSAL_SUCCESS )
            {
                printf( "Could not create task %s!\n", pacTaskName );
            }
        }
        else
        {
            printf( "ERROR: Got into OSALD_vLoadCpu with pre-existing CPU "
                "load tasks. This should never happen!\n" );
        }
    }

    //
    // Finally, we have to make sure OSAL's debug task gets set to one
    // priority level higher than our load tasks. If we *don't* do this, we
    // won't be able to type into the console when the load tasks are at
    // 100% utilization. Also, make sure we don't accidentally try to set
    // the debug tasks to PRIORITY_HIGHEST.
    //

    un8DebugTaskPriority = gsCpuLoadInfo.un8Priority + 1;

    if ( un8DebugTaskPriority > (UN8)( OSAL_TASK_PRIORITY_HIGH ) )
    {
        un8DebugTaskPriority = (UN8)( OSAL_TASK_PRIORITY_HIGH );
    }

    eReturnCode = DEBUG_eTaskChangePriority(
        (OSAL_TASK_PRIORITY_ENUM) un8DebugTaskPriority );

    if ( eReturnCode != OSAL_SUCCESS )
    {
        printf("Could not change the priority on the debug task!\n");
    }

    return;
}

/*****************************************************************************
*
*       OSALD_vUnLoadCpu
*
*****************************************************************************/
static void OSALD_vUnLoadCpu()
{
    UN8 un8Idx;
    OSAL_RETURN_CODE_ENUM eReturnCode;

    //
    // The loop in the load task will only continue so long as
    // .bConsumeCPU is true, so we need to set this to false. The
    // tasks should stop by themselves. The load task is read-only,
    // and generates no usable results, so this is safe (and a
    // bit more compact than creating queueing / messaging
    // infrastructure.)
    //

    gsCpuLoadInfo.bConsumeCPU = FALSE;

    //
    // Just to be on the safe side, send each task a legitimate
    // eTaskDelete, in case we want to add "real" shutdown
    // handleers in the future. For the moment, this effectively
    // does nothing.
    //

    for (un8Idx = 0;
        un8Idx < gsCpuLoadInfo.un8NumTasks;
        ++un8Idx)
    {
        //
        // Generate the name and try to get the handle.
        // On we have that, call eTaskDelete on it.
        //

        OSAL_OBJECT_HDL hTask;
        char pacTaskName[MAX_CPU_LOAD_TASK_NAME_SIZE];

        sprintf(pacTaskName, LOAD_TASK_PREFIX"%02d", un8Idx);
        hTask = OSAL.hTaskGetHandleByName ( pacTaskName );

        if(hTask != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eTaskDelete(hTask);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("Could not delete task %s!\n", pacTaskName);
            }
        }
    }

    //
    // Finally, since we're all done, we can drop the debug task priority back down to
    // LOW.
    //

    DEBUG_eTaskChangePriority( OSAL_TASK_PRIORITY_LOW );

    return;
}

/******************************************************************************
 *
 *      OSALD_n32CpuLoadTask
 *
 *      The load task attempts to keep a certain percentage load
 *      on the CPU by cycling full on and full off. This faux
 *      PWM scheme was both the simplest and most reliable way
 *      to get an effective "load percent" on the CPU in our testing.
 *
 *
 ******************************************************************************/
static N32 OSALD_n32CpuLoadTask( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    //
    // Used for capturing raw values from OSAL.vTimeUp
    //

    UN16 un16MsUp;
    UN32 un32SecsUp;

    //
    // The current time up, and the start of the last "on"
    // phase, both measured in Ms.
    //

    UN32 un32CurTimeUpMs, un32LastOnMs;

    //
    // The calculated on and off times, in Ms, for our
    // current duty cycle, as well as the calculated
    // duty cycle itself.
    //

    UN32 un32OnMs, un32OffMs;
    float fDutyCycle;

    //
    // Coeffecients for the calculation that "pegs" the CPU.
    // They have no particular meaning (aside from z and c,
    // which basically count the number of times through the
    // loops as an int and float, repsectively.)
    //

    UN32   x = 253,       y = 717,         z = 0;
    double a = 808.08173, b = 909.1111157, c = 0.0;

    //
    // Calculate the duty cycle from the integer load percent,
    // then calculate the duration of the on and off portions
    // of our cycle.
    //

    fDutyCycle = (float)gsCpuLoadInfo.un8LoadPct * PCT_TO_DC;
    un32OnMs = (UN32)(fDutyCycle * gsCpuLoadInfo.un16PeriodMs);
    un32OffMs = gsCpuLoadInfo.un16PeriodMs - un32OnMs;

    //
    // gsCpuLoadInfo is read-only from this task. If the main
    // debug task sets bConsumeCPU to false, we know it's time
    // to quit running (and saves us the trouble of setting up
    // task queues and messaging infrastructure.)
    //

    while(gsCpuLoadInfo.bConsumeCPU == TRUE)
    {
        //
        // Calculate the "last on" time so we know when to
        // stop
        //

        OSAL.vTimeUp(&un32SecsUp, &un16MsUp);
        un32LastOnMs = un16MsUp + (SEC_TO_MSEC * un32SecsUp);

        //
        // Loop through our calculation until our up time is
        // done. We just attempt to peg the CPU. The inter-
        // depedencies in the calculation should prevent
        // the compilter from optimizing this away.
        //

        do
        {
            x = (UN32)(b / (c+1.0)) * (-1 ^ (UN32)z);
            a += (x / (float)y);
            y += (UN32)a;
            b += x;
            z += 1;
            c += 1.00;

            //
            // Get the current time to see if we should quit. In
            // testing, the OSAL call did not affect the accuracy
            // of our timing (unlike Sleep)
            //

            OSAL.vTimeUp(&un32SecsUp, &un16MsUp);
            un32CurTimeUpMs = un16MsUp + (SEC_TO_MSEC * un32SecsUp);
        }
        while( (un32LastOnMs + un32OnMs) > un32CurTimeUpMs );

        //
        // Now that our on portion is done, turn off for our
        // required amount of time. This seems to induce a certain
        // amount of "slop" in our timing, which can be ameloriated
        // somewhat by using a longer cycle time (resulting in less
        // calls to .eTaskDelay(). Note this appears to be true
        // for MinGW at least, other platforms way vary.
        //

        eReturnCode = OSAL.eTaskDelay(un32OffMs);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("Error delaying cpu load task!\n");
        }
    }

    return OSAL_TASK_REPORT_NO_ERROR;
}

#if OSAL_TEST == 1

/*****************************************************************************
*
*       OSALD_eDebugTestMenuHandler
**
*       Inputs:
*               *pcTaskParameters	Command line entry string before CR/LF
*
*       Outputs:
*               NONE
*
*****************************************************************************/
static OSAL_DEBUG_STATE_ENUM OSALD_eDebugTestMenuHandler(
        char *pcCmdLine, const void *pvArg )
{
    static char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    static void *pvPtr = NULL;
    static OSAL_OBJECT_HDL hQueue = OSAL_INVALID_OBJECT_HDL;
    static OSAL_OBJECT_HDL hTask = OSAL_INVALID_OBJECT_HDL;
    static OSAL_OBJECT_HDL hSem = OSAL_INVALID_OBJECT_HDL;

    OSAL_DEBUG_STATE_ENUM eState = OSAL_DEBUG_TEST;
    char *acTokens = " \n";
    OSAL_RETURN_CODE_ENUM eReturnCode;

    // Decode the command line
    if (strcmp(pcCmdLine, "allocate") == 0)
    {
        pvPtr = OSAL.pvMemoryAllocate("Test Obj", 1000, TRUE);
        printf("Allocated: %#0x\n", pvPtr);
    }
    else if (strcmp(pcCmdLine, "free") == 0)
    {
        if(pvPtr != NULL)
        {
            OSAL.vMemoryFree(pvPtr);
            printf("freed: %#0x\n", pvPtr);
            pvPtr = NULL;
        }
    }
    else if (strcmp(pcCmdLine, "delay") == 0)
    {
        struct osald_delay {
            TIME_T tTod;
            UN32 un32Time;
            UN32 un32Seconds;
            UN16 un16Msecs;
        };
        N32 n32Msec, n32MsecElapsed;
        struct osald_delay sStart, sEnd;
        char cBuf[OSAL_ASCBUFSIZE];

        // Get next token...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            n32Msec = atoi(pcCmdLine);

            printf("Delay for %d msec...\n", n32Msec);

            /* Get the 'uptime' from the system */
            OSAL.vTimeUp(&sStart.un32Seconds, &sStart.un16Msecs);

            /* Get the number of seconds since 'epoch' from the system */
            OSAL.eTimeGetLocal(&sStart.un32Time);

            /* Get time from the system */
            sStart.tTod = sStart.un32Time;

            // Delay
            OSAL.eTaskDelay(n32Msec);

            /* Get the 'uptime' from the system */
            OSAL.vTimeUp(&sEnd.un32Seconds, &sEnd.un16Msecs);

            /* Get the number of seconds since 'epoch' from the system */
            OSAL.eTimeGetLocal(&sEnd.un32Time);

            /* Get time from the system */
            sEnd.tTod = sEnd.un32Time;

            printf("START\n");
            printf("%u seconds have elapsed since epoch\n",
                sStart.un32Time );
            printf("System uptime: ");
            OSALD_vPrintTime(sStart.un32Seconds, sStart.un16Msecs);
            printf("%s\n", OSAL.ctime_r(&sStart.tTod, cBuf));

            printf("END\n");
            printf("%u seconds have elapsed since epoch\n",
                sEnd.un32Time );
            printf("System uptime: ");
            OSALD_vPrintTime(sEnd.un32Seconds, sEnd.un16Msecs);
            printf("%s\n\n", OSAL.ctime_r(&sEnd.tTod, cBuf));

            n32MsecElapsed =
                (((sEnd.un32Seconds * 1000) + sEnd.un16Msecs) -
            ((sStart.un32Seconds * 1000) + sStart.un16Msecs));

            printf("Delay requested %d msec, "
                "measured is %d msec (delta = %d).\n",
                n32Msec, n32MsecElapsed, abs(n32Msec - n32MsecElapsed));
        }
    }
    else if (strcmp(pcCmdLine, "cs") == 0)
    {
        static UN32 un32SemId = 0;
        N32 n32InitialValue, n32Resources;
        const char *pacSemaphoreNamePrefix = "Semaphore#";

        // Get next token...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            n32InitialValue = atoi(pcCmdLine);
            // Get next token...
            pcCmdLine = strtok( NULL, acTokens );
            if (pcCmdLine != NULL)
            {
                n32Resources = atoi(pcCmdLine);

                snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                        "%s%u", pacSemaphoreNamePrefix,
                        un32SemId++);
                eReturnCode = OSAL.eSemCreate(&hSem, acName, n32InitialValue,
                                          n32Resources, OSAL_SEM_OPTION_NONE);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to allocate semaphore! %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Created semaphore, hdl = %#0x\n", hSem);
                }
            }
        }
    }
    else if (strcmp(pcCmdLine, "ds") == 0)
    {
        if(hSem != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eSemDelete(hSem);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to delete semaphore! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Destroyed semaphore, hdl = %#0x\n", hSem);
                hSem = OSAL_INVALID_OBJECT_HDL;
            }
        }
        else
        {
            printf("Error! Invalid semaphore handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "gs") == 0)
    {
        if(hSem != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eSemGive(hSem);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to give semaphore! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Gave semaphore, hdl = %#0x\n", hSem);
            }
        }
        else
        {
            printf("Error! Invalid semaphore handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "ts") == 0)
    {
        if(hSem != OSAL_INVALID_OBJECT_HDL)
        {
            UN32 un32SecondsBefore = 0, un32SecondsAfter = 0;
            UN16 un16MillisecondsBefore = 0, un16MillisecondsAfter = 0;

            OSAL.vTimeUp(&un32SecondsBefore, &un16MillisecondsBefore);
            eReturnCode = OSAL.eSemTake(hSem, 1250);
            OSAL.vTimeUp(&un32SecondsAfter, &un16MillisecondsAfter);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to take semaphore! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Took semaphore, hdl = %#0x\n", hSem);
            }

            printf("Waited %u seconds, %u milliseconds.\n",
                un32SecondsAfter - un32SecondsBefore,
                abs(un16MillisecondsAfter - un16MillisecondsBefore));
        }
        else
        {
            printf("Error! Invalid semaphore handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "?s") == 0)
    {
        UN32 un32SemNumber = 0;
        OSALS_bPrintSemaphore(hTask, &un32SemNumber);
    }
    else if (strcmp(pcCmdLine, "is") == 0)
    {
        char *pacName;
        OSAL_OBJECT_HDL hTempS;

        // Get name parameter
        pacName = strtok( NULL, "\n" );

        // Find the object this name belongs to
        hTempS = OSAL.hSemGetHandleByName(pacName);
        if(hTempS != OSAL_INVALID_OBJECT_HDL)
        {
            UN32 un32SemNumber = 0;
            hSem = hTempS;
            OSALS_bPrintSemaphore(hSem, &un32SemNumber);
        }
        else
        {
            printf("Error! Invalid semaphore name.\n");
        }
    }
    else if (strcmp(pcCmdLine, "ct") == 0)
    {
        OSAL_TASK_PRIORITY_ENUM ePriority;
        static UN32 un32TaskId = 0;
        const char *pacTaskNamePrefix = "Task#";

        // Get next token...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            ePriority = (OSAL_TASK_PRIORITY_ENUM)atoi(pcCmdLine);

            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     "%s%u", pacTaskNamePrefix,
                    un32TaskId++);

            eReturnCode = OSAL.eTaskCreate(&hTask, acName, OSALD_n32Task,
                (void *)un32TaskId, 4096, ePriority, OSAL_TASK_OPTION_NONE);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to create task! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Created task, hdl = %#0x\n", hTask);
            }
        }
    }
    else if (strcmp(pcCmdLine, "dt") == 0)
    {
        if(hTask != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eTaskDelete(hTask);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to delete task! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Destroyed task, hdl = %#0x\n", hTask);
                hTask = OSAL_INVALID_OBJECT_HDL;
            }
        }
        else
        {
            printf("Error! Invalid task handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "st") == 0)
    {
        if(hTask != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eTaskSuspend(hTask);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to suspend task! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Task suspended, hdl = %#0x\n", hTask);
            }
        }
        else
        {
            printf("Error! Invalid task handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "rt") == 0)
    {
        if(hTask != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eTaskResume(hTask);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to resume task! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Task resumed, hdl = %#0x\n", hTask);
            }
        }
        else
        {
            printf("Error! Invalid task handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "?t") == 0)
    {
        UN32 un32TaskNumber = 0;
        OSALT_bPrintTask(hTask, &un32TaskNumber);
    }
    else if (strcmp(pcCmdLine, "it") == 0)
    {
        char *pacName;
        OSAL_OBJECT_HDL hTempT;

        // Get name parameter
        pacName = strtok( NULL, "\n" );

        // Find the object this name belongs to
        hTempT = OSAL.hTaskGetHandleByName(pacName);
        if(hTempT != OSAL_INVALID_OBJECT_HDL)
        {
            UN32 un32TaskNumber = 0;
            hTask = hTempT;
            OSALT_bPrintTask(hTask, &un32TaskNumber);
        }
        else
        {
            printf("Error! Invalid task name.\n");
        }
    }
    else if (strcmp(pcCmdLine, "cq") == 0)
    {
        UN32 un32Depth, un32Size;
        static UN32 un32QueueId = 0;
        const char *pacQueueNamePrefix = "Queue#";

        // Get next token...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine != NULL)
        {
            un32Depth = atoi(pcCmdLine);

            // Get next token...
            pcCmdLine = strtok( NULL, acTokens );
            if (pcCmdLine != NULL)
            {
                un32Size = atoi(pcCmdLine);

                snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                    "%s%u", pacQueueNamePrefix,
                    un32QueueId++);

                eReturnCode = OSAL.eQueueCreate(&hQueue, acName,
                    un32Depth, un32Size, OSAL_QUEUE_OPTION_FIXED_SIZE);

                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Unable to create queue! %s\n",
                        OSAL.pacGetReturnCodeName(eReturnCode));
                }
                else
                {
                    printf("Created queue, hdl = %#0x\n", hQueue);
                }
            }
        }
    }
    else if (strcmp(pcCmdLine, "dq") == 0)
    {
        if(hQueue != OSAL_INVALID_OBJECT_HDL)
        {
            eReturnCode = OSAL.eQueueDelete(hQueue);

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to delete queue! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("Destroyed queue, hdl = %#0x\n", hQueue);
                hQueue = OSAL_INVALID_OBJECT_HDL;
            }
        }
        else
        {
            printf("Error! Invalid queue handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "pq") == 0)
    {
        OSAL_QUEUE_INFO_STRUCT sQueueInfo;
        char *pacTestMsg, *pacMessage;
        UN32 un32TestMessageSize;

        // Get next tokens...
        pacMessage = strtok( NULL, "\n" );

        if (pacMessage == NULL)
        {
            printf("Error! Invalid arguments! \n");
        }
        else if (hQueue != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eQueueGetInfo(hQueue, &sQueueInfo);

            eReturnCode = OSAL.eMessageAllocate(hQueue, (void *)&pacTestMsg,
                                                OSAL_QUEUE_FLAG_NONBLOCK);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Populate message
                strcpy(pacTestMsg, pacMessage);
                un32TestMessageSize = strlen(pacTestMsg) + 1;

                // Post message
                eReturnCode = OSAL.eQueuePut(pacTestMsg, un32TestMessageSize);
                if(eReturnCode == OSAL_SUCCESS)
                {
#if OSAL_OBJECT_TRACKING == 1
                    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)hQueue;

                    printf("Queue 'put' is successful (sent '%s' to '%s' queue)\n",
                           pacMessage, psObj->pacName);
#else
                    printf("Queue 'put' is successful (sent '%s' to queue)\n",
                           pacMessage);
#endif
                }
                else
                {
                    printf("Error! OSAL.eQueuePut() returned %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
            else
            {
                printf("Error! OSAL.eMessageAllocate() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! Invalid queue handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "pqu") == 0)
    {
        OSAL_QUEUE_INFO_STRUCT sQueueInfo;
        char *pacTestMsg, *pacMessage;
        UN32 un32TestMessageSize;

        // Get next tokens...
        pacMessage = strtok( NULL, "\n" );
        if (pacMessage == NULL)
        {
            printf("Error! Invalid arguments! \n");
        }
        else if(hQueue != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eQueueGetInfo(hQueue, &sQueueInfo);

            eReturnCode = OSAL.eMessageAllocate(hQueue, (void **)&pacTestMsg,
                                                OSAL_QUEUE_FLAG_NONBLOCK|
                                                OSAL_QUEUE_FLAG_URGENT);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Populate message
                strcpy(pacTestMsg, pacMessage);
                un32TestMessageSize = strlen(pacTestMsg) + 1;

                // Post message
                eReturnCode = OSAL.eQueuePut(pacTestMsg, un32TestMessageSize);
                if(eReturnCode == OSAL_SUCCESS)
                {
#if OSAL_OBJECT_TRACKING == 1
                    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)hQueue;

                    printf("Queue 'put' is successful (sent '%s' to '%s' queue)\n",
                           pacMessage, psObj->pacName);
#else
                    printf("Queue 'put' is successful (sent '%s' to queue)\n",
                           pacMessage);
#endif
                }
                else
                {
                    printf("Error! OSAL.eQueuePut() returned %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
            else
            {
                printf("Error! OSAL.eMessageAllocate() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! Invalid queue handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "gq") == 0)
    {
        char *pacTestMsg;
        UN32 un32TestMessageSize;

        if(hQueue != OSAL_INVALID_OBJECT_HDL)
        {
            // Grab from the queue
            eReturnCode = OSAL.eQueueGet(hQueue, (void **)&pacTestMsg,
                                                &un32TestMessageSize, 0);
            if(eReturnCode == OSAL_SUCCESS)
            {
                // Got a message
                printf("Got messasge '%s' of length %u\n",
                       pacTestMsg, un32TestMessageSize);

                // Now free the message
                eReturnCode = OSAL.eMessageFree(pacTestMsg);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    printf("Error! OSAL.eMessageFree() returned %s\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                }
            }
            else
            {
                printf("Error! OSAL.eQueueGet() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! Invalid queue handle\n");
        }
    }
    else if (strcmp(pcCmdLine, "?q") == 0)
    {
        UN32 un32QueueNumber = 0;
        OSALQ_bPrintQueue(hQueue, &un32QueueNumber);
    }
    else if (strcmp(pcCmdLine, "iq") == 0)
    {
        char *pacName;
        OSAL_OBJECT_HDL hTempQ;

        // Get name parameter
        pacName = strtok( NULL, "\n" );

        // Find the object this name belongs to
        hTempQ = OSAL.hQueueGetHandleByName(pacName);
        if(hTempQ != OSAL_INVALID_OBJECT_HDL)
        {
            UN32 un32QueueNumber = 0;
            hQueue = hTempQ;
            OSALQ_bPrintQueue(hQueue, &un32QueueNumber);
        }
        else
        {
            printf("Error! Invalid queue name.\n");
        }
    }
    else if (strcmp(pcCmdLine, "timers") == 0)
    {
        OSAL_OBJECT_HDL hTimer;
        char *pacName = "OSAL_Timer";
        UN32 un32Led;
        UN32 un32InitialOffset, un32PeriodicRate;

        // Get next token...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            return eState;
        }
        un32Led = (UN32)atoi(pcCmdLine);

        // Get next tokens...
        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            return eState;
        }
        un32InitialOffset = (UN32)atoi(pcCmdLine);

        pcCmdLine = strtok( NULL, acTokens );
        if (pcCmdLine == NULL)
        {
            return eState;
        }
        un32PeriodicRate = (UN32)atoi(pcCmdLine);

        // Create a timer for the sample LED blink
        eReturnCode = OSAL.eTimerCreate(
            &hTimer, pacName,
            OSALD_vToggleLed, (void *)un32Led);
        if(eReturnCode == OSAL_SUCCESS)
        {
            printf("Created timer, %s, hdl = %#0x\n", pacName, hTimer);

            // Start timer
            eReturnCode = OSAL.eTimerStartRelative(
                            hTimer,
                            un32InitialOffset,
                            un32PeriodicRate );

            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Unable to start timer! %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
            else
            {
                printf("started timer, io = %u, pr = %u, hdl = %#0x\n",
                       un32InitialOffset, un32PeriodicRate, hTimer);
            }
        }
        else
        {
            printf("Error! OSAL.eTimerCreate() returned %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else if ((strcmp(pcCmdLine, "help") == 0) ||
            (strcmp(pcCmdLine, "HELP") == 0))
    {
        printf("\nAvailable commands:\n");
        printf("\tallocate              -\tAllocate 1000 bytes of memory\n");
        printf("\tfree                  -\tFree 1000 bytes of memory\n");
        printf("\tdelay <msec>          -\tDelay a number of milliseconds\n");
        printf("\tcs init resources     -\tCreate a semaphore\n");
        printf("\tds                    -\tDelete current semaphore\n");
        printf("\tgs                    -\tGive current semaphore\n");
        printf("\tts                    -\tTake current semaphore\n");
        printf("\t?s                    -\tShow current semaphore\n");
        printf("\tis name               -\tInvoke current semaphore\n");
        printf("\tct priority           -\tCreate a task 1=l, 2=m, 3=h\n");
        printf("\tdt                    -\tDelete a task\n");
        printf("\tst                    -\tSuspend current task\n");
        printf("\trt                    -\tResume current task\n");
        printf("\t?t                    -\tShow current task\n");
        printf("\tit name               -\tInvoke current task\n");
        printf("\tcq depth size         -\tCreate a queue\n");
        printf("\tdq                    -\tDelete a queue\n");
        printf("\tpq msg                -\tPost to a queue\n");
        printf("\tpqu msg               -\tPost as URGENT to a queue\n");
        printf("\tgq                    -\tGet from a queue\n");
        printf("\t?q                    -\tShow current queue\n");
        printf("\tiq name               -\tInvoke current queue\n");
        printf("\ttimers led offset rate-\tInstall and test timers\n");
        printf("\thelp                  -\tdisplay this message\n");
        printf("\texit                  -\texit this menu\n");
    }
    else if (strcmp(pcCmdLine, "exit") == 0)
    {
        eState = OSAL_DEBUG_EXIT;
    }
    else
    {
        printf("\nInvalid command.  Type 'help' for a list of commands.\n");
    }

    return eState;
}

/******************************************************************************
 *
 *      OSALD_n32Task
 *
 ******************************************************************************/
static N32 OSALD_n32Task( void *pvArg )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    void *pvTestMemory = NULL;
    UN32 un32Instance = (UN32)pvArg, un32TestMsgSize;
    char *pacTestMessage = NULL;
    OSAL_OBJECT_HDL hQueue = OSAL_INVALID_OBJECT_HDL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    BOOLEAN bReport = TRUE;

    // Construct an appropriate name for this task
    snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%02d",
            "Debug:T#", un32Instance);

    // Install task into the debug handler
    OSAL.eDebugRegister( acName, OSALD_vDebugTaskHandler, pvArg );

    // Construct an appropriate name for queue to be created
    snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%02d",
            "Debug:Q#", un32Instance);

    // Create a queue
    eReturnCode = OSAL.eQueueCreate(
        &hQueue, acName, 3, 100,
        OSAL_QUEUE_OPTION_VARIABLE_SIZE);
    if(eReturnCode != OSAL_SUCCESS)
    {
        printf("Error! Unable to create a queue. %s\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
    }

    // Register Task with OSAL
    eReturnCode = OSAL.eTaskRegister(
                   5, // seconds
                   OSALD_vSleepHandler,
                   OSALD_vWakeupHandler,
                   OSALD_vShutdownHandler,
                   (void *)un32Instance );
    if(eReturnCode != OSAL_SUCCESS)
    {
        return !OSAL_TASK_REPORT_NO_ERROR;
    }

    for(;;)
    {
//        printf("Test My Task #%u\n", un32Instance);

        if(bReport == TRUE)
        {
            // Report to task monitor
            OSAL.eTaskReport(OSAL_TASK_REPORT_NO_ERROR);
        }

        if(pvTestMemory == NULL)
        {
            // Construct an appropriate name for memory to be created
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                    "%s%02d", "Test#", un32Instance);

            // Allocate some memory for testing
            pvTestMemory = OSAL.pvMemoryAllocate(
                         acName, 16 * un32Instance, FALSE);

//            printf("Task #%u allocated memory block %#010x\n",
//                   un32Instance, pvTestMemory);

        }
        else
        {
            // Free the memory
            OSAL.vMemoryFree(pvTestMemory);

//            printf("Task #%u freed memory block %#010x\n",
//                   un32Instance, pvTestMemory);

            pvTestMemory = NULL;
        }

        // Pend on a message from my queue (1000 msec)
        eReturnCode = OSAL.eQueueGet( hQueue, (void**)&pacTestMessage,
                                      &un32TestMsgSize, 1000);
        if(eReturnCode == OSAL_SUCCESS)
        {
            BOOLEAN bExit = FALSE;

            // Got a message, print it
            if(pacTestMessage != NULL)
            {
                printf("%s got message '%s' of length %u\n",
                       acName, pacTestMessage, un32TestMsgSize);

                if(strcmp(pacTestMessage, "sleep") == 0)
                {
                    // Ignore
                }
                else if(strcmp(pacTestMessage, "wakeup") == 0)
                {
                    // Ignore
                }
                else if(strcmp(pacTestMessage, "shutdown") == 0)
                {
                    // Exit task
                    bExit = TRUE;
                }
                else if(strcmp(pacTestMessage, "stop") == 0)
                {
                    // Stop task from reporting to task monitor
                    // this is used to test the report feature.
                    bReport = FALSE;
                }
            }

            // Now we're done with it
            eReturnCode = OSAL.eMessageFree(pacTestMessage);
            if(eReturnCode != OSAL_SUCCESS)
            {
                printf("Error! OSAL.eMessageFree() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }

            if(bExit == TRUE)
            {
                // Exit task
                break;
            }
        }
        else if(eReturnCode == OSAL_TIMEOUT)
        {
            OSAL_QUEUE_INFO_STRUCT sQueueInfo;

            // Timeout occured, do nothing
            eReturnCode = OSAL.eQueueGetInfo(hQueue, &sQueueInfo);
            if(eReturnCode == OSAL_SUCCESS)
            {
//                printf("No message waiting on queue named '%s'\n",
//                       sQueueInfo.acName);
            }
            else
            {
                printf("Error! OSAL.eQueueGetInfo() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            // Some unexpected error
            printf("Error! OSAL.eQueueGet() returned %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
            break;
        }
    }

    // Unregister Task with OSAL
    OSAL.eTaskUnregister();

    // Free any allocated memory
    if(pvTestMemory != NULL)
    {
        // Free the memory
        OSAL.vMemoryFree(pvTestMemory);
        pvTestMemory = NULL;
    }

    // Release queue
    if(hQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eQueueDelete(hQueue);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("Error! Unable to delete the queue. %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
        else
        {
            hQueue = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Unregister us from the debug handler
    OSAL.eDebugUnregister();

    // This task is now done
    return OSAL_TASK_REPORT_NO_ERROR;
}

/*****************************************************************************
*
*       OSALD_vDebugTaskHandler
*
*****************************************************************************/
static void OSALD_vDebugTaskHandler( char *pcCmdLine, const void *pvArg )
{
    char *acTokens = " \n";

    // Make sure NULL pointer is not passed
    if (pcCmdLine != NULL)
    {
        // Tokenize the command line characters
        pcCmdLine = strtok( pcCmdLine, acTokens );

        // need to signal sw to Normal Mode!!
        OSAL.vDebugExitHandler();
    }

    return;
}

/*****************************************************************************
*
*       OSALD_vSleepHandler
*
*****************************************************************************/
static void OSALD_vSleepHandler( void *pvArg )
{
    UN32 un32Instance = (UN32)pvArg;
    char *pacTestMsg, *pacCommand = "sleep";
    OSAL_OBJECT_HDL hQueue;
    UN32 un32TestMessageSize;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acQueueName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    printf("Execute sleep handler for task instance #%u\n", un32Instance);

    snprintf(acQueueName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%02d",
            "Debug:Q#", un32Instance);

    // Find the queue this belongs to
    hQueue = OSAL.hQueueGetHandleByName(acQueueName);
    if(hQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eMessageAllocate(hQueue, (void *)&pacTestMsg,
                                            OSAL_QUEUE_FLAG_BLOCK);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Populate message
            strcpy(pacTestMsg, pacCommand);
            un32TestMessageSize = strlen(pacTestMsg) + 1;

            // Post message
            eReturnCode = OSAL.eQueuePut(pacTestMsg, un32TestMessageSize);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("Queue 'put' is successful (sent '%s' to '%s' queue)\n",
                       pacTestMsg, acQueueName);
            }
            else
            {
                printf("Error! OSAL.eQueuePut() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! OSAL.eMessageAllocate() returned %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        printf("Error! Cannot find queue named %s\n",
               acQueueName);
    }

    return;
}

/*****************************************************************************
*
*       OSALD_vWakeupHandler
*
*****************************************************************************/
static void OSALD_vWakeupHandler( void *pvArg )
{
    UN32 un32Instance = (UN32)pvArg;
    char *pacTestMsg, *pacCommand = "wakeup";
    OSAL_OBJECT_HDL hQueue;
    UN32 un32TestMessageSize;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acQueueName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    printf("Execute wakeup handler for task instance #%u\n", un32Instance);

    snprintf(acQueueName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%02d",
            "Debug:Q#", un32Instance);

    // Find the queue this belongs to
    hQueue = OSAL.hQueueGetHandleByName(acQueueName);
    if(hQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eMessageAllocate(hQueue, (void *)&pacTestMsg,
                                            OSAL_QUEUE_FLAG_BLOCK);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Populate message
            strcpy(pacTestMsg, pacCommand);
            un32TestMessageSize = strlen(pacTestMsg) + 1;

            // Post message
            eReturnCode = OSAL.eQueuePut(pacTestMsg, un32TestMessageSize);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("Queue 'put' is successful (sent '%s' to '%s' queue)\n",
                       pacTestMsg, acQueueName);
            }
            else
            {
                printf("Error! OSAL.eQueuePut() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! OSAL.eMessageAllocate() returned %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        printf("Error! Cannot find queue named %s\n",
               acQueueName);
    }

    return;
}

/*****************************************************************************
*
*       OSALD_vShutdownHandler
*
*****************************************************************************/
static void OSALD_vShutdownHandler( void *pvArg )
{
    UN32 un32Instance = (UN32)pvArg;
    char *pacTestMsg, *pacCommand = "shutdown";
    OSAL_OBJECT_HDL hQueue;
    UN32 un32TestMessageSize;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    char acQueueName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    printf("Execute shutdown handler for task instance #%u\n", un32Instance);

    snprintf(acQueueName, OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
            "%s%02d",
            "Debug:Q#", un32Instance);

    // Find the queue this belongs to
    hQueue = OSAL.hQueueGetHandleByName(acQueueName);
    if(hQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eMessageAllocate(hQueue, (void *)&pacTestMsg,
                                            OSAL_QUEUE_FLAG_BLOCK);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Populate message
            strcpy(pacTestMsg, pacCommand);
            un32TestMessageSize = strlen(pacTestMsg) + 1;

            // Post message
            eReturnCode = OSAL.eQueuePut(pacTestMsg, un32TestMessageSize);
            if(eReturnCode == OSAL_SUCCESS)
            {
                printf("Queue 'put' is successful (sent '%s' to '%s' queue)\n",
                       pacTestMsg, acQueueName);
            }
            else
            {
                printf("Error! OSAL.eQueuePut() returned %s\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
            }
        }
        else
        {
            printf("Error! OSAL.eMessageAllocate() returned %s\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
    }
    else
    {
        printf("Error! Cannot find queue named %s\n",
               acQueueName);
    }

    return;
}

/*******************************************************************************
*
*   OSALD_vToggleLed
*
*******************************************************************************/
static void OSALD_vToggleLed ( OSAL_OBJECT_HDL hTimer, void *pvArg )
{
    static UN32 un32Count = 0;
    UN32 un32Led = (UN32)pvArg;

    // Toggle LED
    printf("TOGGLE LED #%u\n", un32Led);

    // Increment and check counter
    if(++un32Count >= 4)
    {
        // Kill timer from within the timer callback
        OSAL.eTimerStop(hTimer);
        un32Count = 0;
    }

    return;
}

#endif /* OSAL_TEST == 1 */

#if OSAL_FILE_SYSTEM == 1

/*****************************************************************************
*
*       OSALD_eDebugFSMenuHandler
**
*       Inputs:
*               *pcTaskParameters	Command line entry string before CR/LF
*
*       Outputs:
*               NONE
*
*****************************************************************************/
static OSAL_DEBUG_STATE_ENUM OSALD_eDebugFSMenuHandler (
        char *pcCmdLine, char *pcWorkingDir )
{
    OSAL_DEBUG_STATE_ENUM eState = OSAL_DEBUG_FS;
    char acPath[OSALD_MAXIMUM_PATH_LEN] = "";

    // Decode the command line
    if (strcmp(pcCmdLine, "list") == 0)
    {
        OSAL.vFileSystemVolumes(stdout);
    }
    else if (strcmp(pcCmdLine, "vol") == 0)
    {
        const char *pcVolName, *acTokens = "\n";

        // Get next tokens...
        pcVolName = strtok( NULL, acTokens );
        if(pcVolName != NULL)
        {
            printf("vol: '%s'\n", pcVolName);
            OSAL.bFileSystemVolumeInfo(stdout, pcVolName);
        }
    }
    else if (strcmp(pcCmdLine, "dir") == 0)
    {
        const char *pcPathName, *acTokens = "\n";

        // Get next tokens...
        pcPathName = strtok( NULL, acTokens );
        if(pcPathName != NULL)
        {
            // A path was given for this command, get the info for it
            if (strlen(pcPathName) > 0)
            {
                // Generate a path for this mkdir request
                OSALD_GeneratePath( pcWorkingDir, pcPathName, acPath );

                // Attempt to this the directory contents for the
                // requested location.
                printf("dir: '%s'\n", acPath);
                OSAL.bFileSystemDirectory( stdout, acPath );
            }
        }
        // No path was given, use the current working directory
        else
        {
            printf("dir: '%s'\n", pcWorkingDir);
            OSAL.bFileSystemDirectory(stdout, pcWorkingDir);
        }
    }
    else if (strcmp(pcCmdLine, "cd") == 0)
    {
        const char *pcDirName, *acTokens = " \n";
        const char *pcLastDir, *pcVolume;
        ptrdiff_t tCharDifference;
        BOOLEAN bOk;
        UN8 un8Attrs;

        // Get next tokens...
        pcDirName = strtok( NULL, acTokens );
        if(pcDirName != NULL)
        {
            // If no argument was given, cd drops back to the
            // root directory of the currently active volume
            if (strlen(pcDirName) == 0)
            {
                pcVolume = strrchr( pcWorkingDir, ':' );

                tCharDifference = (pcVolume - pcWorkingDir) + 1;

                // Remove them!
                OSAL.bMemSet( pcWorkingDir + tCharDifference,
                              0,
                              OSALD_MAXIMUM_PATH_LEN - tCharDifference );
            }
            // If the user wants to go up a level via use of "..",
            // then we simply need to remove the last directory named
            // in the working directory, and we're done
            else if (strcmp(pcDirName, "..") == 0)
            {
                // Attempt to locate the last directory indicated in the current
                // working path by finding the '/' character
                pcLastDir = strrchr( pcWorkingDir, '/' );

                // The '/' character wasn't found.  There is no place to back
                // up to in the current working path
                if (pcLastDir == NULL)
                {
                    printf( "Error! Already at top level directory for this"
                        " volume\n" );
                }
                // We have found the last directory in the current working
                // path, so we can now remove it
                else
                {
                    // Determine how many characters need to be removed
                    tCharDifference = pcLastDir - pcWorkingDir;

                    // Remove them!
                    OSAL.bMemSet( pcWorkingDir + tCharDifference,
                                  0,
                                  OSALD_MAXIMUM_PATH_LEN - tCharDifference );
                }
            }
            // Only attempt an actual "change directory" if the directory
            // indicated is not the current one (specified via ".")
            else if (strcmp(pcDirName, ".") != 0)
            {
                // Create a full path from the user's argument
                // and the working path we are using
                snprintf( acPath, sizeof(acPath),
                          "%s/%s", pcWorkingDir, pcDirName );

                // Try the full path first
                bOk = OSAL.bFileSystemGetFileAttributes( acPath, &un8Attrs );

                if (bOk == FALSE)
                {
                    // Try the directory alone next
                    bOk = OSAL.bFileSystemGetFileAttributes(
                        pcDirName, &un8Attrs );
                    strncpy(acPath, pcDirName, sizeof(acPath)-1);
                    acPath[sizeof(acPath)-1] = '\0';
                }

                // Either the full path or relative worked -- one is
                // indicating a file which exists
                if (bOk == TRUE)
                {
                    // Only allow "cd" to a directory
                    if ( un8Attrs & OSAL_FILE_ATTR_DIRECTORY )
                    {
                        strncpy( pcWorkingDir, acPath, sizeof(acPath) );
                    }
                    else
                    {
                        printf( "Error! '%s' is not a directory\n",
                                pcDirName );
                    }
                }
                else
                {
                    printf( "Error! Unable to locate '%s'\n",
                            pcDirName );
                }
            }
        }
    }
    else if (strcmp(pcCmdLine, "pwd") == 0)
    {
        // Print the working directory
        printf("%s\n", pcWorkingDir);
    }
    else if (strcmp(pcCmdLine, "rename") == 0)
    {
        const char *pcOldName, *pcNewName, *acTokens = " \n";
        BOOLEAN bOk;
        UN8 un8Attrs;
        UN8 un8RenameResult;
        char acNewPath[OSALD_MAXIMUM_PATH_LEN] = "";


        // Get the next tokens...
        pcOldName = strtok( NULL, acTokens );
        pcNewName = strtok( NULL, acTokens );
        if((pcOldName != NULL) && (pcNewName != NULL))
        {
            // Generate a path for this rename request
            OSALD_GeneratePath( pcWorkingDir, pcOldName, acPath );
            OSALD_GeneratePath( pcWorkingDir, pcNewName, acNewPath );

            // Find out about the file in question
            bOk = OSAL.bFileSystemGetFileAttributes( acPath, &un8Attrs );

            if (bOk == TRUE)
            {
                // Peform rename of the file / directory
                printf("rename: '%s to %s'\n", acPath, acNewPath);
                un8RenameResult = rename( acPath, acNewPath );

                if (un8RenameResult != 0)
                {
                    printf("Unable to rename '%s'", acPath );
                }
            }
            else
            {
                printf( "Error! Unable to locate '%s'\n",
                        acPath );
            }
        }
    }
    else if (strcmp(pcCmdLine, "mkdir") == 0)
    {
        const char *pcDirName, *acTokens = " \n";
        BOOLEAN bOk;

        // Get next tokens...
        pcDirName = strtok( NULL, acTokens );
        if(pcDirName != NULL)
        {
            // Generate a path for this mkdir request
            OSALD_GeneratePath( pcWorkingDir, pcDirName, acPath );

            // Now that the path has been setup, attempt to make
            // the requested directory
            printf("mkdir: '%s'\n", acPath);
            bOk = OSAL.bFileSystemMakeDir( acPath );

            if(bOk == FALSE)
            {
                printf( "Error! Unable to create '%s'\n",
                        pcDirName );
            }
        }
    }
    else if (strcmp(pcCmdLine, "rmdir") == 0)
    {
        const char *pcDirName, *acTokens = " \n";
        BOOLEAN bOk;

        // Get next tokens...
        pcDirName = strtok( NULL, acTokens );
        if(pcDirName != NULL)
        {
            // Generate a path for this rmdir request
            OSALD_GeneratePath( pcWorkingDir, pcDirName, acPath );

            // Now that the path has been setup, attempt to remove
            // the requested directory
            printf("rmdir: '%s'\n", acPath);
            bOk = OSAL.bFileSystemRemoveDir( acPath );

            if(bOk == FALSE)
            {
                printf(  "Error! Unable to remove '%s'\n",
                     pcDirName );
            }
        }
    }
    else if (strcmp(pcCmdLine, "touch") == 0)
    {
        const char *pcFileName, *acTokens = " \n";
        FILE *psFile;

        // Get next tokens...
        pcFileName = strtok( NULL, acTokens );
        if (pcFileName != NULL)
        {
            // Generate a path for this touch request
            OSALD_GeneratePath( pcWorkingDir, pcFileName, acPath );

            printf("touch: '%s'\n", acPath);
            psFile = fopen(acPath, "w");

            if (psFile == NULL)
            {
                printf( "Error! Unable to create '%s'\n",
                        pcFileName );
            }
            else
            {
                fclose(psFile);
            }
        }
    }
    else if (strcmp(pcCmdLine, "del") == 0)
    {
        const char *pcFileName, *acTokens = " \n";
        UN8 un8RemoveResult;

        // Get next tokens...
        pcFileName = strtok( NULL, acTokens );

        if (pcFileName != NULL)
        {
            // Generate a path for this del request
            OSALD_GeneratePath( pcWorkingDir, pcFileName, acPath );

            // Perform remove of the file
            printf("del: '%s'\n", acPath);
            un8RemoveResult = remove( acPath );

            if (un8RemoveResult != 0)
            {
                printf( "Error! Unable to remove '%s'\n",
                        pcFileName );
            }
        }
    }
    else if (strcmp(pcCmdLine, "hex") == 0)
    {
        const char *pcFileName, *acTokens = " \n";
        UN8 un8Attrs;
        BOOLEAN bOk;
        FILE *psFile;

        // Get next tokens...
        pcFileName = strtok( NULL, acTokens );

        if (NULL != pcFileName)
        {
            // Generate a path for this hex request
            OSALD_GeneratePath( pcWorkingDir, pcFileName, acPath );

            // Find out about the file in question
            printf("hex: '%s'\n", acPath);
            bOk = OSAL.bFileSystemGetFileAttributes( acPath, &un8Attrs );

            if (bOk == TRUE)
            {
                // We cannot hex dump a directory -- this command only
                // works on files
                if ( un8Attrs & OSAL_FILE_ATTR_DIRECTORY )
                {
                    printf( "Error! Unable to hex dump a directory\n" );
                }
                else
                {
                    // Open file as read-only
                    psFile = fopen(acPath, "r");
                    if(psFile != NULL)
                    {
                        char acBuffer[16];
                        N32 n32Num;
                        UN16 un16Row = 0;

                        // Dump file a chunk at a time
                        do
                        {
                            // Read a chunk
                            n32Num = fread(&acBuffer[0], 
                                sizeof(acBuffer), 
                                1, 
                                psFile);
                            if(n32Num > 0)
                            {
                                // Dump chunk
                                OSAL.vDump(stdout, 
                                    &un16Row, 
                                    &acBuffer[0],
                                    n32Num * sizeof(acBuffer));
                            }
                            else
                            {
                                // Something went wrong?
                                printf("Error (%d)! Unable to read file '%s'\n",
                                    n32Num,
                                    pcFileName);
                                break;
                            }

                        } while(un16Row < 32);

                        // Close file
                        fclose(psFile);
                    }
                    else
                    {
                        printf("Error! Cannot open '%s'\n", pcFileName);
                    }
                }
            }
            else
            {
                printf( "Error! Unable to locate '%s'\n",
                    pcFileName );
            }
        }
    }
    else if ((strcmp(pcCmdLine, "help") == 0) ||
            (strcmp(pcCmdLine, "HELP") == 0))
    {
        printf("\nAvailable commands:\n");
        printf("\tlist                  -\tDisplay mounted volumes\n");
        printf("\tvol vol_name          -\tDisplay disk info\n");
        printf("\tdir pathname          -\tDisplay directory\n");
        printf("\tcd path               -\tChange current directory\n");
        printf("\tpwd                   -\tPrint working directory\n");
        printf("\ttouch file_name       -\tCreate an empty file\n");
        printf("\tdel file_name         -\tDelete a file\n");
        printf("\tmkdir dir_name        -\tCreate directory\n");
        printf("\trmdir dir_name        -\tRemove directory\n");
        printf("\trename old new        -\tRename a file or directory\n");
        printf("\thex file_name         -\tHex dump a file\n");
        printf("\thelp                  -\tdisplay this message\n");
        printf("\texit                  -\texit this menu\n");
    }
    else if (strcmp(pcCmdLine, "exit") == 0)
    {
        eState = OSAL_DEBUG_EXIT;
    }
    // If the command was a volume label,
    // switch the working directory to that volume
    else if (strchr(pcCmdLine, ':') != 0)
    {
        strncpy(pcWorkingDir, pcCmdLine, strlen(pcCmdLine));
    }
    else
    {
        printf("\nInvalid command.  Type 'help' for a list of commands.\n");
    }

    return eState;
}

/*****************************************************************************
*
*       OSALD_GeneratePath
*
*      	This function is utilized within the FS debug handler to build a
*       valid path to a user-requested element (file or directory) located
*       within a filesystem volume.
*
*
*       Inputs:
*               *pcWorkingDir   The current working directory of the FS
*                               debug handler
*               *pcTarget       The requested filesystem target, given by
*                               name, provided by the user
*               *pcDestBuffer   The destination buffer which will store
*                               the generated path
*
*       Outputs:
*               NONE
*
*****************************************************************************/
static void OSALD_GeneratePath(
        char *pcWorkingDir, const char *pcTarget, char *pcDestBuffer )
{
    // If the target specifies a full path (determined by
    // the presence of an ':' character in the volume label)
    // we don't need to make any adjustments to the path
    if ( strstr(pcTarget, ":") != 0)
    {
        OSAL.bMemCpy( pcDestBuffer, pcTarget, strlen(pcTarget) );
    }
    // Otherwise, this is a relative path and needs to be
    // appended to the current working directory
    else
    {
        // Create the full path
        snprintf( pcDestBuffer, OSALD_MAXIMUM_PATH_LEN,
                  "%s/%s", pcWorkingDir, pcTarget );
    }
}
#endif /* OSAL_FILE_SYSTEM == 1 */

#if OSAL_FIXED_MATH == 1

/*****************************************************************************
*
*       OSALD_eDebugFixedMenuHandler
**
*       Inputs:
*               *pcTaskParameters   Command line entry string before CR/LF
*
*       Outputs:
*               NONE
*
*****************************************************************************/
static OSAL_DEBUG_STATE_ENUM OSALD_eDebugFixedMenuHandler (
        char *pcCmdLine, const void *pvArg )
{
    OSAL_DEBUG_STATE_ENUM eState = OSAL_DEBUG_FIXED;
    // points around the unit circle. (in pi/4 (45 deg) increments)
    const float afX[16] = {1.0f, 0.866f, 0.7071f, 0.5f, 0.0f, -0.5f, -0.7071f, -0.866f, -1.0f, -0.866f, -0.7071f, -0.5f, 0.0f, 0.5f, 0.7071f, 0.866f};
    const float afY[16] = {0.0f, 0.5f, 0.7071f, 0.866f, 1.0f,  0.866f, 0.7071f, 0.5f, 0.0f, -0.5f, -0.7071f, -0.866f, -1.0f, -0.866f, -0.7071f, -0.5f};
    // angles around the unit circle (in pi/4 (45 deg) increments)
    const float afAngle[9] =  {0.0f,           // 0
                        0.7853975f,      // pi/4
                        1.570795f,       // pi/2
                        2.3561925f,      // 3pi/4
                        3.14159f,        // pi
                        3.9269875f,      // 5pi/4
                        4.712385f,       // 3pi/2
                        5.4977825f,      // 7pi/4
                        6.2831853f};     // 2pi


    // Decode the command line
    if (strcmp(pcCmdLine, "atan") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        float fValue = -1;
        N32 n32Value;
        UN8 un8NumBits;
        OSAL_FIXED_OBJECT hFixed;
        OSAL_FIXED_OBJECT hResult;

        hResult = OSAL_FIXED.hCreateFromFixed(0,0);

        while (fValue <= 1)
        {
            n32Value = (N32)(fValue * (1<< 7));
            hFixed = OSAL_FIXED.hCreateFromFixed(n32Value, 7);

            eReturnCode = OSAL_FIXED.eATan(hFixed, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eATan -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);
#if 0
            printf("ATan(%f) is float: %f  fixed: %f\n", fValue,
                   atan(fValue), ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("ATan(%f) is %f\n", fValue,
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif
            fValue += (float)0.1;
            OSAL_FIXED.vDestroy(hFixed);
        }

        OSAL_FIXED.vDestroy(hResult);
    }
    else if (strcmp(pcCmdLine, "atan2") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        N32 n32Value;
        UN8 un8NumBits, un8Index;
        OSAL_FIXED_OBJECT hY, hX;
        OSAL_FIXED_OBJECT hResult =
            OSAL_FIXED.hCreateFromFixed(0,0);

        for (un8Index = 0; un8Index < sizeof(afX)/sizeof(float); un8Index++)
        {
            n32Value = (N32)(afX[un8Index] * (1<< 10));
            hX = OSAL_FIXED.hCreateFromFixed(n32Value, 20);
            n32Value = (N32)(afY[un8Index] * (1<< 10));
            hY = OSAL_FIXED.hCreateFromFixed(n32Value, 20);

            eReturnCode = OSAL_FIXED.eATan2(hY, hX, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eATan2 -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);
#if 0
            printf("ATan2(%f, %f) is float: %f  fixed: %f\n",
                afY[un8Index], afX[un8Index],
                atan2(afY[un8Index], afX[un8Index]),
                ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("ATan2(%f, %f) is %f\n",
                afY[un8Index], afX[un8Index],
                ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif
            OSAL_FIXED.vDestroy(hX);
            OSAL_FIXED.vDestroy(hY);
        }

        OSAL_FIXED.vDestroy(hResult);
    }
    else if (strcmp(pcCmdLine, "pow") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        OSAL_FIXED_OBJECT hFixed =
            OSAL_FIXED.hCreateFromFixed((N32)(1.5 * (1 << 20)),20);
        OSAL_FIXED_OBJECT hResult =
            OSAL_FIXED.hCreateFromFixed(0,0);
        UN8 un8Power = 0;

        while (un8Power < 10)
        {
            eReturnCode = OSAL_FIXED.ePow(hFixed, un8Power, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.ePow -- %d\n", eReturnCode);
                break;
            }

            printf("1.5^%d is %f\n", un8Power,
                ((float)OSAL_FIXED.n32Value(hResult))/(1<< OSAL_FIXED.un8NumFractionalBits(hResult)));

            un8Power += 1;
        }
        OSAL_FIXED.vDestroy(hFixed);
        OSAL_FIXED.vDestroy(hResult);

    }
    else if (strcmp(pcCmdLine, "asin") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        float fValue = -1;
        N32 n32Value;
        UN8 un8NumBits;
        OSAL_FIXED_OBJECT hFixed;
        OSAL_FIXED_OBJECT hResult =
            OSAL_FIXED.hCreateFromFixed(0,0);

        while (fValue <= 1)
        {
            n32Value = (N32)(fValue * (1<< 29));
            hFixed = OSAL_FIXED.hCreateFromFixed(n32Value, 29);

            eReturnCode = OSAL_FIXED.eASin(hFixed, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eASin -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);

#if 0
            printf("ASin(%f) is float:%f  fixed:%f\n", fValue,
                   asin(fValue), ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("ASin(%f) is %f\n", fValue,
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif
            fValue += (float)0.1;
            OSAL_FIXED.vDestroy(hFixed);
        }

        OSAL_FIXED.vDestroy(hResult);

    }
    else if (strcmp(pcCmdLine, "acos") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        float fValue = -1;
        N32 n32Value;
        UN8 un8NumBits;
        OSAL_FIXED_OBJECT hFixed;
        OSAL_FIXED_OBJECT hResult =
            OSAL_FIXED.hCreateFromFixed(0,0);

        while (fValue < 1.1f)
        {
            n32Value = (N32)(fValue * (1<< 20));
            hFixed = OSAL_FIXED.hCreateFromFixed(n32Value, 20);

            eReturnCode = OSAL_FIXED.eACos(hFixed, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eACos -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);

            if (fValue > 1.0f)
            {
                fValue = 1.0f;
            }

#if 0
            printf("ACos(%f) is float:%f  fixed:%f\n", fValue,
                   acos(fValue), ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("ACos(%f) is %f\n", fValue,
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif

            fValue += (float)0.1;
            OSAL_FIXED.vDestroy(hFixed);
        }

        OSAL_FIXED.vDestroy(hResult);

    }
    else if (strcmp(pcCmdLine, "sin") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        N32 n32Value;
        UN8 un8NumBits, un8Index;
        OSAL_FIXED_OBJECT hFixed;
        OSAL_FIXED_OBJECT hResult =
                OSAL_FIXED.hCreateFromFixed(0,0);

        for (un8Index = 0; un8Index < sizeof(afAngle)/sizeof(float); un8Index++)
        {
            n32Value = (N32)(afAngle[un8Index] * (1<< 20));
            hFixed = OSAL_FIXED.hCreateFromFixed(n32Value, 20);

            eReturnCode = OSAL_FIXED.eSin(hFixed, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eSin -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);

#if 0
            printf("Sin(%f (%f deg)) is float:%f  fixed:%f\n", afAngle[un8Index],
                   (float)afAngle[un8Index]*180.f/3.14159f, // convert rad to deg
                   sin(afAngle[un8Index]),
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("Sin(%f (%f deg)) is %f\n", afAngle[un8Index],
                   (float)afAngle[un8Index]*180.f/3.14159f, // convert rad to deg
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif

            OSAL_FIXED.vDestroy(hFixed);
        }

        OSAL_FIXED.vDestroy(hResult);
    }
    else if (strcmp(pcCmdLine, "cos") == 0)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;
        N32 n32Value;
        UN8 un8NumBits, un8Index;
        OSAL_FIXED_OBJECT hFixed;
        OSAL_FIXED_OBJECT hResult =
                OSAL_FIXED.hCreateFromFixed(0,0);

        for (un8Index = 0; un8Index < sizeof(afAngle)/sizeof(float); un8Index++)
        {
            n32Value = (N32)(afAngle[un8Index] * (1<< 20));
            hFixed = OSAL_FIXED.hCreateFromFixed(n32Value, 20);

            eReturnCode = OSAL_FIXED.eCos(hFixed, hResult);

            if (eReturnCode != OSAL_SUCCESS)
            {
                printf("error with OSAL_FIXED.eCos -- %d\n", eReturnCode);
                break;
            }

            n32Value = OSAL_FIXED.n32Value(hResult);
            un8NumBits = OSAL_FIXED.un8NumFractionalBits(hResult);

#if 0
            printf("Cos(%f (%f deg)) is float:%f  fixed:%f\n", afAngle[un8Index],
                   (float)afAngle[un8Index]*180.f/3.14159f, // convert rad to deg
                   cos(afAngle[un8Index]),
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#else
            printf("Cos(%f (%f deg)) is %f\n", afAngle[un8Index],
                   (float)afAngle[un8Index]*180.f/3.14159f, // convert rad to deg
                   ((float)n32Value)/((UN32)(1 << un8NumBits)));
#endif

            OSAL_FIXED.vDestroy(hFixed);
        }

        OSAL_FIXED.vDestroy(hResult);
    }
    else if ((strcmp(pcCmdLine, "help") == 0) ||
            (strcmp(pcCmdLine, "HELP") == 0))
    {
        printf("\nAvailable commands:\n");
        printf("\tpow                   -\tTest the pow function\n");
        printf("\tatan                  -\tTest the atan function\n");
        printf("\tatan2                 -\tTest the atan2 function\n");
        printf("\tasin                  -\tTest the asin function\n");
        printf("\tacos                  -\tTest the acos function\n");
        printf("\tsin                   -\tTest the sin function\n");
        printf("\tcos                   -\tTest the cos function\n");
        printf("\thelp                  -\tdisplay this message\n");
        printf("\texit                  -\texit this menu\n");
    }
    else if (strcmp(pcCmdLine, "exit") == 0)
    {
        eState = OSAL_DEBUG_EXIT;
    }
    else
    {
        printf("\nInvalid command.  Type 'help' for a list of commands.\n");
    }

    return eState;
}
#endif /* OSAL_FIXED_MATH == 1 */

/*****************************************************************************
*
*       OSALD_vDebugMenuHandler
*
*      	This function is a debug handler that a function that has registered
*		itself with the debug handler can use to handle command prompt entries
*		in MENU_SINGLE_TASK_CONTROL mode.
*
*
*       Inputs:
*               *pcTaskParameters	Command line entry string before CR/LF
*
*       Outputs:
*               NONE
*
*****************************************************************************/
void OSALD_vDebugMenuHandler( char *pcCmdLine, const void *pvArg )
{
    char *pcArg;
    char *acTokens = " \n";
    static char acWorkingDir[OSALD_MAXIMUM_PATH_LEN] = "";

    static OSAL_DEBUG_STATE_ENUM eState = OSAL_DEBUG_MAIN;

    // Make sure NULL pointer is not passed
    if (pcCmdLine != NULL)
    {
        // Tokenize the command line characters
        pcArg = strtok( pcCmdLine, acTokens );
        if (pcArg == NULL)
        {
            printf("\nInvalid command.  Type 'help' for a list of commands.\n");
            return;
        }

        switch(eState)
        {
            case OSAL_DEBUG_MAIN:
                eState = OSALD_eDebugMainMenuHandler(pcArg, pvArg);
                if(eState == OSAL_DEBUG_EXIT)
                {
                    eState = OSAL_DEBUG_MAIN;
                    // need to signal sw to Normal Mode!!
                    OSAL.vDebugExitHandler();
                    return;
                }
            break;

#if OSAL_TEST == 1
            case OSAL_DEBUG_TEST:
                eState = OSALD_eDebugTestMenuHandler(pcArg, pvArg);
                if(eState == OSAL_DEBUG_EXIT)
                {
                    eState = OSAL_DEBUG_MAIN;
                }
            break;
#endif /* OSAL_TEST == 1 */

#if OSAL_FILE_SYSTEM == 1

            case OSAL_DEBUG_FS:
                eState = OSALD_eDebugFSMenuHandler(pcArg, acWorkingDir);//pvArg);
                if(eState == OSAL_DEBUG_EXIT)
                {
                    eState = OSAL_DEBUG_MAIN;
                }
            break;

#endif /* OSAL_FILE_SYSTEM == 1 */

#if OSAL_FIXED_MATH == 1

            case OSAL_DEBUG_FIXED:
                eState = OSALD_eDebugFixedMenuHandler(pcArg, pvArg);
                if(eState == OSAL_DEBUG_EXIT)
                {
                    eState = OSAL_DEBUG_MAIN;
                }
            break;

#endif /* OSAL_FIXED_MATH == 1 */

            default:
                eState = OSAL_DEBUG_MAIN;
                // need to signal sw to Normal Mode!!
                OSAL.vDebugExitHandler();
                return;
        }
    }

    return;
}

/*****************************************************************************
*
*       OSALD_vPrintTime
*
*****************************************************************************/
void OSALD_vPrintTime(UN32 un32Seconds, UN16 un16Msecs)
{
    UN16 un16Days;
    UN8 un8Minutes, un8Hours;

    // Compute minutes, hours, days, seconds
    un16Days = un32Seconds / (60 * 60 * 24);
    un32Seconds = un32Seconds % (60 * 60 * 24);

    un8Hours = un32Seconds / (60 * 60);
    un32Seconds = un32Seconds % (60 * 60);

    un8Minutes = un32Seconds / (60);
    un32Seconds = un32Seconds % (60);

    printf("%3u days, %2u hours, %2u minutes, "
           "%2u seconds, %3u milliseconds\n",
           un16Days, un8Hours, un8Minutes, un32Seconds, un16Msecs );

    return;
}

#endif /* OSAL_DEBUG == 1 */
