/******************************************************************************/
/*                     Copyright (c) Sirius XM Radio Inc.                     */
/*                            All Rights Reserved                             */
/*            Licensed Materials - Property of Sirius XM Radio Inc.           */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*DESCRIPTION
*
*	This file contains the OSAL Dynamic Buffer test
*	functions.
*
*******************************************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"
#include "osal.h"

#include "osal_dbuf_test.h"
#include "_osal_dbuf_test.h"

#if OSAL_BUFFER_TEST == 1

/*****************************************************************************
*
*       OSAL_vBufferTest
*
*****************************************************************************/
void OSAL_vBufferTest( void )
{
    // Test OSAL Dynamic Buffers...
    OSAL_RETURN_CODE_ENUM eReturnCode;
    size_t tSize = 0, tTotalRead = 0, tTotalWrite = 0;
    N32 n32ReadTimeout = OSAL_OBJ_TIMEOUT_NONE,
        n32WriteTimeout = OSAL_OBJ_TIMEOUT_NONE;
    OSAL_BUFFER_BLOCK_HDL ahBlock[MAX_NUM_BLOCKS];
    UN8 *apun8Block[MAX_NUM_BLOCKS];
    size_t atBlockSize[MAX_NUM_BLOCKS];
    UN32 un32BlockIndex = 0;
    N8 n8Test = 1;
    BOOLEAN bToggle = FALSE;
    size_t tWriteLen;

    printf("\nTest OSAL Dynamic Buffers...\n\n");

    // Create a block pool for buffers
    eReturnCode =
        OSAL.eBlockPoolCreate (
            &hBlockPool,
            "Test Block Pool",
            22, // size of block
            16, // blocks; 220 bytes
            OSAL_BLOCK_POOL_OPTION_NONE
            );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Failed!
        OSAL_vBufferTestClean();
        printf("Error! Cannot create block pool (%s).\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
        return;
    }
    else
    {
        printf("Created payload block pool.\n");
    }

    // Allocate a buffer from the pool
    hBuffer = OSAL.hBufferAllocate(
        hBlockPool,
        (n32ReadTimeout == OSAL_OBJ_TIMEOUT_NONE) ? FALSE : TRUE,
        (n32WriteTimeout == OSAL_OBJ_TIMEOUT_NONE) ? FALSE : TRUE,
        OSAL_BUFFER_ALLOCATE_OPTION_NONE);
    if(hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        // Failed!
        OSAL_vBufferTestClean();
        printf("Error! Cannot allocate buffer.\n");
        return;
    }
    else
    {
        printf("Allocated a buffer from the pool.\n");
    }

    // Print read/write numbers
    printf("Total bytes read = %u\twritten = %u\n",
           tTotalRead, tTotalWrite);

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    // Grab a block
    ahBlock[un32BlockIndex] =
        OSAL.hBufferGetBlock(
            hBuffer,
            &apun8Block[un32BlockIndex],
            &atBlockSize[un32BlockIndex]
                );
    if(ahBlock[un32BlockIndex] == OSAL_INVALID_BUFFER_BLOCK_HDL)
    {
        // Failed!
        OSAL_vBufferTestClean();
        printf("Error! Cannot allocate a block from buffer.\n");
        return;
    }
    else
    {
        printf("Got Block (%u bytes)\n",
               atBlockSize[un32BlockIndex]);

        // Fill the block
        atBlockSize[un32BlockIndex] /= 2;
        strncpy((char *)apun8Block[un32BlockIndex],
                &acALPHABET[0],
                atBlockSize[un32BlockIndex]);
    }

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    for(;;)
    {
        tWriteLen = strlen(&acAlphabet[0]);
        tSize = OSAL.tBufferWriteTail(
            hBuffer, &acAlphabet[0], tWriteLen, n32WriteTimeout);
        tTotalWrite += tSize;
        printf("Wrote %u bytes to buffer.\n", tSize);
        if(tSize != tWriteLen) break;

        tWriteLen = strlen(&acNumbers[0]);
        tSize = OSAL.tBufferWriteTail(
            hBuffer, &acNumbers[0], tWriteLen, n32WriteTimeout);
        tTotalWrite += tSize;
        printf("Wrote %u bytes to buffer.\n", tSize);
        if(tSize != tWriteLen) break;

        // Grab a block
        un32BlockIndex++;
        if(un32BlockIndex < MAX_NUM_BLOCKS)
        {
            ahBlock[un32BlockIndex] =
                OSAL.hBufferGetBlock(
                    hBuffer,
                    &apun8Block[un32BlockIndex],
                    &atBlockSize[un32BlockIndex]
                        );
            if(ahBlock[un32BlockIndex] == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // Failed!
                OSAL_vBufferTestClean();
                printf("Error! Cannot allocate a block from buffer.\n");
                return;
            }
            else
            {
                printf("Got Block (%u bytes)\n",
                       atBlockSize[un32BlockIndex]);

                // Fill the block
                atBlockSize[un32BlockIndex] /= 2;
                strncpy((char *)apun8Block[un32BlockIndex],
                        &acALPHABET[0],
                        atBlockSize[un32BlockIndex]);
            }
        }
    }

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    do
    {
#if OSAL_DEBUG == 1
    	UN16 un16Row = 0;
#endif /* OSAL_DEBUG == 1 */

        size_t tReadSize = 0;
        BOOLEAN bReadBlock = FALSE;
        tSize = 0;

        do
        {
            tReadSize = 0;
            if(bReadBlock == TRUE)
            {
                unsigned char *pcBuf;
                OSAL_BUFFER_BLOCK_HDL hBlock;
                hBlock = OSAL.hBufferReadBlock(hBuffer, &pcBuf, &tReadSize);
                if(hBlock != OSAL_INVALID_BUFFER_BLOCK_HDL)
                {
                    tTotalRead += tReadSize;
                    printf("Read block.\n");
                    dump(stdout, &un16Row, pcBuf, tReadSize);
                    eReturnCode = OSAL.eBufferReleaseBlock(hBlock);
                    if(eReturnCode != OSAL_SUCCESS)
                    {
                        // Failed!
                        OSAL_vBufferTestClean();
                        printf("Error! Cannot release buffer block (%s).\n",
                               OSAL.pacGetReturnCodeName(eReturnCode));
                        return;
                    }
                    else
                    {
                        printf("Released block.\n");
                    }
                }
                else
                {
                    printf("No more blocks!\n");
                }

                // Toggle read type
                bReadBlock = FALSE;
            }
            else
            {
                printf("Read bytes...\n");
                tReadSize =
                    OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                         sizeof(acBuffer), n32ReadTimeout);
                tTotalRead += tReadSize;
                dump(stdout, &un16Row, &acBuffer[0], tReadSize);

                // Toggle read type
                bReadBlock = TRUE;
            }

            tSize += tReadSize;

        } while(tReadSize > 0);

        printf("Read %u bytes from buffer.\n", tSize);

        un32BlockIndex = 0;

        // Fill allocated blocks
        do
        {
            if(ahBlock[un32BlockIndex] != OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                eReturnCode =
                    OSAL.eBufferWriteBlock(
                        ahBlock[un32BlockIndex],
                        atBlockSize[un32BlockIndex]);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Failed!
                    OSAL_vBufferTestClean();
                    printf("Error! Cannot write block to buffer (%s).\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    return;
                }
                else
                {
                    tTotalWrite += atBlockSize[un32BlockIndex];
                    printf("Wrote block to buffer (%u bytes).\n",
                           atBlockSize[un32BlockIndex]);

                    ahBlock[un32BlockIndex] = OSAL_INVALID_BUFFER_BLOCK_HDL;
                    un32BlockIndex++;
                    break;
                }
            }

            un32BlockIndex++;

        } while(un32BlockIndex < MAX_NUM_BLOCKS);

        // Dump current buffer size
        printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

        // Print read/write numbers
        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);

    } while((tSize != 0) || (un32BlockIndex != MAX_NUM_BLOCKS));

    ////////////////////////////////////////////////////////////////////////////
    // Byte-Read/Write Test
    ////////////////////////////////////////////////////////////////////////////

    printf("\n\nStart Byte-Read/Write Test...\n\n");

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    {
        size_t tReadSize, tOffset;
        UN16 un16Row = 0;

        // Print read/write numbers
        tTotalRead = 0; tTotalWrite = 0;
        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);

        tSize =
            OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                 sizeof(acBuffer), n32ReadTimeout);
        tTotalRead += tSize;
        dump(stdout, &un16Row, &acBuffer[0], tSize);
        printf("Read %u bytes from buffer.\n", tSize);

        for(;;)
        {
            tWriteLen = strlen(&acNumbers[0]);
            tSize = OSAL.tBufferWriteTail(
                hBuffer, &acNumbers[0], tWriteLen, n32WriteTimeout);
            tTotalWrite += tSize;
            printf("Wrote %u bytes to buffer.\n", tSize);
            if(tSize != tWriteLen) break;

            tWriteLen = strlen(&acAlphabet[0]);
            tSize = OSAL.tBufferWriteTail(
                hBuffer, &acAlphabet[0], tWriteLen, n32WriteTimeout);
            tTotalWrite += tSize;
            printf("Wrote %u bytes to buffer.\n", tSize);
            if(tSize != tWriteLen) break;
        }

        // Dump current buffer size
        printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

        // Peek into entire buffer
        printf("\n\nComplete peek into buffer...\n");

        tSize = 0;
        un16Row = 0;
        tOffset = 0;

        do
        {
            tReadSize =
                OSAL.tBufferPeek(hBuffer, &acBuffer[0],
                                 sizeof(acBuffer), tOffset);
            dump(stdout, &un16Row, &acBuffer[0], tReadSize);
            tOffset += tReadSize;
            tSize += tReadSize;
        } while(tReadSize != 0);

        printf("Peek'd %u bytes into buffer.\n", tSize);

        // Peek into last half of buffer
        printf("\n\nPartial peek into last half of buffer...\n");

        tSize = 0;
        un16Row = 0;
        tOffset = OSAL.tBufferGetSize(hBuffer) / 2;

        do
        {
            tReadSize =
                OSAL.tBufferPeek(hBuffer, &acBuffer[0],
                                 sizeof(acBuffer), tOffset);
            dump(stdout, &un16Row, &acBuffer[0], tReadSize);
            tOffset += tReadSize;
            tSize += tReadSize;
        } while(tReadSize != 0);

        printf("Peek'd %u bytes into buffer.\n", tSize);

        // Read all from the buffer
        printf("\n\nComplete read from buffer...\n");

        tSize = 0;
        un16Row = 0;

        do
        {
            tReadSize =
                OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                     sizeof(acBuffer), n32ReadTimeout);
            tTotalRead += tReadSize;
            dump(stdout, &un16Row, &acBuffer[0], tReadSize);
            tSize += tReadSize;
        } while(tReadSize != 0);

        printf("Read %u bytes from buffer.\n", tSize);

        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);
    }

    ////////////////////////////////////////////////////////////////////////////
    // Block-Read/Write Test
    ////////////////////////////////////////////////////////////////////////////

    printf("\n\nStart Block-Read/Write Test...\n\n");

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    {
        UN16 un16Row = 0;
        unsigned char *pcBuf;
        size_t tReadSize, tWriteSize;
        OSAL_BUFFER_BLOCK_HDL hBlock;

        // Print read/write numbers
        tTotalRead = 0; tTotalWrite = 0;
        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);

        hBlock = OSAL.hBufferReadBlock(hBuffer, &pcBuf, &tReadSize);
        if(hBlock != OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            tTotalRead += tReadSize;
            printf("Read block.\n");
            dump(stdout, &un16Row, pcBuf, tReadSize);
            eReturnCode = OSAL.eBufferReleaseBlock(hBlock);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Failed!
                OSAL_vBufferTestClean();
                printf("Error! Cannot release buffer block (%s).\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
                return;
            }
            else
            {
                printf("Released block.\n");
            }
        }
        else
        {
            printf("No more blocks!\n");
        }

        for(;;)
        {
            hBlock =
                OSAL.hBufferGetBlock(
                    hBuffer,
                    &pcBuf,
                    &tWriteSize
                        );
            if(hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // Failed!
                printf("Error! Cannot allocate a block from buffer.\n");
                break;
            }
            else
            {
                printf("Got Block (%u bytes)\n",
                       tWriteSize);

                // Fill the block
                tWriteSize = (strlen(&acAlphabet[0]) <= tWriteSize) ?
                    strlen(&acAlphabet[0]) : tWriteSize;
                strncpy((char *)pcBuf,
                        &acAlphabet[0],
                        tWriteSize);

                // Write block
                eReturnCode =
                    OSAL.eBufferWriteBlock(
                        hBlock,
                        tWriteSize);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Failed!
                    OSAL_vBufferTestClean();
                    printf("Error! Cannot write block to buffer (%s).\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    return;
                }
                else
                {
                    tTotalWrite += tWriteSize;
                    printf("Wrote block to buffer (%u bytes).\n",
                           tWriteSize);
                }
            }

            hBlock =
                OSAL.hBufferGetBlock(
                    hBuffer,
                    &pcBuf,
                    &tWriteSize
                        );
            if(hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // Failed!
                printf("Error! Cannot allocate a block from buffer.\n");
                break;
            }
            else
            {
                printf("Got Block (%u bytes)\n",
                       tWriteSize);

                // Fill the block
                tWriteSize = (strlen(&acNumbers[0]) <= tWriteSize) ?
                    strlen(&acNumbers[0]) : tWriteSize;
                strncpy((char *)pcBuf,
                        &acNumbers[0],
                        tWriteSize);

                // Write block
                eReturnCode =
                    OSAL.eBufferWriteBlock(
                        hBlock,
                        tWriteSize);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Failed!
                    OSAL_vBufferTestClean();
                    printf("Error! Cannot write block to buffer (%s).\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    return;
                }
                else
                {
                    tTotalWrite += tWriteSize;
                    printf("Wrote block to buffer (%u bytes).\n",
                           tWriteSize);
                }
            }

        }

        tSize = 0;
        un16Row = 0;

        // Dump current buffer size
        printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

        do
        {
            tReadSize =
                OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                     sizeof(acBuffer), n32ReadTimeout);
            tTotalRead += tReadSize;
            dump(stdout, &un16Row, &acBuffer[0], tReadSize);
            tSize += tReadSize;
        } while(tReadSize != 0);

        printf("Read %u bytes from buffer.\n", tSize);

        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);
    }

    ////////////////////////////////////////////////////////////////////////////
    // Head/Tail Test
    ////////////////////////////////////////////////////////////////////////////

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    do
    {
        if(n8Test%2 == 0)
        {
            bToggle = TRUE;
        }
        else
        {
            bToggle = FALSE;
        }

        printf("\n\nStart Head/Tail Test(%u)...\n\n", n8Test);

        {
            UN16 un16Row = 0;
            char *pcBuf;

            // Print read/write numbers
            tTotalRead = 0; tTotalWrite = 0;
            printf("Total bytes read = %u\twritten = %u\n",
                   tTotalRead, tTotalWrite);

            tSize =
                OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                     sizeof(acBuffer), n32ReadTimeout);
            tTotalRead += tSize;
            dump(stdout, &un16Row, &acBuffer[0], tSize);
            printf("Read %u bytes from buffer.\n", tSize);

            // Test head/tail
            for(;;)
            {
                tWriteLen = strlen(&acNumbers[0]);
                if(bToggle == FALSE)
                {
                    tSize = OSAL.tBufferWriteHead(
                        hBuffer, &acNumbers[0], tWriteLen);
                }
                else
                {
                    tSize = OSAL.tBufferWriteTail(
                        hBuffer, &acNumbers[0], tWriteLen, n32WriteTimeout);
                }
                tTotalWrite += tSize;
                printf("Wrote %u bytes to buffer.\n", tSize);
                if(tSize != tWriteLen) break;

                tWriteLen = strlen(&acAlphabet[0]);
                if(bToggle == FALSE)
                {
                    tSize = OSAL.tBufferWriteTail(
                        hBuffer, &acAlphabet[0], tWriteLen, n32WriteTimeout);
                }
                else
                {
                    tSize = OSAL.tBufferWriteHead(
                        hBuffer, &acAlphabet[0], tWriteLen);
                }
                tTotalWrite += tSize;
                printf("Wrote %u bytes to buffer.\n", tSize);
                if(tSize != tWriteLen) break;
            }

            un16Row = 0;

            // Dump current buffer size
            printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

            do
            {
                if(bToggle == FALSE)
                {
                    tSize =
                        OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                             sizeof(acBuffer), n32ReadTimeout);
                    printf("Read %u bytes from head of buffer\n", tSize);
                    pcBuf = &acBuffer[0];
                    bToggle = TRUE;
                }
                else
                {
                    tSize =
                        OSAL.tBufferReadTail(hBuffer, &acBuffer[0],
                                             sizeof(acBuffer), n32ReadTimeout);
                    printf("Read %u bytes from tail of buffer\n", tSize);
                    pcBuf = &acBuffer[0] + sizeof(acBuffer) - tSize;
                    bToggle = FALSE;
                }

                tTotalRead += tSize;
                dump(stdout, &un16Row, pcBuf, tSize);

            } while(tSize != 0);

            printf("Total bytes read = %u\twritten = %u\n",
                   tTotalRead, tTotalWrite);
        }

        // Dump current buffer size
        printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    } while(n8Test++ < 4);

    ////////////////////////////////////////////////////////////////////////////
    // Block-Write, Byte-Read Test
    ////////////////////////////////////////////////////////////////////////////

    printf("\n\nStart Block-Write, Byte-Read Test...\n\n");

    // Dump current buffer size
    printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

    {
        UN16 un16Row = 0;
        unsigned char *pcBuf;
        size_t tReadSize, tWriteSize;
        OSAL_BUFFER_BLOCK_HDL hBlock;
        BOOLEAN bWrite = TRUE;
        OSAL_BUFFER_BLOCK_HDL ahBlock[MAX_NUM_BLOCKS];
        size_t atBlockSize[MAX_NUM_BLOCKS];
        UN32 un32BlockIndex = 0;

        // Print read/write numbers
        tTotalRead = 0; tTotalWrite = 0;
        printf("Total bytes read = %u\twritten = %u\n",
               tTotalRead, tTotalWrite);

        hBlock = OSAL.hBufferReadBlock(hBuffer, &pcBuf, &tReadSize);
        if(hBlock != OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            tTotalRead += tReadSize;
            printf("Read block.\n");
            dump(stdout, &un16Row, pcBuf, tReadSize);
            eReturnCode = OSAL.eBufferReleaseBlock(hBlock);
            if(eReturnCode != OSAL_SUCCESS)
            {
                // Failed!
                OSAL_vBufferTestClean();
                printf("Error! Cannot release buffer block (%s).\n",
                       OSAL.pacGetReturnCodeName(eReturnCode));
                return;
            }
            else
            {
                printf("Released block.\n");
            }
        }
        else
        {
            printf("No more blocks!\n");
        }

        for(;;)
        {
            hBlock =
                OSAL.hBufferGetBlock(
                    hBuffer,
                    &pcBuf,
                    &tWriteSize
                        );
            if(hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // Failed!
                printf("Error! Cannot allocate a block from buffer.\n");
                break;
            }
            else
            {
                printf("Got Block (%u bytes)\n",
                       tWriteSize);

                // Fill the block
                tWriteSize = (strlen(&acAlphabet[0]) <= tWriteSize) ?
                    strlen(&acAlphabet[0]) : tWriteSize;
                strncpy((char *)pcBuf,
                        &acAlphabet[0],
                        tWriteSize);

                if(bWrite == TRUE)
                {
                    // Write block
                    eReturnCode =
                        OSAL.eBufferWriteBlock(
                            hBlock,
                            tWriteSize);
                    if(eReturnCode != OSAL_SUCCESS)
                    {
                        // Failed!
                        OSAL_vBufferTestClean();
                        printf("Error! Cannot write block to buffer (%s).\n",
                               OSAL.pacGetReturnCodeName(eReturnCode));
                        return;
                    }
                    else
                    {
                        tTotalWrite += tWriteSize;
                        printf("Wrote block to buffer (%u bytes).\n",
                               tWriteSize);
                    }

                    if(un32BlockIndex < MAX_NUM_BLOCKS)
                    {
                        bWrite = FALSE;
                    }
                }
                else
                {
                    // Replace write with a marker
                    ahBlock[un32BlockIndex] = hBlock;
                    atBlockSize[un32BlockIndex] = tWriteSize;
                    do
                    {
                        pcBuf[--tWriteSize] -= (32 - un32BlockIndex - 1);
                    }
                    while(tWriteSize);
                    un32BlockIndex++;
                    bWrite = TRUE;
                }
            }

            hBlock =
                OSAL.hBufferGetBlock(
                    hBuffer,
                    &pcBuf,
                    &tWriteSize
                        );
            if(hBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // Failed!
                printf("Error! Cannot allocate a block from buffer.\n");
                break;
            }
            else
            {
                printf("Got Block (%u bytes)\n",
                       tWriteSize);

                // Fill the block
                tWriteSize = (strlen(&acNumbers[0]) <= tWriteSize) ?
                    strlen(&acNumbers[0]) : tWriteSize;
                strncpy((char *)pcBuf,
                        &acNumbers[0],
                        tWriteSize);

                // Write block
                eReturnCode =
                    OSAL.eBufferWriteBlock(
                        hBlock,
                        tWriteSize);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Failed!
                    OSAL_vBufferTestClean();
                    printf("Error! Cannot write block to buffer (%s).\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    return;
                }
                else
                {
                    tTotalWrite += tWriteSize;
                    printf("Wrote block to buffer (%u bytes).\n",
                           tWriteSize);
                }
            }

        }

        bWrite = TRUE;
        // Back fill blocks
        for(;;)
        {
            tSize = 0;
            un16Row = 0;

            // Dump current buffer size
            printf("Buffer Size = %u bytes\n", OSAL.tBufferGetSize(hBuffer));

            do
            {
                tReadSize =
                    OSAL.tBufferReadHead(hBuffer, &acBuffer[0],
                                         sizeof(acBuffer), n32ReadTimeout);
                tTotalRead += tReadSize;
                dump(stdout, &un16Row, &acBuffer[0], tReadSize);
                tSize += tReadSize;
            } while(tReadSize != 0);

            printf("Read %u bytes from buffer.\n", tSize);

            printf("Total bytes read = %u\twritten = %u\n",
                   tTotalRead, tTotalWrite);

            if(un32BlockIndex--)
            {
                if(bWrite == FALSE)
                {
                    tWriteSize = 0;
                    bWrite = TRUE;
                }
                else
                {
                    tWriteSize = atBlockSize[un32BlockIndex];
                    bWrite = FALSE;
                }

                // Write block
                eReturnCode =
                    OSAL.eBufferWriteBlock(
                        ahBlock[un32BlockIndex],
                        tWriteSize);
                if(eReturnCode != OSAL_SUCCESS)
                {
                    // Failed!
                    OSAL_vBufferTestClean();
                    printf("Error! Cannot write block to buffer (%s).\n",
                           OSAL.pacGetReturnCodeName(eReturnCode));
                    return;
                }
                else
                {
                    tTotalWrite += tWriteSize;
                    printf("Wrote block to buffer (%u bytes).\n",
                           tWriteSize);
                }
            }
            else
            {
                break;
            }
        }
    }

    //////////////////////
    // Clean up after test
    //////////////////////
    OSAL_vBufferTestClean();

    printf("\nDynamic Buffer Test Complete!\n");

    return;
}

/*****************************************************************************
*
*       OSAL_vBufferTestClean
*
*****************************************************************************/
static void OSAL_vBufferTestClean( void )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;

    printf("\nDynamic Buffer Test Clean-up\n");

    // Free Buffer (if it exists)
    if(hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        eReturnCode = OSAL.eBufferFree(hBuffer);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("Error! Cannot free buffer (%s).\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
        else
        {
            printf("Freed buffer.\n");
        }
    }

    eReturnCode = OSAL.eBlockPoolList();
    if(eReturnCode != OSAL_SUCCESS)
    {
        printf("Error! Cannot get block pool list (%s).\n",
               OSAL.pacGetReturnCodeName(eReturnCode));
    }

    // Kill block pool (if it exists)
    if(hBlockPool != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eBlockPoolDelete(hBlockPool);
        if(eReturnCode != OSAL_SUCCESS)
        {
            printf("Error! Cannot delete block pool (%s).\n",
                   OSAL.pacGetReturnCodeName(eReturnCode));
        }
        else
        {
            printf("Freed block pool.\n");
        }
    }

    printf("Dynamic Buffer Test Clean-up Complete!\n\n");

    return;
}

#endif /* OSAL_BUFFER_TEST == 1 */
