/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio Inc.                      */
/*                            All Rights Reserved                             */
/*             Licensed Materials - Property of Sirius XM Radio Inc.          */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*   DESCRIPTION
*
*   This module will contain all the OSAL Block Pool and
*   Dynamic Buffer APIs.
*
*******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "standard.h"

#include "osal_version.h"

#include "osal.h"

#include "osal_core.h"
#include "osal_buf.h"
#include "_osal_buf.h"

#if (!(defined OSAL_CPU_LITTLE_ENDIAN) && !(defined OSAL_CPU_BIG_ENDIAN))
    #error Error! You must define OSAL_CPU_LITTLE_ENDIAN or OSAL_CPU_BIG_ENDIAN
#endif

/*******************************************************************************
*
*   OSAL_eBlockPoolCreate
*
*   This function is used to create a block pool for use by Dynamic Buffers.
*   A block pool is a pre-allocated, fixed set of memory blocks used to
*   construct stream or block oriented buffers for reading or writing data.
*   The caller must provide the individual block size and the number of blocks
*   required.
*
*   Inputs:
*       phBlockPool - A pointer to a block pool object handle used for other
*           Block Pool and Dynamic Buffer APIs. The supplied object handle will
*           be populated with a valid handle upon success (API returns
*           OSAL_SUCCESS) or will be populated with OSAL_INVALID_OBJECT_HDL
*           if the API fails.
*       pacName - An ASCII Null terminated string used to identify the object
*           being created. A name is limited to OSAL_MAX_OBJECT_NAME_LENGTH
*           characters (not including the NULL) and must be unique.
*       un16BlockSize - The size (in bytes) of each block in the pool.
*       un16NumBlocks - The number of block of size un16BlockSize bytes to
*           create in the block pool.
*       un32Options - One or more block pool creation options.
*           These options are:
*           OSAL_BLOCK_POOL_OPTION_NONE - No options selected.
*           Currently no block pool create options exist, but the options
*           field is present to support possible future expansion.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBlockPoolCreate (
    OSAL_OBJECT_HDL *phBlockPool,
    const char *pacName,
    UN16 un16BlockSize,
    UN16 un16NumBlocks,
    UN32 un32Options
        )
{
#if OSAL_BUFFER == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_OBJECT_HDL hBlockPool;
    OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT *psBlockPoolInfo = NULL;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];
    UN32 un32QueueOptions = OSAL_QUEUE_OPTION_FIXED_SIZE;

    // Check input
    if(phBlockPool == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Initialize return object handle for error cases
    *phBlockPool = OSAL_INVALID_OBJECT_HDL;

    // Initialize local name
    acName[0] = '\0';

    // Verify they have provided valid options
    if( (un32Options & ~OSAL_BLOCK_POOL_OPTION_ALL) !=
       OSAL_BLOCK_POOL_OPTION_NONE)
    {
        return OSAL_ERROR_UNSUPPORTED_OPTION;
    }

    eReturnCode = OSALC_eCreateObject(
        OSAL_OBJECT_TYPE_BLOCK_POOL,
        0,
        pacName,
        &hBlockPool,
        &psObj,
        (OSAL_OBJECT_INFO_UNION **)&psBlockPoolInfo
            );
    if((eReturnCode == OSAL_SUCCESS) && (psBlockPoolInfo != NULL))
    {
        // populate the object info per user input and initialize as necessary
        psBlockPoolInfo->sPublicInfo.un32Options = un32Options;
        psBlockPoolInfo->sPublicInfo.un16BlockSize = un16BlockSize;
        psBlockPoolInfo->sPublicInfo.un16NumBlocks = un16NumBlocks;
        // Determine the actual block size we will request
        // I say 'actual' because we will actually use their block memory
        // to store our own structure which maintains the buffer.
        // Each block needs to be sizeof(OSALB_BLOCK_HEADER_STRUCT) larger than
        // the requested size to track blocks.
        psBlockPoolInfo->sPublicInfo.un32ActualBlockSize =
            sizeof(OSALB_BLOCK_HEADER_STRUCT) +
                    un16BlockSize;
        psBlockPoolInfo->sPublicInfo.un32OverrunCnt = 0;
        psBlockPoolInfo->sPublicInfo.un16Allocated = 0;
        psBlockPoolInfo->sPublicInfo.un16MaxAllocated = 0;
        psBlockPoolInfo->hBufferQueue = OSAL_INVALID_OBJECT_HDL;
        psBlockPoolInfo->hBlockQueue = OSAL_INVALID_OBJECT_HDL;

        // Determine if this block pool will be used in an interrupt
        // if so we must add this option for the queues we need as well
        if((un32Options & OSAL_BLOCK_POOL_OPTION_INTERRUPT) ==
            OSAL_BLOCK_POOL_OPTION_INTERRUPT)
        {
            // Add the OSAL_QUEUE_OPTION_INTERRUPT option to the queues we need
            un32QueueOptions |= OSAL_QUEUE_OPTION_INTERRUPT;

            // Since this block pool is intended to be used in an interrupt
            // we need to choose Interrupt for entering/exiting exclusive access.
            psBlockPoolInfo->bEnterExclusiveAccess =
                bEnterExclusiveAccessInterrupt;
            psBlockPoolInfo->vExitExclusiveAccess =
                vExitExclusiveAccessInterrupt;

            // Block pool does not need a mutex
            psBlockPoolInfo->hMutex =
                OSAL_INVALID_OBJECT_HDL;
        }
        else
        {
            // Since this block pool will not be used in an interrupt
            // we need to choose TaskOnly for entering/exiting exclusive access.
            psBlockPoolInfo->bEnterExclusiveAccess =
                bEnterExclusiveAccessTaskOnly;
            psBlockPoolInfo->vExitExclusiveAccess =
                vExitExclusiveAccessTaskOnly;

            // Construct an appropriate name for semaphore to be created
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     OSAL_NAME_PREFIX"%s:BlkPoolMutex", pacName);

            // Block pool needs a mutex
            eReturnCode = OSAL.eSemCreate(
                &psBlockPoolInfo->hMutex,
                &acName[0], 1, 1, OSAL_SEM_OPTION_NONE);
            if(eReturnCode != OSAL_SUCCESS)
            {
                OSAL_eBlockPoolDelete(hBlockPool);
                return OSAL_ERROR;
            }
        }

        // Construct an appropriate name for the buffer queue
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 OSAL_NAME_PREFIX"%s:bufQ", pacName);

        // Create a message queue which will be used for the
        // buffers requested.
        eReturnCode = OSAL.eQueueCreate(
            &psBlockPoolInfo->hBufferQueue,
            &acName[0],
            un16NumBlocks,
            sizeof(OSALB_BUFFER_STRUCT),
            un32QueueOptions);

        if(eReturnCode != OSAL_SUCCESS)
        {
            OSAL_eBlockPoolDelete(hBlockPool);
            return OSAL_ERROR;
        }

        // Construct an appropriate name for the block queue
        snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                 OSAL_NAME_PREFIX"%s:blkQ", pacName);

        // Create a message queue which will be used for the
        // buffer blocks requested.
        eReturnCode = OSAL.eQueueCreate(
            &psBlockPoolInfo->hBlockQueue,
            &acName[0],
            un16NumBlocks,
            psBlockPoolInfo->sPublicInfo.un32ActualBlockSize,
            un32QueueOptions);

        if(eReturnCode != OSAL_SUCCESS)
        {
            OSAL_eBlockPoolDelete(hBlockPool);
            return OSAL_ERROR;
        }

        // Populate the object
        *phBlockPool = hBlockPool;

        // Buffer created successfully, return handle to caller
        eReturnCode = OSAL_SUCCESS;
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif


}

/*******************************************************************************
*
*   OSAL_eBlockPoolDelete
*
*   This function is used to delete a previously created block pool from the
*   system. Upon return from this call, the block pool's object handle is no
*   longer valid and all memory blocks within the block pool have been
*   destroyed.
*
*   Inputs:
*       hBlockPool - A block pool object handle to perform the operation on.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBlockPoolDelete (
    OSAL_OBJECT_HDL hBlockPool
        )
{
#if OSAL_BUFFER == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT *psBlockPoolInfo;
    void (*vExitExclusiveAccess)(OSAL_OBJECT_HDL hMutex) = NULL;
    BOOLEAN bOk;

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj =
        OSALC_psGetObjectFromHandle( hBlockPool, OSAL_OBJECT_TYPE_BLOCK_POOL );
    if(psObj == NULL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

/*****************************************************************************/
    bOk = psObj->puInfo->sBlockPool.bEnterExclusiveAccess(
        psObj->puInfo->sBlockPool.hMutex);

    if(bOk == FALSE)
    {
        // If we cannot enter exclusive access section, it could mean that
        // the object was already deleted by another task
        return OSAL_ERROR_CANNOT_DELETE_OBJECT;
    }

    // Extract buffer info from object
    psBlockPoolInfo = &psObj->puInfo->sBlockPool;

    // If the Queue hasn't been destroyed yet, destroy it now
    if(psBlockPoolInfo->hBlockQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eQueueDelete(psBlockPoolInfo->hBlockQueue);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate queue handle
            psBlockPoolInfo->hBlockQueue = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // If the Queue hasn't been destroyed yet, destroy it now
    if(psBlockPoolInfo->hBufferQueue != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eQueueDelete(psBlockPoolInfo->hBufferQueue);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate queue handle
            psBlockPoolInfo->hBufferQueue = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // If the mutex hasn't been destroyed yet, destroy it now
    if(psBlockPoolInfo->hMutex != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(psBlockPoolInfo->hMutex);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Invalidate mutex handle
            psBlockPoolInfo->hMutex = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Safe exit exclusive access function
    vExitExclusiveAccess = psBlockPoolInfo->vExitExclusiveAccess;

    // Invalidate appropriate field
    psBlockPoolInfo->vExitExclusiveAccess = NULL;

    // Remove and destroy the object
    eReturnCode = OSALC_eRemoveObject(hBlockPool);

    /*************************************/
    /* Exit the Exclusive Access section */
    if (vExitExclusiveAccess != NULL)
    {
        vExitExclusiveAccess(OSAL_INVALID_OBJECT_HDL);
    }
    /*************************************/

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif


}

/*******************************************************************************
*
*   OSAL_hBufferAllocate
*
*   This function is used to allocate (create) a Dynamic Buffer tied to a
*   provided block pool. Prior to calling this function the block pool
*   associated with this buffer must be created using OSAL.eBlockPoolCreate().
*   Upon success this function will return a handle to a Dynamic Buffer which
*   can be read from or written to using the proceeding Dynamic Buffer
*   I/O calls. The caller may choose to allocate this Dynamic Buffer with
*   read or write blocking features (used to suspend the calling task).
*   The blocking/suspend options for block-wise read or writes have no impact
*   as block-reads and writes may never block/suspend. Only byte-wise reads
*   and writes can block/suspend the caller.
*
*   Inputs:
*       hBlockPool - A block pool object handle to perform the operation on.
*           This is the block pool that read and write APIs will use. A read
*           from a buffer will cause blocks to be released back into the block
*           pool. A write to a buffer will cause blocks to be allocated from
*           the block pool.
*       bBlockingRead - A flag to indicate if blocking reads are desired.
*           If the value provided is FALSE, a read operation will never block.
*           If the value provided is TRUE, a read operation will block when
*           there is no data in the buffer to read. A byte-wse read call will
*           unblock when data becomes available in the provided buffer for
*           reading.
*       bBlockingWrite - A flag to indicate if blocking writes are desired.
*           If the value provided is FALSE, a write operation will never
*           block. If the value provided is TRUE, a write operation will
*           block when there is no more room in the buffer to satisfy the
*           request. A byte-wse write call will unblock when room becomes
*           available in the provided buffer for more writing.
*       un32Options - One or more buffer allocation options. These options are:
*           OSAL_BUFFER_ALLOCATE_OPTION_NONE - No options selected.
*           Currently no buffer allocation options exist, but the options
*           field is present to support possible future expansion.
*
*   Outputs:
*       A valid OSAL_BUFFER_HDL is returned upon success. If unsuccessful,
*       OSAL_INVALID_BUFFER_HDL will be returned. This handle must be provided
*       to all Dynamic Buffer byte- and block-wise read and write APIs.
*
*******************************************************************************/
OSAL_BUFFER_HDL OSAL_hBufferAllocate (
    OSAL_OBJECT_HDL hBlockPool,
    BOOLEAN bBlockingRead,
    BOOLEAN bBlockingWrite,
    UN32 un32Options
        )
{

#if OSAL_BUFFER == 1
    OSAL_OBJECT_STRUCT *psObj;
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT *psBlockPoolInfo;
    OSALB_BUFFER_STRUCT *psBuffer;
    char acName[OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL];

    // Initialize local name
    acName[0] = '\0';

    // Verify they have provided valid options
    if( (un32Options & ~OSAL_BUFFER_ALLOCATE_OPTION_ALL) !=
       OSAL_BUFFER_ALLOCATE_OPTION_NONE)
    {
        return OSAL_INVALID_BUFFER_HDL;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj =
        OSALC_psGetObjectFromHandle( hBlockPool, OSAL_OBJECT_TYPE_BLOCK_POOL );
    if(psObj == NULL)
    {
        return OSAL_INVALID_BUFFER_HDL;
    }

    // Extract buffer info from object
    psBlockPoolInfo = &psObj->puInfo->sBlockPool;

    // Allocate a new buffer from the block pool
    // Overlay new buffer structure onto acquired block
    eReturnCode =
        OSAL.eMessageAllocate (
            psBlockPoolInfo->hBufferQueue,
            (void**)&psBuffer,
            OSAL_QUEUE_FLAG_NONBLOCK );
    if(eReturnCode != OSAL_SUCCESS)
    {
        // Error! Cannot allocate any more buffers from this block queue
        return NULL;
    }

    OSAL_BUFFER_RELEASED_ASSERT(psBuffer);

    // Populate buffer header...
    psBuffer->psBlockPoolInfo = psBlockPoolInfo;
    psBuffer->un32Options = un32Options;
    psBuffer->tSize = 0;
    psBuffer->un8RemainingBits = 0;
    psBuffer->psHead = NULL;
    psBuffer->psTail = NULL;
    psBuffer->un16Allocated = 0;
    psBuffer->un16MaxAllocated = 0;
    psBuffer->hWriteSem = OSAL_INVALID_OBJECT_HDL;
    psBuffer->bBlockingWrite = bBlockingWrite;
#if ((OSAL_DEBUG==1) && (OSAL_BUFFER_OBJECT_STATE_CHECK == 1))
    psBuffer->pvAllocationCheck = (void*)psBuffer;
#endif

    // Configure buffer based on blocking read selection
    if(bBlockingRead == TRUE)
    {
        OSAL_RETURN_CODE_ENUM eReturnCode;

        {
            static UN32 un32Instance = 0;
            const char *pacName;

#if OSAL_OBJECT_TRACKING == 1
            pacName = psObj->pacName;
#else
            pacName = "Unknown";
#endif
            // Construct an appropriate name for semaphore to be created
            // Increment instance number for each semaphore created to
            // generate a unique name.
            snprintf(&acName[0], OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL,
                     OSAL_NAME_PREFIX"BufferRead:%u:%s", un32Instance++,
                     pacName);
        }

        // Create a semaphore with initial value = 0 (unavailable)
        // and a maximum number of resources as 1
        eReturnCode =
            OSAL.eSemCreate(
                &psBuffer->hReadSem,
                &acName[0],
                0,
                1,
                OSAL_SEM_OPTION_NONE);
        if(eReturnCode != OSAL_SUCCESS)
        {
            // Error! Release buffer
            OSAL_eBufferFree((OSAL_BUFFER_HDL)psBuffer);
            return OSAL_INVALID_BUFFER_HDL;
        }
    }
    else
    {
        // No semaphore required
        psBuffer->hReadSem = OSAL_INVALID_OBJECT_HDL;
    }

    // Return buffer pointer as the buffer handle
    return (OSAL_BUFFER_HDL)psBuffer;
#else
    return OSAL_INVALID_BUFFER_HDL;
#endif

}

/*******************************************************************************
*
*   OSAL_eBufferFree
*
*   This function is used to free a previously allocated Dynamic Buffer.
*   It also has the side effect of freeing any blocks, this buffer owns back
*   to its' associated block pool. Once this call completes the buffer handle
*   provided is no longer valid.
*
*   Inputs:
*       hBuffer - The Dynamic Buffer to free.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferFree (
    OSAL_BUFFER_HDL hBuffer
        )
{
#if OSAL_BUFFER == 1
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psBlock;
    UN16 un16Allocated;
    BOOLEAN bOk;

    // Check provided handle
    if(hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);

    if (bOk == FALSE)
    {
        // Cannot enter exclusive access section. Block pool
        // is corrupted or deleted
        return OSAL_ERROR;
    }

    // Start at the head or tail and work back to find the first
    // block in our buffer, then work forward till all blocks are
    // released.
    psBlock = (psBuffer->psHead != NULL) ?
        psBuffer->psHead : psBuffer->psTail;

    // Find the first block in the list
    while( psBlock != NULL )
    {
        // If the previous block is NULL we have found the front
        if(psBlock->psPrev == NULL)
        {
            break;
        }

        // Check the next block
        psBlock = psBlock->psPrev;
    }

    // Remove all blocks, starting from the first block till there
    // ain't no more.
    while (psBlock != NULL)
    {
        // Force this block to be released
        psBlock->bReadReserved = FALSE;

        // Free the current block...
        psBlock = psFreeBlock(psBlock);
    }

    // Additional clean up for some buffer's fields
    psBuffer->tSize = 0;
#if ((OSAL_DEBUG==1) && (OSAL_BUFFER_OBJECT_STATE_CHECK == 1))
    psBuffer->pvAllocationCheck = (void*)(~(size_t)psBuffer);
#endif

    un16Allocated = psBuffer->un16Allocated;

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // Just make sure everything has been freed
    if(un16Allocated != 0)
    {
        printf("OSAL Error! Buffer blocks corrupt (%u != 0).\n",
               psBuffer->un16Allocated);
    }

    // Free read semaphore if one was created
    if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        eReturnCode = OSAL.eSemDelete(psBuffer->hReadSem);
        if(eReturnCode == OSAL_SUCCESS)
        {
            psBuffer->hReadSem = OSAL_INVALID_OBJECT_HDL;
        }
    }

    // Free buffer memory
    OSAL.eMessageFree((void*)psBuffer);

    return OSAL_SUCCESS;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif

}

/*******************************************************************************
*
*   OSAL_tBufferGetSize
*
*   This function is used to return the current size of the provided Dynamic
*   Buffer in bytes. The size of a Dynamic Buffer reported is the actual number
*   of bytes currently written to it and unread as a contiguous stream. The
*   buffer size does not mean its 'capacity'. The buffer size is simply the
*   number of bytes which can be read from the buffer as a contiguous stream.
*
*   Inputs:
*       hBuffer - The Dynamic Buffer to query the size of.
*
*   Outputs:
*       A value of type size_t which contains the current number of bytes
*       available for reading in the provided buffer.
*
*******************************************************************************/
size_t OSAL_tBufferGetSize (
    OSAL_BUFFER_HDL hBuffer
        )
{
#if OSAL_BUFFER == 1
    size_t tSize = 0;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;

    // Check provided handle
    if(hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        BOOLEAN bOk;

        OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

/*****************************************************************************/
        bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
        if (bOk == TRUE)
        {
            tSize = psBuffer->tSize;

            psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
                psBuffer->psBlockPoolInfo->hMutex);
        }
/*****************************************************************************/
    }
    return tSize;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferGetSizeInBits
*
*   This function is used to return the current size of the provided Dynamic
*   Buffer in bits. The size of a Dynamic Buffer reported is the actual number
*   of bits currently written to it and unread as a contiguous stream. The
*   buffer size does not mean its 'capacity'. The buffer size is simply the
*   number of bits which can be read from the buffer as a contiguous stream.
*
*   Inputs:
*       hBuffer - The Dynamic Buffer to query the size of.
*
*   Outputs:
*       A value of type size_t which contains the current number of bits
*       available for reading in the provided buffer.
*
*******************************************************************************/
size_t OSAL_tBufferGetSizeInBits (
    OSAL_BUFFER_HDL hBuffer
        )
{
#if OSAL_BUFFER == 1
    size_t tSize = 0;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;

    // Check provided handle
    if(hBuffer != OSAL_INVALID_BUFFER_HDL)
    {
        BOOLEAN bOk;

        OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

        /*****************************************************************************/
        bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);

        if (bOk == TRUE)
        {
            tSize = (psBuffer->tSize * 8);

            if (tSize < psBuffer->tSize)
            {
                // Overflow, just return SIZE_T_MAX
                tSize = SIZE_T_MAX;
            }
            else
            {
                tSize += psBuffer->un8RemainingBits;
            }

            psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
                psBuffer->psBlockPoolInfo->hMutex);
        }
        /*****************************************************************************/
    }
    return tSize;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_eBufferBlocksIterate
*
*   This function is used to return iterate through all buffer block calling
*   provided callback function and passing start address of the block data
*   together with its size
*
*   Inputs:
*       hBuffer - The Dynamic Buffer to query the size of.
*       teIterator - The callback function which will be called for each
*                    block which is not reserved for block-related operations
*       pvArg - The data which is provided by the caller to be passed to the
*               callback function.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferBlocksIterate(
    OSAL_BUFFER_HDL hBuffer,
    OSAL_BUFFER_ITERATOR_HANDLER teIterator,
    void *pvArg
        )
{
#if OSAL_BUFFER == 1
    BOOLEAN bOk;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;

    // Check input
    if ((hBuffer == OSAL_INVALID_BUFFER_HDL) || (teIterator == NULL))
    {
        return OSAL_ERROR_INVALID_INPUT;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // We perform this check to ensure the buffer
    // is byte-aligned.
    if (psBuffer->un8RemainingBits != 0)
    {
        // Sorry!
        return OSAL_ERROR;
    }

    /*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if (bOk == TRUE)
    {
        OSALB_BLOCK_HEADER_STRUCT *psBlock;
        OSAL_BUFFER_ITERATOR_RESULT_ENUM eResult;
        size_t tDataSize;
        BOOLEAN bMoveToNextBlock = FALSE;

        // Start from the head block
        psBlock = psBuffer->psHead;
        while (psBlock != NULL)
        {
            // Assume we will move to the next block
            bMoveToNextBlock = TRUE;
            
            if ((psBlock->bReadReserved != TRUE) &&
                (psBlock->bWriteReserved != TRUE))
            {
                tDataSize = psBlock->pun8Write - psBlock->pun8Read;
                if (tDataSize != 0)
                {
                    eResult = teIterator(psBlock->pun8Read, tDataSize, pvArg);
                    if (eResult == OSAL_BUFFER_ITERATOR_RESULT_STOP)
                    {
                        break;
                    }
                    else if (eResult == OSAL_BUFFER_ITERATOR_RESULT_REMOVE)
                    {
                        OSALB_BLOCK_HEADER_STRUCT *psBlockToDelete;

                        // Store the block for the removal
                        psBlockToDelete = psBlock;

                        // Get next block
                        psBlock = psBlock->psNext;

                        // Remove the block
                        psFreeBlock(psBlockToDelete);
                        
                        // This step already done
                        bMoveToNextBlock = FALSE;
                    }
                }
            }
            
            if (bMoveToNextBlock == TRUE)
            {
                // Get next block
                psBlock = psBlock->psNext;
            }
        }

        psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
    /*****************************************************************************/
    }

    return OSAL_SUCCESS;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferPeek
*
*   This function is used to read a byte-stream of data from the head of a
*   provided Dynamic Buffer (without removing the data) and copy the requested
*   number of bytes starting with the destination pointer provided. The read
*   operation will start from the value provided by tOffset onward. The bytes
*   read from the buffer however are not removed from the buffer, instead this
*   API provides the ability to peek into data in the provided buffer without
*   removing the data. If the entire read operation cannot be completed, the
*   number of bytes which were read and copied will be returned to the caller.
*   The bytes will be read from the head of the buffer and placed starting at
*   the provided pvDst pointer, until tSize bytes have been copied or all
*   available data has been read.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvDst - A void pointer to the first memory address where the data read
*           from the buffer should be stored.
*       tSize - The amount of data to read from the buffer. The request may be
*           satisified in whole, partial or no data being read depending upon
*           the amount of data presently in the provided buffer. The caller
*           should always use this return value to determine how much data was
*           actually read.
*       tOffset - The offset from the beginning (head) of the buffer to begin
*           reading from.
*
*   Outputs:
*       A size_t value indicating the number of bytes actually read from
*       the buffer.
*
*******************************************************************************/
size_t OSAL_tBufferPeek (
    OSAL_BUFFER_HDL hBuffer,
    void *pvDst,
    size_t tSize,
    size_t tOffset
        )
{
#if OSAL_BUFFER == 1
    size_t tNum = 0;

    UN8 *pun8Dst = (UN8 *)pvDst;
    OSALB_BUFFER_STRUCT const *psBuffer =
        (OSALB_BUFFER_STRUCT const *)hBuffer;
    BOOLEAN bOk;

    // Check provided handle, if they asked for anything and if there is
    // something in the buffer to begin with.
    if( (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
        (tSize == 0) ||
        (pun8Dst == NULL) )
    {
        // Not a valid buffer
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if (bOk == FALSE)
    {
        // Cannot enter exclusive access. Buffer is corrupted
        // or deleted
        return 0;
    }

    if(psBuffer->psHead != NULL)
    {
        // Check to see if we are byte-aligned or not
        if (psBuffer->un8RemainingBits != 0)
        {
            UN32 un32SizeInBits = tSize * 8;
            size_t tBufferOffsetInBits = tOffset * 8;

            // Look for overflow with a simple check
            if (un32SizeInBits < tSize)
            {
                // We had an overflow, so lets cap it at the max
                // value for N32
                un32SizeInBits = N32_MAX;
            }

            // Look for overflow with a simple check
            if (tBufferOffsetInBits < tOffset)
            {
                // We had an overflow, so lets cap it at the max
                // value for size_t
                tBufferOffsetInBits = SIZE_T_MAX;
            }

            // We aren't byte-aligned, so pass this off to the read bits function
            tNum = tBufferPeekBits(
                psBuffer,
                pun8Dst,
                0, // Don't use a bit offset
                (N32)un32SizeInBits,
                tBufferOffsetInBits
                    );

            // Convert tNum from bits to bytes read since the
            // read bits function returns number of bits read.
            tNum = tNum / 8;
        }
        else // We are byte aligned
        {
            OSALB_BLOCK_HEADER_STRUCT *psBlock = psBuffer->psHead;
            size_t tPeekableDataSize = 0;

            // Find out the block where needed data lays based on offset
            // if caller specifies something different than the first
            // readable byte in the buffer, i.e. current head block
            if (tOffset > 0)
            {
                do
                {
                    // Check out number of readable data in bytes in the block.
                    // The logic based on the fact that the *write* position
                    // always not far than the *end-of-data* mark. In case if
                    // whole block is filled it the both those marks point to
                    // the same position (address).
                    tPeekableDataSize = psBlock->pun8Write - psBlock->pun8Read;
                    // Whether the offset lays within that block?
                    if (tPeekableDataSize > tOffset)
                    {
                        // Here we go - we've found needed block with
                        // the data on the caller's offset
                        break;
                    }
                    // Skip whole block from offset point of view
                    tOffset -= tPeekableDataSize;
                    // Get next block
                    psBlock = psBlock->psNext;
                } while (psBlock != NULL);
            }

            // Well, do we have anything to peek from here
            if (psBlock != NULL)
            {
                UN8 *pun8PeekPosition;
                do
                {
                    // Get peeking position in the block
                    pun8PeekPosition = psBlock->pun8Read + tOffset;
                    // Invalidate offset since now for further loops
                    tOffset = 0;
                    // Check out number of readable data in bytes in the block.
                    // The logic based on the fact that the *write* position
                    // always not far than the *end-of-data* mark. In case if
                    // whole block is filled it the both those marks point to
                    // the same position (address).
                    tPeekableDataSize = psBlock->pun8Write - pun8PeekPosition;
                    // Does the block have something to peek?
                    if (tPeekableDataSize > 0)
                    {
                        // Adjust number of bytes to peek based on
                        // remaining space in the caller's buffer
                        if (tSize < tPeekableDataSize)
                        {
                            tPeekableDataSize = tSize;
                        }
                        // Copy the data using system function since it performs operations
                        // using all range of optimizations.
                        OSAL.bMemCpy(pun8Dst, pun8PeekPosition, tPeekableDataSize);
                        // Adjust sizes and position
                        pun8Dst += tPeekableDataSize;
                        tSize -= tPeekableDataSize;
                        tNum += tPeekableDataSize;
                    }

                    // Move to to the next block since we might
                    // need to peek rest data from it also.
                    psBlock = psBlock->psNext;

                } while ((psBlock != NULL) && (tSize != 0));
            }
        }
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/
    // Return the total number of bytes read
    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferPeekBits
*
*   This function is used to read a bits of data from the head of a
*   provided Dynamic Buffer (without removing the data) and copy the requested
*   number of bits starting with the destination pointer provided. The read
*   operation will start from the value provided by tBufferOffset onward.
*   The bits read from the buffer however are not removed from the buffer,
*   instead this API provides the ability to peek into data in the provided
*   buffer without removing the data. If the entire read operation cannot
*   be completed, thenumber of bits which were read and copied will be returned
*   to the caller. The bits will be read from the head of the buffer and placed
*   starting atthe provided pvDst pointer, until tSize bytes have been copied or
*   all available data has been read.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvDst - A void pointer to the first memory address where the data read
*           from the buffer should be stored.
*       tDstBitOffset - The destination bit offset where bit 0 is the leftmost bit
*           at pvDst.
*       n32Bits - The amount of data to read from the buffer. The request may be
*           satisified in whole, partial or no data being read depending upon
*           the amount of data presently in the provided buffer. The caller
*           should always use this return value to determine how much data was
*           actually read.
*       tBufferOffsetBits - The offset from the beginning (head) of the buffer to
*           begin reading from in bits.
*
*   Outputs:
*       A size_t value indicating the number of bits actually read from
*       the buffer.
*
*******************************************************************************/
size_t OSAL_tBufferPeekBits (
    OSAL_BUFFER_HDL hBuffer,
    void *pvDst,
    size_t tDstBitOffset,
    N32 n32Bits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNumBits = 0;

    UN8 *pun8Dst = (UN8 *)pvDst;
    OSALB_BUFFER_STRUCT const *psBuffer =
        (OSALB_BUFFER_STRUCT const *)hBuffer;
    BOOLEAN bOk;

    // Check provided handle, if they asked for anything and if there is
    // something in the buffer to begin with.
    if( (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
        (n32Bits == 0) ||
        (pun8Dst == NULL) )
    {
        // Not a valid buffer
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);

    if (bOk == TRUE)
    {
        tNumBits = tBufferPeekBits(psBuffer, pun8Dst, tDstBitOffset,
            n32Bits, tBufferOffsetBits);

        psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
    }
/*****************************************************************************/
    // Return the total number of bits read
    return tNumBits;
#else
    return (size_t)0;
#endif


}

/*******************************************************************************
*
*   OSAL_tBufferReadHead
*
*   This function is used to read a byte-stream of data from the head of a
*   provided Dynamic Buffer and copy the requested number of bytes starting
*   with the destination pointer provided.
*
*   If the destination pointer is NULL, this function operates as a standard
*   seek operation, starting at the head of a provided Dynamic Buffer and
*   releasing memory along the way.
*
*   If the destination pointer is not NULL, and if the entire read operation
*   cannot be completed, the number of bytes which were read and copied will
*   be returned to the caller. The bytes will be read from the head of the
*   buffer and placed starting at the provided pvDst pointer, until tSize
*   bytes have been copied or all available data has been read.
*
*   This operation may block if the buffer associated with the handle hBuffer
*   was allocated with the blocking-read feature enabled.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvDst - A void pointer to the first memory address where the data
*           read from the buffer should be stored.
*       tSize - The amount of data to read from the buffer. The request may
*           be satisified in whole, partial or no data being read depending
*           upon the amount of data presently in the provided buffer. The
*           caller should always use the return value to determine how much
*           data was actually read.
*       n32Timeout - The time in milliseconds to wait for data to be
*           available while the caller block or suspends the current
*           process/task. Also may be provided OSAL_OBJ_TIMEOUT_INFINITE
*           to have byte-wise read operations wait forever, or
*           OSAL_OBJ_TIMEOUT_NONE to simply read any available data without
*           blocking/suspending. The value provided is applied when performing
*           a byte-wise read from either the head or tail of the provided
*           buffer. Block-wise read and write APIs cannot block or suspend
*           the caller and thus this parameter has no effect on them.
*
*   Outputs:
*       A size_t value indicating the number of bytes actually read from
*       the buffer.
*
*******************************************************************************/
size_t OSAL_tBufferReadHead (
    OSAL_BUFFER_HDL hBuffer,
    void *pvDst,
    size_t tSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNum = 0;

    UN8 *pun8Dst = (UN8 *)pvDst;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle and if they asked for anything
    if( (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
        (tSize == 0) )
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if read blocking is enabled, if so make sure we
    // get a the read semaphore before attempting a read
    if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        N32 n32Timeout;
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, tSize);

        // Pull timeout argument from the list
        n32Timeout = va_arg( tList, N32 );

        // restore stack (push)
        va_end(tList);

        // Wait for data to arrive...
        OSAL.eSemTake(psBuffer->hReadSem, n32Timeout);
    }

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);

    if (bOk == FALSE)
    {
        // Cannot enter exclusive access. Buffer is corrupted
        // or deleted
        if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
        {
            // We have locked semaphore. Need to release it before returning
            OSAL.eSemGive(psBuffer->hReadSem);
        }
        return 0;
    }

    // Read the data
    tNum = tBufferReadHead(psBuffer, pun8Dst, tSize);

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/
    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL))
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }

    // Return the total number of bytes read
    return tNum;
#else
    return (size_t)0;
#endif
}


/*******************************************************************************
*
*   OSAL_tBufferReadHeadBits
*
*   This function provides a simple method of reading variable length
*   bit-fields for the caller.

*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvDest - A pointer to a location where the extracted bits are to be
*           placed (first location/byte).  If it is NULL this works as a seek
*           function.
*       tDstBitOffset - The destination bit offset where bit 0 is the leftmost bit
*           at pvDst.
        n32Bits - The length of the bit string to be copied in (the number of bits)
*
*   Outputs:
*       A size_t value representing the number of bits read from the buffer
*
*******************************************************************************/
size_t OSAL_tBufferReadHeadBits(
    OSAL_BUFFER_HDL hBuffer,
    void *pvDst,
    size_t tDstBitOffset,
    N32 n32Bits,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNumBits = 0;

    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psHead;
    UN8 *pun8Dst = (UN8 *)pvDst;

    size_t tTotalBufferBits, tAlignedBits;
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle and if they asked for anything
    if( (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
        (n32Bits == 0) )
    {
        return 0;
    }

    // If we are attempting a negative seek, abort
    if ( (pvDst == NULL) &&
         (n32Bits < 0) )
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if read blocking is enabled, if so make sure we
    // get a the read semaphore before attempting a read
    if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        N32 n32Timeout;
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, n32Bits);

        // Pull timeout argument from the list
        n32Timeout = va_arg( tList, N32 );

        // restore stack (push)
        va_end(tList);

        // Wait for data to arrive...
        OSAL.eSemTake(psBuffer->hReadSem, n32Timeout);
    }

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Cannot enter exclusive access section. Fail.
        if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemGive(psBuffer->hReadSem);
        }
        return 0;
    }

    // Locally point to the current head block as
    // we know it at this moment.
    psHead = psBuffer->psHead;

    // Check if we have a block at the head of this buffer to read from
    if(psHead != NULL)
    {
        // This is a seek operation.  No need to perform the loop
        // used when we are actually performing a read, which increments
        // a bit at a time.  We can do this in bigger chunks.
        if (pun8Dst == NULL)
        {
            size_t tBufferSizeInBits =
                ((psBuffer->tSize*8) + psBuffer->un8RemainingBits);
            size_t tNumBitsPassed;
            ptrdiff_t tNumAvailableBits;

            // Cap how many bits can be requested
            if ((UN16)abs(n32Bits) > tBufferSizeInBits)
            {
                n32Bits = tBufferSizeInBits;
            }

            do
            {
                // We always seek from the head of the buffer as long as we
                // are able to do so (i.e. as long as we have data to seek past).

                // Can only seek past bits up to the write pointer
                tAlignedBits = ((psHead->pun8Write - psHead->pun8Read) * 8);
                tNumAvailableBits =
                    psHead->tRemainingReadBits +
                    psHead->tRemainingWriteBits +
                    tAlignedBits;

                if (psHead->tRemainingReadBits >= (size_t)abs(n32Bits))
                {
                    // We have more remaining bits then we need to seek
                    // so eat the remaining bits and don't touch the
                    // byte-aligned data
                    psHead->tRemainingReadBits -= n32Bits;

                    // Adjust our leftover value
                    psHead->un8ReadLeftover &=
                        ((1 << psHead->tRemainingReadBits ) -1);

                    tNumBitsPassed = n32Bits;
                }
                else if (tNumAvailableBits >= n32Bits)
                {
                    size_t tSeekBitsToProcess = n32Bits;

                    // First zero out the remaining read bits
                    tSeekBitsToProcess -= psHead->tRemainingReadBits;
                    psHead->tRemainingReadBits = 0;

                    // Clear the read leftover value
                    psHead->un8ReadLeftover = 0;

                    // Increment the read pointer as
                    // far as possible while staying within
                    // the block bounds
                    if (tAlignedBits >= tSeekBitsToProcess)
                    {
                        // move the read pointer ahead the number of
                        // whole bytes we have
                        psHead->pun8Read += (tSeekBitsToProcess / 8);

                        // if we still have bits left, save that data
                        tSeekBitsToProcess = tSeekBitsToProcess % 8;

                        if (tSeekBitsToProcess > 0)
                        {
                            // Adjust the read leftover
                            psHead->tRemainingReadBits = 8 - tSeekBitsToProcess;
                            psHead->un8ReadLeftover = *psHead->pun8Read++;
                            psHead->un8ReadLeftover &=
                                ((1 << psHead->tRemainingReadBits ) -1);
                        }
                    }
                    else // We need to dive into the written bits
                    {
                        psHead->pun8Read += (tAlignedBits / 8);
                        tSeekBitsToProcess -= tAlignedBits;

                        // Use the write leftover
                        psHead->tRemainingReadBits = 8 - tSeekBitsToProcess;
                        psHead->un8ReadLeftover = psHead->un8WriteLeftover;
                        psHead->un8ReadLeftover &=
                            ((1 << psHead->tRemainingReadBits ) -1);
                        psHead->un8WriteLeftover = 0;
                    }

                    tNumBitsPassed = n32Bits;
                }
                else
                {
                    psHead->pun8Read = psHead->pun8Write;

                    psHead->tRemainingReadBits = 0;
                    psHead->un8ReadLeftover = 0;
                    psHead->tRemainingWriteBits = 0;
                    psHead->un8WriteLeftover = 0;

                    tNumBitsPassed = tNumAvailableBits;
                }

                // Increment number of bits we passed so far
                tNumBits += tNumBitsPassed;

                // Decrement number of bits left to seek past
                n32Bits -= tNumBitsPassed;

                // If my current read position is beyond the current block
                // boundaries we need to free this block as well.
                if ( (psHead->pun8Read == psHead->pun8EoB) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psHead = psFreeBlock(psHead);

                    // Check if anything is left
                    if(psHead == NULL)
                    {
                        // No more data to read
                        break;
                    }
                }
            } while ( n32Bits != 0);
        }
        else
        {
            UN8 un8BitMask = 0;
            UN8 un8DstBitMask =0;
            UN8 un8DstShiftBits = 0;
            UN8 un8BitIndex = tDstBitOffset;

            // If we have a offset greater than 8 bits, adjust the destination pointer accordingly
            while (tDstBitOffset >= 8)
            {
                // Subtract 8 bits to get it down to a byte-boundary
                tDstBitOffset = tDstBitOffset - 8;

                if (n32Bits < 0)
                {
                    pun8Dst--;
                }
                else
                {
                    pun8Dst++;
                }
            }

            // Figure out our initial dst shift bits
            if (tDstBitOffset > 0)
            {
                un8DstShiftBits = (8 - tDstBitOffset) - 1;
            }
            else
            {
                if ((n32Bits > 8) || ((n32Bits % 8) == 0))
                {
                    // Set to an "even" value
                    un8DstShiftBits = 7;
                }
                else
                {
                    // Set to an "uneven" value
                    un8DstShiftBits = abs(n32Bits % 8) - 1;
                }
            }

            // Read as many bits as I can...
            do
            {
                // We always read from the head of the buffer as long as
                // we are able to do so (i.e. as long as we have bits to read).

                // If my current read position is beyond the current block
                // boundaries we need to move on to the next block. If this
                // occurs we should free the current block we are reading from
                // before moving on.
                if ( (psHead->pun8Read == psHead->pun8EoB) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psHead = psFreeBlock(psHead);

                    // Check if anything is left
                    if(psHead == NULL)
                    {
                        // No more data to read
                        break;
                    }
                }

                // Check that we are not on the write pointer within this
                // block. If we are, then we need to stop reading
                if ( (psHead->pun8Read == psHead->pun8Write) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // No more data to read from this block. We are now
                    // at the end (tail) of the buffer
                    break;
                }

                // Read data
                if (psHead->tRemainingReadBits == 0)
                {
                    if (psHead->pun8Read != psHead->pun8Write)
                    {
                        // We don't have any remaining bits, so
                        // move some data to our left over bits
                        psHead->un8ReadLeftover = *psHead->pun8Read++;
                        psHead->tRemainingReadBits = 8;
                    }
                    else
                    {
                        psHead->un8ReadLeftover = psHead->un8WriteLeftover;
                        psHead->tRemainingReadBits = psHead->tRemainingWriteBits;
                        psHead->un8WriteLeftover = 0;
                        psHead->tRemainingWriteBits = 0;
                    }
                }

                // Calculate our bit mask using the shift value
                un8DstBitMask = 1 << un8DstShiftBits;

                // Calculate the bit mask for our source
                un8BitMask = 1 << (psHead->tRemainingReadBits - 1);

                // Set the bit to 1 if the source is 1
                if ( (psHead->un8ReadLeftover & un8BitMask) != 0)
                {
                    *pun8Dst |= un8DstBitMask;
                }
                else
                {
                    *pun8Dst &= (~un8DstBitMask);
                }

                // Decrement our bits remaining
                psHead->tRemainingReadBits--;

                // Adjust our leftover value
                psHead->un8ReadLeftover &=
                    ((1 << psHead->tRemainingReadBits ) -1);

                // Increment the bit index -- if we wrap back
                // to zero we're on a new byte
                un8BitIndex = (un8BitIndex + 1 ) % 8;
                if (un8BitIndex == 0)
                {
                    if (n32Bits < 0)
                    {
                        pun8Dst--;
                    }
                    else
                    {
                        pun8Dst++;
                    }
                }

                // Increment number of bits read for this read operation
                tNumBits++;

                // decrement our n16Bits value accordingly
                if (n32Bits < 0)
                {
                    n32Bits++;
                }
                else
                {
                    n32Bits--;
                }

                // Decrement the un8DstShiftBits.  If it is zero,
                // reset it
                // then we'd be byte-aligned in the dst
                if (un8DstShiftBits == 0)
                {
                    un8DstShiftBits = 8 - 1;
                }
                else
                {
                    un8DstShiftBits--;
                }

            } while(n32Bits != 0);
        }

        // Decrement buffer size by the number read
        tTotalBufferBits = psBuffer->tSize * 8;

        if (tTotalBufferBits < psBuffer->tSize)
        {
            tTotalBufferBits = SIZE_T_MAX;
        }
        else
        {
            tTotalBufferBits += psBuffer->un8RemainingBits;
        }

        tTotalBufferBits -= tNumBits;

        psBuffer->tSize = tTotalBufferBits / 8;

        // TODO: calculate this better
        psBuffer->un8RemainingBits = tTotalBufferBits % 8;
    }

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL) )
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    // Return the total number of bits read
    return tNumBits;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferReadTail
*
*   This function is used to read a byte-stream of data  from the tail of a
*   provided Dynamic Buffer and copy the requested number of bytes starting
*   with the end of the destination memory provided.
*
*   If the destination pointer is NULL, this function operates as a standard
*   seek operation, starting at the tail of a provided Dynamic Buffer and
*   releasing memory along the way.

*   If the destination pointer is not NULL, and if the entire read operation
*   cannot be completed, the number of bytes which were read and copied will
*   be returned to the caller. The bytes will be read starting from the tail
*   of the buffer and placed starting at the pvDst + tSize address, until tSize
*   bytes have been copied or all available data has been read.
*
*   This operation may block if the buffer associated with the handle hBuffer
*   was allocated with the blocking-read feature enabled.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvDst - A void pointer to the last memory address where the data read
*           from the buffer should be stored. If the amount of data read is
*           less than the requested amount then the data will appear to be
*           right-justified to the end of the provided destination memory
*           (See 9.12.3.1).
*       tSize - The amount of data to read from the buffer. The request may
*           be satisified in whole, partial or no data being read depending
*           upon the amount of data presently in the provided buffer. The
*           caller should always use the return value to determine how much
*           data was actually read.
*       n32Timeout - The time in milliseconds to wait for data to be
*           available while the caller block or suspends the current
*           process/task. Also may be provided OSAL_OBJ_TIMEOUT_INFINITE
*           to have byte-wise read operations wait forever, or
*           OSAL_OBJ_TIMEOUT_NONE to simply read any available data without
*           blocking/suspending. The value provided is applied when performing
*           a byte-wise read from either the head or tail of the provided
*           buffer. Block-wise read and write APIs cannot block or suspend
*           the caller and thus this parameter has no effect on them.
*
*   Outputs:
*       A size_t value indicating the number of bytes actually read from
*       the buffer.
*
*******************************************************************************/
size_t OSAL_tBufferReadTail (
    OSAL_BUFFER_HDL hBuffer,
    void *pvDst,
    size_t tSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNum = 0;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psTail;
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle and if they asked for anything
    if( (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
        (tSize == 0) )
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if read blocking is enabled, if so make sure we
    // get a the read semaphore before attempting a read
    if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        N32 n32Timeout;
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, tSize);

        // Pull timeout argument from the list
        n32Timeout = va_arg( tList, N32 );

        // restore stack (push)
        va_end(tList);

        // Wait for data to arrive...
        OSAL.eSemTake(psBuffer->hReadSem, n32Timeout);
    }

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Cannot enter exclusive access section. Fail.
        if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemGive(psBuffer->hReadSem);
        }
        return 0;
    }

    // Locally make a copy of the the current tail block as
    // we know it at this moment.
    psTail = psBuffer->psTail;

    // Check if we have a block at the tail of this buffer to read from
    if(psTail != NULL)
    {
        size_t tAvailableBytes;

        // Cap how many bytes can be requested
        if (tSize > psBuffer->tSize)
        {
            tSize = psBuffer->tSize;
        }

        if (pvDst == NULL)
        {
            // This is a seek operation.  No need to perform the loop
            // used when we are actually performing a read, which increments
            // a byte at a time.  We can do this in bigger chunks.
            do
            {
                // We always seek from the tail of the buffer as long as we
                // are able to do so (i.e. as long as we have bytes to seek past.
                // Note: we must do this in reverse order for tail seeking!

                // Can only seek past bytes up to the read pointer
                tAvailableBytes = (size_t)(psTail->pun8Write - psTail->pun8Read);
                if (tAvailableBytes > tSize)
                {
                    tAvailableBytes = tSize;
                }

                // Adjust block's write point
                psTail->pun8Write -= tAvailableBytes;

                // Adjust counters
                tNum += tAvailableBytes;
                tSize -= tAvailableBytes;

                // If my current write position is beyond the current block
                // boundaries we need to free this block as well.
                if (psTail->pun8Write == psTail->pun8SoB)
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psTail = psFreeBlock(psTail);
                }
            } while ((tSize > 0) && (psTail != NULL));
        }
        else
        {
            UN8 *pun8Dst = ((UN8 *)pvDst) + tSize;

            // Read as many bytes as I can from the tail of the buffer
            // Note: Reads from the tail of the buffer must be done
            // in reverse order!
            do
            {
                // We always begin reading from the current tail block provided
                // we are able to do so (i.e. as long as we have bytes to read).

                // Can only read past bytes up to the read pointer
                tAvailableBytes = (size_t)(psTail->pun8Write - psTail->pun8Read);
                if (tAvailableBytes > tSize)
                {
                    tAvailableBytes = tSize;
                }

                // Adjust source and destination points
                psTail->pun8Write -= tAvailableBytes;
                pun8Dst -= tAvailableBytes;

                // Adjust counters
                tNum += tAvailableBytes;
                tSize -= tAvailableBytes;

                // Read data
                OSAL.bMemCpy(pun8Dst, psTail->pun8Write, tAvailableBytes);

                // If my current write position is beyond the current block
                // boundaries we need to free this block as well.
                if (psTail->pun8Write == psTail->pun8SoB)
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psTail = psFreeBlock(psTail);
                }
            } while((tSize > 0) && (psTail != NULL));
        }

        // Decrement buffer size by the number read
        psBuffer->tSize -= tNum;
    }

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL) )
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    // Return the total number of bytes read
    return tNum;
#else
    return (size_t)0;
#endif
}


/*******************************************************************************
*
*   OSAL_tBufferSeekHead
*
*   This function is used to seek from the head of a provided Dynamic Buffer.
*   This operation will free all bytes which are found within tSize.
*   If the entire seek operation cannot be completed, the number of bytes
*   which were freed will be returned to the caller, effectively returning
*   the attainable offset value. This operation may block if the buffer being
*   read was allocated with the blocking-read feature enabled. Bytes will
*   be freed from the head of the buffer, until seek has progressed to tOffset
*   or all available data has been exhausted.
*
*   This function is a simple wrapper of the OSAL_tBufferReadHead API.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       tSize - The offset to which the buffer head should be moved. The
*           request may be satisified in whole, partial or no seek being
*           performed depending upon the amount of data presently in the
*           provided buffer. The caller should always use this return value
*           to determine which offset the seek was able to accomplish.
*
*   Outputs:
*       A size_t value indicating the offset accomplished.
*
*******************************************************************************/
size_t OSAL_tBufferSeekHead (
    OSAL_BUFFER_HDL hBuffer,
    size_t tSize
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    // Just call OSAL_tBufferReadHead with a NULL destination pointer
    tNum = OSAL_tBufferReadHead(
              hBuffer,
              NULL,
              tSize );

    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferSeekHeadBits
*
*   This function is used to seek from the head of a provided Dynamic Buffer.
*   This operation will free all bits which are found within tSizeBits.
*   If the entire seek operation cannot be completed, the number of bits
*   which were freed will be returned to the caller, effectively returning
*   the attainable offset value. This operation may block if the buffer being
*   read was allocated with the blocking-read feature enabled. Bits will
*   be freed from the head of the buffer, until seek has progressed to tOffset
*   or all available data has been exhausted.
*
*   This function is a simple wrapper of the OSAL_tBufferReadHeadBits API.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       tSizeBits - The offset to which the buffer head should be moved. The
*           request may be satisified in whole, partial or no seek being
*           performed depending upon the amount of data presently in the
*           provided buffer. The caller should always use this return value
*           to determine which offset the seek was able to accomplish.
*
*   Outputs:
*       A size_t value indicating the offset accomplished.
*
*******************************************************************************/
size_t OSAL_tBufferSeekHeadBits (
    OSAL_BUFFER_HDL hBuffer,
    size_t tSizeBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    // Just call OSAL_tBufferReadHead with a NULL destination pointer
    // and an empty destination offset
    tNum = OSAL_tBufferReadHeadBits(
              hBuffer,
              NULL,
              0,
              tSizeBits );

    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferSeekTail
*
*   This function is used to seek from the tail of a provided Dynamic Buffer.
*   This operation will free all bytes which are found within tSize.
*   If the entire seek operation cannot be completed, the number of bytes
*   which were freed will be returned to the caller, effectively returning
*   the attainable offset value. This operation may block if the buffer being
*   read was allocated with the blocking-read feature enabled. Bytes will
*   be freed from the head of the buffer, until seek has progressed to tOffset
*   or all available data has been exhausted.
*
*   This function is a simple wrapper of the OSAL_tBufferReadTail API.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       tSize - The offset to which the buffer tail should be moved. The
*           request may be satisified in whole, partial or no seek being
*           performed depending upon the amount of data presently in the
*           provided buffer. The caller should always use this return value
*           to determine which offset the seek was able to accomplish.
*
*   Outputs:
*       A size_t value indicating the offset accomplished.
*
*******************************************************************************/
size_t OSAL_tBufferSeekTail (
    OSAL_BUFFER_HDL hBuffer,
    size_t tSize
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    // Just call OSAL_tBufferReadTail with a NULL destination pointer
    tNum = OSAL_tBufferReadTail(
              hBuffer,
              NULL,
              tSize );

    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_hBufferReadBlock
*
*   This function is used to read the next available block of data from the
*   head (or front) of a provided Dynamic Buffer. The start address and size
*   of the block read from the buffer will be returned to the caller and thus
*   no data is copied from the buffer, the caller manipulates the data
*   in-place, within the buffer it is stored. The caller may read, manipulate
*   or otherwise process the read block until a call to
*   OSAL.eBufferReleaseBlock() is made, at which point control of the block
*   is returned to the buffer. This API provides a way to obtain access to
*   sequential data contained in a Dynamic Buffer without actually copying
*   it out as with a byte-wise read.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       ppun8Data - A pointer to a pointer where the read block's start
*           address will be stored.
*       ptSize - A pointer to a size_t object where the size of the read
*           block will be stored.
*
*   Outputs:
*       A valid OSAL_BUFFER_BLOCK_HDL upon successful read (some data exists
*       and can be read). Otherwise, OSAL_INVALID_BUFFER_BLOCK_HDL is returned
*       on failure. Note that the returned size of the block is arbitrary and
*       must be checked by the caller to determine how much data has actually
*       been obtained through this read block.
*
*******************************************************************************/
OSAL_BUFFER_BLOCK_HDL OSAL_hBufferReadBlock (
    OSAL_BUFFER_HDL hBuffer,
    UN8 **ppun8Data,
    size_t *ptSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    OSAL_BUFFER_BLOCK_HDL hBufferBlock =
        OSAL_INVALID_BUFFER_BLOCK_HDL;

    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle and inputs for validity
    if(
        (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
            (ppun8Data == NULL) ||
                (ptSize == NULL)
                    )
    {
        return OSAL_INVALID_BUFFER_BLOCK_HDL;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if read blocking is enabled, if so make sure we
    // get a the read semaphore before attempting a read
    if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        N32 n32Timeout;
        va_list tList; // variable arguments list

        // grab variable arguments (pop)
        va_start(tList, ptSize);

        // Pull timeout argument from the list
        n32Timeout = va_arg( tList, N32 );

        // restore stack (push)
        va_end(tList);

        // Wait for data to arrive...
        OSAL.eSemTake(psBuffer->hReadSem, n32Timeout);
    }

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Cannot enter exclusive access section. Fail.
        if(psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
        {
            OSAL.eSemGive(psBuffer->hReadSem);
        }
        return OSAL_INVALID_BUFFER_BLOCK_HDL;
    }

    if(psBuffer->psHead != NULL)
    {
        // Calculate amount of data in the read block and return it
        *ptSize = psBuffer->psHead->pun8Write - psBuffer->psHead->pun8Read;

        if(*ptSize > 0)
        {
            // Return start address of the block's data
            *ppun8Data = psBuffer->psHead->pun8Read;

            // Bump our read pointer, past the read data
            psBuffer->psHead->pun8Read += *ptSize;

            // Mark block as read reserved (cannot be freed till
            // a 'put' or 'release' call is made.
            psBuffer->psHead->bReadReserved = TRUE;

            // Decrement buffer size by the number of bytes being read
            psBuffer->tSize -= *ptSize;

            // Assign block pointer to return handle
            hBufferBlock = (OSAL_BUFFER_BLOCK_HDL)psBuffer->psHead;

            // Move head position to the next block, if we have
            // consumed an entire block.
            if(psBuffer->psHead->pun8Read == psBuffer->psHead->pun8EoB)
            {
                psBuffer->psHead = psBuffer->psHead->psNext;
            }
        }
    }

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if((bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL))
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    return hBufferBlock;
#else
    return OSAL_INVALID_BUFFER_BLOCK_HDL;
#endif
}

/*******************************************************************************
*
*   OSAL_eBufferReleaseBlock
*
*   This function is used to release a block back to a Dynamic Buffer
*   previously obtained for reading. This is called when the read block is no
*   longer required and the caller is finished using the block. This call will
*   effectively return the block back to the block pool from which it came.
*
*   Inputs:
*       hBufferBlock - A handle to a buffer's block which has been previously
*       obtained when OSAL.eBufferReadBlock() was called.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferReleaseBlock (
    OSAL_BUFFER_BLOCK_HDL hBufferBlock
        )
{
#if OSAL_BUFFER == 1
    OSALB_BLOCK_HEADER_STRUCT *psBlock =
        (OSALB_BLOCK_HEADER_STRUCT *)hBufferBlock;
    BOOLEAN bOk;

    // Check provided handle
    if(hBufferBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

/*****************************************************************************/
    bOk = psBlock->psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBlock->psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == TRUE)
    {
        // Clear read reserve flag
        psBlock->bReadReserved = FALSE;

        // Attempt to free this block, if we have read all the data possible
        // from it then it will be freed, otherwise it will remain allocated.
        // This call will adjust our next read block accordingly.
        if(psBlock->pun8Read == psBlock->pun8EoB)
        {
            // Free this block, we have read all the data possible from it.
            // This call will adjust our next read block accordingly.
            (void) psFreeBlock(psBlock);
        }

        psBlock->psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBlock->psBuffer->psBlockPoolInfo->hMutex);
    }
/*****************************************************************************/

    return OSAL_SUCCESS;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif

}

/*******************************************************************************
*
*   OSAL_eBufferAppend
*
*   This function is used to append the blocks from one buffer onto
*   another buffer.  This is a destructive append; the source buffer
*   is cleared of data as the append is performed.
*
*   Inputs:
*       hDstBuffer - The buffer to perform the operation on.
*       hBufferDataToAppend - The buffer from which blocks are extracted.
*
*   Outputs:
*       OSAL_SUCCESS on success or an appropriate value from
*       OSAL_RETURN_CODE_ENUM indicating the error experienced.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferAppend (
    OSAL_BUFFER_HDL hDstBuffer,
    OSAL_BUFFER_HDL hBufferDataToAppend
        )
{
#if OSAL_BUFFER == 1
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_SUCCESS;
    OSAL_BUFFER_BLOCK_HDL hReadBlock, hWriteBlock;
    size_t tReadBlockSize, tWriteBlockSize, tBytesToWrite, tDataByteSize,
        tDataBitSize;
    UN8 *pun8ReadData, *pun8WriteData;
    BOOLEAN bSuccess = TRUE;

    OSAL_BUFFER_ALLOCATED_ASSERT((OSALB_BUFFER_STRUCT*)hDstBuffer);
    OSAL_BUFFER_ALLOCATED_ASSERT((OSALB_BUFFER_STRUCT*)hBufferDataToAppend);

    // Read the number of bytes in the append data buffer
    tDataByteSize = OSAL.tBufferGetSize(hBufferDataToAppend);

    // Read the number of bits in the append data buffer
    tDataBitSize = OSAL.tBufferGetSizeInBits(hBufferDataToAppend);

    // These better indicate the name number of bits!
    // We perform this check to ensure the source buffer
    // is byte-aligned.
    if (tDataBitSize != (tDataByteSize * 8))
    {
        // Sorry!
        return OSAL_ERROR;
    }

    do
    {
        // Read a block from the data we're appending
        hReadBlock = OSAL.hBufferReadBlock(hBufferDataToAppend,
            &pun8ReadData, &tReadBlockSize);
        if (hReadBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
        {
            // all is done
            break;
        }

        while (tReadBlockSize > 0)
        {
            // Get a write block from the dest buffer
            hWriteBlock = OSAL.hBufferGetBlock(
                hDstBuffer,
                &pun8WriteData, &tWriteBlockSize);
            // Check we got a block
            if (hWriteBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
            {
                // no blocks available for writing
                eReturnCode = OSAL_ERROR_OUT_OF_MEMORY;
                break;
            }

            if (tReadBlockSize > tWriteBlockSize)
            {
                tBytesToWrite = tWriteBlockSize;
            }
            else
            {
                tBytesToWrite = tReadBlockSize;
            }

            // Copy the data from to the other
            bSuccess = OSAL.bMemCpy(pun8WriteData,
                pun8ReadData, tBytesToWrite);
            if (bSuccess == FALSE)
            {
                break;
            }

            // Write the block back to the destination buffer
            eReturnCode =
                OSAL.eBufferWriteBlock(hWriteBlock, tBytesToWrite);
            if (eReturnCode != OSAL_SUCCESS)
            {
                break;
            }

            // Modify tReadBlockSize and pun8ReadData
            tReadBlockSize -= tBytesToWrite;
            pun8ReadData += tBytesToWrite;
        }

        if (bSuccess == FALSE)
        {
            eReturnCode = OSAL_ERROR;
        }

        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }

        // releasing the copied block from buffer to append
        eReturnCode = OSAL.eBufferReleaseBlock(hReadBlock);
        hReadBlock = OSAL_INVALID_BUFFER_BLOCK_HDL;
        if (eReturnCode != OSAL_SUCCESS)
        {
            break;
        }
    } while (TRUE);

    if (hReadBlock != OSAL_INVALID_BUFFER_BLOCK_HDL)
    {
        OSAL.eBufferReleaseBlock(hReadBlock);
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_eBufferWriteToFile
*
*   This function is used to write the buffer to file
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       bConsumeData - This flag shows if need to keep written blocks 
*                     in the buffer
*       psFile - The file to write to.
*
*   Outputs:
*       OSAL_SUCCESS on success or an appropriate value from
*       OSAL_RETURN_CODE_ENUM indicating the error experienced.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferWriteToFile (
    OSAL_BUFFER_HDL hBuffer,
    BOOLEAN bConsumeData,
    FILE *psFile
        )
{
#if ((OSAL_BUFFER == 1) && (OSAL_FILE_SYSTEM == 1))
    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR_INVALID_INPUT;

    if ((psFile != NULL) && (hBuffer != OSAL_INVALID_BUFFER_HDL))
    {
        OSALB_TO_FILE_ITERATOR_STRUCT sIterator;

        sIterator.psFile = psFile;
        sIterator.bSuccess = TRUE;
        sIterator.bConsumeData = bConsumeData;

        eReturnCode = OSAL_eBufferBlocksIterate(hBuffer, 
            (OSAL_BUFFER_ITERATOR_HANDLER)eWriteBlockToFile,
            &sIterator);
        if (eReturnCode == OSAL_SUCCESS)
        {
            // checking the iterator arg flag
            if (sIterator.bSuccess == FALSE)
            {
                eReturnCode = OSAL_ERROR;
            }
        }
    }

    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferWriteHead
*
*   This function is used to write a byte-stream of data starting with the
*   last byte in the pvSrc memory provided to the head of the provided Dynamic
*   Buffer. If the entire write operation cannot be completed, the number of
*   bytes which were written will be returned to the caller. This operation
*   may block if the buffer being written was allocated with the blocking-write
*   feature enabled. The bytes will be written to the head of the buffer
*   starting at the pvSrc + tSize, until tSize bytes have been copied or all
*   data has been written which can be held by the buffer.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvSrc - A void pointer to the last memory address where the data to be
*           written to the buffer is stored. If the amount of data written is
*           less than the requested amount then only the right most bytes are
*           written to the head of the provided buffer (See 9.12.3.1).
*       tSize - The amount of data to write to the buffer. The request may be
*           satisified in whole, partial or no data being written depending upon
*           the space available in the provided buffer. The caller should always
*           use the return value to determine how much data was actually
*           written.
*
*   Outputs:
*       A size_t value indicating the number of bytes actually written to
*       the buffer.
*
*******************************************************************************/
size_t OSAL_tBufferWriteHead (
    OSAL_BUFFER_HDL hBuffer,
    const void *pvSrc,
    size_t tSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNum = 0;

    const UN8 *pun8Src = ((const UN8 *)pvSrc) + tSize;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psHead;
    N32 n32Timeout;
    va_list tList; // variable arguments list
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle
    if(hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if they want to write anything, if not
    // just allow pending readers to release.
    if(tSize == 0)
    {
        // Might we have a pending reader?
        if(OSAL_INVALID_OBJECT_HDL != psBuffer->hReadSem)
        {
            // Release them
            OSAL.eSemGive(psBuffer->hReadSem);
        }

        return 0; // Nothing written
    }

    // grab variable arguments (pop)
    va_start(tList, tSize);

    // Pull timeout argument from the list
    n32Timeout = (psBuffer->bBlockingWrite == TRUE) ?
        va_arg( tList, N32 ) : OSAL_OBJ_TIMEOUT_NONE;

    // restore stack (push)
    va_end(tList);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Might we have a pending reader?
        if(OSAL_INVALID_OBJECT_HDL != psBuffer->hReadSem)
        {
            // Release them
            OSAL.eSemGive(psBuffer->hReadSem);
        }

        return 0; // Nothing written
    }

    // Locally make a copy of the the current head block as
    // we know it at this moment.
    psHead = psBuffer->psHead;

    // Check if we have a read block to go ahead of, if not go get one
    if(psHead == NULL)
    {
        psHead = psAllocateBlock(psBuffer, FALSE, n32Timeout);
    }

    if(psHead != NULL)
    {
        size_t tAvaliableSpace;

        // Write as many bytes as I can to the head of the buffer (the read block).
        // Note: writes to the head of the buffer must be done in reverse order!
        do
        {
            // We always begin writing to the current block provided we are
            // able to do so (i.e. as long as we have room).
            if (psHead->pun8Read == psHead->pun8SoB)
            {
                // Allocate a new block and move current read block to it
                psHead = psAllocateBlock(psBuffer, FALSE, n32Timeout);
                if(psHead == NULL)
                {
                    break;
                }
            }

            // Check to make sure we have room to write this data, if not
            // attempt to get another block.
            tAvaliableSpace = psHead->pun8Read - psHead->pun8SoB;
            // Since the block has some space there is no need
            // check whether the size is not 0. It is quite enough
            // to adjust number of bytes to be written based on
            // bytes remaining in the source buffer
            if (tAvaliableSpace > tSize)
            {
                tAvaliableSpace = tSize;
            }

            // Adjust source and destination buffer pointers
            psHead->pun8Read -= tAvaliableSpace;
            // Adjust source data pointer
            pun8Src -= tAvaliableSpace;

            // Write data
            OSAL.bMemCpy(psHead->pun8Read, pun8Src, tAvaliableSpace);

            // Increment number of bytes written for this write operation
            tNum += tAvaliableSpace;
            tSize -= tAvaliableSpace;
        } while (tSize > 0);

        // Increment buffer size by the number written
        psBuffer->tSize += tNum;
    }

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL) )
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    // Return the total number of bytes written
    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferWriteTail
*
*   This function is used to write a byte-stream of data starting from the
*   pvSrc pointer provided to the tail of a provided Dynamic Buffer. If the
*   entire write operation cannot be completed, the number of bytes which
*   were written will be returned to the caller. This operation may block if
*   the buffer being written was allocated with the blocking-write feature
*   enabled. The bytes will be written to the buffer starting with the first
*   byte at pvSrc and placed at the tail of the buffer, until tSize bytes have
*   been written or all available space in the buffer has been used.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvSrc - A void pointer to the first memory address where the source
*           data to be written to the buffer is stored.
*       tSize - The amount of data to write to the buffer. The request may be
*           satisfied in whole, partial or no data being written depending
*           upon the space available in the provided buffer. The caller should
*           always use the return value to determine how much data was
*           actually written.
*       n32Timeout - The time in milliseconds to wait for space to be
*           available while the caller block or suspends the current
*           process/task. Also may be provided OSAL_OBJ_TIMEOUT_INFINITE
*           to have byte-wise write operations wait forever, or
*           OSAL_OBJ_TIMEOUT_NONE to simply write any available data without
*           blocking/suspending. The value provided is applied when performing
*           a byte-wise write to either the head or tail of the provided
*           buffer. Block-wise read and write APIs cannot block or suspend
*           the caller and thus this parameter has no effect on them.
*
*   Outputs:
*       A size_t value indicating the number of bytes actually written to the
*       buffer.
*
*******************************************************************************/
size_t OSAL_tBufferWriteTail (
    OSAL_BUFFER_HDL hBuffer,
    const void *pvSrc,
    size_t tSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNum = 0;

    const UN8 *pun8Src = (const UN8 *)pvSrc;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    N32 n32Timeout;
    va_list tList; // variable arguments list
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle
    if(hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if they want to write anything, if not
    // just allow pending readers to release.
    if(tSize == 0)
    {
        // Might we have a pending reader?
        if(OSAL_INVALID_OBJECT_HDL != psBuffer->hReadSem)
        {
            // Release them
            OSAL.eSemGive(psBuffer->hReadSem);
        }

        return 0; // Nothing written
    }

    // grab variable arguments (pop)
    va_start(tList, tSize);

    // Pull timeout argument from the list
    n32Timeout = (psBuffer->bBlockingWrite == TRUE) ?
        va_arg( tList, N32 ) : OSAL_OBJ_TIMEOUT_NONE;

    // restore stack (push)
    va_end(tList);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Might we have a pending reader?
        if(OSAL_INVALID_OBJECT_HDL != psBuffer->hReadSem)
        {
            // Release them
            OSAL.eSemGive(psBuffer->hReadSem);
        }

        return 0; // Nothing written
    }

    // Check to see if we are byte-aligned or not
    if (psBuffer->un8RemainingBits != 0)
    {
        size_t tSizeInBits = tSize * 8;

        // Look for overflow with a simple check
        if (tSizeInBits < tSize)
        {
            // We had an overflow, so lets cap it at the max
            // value for size_t
            tSizeInBits = SIZE_T_MAX;
        }

        // We aren't byte-aligned, so pass this off to the write bits function
        tNum = tBufferWriteTailBits(
            psBuffer,
            pun8Src,
            0,
            tSizeInBits,
            n32Timeout
                );

        // Convert tNum from bits to bytes read since the
        // read bits function returns number of bits read.
        tNum = tNum / 8;
    }
    else
    {
        OSALB_BLOCK_HEADER_STRUCT *psTail;

        // Locally make a copy of the the current tail block as
        // we know it at this moment.
        psTail = psBuffer->psTail;

        // Check if we have a write block to write to, if not go get one
        if(psTail == NULL)
        {
            psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
        }

        if(psTail != NULL)
        {
            size_t tAvailableBytes;

            // Write as many bytes as I can to the tail of the buffer
            do
            {
                // We always begin writing to the tail block provided we are
                // able to do so (i.e. as long as we have room).

                // Get room in bytes available for writing
                tAvailableBytes = psTail->pun8EoB - psTail->pun8Write;

                // Check to make sure we have room to write this data, if not
                // attempt to get another block. Also, if the block is marked as
                // 'reserved for write', then we must move on to the next block.
                // This is because I have no idea how much data the one who
                // reserved the block will actually write.
                if( (tAvailableBytes == 0) ||
                    (psTail->bWriteReserved == TRUE ) )
                {
                    // Allocate a new block and move current write block to it
                    psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
                    if(psTail == NULL)
                    {
                        break;
                    }
                }
                else
                {
                    // Adjust number of bytes to be written based on remaining room
                    // in the block and remaining data in the caller's buffer
                    if (tAvailableBytes > tSize)
                    {
                        // We have more room that caller needs,
                        // so just copy needed part
                        tAvailableBytes = tSize;
                    }

                    // Write data
                    OSAL.bMemCpy(psTail->pun8Write, pun8Src, tAvailableBytes);

                    // Adjust counters and pointers
                    psTail->pun8Write += tAvailableBytes;
                    pun8Src += tAvailableBytes;
                    tNum += tAvailableBytes;
                    tSize -= tAvailableBytes;
                }
            } while (tSize > 0);

            // Increment buffer size by the number written
            psBuffer->tSize += tNum;
        }
    }

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        (psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL) )
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    // Return the total number of bytes written
    return tNum;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_tBufferWriteTailBits
*
*   This function is used to write a byte-stream of data starting from the
*   pvSrc pointer provided to the tail of a provided Dynamic Buffer. If the
*   entire write operation cannot be completed, the number of bits which
*   were written will be returned to the caller. This operation may block if
*   the buffer being written was allocated with the blocking-write feature
*   enabled. The bits will be written to the buffer starting with the first
*   byte at pvSrc and placed at the tail of the buffer, until tSize bits have
*   been written or all available space in the buffer has been used.
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       pvSrc - A void pointer to the first memory address where the source
*           data to be written to the buffer is stored.
*       tSize - The amount of data to write to the buffer. The request may be
*           satisfied in whole, partial or no data being written depending
*           upon the space available in the provided buffer. The caller should
*           always use the return value to determine how much data was
*           actually written.
*       n32Timeout - The time in milliseconds to wait for space to be
*           available while the caller block or suspends the current
*           process/task. Also may be provided OSAL_OBJ_TIMEOUT_INFINITE
*           to have byte-wise write operations wait forever, or
*           OSAL_OBJ_TIMEOUT_NONE to simply write any available data without
*           blocking/suspending. The value provided is applied when performing
*           a byte-wise write to either the head or tail of the provided
*           buffer. Block-wise read and write APIs cannot block or suspend
*           the caller and thus this parameter has no effect on them.
*
*   Outputs:
*       A size_t value indicating the number of bits actually written to the
*       buffer.
*
*******************************************************************************/
size_t OSAL_tBufferWriteTailBits (
    OSAL_BUFFER_HDL hBuffer,
    const void *pvSrc,
    size_t tSrcBitOffset,
    N32 n32Bits,
    ...
        )
{
#if OSAL_BUFFER == 1
    size_t tNumBits = 0;

    const UN8 *pun8Src = (const UN8 *)pvSrc;
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    N32 n32Timeout;
    va_list tList; // variable arguments list
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle
    if(hBuffer == OSAL_INVALID_BUFFER_HDL)
    {
        return 0;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // Check if they want to write anything, if not
    // just allow pending readers to release.
    if(n32Bits == 0)
    {
        // Might we have a pending reader?
        if(OSAL_INVALID_OBJECT_HDL != psBuffer->hReadSem)
        {
            // Release them
            OSAL.eSemGive(psBuffer->hReadSem);
        }

        return 0; // Nothing written
    }

    // grab variable arguments (pop)
    va_start(tList, n32Bits);

    // Pull timeout argument from the list
    n32Timeout = (psBuffer->bBlockingWrite == TRUE) ?
        va_arg( tList, N32 ) : OSAL_OBJ_TIMEOUT_NONE;

    // restore stack (push)
    va_end(tList);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == TRUE)
    {
        tNumBits = tBufferWriteTailBits(
            psBuffer,
            pun8Src,
            tSrcBitOffset,
            n32Bits,
            n32Timeout
        );

        // If the buffer still has bytes in it, then we can return
        // the semaphore we obtained for the next read, otherwise
        // leave the semaphore as taken. When something arrives it
        // will be given.
        if ((psBuffer->tSize > 0) || (psBuffer->un8RemainingBits > 0))
        {
            bGiveSemaphore = TRUE;
        }

        psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
    }
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if( (bGiveSemaphore == TRUE) &&
        psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL)
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive(psBuffer->hReadSem);
    }
    // Return the total number of bits written
    return tNumBits;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_hBufferGetBlock
*
*   This function is used to obtain a pointer to the next available block of
*   data from the tail (or end) of a provided Dynamic Buffer. The start address
*   and size of the block pulled from the buffer will be returned to the caller.
*   The caller then is free to treat this block of memory as its own (i.e. as
*   if it were obtained using a malloc() call) until a call to
*   OSAL.eBufferWriteBlock() is made, at which point control of the block is
*   returned to the buffer. This API provides a way to obtain write-access to
*   sequential data contained in a Dynamic Buffer without actually copying it
*   in as with a byte-wise write. Note that this call causes a "void" or
*   reserved area of memory to be created in the Dynamic Buffer, but the tail
*   of the buffer does not move until the block has been written back to the
*   buffer using OSAL.eBufferWriteBlock(). Additional write-blocks may be
*   retrieved or additional bytes-written to the buffer without a matching
*   OSAL.eBufferWriteBlock(), but this data will not be readable until the
*   "void" or reserved area has been "closed-up" using OSAL.eBufferWriteBlock().
*
*   Inputs:
*       hBuffer - The buffer to perform the operation on.
*       ppun8Data - A pointer to a pointer where the write block's start
*           address will be stored.
*       ptSize - A pointer to a size_t object where the size of the write
*           block will be stored.
*       n32Timeout - The time in milliseconds to wait for space to be
*           available while the caller block or suspends the current
*           process/task. Also may be provided OSAL_OBJ_TIMEOUT_INFINITE
*           to have byte-wise write operations wait forever, or
*           OSAL_OBJ_TIMEOUT_NONE to simply write any available data without
*           blocking/suspending. The value provided is applied when performing
*           a byte-wise write to either the head or tail of the provided
*           buffer. Block-wise read and write APIs cannot block or suspend
*           the caller and thus this parameter has no effect on them.
*
*   Outputs:
*       A valid OSAL_BUFFER_BLOCK_HDL upon successful read (some available
*       space exists and can be written to). Otherwise,
*       OSAL_INVALID_BUFFER_BLOCK_HDL is returned on failure. Note that the
*       returned size of the block is arbitrary and must be checked by the
*       caller to determine how much data has actually been obtained through
*       this write block.
*
*******************************************************************************/
OSAL_BUFFER_BLOCK_HDL OSAL_hBufferGetBlock (
    OSAL_BUFFER_HDL hBuffer,
    UN8 **ppun8Data,
    size_t *ptSize,
    ...
        )
{
#if OSAL_BUFFER == 1
    OSALB_BUFFER_STRUCT *psBuffer =
        (OSALB_BUFFER_STRUCT *)hBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psTail = NULL;
    va_list tList; // variable arguments list
    N32 n32Timeout;
    BOOLEAN bOk;

    // Check provided handle and inputs
    if(
        (hBuffer == OSAL_INVALID_BUFFER_HDL) ||
            (ppun8Data == NULL) ||
                (ptSize == NULL)
                    )
    {
        return OSAL_INVALID_BUFFER_BLOCK_HDL;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    // grab variable arguments (pop)
    va_start(tList, ptSize);

    // Pull timeout argument from the list
    n32Timeout = (psBuffer->bBlockingWrite == TRUE) ?
        va_arg( tList, N32 ) : OSAL_OBJ_TIMEOUT_NONE;

    // restore stack (push)
    va_end(tList);

/*****************************************************************************/
    bOk = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == TRUE)
    {
        // Locally make a copy of the the current tail block as
        // we know it at this moment.
        psTail = psBuffer->psTail;

        // Check if we have a write block to write to, if not go get one
        if(psTail == NULL)
        {
            psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
        }

        if(psTail != NULL)
        {
            // Return a block to the caller...

            // Return current write block to the caller and reserve it
            // for a future write-block operation.
            if ((psTail->pun8Write == psTail->pun8EoB) ||
                (psTail->bWriteReserved == TRUE ))
            {
                // Allocate a new block and move current write block to it
                psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
            }

            if (psTail != NULL)
            {
                // At this point we were already pointing to an available block
                // or we allocated a new one, either way we are ready to go.

                // Mark the current write block as 'write reserved'
                psTail->bWriteReserved = TRUE;

                // Return block parameters
                *ppun8Data = psTail->pun8Write;
                *ptSize = psTail->pun8EoB - psTail->pun8Write;
            }
        }

        psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
    }
/*****************************************************************************/

    // Return pointer as the handle
    return (OSAL_BUFFER_BLOCK_HDL)psTail;
#else
    return OSAL_INVALID_BUFFER_BLOCK_HDL;
#endif
}

/*******************************************************************************
*
*   OSAL_eBufferWriteBlock
*
*   This function is used to write or commit a write-block to the buffer from
*   which it was obtained, effectively making it now available for reading by
*   another process. In the process of doing so it will treat any amount of
*   data written to the block less than or equal to the size obtained as a
*   contiguous stream of data with the previous and next data in the buffer
*   if any exists, effectively closing any gaps in the data (i.e. 6-byte block
*   obtained, but only 4-bytes written as shown below).
*
*   Inputs:
*       hBufferBlock - A handle to a buffer's block which has been previously
*           obtained when OSAL.eBufferGetBlock() was called.
*       tSize - The size in bytes of the data within this block which has
*           been written. tSize may be equal to or less than the block size
*           obtained.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBufferWriteBlock (
    OSAL_BUFFER_BLOCK_HDL hBufferBlock,
    size_t tSize
        )
{
#if OSAL_BUFFER == 1
    OSALB_BLOCK_HEADER_STRUCT *psBlock =
        (OSALB_BLOCK_HEADER_STRUCT *)hBufferBlock;
    BOOLEAN bGiveSemaphore = FALSE;
    BOOLEAN bOk;

    // Check provided handle
    if(hBufferBlock == OSAL_INVALID_BUFFER_BLOCK_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

/*****************************************************************************/
    bOk = psBlock->psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBlock->psBuffer->psBlockPoolInfo->hMutex);
    if(bOk == FALSE)
    {
        // Cannot enter exclusive access section. Aborting.
        return OSAL_ERROR;
    }

    // Truncate the provided size to the maximum size from the write
    // pointer to the end of the block.
    if (tSize > ((size_t)psBlock->pun8EoB - (size_t)psBlock->pun8Write))
    {
        tSize = psBlock->pun8EoB - psBlock->pun8Write;
    }

    // Adjust write pointer from the beginning of the data
    // to the end of the data (which may be the end of the block)
    psBlock->pun8Write += tSize;

    // If the block we are writing is the last block, we don't need
    // to do anything else as future writes will simply continue
    // from this point forward.
    // However, if the amount of data written to the block is
    // shorter than an entire block we must adjust the
    // End-of-Block pointer to the current write position. This is because
    // there is now a discontinuity in the data and we will need to
    // move on to the next block.
    if(psBlock->psNext != NULL)
    {
        // This is not the last block, so we must truncate the
        // length of this block to what was written. This is because
        // someone has already written data after this block.
        psBlock->pun8EoB = psBlock->pun8Write;
    }

    // Increment buffer size by the number written
    psBlock->psBuffer->tSize += tSize;

    // Now simply place the block back in the buffer by clearing the
    // 'write reserved' flag
    psBlock->bWriteReserved = FALSE;

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((psBlock->psBuffer->tSize > 0) ||
        (psBlock->psBuffer->un8RemainingBits > 0))
    {
        bGiveSemaphore = TRUE;
    }

    psBlock->psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBlock->psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/

    // If the buffer still has bytes in it, then we can return
    // the semaphore we obtained for the next read, otherwise
    // leave the semaphore as taken. When something arrives it
    // will be given.
    if ((bGiveSemaphore == TRUE) &&
        (psBlock->psBuffer->hReadSem != OSAL_INVALID_OBJECT_HDL))
    {
        // Indicate that the buffer still has data in it to receive
        OSAL.eSemGive( psBlock->psBuffer->hReadSem);
    }

    return OSAL_SUCCESS;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_hBlockPoolGetHandleByName
*
*   This function is used to obtain the block pool handle given the name
*   provided.
*
*   Inputs:
*       pacName - An ASCII Null terminated string used to identify the object
*       being querried.
*
*   Outputs:
*       The block pool's  object handle or OSAL_INVALID_OBJECT_HDL if no block
*       pool has this name.
*
*******************************************************************************/
OSAL_OBJECT_HDL OSAL_hBlockPoolGetHandleByName (
    const char *pacName
        )
{
#if OSAL_BUFFER == 1
    OSAL_OBJECT_HDL hObjectHdl = OSAL_INVALID_OBJECT_HDL;
    OSAL_OBJECT_STRUCT *psObj;

    // Search for the object by name
    psObj = OSALC_psFindObjectFromName(OSAL_OBJECT_TYPE_BLOCK_POOL, pacName);
    if(psObj != NULL)
    {
        hObjectHdl = (OSAL_OBJECT_HDL)psObj;
    }
    return hObjectHdl;
#else
    return OSAL_INVALID_OBJECT_HDL;
#endif
}

/*******************************************************************************
*
*   OSAL_eBlockPoolGetInfo
*
*   This function is used to return the information regarding a specified
*   block pool object.
*
*   Inputs:
*       hBlockPoolObj - A block pool's object handle to perform the operation
*           on.
*       psBlockPoolInfo - A pointer to OSAL_BLOCK_POOL_INFO_STRUCT which
*           will be populated by the API call with information about this
*           specified block pool object.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBlockPoolGetInfo (
    OSAL_OBJECT_HDL hBlockPoolObj,
    OSAL_BLOCK_POOL_INFO_STRUCT *psBlockPoolInfo
        )
{
#if OSAL_BUFFER == 1
    OSAL_RETURN_CODE_ENUM eReturnCode =
        OSAL_ERROR_INVALID_HANDLE;
    OSAL_OBJECT_STRUCT *psObj;

    // Check provided structure pointer
    if(psBlockPoolInfo == NULL)
    {
        return OSAL_ERROR_INVALID_POINTER;
    }

    // Extract object structure from handle provided and
    // verify a valid handle was provided
    psObj = OSALC_psGetObjectFromHandle(
        hBlockPoolObj,
        OSAL_OBJECT_TYPE_BLOCK_POOL
            );
    if(psObj != NULL)
    {
        BOOLEAN bOk;
/*****************************************************************************/
        bOk = psObj->puInfo->sBlockPool.bEnterExclusiveAccess(
            psObj->puInfo->sBlockPool.hMutex);
        if(bOk == TRUE)
        {
            // Copy object info I have into object info provided.
            *psBlockPoolInfo = psObj->puInfo->sBlockPool.sPublicInfo;

            psObj->puInfo->sBlockPool.vExitExclusiveAccess(
                psObj->puInfo->sBlockPool.hMutex);
/*****************************************************************************/
            // All is well
            eReturnCode = OSAL_SUCCESS;
        }
        else
        {
            eReturnCode = OSAL_ERROR;
        }
    }
    return eReturnCode;
#else
    return OSAL_ERROR_UNSUPPORTED_API;
#endif
}

/*******************************************************************************
*
*   OSAL_eBlockPoolList
*
*   This function is used to verbosely list all block pools that the OSAL is
*   managing.
*
*   Inputs:
*       none.
*
*   Outputs:
*       An OSAL return code. See Section 7 for details.
*
*******************************************************************************/
OSAL_RETURN_CODE_ENUM OSAL_eBlockPoolList ( void )
{
#if (OSAL_BUFFER == 1) && (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

    OSAL_RETURN_CODE_ENUM eReturnCode = OSAL_ERROR;
    OSAL_OBJECT_HDL hBlockPoolList;
    UN32 un32BlockPools = 0, un32BlockPoolNumber = 0;
    OSAL_STATISTICS_STRUCT sStatistics;

    // Iterate the list of block pools and list each entry

    // Grab the buffer linked list
    hBlockPoolList = OSALC_hGetObjectList(OSAL_OBJECT_TYPE_BLOCK_POOL);
    if(hBlockPoolList == OSAL_INVALID_OBJECT_HDL)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the number of buffers in the list
    eReturnCode = OSAL.eLinkedListItems(hBlockPoolList, &un32BlockPools);
    if(eReturnCode != OSAL_SUCCESS)
    {
        return OSAL_ERROR_INVALID_HANDLE;
    }

    // Get the object's statistics
    OSALC_vGetObjectStatistics(OSAL_OBJECT_TYPE_BLOCK_POOL, &sStatistics);

    // Check that one or more elements exist in the list
    if(un32BlockPools > 0)
    {
        // Print title
        printf("===========================\n");
        printf("| OSAL Block Pool List    |\n");
        printf("===========================\n");
    }

    // Iterate the list dumping each element individually
    eReturnCode = OSAL.eLinkedListIterate(hBlockPoolList,
        bPrintBlockPool, &un32BlockPoolNumber);
    if(eReturnCode == OSAL_NO_OBJECTS)
    {
        printf("Object list is empty!\n");
        return eReturnCode;
    }
    else if(eReturnCode != OSAL_SUCCESS)
    {
        return eReturnCode;
    }
    else
    {
        // Print summary
        printf("%u Block Pool(s) currently allocated by the system.\n",
                    un32BlockPools);

        // Print statistics
        printf("Current = %u, Minimum = %u, Maximum = %u\n",
               sStatistics.un32Current,
               sStatistics.un32Minimum,
               sStatistics.un32Maximum );

    }

    return OSAL_SUCCESS;
#else

    return OSAL_ERROR_UNSUPPORTED_API;

#endif /*(OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */
}

// future endinanless functions possible here?

/*******************************************************************************
*
*   OSAL_tBufferReadString
*
*******************************************************************************/
size_t OSAL_tBufferReadString (
    OSAL_BUFFER_HDL hBuffer,
    char *pacString,
    size_t tSize
        )
{
#if OSAL_BUFFER == 1
    size_t tStringLen = 0;
    char cRead;
    size_t tNum;
    BOOLEAN bLocked;
    OSALB_BUFFER_STRUCT *psBuffer = (OSALB_BUFFER_STRUCT *)hBuffer;

    // Check provided handle and if they asked for anything
    if((hBuffer == OSAL_INVALID_BUFFER_HDL) ||
       (tSize == 0))
    {
        return FALSE;
    }

    OSAL_BUFFER_ALLOCATED_ASSERT(psBuffer);

    /*****************************************************************************/
    bLocked = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    if (bLocked == FALSE)
    {
        return FALSE;
    }

    // Read out string one-by-one symbol
    do
    {
        // Read next symbol from the stream
        tNum = tBufferReadHead(psBuffer, (UN8*)&cRead, sizeof(cRead));
        if (tNum != sizeof(cRead))
        {
            // Some problem reading from the buffer, error out
            break;
        }

        // Put symbol in the output buffer
        pacString[tStringLen++] = cRead;
    } while ((cRead != '\0') && (tStringLen < tSize));

    psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
        psBuffer->psBlockPoolInfo->hMutex);
    /*****************************************************************************/

    return tStringLen;
#else
    return (size_t)0;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToUN16
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToUN16(
    OSAL_BUFFER_HDL hBuffer,
    UN16 *pun16Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > UN16_BITLEN) || (tBits == 0) || (pun16Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pun16Dest) + 1,
        UN16_BITLEN - tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        (UN8 *)pun16Dest,
        UN16_BITLEN - tBits,
        (N32)tBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToN16
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToN16(
    OSAL_BUFFER_HDL hBuffer,
    N16 *pn16Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N16_BITLEN) || (tBits == 0) || (pn16Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pn16Dest) + 1,
        N16_BITLEN-tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        (UN8 *)pn16Dest,
        N16_BITLEN - tBits,
        (N32)tBits);
#endif

    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn16Dest, N16, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToUN32
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToUN32(
    OSAL_BUFFER_HDL hBuffer,
    UN32 *pun32Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > UN32_BITLEN) || (tBits == 0) || (pun32Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pun32Dest) + 3,
        UN32_BITLEN - tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        (UN8 *)pun32Dest,
        UN32_BITLEN - tBits,
        (N32)tBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToN32
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToN32(
    OSAL_BUFFER_HDL hBuffer,
    N32 *pn32Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N32_BITLEN) || (tBits == 0) || (pn32Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pn32Dest) + 3,
        N32_BITLEN - tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pn32Dest),
        N32_BITLEN - tBits,
        (N32)tBits);
#endif

    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn32Dest, N32, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToUN64
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToUN64(
    OSAL_BUFFER_HDL hBuffer,
    UN64 *pun64Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > UN64_BITLEN) || (tBits == 0) || (pun64Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pun64Dest) + 7,
        UN64_BITLEN - tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        (UN8 *)pun64Dest,
        UN64_BITLEN - tBits,
        (N32)tBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferReadBitsToN64
*
*******************************************************************************/
BOOLEAN OSAL_bBufferReadBitsToN64(
    OSAL_BUFFER_HDL hBuffer,
    N64 *pn64Dest,
    size_t tBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N64_BITLEN) || (tBits == 0) || (pn64Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        ((UN8 *)pn64Dest) + 7,
        N64_BITLEN-tBits,
        -(N32)tBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferReadHeadBits(
        hBuffer,
        (UN8 *)pn64Dest,
        N64_BITLEN - tBits,
        (N32)tBits);
#endif

    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn64Dest, N64, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToUN16
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToUN16(
    OSAL_BUFFER_HDL hBuffer,
    UN16 *pun16Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N16_BITLEN) || (tBits == 0) || (pun16Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pun16Dest) + 1,
        N16_BITLEN - tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        (UN8 *)pun16Dest,
        N16_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToN16
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToN16(
    OSAL_BUFFER_HDL hBuffer,
    N16 *pn16Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N16_BITLEN) || (tBits == 0) || (pn16Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pn16Dest) + 1,
        N16_BITLEN-tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        (UN8 *)pn16Dest,
        N16_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif

    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn16Dest, N16, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToUN32
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToUN32(
    OSAL_BUFFER_HDL hBuffer,
    UN32 *pun32Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N32_BITLEN) || (tBits == 0) || (pun32Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pun32Dest) + 3,
        N32_BITLEN - tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        (UN8 *)pun32Dest,
        N32_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToN32
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToN32(
    OSAL_BUFFER_HDL hBuffer,
    N32 *pn32Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N32_BITLEN) || (tBits == 0) || (pn32Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pn32Dest) + 3,
        N32_BITLEN-tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        (UN8 *)pn32Dest,
        N32_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif

    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn32Dest, N32, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToUN64
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToUN64(
    OSAL_BUFFER_HDL hBuffer,
    UN64 *pun64Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N64_BITLEN) || (tBits == 0) || (pun64Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pun64Dest) + 7,
        N64_BITLEN - tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        (UN8 *)pun64Dest,
        N64_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif
    return (tNum == tBits);
#else
    return FALSE;
#endif
}

/*******************************************************************************
*
*   OSAL_bBufferPeekBitsToN64
*
*******************************************************************************/
BOOLEAN OSAL_bBufferPeekBitsToN64(
    OSAL_BUFFER_HDL hBuffer,
    N64 *pn64Dest,
    size_t tBits,
    size_t tBufferOffsetBits
        )
{
#if OSAL_BUFFER == 1
    size_t tNum;

    if ((tBits > N64_BITLEN) || (tBits == 0) || (pn64Dest == NULL))
    {
        return FALSE;
    }

#ifdef OSAL_CPU_LITTLE_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pn64Dest) + 7,
        N64_BITLEN - tBits,
        -(N32)tBits,
        tBufferOffsetBits);
#elif defined OSAL_CPU_BIG_ENDIAN
    tNum = OSAL_tBufferPeekBits(
        hBuffer,
        ((UN8 *)pn64Dest),
        N64_BITLEN - tBits,
        (N32)tBits,
        tBufferOffsetBits);
#endif
    // If requested number of bits is less than the type's size
    OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(pn64Dest, N64, tNum, tBits);

    return (tNum == tBits);
#else
    return FALSE;
#endif
}

// ************************************************************************** //
// ***************************** LOCAL FUNCTIONS **************************** //
// ************************************************************************** //

#if OSAL_BUFFER == 1

#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)

/*******************************************************************************
*
*   bPrintBlockPool
*
*   This is a local-helper function which actually prints the block pool
*   information for an individual block pool. This is a callback implementation
*   for a linked list iteration.
*
*   Inputs:
*       pvData - A pointer to an OSAL_OBJECT_STRUCT in the linked list of
*           maintained block pools.
*       pvArg - A pointer to an argument to process during each iteration. In
*           this context it is used to increment the block pool number for
*           display.
*
*   Outputs:
*       A BOOLEAN result which is always TRUE to indicate the iteration
*       should continue, until the end of the list.
*
*******************************************************************************/
static BOOLEAN bPrintBlockPool ( void *pvData, void *pvArg )
{
    OSAL_OBJECT_STRUCT *psObj = (OSAL_OBJECT_STRUCT *)pvData,
        *psCreatorObj;
    UN32 *pun32BlockPoolNumber = (UN32 *)pvArg;
    const char *pacCreatorObjName = "Unknown";

    // Check if argument is valid
    if(pun32BlockPoolNumber == NULL)
    {
        return FALSE;
    }

    // Prepare buffer number
    (*pun32BlockPoolNumber)++;

    // Check if object is valid
    if(psObj != NULL)
    {
        // Point to this object's info
        OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT sBlockPoolInfo;
        BOOLEAN bOk;

/*****************************************************************************/
        bOk = psObj->puInfo->sBlockPool.bEnterExclusiveAccess(
            psObj->puInfo->sBlockPool.hMutex);
        if(bOk == FALSE)
        {
            // Cannot enter exclusive access section. Fail.
            return FALSE;
        }

        // Copy object info I have into object info provided.
        sBlockPoolInfo = psObj->puInfo->sBlockPool;

        psObj->puInfo->sBlockPool.vExitExclusiveAccess(
            psObj->puInfo->sBlockPool.hMutex);
/*****************************************************************************/

        // Extract the creator task's name
        psCreatorObj = (OSAL_OBJECT_STRUCT *)psObj->hCreatorObj;
        if(psCreatorObj != NULL)
        {
            // Extract name of creator
            pacCreatorObjName = psCreatorObj->pacName;
        }

        // Print all block pool info...
        printf("Block Pool #%u\n", *pun32BlockPoolNumber);
        printf("Name = '%s' (hdl = 0x%p)\n",
            psObj->pacName,
               psObj
                   );
        printf("\tCreator = %s\n", pacCreatorObjName);
        printf("\tOptions = %#010x\n",
               sBlockPoolInfo.sPublicInfo.un32Options);
        printf("\tBlock Size = %u bytes\n"
               "\tNumber of blocks = %u blocks\n"
               "\tActual Block Size = %u bytes\n",
               sBlockPoolInfo.sPublicInfo.un16BlockSize,
               sBlockPoolInfo.sPublicInfo.un16NumBlocks,
               sBlockPoolInfo.sPublicInfo.un32ActualBlockSize
                   );
        printf("\tBlock Queue = 0x%p\n", sBlockPoolInfo.hBlockQueue);
        printf("\tBuffer Pool Capacity = %u bytes\n",
               sBlockPoolInfo.sPublicInfo.un16BlockSize * sBlockPoolInfo.sPublicInfo.un16NumBlocks);
        printf("\tBlocks Allocated = %u\n", sBlockPoolInfo.sPublicInfo.un16Allocated);
        printf("\tMaximum Blocks Allocated = %u\n",
               sBlockPoolInfo.sPublicInfo.un16MaxAllocated);
        printf("\tOverruns = %u\n", sBlockPoolInfo.sPublicInfo.un32OverrunCnt);
        puts("");
    }

    // Keep iterating
    return TRUE;
}

#endif /* (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */


/*******************************************************************************
*
*   psAllocateBlock
*
*   This is a local-helper function which is used to allocate a block from
*   a block pool to the provided buffer. This function returns the next
*   available block in the buffer.
*
*   Inputs:
*       psBuffer - A pointer to the buffer for which this new block is to
*           be added to.
*       bTail - A flag to indicate if the block is being added to the head
*           or tail of the buffer.
*       n32Timeout - A timeout to indicate how much time the call should
*           suspend if no blocks are available. The timeout is consistent
*           with other OSAL timeouts (i.e. -1 = infinite, 0 = no timeout, etc.)
*
*   Outputs:
*       A pointer to the next available block, NULL if no more blocks available
*
*******************************************************************************/
static OSALB_BLOCK_HEADER_STRUCT *psAllocateBlock (
    OSALB_BUFFER_STRUCT *psBuffer,
    BOOLEAN bTail,
    N32 n32Timeout
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    UN8 *pun8Data;
    OSALB_BLOCK_HEADER_STRUCT *psNewBlock = NULL;
    UN32 un32MessageAllocateFlags;
    BOOLEAN bLocked = TRUE;

    // Check if a blocking call is requested and set flag appropriately
    if(n32Timeout == OSAL_OBJ_TIMEOUT_NONE)
    {
        un32MessageAllocateFlags = OSAL_QUEUE_FLAG_NONBLOCK;
    }
    else
    {
        un32MessageAllocateFlags = OSAL_QUEUE_FLAG_BLOCK;

        // We are always in an exclusive access section when
        // this is called.
        psBuffer->psBlockPoolInfo->vExitExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
/*****************************************************************************/
    }

    // Allocate a new block from the block pool
    // Overlay new block header onto acquired block
    eReturnCode =
        OSAL.eMessageAllocate (
            psBuffer->psBlockPoolInfo->hBlockQueue,
            (void*)((void *)&psNewBlock),
            un32MessageAllocateFlags );
    if(un32MessageAllocateFlags == OSAL_QUEUE_FLAG_BLOCK)
    {
        // Re-enter exclusive access.
/*****************************************************************************/
        bLocked = psBuffer->psBlockPoolInfo->bEnterExclusiveAccess(
            psBuffer->psBlockPoolInfo->hMutex);
    }

    // Checking that message was successfully allocated and
    // exclusive access section is entered (or not attempted to)
    if((eReturnCode == OSAL_SUCCESS) && (bLocked == TRUE))
    {
        // Increment and update block pool statistics
        if(psBuffer->psBlockPoolInfo->sPublicInfo.un16Allocated <
             psBuffer->psBlockPoolInfo->sPublicInfo.un16NumBlocks)
        {
            psBuffer->psBlockPoolInfo->sPublicInfo.un16Allocated++;
            if(psBuffer->psBlockPoolInfo->sPublicInfo.un16Allocated >
               psBuffer->psBlockPoolInfo->sPublicInfo.un16MaxAllocated)
            {
                psBuffer->psBlockPoolInfo->sPublicInfo.un16MaxAllocated =
                    psBuffer->psBlockPoolInfo->sPublicInfo.un16Allocated;
            }
        }

        // Increment and update buffer statistics
        if(psBuffer->un16Allocated <
           psBuffer->psBlockPoolInfo->sPublicInfo.un16NumBlocks)
        {
            psBuffer->un16Allocated++;
            if(psBuffer->un16Allocated >
               psBuffer->un16MaxAllocated)
            {
                psBuffer->un16MaxAllocated =
                    psBuffer->un16Allocated;
            }
        }

        // Overlay new data area onto acquired block
        pun8Data = (UN8 *)(psNewBlock + 1);

        // Populate block header...
        psNewBlock->psBuffer = psBuffer;
        psNewBlock->pun8SoB = pun8Data;
        psNewBlock->pun8EoB =
            psNewBlock->pun8SoB +
                psBuffer->psBlockPoolInfo->sPublicInfo.un32ActualBlockSize -
                        sizeof(OSALB_BLOCK_HEADER_STRUCT);
        psNewBlock->bWriteReserved = FALSE;
        psNewBlock->bReadReserved = FALSE;
        psNewBlock->tRemainingReadBits = 0;
        psNewBlock->tRemainingWriteBits = 0;
        psNewBlock->un8ReadLeftover = 0;
        psNewBlock->un8WriteLeftover = 0;

        if(bTail == TRUE)
        {
            // This block is being added to the tail of the current buffer
            psNewBlock->psNext =
                (OSALB_BLOCK_HEADER_STRUCT *)NULL;
            psNewBlock->pun8Read = psNewBlock->pun8SoB;
            psNewBlock->pun8Write = psNewBlock->pun8SoB;

            // The next block is added to the tail so the previous block
            // is always the current tail.
            psNewBlock->psPrev = psBuffer->psTail;

            // Link this new block after the current tail block,
            // if the current tail block exists.
            if(psBuffer->psTail != NULL)
            {
                // Link current write block to the new block
                psBuffer->psTail->psNext = psNewBlock;
            }

            // Make the new block the current tail block
            psBuffer->psTail = psNewBlock;

            // Check if this is the first block, if so, make it also
            // the current head block.
            if(psBuffer->psHead == NULL)
            {
                psBuffer->psHead = psNewBlock;
            }
        }
        else
        {
            // This block is being added to the head of the buffer
            psNewBlock->psPrev =
                (OSALB_BLOCK_HEADER_STRUCT *)NULL;
            psNewBlock->pun8Read = psNewBlock->pun8EoB;
            psNewBlock->pun8Write = psNewBlock->pun8EoB;

            // The new block is added to the head of the buffer so the next
            // block from this one is always the current head.
            psNewBlock->psNext = psBuffer->psHead;

            // Link this new block before the current head,
            // if the current head block exists.
            if(psBuffer->psHead != NULL)
            {
                // Link current read block to the new block
                psBuffer->psHead->psPrev = psNewBlock;
            }

            // Make the new block the current head block
            psBuffer->psHead = psNewBlock;

            // Check if this is the first block, if so, make it also
            // the current tail block.
            if(psBuffer->psTail == NULL)
            {
                psBuffer->psTail = psNewBlock;
            }
        }
    }
    else
    {
        // Error! Update statistics for the block pool
        psBuffer->psBlockPoolInfo->sPublicInfo.un32OverrunCnt++;
    }

    return psNewBlock;
}

/*******************************************************************************
*
*   psFreeBlock
*
*   This is a local-helper function which is used to free a previously
*   allocated block back to the block pool from which it came.
*
*   Inputs:
*       psFreeBlock - A pointer to a previously allocated block which is to be
*       freed.
*
*   Outputs:
*       A pointer to the next block in the buffer.
*
*******************************************************************************/
static OSALB_BLOCK_HEADER_STRUCT *psFreeBlock (
    OSALB_BLOCK_HEADER_STRUCT *psFreeBlock
        )
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    OSALB_BUFFER_STRUCT *psBuffer =
        psFreeBlock->psBuffer;
    OSALB_BLOCK_HEADER_STRUCT *psNext = NULL;

    // Someone has decided that this block is no longer needed as
    // part of the buffer, so it can be de-linked from the buffer

    // Sew up the ends...delink block

    // Tie this block's previous block to the next block
    if(psFreeBlock->psPrev != NULL)
    {
        psFreeBlock->psPrev->psNext =
            psFreeBlock->psNext;
    }

    // Tie this block's next block to the previous block
    if(psFreeBlock->psNext != NULL)
    {
        psFreeBlock->psNext->psPrev =
            psFreeBlock->psPrev;
    }

    // Now we need to make sure we modify any head/tail block
    // references to this block.

    // If the block being freed is the current head block,
    // make the next block the new head
    if(psFreeBlock == psBuffer->psHead)
    {
        psBuffer->psHead = psFreeBlock->psNext;

        // Return the next block
        psNext = psBuffer->psHead;
    }

    // If the block being freed is the current tail block,
    // make the previous block the new tail
    if(psFreeBlock == psBuffer->psTail)
    {
        psBuffer->psTail = psFreeBlock->psPrev;

        // Return the next block
        psNext = psBuffer->psTail;
    }

    // Isolate block
    psFreeBlock->psPrev = NULL;
    psFreeBlock->psNext = NULL;

    // At this point the block is no longer a part of the buffer
    // it is still allocated from the block pool.

    // First check if block can be freed, if not, don't
    if(psFreeBlock->bReadReserved == FALSE)
    {
        // Free this block, we have read all the data possible from it.

        // Free this block...
        eReturnCode = OSAL.eMessageFree(
            (void *)psFreeBlock);
        if(eReturnCode == OSAL_SUCCESS)
        {
            // Decrement block pool statistics
            psBuffer->psBlockPoolInfo->sPublicInfo.un16Allocated--;

            // Decrement buffer statistics
            psBuffer->un16Allocated--;
        }
    }

    return psNext;
}

/*******************************************************************************
*
*   tBufferPeekBits
*
*******************************************************************************/
static size_t tBufferPeekBits (
    OSALB_BUFFER_STRUCT const *psBuffer,
    UN8 *pun8Dst,
    size_t tDstBitOffset,
    N32 n32Bits,
    size_t tBufferOffsetBits
        )
{
    size_t tNumBits = 0;
    UN8 un8BitIndex, un8BitMask = 0, un8DstBitMask = 0, un8DstShiftBits;

    if(psBuffer->psHead != NULL)
    {
        OSALB_BLOCK_HEADER_STRUCT sHead = *psBuffer->psHead;
            // we are just taking a peek

        // If we have a offset greater than 8 bits, adjust the
        // destination pointer accordingly
        while (tDstBitOffset >=8)
        {
            // Subtract 8 bits to get it down to a byte-boundary
            tDstBitOffset = tDstBitOffset - 8;

            if (n32Bits < 0)
            {
                pun8Dst--;
            }
            else
            {
                pun8Dst++;
            }
        }

        // Figure out our initial dst shift bits
        if (tDstBitOffset > 0)
        {
            un8DstShiftBits = (8- tDstBitOffset) - 1;
        }
        else
        {
            if ((n32Bits > 8) || ((n32Bits % 8) == 0))
            {
                // Set to an "even" value
                un8DstShiftBits = 7;
            }
            else
            {
                // Set to an "uneven" value
                un8DstShiftBits = abs(n32Bits % 8) - 1;
            }
        }

        // Set up our starting bit index
        un8BitIndex = tDstBitOffset;

        // Read as many bits as I can...
        do
        {
            // We always read from the head of the buffer as long as
            // we are able to do so (i.e. as long as we have bits to read).

            // If my current read position is beyond the current block boundaries
            // and there are no unaligned bits waiting to be read then
            // we need to move on to the next block.
            if ( (sHead.pun8Read == sHead.pun8EoB) &&
                 (sHead.tRemainingReadBits == 0) &&
                 (sHead.tRemainingWriteBits == 0) )
            {
                // Check if anything is left
                if(sHead.psNext == NULL)
                {
                    // No more data to read
                    break;
                }

                // Adjust to our next head block accordingly.
                sHead = *sHead.psNext;
            }

            // Check that we are not on the write pointer within this
            // block. If we are, then we need to stop reading
            if ( (sHead.pun8Read == sHead.pun8Write) &&
                  (sHead.tRemainingReadBits == 0) &&
                  (sHead.tRemainingWriteBits == 0))
            {
                // No more data to read from this block. We are now
                // at the end (tail) of the buffer
                break;
            }

            // Adjust our internal bit values in the head
            if (sHead.tRemainingReadBits == 0)
            {
                if (sHead.pun8Read == sHead.pun8Write)
                {
                    // We're out of data.  Use the remaining write
                    // bits now
                    sHead.un8ReadLeftover = sHead.un8WriteLeftover;
                    sHead.tRemainingReadBits = sHead.tRemainingWriteBits;
                    sHead.un8WriteLeftover = 0;
                    sHead.tRemainingWriteBits = 0;
                }
                else
                {
                    // We don't have any remaining bits, so
                    // move some data to our left over bits
                    sHead.un8ReadLeftover = *sHead.pun8Read++;
                    sHead.tRemainingReadBits = 8;
                }
            }

            // Check if we can begin copying data, if so go ahead, otherwise
            // just decrement the offset and move on.
            if(tBufferOffsetBits == 0)
            {
                // Calculate our destination bit mask using the shift value
                un8DstBitMask = 1 << un8DstShiftBits;

                // Calculate the bit mask for our source
                un8BitMask = 1 << (sHead.tRemainingReadBits - 1);

                // Set the bit to 1 if the source is 1
                if ( (sHead.un8ReadLeftover & un8BitMask) != 0)
                {
                    *pun8Dst |= un8DstBitMask;
                }
                else
                {
                    *pun8Dst &= (~un8DstBitMask);
                }

                // Decrement our bits remaining
                sHead.tRemainingReadBits--;

                // Adjust our leftover value
                sHead.un8ReadLeftover &=
                    ((1 << sHead.tRemainingReadBits ) -1);

                // Increment the bit index -- if we wrap back
                // to zero we're on a new byte
                un8BitIndex = (un8BitIndex + 1 ) % 8;
                if (un8BitIndex == 0)
                {
                    if (n32Bits < 0)
                    {
                        pun8Dst--;
                    }
                    else
                    {
                        pun8Dst++;
                    }
                }

                // Increment number of bits read for this read operation
                tNumBits++;

                // decrement our n16Bits value accordingly to keep track
                // of how many we have left to read
                if (n32Bits < 0)
                {
                    n32Bits++;
                }
                else
                {
                    n32Bits--;
                }

                // Decrement the un8DstShiftBits.  If it is zero,
                // reset it
                // then we'd be byte-aligned in the dst
                if (un8DstShiftBits == 0)
                {
                    un8DstShiftBits = 8 - 1;
                }
                else
                {
                    un8DstShiftBits--;
                }
            }
            else
            {
                // Do not read data, but do update the internal
                // bit counter/value
                // Decrement our bits remaining
                sHead.tRemainingReadBits--;

                // Adjust our leftover value
                sHead.un8ReadLeftover &=
                    ((1 << sHead.tRemainingReadBits ) -1);

                // Decrement offset
                tBufferOffsetBits--;
            }

        } while(n32Bits);
    }

    // Return the total number of bits read
    return tNumBits;
}

/*******************************************************************************
*
*   tBufferReadHeadBits
*
*******************************************************************************/
static size_t tBufferReadHeadBits(
    OSALB_BUFFER_STRUCT *psBuffer,
    UN8 *pun8Dst,
    size_t tDstBitOffset,
    N32 n32Bits
        )
{
    OSALB_BLOCK_HEADER_STRUCT *psHead;
    size_t tNumBits = 0;
    size_t tTotalBufferBits, tAlignedBits;

    // If we are attempting a negative seek, abort
    if ( (pun8Dst == NULL) &&
         (n32Bits < 0) )
    {
        return 0;
    }

    // Locally point to the current head block as
    // we know it at this moment.
    psHead = psBuffer->psHead;

    // Check if we have a block at the head of this buffer to read from
    if(psHead != NULL)
    {
        // This is a seek operation.  No need to perform the loop
        // used when we are actually performing a read, which increments
        // a bit at a time.  We can do this in bigger chunks.
        if (pun8Dst == NULL)
        {
            size_t tBufferSizeInBits =
                ((psBuffer->tSize * 8) + psBuffer->un8RemainingBits);
            size_t tNumBitsPassed;
            ptrdiff_t tNumAvailableBits;

            // Cap how many bits can be requested
            if ((UN16)abs(n32Bits) > tBufferSizeInBits)
            {
                n32Bits = tBufferSizeInBits;
            }

            do
            {
                // We always seek from the head of the buffer as long as we
                // are able to do so (i.e. as long as we have data to seek past).

                // Can only seek past bits up to the write pointer
                tAlignedBits = ((psHead->pun8Write - psHead->pun8Read) * 8);
                tNumAvailableBits =
                    psHead->tRemainingReadBits +
                    psHead->tRemainingWriteBits +
                    tAlignedBits;

                if (psHead->tRemainingReadBits >= (size_t)abs(n32Bits))
                {
                    // We have more remaining bits then we need to seek
                    // so eat the remaining bits and don't touch the
                    // byte-aligned data
                    psHead->tRemainingReadBits -= n32Bits;

                    // Adjust our leftover value
                    psHead->un8ReadLeftover &=
                        ((1 << psHead->tRemainingReadBits ) -1);

                    tNumBitsPassed = n32Bits;
                }
                else if (tNumAvailableBits >= n32Bits)
                {
                    size_t tSeekBitsToProcess = n32Bits;

                    // First zero out the remaining read bits
                    tSeekBitsToProcess -= psHead->tRemainingReadBits;
                    psHead->tRemainingReadBits = 0;

                    // Clear the read leftover value
                    psHead->un8ReadLeftover = 0;

                    // Increment the read pointer as
                    // far as possible while staying within
                    // the block bounds
                    if (tAlignedBits >= tSeekBitsToProcess)
                    {
                        // move the read pointer ahead the number of
                        // whole bytes we have
                        psHead->pun8Read += (tSeekBitsToProcess / 8);

                        // if we still have bits left, save that data
                        tSeekBitsToProcess = tSeekBitsToProcess % 8;

                        if (tSeekBitsToProcess > 0)
                        {
                            // Adjust the read leftover
                            psHead->tRemainingReadBits = 8 - tSeekBitsToProcess;
                            psHead->un8ReadLeftover = *psHead->pun8Read++;
                            psHead->un8ReadLeftover &=
                                ((1 << psHead->tRemainingReadBits ) - 1);
                        }
                    }
                    else // We need to dive into the written bits
                    {
                        psHead->pun8Read += (tAlignedBits / 8);
                        tSeekBitsToProcess -= tAlignedBits;

                        // Use the write leftover
                        psHead->tRemainingReadBits = 8 - tSeekBitsToProcess;
                        psHead->un8ReadLeftover = psHead->un8WriteLeftover;
                        psHead->un8ReadLeftover &=
                            ((1 << psHead->tRemainingReadBits ) - 1);
                        psHead->un8WriteLeftover = 0;
                    }

                    tNumBitsPassed = n32Bits;
                }
                else
                {
                    psHead->pun8Read = psHead->pun8Write;

                    psHead->tRemainingReadBits = 0;
                    psHead->un8ReadLeftover = 0;
                    psHead->tRemainingWriteBits = 0;
                    psHead->un8WriteLeftover = 0;

                    tNumBitsPassed = tNumAvailableBits;
                }

                // Increment number of bits we passed so far
                tNumBits += tNumBitsPassed;

                // Decrement number of bits left to seek past
                n32Bits -= tNumBitsPassed;

                // If my current read position is beyond the current block
                // boundaries we need to free this block as well.
                if ( (psHead->pun8Read == psHead->pun8EoB) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psHead = psFreeBlock(psHead);

                    // Check if anything is left
                    if(psHead == NULL)
                    {
                        // No more data to read
                        break;
                    }
                }
            }
            while ( n32Bits );
        }
        else
        {
            UN8 un8BitMask = 0;
            UN8 un8DstBitMask =0;
            UN8 un8DstShiftBits = 0;
            UN8 un8BitIndex = tDstBitOffset;

            // If we have a offset greater than 8 bits, adjust the destination pointer accordingly
            while (tDstBitOffset >= 8)
            {
                // Subtract 8 bits to get it down to a byte-boundary
                tDstBitOffset = tDstBitOffset - 8;

                if (n32Bits < 0)
                {
                    pun8Dst--;
                }
                else
                {
                    pun8Dst++;
                }
            }

            // Figure out our initial dst shift bits
            if (tDstBitOffset > 0)
            {
                un8DstShiftBits = (8 - tDstBitOffset) - 1;
            }
            else
            {
                if ((n32Bits > 8) || ((n32Bits % 8) == 0))
                {
                    // Set to an "even" value
                    un8DstShiftBits = 7;
                }
                else
                {
                    // Set to an "uneven" value
                    un8DstShiftBits = abs(n32Bits % 8) - 1;
                }
            }

            // Read as many bits as I can...
            do
            {
                // We always read from the head of the buffer as long as
                // we are able to do so (i.e. as long as we have bits to read).

                // If my current read position is beyond the current block
                // boundaries we need to move on to the next block. If this
                // occurs we should free the current block we are reading from
                // before moving on.
                if ( (psHead->pun8Read == psHead->pun8EoB) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // Free this block (if we can), and adjust to our next head
                    // block accordingly.
                    psHead = psFreeBlock(psHead);

                    // Check if anything is left
                    if(psHead == NULL)
                    {
                        // No more data to read
                        break;
                    }
                }

                // Check that we are not on the write pointer within this
                // block. If we are, then we need to stop reading
                if ( (psHead->pun8Read == psHead->pun8Write) &&
                     (psHead->tRemainingReadBits == 0) &&
                     (psHead->tRemainingWriteBits == 0))
                {
                    // No more data to read from this block. We are now
                    // at the end (tail) of the buffer
                    break;
                }

                // Read data
                if (psHead->tRemainingReadBits == 0)
                {
                    if (psHead->pun8Read != psHead->pun8Write)
                    {
                        // We don't have any remaining bits, so
                        // move some data to our left over bits
                        psHead->un8ReadLeftover = *psHead->pun8Read++;
                        psHead->tRemainingReadBits = 8;
                    }
                    else
                    {
                        psHead->un8ReadLeftover = psHead->un8WriteLeftover;
                        psHead->tRemainingReadBits = psHead->tRemainingWriteBits;
                        psHead->un8WriteLeftover = 0;
                        psHead->tRemainingWriteBits = 0;
                    }
                }

                // Calculate our bit mask using the shift value
                un8DstBitMask = 1 << un8DstShiftBits;

                // Calculate the bit mask for our source
                un8BitMask = 1 << (psHead->tRemainingReadBits - 1);

                // Set the bit to 1 if the source is 1
                if ( (psHead->un8ReadLeftover & un8BitMask) != 0)
                {
                    *pun8Dst |= un8DstBitMask;
                }
                else
                {
                    *pun8Dst &= (~un8DstBitMask);
                }

                // Decrement our bits remaining
                psHead->tRemainingReadBits--;

                // Adjust our leftover value
                psHead->un8ReadLeftover &=
                    ((1 << psHead->tRemainingReadBits ) -1);

                // Increment the bit index -- if we wrap back
                // to zero we're on a new byte
                un8BitIndex = (un8BitIndex + 1 ) % 8;
                if (un8BitIndex == 0)
                {
                    if (n32Bits < 0)
                    {
                        pun8Dst--;
                    }
                    else
                    {
                        pun8Dst++;
                    }
                }

                // Increment number of bits read for this read operation
                tNumBits++;

                // decrement our n16Bits value accordingly
                if (n32Bits < 0)
                {
                    n32Bits++;
                }
                else
                {
                    n32Bits--;
                }

                // Decrement the un8DstShiftBits.  If it is zero,
                // reset it
                // then we'd be byte-aligned in the dst
                if (un8DstShiftBits == 0)
                {
                    un8DstShiftBits = 8 - 1;
                }
                else
                {
                    un8DstShiftBits--;
                }

            } while(n32Bits);
        }

        // Decrement buffer size by the number read
        tTotalBufferBits = psBuffer->tSize * 8;

        if (tTotalBufferBits < psBuffer->tSize)
        {
            tTotalBufferBits = SIZE_T_MAX;
        }
        else
        {
            tTotalBufferBits += psBuffer->un8RemainingBits;
        }

        tTotalBufferBits -= tNumBits;

        psBuffer->tSize = tTotalBufferBits / 8;

        // TODO: calculate this better
        psBuffer->un8RemainingBits = tTotalBufferBits % 8;
    }

    // Return the total number of bits read
    return tNumBits;
}

/*******************************************************************************
*
*   tBufferReadHead
*
*******************************************************************************/
static size_t tBufferReadHead (
    OSALB_BUFFER_STRUCT *psBuffer,
    UN8 *pun8Dst,
    size_t tSize
        )
{
    size_t tNum = 0;

    // Check to see if we are byte-aligned or not
    if (psBuffer->un8RemainingBits != 0)
    {
        N32 n32SizeInBits = tSize * 8;

        // Look for overflow with a simple check
        if ((UN32)abs(n32SizeInBits) < tSize)
        {
            // We had an overflow, so lets cap it at the max
            // value for size_t
            n32SizeInBits = N32_MAX;
        }

        // We aren't byte-aligned, so pass this off to the read bits function
        tNum = tBufferReadHeadBits(
            psBuffer,
            pun8Dst,
            0, // start from the left bit
            n32SizeInBits
                );

        // Convert tNum from bits to bytes read since the
        // read bits function returns number of bits read.
        tNum = tNum / 8;
    }
    else
    {
        OSALB_BLOCK_HEADER_STRUCT *psHead;
        size_t tAvailableBytes;

        // Locally point to the current head block as
        // we know it at this moment.
        psHead = psBuffer->psHead;

        // Cap how many bytes can be requested
        if (tSize > psBuffer->tSize)
        {
            tSize = psBuffer->tSize;
        }

        // Check if we have a block at the head of this buffer to read from
        if (psHead != NULL)
        {
            if (pun8Dst == NULL)
            {
                // This is a seek operation.  No need to perform the loop
                // used when we are actually performing a read, which increments
                // a byte at a time.  We can do this in bigger chunks.
                do
                {
                    // We always seek from the head of the buffer as long as we
                    // are able to do so (i.e. as long as we have bytes to seek past).

                    // Can only seek past bytes up to the write pointer
                    tAvailableBytes =
                        (size_t)(psHead->pun8Write - psHead->pun8Read);

                    // Adjust size to skip
                    if (tAvailableBytes > tSize)
                    {
                        tAvailableBytes = tSize;
                    }

                    if (tAvailableBytes > 0)
                    {
                        // Adjust block's read point
                        psHead->pun8Read += tAvailableBytes;

                        // Adjust counters
                        tNum += tAvailableBytes;
                        tSize -= tAvailableBytes;
                    }
                    else
                    {
                        break;
                    }

                    // If my current read position is beyond the current block
                    // boundaries we need to free this block as well.
                    if (psHead->pun8Read == psHead->pun8EoB)
                    {
                        // Free this block (if we can), and adjust to our next head
                        // block accordingly.
                        psHead = psFreeBlock(psHead);
                    }
                } while ((tSize > 0) && (psHead != NULL));
            }
            else
            {
                // Read as many bytes as I can...
                do
                {
                    // We always read from the head of the buffer as long
                    // as we are able to do so (i.e. as long as we have
                    // bytes to read).

                    // Can only seek past bytes up to the write pointer
                    tAvailableBytes =
                        (size_t)(psHead->pun8Write - psHead->pun8Read);

                    // Adjust size to skip
                    if (tAvailableBytes > tSize)
                    {
                        tAvailableBytes = tSize;
                    }

                    if (tAvailableBytes > 0)
                    {
                        // Read data
                        OSAL.bMemCpy(pun8Dst, psHead->pun8Read, tAvailableBytes);
                        // Adjust source and destination points
                        psHead->pun8Read += tAvailableBytes;
                        pun8Dst += tAvailableBytes;

                        // Adjust counters
                        tNum += tAvailableBytes;
                        tSize -= tAvailableBytes;
                    }

                    if (psHead->pun8Read == psHead->pun8EoB)
                    {
                        // Free this block (if we can), and adjust to our next head
                        // block accordingly.
                        psHead = psFreeBlock(psHead);
                    }
                    else if (tAvailableBytes == 0)
                    {
                        // It means that we cannot read data from this block since
                        // it's empty, but has not be fully utilized. Thus, just stop
                        // processing the data then.
                        break;
                    }

                } while ((tSize > 0) && (psHead != NULL));
            }

            // Decrement buffer size by the number read
            psBuffer->tSize -= tNum;
        }
    }

    return tNum;
}

/*******************************************************************************
*
*   tBufferWriteTailBits
*
*******************************************************************************/
static size_t tBufferWriteTailBits (
    OSALB_BUFFER_STRUCT *psBuffer,
    const UN8 *pun8Src,
    size_t tSrcBitOffset,
    N32 n32Bits,
    N32 n32Timeout
        )
{
    UN8 un8BitMask =0, un8SrcShiftBits;
    size_t tNumBits = 0;
    size_t tBitsReadFromSrc = 0;
    OSALB_BLOCK_HEADER_STRUCT *psTail;

    // Locally make a copy of the the current tail block as
    // we know it at this moment.
    psTail = psBuffer->psTail;

    // Check if we have a write block to write to, if not go get one
    if(psTail == NULL)
    {
        psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
    }

    if(psTail != NULL)
    {
        // Write as many bits as I can to the tail of the buffer
        do
        {
            // We always begin writing to the tail block provided we are
            // able to do so (i.e. as long as we have room).

            // Check to make sure we have room to write this data, if not
            // attempt to get another block. Also, if the block is marked as
            // 'reserved for write', then we must move on to the next block.
            // This is because I have no idea how much data the one who
            // reserved the block will actually write.
            if( (psTail->pun8Write == psTail->pun8EoB &&
                 psTail->tRemainingWriteBits == 0)
                 ||
                (psTail->bWriteReserved == TRUE ) )
            {
                // Allocate a new block and move current write block to it
                psTail = psAllocateBlock(psBuffer, TRUE, n32Timeout);
                if(psTail == NULL)
                {
                    break;
                }
            }

            if (psTail->tRemainingWriteBits == 8)
            {
                // Push our "leftovers" into the main buffer
                *psTail->pun8Write++ = psTail->un8WriteLeftover;

                // Reset the write bits count and clear the leftovers
                psTail->tRemainingWriteBits = 0;
                psTail->un8WriteLeftover = 0;

                // Move to the next byte in our src if
                // we've read an entire byte
                if (tBitsReadFromSrc == 8)
                {
                    if (n32Bits > 0)
                    {
                        pun8Src++;
                    }
                    else
                    {
                        pun8Src--;
                    }

                    tBitsReadFromSrc = 0;
                }
            }

            // Write data
            // Calculate the bit mask
            un8SrcShiftBits = 7 - (psTail->tRemainingWriteBits + tSrcBitOffset);

            if (un8SrcShiftBits > 7)
            {
                un8SrcShiftBits = 7;
            }

            un8BitMask = 1 << un8SrcShiftBits;

            // Set the bit to 1 if the source is 1
            if ( ( (*pun8Src) & un8BitMask) != 0)
            {
                psTail->un8WriteLeftover |= un8BitMask;
            }

            // Update the number of bits in our "leftover" byte
            psTail->tRemainingWriteBits++;
            tBitsReadFromSrc++;

            // Increment number of bits written for this write operation
            tNumBits++;

            if (n32Bits < 0)
            {
                n32Bits++;
            }
            else
            {
                n32Bits--;
            }

            // Decrement the dest bit offset until we hit it to zero.
            // then we'd be byte-aligned in the dst
            if (tSrcBitOffset > 0)
            {
                tSrcBitOffset--;
            }

        } while(n32Bits);

        // Increment buffer size by the number of bits written
        psBuffer->tSize += (tNumBits / 8);

        // TODO: calculate this better
        psBuffer->un8RemainingBits = tNumBits % 8;
    }

    // Return the total number of bits written
    return tNumBits;
}

#if (OSAL_FILE_SYSTEM == 1)
/*******************************************************************************
*
*   eWriteBlockToFile
*
*******************************************************************************/
static OSAL_BUFFER_ITERATOR_RESULT_ENUM eWriteBlockToFile(
    void *pvData,
    size_t tDataSize,
    OSALB_TO_FILE_ITERATOR_STRUCT *psIterator
        )
{
    size_t tBlocksWritten;

    tBlocksWritten =
        fwrite(pvData, tDataSize, 1, psIterator->psFile);
    if (tBlocksWritten != 1)
    {
        psIterator->bSuccess = FALSE;

        return OSAL_BUFFER_ITERATOR_RESULT_STOP;
    }

    if (psIterator->bConsumeData == FALSE)
    {
        return OSAL_BUFFER_ITERATOR_RESULT_KEEP;
    }
    else
    {
        return OSAL_BUFFER_ITERATOR_RESULT_REMOVE;
    }
}
#endif

/*******************************************************************************
*
*   bEnterExclusiveAccessTaskOnly
*
*   This is a local-helper function which is used to enter an exclusive access
*   section by simply preventing any other task from accessing these same
*   internal structures. Interrupts may still occur since we were told that
*   the caller does not use this object from an ISR.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static BOOLEAN bEnterExclusiveAccessTaskOnly(OSAL_OBJECT_HDL hMutex)
{
    OSAL_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bSuccess = FALSE;

    eReturnCode = OSAL.eSemTake(hMutex, OSAL_OBJ_TIMEOUT_INFINITE);
    if (eReturnCode == OSAL_SUCCESS)
    {
        bSuccess = TRUE;
    }

    return bSuccess;
}

/*******************************************************************************
*
*   vExitExclusiveAccessTaskOnly
*
*   This is a local-helper function which is used to exit an exclusive access
*   section which was entered by the call to bEnterExclusiveAccessTaskOnly().
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static void vExitExclusiveAccessTaskOnly(OSAL_OBJECT_HDL hMutex)
{
    OSAL.eSemGive(hMutex);
    return;
}
/*******************************************************************************
*
*   bEnterExclusiveAccessInterrupt
*
*   This is a local-helper function which is used to enter an exclusive access
*   section by disabling interrupts as these APIs for this object might
*   be used by an ISR as well as between tasks. So this method disables
*   interrupts to protect the shared data. This is much more struct than
*   just blocking tasks as in TaskOnly.
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static BOOLEAN bEnterExclusiveAccessInterrupt(OSAL_OBJECT_HDL hMutex)
{
    OSAL.vEnterCriticalSection();

    // This function always returns TRUE, because it is not possible to destroy
    // OSAL control structure handling Critical Section semaphore before buffers
    return TRUE;
}

/*******************************************************************************
*
*   vExitExclusiveAccessInterrupt
*
*   This is a local-helper function which is used to exit an exclusive access
*   section which was entered by the call to bEnterExclusiveAccessInterrupt().
*
*   Inputs:
*       None.
*
*   Outputs:
*       None.
*
*******************************************************************************/
static void vExitExclusiveAccessInterrupt(OSAL_OBJECT_HDL hMutex)
{
    OSAL.vExitCriticalSection();
    return;
}

#ifdef SUPPORT_CUNIT
#include <osal_buf.cunit>
#endif

#endif /* #if OSAL_BUFFER == 1 for local functions */
