/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  LOCAL HEADER
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _OSAL_CRC_H_
#define _OSAL_CRC_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <standard.h>

#include "osal.h"
#include "osal_core.h"

#include "osal_crc.h"
  /***************/
 /** CONSTANTS **/
/***************/

#define CRC_BUFFER_SIZE (32)

  /**************/
 /** TYPEDEFS **/
/**************/

/* CRC Interface typedefs -- don't change these! */

// Lookup table generation
typedef BOOLEAN (*tGenerateCRCLookupTableFunc) (
	UN8 *pun8LookupTable
		);

// CRC computation initializer
typedef BOOLEAN (*tCRCInitializerFunc)(
	OSAL_CRC_RESULT *ptInitialValue
        );

// CRC computation
typedef OSAL_CRC_RESULT (*tComputeCRCFunc)(
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        );

// Structure used to define CRC interfaces.  This struct
// type is only used for static constant definitions -
// don't change it!  Hey! Did you hear me?!?
typedef struct osal_crc_impl_struct
{
    const tGenerateCRCLookupTableFunc tGenerateTable;
    const tCRCInitializerFunc tInitialize;
    const tComputeCRCFunc tCompute;
    const size_t tLookupTableSize;

} OSAL_CRC_IMPL_STRUCT;

typedef struct osal_buffer_iterator_arg_struct
{
    OSAL_CRC_INFO_STRUCT *psCRC;
    OSAL_CRC_IMPL_STRUCT *psImpl;
    OSAL_CRC_RESULT tResultCRC;
    size_t tByteOffset;
    size_t tNumBytesToProcess;
    size_t tBytesProcessed;
} OSAL_BUFFER_ITERATOR_ARG_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

static OSAL_RETURN_CODE_ENUM eInitCRCObject (
    const char *pacName,
    OSAL_CRC_INFO_STRUCT *psCRC,
    OSAL_CRC_TYPE_ENUM eType
        );

static void vUnInitCRCObject (
    OSAL_CRC_INFO_STRUCT *psCRC
        );

static const char *pacEnumText (
    OSAL_CRC_TYPE_ENUM eCRCType
        );

static OSAL_BUFFER_ITERATOR_RESULT_ENUM eBufferIteratorCallback (
    void *pvData,
    size_t tDataSize,
    void *pvArg
        );

  /*******************************/
 /** CRC Interface definitions **/
/*******************************/

    /****************************************
     * Adding CRCs - Step 1:                *
     * Add value in OSAL_CRC_TYPE_ENUM      *
     * (defined in osal.h) for new CRC type *
     * Define all necessary constants.      *
     * Define all required prototypes:      *
     *  Lookup Table Generator              *
     *  CRC computation initializer         *
     *  CRC computation engine              *
     * Expand functionality in pacEnumText  *
     * to include the newly defined CRC.    *
     ****************************************/

/*
 * CCITT-CRC16:
 *  As referenced in RX000135-D Appendix A
 */

// Types
typedef UN16 CCITT_CRC16;

// Constants
#define CCITT_CRC16_NAME "CCITT-CRC16"
#define CCITT_CRC16_INITIAL_VALUE (0x1d0f)
#define CCITT_CRC16_COMPUTE_VALUE (0x1021)
#define CCITT_CRC16_COMPUTE_MASK (0x8000)

#define CCITT_CRC16_NUM_LOOKUP_ENTRIES (256)
#define CCITT_CRC16_ENTRY_SIZE (2)
#define CCITT_CRC16_LOOKUP_TABLE_SIZE \
    (CCITT_CRC16_NUM_LOOKUP_ENTRIES * CCITT_CRC16_ENTRY_SIZE)

// Function Prototypes

static BOOLEAN bGenerateCCITT_CRC16LookupTable (
    UN8 *pun8LookupTable
        );

static BOOLEAN bInitializeCCITT_CRC16 (
    OSAL_CRC_RESULT *ptInitialValue
        );

static OSAL_CRC_RESULT tComputeCCITT_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        );

/*
 * NAVTEQ-CRC16:
 *  As referenced in TDFSpecs106.pdf, Appendix A
 */

// Types
typedef N16 NAVTEQ_CRC16;

// Constants
// Note -- the NAVTEQ-CRC16 CRC uses the same look-up table
// as the CCITT-CRC16.  The only difference is the calculation
// function and the initiliziation function
#define NAVTEQ_CRC16_NAME "NAVTEQ-CRC16"
#define NAVTEQ_CRC16_INITIAL_VALUE (0)

// Function Prototypes

static BOOLEAN bInitializeNAVTEQ_CRC16(
    OSAL_CRC_RESULT *ptInitialValue
        );

static OSAL_CRC_RESULT tComputeNAVTEQ_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        );

/*
 * XM AppID Packet CRC16:
 *  As referenced in  Appendix A of
 *  AN-10_ Parsing_Application_Data_r1.7_2007-10-20.pdf
 */

// Types
typedef U16 XMAPPID_CRC16;

// Constants
// Note -- the XMAPPID_CRC16 CRC uses the same look-up table
// as the CCITT-CRC16.  The only difference is the calculation
// function and the initiliziation function
#define XMAPPID_CRC16_NAME              "XMAPPID_CRC16"
#define XMAPPID_CRC16_INITIAL_VALUE     (0xFFFF)
#define XMAPPID_CRC16_WIDTH             (16)
#define XMAPPID_CRC16_FINAL_XOR_VALUE   (0xFFFF)

// Function Prototypes

static BOOLEAN bInitializeXMAPPID_CRC16(
    OSAL_CRC_RESULT *ptInitialValue
        );

static OSAL_CRC_RESULT tComputeXMAPPID_CRC16 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        );

/*
 * ISO 3309 CRC32:
 *  As referenced in Appendix D of
 *  http://www.w3.org/TR/PNG/
 */

// Types
typedef UN32 ISO3309_CRC32;

// Constants
#define ISO3309_CRC32_NAME              "ISO3309_CRC32"
#define ISO3309_CRC32_INITIAL_VALUE     (0xFFFFFFFFL)
#define ISO3309_CRC32_COMPUTE_VALUE (0xedb88320L)
#define ISO3309_CRC32_COMPUTE_MASK (0x1)
#define ISO3309_CRC32_NUM_LOOKUP_ENTRIES (256)
#define ISO3309_CRC32_ENTRY_SIZE (4)
#define ISO3309_CRC32_LOOKUP_TABLE_SIZE \
    (ISO3309_CRC32_NUM_LOOKUP_ENTRIES * ISO3309_CRC32_ENTRY_SIZE)

#if OSAL_CRC == 1
// Function Prototypes
static BOOLEAN bGenerateISO3309_CRC32LookupTable (
    UN8 *pun8LookupTable
        );

static BOOLEAN bInitializeISO3309_CRC32 (
    OSAL_CRC_RESULT *ptInitialValue
        );

static OSAL_CRC_RESULT tComputeISO3309_CRC32 (
    OSAL_CRC_RESULT tCurrentCRC,
    UN8 *pun8Buffer,
    size_t tBufferSize,
    UN8 *pun8LookupTable
        );

  /***************/
 /** VARIABLES **/
/***************/

    /****************************************
     * Adding CRCs - Step 2:                *
     * Plug functions defined above into    *
     * gsCRCs below IN THE ORDER WHICH CRCs *
     * APPREAR IN OSAL_CRC_TYPE_ENUM        *
     * (defined in osal.h)                  *
     ****************************************/

// CRC implementations are added here...
static const OSAL_CRC_IMPL_STRUCT gsCRCs[OSAL_CRC_TYPE_MAX] =
{
    // CCITT-CRC16 = 0
    {
     /*.tGenerateTableFunc = */bGenerateCCITT_CRC16LookupTable,
     /*.tInitializeFunc = */bInitializeCCITT_CRC16,
     /*.tComputeCRCFunc = */tComputeCCITT_CRC16,
     /*.tLookupTableSize = */CCITT_CRC16_LOOKUP_TABLE_SIZE
    },

    // NAVTEQ-CRC16 = 1
    // Note this CRC uses the same table as the CCITT-CRC16
    // implementation.
    {
     /*.tGenerateTableFunc = */bGenerateCCITT_CRC16LookupTable,
     /*.tInitializeCRCFunc = */bInitializeNAVTEQ_CRC16,
     /*.tComputeFunc = */tComputeNAVTEQ_CRC16,
     /*.tLookupTableSize = */CCITT_CRC16_LOOKUP_TABLE_SIZE
    },

    // XMAPPID-CRC16 = 2
    // Note this CRC uses the same table as the CCITT-CRC16
    // implementation.
    {
     /*.tGenerateTableFunc = */bGenerateCCITT_CRC16LookupTable,
     /*.tInitializeFunc = */bInitializeXMAPPID_CRC16,
     /*.tComputeCRCFunc = */tComputeXMAPPID_CRC16,
     /*.tLookupTableSize = */CCITT_CRC16_LOOKUP_TABLE_SIZE
    },

    // ISO3309_CRC32 = 3
    {
     /*.tGenerateTableFunc = */bGenerateISO3309_CRC32LookupTable,
     /*.tInitializeFunc = */bInitializeISO3309_CRC32,
     /*.tComputeCRCFunc = */tComputeISO3309_CRC32,
     /*.tLookupTableSize = */ISO3309_CRC32_LOOKUP_TABLE_SIZE
    },
};
#endif

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _OSAL_H_
