/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
*
*
*
*
*
*	LOCAL HEADER
*******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _OSAL_BUF_H_
#define _OSAL_BUF_H_

  /**************/
 /** INCLUDES **/
/**************/
#include <stdlib.h>
#include "standard.h"

#include "osal_buf.h"

  /***************/
 /** CONSTANTS **/
/***************/

// This definition triggers OSAL Buffers checks which cause *crash* in case
// of error cases
#ifndef OSAL_BUFFER_OBJECT_STATE_CHECK
#define OSAL_BUFFER_OBJECT_STATE_CHECK (0)
#endif

  /**************/
 /** TYPEDEFS **/
/**************/

typedef struct osalb_buffer_struct
{
    // Block Pool to extract blocks from...
    OSAL_BLOCK_POOL_PRIVATE_INFO_STRUCT *psBlockPoolInfo;

    // Buffer info
    UN32 un32Options; // Options associated with this buffer
    size_t tSize; // Size of entire buffer
    UN8 un8RemainingBits; // Any remaining bits. A value between 1 and 7

    // A pointer to the head block in this buffer
    struct osalb_block_header_struct *psHead;

    // A pointer to the tail block in this buffer
    struct osalb_block_header_struct *psTail;

    // Flag to indicate if byte writes are blocking
    BOOLEAN bBlockingWrite;

    // A handle to an optional semaphore used if reads from
    // the buffer are blocking.
    OSAL_OBJECT_HDL hReadSem;
    OSAL_OBJECT_HDL hWriteSem;

    // Allocation Statistics
    UN16 un16Allocated;
    UN16 un16MaxAllocated;

#if ((OSAL_DEBUG==1) && (OSAL_BUFFER_OBJECT_STATE_CHECK == 1))
    void *pvAllocationCheck;
#endif

} OSALB_BUFFER_STRUCT;

typedef struct osalb_block_header_struct
{
    // Buffer this block belongs to
    OSALB_BUFFER_STRUCT *psBuffer;

    // Block organization/position info
    struct osalb_block_header_struct *psNext;
    struct osalb_block_header_struct *psPrev;

    // This block's info
    UN8 *pun8SoB; // Start address of this block
    UN8 *pun8EoB; // End address of this block

    UN8 *pun8Read; // Marker where the next read in a block will occur
    UN8 *pun8Write; // Marker where the next write in a block will occur
    BOOLEAN bWriteReserved; // Flag to indicate everything from the write
                            // marker to the end of the block is reserved
                            // for block-wise writing.
    BOOLEAN bReadReserved; // Flag to indicate this block is reserved
                           // for reading.

    size_t tRemainingReadBits;  // The number of bits left in un8ReadLeftover
    size_t tRemainingWriteBits;  // The number of bits left in un8WriteLeftover

    UN8 un8ReadLeftover; // Any leftover bits from the last read byte
                         // Mom says we have to eat our leftover since there
                         // are starving kids in Africa.

    UN8 un8WriteLeftover; // Any leftover bits from the last write
                         // We tell mom we'll save our leftovers and she can
                         // FedEx them to Africa if she is so worried.
} OSALB_BLOCK_HEADER_STRUCT;

typedef struct osalb_to_file_iterator_struct
{
    FILE *psFile;
    BOOLEAN bSuccess;
    BOOLEAN bConsumeData;
} OSALB_TO_FILE_ITERATOR_STRUCT;

  /***************/
 /** VARIABLES **/
/***************/

  /************/
 /** MACROS **/
/************/

// This macro provides expanding value which is read as signed, but
// may have not enough bits to fill in all bits in the data type
#define OSAL_BUFFER_EXPAND_TO_SIGNED_VALUE(_pBuf, _DataType, _tNum, _tBits) \
    if ((1 < _tBits) && (_tBits < _DataType##_BITLEN) && (_tNum == _tBits)) \
    {                                                                       \
        /* Check the major significant bit of the value just read      */   \
        const _DataType tMask = ((_DataType)1 << ((tBits) - 1));            \
        if ((*(_pBuf) & tMask) == tMask)                                    \
        {                                                                   \
            /* If the bit is set, the number is negative. To keep it   */   \
            /* negative we need to set all '1' up to the requested type*/   \
            /* size.                                                   */   \
            *(_pBuf) |= ~(tMask - 1);                                       \
        }                                                                   \
    }

#if ((OSAL_DEBUG == 1) && (OSAL_BUFFER_OBJECT_STATE_CHECK == 1))
#define OSAL_BUFFER_ALLOCATED_ASSERT(_pBuf) \
    if ((_pBuf)->pvAllocationCheck != (_pBuf))                              \
    {                                                                       \
        printf("OSAL Buffer error: %p is released\n", (_pBuf));             \
        *((int*)0) = 0; /* CRASH */                                         \
    }

#define OSAL_BUFFER_RELEASED_ASSERT(_pBuf) \
    if ((_pBuf)->pvAllocationCheck == (_pBuf))                              \
    {                                                                       \
        printf("OSAL Buffer error: %p is allocated\n", (_pBuf));            \
        *((int*)0) = 0; /* CRASH */                                         \
    }
#else
#define OSAL_BUFFER_ALLOCATED_ASSERT(_pBuf)
#define OSAL_BUFFER_RELEASED_ASSERT(_pBuf)
#endif

  /****************/
 /** PROTOTYPES **/
/****************/

#if (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1)
static BOOLEAN bPrintBlockPool (
    void *pvData,
    void *pvArg
        );
#endif /* (OSAL_DEBUG == 1) && (OSAL_OBJECT_TRACKING == 1) */

static OSALB_BLOCK_HEADER_STRUCT *psAllocateBlock (
    OSALB_BUFFER_STRUCT *psBuffer,
    BOOLEAN bTail,
    N32 n32Timeout
        );

static OSALB_BLOCK_HEADER_STRUCT *psFreeBlock (
    OSALB_BLOCK_HEADER_STRUCT *psFreeBlock
        );

static size_t tBufferPeekBits (
    OSALB_BUFFER_STRUCT const *psBuffer,
    UN8 *pun8Dst,
    size_t tDstBitOffset,
    N32 n32Bits,
    size_t tBufferOffsetBits
        );

static size_t tBufferReadHeadBits(
    OSALB_BUFFER_STRUCT *psBuffer,
    UN8 *pun8Dst,
    size_t tDstBitOffset,
    N32 n32Bits
        );

static size_t tBufferWriteTailBits (
    OSALB_BUFFER_STRUCT *psBuffer,
    const UN8 *pun8Src,
    size_t tSrcBitOffset,
    N32 n32Bits,
    N32 n32Timeout
        );

static size_t tBufferReadHead (
    OSALB_BUFFER_STRUCT *psBuffer,
    UN8 *pun8Dst,
    size_t tSize
        );

static OSAL_BUFFER_ITERATOR_RESULT_ENUM eWriteBlockToFile(
    void *pvData,
    size_t tDataSize,
    OSALB_TO_FILE_ITERATOR_STRUCT *psIterator
        );

static BOOLEAN bEnterExclusiveAccessTaskOnly (
    OSAL_OBJECT_HDL hMutex
        );

static void vExitExclusiveAccessTaskOnly (
    OSAL_OBJECT_HDL hMutex
        );

static BOOLEAN bEnterExclusiveAccessInterrupt (
    OSAL_OBJECT_HDL hMutex
        );

static void vExitExclusiveAccessInterrupt (
    OSAL_OBJECT_HDL hMutex
        );

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif	// _OSAL_BUF_H_
