/******************************************************************************/
/*                    Copyright (c) Sirius-XM Satellite Radio Inc.            */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius-XM Satellite Radio Inc.    */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This is a generic SRH header file for all SRH devices.
 *  This file need not, and should not be modified to accomodate specific
 *  SRH devices.
 *
 ******************************************************************************/

#ifndef SRH_H_
#define SRH_H_

#ifdef __cplusplus
extern "C" {
#endif


#include "standard.h"
#include "osal.h"
#include "devices.h"
#include "sms_api.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

/* SRH 'device' ioctl id's */
#define SRH_IOCTL_ANTENNA_POWER_ENABLE       (DEV_IOCTL_SRH + 0)
#define SRH_IOCTL_ANTENNA_POWER_DISABLE      (DEV_IOCTL_SRH + 1)
#define SRH_IOCTL_POWER_ON                   (DEV_IOCTL_SRH + 2)
#define SRH_IOCTL_POWER_OFF                  (DEV_IOCTL_SRH + 3)
#define SRH_IOCTL_RESET_ENABLE               (DEV_IOCTL_SRH + 4)
#define SRH_IOCTL_RESET_DISABLE              (DEV_IOCTL_SRH + 5)
#define SRH_IOCTL_RESET                      (DEV_IOCTL_SRH + 6)
#define SRH_IOCTL_GET_NUM_ANTENNAS           (DEV_IOCTL_SRH + 7)
#define SRH_IOCTL_MODULE_EXISTS              (DEV_IOCTL_SRH + 8)
#define SRH_IOCTL_PROTOCOL_IDENTIFIER        (DEV_IOCTL_SRH + 9)
#define SRH_IOCTL_DECODER_EXISTS             (DEV_IOCTL_SRH + 10)
#define SRH_IOCTL_DECODER_CAPABILITIES       (DEV_IOCTL_SRH + 11)
#define SRH_IOCTL_DESCRIBE_DECODERS          (DEV_IOCTL_SRH + 12)
#define SRH_IOCTL_DESCRIBE_MODULES           (DEV_IOCTL_SRH + 13)
#define SRH_IOCTL_DESCRIBE_SRMS              (DEV_IOCTL_SRH + 14)
#define SRH_IOCTL_SRM_EXISTS                 (DEV_IOCTL_SRH + 15)
#define SRH_IOCTL_WRITE_DRIVER_CONFIGURATION (DEV_IOCTL_SRH + 16)
#define SRH_IOCTL_DESCRIBE_PORT              (DEV_IOCTL_SRH + 17)
#define SRH_IOCTL_DEVICE_GROUP               (DEV_IOCTL_SRH + 18)
#define SRH_IOCTL_MODULE_CONFIGURATION       (DEV_IOCTL_SRH + 19)
#define SRH_IOCTL_GET_APP_RESET_CONTROL      (DEV_IOCTL_SRH + 20)
#define SRH_IOCTL_APP_RESET                  (DEV_IOCTL_SRH + 21)

// Antenna definitions
#define SRH_SRM_MAX_NUM_ANTENNAS    (2) // Single or Diversity only

// Device Query Definitions
//**************************************************
// defines used for Device Capabilities
//**************************************************
#define SRH_DEVICE_CAPABILITY_NONE        (0x00000000)
#define SRH_DEVICE_CAPABILITY_AUDIO       (0x00000001)
#define SRH_DEVICE_CAPABILITY_VIDEO       (0x00000002)
#define SRH_DEVICE_CAPABILITY_DATA        (0x00000004)
#define SRH_DEVICE_CAPABILITY_RESERVED    (0xFFFFFFF8)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Module Id
typedef N16 MODULE_ID; // use N16 so we have an invalid number value as well
/* Module ID definition */
#define MODULE_ID_INVALID  ((MODULE_ID)(-1))

// Device GROUP & Mask
typedef UN8 DEVICE_GROUP;
#define DEVICE_GROUP_MIN ((DEVICE_GROUP)1)
#define DEVICE_GROUP_MAX ((DEVICE_GROUP)32)
#define DEVICE_GROUP_NOT_PROVISIONED ((DEVICE_GROUP)DEVICE_GROUP_MAX + 1)

typedef UN32 DEVICE_MASK;
#define DEVICE_MASK_EMPTY ((DEVICE_MASK)0)

/*
 * From SX-9845-0166, Section 7.6.2 "The Device Mask Field":
 *
 * Device Groups are assigned a Device Mask bit position
 * (0 to 31). Radios process the message if the bit position
 * is set (1) for the bit position assigned to the Device Group
 * to which the Radio is assigned. Otherwise, if this bit is
 * cleared (0) these Radios do Not process the message.
 *
 * And Further:
 *
 * Device Mask Bit Position = Device Group Number  1
 * Bit Position 0 is the least significant bit of the Device Mask field
 * and Bit Position 31 is the most significant bit.
 *
 * */
#define DEVICE_GROUP_TO_MASK(group) \
    (0x1 << ((group) - 1))

/*
    General Purpose I/O Interface Prototype
*/
typedef void (*SRH_GPIO)(void);
#define SRH_NULL_GPIO ((SRH_GPIO)NULL)

/*
    General Purpose IO Functions
    One interface to enable, another to disable
    pvReset interface is used if application desired to control GPIO lines
    manually. Driver should perform full reset sequence (enable-timeout-disable)
    when this method is called. Used for RESET line.
*/
typedef struct srh_device_gpio_intf_struct
{
    SRH_GPIO pvEnable;
    SRH_GPIO pvDisable;
    SRH_GPIO pvReset;

} SRH_DEVICE_GPIO_INTF_STRUCT;

/*
    SSP Protocol Identifier
*/
typedef enum protocol_identifier_enum
{
    PROTOCOL_IDENTIFIER_SSP = 0x03,
    PROTOCOL_IDENTIFIER_SSP_V2 = 0x06,
    PROTOCOL_IDENTIFIER_SXI

} PROTOCOL_IDENTIFIER_ENUM;

/*
    SXi Module Configuration
 */
// Please refer to the document SX-9845-0097 SXi Messages
// for complete definitions of these parameters.
typedef struct sxi_module_cfg_struct
{
    // Fade
    UN8 un8Fade;

    // CategoryLabelLength
    UN8 un8CategoryLabelLength;

    // ChannelLabelLength
    UN8 un8ChannelLabelLength;

    // MetadataLabelLength
    UN8 un8MetadataLabelLength;

    // MaxPendingIndications
    UN8 un8MaxPendingIndications;

    // ConfirmationWaitTime
    UN8 un8ConfirmationWaitTime;

    // IRControl
    UN8 un8IRControl;

    // IRDeleteOnTune
    UN8 un8IRDeleteOnTune;

    // IRMarkNewTrack
    UN8 un8IRMarkNewTrack;

    // ExtendedControl
    UN8 un8ExtendedControl;

    // PrioritySmartFavCnt
    UN8 un8PrioritySmartFavCnt;

} SXI_MODULE_CFG_STRUCT;

/*
    Device Port Structure
    Always contains a device name(fname) and mode
*/
typedef struct srh_device_port_struct
{
    // BSP Device Name (i.e. "com1:", "com2:", etc.
    const char *pcDeviceName;

    // Open mode (i.e. "115200,N,8,1")
    const char *pcMode;

} SRH_DEVICE_PORT_STRUCT;

/*
    SDTP Port Structure
    Add a port number to the basic
    port structure
*/
typedef struct srh_device_sdtp_port_struct
{
    SRH_DEVICE_PORT_STRUCT sPort;
    UN8 un8PortId;

} SRH_DEVICE_SDTP_PORT_STRUCT;

/*
    Port Description Type (parity) used
    for reporting via ioctl()
*/
typedef enum srh_device_port_parity_enum
{
    SRH_DEVICE_PORT_PARITY_EVEN,
    SRH_DEVICE_PORT_PARITY_ODD,
    SRH_DEVICE_PORT_PARITY_NONE,
    SRH_DEVICE_PORT_PARITY_UNKNOWN

} SRH_DEVICE_PORT_PARITY_ENUM;

/*
    Port Descriptor Tuple
*/
typedef struct srh_device_port_tuple_struct
{
    UN32 un32BaudRate;
    UN8  un8DataBits;
    SRH_DEVICE_PORT_PARITY_ENUM eParity;
    UN8  un8StopBits;
    UN8 un8PortId;

} SRH_DEVICE_PORT_TUPLE_STRUCT;

// Type to describe device capabilities
typedef UN32 SRH_DEVICE_CAPABILITIES_MASK;

typedef struct srh_device_decoder_map_struct
{
    // Decoder Name
    const char *pacDecoderName;

    // Decoder Id
    DECODER_ID tDecoderId;

    // Decoder Type
    SRH_DEVICE_CAPABILITIES_MASK tDecoderCapabilities;

    // SDTP Port
    SRH_DEVICE_SDTP_PORT_STRUCT *psSDTP;


} SRH_DEVICE_DECODER_MAP_STRUCT;

typedef struct srh_device_module_map_struct
{
    // Module Id
    MODULE_ID tModuleId;

    // bPrimary TRUE if Primary module, FALSE if secondary module
    // secondary modules only present in multi-module chipsets
    BOOLEAN   bPrimary;

    // SXi Module Configuration
    SXI_MODULE_CFG_STRUCT sSXiConfig;

    // Decoders
    UN32 un32NumDecoders;
    SRH_DEVICE_DECODER_MAP_STRUCT *pasDecoders;

} SRH_DEVICE_MODULE_MAP_STRUCT;

typedef struct srh_device_srm_map_struct
{
    // SRM Name
    const char *pacSRMName;

    // Boolean flag which specifies whenever RESET procedure is
    // carried over by application itself
    BOOLEAN bAppResetControl;
    /* SRM GPIO Interfaces */
    SRH_DEVICE_GPIO_INTF_STRUCT sPower;
    SRH_DEVICE_GPIO_INTF_STRUCT sAntenna;
    SRH_DEVICE_GPIO_INTF_STRUCT sReset;

    /* SRM Physical Description */

    /*
        Module Description.

        The following elements describe the SRM's internal module
        configuration. It lists the number of modules supported
        and each of the module ids. The SRM device driver registration
        must provide a pointer to an array of module ids supported.
        This pointer must point to a constant array.
    */
    UN8 un8NumberOfModules;
    SRH_DEVICE_MODULE_MAP_STRUCT *pasModuleMap;

    /* SRM Physical Antenna Support */
    UN8 un8NumberOfAntennas;

    /*
        Protocol Configuration

        There is always one and only one Protocol port per SRM
    */
    PROTOCOL_IDENTIFIER_ENUM eProtocolId;
    SRH_DEVICE_PORT_STRUCT sPort;

} SRH_DEVICE_SRM_MAP_STRUCT;

typedef struct srh_device_struct
{
    /* SRH Device Struct */

    // The collection of SRMs
    UN8 un8NumberOfSRMs;
    SRH_DEVICE_SRM_MAP_STRUCT *pasSRMMap;

    // This device's provisioned device group
    DEVICE_GROUP tDeviceGroup;

} SRH_DEVICE_STRUCT;

typedef struct srh_device_decoder_tuple_struct
{
	// Decoder Name
    const char *pacDecoderName;

    // Decoder Id
    DECODER_ID tDecoderPathId;

	// Decoder Type
	SRH_DEVICE_CAPABILITIES_MASK tDecoderCapabilities;

} SRH_DEVICE_DECODER_TUPLE_STRUCT;

typedef struct srh_device_module_tuple_struct
{
    // Module Id
    MODULE_ID tModuleId;

    // bPrimary TRUE if Primary module, FALSE if secondary module
    // this flag is used to help solve power sequencing issues
    // when utilizing dual module G3 devices.
    BOOLEAN   bPrimary;

    // Module's Supported Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tModuleCapabilities;

} SRH_DEVICE_MODULE_TUPLE_STRUCT;

typedef struct srh_device_srm_tuple_struct
{
    // SRM Name
    const char *pacSRMName;

    // SRM's Supported Capabilities
    SRH_DEVICE_CAPABILITIES_MASK tSRMCapabilities;

} SRH_DEVICE_SRM_TUPLE_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

/* An external constant declaration of the SRM interface */
extern const OSAL_DEV_INTERFACE_STRUCT GsSRHInterface;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

void SRH_vRegisterDriver( char *pacUARTConfig );
void SRH_vUnRegisterDriver( void );


#ifdef __cplusplus
}
#endif

#endif /* SRH_H */
