/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 *  This module contains the Serial-I/O device driver implementation. This
 *  driver is a very simple "shim" driver used to provide separate connections
 *  for communication and I/O (modem signals). Typically a serial port driver
 *  only allows one connection (or node). So if one part of the software needs
 *  to control the modem signals (for a completely independant reason) and
 *  another part to use the tx/rx comm, this driver can make that happen.
 *
 *  The device being shim'd can be opened by using the normal mode supplied
 *  by the native driver (e.g. "57600,N,8,1") but if I/O control is needed
 *  this is indicated by opening with a mode of "io".
 *
 ******************************************************************************/

#include <string.h>

#include "standard.h"
#include "osal.h"

// Device Driver Framework Common Serial Communications Port Header
#include "com.h"

// Serial-I/O Shim Driver Includes
#include "_sio_shim.h"
#include "sio_shim.h"

/*****************************************************************************
*
*       _bInit
*
*   Performs any device initialization for the physical device this driver
*   is written for. This method is called only when the first driver instance
*   is opened.
*
*   Inputs:
*       pcDevName - A pointer to an ASCII null terminated string which
*           identifies this device. A copy of the provided name is retained
*           in the device node and this is a pointer to that name.
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       ppvPublicData - A pointer to a pointer which represents any
*           persistent object which contains device-global or public
*           data required by all driver instances.
*
*   Outputs:
*       TRUE on success. Otherwise FALSE should be returned on error.
*
*****************************************************************************/
static BOOLEAN _bInit (
                        const char *pcDevName,
                        void const *pvProperties,
                        void **ppvPublicData
                        )
{
    SIO_SHIM_PUBLIC_STRUCT *psPublicData;

	// Must provide a valid 'properties' pointer. For this driver the
    // 'properties' provided is the serial port device name this
    // driver is going to 'shim' on top of.
	if(pvProperties == NULL)
	{
        return FALSE;
	}

    // Allocate a sturcture to hold the public control structure
    // required by the driver.
    // Set the public data pointer for the Framework to maintain
    psPublicData =
    *ppvPublicData =
        OSAL.pvMemoryAllocate(
            pcDevName,
            sizeof(SIO_SHIM_PUBLIC_STRUCT),
            TRUE );
    if(psPublicData == NULL)
    {
        // Error!
        printf("SIO_SHIM Error! Cannot allocate memory.\n");
        return FALSE;
    }

    // Initialize to indicate we are not shimming for a comm session.
    psPublicData->bCommAssigned = FALSE;

    // Open device we are to 'shim' - defaulted to some known
    // working baud rate.
    psPublicData->psFile = fopen(
        (const char*)pvProperties, SIO_SHIM_DEFAULT_MODE);
    if(psPublicData->psFile == NULL)
    {
        printf("SIO_SHIM Error! Cannot open device '%s' with mode '%s'.\n",
            (const char*)pvProperties, SIO_SHIM_DEFAULT_MODE);
        return FALSE;
    }

    return TRUE;
}

/*****************************************************************************
*
*       _vExit
*
*   Performs any device un-initialization or clean-up procedures required
*   when the device is no longer needed. This method is called only when
*   all open connections to the driver have been closed.
*
*   Inputs:
*       pvProperties - A pointer to the device properties structure provided
*           when the device node was registered (typically by the BSP).
*       pvPublicData - - A pointer to a persistent object which contains
*           device-global or public data required by all driver instances.
*
*   Outputs:
*       None.
*
*****************************************************************************/
static void _vExit (
                        void const *pvProperties,
                        void *pvPublicData
                        )
{
    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)pvPublicData;

    // Close the shim'd device
    if(psPublicData->psFile != NULL)
    {
        fclose(psPublicData->psFile);
    }

    // Now, just free any public data allocated by _bInit()
    OSAL.vMemoryFree(pvPublicData);

    return;
}

/*****************************************************************************
*
*       _n32Open
*
*   Opens a connection (instance) to the specified device by allocating any
*   resources or state information required for this instance.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver.
*       pvArg - A pointer to some device driver specific data which represents
*           a mode or configuration this instance is to be open with.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Open (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        const void *pvArg
                        )
{
    N32 n32RetVal = DEV_ERROR;

    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;
    const char *pacMode = (const char *)pvArg;

    // Check input
    if(pvArg == NULL)
    {
        // Error!
        return DEV_ERROR;
    }

    // When we open this driver, we are doing so either for comm or signals.
    // Only one comm instance is available at a time, but we can have many signals
    // instances open.

    // Check if we are opening for a "modem signals" or I/O node
    if(strcmp(pacMode, "io") == 0)
    {
        // Opening for signals...

        // We just need to know if this is for signals(io) or not.
        // Since we just need one value there is nothing to allocate
        psDeviceInfo->pvPrivateData = (void *)(size_t)TRUE; // Signals
        n32RetVal = DEV_OK;
    }
    // If not already opened for comm...
    else if(psPublicData->bCommAssigned == FALSE)
    {
        // Opening for comm...emulate native open

        // We just need to know if this is for signals or not.
        // Since we just need one value there is nothing to allocate
        psDeviceInfo->pvPrivateData = (void *)FALSE; // Comm

        // Modify serial port params(mode) according to what is provided
        // which models what the original "open" would do.
        n32RetVal = ioctl(psPublicData->psFile,
            COM_IOCTL_CHANGE_SETTINGS, (const char*)pvArg);
        if(n32RetVal == DEV_ERROR)
        {
            // Error!
            printf("SIO_SHIM Error! Cannot change settings to '%s'.\n",
                (const char*)pvArg);
        }
        else
        {
            // Ok, we're open for business...
            psPublicData->bCommAssigned = TRUE;
        }
    }

    return n32RetVal;
}

/*****************************************************************************
*
*       _n32Close
*
*   Closes a connection (instance) to the specified device and releases any
*   resources obtained when the device was opened.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members
*           can be accessed to retrieve information about the device itself
*           and the driver instance. See Section 4.10.2 for more detail on
*           this structures members.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is returned
*       on error. See section 4.11 for possible error codes. If anything other
*       than DEV_OK is returned the instance handle will not be released and
*       will remain open.
*
*****************************************************************************/
static N32 _n32Close (
                        OSAL_DEV_INFO_STRUCT *psDeviceInfo
                        )
{
    N32 n32RetVal = DEV_OK;

    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    // Extract private data from the Device Driver Framework
    // so that we can easily determine what type of node this is.
    BOOLEAN bSignals = (BOOLEAN)(size_t)psDeviceInfo->pvPrivateData;

    // Check what type is being closed, if it is not a 'signals'
    // or io node then just return ok.
    if(bSignals == FALSE)
    {
        // Just mark as unassigned now, no need to do anything else
        psPublicData->bCommAssigned = FALSE;
    }

    return n32RetVal;
}

/*****************************************************************************
*
*       _tRead
*
*   Allows byte wide (stream oriented) reads from the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcDst - The destination pointer where device data is to be read.
*       tSize - The size of each object to read.
*       tNumObjs - The number of objects to read.
*
*   Outputs:
*       The number of objects successfully read from the device and placed
*       in pcDst. If an error occurs or no data was read a zero is returned.
*       It should not be assumed that every read request must always be carried
*       out to completion in a single call. The caller is required to check the
*       return value to determine if all the requested data was read or if a
*       partial read was performed. If a partial read occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tRead (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        char *pcDst,
                        size_t tSize,
                        size_t tNumObjs
                        )
{
    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    // Extract private data from the Device Driver Framework
    BOOLEAN bSignals = (BOOLEAN)(size_t)psDeviceInfo->pvPrivateData;

    // Only allowed for a non-signals handles(nodes)
    if(bSignals == FALSE)
    {
        // Pass thru to native driver
        tNumObjs = fread(pcDst, tSize, tNumObjs, psPublicData->psFile);
    }
    else
    {
        // Do nothing
        tNumObjs = 0;
    }

    return tNumObjs;
}

/*****************************************************************************
*
*       _tWrite
*
*   Allows byte wide (stream oriented) writes to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       pcSrc - The source pointer where device data is to be written from.
*       tSize - The size of each object to write.
*       tNumObjs - The number of objects to write.
*
*   Outputs:
*       The number of objects successfully written to the device from pcSrc.
*       If an error occurs or no data was written a zero is returned. It should
*       not be assumed that every write request must always be carried out to
*       completion in a single call. The caller is required to check the return
*       value to determine if all the requested data was written or if a
*       partial write was performed. If a partial write occurs the caller may
*       wish to call the API again to complete the operation. If additional
*       error information is desired to be communicated to the application,
*       this should be implemented as some type of ioctl command defined by
*       the driver and supported by the _n32Ioctl() implementation.
*
*****************************************************************************/
static size_t _tWrite (
                         const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                         const char *pcSrc,
                         size_t tSize,
                         size_t tNumObjs
                         )
{
    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    // Extract private data from the Device Driver Framework
    BOOLEAN bSignals = (BOOLEAN)(size_t)psDeviceInfo->pvPrivateData;

    // Only allowed for a non-signals handles(nodes)
    if(bSignals == FALSE)
    {
        // Pass thru to native driver
        tNumObjs = fwrite(pcSrc, tSize, tNumObjs, psPublicData->psFile);
    }
    else
    {
        // Do nothing
        tNumObjs = 0;
    }

    return tNumObjs;
}

/*****************************************************************************
*
*       _n32Ioctl
*
*   Allows driver specific command and control to the device.
*
*   Inputs:
*       psDeviceInfo - A pointer to the device info pertaining to this
*           instance of the driver. Using this pointer various members can
*           be accessed to retrieve information about the device itself and
*           the driver instance. See Section 4.10.2 for more detail on this
*           structures members.
*       n32Cmd - A number assigned to a particular device command. Typically
*           this is defined in the device driver's public header file
*           (i.e. <device_driver_name>.h)
*       tList - A pointer to a variable argument list which contains zero
*           or more command specific arguments. The number and type of each
*           argument must be defined in a device driver document for this
*           particular device driver.
*
*   Outputs:
*       DEV_OK should be returned on success. Otherwise a value < 0 is
*       returned on error. Any unsupported ioctl commands should return
*       DEV_UNSUPPORTED_COMMAND. See section 4.11 for possible error codes.
*
*****************************************************************************/
static N32 _n32Ioctl (
                        const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                        N32 n32Cmd,
                        va_list *ptList
                        )
{
    N32 n32Err = DEV_UNSUPPORTED_CMD;

    // Extract public data from the Device Driver Framework
    SIO_SHIM_PUBLIC_STRUCT *psPublicData =
        (SIO_SHIM_PUBLIC_STRUCT *)psDeviceInfo->pvPublicData;

    // Extract private data from the Device Driver Framework
    BOOLEAN bSignals = (BOOLEAN)(size_t)psDeviceInfo->pvPrivateData;

    // We can call the native driver's ioctl only if it is either
    // a comm node, or a signal node which is trying to get, set
    // or clear signals. Otherwise it is not permitted.
    if((bSignals == FALSE) ||
           ((bSignals == TRUE) &&
                    (   (n32Cmd == COM_IOCTL_SET_SIGNALS) ||
                        (n32Cmd == COM_IOCTL_GET_SIGNALS) ||
                        (n32Cmd == COM_IOCTL_CLR_SIGNALS)
                    )
           )
      )
    {
        // Pass-thru to native driver
        n32Err =
            psPublicData->psFile->psIntf->iIoctl(
                (FILE *)psPublicData->psFile->pvHdl, n32Cmd, ptList);
    }

    return n32Err;
}
