/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * PRIVATE HEADER
 *
 ******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef _SQL_INTERFACE_OBJ_H_
#define _SQL_INTERFACE_OBJ_H_

  /**************/
 /** INCLUDES **/
/**************/

#include <stdio.h>

#include "standard.h"

#include "sqlite3.h"

#include "sql_interface_obj.h"

  /***************/
 /** CONSTANTS **/
/***************/

// Setting ENABLE_SQL_LOGGING to 1 will enable the
// logging configuration option in sqlite which can be
// helpful with debugging
#define ENABLE_SQL_LOGGING 0

/* Object name prefix for objects */
#define SQL_INTERFACE_OBJECT_NAME   "SQL"

#define SQL_CONTROL_OBJECT_NAME     "SQL Control Object"
#define SQL_CONTROL_SEMAPHORE_NAME  "SqlIntCntrlSem"

// Constant used to ID osal for SQLite
#define OSAL_VFS_NAME "osal"

// Constant command strings for SQL
#define SQL_START_TRANSACTION "begin;"
#define SQL_END_TRANSACTION "end;"

// The maximum path length; this is 255 for Posix, and 260 for
// Win32, so we just set this to be the smaller of the two (lest
// we be able to sneak 260-character paths in under Posix OSes.)
#define SQL_MAX_PATH_LEN    (255)

// We set the maximum "attach" name at 20, which should be large enough
// for most uses.
#define SQL_MAX_ATTACH_NAME_LEN (20)

// The %Q in the format string below is part of SQLite's extended printf
// format set. It single-quotes the string, then properly escapes any
// inner single-quotes passed as part of a filename. This protects SQL
// calls in the (presumably rare) case of an application providing a
// database path that contains a single-quote / apostrophe.

#define SQL_INTERFACE_FORMAT_PATH_QUOTE "%Q"

// The attach operation below makes use of the special SQLite printf()
// function to ensure that any options (the path, in this case) are
// escaped according to SQL format rules.

#define SQL_ATTACH_DB \
    "attach "SQL_INTERFACE_FORMAT_PATH_QUOTE" as %s;"

#define SQL_DETACH_DB \
    "detach %s;"

// Using the above, we can figure out the maximum size of our attach
// and detach commands

#define SQL_MAX_ATTACH_CMD_LEN (SQL_MAX_PATH_LEN + SQL_MAX_ATTACH_NAME_LEN + 13)
#define SQL_MAX_DETACH_CMD_LEN (SQL_MAX_ATTACH_NAME_LEN + 9)

#define SQL_BUFFER_INCREASE_MARGIN  (20)

///////////////////////////////////////
//          SQLITE PRAGMAS           //
///////////////////////////////////////
#define SQL_INTEGRITY_CHECK "pragma integrity_check;"
#define SQL_INTEGRITY_VERIFICATION "ok"

#define SQLITE_GET_PAGESIZE "PRAGMA page_size"

//SQLITE.org says max page size is 65536, so
// we can limit the input number to 5 characters
#define SQLITE_SET_PAGESIZE "PRAGMA page_size=%5u"
#define SQLITE_SET_PAGESIZE_LEN (32)


  /**************/
 /** TYPEDEFS **/
/**************/

/* Private object elements */

// Structure which groups an SQL query result
// with metadata about the result
typedef struct sql_query_result_struct
{
    int iResultRows;
    int iResultColumns;
    char **ppacResultString;

} SQL_QUERY_RESULT_STRUCT;

typedef struct sql_pramga_result_struct
{
    BOOLEAN bSuccess;
    UN32 un32PragmaValue;

} SQL_PRAGMA_RESULT_STRUCT;

typedef struct sql_object_struct
{
    // SQLite connection handle
    sqlite3 *psDB;

    // Last error code received from SQLite
    // Declared as "int" to match SQL API
    // Just used for debugging
    int iLastSQLErrorCode;

    // Work buffer and the buffer's current size
    char *pacBuffer;
    size_t tBufferSize;

    // Prepared statements for start/end transactions
    SQL_PREPARED_STATEMENT_HANDLE hStartTransactionStmt;
    SQL_PREPARED_STATEMENT_HANDLE hEndTransactionStmt;

} SQL_OBJECT_STRUCT;

// The SQL control object which manages
// data for all SQL connections
typedef struct sql_interface_ctrl_struct
{
    // number of sql connections
    UN16 un16NumConnections;

    // sem to control access to this object
    OSAL_OBJECT_HDL hSem;

} SQL_INTERFACE_CTRL_STRUCT;

  /************/
 /** MACROS **/
/************/

  /****************/
 /** PROTOTYPES **/
/****************/

/* Object Public Prototypes */

static BOOLEAN bInitializeSqlite ( void );

static void vUninitializeSqlite ( void );

static SQL_INTERFACE_OBJECT hConnectSqlite (
    const char *pacDatabaseFile,
    SQL_INTERFACE_OPTIONS_MASK t2Options,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        );

static BOOLEAN bCopyMainDatabase (
    SQL_INTERFACE_OBJECT hDest,
    SQL_INTERFACE_OBJECT hSrc
        );

static void vDisconnectSqlite (
    SQL_INTERFACE_OBJECT hSQL
        );

static BOOLEAN bCheckIntegrity (
    SQL_INTERFACE_OBJECT hSQL
        );

static BOOLEAN bAttachDBToConnection (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacDatabaseFile,
    const char *pacAttachName,
    DATASERVICE_ERROR_CODE_ENUM *peErrorCode
        );

static void vDetachDBFromConnection (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacAttachName
        );

static BOOLEAN bExecuteCommand (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacCommandString
        );

static BOOLEAN bExecutePreparedCommand (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacCommandString,
    size_t tNumColumns,
    PREPARED_QUERY_COLUMN_CALLBACK bColumnCallback,
    void *pvCallbackArg
        );

static BOOLEAN bStartTransaction (
    SQL_INTERFACE_OBJECT hSQL
        );

static BOOLEAN bEndTransaction (
    SQL_INTERFACE_OBJECT hSQL
        );

static BOOLEAN bQuery (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacQueryString,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg
        );

static SQL_PREPARED_STATEMENT_HANDLE hCreatePreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    const char *pacStatementString
        );

static void vDestroyPreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE hStatement
        );

static BOOLEAN bExecutePreparedStatement (
    SQL_INTERFACE_OBJECT hSQL,
    SQL_PREPARED_STATEMENT_HANDLE hStatement,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg,
    size_t tNumParams,
    ...
        );

static BOOLEAN bGetLastInsertRowID (
    SQL_INTERFACE_OBJECT hSQL,
    UN32* pun32RowID
        );

/* Object Private Prototypes */

static BOOLEAN bExpandBuffer (
    SQL_OBJECT_STRUCT *psObj,
    size_t tNewSizeRequired,
    BOOLEAN bCopy
        );

static int iBindParameters (
    SQL_OBJECT_STRUCT *psObj,
    sqlite3_stmt *psStatement,
    const SQL_BIND_PARAMETER_STRUCT *psParams,
    size_t tNumParams
        );

static int iBindString (
    SQL_OBJECT_STRUCT *psObj,
    sqlite3_stmt *psStatement,
    int iColIndex,
    STRING_OBJECT hString
        );

static BOOLEAN bIntegrityQueryResult (
    SQL_QUERY_COLUMN_STRUCT *psColumns,
    N32 n32NumberOfColumns,
    void *pvArg
        );

static BOOLEAN bSetDBPageSize (
    SQL_INTERFACE_OBJECT hSQL,
    UN32 un32PageSize
        );

static UN32 un32DBPageSize (
    SQL_INTERFACE_OBJECT hSQL
        );

static BOOLEAN bProcessQueryPragma(
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    SQL_PRAGMA_RESULT_STRUCT *psResult
        );

static int iProcessQuery (
    sqlite3_stmt * psStatement,
    SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
    void *pvArg
        );

static void vDestroyObject (
    SQL_OBJECT_STRUCT *psObj
        );

#if ENABLE_SQL_LOGGING == 1
static void vLogSQL (
    void *pvArg,
    int iCode,
    const char *pcMsg
        );
#endif

  /***************/
 /** VARIABLES **/
/***************/

// Global instance of an interface for this object
const SQL_PORT_INTERFACE_STRUCT SQL_INTERFACE =
{
    /*.bInitializeSqlite = */           bInitializeSqlite,
    /*.vUninitializeSqlite = */         vUninitializeSqlite,
    /*.hConnect = */                    hConnectSqlite,
    /*.vDisconnect = */                 vDisconnectSqlite,
    /*.bCopyMainDatabase = */           bCopyMainDatabase,
    /*.bAttachDBToConnection = */       bAttachDBToConnection,
    /*.vDetatchDBFromConnection = */    vDetachDBFromConnection,
    /*.bExecuteCommand = */             bExecuteCommand,
    /*.bExecutePreparedCommand = */     bExecutePreparedCommand,
    /*.bStartTransaction = */           bStartTransaction,
    /*.bEndTransaction = */             bEndTransaction,
    /*.bQuery = */                      bQuery,
    /*.hCreatePreparedStatement = */    hCreatePreparedStatement,
    /*.vDestroyPreparedStatement = */   vDestroyPreparedStatement,
    /*.bExecutePreparedStatement = */   bExecutePreparedStatement,
    /*.bGetLastInsertRowID =*/          bGetLastInsertRowID
};

static SQL_INTERFACE_CTRL_STRUCT *gpsCntrlStruct = (SQL_INTERFACE_CTRL_STRUCT *)NULL;

  /**********************/
 /** INLINE FUNCTIONS **/
/**********************/

#endif    // _SQL_INTERFACE_OBJ_H_
