/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the Object:SQL_INTERFACE header file.
 *
 ******************************************************************************/

#ifndef SQL_INTERFACE_OBJ_H_
#define SQL_INTERFACE_OBJ_H_

#include "standard.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Connection Options
#define SQL_INTERFACE_OPTIONS_NONE                      (0x0000)

#define SQL_INTERFACE_OPTIONS_READONLY                  (0x0001)
#define SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK     (0x0002)
#define SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND       (0x0004)

#define SQL_INTERFACE_OPTIONS_ALL            (     \
    (SQL_INTERFACE_OPTIONS_MASK)                   \
    SQL_INTERFACE_OPTIONS_READONLY |               \
    SQL_INTERFACE_OPTIONS_SKIP_CORRUPTION_CHECK |  \
    SQL_INTERFACE_OPTIONS_CREATE_IF_NOT_FOUND      \
    )

// Name used when attempting to open a memory database with sqlite
#define SQL_MEMORY_DATABASE_NAME    ":memory:"

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// SQL Interface handle
typedef void * SQL_INTERFACE_OBJECT;
#define SQL_INTERFACE_INVALID_OBJECT (SQL_INTERFACE_OBJECT)NULL

// SQL Interface handle
typedef void * SQL_PREPARED_STATEMENT_HANDLE;
#define SQL_PREPARED_STATEMENT_INVALID_HANDLE (SQL_PREPARED_STATEMENT_HANDLE)NULL

typedef size_t SQL_COLUMN_INDEX;

typedef UN32 SQL_INTERFACE_OPTIONS_MASK;

typedef enum sql_column_type_enum
{
    SQL_COLUMN_TYPE_INTEGER = 0,
    SQL_COLUMN_TYPE_C_STRING,
    SQL_COLUMN_TYPE_BLOB,
    SQL_COLUMN_TYPE_UNKNOWN
} SQL_COLUMN_TYPE_ENUM;

typedef enum sql_bind_type_enum
{
    SQL_BIND_TYPE_UN32 = 0,
    SQL_BIND_TYPE_N64,
    SQL_BIND_TYPE_C_STRING,
    SQL_BIND_TYPE_BLOB,
    SQL_BIND_TYPE_STRING_OBJECT,
    SQL_BIND_TYPE_UNKNOWN
} SQL_BIND_TYPE_ENUM;

typedef struct sql_query_column_un32_data
{
    UN32 un32Data;
} SQL_QUERY_COLUMN_UN32_DATA;

typedef struct sql_query_column_n64_data
{
    N64 n64Data;
} SQL_QUERY_COLUMN_N64_DATA;

typedef struct sql_query_column_cstring_data
{
    const unsigned char *pcData;
} SQL_QUERY_COLUMN_CSTRING_DATA;

typedef struct sql_query_column_blob_data
{
    const void *pvData;
    size_t tSize;
} SQL_QUERY_COLUMN_BLOB_DATA;

typedef union sql_query_column_data_union
{
    SQL_QUERY_COLUMN_UN32_DATA sUN32;
    SQL_QUERY_COLUMN_N64_DATA sN64;
    SQL_QUERY_COLUMN_CSTRING_DATA sCString;
    SQL_QUERY_COLUMN_BLOB_DATA sBLOB;

} SQL_QUERY_COLUMN_DATA_UNION;

typedef struct sql_query_column_struct
{
    const char *pcColName;
    N64 n64NativeInteger;
    SQL_COLUMN_TYPE_ENUM eType;
    SQL_QUERY_COLUMN_DATA_UNION uData;

} SQL_QUERY_COLUMN_STRUCT;

typedef struct sql_bind_parameter_struct
{
    SQL_BIND_TYPE_ENUM eType;   // parameter type
    void *pvData;               // parameter value
    size_t tSize;               // BLOB size

} SQL_BIND_PARAMETER_STRUCT;

// Query result table iterator prototype
typedef BOOLEAN (*SQL_QUERY_RESULT_HANDLER) (
    SQL_QUERY_COLUMN_STRUCT *psColumn,
    N32 n32NumberOfColumns,
    void *pvArg
       );

typedef BOOLEAN (*PREPARED_QUERY_COLUMN_CALLBACK)(
    SQL_COLUMN_INDEX tIndex,
    SQL_BIND_TYPE_ENUM *peType,
    size_t *ptDataSize,
    void **ppvData,
    void *pvCallbackArg
        );

// SQL Port Interface Struct
typedef struct sql_port_interface_struct
{
    BOOLEAN (*bInitialize) ( void );

    void (*vUninitialize) ( void );

    SQL_INTERFACE_OBJECT (*hConnect) (
        const char *pacDatabaseFile,
        SQL_INTERFACE_OPTIONS_MASK tOptions,
        DATASERVICE_ERROR_CODE_ENUM *peErrorCode
            );

    void (*vDisconnect) (
        SQL_INTERFACE_OBJECT hSQL
            );

    BOOLEAN (*bCopyMainDatabase) (
        SQL_INTERFACE_OBJECT hDest,
        SQL_INTERFACE_OBJECT hSrc
            );

    BOOLEAN (*bAttachDBToConnection) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacDatabaseFile,
        const char *pacAttachName,
        DATASERVICE_ERROR_CODE_ENUM *peErrorCode
            );

    void (*vDetachDBFromConnection) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacAttachName
            );

    BOOLEAN (*bExecuteCommand) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacCommandString
            );

    BOOLEAN (*bExecutePreparedCommand) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacCommandString,
        size_t tNumColumns,
        PREPARED_QUERY_COLUMN_CALLBACK bColumnCallback,
        void *pvCallbackArg
            );

    BOOLEAN (*bStartTransaction) (
        SQL_INTERFACE_OBJECT hSQL
            );

    BOOLEAN (*bEndTransaction) (
        SQL_INTERFACE_OBJECT hSQL
            );

    BOOLEAN (*bQuery) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacQueryString,
        SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
        void *pvArg
            );

    SQL_PREPARED_STATEMENT_HANDLE (*hCreatePreparedStatement) (
        SQL_INTERFACE_OBJECT hSQL,
        const char *pacStatementString
            );

    void (*vDestroyPreparedStatement) (
        SQL_INTERFACE_OBJECT hSQL,
        SQL_PREPARED_STATEMENT_HANDLE hStatement
            );

    BOOLEAN (*bExecutePreparedStatement) (
        SQL_INTERFACE_OBJECT hSQL,
        SQL_PREPARED_STATEMENT_HANDLE hStatement,
        SQL_QUERY_RESULT_HANDLER tbRowResultHandler,
        void *pvArg,
        size_t tNumParams,
        ...
            );

    BOOLEAN (*bGetLastInsertRowID) (
        SQL_INTERFACE_OBJECT hSQL,
        UN32* pun32RowID
            );

} SQL_PORT_INTERFACE_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/
extern const SQL_PORT_INTERFACE_STRUCT SQL_INTERFACE;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

/* Object Friend Prototypes */



#endif /* SQL_INTERFACE_OBJ_H_ */
