/******************************************************************************/
/*                    Copyright (c) Sirius XM Radio, Inc.                     */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius XM Radio, Inc.             */
/******************************************************************************/
/*******************************************************************************
 *
 * DESCRIPTION
 *
 * This is the SMS API header file.
 *
 ******************************************************************************/

#ifndef SMS_API_H_
#define SMS_API_H_

#include "standard.h"
#include <stddef.h>

#include "osal.h"

#ifdef __cplusplus
extern "C" {
#endif

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Provide initial definitions of SMS compile options if not defined
// at compile time as compile flags. This way they at least are defined
// as something.
#ifndef SMS_DEBUG
#define SMS_DEBUG 0
#endif

// SRM Event Mask
#define SRM_OBJECT_EVENT_NONE              (0x0000)

#define SRM_OBJECT_EVENT_STATE             (0x0001)

#define SRM_OBJECT_EVENT_ALL               (      \
    (SRM_EVENT_MASK)                              \
    SRM_OBJECT_EVENT_STATE                        \
    )

// Module Event Mask
#define MODULE_OBJECT_EVENT_NONE                    (0x0000)

#define MODULE_OBJECT_EVENT_STATE                   (0x0001)
#define MODULE_OBJECT_EVENT_SUBSTATUS               (0x0002)
#define MODULE_OBJECT_EVENT_UPDATE_PROGRESS         (0x0004)

#define MODULE_OBJECT_EVENT_ALL            (      \
    (MODULE_EVENT_MASK)                           \
    MODULE_OBJECT_EVENT_STATE |                   \
    MODULE_OBJECT_EVENT_SUBSTATUS |               \
    MODULE_OBJECT_EVENT_UPDATE_PROGRESS           \
    )

// Decoder Event Mask
#define DECODER_OBJECT_EVENT_NONE                   (0x00000000)

#define DECODER_OBJECT_EVENT_STATE                  (0x00000001)
#define DECODER_OBJECT_EVENT_CHANNEL                (0x00000002)
#define DECODER_OBJECT_EVENT_SIGNAL                 (0x00000004)
#define DECODER_OBJECT_EVENT_ANTENNA                (0x00000008)
#define DECODER_OBJECT_EVENT_TIME                   (0x00000010)
#define DECODER_OBJECT_EVENT_TUNE                   (0x00000020)
#define DECODER_OBJECT_EVENT_PLAYBACK               (0x00000040)
#define DECODER_OBJECT_EVENT_UPDATE_PROGRESS        (0x00000080)
#define DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED   (0x00000100)
#define DECODER_OBJECT_EVENT_SEEK_ALERT             (0x00000200)
#define DECODER_OBJECT_EVENT_ENGINEERING_DATA       (0x00000400)
#define DECODER_OBJECT_EVENT_EPG_UPDATE_DONE        (0x00000800)
#define DECODER_OBJECT_EVENT_SUB_NOTIFICATION_STATE (0x00001000)
#define DECODER_OBJECT_EVENT_TUNE_SCAN              (0x00002000)
#define DECODER_OBJECT_EVENT_ANTENNA_AIMING         (0x00004000)
#define DECODER_OBJECT_EVENT_TUNEMIX                (0x00008000)
#define DECODER_OBJECT_EVENT_AUDIO_PRESENCE         (0x00010000)

#define DECODER_OBJECT_EVENT_ANTENNA_SIGNAL         (   \
    (DECODER_EVENT_MASK)(                               \
    DECODER_OBJECT_EVENT_SIGNAL |                       \
    DECODER_OBJECT_EVENT_ANTENNA                    ))

#define DECODER_OBJECT_EVENT_ALL                    (   \
   (DECODER_EVENT_MASK)(                                \
    DECODER_OBJECT_EVENT_STATE |                        \
    DECODER_OBJECT_EVENT_CHANNEL |                      \
    DECODER_OBJECT_EVENT_SIGNAL |                       \
    DECODER_OBJECT_EVENT_ANTENNA |                      \
    DECODER_OBJECT_EVENT_TIME |                         \
    DECODER_OBJECT_EVENT_TUNE |                         \
    DECODER_OBJECT_EVENT_PLAYBACK |                     \
    DECODER_OBJECT_EVENT_UPDATE_PROGRESS |              \
    DECODER_OBJECT_EVENT_SUBSCRIPTION_UPDATED |         \
    DECODER_OBJECT_EVENT_SEEK_ALERT |                   \
    DECODER_OBJECT_EVENT_ENGINEERING_DATA |             \
    DECODER_OBJECT_EVENT_EPG_UPDATE_DONE |              \
    DECODER_OBJECT_EVENT_SUB_NOTIFICATION_STATE |       \
    DECODER_OBJECT_EVENT_TUNE_SCAN |                    \
    DECODER_OBJECT_EVENT_ANTENNA_AIMING |               \
    DECODER_OBJECT_EVENT_TUNEMIX |                      \
    DECODER_OBJECT_EVENT_AUDIO_PRESENCE            ))

//Engineering Data Event Mask
#define ENGINEERING_DATA_OBJECT_EVENT_NONE                      (0x0000)
#define ENGINEERING_DATA_OBJECT_EVENT_SIGNAL_DETAILED           (0x0001)
#define ENGINEERING_DATA_OBJECT_EVENT_OVERLAY_SIGNAL_DETAILED   (0x0002)
#define ENGINEERING_DATA_OBJECT_EVENT_AUDIO_DECODER_BITRATE     (0x0004)
#define ENGINEERING_DATA_OBJECT_EVENT_LINK_STATUS               (0x0008)

#define ENGINEERING_DATA_OBJECT_EVENT_ALL (                 \
    (ENGINEERING_DATA_EVENT_MASK)                           \
    ENGINEERING_DATA_OBJECT_EVENT_SIGNAL_DETAILED |         \
    ENGINEERING_DATA_OBJECT_EVENT_OVERLAY_SIGNAL_DETAILED | \
    ENGINEERING_DATA_OBJECT_EVENT_AUDIO_DECODER_BITRATE |   \
    ENGINEERING_DATA_OBJECT_EVENT_LINK_STATUS               \
    )

// Playback Event Mask
#define PLAYBACK_OBJECT_EVENT_NONE                  (0x0000)

#define PLAYBACK_OBJECT_EVENT_PLAY                  (0x0001)
#define PLAYBACK_OBJECT_EVENT_PAUSE                 (0x0002)
#define PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE       (0x0004)
#define PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE       (0x0008)
#define PLAYBACK_OBJECT_EVENT_TIME_OFFSET           (0x0010)
#define PLAYBACK_OBJECT_EVENT_TIME_FROM_TRACK_START (0x0020)
#define PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO (0x0040)
#define PLAYBACK_OBJECT_EVENT_WARNING               (0x0080)
#define PLAYBACK_OBJECT_EVENT_LIMIT                 (0x0100)
#define PLAYBACK_OBJECT_EVENT_ERROR                 (0x0200)
#define PLAYBACK_OBJECT_EVENT_DURATION_OF_TRACK     (0x0400)
#define PLAYBACK_OBJECT_EVENT_TRACKS_BEFORE         (0x0800)
#define PLAYBACK_OBJECT_EVENT_TRACKS_REMAINING      (0x1000)


#define PLAYBACK_OBJECT_EVENT_ALL           (       \
    (PLAYBACK_OBJECT_EVENT_MASK)                    \
    PLAYBACK_OBJECT_EVENT_PLAY |                    \
    PLAYBACK_OBJECT_EVENT_PAUSE |                   \
    PLAYBACK_OBJECT_EVENT_PLAY_PERCENTAGE |         \
    PLAYBACK_OBJECT_EVENT_FILL_PERCENTAGE |         \
    PLAYBACK_OBJECT_EVENT_TIME_OFFSET |             \
    PLAYBACK_OBJECT_EVENT_TIME_FROM_TRACK_START |   \
    PLAYBACK_OBJECT_EVENT_RECORDED_CONTENT_INFO |   \
    PLAYBACK_OBJECT_EVENT_WARNING |                 \
    PLAYBACK_OBJECT_EVENT_LIMIT |                   \
    PLAYBACK_OBJECT_EVENT_DURATION_OF_TRACK |       \
    PLAYBACK_OBJECT_EVENT_TRACKS_BEFORE |           \
    PLAYBACK_OBJECT_EVENT_TRACKS_REMAINING |        \
    PLAYBACK_OBJECT_EVENT_ERROR             \
    )

// Song Event Mask
#define SONG_OBJECT_EVENT_NONE          (0x0000)

#define SONG_OBJECT_EVENT_TITLE         (0x0001)
#define SONG_OBJECT_EVENT_ARTIST        (0x0002)
#define SONG_OBJECT_EVENT_CHANNEL_ID    (0x0004)
#define SONG_OBJECT_EVENT_DURATION      (0x0008)
#define SONG_OBJECT_EVENT_TIME_STAMP    (0x0010)
#define SONG_OBJECT_EVENT_STATUS        (0x0020)

#define SONG_OBJECT_EVENT_ALL           (      \
    (SONG_EVENT_MASK)                          \
    SONG_OBJECT_EVENT_TITLE      |                 \
    SONG_OBJECT_EVENT_ARTIST     |                 \
    SONG_OBJECT_EVENT_CHANNEL_ID |                 \
    SONG_OBJECT_EVENT_DURATION   |                 \
    SONG_OBJECT_EVENT_TIME_STAMP |                 \
    SONG_OBJECT_EVENT_STATUS                       \
    )


// SongList Event Mask
#define SONGLIST_OBJECT_EVENT_NONE          (0x0000)

#define SONGLIST_OBJECT_EVENT_UPDATE        (0x0001)
#define SONGLIST_OBJECT_EVENT_SONG          (0x0002)

#define SONGLIST_OBJECT_EVENT_ALL           (      \
    (SONGLIST_OBJECT_EVENT_MASK)                   \
    SONGLIST_OBJECT_EVENT_UPDATE |                 \
    SONGLIST_OBJECT_EVENT_SONG                     \
    )

// Channel Event Mask
#define CHANNEL_OBJECT_EVENT_NONE          ((CHANNEL_EVENT_MASK)0x00000000)

#define CHANNEL_OBJECT_EVENT_SERVICE_ID    ((CHANNEL_EVENT_MASK)0x00000001)
#define CHANNEL_OBJECT_EVENT_CHANNEL_ID    ((CHANNEL_EVENT_MASK)0x00000002)
#define CHANNEL_OBJECT_EVENT_ATTRIBUTES    ((CHANNEL_EVENT_MASK)0x00000004)
#define CHANNEL_OBJECT_EVENT_NAME          ((CHANNEL_EVENT_MASK)0x00000008)
#define CHANNEL_OBJECT_EVENT_CATEGORY      ((CHANNEL_EVENT_MASK)0x00000010)
#define CHANNEL_OBJECT_EVENT_TITLE         ((CHANNEL_EVENT_MASK)0x00000020)
#define CHANNEL_OBJECT_EVENT_ARTIST        ((CHANNEL_EVENT_MASK)0x00000040)
#define CHANNEL_OBJECT_EVENT_COMPOSER      ((CHANNEL_EVENT_MASK)0x00000080)
#define CHANNEL_OBJECT_EVENT_CONTENTINFO   ((CHANNEL_EVENT_MASK)0x00000100)
#define CHANNEL_OBJECT_EVENT_DESCRIPTION   ((CHANNEL_EVENT_MASK)0x00000200)
#define CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS \
                                           ((CHANNEL_EVENT_MASK)0x00000400)
#define CHANNEL_OBJECT_EVENT_ART           ((CHANNEL_EVENT_MASK)0x00000800)
#define CHANNEL_OBJECT_EVENT_REMOVED       ((CHANNEL_EVENT_MASK)0x00001000)
#define CHANNEL_OBJECT_EVENT_TYPE_INFO     ((CHANNEL_EVENT_MASK)0x00002000)
#define CHANNEL_OBJECT_EVENT_EPG           ((CHANNEL_EVENT_MASK)0x00004000)
#define CHANNEL_OBJECT_EVENT_IR_NAV_CLASS  ((CHANNEL_EVENT_MASK)0x00008000)
#define CHANNEL_OBJECT_EVENT_ACO           ((CHANNEL_EVENT_MASK)0x00010000)
#define CHANNEL_OBJECT_EVENT_ALBUM         ((CHANNEL_EVENT_MASK)0x00020000)

#define CHANNEL_OBJECT_EVENT_ALL           (      \
    (CHANNEL_EVENT_MASK)                          \
    CHANNEL_OBJECT_EVENT_SERVICE_ID |             \
    CHANNEL_OBJECT_EVENT_CHANNEL_ID |             \
    CHANNEL_OBJECT_EVENT_ATTRIBUTES |             \
    CHANNEL_OBJECT_EVENT_NAME |                   \
    CHANNEL_OBJECT_EVENT_CATEGORY |               \
    CHANNEL_OBJECT_EVENT_TITLE |                  \
    CHANNEL_OBJECT_EVENT_COMPOSER |               \
    CHANNEL_OBJECT_EVENT_ARTIST |                 \
    CHANNEL_OBJECT_EVENT_ALBUM |                  \
    CHANNEL_OBJECT_EVENT_CONTENTINFO |            \
    CHANNEL_OBJECT_EVENT_DESCRIPTION |            \
    CHANNEL_OBJECT_EVENT_SIMILAR_CHANNELS |       \
    CHANNEL_OBJECT_EVENT_ART |                    \
    CHANNEL_OBJECT_EVENT_REMOVED |                \
    CHANNEL_OBJECT_EVENT_TYPE_INFO |              \
    CHANNEL_OBJECT_EVENT_EPG |                    \
    CHANNEL_OBJECT_EVENT_IR_NAV_CLASS |           \
    CHANNEL_OBJECT_EVENT_ACO                      \
        )

// Category Event Mask
#define CATEGORY_OBJECT_EVENT_NONE               (0x0000)

// Category event types
#define CATEGORY_OBJECT_EVENT_CONTENTS           (0x0001)
#define CATEGORY_OBJECT_EVENT_ART                (0x0002)
#define CATEGORY_OBJECT_EVENT_NAME               (0x0004)
#define CATEGORY_OBJECT_EVENT_REMOVED            (0x0008)

#define CATEGORY_OBJECT_EVENT_ALL           (      \
    (CATEGORY_EVENT_MASK)                          \
    CATEGORY_OBJECT_EVENT_CONTENTS |               \
    CATEGORY_OBJECT_EVENT_ART |                    \
    CATEGORY_OBJECT_EVENT_NAME     |               \
    CATEGORY_OBJECT_EVENT_REMOVED                  \
        )

// Presets Event Mask
#define PRESETS_OBJECT_EVENT_NONE               (0x0000)

// Presets Event Types
#define PRESETS_OBJECT_EVENT_BAND_ADDED         (0x0001)
#define PRESETS_OBJECT_EVENT_BAND_REMOVED       (0x0002)
#define PRESETS_OBJECT_EVENT_BAND_MODIFIED      (0x0004)

#define PRESETS_OBJECT_EVENT_ALL                (   \
       (PRESETS_EVENT_MASK)                         \
        PRESETS_OBJECT_EVENT_BAND_ADDED     |       \
        PRESETS_OBJECT_EVENT_BAND_REMOVED   |       \
        PRESETS_OBJECT_EVENT_BAND_MODIFIED             )

// Channel Art Service Event Mask
#define CHANNEL_ART_SERVICE_EVENT_NONE             (0x0000)
#define CHANNEL_ART_SERVICE_EVENT_SERVICE_STATE    (0x0001)
#define CHANNEL_ART_SERVICE_EVENT_CHAN_PROD_STATE  (0x0002)
#define CHANNEL_ART_SERVICE_EVENT_ALBUM_PROD_STATE (0x0004)

#define CHANNEL_ART_SERVICE_EVENT_ALL     (        \
    (CHANNEL_ART_SERVICE_EVENT_MASK)               \
    CHANNEL_ART_SERVICE_EVENT_SERVICE_STATE |      \
    CHANNEL_ART_SERVICE_EVENT_CHAN_PROD_STATE |    \
    CHANNEL_ART_SERVICE_EVENT_ALBUM_PROD_STATE     \
    )

// Data Services Event Mask
#define DATASERVICE_EVENT_NONE      (0x0000)

#define DATASERVICE_EVENT_STATE      (0x0001)
#define DATASERVICE_EVENT_NEW_DATA   (0x0002)
#define DATASERVICE_EVENT_TIMEOUT    (0x0004)
#define DATASERVICE_EVENT_SERVICE    (0x0008)

#define DATASERVICE_EVENT_ALL     (        \
    (DATASERVICE_EVENT_MASK)               \
    DATASERVICE_EVENT_STATE |              \
    DATASERVICE_EVENT_NEW_DATA |           \
    DATASERVICE_EVENT_TIMEOUT              \
    )

#define CDO_FIELD_NONE                 (0x0000)
#define CDO_FIELD_ARTIST               (0x0001)
#define CDO_FIELD_TITLE                (0x0002)
#define CDO_FIELD_COMPOSER             (0x0004)
#define CDO_FIELD_CONTENTINFO          (0x0008)
#define CDO_FIELD_ALBUM                (0x0010)

#define CDO_FIELD_ALL                  (      \
    (CDO_FIELD_MASK)                          \
    CDO_FIELD_ARTIST   |                      \
    CDO_FIELD_TITLE    |                      \
    CDO_FIELD_COMPOSER |                      \
    CDO_FIELD_ALBUM    |                      \
    CDO_FIELD_CONTENTINFO                     \
    )

#define MODULE_SUSPED_DATE_INVALID (0)

// Sports service related constants.
#define SPORTS_MONITOR_EVENT_NONE                   (0x0000)
#define SPORTS_MONITOR_EVENT_SPORTLIST              (0x0001)
#define SPORTS_MONITOR_EVENT_AFFILIATE_EXISTANCE    (0x0002)
#define SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL   (0x0004)
#define SPORTS_MONITOR_EVENT_ALL           (        \
    (SPORTS_MONITOR_EVENT_MASK)                     \
    SPORTS_MONITOR_EVENT_SPORTLIST |                \
    SPORTS_MONITOR_EVENT_AFFILIATE_EXISTANCE |      \
    SPORTS_MONITOR_EVENT_AFFILIATE_INDIVIDUAL       \
    )

// Known information classes.  Information classes also represent
// the bit position of class in a SPORTS_MONITOR_EVENT_MASK.
#define INFO_CLASS_HEAD_TO_HEAD_EVENT               (0)
#define INFO_CLASS_RANKED_LIST_EVENT                (1)
#define INFO_CLASS_SEASON_SUMMARY                   (2)
#define INFO_CLASS_NEWS_ITEM                        (3)
#define INFO_CLASS_PARTICIPANT_STATISTICS           (4)
#define INFO_CLASS_PARTICIPANT_INDEX                (5)
#define INFO_CLASS_EVENT_DESCRIPTION                (6)

#define DEVICE_EVENT_NONE                        (0x0000)
#define DEVICE_EVENT_INITIAL_POSITION_CONFIGURED (0x0001)
#define DEVICE_EVENT_POSITION_UPDATE             (0x0002)
#define DEVICE_EVENT_REGISTRATION_UPDATE         (0x0004)

#define DEVICE_EVENT_ALL     (                 \
    (DEVICE_EVENT_MASK)                        \
    DEVICE_EVENT_INITIAL_POSITION_CONFIGURED | \
    DEVICE_EVENT_POSITION_UPDATE             | \
    DEVICE_EVENT_REGISTRATION_UPDATE           \
    )

#define AFFILIATE_EVENT_BODY_BIT_POSITION           (31)

// Behaviors
#define SMS_BEHAVIOR_DECODER_SELF_TUNE_MATURE (0)
#define SMS_BEHAVIOR_RESTORE_FACTORY_DEFAULTS (1)

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

// Basic Types
typedef N16 CHANNEL_ID;
#define CHANNEL_INVALID_ID N16_MAX

typedef N16 SERVICE_ID;
#define SERVICE_INVALID_ID N16_MAX

typedef UN16 TOPIC_ID;
#define TOPIC_INVALID_ID UN16_MAX

typedef UN32 SERIES_ID;
#define SERIES_INVALID_ID UN32_MAX

typedef UN32 PROGRAM_ID;
#define PROGRAM_INVALID_ID UN32_MAX

typedef N16 CATEGORY_ID;
#define CATEGORY_INVALID_ID N16_MAX

typedef UN16 USER_ID;
#define USER_INVALID_ID UN16_MAX

typedef N16 CATEGORY_CHANNEL_INDEX;
#define CATEGORY_CHANNEL_INVALID_INDEX ((CATEGORY_CHANNEL_INDEX)-1)

/* Object handle definitions */
typedef struct sms_object * SMS_OBJECT;
#define SMS_INVALID_OBJECT ((SMS_OBJECT)NULL)

typedef struct srm_object * SRM_OBJECT;
#define SRM_INVALID_OBJECT (SRM_OBJECT)NULL

typedef struct module_object * MODULE_OBJECT;
#define MODULE_INVALID_OBJECT (MODULE_OBJECT)NULL

typedef struct decoder_object * DECODER_OBJECT;
#define DECODER_INVALID_OBJECT (DECODER_OBJECT)NULL

typedef struct detailed_signal_quality_object * DETAILED_SIGNAL_QUALITY_OBJECT;
#define DETAILED_SIGNAL_QUALITY_INVALID_OBJECT (DETAILED_SIGNAL_QUALITY_OBJECT)NULL

typedef struct detailed_overlay_signal_quality_object * DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT;
#define DETAILED_OVERLAY_SIGNAL_QUALITY_INVALID_OBJECT (DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT)NULL

typedef struct link_status_information_object * LINK_STATUS_INFORMATION_OBJECT;
#define LINK_STATUS_INFORMATION_INVALID_OBJECT (LINK_STATUS_INFORMATION_OBJECT)NULL

typedef struct playback_object * PLAYBACK_OBJECT;
#define PLAYBACK_INVALID_OBJECT (PLAYBACK_OBJECT)NULL

typedef struct song_object * SONG_OBJECT;
#define SONG_INVALID_OBJECT (SONG_OBJECT)NULL

typedef struct songlist_object * SONGLIST_OBJECT;
#define SONGLIST_INVALID_OBJECT (SONGLIST_OBJECT)NULL

typedef struct channel_object * CHANNEL_OBJECT;
#define CHANNEL_INVALID_OBJECT (CHANNEL_OBJECT)NULL

typedef struct channellist_object * CHANNELLIST_OBJECT;
#define CHANNELLIST_INVALID_OBJECT (CHANNELLIST_OBJECT)NULL

typedef struct category_object * CATEGORY_OBJECT;
#define CATEGORY_INVALID_OBJECT (CATEGORY_OBJECT)NULL

typedef struct cdo_object * CD_OBJECT;
#define CD_INVALID_OBJECT (CD_OBJECT)NULL

typedef struct cid_object * CID_OBJECT;
#define CID_INVALID_OBJECT (CID_OBJECT)NULL

typedef struct team_object * TEAM_OBJECT;
#define TEAM_INVALID_OBJECT (TEAM_OBJECT)NULL

typedef TEAM_OBJECT CAR_OBJECT;
#define CAR_INVALID_OBJECT (CAR_OBJECT)NULL

typedef struct league_object * LEAGUE_OBJECT;
#define LEAGUE_INVALID_OBJECT (LEAGUE_OBJECT)NULL

typedef struct string_object * STRING_OBJECT;
#define STRING_INVALID_OBJECT (STRING_OBJECT)NULL

typedef struct content_event_monitoring_list_object * CEML_OBJECT;
#define CEML_INVALID_OBJECT (CEML_OBJECT)NULL

typedef struct content_alert_list_object * CAL_OBJECT;
#define CAL_INVALID_OBJECT (CAL_OBJECT)NULL

typedef struct cal_content_object * CAL_CONTENT_OBJECT;
#define CAL_CONTENT_INVALID_OBJECT (CAL_CONTENT_OBJECT)NULL

typedef struct cal_alert_object * CAL_ALERT_OBJECT;
#define CAL_ALERT_INVALID_OBJECT (CAL_ALERT_OBJECT)NULL

typedef struct seek_service_object * SEEK_SERVICE_OBJECT;
#define SEEK_SERVICE_INVALID_OBJECT (SEEK_SERVICE_OBJECT)NULL

typedef CAL_CONTENT_OBJECT SEEK_CONTENT_OBJECT;
#define SEEK_CONTENT_INVALID_OBJECT CAL_CONTENT_INVALID_OBJECT

typedef CAL_ALERT_OBJECT SEEK_EVENT_OBJECT;
#define SEEK_EVENT_INVALID_OBJECT CAL_ALERT_INVALID_OBJECT

typedef struct sport_zone_service_object * SPORT_ZONE_SERVICE_OBJECT;
#define SPORT_ZONE_SERVICE_INVALID_OBJECT (SPORT_ZONE_SERVICE_OBJECT)NULL

typedef struct sport_zone_object * SPORT_ZONE_OBJECT;
#define SPORT_ZONE_INVALID_OBJECT (SPORT_ZONE_OBJECT)NULL

typedef struct presets_object * PRESETS_OBJECT;
#define PRESETS_INVALID_OBJECT (PRESETS_OBJECT)NULL

typedef struct preset_band_object * PRESET_BAND_OBJECT;
#define PRESET_BAND_INVALID_OBJECT (PRESET_BAND_OBJECT)NULL

typedef struct dataservice_mgr_object * DATASERVICE_MGR_OBJECT;
#define DATASERVICE_MGR_INVALID_OBJECT (DATASERVICE_MGR_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT CHANNEL_ART_SERVICE_OBJECT;
#define CHANNEL_ART_SERVICE_INVALID_OBJECT (CHANNEL_ART_SERVICE_OBJECT)NULL

typedef struct channel_art_object * CHANNEL_ART_OBJECT;
#define CHANNEL_ART_INVALID_OBJECT (CHANNEL_ART_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT EPG_SERVICE_OBJECT;
#define EPG_SERVICE_INVALID_OBJECT (EPG_SERVICE_OBJECT)NULL

typedef struct eprog_event_struct * PROGRAM_OBJECT;
#define PROGRAM_INVALID_OBJECT (PROGRAM_OBJECT)NULL

typedef struct epg_topics_list_struct * EPG_TOPICS_LIST;
#define EPG_INVALID_TOPICS_LIST_OBJECT (EPG_TOPICS_LIST)NULL

typedef struct epg_topic_struct * TOPIC_OBJECT;
#define TOPIC_INVALID_OBJECT (TOPIC_OBJECT)NULL

typedef struct epg_channel_object_struct * EPG_CHANNEL_OBJECT;
#define EPG_INVALID_CHANNEL_OBJECT (EPG_CHANNEL_OBJECT)NULL

typedef struct image_object * IMAGE_OBJECT;
#define IMAGE_INVALID_OBJECT (IMAGE_OBJECT)NULL

typedef struct DSRL_TARGET_object * DSRL_TARGET_OBJECT;
#define DSRL_TARGET_INVALID_OBJECT (DSRL_TARGET_OBJECT)NULL

typedef struct DSRL_ENTRY_object * DSRL_ENTRY_OBJECT;
#define DSRL_ENTRY_INVALID_OBJECT (DSRL_ENTRY_OBJECT)NULL

typedef struct locid_object * LOCID_OBJECT;
#define LOCID_INVALID_OBJECT (LOCID_OBJECT)NULL

typedef struct distance_object * DISTANCE_OBJECT;
#define DISTANCE_INVALID_OBJECT (DISTANCE_OBJECT)NULL

typedef DSRL_TARGET_OBJECT LOCATION_OBJECT;
#define LOCATION_INVALID_OBJECT (LOCATION_OBJECT)NULL

typedef struct shape_object *SHAPE_OBJECT;
#define SHAPE_INVALID_OBJECT (SHAPE_OBJECT)NULL

typedef struct dataservice_result_list_object * DSRL_OBJECT;
#define DSRL_INVALID_OBJECT (DSRL_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT TRAFFIC_SERVICE_OBJECT;
#define TRAFFIC_SERVICE_INVALID_OBJECT (TRAFFIC_SERVICE_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT WEATHER_SERVICE_OBJECT;
#define WEATHER_SERVICE_INVALID_OBJECT (WEATHER_SERVICE_OBJECT)NULL

typedef struct device_registration_object * DEVICE_REGISTRATION_OBJECT;
#define DEVICE_REGISTRATION_INVALID_OBJECT (DEVICE_REGISTRATION_OBJECT)NULL

typedef struct weather_msg_object * WEATHER_MSG_OBJECT;
#define WEATHER_MSG_INVALID_OBJECT (WEATHER_MSG_OBJECT)NULL

typedef struct weather_event_object * WEATHER_EVENT_OBJECT;
#define WEATHER_EVENT_INVALID_OBJECT (WEATHER_EVENT_OBJECT)NULL

typedef struct temperature_object * TEMPERATURE_OBJECT;
#define TEMPERATURE_INVALID_OBJECT (TEMPERATURE_OBJECT)NULL

typedef struct precipitation_object * PRECIPITATION_OBJECT;
#define PRECIPITATION_INVALID_OBJECT (PRECIPITATION_OBJECT)NULL

typedef struct wind_object * WIND_OBJECT;
#define WIND_INVALID_OBJECT (WIND_OBJECT)NULL

typedef struct humidity_object * HUMIDITY_OBJECT;
#define HUMIDITY_INVALID_OBJECT (HUMIDITY_OBJECT)NULL

typedef struct uvinfo_object * UVINFO_OBJECT;
#define UVINFO_INVALID_OBJECT (UVINFO_OBJECT)NULL

typedef struct air_object * AIR_OBJECT;
#define AIR_INVALID_OBJECT (AIR_OBJECT)NULL

typedef struct forecast_object * FORECAST_OBJECT;
#define FORECAST_INVALID_OBJECT (FORECAST_OBJECT)NULL

typedef struct snow_conditions_object * SNOW_CONDITIONS_OBJECT;
#define SNOW_CONDITIONS_INVALID_OBJECT (SNOW_CONDITIONS_OBJECT)NULL

typedef struct ski_attributes_object * SKI_ATTRIBUTES_OBJECT;
#define SKI_ATTRIBUTES_INVALID_OBJECT (SKI_ATTRIBUTES_OBJECT)NULL

typedef struct ski_conditions_object * SKI_CONDITIONS_OBJECT;
#define SKI_CONDITIONS_INVALID_OBJECT (SKI_CONDITIONS_OBJECT)NULL

typedef LOCID_OBJECT TRAFFIC_LOCID_OBJECT;
#define TRAFFIC_LOCID_INVALID_OBJECT (TRAFFIC_LOCID_OBJECT)NULL

typedef struct traffic_msg_object * TRAFFIC_MSG_OBJECT;
#define TRAFFIC_MSG_INVALID_OBJECT (TRAFFIC_MSG_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT FUEL_SERVICE_OBJECT;
#define FUEL_SERVICE_INVALID_OBJECT (FUEL_SERVICE_OBJECT)NULL

typedef struct fuel_station_object * FUEL_STATION_OBJECT;
#define FUEL_STATION_INVALID_OBJECT (FUEL_STATION_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT MOVIES_SERVICE_OBJECT;
#define MOVIES_SERVICE_INVALID_OBJECT (MOVIES_SERVICE_OBJECT)NULL

typedef struct movie_object * MOVIE_OBJECT;
#define MOVIE_INVALID_OBJECT (MOVIE_OBJECT)NULL

typedef struct theater_object * THEATER_OBJECT;
#define THEATER_INVALID_OBJECT (THEATER_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT PHONETICS_SERVICE_OBJECT;
#define PHONETICS_SERVICE_INVALID_OBJECT (PHONETICS_SERVICE_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT AGW_SERVICE_OBJECT;
#define AGW_SERVICE_INVALID_OBJECT (AGW_SERVICE_OBJECT)NULL

typedef struct agw_tile_object * AGW_TILE_OBJECT;
#define AGW_TILE_INVALID_OBJECT (AGW_TILE_OBJECT)NULL

typedef struct agw_shape_object * AGW_SHAPE_OBJECT;
#define AGW_SHAPE_INVALID_OBJECT (AGW_SHAPE_OBJECT)NULL

typedef struct agw_front_object * AGW_FRONT_OBJECT;
#define AGW_FRONT_INVALID_OBJECT (AGW_FRONT_OBJECT)NULL

typedef struct agw_isobar_object * AGW_ISOBAR_OBJECT;
#define AGW_ISOBAR_INVALID_OBJECT (AGW_ISOBAR_OBJECT)NULL

typedef struct agw_pressure_center_object * AGW_PRESSURE_CENTER_OBJECT;
#define AGW_PRESSURE_CENTER_INVALID_OBJECT (AGW_PRESSURE_CENTER_OBJECT)NULL

typedef struct agw_storm_description_object * AGW_STORM_ATTRIBUTES_OBJECT;
#define AGW_STORM_ATTRIBUTES_INVALID_OBJECT (AGW_STORM_ATTRIBUTES_OBJECT)NULL

typedef struct agw_storm_position_object * AGW_STORM_POSITION_OBJECT;
#define AGW_STORM_POSITION_INVALID_OBJECT (AGW_STORM_POSITION_OBJECT)NULL

typedef struct agw_wind_radii_area_object * AGW_WIND_RADII_AREA_OBJECT;
#define AGW_WIND_RADII_AREA_INVALID_OBJECT (AGW_WIND_RADII_AREA_OBJECT)NULL

typedef UN32 CDO_FIELD_MASK;

typedef DATASERVICE_MGR_OBJECT SPORTS_SERVICE_OBJECT;
#define SPORTS_SERVICE_INVALID_OBJECT (SPORTS_SERVICE_OBJECT)NULL

typedef struct sports_monitor_object * SPORTS_MONITOR_OBJECT;
#define SPORTS_MONITOR_INVALID_OBJECT (SPORTS_MONITOR_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT WS_ALERTS_SERVICE_OBJECT;
#define WS_ALERTS_SERVICE_INVALID_OBJECT (WS_ALERTS_SERVICE_OBJECT)NULL

typedef struct ws_alert_msg_object * WS_ALERT_MSG_OBJECT;
#define WS_ALERT_MSG_INVALID_OBJECT (WS_ALERT_MSG_OBJECT)NULL

typedef struct ws_alerts_location_object * WS_ALERTS_LOCATION_OBJECT;
#define WS_ALERTS_LOCATION_INVALID_OBJECT (WS_ALERTS_LOCATION_OBJECT)NULL

typedef struct subscription_object * SUB_NOTIFICATION_OBJECT;
#define SUB_NOTIFICATION_INVALID_OBJECT (SUB_NOTIFICATION_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT SAFETY_CAMERAS_SERVICE_OBJECT;
#define SAFETY_CAMERAS_SERVICE_INVALID_OBJECT (SAFETY_CAMERAS_SERVICE_OBJECT)NULL

typedef struct safety_cameras_location_object * SAFETY_CAMERAS_LOCATION_OBJECT;
#define SAFETY_CAMERAS_LOCATION_INVALID_OBJECT (SAFETY_CAMERAS_LOCATION_OBJECT)NULL

typedef struct safety_cameras_sublocation_object * SAFETY_CAMERAS_SUBLOCATION_OBJECT;
#define SAFETY_CAMERAS_SUBLOCATION_INVALID_OBJECT (SAFETY_CAMERAS_SUBLOCATION_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT STOCK_TICKER_SERVICE_OBJECT;
#define STOCK_TICKER_SERVICE_INVALID_OBJECT (STOCK_TICKER_SERVICE_OBJECT)NULL

typedef struct stock_msg_object *STOCK_MSG_OBJECT;
#define STOCK_MSG_INVALID_OBJECT (STOCK_MSG_OBJECT)NULL

typedef DSRL_TARGET_OBJECT STOCK_SYMBOL_OBJECT;
#define STOCK_SYMBOL_INVALID_OBJECT (STOCK_SYMBOL_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT MAPS_SERVICE_OBJECT;
#define MAPS_SERVICE_INVALID_OBJECT (MAPS_SERVICE_OBJECT)NULL

typedef DATASERVICE_MGR_OBJECT TRAFFIC_CAMERAS_SERVICE_OBJECT;
#define TRAFFIC_CAMERAS_SERVICE_INVALID_OBJECT (TRAFFIC_CAMERAS_SERVICE_OBJECT)NULL

typedef struct tunemix_object *TUNEMIX_OBJECT;
#define TUNEMIX_INVALID_OBJECT (TUNEMIX_OBJECT)NULL

/*******************/
/*** SMS OBJECT ***/
/******************/

/* SMS Return Codes */
typedef enum smsapi_return_code_enum
{
    // Informational return codes
    SMSAPI_RETURN_CODE_SUCCESS,
    SMSAPI_RETURN_CODE_OP_NOT_SUPPORTED,
    SMSAPI_RETURN_CODE_OUT_OF_RANGE_PARAMETER,
    SMSAPI_RETURN_CODE_NO_OBJECTS,

    // Error return codes

    // Library state return codes
    SMSAPI_RETURN_CODE_ALREADY_INITIALIZED,
    SMSAPI_RETURN_CODE_NOT_INITIALIZED,

    // General use error return codes
    SMSAPI_RETURN_CODE_NOT_OWNER,
    SMSAPI_RETURN_CODE_NOT_FOUND,
    SMSAPI_RETURN_CODE_STILL_ACTIVE,
    SMSAPI_RETURN_CODE_OUT_OF_MEMORY,
    SMSAPI_RETURN_CODE_INVALID_INPUT,            //10
    SMSAPI_RETURN_CODE_BAD_ARGUMENT = SMSAPI_RETURN_CODE_INVALID_INPUT,
    SMSAPI_RETURN_CODE_INVALID_DECODER,
    SMSAPI_RETURN_CODE_UNSUPPORTED_API,
    SMSAPI_RETURN_CODE_API_NOT_ALLOWED,
    SMSAPI_RETURN_CODE_INVALID_OPTIONS,
    SMSAPI_RETURN_CODE_INSUFFICIENT_SIZE,
    SMSAPI_RETURN_CODE_PARTIAL_TRANSACTION,

    // content monitoring related codes
    SMSAPI_RETURN_CODE_CID_NOT_ALLOWED,
    SMSAPI_RETURN_CODE_INVALID_CID_TYPE,
    SMSAPI_RETURN_CODE_DUPLICATE_CONTENT,
    SMSAPI_RETURN_CODE_CONTENT_DOES_NOT_EXIST,  //20
    SMSAPI_RETURN_CODE_UNABLE_TO_CREATE_CONTENT,
    SMSAPI_RETURN_CODE_UNABLE_TO_ADD_CONTENT,

    // used for seek services
    SMSAPI_RETURN_CODE_WRONG_SEEK_SERVICE,

    // used in tuning attempts
    SMSAPI_RETURN_CODE_INVALID_CHANNEL,
    SMSAPI_RETURN_CODE_LOCKED_CHANNEL,
    SMSAPI_RETURN_CODE_MATURE_CHANNEL,

    // Service related return codes
    SMSAPI_RETURN_CODE_SERVICE_ALREADY_STARTED,
    SMSAPI_RETURN_CODE_SERVICE_NOT_STARTED,

    // Preset related return codes
    SMSAPI_RETURN_CODE_NEW_SERVICE,
    SMSAPI_RETURN_CODE_LOAD_SERVICE,             //30
    SMSAPI_RETURN_CODE_EMPTY_PRESET,

    // config related
    SMSAPI_RETURN_CODE_CFG_VALUE_TOO_LONG,
    SMSAPI_RETURN_CODE_CFG_NO_VALUE_SET,
    SMSAPI_RETURN_CODE_CFG_TAG_REMOVED,
    SMSAPI_RETURN_CODE_CFG_MALFORMED_FILE,
    SMSAPI_RETURN_CODE_CFG_FILE_BAD_CHECKSUM,
    SMSAPI_RETURN_CODE_CFG_BASE64_ERROR,
    SMSAPI_RETURN_CODE_CFG_NO_PARENT,

    // module firmware update error codes
    SMSAPI_RETURN_CODE_MODULE_FWUPDATE_FILE_VERSION_INVALID,
    SMSAPI_RETURN_CODE_MODULE_FWUPDATE_ALREADY_IN_PROGRESS,   //40

    // Catch-all error return code
    SMSAPI_RETURN_CODE_ERROR
} SMSAPI_RETURN_CODE_ENUM;

typedef enum smsapi_output_option_enum
{
    SMS_OUTPUT_OPTION_TERSE,
    SMS_OUTPUT_OPTION_VERBOSE,
    SMS_OUTPUT_OPTION_GROSS

}SMSAPI_OUTPUT_OPTION_ENUM;

// Channel/Category Navigation
typedef enum smsapi_direction_enum
{
    SMSAPI_DIRECTION_PREVIOUS = -1,
    SMSAPI_DIRECTION_DIRECT = 0,
    SMSAPI_DIRECTION_NEXT = 1

} SMSAPI_DIRECTION_ENUM;

typedef enum sms_language_enum
{
    SMS_LANGUAGE_ENGLISH,
    SMS_LANGUAGE_SPANISH,
    SMS_LANGUAGE_FRENCH,
    SMS_INVALID_LANGUAGE

} SMS_LANGUAGE_ENUM;

// SMS Object Event Mask
typedef int SMSAPI_EVENT_MASK;

typedef int SMS_BEHAVIOR_CODE;

// SMS Event Mask Modification actions
typedef enum smsapi_modify_event_mask_enum
{
    SMSAPI_MODIFY_EVENT_MASK_START,
    SMSAPI_MODIFY_EVENT_MASK_REPLACE = SMSAPI_MODIFY_EVENT_MASK_START,
    SMSAPI_MODIFY_EVENT_MASK_ENABLE,
    SMSAPI_MODIFY_EVENT_MASK_DISABLE,
    SMSAPI_MODIFY_EVENT_MASK_INVALID

} SMSAPI_MODIFY_EVENT_MASK_ENUM;

// SMS Object Update callback
typedef void (*SMSAPI_OBJECT_EVENT_CALLBACK)(
    void *hObj,
    SMSAPI_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

/* External object interface */
typedef struct sms_object_interface_struct
{
    /* Object Members */

    const char * (*pacGetVersion) (
        UN8 *pun8MajorVersion,
        UN8 *pun8MinorVersion,
        UN8 *pun8SubVersion
            );

    SMSAPI_RETURN_CODE_ENUM (*eInitialize) (
        const char *pacConfigPath,
        const char *pacConfigInitializerPath,
        const char *pacConfigInitializerFile,
        BOOLEAN *pbConfigFileReset
            );

    SMSAPI_RETURN_CODE_ENUM (*eUninitialize) (
        void
            );

    SMSAPI_RETURN_CODE_ENUM (*eBehavior) (
        SMS_BEHAVIOR_CODE eBehaviorCode,
        ...
            );

} SMS_OBJECT_INTERFACE_STRUCT;

/******************/
/*** SRM OBJECT ***/
/******************/

// SRM Object Event Mask
typedef SMSAPI_EVENT_MASK SRM_EVENT_MASK;

// SRM Object Update callback
typedef void (*SRM_OBJECT_EVENT_CALLBACK)(
    SRM_OBJECT hObj,
    SRM_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// SRM State
typedef enum srm_state_enum
{
    SRM_STATE_INITIAL,
    SRM_STATE_READY,
    SRM_STATE_STOPPED,
    SRM_STATE_ERROR,
    SRM_STATE_INVALID

} SRM_STATE_ENUM;

// SRM Error Codes
typedef enum srm_error_code_enum
{
    SRM_ERROR_CODE_NONE,
    SRM_ERROR_CODE_INVALID_OBJECT,
    SRM_ERROR_CODE_EVENT_POST_ERROR,
    SRM_ERROR_CODE_EVENT_ALLOCATION_ERROR,
    SRM_ERROR_CODE_NO_CAPABLE_MODULE,
    SRM_ERROR_CODE_INITIALIZATION_ERROR,
    SRM_ERROR_CODE_UNABLE_TO_REMOVE_SRM,
    SRM_ERROR_CODE_UNABLE_TO_LOCK_SMS,
    SRM_ERROR_CODE_MODULE_LIST_ERROR,
    SRM_ERROR_CODE_GENERAL,
    SRM_ERROR_CODE_UNKNOWN

} SRM_ERROR_CODE_ENUM;

/* External object interface */
typedef struct srm_object_interface_struct
{
    /* Object Members */

    SRM_OBJECT (*hGet) (
        const char *pacDriverName,
        const char *pacSRMName,
        SRM_EVENT_MASK tEventRequestMask,
        SRM_OBJECT_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vRelease) (
        SRM_OBJECT hSRM
            );

    SRM_STATE_ENUM (*eState) (
        SRM_OBJECT hSRM
            );

    SRM_ERROR_CODE_ENUM (*eErrorCode) (
        SRM_OBJECT hSRM
            );

    SRM_EVENT_MASK (*tEventMask) (
        SRM_OBJECT hSRM
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyRegisteredEventMask) (
        SRM_OBJECT hSRM,
        SRM_EVENT_MASK tEventMask,
        SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
            );

    BOOLEAN (*bUpdate) (
        SRM_OBJECT hSRM,
        SRM_EVENT_MASK tMask
            );

} SRM_OBJECT_INTERFACE_STRUCT;

  /*********************/
 /*** MODULE OBJECT ***/
/*********************/

// Module Object Event Mask
typedef SMSAPI_EVENT_MASK MODULE_EVENT_MASK;

// Update callback
typedef void (*MODULE_OBJECT_EVENT_CALLBACK)(
    MODULE_OBJECT hModule,
    MODULE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Module state
typedef enum module_state_enum
{
    MODULE_STATE_INITIAL,
    MODULE_STATE_UPDATING,
    MODULE_STATE_READY,
    MODULE_STATE_STOPPED,
    MODULE_STATE_ERROR,
    MODULE_STATE_INVALID

} MODULE_STATE_ENUM;

// MODULE Error Codes
typedef enum module_error_code_enum
{
    MODULE_ERROR_CODE_NONE,
    MODULE_ERROR_CODE_INVALID_OBJECT,
    MODULE_ERROR_CODE_INVALID_INPUT,
    MODULE_ERROR_CODE_EVENT_POST_ERROR,
    MODULE_ERROR_CODE_EVENT_ALLOCATION_ERROR,
    MODULE_ERROR_CODE_UNABLE_TO_RESET_SRM,
    MODULE_ERROR_CODE_NO_CAPABLE_DECODER,
    MODULE_ERROR_CODE_INITIALIZATION_ERROR,
    MODULE_ERROR_CODE_UNABLE_TO_REMOVE_MODULE,
    MODULE_ERROR_CODE_DECODER_LIST_ERROR,
    MODULE_ERROR_CODE_UNABLE_TO_STOP_DECODER,
    MODULE_ERROR_CODE_RADIO_OBJECT_ERROR,
    MODULE_ERROR_CODE_RADIO_REQUESTED_RESET,
    MODULE_ERROR_CODE_RADIO_COMM_FAILURE,
    MODULE_ERROR_CODE_RADIO_DEVICE_ERROR,
    MODULE_ERROR_CODE_SRM_ERROR,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_FAILED,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PROGRAMMING_TIMEOUT,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_CONFIGURE_FAILED,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_FAILED,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_ERASE_TIMEOUT,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_FAILED,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_PACKET_TIMEOUT,
    MODULE_ERROR_CODE_FIRMWARE_UPDATE_GENERAL,
    MODULE_ERROR_CODE_GENERAL,
    MODULE_ERROR_CODE_UNKNOWN

} MODULE_ERROR_CODE_ENUM;

typedef enum module_substatus_enum
{
    MODULE_SUBSTATUS_NOT_SUBSCRIBED = 0,
    MODULE_SUBSTATUS_SUBSCRIBED,
    MODULE_SUBSTATUS_SUSPEND_ALERT,
    MODULE_SUBSTATUS_SUSPENDED,
    MODULE_SUBSTATUS_INVALID

} MODULE_SUBSTATUS_ENUM;

typedef enum module_package_cmd_enum
{
    MODULE_PACKGE_CMD_SELECT,
    MODULE_PACKGE_CMD_QUERY,
    MODULE_PACKGE_CMD_VALIDATE,
    MODULE_PACKGE_CMD_REPORT,
    MODULE_PACKGE_CMD_INVALID,

} MODULE_PACKGE_CMD_ENUM;

typedef UN8 MODULE_SUBSTATUS_REASON_CODE;
#define MODULE_SUBSTATUS_REASON_CODE_INVALID 0xFF

#define MODULE_EVENT_DATA_MAX_SIZE          (10)
typedef struct module_event_data_struct
{
    UN16 aun16Data[MODULE_EVENT_DATA_MAX_SIZE];

} MODULE_EVENT_DATA_STRUCT;

/* External object interface */
typedef struct module_object_interface_struct
{
    /* Object Members */

    MODULE_OBJECT (*hGet) (
        SRM_OBJECT hSRM,
        const char *pacDecoderName,
        MODULE_EVENT_MASK tEventRequestMask,
        MODULE_OBJECT_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vRelease) (
        MODULE_OBJECT hModule
            );

    MODULE_STATE_ENUM (*eState) (
        MODULE_OBJECT hModule
            );

    MODULE_ERROR_CODE_ENUM (*eErrorCode) (
        MODULE_OBJECT hModule
            );

    MODULE_EVENT_MASK (*tEventMask) (
        MODULE_OBJECT hModule
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyRegisteredEventMask) (
        MODULE_OBJECT hModule,
        MODULE_EVENT_MASK tEventMask,
        SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
            );

    BOOLEAN (*bUpdate) (
        MODULE_OBJECT hModule,
        MODULE_EVENT_MASK tMask
            );

    N16 (*n16GmtOffsetGet) (
        MODULE_OBJECT hModule
            );

    SMSAPI_RETURN_CODE_ENUM (*eGmtOffsetSet) (
        MODULE_OBJECT hModule,
        N16 n16GmtOffset
            );

    BOOLEAN (*bDaylightSavingsTimeObservedGet) (
        MODULE_OBJECT hModule
            );

    SMSAPI_RETURN_CODE_ENUM (*eDaylightSavingsTimeObservedSet) (
        MODULE_OBJECT hModule,
        BOOLEAN bDaylightSavingsTimeObserved
            );

    STRING_OBJECT (*hESN) (
        MODULE_OBJECT hModule
            );

    SMSAPI_RETURN_CODE_ENUM (*eSubscription) (
        MODULE_OBJECT hModule,
        MODULE_SUBSTATUS_ENUM *peStatus,
        MODULE_SUBSTATUS_REASON_CODE *ptReasonCode,
        UN32 *pun32UTCSuspendDate,
        STRING_OBJECT hReasonText,
        STRING_OBJECT hPhoneNumber,
        BOOLEAN *pbIsAudioSubscribed
            );

    SMSAPI_RETURN_CODE_ENUM (*eFirmwareUpdate) (
        MODULE_OBJECT hModule,
        const char *pcFirmwareFileName
            );

    SMSAPI_RETURN_CODE_ENUM (*eUpdateProgress) (
        MODULE_OBJECT hModule,
        UN8 *pun8UpdateProgress
            );

    SMSAPI_RETURN_CODE_ENUM (*eEventData) (
        MODULE_OBJECT hModule,
        MODULE_EVENT_DATA_STRUCT *psEventData
            );

    SMSAPI_RETURN_CODE_ENUM (*ePackage) (
        MODULE_OBJECT hModule,
        MODULE_PACKGE_CMD_ENUM eCmd,
        UN8 un8PackageIndex,
        UN8 *pun8Resp,
        size_t tRespSize,
        size_t *ptBytesRxd
            );

} MODULE_OBJECT_INTERFACE_STRUCT;

  /*****************************/
 /*** MODULE VERSION OBJECT ***/
/*****************************/

typedef struct module_version_object_interface_struct
{
    /* Object Members */
    STRING_OBJECT (*hTypeID) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hHardwareRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hSoftwareRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hProtocolRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hBaseBandRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hDecoderRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hRFRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hSPLRevision) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hCapability) (
        MODULE_OBJECT hModule
            );

    STRING_OBJECT (*hModuleHwName) (
        MODULE_OBJECT hModule
            );

    SMSAPI_RETURN_CODE_ENUM (*eFWUpdateFileVersion) (
        const char *pcFile,
        STRING_OBJECT  hFileVersion,
        STRING_OBJECT  hFileEarliestVersion,
        STRING_OBJECT  hFileLatestVersion,
        STRING_OBJECT  hFileTypeID
            );

    N32 (*n32FPrintf) (
        MODULE_OBJECT hModule,
        FILE *psFile
            );

} MODULE_VERSION_OBJECT_INTERFACE_STRUCT;

/**********************/
/*** DECODER OBJECT ***/
/**********************/

// Decoder Object Event Mask
typedef SMSAPI_EVENT_MASK DECODER_EVENT_MASK;

// Channel Object Event Mask
typedef SMSAPI_EVENT_MASK CHANNEL_EVENT_MASK;

// Update callback
typedef void (*DECODER_OBJECT_EVENT_CALLBACK)(
    DECODER_OBJECT hDecoder,
    DECODER_EVENT_MASK tEventMask,
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
            );

typedef void (*DECODER_CHANNEL_ACCESS_CALLBACK)(
    DECODER_OBJECT hDecoder,
    CHANNEL_OBJECT hChannel,
    void *pvChannelCallbackArg
            );

typedef void (*DECODER_CATEGORY_ACCESS_CALLBACK)(
    DECODER_OBJECT hDecoder,
    CATEGORY_OBJECT hCategory,
    void *pvCategoryCallbackArg
            );

typedef BOOLEAN (*BROWSE_CHANNEL_COMPARE_HANDLER)(
        DECODER_OBJECT hDecoder,
        CHANNEL_OBJECT hChannel,
        void *pvIterateArg
            );

typedef BOOLEAN (*BROWSE_CATEGORY_COMPARE_HANDLER)(
        DECODER_OBJECT hDecoder,
        CATEGORY_OBJECT hCategory,
        void *pvIterateArg,
        N16 *pn16Offset
            );

typedef BOOLEAN (*CHANNEL_OBJECT_ITERATOR)(
    CHANNEL_OBJECT hChannel,
    void *pvIteratorArg
            );

typedef BOOLEAN (*CHANNEL_OBJECT_ITERATOR_INCLUDE_HANDLER)(
    CHANNEL_OBJECT hChannel,
    void *pvIteratorArg
        );

typedef BOOLEAN (*CATEGORY_OBJECT_ITERATOR)(
    CATEGORY_OBJECT hCategory,
    void *pvIteratorArg
            );

typedef BOOLEAN (*CATEGORY_OBJECT_ITERATOR_INCLUDE_HANDLER)(
    CATEGORY_OBJECT hCategory,
    void *pvIteratorArg
        );

typedef enum signal_state_enum
{
    SIGNAL_STATE_ACQUIRED,
    SIGNAL_STATE_ACQUIRING,
    SIGNAL_STATE_ANTENNA_ERROR,
    SIGNAL_STATE_UNKNOWN

} SIGNAL_STATE_ENUM;

typedef enum signal_quality_enum
{
    SIGNAL_QUALITY_NONE = 0,
    SIGNAL_QUALITY_WEAK = 1,
    SIGNAL_QUALITY_GOOD = 2,
    SIGNAL_QUALITY_EXCELLENT = 3,
    SIGNAL_QUALITY_INVALID = UN8_MAX

} SIGNAL_QUALITY_ENUM;

typedef enum detailed_signal_strength_enum
{
    DETAILED_SIGNAL_QUALITY_NONE = 0,
    DETAILED_SIGNAL_QUALITY_VERY_WEAK = 1,
    DETAILED_SIGNAL_QUALITY_WEAK = 2,
    DETAILED_SIGNAL_QUALITY_MARGINAL = 3,
    DETAILED_SIGNAL_QUALITY_GOOD = 4,
    DETAILED_SIGNAL_QUALITY_INVALID = UN8_MAX

} DETAILED_SIGNAL_STRENGTH_ENUM;

typedef struct antenna_aiming_signal_struct
{
    UN8 un8SatPercentSigLevel;
    UN8 un8TerPercentSigLevel;

} ANTENNA_AIMING_SIGNAL_STRUCT;

typedef struct signal_quality_struct
{
    SIGNAL_STATE_ENUM eState;
    SIGNAL_QUALITY_ENUM eComposite;
    SIGNAL_QUALITY_ENUM eSatellite;
    SIGNAL_QUALITY_ENUM eTerrestrial;
    ANTENNA_AIMING_SIGNAL_STRUCT sAntennaAimingData;

} SIGNAL_QUALITY_STRUCT;

// Antenna State
typedef enum antenna_state_enum
{
    ANTENNA_STATE_UNSUPPORTED,
    ANTENNA_STATE_DETECTED,
    ANTENNA_STATE_NOT_DETECTED,
    ANTENNA_STATE_SHORTED,
    ANTENNA_STATE_UNKNOWN

} ANTENNA_STATE_ENUM;

// Channel Browse style
typedef enum channel_browse_style_enum
{
    CHANNEL_BROWSE_STYLE_ALL_CHANNELS,
    CHANNEL_BROWSE_STYLE_WITHIN_CATEGORY,
    CHANNEL_BROWSE_STYLE_INVALID

} CHANNEL_BROWSE_STYLE_ENUM;

// Decoder Tune State
typedef enum tune_state_enum
{
    TUNE_STATE_TUNED,
    TUNE_STATE_TUNING_IN_PROGRESS,

    // Initialized state
    TUNE_STATE_UNKNOWN,

    // Invalid state
    TUNE_STATE_INVALID

} TUNE_STATE_ENUM;

// Decoder State
typedef enum decoder_state_enum
{
    DECODER_STATE_INITIAL,
    DECODER_STATE_UPDATING,
    DECODER_STATE_READY,
    DECODER_STATE_RELEASED,
    DECODER_STATE_ERROR,
    DECODER_STATE_INVALID

} DECODER_STATE_ENUM;

// Decoder Error Codes
typedef enum decoder_error_code_enum
{
    DECODER_ERROR_CODE_NONE,
    DECODER_ERROR_CODE_INVALID_OBJECT,
    DECODER_ERROR_CODE_EVENT_POST_ERROR,
    DECODER_ERROR_CODE_EVENT_ALLOCATION_ERROR,
    DECODER_ERROR_CODE_UNABLE_TO_RESET_MODULE,
    DECODER_ERROR_CODE_INITIALIZATION_ERROR,
    DECODER_ERROR_CODE_RADIO_ERROR,
    DECODER_ERROR_CODE_UNABLE_TO_REMOVE_DECODER,
    DECODER_ERROR_CODE_CCACHE_ERROR,
    DECODER_ERROR_CODE_MODULE_ERROR,
    DECODER_ERROR_CODE_TIMER_ERROR,
    DECODER_ERROR_CODE_GENERAL,
    DECODER_ERROR_CODE_UNKNOWN

} DECODER_ERROR_CODE_ENUM;

typedef enum decoder_tone_generation_balance_enum
{
    DECODER_TONE_GENERATION_BALANCE_LEFT,
    DECODER_TONE_GENERATION_BALANCE_RIGHT,
    DECODER_TONE_GENERATION_BALANCE_BOTH

} DECODER_TONE_GENERATION_BALANCE_ENUM;

typedef enum decode_tune_scan_style_enum
{
    DECODER_TUNE_SCAN_STYLE_SMART_FAVORITES,
    DECODER_TUNE_SCAN_STYLE_ALL_CHANNELS

} DECODER_TUNE_SCAN_STYLE_ENUM;

typedef enum decoder_tune_scan_cmd_enum
{
    DECODER_TUNE_SCAN_CMD_START,
    DECODER_TUNE_SCAN_CMD_STOP,
    DECODER_TUNE_SCAN_CMD_ABORT,
    DECODER_TUNE_SCAN_CMD_SKIP_FORWARD,
    DECODER_TUNE_SCAN_CMD_SKIP_BACKWARD

} DECODER_TUNE_SCAN_CMD_ENUM;

// Decoder Tune Scan Status Mask
typedef UN8 DECODER_TUNE_SCAN_STATUS_MASK;

#define DECODER_TUNE_SCAN_STATUS_NONE               (0x00)
#define DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE  (0x01)
#define DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED       (0x02)

#define DECODER_TUNE_SCAN_STATUS_ALL                (   \
       (DECODER_TUNE_SCAN_STATUS_MASK)                  \
        DECODER_TUNE_SCAN_STATUS_CONTENT_AVAILABLE |    \
        DECODER_TUNE_SCAN_STATUS_SCAN_ABORTED               )

// Audio Presence status
typedef enum audio_presence_enum
{
    DECODER_AUDIO_PRESENCE_AUDIO_NOT_PRESENT,
    DECODER_AUDIO_PRESENCE_AUDIO_PRESENT,
    DECODER_AUDIO_PRESENCE_UNKNOWN

} DECODER_AUDIO_PRESENCE_ENUM;

/* Decoder ID definition */
typedef UN8 DECODER_ID;
#define DECODER_ID_INVALID  (DECODER_ID)0xFF

#define SMSAPI_AUDIO_LEVEL_MUTE (N16)N16_MIN

/* External object interface */
typedef struct decoder_object_interface_struct
{
    /* Object Members */

    DECODER_OBJECT (*hGet) (
        SRM_OBJECT hSRM,
        const char *pacDecoderName,
        DECODER_EVENT_MASK tEventRequestMask,
        DECODER_OBJECT_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vRelease) (
        DECODER_OBJECT hDecoder
            );

    DECODER_STATE_ENUM (*eState) (
        DECODER_OBJECT hDecoder
            );

    DECODER_ERROR_CODE_ENUM (*eErrorCode) (
        DECODER_OBJECT hDecoder
            );

    DECODER_EVENT_MASK (*tEventMask) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyRegisteredEventMask) (
        DECODER_OBJECT hDecoder,
        DECODER_EVENT_MASK tEventMask,
        SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
            );

    BOOLEAN (*bUpdate) (
        DECODER_OBJECT hDecoder,
        DECODER_EVENT_MASK tMask
            );

    SMSAPI_RETURN_CODE_ENUM (*eTuneDirect) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN bOverride
            );

    TUNE_STATE_ENUM (*eTuneState) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eSignalQuality) (
        DECODER_OBJECT hDecoder,
        SIGNAL_QUALITY_STRUCT *psSignalQuality
            );

    N8 (*n8NumAntennas) (
        DECODER_OBJECT hDecoder
            );

    ANTENNA_STATE_ENUM (*eAntennaState) (
        DECODER_OBJECT hDecoder,
        N8 n8Antenna
            );

    SMSAPI_RETURN_CODE_ENUM (*eUpdateProgress)  (
        DECODER_OBJECT hDecoder,
        UN8 *pun8UpdateProgress
            );

    CHANNEL_ID (*tGetChannelId) (
        DECODER_OBJECT hDecoder,
        SERVICE_ID tServiceId
            );

    CHANNEL_ID (*tCurrentChannelId) (
        DECODER_OBJECT hDecoder
            );

    SERVICE_ID (*tCurrentServiceId) (
        DECODER_OBJECT hDecoder
            );

    CATEGORY_ID (*tCurrentCategoryId) (
        DECODER_OBJECT hDecoder
            );

    CHANNEL_ID (*tLastTunedChannelId) (
        DECODER_OBJECT hDecoder
            );

    CATEGORY_ID (*tLastTunedCategoryId) (
        DECODER_OBJECT hDecoder
            );

    CHANNEL_ID (*tBrowsedChannelId) (
        DECODER_OBJECT hDecoder
            );

    CATEGORY_ID (*tBrowsedCategoryId) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseChannelStyleSet) (
        DECODER_OBJECT hDecoder,
        BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
        void *pvBrowseChannelCompareArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseChannel) (
        DECODER_OBJECT hDecoder,
        SMSAPI_DIRECTION_ENUM eDirection
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseToChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseCategoryStyleSet) (
        DECODER_OBJECT hDecoder,
        BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
        void *pvBrowseCategoryCompareArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseCategory) (
        DECODER_OBJECT hDecoder,
        SMSAPI_DIRECTION_ENUM eDirection
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseToCategory) (
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId
            );

    SMSAPI_RETURN_CODE_ENUM (*eUseChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        DECODER_CHANNEL_ACCESS_CALLBACK vChannelAccessCallback,
        void *pvChannelAccessCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eUseCategory) (
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        DECODER_CATEGORY_ACCESS_CALLBACK vCategoryAccessCallback,
        void *pvCategoryAccessCallbackArg
            );

    BOOLEAN (*bLockChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId
            );

    BOOLEAN (*bUnlockChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId
            );

    BOOLEAN (*bSkipChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId
            );

    BOOLEAN (*bUnskipChannel) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsChannelLocked) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN *pbLocked
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsChannelSkipped) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN *pbSkipped
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsChannelSubscribed) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN *pbSubscribed
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsChannelMature) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN *pbMature
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsChannelFreeToAir) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        BOOLEAN *pbFreeToAir
            );

    PLAYBACK_OBJECT (*hPlayback) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetUnsubscribedText) (
        DECODER_OBJECT hDecoder,
        const char *pacUnsubscribedText,
        CDO_FIELD_MASK tMask
            );

    SMSAPI_RETURN_CODE_ENUM (*eStartToneGeneration) (
        DECODER_OBJECT hDecoder,
        UN32 un32ToneFreqHz,
        N8 n8Volume,
        DECODER_TONE_GENERATION_BALANCE_ENUM eBalance
            );

    SMSAPI_RETURN_CODE_ENUM (*eStopToneGeneration) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eAlertToneGeneration) (
        DECODER_OBJECT hDecoder,
        N8 n8Volume,
        int iCmd,
        ...
            );

    SMSAPI_RETURN_CODE_ENUM (*eAudio) (
        DECODER_OBJECT hDecoder,
        N16 n16Level
            );

    SMSAPI_RETURN_CODE_ENUM (*eTuneScanConfigure ) (
        DECODER_OBJECT hDecoder,
        UN8 un8PlaySeconds,
        BOOLEAN bScanLockedMature,
        BOOLEAN bScanSkipped,
        DECODER_TUNE_SCAN_STYLE_ENUM eScanStyle
            );

    SMSAPI_RETURN_CODE_ENUM (*eTuneScan ) (
        DECODER_OBJECT hDecoder,
        DECODER_TUNE_SCAN_CMD_ENUM eScanCmd
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsTuneScanContentAvailable ) (
        DECODER_OBJECT hDecoder,
        DECODER_TUNE_SCAN_STATUS_MASK *ptStatusMask
            );

    DECODER_AUDIO_PRESENCE_ENUM (*eAudioPresence) (
        DECODER_OBJECT hDecoder
            );

} DECODER_OBJECT_INTERFACE_STRUCT;

  /*******************************/
 /*** ENGINEERING DATA OBJECT ***/
/*******************************/

// Engineering Data Object Event Mask
typedef SMSAPI_EVENT_MASK ENGINEERING_DATA_EVENT_MASK;

typedef UN8 AUDIO_DECODER_BITRATE;

typedef struct engineering_data_object_interface_struct
{
    /* Object Members */

    ENGINEERING_DATA_EVENT_MASK (*tEventMask) (
        DECODER_OBJECT hDecoder
            );

    DETAILED_SIGNAL_QUALITY_OBJECT (*hSignalQualityDetailed) (
        DECODER_OBJECT hDecoder
            );

    DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT (*hOverlaySignalQualityDetailed) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eAudioDecoderBitrate) (
        DECODER_OBJECT hDecoder,
        AUDIO_DECODER_BITRATE *ptBitrate
            );

    LINK_STATUS_INFORMATION_OBJECT (*hLinkStatusInformation) (
        DECODER_OBJECT hDecoder
            );

    N32 (*n32FPrintf) (
        DECODER_OBJECT hDecoder,
        FILE *psFile
            );

} ENGINEERING_DATA_OBJECT_INTERFACE_STRUCT;


  /**************************************/
 /*** DETAILED SIGNAL QUALITY OBJECT ***/
/**************************************/

typedef struct detailed_signal_quality_object_interface_struct
{
    /* Object Members */
    STRING_OBJECT (*hSigStrength) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hTunerStatus) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hENSALockStatus) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hENSBLockStatus) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hBERS1) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hBERS2) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hBERT) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hCNS1A) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hCNS1B) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hCNS2A) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hCNS2B) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hRSErrsWord) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hRSErrsSatSymb) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hRSErrsTerrSymb) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hTunerCarrierFreqOffset) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    STRING_OBJECT (*hRSSI) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality
            );

    N32 (*n32FPrintf) (
        DETAILED_SIGNAL_QUALITY_OBJECT hDetSigQuality,
        FILE *psFile
            );

} DETAILED_SIGNAL_QUALITY_OBJECT_INTERFACE_STRUCT;

  /**********************************************/
 /*** DETAILED OVERLAY SIGNAL QUALITY OBJECT ***/
/**********************************************/

typedef struct detailed_overlay_signal_quality_object_interface_struct
{
    /* Object Members */
    STRING_OBJECT (*hReceiverState) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERS1A) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERS1B) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERS2A) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERS2B) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTerrA) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTerrB) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate0A) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate1A) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate2A) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate0B) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate1B) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    STRING_OBJECT (*hOBERTurboWordErrRate2B) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality
            );

    N32 (*n32FPrintf) (
        DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT hDetOverlaySigQuality,
        FILE *psFile
            );

} DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT_INTERFACE_STRUCT;


  /**************************************/
 /*** LINK STATUS INFORMATION OBJECT ***/
/**************************************/

typedef UN32 NUMBER_CONFIRMATION_TIMEOUTS;
typedef UN32 PACKET_COUNTER_NUMBER;

typedef struct link_status_information_object_interface_struct
{
    /* Object Members */
    SMSAPI_RETURN_CODE_ENUM (*eNumberConfirmationTimeouts) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        NUMBER_CONFIRMATION_TIMEOUTS *ptNumConfirmTimeouts
            );

    SMSAPI_RETURN_CODE_ENUM (*eNumberDataPacketsDropped) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        PACKET_COUNTER_NUMBER *ptDataPacketsDropped
            );

    SMSAPI_RETURN_CODE_ENUM (*eNumberDataPacketsTransmitted) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        PACKET_COUNTER_NUMBER *ptDataPacketsTransmitted
            );

    SMSAPI_RETURN_CODE_ENUM (*eNumberAudioPacketsDropped) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        PACKET_COUNTER_NUMBER *ptAudioPacketsDropped
            );

    SMSAPI_RETURN_CODE_ENUM (*eNumberAudioPacketsTransmitted) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        PACKET_COUNTER_NUMBER *ptAudioPacketsTransmitted
            );

    N32 (*n32FPrintf) (
        LINK_STATUS_INFORMATION_OBJECT hLinkStatusInfo,
        FILE *psFile
            );

} LINK_STATUS_INFORMATION_OBJECT_INTERFACE_STRUCT;

  /**********************/
 /*** SONG OBJECT ***/
/**********************/
typedef enum smsapi_song_status_enum
{
    SMSAPI_SONG_STATUS_UNKNOWN,
    SMSAPI_SONG_STATUS_INCOMPLETE,
    SMSAPI_SONG_STATUS_COMPLETE
} SMSAPI_SONG_STATUS_ENUM;

// Song Object Event Mask
typedef SMSAPI_EVENT_MASK SONG_EVENT_MASK;

typedef UN32 SONG_ID;
#define SONG_MAX_ID UN16_MAX
#define SONG_INVALID_ID SONG_MAX_ID+2
#define SONG_LIVE_ID SONG_MAX_ID+1

/* External object interface */
typedef struct song_object_interface_struct
{
    /* Object Members */

    SONG_EVENT_MASK (*tEventMask) (
        SONG_OBJECT hSong
            );

    STRING_OBJECT (*hArtist) (
        SONG_OBJECT hSong
            );

    STRING_OBJECT (*hTitle) (
        SONG_OBJECT hSong
            );

    SMSAPI_RETURN_CODE_ENUM (*eTimeStamp) (
        SONG_OBJECT hSong,
        UN32 *pun32TimeStamp
            );

    SMSAPI_RETURN_CODE_ENUM (*eDuration) (
        SONG_OBJECT hSong,
        UN32 *pun32Duration
            );

    SMSAPI_RETURN_CODE_ENUM (*eChannelId) (
        SONG_OBJECT hSong,
        CHANNEL_ID *ptChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eStatus) (
        SONG_OBJECT hSong,
        SMSAPI_SONG_STATUS_ENUM *peStatus
            );

} SONG_OBJECT_INTERFACE_STRUCT;

  /**********************/
 /*** SONGLIST OBJECT ***/
/**********************/

// Song Object Event Mask
typedef SMSAPI_EVENT_MASK SONGLIST_OBJECT_EVENT_MASK;


// Update callback
typedef void (*SONGLIST_OBJECT_EVENT_CALLBACK)(
    SONGLIST_OBJECT hSongList,
    SONGLIST_OBJECT_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

typedef enum songlist_type_enum
{
    SONGLIST_TYPE_TIME_SHIFTED_CONTENT,
    SONGLIST_TYPE_INVALID

} SONGLIST_TYPE_ENUM;

/* External object interface */
typedef struct songlist_object_interface_struct
{
    /* Object Members */

    SONGLIST_OBJECT (*hCreate) (
        PLAYBACK_OBJECT hPlayback,
        SONGLIST_TYPE_ENUM eSongListType,
        UN16 un16Before,
        UN16 un16After,
        SONG_EVENT_MASK tEventRequestMask,
        SONGLIST_OBJECT_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        ...
            );

    void (*vDestroy) (
        SONGLIST_OBJECT hSongliist
            );

    SONG_OBJECT (*hSong) (
        SONGLIST_OBJECT hSongliist,
        N16 n16Offset
            );

    SMSAPI_RETURN_CODE_ENUM (*eBrowseList) (
        SONGLIST_OBJECT hSonglist,
        N16 n16Offset
            );

    SMSAPI_RETURN_CODE_ENUM (*eSize) (
        SONGLIST_OBJECT hSonglist,
        size_t *ptSize,
        UN16 *pun16Before,
        UN16 *pun16After
        );

    SONGLIST_OBJECT_EVENT_MASK (*tEventMask) (
        SONGLIST_OBJECT hSonglist
            );

} SONGLIST_OBJECT_INTERFACE_STRUCT;

  /**********************/
 /*** CHANNEL OBJECT ***/
/**********************/

typedef UN16 CHANNEL_ACO;
#define CHANNEL_ACO_DEFAULT  ((CHANNEL_ACO)UN16_MAX)

// Play On Select Method
typedef enum play_on_select_method_enum
{
    CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_A,
    CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_B,
    CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_C,
    CHANNEL_PLAY_ON_SELECT_METHOD_NEWEST_TYPE_D,
    CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_A,
    CHANNEL_PLAY_ON_SELECT_METHOD_CONSTRAINED_TYPE_B,
    CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_A,
    CHANNEL_PLAY_ON_SELECT_METHOD_REALTIME_TYPE_B

} CHANNEL_PLAY_ON_SELECT_METHOD_ENUM;

// iterator callback
typedef BOOLEAN (*CHANNEL_SIMILAR_ITERATOR_CALLBACK)(
    CHANNEL_OBJECT hChannel,
    void *pvEventCallbackArg
        );

typedef BOOLEAN (*PROGRAM_ITERATOR_CALLBACK)(
    PROGRAM_OBJECT hEProgEvent,
    void *pvEventCallbackArg
        );

typedef BOOLEAN (*EPG_TOPIC_ITERATOR_CALLBACK)(
    TOPIC_OBJECT hEpgTopic,
    void *pvEventCallbackArg
        );

// Update callback
typedef void (*CHANNEL_OBJECT_EVENT_CALLBACK)(
  CHANNEL_OBJECT hChannel,
  CHANNEL_EVENT_MASK tEventMask,
  void *pvEventCallbackArg
      );

// Seek content access callback
typedef void (*SEEK_CONTENT_ACCESS_CALLBACK)(
    SEEK_CONTENT_OBJECT hSeekContent,
    void *pvSeekContentCallbackArg
            );

/* External object interface */
typedef struct channel_object_interface_struct
{
    /* Object Members */

    CHANNEL_EVENT_MASK (*tEventMask) (
        CHANNEL_OBJECT hChannel
    );

    SERVICE_ID (*tServiceId) (
        CHANNEL_OBJECT hChannel
    );

    CHANNEL_ID (*tChannelId) (
        CHANNEL_OBJECT hChannel
    );

    STRING_OBJECT (*hShortName) (
        CHANNEL_OBJECT hChannel
    );

    STRING_OBJECT (*hMediumName) (
        CHANNEL_OBJECT hChannel
    );

    STRING_OBJECT (*hLongName) (
        CHANNEL_OBJECT hChannel
    );

    STRING_OBJECT (*hShortDescription) (
        CHANNEL_OBJECT hChannel
    );

    STRING_OBJECT (*hLongDescription) (
        CHANNEL_OBJECT hChannel
    );

    SMSAPI_RETURN_CODE_ENUM (*eCategoryOffset)(
        CHANNEL_OBJECT hChannel,
        CATEGORY_ID tCategoryId,
        N16 *pn16Offset
    );

    SMSAPI_RETURN_CODE_ENUM (*eNumCategories)(
        CHANNEL_OBJECT hChannel,
        N16 *pn16NumCategories
    );

    CATEGORY_OBJECT (*hCategory) (
        CHANNEL_OBJECT hChannel,
        N16 n16Offset
    );

    CD_OBJECT (*hCDO) (
        CHANNEL_OBJECT hChannel
    );

    CHANNEL_ART_OBJECT (*hArt) (
        CHANNEL_OBJECT hChannel
    );

    SMSAPI_RETURN_CODE_ENUM (*ePreset) (
        CHANNEL_OBJECT hChannel,
        PRESET_BAND_OBJECT *phBand,
        STRING_OBJECT *phPresetName,
        size_t *ptPresetIndex
    );

    SMSAPI_RETURN_CODE_ENUM (*eIsLocked) (
        CHANNEL_OBJECT hChannel,
        BOOLEAN *pbLocked
    );

    SMSAPI_RETURN_CODE_ENUM (*eIsSkipped) (
        CHANNEL_OBJECT hChannel,
        BOOLEAN *pbSkipped
    );

    SMSAPI_RETURN_CODE_ENUM (*eIsSubscribed) (
        CHANNEL_OBJECT hChannel,
        BOOLEAN *pbSubscribed
    );

    SMSAPI_RETURN_CODE_ENUM (*eIsMature) (
        CHANNEL_OBJECT hChannel,
        BOOLEAN *pbMature
    );

    SMSAPI_RETURN_CODE_ENUM (*eIsFreeToAir) (
        CHANNEL_OBJECT hChannel,
        BOOLEAN *pbFreeToAir
    );


    SMSAPI_RETURN_CODE_ENUM (*eNotifyOnChange) (
        DECODER_OBJECT hDecoder,
        CHANNEL_OBJECT_EVENT_CALLBACK vNotifierCallback,
        void *pvNotifierCallbackArg
    );

    SMSAPI_RETURN_CODE_ENUM (*eIterateAll) (
        DECODER_OBJECT hDecoder,
        CHANNEL_OBJECT_ITERATOR bIterator,
        CHANNEL_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude,
        void *pvArg
    );

    SMSAPI_RETURN_CODE_ENUM (*eIterateSimilarChannels) (
        CHANNEL_OBJECT hChannel,
        CHANNEL_SIMILAR_ITERATOR_CALLBACK bSimilarIterator,
        void *pvIteratorArg
    );

    SMSAPI_RETURN_CODE_ENUM (*eIteratePrograms) (
        CHANNEL_OBJECT hChannel,
        PROGRAM_ITERATOR_CALLBACK bEpgIterator,
        void *pvIteratorArg
    );

    CHANNEL_ACO (*tACO ) (
        CHANNEL_OBJECT hChannel
            );

    BOOLEAN (*bQualifiedForTuneMix) (
        CHANNEL_OBJECT hChannel
            );

    SMSAPI_RETURN_CODE_ENUM (*ePlayOnSelect) (
        CHANNEL_OBJECT hChannel,
        CHANNEL_PLAY_ON_SELECT_METHOD_ENUM *pePlayOnSelect
            );

} CHANNEL_OBJECT_INTERFACE_STRUCT;

/**************************/
/*** CHANNELLIST OBJECT ***/
/**************************/

// Update callback
typedef void (*CHANNELLIST_OBJECT_EVENT_CALLBACK)(
  CHANNELLIST_OBJECT hChannelList,
  CHANNEL_EVENT_MASK tEventMask,
  void *pvEventCallbackArg
      );

// Browse type
typedef enum browse_type_enum
{
  BROWSE_TYPE_CATEGORY,
  BROWSE_TYPE_ALL_CHANNELS,
  BROWSE_TYPE_INVALID

} BROWSE_TYPE_ENUM;

/* External object interface */
typedef struct channellist_object_interface_struct
{
  /* Object Members */

  CHANNELLIST_OBJECT (*hCreate) (
      DECODER_OBJECT hDecoder,
      BROWSE_TYPE_ENUM eBrowseType,
      CHANNEL_ID tReferenceChannelId,
      CATEGORY_ID tReferenceCategoryId,
      UN16 un16Before,
      UN16 un16After,
      CHANNEL_EVENT_MASK tEventRequestMask,
      CHANNELLIST_OBJECT_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
      void *pvBrowseChannelCompareArg
          );

  void (*vDestroy) (
      CHANNELLIST_OBJECT hChannelList
          );

  BOOLEAN (*bUpdate) (
      CHANNELLIST_OBJECT hChannelList,
      CHANNEL_EVENT_MASK tMask
          );

  SMSAPI_RETURN_CODE_ENUM (*eModifyRegisteredEventMask) (
      CHANNELLIST_OBJECT hChannelList,
      CHANNEL_EVENT_MASK tEventMask,
      SMSAPI_MODIFY_EVENT_MASK_ENUM eModification
          );

  CHANNEL_OBJECT (*hChannel) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16Offset
          );

  CHANNEL_ID (*tChannelId) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16Offset
          );

  CHANNEL_EVENT_MASK (*tChannelEventMask) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16Offset
          );

  CATEGORY_OBJECT (*hCategory) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16Offset
          );

  CATEGORY_ID (*tCategoryId) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16Offset
          );

  SMSAPI_RETURN_CODE_ENUM (*eBrowseList) (
      CHANNELLIST_OBJECT hChannelList,
      N16 n16BrowseOffset
          );

  SMSAPI_RETURN_CODE_ENUM (*eBrowseStyleSet)(
      CHANNELLIST_OBJECT hChannelList,
      BROWSE_CATEGORY_COMPARE_HANDLER bBrowseCategoryCompareHandler,
      void *pvBrowseCategoryCompareArg,
      BROWSE_CHANNEL_COMPARE_HANDLER bBrowseChannelCompareHandler,
      void *pvBrowseChannelCompareArg
          );

  SMSAPI_RETURN_CODE_ENUM (*eBrowseCategory) (
      CHANNELLIST_OBJECT hChannelList,
      SMSAPI_DIRECTION_ENUM eDirection
          );

  SMSAPI_RETURN_CODE_ENUM (*eBrowseToCategory) (
      CHANNELLIST_OBJECT hChannelList,
      CATEGORY_ID tCategoryId
          );

  size_t (*tSize) (
      CHANNELLIST_OBJECT hChannelList,
      UN16 *pun16Before,
      UN16 *pun16After
          );

} CHANNELLIST_OBJECT_INTERFACE_STRUCT;

  /***********************/
 /*** CATEGORY OBJECT ***/
/***********************/

// Category type enumeration
typedef enum category_type_enum
{
   CATEGORY_TYPE_BROADCAST,
   CATEGORY_TYPE_USER,
   CATEGORY_TYPE_VIRTUAL,
   CATEGORY_TYPE_INVALID

} CATEGORY_TYPE_ENUM;

// Category Object Event Mask
typedef SMSAPI_EVENT_MASK CATEGORY_EVENT_MASK;

// Category callbacks
typedef N16 (*CATEGORY_ITERATION_HANDLER)(
    CATEGORY_OBJECT hCategory,
    CATEGORY_CHANNEL_INDEX tCurrentIndex,
    CHANNEL_OBJECT hChannel,
    void *pvIterateArg
        );

// Category Evente callback
typedef void (*CATEGORY_OBJECT_EVENT_CALLBACK)(
    CATEGORY_OBJECT hCategory,
    CATEGORY_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

typedef N16 (*CATEGORY_SORT_HANDLER)(
    CHANNEL_OBJECT hChan1,
    CHANNEL_OBJECT hChan2
        );

/* External object interface */
typedef struct category_object_interface_struct
{
    /* Object Members */

    CATEGORY_ID (*tGetCategoryId)(
        CATEGORY_OBJECT hCategory
            );

    USER_ID (*tGetUserId)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetUserId)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        USER_ID tUserId
            );

    size_t (*tSize)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId
            );

    CATEGORY_EVENT_MASK (*tEventMask) (
        CATEGORY_OBJECT hCategory
            );

    STRING_OBJECT (*hShortName)(
        CATEGORY_OBJECT hCategory
            );

    STRING_OBJECT (*hMediumName)(
        CATEGORY_OBJECT hCategory
            );

    STRING_OBJECT (*hLongName)(
        CATEGORY_OBJECT hCategory
            );

    SMSAPI_RETURN_CODE_ENUM (*eRename) (
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        const char *pacNewLongName,
        const char *pacNewShortName
            );

    CATEGORY_TYPE_ENUM (*eType) (
        CATEGORY_OBJECT hCategory
            );

    CHANNEL_ART_OBJECT (*hArt) (
        CATEGORY_OBJECT hCategory
            );

    CATEGORY_ID (*tCreate)(
        DECODER_OBJECT hDecoder,
        const char *pacLongName,
        const char *pacShortName,
        CATEGORY_CHANNEL_INDEX tInitialNumChannels
            );

    void (*vDestroy)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        CATEGORY_ITERATION_HANDLER n16IterateFxn,
        void *pvIterateArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eInsertNewChannel)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eInsertAfterChannel)(
        CATEGORY_OBJECT hCategory,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eInsertBeforeChannel)(
        CATEGORY_OBJECT hCategory,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eReplaceChannel)(
        CATEGORY_OBJECT hCategory,
        CHANNEL_ID tChannelId
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveChannel)(
        CATEGORY_OBJECT hCategory
            );

    SMSAPI_RETURN_CODE_ENUM (*eSort)(
        DECODER_OBJECT hDecoder,
        CATEGORY_ID tCategoryId,
        CATEGORY_SORT_HANDLER n16SortFxn
            );

    SMSAPI_RETURN_CODE_ENUM (*eNotifyOnChange) (
        DECODER_OBJECT hDecoder,
        CATEGORY_OBJECT_EVENT_CALLBACK vNotifierCallback,
        void *pvNotifierCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateAll) (
        DECODER_OBJECT hDecoder,
        CATEGORY_OBJECT_ITERATOR bIterator,
        CATEGORY_OBJECT_ITERATOR_INCLUDE_HANDLER bInclude,
        void *pvArg
            );

} CATEGORY_OBJECT_INTERFACE_STRUCT;

  /******************************/
 /*** SMART FAVORITES OBJECT ***/
/******************************/

// Advanced IR Play Point Ctrl
typedef enum smart_favorites_play_point_ctrl_enum
{
    SMART_FAVORITES_PLAY_POINT_CTRL_LIVE,
    SMART_FAVORITES_PLAY_POINT_CTRL_START,
    SMART_FAVORITES_PLAY_POINT_CTRL_AUTOMATIC

} SMART_FAVORITES_PLAY_POINT_CTRL_ENUM;

/* External object interface */
typedef struct smart_favorites_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eStart ) (
        DECODER_OBJECT hDecoder,
        BOOLEAN bCurrentBand
            );

    void (*vStop ) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable ) (
        DECODER_OBJECT hDecoder,
        SMART_FAVORITES_PLAY_POINT_CTRL_ENUM ePlayPoint
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable ) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetCurrentConfig ) (
        DECODER_OBJECT hDecoder,
        BOOLEAN *pbEnabled,
        SMART_FAVORITES_PLAY_POINT_CTRL_ENUM *pePlayPoint
            );

    CATEGORY_ID (*tCategoryId ) (
        DECODER_OBJECT hDecoder
            );

    N32 (*n32FPrintf ) (
        DECODER_OBJECT hDecoder,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOptions
            );

} SMART_FAVORITES_OBJECT_INTERFACE_STRUCT;

  /******************/
 /*** CDO OBJECT ***/
/******************/

// Content Description Object Type (CDO Type)
typedef enum cdo_type_enum
{
    CDO_REPORT =        '*',
    CDO_MUSIC =         '$',
    CDO_SPORTS =        '@',
    CDO_NEWS =          '?',
    CDO_ENTERTAINMENT = '!',
    CDO_NON_PROGRAM =   '^',

    CDO_UNKNOWN =       'U',
    CDO_INVALID =       'I'

} CDO_TYPE_ENUM;

typedef BOOLEAN (*CDO_ITERATOR_CALLBACK)(
    CID_OBJECT hId,
    CDO_TYPE_ENUM eType,
    const char *pacDescription,
    void *pvContentIteratorCallbackArg
        );

/* External object interface */
typedef struct cdo_object_interface_struct
{
    /* Object Members */

    CDO_TYPE_ENUM (*eType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

    const char *(*pacDescription) (
        CD_OBJECT hCDO
            );

    STRING_OBJECT (*hArtist) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hArtistId) (
        CD_OBJECT hCDO
            );

    STRING_OBJECT (*hTitle) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hTitleId) (
        CD_OBJECT hCDO
            );

    STRING_OBJECT (*hComposer) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hComposerId) (
        CD_OBJECT hCDO
            );

    STRING_OBJECT (*hAlbum) (
        CD_OBJECT hCDO
            );

    STRING_OBJECT (*hContentInfo) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hContentInfoId) (
        CD_OBJECT hCDO
            );

    CHANNEL_ART_OBJECT (*hArt) (
        CD_OBJECT hCDO
            );

    N16 (*n16Compare) (
        CD_OBJECT hCDO1,
        CD_OBJECT hCDO2
            );

    BOOLEAN (*bIterateTypes) (
        CDO_ITERATOR_CALLBACK bIteratorCallback,
        void *pvIteratorCallbackArg
            );

    N32 (*n32FPrintf) (
        CD_OBJECT hCDO,
        FILE *psFile
            );

} CD_OBJECT_INTERFACE_STRUCT;

  /******************/
 /*** CID OBJECT ***/
/******************/

/* External object interface */
typedef struct cid_object_interface_struct
{
    /* Object Members */

    void (*vDestroy) (
        CID_OBJECT hCID
            );

    N16 (*n16Equal) (
        CID_OBJECT hCID1,
        CID_OBJECT hCID2
            );

    N16 (*n16Compare) (
        CID_OBJECT hCID1,
        CID_OBJECT hCID2
            );

    CID_OBJECT (*hDuplicate) (
        CID_OBJECT hCID
            );

    BOOLEAN (*bSetEquivalent) (
        CID_OBJECT hCID,
        CID_OBJECT hEquivalentCID
            );

    N32 (*n32FWrite) (
        CID_OBJECT hCDO,
        FILE *psFile
            );

    CID_OBJECT (*hFRead) (
        FILE *psFile
            );

    N32 (*n32FPrintf) (
        CID_OBJECT hCID,
        FILE *psFile
            );

} CID_OBJECT_INTERFACE_STRUCT;

  /**************************/
 /*** NON_PROGRAM OBJECT ***/
/**************************/

// Non-Program Type
typedef enum non_program_enum
{
    NON_PROGRAM_VOICE =         'V',
    NON_PROGRAM_INTERSTITIAL =  'I',
    NON_PROGRAM_COMMERCIAL =    'C',
    NON_PROGRAM_OTHER =         'O',

    NON_PROGRAM_UNKNOWN =       ' '

} NON_PROGRAM_ENUM;

/* External object interface */
typedef struct non_program_object_interface_struct
{
    /* Object Members */

    NON_PROGRAM_ENUM (*eSubType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

} NON_PROGRAM_OBJECT_INTERFACE_STRUCT;

  /****************************/
 /*** ENTERTAINMENT OBJECT ***/
/****************************/

// Entertainment Type
typedef enum entertainment_enum
{
    ENTERTAINMENT_COMEDY =          'C',
    ENTERTAINMENT_TALK =            'T',
    ENTERTAINMENT_ENTERTAINMENT =   'E',
    ENTERTAINMENT_POLITICAL =       'P',
    ENTERTAINMENT_RELIGION =        'R',
    ENTERTAINMENT_SPANISH =         'S',
    ENTERTAINMENT_OTHER =           'O',

    ENTERTAINMENT_UNKNOWN =         ' '

} ENTERTAINMENT_ENUM;

/* External object interface */
typedef struct entertainment_object_interface_struct
{
    /* Object Members */

    ENTERTAINMENT_ENUM (*eSubType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

} ENTERTAINMENT_OBJECT_INTERFACE_STRUCT;

  /*******************/
 /*** NEWS OBJECT ***/
/*******************/

// News Type
typedef enum news_enum
{
    NEWS_GLOBAL =       'G',
    NEWS_WEATHER =      'W',
    NEWS_FINANCIAL =    'F',
    NEWS_US =           'U',
    NEWS_PUBLIC_RADIO = 'P',
    NEWS_OTHER =        'O',

    NEWS_UNKNOWN =      ' '

} NEWS_ENUM;

/* External object interface */
typedef struct news_object_interface_struct
{
    /* Object Members */

    NEWS_ENUM (*eSubType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

} NEWS_OBJECT_INTERFACE_STRUCT;

  /*********************/
 /*** SPORTS OBJECT ***/
/*********************/

// Sports Type
typedef enum sports_enum
{
    SPORTS_FOOTBALL = 0x01000000,
    SPORTS_HOCKEY = 0x02000000,
    SPORTS_BASKETBALL = 0x03000000,
    SPORTS_BASEBALL = 0x04000000,
    SPORTS_SOCCER = 0x05000000,
    SPORTS_AUTO_RACING = 0x06000000,
    SPORTS_OTHER = 0x07000000,
    SPORTS_UNKNOWN = 0x08000000
} SPORTS_ENUM;

typedef enum sports_broadcast_enum
{
    SPORTS_BROADCAST_NATIONAL,
    SPORTS_BROADCAST_TEAM,
    SPORTS_BROADCAST_OTHER,
    SPORTS_BROADCAST_UNKNOWN,
} SPORTS_BROADCAST_ENUM;

typedef BOOLEAN (*SPORTS_CONTENT_ITERATOR_CALLBACK)(
    const char *pacId,
    const char *pacName,
    SPORTS_ENUM eSubType,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        );

/* External object interface */

typedef struct sports_object_interface_struct
{
    /* Object Members */

    SPORTS_ENUM (*eSubType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

    LEAGUE_OBJECT (*hLeague) (
        CD_OBJECT hCDO
            );

    UN8 (*un8Teams) (
        CD_OBJECT hCDO
            );

    TEAM_OBJECT (*hTeam) (
        CD_OBJECT hCDO,
        UN8 un8Index
            );

    UN8 (*un8Cars) (
        CD_OBJECT hCDO
            );

    CAR_OBJECT (*hCar) (
        CD_OBJECT hCDO,
        UN8 un8Index
            );

    SPORTS_BROADCAST_ENUM (*eBroadcast) (
        CD_OBJECT hCDO,
        UN8 *pun8Index
            );

    BOOLEAN (*bIterateContent) (
        SPORTS_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
        void *pvContentIteratorCallbackArg
            );

} SPORTS_CONTENT_OBJECT_INTERFACE_STRUCT;

  /*********************/
 /*** LEAGUE OBJECT ***/
/*********************/

// League Type
typedef enum league_enum
{
    LEAGUE_NFL =                       'F',
    LEAGUE_MLB =                       'M',
    LEAGUE_NBA =                       'B',
    LEAGUE_NHL =                       'H',
    LEAGUE_COLLEGE_FOOTBALL =          'f',
    LEAGUE_COLLEGE_BASKETBALL =        'b',
    LEAGUE_WOMENS_COLLEGE_BASKETBALL = 'w',
    LEAGUE_COLLEGE_OTHER =             'c',
    LEAGUE_AUTORACING =                'A',
    LEAGUE_SOCCER =                    'S',
    LEAGUE_UNKNOWN =                   ' '

} LEAGUE_ENUM;

#define LEAGUE_INVALID_ID 0xFF

/* External object interface */

typedef BOOLEAN (*LEAGUE_CONTENT_ITERATOR_CALLBACK)(
    LEAGUE_OBJECT hLeague,
    void *pvContentIteratorCallbackArg
        );

typedef struct league_object_interface_struct
{
    /* Object Members */

    STRING_OBJECT (*hName) (
        LEAGUE_OBJECT hLeague
            );

    STRING_OBJECT (*hAbbreviation) (
        LEAGUE_OBJECT hLeague
            );

    LEAGUE_ENUM (*eLeague) (
        LEAGUE_OBJECT hLeague
            );

    BOOLEAN (*bIterateContent) (
        LEAGUE_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
        void *pvContentIteratorCallbackArg
            );

    CID_OBJECT (*hId) (
        LEAGUE_OBJECT hLeague
            );

    CID_OBJECT (*hTeamId) (
        LEAGUE_OBJECT hLeague,
        TEAM_OBJECT hTeam
            );

    SPORTS_ENUM (*eSport) (
        LEAGUE_OBJECT hLeague
            );

    N32 (*n32Version) (
        LEAGUE_OBJECT hLeague
            );

    BOOLEAN (*bGetTableVersion) (
        N32 *pn32Version,
        BOOLEAN *bComplete
            );

    UN8 (*un8Id) (
        LEAGUE_OBJECT hLeague
            );

    BOOLEAN (*bIsSportsFlashEnabled)(
        LEAGUE_OBJECT hLeague
            );

} LEAGUE_OBJECT_INTERFACE_STRUCT;

  /*******************/
 /*** TEAM OBJECT ***/
/*******************/

/* External object interface */
#define MAX_NUM_LEAGUES (128) // 7-bit value

#define NUMBER_OF_LEAGUE_ELEMENTS ((MAX_NUM_LEAGUES + UN8_BITLEN - 1) / 8)

typedef struct league_membership_struct
{
    UN8 un8Mask[NUMBER_OF_LEAGUE_ELEMENTS];

} LEAGUE_MEMBERSHIP_STRUCT;

typedef BOOLEAN (*TEAM_CONTENT_ITERATOR_CALLBACK)(
    TEAM_OBJECT hTeam,
    void *pvContentIteratorCallbackArg
        );

typedef struct team_object_interface_struct
{
    /* Object Members */
#ifdef SXM_LEGACY_SSP

    TEAM_OBJECT (*hCreate) (
        LEAGUE_ENUM eLeague,
        CID_OBJECT hTeamId,
        const char *pacAbbrev,
        const char *pacName,
        const char *pacNickname
               );

#endif

    STRING_OBJECT (*hName) (
        TEAM_OBJECT hTeam
            );

    STRING_OBJECT (*hNickname) (
        TEAM_OBJECT hTeam
            );

    STRING_OBJECT (*hAbbreviation) (
        TEAM_OBJECT hTeam
            );

    CID_OBJECT (*hId) (
        TEAM_OBJECT hTeam
            );

    BOOLEAN (*bIterateContent) (
        LEAGUE_OBJECT hLeague,
        TEAM_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
        void *pvContentIteratorCallbackArg
            );

    N32 (*n32Version) (
        TEAM_OBJECT hTeam
            );

    BOOLEAN (*bGetTableVersion) (
        N32 *pn32Version,
        BOOLEAN *bComplete
            );

    UN16 (*un16Id) (
        TEAM_OBJECT hTeam
            );

    SMSAPI_RETURN_CODE_ENUM (*eLeagueMembership) (
        TEAM_OBJECT hTeam,
        LEAGUE_MEMBERSHIP_STRUCT *psLeagueMembership
            );

    BOOLEAN (*bIsSportsFlashEligible) (
        LEAGUE_OBJECT hLeague,
        TEAM_OBJECT hTeam
            );

} TEAM_OBJECT_INTERFACE_STRUCT;

  /*******************/
 /*** CAR OBJECT ***/
/*******************/

/* External object interface */

typedef BOOLEAN (*CAR_CONTENT_ITERATOR_CALLBACK)(
    CAR_OBJECT hCar,
    void *pvContentIteratorCallbackArg
        );

typedef struct car_object_interface_struct
{
    /* Object Members */

#ifdef SXM_LEGACY_SSP
    CAR_OBJECT (*hCreate) (
          LEAGUE_ENUM eLeague,
          CID_OBJECT hCarId,
          const char *pacCarNumber,
          const char *pacDriverName,
          const char *pacNickname
              );
#endif

    STRING_OBJECT (*hDriverName) (
        CAR_OBJECT hCar
            );

    STRING_OBJECT (*hCarNumber) (
        CAR_OBJECT hCar
            );

    BOOLEAN (*bIterateContent) (
        LEAGUE_OBJECT hLeague,
        CAR_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
        void *pvContentIteratorCallbackArg
            );

} CAR_OBJECT_INTERFACE_STRUCT;

  /*************************/
 /*** SPORT_ZONE OBJECT ***/
/*************************/

// Sport Zone Iterator

typedef BOOLEAN (*SPORT_ZONE_ITERATION_HANDLER)(
    SPORT_ZONE_OBJECT hZone,
    void *pvArg
        );

// External object interface

typedef struct sport_zone_object_interface_struct
{
    /* Object Members */
    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        DECODER_OBJECT hDecoder
            );

    void (*vStop) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        DECODER_OBJECT hDecoder,
        SPORT_ZONE_ITERATION_HANDLER bIterator,
        void *pvArg
            );

    SPORT_ZONE_OBJECT (*hCreate) (
        DECODER_OBJECT hDecoder,
        LEAGUE_OBJECT hLeague
          );

    void (*vDestroy) (
        SPORT_ZONE_OBJECT hSportZone
            );

    CATEGORY_ID (*tCategoryId) (
        SPORT_ZONE_OBJECT hSportZone
            );

    LEAGUE_OBJECT (*hLeague) (
        SPORT_ZONE_OBJECT hSportZone
            );

} SPORT_ZONE_OBJECT_INTERFACE_STRUCT;

// Song Taging Services
typedef enum song_tag_service_enum
{
    SONG_TAG_SERVICE_ITUNES = 0,
    SONG_TAG_SERVICE_INVALID,
    SONG_TAG_SERVICE_MAX = SONG_TAG_SERVICE_INVALID
} SONG_TAG_SERVICE_ENUM;

typedef BOOLEAN (*SONG_TAG_SERVICE_ITERATOR_CALLBACK)(
    SONG_TAG_SERVICE_ENUM eService,
    STRING_OBJECT hURL,
    void *pvIteratorArg
        );

typedef struct song_tag_service_interface_struct
{
    STRING_OBJECT (*hURL) (
        SONG_TAG_SERVICE_ENUM eService
            );

    BOOLEAN (*bIterate) (
        SONG_TAG_SERVICE_ITERATOR_CALLBACK bIteratorCallback,
        void *pvContentIteratorArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetITunesTagValue) (
        CID_OBJECT hTag,
        UN32 *pun32TagValue
            );

    CID_OBJECT (*hCreateITunesTag) (
        UN32 un32ITunesTagValue
            );

} SONG_TAG_SERVICE_INTERFACE_STRUCT;

  /********************/
 /*** MUSIC OBJECT ***/
/********************/

// Music Type
typedef enum music_enum
{
    MUSIC_SHOW =    'M',
    MUSIC_MIX =     'X',
    MUSIC_OTHER =   'O',

    MUSIC_UNKNOWN = ' '

} MUSIC_ENUM;

/* External object interface */
typedef struct music_object_interface_struct
{
    /* Object Members */

    MUSIC_ENUM (*eSubType) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hArtistId) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hSongId) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hSongTag) (
        CD_OBJECT hCDO,
        SONG_TAG_SERVICE_ENUM eService
            );

} MUSIC_OBJECT_INTERFACE_STRUCT;

  /*********************/
 /*** REPORT OBJECT ***/
/*********************/

/* External object interface */

typedef BOOLEAN (*REPORT_CONTENT_ITERATOR_CALLBACK)(
    const char *pacId,
    const char *pacName,
    CID_OBJECT hId,
    void *pvContentIteratorCallbackArg
        );

typedef struct report_object_interface_struct
{
    /* Object Members */

    const char *(*pacId) (
        CD_OBJECT hCDO
            );

    const char *(*pacName) (
        CD_OBJECT hCDO
            );

    CID_OBJECT (*hId) (
        CD_OBJECT hCDO
            );

    BOOLEAN (*bIterateContent) (
        REPORT_CONTENT_ITERATOR_CALLBACK bContentIteratorCallback,
        void *pvContentIteratorCallbackArg
            );

    BOOLEAN (*bContentComplete) ( void );

} REPORT_OBJECT_INTERFACE_STRUCT;

#ifdef SXM_LEGACY_SSP
/***********************************/
/*** PID_OBJECT_INTERFACE_STRUCT ***/
/***********************************/

// For support of low-band PIDs
typedef struct pid_object_interface_struct
{
    // Special CID Create Methods

    CID_OBJECT (*hCreateTeam) (
        LEAGUE_ENUM eLeague,
        const char *pacTeamIdentifier
            );

    CID_OBJECT (*hCreateCar) (
        const char *pacCarNumber
            );

} PID_OBJECT_INTERFACE_STRUCT;
#endif

/*********************/
/*** STRING OBJECT ***/
/*********************/

/* External object interface */
typedef struct string_object_interface_struct
{
  /* Object Members */

  STRING_OBJECT (*hCreate) (
      const char *pacString,
      size_t tMinimumLength
          );

  size_t (*tCopy) (
      STRING_OBJECT hSrcString,
      STRING_OBJECT hDstString
          );

  size_t (*tCopyToCStr) (
      STRING_OBJECT hString,
      char *pacDst,
      size_t tDstSize
          );

  const char *(*pacCStr) (
      STRING_OBJECT hString
          );

  size_t (*tSize) (
      STRING_OBJECT hString
          );

  N16 (*n16Compare) (
      STRING_OBJECT hString1,
      STRING_OBJECT hString2,
      BOOLEAN bBinary
          );

  N16 (*n16CaseCompare) (
      STRING_OBJECT hString1,
      STRING_OBJECT hString2
          );

  N16 (*n16CompareCStr) (
      const char *pacString,
      size_t tLength,
      STRING_OBJECT hString
          );

  N16 (*n16SubstringCStr) (
      STRING_OBJECT hString,
      const char *pacString,
      BOOLEAN bCaseInsensitive
          );

  STRING_OBJECT (*hDuplicate) (
      STRING_OBJECT hString
          );

  N32 (*n32FWrite) (
      STRING_OBJECT hString,
      FILE *psFile
          );

  STRING_OBJECT (*hFRead) (
      FILE *psFile
          );

  N32 (*n32FPrintf) (
      STRING_OBJECT hString,
      FILE *psFile
          );

  BOOLEAN (*bModifyCStr) (
      STRING_OBJECT hString,
      const char *pacSrc
          );

    size_t (*tAppendString) (
        STRING_OBJECT hSrcString,
        STRING_OBJECT hDstString
            );

    size_t (*tAppendCStr) (
        STRING_OBJECT hString,
        const char *pacSrc
            );

  void (*vDestroy) (
      STRING_OBJECT hString
          );

} STRING_OBJECT_INTERFACE_STRUCT;

/*********************/
/******* BAUDOT ******/
/*********************/

typedef enum baudot_behavior_enum
{
    BAUDOT_BEHAVIOR_PROCESS_TO_MAX_SYMBOLS,
    BAUDOT_BEHAVIOR_PROCESS_TO_END,
    BAUDOT_BEHAVIOR_INVALID

} BAUDOT_BEHAVIOR_ENUM;

typedef struct baudot_interface_struct
{
    STRING_OBJECT (*hToString) (
        OSAL_BUFFER_HDL hBuffer,
        BAUDOT_BEHAVIOR_ENUM eProcessBehavior,
        BOOLEAN bStringStartsWithLetters,
        BOOLEAN bStringStartsWithCaps,
        size_t tMaxBaudotSymbolsToRead,
        size_t *ptNumSymbolsFound
            );

} BAUDOT_INTERFACE_STRUCT;

  /*********************/
 /*** PLAYBACK OBJECT */
/*********************/
#define PLAYBACK_SONG_OFFSET_START   N16_MIN
#define PLAYBACK_SONG_OFFSET_LIVE    N16_MAX
#define PLAYBACK_TIME_OFFSET_START   N32_MIN
#define PLAYBACK_TIME_OFFSET_LIVE    N32_MAX

typedef enum playback_error_code_enum
{
    PLAYBACK_ERROR_CODE_NONE = 0,
    PLAYBACK_ERROR_CODE_INVALID_OBJECT,
    PLAYBACK_ERROR_CODE_PLAYBACK_NOT_SUPPORTED,
    PLAYBACK_ERROR_CODE_PLAYBACK_OP_FAILED,
    PLAYBACK_ERROR_CODE_OUT_OF_RANGE

} PLAYBACK_ERROR_CODE_ENUM;

// Playback Object Event Mask
typedef SMSAPI_EVENT_MASK PLAYBACK_OBJECT_EVENT_MASK;

// Global (re-usable) instance of an interface for this object
typedef struct playback_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*ePause) (
        PLAYBACK_OBJECT hPlayback
            );

    SMSAPI_RETURN_CODE_ENUM (*ePlay) (
        PLAYBACK_OBJECT hPlayback
            );

    SMSAPI_RETURN_CODE_ENUM (*eSeekTime) (
        PLAYBACK_OBJECT hPlayback,
        N32 n32Seconds,
        BOOLEAN bPauseAfterSeek
            );

    SMSAPI_RETURN_CODE_ENUM (*eSeekSong) (
        PLAYBACK_OBJECT hPlayback,
        N16 n16RelativeOffset,
        BOOLEAN bPauseAfterSeek
            );

    SMSAPI_RETURN_CODE_ENUM (*eSeekDirect) (
        PLAYBACK_OBJECT hPlayback,
        SONGLIST_OBJECT hSonglist,
        N16 n16Offset,
        BOOLEAN bPauseAfterSeek
            );

    SMSAPI_RETURN_CODE_ENUM (*ePlayPercentage) (
        PLAYBACK_OBJECT hPlayback,
        UN8 *pun8PlayPercentage
            );

    SMSAPI_RETURN_CODE_ENUM (*eFillPercentage) (
        PLAYBACK_OBJECT hPlayback,
        UN8 *pun8FillPercentage
            );

    SMSAPI_RETURN_CODE_ENUM (*eTimeOffset) (
        PLAYBACK_OBJECT hPlayback,
        N32 *pn32TimeOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eTimeFromTrackStart) (
        PLAYBACK_OBJECT hPlayback,
        UN32 *pun32TimeFromTrackStart
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsPaused) (
        PLAYBACK_OBJECT hPlayback,
        BOOLEAN *pbPaused
            );

    PLAYBACK_OBJECT_EVENT_MASK (*tEventMask) (
        PLAYBACK_OBJECT hPlayback
            );

    PLAYBACK_ERROR_CODE_ENUM (*eErrorCode) (
        PLAYBACK_OBJECT hPlayback
            );

    SMSAPI_RETURN_CODE_ENUM (*eWarningOffsetGet) (
        PLAYBACK_OBJECT hPlayback,
        UN32 *pun32WarningOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eWarningOffsetSet) (
        PLAYBACK_OBJECT hDecoder,
        UN32 un32WarningOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eRecordedContentTimeInfo) (
        PLAYBACK_OBJECT hPlayback,
        UN32 *pun32Duration,
        UN32 *pun32ElapsedTime,
        UN32 *pun32RemainingTime
            );

    UN32 (*un32IRBufferDuration) (
        PLAYBACK_OBJECT hPlayback
            );

    SMSAPI_RETURN_CODE_ENUM (*eTrackDuration ) (
        PLAYBACK_OBJECT hPlayback,
        UN32 *pun32Duration
            );

    SMSAPI_RETURN_CODE_ENUM (*eTracksBefore ) (
        PLAYBACK_OBJECT hPlayback,
        UN16 *pun16TracksBefore
            );

    SMSAPI_RETURN_CODE_ENUM (*eTracksRemaining ) (
        PLAYBACK_OBJECT hPlayback,
        UN16 *pun16TracksRemaining
            );

    SMSAPI_RETURN_CODE_ENUM (*eSeekPrevious ) (
        PLAYBACK_OBJECT hPlayback,
        BOOLEAN bPauseAfterSeek
            );

    SMSAPI_RETURN_CODE_ENUM (*eSeekNext ) (
        PLAYBACK_OBJECT hPlayback,
        BOOLEAN bPauseAfterSeek
            );

} PLAYBACK_OBJECT_INTERFACE_STRUCT;

  /*********************************************/
 /*** CONTENT EVENT MONITORING SERVICE OBJECT  */
/***********************************************/

// Content Compare/Sort Handler definition
typedef N16 (*CONTENT_COMPARE_HANDLER)(
    void *pvContentArg1,
    void *pvContentArg2
        );

// Registered Content Iterator Callback definition
typedef BOOLEAN (*CONTENT_ITERATOR_CALLBACK)(
    CEML_OBJECT hCEML,
    CID_OBJECT hCID,
    void *pvContentArg,
    UN32 un32Options,
    void *pvContentIteratorArg
        );

// Active Event Iterator Callback definition
typedef BOOLEAN (*ACTIVE_EVENT_ITERATOR_CALLBACK)(
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventIteratorArg
        );

// Active Event Callback definition
typedef void (*ACTIVE_EVENT_CALLBACK) (
    CEML_OBJECT hCEML,
    CHANNEL_OBJECT hChannel,
    void *pvContentArg,
    UN32 un32Flags,
    void *pvEventCallbackArg
        );

/////////////////////////////////
// Content Registration Options
/////////////////////////////////

// Either of these event options may be chosen in any combination...
#define SMS_EVENT_REGISTRATION_OPTION_INITIAL     (0x00000001)
#define SMS_EVENT_REGISTRATION_OPTION_CURRENT     (0x00000002)
#define SMS_EVENT_REGISTRATION_OPTION_END         (0x00000004)

// These event options are mutually exclusive as only one or the
// other may be chosen.
#define SMS_EVENT_REGISTRATION_OPTION_RETRIGGER   (0x00000008)
#define SMS_EVENT_REGISTRATION_OPTION_ONETIME     (0x00000010)

// Reserved options
#define SMS_EVENT_REGISTRATION_OPTION_RESERVED    (0xFFFFFFE0)

// None/All options
#define SMS_EVENT_REGISTRATION_OPTION_NONE        (0x00000000)

#define SMS_EVENT_REGISTRATION_OPTION_ALL         ( \
    SMS_EVENT_REGISTRATION_OPTION_INITIAL | \
    SMS_EVENT_REGISTRATION_OPTION_CURRENT | \
    SMS_EVENT_REGISTRATION_OPTION_END | \
    SMS_EVENT_REGISTRATION_OPTION_RETRIGGER | \
    SMS_EVENT_REGISTRATION_OPTION_ONETIME \
        )

////////////////////////
// Active Event Flags
////////////////////////

// Event has "started"
#define ACTIVE_EVENT_FLAG_INITIAL       (0x00000001)
// Event is no longer occuring "ended"
#define ACTIVE_EVENT_FLAG_END           (0x00000002)
// Event will no longer occur and has been removed from the registered list
#define ACTIVE_EVENT_FLAG_FINAL         (0x00000004)
// Event flags reserved
#define ACTIVE_EVENT_FLAG_RESERVED      (0xFFFFFFF8)

// Global (re-usable) instance of an interface for this object
typedef struct ceml_object_interface_struct
{
    /* Object Members */

    CEML_OBJECT (*hCreateList) (
        DECODER_OBJECT hDecoder,
        CONTENT_COMPARE_HANDLER n16CompareHandler,
        ACTIVE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vDestroyList) (
        CEML_OBJECT hCEML
            );

    size_t (*tContentListItems) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateContentList) (
        CEML_OBJECT hCEML,
        CONTENT_ITERATOR_CALLBACK bContentIterator,
        void *pvContentIteratorArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateActiveList) (
        CEML_OBJECT hCEML,
        ACTIVE_EVENT_ITERATOR_CALLBACK bEventIterator,
        void *pvEventIteratorArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegister) (
        CEML_OBJECT hCEML,
        CID_OBJECT hCID,
        void *pvContentArg,
        UN32 un32Options
            );

    CID_OBJECT (*hCID) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eReplace) (
        CEML_OBJECT hCEML,
        UN32 un32Options
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemove) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveAll) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eContentEnabled) (
        CEML_OBJECT hCEML,
        BOOLEAN *pbEnabled
            );

    SMSAPI_RETURN_CODE_ENUM (*eAcknowledge) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eBlockAllEvents) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eUnBlockAllEvents) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eEventsBlocked) (
        CEML_OBJECT hCEML,
        BOOLEAN *pbBlocked
            );

    SMSAPI_RETURN_CODE_ENUM (*eSearchForCurrent) (
        CEML_OBJECT hCEML
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetEndedEventAttributes) (
        CEML_OBJECT hCEML,
        CHANNEL_ID *ptChannelId,
        STRING_OBJECT *phArtist,
        STRING_OBJECT *phTitle
            );

} CEML_OBJECT_INTERFACE_STRUCT;

  /*********************************************/
 /*** CONTENT ALERT SERVICE OBJECT  */
/***********************************************/

typedef BOOLEAN (*CAL_CONTENT_ITERATOR_CALLBACK)(
    CAL_CONTENT_OBJECT hContent,
    void *pvContentIteratorArg
        );

typedef void (*CAL_CALLBACK)(
    CAL_OBJECT hCAL,
    CAL_ALERT_OBJECT hAlert,
    UN32 un32Flags,
    void *pvCALCallbackArg
        );


typedef int CAL_AUTO_ADD_OPTIONS;

#define CAL_AUTO_ADD_OPTION_NONE           (0x00000000)

#define CAL_AUTO_ADD_OPTION_OFF            (0x00000001)
#define CAL_AUTO_ADD_OPTION_UNIQUE         (0x00000002)
#define CAL_AUTO_ADD_OPTION_IF_TUNED       (0x00000004)
#define CAL_AUTO_ADD_OPTION_IF_NOT_TUNED   (0x00000008)
#define CAL_AUTO_ADD_OPTION_TOP            (0x00000010)
#define CAL_AUTO_ADD_OPTION_BOTTOM         (0x00000020)

// Event has "started"
#define CAL_INITIAL       ACTIVE_EVENT_FLAG_INITIAL
// Event is no longer occurring "ended"
#define CAL_END           ACTIVE_EVENT_FLAG_END
// Event will no longer occur and has been removed from the registered list
#define CAL_FINAL         ACTIVE_EVENT_FLAG_FINAL
// Event flags reserved
#define CAL_RESERVED      ACTIVE_EVENT_FLAG_RESERVED

// Content Alert Service Interface
typedef struct content_alert_service_interface_struct
{
   CAL_OBJECT (*hCreateList) (
        DECODER_OBJECT hDecoder,
        CAL_CALLBACK vCALCallback,
        void *vCALCallbackArg
            );

   void (*vDestroyList) (
        CAL_OBJECT hCAL
            );

} CONTENT_ALERT_SERVICE_INTERFACE_STRUCT;

// Content Alert List Interface
typedef struct content_alert_list_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eRegister) (
        CAL_OBJECT hCAL,
        CID_OBJECT hCID,
        void *pvContentArg,
        UN32 un32Options,
        STRING_OBJECT hArtistText,
        STRING_OBJECT hTitleText,
        CAL_CONTENT_OBJECT *phCalContent
            );

   SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        CAL_OBJECT hCAL,
        CAL_CONTENT_ITERATOR_CALLBACK bContentIterator,
        void *pvContentIteratorArg
            );

    BOOLEAN (*bEnabled) (
        CAL_OBJECT hCAL
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable) (
        CAL_OBJECT hCAL
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable) (
        CAL_OBJECT hCAL
            );

    size_t (*tNumItems) (
        CAL_OBJECT hCAL
            );

    BOOLEAN (*bExists) (
        CAL_OBJECT hCAL,
        CID_OBJECT hCID
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveAll) (
        CAL_OBJECT hCAL
            );

    CATEGORY_ID (*tCategoryId) (
        CAL_OBJECT hCAL
            );

    SMSAPI_RETURN_CODE_ENUM (*eAttachCategory) (
        CAL_OBJECT hCAL,
        CATEGORY_ID tCategoryId,
        CAL_AUTO_ADD_OPTIONS tCatAddOptions,
        ...
            );

    SMSAPI_RETURN_CODE_ENUM (*eDetachCategory) (
        CAL_OBJECT hCAL
            );

} CONTENT_ALERT_LIST_INTERFACE_STRUCT;

  /*********************************************/
 /*** CAL ALERT OBJECT  */
/***********************************************/

// CAL_ALERT Interface
typedef struct cal_alert_object_interface_struct
{

    SMSAPI_RETURN_CODE_ENUM (*eAddToCategory) (
        CAL_ALERT_OBJECT hAlert,
        BOOLEAN bAddDuplicate
            );

    SMSAPI_RETURN_CODE_ENUM (*eTune) (
        CAL_ALERT_OBJECT hAlert,
        BOOLEAN bLockOverride
            );

    CHANNEL_OBJECT (*hChannel) (
        CAL_ALERT_OBJECT hAlert
            );

    CAL_CONTENT_OBJECT (*hContent) (
        CAL_ALERT_OBJECT hAlert
            );

}CAL_ALERT_OBJECT_INTERFACE_STRUCT;

  /*********************************************/
 /*** CAL CONTENT OBJECT  */
/***********************************************/

// CAL Content State
typedef enum cal_content_state_enum
{
    CAL_CONTENT_STATE_DISABLED,
    CAL_CONTENT_STATE_ACTIVE,
    CAL_CONTENT_STATE_SEARCHING,
    CAL_CONTENT_STATE_INVALID
} CAL_CONTENT_STATE_ENUM;

// CAL_CONTENT Interface
typedef struct cal_content_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eRemove) (
        CAL_CONTENT_OBJECT hContent
            );

    CID_OBJECT (*hCID) (
        CAL_CONTENT_OBJECT hContent
            );

    CAL_CONTENT_STATE_ENUM (*eState) (
        CAL_CONTENT_OBJECT hContent
            );

    void * (*pvContentArg) (
        CAL_CONTENT_OBJECT hContent
            );

    UN32 (*un32Options) (
        CAL_CONTENT_OBJECT hContent
            );

    BOOLEAN (*bEnabled) (
        CAL_CONTENT_OBJECT hContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable) (
        CAL_CONTENT_OBJECT hContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable) (
        CAL_CONTENT_OBJECT hContent
            );

    STRING_OBJECT (*hArtistText) (
        CAL_CONTENT_OBJECT hContent
            );

    STRING_OBJECT (*hTitleText) (
        CAL_CONTENT_OBJECT hContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eSearchForCurrent) (
        CAL_CONTENT_OBJECT hContent
            );

} CAL_CONTENT_OBJECT_INTERFACE_STRUCT;

  /*********************************************/
 /*** SEEK SERVICE OBJECT  */
/***********************************************/

// the possible types of seek services
typedef enum seek_service_enum
{
    SEEK_SERVICE_ARTIST_TITLE,
    SEEK_SERVICE_TRAFFIC_WEATHER,
    SEEK_SERVICE_SPORTS,
    SEEK_SERVICE_INVALID
} SEEK_SERVICE_ENUM;

// the possible states of a seek Service
typedef enum seek_state_enum
{
    SEEK_STATE_ENABLED,
    SEEK_STATE_DISABLED,
    SEEK_STATE_INVALID
} SEEK_STATE_ENUM;

// indication of whether content can be registered
typedef enum seek_availability_enum
{
    SEEK_AVAILABILTY_AVAILABLE,
    SEEK_AVAILIBILITY_UNAVAILABLE,
    SEEK_AVAILIBILITY_ALREADY_REGISTERED,
    SEEK_AVAILIBILITY_NOT_ALLOWED,
    SEEK_AVAILIBILITY_UNKNOWN
} SEEK_AVAILABILTY_ENUM;

typedef BOOLEAN (*SEEK_CONTENT_ITERATOR_CALLBACK)(
    DECODER_OBJECT hDecoder,
    SEEK_SERVICE_ENUM eService,
    SEEK_CONTENT_OBJECT hContent,
    void *pvSeekContentIteratorArg
        );

// Generic Seek Service Interface
typedef struct seek_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    void (*vStop) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SEEK_STATE_ENUM (*eState) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    CATEGORY_ID (*tCategoryId) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    N32 (*n32FPrintf) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

    size_t (*tNumItems) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveAll) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService,
        SEEK_CONTENT_ITERATOR_CALLBACK bSeekContentIterator,
        void *pvSeekContentIteratorArg
            );

    SEEK_EVENT_OBJECT (*hGetNextEvent) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetNextEndedEventAttributes) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID *ptChannelId,
        STRING_OBJECT *phArtist,
        STRING_OBJECT *phTitle
            );

    BOOLEAN (*bGetAlertsTunedStatus) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnableAlertsTuned) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService,
        BOOLEAN bEnable
            );

    BOOLEAN (*bGetAlertsFinishedStatus) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnableAlertsFinished) (
        DECODER_OBJECT hDecoder,
        SEEK_SERVICE_ENUM eService,
        BOOLEAN bEnable
            );

} SEEK_INTERFACE_STRUCT;

  /*********************************************/
 /*** SEEK CONTENT OBJECT  */
/***********************************************/

// Generic Seek Content Interface
typedef struct seek_content_interface_struct
{
    SEEK_SERVICE_ENUM (*eService) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemove) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnable) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisable) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    BOOLEAN (*bEnabled) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eUseSeekContent) (
        SEEK_CONTENT_OBJECT hSeekContent,
        SEEK_CONTENT_ACCESS_CALLBACK vSeekContentAccessCallback,
        void *pvCallbackArg
            );

    N32 (*n32FPrintf) (
        SEEK_CONTENT_OBJECT hSeekContent,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} SEEK_CONTENT_INTERFACE_STRUCT;

  /*********************************************/
 /*** SEEK EVENT OBJECT  */
/***********************************************/

// Seek Event Interface
typedef struct seek_event_interface_struct
{
    CHANNEL_OBJECT (*hChannel) (
        SEEK_EVENT_OBJECT hSeekEvent
            );

    SEEK_CONTENT_OBJECT (*hContent) (
        SEEK_EVENT_OBJECT hSeekEvent
            );

} SEEK_EVENT_INTERFACE_STRUCT;

  /*********************************************/
 /*** ARTIST / TITLE SEEK OBJECT  */
/***********************************************/
// these are seek types specific to the artist/title seek service
typedef enum at_seek_enum
{
    AT_SEEK_ARTIST,
    AT_SEEK_TITLE,
    AT_SEEK_UNKNOWN
} AT_SEEK_ENUM;

// Artist/Title Seek Service Interface
typedef struct at_seek_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eAvailableToRegister) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        SEEK_AVAILABILTY_ENUM *peTitleAvailablility,
        SEEK_AVAILABILTY_ENUM *peArtistAvailablility
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegister) (
        DECODER_OBJECT hDecoder,
        AT_SEEK_ENUM eType,
        CHANNEL_ID tChannelId,
        SEEK_CONTENT_OBJECT *phSeekContent
            );

} AT_SEEK_INTERFACE_STRUCT;

  /*********************************************/
 /*** ARTIST / TITLE SEEK CONTENT OBJECT  */
/***********************************************/

// Artist/Title Seek Content Interface
typedef struct at_seek_content_interface_struct
{
    STRING_OBJECT (*hRegisteredArtistText) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    STRING_OBJECT (*hRegisteredTitleText) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    AT_SEEK_ENUM (*eType) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    BOOLEAN (*bConvertible) (
        SEEK_CONTENT_OBJECT hSeekContent,
        AT_SEEK_ENUM eConvertToType
            );

    SMSAPI_RETURN_CODE_ENUM (*eConvert) (
        SEEK_CONTENT_OBJECT hSeekContent,
        AT_SEEK_ENUM eConvertToType
            );

} AT_SEEK_CONTENT_INTERFACE_STRUCT;

  /*********************************************/
 /*** TRAFFIC / WEATHER SEEK OBJECT  */
/***********************************************/

// Traffic/Weather Seek Service Interface
typedef struct tw_seek_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eAvailableToRegister) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        SEEK_AVAILABILTY_ENUM *peTrafficWeatherAvailablility
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegister) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        SEEK_CONTENT_OBJECT *phSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegisterMarket) (
        DECODER_OBJECT hDecoder,
        CID_OBJECT hReportId,
        SEEK_CONTENT_OBJECT *phSeekContent
            );

} TW_SEEK_INTERFACE_STRUCT;

  /*********************************************/
 /*** TRAFFIC / WEATHER SEEK CONTENT OBJECT  */
/***********************************************/

// Traffic/Weather Seek Content Interface
typedef struct tw_seek_content_interface_struct
{

    CID_OBJECT (*hReportId) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    STRING_OBJECT (*hMarketName) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    STRING_OBJECT (*hMarketId) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

} TW_SEEK_CONTENT_INTERFACE_STRUCT;

  /*********************************************/
 /*** SPORTS SEEK OBJECT  */
/***********************************************/
// these are seek types specific to the sports seek service
typedef enum sp_seek_enum
{
    SP_SEEK_TEAM,
    SP_SEEK_LEAGUE,
    SP_SEEK_UNKNOWN
} SP_SEEK_ENUM;

// Sports Seek Service Interface
typedef struct sp_seek_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eAvailableToRegister) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        SEEK_AVAILABILTY_ENUM *peAvailablility,
        SP_SEEK_ENUM eType,
        ...
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegister) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID tChannelId,
        SP_SEEK_ENUM eType,
        SEEK_CONTENT_OBJECT *phSeekContent,
        ...
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegisterTeam) (
        DECODER_OBJECT hDecoder,
        LEAGUE_OBJECT hLeague,
        TEAM_OBJECT hTeam,
        SEEK_CONTENT_OBJECT *phSeekContent
            );

    SMSAPI_RETURN_CODE_ENUM (*eRegisterLeague) (
        DECODER_OBJECT hDecoder,
        LEAGUE_OBJECT hLeague,
        SEEK_CONTENT_OBJECT *phSeekContent
            );

} SP_SEEK_INTERFACE_STRUCT;

  /*********************************************/
 /*** SPORTS SEEK CONTENT OBJECT  */
/***********************************************/

// Sports Seek Content Interface
typedef struct sp_seek_content_interface_struct
{
    STRING_OBJECT (*hRegisteredName) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    STRING_OBJECT (*hRegisteredNickname) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    CID_OBJECT (*hRegisteredLeagueId) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    CID_OBJECT (*hRegisteredTeamId) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

    SP_SEEK_ENUM (*eType) (
        SEEK_CONTENT_OBJECT hSeekContent
            );

} SP_SEEK_CONTENT_INTERFACE_STRUCT;

  /**********************/
 /*** PRESETS OBJECT ***/
/**********************/

// Featured Favorites Band Capacity
typedef UN8 FAVORITES_BAND_CAPACITY;

#define FAVORITES_BAND_CAPACITY_MIN             (1)

// Featured Favorites Bands Number
typedef UN8 FAVORITES_BAND_NUMBER;

#define FAVORITES_BANDS_NUMBER_MIN              (1)
#define FAVORITES_BANDS_NUMBER_UNLIMITED        (UN8_MAX)

// Featured Favorites Band purpose (2-bit field)
typedef UN8 FAVORITES_BAND_PURPOSE_MASK;

#define FAVORITES_BAND_PURPOSE_NONE             (0x00)
#define FAVORITES_BAND_PURPOSE_CONVENIENCE      (0x01)
#define FAVORITES_BAND_PURPOSE_SCANNING         (0x02)

#define FAVORITES_BAND_PURPOSE_ALL              (  \
        FAVORITES_BAND_PURPOSE_CONVENIENCE  |      \
        FAVORITES_BAND_PURPOSE_SCANNING              )

// Preset Band type enumeration to distinguish
// user preset band from the Featured (broadcasted) one
typedef enum preset_band_type_enum
{
    PRESET_BAND_TYPE_USER,
    PRESET_BAND_TYPE_FEATURED,
    PRESET_BAND_TYPE_INVALID

} PRESET_BAND_TYPE_ENUM;

// Presets Object Event Mask
typedef SMSAPI_EVENT_MASK PRESETS_EVENT_MASK;

// Presets Iterator Callback definition
typedef N16 (*PRESETS_ITERATOR_HANDLER) (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    size_t tBandIndex,
    CHANNEL_OBJECT hChannel,
    STRING_OBJECT hPresetName,
    size_t tPresetIndex,
    void *pvIterateArg
        );

// Presets band update notification callback
typedef void (*PRESETS_OBJECT_EVENT_CALLBACK ) (
    PRESETS_OBJECT hPresets,
    PRESET_BAND_OBJECT hBand,
    PRESETS_EVENT_MASK tEventMask,
    void *pvCallbackArg
        );

// Global (re-usable) instance of an interface for this object
typedef struct presets_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        PRESETS_OBJECT *phPresets,
        DECODER_OBJECT hDecoder,
        const char *pacLongName,
        const char *pacShortName
            );

    SMSAPI_RETURN_CODE_ENUM (*eStop) (
        PRESETS_OBJECT hPresets
            );

    CATEGORY_ID (*tCategoryId) (
        PRESETS_OBJECT hPresets
            );

    size_t (*tNumBands) (
        PRESETS_OBJECT hPresets
            );

    STRING_OBJECT (*hShortName) (
        PRESETS_OBJECT hPresets
            );

    STRING_OBJECT (*hLongName) (
        PRESETS_OBJECT hPresets
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        PRESETS_OBJECT hPresets,
        PRESETS_ITERATOR_HANDLER n16Iterator,
        void *pvIterateArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetCurrentBand) (
        PRESET_BAND_OBJECT hBand
            );

    PRESET_BAND_OBJECT (*hCurrentBand) (
        PRESETS_OBJECT hPresets
            );

    PRESET_BAND_OBJECT (*hPreviousBand) (
        PRESETS_OBJECT hPresets
            );

    PRESET_BAND_OBJECT (*hNextBand) (
        PRESETS_OBJECT hPresets
            );

    N32 (*n32FPrintf) (
        PRESETS_OBJECT hPresets,
        FILE *psFile
            );

    SMSAPI_RETURN_CODE_ENUM (*eNotifyOnChange ) (
        PRESETS_OBJECT hPresets,
        PRESETS_OBJECT_EVENT_CALLBACK vCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eFeaturedFavoritesStart ) (
        PRESETS_OBJECT hPresets,
        FAVORITES_BAND_CAPACITY tBandCapacity,
        FAVORITES_BAND_NUMBER tBandsNumber,
        FAVORITES_BAND_PURPOSE_MASK tPurpose
            );

    SMSAPI_RETURN_CODE_ENUM (*eFeaturedFavoritesStop ) (
        PRESETS_OBJECT hPresets,
        BOOLEAN bRemoveBands
            );

} PRESETS_OBJECT_INTERFACE_STRUCT;

typedef struct preset_band_object_interface_struct
{
    PRESET_BAND_OBJECT (*hCreate) (
        PRESETS_OBJECT hPresets,
        size_t tCapacity,
        const char *pacName
            );

    SMSAPI_RETURN_CODE_ENUM (*eDestroy) (
        PRESET_BAND_OBJECT hBand
            );

    STRING_OBJECT (*hName) (
        PRESET_BAND_OBJECT hBand
            );

    STRING_OBJECT (*hLongName ) (
        PRESET_BAND_OBJECT hBand
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetLongName ) (
        PRESET_BAND_OBJECT hBand,
        char *pacLongName
            );

    STRING_OBJECT (*hVerboseName ) (
        PRESET_BAND_OBJECT hBand
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetVerboseName ) (
        PRESET_BAND_OBJECT hBand,
        char *pacVerboseName
            );

    STRING_OBJECT (*hDescription ) (
        PRESET_BAND_OBJECT hBand
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetDescription ) (
        PRESET_BAND_OBJECT hBand,
        char *pacDescription
            );

    PRESET_BAND_TYPE_ENUM (*eType) (
        PRESET_BAND_OBJECT hBand
            );

    size_t (*tCapacity) (
        PRESET_BAND_OBJECT hBand
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetPreset) (
        PRESET_BAND_OBJECT hBand,
        size_t tPresetIndex,
        CHANNEL_ID tChannelId
            );

    CHANNEL_ID (*tGetPreset) (
        PRESET_BAND_OBJECT hBand,
        size_t tPresetIndex
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetPresetName) (
        PRESET_BAND_OBJECT hBand,
        size_t tPresetIndex,
        const char *pacName
            );

    STRING_OBJECT (*hGetPresetName) (
        PRESET_BAND_OBJECT hBand,
        size_t tPresetIndex
            );

    SMSAPI_RETURN_CODE_ENUM (*eClearAllPresets) (
        PRESET_BAND_OBJECT hBand,
        BOOLEAN bClearNames
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        PRESET_BAND_OBJECT hBand,
        PRESETS_ITERATOR_HANDLER n16Iterator,
        void *pvIterateArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eTune) (
        PRESET_BAND_OBJECT hBand,
        size_t tPresetIndex,
        BOOLEAN bLockOverride
            );

    N32 (*n32FPrintf) (
        PRESET_BAND_OBJECT hBand,
        FILE *psfile
            );

} PRESET_BAND_OBJECT_INTERFACE_STRUCT;

  /*********************************************/
 /***       SUB_NOTIFICATION                   */
/***********************************************/
typedef enum sub_notification_state_enum
{
    SUB_NOTIFICATION_NOT_REQD,
    SUB_NOTIFICATION_INITIAL,
    SUB_NOTIFICATION_WAIT,
    SUB_NOTIFICATION_TUNE,
    SUB_NOTIFICATION_METADATA,
    SUB_NOTIFICATION_COMPLETE,
    SUB_NOTIFICATION_INVALID

} SUB_NOTIFICATION_STATE_ENUM;

typedef struct sub_notification_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        DECODER_OBJECT hDecoder
            );

    void (*vStop) (
        DECODER_OBJECT hDecoder
            );

    SUB_NOTIFICATION_STATE_ENUM (*eState) (
        DECODER_OBJECT hDecoder
            );

} SUB_NOTIFICATION_OBJECT_INTERFACE_STRUCT;


  /*********************************/
 /***       IMAGE Object        ***/
/*********************************/

typedef enum image_format_enum
{
    IMAGE_FORMAT_PNG,
    IMAGE_FORMAT_JPEG,
    IMAGE_FORMAT_BMP,
    IMAGE_FORMAT_RAW,

    IMAGE_INVALID_FORMAT

} IMAGE_FORMAT_ENUM;

typedef enum image_property_enum
{
    IMAGE_PROPERTY_WIDTH,
    IMAGE_PROPERTY_HEIGHT,
    IMAGE_PROPERTY_BITS_PER_PIXEL,
    IMAGE_PROPERTY_CAPTION

} IMAGE_PROPERTY_ENUM;

typedef enum image_property_type_enum
{
    IMAGE_PROPERTY_TYPE_UN32,
    IMAGE_PROPERTY_TYPE_STRING

} IMAGE_PROPERTY_TYPE_ENUM;

typedef struct image_property_value_struct
{
    IMAGE_PROPERTY_TYPE_ENUM eType;

    union
    {
        UN32 un32Value;
        STRING_OBJECT hStringValue;
    } uData;

} IMAGE_PROPERTY_VALUE_STRUCT;

typedef struct rgb_struct
{
    UN8 un8Red;
    UN8 un8Green;
    UN8 un8Blue;

} RGB_STRUCT;

/* External object interface */
typedef struct image_interface_struct
{
    /* Object members */

    IMAGE_FORMAT_ENUM (*eFormat) (
        IMAGE_OBJECT hImage
            );

    STRING_OBJECT (*hFileName) (
        IMAGE_OBJECT hImage
            );

    SMSAPI_RETURN_CODE_ENUM (*eProperty)(
        IMAGE_OBJECT hImage,
        IMAGE_PROPERTY_ENUM eProperty,
        IMAGE_PROPERTY_VALUE_STRUCT *psValue
            );

    SMSAPI_RETURN_CODE_ENUM (*eImageBackgroundDisplayRules) (
        IMAGE_OBJECT hImage,
        BOOLEAN *pbUseBackgroundColor,
        BOOLEAN *pbUseLineBitmap,
        size_t  *tNumLineBitmapEntries,
        RGB_STRUCT *psBackgroundColor
            );

    BOOLEAN (*bGetLineBitmap) (
        IMAGE_OBJECT hImage,
        RGB_STRUCT *psLineBitmap
            );

} IMAGE_INTERFACE_STRUCT;

  /***************************************/
 /***----       Data Services      ----**/
/***************************************/

// Data Service Identifier
typedef UN16 DSI;
#define DSI_INVALID_ID (UN16_MAX)

// Data Multiplex Identifier
typedef UN16 SXM_DMI;
#define SXM_DMI_INVALID_ID (UN16_MAX)

// Data Service States
typedef enum dataservice_state_enum
{
    DATASERVICE_STATE_INITIAL = 0,
    DATASERVICE_STATE_UNSUBSCRIBED,
    DATASERVICE_STATE_UNAVAILABLE,
    DATASERVICE_STATE_POI_UPDATES_ONLY,
    DATASERVICE_STATE_READY,
    DATASERVICE_STATE_STOPPED,
    DATASERVICE_STATE_ERROR,
    DATASERVICE_STATE_INVALID

} DATASERVICE_STATE_ENUM;

// Data Service Error Codes
typedef enum dataservice_error_code_enum
{
    DATASERVICE_ERROR_CODE_NONE = 0,
    DATASERVICE_ERROR_CODE_GENERAL,
    DATASERVICE_ERROR_CODE_DEVICE_FAILURE,
    DATASERVICE_ERROR_CODE_INVALID_DEVICE,
    DATASERVICE_ERROR_CODE_DATABASE_NOT_FOUND,
    DATASERVICE_ERROR_CODE_DATABASE_VERSION_MISMATCH,
    DATASERVICE_ERROR_CODE_DATABASE_ACCESS_FAILURE,
    DATASERVICE_ERROR_CODE_DATABASE_CORRUPT,
    DATASERVICE_ERROR_CODE_DATABASE_READONLY,
    DATASERVICE_ERROR_CODE_DATABASE_TOO_OLD,
    DATASERVICE_ERROR_UNKNOWN

} DATASERVICE_ERROR_CODE_ENUM;

// Data Service Type
typedef enum dataservice_type_enum
{
    DATASERVICE_TYPE_TRAFFIC,
    DATASERVICE_TYPE_EV_CHARGING,
    DATASERVICE_TYPE_FUEL,
    DATASERVICE_TYPE_CANFUEL,
    DATASERVICE_TYPE_MOVIES,
    DATASERVICE_TYPE_PHONETICS,
    DATASERVICE_TYPE_EPG,
    DATASERVICE_TYPE_AGW,
    DATASERVICE_TYPE_WEATHER,
    DATASERVICE_TYPE_SPORTSDATA,
    DATASERVICE_TYPE_SAFETY_CAMERAS,
    DATASERVICE_TYPE_STOCK_TICKER,
    DATASERVICE_TYPE_WS_ALERTS,
    DATASERVICE_TYPE_TRAFFIC_CAMERAS,
    DATASERVICE_TYPE_MAPS,
    DATASERVICE_TYPE_UNKNOWN

} DATASERVICE_TYPE_ENUM;

// Data Service Event Mask
typedef SMSAPI_EVENT_MASK DATASERVICE_EVENT_MASK;

// Data Service callback for all interactive data services
typedef void (*DATASERVICE_EVENT_CALLBACK) (
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

typedef UN32 DATASERVICE_OPTIONS_MASK;

#define DATASERVICE_OPTION_NONE                   (0x00000000)
#define DATASERVICE_OPTION_REFERENCE_DB_PATH      (0x00000001)
#define DATASERVICE_OPTION_DISABLE_REF_DB_UPDATES (0x00000002)
#define DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT    (0x00000004)

typedef struct dataservice_options_struct
{
    // Fields mask
    DATASERVICE_OPTIONS_MASK tMask;

    // Argument for DATASERVICE_OPTION_REFERENCE_DB_PATH
    char *pcRefDBPath;

    // Argument for DATASERVICE_OPTION_IMAGE_OUTPUT_FORMAT
    IMAGE_FORMAT_ENUM eImageOutputFormat;

    // Argument(s) for something else goes here...

} DATASERVICE_OPTIONS_STRUCT;

/***************************************/
/*****--------- Amenities ---------*****/
/***************************************/

typedef enum amenity_status_enum
{
    AMENITY_STATUS_AVAILABLE,
    AMENITY_STATUS_UNAVAILABLE,
    AMENITY_STATUS_INVALID

} AMENITY_STATUS_ENUM;

typedef enum fuel_station_amenity_enum
{
    FUEL_STATION_AMENITY_OPEN_24HR,
    FUEL_STATION_AMENITY_EMERGENCY_ROAD_SERVICE,
    FUEL_STATION_AMENITY_FULL_SERVICE,
    FUEL_STATION_AMENITY_OIL_CHANGE,
    FUEL_STATION_AMENITY_INTERSTATE_ACCESS_UNDER_HALF_MILE,
    FUEL_STATION_AMENITY_INTERSTATE_ACCESS_OVER_HALF_MILE,
    FUEL_STATION_AMENITY_CASH_DISCOUNT,
    FUEL_STATION_AMENITY_COVENIENCE_STORE,
    FUEL_STATION_AMENITY_SUPERMARKET,
    FUEL_STATION_AMENITY_SNACKS_FAST_FOOD,
    FUEL_STATION_AMENITY_RESTAURANT,
    FUEL_STATION_AMENITY_TRUCK_STOP,
    FUEL_STATION_AMENITY_TRUCK_STOP_WITH_HOTEL,
    FUEL_STATION_AMENITY_AFFILIATED_NETWORK_CARDS_ACCEPTED,
    FUEL_STATION_AMENITY_ALL_NETWORK_CARDS_ACCEPTED,
    FUEL_STATION_AMENITY_CREDIT_DEBIT,
    FUEL_STATION_AMENITY_PUBLIC_ACCESS,
    FUEL_STATION_AMENITY_PATRONS_ONLY,
    FUEL_STATION_AMENITY_CASH_ACCEPTED,
    FUEL_STATION_AMENITY_RESERVATIONS,
    FUEL_STATION_NUM_AMENITIES

} FUEL_STATION_AMENITY_ENUM;

typedef enum theater_amenity_enum
{
    THEATER_AMENITY_STADIUM_SEATING,
    THEATER_AMENITY_STADIUM_ROCKER_SEATING,
    THEATER_NUM_AMENITIES

} THEATER_AMENITY_ENUM;

typedef enum ski_amenity_enum
{
    SKI_AMENITY_SKI_LIFTS,
    SKI_AMENITY_SKI_TRAILS,
    SKI_AMENITY_SNOW_MADE_LAST_24HRS,
    SKI_AMENITY_SNOW_GROOMED_LAST_24HRS,
    SKI_AMENITY_NIGHT_SKIING_PERMITTED,
    SKI_AMENITY_SNOWBOARDING,
    SKI_NUM_AMENITIES

} SKI_AMENITY_ENUM;

typedef union amenity_union
{
    FUEL_STATION_AMENITY_ENUM eFuelStationAmenity;
    THEATER_AMENITY_ENUM eTheaterAmenity;
    SKI_AMENITY_ENUM eSkiAmenity;

} AMENITY_UNION;

typedef struct amenity_struct
{
    AMENITY_STATUS_ENUM eStatus;
    AMENITY_UNION uAmenity;

} AMENITY_STRUCT;

/***************************************/
/*****--------- Favorites ---------*****/
/***************************************/
typedef struct favorites_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eSetFavorite) (
        DSRL_ENTRY_OBJECT hEntry,
        BOOLEAN bIsFavorite
            );

    BOOLEAN (*bIsFavorite) (
        DSRL_ENTRY_OBJECT hEntry
            );

    void (*vClearAllFavorites) (
        DATASERVICE_MGR_OBJECT hManager
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetFavoriteByLocID) (
        LOCID_OBJECT hLocID,
        DATASERVICE_MGR_OBJECT hManager,
        BOOLEAN bIsFavorite
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetFavoriteByStockSymbol) (
        STOCK_SYMBOL_OBJECT hStockSymbol,
        DATASERVICE_MGR_OBJECT hManager,
        BOOLEAN bIsFavorite
            );

} FAVORITES_INTERFACE_STRUCT;

/***************************************/
/*****---- Generic Data Service ----****/
/***************************************/
typedef UN32 DATASERVICE_REF_DATA_VER;
#define DATASERVICE_REF_DATA_INVALID_VER (UN32_MAX)

typedef struct dataservice_state_change_struct
{
    DATASERVICE_STATE_ENUM ePreviousState;
    DATASERVICE_STATE_ENUM eCurrentState;

} DATASERVICE_STATE_CHANGE_STRUCT;

// Generic Data Service callback
typedef void (*GENERIC_DATASERVICE_EVENT_CALLBACK)(
    DATASERVICE_MGR_OBJECT hDataService,
    DATASERVICE_EVENT_MASK tCurrentEvent,
    void *pvEventArg,
    void *pvEventCallbackArg
        );

// Structure used when configuring the
// data stream for a generic data service
typedef struct dataservice_dmi_config_struct
{
    // The SXM_DMI to configure
    SXM_DMI tDMI;

    // Flag indicating if this SXM_DMI is
    // enabled (bEnable == TRUE) or
    // disabled (bEnable == FALSE)
    BOOLEAN bEnable;
} DATASERVICE_DMI_CONFIG_STRUCT;

// Data product types
typedef enum data_product_type_enum
{
    // Service-agnostic products
    DATA_PRODUCT_TYPE_DB_UPDATES,

    // Service-specific products

    /* Fuel & EV */
    DATA_PRODUCT_TYPE_FUEL_STATION_LOGO_UPDATES,
    DATA_PRODUCT_TYPE_FUEL_AVAILABILITY_UPDATES,
    DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES,

    // Art
    DATA_PRODUCT_TYPE_ALBUM_ART,
    DATA_PRODUCT_TYPE_CHANNEL_ART,

    // Invalid product type
    DATA_PRODUCT_TYPE_INVALID

} DATA_PRODUCT_TYPE_ENUM;

// Data product states
typedef enum data_product_state_enum
{
    DATA_PRODUCT_STATE_INITIAL,
    DATA_PRODUCT_STATE_ERROR,
    DATA_PRODUCT_STATE_READY,
    DATA_PRODUCT_STATE_UNSUBSCRIBED,
    DATA_PRODUCT_STATE_DISABLED

} DATA_PRODUCT_STATE_ENUM;

// Data product mask type
typedef UN32 DATA_PRODUCT_MASK;
#define DATA_PRODUCT_MASK_NONE ((DATA_PRODUCT_MASK)0)

// Data product iterator callback
typedef BOOLEAN (* DATA_PRODUCT_ITERATOR_CALLBACK)(
    DATASERVICE_MGR_OBJECT hDataService,
    DATA_PRODUCT_TYPE_ENUM eProductType,
    DATA_PRODUCT_MASK tProductMask,
    DATA_PRODUCT_STATE_ENUM eState,
    void *pvCallbackArg
            );

/* External object interface */
typedef struct generic_dataservice_interface_struct
{
    /* Object Members */

    DATASERVICE_MGR_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        DSI tDSI,
        BOOLEAN bEnableAllDMIs,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        GENERIC_DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vStop) (
        DATASERVICE_MGR_OBJECT hDataService
            );

    DATASERVICE_STATE_ENUM (*eState) (
        DATASERVICE_MGR_OBJECT hDataService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        DATASERVICE_MGR_OBJECT hDataService
            );

    SMSAPI_RETURN_CODE_ENUM (*eManageDataStream) (
        DATASERVICE_MGR_OBJECT hDataService,
        DATASERVICE_DMI_CONFIG_STRUCT const *psDMIsToConfigure,
        size_t tNumDMIsToConfigure
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnableProduct)(
        DATASERVICE_MGR_OBJECT hDataService,
        DATA_PRODUCT_TYPE_ENUM eProductType,
        DATA_PRODUCT_MASK tProductMask
            );

    SMSAPI_RETURN_CODE_ENUM (*eDisableProduct)(
        DATASERVICE_MGR_OBJECT hDataService,
        DATA_PRODUCT_TYPE_ENUM eProductType
            );

    SMSAPI_RETURN_CODE_ENUM (*eProductState)(
        DATASERVICE_MGR_OBJECT hDataService,
        DATA_PRODUCT_TYPE_ENUM eProductType,
        DATA_PRODUCT_MASK *ptProductMask,
        DATA_PRODUCT_STATE_ENUM *peProductState
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateProducts)(
        DATASERVICE_MGR_OBJECT hDataService,
        DATA_PRODUCT_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eStartTimedEvent)(
        DATASERVICE_MGR_OBJECT hDataService,
        BOOLEAN bRepeat,
        UN32 un32IntervalInSeconds
            );

    void (*vStopTimedEvent)(
        DATASERVICE_MGR_OBJECT hDataService
            );

    SMSAPI_RETURN_CODE_ENUM (*eSendServiceEvent)(
        DATASERVICE_MGR_OBJECT hDataService
            );

} GENERIC_DATASERVICE_INTERFACE_STRUCT;

  /***************************************/
 /***---- Channel Art Data Service ----**/
/***************************************/

// Channel Art Service Event Mask
typedef SMSAPI_EVENT_MASK CHANNEL_ART_SERVICE_EVENT_MASK;

// Channel Art Object Available Image mask
typedef UN16 CHANNEL_ART_AVAILABLE_IMAGE_MASK;

// Channel Art Image Type Enumeration
typedef enum channel_art_imagetype_enum
{
    CHANNEL_ART_IMAGETYPE_LOGO = 0,
    CHANNEL_ART_IMAGETYPE_SECONDARY_LOGO,
    CHANNEL_ART_IMAGETYPE_BKGRND,
    CHANNEL_ART_IMAGETYPE_ALBUM,
    CHANNEL_ART_IMAGETYPE_MAX

} CHANNEL_ART_IMAGETYPE_ENUM;

// Channel Art Service Update callback
typedef void (*CHANNEL_ART_EVENT_CALLBACK)(
    CHANNEL_ART_SERVICE_OBJECT hChannelArtService,
    CHANNEL_ART_SERVICE_EVENT_MASK tEventMask,
    void *pvEventCallbackArg
        );

// Channel Art Access Callback
typedef void (*CHANNEL_ART_ACCESS_CALLBACK)(
    CHANNEL_ART_OBJECT hChannelArt,
    CHANNEL_ART_AVAILABLE_IMAGE_MASK tAvailableImages,
    void *pvChannelArtAccessCallbackArg
            );

  /*********************************/
 /*** CHANNEL_ART Object        ***/
/*********************************/

// Channel Art Object Event mask
typedef SMSAPI_EVENT_MASK CHANNEL_ART_EVENT_MASK;

typedef UN32 CHANNEL_ART_OPTIONS_MASK;

#define CHANNEL_ART_OPTION_NONE                   (0x00000000)
#define CHANNEL_ART_OPTION_FILE_PATH              (0x00000001)
#define CHANNEL_ART_OPTION_EPHEMERAL_FILE_PATH    (0x00000002)

#define CHANNEL_ART_AVAILABLE_IMAGE_NONE              (0x0000)
#define CHANNEL_ART_AVAILABLE_IMAGE_LOGO              (0x0001)
#define CHANNEL_ART_AVAILABLE_IMAGE_SECONDARY_LOGO    (0x0002)
#define CHANNEL_ART_AVAILABLE_IMAGE_BACKGROUND        (0x0004)
#define CHANNEL_ART_AVAILABLE_IMAGE_ALBUM             (0x0008)

/* External object interface */
typedef struct channel_art_interface_struct
{
    /* Object Members */
    CHANNEL_ART_SERVICE_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        CHANNEL_ART_SERVICE_EVENT_MASK tEventRequestMask,
        CHANNEL_ART_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        CHANNEL_ART_AVAILABLE_IMAGE_MASK tSelectedImageTypes,
        CHANNEL_ART_OPTIONS_MASK tOptions,
        ...
            );

    void (*vStop) (
        CHANNEL_ART_SERVICE_OBJECT hChannelArtService
            );

    DATASERVICE_STATE_ENUM (*eState) (
        CHANNEL_ART_SERVICE_OBJECT hChannelArtService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        CHANNEL_ART_SERVICE_OBJECT hChannelArtService
            );

    SMSAPI_RETURN_CODE_ENUM (*eUseArt) (
        CHANNEL_ART_OBJECT hArt,
        CHANNEL_ART_ACCESS_CALLBACK vChannelArtAccessCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eImage) (
        CHANNEL_ART_OBJECT hArt,
        CHANNEL_ART_IMAGETYPE_ENUM eImageType,
        IMAGE_OBJECT *phImage,
        BOOLEAN *pbIsDefault
            );

} CHANNEL_ART_INTERFACE_STRUCT;

/***************************************/
/***        DSRL_TARGET_OBJECT       ***/
/***************************************/

typedef enum DSRL_TARGET_type_enum
{
    DSRL_TARGET_TYPE_LOCATION = 0,
    DSRL_TARGET_TYPE_STOCK_SYMBOL,
    DSRL_TARGET_TYPE_UNKNOWN

} DSRL_TARGET_TYPE_ENUM;

typedef struct DSRL_TARGET_object_interface_struct
{
    /* Object members */

    DSRL_TARGET_TYPE_ENUM (*eType) (
        DSRL_TARGET_OBJECT hTarget
            );

    LOCATION_OBJECT (*hLocation) (
        DSRL_TARGET_OBJECT hTarget
            );

    STOCK_SYMBOL_OBJECT (*hStockSymbol) (
        DSRL_TARGET_OBJECT hTarget
            );

} DSRL_TARGET_OBJECT_INTERFACE_STRUCT;

/***************************************/
/***        DSRL_ENTRY_OBJECT        ***/
/***************************************/

typedef enum DSRL_ENTRY_type_enum
{
    DSRL_ENTRY_TYPE_TRAFFIC_MSG = 0,
    DSRL_ENTRY_TYPE_FUEL_STATION,
    DSRL_ENTRY_TYPE_THEATER,
    DSRL_ENTRY_TYPE_AGW_TILE,
    DSRL_ENTRY_TYPE_AGW_SHAPE,
    DSRL_ENTRY_TYPE_WEATHER_MSG,
    DSRL_ENTRY_TYPE_STOCK_MSG,
    DSRL_ENTRY_TYPE_SAFETY_CAMERAS_LOCATION,
    DSRL_ENTRY_TYPE_WS_ALERTS_MSG,
    DSRL_ENTRY_TYPE_UNKNOWN
} DSRL_ENTRY_TYPE_ENUM;

typedef struct DSRL_ENTRY_object_interface_struct
{
    /* Object members */

    DSRL_ENTRY_TYPE_ENUM (*eType) (
        DSRL_ENTRY_OBJECT hEntry
            );

    TRAFFIC_MSG_OBJECT (*hTrafficMsg) (
        DSRL_ENTRY_OBJECT hEntry
            );

    FUEL_STATION_OBJECT (*hFuelStation) (
        DSRL_ENTRY_OBJECT hEntry
            );

    THEATER_OBJECT (*hTheater) (
        DSRL_ENTRY_OBJECT hEntry
            );

    AGW_TILE_OBJECT (*hAgwTile) (
        DSRL_ENTRY_OBJECT hEntry
            );

    AGW_SHAPE_OBJECT (*hAgwShape) (
        DSRL_ENTRY_OBJECT hEntry
            );

    WEATHER_MSG_OBJECT (*hWeatherMsg) (
        DSRL_ENTRY_OBJECT hEntry
            );

    STOCK_MSG_OBJECT (*hStockMsg) (
        DSRL_ENTRY_OBJECT hEntry
            );

    SAFETY_CAMERAS_LOCATION_OBJECT (*hSafetyCamerasLocation) (
        DSRL_ENTRY_OBJECT hEntry
            );

    WS_ALERT_MSG_OBJECT (*hWSAlertMsg) (
        DSRL_ENTRY_OBJECT hEntry
            );
} DSRL_ENTRY_OBJECT_INTERFACE_STRUCT;


/********************/
/*** LOCID OBJECT ***/
/********************/

// A generic ID value that is used to define an ID value
// for a location.  The principal member of the LOCID
typedef UN32 LOC_ID;
#define LOC_INVALID_ID UN32_MAX


// An enum that defines the type of LOCID.
typedef enum locid_type_enum
{
  LOCID_TYPE_UNKNOWN = 0,
  LOCID_TYPE_TRAFFIC,
  LOCID_TYPE_THEATER,
  LOCID_TYPE_FUEL_STATION,
  LOCID_TYPE_SAFETY_CAMERAS,
  LOCID_TYPE_WEATHER
} LOCID_TYPE_ENUM;

/* External object interface */
typedef struct locid_object_interface_struct
{
  LOCID_OBJECT (*hCreate) (
      LOC_ID tID
          );

  LOC_ID (*tID) (
      LOCID_OBJECT hLocID
          );

  LOCID_TYPE_ENUM (*eType) (
      LOCID_OBJECT hLocID
          );

  LOCID_OBJECT (*hDuplicate) (
    LOCID_OBJECT hLocID
        );

  N16 (*n16Compare) (
      LOCID_OBJECT hLocID1,
      LOCID_OBJECT hLocID2
        );

  N32 (*n32FWrite) (
      LOCID_OBJECT hLocID,
      FILE *psFile
          );

  LOCID_OBJECT (*hFRead) (
      FILE *psFile
          );

  N32 (*n32FPrintf) (
      LOCID_OBJECT hLocID,
      FILE *psFile
          );

  void (*vDestroy) (
      LOCID_OBJECT hLocID
          );
} LOCID_OBJECT_INTERFACE_STRUCT;

/***********************/
/*** DISTANCE OBJECT ***/
/***********************/
// An enum that defines a type of distance units
typedef enum distance_unit_type_enum
{
  DISTANCE_UNIT_TYPE_UNKNOWN = 0,
  DISTANCE_UNIT_TYPE_MILES,
  DISTANCE_UNIT_TYPE_KILOMETERS
} DISTANCE_UNIT_TYPE_ENUM;

/* External object interface */
typedef struct distance_object_interface_struct
{
  DISTANCE_OBJECT (*hCreate) (
      N32 n32Value,
      DISTANCE_UNIT_TYPE_ENUM eUnits
          );

  DISTANCE_OBJECT (*hDuplicate) (
       DISTANCE_OBJECT hDistance
          );

  OSAL_FIXED_OBJECT (*hValue) (
      DISTANCE_OBJECT hDistance,
      DISTANCE_UNIT_TYPE_ENUM eUnits
          );

  N16 (*n16Compare) (
      DISTANCE_OBJECT hDistance1,
      DISTANCE_OBJECT hDistance2
          );

  N32 (*n32FWrite) (
      DISTANCE_OBJECT hDistance,
      FILE *psFile
          );

  DISTANCE_OBJECT (*hFRead) (
      FILE *psFile
          );

  N32 (*n32FPrintf) (
      DISTANCE_OBJECT hDistance,
      FILE *psFile
          );

  void (*vDestroy) (
      DISTANCE_OBJECT hDistance
          );
} DISTANCE_OBJECT_INTERFACE_STRUCT;

/***********************/
/*** LOCATION OBJECT ***/
/***********************/
/* Location area type */
typedef enum location_type_enum
{
    LOCATION_TYPE_CIRCLE,
    LOCATION_TYPE_RECTANGLE,
    LOCATION_TYPE_UNKNOWN

}  LOCATION_TYPE_ENUM;

/* External object interface */
typedef struct location_object_interface_struct
{
  LOCATION_OBJECT (*hCreateForRadius) (
      OSAL_FIXED_OBJECT  hLat,
      OSAL_FIXED_OBJECT  hLon,
      DISTANCE_OBJECT hRadius
          );

  LOCATION_OBJECT (*hCreateForLocID) (
      OSAL_FIXED_OBJECT  hLat,
      OSAL_FIXED_OBJECT  hLon,
      LOCID_OBJECT    hLocID
          );
  LOCATION_OBJECT (*hCreateRectangle) (
      OSAL_FIXED_OBJECT  hSouthernLat,
      OSAL_FIXED_OBJECT  hWesternLon,
      DISTANCE_OBJECT hDistanceLat,
      DISTANCE_OBJECT hDistanceLon
          );

  LOCATION_OBJECT (*hDuplicate) (
      LOCATION_OBJECT hLocation
          );

  LOCATION_TYPE_ENUM (*eType) (
      LOCATION_OBJECT hLocation
          );

  OSAL_FIXED_OBJECT (*hLat) (
      LOCATION_OBJECT hLocation
          );

  OSAL_FIXED_OBJECT (*hLon) (
      LOCATION_OBJECT hLocation
          );

  DISTANCE_OBJECT (*hRadius) (
      LOCATION_OBJECT hLocation
          );

  DISTANCE_OBJECT (*hWidth) (
      LOCATION_OBJECT hLocation
          );

  DISTANCE_OBJECT (*hHeight) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hDescription) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hStreetNum) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hStreetName) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hCity) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hState) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hZipCode) (
      LOCATION_OBJECT hLocation
          );

  STRING_OBJECT (*hPhone) (
      LOCATION_OBJECT hLocation
          );

  LOCID_OBJECT (*hLocID) (
      LOCATION_OBJECT hLocation
          );

  DISTANCE_OBJECT (*hDistance) (
      LOCATION_OBJECT hLocationFrom,
      LOCATION_OBJECT hLocationTo
          );

  LOCATION_OBJECT (*hLocationAtDistance) (
      LOCATION_OBJECT hLocationFrom,
      DISTANCE_OBJECT hDistance,
      OSAL_FIXED_OBJECT hBearing
          );

  BOOLEAN (*bCoordinatesWithinArea) (
      LOCATION_OBJECT hLocation,
      OSAL_FIXED_OBJECT hLat,
      OSAL_FIXED_OBJECT hLon
          );

  LOCATION_OBJECT (*hClosest) (
      LOCATION_OBJECT hTarget,
      LOCATION_OBJECT hLocation1,
      LOCATION_OBJECT hLocation2
          );

  N32 (*n32FWrite) (
      LOCATION_OBJECT hLocation,
      FILE *psFile
          );

  LOCATION_OBJECT (*hFRead) (
      FILE *psFile
          );

  N32 (*n32FPrintf) (
      LOCATION_OBJECT hLocation,
      FILE *psFile
          );

  void (*vDestroy) (
      LOCATION_OBJECT hLocation
          );

  SMSAPI_RETURN_CODE_ENUM (*eIsPoint) (
      LOCATION_OBJECT hLocation,
      BOOLEAN *pbIsPoint
          );

} LOCATION_OBJECT_INTERFACE_STRUCT;

/*************************************/
/***   WS_ALERTS_LOCATION OBJECT  ***/
/***********************************/
typedef struct ws_alerts_location_interface_struct
{

    SHAPE_OBJECT (*hShape) (
        WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
            );

    STRING_OBJECT (*hLocationExtType) (
        WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
            );

    STRING_OBJECT (*hDescription) (
        WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation
            );

     N32 (*n32FPrintf) (
        WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} WS_ALERTS_LOCATION_INTERFACE_STRUCT;

  /***********************/
 /***   SHAPE OBJECT  ***/
/***********************/

typedef enum shape_type_enum
{
    SHAPE_TYPE_INVALID,
    SHAPE_TYPE_OPEN,
    SHAPE_TYPE_CLOSED
} SHAPE_TYPE_ENUM;

typedef BOOLEAN (*SHAPE_POINTS_ITERATOR_CALLBACK) (
    LOCATION_OBJECT hLocation,
    void *pvArg
        );

typedef BOOLEAN (*SHAPE_LINES_ITERATOR_CALLBACK) (
    OSAL_FIXED_OBJECT hLatBegin,
    OSAL_FIXED_OBJECT hLonBegin,
    OSAL_FIXED_OBJECT hLatEnd,
    OSAL_FIXED_OBJECT hLonEnd,
    void *pvArg
        );

/* External object interface */
typedef struct shape_object_interface_struct
{
    SHAPE_TYPE_ENUM (*eShapeType) (
        SHAPE_OBJECT hShape
            );

    SMSAPI_RETURN_CODE_ENUM (*eIteratePoints) (
        SHAPE_OBJECT hShape,
        SHAPE_POINTS_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    N32 (*n32FPrintf) (
        SHAPE_OBJECT hShape,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} SHAPE_OBJECT_INTERFACE_STRUCT;

  /***************************/
 /*** LOCALIZATION OBJECT ***/
/***************************/

/* Localization type */
typedef enum localization_measurement_system_enum
{
    // US Localization
    LOCALIZATION_MEASUREMENT_SYSTEM_US
} LOCALIZATION_MEASUREMENT_SYSTEM_ENUM;

/* Supported measurement units */
typedef enum locallization_measurement_units_enum
{
  LOCALIZATION_MEASUREMENT_UNITS_UNKNOWN,
  LOCALIZATION_MEASUREMENT_UNITS_INCH,
  LOCALIZATION_MEASUREMENT_UNITS_FAHRENHEIT,
  LOCALIZATION_MEASUREMENT_UNITS_MILES_PER_HOUR

} LOCALIZATION_MEASUREMENT_UNITS_ENUM;

/* External object interface */
typedef struct localization_object_interface_struct
{
  BOOLEAN (*bSetMeasurementSystem) (
      LOCALIZATION_MEASUREMENT_SYSTEM_ENUM eMeasureSystem
          );

  LOCALIZATION_MEASUREMENT_UNITS_ENUM (*eGetTemperatureUnits) (
          void );

  LOCALIZATION_MEASUREMENT_UNITS_ENUM (*eGetDistanceUnits) (
          void );

  LOCALIZATION_MEASUREMENT_UNITS_ENUM (*eGetSpeedUnits) (
          void );

  LOCALIZATION_MEASUREMENT_UNITS_ENUM (*eGetPrecipitationUnits) (
          void );

  SMS_LANGUAGE_ENUM (*eGetLanguage) (
          void );

} LOCALIZATION_OBJECT_INTERFACE_STRUCT;

  /*********************/
 /*** DEVICE OBJECT ***/
/*********************/

typedef SMSAPI_EVENT_MASK DEVICE_EVENT_MASK;

/* Device Location update callback */
typedef void (*DEVICE_UPDATE_CALLBACK) (
    void *pvRegistrationArg,
    DEVICE_EVENT_MASK tEventMask,
    DISTANCE_OBJECT hDistance,
    void *pvCallbackArg
        );

/* External object interface */
typedef struct device_object_interface_struct
{
    BOOLEAN (*bSetPosition) (
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon
            );

    BOOLEAN (*bGetPosition) (
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon
            );

    DEVICE_REGISTRATION_OBJECT (*hRegisterForUpdates) (
        DISTANCE_OBJECT hDistance,
        void *pvRegistrationArg,
        DEVICE_UPDATE_CALLBACK vCallback,
        void *pvCallbackArg
            );

    void (*vUnregisterForUpdates) (
        DEVICE_REGISTRATION_OBJECT hRegistration
            );

    void * (*pvRegistrationArgument) (
        DEVICE_REGISTRATION_OBJECT hRegistration
            );

    BOOLEAN (*bReplaceRegistrationArgument) (
        DEVICE_REGISTRATION_OBJECT hRegistration,
        BOOLEAN bNotify,
        void *pvRegistrationArg
            );

} DEVICE_OBJECT_INTERFACE_STRUCT;

/***************************************/
/***- Data Service Result List (DSRL)**/
/***************************************/

typedef UN32 DSRL_ENTRY_ID;
#define DSRL_ENTRY_INVALID_ID UN32_MIN

typedef enum dsrl_entry_status_enum
{
    DSRL_ENTRY_STATUS_NEW = 0,
    DSRL_ENTRY_STATUS_UNCHANGED,
    DSRL_ENTRY_STATUS_CHANGED,
    DSRL_ENTRY_STATUS_REMOVED,
    DSRL_ENTRY_STATUS_UNKNOWN
} DSRL_ENTRY_STATUS_ENUM;

typedef enum dsrl_state_enum
{
    DSRL_STATE_INITIAL = 0,
    DSRL_STATE_UPDATING,
    DSRL_STATE_READY,
    DSRL_STATE_STOPPED,
    DSRL_STATE_ERROR,
    DSRL_STATE_UNKNOWN

} DSRL_STATE_ENUM;

// Callback when there are changes to the content of the DSRL
typedef void (*DSRL_CALLBACK) (
    DSRL_OBJECT hDSRL,
    DSRL_STATE_ENUM eState,
    void *pvCallbackArg
        );

typedef void (*DSRL_EXCLUSIVE_ACCESS_CALLBACK) (
    DSRL_OBJECT hDSRL,
    void *pvCallbackArg
        );

// Registered Content Iterator Callback definition
typedef BOOLEAN (*DSRL_ITERATOR_CALLBACK) (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_ID tEntryID,
    DSRL_ENTRY_STATUS_ENUM eStatus,
    DSRL_ENTRY_OBJECT hEntry,
    void *pvIteratorArg
        );

// Application provided sort function for the DSRL
typedef N16 (*DSRL_SORT_FUNCTION) (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry1,
    DSRL_ENTRY_OBJECT hEntry2,
    void *pvSortArg
        );

// Application provided filter function for the DSRL
typedef BOOLEAN (*DSRL_FILTER_FUNCTION) (
    DSRL_OBJECT hDSRL,
    DSRL_ENTRY_OBJECT hEntry,
    void *pvFilterArg
        );

// Types of operations which may be performed
// when modifying a DSRL
typedef enum dsrl_modify_operation_enum
{
    DSRL_MODIFY_OPERATION_ADD,
    DSRL_MODIFY_OPERATION_REPLACE,
    DSRL_MODIFY_OPERATION_REMOVE,
    DSRL_MODIFY_OPERATION_REMOVEALL,
    DSRL_MODIFY_OPERATION_INVALID

} DSRL_MODIFY_OPERATION_ENUM;

/* External object interface */
typedef struct dsrl_object_interface_struct
{
    DSRL_OBJECT (*hCreate) (
        DATASERVICE_MGR_OBJECT hDataServiceMgr,
        DSRL_TARGET_OBJECT *phTargetList,
        size_t tNumTargets,
        size_t tCapacity,
        DSRL_SORT_FUNCTION n16Sort,
        void *pvSortArg,
        DSRL_FILTER_FUNCTION bFilterItem,
        void *pvFilterArg,
        DSRL_CALLBACK vCallback,
        void *pvCallbackArg
            );

    DSRL_OBJECT (*hCreateForDeviceLocation) (
        DATASERVICE_MGR_OBJECT hDataServiceMgr,
        DISTANCE_OBJECT hRadiusOfInterest,
        size_t tCapacity,
        DSRL_SORT_FUNCTION n16Sort,
        void *pvSortArg,
        DSRL_FILTER_FUNCTION bFilterItem,
        void *pvFilterArg,
        DSRL_CALLBACK vCallback,
        void *pvCallbackArg
            );

    DSRL_OBJECT (*hCreateFavorites) (
        DATASERVICE_MGR_OBJECT hDataServiceMgr,
        DSRL_SORT_FUNCTION n16Sort,
        void *pvSortArg,
        DSRL_CALLBACK vCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyTargets) (
        DSRL_OBJECT hDSRL,
        DSRL_MODIFY_OPERATION_ENUM eOperation,
        DSRL_TARGET_OBJECT *phTargetList,
        size_t tNumTargets
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyCapacity) (
        DSRL_OBJECT hDSRL,
        size_t tCapacity
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyFilter) (
        DSRL_OBJECT hDSRL,
        DSRL_FILTER_FUNCTION bFilterItem,
        void *pvFilterArgs
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifySort) (
        DSRL_OBJECT hDSRL,
        DSRL_SORT_FUNCTION n16Sort,
        void *pvSortArgs
            );

    SMSAPI_RETURN_CODE_ENUM (*eModifyDeviceRadius) (
        DSRL_OBJECT hDSRL,
        DISTANCE_OBJECT hRadius
            );

    void (*vDestroy) (
        DSRL_OBJECT hDSRL
            );

    DATASERVICE_MGR_OBJECT (*hDataServiceMgr) (
        DSRL_OBJECT hDSRL
            );

    LOCATION_OBJECT (*hDeviceLocation) (
        DSRL_OBJECT hDSRL
            );

    DATASERVICE_TYPE_ENUM (*eType) (
        DSRL_OBJECT hDSRL
            );

    DSRL_STATE_ENUM (*eState) (
        DSRL_OBJECT hDSRL
            );

    size_t (*tCapacity) (
       DSRL_OBJECT hDSRL
            );

    BOOLEAN (*bTargetsUpdated) (
        DSRL_OBJECT hDSRL
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        DSRL_OBJECT hDSRL,
        DSRL_ITERATOR_CALLBACK bEntryIterator,
        void *pvIteratorArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eExclusiveAccess) (
        DSRL_OBJECT hDSRL,
        DSRL_EXCLUSIVE_ACCESS_CALLBACK vAccessCallback,
        void *pvAccessArg
            );

} DSRL_OBJECT_INTERFACE_STRUCT;

/**************************************************/
/***--- Electronic Program Guide Data Service ---**/
/**************************************************/
// Number of EPG segments
#define EPG_SEGMENTS_MAX (8)
#define EPG_SEGMENTS_MIN (1)

// EPG Schedule Segment states
typedef enum epg_seg_state_enum
{
    EPG_SEG_STATE_WAIT   = 'W',
    EPG_SEG_STATE_READY  = 'R',
    EPG_SEG_STATE_LOADED = 'L',
    EPG_SEG_STATE_SAVED  = 'S'
} EPG_SEG_STATE_ENUM;

// Program flags
typedef UN16 PROGRAM_FLAGS;

#define EPG_PROGRAM_UNDEFINED       ((PROGRAM_FLAGS)(0x0000))
#define EPG_PROGRAM_FEATURED        ((PROGRAM_FLAGS)(0x0001))
#define EPG_PROGRAM_HIGHLIGHTED     ((PROGRAM_FLAGS)(0x0002))
#define EPG_PROGRAM_LIVE            ((PROGRAM_FLAGS)(0x0004))
#define EPG_PROGRAM_NEW_THIS        ((PROGRAM_FLAGS)(0x0008))

#define EPG_PROGRAM_ALL_FLAGS       ((PROGRAM_FLAGS)(   \
    EPG_PROGRAM_FEATURED    |     \
    EPG_PROGRAM_HIGHLIGHTED |     \
    EPG_PROGRAM_LIVE        |     \
    EPG_PROGRAM_NEW_THIS          \
                              ))
#define EPG_PROGRAM_INVALID_FLAGS ((PROGRAM_FLAGS)(~EPG_PROGRAM_ALL_FLAGS))

// Program recording options
typedef enum epg_program_recording_option_enum
{
    EPG_PROGRAM_RECORD_UNDEFINED = 0,
    EPG_PROGRAM_RECORD_NONE,
    EPG_PROGRAM_RECORD_RESTRICTED,
    EPG_PROGRAM_RECORD_APPROVED,
    EPG_PROGRAM_RECORD_ALWAYS

} EPG_PROGRAM_RECORDING_OPTION_ENUM;

// EPG Schedule segments
typedef enum epg_program_day_enum
{
    PROGRAM_DAY_TODAY = 0,
    PROGRAM_DAY_DAY0 = PROGRAM_DAY_TODAY,
    PROGRAM_DAY_TOMORROW = 1,
    PROGRAM_DAY_DAY1 = PROGRAM_DAY_TOMORROW,
    PROGRAM_DAY_DAY2 = 2,
    PROGRAM_DAY_DAY3 = 3,
    PROGRAM_DAY_DAY4 = 4,
    PROGRAM_DAY_DAY5 = 5,
    PROGRAM_DAY_DAY6 = 6,
    PROGRAM_DAY_DAY7 = 7,
    PROGRAM_DAY_MAX = PROGRAM_DAY_DAY7,
    PROGRAM_INVALID_DAY_ID = UN8_MAX
} PROGRAM_DAY_ID;

typedef enum epg_schedule_state_enum
{
    SCHEDULE_STATE_INVALID,
    SCHEDULE_STATE_PROGRESS,
    SCHEDULE_STATE_READY_PARTIAL,
    SCHEDULE_STATE_READY_FULL

} SCHEDULE_STATE_ENUM;

typedef struct epg_schedule_state_struct
{
    UN8 un8Version;
    UN16 un16Epoch;
    SCHEDULE_STATE_ENUM eState;
    UN8 un8NumSegments;
    EPG_SEG_STATE_ENUM aeSegState[EPG_SEGMENTS_MAX];

} SCHEDULE_STATE_STRUCT;

typedef struct epg_filter_struct
{
    /*Starting time of searching programs*/
    TIME_T      tStartTime;

    /*Ending time of searching programs*/
    TIME_T      tEndTime;

    /*Amount of channels in array to search programs in*/
    UN8             u8NumChannels;

    /*Array of channel's SIDs to search programs in*/
    SERVICE_ID      *ptChannels;

#if 0 // Currently not supported. Reserved for future use.
    /*Amount of topics in array to search programs with*/
    UN16             u16NumTopics;

    /*Array of topics to search programs with*/
    TOPIC_ID        *ptTopics;
#endif

    /*Amount of Series IDs in array to search*/
    UN16             u16NumSeries;

    /*Array of Series IDs to search*/
    SERIES_ID        *ptSeries;

    /*Amount of Program IDs in array to search*/
    UN16             u16NumPrograms;

    /*Array of Program IDs to search*/
    PROGRAM_ID        *ptPrograms;

    /*Flags to search programs with*/
    PROGRAM_FLAGS   tEpgFlags;

    /* Recording Option to search program with */
    EPG_PROGRAM_RECORDING_OPTION_ENUM eRecordingOption;

    /*Substring to be searched in all text fields of each program object*/
    STRING_OBJECT hSearchSubstring;

#if 0 // Currently not supported. Reserved for future use.
    /*Get the list of today's playing programs according to EpgSdk recommendations */
    BOOLEAN bHotNow;

    /*Get the list of this week's playing programs according to EpgSdk recommendations */
    BOOLEAN bHotThisWeek;

    /*this parameter allows to get list of programs for particular day\segment (SHSEG)*/
    PROGRAM_DAY_ID  tDayID;//!!!should be initialized by UN8_MAX if unused!!!
#endif

    /*Get the list of unique programs for each channel (without repeats) */
    BOOLEAN bUnique;

} EPG_FILTER_STRUCT;

// EPG iterator's input parameters
typedef struct epg_events_iterator_struct
{
    PROGRAM_ITERATOR_CALLBACK bEpgIterator;
    void *pvIteratorArg;
    EPG_FILTER_STRUCT *psEpgFilter;
    BOOLEAN bContinue;
    UN8 un8NumSegmentsAvailable;
} EPG_EVENTS_ITERATOR_STRUCT;

/*********************************/
/********** EPG Object ***********/
/*********************************/

typedef struct epg_printf_struct
{
    N32 n32CCount;
    FILE *psFile;
} PRINTF_STRUCT;

/* External object interface */
typedef struct eprogram_object_interface_struct
{
    /* Object Members */

    SERVICE_ID (*tChannelSID) (
        PROGRAM_OBJECT hEpgEvent
            );

    SERIES_ID (*tSeriesId)(
        PROGRAM_OBJECT hEpgEvent
            );

    PROGRAM_ID (*tProgramId)(
        PROGRAM_OBJECT hEpgEvent
            );

    TIME_T (*tStartTime)(
            PROGRAM_OBJECT hEpgEvent
                );

    TIME_T (*tEndTime) (
            PROGRAM_OBJECT hEpgEvent
                );

    PROGRAM_DAY_ID (*tEventDay) (
            PROGRAM_OBJECT hEpgEvent
                );

    PROGRAM_FLAGS (*tPFlags) (
            PROGRAM_OBJECT hEpgEvent
                ); // 12bit mask - featured/highlited/live/new...

    BOOLEAN (*bGetShortName)(
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT  hShortName);

    BOOLEAN (*bGetLongName)(
        PROGRAM_OBJECT hEpgEvent,
        STRING_OBJECT hLongName);

    BOOLEAN (*bGetSeriesDescription) (
            PROGRAM_OBJECT hEpgEvent,
            STRING_OBJECT hSeriesDescription); // handle to series description string

    BOOLEAN (*bGetProgramDescription) (
            PROGRAM_OBJECT hEpgEvent,
            STRING_OBJECT hProgramDescription); // handle to program description string

    UN16 (*un16OrigDate) (
            PROGRAM_OBJECT hEpgEvent
                ); // day (epoch) when program was originally aired

    EPG_TOPICS_LIST (*hEpgTopics) (
            PROGRAM_OBJECT hEpgEvent
                );

    N32 (*n32FPrintf) (
        PROGRAM_OBJECT hEpgEvent,
        FILE *psFile
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateTopics) (
        EPG_TOPICS_LIST hEpgTopics,
        EPG_TOPIC_ITERATOR_CALLBACK bEpgTopicIterator,
        void *pvIteratorArg
            );

    EPG_PROGRAM_RECORDING_OPTION_ENUM (*eRecordingOption) (
        PROGRAM_OBJECT hEpgEvent
            );

} PROGRAM_OBJECT_INTERFACE_STRUCT;

typedef struct epg_topic_interface_struct
{
    TOPIC_ID (*tTopicId) (
            TOPIC_OBJECT    hEpgTopic
        );

    STRING_OBJECT (*hTopicName) (
            TOPIC_OBJECT hEpgTopic
                );

    N32 (*n32FPrintf) (
            TOPIC_OBJECT hEpgEvent,
            FILE *psFile
                );

} EPG_TOPIC_INTERFACE_STRUCT;

// EPG Object Event mask
typedef SMSAPI_EVENT_MASK EPG_EVENT_MASK;

/* External object interface */
typedef struct epg_interface_struct
{
  /* Object Members */

  EPG_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      UN8 un8NumSegmentsForRecieve,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
          );

  void (*vStop) (
      EPG_SERVICE_OBJECT hEpgService
          );

  DATASERVICE_STATE_ENUM (*eState) (
      EPG_SERVICE_OBJECT hEpgService
          );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      EPG_SERVICE_OBJECT hEpgService
          );

  SMSAPI_RETURN_CODE_ENUM (*eIteratePrograms) (
      EPG_SERVICE_OBJECT hEpgService,
      PROGRAM_ITERATOR_CALLBACK bEpgIterator,
      void *pvIteratorArg,
      EPG_FILTER_STRUCT *psEpgFilter
  );

  SMSAPI_RETURN_CODE_ENUM (*eScheduleStates) (
      EPG_SERVICE_OBJECT hEpgService,
      SCHEDULE_STATE_STRUCT *psCurrent,
      SCHEDULE_STATE_STRUCT *psInProcess
  );

  SMSAPI_RETURN_CODE_ENUM (*eEnableCheckVersionsBySegments) (
      EPG_SERVICE_OBJECT hEpgService,
      BOOLEAN bEnable
  );
} EPG_INTERFACE_STRUCT;

/***************************************/
/***----   TRAFFIC Data Service   ----**/
/***************************************/

// The position code is derived from the code list
// published by the Navigation Consortium communizing
// the position codes between the NavTeq and TeleAtlas
// navigation map data bases.  The position codes utilized
// in the SXM traffic system shall adhere to the latest
// Navigation Consortium US TMC location table release.
typedef LOC_ID TRAFFIC_POS_CODE;
#define TRAFFIC_INVALID_POS_CODE UN32_MIN

typedef UN8 TRAFFIC_MARKET;
#define TRAFFIC_INVALID_MARKET UN8_MAX

typedef TRAFFIC_POS_CODE TRAFFIC_BSA;
#define TRAFFIC_INVALID_BSA TRAFFIC_INVALID_POS_CODE

typedef UN16 TRAFFIC_EVENT_CODE;
#define TRAFFIC_INVALID_EVENT_CODE ((TRAFFIC_EVENT_CODE)0)

typedef N8 TRAFFIC_OFFSET;

// This enumeration defines the flow of traffic the
// traffic message is using
typedef enum traffic_direction_enum
{
  TRAFFIC_DIRECTION_POSITIVE = 0,        //With Traffic
  TRAFFIC_DIRECTION_NEGATIVE,            //Against Traffic
  TRAFFIC_DIRECTION_UNKNOWN
} TRAFFIC_DIRECTION_ENUM;

typedef enum traffic_speed_enum
{
    TRAFFIC_SPEED_UNKNOWN=0,
    TRAFFIC_SPEED_5MPH,
    TRAFFIC_SPEED_10MPH,
    TRAFFIC_SPEED_20MPH,
    TRAFFIC_SPEED_25MPH,
    TRAFFIC_SPEED_30MPH,
    TRAFFIC_SPEED_40MPH,
    TRAFFIC_SPEED_45MPH,
    TRAFFIC_SPEED_10KMH = TRAFFIC_SPEED_5MPH,
    TRAFFIC_SPEED_20KMH = TRAFFIC_SPEED_10MPH,
    TRAFFIC_SPEED_30KMH = TRAFFIC_SPEED_20MPH,
    TRAFFIC_SPEED_40KMH = TRAFFIC_SPEED_25MPH,
    TRAFFIC_SPEED_50KMH = TRAFFIC_SPEED_30MPH,
    TRAFFIC_SPEED_60KMH = TRAFFIC_SPEED_40MPH,
    TRAFFIC_SPEED_70KMH = TRAFFIC_SPEED_45MPH,
    TRAFFIC_SPEED_FLOWING_FREELY,

} TRAFFIC_SPEED_ENUM;

typedef enum traffic_duration_enum
{
    TRAFFIC_DURATION_UNKNOWN=0,
    TRAFFIC_DURATION_15_MINUTES,
    TRAFFIC_DURATION_30_MINUTES,
    TRAFFIC_DURATION_1_HOUR,
    TRAFFIC_DURATION_2_HOURS,
    TRAFFIC_DURATION_3_HOURS,
    TRAFFIC_DURATION_4_HOURS,
    TRAFFIC_DURATION_REST_OF_DAY,
    TRAFFIC_DURATION_COUNT

} TRAFFIC_DURATION_ENUM;

// Traffic Position Code Callback
typedef void (*TRAFFIC_POSCODE_LOOKUP_CALLBACK)(
  TRAFFIC_SERVICE_OBJECT hTrafficService,
  TRAFFIC_MARKET tMarket,
  TRAFFIC_POS_CODE tPosCode,
  TRAFFIC_DIRECTION_ENUM eDirection,
  UN8 un8MaxLocationsReq,
  UN8 *un8NumLocations,
  LOCATION_OBJECT *pahLocations,
  void *pvArg
      );

/*********************************/
/***      TRAFFIC Object       ***/
/*********************************/

/* External object interface */
typedef struct traffic_interface_struct
{
/* Object Members */

  TRAFFIC_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      TRAFFIC_POSCODE_LOOKUP_CALLBACK vTrafficPosLookUpCallback,
      void *pvPosLookUpCallbackArg,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

  void (*vStop) (
      TRAFFIC_SERVICE_OBJECT hTrafficService
        );

  DATASERVICE_STATE_ENUM (*eState) (
      TRAFFIC_SERVICE_OBJECT hTrafficService
        );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      TRAFFIC_SERVICE_OBJECT hTrafficService
        );

  SMSAPI_RETURN_CODE_ENUM (*eAddMarket) (
      TRAFFIC_SERVICE_OBJECT hTrafficService,
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  SMSAPI_RETURN_CODE_ENUM (*eRemoveMarket) (
      TRAFFIC_SERVICE_OBJECT hTrafficService,
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

} TRAFFIC_INTERFACE_STRUCT;

/****************************/
/*** TRAFFIC_LOCID OBJECT ***/
/****************************/

typedef enum traffic_locid_type_enum
{
    TRAFFIC_LOCID_TYPE_INVALID = 0,
    TRAFFIC_LOCID_TYPE_MARKET,
    TRAFFIC_LOCID_TYPE_BSA,
    TRAFFIC_LOCID_TYPE_POINT
} TRAFFIC_LOCID_TYPE_ENUM;


/* External object interface */
typedef struct traffic_locid_object_interface_struct
{
  TRAFFIC_LOCID_OBJECT (*hCreate) (
      TRAFFIC_POS_CODE tPosCode,
      TRAFFIC_MARKET tMarket,
      TRAFFIC_BSA tBSA
          );

  TRAFFIC_LOCID_OBJECT (*hCreateExt) (
      TRAFFIC_POS_CODE tPosCode,
      TRAFFIC_MARKET tMarket,
      TRAFFIC_BSA tBSA,
      TRAFFIC_OFFSET tOffset
          );

  TRAFFIC_POS_CODE (*tPosCode) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  TRAFFIC_MARKET (*tMarket) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  TRAFFIC_BSA (*tBSA) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  TRAFFIC_OFFSET (*tOffset) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  TRAFFIC_LOCID_TYPE_ENUM (*eType) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

  N16 (*n16CompareMarkets) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID1,
      TRAFFIC_LOCID_OBJECT hTrafficLocID2
          );

  N16 (*n16CompareBSAs) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID1,
      TRAFFIC_LOCID_OBJECT hTrafficLocID2
          );

  TRAFFIC_LOCID_OBJECT (*hFRead) (
      FILE *psFile
          );

  void (*vDestroy) (
      TRAFFIC_LOCID_OBJECT hTrafficLocID
          );

} TRAFFIC_LOCID_OBJECT_INTERFACE_STRUCT;

/****************************/
/*** TRAFFIC_MSG OBJECT ***/
/****************************/
typedef enum alertc_freeform_data_label_enum
{
    ALERTC_FREEFORM_DATA_LABEL_DURATION= 0,
    ALERTC_FREEFORM_DATA_LABEL_CONTROL,
    ALERTC_FREEFORM_DATA_LABEL_LENGTH_OF_AFFECTED_ROUTE,
    ALERTC_FREEFORM_DATA_LABEL_SPEED_LIMIT,
    ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_5,
    ALERTC_FREEFORM_DATA_LABEL_QUANTIFIER_8,
    ALERTC_FREEFORM_DATA_LABEL_SUPPL_INFO,
    ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_START_TIME,
    ALERTC_FREEFORM_DATA_LABEL_EXPLICIT_STOP_TIME,
    ALERTC_FREEFORM_DATA_LABEL_ADDITIONAL_EVENT,
    ALERTC_FREEFORM_DATA_LABEL_DETAILED_DIVERSION_ADVISE,
    ALERTC_FREEFORM_DATA_LABEL_DESTINATION,
    ALERTC_FREEFORM_DATA_LABEL_RESERVED_12,
    ALERTC_FREEFORM_DATA_LABEL_SOURCE_OF_PROBLEM,
    ALERTC_FREEFORM_DATA_LABEL_SEPARATOR,
    ALERTC_FREEFORM_DATA_LABEL_RESERVED_15,
    ALERTC_FREEFORM_DATA_LABEL_UNKNOWN
} ALERTC_FREEFORM_DATA_LABEL_ENUM;

typedef enum traffic_message_type_enum
{
    TRAFFIC_MSG_TYPE_UNKNOWN,
    TRAFFIC_MSG_TYPE_INCIDENT,
    TRAFFIC_MSG_TYPE_SPEED_AND_FLOW,
    TRAFFIC_MSG_TYPE_CAMERA_IMAGE
} TRAFFIC_MSG_TYPE_ENUM;

typedef enum traffic_message_class_enum
{
    TRAFFIC_MSG_CLASS_UNKNOWN = 0,
    TRAFFIC_MSG_CLASS_LEVEL_OF_SERVICE,
    TRAFFIC_MSG_CLASS_EXPECTED_LEVEL_OF_SERVICE,
    TRAFFIC_MSG_CLASS_ACCIDENTS,
    TRAFFIC_MSG_CLASS_INCIDENTS,
    TRAFFIC_MSG_CLASS_CLOSURES_AND_RESTRICTIONS,
    TRAFFIC_MSG_CLASS_ROAD_RESTRICTIONS,
    TRAFFIC_MSG_CLASS_EXIT_RESTRICTIONS ,
    TRAFFIC_MSG_CLASS_ENTRY_RESTRICTIONS ,
    TRAFFIC_MSG_CLASS_TRAFFIC_RESTRICTIONS,
    TRAFFIC_MSG_CLASS_CARPOOL_INFO,
    TRAFFIC_MSG_CLASS_ROADWORKS,
    TRAFFIC_MSG_CLASS_OBSTRUCTION_HAZARDS,
    TRAFFIC_MSG_CLASS_DANGER_SITUATIONS,
    TRAFFIC_MSG_CLASS_ROAD_CONDITIONS,
    TRAFFIC_MSG_CLASS_TEMPERATURES,
    TRAFFIC_MSG_CLASS_PERCIPITATION_VISIBILITY,
    TRAFFIC_MSG_CLASS_WIND_AND_AIR_QUALITY,
    TRAFFIC_MSG_CLASS_ACTIVITIES,
    TRAFFIC_MSG_CLASS_SECURITY_ALERTS,
    TRAFFIC_MSG_CLASS_DELAYS,
    TRAFFIC_MSG_CLASS_CANCELATIONS,
    TRAFFIC_MSG_CLASS_TRAVEL_TIME_INFO,
    TRAFFIC_MSG_CLASS_DANGEROUS_VEHICLES,
    TRAFFIC_MSG_CLASS_EXCEPTIONAL_LOAD_VEHICLES,
    TRAFFIC_MSG_CLASS_TRAFFIC_EQUIP_STATUS,
    TRAFFIC_MSG_CLASS_SIZE_WEIGHT_LIMITS,
    TRAFFIC_MSG_CLASS_PARKING_RESTRICTIONS,
    TRAFFIC_MSG_CLASS_PARKING,
    TRAFFIC_MSG_CLASS_REF_TO_AUDIO_BDCAST,
    TRAFFIC_MSG_CLASS_SERVICE_MESSAGES,
    TRAFFIC_MSG_CLASS_SPECIAL_MESSAGES,
    TRAFFIC_MSG_CLASS_32,
    TRAFFIC_MSG_CLASS_33,
    TRAFFIC_MSG_CLASS_34,
    TRAFFIC_MSG_CLASS_35,
    TRAFFIC_MSG_CLASS_36,
    TRAFFIC_MSG_CLASS_37,
    TRAFFIC_MSG_CLASS_38,
    TRAFFIC_MSG_CLASS_39,
    TRAFFIC_MSG_CLASS_COUNT

} TRAFFIC_MSG_CLASS_ENUM;

// Iterator callback function used with the TRAFFIC_MSG.eIterateLocations
// API to iterate the list of LOCATION objects that are currently
// in the TRAFFIC MSG.
typedef BOOLEAN (*TRAFFIC_MSG_LOCATION_ITERATOR) (
    LOCATION_OBJECT hLocation,
    void *pvIteratorArg
        );

/* External object interface */
typedef struct traffic_msg_object_interface_struct
{
  TRAFFIC_MSG_TYPE_ENUM (*eType) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  TRAFFIC_MSG_CLASS_ENUM (*eClass) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  TRAFFIC_SPEED_ENUM (*eSpeed) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  TRAFFIC_DURATION_ENUM (*eDuration) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  TRAFFIC_DIRECTION_ENUM (*eDirection) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  BOOLEAN (*bDiversionAdvised) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  TRAFFIC_EVENT_CODE (*tEventCode) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  STRING_OBJECT (*hText) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  STRING_OBJECT (*hAdditionalText) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  UN32 (*un32StartTime) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  UN32 (*un32StopTime) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  UN8 (*un8GetFreeFormData) (
      TRAFFIC_MSG_OBJECT hTrafficMsg,
      ALERTC_FREEFORM_DATA_LABEL_ENUM eLabel,
      UN16 *paun16Values,
      size_t tCount
          );

  UN8 (*un8NumLocations) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

  LOCATION_OBJECT (*hGetLocation) (
      TRAFFIC_MSG_OBJECT hTrafficMsg,
      UN8 un8Index
          );

  N32 (*n32FPrintf) (
      TRAFFIC_MSG_OBJECT hTrafficMsg,
      FILE *psFile,
      SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
          );

  SMSAPI_RETURN_CODE_ENUM  (*eIterateLocations) (
      TRAFFIC_MSG_OBJECT hTrafficMsg,
      TRAFFIC_MSG_LOCATION_ITERATOR bIterator,
      void *pvIteratorArg
          );

  SMSAPI_RETURN_CODE_ENUM (*eCameraDirection) (
      TRAFFIC_MSG_OBJECT hTrafficMsg,
      UN16 *pun16Direction
          );

  IMAGE_OBJECT (*hImage) (
      TRAFFIC_MSG_OBJECT hTrafficMsg
          );

} TRAFFIC_MSG_OBJECT_INTERFACE_STRUCT;

/***************************************/
/***---  Fuel Price Data Service ----***/
/***************************************/

// Enumeration used to specify the fuel
// sorting method
typedef enum fuel_price_sort_method_enum
{
    FUEL_PRICE_SORT_METHOD_PRICE,
    FUEL_PRICE_SORT_METHOD_FUELTYPE,
    FUEL_PRICE_SORT_METHOD_INVALID

} FUEL_PRICE_SORT_METHOD_ENUM;

// Well Known Fuel Types
typedef enum fuel_type_enum
{
    FUEL_TYPE_REGULAR,
    FUEL_TYPE_MIDRANGE,
    FUEL_TYPE_PREMIUM,
    FUEL_TYPE_DIESEL,
    FUEL_TYPE_ELECTRIC_L1_120V,
    FUEL_TYPE_ELECTRIC_L2_240V,
    FUEL_TYPE_ELECTRIC_DC,
    FUEL_TYPE_UNKNOWN

} FUEL_TYPE_ENUM;

/*********************************/
/***    FUEL Service Object    ***/
/*********************************/

// Iterator callback for listing fuel types
typedef BOOLEAN (*FUEL_TYPE_ITERATOR) (
    STRING_OBJECT hFuelType,
    FUEL_TYPE_ENUM eFuelType,
    void *pvIteratorArg
        );

/* External object interface */
typedef struct fuel_interface_struct
{
/* Object Members */

  FUEL_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
          );

  FUEL_SERVICE_OBJECT (*hEVStart) (
      const char *pacSRHDriverName,
      FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
          );

  FUEL_SERVICE_OBJECT (*hCanadianStart) (
      const char *pacSRHDriverName,
      FUEL_PRICE_SORT_METHOD_ENUM eFuelPriceSortMethod,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
          );

  void (*vStop) (
      FUEL_SERVICE_OBJECT hFuelService
          );

  DATASERVICE_STATE_ENUM (*eState) (
      FUEL_SERVICE_OBJECT hFuelService
          );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      FUEL_SERVICE_OBJECT hFuelService
          );

  SMSAPI_RETURN_CODE_ENUM (*eFuelTypes) (
      FUEL_SERVICE_OBJECT hFuelService,
      FUEL_TYPE_ITERATOR bIterator,
      void *pvIteratorArg
          );

  DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
          );

  DATASERVICE_ERROR_CODE_ENUM (*eEVGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
          );

  DATASERVICE_ERROR_CODE_ENUM (*eCanadianGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
          );

} FUEL_INTERFACE_STRUCT;

/*********************************/
/***    FUEL Station Object    ***/
/*********************************/

// Mask which provides information regarding which fields
// of the FUEL_INFO_STRUCT have usable data
typedef UN32 FUEL_INFO_MASK;
#define FUEL_INFO_NONE              ((FUEL_INFO_MASK) (0x000))
#define FUEL_INFO_NUM_POSITIONS     ((FUEL_INFO_MASK) (0x001))
#define FUEL_INFO_DYNAMIC_POSITIONS ((FUEL_INFO_MASK) (0x002))
#define FUEL_INFO_PRICE             ((FUEL_INFO_MASK) (0x004))
#define FUEL_INFO_TIME              ((FUEL_INFO_MASK) (0x008))

typedef enum fuel_availability_enum
{
    FUEL_AVAILABLE,
    FUEL_NOT_AVAILABLE

} FUEL_AVAILABILITY_ENUM;

typedef struct fuel_info_dynamic_position_struct
{
    UN16 un16NumberAvailableRefuelingPositions;
    UN16 un16NumberInUseRefuelingPositions;
    UN16 un16NumberOfflineRefuelingPositions;

} FUEL_INFO_DYNAMIC_POSITION_STRUCT;

typedef struct fuel_info_struct
{
    /* Attributes which aren't masked*/
    FUEL_TYPE_ENUM eFuelType;
    STRING_OBJECT hShortFuelName;
    STRING_OBJECT hLongFuelName;
    FUEL_AVAILABILITY_ENUM eAvailability;
    FUEL_INFO_MASK tInfoMask;

    /* Masked Attributes */

    // Price Value
    UN32 un32PriceInTenthsOfLowestDenomination;

    // Time of report (service dependent)
    UN32 un32ReportTime;

    // Refueling Position Data
    UN16 un16NumberOfRefuelingPositions;
    FUEL_INFO_DYNAMIC_POSITION_STRUCT sDynamicPositionInfo;

} FUEL_INFO_STRUCT;

typedef enum fuel_brand_logo_image_type_enum
{
    FUEL_BRAND_LOGO_TYPE_LOW_RES = 0,
    FUEL_BRAND_LOGO_TYPE_MED_RES,
    FUEL_BRAND_LOGO_TYPE_HIGH_RES,
    FUEL_BRAND_LOGO_INVALID_TYPE

} FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM;

/* External object interface */
typedef struct fuel_station_object_interface_struct
{
  /* Object Members */

  STRING_OBJECT (*hBrand) (
      FUEL_STATION_OBJECT hFuelStation
          );

  IMAGE_OBJECT (*hBrandLogo) (
      FUEL_STATION_OBJECT hFuelStation,
      FUEL_BRAND_LOGO_IMAGE_TYPE_ENUM eLogoType
        );

  STRING_OBJECT (*hAdditionalInfo) (
      FUEL_STATION_OBJECT hFuelStation
          );

  LOCATION_OBJECT (*hLocation) (
      FUEL_STATION_OBJECT hFuelStation
          );

  UN8 (*un8NumAvailableFuelTypes) (
      FUEL_STATION_OBJECT hFuelStation
          );

  SMSAPI_RETURN_CODE_ENUM (*eFuelInfo) (
      FUEL_STATION_OBJECT hFuelStation,
      UN8 un8FuelIndex,
      FUEL_INFO_STRUCT *psFuelInfo
        );

  SMSAPI_RETURN_CODE_ENUM (*eAmenities) (
      FUEL_STATION_OBJECT hFuelStation,
      size_t tNumAmenities,
      AMENITY_STRUCT *pasFuelAmenities
        );

  N32 (*n32FPrintf) (
      FUEL_STATION_OBJECT hFuelStation,
      FILE *psFile
          );

} FUEL_STATION_OBJECT_INTERFACE_STRUCT;

/***************************************/
/***----   MOVIES Data Service    ----**/
/***************************************/

/****************************/
/***     MOVIE OBJECT     ***/
/****************************/

// A unique ID to identify a movie by within the current
// set of movies
typedef N16 MOVIE_ID;
#define MOVIE_INVALID_ID ((MOVIE_ID)-1)

// Known Rating Systems
typedef enum movie_rating_system_enum
{
    MOVIE_RATING_SYSTEM_US_MPAA = 0,
    MOVIE_RATING_SYSTEM_CANADA1,
    MOVIE_RATING_SYSTEM_CANADA2,
    MOVIE_RATING_SYSTEM_QUEBEC,
    MOVIE_RATING_SYSTEM_MEXICO,
    MOVIE_RATING_SYSTEM_UNKNOWN

} MOVIE_RATING_SYSTEM_ENUM;

// A set of enums that can be used to sort movies on
// If a value of MOVIE_RATING_UNKNOWN is used,
// the application should rely on the MOVIE.hRatingText method
// to get the string name for the rating.
typedef enum movie_rating_enum
{
    // US-MPAA
    MOVIE_RATING_MPAA_UNRATED = 0,
    MOVIE_RATING_MPAA_G,
    MOVIE_RATING_MPAA_PG,
    MOVIE_RATING_MPAA_PG13,
    MOVIE_RATING_MPAA_R,
    MOVIE_RATING_MPAA_NC17,

    // CANADA 1
    MOVIE_RATING_CANADA1_UNRATED,
    MOVIE_RATING_CANADA1_G,
    MOVIE_RATING_CANADA1_PG,
    MOVIE_RATING_CANADA1_14A,
    MOVIE_RATING_CANADA1_18A,
    MOVIE_RATING_CANADA1_R,

    // CANADA 2
    MOVIE_RATING_CANADA2_UNRATED,
    MOVIE_RATING_CANADA2_G,
    MOVIE_RATING_CANADA2_PG,
    MOVIE_RATING_CANADA2_14,
    MOVIE_RATING_CANADA2_18,
    MOVIE_RATING_CANADA2_E,
    MOVIE_RATING_CANADA2_R,

    // QUEBECOIS
    MOVIE_RATING_QUEBEC_NONCLASS,
    MOVIE_RATING_QUEBEC_G,
    MOVIE_RATING_QUEBEC_13PLUS,
    MOVIE_RATING_QUEBEC_16PLUS,
    MOVIE_RATING_QUEBEC_18PLUS,

    // MEXICO
    MOVIE_RATING_MEXICO_SC,
    MOVIE_RATING_MEXICO_AA,
    MOVIE_RATING_MEXICO_A,
    MOVIE_RATING_MEXICO_B,
    MOVIE_RATING_MEXICO_B15,
    MOVIE_RATING_MEXICO_C,
    MOVIE_RATING_MEXICO_D,

    // Other
    MOVIE_RATING_UNKNOWN,
    MOVIE_RATING_ERROR

} MOVIE_RATING_ENUM;

// Iterator callback for stepping through the ratings of a particular movie
typedef BOOLEAN (*MOVIE_RATINGS_ITERATOR) (
    MOVIE_RATING_SYSTEM_ENUM eRatingSystem,
    STRING_OBJECT hRatingSystemName,
    MOVIE_RATING_ENUM eRating,
    STRING_OBJECT hRatingText,
    void *pvArg
        );

/* External object interface */
typedef struct movie_object_interface_struct
{
    MOVIE_ID (*tID) (
        MOVIE_OBJECT hMovie
            );

    // Always provides english-language name
    STRING_OBJECT (*hName) (
        MOVIE_OBJECT hMovie
            );

    STRING_OBJECT (*hNameInLanguage) (
        MOVIE_OBJECT hMovie,
        SMS_LANGUAGE_ENUM *peLanguage
            );

    UN8 (*un8RunTime) (
        MOVIE_OBJECT hMovie
            );

    STRING_OBJECT (*hActors) (
        MOVIE_OBJECT hMovie
            );

    // Always provides english-language synopsis
    STRING_OBJECT (*hSynopsis) (
        MOVIE_OBJECT hMovie
            );

    STRING_OBJECT (*hSynopsisInLanguage) (
        MOVIE_OBJECT hMovie,
        SMS_LANGUAGE_ENUM *peLanguage
            );

    // Always provides US-MPAA rating
    MOVIE_RATING_ENUM (*eRating) (
        MOVIE_OBJECT hMovie
            );

    // Always provides US-MPAA rating text
    STRING_OBJECT (*hRatingText) (
        MOVIE_OBJECT hMovie
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateRatings) (
        MOVIE_OBJECT hMovie,
        MOVIE_RATINGS_ITERATOR bIterator,
        void *pvIteratorArg
            );

    N32 (*n32FPrintf) (
        MOVIE_OBJECT hMovie,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} MOVIE_OBJECT_INTERFACE_STRUCT;

/*******************************/
/***     THEATER OBJECT     ***/
/*****************************/

// A unique ID to identify a movie theater within the broadcast
typedef LOC_ID THEATER_ID;
#define THEATER_INVALID_ID ((THEATER_ID)LOC_INVALID_ID)

// Iterator callback for stepping through the movies at a particular theater
typedef BOOLEAN (*THEATER_MOVIES_ITERATOR) (
    THEATER_OBJECT hTheater,
    MOVIE_OBJECT hMovie,
    void *pvIteratorArg
        );

// Iterator callback for stepping through the times for a
// particular movie at a particular theater
typedef BOOLEAN (*THEATER_MOVIE_TIMES_ITERATOR) (
    THEATER_OBJECT hTheater,
    MOVIE_OBJECT hMovie,
    UN32 un32StartTime,
    void *pvIteratorArg
        );

/* External object interface */
typedef struct theater_object_interface_struct
{
    LOCATION_OBJECT (*hLocation) (
        THEATER_OBJECT hTheater
          );

    UN8 (*un8NumMovies) (
        THEATER_OBJECT hTheater
          );

    SMSAPI_RETURN_CODE_ENUM  (*eIterateMovies) (
        THEATER_OBJECT hTheater,
        THEATER_MOVIES_ITERATOR bIterator,
        void *pvIteratorArg
          );

    SMSAPI_RETURN_CODE_ENUM  (*eIterateTimes) (
        THEATER_OBJECT hTheater,
        THEATER_MOVIE_TIMES_ITERATOR bIterator,
        void *pvIteratorArg
          );

    SMSAPI_RETURN_CODE_ENUM (*eAmenities) (
        THEATER_OBJECT hTheater,
        size_t tNumAmenities,
        AMENITY_STRUCT *pasTheaterAmenities
          );

    N32 (*n32FPrintf) (
        THEATER_OBJECT hTheater,
        FILE *psFile
          );

} THEATER_OBJECT_INTERFACE_STRUCT;


/*********************************/
/***      MOVIES Object       ***/
/*******************************/

// Itereator callback function used with the MOVIES.eIterate
// API to iterate the list of MOVIE objects that are currently
// in the broadcast.
typedef BOOLEAN (*MOVIES_ITERATOR) (
    MOVIE_OBJECT hMovie,
    void *pvIteratorArg
        );

/* External object interface */
typedef struct movies_interface_struct
{
  MOVIES_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

  void (*vStop) (
      MOVIES_SERVICE_OBJECT hMoviesService
        );

  DATASERVICE_STATE_ENUM (*eState) (
      MOVIES_SERVICE_OBJECT hMoviesService
        );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      MOVIES_SERVICE_OBJECT hMoviesService
        );

  SMSAPI_RETURN_CODE_ENUM (*eIterate) (
      MOVIES_SERVICE_OBJECT hMoviesService,
      MOVIES_ITERATOR bIterator,
      void *pvIteratorArg
        );

  DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

} MOVIES_INTERFACE_STRUCT;


/***************************************/
/***----   WEATHER Data Service   ----**/
/***************************************/

//Area ID type definition
typedef UN8 AREA_ID;
#define AREA_INVALID_ID ((AREA_ID)UN8_MIN)

//Macro to generate loc_id, state_id and area_id
#define GET_LOC_ID_FROM_STATE_AREA(x, y) \
    ((((LOC_ID)x) << 8) | y)

#define GET_STATE_ID_FROM_LOC_ID(x) \
    ((STATE_ID)((x >> 8) & 0x000000FF))

#define GET_AREA_ID_FROM_LOC_ID(x) \
    ((UN8)(x & 0x000000FF))

/*********************************/
/***      WEATHER Object       ***/
/*********************************/

// Locations Iterator Callback definition
typedef BOOLEAN (*WEATHER_LOCATION_ITERATOR) (
    LOCATION_OBJECT hLocation,
    void *pvIterateArg
        );

/* External object interface */
typedef struct weather_interface_struct
{
/* Object Members */

  WEATHER_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

  void (*vStop) (
      WEATHER_SERVICE_OBJECT hWeatherService
        );

  DATASERVICE_STATE_ENUM (*eState) (
      WEATHER_SERVICE_OBJECT hWeatherService
        );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      WEATHER_SERVICE_OBJECT hWeatherService
        );

  SMSAPI_RETURN_CODE_ENUM (*eEnableSkiReports) (
      WEATHER_SERVICE_OBJECT hWeatherService
        );

  SMSAPI_RETURN_CODE_ENUM (*eDisableSkiReports) (
      WEATHER_SERVICE_OBJECT hWeatherService
        );

  SMSAPI_RETURN_CODE_ENUM (*eIterateLocations) (
      WEATHER_SERVICE_OBJECT hWeatherService,
      BOOLEAN bWeatherSkiFlag,
      STRING_OBJECT hSortName,
      STRING_OBJECT hState,
      WEATHER_LOCATION_ITERATOR bIterator,
      void *pvArg
        );

  DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

} WEATHER_INTERFACE_STRUCT;


  /*********************************/
 /***      TEMPERATURE Object   ***/
/*********************************/

/* External object interface */
typedef struct temperature_object_interface_struct
{
/* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eHigh) (
        TEMPERATURE_OBJECT hTemperature,
        N16 *pn16HighTemperature
            );

    SMSAPI_RETURN_CODE_ENUM (*eLow) (
        TEMPERATURE_OBJECT hTemperature,
        N16 *pn16LowTemperature
            );

    SMSAPI_RETURN_CODE_ENUM (*eCurrent) (
        TEMPERATURE_OBJECT hTemperature,
        N16 *pn16CurrentTemperature
            );

    N32 (*n32FPrintf) (
        TEMPERATURE_OBJECT hTemperature,
        FILE *psFile
            );

} TEMPERATURE_OBJECT_INTERFACE_STRUCT;


  /*********************************/
 /***    WEATHER_EVENT Object   ***/
/*********************************/
typedef enum weather_event_code_enum
{
    WEATHER_EVENT_DATA_NA = 0,
    WEATHER_EVENT_UNKNOWN_PRECIP,
    WEATHER_EVENT_ISOLATED_THUNDERSTORMS,
    WEATHER_EVENT_SCATTERED_THUNDERSTORMS,
    WEATHER_EVENT_SCATTERED_THUNDERSTORMS_NIGHT,
    WEATHER_EVENT_SEVERE_THUNDERSTORMS,
    WEATHER_EVENT_THUNDERSTORMS,
    WEATHER_EVENT_RAIN,
    WEATHER_EVENT_LIGHT_RAIN,
    WEATHER_EVENT_HEAVY_RAIN,
    WEATHER_EVENT_SCATTERED_SHOWERS,
    WEATHER_EVENT_SCATTERED_SHOWERS_NIGHT,
    WEATHER_EVENT_SHOWERS,
    WEATHER_EVENT_DRIZZLE,
    WEATHER_EVENT_FREEZING_DRIZZLE,
    WEATHER_EVENT_FREEZING_RAIN,
    WEATHER_EVENT_WINTRY_MIX,
    WEATHER_EVENT_MIXED_RAIN_AND_SNOW,
    WEATHER_EVENT_MIXED_RAIN_AND_SLEET,
    WEATHER_EVENT_MIXED_RAIN_AND_HAIL,
    WEATHER_EVENT_HAIL,
    WEATHER_EVENT_SLEET,
    WEATHER_EVENT_ICE_PELLETS,
    WEATHER_EVENT_FLURRIES,
    WEATHER_EVENT_LIGHT_SNOW,
    WEATHER_EVENT_MODERATE_SNOW,
    WEATHER_EVENT_SNOW,
    WEATHER_EVENT_HEAVY_SNOW,
    WEATHER_EVENT_SCATTERED_SNOW_SHOWERS,
    WEATHER_EVENT_SCATTERED_SNOW_SHOWERS_NIGHT,
    WEATHER_EVENT_SNOW_SHOWERS,
    WEATHER_EVENT_BLOWING_SNOW,
    WEATHER_EVENT_BLIZZARD,
    WEATHER_EVENT_SANDSTORM,
    WEATHER_EVENT_BLOWING_DUST,
    WEATHER_EVENT_DUST,
    WEATHER_EVENT_FOGGY,
    WEATHER_EVENT_LIGHT_FOG,
    WEATHER_EVENT_MODERATE_FOG,
    WEATHER_EVENT_HEAVY_FOG,
    WEATHER_EVENT_MIST,
    WEATHER_EVENT_HAZY,
    WEATHER_EVENT_SMOKY,
    WEATHER_EVENT_BLUSTERY,
    WEATHER_EVENT_WINDY,
    WEATHER_EVENT_COLD,
    WEATHER_EVENT_HOT,
    WEATHER_EVENT_SUNNY,
    WEATHER_EVENT_MOSTLY_SUNNY,
    WEATHER_EVENT_CLEAR_NIGHT,
    WEATHER_EVENT_MOSTLY_CLEAR_NIGHT,
    WEATHER_EVENT_PARTLY_CLOUDY,
    WEATHER_EVENT_PARTLY_CLOUDY_NIGHT,
    WEATHER_EVENT_MOSTLY_CLOUDY,
    WEATHER_EVENT_MOSTLY_CLOUDY_NIGHT,
    WEATHER_EVENT_CLOUDY,
    WEATHER_EVENT_TROPICAL_STORM,
    WEATHER_EVENT_HURRICANE,
    WEATHER_EVENT_FUNNEL_CLOUD,
    WEATHER_EVENT_TORNADO,
    WEATHER_EVENT_INVALID

} WEATHER_EVENT_CODE_ENUM;

/* External object interface */
typedef struct weather_event_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eCode) (
        WEATHER_EVENT_OBJECT hWeatherEvent,
        WEATHER_EVENT_CODE_ENUM *peEventCode
            );

    STRING_OBJECT (*hDescription) (
        WEATHER_EVENT_OBJECT hWeatherEvent
            );

    SMSAPI_RETURN_CODE_ENUM (*eIconCode) (
        WEATHER_EVENT_OBJECT hWeatherEvent,
        UN8 *pun8EventIconCode
            );

    N32 (*n32FPrintf) (
        WEATHER_EVENT_OBJECT hWeatherEvent,
        FILE *psFile
            );

} WEATHER_EVENT_OBJECT_INTERFACE_STRUCT;

  /*********************************/
 /***    PRECIPITATION Object   ***/
/*********************************/

typedef enum precipitation_type_enum
{
    PRECIPITATION_TYPE_RAIN = 0,
    PRECIPITATION_TYPE_SNOW,
    PRECIPITATION_TYPE_MIXED,
    PRECIPITATION_TYPE_UNKNOWN,
} PRECIPITATION_TYPE_ENUM;


/* External object interface */
typedef struct precipitation_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eType) (
        PRECIPITATION_OBJECT hPrecipitation,
        PRECIPITATION_TYPE_ENUM *pePrecipitationType
            );

    SMSAPI_RETURN_CODE_ENUM (*eAmountRange) (
        PRECIPITATION_OBJECT hPrecipitation,
        OSAL_FIXED_OBJECT hRangeLow,
        OSAL_FIXED_OBJECT hRangeHigh
            );

    SMSAPI_RETURN_CODE_ENUM (*eChance) (
        PRECIPITATION_OBJECT hPrecipitation,
        UN8 *un8PrecipitationChance
            );

    N32 (*n32FPrintf) (
        PRECIPITATION_OBJECT hPrecipitation,
        FILE *psFile
            );

} PRECIPITATION_OBJECT_INTERFACE_STRUCT;


  /*********************************/
 /***             WIND Object   ***/
/*********************************/

/* External object interface */
typedef struct wind_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eDirection) (
        WIND_OBJECT hWind,
        UN16 *pun16WindDirection
            );

    SMSAPI_RETURN_CODE_ENUM (*eSpeed) (
        WIND_OBJECT hWind,
        UN8 *pun8WindSpeed
            );

    N32 (*n32FPrintf) (
        WIND_OBJECT hWind,
        FILE *psFile
            );

} WIND_OBJECT_INTERFACE_STRUCT;


  /*********************************/
 /***       HUMIDITY Object     ***/
/*********************************/

/* External object interface */
typedef struct humidity_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eRange) (
        HUMIDITY_OBJECT hHumidity,
        UN8 *pun8HumidityRangeLow,
        UN8 *pun8HumidityRangeHigh
            );

    N32 (*n32FPrintf) (
        HUMIDITY_OBJECT hHumidity,
        FILE *psFile
            );

} HUMIDITY_OBJECT_INTERFACE_STRUCT;

  /*********************************/
 /***       UVINFO Object       ***/
/*********************************/

typedef enum uv_risk_enum
{
    UV_RISK_LOW = 0,
    UV_RISK_MODERATE,
    UV_RISK_HIGH,
    UV_RISK_VERYHIGH,
    UV_RISK_EXTREME
} UV_RISK_ENUM;


/* External object interface */
typedef struct uvinfo_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eIndexRange) (
        UVINFO_OBJECT hUVInfo,
        UN8 *pun8UVIndexRangeLow,
        UN8 *pun8UVIndexRangeHigh
            );

    SMSAPI_RETURN_CODE_ENUM (*eRiskRating) (
        UVINFO_OBJECT hUVInfo,
        UV_RISK_ENUM *peUVRisk
            );

    N32 (*n32FPrintf) (
        UVINFO_OBJECT hUVInfo,
        FILE *psFile
            );

} UVINFO_OBJECT_INTERFACE_STRUCT;


  /*********************************/
 /***       AIR Object          ***/
/*********************************/

typedef enum air_quality_enum
{
    AIR_QUALITY_GOOD = 0,
    AIR_QUALITY_MODERATE,
    AIR_QUALITY_UNHEALTHY_FOR_SENS,
    AIR_QUALITY_UNHEALTHY,
    AIR_QUALITY_VERY_UNHEALTHY,
    AIR_QUALITY_DANGEROUS,
    AIR_QUALITY_NO_DATA
} AIR_QUALITY_ENUM;

typedef enum air_pollen_level_enum
{
    AIR_POLLEN_LEVEL_LOW = 0,
    AIR_POLLEN_LEVEL_MODERATE,
    AIR_POLLEN_LEVEL_HIGH
} AIR_POLLEN_LEVEL_ENUM;

/* External object interface */
typedef struct air_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eQuality) (
        AIR_OBJECT hAir,
        AIR_QUALITY_ENUM *peAirQuality
            );

    SMSAPI_RETURN_CODE_ENUM (*ePollenCount) (
        AIR_OBJECT hAir,
        UN8 *pun8PollenCount
            );

    SMSAPI_RETURN_CODE_ENUM (*ePollenLevel) (
        AIR_OBJECT hAir,
        AIR_POLLEN_LEVEL_ENUM *pePollenLevel
            );

    N32 (*n32FPrintf) (
        AIR_OBJECT hAir,
        FILE *psFile
            );

} AIR_OBJECT_INTERFACE_STRUCT;

  /*********************************/
 /***     FORECAST Object       ***/
/*********************************/

typedef enum forecast_cloud_cover_enum
{
    FORECAST_CLOUD_COVER_CLEAR = 0,
    FORECAST_CLOUD_COVER_FEW,
    FORECAST_CLOUD_COVER_SCATTERED,
    FORECAST_CLOUD_COVER_BROKEN,
    FORECAST_CLOUD_COVER_OVERCAST,
    FORECAST_CLOUD_COVER_NO_DATA

} FORECAST_CLOUD_COVER_ENUM;

typedef struct forecast_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eWeatherEvent) (
        FORECAST_OBJECT hForecast,
        WEATHER_EVENT_OBJECT *phWeatherEvent
            );

    SMSAPI_RETURN_CODE_ENUM (*eTemperature) (
        FORECAST_OBJECT hForecast,
        TEMPERATURE_OBJECT *phTemperature
            );

    SMSAPI_RETURN_CODE_ENUM (*ePrecipitation) (
        FORECAST_OBJECT hForecast,
        PRECIPITATION_OBJECT *phPrecipitation
            );

    SMSAPI_RETURN_CODE_ENUM (*eWind) (
        FORECAST_OBJECT hForecast,
        WIND_OBJECT *phWind
            );

    SMSAPI_RETURN_CODE_ENUM (*eHumidity) (
        FORECAST_OBJECT hForecast,
        HUMIDITY_OBJECT *phHumidity
            );

    SMSAPI_RETURN_CODE_ENUM (*eUVInfo) (
        FORECAST_OBJECT hForecast,
        UVINFO_OBJECT *phUVInfo
            );

    SMSAPI_RETURN_CODE_ENUM (*eAir) (
        FORECAST_OBJECT hForecast,
        AIR_OBJECT *phAir
            );

    SMSAPI_RETURN_CODE_ENUM (*eCloudCover) (
        FORECAST_OBJECT hForecast,
        FORECAST_CLOUD_COVER_ENUM *peCloudCover
            );

    UN32 (*un32TimeStamp) (
        FORECAST_OBJECT hForecast
            );

    N32 (*n32FPrintf) (
        FORECAST_OBJECT hForecast,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} FORECAST_OBJECT_INTERFACE_STRUCT;

  /*********************************/
 /***    WEATHER_MSG Object     ***/
/*********************************/

typedef enum weather_forecast_type_enum
{
    WEATHER_FORECAST_TYPE_CURRENT = 0,
    WEATHER_FORECAST_TYPE_0_3_HOUR,
    WEATHER_FORECAST_TYPE_3_6_HOUR,
    WEATHER_FORECAST_TYPE_SUNDAY,
    WEATHER_FORECAST_TYPE_MONDAY,
    WEATHER_FORECAST_TYPE_TUESDAY,
    WEATHER_FORECAST_TYPE_WEDNESDAY,
    WEATHER_FORECAST_TYPE_THURSDAY,
    WEATHER_FORECAST_TYPE_FRIDAY,
    WEATHER_FORECAST_TYPE_SATURDAY,
    WEATHER_FORECAST_TYPE_MAX
} WEATHER_FORECAST_TYPE_ENUM;

typedef enum weather_msg_object_type_enum
{
    WEATHER_MSG_TYPE_FORECAST = 0,
    WEATHER_MSG_TYPE_SKI_REPORT,
    WEATHER_MSG_TYPE_INVALID

} WEATHER_MSG_OBJECT_TYPE_ENUM;

typedef struct weather_msg_object_interface_struct
{

    SMSAPI_RETURN_CODE_ENUM (*eGetForecast) (
        WEATHER_MSG_OBJECT hWeatherMsg,
        WEATHER_FORECAST_TYPE_ENUM eForecastType,
        FORECAST_OBJECT *phForecast
            );

    LOCATION_OBJECT (*hLocation) (
        WEATHER_MSG_OBJECT hWeatherMsg
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetSkiConditions) (
        WEATHER_MSG_OBJECT hWeatherMsg,
        SKI_CONDITIONS_OBJECT *phSkiCond
            );

    WEATHER_MSG_OBJECT_TYPE_ENUM (*eType) (
        WEATHER_MSG_OBJECT hWeatherMsg
            );

    STRING_OBJECT (*hICAO)(
        WEATHER_MSG_OBJECT hWeatherMsg
            );

    N32 (*n32FPrintf) (
        WEATHER_MSG_OBJECT hWeatherMsg,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} WEATHER_MSG_OBJECT_INTERFACE_STRUCT;

/*********************************/
/***  SKI_CONDITIONS Object    ***/
/*********************************/
typedef enum ski_wind_condition_enum
{
    SKI_WIND_CONDITION_CALM = 0,
    SKI_WIND_CONDITION_MILD,
    SKI_WIND_CONDITION_MODERATE,
    SKI_WIND_CONDITION_STRONG,
    SKI_WIND_CONDITION_GUSTY,
    SKI_WIND_CONDITION_VERY_STRONG,
    SKI_WIND_CONDITION_NOT_RECOMMENDED,
    SKI_WIND_CONDITION_UNKNOWN

} SKI_WIND_CONDITION_ENUM;

typedef struct ski_conditions_object_interface_struct
{

    SMSAPI_RETURN_CODE_ENUM (*eOperationalStatus) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        BOOLEAN *pbOperStatus
            );

    SMSAPI_RETURN_CODE_ENUM (*eWeatherEvent) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        WEATHER_EVENT_OBJECT *phWeatherEvent
            );

    SMSAPI_RETURN_CODE_ENUM (*eTemperature) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        TEMPERATURE_OBJECT *phTemperature
            );

    SMSAPI_RETURN_CODE_ENUM (*eWindCondition) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        SKI_WIND_CONDITION_ENUM *peWindCond
            );

    SMSAPI_RETURN_CODE_ENUM (*eSnowConditions) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        SNOW_CONDITIONS_OBJECT *phSnowCond
            );

    SMSAPI_RETURN_CODE_ENUM (*eAttributes) (
        SKI_CONDITIONS_OBJECT hSkiCond,
        SKI_ATTRIBUTES_OBJECT *phSkiAttributes
            );

    UN32 (*un32TimeStamp) (
        SKI_CONDITIONS_OBJECT hSkiCond
            );

    N32 (*n32FPrintf) (
        SKI_CONDITIONS_OBJECT hSkiMsg,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );
} SKI_CONDITIONS_OBJECT_INTERFACE_STRUCT;

/*********************************/
/***  SNOW_CONDITIONS Object   ***/
/*********************************/
typedef enum ski_snow_conditions_enum
{
    SKI_SNOW_CONDITIONS_POWDER = 0,
    SKI_SNOW_CONDITIONS_HARD_PACKED,
    SKI_SNOW_CONDITIONS_WET_PACKED,
    SKI_SNOW_CONDITIONS_LOOSE_GRANULAR,
    SKI_SNOW_CONDITIONS_WET_GRANULAR,
    SKI_SNOW_CONDITIONS_FROZEN_GRANULAR,
    SKI_SNOW_CONDITIONS_ICY,
    SKI_SNOW_CONDITIONS_MIXED_CONDITIONS

} SKI_SNOW_CONDITIONS_ENUM;

typedef struct snow_conditions_object_interface_struct
{

    SMSAPI_RETURN_CODE_ENUM (*eMinBaseDepth) (
        SNOW_CONDITIONS_OBJECT hSnowCond,
        UN8 *pun8MinBaseDepth
            );

    SMSAPI_RETURN_CODE_ENUM (*eMaxBaseDepth) (
        SNOW_CONDITIONS_OBJECT hSnowCond,
        UN8 *pun8MaxBaseDepth
            );

    SMSAPI_RETURN_CODE_ENUM (*eNewSnowRange) (
        SNOW_CONDITIONS_OBJECT hSnowCond,
        UN8 *pun8NewSnowRangeLow,
        UN8 *pun8NewSnowRangeHigh
            );

    SMSAPI_RETURN_CODE_ENUM (*eConditions) (
        SNOW_CONDITIONS_OBJECT hSnowCond,
        SKI_SNOW_CONDITIONS_ENUM *peConditions
            );

    N32 (*n32FPrintf) (
        SNOW_CONDITIONS_OBJECT hSnowCond,
        FILE *psFile
            );
} SNOW_CONDITIONS_OBJECT_INTERFACE_STRUCT;


/*********************************/
/***    SKI_ATTRIBUTES Object   ***/
/*********************************/
typedef struct ski_attributes_object_interface_struct
{

    SMSAPI_RETURN_CODE_ENUM (*eNumOfLifts) (
        SKI_ATTRIBUTES_OBJECT hSkiAttrs,
        UN8 *pun8NumOfLifts
            );

    SMSAPI_RETURN_CODE_ENUM (*eNumOfTrailsRange) (
        SKI_ATTRIBUTES_OBJECT hSkiAttrs,
        UN8 *pun8NumOfTrailsRangeLow,
        UN8 *pun8NumOfTrailsRangeHigh
            );

    SMSAPI_RETURN_CODE_ENUM (*eAmenities) (
        SKI_ATTRIBUTES_OBJECT hSkiAttrs,
        size_t tNumAmenities,
        AMENITY_STRUCT *pasSkiAmenities
          );

    N32 (*n32FPrintf) (
        SKI_ATTRIBUTES_OBJECT hSkiAttrs,
        FILE *psFile
            );
} SKI_ATTRIBUTES_OBJECT_INTERFACE_STRUCT;

  /***************************************/
 /***----  Phonetics Data Service  ----**/
/***************************************/

  /*******************************/
 /***     Phonetics Object    ***/
/*******************************/

typedef enum phonetics_file_type_enum
{
    PHONETICS_FILE_TYPE_CHANNELS,
    PHONETICS_FILE_TYPE_CATEGORIES,
    PHONETICS_FILE_TYPE_MARKETS,
    PHONETICS_FILE_TYPE_TEAMS,
    PHONETICS_FILE_TYPE_LEAGUES,
    PHONETICS_FILE_INVALID_TYPE

} PHONETICS_FILE_TYPE_ENUM;

// Callback which this service uses to inform
// the application when a file is updated
typedef void (*PHONETICS_FILE_UPDATE_CALLBACK) (
    PHONETICS_SERVICE_OBJECT hPhoneticsService,
    PHONETICS_FILE_TYPE_ENUM eFileType,
    SMS_LANGUAGE_ENUM eLanguage,
    const char *pacTTSFilePath,
    const char *pacRecFilePath,
    void *pvFileCallbackArg
        );

/* External object interface */
typedef struct phonetics_interface_struct
{
    PHONETICS_SERVICE_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        const char *pacOutputDir,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        PHONETICS_FILE_UPDATE_CALLBACK vFileCallback,
        void *pvFileCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

    void (*vStop) (
        PHONETICS_SERVICE_OBJECT hPhoneticsService
            );

    BOOLEAN (*bGenerateDistributionFiles) (
        const char *pacTTSOutputDir,
        const char *pacRecOutputDir,
        const char *pacDBPath
            );

    SERVICE_ID (*tReadServiceIdFromDictionary) (
        FILE *psDictionaryFile
            );

    CATEGORY_ID (*tReadCategoryIdFromDictionary) (
        FILE *psDictionaryFile
            );

    DATASERVICE_STATE_ENUM (*eState) (
        PHONETICS_SERVICE_OBJECT hPhoneticsService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        PHONETICS_SERVICE_OBJECT hPhoneticsService
            );

} PHONETICS_INTERFACE_STRUCT;

  /***************************************/
 /***----     AGW Data Service     ----**/
/***************************************/

  /*******************************/
 /***        AGW Object       ***/
/*******************************/

// Shape points iterator
typedef BOOLEAN (*AGW_SHAPE_POINTS_ITERATOR_CALLBACK)(
    OSAL_FIXED_OBJECT hLat,
    OSAL_FIXED_OBJECT hLon,
    void *pvArg
        );

// Shapes iterator
typedef BOOLEAN (*AGW_SHAPE_CALLBACK)(
    AGW_SHAPE_OBJECT hShapeObj,
    void *pvArg
        );

// Defines supported data services
typedef enum agw_product_type_enum
{
    AGW_PRODUCT_TYPE_NOWRAD,
    AGW_PRODUCT_TYPE_STORM_ATTRIBUTES,
    AGW_PRODUCT_TYPE_SURFACE_FEATURES,
    AGW_PRODUCT_TYPE_STORM_TRACK,
    AGW_PRODUCT_TYPE_WIND_MAGNITUDE,
    AGW_PRODUCT_TYPE_WIND_DIRECTION,
    AGW_PRODUCT_TYPE_UNKNOWN
} AGW_PRODUCT_TYPE_ENUM;

// Defines supported shapes
typedef enum agw_shape_type_enum
{
    AGW_SHAPE_TYPE_FRONT,
    AGW_SHAPE_TYPE_PRESSURE_CENTER ,
    AGW_SHAPE_TYPE_ISOBAR,
    AGW_SHAPE_TYPE_STORM_POSITION,
    AGW_SHAPE_TYPE_WIND_RADII_FIELD,
    AGW_SHAPE_TYPE_STORM_ATTRIBUTES,
    AGW_SHAPE_TYPE_UNKNOWN
} AGW_SHAPE_TYPE_ENUM;

// Defines supported front types
typedef enum agw_front_type_enum
{
    AGW_FRONT_TYPE_COLD,
    AGW_FRONT_TYPE_WARM,
    AGW_FRONT_TYPE_STATIONARY,
    AGW_FRONT_TYPE_OCCLUDED,
    AGW_FRONT_TYPE_TROUGH,
    AGW_FRONT_TYPE_SQUALL_LINE,
    AGW_FRONT_TYPE_DRY_LINE,
    AGW_FRONT_TYPE_DISSIPATING_COLD,
    AGW_FRONT_TYPE_DISSIPATING_WARM,
    AGW_FRONT_TYPE_DISSIPATING_STATIONARY,
    AGW_FRONT_TYPE_DISSIPATING_OCCLUDED,
    AGW_FRONT_TYPE_FORMING_COLD,
    AGW_FRONT_TYPE_FORMING_WARM,
    AGW_FRONT_TYPE_FORMING_STATIONARY,
    AGW_FRONT_TYPE_FORMING_OCCLUDED,
    AGW_FRONT_TYPE_UNKNOWN
} AGW_FRONT_TYPE_ENUM;

// Defines supported pressure types
typedef enum agw_pressure_type_enum
{
    AGW_PRESSURE_TYPE_UNKNOWN,
    AGW_PRESSURE_TYPE_HIGH,
    AGW_PRESSURE_TYPE_LOW
} AGW_PRESSURE_TYPE_ENUM;

// Defines supported storm position types
typedef enum agw_storm_position_type_enum
{
    AGW_STORM_POSITION_TYPE_UNKNOWN,
    AGW_STORM_POSITION_TYPE_HISTORIC,
    AGW_STORM_POSITION_TYPE_CURRENT,
    AGW_STORM_POSITION_TYPE_FORECAST
} AGW_STORM_POSITION_TYPE_ENUM;

// Defines supported storm type types
typedef enum agw_storm_type_enum
{
    AGW_STORM_TYPE_UNKNOWN,
    AGW_STORM_TYPE_TROPICAL_DISTURBANCE,
    AGW_STORM_TYPE_HURRICANE_CAT_1,
    AGW_STORM_TYPE_HURRICANE_CAT_2,
    AGW_STORM_TYPE_HURRICANE_CAT_3,
    AGW_STORM_TYPE_HURRICANE_CAT_4,
    AGW_STORM_TYPE_HURRICANE_CAT_5,
    AGW_STORM_TYPE_TROPICAL_STORM,
    AGW_STORM_TYPE_TROPICAL_DEPRESSION,
    AGW_STORM_TYPE_TYPHOON,
    AGW_STORM_TYPE_SUPER_TYPHOON,
    AGW_STORM_TYPE_TROPICAL_CYCLONE
} AGW_STORM_TYPE_ENUM;

/* External object interface */
typedef struct agw_interface_struct
{
/* Object Members */

  AGW_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      const char *pacRasterPath,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
         );

  void (*vStop) (
      AGW_SERVICE_OBJECT hAgwService
         );

  DATASERVICE_STATE_ENUM (*eState) (
      AGW_SERVICE_OBJECT hAgwService
         );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      AGW_SERVICE_OBJECT hAgwService
         );

  SMSAPI_RETURN_CODE_ENUM (*eAddFilter) (
      AGW_SERVICE_OBJECT hAgwService,
      AGW_PRODUCT_TYPE_ENUM eProductType
          );

  SMSAPI_RETURN_CODE_ENUM (*eRemoveFilter) (
      AGW_SERVICE_OBJECT hAgwService,
      AGW_PRODUCT_TYPE_ENUM eProductType
          );

} AGW_INTERFACE_STRUCT;

  /************************/
 /*** AGW TILE Object  ***/
/************************/

/* External object interface */
typedef struct agw_tile_object_interface_struct
{
    AGW_PRODUCT_TYPE_ENUM (*eProductType) (
        AGW_TILE_OBJECT hAgwTile
            );

    TIME_T (*tTimeStamp)(
        AGW_TILE_OBJECT hAgwTile
            );

    LOCATION_OBJECT (*hNorthWest) (
        AGW_TILE_OBJECT hAgwTile
            );

    LOCATION_OBJECT (*hSouthEast) (
        AGW_TILE_OBJECT hAgwTile
            );

    IMAGE_OBJECT (*hImage)(
        AGW_TILE_OBJECT hAgwTile
            );

    N32 (*n32FPrintf) (
        AGW_TILE_OBJECT hAgwTile,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );
} AGW_TILE_OBJECT_INTERFACE_STRUCT;

  /**************************/
 /***  AGW SHAPE Object  ***/
/**************************/
/* External object interface */
typedef struct agw_shape_object_interface_struct
{
    AGW_PRODUCT_TYPE_ENUM (*eProductType) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_SHAPE_TYPE_ENUM (*eShapeType) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    TIME_T (*tTimeStamp) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_FRONT_OBJECT (*hFront) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_ISOBAR_OBJECT (*hIsobar) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_PRESSURE_CENTER_OBJECT (*hPressureCenter) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_STORM_ATTRIBUTES_OBJECT (*hStormAttributes) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_STORM_POSITION_OBJECT (*hStormPosition) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    AGW_WIND_RADII_AREA_OBJECT (*hWindRadiiArea) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    UN16 (*un16Points) (
        AGW_SHAPE_OBJECT hAgwShape
            );

    SMSAPI_RETURN_CODE_ENUM (*eIteratePoints) (
        AGW_SHAPE_OBJECT hAgwShape,
        AGW_SHAPE_POINTS_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetPoint) (
        AGW_SHAPE_OBJECT hAgwShape,
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon,
        UN16 un16Point
            );

    N32 (*n32FPrintf) (
        AGW_SHAPE_OBJECT hAgwShape,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );
} AGW_SHAPE_OBJECT_INTERFACE_STRUCT;

  /**************************/
 /***  AGW FRONT Object  ***/
/**************************/
/* External object interface */
typedef struct agw_front_object_interface_struct
{
    AGW_FRONT_TYPE_ENUM (*eType) (
        AGW_FRONT_OBJECT hAgwFront
            );

    N32 (*n32FPrintf) (
        AGW_FRONT_OBJECT hAgwFront,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_FRONT_OBJECT_INTERFACE_STRUCT;

  /***************************/
 /***  AGW ISOBAR Object  ***/
/***************************/
/* External object interface */
typedef struct agw_isobar_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*ePressure) (
        AGW_ISOBAR_OBJECT hAgwIsobar,
        OSAL_FIXED_OBJECT hPressure
            );

    N32 (*n32FPrintf) (
        AGW_ISOBAR_OBJECT hAgwIsobar,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_ISOBAR_OBJECT_INTERFACE_STRUCT;

  /************************************/
 /***  AGW PRESSURE CENTER Object  ***/
/************************************/
/* External object interface */
typedef struct agw_pressure_center_object_interface_struct
{
    AGW_PRESSURE_TYPE_ENUM (*eType) (
        AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr
            );

    SMSAPI_RETURN_CODE_ENUM (*ePressure) (
        AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr,
        OSAL_FIXED_OBJECT hPressure
            );

    N32 (*n32FPrintf) (
        AGW_PRESSURE_CENTER_OBJECT hAgwPressCntr,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_PRESSURE_CENTER_OBJECT_INTERFACE_STRUCT;

  /************************************/
 /***  AGW STORM ATTRIBUTES Object ***/
/************************************/
/* External object interface */
typedef struct agw_storm_attributes_object_interface_struct
{
    STRING_OBJECT (*hStormID) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    TIME_T (*tObservationTime) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    SMSAPI_RETURN_CODE_ENUM (*eEchoTop) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
        OSAL_FIXED_OBJECT hEchoTop
            );

    SMSAPI_RETURN_CODE_ENUM (*eDirection) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
        OSAL_FIXED_OBJECT hDirection
            );

    SMSAPI_RETURN_CODE_ENUM (*eSpeed) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
        OSAL_FIXED_OBJECT hSpeed
            );

    BOOLEAN (*bIsHailProbable) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    BOOLEAN (*bIsHailPositive) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    BOOLEAN (*bIsMesocyclonic) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    BOOLEAN (*bIsTornadic) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr
            );

    N32 (*n32FPrintf) (
        AGW_STORM_ATTRIBUTES_OBJECT hAgwStormAttr,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_STORM_ATTRIBUTES_OBJECT_INTERFACE_STRUCT;

  /************************************/
 /***   AGW STORM POSITION Object  ***/
/************************************/
/* External object interface */
typedef struct agw_storm_position_object_interface_struct
{
    STRING_OBJECT (*hStormID) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    STRING_OBJECT (*hStormName) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    AGW_STORM_POSITION_TYPE_ENUM (*ePositionType) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    SMSAPI_RETURN_CODE_ENUM (*eDirection) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        OSAL_FIXED_OBJECT hDirection
            );

    SMSAPI_RETURN_CODE_ENUM (*eSpeed) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        OSAL_FIXED_OBJECT hSpeed
            );

    SMSAPI_RETURN_CODE_ENUM (*eMaxSustainedWinds) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        OSAL_FIXED_OBJECT hMaxWinds
            );

    SMSAPI_RETURN_CODE_ENUM (*eGusts) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        OSAL_FIXED_OBJECT hGusts
            );

    SMSAPI_RETURN_CODE_ENUM (*ePressure) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        OSAL_FIXED_OBJECT hPressure
            );

    TIME_T (*tObservationTime) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    AGW_STORM_TYPE_ENUM (*eStormType) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    UN16 (*un16SequenceID) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateWindRadii) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        AGW_SHAPE_CALLBACK bCallback,
        void *pvArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateStormPosition) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        AGW_SHAPE_CALLBACK bCallback,
        void *pvArg
            );

    N32 (*n32FPrintf) (
        AGW_STORM_POSITION_OBJECT hAgwStormPos,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_STORM_POSITION_OBJECT_INTERFACE_STRUCT;


  /************************************/
 /***  AGW WIND RADII AREA Object  ***/
/************************************/
/* External object interface */
typedef struct agw_wind_radii_area_object_interface_struct
{
    SMSAPI_RETURN_CODE_ENUM (*eWindSpeed) (
        AGW_WIND_RADII_AREA_OBJECT hAgwWindRd,
        OSAL_FIXED_OBJECT hWindSpeed
            );

    N32 (*n32FPrintf) (
        AGW_WIND_RADII_AREA_OBJECT hAgwWindRd,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} AGW_WIND_RADII_AREA_OBJECT_INTERFACE_STRUCT;

/***************************************/
/***---- Sports Data Service ----**/
/***************************************/

typedef UN32 SPORT_ID;
typedef UN32 AFFILIATE_ID;
// Be careful if you change this %llu format is used with table ids.
typedef N64 TABLE_ID;
typedef UN32 LEAGUE_ID;
typedef TABLE_ID REFERENCED_TABLE_ID;
typedef UN32 INFORMATION_CLASS;
typedef UN32 INSTANCE_ID;

typedef enum sports_service_sort_type_enum
{
    SORT_TYPE_ASCEND = 0,
    SORT_TYPE_DECEND,
    SORT_TYPE_NONE

}SORT_TYPE_ENUM;

/* Sports Data Service Access Callback */
typedef void (*SPORTS_SERVICE_ACCESS_CALLBACK)(
    SPORTS_SERVICE_OBJECT hSportsService,
    void *pvCallbackArg
            );

/***************************************/
/***   SPORTS DATA SERVICE Object    ***/
/***************************************/

/* External object interface */
typedef struct sports_service_interface_struct
{
/* Object Members */

  SPORTS_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
        );

  void (*vStop) (
      SPORTS_SERVICE_OBJECT hSportsDataService
        );

  DATASERVICE_STATE_ENUM (*eState) (
      SPORTS_SERVICE_OBJECT hSportsDataService
        );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      SPORTS_SERVICE_OBJECT hSportsDataService
        );

    SMSAPI_RETURN_CODE_ENUM (*eUseDataSet) (
        SPORTS_SERVICE_OBJECT hSportsService,
        SPORTS_SERVICE_ACCESS_CALLBACK vCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetSportList) (
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetRootAffiliateList) (
        SPORTS_SERVICE_OBJECT hSportsService,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetAffiliateChildren) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tId,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetAffiliateInfoClassList) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tId,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetAffiliateTableList) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        INFORMATION_CLASS tInfoClass,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetAffiliate) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tId,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetTable) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        TABLE_ID tTableId,
        char *pDest,
        UN32 *pun32DestSize,
        UN32 tRowOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetTableHeader) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        TABLE_ID tTableId,
        char *pDest,
        UN32 *pun32DestSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetTableByColumnMatch) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        TABLE_ID tTableId,
        UN32 tColumnId,
        char *pValue,
        char *pDest,
        UN32 *pun32DestSize,
        UN32 tRowOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetTableByColumnRange) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        TABLE_ID tTableId,
        UN32 tColumnId,
        UN32 tValue1,
        UN32 tValue2,
        SORT_TYPE_ENUM eSortType,
        char *pDest,
        UN32 *pun32DestSize,
        UN32 tRowOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetTableByTeamPresence) (
        SPORTS_SERVICE_OBJECT hSportsService,
        AFFILIATE_ID tAffiliateId,
        TABLE_ID tTableId,
        UN32 un32ColumnOneID,
        UN32 un32ColumnTwoID,
        UN32 tTeamId,
        char *pDest,
        UN32 *pun32DestSize,
        UN32 tRowOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetReferencedTable) (
        SPORTS_SERVICE_OBJECT hSportsService,
        REFERENCED_TABLE_ID tId,
        char *pDest,
        UN32 *pun32DestSize,
        UN32 tRowOffset
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetReferencedTableHeader) (
        SPORTS_SERVICE_OBJECT hSportsService,
        REFERENCED_TABLE_ID tId,
        char *pDest,
        UN32 *pun32DestSize
            );

    CID_OBJECT (*hCreateLeagueCid) (
        UN32 un32LeagueId
            );

    CID_OBJECT (*hCreateTeamCid) (
        UN32 un32LeagueId,
        UN32 un32TeamId
            );

} SPORTS_SERVICE_INTERFACE_STRUCT;

/***************************************/
/***---- Sports Monitor Object ----**/
/***************************************/

typedef UN32 SPORTS_MONITOR_EVENT_MASK;

/* Monitor callback to indicate change. */
typedef void (*SPORTS_MONITOR_CALLBACK)(
    SPORTS_MONITOR_OBJECT hMonitor,
    SPORTS_MONITOR_EVENT_MASK tEventMask,
    void *pvCallbackArg
            );
/* Iterator callback to let applications process change. */
typedef BOOLEAN (*SPORT_MONITOR_ITERATOR) (
    SPORTS_MONITOR_OBJECT hMonitor,
    AFFILIATE_ID tAffiliate,
    SPORTS_MONITOR_EVENT_MASK tEventMask,
    void *pvCallbackArg
        );

/* External object interface */
typedef struct sports_monitor_interface_struct
{
/* Object Members */

    SPORTS_MONITOR_OBJECT (*hCreate) (
        SPORTS_SERVICE_OBJECT hService,
        SPORTS_MONITOR_CALLBACK vCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetEventMask) (
        SPORTS_MONITOR_OBJECT hMonitor,
        SPORTS_MONITOR_EVENT_MASK *ptEventMask
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetEventMask) (
        SPORTS_MONITOR_OBJECT hMonitor,
        SPORTS_MONITOR_EVENT_MASK tEventMask
            );

    void (*vDestroy) (
        SPORTS_SERVICE_OBJECT hService,
        SPORTS_MONITOR_OBJECT hMonitor
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetAffiliateMask) (
        SPORTS_MONITOR_OBJECT hMonitor,
        AFFILIATE_ID tAffiliate,
        SPORTS_MONITOR_EVENT_MASK tEventMask
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetAffiliateMask) (
        SPORTS_MONITOR_OBJECT hMonitor,
        AFFILIATE_ID tAffiliate,
        SPORTS_MONITOR_EVENT_MASK *ptEventMask
                );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveAllAffiliates) (
        SPORTS_MONITOR_OBJECT hMonitor
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateChangedAffiliates) (
        SPORTS_MONITOR_OBJECT hMonitor,
        SPORT_MONITOR_ITERATOR bIterator,
        void *pvIteratorArg
            );

} SPORTS_MONITOR_INTERFACE_STRUCT;

  /****************************************/
 /***---- STOCK TICKER Data Service ----**/
/****************************************/

typedef BOOLEAN (*STOCK_TICKER_SYMBOL_ITERATOR) (
    STOCK_TICKER_SERVICE_OBJECT hService,
    const char *pacStockSymbol,
    void *pvArg
        );

typedef enum stock_msg_price_direction_enum
{
    STOCK_MSG_PRICE_DIRECTION_INVALID = 0,
    STOCK_MSG_PRICE_DIRECTION_UP,
    STOCK_MSG_PRICE_DIRECTION_UNCHANGED,
    STOCK_MSG_PRICE_DIRECTION_DOWN
} STOCK_MSG_PRICE_DIRECTION_ENUM;

typedef struct stock_symbol_interface_struct
{
    STOCK_SYMBOL_OBJECT (*hCreate) (
        STRING_OBJECT hName
            );

    N32 (*n32FWrite) (
        STOCK_SYMBOL_OBJECT hTarget,
        FILE *psFile
            );

    STOCK_SYMBOL_OBJECT (*hFRead) (
        FILE *psFile
            );

    N32 (*n32FPrintf) (
        STOCK_SYMBOL_OBJECT hTarget,
        FILE *psFile
            );

    void (*vDestroy) (
        STOCK_SYMBOL_OBJECT hTarget
            );
} STOCK_SYMBOL_INTERFACE_STRUCT;

/* External object interface */
typedef struct stock_ticker_interface_struct
{
/* Object Members */

  STOCK_TICKER_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
         );

  void (*vStop) (
      STOCK_TICKER_SERVICE_OBJECT hService
         );

  DATASERVICE_STATE_ENUM (*eState) (
      STOCK_TICKER_SERVICE_OBJECT hService
         );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      STOCK_TICKER_SERVICE_OBJECT hService
         );

  BOOLEAN (*bStockSymbolsUpdated) (
      STOCK_TICKER_SERVICE_OBJECT hService,
      UN32 *pun32ContentVersion
         );

  SMSAPI_RETURN_CODE_ENUM (*eIterateStockSymbols) (
      STOCK_TICKER_SERVICE_OBJECT hService,
      STOCK_TICKER_SYMBOL_ITERATOR bIterator,
      void *pvIteratorArg
         );

  STRING_OBJECT (*hDataProviderInfo) (
      STOCK_TICKER_SERVICE_OBJECT hService
         );

} STOCK_TICKER_SERVICE_INTERFACE_STRUCT;

typedef struct stock_msg_interface_struct
{
  STRING_OBJECT (*hSymbolName) (
      STOCK_MSG_OBJECT hMsg
         );

  SMSAPI_RETURN_CODE_ENUM (*eValues) (
      STOCK_MSG_OBJECT hMsg,
      UN32 *pun32Price,
      UN8 *pun8PriceFrac,
      UN32 *pun32PriceDelta,
      UN8 *pun8PriceDeltaFrac,
      STOCK_MSG_PRICE_DIRECTION_ENUM *peDirection
         );

  STOCK_SYMBOL_OBJECT (*hStockSymbol) (
      STOCK_MSG_OBJECT hMsg
          );

  N32 (*n32FPrintf) (
      STOCK_MSG_OBJECT hMsg,
      FILE *psFile,
      SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
         );

} STOCK_MSG_INTERFACE_STRUCT;

  /****************************************/
 /***---- NON NAV MAPS Data Service ----**/
/****************************************/
typedef enum map_service_detail_level_enum
{
    MAPS_SERVICE_DETAIL_LEVEL_LOW = 0,
    MAPS_SERVICE_DETAIL_LEVEL_MEDIUM,
    MAPS_SERVICE_DETAIL_LEVEL_HIGH,
    MAPS_SERVICE_DETAIL_LEVEL_HIGHEST,
    MAPS_SERVICE_DETAIL_LEVEL_UNKNOWN,
    MAPS_SERVICE_DETAIL_LEVEL_MAX = MAPS_SERVICE_DETAIL_LEVEL_UNKNOWN
} MAPS_SERVICE_DETAIL_LEVEL_ENUM;

/* External object interface */
typedef struct maps_interface_struct
{
    /* Object Members */
    MAPS_SERVICE_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        const char *pacOutputDir,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

    void (*vStop) (
        MAPS_SERVICE_OBJECT hService
            );

    DATASERVICE_STATE_ENUM (*eState) (
        MAPS_SERVICE_OBJECT hService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        MAPS_SERVICE_OBJECT hService
            );

    SMSAPI_RETURN_CODE_ENUM (*ePersistentDataCleanup) (
        MAPS_SERVICE_OBJECT hService,
        const char *pcReferenceDBPath,
        const char *pcPesistentDBPath
            );

    SMSAPI_RETURN_CODE_ENUM (*eFindNearestMap) (
        MAPS_SERVICE_OBJECT hService,
        OSAL_FIXED_OBJECT hLat,
        OSAL_FIXED_OBJECT hLon,
        MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
        char *pacImageFilePath,
        size_t tImageFilePathSize
            );

    SMSAPI_RETURN_CODE_ENUM (*eLookupMapByName) (
        MAPS_SERVICE_OBJECT hService,
        const char *pacMapName,
        MAPS_SERVICE_DETAIL_LEVEL_ENUM eDetailLevel,
        char *pacImageFilePath,
        size_t tImageFilePathSize
            );

    SMSAPI_RETURN_CODE_ENUM (*ePrioritizeMarket) (
        MAPS_SERVICE_OBJECT hService,
        TRAFFIC_MARKET tMarketId
            );
} MAPS_INTERFACE_STRUCT;

  /***************************************/
 /***-- Safety Cameras Data Service ----**/
/***************************************/

  /*******************************/
 /***  SAFETY_CAMERAS Object  ***/
/*******************************/

typedef UN8 SAFETY_CAMERAS_ALERT_ID;

#define SAFETY_CAMERAS_ALERT_INVALID_ID (SAFETY_CAMERAS_ALERT_ID)(0)

typedef struct safety_cameras_alert_type_struct {
    SAFETY_CAMERAS_ALERT_ID tID;
    STRING_OBJECT hDescription;
} SAFETY_CAMERAS_ALERT_TYPE_STRUCT;

typedef BOOLEAN (*SAFETY_CAMERAS_ALERT_TYPES_ITERATOR_CALLBACK)(
    SAFETY_CAMERAS_ALERT_TYPE_STRUCT *psAlertType,
    void *pvCallbackArg
        );

/* External object interface */
typedef struct safety_cameras_interface_struct
{
/* Object Members */

    SAFETY_CAMERAS_SERVICE_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

    void (*vStop) (
        SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
            );

    DATASERVICE_STATE_ENUM (*eState) (
        SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateAlertTypes) (
        SAFETY_CAMERAS_SERVICE_OBJECT hSafeCamService,
        SAFETY_CAMERAS_ALERT_TYPES_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
        const char *pcContainingDirectoryPath,
        DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
        DATASERVICE_REF_DATA_VER *ptNextRefDataVer
            );

} SAFETY_CAMERAS_INTERFACE_STRUCT;

  /***********************************/
 /** SAFETY CAMERAS LOCATION Object **/
/***********************************/

/* Safety Cameras Location callback to iterate through all Sub Locations */
typedef BOOLEAN (*SAFETY_CAMERAS_SUBLOCATIONS_ITERATOR_CALLBACK)(
    SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
    void *pvCallbackArg
        );

/* External object interface */
typedef struct safety_cameras_location_object_interface_struct
{
    /* Object Members */

    LOCATION_OBJECT (*hLocation) (
        SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateSubLocations) (
        SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
        SAFETY_CAMERAS_SUBLOCATIONS_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    N32 (*n32FPrintf) (
        SAFETY_CAMERAS_LOCATION_OBJECT hSafeCamLoc,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} SAFETY_CAMERAS_LOCATION_OBJECT_INTERFACE_STRUCT;


  /*****************************************/
 /*** SAFETY CAMERAS SUBLOCATION Object ***/
/*****************************************/

/* External object interface */
typedef struct safety_cameras_sublocation_object_interface_struct
{
    /* Object Members */

    LOCATION_OBJECT (*hLocation) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
            );

    SMSAPI_RETURN_CODE_ENUM (*eDirection) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
        UN16* pun16Direction
            );

    SMSAPI_RETURN_CODE_ENUM (*eHighSpeedLimit) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
        OSAL_FIXED_OBJECT hHighSpeedLimit
            );

    SMSAPI_RETURN_CODE_ENUM (*eLowSpeedLimit) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
        OSAL_FIXED_OBJECT hLowSpeedLimit
            );

    SAFETY_CAMERAS_ALERT_ID (*tAlertType) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc
            );

    N32 (*n32FPrintf) (
        SAFETY_CAMERAS_SUBLOCATION_OBJECT hSafeCamSubLoc,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} SAFETY_CAMERAS_SUBLOCATION_OBJECT_INTERFACE_STRUCT;

  /**************************************/
 /***     WEATHER/SECURITY ALERTS    ***/
/**************************************/

// A unique ID to identify an alert message by within the current
// set of alerts
typedef UN32 WS_ALERT_MSG_ID;
#define WS_ALERT_MSG_INVALID_ID ((WS_ALERT_MSG_ID)0)

// Alert locations iterator
typedef BOOLEAN (*WS_ALERT_MSG_LOCATIONS_ITERATOR_CALLBACK)(
    WS_ALERTS_LOCATION_OBJECT hWsAlertsLocation,
    void *pvArg
        );

// Alert types iterator
typedef BOOLEAN (*WS_ALERT_MSG_TYPES_ITERATOR_CALLBACK) (
    STRING_OBJECT hMsgType,
    void *pvArg
        );

  /**************************************************/
 /***   WEATHER/SECURITY ALERTS Service Object   ***/
/**************************************************/

/* External object interface */
typedef struct ws_alerts_interface_struct
{
/* Object Members */

  WS_ALERTS_SERVICE_OBJECT (*hStart) (
      const char *pacSRHDriverName,
      DATASERVICE_EVENT_MASK tEventRequestMask,
      DATASERVICE_EVENT_CALLBACK vEventCallback,
      void *pvEventCallbackArg,
      DATASERVICE_OPTIONS_STRUCT const *psOptions
          );

  void (*vStop) (
      WS_ALERTS_SERVICE_OBJECT hWSAlertsService
          );

  DATASERVICE_STATE_ENUM (*eState) (
      WS_ALERTS_SERVICE_OBJECT hWSAlertsService
          );

  DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
      WS_ALERTS_SERVICE_OBJECT hWSAlertsService
          );

  DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
      const char *pcContainingDirectoryPath,
      DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
      DATASERVICE_REF_DATA_VER *ptNextRefDataVer
        );

} WS_ALERTS_INTERFACE_STRUCT;

  /*******************************************/
 /***  WEATHER/SECURITY ALERT MSG Object  ***/
/*******************************************/
/* External object interface */
typedef struct ws_alert_msg_object_interface_struct
{
    STRING_OBJECT (*hMsgText) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    STRING_OBJECT (*hAlertText) (
        WS_ALERT_MSG_OBJECT hAlertMsg,
        STRING_OBJECT hAlertType
            );

    TIME_T (*tStartTime) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    TIME_T (*tEndTime) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    SMS_LANGUAGE_ENUM (*eMsgLang) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    UN16 (*un16Priority) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateLocations) (
        WS_ALERT_MSG_OBJECT hAlertMsg,
        WS_ALERT_MSG_LOCATIONS_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateAlertTypes) (
        WS_ALERT_MSG_OBJECT hAlertMsg,
        WS_ALERT_MSG_TYPES_ITERATOR_CALLBACK bCallback,
        void *pvCallbackArg
            );

    WS_ALERT_MSG_ID (*tID) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    BOOLEAN (*bIsMarineZone) (
        WS_ALERT_MSG_OBJECT hAlertMsg
            );

    N32 (*n32FPrintf) (
        WS_ALERT_MSG_OBJECT hAlertMsg,
        FILE *psFile,
        SMSAPI_OUTPUT_OPTION_ENUM eOutputOption
            );

} WS_ALERT_MSG_OBJECT_INTERFACE_STRUCT;

  /***************************************/
 /***-- Traffic Cameras Data Service ---**/
/***************************************/

  /*******************************/
 /***  TRAFFIC_CAMERAS Object  ***/
/*******************************/

/* External object interface */
typedef struct traffic_cameras_interface_struct
{
    /* Object Members */

    TRAFFIC_CAMERAS_SERVICE_OBJECT (*hStart) (
        const char *pacSRHDriverName,
        DATASERVICE_EVENT_MASK tEventRequestMask,
        DATASERVICE_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg,
        DATASERVICE_OPTIONS_STRUCT const *psOptions
            );

    void (*vStop) (
        TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService
            );

    DATASERVICE_STATE_ENUM (*eState) (
        TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eErrorCode) (
        TRAFFIC_CAMERAS_SERVICE_OBJECT hTrafficCamerasService
            );

    DATASERVICE_ERROR_CODE_ENUM (*eGetReferenceDataVersion) (
        const char *pcContainingDirectoryPath,
        DATASERVICE_REF_DATA_VER *ptCurrentRefDataVer,
        DATASERVICE_REF_DATA_VER *ptNextRefDataVer
            );

} TRAFFIC_CAMERAS_INTERFACE_STRUCT;

  /****************************/
 /***    TUNEMIX Object    ***/
/****************************/

typedef enum tunemix_status_enum
{
    TUNEMIX_STATUS_UNDEFINED = 0,
    TUNEMIX_STATUS_AVAILABLE,
    TUNEMIX_STATUS_UNAVAILABLE

} TUNEMIX_STATUS_ENUM;

typedef BOOLEAN (*TUNEMIX_ITERATOR_CALLBACK) (
    TUNEMIX_OBJECT hTuneMix,
    void *pvIteratorArg
        );

typedef BOOLEAN (*TUNEMIX_CHANNELS_ITERATOR_CALLBACK) (
    CHANNEL_OBJECT hChannel,
    void *pvIteratorArg
        );

/* External object interface */
typedef struct tunemix_object_interface_struct
{
    /* Object Members */
    TUNEMIX_OBJECT (*hCreate) (
        DECODER_OBJECT hDecoder,
        BOOLEAN bOverride,
        BOOLEAN bPlayUnrestricted
            );

    void (*vDestroy) (
        TUNEMIX_OBJECT hTuneMix
            );

    TUNEMIX_OBJECT (*hActive) (
        DECODER_OBJECT hDecoder
            );

    CHANNEL_ID (*tChannelID) (
        TUNEMIX_OBJECT hTuneMix
            );

    SMSAPI_RETURN_CODE_ENUM (*eTune) (
        TUNEMIX_OBJECT hTuneMix
            );

    SMSAPI_RETURN_CODE_ENUM (*eConfigure) (
        TUNEMIX_OBJECT hTuneMix,
        CHANNEL_ID *ptChannelId,
        UN16 un16Count
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterate) (
        DECODER_OBJECT hDecoder,
        TUNEMIX_ITERATOR_CALLBACK bIterateFxn,
        void *pvIteratorArg
            );

    TUNEMIX_STATUS_ENUM (*eStatus) (
        TUNEMIX_OBJECT hTuneMix
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateChannels) (
        TUNEMIX_OBJECT hTuneMix,
        TUNEMIX_CHANNELS_ITERATOR_CALLBACK bIterateFxn,
        void *pvIteratorArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetOverrides) (
        TUNEMIX_OBJECT hTuneMix,
        BOOLEAN bOverride,
        BOOLEAN bPlayUnrestricted
            );

    SMSAPI_RETURN_CODE_ENUM (*eGetOverrides) (
        TUNEMIX_OBJECT hTuneMix,
        BOOLEAN *pbOverride,
        BOOLEAN *pbPlayUnrestricted
            );

    STRING_OBJECT (*hGetName) (
        TUNEMIX_OBJECT hTuneMix
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetName) (
        TUNEMIX_OBJECT hTuneMix,
        char *pacName
            );

} TUNEMIX_OBJECT_INTERFACE_STRUCT;

  /**************************/
 /*** SPORT_FLASH OBJECT ***/
/**************************/

// This type is used to pass Sports Flash Event ID.
typedef UN32 SPORTS_FLASH_EVENT_ID;

// Invalid Sports Flash ID value is defined as
#define SPORTS_FLASH_INVALID_EVENT_ID UN32_MAX

// This structure is used to handle data about game event.
typedef struct sports_flash_game_event_info_struct
{
    CHANNEL_ID  tChannelID;
    BOOLEAN     bEnded;
    CID_OBJECT  hHomeLeagueId;
    CID_OBJECT  hHomeTeamId;
    CID_OBJECT  hAwayLeagueId;
    CID_OBJECT  hAwayTeamId;
    UN16        un16Bias;

} SPORTS_FLASH_GAME_EVENT_INFO_STRUCT;

// This enum is used to mark the program status of the Game Event.
typedef enum sports_flash_program_status_enum
{
    SPORTS_FLASH_PROGRAM_AVAILABLE = 0,
    SPORTS_FLASH_PROGRAM_ENDED

} SPORTS_FLASH_PROGRAM_STATUS_ENUM;

// This enum is used to mark the status of the Flash Event.
typedef enum sports_flash_event_status_enum
{
    SPORTS_FLASH_EVENT_OCCURRED = 0,
    SPORTS_FLASH_EVENT_EXPIRED,
    SPORTS_FLASH_EVENT_PLAYED

} SPORTS_FLASH_EVENT_STATUS_ENUM;

// Sports Flash Event Callbacks

typedef void ( *SPORTS_FLASH_GAME_EVENT_CALLBACK ) (
    DECODER_OBJECT hDecoder,
    SPORTS_FLASH_PROGRAM_STATUS_ENUM eProgramStatus,
    const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGameInfo,
    void *pvEventCallbackArg
        );

typedef void ( *SPORTS_FLASH_FLASH_EVENT_CALLBACK ) (
    DECODER_OBJECT hDecoder,
    CHANNEL_ID tChannelID,
    SPORTS_FLASH_EVENT_ID tFlashEventID,
    SPORTS_FLASH_EVENT_STATUS_ENUM eFlashEventStatus,
    void *pvEventCallbackArg
        );

// Sports Flash Iterator Callbacks
typedef BOOLEAN (*SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK) (
    CID_OBJECT hLeagueId,
    CID_OBJECT hTeamId,
    void *pvTeamsIteratorCallbackArg
        );

typedef BOOLEAN (*SPORTS_FLASH_GAMES_ITERATOR_CALLBACK) (
    const SPORTS_FLASH_GAME_EVENT_INFO_STRUCT *psGameInfo,
    void *pvGamesIteratorCallbackArg
        );

typedef BOOLEAN (*SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK) (
    CHANNEL_ID tChannelID,
    SPORTS_FLASH_EVENT_ID tFlashEventID,
    void *pvEventsIteratorCallbackArg
        );

// External object interface

typedef struct sports_flash_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        DECODER_OBJECT hDecoder,
        UN32 *pun32MaxSportsFlash,
        SPORTS_FLASH_GAME_EVENT_CALLBACK vGameEventCallback,
        void *pvGameEventCallbackArg
            );

    void (*vStop) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eAddFavoriteTeam) (
        DECODER_OBJECT hDecoder,
        CID_OBJECT hLeagueId,
        CID_OBJECT hTeamId
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveFavoriteTeam) (
        DECODER_OBJECT hDecoder,
        CID_OBJECT hLeagueId,
        CID_OBJECT hTeamId
            );

    SMSAPI_RETURN_CODE_ENUM (*eRemoveAllFavoriteTeams) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateFavoriteTeams) (
        DECODER_OBJECT hDecoder,
        SPORTS_FLASH_TEAMS_ITERATOR_CALLBACK bTeamsIteratorCallback,
        void *pvTeamsIteratorCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateCurrentGames) (
        DECODER_OBJECT hDecoder,
        SPORTS_FLASH_GAMES_ITERATOR_CALLBACK bGamesIteratorCallback,
        void *pvGamesIteratorCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetGamesMonitor) (
        DECODER_OBJECT hDecoder,
        CHANNEL_ID *ptChannelIDList,
        UN16 un16Count,
        SPORTS_FLASH_FLASH_EVENT_CALLBACK vFlashEventCallback,
        void *pvFlashEventCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*eIterateCurrentFlashEvents) (
        DECODER_OBJECT hDecoder,
        SPORTS_FLASH_EVENTS_ITERATOR_CALLBACK bEventsIteratorCallback,
        void *pvEventsIteratorCallbackArg
            );

    SMSAPI_RETURN_CODE_ENUM (*ePlayFlashEvent) (
        DECODER_OBJECT hDecoder,
        SPORTS_FLASH_EVENT_ID tFlashEventID
            );

    SMSAPI_RETURN_CODE_ENUM (*eAbortFlashEvent) (
        DECODER_OBJECT hDecoder
            );

    SPORTS_FLASH_EVENT_ID (*tCurrentFlashEventID) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eEnableNotification) (
        DECODER_OBJECT hDecoder,
        BOOLEAN bEnable
            );

    SMSAPI_RETURN_CODE_ENUM (*eIsNotificationEnabled) (
        DECODER_OBJECT hDecoder,
        BOOLEAN *pbEnable
            );

} SPORTS_FLASH_OBJECT_INTERFACE_STRUCT;

  /**************************/
 /****** TW_NOW OBJECT *****/
/**************************/

// This type is used to pass TW Now Bulletin ID.
typedef UN32 TW_NOW_BULLETIN_ID;

// Invalid Bulletin ID value is defined as
#define TW_NOW_INVALID_BULLETIN_ID (TW_NOW_BULLETIN_ID)UN32_MAX

// This enum is used to mark the status of the Bulletin.
typedef enum tw_now_bulletin_status_enum
{
    TW_NOW_NO_BULLETIN_AVAILABLE,
    TW_NOW_NEW_BULLETIN_AVAILABLE,
    TW_NOW_PLAYED_BULLETIN_AVAILABLE
} TW_NOW_BULLETIN_STATUS_ENUM;

// TW Now Event Callback

typedef void ( *TW_NOW_EVENT_CALLBACK ) (
    DECODER_OBJECT hDecoder,
    TW_NOW_BULLETIN_ID tBulletinID,
    TW_NOW_BULLETIN_STATUS_ENUM eBulletinStatus,
    CID_OBJECT hMarketId,
    void *pvEventCallbackArg
        );

// External object interface

typedef struct tw_now_object_interface_struct
{
    /* Object Members */

    SMSAPI_RETURN_CODE_ENUM (*eStart) (
        DECODER_OBJECT hDecoder,
        TW_NOW_EVENT_CALLBACK vEventCallback,
        void *pvEventCallbackArg
            );

    void (*vStop) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*eSetMarketID) (
        DECODER_OBJECT hDecoder,
        CID_OBJECT hMarketID
            );

    CID_OBJECT (*hGetMarketID) (
        DECODER_OBJECT hDecoder
            );

    SMSAPI_RETURN_CODE_ENUM (*ePlayBulletinAudio) (
        DECODER_OBJECT hDecoder,
        TW_NOW_BULLETIN_ID tBulletinID
            );

    SMSAPI_RETURN_CODE_ENUM (*eAbortBulletinPlayback) (
        DECODER_OBJECT hDecoder
            );

    TW_NOW_BULLETIN_ID (*tCurrentBulletinID) (
        DECODER_OBJECT hDecoder
            );

} TW_NOW_OBJECT_INTERFACE_STRUCT;

  /*****************************/
 /** GLOBAL STRUCT TEMPLATES **/
/*****************************/

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

extern const SMS_OBJECT_INTERFACE_STRUCT SMS;
extern const SRM_OBJECT_INTERFACE_STRUCT SRM;
extern const MODULE_OBJECT_INTERFACE_STRUCT MODULE;
extern const MODULE_VERSION_OBJECT_INTERFACE_STRUCT MODULE_VERSION;
extern const DECODER_OBJECT_INTERFACE_STRUCT DECODER;
extern const ENGINEERING_DATA_OBJECT_INTERFACE_STRUCT ENGINEERING_DATA;
extern const DETAILED_SIGNAL_QUALITY_OBJECT_INTERFACE_STRUCT DETAILED_SIGNAL_QUALITY;
extern const DETAILED_OVERLAY_SIGNAL_QUALITY_OBJECT_INTERFACE_STRUCT DETAILED_OVERLAY_SIGNAL_QUALITY;
extern const LINK_STATUS_INFORMATION_OBJECT_INTERFACE_STRUCT LINK_STATUS_INFORMATION;
extern const CHANNEL_OBJECT_INTERFACE_STRUCT CHANNEL;
extern const CHANNELLIST_OBJECT_INTERFACE_STRUCT CHANNELLIST;
extern const CATEGORY_OBJECT_INTERFACE_STRUCT CATEGORY;
extern const CD_OBJECT_INTERFACE_STRUCT CDO;
extern const CID_OBJECT_INTERFACE_STRUCT CID;
extern const NON_PROGRAM_OBJECT_INTERFACE_STRUCT NON_PROGRAM;
extern const ENTERTAINMENT_OBJECT_INTERFACE_STRUCT ENTERTAINMENT;
extern const NEWS_OBJECT_INTERFACE_STRUCT NEWS;
extern const TEAM_OBJECT_INTERFACE_STRUCT TEAM;
extern const CAR_OBJECT_INTERFACE_STRUCT CAR;
extern const LEAGUE_OBJECT_INTERFACE_STRUCT LEAGUE;
extern const SPORTS_CONTENT_OBJECT_INTERFACE_STRUCT SPORTS;
extern const MUSIC_OBJECT_INTERFACE_STRUCT MUSIC;
extern const REPORT_OBJECT_INTERFACE_STRUCT REPORT;
#ifdef SXM_LEGACY_SSP
extern const PID_OBJECT_INTERFACE_STRUCT PID;
#endif
extern const STRING_OBJECT_INTERFACE_STRUCT STRING;
extern const PLAYBACK_OBJECT_INTERFACE_STRUCT PLAYBACK;
extern const SMART_FAVORITES_OBJECT_INTERFACE_STRUCT SMART_FAVORITES;
extern const SONGLIST_OBJECT_INTERFACE_STRUCT SONGLIST;
extern const SONG_OBJECT_INTERFACE_STRUCT SONG;
extern const CEML_OBJECT_INTERFACE_STRUCT CEML;
extern const CONTENT_ALERT_SERVICE_INTERFACE_STRUCT CAS;
extern const CAL_CONTENT_OBJECT_INTERFACE_STRUCT CAL_CONTENT;
extern const CAL_ALERT_OBJECT_INTERFACE_STRUCT CAL_ALERT;
extern const AT_SEEK_INTERFACE_STRUCT AT_SEEK;
extern const TW_SEEK_INTERFACE_STRUCT TW_SEEK;
extern const SP_SEEK_INTERFACE_STRUCT SP_SEEK;
extern const SEEK_INTERFACE_STRUCT SEEK;
extern const SEEK_EVENT_INTERFACE_STRUCT SEEK_EVENT;
extern const SEEK_CONTENT_INTERFACE_STRUCT SEEK_CONTENT;
extern const AT_SEEK_CONTENT_INTERFACE_STRUCT AT_SEEK_CONTENT;
extern const TW_SEEK_CONTENT_INTERFACE_STRUCT TW_SEEK_CONTENT;
extern const SP_SEEK_CONTENT_INTERFACE_STRUCT SP_SEEK_CONTENT;
extern const SPORT_ZONE_OBJECT_INTERFACE_STRUCT SPORT_ZONE;
extern const PRESETS_OBJECT_INTERFACE_STRUCT PRESETS;
extern const PRESET_BAND_OBJECT_INTERFACE_STRUCT BAND;
extern const FAVORITES_INTERFACE_STRUCT FAVORITES;
extern const GENERIC_DATASERVICE_INTERFACE_STRUCT DATA;
extern const CHANNEL_ART_INTERFACE_STRUCT CHANNEL_ART;
extern const SPORTS_SERVICE_INTERFACE_STRUCT SPORTS_SERVICE;
extern const SPORTS_MONITOR_INTERFACE_STRUCT SPORTS_MONITOR;
extern const STOCK_TICKER_SERVICE_INTERFACE_STRUCT STOCK_TICKER;
extern const STOCK_MSG_INTERFACE_STRUCT STOCK_MSG;
extern const STOCK_SYMBOL_INTERFACE_STRUCT STOCK_SYMBOL;
extern const IMAGE_INTERFACE_STRUCT IMAGE;
extern const MAPS_INTERFACE_STRUCT MAPS;
extern const DSRL_TARGET_OBJECT_INTERFACE_STRUCT DSRL_TARGET;
extern const DSRL_ENTRY_OBJECT_INTERFACE_STRUCT DSRL_ENTRY;
extern const LOCID_OBJECT_INTERFACE_STRUCT LOCID;
extern const DISTANCE_OBJECT_INTERFACE_STRUCT DISTANCE;
extern const LOCATION_OBJECT_INTERFACE_STRUCT LOCATION;
extern const SHAPE_OBJECT_INTERFACE_STRUCT SHAPE;
extern const DSRL_OBJECT_INTERFACE_STRUCT DSRL;
extern const TRAFFIC_INTERFACE_STRUCT TRAFFIC;
extern const TRAFFIC_LOCID_OBJECT_INTERFACE_STRUCT TRAFFIC_LOCID;
extern const TRAFFIC_MSG_OBJECT_INTERFACE_STRUCT TRAFFIC_MSG;
extern const FUEL_INTERFACE_STRUCT FUEL;
extern const FUEL_STATION_OBJECT_INTERFACE_STRUCT FUEL_STATION;
extern const BAUDOT_INTERFACE_STRUCT BAUDOT;
extern const SONG_TAG_SERVICE_INTERFACE_STRUCT SONG_TAG_SERVICE;
extern const MOVIES_INTERFACE_STRUCT MOVIES;
extern const MOVIE_OBJECT_INTERFACE_STRUCT MOVIE;
extern const THEATER_OBJECT_INTERFACE_STRUCT THEATER;
extern const EPG_INTERFACE_STRUCT EPG;
extern const PROGRAM_OBJECT_INTERFACE_STRUCT PROGRAM;
extern const EPG_TOPIC_INTERFACE_STRUCT TOPIC;
extern const WEATHER_INTERFACE_STRUCT WEATHER;
extern const WEATHER_MSG_OBJECT_INTERFACE_STRUCT WEATHER_MSG;
extern const LOCALIZATION_OBJECT_INTERFACE_STRUCT LOCALIZATION;
extern const DEVICE_OBJECT_INTERFACE_STRUCT DEVICE;
extern const FORECAST_OBJECT_INTERFACE_STRUCT FORECAST;
extern const TEMPERATURE_OBJECT_INTERFACE_STRUCT TEMPERATURE;
extern const WEATHER_EVENT_OBJECT_INTERFACE_STRUCT WEATHER_EVENT;
extern const WIND_OBJECT_INTERFACE_STRUCT WIND;
extern const HUMIDITY_OBJECT_INTERFACE_STRUCT HUMIDITY;
extern const UVINFO_OBJECT_INTERFACE_STRUCT UVINFO;
extern const AIR_OBJECT_INTERFACE_STRUCT AIR;
extern const PRECIPITATION_OBJECT_INTERFACE_STRUCT PRECIPITATION;
extern const SKI_CONDITIONS_OBJECT_INTERFACE_STRUCT SKI_CONDITIONS;
extern const SKI_ATTRIBUTES_OBJECT_INTERFACE_STRUCT SKI_ATTRIBUTES;
extern const SNOW_CONDITIONS_OBJECT_INTERFACE_STRUCT SNOW_CONDITIONS;
extern const AGW_INTERFACE_STRUCT AGW;
extern const AGW_TILE_OBJECT_INTERFACE_STRUCT AGW_TILE;
extern const AGW_SHAPE_OBJECT_INTERFACE_STRUCT AGW_SHAPE;
extern const AGW_FRONT_OBJECT_INTERFACE_STRUCT AGW_FRONT;
extern const AGW_ISOBAR_OBJECT_INTERFACE_STRUCT AGW_ISOBAR;
extern const AGW_PRESSURE_CENTER_OBJECT_INTERFACE_STRUCT AGW_PRESSURE_CENTER;
extern const AGW_STORM_ATTRIBUTES_OBJECT_INTERFACE_STRUCT AGW_STORM_ATTRIBUTES;
extern const AGW_STORM_POSITION_OBJECT_INTERFACE_STRUCT AGW_STORM_POSITION;
extern const AGW_WIND_RADII_AREA_OBJECT_INTERFACE_STRUCT AGW_WIND_RADII_AREA;
extern const WS_ALERTS_LOCATION_INTERFACE_STRUCT WS_ALERTS_LOCATION;
extern const WS_ALERTS_INTERFACE_STRUCT WS_ALERTS;
extern const WS_ALERT_MSG_OBJECT_INTERFACE_STRUCT WS_ALERT_MSG;
extern const SUB_NOTIFICATION_OBJECT_INTERFACE_STRUCT SUB_NOTIFICATION;
extern const SAFETY_CAMERAS_INTERFACE_STRUCT SAFETY_CAMERAS;
extern const SAFETY_CAMERAS_LOCATION_OBJECT_INTERFACE_STRUCT SAFETY_CAMERAS_LOCATION;
extern const SAFETY_CAMERAS_SUBLOCATION_OBJECT_INTERFACE_STRUCT SAFETY_CAMERAS_SUBLOCATION;
extern const PHONETICS_INTERFACE_STRUCT PHONETICS;
extern const TRAFFIC_CAMERAS_INTERFACE_STRUCT TRAFFIC_CAMERAS;
extern const TUNEMIX_OBJECT_INTERFACE_STRUCT TUNEMIX;
extern const SPORTS_FLASH_OBJECT_INTERFACE_STRUCT SPORTS_FLASH;
extern const TW_NOW_OBJECT_INTERFACE_STRUCT TW_NOW;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* SMS_API_H */
