/******************************************************************************/
/*                    Copyright (c) Sirius-XM Satellite Radio Inc.            */
/*                            All Rights Reserved                             */
/*         Licensed Materials - Property of Sirius-XM Satellite Radio Inc.    */
/*                           Proprietary & Confidential	                      */
/******************************************************************************/

  /*********************************/
 /** PREVENT REDUNDANT INCLUSION **/
/*********************************/
#ifndef OSAL_H_
#define OSAL_H_

#ifdef __cplusplus
extern "C" {
#endif


  /**************/
 /** INCLUDES **/
/**************/

#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <time.h>

#include "standard.h"

  /**********************/
 /** GLOBAL CONSTANTS **/
/**********************/

// Provide initial definitions of OSAL compile options if not defined
// at compile time as compile flags
#ifndef OSAL_DEBUG
#define OSAL_DEBUG 1
#endif

#ifndef OSAL_OBJECT_TRACKING
#if OSAL_DEBUG == 1
#define OSAL_OBJECT_TRACKING 1
#else
#define OSAL_OBJECT_TRACKING 0
#endif
#endif

#ifndef OSAL_FILE_SYSTEM
#define OSAL_FILE_SYSTEM 1
#endif

#ifndef OSAL_LOG
#define OSAL_LOG 1
#endif

#ifndef OSAL_BUFFER
#define OSAL_BUFFER 1
#endif

#ifndef OSAL_CRC
#define OSAL_CRC 1
#endif

#ifndef OSAL_INTERRUPT
#define OSAL_INTERRUPT 0
#endif

#ifndef OSAL_DEVICES
#define OSAL_DEVICES 1
#endif

#ifndef OSAL_TEST
#define OSAL_TEST 0
#endif

#ifndef OSAL_FIXED_MATH
#define OSAL_FIXED_MATH 1
#endif


/* Object name prefix for OSAL objects */
#define OSAL_NAME_PREFIX "OS:"

/* The maximum length allowed for a object (task, queue....) name */
/* Name length does not include the NULL */
#define OSAL_MAX_OBJECT_NAME_LENGTH (63)
#define OSAL_MAX_OBJECT_NAME_LENGTH_WITH_NULL \
    (OSAL_MAX_OBJECT_NAME_LENGTH + 1)

/* Timeout Options */
#define OSAL_OBJ_TIMEOUT_INFINITE   (-1)
#define OSAL_OBJ_TIMEOUT_NONE       (0)

/* OSAL configuration mask definitions */
typedef UN32 OSAL_CONFIG_INFO_MASK;
#define OSAL_CONFIG_NONE                  ((OSAL_CONFIG_INFO_MASK)0)
#define OSAL_CONFIG_FILE_SYSTEM           ((OSAL_CONFIG_INFO_MASK)(1 << 0))
#define OSAL_CONFIG_LOG                   ((OSAL_CONFIG_INFO_MASK)(1 << 1))
#define OSAL_CONFIG_BUFFER                ((OSAL_CONFIG_INFO_MASK)(1 << 2))
#define OSAL_CONFIG_CRC                   ((OSAL_CONFIG_INFO_MASK)(1 << 3))
#define OSAL_CONFIG_INTERRUPT             ((OSAL_CONFIG_INFO_MASK)(1 << 4))
#define OSAL_CONFIG_DEVICES               ((OSAL_CONFIG_INFO_MASK)(1 << 5))
#define OSAL_CONFIG_TEST                  ((OSAL_CONFIG_INFO_MASK)(1 << 6))
#define OSAL_CONFIG_FIXED_MATH            ((OSAL_CONFIG_INFO_MASK)(1 << 7))
#define OSAL_CONFIG_NATIVE_STDIO          ((OSAL_CONFIG_INFO_MASK)(1 << 8))

/* Time Update definitions */

// Time update mask
typedef UN32 OSAL_TIME_UPDATE_MASK;

// Time update masks
#define OSAL_TIME_UPDATE_MASK_NONE             (0x0000)

#define OSAL_TIME_UPDATE_MASK_TIME             (0x0001)
#define OSAL_TIME_UPDATE_MASK_GMT              (0x0002)
#define OSAL_TIME_UPDATE_MASK_DST              (0x0004)

#define OSAL_TIME_UPDATE_MASK_ALL               (   \
    (OSAL_TIME_UPDATE_MASK)                         \
    OSAL_TIME_UPDATE_MASK_TIME                    | \
    OSAL_TIME_UPDATE_MASK_GMT                     | \
    OSAL_TIME_UPDATE_MASK_DST                       \
        )

/* Task Reporting Definition */
#define OSAL_TASK_REPORT_NO_ERROR               (0)
#define OSAL_TASK_REPORT_MAXIMUM_INTERVAL_SEC   (60)

/* Task Create Options */

#define OSAL_TASK_OPTION_NONE              (0x00000000UL)
#define OSAL_TASK_OPTION_DEBUG_OUTPUT      (0x00000001UL)

#define OSAL_TASK_OPTION_ALL                ( \
    OSAL_TASK_OPTION_NONE | \
    OSAL_TASK_OPTION_DEBUG_OUTPUT \
        )

/* Queue Create Options */

#define OSAL_QUEUE_OPTION_NONE             (0x00000000UL)
/* use fixed size queues */
#define OSAL_QUEUE_OPTION_FIXED_SIZE       (0x00000001UL)
/* use variable sized queues */
#define OSAL_QUEUE_OPTION_VARIABLE_SIZE    (0x00000002UL)
/* use if queues posted in an interrupt */
#define OSAL_QUEUE_OPTION_INTERRUPT        (0x00000004UL)

#define OSAL_QUEUE_OPTION_ALL              ( \
    OSAL_QUEUE_OPTION_FIXED_SIZE | \
    OSAL_QUEUE_OPTION_VARIABLE_SIZE | \
    OSAL_QUEUE_OPTION_INTERRUPT \
        )

/* Queue Allocate Flags */

#define OSAL_QUEUE_FLAG_NONE               (0x00000000UL)
/* specify that the task should block on a full queue during the send. */
#define OSAL_QUEUE_FLAG_BLOCK              (0x00000001UL)
/* this is the default behavior where the call will return an error on
   a full queue. */
#define OSAL_QUEUE_FLAG_NONBLOCK           (0x00000002UL)
/* In the systems that support this feature, the message will be marked as
   high priority (front of queue). */
#define OSAL_QUEUE_FLAG_URGENT             (0x00000004UL)
/* In the systems that support this feature, the message will be marked as
   the lowest priority (always last message in the queue).
   The message is always placed in the tale of the queue and is sent
   when it is the last and only one message in the queue.
   The event was introduced for internal SMS use only
   and generally is not required to use by the application.*/
#define OSAL_QUEUE_FLAG_DEFERRED           (0x00000008UL)

#define OSAL_QUEUE_FLAG_ALL                ( \
    OSAL_QUEUE_FLAG_BLOCK       | \
    OSAL_QUEUE_FLAG_NONBLOCK    | \
    OSAL_QUEUE_FLAG_URGENT      | \
    OSAL_QUEUE_FLAG_DEFERRED      \
        )

/* Semaphore Create Flags */

#define OSAL_SEM_OPTION_NONE               (0x00000000UL)
#define OSAL_SEM_OPTION_MUTEX              (0x00000001UL)
#define OSAL_SEM_OPTION_ALL                ( \
    OSAL_SEM_OPTION_MUTEX \
        )

/* Linked List Options */

#define OSAL_LL_OPTION_NONE               (0x00000000UL)
#define OSAL_LL_OPTION_PROTECT            (0x00000001UL)
#define OSAL_LL_OPTION_CIRCULAR           (0x00000002UL)
#define OSAL_LL_OPTION_LINEAR             (0x00000004UL)
#define OSAL_LL_OPTION_UNIQUE             (0x00000008UL)
#define OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS \
                                          (0x00000010UL)
#define OSAL_LL_OPTION_BINARY_SEARCH      (0x00000020UL)
#define OSAL_LL_OPTION_RBTREE             (0x00000040UL)

#define OSAL_LL_OPTION_ALL                ( \
    OSAL_LL_OPTION_PROTECT | \
    OSAL_LL_OPTION_CIRCULAR | \
    OSAL_LL_OPTION_LINEAR | \
    OSAL_LL_OPTION_UNIQUE | \
    OSAL_LL_OPTION_USE_PRE_ALLOCATED_ELEMENTS | \
    OSAL_LL_OPTION_BINARY_SEARCH | \
    OSAL_LL_OPTION_RBTREE \
        )

/* Log Create Options */

#define OSAL_LOG_OPTION_NONE             (0x00000000UL)
#define OSAL_LOG_OPTION_FORMAT           (0x00000001UL)

#define OSAL_LOG_OPTION_ALL              ( \
    OSAL_LOG_OPTION_FORMAT \
        )

/* Block Pool Create Options */

#define OSAL_BLOCK_POOL_OPTION_NONE      (0x00000000UL)
/* use if block pool accessed in an interrupt */
#define OSAL_BLOCK_POOL_OPTION_INTERRUPT (0x00000001UL)

#define OSAL_BLOCK_POOL_OPTION_ALL       ( \
    OSAL_BLOCK_POOL_OPTION_INTERRUPT \
        )

/* Buffer Allocate Options */

#define OSAL_BUFFER_ALLOCATE_OPTION_NONE (0x00000000UL)

#define OSAL_BUFFER_ALLOCATE_OPTION_ALL  ( \
    OSAL_BUFFER_ALLOCATE_OPTION_NONE \
        )

/* OSAL File Attributes */

// Read / Write Access Rights Attributes
#define OSAL_FILE_ATTR_READ	     (0x01)
#define OSAL_FILE_ATTR_WRITE	 (0x02)

// File Attributes
#define OSAL_FILE_ATTR_HIDDEN    (0x04)
#define OSAL_FILE_ATTR_SYSTEM    (0x08)
#define OSAL_FILE_ATTR_DIRECTORY (0x20)

// ASC Time Buffer Size
#define OSAL_ASCBUFSIZE              (26)

// Size of the FIXED_OBJECT_STRUCT, in terms of OSAL_FIXED_OBJECT_DATA
#define OSAL_FIXED_OBJECT_SIZE (2)

// A typedef to clear up code when using OSAL_FIXED_OBJECT_SIZE
// to create a OSAL_FIXED_OBJECT on the stack
// To support 64-bit porting
typedef N32 OSAL_FIXED_OBJECT_DATA;

/******************************************************************************/
/* ANSI-C File-IO                                                             */
/******************************************************************************/

// Type of object holding information necessary to control a stream.
typedef struct osal_file_struct
{
    // Interface
    struct osal_file_interface_struct const *psIntf;

    // Object sharing protection handle (typically OSAL_OBJECT_HDL)
    void *pvUsageProtection;
    size_t tProtectionNesting;

    // Object handle (typically FILE *)
    void *pvHdl;

} OSAL_FILE_STRUCT;

// Re-define ANSI-C FILE structure as OSAL file structure
#define FILE OSAL_FILE_STRUCT

// ANSI-C File I/O Prototypes (implemented by OSAL)
#define fopen OSAL.psFopen
#define fclose OSAL.iFclose
#define fread OSAL.tFread
#define fwrite OSAL.tFwrite
#define ftell OSAL.lFtell
#define ioctl OSAL.iIoctl
#define fseek OSAL.iFseek
#define remove OSAL.iRemove
#define rename OSAL.iRename
#define fflush OSAL.iFflush
#ifdef feof
#undef feof
#endif
#define feof OSAL.iFeof
#ifdef ferror
#undef ferror
#endif
#define ferror OSAL.iFerror
#define clearerr OSAL.vClearerr

// ANSI-C File I/O Prototypes (derivated by OSAL implementations)
#define fgetc OSAL.iFgetc
#define fgets OSAL.pcFgets
#define fputc OSAL.iFputc
#define fputs OSAL.iFputs
#define fprintf OSAL.iFPrintf


#ifndef EOK
#define EOK 0
#endif

#ifdef stdin
#undef stdin
#endif

#ifdef stdout
#undef stdout
#endif

#ifdef stderr
#undef stderr
#endif

// ANSI-C File pointers
extern OSAL_FILE_STRUCT GsStdin;
#define stdin (&GsStdin)
extern OSAL_FILE_STRUCT GsStdout;
#define stdout (&GsStdout)
extern OSAL_FILE_STRUCT GsStderr;
#define stderr (&GsStderr)

/*
    OSAL FILE ioctl group/class id's

    Certain drivers sometimes belong to generalized groups or classes. In
    this case common operations that devices have which belong to a group
    or class can be generalized and supported by drivers which implement
    them. To accomplish this driver group or class commands can be created.
    To help organize these groups or classes, id's can be constructed which
    allow individual class drivers to define the commands which make up the
    groups. The commands which belong to the FILE_IOCTL group are handled
    in the OSAL abstraction layer rather than in particular driver files.  Thus,
    the actual commands are defined here.

    Each device class is given 256 possible entries.
*/

// General 'file' ioctl group id
#define FILE_IOCTL                          (0x00000000)

// Device General Ioctl Group Members
#define FILE_IOCTL_USAGE_PROTECT            (FILE_IOCTL + 0)
#define FILE_IOCTL_USAGE_UNPROTECT          (FILE_IOCTL + 1)

// ANSI-C Stdio Definitions (prepare for redefinition)
#ifndef OSAL_USE_NATIVE_STDIO

// ANSI-C stdio APIs
#ifdef printf
#undef printf
#endif
#if OSAL_DEBUG == 1
#define printf OSAL.iPrintf
#else
#if __STDC_VERSION__ >= 199901L
#define printf(...)
#else
#define printf if(1);else OSAL.iPrintf
#endif
#endif /* OSAL_DEBUG == 1 */

#ifdef putchar
#undef putchar
#endif
#if OSAL_DEBUG == 1
#define putchar OSAL.iPutchar
#else
#if __STDC_VERSION__ >= 199901L
#define putchar(...)
#else
#define putchar if(1);else OSAL.iPutchar
#endif
#endif /* OSAL_DEBUG == 1 */

#ifdef puts
#undef puts
#endif
#if OSAL_DEBUG == 1
#define puts OSAL.iPuts
#else
#if __STDC_VERSION__ >= 199901L
#define puts(...)
#else
#define puts if(1);else OSAL.iPuts
#endif
#endif /* OSAL_DEBUG == 1 */

#ifdef getchar
#undef getchar
#endif
#if OSAL_DEBUG == 1
#define getchar OSAL.iGetchar
#else
#if __STDC_VERSION__ >= 199901L
#define getchar(...)
#else
#define getchar if(1);else OSAL.iGetchar
#endif
#endif /* OSAL_DEBUG == 1 */

#ifdef gets
#undef gets
#endif
#if OSAL_DEBUG == 1
#define gets OSAL.pcGets
#else
#if __STDC_VERSION__ >= 199901L
#define gets(...)
#else
#define gets if(1);else OSAL.pcGets
#endif
#endif /* OSAL_DEBUG == 1 */

#endif /* OSAL_USE_NATIVE_STDIO */

/******************************************************************************/

  /*********************/
 /** GLOBAL TYPEDEFS **/
/*********************/

/* OS-Specific Object Handle */
typedef struct os_object_hdl *OS_OBJECT_HDL;
#define OS_INVALID_OBJECT_HDL ((OS_OBJECT_HDL)(0))

// OSAL Object Handle
typedef struct osal_object_hdl *OSAL_OBJECT_HDL;
#define OSAL_INVALID_OBJECT_HDL ((OSAL_OBJECT_HDL)(0))

// OSAL Linked-List Entry
typedef struct osal_linked_list_entry *OSAL_LINKED_LIST_ENTRY;
#define OSAL_INVALID_LINKED_LIST_ENTRY ((OSAL_LINKED_LIST_ENTRY)(0))
#define OSAL_INVALID_LINKED_LIST_ENTRY_PTR ((OSAL_LINKED_LIST_ENTRY*)(0))

// OSAL Buffer
typedef struct osal_buffer_hdl * OSAL_BUFFER_HDL;
#define OSAL_INVALID_BUFFER_HDL ((OSAL_BUFFER_HDL)(0))

// OSAL Buffer Block
typedef struct osal_buffer_block_hdl *OSAL_BUFFER_BLOCK_HDL;
#define OSAL_INVALID_BUFFER_BLOCK_HDL ((OSAL_BUFFER_BLOCK_HDL)(0))

// OSAL FIXED Object
typedef struct osal_fixed_object *OSAL_FIXED_OBJECT;
#define OSAL_FIXED_INVALID_OBJECT ((OSAL_FIXED_OBJECT)(0))

// OSAL Time Notification Object
typedef struct osal_time_notification_object * OSAL_TIME_NOTIFICATION_OBJECT;
#define OSAL_TIME_NOTIFICATION_INVALID_OBJECT \
    ((OSAL_TIME_NOTIFICATION_OBJECT)(0))

// OSAL CRC Value
typedef UN32 OSAL_CRC_RESULT;

// OSAL Object Task Handler Prototype
typedef N32 (*OSAL_TASK_HANDLER) ( void *pvArg );

// OSAL Object Sleep Handler Prototype
typedef void (*OSAL_SLEEP_HANDLER) ( void *pvArg );

// OSAL Object Wake-up Handler Prototype
typedef void (*OSAL_WAKEUP_HANDLER) ( void *pvArg );

// OSAL Object Shutdown Handler Prototype
typedef void (*OSAL_SHUTDOWN_HANDLER) ( void *pvArg );

// OSAL Object Timer Handler Prototype
typedef void (*OSAL_TIMER_HANDLER) ( OSAL_OBJECT_HDL hTimer, void *pvArg );

// OSAL Object Time Update Handler Prototype
typedef void (*OSAL_TIME_UPDATE_HANDLER) (
    OSAL_TIME_UPDATE_MASK tUpdateMask,
    void *pvArg
        );

// OSAL Object Interrupt Handler Prototype
typedef void (*OSAL_INTERRUPT_HANDLER) ( OSAL_OBJECT_HDL hInt, void *pvArg );

// OSAL Log 'iterator' function prototype
typedef BOOLEAN (*OSAL_LOG_ITERATOR_HANDLER) ( OSAL_OBJECT_HDL hLog, const char *pacName, void *pvArg );

// OSAL Linked List 'sorted' compare function prototype
typedef N16 (*OSAL_LL_COMPARE_HANDLER)( void *pvObj1, void *pvObj2 );

// OSAL Linked List 'iterator' function prototype
typedef BOOLEAN (*OSAL_LL_ITERATOR_HANDLER) ( void *pvData, void *pvArg );

// OSAL Linked List element's data 'release' function prototype
typedef void (*OSAL_LL_RELEASE_HANDLER)( void *pvData );

// OSAL Buffer Iterator Return Codes
typedef enum osal_buffer_iterator_result_enum
{
    OSAL_BUFFER_ITERATOR_RESULT_STOP,
    OSAL_BUFFER_ITERATOR_RESULT_KEEP,
    OSAL_BUFFER_ITERATOR_RESULT_REMOVE

} OSAL_BUFFER_ITERATOR_RESULT_ENUM;

// OSAL Buffer 'iterator' function prototype
typedef OSAL_BUFFER_ITERATOR_RESULT_ENUM (*OSAL_BUFFER_ITERATOR_HANDLER) (
    void *pvData, size_t tDataSize, void *pvArg
        );

// OSAL Debug Function Handler Prototype
typedef void(*OSAL_DEBUG_HANDLER)( char *pacInput, const void *pvArg );

// OSAL Task ID
typedef UN32 OSAL_TASK_ID;

// OSAL File System Directory Items Qualifier Handler prototype
typedef BOOLEAN (*OSAL_FS_FILE_QUALIFIER)( const char *pcItemName, const void *pvArg );

// OSAL Linked List Add Type
typedef enum osal_ll_add_enum
{
    OSAL_LL_ADD_BEFORE_ENTRY,
    OSAL_LL_ADD_AFTER_ENTRY,
    OSAL_LL_ADD_REPLACE_ENTRY,
    OSAL_LL_ADD_NEW_ENTRY,
    OSAL_LL_ADD_MAX,
} OSAL_LL_ADD_ENUM;

// OSAL CRC Types
typedef enum osal_crc_type_enum
{
    OSAL_CRC_TYPE_CCITT_CRC16 = 0,
    OSAL_CRC_TYPE_NAVTEQ_CRC16,
    OSAL_CRC_TYPE_XMAPPID_CRC16,
    OSAL_CRC_TYPE_ISO3309_CRC32,
    OSAL_CRC_TYPE_MAX

} OSAL_CRC_TYPE_ENUM;

// OSAL Object Return Codes
typedef enum osal_return_code_enum
{
    // Informational return codes
    OSAL_SUCCESS = 0,
    OSAL_TASK_MONITOR_OK = OSAL_SUCCESS,
    OSAL_TIMEOUT,
    OSAL_MESSAGE_NOT_AVAILABLE,
    OSAL_QUEUE_EMPTY,
    OSAL_QUEUE_FULL,
    OSAL_QUEUE_MESSAGE_ALREADY_QUEUED,
    OSAL_SEM_AVAILABLE,
    OSAL_SEM_NOT_AVAILABLE,
    OSAL_TIMER_NOT_ACTIVE,
    OSAL_NO_OBJECTS,
    OSAL_OBJECT_NOT_FOUND,                  // 10

    // Error return codes
    OSAL_ERROR,
    OSAL_ERROR_CANNOT_CREATE_OBJECT,
    OSAL_ERROR_CANNOT_DELETE_OBJECT,
    OSAL_ERROR_CANNOT_ADD_OBJECT,
    OSAL_ERROR_CANNOT_REMOVE_OBJECT,
    OSAL_ERROR_INVALID_INPUT,
    OSAL_ERROR_CORRUPTED_OBJECT_LIST,
    OSAL_ERROR_INVALID_POINTER,
    OSAL_ERROR_INVALID_HANDLE,
    OSAL_ERROR_NAME_TOO_LONG,               // 20
    OSAL_ERROR_NO_FREE_HANDLES,
    OSAL_ERROR_NAME_NOT_FOUND,
    OSAL_ERROR_DUPLICATE_NAME,
    OSAL_ERROR_INVALID_PRIORITY,
    OSAL_ERROR_ADDRESS_MISALIGNED,
    OSAL_ERROR_TASK_NOT_REGISTERED,
    OSAL_ERROR_INVALID_TIME,
    OSAL_ERROR_UNSUPPORTED_API,
    OSAL_ERROR_UNSUPPORTED_OPTION,
    OSAL_ERROR_UNSUPPORTED_FLAG,            // 30
    OSAL_ERROR_OUT_OF_MEMORY,
    OSAL_ERROR_LIST_NOT_EMPTY,
    OSAL_ERROR_LIST_ITEM_NOT_UNIQUE,
    OSAL_ERROR_TIMER_ACTIVE,
    OSAL_ERROR_MESSAGE_NOT_ALLOCATED,

    // Math-specific error codes
    OSAL_ERROR_MATH_OVERFLOW,
    OSAL_ERROR_MATH_DIVIDE_BY_ZERO,

    // Task monitoring error codes
    OSAL_ERROR_TASK_MONITOR_REPORT_TIMEOUT,
    OSAL_ERROR_TASK_MONITOR_STACK_EXCEEDED,
    OSAL_ERROR_TASK_MONITOR_MEMORY_ALLOCATION_EXCEEDED, // 40
    OSAL_ERROR_TASK_MONITOR_CPU_UTILIZATION_EXCEEDED,
    OSAL_ERROR_TASK_MONITOR_DEVICE_DRIVER_FAILURE,

    OSAL_ERROR_UNKNOWN

} OSAL_RETURN_CODE_ENUM;

// OSAL Start Handler Prototype
typedef BOOLEAN (*OSAL_START_HANDLER) ( const void *pvArg );

// OSAL Driver Registration Prototypes
typedef BOOLEAN (*OSAL_REGISTER_DRIVERS_HANDLER) ( void *pvArg );
typedef void (*OSAL_UNREGISTER_DRIVERS_HANDLER) ( void *pvArg );

/* Task Priority Options */
typedef enum osal_task_priority_enum
{
    OSAL_TASK_PRIORITY_LOWEST = 0,
    OSAL_TASK_PRIORITY_LOW,
    OSAL_TASK_PRIORITY_MEDIUM,
    OSAL_TASK_PRIORITY_HIGH,
    OSAL_TASK_PRIORITY_HIGHEST,
    OSAL_TASK_PRIORITY_INVALID

} OSAL_TASK_PRIORITY_ENUM;


/* Device Driver Node Definitions */

// The maximum size of a device name
#define OSAL_DEV_MAX_NAME_LENGTH (16)
#define OSAL_DEV_MAX_NAME_LENGTH_WITH_NULL \
    (OSAL_DEV_MAX_NAME_LENGTH + 1)

/*
    Device Driver Information Structure

    This structure is provided to each of the device driver method
    implementations. It contains information about the device node itself
    (the device's public data) as well as any data relevant to the specific
    instance of this device driver.
*/

typedef struct osal_dev_info_struct
{
    // This driver's node name
    const char *pcDevName;

    // This driver's instance number or 'open count'
    UN32 un32Instance;

    // The current 'byte-stream' position
    UN32 un32Position;

    // A pointer to any device specific attribute data. A structure
    // which must be cast the the driver itself to any relevant structure
    void const *pvProperties;

    // This driver's node public data (if any) which is accessible via
    // any instance of the driver, but common to all. This is set-up by the
    // bInit() method and tore-down by the vExit() method.
    void *pvPublicData;

    // This driver's private data (if any) unique to any one
    // instance of the device driver. This is set-up by the
    // n32Open() method and tore-down by the n32Close() method.
    void *pvPrivateData;

} OSAL_DEV_INFO_STRUCT;

/*
    This structure defines the generic method interface structure for all
    device drivers. Each device driver is registered via the DEV_un32RegNode()
    API where a pointer to an instance of this structure must be provided
    which represents the drivers interface implementation. Not all methods
    need to have an implementation and thus can be defined as NULL or
    use the alternative 'tagged' initialization method which is preferred.
*/

typedef struct osal_dev_interface_struct
{
    /*
        bInit()
    */
    BOOLEAN     (*bInit)           (
                    const char *pcDevName,
                    void const *pvProperties,
                    void **ppvPublicData
                        );

    /*
        vExit()
    */
    void        (*vExit)            (
                    void const *pvProperties,
                    void *pvPublicData
                        );

    /*
        n32Open()
    */
    N32         (*n32Open)          (
                    OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                    const void *pvArg
                        );

    /*
        n32Close()
    */
    N32         (*n32Close)         (
                    OSAL_DEV_INFO_STRUCT *psDeviceInfo
                        );

    /*
        tRead()
    */
    size_t      (*tRead)            (
                    const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                    char *pcDst,
                    size_t tSize,
                    size_t tNumObjs
                        );

    /*
        tWrite()
    */
    size_t      (*tWrite)            (
                    const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                    const char *pcSrc,
                    size_t tSize,
                    size_t tNumObjs
                        );

    /*
        n32Ioctl()
    */
    N32         (*n32Ioctl)         (
                    const OSAL_DEV_INFO_STRUCT *psDeviceInfo,
                    N32 n32Cmd,
                    va_list *ptList
                        );

} OSAL_DEV_INTERFACE_STRUCT;

/*
    General Device Driver Return Codes
*/

// API completed successfully, device is operating correctly
#define DEV_OK                               ( 0)

// An API or device error occured
#define DEV_ERROR                            (-1)

// Device name provided is too short
#define DEV_NAME_TOO_SHORT                   (-2)

// Device name provided is too long
#define DEV_NAME_TOO_LONG                    (-3)

// Device name provided already exists
#define DEV_NAME_DUPLICATE                   (-4)

// Registration of a driver cannot occur without an interface
#define DEV_NO_INTERFACE                     (-5)

// An invalid input (such as a NULL pointer) was provided
#define DEV_INVALID_INPUT                    (-6)

// The maximum number of registered drivers has been exceeded
#define DEV_MAX_DRIVERS_EXCEEDED             (-7)

// API failed to allocate required memory
#define DEV_OUT_OF_MEMORY                    (-8)

// API failed to create or acquire a mutex for the operation requested
#define DEV_MUTEX_FAILURE                    (-9)

// Device does not support the provided ioctl() command
#define DEV_UNSUPPORTED_CMD                  (-10)

/*
    Device ioctl group/class id's

    Certain drivers sometimes belong to generalized groups or classes. In
    this case common operations that devices have which belong to a group
    or class can be generalized and supported by drivers which implement
    them. To accomplish this driver group or class commands can be created.
    To help organize these groups or classes, id's can be constructed which
    allow individual class drivers to define the commands which make up the
    groups. The actual commands themselves are defined in a driver class
    header file (under \Projects\Devices\) but the group ID's are defined here.

    Each device class is given 256 possible entries.
*/

// General 'device' ioctl group id (for all drivers)
#define DEV_IOCTL                            (0x00000100)

// Device General Ioctl Group Members
#define DEV_IOCTL_SEEK_POSITION              (DEV_IOCTL + 0)
#define DEV_IOCTL_SET_TIMEOUT                (DEV_IOCTL + 1)
#define DEV_IOCTL_GET_ERROR                  (DEV_IOCTL + 2)
#define DEV_IOCTL_GET_EOF                    (DEV_IOCTL + 3)
#define DEV_IOCTL_CLEAR_ERROR                (DEV_IOCTL + 4)

/*  Tables for the properties of objects */

/* Tasks */

#if OSAL_DEBUG == 1
typedef struct osal_task_debug_struct
{
    // Debug handler function callback
    OSAL_DEBUG_HANDLER pvFxnCallback;

    // Pointer to the Argument for the fxn Callback
    const void *pvArg;

    // The name the provided for this debug registration
    const char *pacName;

    // Output enable flag
    BOOLEAN bOutputEnabled;

} OSAL_TASK_DEBUG_STRUCT;
#endif

typedef struct osal_task_info_struct
{
    // Task (user provided) parameters
    OSAL_TASK_HANDLER tTaskHandler;
    void *pvTaskHandlerArgument;
    UN32 un32StackSize;
    OSAL_TASK_PRIORITY_ENUM ePriority;
    UN32 un32Options;

    // Task Monitor Application Report Status
    N32 n32ReportedStatus;

    // Task Monitor (user provided) registration info
    UN16 un16ReportingInterval;
    OSAL_SLEEP_HANDLER tSleepHandler;
    OSAL_WAKEUP_HANDLER tWakeupHandler;
    OSAL_SHUTDOWN_HANDLER tShutdownHandler;
    void *pvHandlerArgument;

    // Task Monitor info
    UN16 un16RemainingClicksForReport;
    OSAL_LINKED_LIST_ENTRY hMonitorEntry;

    // Task statistics
    UN8 un8StackPercentUsed;
    UN32 un32StackBytesUsed;
    UN32 un32StackBytesFree;

#if OSAL_DEBUG == 1
    // Task Debug Info
    OSAL_TASK_DEBUG_STRUCT sDebug;
#endif

} OSAL_TASK_INFO_STRUCT;

/* Semaphores */

typedef struct osal_sem_info_struct
{
    UN32 un32Resources;
    UN32 un32Options;

    // Resources consumed
    UN32 un32Count;

    // Statistics
    UN32 un32MinCount;
    UN32 un32MaxCount;
    UN32 un32NotAvailableCount;
    UN32 un32TimeoutCount;
    UN32 un32TakeAttemptCount;

} OSAL_SEM_INFO_STRUCT;

/* Queues */
typedef struct osal_queue_info_struct
{
    // Queue create parameters
    UN32 un32QueueDepth;
    UN32 un32MessageSize;
    UN32 un32Options;

    // Queue real-time info
    UN32 un32MessagesAvailable;
    UN32 un32MessagesQueued;
    UN32 un32MessagesTaken;
    UN32 un32FailedAllocations;

    OSAL_SEM_INFO_STRUCT sAllocSemInfo;
    OSAL_SEM_INFO_STRUCT sPendSemInfo;

} OSAL_QUEUE_INFO_STRUCT;

/* Timers */

typedef enum osal_timer_overflow_hdlr_enum
{
    OSAL_TIMER_OVERFLOW_HANDLE_MAX_TIME,
    OSAL_TIMER_OVERFLOW_HANDLE_REMAINDER,
    OSAL_TIMER_OVERFLOW_HANDLE_EXPIRATION,

} OSAL_TIMER_OVERFLOW_HDLR_ENUM;

typedef struct osal_absolute_time_struct
{
    UN8 un8Second;
    UN8 un8Minute;
    UN8 un8Hour;
    UN16 un16Year;
    UN8 un8Day;
    UN8 un8Month;

} OSAL_ABSOLUTE_TIME_STRUCT;

typedef struct osal_timer_info_struct
{
    // User defined timer info...

    // User provided callback and arg
    OSAL_TIMER_HANDLER tTimerHandler;
    const void *pvTimerHandlerArgument;

    // Relative timers have an initial offset of something and
    // a periodic rate of some value or zero.
    // Absolute timers will calculate an initial offset (the absolute time
    // from the time the timer was started) and a zero periodic rate
    UN32 un32InitialOffset;
    UN32 bInitialOffsetOverflow;
    UN32 un32PeriodicRate;
    OSAL_TIMER_OVERFLOW_HDLR_ENUM ePeriodicRateOverflowHdlrMode;

    // Timer maintenance info...

    // Timer object this timer belongs to
    OSAL_OBJECT_HDL hTimerObj;

    // Entry handle in the active list (if active)
    OSAL_LINKED_LIST_ENTRY hActiveEntry;

    // Active flag
    BOOLEAN bActive;

    UN32 un32ExpirationDelta; // msec

    // Absolute timer struct
    BOOLEAN bAbsolute;
    OSAL_ABSOLUTE_TIME_STRUCT sAbsoluteTime;

} OSAL_TIMER_INFO_STRUCT;

/* Memory Blocks */

typedef struct osal_memory_info_struct
{
    size_t tSize;

} OSAL_MEMORY_INFO_STRUCT;

/* Interrupts */

typedef struct osal_int_info_struct
{
    UN32 un32InterruptNumber;
    UN32 un32InterruptCount;
    BOOLEAN bEnabled;

} OSAL_INT_INFO_STRUCT;

/* Logs */

typedef struct osal_log_info_struct
{
    char *pcBuffer;
    N32 n32Size;
    UN32 un32Options;

    N32 n32Index;
    BOOLEAN bLocked;
    BOOLEAN bEnabled;
    BOOLEAN bPersist;

} OSAL_LOG_INFO_STRUCT;

/* Dynamic Buffers */

typedef struct osal_block_pool_info_struct
{
    // Pool configuration
    UN32 un32Options;

    UN16 un16BlockSize;
    UN16 un16NumBlocks;
    UN32 un32ActualBlockSize;

    // Pool Allocation Statistics
    UN32 un16Allocated;
    UN32 un16MaxAllocated;
    UN32 un32OverrunCnt;

} OSAL_BLOCK_POOL_INFO_STRUCT;

// OSAL Monitor Error Handler Prototype
typedef void (*OSAL_MONITOR_ERROR_HANDLER) (
                OSAL_RETURN_CODE_ENUM eErrorCode,
                const OSAL_TASK_INFO_STRUCT *psTaskInfo,
                void *pvArg
                    );

// OSAL Heart Beat Handler Prototype
typedef void (*OSAL_HEARTBEAT_HANDLER) ( void *pvArg );

// OSAL Watchdog Handler Prototypes
typedef void (*OSAL_WATCHDOG_START_HANDLER) ( void *pvArg );
typedef void (*OSAL_WATCHDOG_STOP_HANDLER) ( void *pvArg );
typedef void (*OSAL_WATCHDOG_KICK_HANDLER) ( void *pvArg );

// OSAL Monitor Handler Structure
typedef struct osal_monitor_handler_struct {

    // OSAL Monitoring
    OSAL_MONITOR_ERROR_HANDLER vMonitorErrorHandler;
    void *pvMonitorErrorHandlerArg;

    // OSAL Heart Beat
    OSAL_HEARTBEAT_HANDLER vHeartBeatHandler;
    void *pvHeartBeatHandlerArg;

    // OSAL Watchdog
    OSAL_WATCHDOG_START_HANDLER vWatchdogStartHandler;
    OSAL_WATCHDOG_STOP_HANDLER vWatchdogStopHandler;
    OSAL_WATCHDOG_KICK_HANDLER vWatchdogKickHandler;
    void *pvWatchdogHandlerArg;

} OSAL_MONITOR_HANDLER_STRUCT;

// OSAL Start Handler Structure
typedef struct osal_start_handler_struct {

    // OSAL Device Driver Registration
    OSAL_REGISTER_DRIVERS_HANDLER bRegisterDrivers;
    OSAL_UNREGISTER_DRIVERS_HANDLER vUnregisterDrivers;
    void *pvRegisterDriversHandlerArg;

    // OSAL Bootstrap(Start)
    OSAL_START_HANDLER bStartHandler;
    void *pvStartHandlerArg;

    // OSAL Monitor Handlers
    OSAL_MONITOR_HANDLER_STRUCT sMonitorHandlers;

} OSAL_START_HANDLER_STRUCT;

// OSAL API Structure

// File Interface, for use by various OS-Dependent File I/O APIs
typedef struct osal_file_interface_struct
{
    void* (*psFopen)(const char *pcFileName, const char *pcMode);
    int (*iFclose)(void *psFile);
    size_t (*tFread)(void *pvDst, size_t tSize, size_t tNumObj,
                    void *psFile);
    size_t (*tFwrite)(const void *pvSrc, size_t tSize, size_t tNumObj,
                     void *psFile);
    long (*lFtell)(void *psFile);
    int (*iIoctl)(void *psFile, int iCmd, va_list *ptList);
    int (*iFseek)(void *psFile, long lOffset, int iMode);
    int (*iRemove)(const char *pcFileName);
    int (*iRename)(const char *pcOldFileName, const char *pcNewFileName);
    int (*iFflush)(void *psFile);
    int (*iFerror)(void *psFile);
    int (*iFeof)(void *psFile);
    void (*vClearerror)(void *psFile);
} OSAL_FILE_INTERFACE_STRUCT;

typedef struct osal_api_struct
{
    /*
        General APIs
    */
    OSAL_CONFIG_INFO_MASK (*tConfigInfo)( void );

    BOOLEAN (*bStart)(
        const OSAL_START_HANDLER_STRUCT *psStartHandlers
            );

    BOOLEAN (*bSleep)( void );

    BOOLEAN (*bWakeup)( void );

    BOOLEAN (*bShutdown)( void );

    void (*vEnterCriticalSection) ( void );

    void (*vExitCriticalSection) ( void );

    OSAL_RETURN_CODE_ENUM (*eEnterTaskSafeSection) ( void );

    OSAL_RETURN_CODE_ENUM (*eExitTaskSafeSection) ( void );

    OSAL_RETURN_CODE_ENUM (*eGetUtilizationCPU) (
        UN8 *pun8Current,
        UN8 *pun8Max,
        UN8 *pun8Min
            );

    void (*vResetUtilizationCPU) ( void );

    const char * (*pacGetReturnCodeName) (
        OSAL_RETURN_CODE_ENUM eReturnCode
            );

    const char * (*pacGetVersion) (
        UN8 *pun8MajorVersion,
        UN8 *pun8MinorVersion,
        UN8 *pun8SubVersion
            );

    const char * (*pacGetObjectName) (
        OSAL_OBJECT_HDL hObj
            );

    void (*vReportError) (
        OSAL_RETURN_CODE_ENUM eErrorCode
            );

    /*
        Task Management APIs
    */

    OSAL_RETURN_CODE_ENUM (*eTaskCreate) (
        OSAL_OBJECT_HDL *phTaskObj,
        const char *pacName,
        OSAL_TASK_HANDLER tTaskHandler,
        void *pvTaskHandlerArgument,
        UN32 un32StackSize,
        OSAL_TASK_PRIORITY_ENUM ePriority,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eTaskDelete) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskSuspend) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskResume) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskDelay) (
        UN32 un32Milliseconds
            );

    void (*vTaskYield) ( void );

    OSAL_RETURN_CODE_ENUM (*eTaskChangePriority) (
        OSAL_OBJECT_HDL hTaskObj, OSAL_TASK_PRIORITY_ENUM eNewPriority
            );

    OSAL_RETURN_CODE_ENUM (*eTaskRegister) (
        UN16 un16ReportingInterval,
        OSAL_SLEEP_HANDLER tSleepHandler,
        OSAL_WAKEUP_HANDLER tWakeupHandler,
        OSAL_SHUTDOWN_HANDLER tShutdownHandler,
        void *pvHandlerArgument
            );

    OSAL_RETURN_CODE_ENUM (*eTaskUnregister) ( void );

    OSAL_RETURN_CODE_ENUM (*eTaskReport) (
        N32 n32Status
            );

    OSAL_OBJECT_HDL (*hTaskGetHandle) ( void );

    OSAL_OBJECT_HDL (*hTaskGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eTaskGetInfo ) (
        OSAL_OBJECT_HDL hTaskObj,
        OSAL_TASK_INFO_STRUCT *psTaskInfo
            );

    BOOLEAN (*bTaskGetId) ( OSAL_TASK_ID *ptTaskId );

    OSAL_RETURN_CODE_ENUM (*eTaskList) ( void );

    /*
        Task Communication APIs
    */

    OSAL_RETURN_CODE_ENUM (*eQueueCreate) (
        OSAL_OBJECT_HDL *phQueueObj,
        const char *pacName,
        UN32 un32QueueDepth, UN32 un32MessageSize,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eQueueDelete) (
        OSAL_OBJECT_HDL hQueueObj
            );

    OSAL_RETURN_CODE_ENUM (*eQueueGet) (
        OSAL_OBJECT_HDL hQueueObj,
        void **ppvMessage, UN32 *pun32MessageSize,
        N32 n32Timeout
            );

    OSAL_RETURN_CODE_ENUM (*eQueuePut) (
        const void *pvMessage,
        UN32 un32MessageSize
            );

    OSAL_OBJECT_HDL (*hQueueGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eQueueGetInfo) (
        OSAL_OBJECT_HDL hQueueObj,
        OSAL_QUEUE_INFO_STRUCT *psQueueInfo
            );

    OSAL_RETURN_CODE_ENUM (*eQueueList) ( void );

    OSAL_RETURN_CODE_ENUM (*eMessageAllocate) (
        OSAL_OBJECT_HDL hQueueObj,
        void **ppvMessage,
        UN32 un32Flags
            );

    OSAL_RETURN_CODE_ENUM (*eMessageFree) (
        const void *pvMessage
            );

    /*
        Task Synchronization APIs
    */

    OSAL_RETURN_CODE_ENUM (*eSemCreate) (
        OSAL_OBJECT_HDL *phSemObj,
        const char *pacName,
        UN32 un32InitialValue,
        UN32 un32Resources,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eSemDelete) (
        OSAL_OBJECT_HDL hSemObj
            );

    OSAL_RETURN_CODE_ENUM (*eSemGive) (
        OSAL_OBJECT_HDL hSemObj
            );

    OSAL_RETURN_CODE_ENUM (*eSemTake) (
        OSAL_OBJECT_HDL hSemObj,
        N32 n32Timeout
            );

    OSAL_OBJECT_HDL (*hSemGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eSemGetInfo) (
        OSAL_OBJECT_HDL hSemObj,
        OSAL_SEM_INFO_STRUCT *psSemInfo
            );

    OSAL_RETURN_CODE_ENUM (*eSemList) ( void );

    /*
        Timer Management APIs
    */

    OSAL_RETURN_CODE_ENUM (*eTimerCreate) (
        OSAL_OBJECT_HDL *phTimerObj,
        const char *pacName,
        OSAL_TIMER_HANDLER tTimerHandler,
        const void *pvTimerHandlerArgument
            );

    OSAL_RETURN_CODE_ENUM (*eTimerDelete) (
        OSAL_OBJECT_HDL hTimerObj
            );

    OSAL_RETURN_CODE_ENUM (*eTimerSetHandler) (
        OSAL_OBJECT_HDL hTimerObj,
        OSAL_TIMER_HANDLER tTimerHandler
            );

    OSAL_RETURN_CODE_ENUM (*eTimerSetHandlerArg) (
        OSAL_OBJECT_HDL hTimerObj,
        const void *pvTimerHandlerArgument
            );

    OSAL_RETURN_CODE_ENUM (*eTimerStartRelative) (
        OSAL_OBJECT_HDL hTimerObj,
        UN32 un32InitialOffset,
        UN32 un32PeriodicRate
            );

    OSAL_RETURN_CODE_ENUM (*eTimerStartAbsolute) (
        OSAL_OBJECT_HDL hTimerObj,
        UN8 un8Second,
        UN8 un8Minute,
        UN8 un8Hour,
        UN16 un16Year,
        UN8 un8Day,
        UN8 un8Month
            );

    OSAL_RETURN_CODE_ENUM (*eTimerStop) (
        OSAL_OBJECT_HDL hTimerObj
            );

    OSAL_RETURN_CODE_ENUM (*eTimerRemaining) (
        OSAL_OBJECT_HDL hTimerObj,
        UN32 *pun32TimeRemaining
            );

    OSAL_OBJECT_HDL (*hTimerGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eTimerGetInfo) (
        OSAL_OBJECT_HDL hTimerObj,
        OSAL_TIMER_INFO_STRUCT *psTimerInfo
            );

    OSAL_RETURN_CODE_ENUM (*eTimerList) ( void );

    /*
        Time Management APIs
    */

    OSAL_RETURN_CODE_ENUM (*eTimeSet) (
        UN32 un32Seconds
            );

    OSAL_RETURN_CODE_ENUM (*eTimeSetGMToffset) (
        N32 n32Minutes
            );

    OSAL_RETURN_CODE_ENUM (*eTimeSetDSTadjustment) (
        UN32 un32Minutes
            );

    OSAL_RETURN_CODE_ENUM (*eTimeSetRegisterNotification) (
        OSAL_TIME_NOTIFICATION_OBJECT *phNotificationHandle,
        OSAL_TIME_UPDATE_MASK tUpdateMask,
        OSAL_TIME_UPDATE_HANDLER vHandler,
        void *pvHandlerArg
            );

    OSAL_RETURN_CODE_ENUM (*eTimeSetUnRegisterNotification) (
        OSAL_TIME_NOTIFICATION_OBJECT hNotificationHandle
            );

    OSAL_RETURN_CODE_ENUM (*eTimeGet) (
        UN32 *pun32Seconds
            );

    OSAL_RETURN_CODE_ENUM (*eTimeGetLocal) (
        UN32 *pun32Seconds
            );

    OSAL_RETURN_CODE_ENUM (*eTimeGetGMToffset) (
        N32 *pn32Minutes
            );

    OSAL_RETURN_CODE_ENUM (*eTimeGetDSTadjustment) (
        UN32 *pun32Minutes
            );

    void (*vTimeUp) (
        UN32 *pun32Seconds,
        UN16 *pun16Msecs
            );

    struct tm *(*gmtime_r) (
        const TIME_T *timer,
        struct tm *tmbuf
            );

    struct tm *(*localtime_r) (
        const TIME_T *timer,
        struct tm *tmbuf
            );

    TIME_T (*mktime) (
        struct tm *tmbuf
            );

    double (*difftime) (
        TIME_T time1,
        TIME_T time0
            );

    char * (*asctime_r) (
        const struct tm *tp,
        char *buf
            );

    char * (*ctime_r) (
        const TIME_T *tp,
        char *buf
            );

    /*
        Memory Management and Interface APIs
    */

    BOOLEAN (*bMemRead8) (
        const void *pvMemoryAddress,
        UN8 *pun8Value
            );

    BOOLEAN (*bMemWrite8) (
         void *pvMemoryAddress,
         UN8 un8Value
             );

    BOOLEAN (*bMemRead16) (
         const void *pvMemoryAddress,
         UN16 *pun16Value
             );

    BOOLEAN (*bMemWrite16) (
          void *pvMemoryAddress,
          UN16 un16Value
              );

    BOOLEAN (*bMemRead32) (
         const void *pvMemoryAddress,
         UN32 *pun32Value
             );

    BOOLEAN (*bMemWrite32) (
          void *pvMemoryAddress,
          UN32 un32Value
              );

    BOOLEAN (*bPortRead8) (
         const void *pvPortAddress,
         UN8 *pun8Value
             );

    BOOLEAN (*bPortWrite8) (
          void *pvPortAddress,
          UN8 un8Value
              );

    BOOLEAN (*bPortRead16) (
          const void *pvPortAddress,
          UN16 *pun16Value
              );

    BOOLEAN (*bPortWrite16) (
           void *pvPortAddress,
           UN16 un16Value
               );

    BOOLEAN (*bPortRead32) (
          const void *pvPortAddress,
          UN32 *pun32Value
              );

    BOOLEAN (*bPortWrite32) (
           void *pvPortAddress,
           UN32 un32Value
               );

    BOOLEAN (*bMemCpy) (
          void *pvDst,
          const void *pvSrc,
          size_t tSize
              );

    BOOLEAN (*bMemSet) (
          void *pvDst,
          UN8 un8Value,
          size_t tSize
              );

    void * (*pvMemoryAllocate) (
          const char *pacName,
          size_t tSize,
          BOOLEAN bZeroInitialize
              );

    void (*vMemoryFree) (
          void *pvMemoryAddress
              );

    BOOLEAN (*bMemoryUsage) (
          UN32 *pun32CurrentAllocatedBlocks,
          UN32 *pun32CurrentActualAllocatedBlocks,
          UN32 *pun32CurrentUserBytes,
          UN32 *pun32CurrentActualBytes,
          UN32 *pun32MaxAllocatedBlocks,
          UN32 *pun32MaxActualAllocatedBlocks,
          UN32 *pun32MaxUserBytes,
          UN32 *pun32MaxActualBytes,
          UN32 *pun32TotalSystemBytes
               );

    OSAL_RETURN_CODE_ENUM (*eMemoryList) (
        const char *pacFilterText
            );

    /* Linked List APIs */

    OSAL_RETURN_CODE_ENUM (*eLinkedListCreate) (
        OSAL_OBJECT_HDL *phLL,
        const char *pacName,
        OSAL_LL_COMPARE_HANDLER tn16CompareFunction,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListDelete) (
        OSAL_OBJECT_HDL hLL
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListAdd) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY *phEntry,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListReplaceEntry)(
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY hEntry,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListAddBeforeEntry)(
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY *phEntry,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListAddAfterEntry)(
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY *phEntry,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListRemove) (
        OSAL_LINKED_LIST_ENTRY hEntry
            );


    OSAL_RETURN_CODE_ENUM (*eLinkedListSearch) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY *phEntry,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListLinearSearch) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LINKED_LIST_ENTRY *phEntry,
        OSAL_LL_COMPARE_HANDLER tn16CompareFunction,
        void *pvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListSort) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LL_COMPARE_HANDLER tn16Compare
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListRemoveAll) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LL_RELEASE_HANDLER tvRelease
            );

    void *(*pvLinkedListThis) (
        OSAL_LINKED_LIST_ENTRY hEntry
            );

    OSAL_LINKED_LIST_ENTRY (*hLinkedListFirst) (
        OSAL_OBJECT_HDL hLL,
        void **ppvData
            );

    OSAL_LINKED_LIST_ENTRY (*hLinkedListLast) (
        OSAL_OBJECT_HDL hLL,
        void **ppvData
            );

    OSAL_LINKED_LIST_ENTRY (*hLinkedListNext) (
        OSAL_LINKED_LIST_ENTRY hCurrent,
        void **ppvData
            );

    OSAL_LINKED_LIST_ENTRY (*hLinkedListPrev) (
        OSAL_LINKED_LIST_ENTRY hCurrent,
        void **ppvData
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListIterate) (
        OSAL_OBJECT_HDL hLL,
        OSAL_LL_ITERATOR_HANDLER tvIterator,
        void *pvArg
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListItems) (
        OSAL_OBJECT_HDL hLL,
        UN32 *pun32Items
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListProtect) (
        OSAL_OBJECT_HDL hLL
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListUnprotect) (
        OSAL_OBJECT_HDL hLL
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListLock) (
        OSAL_OBJECT_HDL hLL
            );

    OSAL_RETURN_CODE_ENUM (*eLinkedListUnlock) (
        OSAL_OBJECT_HDL hLL
            );

    void * (*pvLinkedListMemoryAllocate)(
        const char *pacName,
        size_t tSize,
        BOOLEAN bZeroInitialize
            );

    void (*vLinkedListMemoryFree)(
          void *pvObject
              );

    /* Debug APIs */

    OSAL_RETURN_CODE_ENUM (*eDebugRegister)(
        const char *pacName,
        OSAL_DEBUG_HANDLER hCallBackFunc,
        const void *pvArg);

    OSAL_RETURN_CODE_ENUM (*eDebugUnregister)( void );

    BOOLEAN (*bOutputEnabledThisTask)( void );

    void (*vEnableOutputThisTask)( void );

    void (*vDisableOutputThisTask)( void );

    void (*vControlOutputThisTask)( BOOLEAN );

    BOOLEAN (*bDebugUpdateName) (const char *pacName );

    void (*vDebugExitHandler)(void);

    char * (*pacSetDebugCommandLineText) ( char *pacText );

    /* Loggin APIs */

    OSAL_RETURN_CODE_ENUM (*eLogCreate)(
        OSAL_OBJECT_HDL *phLog,
        const char *pacName,
        N32 n32Size,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eLogDelete)(
        OSAL_OBJECT_HDL hLog
            );

    N32 (*n32LogWrite)(
        OSAL_OBJECT_HDL hLog,
        const char *pcFormat,
        ...
            );

    N32 (*n32VLogWrite)(
        OSAL_OBJECT_HDL hLog,
        const char *pcFormat,
        va_list *ptArgList
            );

    N32 (*n32LogRead)(
        OSAL_OBJECT_HDL hLog,
        char *pcDest,
        N32 n32Position,
        N32 n32Length
            );

    OSAL_RETURN_CODE_ENUM (*eLogClear)(
        OSAL_OBJECT_HDL hLog
            );

    void (*vLogDisable)(
        OSAL_OBJECT_HDL hLog
            );

    void (*vLogEnable)(
        OSAL_OBJECT_HDL hLog
            );

    void (*vLogPersist)(
        OSAL_OBJECT_HDL hLog,
        BOOLEAN bPersist
            );

    OSAL_OBJECT_HDL (*hLogGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eLogGetInfo) (
        OSAL_OBJECT_HDL hLog,
        OSAL_LOG_INFO_STRUCT *psLogInfo
            );

    OSAL_RETURN_CODE_ENUM (*eLogList)  ( BOOLEAN bVerbose );

    OSAL_RETURN_CODE_ENUM (*eLogIterate)  (
        OSAL_LOG_ITERATOR_HANDLER pbIterator,
        void *pvIteratorArg
            );

    /* Buffer APIs */

    OSAL_RETURN_CODE_ENUM (*eBlockPoolCreate) (
        OSAL_OBJECT_HDL *phBlockPool,
        const char *pacName,
        UN16 un16BlockSize,
        UN16 un16NumBlocks,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eBlockPoolDelete) (
        OSAL_OBJECT_HDL hBlockPool
            );

    OSAL_BUFFER_HDL (*hBufferAllocate) (
        OSAL_OBJECT_HDL hBlockPool,
        BOOLEAN bBlockingRead,
        BOOLEAN bBlockingWrite,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eBufferFree) (
        OSAL_BUFFER_HDL hBuffer
            );

    size_t (*tBufferGetSize) (
            OSAL_BUFFER_HDL hBuffer
                );

    size_t (*tBufferGetSizeInBits) (
        OSAL_BUFFER_HDL hBuffer
            );

    OSAL_RETURN_CODE_ENUM (*eBufferBlocksIterate) (
        OSAL_BUFFER_HDL hBuffer,
        OSAL_BUFFER_ITERATOR_HANDLER teIterator,
        void *pvArg
            );

    size_t (*tBufferPeek) (
        OSAL_BUFFER_HDL hBuffer,
        void *pvDst,
        size_t tSize,
        size_t tOffset
            );

    size_t (*tBufferPeekBits) (
        OSAL_BUFFER_HDL hBuffer,
        void *pvDst,
        size_t tDstBitOffset,
        N32 n32Bits,
        size_t tBufferOffsetBits
            );

    size_t (*tBufferReadHead) (
        OSAL_BUFFER_HDL hBuffer,
        void *pvDst,
        size_t tSize,
        ...
            );

    size_t (*tBufferReadHeadBits) (
        OSAL_BUFFER_HDL hBuffer,
        void *pvDst,
        size_t tDstBitOffset,
        N32 n32Bits,
        ...
            );

    size_t (*tBufferReadTail) (
        OSAL_BUFFER_HDL hBuffer,
        void *pvDst,
        size_t tSize,
        ...
            );

    size_t (*tBufferSeekHead) (
        OSAL_BUFFER_HDL hBuffer,
        size_t tSize
        );

    size_t (*tBufferSeekHeadBits) (
        OSAL_BUFFER_HDL hBuffer,
        size_t tSizeBits
        );

    size_t (*tBufferSeekTail) (
        OSAL_BUFFER_HDL hBuffer,
        size_t tSize
        );

    OSAL_BUFFER_BLOCK_HDL (*hBufferReadBlock) (
        OSAL_BUFFER_HDL hBuffer,
        UN8 **ppun8Data,
        size_t *ptSize,
        ...
            );

    OSAL_RETURN_CODE_ENUM (*eBufferReleaseBlock) (
        OSAL_BUFFER_BLOCK_HDL hBufferBlock
            );

    OSAL_RETURN_CODE_ENUM (*eBufferAppend) (
        OSAL_BUFFER_HDL hDstBuffer,
        OSAL_BUFFER_HDL hBufferDataToAppend
            );

    OSAL_RETURN_CODE_ENUM (*eBufferWriteToFile) (
        OSAL_BUFFER_HDL hBuffer,
        BOOLEAN bConsumeData,
        FILE *psFile
            );

    size_t (*tBufferWriteHead) (
        OSAL_BUFFER_HDL hBuffer,
        const void *pvSrc,
        size_t tSize,
        ...
            );

    size_t (*tBufferWriteTail) (
        OSAL_BUFFER_HDL hBuffer,
        const void *pvSrc,
        size_t tSize,
        ...
            );

    size_t (*tBufferWriteTailBits) (
        OSAL_BUFFER_HDL hBuffer,
        const void *pvSrc,
        size_t tSrcBitOffset,
        N32 n32Bits,
        ...
            );

    OSAL_BUFFER_BLOCK_HDL (*hBufferGetBlock) (
        OSAL_BUFFER_HDL hBuffer,
        UN8 **ppun8Data,
        size_t *ptSize,
        ...
            );

    OSAL_RETURN_CODE_ENUM (*eBufferWriteBlock) (
        OSAL_BUFFER_BLOCK_HDL hBufferBlock,
        size_t tSize
            );

    OSAL_OBJECT_HDL (*hBufferGetHandleByName) (
        const char *pacName
            );

    OSAL_RETURN_CODE_ENUM (*eBlockPoolGetInfo) (
        OSAL_OBJECT_HDL hBlockPoolObj,
        OSAL_BLOCK_POOL_INFO_STRUCT *psBlockPoolInfo
            );

    OSAL_RETURN_CODE_ENUM (*eBlockPoolList) ( void );

    size_t (*tBufferReadString) (
        OSAL_BUFFER_HDL hBuffer,
        char *pacString,
        size_t tSize
            );

    BOOLEAN (*bBufferReadBitsToUN16) (
        OSAL_BUFFER_HDL hBuffer,
        UN16 *pun16Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferReadBitsToN16) (
        OSAL_BUFFER_HDL hBuffer,
        N16 *pn16Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferReadBitsToUN32) (
        OSAL_BUFFER_HDL hBuffer,
        UN32 *pun32Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferReadBitsToN32) (
        OSAL_BUFFER_HDL hBuffer,
        N32 *pn32Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferReadBitsToUN64) (
        OSAL_BUFFER_HDL hBuffer,
        UN64 *pun64Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferReadBitsToN64) (
        OSAL_BUFFER_HDL hBuffer,
        N64 *pn64Dest,
        size_t tBits
            );

    BOOLEAN (*bBufferPeekBitsToUN16) (
        OSAL_BUFFER_HDL hBuffer,
        UN16 *pun16Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    BOOLEAN (*bBufferPeekBitsToN16) (
        OSAL_BUFFER_HDL hBuffer,
        N16 *pn16Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    BOOLEAN (*bBufferPeekBitsToUN32) (
        OSAL_BUFFER_HDL hBuffer,
        UN32 *pun32Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    BOOLEAN (*bBufferPeekBitsToN32) (
        OSAL_BUFFER_HDL hBuffer,
        N32 *pn32Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    BOOLEAN (*bBufferPeekBitsToUN64) (
        OSAL_BUFFER_HDL hBuffer,
        UN64 *pun64Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    BOOLEAN (*bBufferPeekBitsToN64) (
        OSAL_BUFFER_HDL hBuffer,
        N64 *pn64Dest,
        size_t tBits,
        size_t tBufferOffsetBits
            );

    /* CRC APIs */

    OSAL_RETURN_CODE_ENUM (*eGetCRC) (
        OSAL_OBJECT_HDL *phCRC,
        OSAL_CRC_TYPE_ENUM eCRCType
            );

    OSAL_RETURN_CODE_ENUM (*eReleaseCRC) (
        OSAL_OBJECT_HDL hCRC
            );

    BOOLEAN (*bInitializeCRC) (
        OSAL_OBJECT_HDL hCRC,
        OSAL_CRC_RESULT *ptInitialValue
            );

    OSAL_CRC_RESULT (*tComputeCRC) (
        OSAL_OBJECT_HDL hCRC,
        OSAL_CRC_RESULT tInputCRC,
        OSAL_BUFFER_HDL hBuffer,
        size_t tByteOffset,
        size_t tNumBytesToProcess,
        size_t *ptBytesProcessed
            );

    OSAL_RETURN_CODE_ENUM (*eCRCList) ( void );

    /* Device Framework APIs */

    N32 (*n32RegNode)(
        const char *pcDevName,
        const OSAL_DEV_INTERFACE_STRUCT *psInterface,
        const void *pvProperties,
        size_t tPropertySize
            );

    BOOLEAN (*bUnRegNode)(
        const char *pcDevName
            );

    /* File System APIs */

    BOOLEAN (*bFileSystemStart) ( void );

    BOOLEAN (*bFileSystemStop) ( void );

    void (*vFileSystemVolumes) ( OSAL_FILE_STRUCT *psFile );

    BOOLEAN (*bFileSystemVolumeInfo) (
         OSAL_FILE_STRUCT *psFile,
         const char *pcVolName );

    BOOLEAN (*bFileSystemDirectory) (
         OSAL_FILE_STRUCT *psFile,
         const char *pcPathName );

    BOOLEAN (*bFileSystemMakeDir) (
         const char *pcDirName );

    BOOLEAN (*bFileSystemRemoveDir) (
         const char *pcDirName );

    OSAL_RETURN_CODE_ENUM (*eFileSystemGetDirItems) (
                    OSAL_OBJECT_HDL *hDirItems,
                    const char *pcPathName,
                    OSAL_FS_FILE_QUALIFIER bFileQualifierFunc,
                    void *pvQualiferArg );

    OSAL_RETURN_CODE_ENUM (*eFileSystemReleaseDirItems) (
                    OSAL_OBJECT_HDL hDirItems );

    BOOLEAN (*bFileSystemTruncateFile) ( OSAL_FILE_STRUCT *psFile,
                                         size_t tNewSize );

    BOOLEAN (*bFileSystemGetFileSize) ( OSAL_FILE_STRUCT *psFile,
                                        size_t *ptFileSize );

    BOOLEAN (*bFileSystemGetFileAttributes) ( const char *pcPath,
                                              UN8 *pun8FileAttributes );

    BOOLEAN (*bFileSystemSetFileAttributes) ( const char *pcPath,
                                              UN8 un8FileAttributes );

    BOOLEAN (*bFileSystemSetTempPath) ( const char *pacTempPath );

    size_t (*tFileSystemGetTempPath) ( size_t tDstSize,
                                       char *pacDst );
    
    BOOLEAN (*bFileSystemSyncFile) ( FILE *psFile );

    BOOLEAN (*bFileSystemUnconditionalFileUnlink) ( const char *pcPath );

    /* ANSI-C File-IO APIs */

    OSAL_FILE_STRUCT* (*psFopen)(const char *pcFileName, const char *pcMode);
    int (*iFclose)(OSAL_FILE_STRUCT *psFile);
    size_t (*tFread)(void *pvDst, size_t tSize, size_t tNumObj,
                      OSAL_FILE_STRUCT *psFile);
    size_t (*tFwrite)(const void *pvSrc, size_t tSize, size_t tNumObj,
                       OSAL_FILE_STRUCT *psFile);
    long (*lFtell)(OSAL_FILE_STRUCT *psFile);
    int (*iIoctl)(OSAL_FILE_STRUCT *psFile, int iCmd, ...);
    int (*iFseek)(OSAL_FILE_STRUCT *psFile, long lOffset, int iMode);
    int (*iRemove)(const char *pcFileName);
    int (*iRename)(const char *pcOldFileName, const char *pcNewFileName);
    int (*iCopyFile)(const char *pcSrcFileName, const char *pcDstFileName, BOOLEAN bForce);
    int (*iFflush)(OSAL_FILE_STRUCT *psFile);
    int (*iFeof)(OSAL_FILE_STRUCT *psFile);
    int (*iFerror)(OSAL_FILE_STRUCT *psFile);
    void (*vClearerr)(OSAL_FILE_STRUCT *psFile);

    int (*iFgetc)(OSAL_FILE_STRUCT *psFile);
    char* (*pcFgets)(char *pcDest, int iNum, OSAL_FILE_STRUCT *psFile);
    int (*iFputc)(int iChar, OSAL_FILE_STRUCT *psFile);
    int (*iFputs)(const char *pcSrc, OSAL_FILE_STRUCT *psFile);
    int (*iFPrintf)(OSAL_FILE_STRUCT *psFile, const char *pcFormat, ...);
    void (*vDump)( OSAL_FILE_STRUCT *psFile, UN16 *pun16Row,
                    const void *pvData, UN32 un32Length );

    /* ANSI-C stdio APIs */

    int (*iPrintf)(const char *pcFormat, ...);
    int (*iPutchar)(int iChar);
    int (*iPuts)(const char *pcSrc);
    int (*iGetchar)(void);
    char* (*pcGets)(char *pcDest);

} OSAL_API_STRUCT;

typedef struct fixed_interface_struct
{
    OSAL_FIXED_OBJECT (*hCreate) (
        N32 n32WholeValue,
        N32 n32FractionalValue,
        UN8 un8Pow10
        );

    OSAL_FIXED_OBJECT (*hCreateFromFixed) (
        N32 n32FixedValue,
        UN8 un8NumFractionalBits
            );

    OSAL_FIXED_OBJECT (*hCreateInMemory) (
        N32 n32FixedValue,
        UN8 un8NumFractionalBits,
        void *pvMemory
            );

    OSAL_FIXED_OBJECT (*hDuplicate) (
        OSAL_FIXED_OBJECT hFixed
            );

    OSAL_FIXED_OBJECT (*hDuplicateInMemory) (
        OSAL_FIXED_OBJECT hFixed,
        void *pvMemory
            );

    BOOLEAN (*bCopyToMemory) (
        OSAL_FIXED_OBJECT hFixed,
        void *pvMemory
            );

    void (*vDestroy) (
        OSAL_FIXED_OBJECT hFixed
            );

    N32 (*n32Value) (
        OSAL_FIXED_OBJECT hFixed
            );

    UN8 (*un8NumFractionalBits) (
        OSAL_FIXED_OBJECT hFixed
            );

    N32 (*n32ScaledValue) (
        OSAL_FIXED_OBJECT hFixed,
        UN8 un8NumFractionalBits
            );

    N32 (*n32Ceiling) (
        OSAL_FIXED_OBJECT hFixed
            );

    N32 (*n32Floor) (
        OSAL_FIXED_OBJECT hFixed
            );

    OSAL_RETURN_CODE_ENUM (*eAdd) (
        OSAL_FIXED_OBJECT hFixed1,
        OSAL_FIXED_OBJECT hFixed2,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eSubtract) (
        OSAL_FIXED_OBJECT hFixed1,
        OSAL_FIXED_OBJECT hFixed2,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eMultiply) (
        OSAL_FIXED_OBJECT hFixed1,
        OSAL_FIXED_OBJECT hFixed2,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eDivide) (
        OSAL_FIXED_OBJECT hFixedNumerator,
        OSAL_FIXED_OBJECT hFixedDenominator,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eDegreeToRadian) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eRadianToDegree) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eSin) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );
    OSAL_RETURN_CODE_ENUM (*eASin) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eCos) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eACos) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eATan) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eATan2) (
        OSAL_FIXED_OBJECT hFixedY,
        OSAL_FIXED_OBJECT hFixedX,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eAbs) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eSqrt) (
        OSAL_FIXED_OBJECT hFixed,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*ePow) (
        OSAL_FIXED_OBJECT hFixed,
        UN8 un8Power,
        OSAL_FIXED_OBJECT hResult
            );

    OSAL_RETURN_CODE_ENUM (*eEval) (
        OSAL_FIXED_OBJECT hResult,
        void *pvData,
        size_t tDataSize,
        const char *pacExpr,
        ...
            );

    N16 (*n16Compare) (
        OSAL_FIXED_OBJECT hFixed1,
        OSAL_FIXED_OBJECT hFixed2
            );

    N32 (*n32FWrite) (
        OSAL_FIXED_OBJECT hFixed,
        FILE *psFile
            );

    OSAL_FIXED_OBJECT (*hFRead) (
        FILE *psFile
            );

    N32 (*n32FPrintf) (
        OSAL_FIXED_OBJECT hFixed,
        FILE *psFile,
        BOOLEAN bIsTerse
            );
} OSAL_FIXED_INTERFACE_STRUCT;

// OS Interface Structure

typedef struct os_intf_struct
{
    /*
        General APIs
    */
    BOOLEAN (*bStart)(
        OSAL_START_HANDLER tStartHandler,
        const void *pvStartHandlerArg
            );

    void (*vEnterCriticalSection) ( void );

    void (*vExitCriticalSection) ( void );

    OSAL_RETURN_CODE_ENUM (*eGetUtilizationCPU) (
        UN8 *pun8Current,
        UN8 *pun8Max,
        UN8 *pun8Min
            );

    void (*vResetUtilizationCPU) ( void );

    void (*vUpdateCPUState) ( void );

    /*
        Task Management APIs
    */

    OSAL_RETURN_CODE_ENUM (*eTaskCreate) (
        OSAL_OBJECT_HDL *phTaskObj,
        const char *pacName,
        OSAL_TASK_HANDLER tTaskHandler,
        void *pvTaskHandlerArgument,
        UN32 un32StackSize,
        OSAL_TASK_PRIORITY_ENUM ePriority,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eTaskDelete) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskSuspend) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskResume) (
        OSAL_OBJECT_HDL hTaskObj
            );

    OSAL_RETURN_CODE_ENUM (*eTaskDelay) (
        UN32 un32Milliseconds
            );

    void (*vTaskYield) ( void );

    OSAL_RETURN_CODE_ENUM (*eTaskChangePriority) (
        OSAL_OBJECT_HDL hTaskObj, OSAL_TASK_PRIORITY_ENUM eNewPriority
            );

    OSAL_OBJECT_HDL (*hTaskGetHandle) ( void );

    OSAL_RETURN_CODE_ENUM (*eTaskGetInfo ) (
        OSAL_OBJECT_HDL hTaskObj,
        OSAL_TASK_INFO_STRUCT *psTaskInfo
            );

    BOOLEAN (*bTaskGetId) ( OSAL_TASK_ID *ptTaskId );

    OSAL_RETURN_CODE_ENUM (*eTaskList) ( void );

    /*
        Task Synchronization APIs
    */

    OSAL_RETURN_CODE_ENUM (*eSemCreate) (
        OSAL_OBJECT_HDL *phSemObj,
        const char *pacName,
        UN32 un32InitialValue,
        UN32 un32Resources,
        UN32 un32Options
            );

    OSAL_RETURN_CODE_ENUM (*eSemDelete) (
        OSAL_OBJECT_HDL hSemObj
            );

    OSAL_RETURN_CODE_ENUM (*eSemGive) (
        OSAL_OBJECT_HDL hSemObj
            );

    OSAL_RETURN_CODE_ENUM (*eSemTake) (
        OSAL_OBJECT_HDL hSemObj,
        N32 n32Timeout
            );

    /*
        Time Management APIs
    */

    OSAL_RETURN_CODE_ENUM (*eGetTime) (
        UN32 *pun32Sec,
        UN16 *pun16MSec
            );

    /*
        Memory Management and Interface APIs
    */

    BOOLEAN (*bMemRead8) (
        const void *pvMemoryAddress,
        UN8 *pun8Value
            );

    BOOLEAN (*bMemWrite8) (
         void *pvMemoryAddress,
         UN8 un8Value
             );

    BOOLEAN (*bMemRead16) (
         const void *pvMemoryAddress,
         UN16 *pun16Value
             );

    BOOLEAN (*bMemWrite16) (
          void *pvMemoryAddress,
          UN16 un16Value
              );

    BOOLEAN (*bMemRead32) (
         const void *pvMemoryAddress,
         UN32 *pun32Value
             );

    BOOLEAN (*bMemWrite32) (
          void *pvMemoryAddress,
          UN32 un32Value
              );

    BOOLEAN (*bPortRead8) (
         const void *pvPortAddress,
         UN8 *pun8Value
             );

    BOOLEAN (*bPortWrite8) (
          void *pvPortAddress,
          UN8 un8Value
              );

    BOOLEAN (*bPortRead16) (
          const void *pvPortAddress,
          UN16 *pun16Value
              );

    BOOLEAN (*bPortWrite16) (
           void *pvPortAddress,
           UN16 un16Value
               );

    BOOLEAN (*bPortRead32) (
          const void *pvPortAddress,
          UN32 *pun32Value
              );

    BOOLEAN (*bPortWrite32) (
           void *pvPortAddress,
           UN32 un32Value
               );

    BOOLEAN (*bMemCpy) (
          void *pvDst,
          const void *pvSrc,
          size_t tSize
              );

    BOOLEAN (*bMemSet) (
          void *pvDst,
          UN8 un8Value,
          size_t tSize
              );

    void * (*pvMemoryAllocate) (
          const char *pacName,
          size_t tSize,
          BOOLEAN bZeroInitialize,
          size_t *ptActualSizeAllocated
              );

    void (*vMemoryFree) (
          void *pvMemoryAddress,
          size_t *ptNumBytesFreed
              );

    UN32 (*un32TotalSystemBytes) (
           void
              );

    /* File System APIs */

    BOOLEAN (*bFileSystemStart) ( void );

    BOOLEAN (*bFileSystemStop) ( void );

    void (*vFileSystemVolumes) ( OSAL_FILE_STRUCT *psFile );

    BOOLEAN (*bFileSystemVolumeInfo) (
         OSAL_FILE_STRUCT *psFile,
         const char *pcVolName );

    BOOLEAN (*bFileSystemDirectory) (
         OSAL_FILE_STRUCT *psFile,
         const char *pcPathName );

    BOOLEAN (*bFileSystemMakeDir) (
         const char *pcDirName );

    BOOLEAN (*bFileSystemRemoveDir) (
         const char *pcDirName );

    OSAL_RETURN_CODE_ENUM (*eFileSystemGetDirItems) (
                    OSAL_OBJECT_HDL *hDirItems,
                    const char *pcPathName,
                    OSAL_FS_FILE_QUALIFIER bFileQualifierFunc,
                    void *pvQualiferArg );

    OSAL_RETURN_CODE_ENUM (*eFileSystemReleaseDirItems) (
                    OSAL_OBJECT_HDL hDirItems );

    BOOLEAN (*bFileSystemTruncateFile) ( OSAL_FILE_STRUCT *psFile,
                                         size_t tNewSize );

    BOOLEAN (*bFileSystemGetFileSize) ( OSAL_FILE_STRUCT *psFile,
                                        size_t *ptFileSize );

    BOOLEAN (*bFileSystemGetFileAttributes) ( const char *pcPath,
                                              UN8 *pun8FileAttributes );

    BOOLEAN (*bFileSystemSetFileAttributes) ( const char *pcPath,
                                              UN8 un8FileAttributes );

    BOOLEAN (*bFileSystemSyncFile) ( FILE *psFile );

    BOOLEAN (*bFileSystemUnconditionalFileUnlink) ( const char *pcPath );

} OS_INTF_STRUCT;

  /****************************/
 /** GLOBAL UNION TEMPLATES **/
/****************************/

  /**********************/
 /** GLOBAL VARIABLES **/
/**********************/

// This is an external declaration reference to the API
// object. This is allocated at run-time

// Note: The Coding standards have been intentionally violated in order
// to preserve the interface as a 'true' API. APIs are by there nature
// global (as are all "C" functions) and thus the notation identifying
// it as globally-global is unnecessary.
extern const OSAL_API_STRUCT OSAL;

// This is an external declaration for all the FIXED related APIs
extern const OSAL_FIXED_INTERFACE_STRUCT OSAL_FIXED;

  /*******************/
 /** GLOBAL MACROS **/
/*******************/

// MACRO to run short functions which need to run often quickly
#ifdef __ICCARM__
    #define OSAL_OPTIMIZE __ramfunc
#else
    #define OSAL_OPTIMIZE
#endif /* __ICCARM__ */

  /***********************/
 /** GLOBAL PROTOTYPES **/
/***********************/

  /*****************************/
 /** GLOBAL INLINE FUNCTIONS **/
/*****************************/


#ifdef __cplusplus
}
#endif

#endif	// OSAL_H_
