/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_wsalerts_utils.cpp
* @brief       Implementation handling of the Util class to find distance information
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/
#include "fc_sxm_common.h"
#include "fc_sxm_tcl_wsalerts_utils.h"
#include "fc_sxm_wsalerts_types.h"
#include "fc_sxm_tcl_base_dsrl.h"
#include <limits>

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_WSALERTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_wsalerts_utils.cpp.trc.h"
#endif

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/
tBool fc_sxm_tclWsAlertsUtils::bIsPointInsidePolygon(
//!(I) The polygon that is checked
         const vector<midw_ext_fi_tcl_PositionWGS84>& vecShapeGeoPoints,

         //!(I) The position that is checked
         const midw_ext_fi_tcl_PositionWGS84& rDevLocation)
{
   const tU16 NumberOfPolygonPositions = static_cast<tU16>(vecShapeGeoPoints.size());
   tU16 j = static_cast<tU16>(NumberOfPolygonPositions - 1);

   tBool bOddNodes = FALSE;
   for (tU16 i = 0; i < NumberOfPolygonPositions; i++)
   {
      tS32 s32PiLong = vecShapeGeoPoints[i].Longitude.s32Value;
      tS32 s32PiLat = vecShapeGeoPoints[i].Latitude.s32Value;
      tS32 s32PjLong = vecShapeGeoPoints[j].Longitude.s32Value;
      tS32 s32PjLat = vecShapeGeoPoints[j].Latitude.s32Value;
      tS32 s32PLong = rDevLocation.Longitude.s32Value;
      tS32 s32PLat = rDevLocation.Latitude.s32Value;

      tS32 s32SubPjPi = s32PjLat - s32PiLat;

      if ((s32PiLat < s32PLat && s32PjLat >= s32PLat) || (s32PiLat >= s32PLat && s32PjLat < s32PLat))
      {
         tFloat f32Temp = 0.0F;
         if(s32SubPjPi == 0)
         {
            f32Temp = 0.0F;
         }
         else
         {
            f32Temp = static_cast<tFloat>((s32PjLong - s32PiLong) / s32SubPjPi);
         }

         if (s32PiLong + (s32PLat - s32PiLat) * static_cast<tS32>(f32Temp) < s32PLong)
         {
            bOddNodes = (bOddNodes == TRUE ? FALSE : TRUE);
         }
      }
      j = i;
   }
   return bOddNodes;
}

tS32 fc_sxm_tclWsAlertsUtils::s32GetDotProduct(
//!(I) The first position that defines the first vector.
         const midw_ext_fi_tcl_PositionWGS84& oPosition1,

         //!(I) The second position that defines the first vector.
         //!    It is also The first position that defines the second vector.
         const midw_ext_fi_tcl_PositionWGS84& oPosition2,

         //!(I) The second position that defines the second vector.
         const midw_ext_fi_tcl_PositionWGS84& oPosition3)
{
   tS32 P2Long_P1Long = oPosition2.Longitude.s32Value - oPosition1.Longitude.s32Value;
   tS32 P2Lat_P1Lat = oPosition2.Latitude.s32Value - oPosition1.Latitude.s32Value;

   tS32 P3Long_P2Long = oPosition3.Longitude.s32Value - oPosition2.Longitude.s32Value;
   tS32 P3Lat_P2Lat = oPosition3.Latitude.s32Value - oPosition2.Latitude.s32Value;

   return P2Long_P1Long * P3Long_P2Long + P2Lat_P1Lat * P3Lat_P2Lat;
}

tF32 fc_sxm_tclWsAlertsUtils::s32GetDistance(
//!(I) The first position
         const midw_ext_fi_tcl_PositionWGS84& oPosition1,

         //!(I) The second position
         const midw_ext_fi_tcl_PositionWGS84& oPosition2)
{
   tF32 f32RetVal = 0.0F;

   DISTANCE_OBJECT hNewRadius= fc_sxm_tclDsrlTypeAdapter::hCreateSMSRadius(0);

   LOCATION_OBJECT hWsAlertLocation = fc_sxm_tclDsrlTypeAdapter::
            hCreateSMSLocation(oPosition1.Latitude.s32Value,oPosition1.Longitude.s32Value, hNewRadius);
   // Device location object
   LOCATION_OBJECT hDevLocation = fc_sxm_tclDsrlTypeAdapter::
            hCreateSMSLocation(oPosition2.Latitude.s32Value,oPosition2.Longitude.s32Value, hNewRadius);

   //delete Radius Obj,not required anymore
   DISTANCE.vDestroy(hNewRadius);

   //check if Device location is Valid
   if((hDevLocation != LOCATION_INVALID_OBJECT))
   {
      //Check Alert Location Obj is valid
      if(hWsAlertLocation != LOCATION_INVALID_OBJECT)
      {
         //Get Distance from Alert location to Device location
         f32RetVal = fc_sxm_fGetLocationDistKm(hWsAlertLocation,hDevLocation);

         //We are done with Alert Location
         //Destroy it
         LOCATION.vDestroy(hWsAlertLocation);
      }
      //We are done with Device location
      //Destroy it
      LOCATION.vDestroy(hDevLocation);
   }

   //finally return the value
   return f32RetVal;
}


tF32 fc_sxm_tclWsAlertsUtils::s32GetDistanceToPolygon(
//!(I) The list of positions which define the polygon.
         const vector<midw_ext_fi_tcl_PositionWGS84>& vecShapeGeoPoints,

         //!(I) Position, from where the nearest distance will be searched
         const midw_ext_fi_tcl_PositionWGS84& rDevLocation)
{
   tF32 f32DistMin = numeric_limits<tF32>::max();

   //Get the number of positions which define the polygon
   const tU16 NumberOfPolygonPositions = static_cast<tU16>(vecShapeGeoPoints.size());
   if (NumberOfPolygonPositions >= 2)
   {
      tU16 j = static_cast<tU16>(NumberOfPolygonPositions - 1);

      //Loop over each edge of the polygon and find out the minimum distance between oPosition and the edge
      for (tU16 i = 0; i < NumberOfPolygonPositions; i++) {
         //try to get the nearest point to oPosition that is located on the line segement
         //between oPolygon[i] and oPolygon[j]
         midw_ext_fi_tcl_PositionWGS84 oNearestPosition;

         bGetNearestPoint(vecShapeGeoPoints[j], vecShapeGeoPoints[i], rDevLocation, true, oNearestPosition);
         j = i;

         //Get the distance between oNearestPosition and oPosition
         tF32 f32Distance = s32GetDistance(oNearestPosition, rDevLocation);
         f32DistMin = ENAVI_MIN(f32Distance, f32DistMin);
      }
   }
   return f32DistMin;
}

tS32 fc_sxm_tclWsAlertsUtils::s32RoundVal(tDouble number)
{
   if (number < 0.0) {
      number -= 0.5;
   } else {
      number += 0.5;
   }
   return ((tS32) number);
}

tS32 fc_sxm_tclWsAlertsUtils::tDoubleS32(tDouble number)
{
   tS32 s32Result = 0;
   const tDouble maxAllowedDouble = (tDouble) (OSAL_C_S32_MAX) - 0.5;
   if (number > maxAllowedDouble)
   {
      s32Result = OSAL_C_S32_MAX;
   }
   else
   {
      const tDouble minAllowedNumber = (tDouble) (-OSAL_C_S32_MAX) + 0.5;
      if (number < minAllowedNumber)
      {
         s32Result = -OSAL_C_S32_MAX;
      }
      else
      {
         s32Result = s32RoundVal(number);
      }
   }
   return (s32Result);
}

tBool fc_sxm_tclWsAlertsUtils::bGetNearestPoint(
//!(I) First position that defines the line
         const midw_ext_fi_tcl_PositionWGS84& oPosition1,

         //!(I) Second position that defines the line
         const midw_ext_fi_tcl_PositionWGS84& oPosition2,

         //!(I) Position, from which the closest position on the line is searched
         const midw_ext_fi_tcl_PositionWGS84& oPosition,

         //!(I) Bool defining the type of linesegment
         //!    If \c FALSE, the returned nearest point will be searched on an infinite line through
         //!    \c oPosition1 and \c oPosition2
         //!    Otherwise only the line segement between oPosition1 and oPosition2 is considered
         tBool bLineSegment,

         //!(O) The nearest position from oPositiion to the provided line
         midw_ext_fi_tcl_PositionWGS84& oNearestPosition)
{
   if (oPosition1.Latitude.s32Value == oPosition2.Latitude.s32Value
            && oPosition1.Longitude.s32Value == oPosition2.Longitude.s32Value)
   {
      if (bLineSegment == FALSE)
      {
         //Case 1: infinite line not defined => we cannot search a nearest point
         return FALSE;
      }

      //Case 2: The line segment defined, which is one point => the point is the nearest position to provided position
      oNearestPosition.Latitude.s32Value = oPosition1.Latitude.s32Value;
      oNearestPosition.Longitude.s32Value = oPosition1.Longitude.s32Value;

      return TRUE;
   }

   //If the both provided points oPosition1 and oPosition2 are not the same
   if (bLineSegment == TRUE) {
      /*Case 3: If only the line segment between two provided positions is considered
       //        and the nearest position is not on the segment, one of the end point
       //        of line segment must be the nearest position
       //
       //        e.g.                           or
       //                       oPosition1 *            oPosition *
       //                                 /                      /
       //                                /           oPosition1 *
       //                    oPosition2 *                        \
    //                                \                        \
    //                       oPosition *            oPosition2  *
       //
       //(Idea based on https://hi-dms.de.bosch.com/docushare/dsweb/Get/Document-314752/10041_WPD_Dist%20to%20Line.pdf)
       */

      if (s32GetDotProduct(oPosition1, oPosition2, oPosition) >= 0) {
         oNearestPosition = oPosition2;
         return TRUE;
      }

      if (s32GetDotProduct(oPosition2, oPosition1, oPosition) >= 0) {
         oNearestPosition = oPosition1;
         return TRUE;
      }
   }

   //Case4: If the nearest point is located on the searched line, the longitude and the latitude are calculated
   tS32 s32P1Long = oPosition1.Longitude.s32Value;
   tS32 s32P1Lat = oPosition1.Latitude.s32Value;
   tS32 s32P2Long = oPosition2.Longitude.s32Value;
   tS32 s32P2Lat = oPosition2.Latitude.s32Value;
   tS32 s32PLong = oPosition.Longitude.s32Value;
   tS32 s32PLat = oPosition.Latitude.s32Value;
   tS32 s32P2Long_P1Long = s32P2Long - s32P1Long;
   tS32 s32P2Lat_P1Lat = s32P2Lat - s32P1Lat;

   tS32 s32P=0;

/* To suppress prio2: lint warning */
   tS32 s32p1 =  ((s32PLong - s32P1Long) * s32P2Long_P1Long + (s32PLat - s32P1Lat) * s32P2Lat_P1Lat);
   tS32 s32p2 =  (s32P2Long_P1Long * s32P2Long_P1Long + s32P2Lat_P1Lat * s32P2Lat_P1Lat);

   if((s32p2 == 0) || (s32p1 == 0))
   {
      s32P = 0;
      return FALSE;
   }
   else
   {
      s32P = s32p1/s32p2;
   }

   tFloat u = static_cast<tFloat> (s32P);
   tFloat fLong = static_cast<tFloat>(s32P1Long) + u * static_cast<tFloat>(s32P2Long_P1Long);
   tFloat fLat = static_cast<tFloat>(s32P1Lat) + u * static_cast<tFloat>(s32P2Lat_P1Lat);
/* To suppress prio2: lint warning */

   oNearestPosition.Longitude.s32Value = tDoubleS32(tDouble(fLong));
   oNearestPosition.Latitude.s32Value = tDoubleS32(tDouble(fLat));

   return TRUE;
}
