/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_wsalerts_list.h
* @brief       Declaration of handling Weather and Security Alert information to HMI
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            DEFINES                              =======
=============================================================================*/
#ifndef _FC_SXM_TCL_WSALERTS_LIST_H_
#define _FC_SXM_TCL_WSALERTS_LIST_H_

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/
#include "fc_sxm_tcl_data_list.hpp"
#include "fc_sxm_wsalerts_types.h"
#include "fc_sxm_wsalerts_fi.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/

/*=============================================================================
=======                     FORWARD DECLARATIONS                        =======
=============================================================================*/
class midw_ext_fi_tcl_WSAlertMsg;
class midw_ext_fi_tcl_PositionWGS84;


/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/
/*
  sort-methods supported
*/
typedef enum
{
   fc_sxm_enWsAlertListSortMethod_NONE,
   fc_sxm_enWsAlertListSortMethod_PRIORITY,
   fc_sxm_enWsAlertListSortMethod_DISTANCE

} fc_sxm_tenWsAlertsListSortMethod;

struct fc_sxm_trWsAlertGeoPoint{
    tFloat fLat;
    tFloat fLon;

    fc_sxm_trWsAlertGeoPoint() :
       fLat(0.0f),
       fLon(0.0f)
    {}
};

struct fc_sxm_trWsAlertLocation{
   string oDescription;
   string oLocationExtType;
   vector <fc_sxm_trWsAlertGeoPoint> vecShapeGeoPoints;

   fc_sxm_trWsAlertLocation() :
      oDescription(""),
      oLocationExtType("")
   {
      vecShapeGeoPoints.clear();
   }
};

struct fc_sxm_trWsAlertType{
   tU32 u32AlertTypeId;
   string oAlertExtRefCode;

   fc_sxm_trWsAlertType() :
      u32AlertTypeId(midw_ext_fi_tcl_e8_WSAlertTypes::FI_EN_INVALID_ALERT_TYPE),
      oAlertExtRefCode("")
   {}
};

// structure with data of a list-entry for HMI
struct fc_sxm_trWsAlertsListEntry:public fc_sxm_trBaseListEntry {
public:

    tU32 u32AlertMSGID;
    string oMsgText;
    TIME_T oStartTime;
    TIME_T oEndTime;
    TIME_T oTimeStamp;
    tF32   f32Distance;
    string oLocationName;
    SMS_LANGUAGE_ENUM enMessageLang;
    tBool bIsMarineZone;
    tU16  u16MsgPriority;

    vector<fc_sxm_trWsAlertType>  vecAlertTypes;
    vector<fc_sxm_trWsAlertLocation>  vecWSAlertLocations;

    fc_sxm_trWsAlertsListEntry() :
       u32AlertMSGID(0),
       oMsgText(""),
       oStartTime(0),
       oEndTime(0),
       oTimeStamp(0),
       f32Distance(0.0),
       oLocationName(""),
       enMessageLang(SMS_INVALID_LANGUAGE),
       bIsMarineZone(FALSE),
       u16MsgPriority(0)
    {
       vecAlertTypes.clear();
       vecWSAlertLocations.clear();
    }

};

/*
  sort-configuration consisting of filter-method and additional parameters
  e.g. for which brand shall we filter
*/
struct fc_sxm_trWsAlertsListFilterCfg {
   fc_sxm_tenWsAlertFilterMethod enWsAlertsMsgObjectType;

   fc_sxm_trWsAlertsListFilterCfg():
      enWsAlertsMsgObjectType(fc_sxm_enWsAlertsFilterMethod_NONE)
   {}
    /* to check if filter has changed, configuration has to provide operator!=
       see sort-configuration
    */
    tBool operator!=(fc_sxm_trWsAlertsListFilterCfg const &b) const {
        SXM_RET_1_IF_NEQ(enWsAlertsMsgObjectType);
        return FALSE;
    }
};


/* sort-configuration consisting of sort-method and additional parameters */
struct fc_sxm_trWsAlertsListSortCfg {
    fc_sxm_tenWsAlertsListSortMethod enSortMethod;
    /* Check if configuration has changed.
       to check if sorting has chanded, configuration has to provice operator!=
       attention:
       operator is used in the way that it only returns TRUE if
       the changes are relevant for the current configuration.
       e.g. if sorting by distance is not enabled, changing of the 
       location has no effect.
    */

    fc_sxm_trWsAlertsListSortCfg() :
       enSortMethod(fc_sxm_enWsAlertListSortMethod_NONE)
    {}

    tBool operator!=(fc_sxm_trWsAlertsListSortCfg const &b) const {
    	(tVoid)b;
        return FALSE;
    }
};

/*
  struct containing all our configuration.
  Must be dericed from fc_sxm_trBaseListConfig
*/

struct fc_sxm_trWsAlertsListConfig:public fc_sxm_trBaseListConfig {
	typedef fc_sxm_trWsAlertsListEntry trListEntry;
    fc_sxm_trWsAlertsListSortCfg rSortCfg;
    fc_sxm_trWsAlertsListFilterCfg rFilterCfg;

    fc_sxm_trWsAlertsListConfig()
    {}
    // see sort-config
    tBool operator!=(fc_sxm_trWsAlertsListConfig const &b) const {
        SXM_RET_1_IF_NEQ(rSortCfg);
        SXM_RET_1_IF_NEQ(rFilterCfg);
        return FALSE;
    }
    virtual tVoid vPrintSpecific() const;
};

/*=============================================================================
=======                              CLASSES                            =======
=============================================================================*/

/*
  The WsAlerts-list, derive from template fc_sxm_tclDataListTyped.
  Template-parameters are our list-class and our configuration.
*/
class fc_sxm_trWsAlertsList:
   public fc_sxm_tclDataListTyped<fc_sxm_trWsAlertsList,fc_sxm_trWsAlertsListConfig>{
 public:
    /*
      constructor:
      Simply pass parameters to base-class
    */
   fc_sxm_trWsAlertsList():
      fc_sxm_tclDataListTyped<fc_sxm_trWsAlertsList, fc_sxm_trWsAlertsListConfig>(0, fc_sxm_tenListType_Invalid, OSAL_NULL, TR_CLASS_FC_SXM_WSALERTS_APP)
   {}
	fc_sxm_trWsAlertsList(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp):
        fc_sxm_tclDataListTyped<fc_sxm_trWsAlertsList, fc_sxm_trWsAlertsListConfig>(u32ListId, enListType, poDataApp, TR_CLASS_FC_SXM_WSALERTS_APP)
    {}

        /* compare-fn that shall compare according to fc_sxm_trWsAlertsSortCfg
           Attention:
           Strict ordering has to be proviced: *prLeft<*prRight || *prRights<*prLeft
           (not strict weak ordering as for stl-containers)
           This is accomplished by comparing a unique id as fallback in case of *prLeft==*prRight
        */
    virtual bool bCompareItem(fc_sxm_trWsAlertsListEntry const *prLeft, fc_sxm_trWsAlertsListEntry const *prRight) const;
    tBool bCompareWsAlertPriority(fc_sxm_trWsAlertsListEntry const *prLeft, fc_sxm_trWsAlertsListEntry const *prRight) const;
    // compare-fn that shall filter according to fc_sxm_trWsAlertsListFilterCfg
    virtual bool bFilterPassed(fc_sxm_trWsAlertsListEntry const &rItem)  const;
    tVoid vCopyAlertInfoToFi(fc_sxm_trWsAlertsListEntry const &roEntry,midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const;
    tVoid vFillWSAlertInfo(fc_sxm_trWsAlertsListEntry const &roEntry,
             midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const;
    tVoid vFillWSAlertInfoMap(fc_sxm_trWsAlertsListEntry const &roEntry,
             midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const;
    tVoid vTraceFiEntry(midw_ext_fi_tcl_WSAlertMsg const &oFiEntry) const;

    tVoid vEmitCombinedList(fc_sxm_trAdressing const &rAdressing);
    tVoid vEmitMapList(fc_sxm_trAdressing const &rAdressing);
    tVoid vEmitStandardList(fc_sxm_trAdressing const &rAdressing);

 protected:
    /*
      will be called by base-class when data are ready and the list shall be sent to hmi
     */
    virtual tVoid vEmit(fc_sxm_trAdressing const &rAdressing);
    tVoid vFillListEntry(fc_sxm_trWsAlertsListEntry const &roEntry, midw_ext_fi_tcl_WSAlertMsg &oFiEntry) const;
    tBool bIsAlertEnabled(fc_sxm_trWsAlertsListEntry const &roEntry) const;
    tF32 f32GetDistanceInfo(const vector<midw_ext_fi_tcl_PositionWGS84>& vecShapeGeoPoints,
             const midw_ext_fi_tcl_PositionWGS84& rDevLocation) const;
    tBool bIsAlertSupported(fc_sxm_trWsAlertsListEntry const &roEntry) const;
    virtual tVoid vAddEntry(const fc_sxm_trWsAlertsListEntry& rListEntry);
    virtual tVoid vDeleteEntry(const fc_sxm_trWsAlertsListEntry& rListEntry);
#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
    virtual tVoid vStartTimer();
    virtual tBool bIsListModeTimerRunning();
    virtual tVoid vStopTimer();
#endif

 private:
     static set<_trListEntry*, trCompareListEntry> _setCombinedSortedEntries;


};
#endif
/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/
