/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_wsalerts_app.cpp
* @brief       Implementation of handling HMI requests and functionalities of the
*              Weather and Security Alerts service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
=======                            INCLUDES                             =======
=============================================================================*/
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_wsalerts.h"
#include "fc_sxm_tcl_wsalerts_app.h"
#include "fc_sxm_tcl_wsalerts_dsrl.h"

/*=============================================================================
=======               CONSTANTS & MACROS FOR GENERAL PURPOSE            =======
=============================================================================*/
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_WSALERTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_wsalerts_app.cpp.trc.h"
#endif

/*=============================================================================
=======                             TYPES                               =======
=============================================================================*/
typedef enum {
    fc_sxm_enTtfisCmdsWsAlerts_MStartGetWSAlertMsgList=1,
    fc_sxm_enTtfisCmdWsAlerts_MStartSetWSAlertsConfig,
    fc_sxm_enTtfisCmdsWsAlerts_MStartGetWSAlertMsgDetails,
    fc_sxm_enTtfisCmdsWsAlerts_MStartSetListsConfig,
    fc_sxm_enTtfisCmdsWsAlerts_SetListMode,
    fc_sxm_enTtfisCmdsWsAlerts_MStartGetWeatherAlertData,
    fc_sxm_enTtfisCmdsWsAlerts_MStartAnnounceWSALerts,
    fc_sxm_enTtfisCmdsWsAlerts_MStartSetWSAlertsProjectConfig,
    fc_sxm_enTtfisCmdsWsAlerts_MStartValidateWSAlertMessage
} fc_sxm_tenTtfisCmdsWsAlerts;

/*=============================================================================
=======                              METHODS                            =======
=============================================================================*/

/****************************************************************************
 *FUNCTION:     fc_sxm_tclWSAlertsApp
 *DESCRIPTION:  Constructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ****************************************************************************/
/*
   peha: constructor: give it a name and provide the used sxmServiceId (not the ccaServiceId!!)
*/
fc_sxm_tclWsAlertsApp::fc_sxm_tclWsAlertsApp():
    fc_sxm_tclDataApp("sxmWeatherSecurityAlertsApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_WEATHER_ALERTS, "ws_alerts", TR_CLASS_FC_SXM_WSALERTS_APP),
       _hWSAlertsService (WS_ALERTS_SERVICE_INVALID_OBJECT),
       _poNearByDsrl (OSAL_NULL),
       _poNearDestDsrl (OSAL_NULL),
       _poNearByList (OSAL_NULL),
       _poNearDestList (OSAL_NULL),
       _bSendUpdate (FALSE),
       _u8ListReqType(OSAL_NULL),
       _u32WSAlertMsgID(0),
       _u8RetryCounter(0),
       _u32UpdateCounter(0)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp constructor"));
    _mapAlertConfig.clear();
    _enTestListMode=fc_sxm_enListMode_Invalid;
    enAudioSourceActivity = FC_SXM_EN_AUDIO_SOURCE_OFF;

}

/*********************************************************************
 *FUNCTION:     fc_sxm_tclWSAlertsApp
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclWsAlertsApp::~fc_sxm_tclWsAlertsApp()
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp destructor Entered"));

   /* reset pointers, cleanup will be done by framework */
   _poNearByDsrl = OSAL_NULL;
   _poNearDestDsrl = OSAL_NULL;
   _poNearByList = OSAL_NULL;
   _poNearDestList = OSAL_NULL;
   
   _hWSAlertsService = WS_ALERTS_SERVICE_INVALID_OBJECT;
   _u32UpdateCounter = 0;
   _u8RetryCounter = 0;
   _mapAlertConfig.clear();

   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp destructor exited"));
}

/*********************************************************************
 *FUNCTION:     vConfigureWSAlerts
 *DESCRIPTION:  called from vProcess of SetWSAlertProjectConfig method
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vConfigureWSAlerts(fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig  const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts Entered"));
   midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsProjectConfigMethodStart const &oMStart=prMsg->oFiMsg;
   
   for(tU32 u32Index=0; u32Index < oMStart.AlertConfig.size(); ++u32Index)
   {
      fc_sxm_trWSAlertConfig rWSAlertConfig;
      fc_sxm_trRGBAcolor rWSAlertRGBAcolor;

      rWSAlertConfig.u32AlertTypeId = static_cast<tU32>(oMStart.AlertConfig[u32Index].AlertTypeID.enType);

      tString chExtRefCode = oMStart.AlertConfig[u32Index].AlertExtRefCode.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
      rWSAlertConfig.oExtRefCode = (string) chExtRefCode;
      rWSAlertConfig.u32AlertNameID = oMStart.AlertConfig[u32Index].AlertNameID;
      rWSAlertConfig.u32AlertGroupId = oMStart.AlertConfig[u32Index].AlertGroupId;
      rWSAlertConfig.u32AlertPrio = oMStart.AlertConfig[u32Index].AlertPrio;
      rWSAlertConfig.bIsEnabledInHMI = oMStart.AlertConfig[u32Index].IsEnabledInHMI;
      rWSAlertConfig.bIsEnabledInMap = oMStart.AlertConfig[u32Index].IsEnabledInMap;

      rWSAlertRGBAcolor.u8Red = oMStart.AlertConfig[u32Index].AlertColor.Red;
      rWSAlertRGBAcolor.u8Green = oMStart.AlertConfig[u32Index].AlertColor.Green;
      rWSAlertRGBAcolor.u8Blue = oMStart.AlertConfig[u32Index].AlertColor.Blue;
      rWSAlertRGBAcolor.u8Alpha = oMStart.AlertConfig[u32Index].AlertColor.Alpha;

      rWSAlertConfig.trRGBAcolors = rWSAlertRGBAcolor;

      _mapAlertConfig.insert(make_pair(rWSAlertConfig.oExtRefCode, rWSAlertConfig));

      // Debug traces
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts u32AlertTypeId :%d",rWSAlertConfig.u32AlertTypeId));
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts oExtRefCode :%s",rWSAlertConfig.oExtRefCode.c_str()));
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts IsEnabledInHMI :%d",rWSAlertConfig.bIsEnabledInHMI));
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts u32AlertGroupId :%d",rWSAlertConfig.u32AlertGroupId));
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts u32Priority :%d",rWSAlertConfig.u32AlertPrio));
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts u8Red :%d u8Alpha:%d",rWSAlertRGBAcolor.u8Red,rWSAlertRGBAcolor.u8Alpha));

      OSAL_DELETE[] chExtRefCode;
   }
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vConfigureWSAlerts Exit"));
}

/*********************************************************************
 *FUNCTION:     vInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vInitialize"));

    //Initialize the class members
   _hWSAlertsService = WS_ALERTS_SERVICE_INVALID_OBJECT;
   _poNearByDsrl = OSAL_NULL;
   _poNearDestDsrl = OSAL_NULL;
   _poNearByList = OSAL_NULL;
   _poNearDestList = OSAL_NULL;

   fc_sxm_trWsAlertsList *poAlertMapList = OSAL_NULL;
   fc_sxm_tclWsAlertsDSRL *poAlertMapDsrl = OSAL_NULL;

    //Create the worker thread
    /* call vInitialize of base-class to start the worker thread */
    fc_sxm_tclBaseApp::vInitialize();

    fc_sxm_trWsAlertsDSRLCfg rDsrlCfg;
    rDsrlCfg.enSortMethod = fc_sxm_enWsAlertSortMethod_PRIORITY;
    rDsrlCfg.rFilterCfg.enWsAlertsMsgObjectType= (fc_sxm_tenWsAlertFilterMethod)fc_sxm_enWsAlertsFilterMethod_NONE;

    /*DSRL id - 1*/
    /* Get Capacity & Radius defined for WsAlerts Nearby type */
    rDsrlCfg.u32Capacity= FC_SXM_WSALERTS_DEVICE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_WSALERTS_DEVICE_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: call vCreateDsrlAndList: NearBy Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius));

    vCreateDsrlAndList(fc_sxm_enDSRLType_Device, fc_sxm_tenListType_NearBy, &_poNearByDsrl, &_poNearByList, rDsrlCfg);

    /*DSRL id - 2*/
    /* Get Capacity & Radius defined for WsAlerts NearDest type */
    rDsrlCfg.u32Capacity = FC_SXM_WSALERTS_DESTINATION_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_WSALERTS_DESTINATION_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: call vCreateDsrlAndList: NearDest Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Dest, fc_sxm_tenListType_NearDest, &_poNearDestDsrl, &_poNearDestList, rDsrlCfg);

    /*DSRL id - 3*/
    /* Get Capacity & Radius defined for WsAlerts Alert Map type */
    float fLat = FC_SXM_WSALERTS_MAP_LATITUDE;
    float fLon = FC_SXM_WSALERTS_MAP_LONGITUDE;
    tS32 u32Lat = (tS32)(fLat*0xFFFF);
    tS32 u32Lon = (tS32)(fLon*0xFFFF);

    rDsrlCfg.rLocation = fc_sxm_trDsrlLocation(u32Lat, u32Lon);
    rDsrlCfg.u32Capacity = FC_SXM_WSALERTS_STANDARD_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_WSALERTS_STANDARD_DSRL_RADIUS;


    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: call vCreateDsrlAndList: Another Location Capacity =%d ,Radius = %d",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Standard, fc_sxm_tenListType_AnotherLocation, &poAlertMapDsrl, &poAlertMapList, rDsrlCfg);


    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:NearBy=%08x,NearDest=%08x,AlertMap=%08x",
                    _poNearByList,
                    _poNearDestList,
                    poAlertMapList));
}

/*********************************************************************
 *FUNCTION:     vDeInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vDeInitialize"));
    /* call vDeInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vDeInitialize();


    /* Deinit of all other stuff starts from here */
   _poNearByDsrl=OSAL_NULL;
   _poNearDestDsrl=OSAL_NULL;

   _poNearByList=OSAL_NULL;
   _poNearDestList=OSAL_NULL;

}


/*********************************************************************
 *FUNCTION:     vDispatchMsgFromQ
 *DESCRIPTION:  this method is called by the thread-function.
   For each message that shall be handled put and entry here
 *PARAMETER:    fc_sxm_tclMessage
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vDispatchMsgFromQ: SID=%x action=%d",
    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()),poThreadMsg->u16GetActionOnly()));
    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action)
    {
        // use macro SXM_MSGQ_DISPATCH  for each message specific for Weather alerts
        //Previously used, For getting dataservice ready
        SXM_MSGQ_DISPATCH(fc_sxm_trMsgDataSmsEvtServiceUp);

        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsAudioSource,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultAbortPrompt,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultPausePrompt,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientMResultContinuePrompt,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTTSClientError,fc_sxm_tclWsAlertsApp::instance());
        /* method-start messages from cca-service */
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartGetWSAlertsMsgList,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartSetListsConfig,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartListModeSet,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig,fc_sxm_tclWsAlertsApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgWsAlertsMStartValidateWSAlertMessage,fc_sxm_tclWsAlertsApp::instance());
        default:
        ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",u32Action));
        break;
    }
}

/*********************************************************************
 *FUNCTION:     vCreateDsrlAndList
 *DESCRIPTION:  Create DSRL and List
 *PARAMETER:    fc_sxm_tenDSRLType, fc_sxm_tenListType, fc_sxm_tclWsAlertsDSRL
 *              fc_sxm_trWsAlertsDSRLCfg
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vCreateDsrlAndList(fc_sxm_tenDSRLType enDsrlType,
                                                fc_sxm_tenListType enListType,
                                                fc_sxm_tclWsAlertsDSRL **ppoDsrl,
                                                fc_sxm_trWsAlertsList **ppoList,
                                                fc_sxm_trWsAlertsDSRLCfg rDsrlCfg)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vCreateDsrlAndList: enDsrlType=%d enListType=%d",
                    ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType),
                    ETG_CENUM(fc_sxm_tenListType, enListType)
                    ));
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::construct dsrl"));
    /* create DSRL, dsrl will remain inactive while not configured*/
    fc_sxm_tclWsAlertsDSRL *poDsrl;
    poDsrl=OSAL_NEW fc_sxm_tclWsAlertsDSRL(enDsrlType);
    if (OSAL_NULL == poDsrl)
    {
       ETG_TRACE_ERR(("RETURNING from METHOD as MEMORY IS NOT ALLOTED"));
       return ;
    }

    /* register the dsrl */
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::poRegisterDsrl"));
    *ppoDsrl=poDsrl;
    poRegisterDsrl(poDsrl);

    /* create list*/
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::fc_sxm_trWsAlertsList Create"));

    fc_sxm_trWsAlertsList *poWsAlertsList=OSAL_NEW fc_sxm_trWsAlertsList((tU32)enListType, enListType, this);

    if (OSAL_NULL == poWsAlertsList)
    {
       ETG_TRACE_ERR(("RETURNING from METHOD as MEMORY IS NOT ALLOTED"));
       return;
    }

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::poWsAlertsList setDsrl"));
    poWsAlertsList->vSetDsrl(poDsrl);

    /* configure the list */
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::fc_sxm_trWsAlertsList setCfg"));
    fc_sxm_trWsAlertsListConfig rListCfg;
    rListCfg.rSortCfg.enSortMethod = (fc_sxm_tenWsAlertsListSortMethod)rDsrlCfg.enSortMethod;
    rListCfg.rFilterCfg.enWsAlertsMsgObjectType=(fc_sxm_tenWsAlertFilterMethod)fc_sxm_enWsAlertsFilterMethod_NONE;
    poWsAlertsList->vSetCfg(rListCfg);

    *ppoList = poWsAlertsList;

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::poDsrl  vSetNextConfig"));
    vPatchDsrlConfig(enDsrlType, rDsrlCfg);
    poDsrl->vSetNextConfig(rDsrlCfg);

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vCreateDsrlAndList: DONE"));
}


/*********************************************************************
 *FUNCTION:     vStartSmsService
 *DESCRIPTION:  handle command from sxmAppManager to start service (Mandatory)
 *PARAMETER:
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vStartSmsService()
{
   //tString atext;

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vStartSmsService"));

    /*Check if the WsAlert service is already started*/
    if (_hWSAlertsService != WS_ALERTS_SERVICE_INVALID_OBJECT)
    {
        ETG_TRACE_ERR(("WsAlertsApp service already started!"));
        return;
    }

    /*Start the Weather Security Alert service */
    _hWSAlertsService = WS_ALERTS.hStart(FC_SXM_SRH_DRIVER_NAME,
                                  DATASERVICE_EVENT_ALL,
                                  cb_vDataServiceEventCallback,
                                  (tVoid*)this, 
                                  DATASERVICE_OPTION_NONE);

   if (_hWSAlertsService == WS_ALERTS_SERVICE_INVALID_OBJECT)
   {
       ETG_TRACE_ERR(("Could not start WsAlert service"));
   }
   else
   {
       ETG_TRACE_USR4(("Start WsAlert service success"));
   }
}

/* all messages are handled via vProcess(tMsgType const *prMsg) */

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd (Mandatory)
 *PARAMETER:    fc_sxm_trMsgCmdAppTtfisCmd
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
   fc_sxm_tenTtfisCmdsWsAlerts enMsgCode = (fc_sxm_tenTtfisCmdsWsAlerts) prMsg->u8MsgCode;
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) code=%d",
            ETG_CENUM(fc_sxm_tenTtfisCmdsWsAlerts, enMsgCode)));
   tU8 const *au8Data = prMsg->au8Data;

   switch (enMsgCode)
   {
      case fc_sxm_enTtfisCmdsWsAlerts_MStartGetWSAlertMsgList:
      {
         /* This case handled to Get Weather alerts info the selected list id*/
         midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgListMethodStart oMStart;
         oMStart.ListId = au8Data[0];

         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartGetWSAlertMsgList listId=%d", oMStart.ListId));
         if ((tU32) fc_sxm_tenListType_Invalid != oMStart.ListId)
         {
            fc_sxm_vSendSelfMessage(oMStart);
         }
      }
      break;

      case fc_sxm_enTtfisCmdWsAlerts_MStartSetWSAlertsConfig:
      {
         midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsConfigMethodStart oMStart;

         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartSetWSAlertsConfig AlertID=%d",
                  (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) au8Data[0]));
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartSetWSAlertsConfig Action=%d",
                  (midw_ext_fi_tcl_e8_WSAlertActionType::tenType) au8Data[2]));

         /* Copy AlertTypeIds and Status (Enable/Disable/UnSupported) from TTFis  */
         midw_ext_fi_tcl_e8_WSAlertTypes enAlerTypes;

         enAlerTypes.enType = (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) au8Data[0];
         oMStart.AlertTypeIDs.push_back(enAlerTypes);

         enAlerTypes.enType = (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) au8Data[1];
         oMStart.AlertTypeIDs.push_back(enAlerTypes);

         oMStart.WSAlertAction.enType = (midw_ext_fi_tcl_e8_WSAlertActionType::tenType) au8Data[2];

         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartGetWSAlertMsgDetails:
      {
         /* This case handled to Get Weather alerts info the selected list id*/
         midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgDetailsMethodStart oMStart;
         oMStart.MessageID = (tS32)SXM_GET_U32(&au8Data[0]);
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartGetWSAlertMsgDetails MessageID=%d", oMStart.MessageID));
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartSetListsConfig:
      {
         /* This case handled to do Method start for the set list capacity */
         midw_ext_sxm_wsalertsfi_tclMsgSetListsConfigMethodStart oMStart;
         oMStart.Config.Capacity = prMsg->au8Data[0];
         ETG_TRACE_USR4(
                  ("fc_sxm_tclWsAlertsApp::fc_sxm_enTtfisCmdsWsAlerts_MStartSetListsConfig Config.Capacity=%d", oMStart.Config.Capacity));
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_SetListMode:
      {
         midw_ext_sxm_wsalertsfi_tclMsgSxmListModeSet oPSet;
         /* Copy ListId param from TTFis*/
         oPSet.ListMode.ListID = (tU32) au8Data[0];
         oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType) au8Data[1];
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd SetListId ListID=%d", oPSet.ListMode.ListID));
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd SetListMode ListMode=%d", oPSet.ListMode.Mode.enType));
         fc_sxm_vSendSelfMessage(oPSet);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartGetWeatherAlertData:
      {
         /* This case handled to do Method start for the set list capacity */
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartGetWeatherAlertData Entered"));
         midw_ext_sxm_wsalertsfi_tclMsgGetWeatherAlertDataMethodStart oMStart;
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartAnnounceWSALerts:
      {
         midw_ext_sxm_wsalertsfi_tclMsgAnnounceWSAlertMethodStart oMStart;
         oMStart.MessageID = (tU32) prMsg->au8Data[0];
         oMStart.Action.enType = (midw_ext_fi_tcl_e8_WSAlertAnnounceAction::tenType) prMsg->au8Data[1];
         vConvertTTFisString2FIString((tCString) &prMsg->au8Data[2], oMStart.Text);
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartSetWSAlertsProjectConfig:
      {
         midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsProjectConfigMethodStart oMStart;
         midw_ext_fi_tcl_WSAlertConfig oWSAlertConfig;

         // Get the alert params from HMI
         oWSAlertConfig.AlertTypeID.enType = (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) au8Data[0];
         oWSAlertConfig.AlertNameID = au8Data[1];
         oWSAlertConfig.AlertGroupId = au8Data[2];
         oWSAlertConfig.AlertPrio = au8Data[3];
         oWSAlertConfig.IsEnabledInHMI = au8Data[4];
         oWSAlertConfig.IsEnabledInMap = au8Data[5];
         oWSAlertConfig.AlertColor.Red = au8Data[6];
         oWSAlertConfig.AlertColor.Green = au8Data[7];
         oWSAlertConfig.AlertColor.Blue = au8Data[8];
         oWSAlertConfig.AlertColor.Alpha = au8Data[9];
         vConvertTTFisString2FIString((tCString)&prMsg->au8Data[10], oWSAlertConfig.AlertExtRefCode);

         oMStart.AlertConfig.push_back(oWSAlertConfig);

        // Debugging Traces
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartSetWSAlertsProjectConfig AlertTypeID=%d",
                  (midw_ext_fi_tcl_e8_WSAlertTypes::tenType) oWSAlertConfig.AlertTypeID.enType));
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartSetWSAlertsProjectConfig AlertPrio=%d IsEnabledInHMI=%d",
                         oWSAlertConfig.AlertPrio,oWSAlertConfig.IsEnabledInHMI));

         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      case fc_sxm_enTtfisCmdsWsAlerts_MStartValidateWSAlertMessage:
      {
         midw_ext_sxm_wsalertsfi_tclMsgValidateWSAlertMessageMethodStart oMStart;
         oMStart.MessageID = (tS32)SXM_GET_U32(&au8Data[0]);
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::ttfis-cmd MStartValidateWSAlertMessage ID:%d",oMStart.MessageID));
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;

      default:
         ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                  ETG_CENUM(fc_sxm_tenTtfisCmdsWsAlerts, enMsgCode)));
      break;
   }
}

/*********************************************************************
 *
 *FUNCTION:    vProcess(fc_sxm_trMsgListUpdateWsAlertTimer const *prMsg)
 *
 *DESCRIPTION:  message containing List update timer for WsAlert
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
tVoid fc_sxm_tclWsAlertsApp::vProcessTimer(fc_sxm_trMsgListUpdateWsAlertTimer const *prMsg) {
   (tVoid)prMsg; /* To Remove Lint warning*/
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess"
                    "(fc_sxm_trMsgListUpdateTimer)"));
   vStopListTimer();
   /* find the correct list by ListID */
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess _u32ListID %d",_u32ListID ));
   fc_sxm_trWsAlertsList *poList=fc_sxm_trWsAlertsList::poGetTypedListById(_u32ListID);
   if(OSAL_NULL!=poList)  {
      poList->vOnListTimerExpire();
   }
}

/*********************************************************************
 *
 *FUNCTION:    vStartListTimer
 *
 *DESCRIPTION:  start list timer for WsAlert app
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vStartListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vStartListTimer()"));

    _mListUpdateTimer.vStart(FC_SXM_LIST_DATA_UPDATE_PERIOD);
}
/*********************************************************************
 *
 *FUNCTION:    vStopListTimer
 *
 *DESCRIPTION: stop the list timer for WsAlert app
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vStopListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vStopListTimer()"));

    _mListUpdateTimer.vStop();
}
/*********************************************************************
 *
 *FUNCTION:    bIsListUpdateTimerRunning
 *
 *DESCRIPTION:  check the timer is running or not
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bIsListUpdateTimerRunning(tVoid) const {
    return(_mListUpdateTimer.bIsRunning());
}
#endif

/*********************************************************************
 *
 *FUNCTION:    vPrepareCustomList
 *
 *DESCRIPTION: Method to send FI message for NearBy and Near Destination list
 *
 *PARAMETER:    1) Pointer to GetWsAlertsInfoList prMsg
 *
 *RETURNVALUE:  void
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vPrepareCustomList(fc_sxm_trMsgWsAlertsMStartGetWSAlertsMsgList const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareCustomList START"));

    tU32 u32NearbyEnstate = fc_sxm_trWsAlertsList::poGetTypedListById((tU32)fc_sxm_tenListType_NearBy)->rGetStatusInfo().enListState;
    tU32 u32NearDestEnstate = fc_sxm_trWsAlertsList::poGetTypedListById((tU32)fc_sxm_tenListType_NearDest)->rGetStatusInfo().enListState;

    fc_sxm_trWsAlertsListConfig rListCfg;
    rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenWsAlertsListSortMethod)fc_sxm_enWsAlertListSortMethod_PRIORITY;
    rListCfg.rFilterCfg.enWsAlertsMsgObjectType=(fc_sxm_tenWsAlertFilterMethod)fc_sxm_enWsAlertsFilterMethod_NONE;

    tU32 u32EnListStateReady = static_cast<tU32>(fc_sxm_enListState_Ready);
    tU32 u32ListTypeNearBy = static_cast<tU32>(fc_sxm_tenListType_NearBy);
    tU32 u32ListTypeNearDest = static_cast<tU32>(fc_sxm_tenListType_NearDest);

    // Use case 1: If Nearby and Near destination data is ready
   if ((u32NearbyEnstate == u32EnListStateReady) && (u32NearDestEnstate == u32EnListStateReady))
   {
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareCustomList ----BOTH ARE READY "));
      //Set the configuration
      fc_sxm_trWsAlertsList::vSetCfg(u32ListTypeNearBy, rListCfg);
      //finally,send list to HMI
      vSendList(u32ListTypeNearBy, prMsg->rAdressing);
      //then,set the flag to send the data to HMI
      _bSendUpdate = true;
      //Set the configuration
      fc_sxm_trWsAlertsList::vSetCfg(u32ListTypeNearDest, rListCfg);
      //finally,send list to HMI
      vSendList(u32ListTypeNearDest, prMsg->rAdressing);

   }
   // Use case 2: If only NearBy data is ready
   else if (u32NearbyEnstate == u32EnListStateReady)
   {
      //OK,check if Near Destination is not ready
      if (u32NearDestEnstate != u32EnListStateReady)
      {
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareCustomList ----Near By IS READY "));
         //then,set the flag to send the data to HMI
         _bSendUpdate = true;
         //Set the configuration
         fc_sxm_trWsAlertsList::vSetCfg(u32ListTypeNearBy, rListCfg);
         //finally,send list to HMI
         vSendList(u32ListTypeNearBy, prMsg->rAdressing);
      }
   }
   // Use case 3: If only Near destination data is ready
   else if (u32NearDestEnstate == u32EnListStateReady)
   {
      if (u32NearbyEnstate != u32EnListStateReady)
      {
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareCustomList ----Near Dest IS READY "));
         //then,set the flag to send the data to HMI
         _bSendUpdate = true;
         //Set the configuration
         fc_sxm_trWsAlertsList::vSetCfg(u32ListTypeNearDest, rListCfg);
         //finally,send list to HMI
         vSendList(u32ListTypeNearDest, prMsg->rAdressing);
      }
   }
   else
   {
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareCustomList ----BOTH ARE NOT READY "));
   }
}

/*********************************************************************
 *
 *FUNCTION:    vPrepareList
 *
 *DESCRIPTION: Method to send FI message for NearBy and Near Destination list
 *
 *PARAMETER:    1) Pointer to GetWsAlertsInfoList message
 *              2) u32ListId
 *
 *RETURNVALUE:  void
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vPrepareList(fc_sxm_trMsgWsAlertsMStartGetWSAlertsMsgList const *prMsg,tU32 u32ListId)
{
    //prepare local method start
    midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgListMethodStart const &oMStart=prMsg->oFiMsg;

    //prepare the configuration requested by HMI
    fc_sxm_trWsAlertsListConfig rListCfg;
    rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenWsAlertsListSortMethod)fc_sxm_enWsAlertListSortMethod_PRIORITY;
    rListCfg.rFilterCfg.enWsAlertsMsgObjectType=(fc_sxm_tenWsAlertFilterMethod)fc_sxm_enWsAlertsFilterMethod_NONE;

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareList(fc_sxm_trMsgWsAlertsMStartGetWSAlertsInfoList) vSetCfg: listId=%d enSortMethod=%d",
                     u32ListId,
                     ETG_CENUM(fc_sxm_tenWsAlertsListSortMethod, rListCfg.rSortCfg.enSortMethod)
                   ));
    //check if it is for combined list
    if (oMStart.ListId == static_cast<tU8>(fc_sxm_tenListType_Custom))
    {
        ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareList for Combined list case"));
        vSetListReqType(static_cast<tU8>(fc_sxm_tenListType_Custom));
        vPrepareCustomList(prMsg);
    }
    else
    {
        ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareList for Normal case"));

        //Set the configuration
        fc_sxm_trWsAlertsList::vSetCfg(u32ListId, rListCfg);

        //OK,this is for normal case (Nearby or Near destination) :just send list to HMI.
        vSendList(oMStart.ListId, prMsg->rAdressing);
    }
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vPrepareList(fc_sxm_trMsgWsAlertsMStartGetWSAlertsInfoList) vSendList"));
}

/*
  Handle method-starts
*/
/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartGetWSAlertsInfoList
 *PARAMETER:    1) Pointer to GetWsAlertsInfoList message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWSAlertsMsgList const *prMsg)
{

    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWSAlertsMStartGetWSAlertsInfoList"));
    midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgListMethodStart const &oMStart=prMsg->oFiMsg;

    //Set the request type to be checked while sending method result
    vSetListReqType(static_cast<tU8>(oMStart.ListId));

    //finally prepare list to be send to HMI
    vPrepareList(prMsg,oMStart.ListId);
}


/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartSetListsConfig
 *PARAMETER:    1) Pointer to fc_sxm_trMsgWsAlertsMStartSetListsConfig message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartSetListsConfig const *prMsg)
{
   midw_ext_sxm_wsalertsfi_tclMsgSetListsConfigMethodStart const &oFiMsg = prMsg->oFiMsg;
    tU32 u32Capacity=oFiMsg.Config.Capacity;

    tBool bNearByStatus = FALSE;
    tBool bDestStatus = FALSE;

    /* configure dsrl with new capacity */
    fc_sxm_trWsAlertsDSRLCfg rDsrlCfg;
    rDsrlCfg.u32Capacity = u32Capacity;

    /* configure the radius & Sort Types*/
    rDsrlCfg.enSortMethod = fc_sxm_enWsAlertSortMethod_PRIORITY;
    rDsrlCfg.u32Radius = FC_SXM_WSALERTS_STANDARD_DSRL_RADIUS;



    ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess() call Capacity=%d",
                     u32Capacity));

    /* Prapare method result with success Status*/
    midw_ext_sxm_wsalertsfi_tclMsgSetListsConfigMethodResult oMRes;

    if( OSAL_NULL != _poNearByDsrl)
    {
        /* So Set new capacity to NearBy DSRL*/
        _poNearByDsrl->vSetNextConfig(rDsrlCfg);
        /* Set the Method reult to true*/
        oMRes.Status = TRUE;
        bNearByStatus = TRUE;
        ETG_TRACE_USR4(("fc_sxm_trMsgWsAlertsMStartSetListsConfig vSetNextConfig Nearby"));
    }

    /* We are going to change the capacity only for Destination  dsrl*/
    /* And Check if _poNearDestDsrl is created */

     if( OSAL_NULL != _poNearDestDsrl)
     {
         ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::Set config for Destination"));
         /* So Set new capacity to Destination DSRL*/
         _poNearDestDsrl->vSetNextConfig(rDsrlCfg);
         /* Set the Method result to true*/
         bDestStatus = TRUE;
     }
     oMRes.Status = (bNearByStatus && bDestStatus);

     ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartSetListsConfig) call enSendFiMessage"));

     /* Send the method result to our service*/
    fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartSetListsConfig) END"));

}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Set property to Handle the List Mode
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartListModeSet message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartListModeSet  const *prMsg)
{
   ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgWsAlertsMStartListModeSet" ));

   midw_ext_sxm_wsalertsfi_tclMsgSxmListModeSet const &oPSet=prMsg->oFiMsg;

   /* Initialize default values*/
   tU32 u32ListId = oPSet.ListMode.ListID;
   ETG_TRACE_USR4(( "fc_sxm_trMsgWsAlertsMStartListModeSet: ListID %d",u32ListId));

   fc_sxm_tenListMode enListMode = (fc_sxm_tenListMode)oPSet.ListMode.Mode.enType;
   ETG_TRACE_USR4(( "fc_sxm_trMsgWsAlertsMStartListModeSet: enListMode %d",enListMode));
   vSetListMode(u32ListId, prMsg->rAdressing,enListMode);
}

/*********************************************************************
*FUNCTION:     enGetAlertStatus
*DESCRIPTION:  Get the Alert Status (Enable/Disable) from HMI
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig message
*RETURNVALUE:  fc_sxm_tenWSAlertStatus
********************************************************************/
fc_sxm_tenWSAlertStatus fc_sxm_tclWsAlertsApp::enGetAlertStatus(fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig  const *prMsg) const
{
   midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsConfigMethodStart const &oFiMsg=prMsg->oFiMsg;
   fc_sxm_tenWSAlertStatus enRetVal = fc_sxm_WSAlert_enStatus_UnSupported;

   if( oFiMsg.WSAlertAction.enType == midw_ext_fi_tcl_e8_WSAlertActionType::FI_EN_ENABLE)
   {
       enRetVal = fc_sxm_WSAlert_enStatus_Enabled;
   }
   else if (oFiMsg.WSAlertAction.enType == midw_ext_fi_tcl_e8_WSAlertActionType::FI_EN_DISABLE)
   {
       enRetVal = fc_sxm_WSAlert_enStatus_Disabled;
   }
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::enGetAlertStatus : %d",enRetVal));
   return enRetVal;
}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartSetListsConfig
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig  const *prMsg)
{
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig entered" ));

   midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsConfigMethodStart const &oFiMsg=prMsg->oFiMsg;
   midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsConfigMethodResult oMRes;
   oMRes.Status = TRUE;

   //for each alert TypeId from HMI
   SXM_FOREACH_CONST(vector<midw_ext_fi_tcl_e8_WSAlertTypes>, vecAlertTypeIter, oFiMsg.AlertTypeIDs)
   {
       //for each AlertTypeId from our internal Map
       SXM_FOREACH_MAP(string,fc_sxm_trWSAlertConfig,mapConfigIter,_mapAlertConfig)
       {
           //check if it is found
           if( (*vecAlertTypeIter).enType == mapConfigIter->second.u32AlertTypeId)
           {
               //OK,copy the status what HMI is set
               mapConfigIter->second.bIsEnabledInHMI = enGetAlertStatus(prMsg);

               ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig bIsEnabledInHMI:%d",
                                mapConfigIter->second.bIsEnabledInHMI));
               break;
           }
       }
   }
   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartSetWSAlertsConfig Exit"));
}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails
*PARAMETER:  Pointer to fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails  const *prMsg)
{
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails entered" ));

   midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgDetailsMethodStart const &oMStart=prMsg->oFiMsg;
   midw_ext_sxm_wsalertsfi_tclMsgGetWSAlertMsgDetailsMethodResult oMRes;

   tU32 u32MessageID = (tU32)oMStart.MessageID;

   /* entry to be filled from list: */
   fc_sxm_trWsAlertsListEntry rEntry;
   fc_sxm_trWsAlertsList *poList = fc_sxm_trWsAlertsList::poGetTypedListById(static_cast<tU32>(fc_sxm_tenListType_NearBy));

   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::u32MessageID=%d",u32MessageID));


   /* Check poList for NULL type*/
   if (OSAL_NULL != poList)
   {
      /* OK, we found the list */
      /* Check List for the MessageID*/
      tBool bEntryFound = poList->bGetListEntry(u32MessageID, rEntry);
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::NearBy State :%d",bEntryFound));
      if(bEntryFound == FALSE)
      {
         poList = fc_sxm_trWsAlertsList::poGetTypedListById(static_cast<tU32>(fc_sxm_tenListType_NearDest));
         if(OSAL_NULL != poList)
         {
            bEntryFound = poList->bGetListEntry(u32MessageID, rEntry);
            ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::NearDest State :%d",bEntryFound));
            if(bEntryFound == FALSE)
            {
               ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails) Entry not found"));
               /* Not Found MessageID in List,So Send an Error to FI*/
               fc_sxm_tclWsAlertsService::vSendError(prMsg->rAdressing,
                        (tU16) midw_ext_fi_tcl_e8_WSAlerts_ErrorTypes::FI_EN_OPERATION_FAILED);
               return;
            }
         }
         else
         {
            ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails) poList NULL"));
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails) poList NULL"));
      return;
   }

   /* Weather Alerts Info to be filled from our internal list*/
   midw_ext_fi_tcl_WSAlertMsg &rWSAlertMsgInfo = oMRes.WSAlertMsgInfo;
   if (OSAL_NULL != poList)
   {
      poList->vFillWSAlertInfo(rEntry,rWSAlertMsgInfo);
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::poList NULL"));
   }

   /* Send the Method Result to CCA*/
   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartGetWSAlertMsgDetails Exit"));
}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData
*PARAMETER:  Pointer to fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData  const *prMsg)
{
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData entered" ));
   midw_ext_sxm_wsalertsfi_tclMsgGetWeatherAlertDataMethodStart const &oMStart= prMsg->oFiMsg;
   (tVoid)oMStart;

   fc_sxm_trWsAlertsListConfig rListCfg;
   rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenWsAlertsListSortMethod)fc_sxm_enWsAlertListSortMethod_PRIORITY;
   rListCfg.rFilterCfg.enWsAlertsMsgObjectType = (fc_sxm_tenWsAlertFilterMethod) fc_sxm_enWsAlertsFilterMethod_NONE;

   vSetListReqType(static_cast<tU8>(fc_sxm_tenListType_AnotherLocation));
   fc_sxm_trWsAlertsList::vSetCfg((tU32) fc_sxm_tenListType_AnotherLocation, rListCfg);
   //OK,this is for Shape list case :just send list to HMI.
   vSendList(static_cast<tU32>(fc_sxm_tenListType_AnotherLocation), prMsg->rAdressing);
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartGetWeatherAlertData exit" ));
}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig  const *prMsg)
{
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig entered" ));
   midw_ext_sxm_wsalertsfi_tclMsgSetWSAlertsProjectConfigMethodResult oMRes;
   oMRes.Status = TRUE;

   // Configure the default Weather Alerts
   vConfigureWSAlerts(prMsg);

   fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
   ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcess fc_sxm_trMsgWsAlertsMStartSetWSAlertsProjectConfig Exit"));
}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Method to send FI message for fc_sxm_trMsgWsAlertsMStartValidateWSAlertMessage
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartValidateWSAlertMessage message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartValidateWSAlertMessage  const *prMsg)
{
   midw_ext_sxm_wsalertsfi_tclMsgValidateWSAlertMessageMethodStart const &oMStart= prMsg->oFiMsg;
   midw_ext_sxm_wsalertsfi_tclMsgValidateWSAlertMessageMethodResult oMRes;

   tU32 u32MessageID = static_cast<tU32>(oMStart.MessageID);
   oMRes.MessageID = u32MessageID;
   oMRes.Status = FALSE;

   /* entry to be filled from list: */
   fc_sxm_trWsAlertsListEntry rEntry;

   fc_sxm_trWsAlertsList *poListNearBy = fc_sxm_trWsAlertsList::poGetTypedListById(static_cast<tU32>(fc_sxm_tenListType_NearBy));

   if(OSAL_NULL != poListNearBy)
   {
      tBool bNearByEntry = poListNearBy->bGetListEntry(u32MessageID, rEntry);
      //first check in NearBy
      if(bNearByEntry == TRUE)
      {
         //If found, set status TRUE
         oMRes.Status = TRUE;
      }
      else
      {
         fc_sxm_trWsAlertsList *poListDest = fc_sxm_trWsAlertsList::poGetTypedListById(static_cast<tU32>(fc_sxm_tenListType_NearDest));
         //check in Near Destination list
         if(OSAL_NULL != poListDest)
         {
            tBool bNearDestEntry = poListDest->bGetListEntry(u32MessageID, rEntry);
            if(bNearDestEntry == TRUE)
            {
               //If found, set status TRUE
               oMRes.Status = TRUE;
            }
         }// end poListDest
      }// end bNearByEntry
   }
   //Send CCA message with Method result
  fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}



/*********************************************************************
 *FUNCTION:    vProcess(fc_sxm_trMsgTTSTimer const *prMsg)
 *DESCRIPTION:  message containing TTS timer
 *PARAMETER:    prMsg
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcessTimer(fc_sxm_trMsgTTSTimer const * /*prMsg*/)
{

   _otTTSTimer.vStop();
   if(_u8RetryCounter < FC_SXM_WSALERTS_TTS_RETRY_COUNTER)
   {
      tBool bIsCurrentStateInitial = (fc_sxm_tclTTSContext::instance()->poGetCurrentState() == fc_sxm_tclTTSStateInitial::instance());
      if(bIsCurrentStateInitial)
      {
         ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcessTimer(fc_sxm_trMsgTTSTimer), _otTTSTimer.vStart, _u8RetryCounter=%d",
                  _u8RetryCounter));
         _otTTSTimer.vStart(FC_SXM_WSALERTS_TTS_TIMEOUT);
         ++_u8RetryCounter;
      }
      else
      {
         _u8RetryCounter = 0;
         ETG_TRACE_ERR(( "fc_sxm_tclWsAlertsApp::vProcessTimer(fc_sxm_trMsgTTSTimer), TTS State is invalid"));
      }
   }
   else
   {
	   _u8RetryCounter = 0;
       fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateError::instance());
       vNotifyHmiErrorState();
       fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePending::instance());
       fc_sxm_tclTTSContext::instance()->vDoAction();
       ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vProcessTimer(fc_sxm_trMsgTTSTimer), Announcement is in pending state, _u8RetryCounter=%d",
               _u8RetryCounter));
   }
}

tVoid fc_sxm_tclWsAlertsApp::vNotifyHmiErrorState()
{
	ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vNotifyHmiErrorState"));
	//clear the alert message
	tBool bResult = fc_sxm_tclTTSClient::instance()->bClearSynthesizeText();
	ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vNotifyHmiErrorState bResult =%d",bResult));
	vNotifyWSAlertAnnounceStatus(midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_ERROR);

}

/*********************************************************************
*FUNCTION:     vStartWSALertAnnouncement
*DESCRIPTION:  Start the ws alert announcement
*PARAMETER:    poCurrentState of TTS Client handler
*RETURNVALUE:  None
********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bStartWSALertAnnouncement(fc_sxm_tclTTSState* poCurrentState, string sWSAlertText)
{
   tBool bResult = FALSE;
   if((poCurrentState == fc_sxm_tclTTSStateInitial::instance()))
   {
      bResult = fc_sxm_tclTTSClient::instance()->bStartAnnouncement(sWSAlertText);
      if(bResult)
      {
          _otTTSTimer.vStart(FC_SXM_WSALERTS_TTS_TIMEOUT);
          ++_u8RetryCounter;

      }
	  else
	  {
		fc_sxm_tclTTSClient::instance()->vHandleAudioSourceOn();
		bResult = true; 
	  }
	  
      ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::bStartWSALertAnnouncement, _otTTSTimer.vStart, _u8RetryCounter=%d",
               _u8RetryCounter));

   }
   else
   {
	  bResult = true;
      ETG_TRACE_ERR(( "bAnnounceWsAlert enAction=START_ANNOUNCEMENT, Invalid TTS Client State"));
   }
   return bResult;
}

/*********************************************************************
*FUNCTION:     vStopOrAbortWSALertAnnouncement
*DESCRIPTION:  Stop or Abort the ws alert announcement
*PARAMETER:    poCurrentState of TTS Client handler
*RETURNVALUE:  None
********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bStopOrAbortWSALertAnnouncement(const fc_sxm_tclTTSState* poCurrentState, tBool bIsStop) const
{
   tBool bResult = FALSE;
   if((poCurrentState == fc_sxm_tclTTSStatePrepared::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateStarted::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStatePaused::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateResumed::instance()))
   {
      bResult = bIsStop ?
               fc_sxm_tclTTSClient::instance()->bStopAnnouncement() :
               fc_sxm_tclTTSClient::instance()->bAbortAnnouncement();
   }
   else
   {
      ETG_TRACE_ERR(("bAnnounceWsAlert enAction=STOP_ANNOUNCEMENT/ABORT_ANNOUNCEMENT, Invalid TTS Client State"));
   }
   return bResult;
}

/*********************************************************************
*FUNCTION:     vPasueWSALertAnnouncement
*DESCRIPTION:  Pause ws alert announcement
*PARAMETER:    poCurrentState of TTS Client handler
*RETURNVALUE:  None
********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bPasueWSALertAnnouncement(const fc_sxm_tclTTSState* poCurrentState) const
{
   tBool bResult = FALSE;
   if((poCurrentState == fc_sxm_tclTTSStateStarted::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateResumed::instance()))
   {
      bResult = fc_sxm_tclTTSClient::instance()->bPauseAnnouncement();
   }
   else
   {
      ETG_TRACE_ERR(( "bAnnounceWsAlert enAction=PAUSE_ANNOUNCEMENT, Invalid TTS Client State"));
   }
   return bResult;
}

/*********************************************************************
*FUNCTION:     vResumeWSALertAnnouncement
*DESCRIPTION:  Resume ws alert announcement
*PARAMETER:    poCurrentState of TTS Client handler
*RETURNVALUE:  None
********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bResumeWSALertAnnouncement(const fc_sxm_tclTTSState* poCurrentState) const
{
   tBool bResult = FALSE;
   if((poCurrentState == fc_sxm_tclTTSStatePaused::instance()))
   {
      bResult = fc_sxm_tclTTSClient::instance()->bResumeAnnouncement();
   }
   else
   {
      ETG_TRACE_ERR(( "bAnnounceWsAlert enAction=RESUME_ANNOUNCEMENT, Invalid TTS Client State"));
   }
   return bResult;
}

/*********************************************************************
*FUNCTION:     bAnnounceWsAlert
*DESCRIPTION:  Parse the WSAlertAnnounce request and call TTS Client method
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert message
*RETURNVALUE:  TRUE when request is valid else FALSE
********************************************************************/
tBool fc_sxm_tclWsAlertsApp::bAnnounceWsAlert(fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert const *prMsg)
{
   tBool bResult = FALSE;
   if(prMsg)
   {
      _u32WSAlertMsgID = prMsg->oFiMsg.MessageID;
      midw_ext_fi_tcl_e8_WSAlertAnnounceAction::tenType enAction = prMsg->oFiMsg.Action.enType;
      tString chWSAlertText = prMsg->oFiMsg.Text.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
      string sWSAlertText = (string)chWSAlertText;
      fc_sxm_tclTTSState* poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
      switch(enAction)
      {
      case midw_ext_fi_tcl_e8_WSAlertAnnounceAction::FI_EN_ACTION_START:
         bResult = bStartWSALertAnnouncement(poCurrentState, sWSAlertText);
         break;

      case midw_ext_fi_tcl_e8_WSAlertAnnounceAction::FI_EN_ACTION_STOP:
         bResult = bStopOrAbortWSALertAnnouncement(poCurrentState, TRUE);
         break;
      case midw_ext_fi_tcl_e8_WSAlertAnnounceAction::FI_EN_ACTION_ABORT:
         bResult = bStopOrAbortWSALertAnnouncement(poCurrentState);
         break;

      case midw_ext_fi_tcl_e8_WSAlertAnnounceAction::FI_EN_ACTION_PAUSE:
         bResult = bPasueWSALertAnnouncement(poCurrentState);
         break;

      case midw_ext_fi_tcl_e8_WSAlertAnnounceAction::FI_EN_ACTION_RESUME:
         bResult = bResumeWSALertAnnouncement(poCurrentState);
         break;
      default:
         ETG_TRACE_ERR(( "fc_sxm_tclWsAlertsApp::bAnnounceWsAlert, enAction=%d is incorrect",
                  ETG_CENUM(midw_ext_fi_tcl_e8_WSAlertAnnounceAction::tenType, enAction)));
      }
	  OSAL_DELETE[] chWSAlertText;   
   } 
   return bResult;
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert)
*DESCRIPTION:  Handle the method start request for fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsMStartAnnounceWSAlert  const *prMsg)
{
   if(bAnnounceWsAlert(prMsg))
   {
      midw_ext_sxm_wsalertsfi_tclMsgAnnounceWSAlertMethodResult oMRes;
      oMRes.Status = TRUE;
      fc_sxm_tclWsAlertsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
   }
   else
   {
      fc_sxm_tclWsAlertsService::instance()->vSendError(prMsg->rAdressing,
                              (tU16) midw_ext_fi_tcl_e8_WSAlerts_ErrorTypes::FI_EN_OPERATION_FAILED);
   }
}



/*********************************************************************
*FUNCTION:     vHandleAudioSourceOn
*DESCRIPTION:  Handle the audio source activities
*PARAMETER:    None
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vHandleAudioSourceOn()
{

    fc_sxm_tclTTSState* poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
    ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vHandleAudioSourceOn, _u8RetryCounter=%d ",
            _u8RetryCounter));
    if((poCurrentState == fc_sxm_tclTTSStateInitial::instance()) &&
      (_otTTSTimer.bIsRunning()) &&
      (_u8RetryCounter <= FC_SXM_WSALERTS_TTS_RETRY_COUNTER))
    {
	    ETG_TRACE_USR4(( "fc_sxm_tclWsAlertsApp::vHandleAudioSourceOn,initial _u8RetryCounter=%d ",
	               _u8RetryCounter));
        _otTTSTimer.vStop();
        fc_sxm_tclTTSClient::instance()->vHandleAudioSourceOn();
    }
    else if(poCurrentState == fc_sxm_tclTTSStatePending::instance())
    {
		fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateInitial::instance());
		vNotifyWSAlertAnnounceStatus(midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_AVAILABLE);
    }
    else
    {
       fc_sxm_tclTTSClient::instance()->bReleaseAudioChannel();
    }
    _u8RetryCounter = 0;
}

/*********************************************************************
*FUNCTION:     vHandleAudioSourceOff
*DESCRIPTION:  Handle the audio source activities
*PARAMETER:    None
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vHandleAudioSourceOff() const
{
	ETG_TRACE_USR4(("vHandleAudioSourceOff, enSourceActivity=%d",enGetAudioSourceActivity()));
   fc_sxm_tclTTSState*  poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
   if((poCurrentState == fc_sxm_tclTTSStateStopped::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateError::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStatePending::instance() ||
      (poCurrentState == fc_sxm_tclTTSStateAborted::instance())))
   {
      midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
      vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
      fc_sxm_tclTTSClient::instance()->vHandleAudioSourceOff();
   }
   else if (poCurrentState == fc_sxm_tclTTSStateInitial::instance())
   {
	   ETG_TRACE_USR4(("vHandleAudioSourceOff, current state is initial still waiting for methodstart"));
   }
   else
   {
      fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateError::instance());
	  // Send as abort announcement to close acuostic device. 
	  // This part is execute when announcement is aborted/interuupted by SIRI/SMS/Phone
      fc_sxm_tclTTSClient::instance()->bAbortAnnouncement();
   }
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgWsAlertsAudioSource)
*DESCRIPTION:  Handle the Audio source activity
*PARAMETER:    Pointer to fc_sxm_trMsgWsAlertsAudioSource message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgWsAlertsAudioSource const *prMsg)
{
   if(prMsg)
   {
	  //variable to check the source availability
	  //and to ignore source request when the source is already available
	  vSetAudioSourceActivity(prMsg->enSourceActivity);
      switch(prMsg->enSourceActivity)
      {
      case FC_SXM_EN_AUDIO_SOURCE_ON:
         vHandleAudioSourceOn();
         break;
      case FC_SXM_EN_AUDIO_SOURCE_OFF:
         vHandleAudioSourceOff();
         break;
      case FC_SXM_EN_AUDIO_SOURCE_PAUSE:
         fc_sxm_tclTTSClient::instance()->vHandleAudioSourcePause();
         break;
      default:
         ETG_TRACE_ERR(("fc_sxm_tclWsAlertsApp::vProcess, Invalid Source Activity %d", prMsg->enSourceActivity));
      }
   }
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput)
*DESCRIPTION:  Handle the method result for Prepare Voice Output
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput  const *prMsg) const
{
   fc_sxm_tclTTSClient::instance()->vHandlePrepareVoiceOutputResult(prMsg);
}

/*********************************************************************
*FUNCTION:     vNotifyWSAlertAnnounceStatus
*DESCRIPTION:  Notify the WS alert announcement status to HMI
*PARAMETER:    enStatus status of the WS Alert announcement
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vNotifyWSAlertAnnounceStatus(midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enStatus) const
{
   fc_sxm_tcl_trWsAlertsPropertyWSAlertAnnounceStatus oWSAlertAnnounceStatus;
   oWSAlertAnnounceStatus.oFiMsg.MessageID = _u32WSAlertMsgID;
   oWSAlertAnnounceStatus.oFiMsg.WSAlertAnnounceStatus.enType = enStatus;
   fc_sxm_tclWsalerts_properties::instance()->oWSAlertAnnounceStatus.vSet(oWSAlertAnnounceStatus);
   fc_sxm_tclWsalerts_properties::instance()->oWSAlertAnnounceStatus.vNotify();

   ETG_TRACE_USR4(("vNotifyWSAlertAnnounceStatus, WSAlertAnnounceStatus=%d",
            ETG_CENUM(midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType, enStatus)));
}

/*********************************************************************
*FUNCTION:     enGetWSALertStatus
*DESCRIPTION:  Get ws alert announcement status
*PARAMETER:    None
*RETURNVALUE:  enWSALertStatus which depends on the TTS Client state
********************************************************************/
midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType fc_sxm_tclWsAlertsApp::enGetWSALertStatus() const
{
   midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSALertStatus =
            midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_INVALID;
   fc_sxm_tclTTSState*  poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
   if(poCurrentState == fc_sxm_tclTTSStateStarted::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_STARTED;
   }
   else if(poCurrentState == fc_sxm_tclTTSStatePending::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_AVAILABLE;
   }
   else if(poCurrentState == fc_sxm_tclTTSStatePaused::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_PAUSED;
   }
   else if(poCurrentState == fc_sxm_tclTTSStateResumed::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_RESUMED;
   }
   else if(poCurrentState == fc_sxm_tclTTSStateStopped::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_STOPPED;
   }
   else if(poCurrentState == fc_sxm_tclTTSStateAborted::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_ABORTED;
   }
   else if(poCurrentState == fc_sxm_tclTTSStateError::instance())
   {
      enWSALertStatus = midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::FI_EN_STATUS_ERROR;
   }
   return enWSALertStatus;
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak)
*DESCRIPTION:  Handle the first method result for Synthesize Speak
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak  const *prMsg) const
{
   fc_sxm_tclTTSClient::instance()->vHandleMethodResultFirstSynthesizeSpeak(prMsg);
   midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
   vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak)
*DESCRIPTION:  Handle the middle method result for Synthesize Speak
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak  const *prMsg) const
{
   fc_sxm_tclTTSClient::instance()->vHandleMethodResultMiddleSynthesizeSpeak(prMsg);
   midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
   vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak)
*DESCRIPTION:  Handle the last method result for Synthesize Speak
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak  const *prMsg) const
{
   fc_sxm_tclTTSState*  poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
   if((poCurrentState == fc_sxm_tclTTSStateStarted::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStatePaused::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateResumed::instance()))
   {
      fc_sxm_tclTTSClient::instance()->vHandleMethodResultLastSynthesizeSpeak(prMsg);
   }
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultAbortPrompt)
*DESCRIPTION:  Handle the last method result for Abort prompt
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultAbortPrompt message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultAbortPrompt const *prMsg) const
{
   fc_sxm_tclTTSState*  poCurrentState = fc_sxm_tclTTSContext::instance()->poGetCurrentState();
   // Check for valid abort announcement
   if((poCurrentState == fc_sxm_tclTTSStateAborted::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateStarted::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStatePaused::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStatePrepared::instance()) ||
      (poCurrentState == fc_sxm_tclTTSStateResumed::instance()))
   {
      fc_sxm_tclTTSClient::instance()->vHandleAbortPromptResult(prMsg);
      fc_sxm_tclTTSClient::instance()->bReleaseAudioChannel();
   }
   else if(poCurrentState == fc_sxm_tclTTSStateError::instance())
   {
      fc_sxm_tclTTSClient::instance()->vHandleAbortPromptResult(prMsg);
      midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
      vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
      fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateInitial::instance());
   }
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultPausePrompt)
*DESCRIPTION:  Handle the last method result for pause prompt
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultPausePrompt message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultPausePrompt const *prMsg) const
{
   fc_sxm_tclTTSClient::instance()->vHandlePausePromptResult(prMsg);
   midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
   vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientMResultContinuePrompt)
*DESCRIPTION:  Handle the last method result for continue prompt
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientMResultContinuePrompt message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientMResultContinuePrompt const *prMsg) const
{
   fc_sxm_tclTTSClient::instance()->vHandleContinuePromptResult(prMsg);
   midw_ext_fi_tcl_e8_WSAlertAnnounceStatus::tenType enWSAlertStatus = enGetWSALertStatus();
   vNotifyWSAlertAnnounceStatus(enWSAlertStatus);
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgTTSClientError)
*DESCRIPTION:  Handle the method error for all sds method starts
*PARAMETER:    Pointer to fc_sxm_trMsgTTSClientError message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgTTSClientError  const * /* prMsg */ ) const
{
   fc_sxm_tclTTSClient::instance()->vHandleError();
}

/*********************************************************************
*FUNCTION:     vProcessGenericMsg
*DESCRIPTION:
*PARAMETER:  Pointer to fc_sxm_trMsgDataSmsEvtServiceUp message
*RETURNVALUE:  None
**********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg)
{
    (tVoid) prMsg;
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vProcess(fc_sxm_trMsgDataSmsEvtServiceUp)"));
}


/****************************************************************************
*
*FUNCTION:          vClearServiceData
*
*DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.
                    This is a overridden method from the data app.
*PARAMETERS:
*
*RETURNVALUES:      None
*
******************************************************************************/
/*
   Method to be Overriden by each Service
   to clear Service specific internal data
   - Clear User Data - WsAlertObject, Alert MAP List Request
*/
tVoid fc_sxm_tclWsAlertsApp::vClearServiceData()
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vClearServiceData"));
   _u32UpdateCounter = 0;
   _u8RetryCounter = 0;
}

/*********************************************************************************
 *FUNCTION:     enGetCombinedListState
 *DESCRIPTION:  Method to send the WSAlertsInfoListStatus property to HMI
 *PARAMETER:    enNearByListState
 *PARAMETER:    enNearDestListState
 *RETURNVALUE:  fc_sxm_tenListState (Initial,Ready,updating,changed,error,frozen)
 *********************************************************************************/
fc_sxm_tenListState fc_sxm_tclWsAlertsApp::enGetCombinedListState(const fc_sxm_listStatusInfo& rNearByListInfo,
         const fc_sxm_listStatusInfo& rNearDestListInfo,
         fc_sxm_tenListState enNearByListState, fc_sxm_tenListState enNearDestListState) const
{
    fc_sxm_listStatusInfo rCombinedListInfo;
    rCombinedListInfo.u32Id = (tU32)fc_sxm_tenListType_Custom;
    rCombinedListInfo.enListType = fc_sxm_tenListType_Custom;

    fc_sxm_tenListState enRetValue = fc_sxm_enListState_Initial;

    if((rNearByListInfo.bActivated == TRUE) || (rNearDestListInfo.bActivated == TRUE))
    {
        rCombinedListInfo.bActivated = TRUE;
    }
    if((rNearByListInfo.bFrozen == TRUE) || (rNearDestListInfo.bFrozen == TRUE))
    {
       rCombinedListInfo.bFrozen = TRUE;
    }
    rCombinedListInfo.u32Size = rNearByListInfo.u32Size + rNearDestListInfo.u32Size;

    //Check for List state
    if((enNearByListState == fc_sxm_enListState_Initial) &&
             (enNearDestListState == fc_sxm_enListState_Initial))
    {
       enRetValue = fc_sxm_enListState_Initial;
    }
    else if((enNearByListState == fc_sxm_enListState_Ready) ||
             (enNearDestListState == fc_sxm_enListState_Ready))
    {
       enRetValue = fc_sxm_enListState_Ready;
    }
    else if((enNearByListState == fc_sxm_enListState_Changed) ||
             (enNearDestListState == fc_sxm_enListState_Changed))
    {
       enRetValue = fc_sxm_enListState_Changed;
    }
    else if((enNearByListState == fc_sxm_enListState_Updating) &&
             (enNearDestListState == fc_sxm_enListState_Updating))
    {
       enRetValue = fc_sxm_enListState_Updating;

    }
    else if((enNearByListState == fc_sxm_enListState_Activated) ||
             (enNearDestListState == fc_sxm_enListState_Activated))
    {
       enRetValue = fc_sxm_enListState_Activated;

    }
    else if((enNearByListState == fc_sxm_enListState_Frozen) ||
             (enNearDestListState == fc_sxm_enListState_Frozen))
    {
       enRetValue = fc_sxm_enListState_Frozen;

    }
    else if((enNearByListState == fc_sxm_enListState_Error) &&
             (enNearDestListState == fc_sxm_enListState_Error))
    {
       enRetValue = fc_sxm_enListState_Error;
    }
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::enGetCombinedListState :%d",enRetValue));
    return enRetValue;
}

/*********************************************************************************
 *FUNCTION:     vSendWSAlertInfoListStatus
 *DESCRIPTION:  send the WSAlertsInfoListStatus property to HMI
 *PARAMETER:    None
 *RETURNVALUE:  None
 *********************************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vSendWSAlertInfoListStatus()
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: vSendWSAlertInfoListStatus START"));
   fc_sxm_tcl_trWsAlertsPropertyNewUpdateWSAlertsInfoListStatus oProperty;
   midw_ext_sxm_wsalertsfi_tclMsgWSAlertsInfoListStatusStatus &oFiMsg = oProperty.oFiMsg;
   oGetDataListsInfo().vCopyListInfosToFi(oFiMsg.WSAlertsListStatus);

   fc_sxm_listStatusInfo rNearByListInfo = fc_sxm_trWsAlertsList::poGetTypedListById(
            static_cast<tU32>(fc_sxm_tenListType_NearBy))->rGetStatusInfo();

   fc_sxm_listStatusInfo rNearDestListInfo = fc_sxm_trWsAlertsList::poGetTypedListById(
            static_cast<tU32>(fc_sxm_tenListType_NearDest))->rGetStatusInfo();

   fc_sxm_listStatusInfo rCombinedListInfo;
   rCombinedListInfo.enListState = enGetCombinedListState(rNearByListInfo, rNearDestListInfo,
            rNearDestListInfo.enListState, rNearDestListInfo.enListState);

   oFiMsg.WSAlertsListStatus.ListInfo.push_back(rCombinedListInfo);

   /*Send FI Status*/
   fc_sxm_tclWsalerts_properties::instance()->oNewUpdateWSAlertsInfoListStatus.vSet(oProperty);
   fc_sxm_tclWsalerts_properties::instance()->oNewUpdateWSAlertsInfoListStatus.vNotify();
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: vSendWSAlertInfoListStatus END"));
}

/*********************************************************************
 *
 *FUNCTION:    vAddEntry
 *DESCRIPTION: Add the Entries into the combined Map (Nearby and Near Destination)
 *PARAMETER:   &rListEntry
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vAddEntry(const fc_sxm_trWsAlertsListEntry& rListEntry)
{
   _mapCombinedEntries.insert(make_pair(rListEntry.u32AlertMSGID,rListEntry));
}

/*********************************************************************
 *
 *FUNCTION:    vDeleteEntry
 *DESCRIPTION: Clear the mapEntries if the DSRL entry of the AlertMSGID is expired
 *PARAMETER:   &rListEntry
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vDeleteEntry(const fc_sxm_trWsAlertsListEntry& rListEntry)
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: vDeleteEntry AlertMSGID removed!"));
   _mapCombinedEntries.erase(rListEntry.u32AlertMSGID);
}

/*********************************************************************************
 *FUNCTION:     vSendCurrentWSAlerts
 *DESCRIPTION:  send the CurrentWSAlerts (alert pop-up) property to HMI
 *PARAMETER:    Pointer to the DataList
 *RETURNVALUE:  None
 *********************************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vSendCurrentWSAlerts(fc_sxm_tclDataListBase *poDataList)
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vSendCurrentWSAlerts START"));
   fc_sxm_tcl_trWsAlertsPropertyCurrentWSAlertsStatus oPropertyCurrentAlerts;
   midw_ext_sxm_wsalertsfi_tclMsgCurrentWSAlertsStatus &oFiMsgWS=oPropertyCurrentAlerts.oFiMsg;

   //Set the list request type to know which request type data (Nearby or Near Destination) to be sent
   vSetListReqType(static_cast<tU8>(poDataList->rGetStatusInfo().enListType));

   fc_sxm_trWsAlertsList *poList = OSAL_NULL;

   if (fc_sxm_tenListType_NearBy == (poDataList->rGetStatusInfo().enListType))
   {
      poList = fc_sxm_trWsAlertsList::poGetTypedListById(fc_sxm_tenListType_NearBy);

   }
   else if (fc_sxm_tenListType_NearDest == (poDataList->rGetStatusInfo().enListType))
   {
      poList = fc_sxm_trWsAlertsList::poGetTypedListById(fc_sxm_tenListType_NearDest);

   }
   if(poList != OSAL_NULL)
   {
      /*find with getSortedEntries from map if any new DSRL Entry found into property
      move new to normal and send fi*/
      SXM_FOREACH_MAP(tU32,fc_sxm_trWsAlertsListEntry,itr,_mapCombinedEntries)
      {
         //for each alert TypeId our internal list
         SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, itr->second.vecAlertTypes)
         {
            tBool bEnabled = FALSE;
            //for each AlertTypeId from our internal configuration
            SXM_FOREACH_MAP(string,fc_sxm_trWSAlertConfig,mapConfigIter,_mapAlertConfig)
            {
               //check if it is found
               if ((*vecAlertTypeIter).u32AlertTypeId == mapConfigIter->second.u32AlertTypeId)
               {
                  // Check if Alert is enabled in HMI
                  if (mapConfigIter->second.bIsEnabledInHMI == TRUE)
                  {
                     bEnabled = TRUE;
                     break;
                  }
                  ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vSendCurrentWSAlerts bEnabled:%d", bEnabled));
               }
            }
            if (itr->second.enState == fc_sxm_enListEntryState_New && bEnabled == TRUE)
            {
               midw_ext_fi_tcl_WSAlertMsg oMsg;
               poList->vFillWSAlertInfo(itr->second, oMsg);
               oFiMsgWS.WSAlertList.push_back(oMsg);
               //finally remove new
               itr->second.enState = fc_sxm_enListEntryState_Normal;
            }
         } // mapAlertConfig End loop
      } // mapAlertList End loop (Whole US)
   }
   /*Send FI Status*/
   fc_sxm_tclWsalerts_properties::instance()->oCurrentWSAlertsStatus.vSet(oPropertyCurrentAlerts);
   fc_sxm_tclWsalerts_properties::instance()->oCurrentWSAlertsStatus.vNotify();
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vSendCurrentWSAlerts END"));
}

/*********************************************************************************
 *FUNCTION:     vSendWeatherAlertData
 *DESCRIPTION:  send the WeatherAlertData (Weather Alert Map) property to HMI
 *PARAMETER:    None
 *RETURNVALUE:  None
 *********************************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vSendWeatherAlertData()
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: vSendWeatherAlertData START"));
   fc_sxm_tcl_trWsAlertsPropertyWeatherAlertData oProperty;
   midw_ext_sxm_wsalertsfi_tclMsgWeatherAlertDataStatus &oFiMsg = oProperty.oFiMsg;
   tBool bIsSupported = FALSE;
   fc_sxm_trWsAlertsList *poList = OSAL_NULL;

   poList = fc_sxm_trWsAlertsList::poGetTypedListById(static_cast<tU32>(fc_sxm_tenListType_AnotherLocation));

   if (poList != OSAL_NULL)
   {
      map<tU32, fc_sxm_trWsAlertsListEntry> &mapAlertList = poList->rGetMapEntries();
      /*find with getSortedEntries from map if any new DSRL Entry found into property
       move new to normal and send fi*/
      SXM_FOREACH_MAP(tU32,fc_sxm_trWsAlertsListEntry,itr,mapAlertList)
      {
         if (itr->second.enState == fc_sxm_enListEntryState_New)
         {
            //for each alert TypeId our internal list
            SXM_FOREACH_CONST(vector<fc_sxm_trWsAlertType>, vecAlertTypeIter, itr->second.vecAlertTypes)
            {
               //for each AlertTypeId from our internal configuration
               SXM_FOREACH_MAP(string,fc_sxm_trWSAlertConfig,mapConfigIter,_mapAlertConfig)
               {
                  //check if for each alertType Id
                  if ((*vecAlertTypeIter).u32AlertTypeId == mapConfigIter->second.u32AlertTypeId)
                  {
                     //check  if the alert is supported
                     if (mapConfigIter->second.bIsEnabledInMap == TRUE)
                     {
                        bIsSupported = TRUE;
                        ETG_TRACE_USR2(("fc_sxm_tclWsAlertsApp:: SXM_FOREACH_MAP bIsSupported true"));
                        // break the alert config map
                        break;
                     }
                     else
                     {
                        //bIsSupported is false
                     }
                  }
                  else
                  {
                    //Alert Type id Not found
                  }
               } // map AlertConfig loop end
               if (bIsSupported == TRUE)
               {
                  // break the FOR loop for each alert type id
                  break;
               }
            }// map AlertType loop end
            // finally remove new
            itr->second.enState = fc_sxm_enListEntryState_Normal;
         }
      } //mapAlertList for whole US
   }
   // if the alert is supported, update the counter and notify the HMI
   if(bIsSupported)
   {
      oFiMsg.UpdateCounter = ++_u32UpdateCounter;
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::_u32UpdateCounter : %d",_u32UpdateCounter));
      /*Send FI Status*/
      fc_sxm_tclWsalerts_properties::instance()->oWeatherAlertData.vSet(oProperty);
      fc_sxm_tclWsalerts_properties::instance()->oWeatherAlertData.vNotify();
   }
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp:: vSendWeatherAlertData END"));
}


/*********************************************************************
 *FUNCTION:     vOnListStateChanged
 *DESCRIPTION:  Method to send the WSAlertsInfoListStatus property to HMI
                Method to send the CurrentWSAlertsStatus property to HMI
 *PARAMETER:    poDataList
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vOnListStateChanged(fc_sxm_tclDataListBase *poDataList)
{
   ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vOnListStateChanged, List state=%d, List type=%d",
                                                         poDataList->rGetStatusInfo().enListState,
                                                         poDataList->rGetStatusInfo().enListType));

   if (fc_sxm_tenListType_Invalid != (poDataList->rGetStatusInfo().enListType))
   {
      //Weather Alert Shape List property status
      if(fc_sxm_tenListType_AnotherLocation == (poDataList->rGetStatusInfo().enListType))
      {
         if(fc_sxm_enListState_Ready == (poDataList->rGetStatusInfo().enListState))
         {
            vSendWeatherAlertData();
         }
         else
         {
            ETG_TRACE_USR4(("AnotherLocation list not ready yet"));
         }
      }
      else
      {
         //Weather Alert Message List property status
         vSendWSAlertInfoListStatus();

         //Current Weather Alerts POP-UP
         vSendCurrentWSAlerts(poDataList);
      }
   }
   else
   {
      ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vOnListStateChange  tenListType_Invalid"));
   }
}


/*********************************************************************
 *FUNCTION:     vOnFiServiceStateChanged
 *DESCRIPTION:  Application get notified about the new fi-service-state that shall be sent to clients
 *PARAMETER:     oFiState
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState) {
    ETG_TRACE_USR4(("fc_sxm_tclWsAlertsApp::vOnFiServiceStateChanged : State:%20s (%d)",
                    fc_sxm_szSmsDataServiceStateFiToString(oFiState.Status), 
                    oFiState.Status.enType));
     fc_sxm_tcl_trWsAlertsPropertySxmServiceStatus oProperty;
     oProperty.oFiMsg.DataServiceState=oFiState;
     fc_sxm_tclWsalerts_properties::instance()->oSxmServiceStatus.vSetAndNotify(oProperty);
}

/*********************************************************************
 *
 *FUNCTION:    vPrintReportSpecific
 *
 *DESCRIPTION:  for debuging
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclWsAlertsApp::vPrintReportSpecific() {
   /* for debuging */
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tspecific data:"));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\ttestListMode=%u", ETG_CENUM(fc_sxm_tenListMode, _enTestListMode)));
}



/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/
