/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_traffic_data.cpp
* @brief       Implementation for traffic data. This class will parse and forms the traffic data packet for TPEG.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*===========================================================================
===
===                           Includes
=============================================================================*/

#include "fc_sxm_common.h"
#include "fc_sxm_sms.h"

#include "fc_sxm_service_sxm_traffic.h"

#include "fc_sxm_tcl_traffic_app.h"
#include "fc_sxm_tcl_traffic_data.h"
#include "fc_sxm_tcl_traffic_types.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_TRAFFIC_DATA
#include "trcGenProj/Header/fc_sxm_tcl_traffic_data.cpp.trc.h"
#endif


/******************************************************************************
*FUNCTION:          fc_sxm_tclTrafficData
*
*DESCRIPTION:       constructor

*PARAMETERS:        None                  
*
*RETURNVALUES:      None
*******************************************************************************/

fc_sxm_tclTrafficData::fc_sxm_tclTrafficData():
   _bTmcDataSentCompleted(TRUE)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficData constructor"));
    _mmapRdbid.clear();
    _mMultipleSdtpMsg.clear();
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficData constructor"));  
}

/******************************************************************************
*FUNCTION:          ~fc_sxm_tclTrafficData
*
*DESCRIPTION:       destructor

*PARAMETERS:        None                  
*
*RETURNVALUES:      None
*******************************************************************************/

fc_sxm_tclTrafficData::~fc_sxm_tclTrafficData()
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficData destructor"));
    _mmapRdbid.clear(); /* clear the map */
    _bTmcDataSentCompleted = TRUE;
    _mMultipleSdtpMsg.clear();
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficData destructor"));

}

/******************************************************************************
*FUNCTION:          vClearTrafficServiceData
*
*DESCRIPTION:       Function to clear the stored traffic service related data during the
                    low power scenario

*PARAMETERS:
*
*RETURNVALUES:
*******************************************************************************/

tVoid fc_sxm_tclTrafficData::vClearTrafficServiceData()
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficData vClearTrafficServiceData"));
    _mmapRdbid.clear(); /* clear the map */
    _bTmcDataSentCompleted = TRUE;
    _mMultipleSdtpMsg.clear();
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficData vClearTrafficServiceData"));

}
/******************************************************************************
*FUNCTION:          vSetRdbid
*
*DESCRIPTION:       stores the Rdbidwishlist in a map to use later

*PARAMETERS:        poTrafficRdbidList: pointer to rdbidlist received from tima
*                  
*
*RETURNVALUES:      None
*******************************************************************************/

tVoid fc_sxm_tclTrafficData::vSetRdbid(trTrafficRdbidList const *poTrafficRdbidList)
{

    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vSetRdbid")); 
     /* clear Rdbid list in multimap */
    _mmapRdbid.clear();

    if(OSAL_NULL != poTrafficRdbidList)
    {
        /*size of the rdbid list */
        tU8 u8ListSize = poTrafficRdbidList->u8ListSize;

        /* index value doesn't exceed list size */
        for(tU8 u8Index = 0; u8Index < u8ListSize; u8Index++)
        {
            trBsaAndCountryCode rBsaAndCountryCode;
            rBsaAndCountryCode.u16Bsa = 
                poTrafficRdbidList->aoTrafficRdbid[u8Index].u16BSA;
            rBsaAndCountryCode.u8CountryCode = 
                poTrafficRdbidList->aoTrafficRdbid[u8Index].u8CountryCode;
            /* fill the map with LTN and bsa,country code */
            _mmapRdbid.insert(pair<tU8, trBsaAndCountryCode>(
                          poTrafficRdbidList->aoTrafficRdbid[u8Index].u8LTN,
                                                        rBsaAndCountryCode));

            ETG_TRACE_USR1((" LTN = 0x%x BSA = 0x%x",
                            poTrafficRdbidList->aoTrafficRdbid[u8Index].u8LTN,
                            poTrafficRdbidList->aoTrafficRdbid[u8Index].u16BSA)); 
         }/* end of for loop */
    }
        
    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vSetRdbid"));
 }
/*****************************************************************************
*FUNCTION:       vGetRdbidList
*
*DESCRIPTION:    Get the RdbidList to send the Actuallist

*PARAMETERS:     rTrafficRdbidList: reference to rdbidlist received from tima
*                  
*
*RETURNVALUES:   None
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vGetRdbidList(
                                      trTrafficRdbidList& rTrafficRdbidList)
{
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::vGetRdbidList"));

    /* size of the rdbid array , also used as index parameter to push
       rdbid elemnts to array. */
    rTrafficRdbidList.u8ListSize = 0;
    std::multimap<tU8 , trBsaAndCountryCode >::iterator iterValue;
    iterValue = _mmapRdbid.begin();
    /* iterate the map to get the RdbidList */
    while (iterValue != _mmapRdbid.end())
    {                                                                         
        rTrafficRdbidList.aoTrafficRdbid[rTrafficRdbidList.u8ListSize].u8LTN =
                                                           (*iterValue).first;
        trBsaAndCountryCode rBsaAndCountryCode = (*iterValue).second;
        rTrafficRdbidList.aoTrafficRdbid[rTrafficRdbidList.u8ListSize].u16BSA =
                                                     rBsaAndCountryCode.u16Bsa;
        rTrafficRdbidList.aoTrafficRdbid[
                               rTrafficRdbidList.u8ListSize].u8CountryCode =
                                              rBsaAndCountryCode.u8CountryCode;
        ETG_TRACE_USR3(("vGetRdbidList:LTN=0x%x BSA=0x%x",
                              (*iterValue).first, rBsaAndCountryCode.u16Bsa));
        ++iterValue;
        ++rTrafficRdbidList.u8ListSize;
            
    }/* end of while */
    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vGetRdbidList"));
}

/*****************************************************************************
*FUNCTION:       bIsLtnInRdbidList
*
*DESCRIPTION:    To check the Ltn received from payload is present in the List

*PARAMETERS:     u8LTN: Ltn received from payload
*                  
*
*RETURNVALUES:   None
******************************************************************************/
tBool fc_sxm_tclTrafficData::bIsLTNInRdbidList(tU8 u8LTN)
{
   ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::bIsLtnInRdbidList"));

   trTrafficRdbidList rTrafficRdbidList;

   vGetRdbidList(rTrafficRdbidList);
   for (tU8 u8LtnCount = 0; u8LtnCount < rTrafficRdbidList.u8ListSize; u8LtnCount++ )
   {
      if (u8LTN == rTrafficRdbidList.aoTrafficRdbid[u8LtnCount].u8LTN )
      {
         ETG_TRACE_USR4(("bIsLtnInRdbidList:In RdbidList"));
         return TRUE;
      }
   }

   ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::bIsLtnInRdbidList"));
   return FALSE;
}

/******************************************************************************
*FUNCTION:          vParsePayloadData
*
*DESCRIPTION:       Extracts the msg from pay load and call the parse function
                    irresepective of the completeness of sdtp packet.

*PARAMETERS:        pvPayload: payload packt from SMS
*                  
*
*RETURNVALUES:      tVoid 
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vParsePayloadData(fc_sxm_trMsgDataPayLoad rMsg)
{
   ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::vParsePayloadData"));

   //For testing
   static tU32 u32CountTrafficPlRx = 0;
   static tU32 u32CountTrafficPlNotValid = 0;

   tU8 u8Market = rMsg.u8Ltn;
   /* size of the msg */
   tU32 u32AlertCMsgSize = rMsg.u32PayLoadDataSize;

   //testing
   u32CountTrafficPlRx++;

   /* iterator for the map */
   if (bIsLTNInRdbidList(u8Market))
   {
      trSXMTrafficSDTPPacket rSdtpMsg;
      rSdtpMsg.u32Size = u32AlertCMsgSize;

      /* extract the msg from payload */
      OSAL_pvMemoryCopy((rSdtpMsg.u8Data),(rMsg.u8PayloadData),
                     u32AlertCMsgSize);

      trSXMTrafficMessage rTrafficMessage;
      /* Set the flag to indicate all the msgs are transmitted */
      rTrafficMessage.rSXMTrafficMessageDetails.bAllMsgTransmitted = TRUE;

      /* Set Duplicate flag to False as we are not checking duplicate message */
      rTrafficMessage.rSXMTrafficMessageDetails.bDuplicate = FALSE; 

      /* message for particular Market,Spread across multiple SDTP packets */
      rTrafficMessage.rSXMTrafficMessageDetails.bStartflag = FALSE; 

      /* setting the BSA to False as we are not supporting BSA */
      rTrafficMessage.rSXMTrafficMessageDetails.u16BSA = FALSE;

      rTrafficMessage.rSXMTrafficMessageDetails.u8LTN = u8Market; /* store the LTN */

      /* store the country code based on the LTN */
      rTrafficMessage.rSXMTrafficMessageDetails.u8CountryCode = 
                                        u8GetCountryCode(u8Market); 

      ETG_TRACE_USR3(("Incoming LTN = 0x%x CountryCode = 0x%x",
               rTrafficMessage.rSXMTrafficMessageDetails.u8LTN,
               rTrafficMessage.rSXMTrafficMessageDetails.u8CountryCode));
      /* parse the stored msg */
      vExtractAlertCMsg(rTrafficMessage,rSdtpMsg.u8Data,
               rSdtpMsg.u32Size);

      vStoreAlertCMsg(rTrafficMessage);
   }
   else
   {
      //testing
      u32CountTrafficPlNotValid++;
      ETG_TRACE_USR4(("Not valid LTN present in the wishlist"));
   }
   ETG_TRACE_USR4(("vParsePayloadData:PayloadRx = %u : PayloadStored = %u",
                     u32CountTrafficPlRx,(u32CountTrafficPlRx-u32CountTrafficPlNotValid)));

   ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vParsePayloadData"));  
}

/******************************************************************************
*FUNCTION:          vParseAndStorePayloadData
*
*DESCRIPTION:       Extracts the msg from pay load and call the parse function
                    based on the sdtp packet.

*PARAMETERS:        pvPayload: payload packt from SMS
                    
*                  
*
*RETURNVALUES:      tVoid 
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vParseAndStorePayloadData(fc_sxm_trMsgDataPayLoad rMsg)
{
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::vParseAndStorePayloadData"));

    tU8 u8Market = rMsg.u8Ltn;

    /* iterator for the map */
    if (bIsLTNInRdbidList(u8Market))
    {
       /* size of the msg */
       tU32 u32AlertCMsgSize = rMsg.u32PayLoadDataSize;

       ETG_TRACE_USR4(("Ltn is present in the RdbidList"));

       map<tU8,trSXMIncidentAndSFMessage>::iterator iMsgitr;

       /* find the market in the map */
       iMsgitr = _mMultipleSdtpMsg.find(u8Market);

       /* if multiple sdtp */
       if(!(rMsg.bAllMsgTransmitted))
       {
           if(iMsgitr !=_mMultipleSdtpMsg.end()) /* market exist */
           {               
               ETG_TRACE_USR4(("Multiple SDTP Mrkt exist,MsgSize:%d",u32AlertCMsgSize));
               trSXMTrafficSDTPPacket rSdtpMsg;
               rSdtpMsg.u32Size = u32AlertCMsgSize;

               /* extract the msg from payload */
               OSAL_pvMemoryCopy((rSdtpMsg.u8Data),(rMsg.u8PayloadData),
                    u32AlertCMsgSize);

               if (rMsg.bSpeedFlowMsg) /* speed&flow msg */
               {
                   ETG_TRACE_USR4(("Speed&FlowMsg Multiple SDTP Mrkt exist"));
                   /* push the msg into the S&F queue */
                   (*iMsgitr).second.qMultipleSFSDTP.push(rSdtpMsg);
               }

               if(rMsg.bIncidentMsg)/* incident msg */
               {
                   ETG_TRACE_USR4(("Incident Multiple SDTP Mrkt exist"));
                   /* push the msg into the Incident queue */
                   (*iMsgitr).second.qMultipleIncidentSDTP.push(rSdtpMsg);   
               }
           }
           else /* new market */
           {
               ETG_TRACE_USR4(("Multiple SDTP New Mrkt, MsgSize:%d",u32AlertCMsgSize));

               trSXMIncidentAndSFMessage rIncidentSpeedandFlow;
               trSXMTrafficSDTPPacket rSdtpMsg;
               rSdtpMsg.u32Size = u32AlertCMsgSize;

               /* extract the msg from the payload */
               OSAL_pvMemoryCopy((rSdtpMsg.u8Data),rMsg.u8PayloadData,
                      u32AlertCMsgSize);

               if (rMsg.bSpeedFlowMsg)/* for speed and flow msg */
               {
                   ETG_TRACE_USR4(("Speed&FlowMsg Multiple SDTP New Mrkt"));
                   rIncidentSpeedandFlow.qMultipleSFSDTP.push(rSdtpMsg);

                   /* insert the msg into the map */
                   _mMultipleSdtpMsg.insert(pair<tU8,trSXMIncidentAndSFMessage>(
                                           u8Market, rIncidentSpeedandFlow));
               }

               if(rMsg.bIncidentMsg) /* for incident msg */
               {
                   ETG_TRACE_USR4(("Incident Multiple SDTP New Mrkt"));
                   rIncidentSpeedandFlow.qMultipleIncidentSDTP.push(rSdtpMsg);

                   /* insert the msg into the map */
                   _mMultipleSdtpMsg.insert(pair<tU8,trSXMIncidentAndSFMessage>(
                                           u8Market, rIncidentSpeedandFlow));
               }
            }
       }
       else /* completed msg in the sdtp packt */
       {
           trSXMTrafficSDTPPacket rSdtpMsg;
           rSdtpMsg.u32Size = u32AlertCMsgSize;

           /* extract the msg from payload */
           OSAL_pvMemoryCopy((rSdtpMsg.u8Data), rMsg.u8PayloadData,
                    u32AlertCMsgSize);

           if(iMsgitr !=_mMultipleSdtpMsg.end()) /* markt present in the map */
           {
               ETG_TRACE_USR4(("Single SDTP Existing Mrkt:MsgSize:%d",u32AlertCMsgSize));

               /* send the msg for parsing */
               if(rMsg.bSpeedFlowMsg)
               {
                   ETG_TRACE_USR4(("SpeedFlowMsg Single SDTP Existing Mrkt"));

                   /* push the msg into the S&F queue */
                   (*iMsgitr).second.qMultipleSFSDTP.push(rSdtpMsg);

                   /* parse the last packet of msg */
                   vParseAlertCMsg(u8Market,rMsg.bSpeedFlowMsg,rMsg.bIncidentMsg);
                                                           
                   ETG_TRACE_USR4(("SpeedFlowMsg Single SDTP Existing Mrkt end"));
               }
               /* send the msg for parsing */
               if(rMsg.bIncidentMsg)
               {
                   ETG_TRACE_USR4(("Incident Single SDTP Existing Mrkt"));

                   /* push the msg into the Incident queue */
                   (*iMsgitr).second.qMultipleIncidentSDTP.push(rSdtpMsg);                

                   /* parse the last packet of msg */
                   vParseAlertCMsg(u8Market,rMsg.bSpeedFlowMsg,rMsg.bIncidentMsg);
                                                         
                   ETG_TRACE_USR4(("Incident Single SDTP Existing Mrkt end"));
               }
    
           }
           else /* new market with complete msg */
           {
               ETG_TRACE_USR4(("Single SDTP New Mrkt"));
               trSXMIncidentAndSFMessage rIncidentSpeedandFlow;
               
               if(rMsg.bSpeedFlowMsg)/* speed and flow msg */
               {
                   ETG_TRACE_USR4(("Speed&FlowMsg Single SDTP New Mrkt"));

                   rIncidentSpeedandFlow.qMultipleSFSDTP.push(rSdtpMsg);

                   /* insert the msg into the map */
                   _mMultipleSdtpMsg.insert(pair<tU8,trSXMIncidentAndSFMessage>(
                                           u8Market, rIncidentSpeedandFlow));
                   /* parse the msg */
                   vParseAlertCMsg(u8Market,rMsg.bSpeedFlowMsg,rMsg.bIncidentMsg);
               }
               if(rMsg.bIncidentMsg) /* incident msg */
               {
                   ETG_TRACE_USR4(("Incident Single SDTP New Mrkt"));

                   rIncidentSpeedandFlow.qMultipleIncidentSDTP.push(rSdtpMsg);

                   /* insert the msg into the map */
                   _mMultipleSdtpMsg.insert(pair<tU8,trSXMIncidentAndSFMessage>(
                                           u8Market, rIncidentSpeedandFlow));
                   /* parse the msg */
                   vParseAlertCMsg(u8Market,rMsg.bSpeedFlowMsg,rMsg.bIncidentMsg);
               }
           }
       }
    }

    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vParseAndStorePayloadData"));
}
/******************************************************************************
*FUNCTION:          vParseAlertCMsg
*
*DESCRIPTION:       prepare the Tmc data for TMC data properties

*PARAMETERS:        rTmcDataStatus: stores the TMCdata and returns to
                    traffic application
*                  
*
*RETURNVALUES:      tBool
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vParseAlertCMsg(tU8 u8Market,tBool bSpeedFlowMsg,
                                                         tBool bIncidentMsg)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vParseAlertCMsg"));
    trSXMTrafficMessage rTrafficMessage;

    /* Set the flag to indicate all the msgs are transmitted */
    rTrafficMessage.rSXMTrafficMessageDetails.bAllMsgTransmitted = TRUE;

    /* Set Duplicate flag to False as we are not checking duplicate message */
    rTrafficMessage.rSXMTrafficMessageDetails.bDuplicate = FALSE; 

    /* message for particular Market,Spread across multiple SDTP packets */
    rTrafficMessage.rSXMTrafficMessageDetails.bStartflag = FALSE; 

    /* setting the BSA to False as we are not supporting BSA */
    rTrafficMessage.rSXMTrafficMessageDetails.u16BSA = FALSE;

    rTrafficMessage.rSXMTrafficMessageDetails.u8LTN = u8Market; /* store the LTN */

    /* store the country code based on the LTN */
    rTrafficMessage.rSXMTrafficMessageDetails.u8CountryCode = 
                                     u8GetCountryCode(u8Market); 

    ETG_TRACE_USR3(("Incoming LTN = 0x%x CountryCode = 0x%x",
                     rTrafficMessage.rSXMTrafficMessageDetails.u8LTN,
                     rTrafficMessage.rSXMTrafficMessageDetails.u8CountryCode));

    /* find the market id in the map */
    map<tU8,trSXMIncidentAndSFMessage>::iterator iMsgitr;

    iMsgitr = _mMultipleSdtpMsg.find(u8Market);

    if(bSpeedFlowMsg) /* for speed and flow msg */
    {
        ETG_TRACE_USR4(("Speed&Flow Msg storing"));

        while(TRUE != (*iMsgitr).second.qMultipleSFSDTP.empty())
        {
            ETG_TRACE_USR4(("Start storing Alertc of S&F"));

            trSXMTrafficSDTPPacket rSdtpMsg;
            rSdtpMsg = (*iMsgitr).second.qMultipleSFSDTP.front();

            /* clear the queue from the map */
            (*iMsgitr).second.qMultipleSFSDTP.pop();

            /* parse the stored msg */
            vExtractAlertCMsg(rTrafficMessage,rSdtpMsg.u8Data,
                          rSdtpMsg.u32Size);
        } /* end of while */

        ETG_TRACE_USR4(("End storing Alertc of S&F"));

    }

    if(bIncidentMsg) /* for incident msg */
    {
        ETG_TRACE_USR4(("Incident Msg Storing"));

        while(TRUE != (*iMsgitr).second.qMultipleIncidentSDTP.empty())
        {
            ETG_TRACE_USR4(("Start storing Alertc of Incident"));

            trSXMTrafficSDTPPacket rSdtpMsg;
            rSdtpMsg = (*iMsgitr).second.qMultipleIncidentSDTP.front();

            /* clear the queue from the map */
            (*iMsgitr).second.qMultipleIncidentSDTP.pop();

            /* store the alert-c msg */
            vExtractAlertCMsg(rTrafficMessage,rSdtpMsg.u8Data,
                          rSdtpMsg.u32Size);
        } /* end of while */

        ETG_TRACE_USR4(("End storing Alertc of Incident"));
    }
    
    vStoreAlertCMsg(rTrafficMessage);

    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vParseAlertCMsg"));   

}

/****************************************************************************
*FUNCTION:          vExtractAlertCMsg
*
*DESCRIPTION:       store the Alert-c messages in the queue

*PARAMETERS:        rTrafficMessage:reference to store the Alertc messages
                    u32AlertCMsgSize: size of the msg
*                   pu8AlertCMessage: msg buffer
*
*RETURNVALUES:      tVoid
******************************************************************************/
tVoid fc_sxm_tclTrafficData::vExtractAlertCMsg(
                                       trSXMTrafficMessage &rTrafficMessage,
                          tU8 const *pu8AlertCMessage,tU32 u32AlertCMsgSize) const
{
    tU32 u32AlertCOffset = OSAL_NULL;
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::vExtractAlertCMsg"));
    ETG_TRACE_USR4(("vExtractAlertCMsg u32AlertCMsgSize:%d",u32AlertCMsgSize));

    while(u32AlertCMsgSize > OSAL_NULL)
    {
        if (((pu8AlertCMessage[u32AlertCOffset] >> 3) && 0xFF) == 
                                                     SXM_SINGLE_GRP_MSG_INDCTN)
        {
           /* ETG_TRACE_USR4(("fc_sxm_tclTrafficData::vParseAlertCMsg"
                                         " u32AlertCMsgSize Single Group")); */
            trSXMAlertCmessage rAlertCMessage;
            rAlertCMessage.u32SizeOfAlert8Msg = 
                                  SXM_TRAFFIC_SIZEOF_SINGLE_GRP_TMC_MSG;

            OSAL_pvMemoryCopy(rAlertCMessage.u8AlertCMessage,
                       static_cast<tPCU8>(&pu8AlertCMessage[u32AlertCOffset]),
                       SXM_TRAFFIC_SIZEOF_SINGLE_GRP_TMC_MSG);


            rTrafficMessage.qoTrafficAlertCMessage.push_back(rAlertCMessage);
            u32AlertCMsgSize -= SXM_TRAFFIC_SIZEOF_SINGLE_GRP_TMC_MSG;
            u32AlertCOffset +=  SXM_TRAFFIC_SIZEOF_SINGLE_GRP_TMC_MSG;

        }
        else if (((pu8AlertCMessage[u32AlertCOffset] >> 3) && 0xFF) == 
                                                     SXM_MULTI_GRP_MSG_INDCTN)
        {
            /* ETG_TRACE_USR4(("fc_sxm_tclTrafficData::vParseAlertCMsg"
                                   " u32AlertCMsgSize MultiGroupMsg")); */
            tU32 u32SizeOfMGM = OSAL_NULL;
            tU32 u32Index = u32AlertCOffset;
            ETG_TRACE_USR4(("vParseAlertCMsg u32AlertCMsgSize"
                 " MultiGroupMsg:%0x",pu8AlertCMessage[u32AlertCOffset + 1]));

            if (((pu8AlertCMessage[u32AlertCOffset + 1] & 
                                                SXM_MULTIGRP_FG_MSG_INDCTN) == 
                                                SXM_MULTIGRP_FG_MSG_INDCTN))
            {
                u32Index += SXM_TRAFFIC_SIZEOF_SINGLE_GRP_TMC_MSG;
                /* find the Group Sequence Identifier, which counts the total
                   no of MGM followed */
                tU8 u8GsiData = ((pu8AlertCMessage[u32Index + 1] >> 4) &
                                                                         0x03);
                /* ETG_TRACE_USR4(("fc_sxm_tclTrafficData::vParseAlertCMsg"
                                          " MGM u8GsiData: %d",u8GsiData)); */
                
                switch(u8GsiData)
                {                    
                    case SXM_MGM_SUBSQENT_GSI_VALUE_ZERO:
                        u32SizeOfMGM = SXM_MGM_MNIMUM_SIZE ;
                        break;
                    case SXM_MGM_SUBSQENT_GSI_VALUE_ONE:
                        u32SizeOfMGM = SXM_MGM_MNIMUM_SIZE + 5;
                        break;
                    case SXM_MGM_SUBSQENT_GSI_VALUE_TWO:
                        u32SizeOfMGM = SXM_MGM_MNIMUM_SIZE + 10;
                        break;
                    default:
                        u32SizeOfMGM = SXM_MGM_MAXIMUM_SIZE;
                        break;
                }
                                                                 
            }

            else
            {
                ETG_TRACE_USR4(("vParseAlertCMsg MGM Error"));
                break;

            }

            trSXMAlertCmessage rAlertCMessage;

            /* ETG_TRACE_USR4(("fc_sxm_tclTrafficData::vParseAlertCMsg"
                                       " u32SizeOfMGM:%d",u32SizeOfMGM)); */
            rAlertCMessage.u32SizeOfAlert8Msg = u32SizeOfMGM; 
            OSAL_pvMemoryCopy(rAlertCMessage.u8AlertCMessage,
                      static_cast<tPCU8>(&pu8AlertCMessage[u32AlertCOffset]),
                      u32SizeOfMGM);

            rTrafficMessage.qoTrafficAlertCMessage.push_back(rAlertCMessage);
            u32AlertCMsgSize -= u32SizeOfMGM;
            u32AlertCOffset +=  u32SizeOfMGM;
        }
        else
        {
            ETG_TRACE_USR4(("vParseAlertCMsg Not Alert-c message"));
            break;
        }
    }/* end of while */
    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vExtractAlertCMsg"));
}

/****************************************************************************
*FUNCTION:          u8GetCountryCode
*
*DESCRIPTION:       copy the message to the queue

*PARAMETERS:        rSXMTrafficMessage:Alertc messages
*                  
*
*RETURNVALUES:      tVoid
******************************************************************************/
tU8 fc_sxm_tclTrafficData::u8GetCountryCode(tU8 u8Ltn)
{
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::u8GetCountryCode"));
    /* default value of country code */
    tU8 u8CountryCode = SXM_TRAFFIC_TMCCOUNTRYCODE;
    std::multimap <tU8, trBsaAndCountryCode>::iterator iterValue;
    /* iterValue to the entry for the LTN in the multimap */
    iterValue = _mmapRdbid.find(u8Ltn);
    /* if there is an entry, take the country code */
    if(_mmapRdbid.end() != iterValue)
    {
        u8CountryCode=(*iterValue).second.u8CountryCode;
    }
    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::u8GetCountryCode"));
    return u8CountryCode;
}

/******************************************************************************
*FUNCTION:          vStoreAlertCMsg
*
*DESCRIPTION:       copy the message to the queue

*PARAMETERS:        rSXMTrafficMessage:Alertc messages
*                  
*
*RETURNVALUES:      tVoid
*******************************************************************************/
tVoid fc_sxm_tclTrafficData::vStoreAlertCMsg(
                                  trSXMTrafficMessage const &rSXMTrafficMessage)
{
   ETG_TRACE_USR4(("---> fc_sxm_tclTrafficData::vStoreAlertCMsg"));
    
   /* if TMC messages are present */
   if( !rSXMTrafficMessage.qoTrafficAlertCMessage.empty())
   {
      ETG_TRACE_USR3(("Push TMC to Queue"));

      /* push the message in a queue */
      _qTrafficMessage.push(rSXMTrafficMessage);

      ETG_TRACE_USR4(("_qTrafficMessage size:%d",_qTrafficMessage.size()));
   }

   /* if timer is not running and data is complete, send the data */
   if ( bIsTmcDataCompleted() && 
           !fc_sxm_tclTrafficApp::instance()->bIsTmcDataTimerRunning())
   {
       vReadyToSendTMC();
   }

   ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficData::vStoreAlertCMsg"));      
}


/******************************************************************************
*FUNCTION:          vTmcDataCompleted
*
*DESCRIPTION:       To keep trac of the data is completely sent or not

*PARAMETERS:        rSXMTrafficMessage:Alertc messages
*                  
*
*RETURNVALUES:      tVoid
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vSetTmcDataCompleted(tBool bIsDataCompleted)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vTmcDataCompleted")); 
    _bTmcDataSentCompleted = bIsDataCompleted;    
    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vTmcDataCompleted")); 
}


/******************************************************************************
*FUNCTION:          vTmcDataCompleted
*
*DESCRIPTION:       To keep trac of the data is completely sent or not

*PARAMETERS:        rSXMTrafficMessage:Alertc messages
*                  
*
*RETURNVALUES:      tVoid
******************************************************************************/

tBool fc_sxm_tclTrafficData::bIsTmcDataCompleted(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclTrafficData::bIsTmcDataCompleted:%d",
                                _bTmcDataSentCompleted )); 
    return _bTmcDataSentCompleted;  
}

/******************************************************************************
*FUNCTION:          vReadyToSendTMC
*
*DESCRIPTION:       Copy the TMc data from the queue

*PARAMETERS:        tVoid
*                  
*
*RETURNVALUES:      tVoid
******************************************************************************/


tVoid fc_sxm_tclTrafficData::vReadyToSendTMC(tVoid)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vReadyToSendTMC"));

    /* if TMC present in the queue */
    if( !_qTrafficMessage.empty() )
    {
        /* read the traffic message to send to FC_TIMA */
        _oReadyToSendTMC = _qTrafficMessage.front();

        /*pop the stored message from queue */
        _qTrafficMessage.pop();

        /* post to the application data is ready */
        fc_sxm_tclTrafficApp::instance()->vPostTmcData();
        
        ETG_TRACE_USR4(("Ready to Send TMC Details LTN = "
                      "0x%x BSA = 0x%x bDuplicate = 0x%x  bStartflag = 0x%x ListSize = %d",
                      _oReadyToSendTMC.rSXMTrafficMessageDetails.u8LTN,
                      _oReadyToSendTMC.rSXMTrafficMessageDetails.u16BSA,
                      _oReadyToSendTMC.rSXMTrafficMessageDetails.bDuplicate,
                      _oReadyToSendTMC.rSXMTrafficMessageDetails.bStartflag,
                      _oReadyToSendTMC.qoTrafficAlertCMessage.size())); 
     
    }
    else
    {             
         ETG_TRACE_USR4(("No data in queue"));
    }

    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vReadyToSendTMC")); 
}
/****************************************************************************
*FUNCTION:          vGetTMCData
*
*DESCRIPTION:       prepare the Tmc data for TMC data properties

*PARAMETERS:        rTmcDataStatus: stores the TMCdata and returns to traffic
                    application
*                  
*
*RETURNVALUES:      tVoid
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vGetTMCData(
                                   sxm_tclMsgFID_TRAFFIC_G_TmcDataStatus& rTmcDataStatus,
                                   tBool &bDataForRdbIdComplete)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vGetTMCData"));
    /* TRUE:When Multiple SDTP packets for a Market,FALSE:Single SDTP packet*/
    rTmcDataStatus.CycleStart =
                    _oReadyToSendTMC.rSXMTrafficMessageDetails.bStartflag;
    /* TRUE: Only duplicate for given RdbId,FALSE:No duplicate,new information
       Always set to False as we can not check */
    rTmcDataStatus.Duplicate =
                        _oReadyToSendTMC.rSXMTrafficMessageDetails.bDuplicate;

    /* BSA Setting to 0 as SXM is not sending BSA */
    rTmcDataStatus.RDBId.u16Bsa = FALSE;

    /* Location table number (Market) */
    rTmcDataStatus.RDBId.u8RdbidLTN =
                             _oReadyToSendTMC.rSXMTrafficMessageDetails.u8LTN;
    /* No encripted data */
    rTmcDataStatus.RDBId.bEncFlag = FALSE;

    /* service id is 2 */
    rTmcDataStatus.RDBId.u8RdbidSID =
                             SXM_TRAFFIC_SERVICE_ID;

    /* country code based on the wishlist */
    rTmcDataStatus.RDBId.u8RdbidCC =
                     _oReadyToSendTMC.rSXMTrafficMessageDetails.u8CountryCode;

    rTmcDataStatus.DataForRdbIdComplete = FALSE;
  
    /* clear the traffic info vector */
    rTmcDataStatus.ListOfTrafficInfo.clear();

    /* copy the alert-c messages based on the Maximum amount of
    data set in capabilities */
    while (!_oReadyToSendTMC.qoTrafficAlertCMessage.empty())
    {
        /* object to traffic info class */
        sxm_tcl_e16_TrafficInfo oTrafficInfo;

        oTrafficInfo.ApplicationId.enType =
                          sxm_tcl_e16_TrafficTmcApplicationId::FI_EN_TMCDATA;

     	  oTrafficInfo.Encryption.Encrypted = FALSE;

        /* message is not encrypted, so set it to 0 */
        oTrafficInfo.Encryption.Encid = 0;

        /* Test bits for encrypted messages, 0 if message not encrypted */
        oTrafficInfo.Encryption.TestBits = 0;

        trSXMAlertCmessage rAlertCMsgs =
                           _oReadyToSendTMC.qoTrafficAlertCMessage.front();

        //ETG_TRACE_USR4(("TMC to Client DATA size = %d ",rAlertCMsgs.u32SizeOfAlert8Msg));

        /* push the message to the list */
        for(tU8 u8Index = 0; u8Index < (rAlertCMsgs.u32SizeOfAlert8Msg);
                                                                u8Index++)
        {
             oTrafficInfo.ListOf8AData.push_back(
                                       rAlertCMsgs.u8AlertCMessage[u8Index]);
                                
        }/* end of for loop */

        /* clear the queue */
        _oReadyToSendTMC.qoTrafficAlertCMessage.pop_front();

        oTrafficInfo.MsgSource.MediumTypeId.enType =
                              sxm_tcl_e8_TrafficTmcMediumType::FI_EN_SATELLITE;
        /* Message source */
        oTrafficInfo.MsgSource.Number = 1;

        /* identifying the program service, the message is
         sent from. */
        oTrafficInfo.ProgramID = SXM_TRAFFIC_PROGRAM_ID;

        /* set the RDBID */
        oTrafficInfo.ProgramServiceName = SXM_TRAFFIC_PROGRAMSERVICENAME;

        oTrafficInfo.RDBId.u16Bsa =
                  _oReadyToSendTMC.rSXMTrafficMessageDetails.u16BSA;
        oTrafficInfo.RDBId.bEncFlag = FALSE;
        oTrafficInfo.RDBId.u8RdbidSID = SXM_TRAFFIC_SERVICE_ID;
        oTrafficInfo.RDBId.u8RdbidLTN =
                  _oReadyToSendTMC.rSXMTrafficMessageDetails.u8LTN;
        oTrafficInfo.RDBId.u8RdbidCC =
                  _oReadyToSendTMC.rSXMTrafficMessageDetails.u8CountryCode;
            
        oTrafficInfo.ServiceProviderName =
                                        SXM_TRAFFIC_SERVICEPROVIDERNAME;

        rTmcDataStatus.ListOfTrafficInfo.push_back(oTrafficInfo);
        oTrafficInfo.ListOf8AData.clear();
    }
 
    if(_oReadyToSendTMC.qoTrafficAlertCMessage.empty())
    {
        /* no traffic message left in the queue for the SDTP,return false */
        rTmcDataStatus.DataForRdbIdComplete = TRUE;
        bDataForRdbIdComplete = TRUE;
    }
   
    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vGetTMCData"));
}

/****************************************************************************
*FUNCTION:          vClearOldData
*
*DESCRIPTION:       Based on the rdbid wishlist it clears the data.
                    If rdbid wishlist id empty it clears all the previous data.

*PARAMETERS:        u8Market:market id
*                  
*
*RETURNVALUES:      None
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vClearOldData(tU8 u8Ltn)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vClearOldData"));

    tU8 u8Market = u8Ltn;
    map<tU8,trSXMIncidentAndSFMessage>::iterator iMsgitr;

    /* find the market */
    iMsgitr = _mMultipleSdtpMsg.find(u8Market);
    if(iMsgitr !=_mMultipleSdtpMsg.end()) /* market exist */
    {
        ETG_TRACE_USR4(("vClearOldData Market ID:%d",u8Market));

        /* erase the market */
        _mMultipleSdtpMsg.erase(iMsgitr);
    }

    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vClearOldData")); 
}


/****************************************************************************
*FUNCTION:          vClearDataonNoSignal
*
*DESCRIPTION:       when there is no signal, this will clear all the stored
                    data

*PARAMETERS:        void
*                  
*
*RETURNVALUES:      None
******************************************************************************/

tVoid fc_sxm_tclTrafficData::vHandleNoSignal(tVoid)
{
    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficData::vClearDataonNoSignal"));
    /* clear the data from map */
    _mMultipleSdtpMsg.clear();

    ETG_TRACE_USR4(("<-- fc_sxm_tclTrafficData::vClearDataonNoSignal")); 
}
