/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_traffic_app.cpp
* @brief       Implements the traffic application handler.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/
/*===========================================================================
===
===                           Includes
=============================================================================*/
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_traffic.h"
#include "fc_sxm_tcl_traffic_app.h"
#include "fc_sxm_tcl_traffic_data.h"
#include "fc_sxm_tcl_traffic_properties.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_TRAFFIC_APP
#include "trcGenProj/Header/fc_sxm_tcl_traffic_app.cpp.trc.h"
#endif

typedef enum
{
    fc_sxm_enTtfisCmdsTraffic_METHODSTART_WISHLIST=1
} fc_sxm_tenTtfisCmdsTraffic;

//#define FC_SXM_TRAFFIC_PROCESS_ALL_BIT_FLAG              1

/*===========================================================================
====
====                          Methods
=============================================================================*/


/****************************************************************************
*
*FUNCTION:          fc_sxm_tclTrafficApp
*
*DESCRIPTION:       Constructor
*
*PARAMETERS:        None 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tBool fc_sxm_tclTrafficApp::bModifyFilter(tBool bEnable, SXM_DMI tDMI) {
    DATASERVICE_DMI_CONFIG_STRUCT rCfg;
    rCfg.tDMI=tDMI;
    rCfg.bEnable=bEnable;
    tBool bRet=( DATA.eManageDataStream(_hTraffic, &rCfg, 1) ==SMSAPI_RETURN_CODE_SUCCESS);
    if (!bRet) {
        ETG_TRACE_ERR(("fc_sxm_tclTrafficApp::bModifyFilter FAILED"));
    }
    return bRet;
}


fc_sxm_tclTrafficApp::fc_sxm_tclTrafficApp():
    fc_sxm_tclDataApp("sxmTrafficApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_TRAFFIC, "traffic", TR_CLASS_FC_SXM_TRAFFIC_APP)
{

    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp constructor"));
    /* initialize the default values to the member variables */
    _hTraffic = DATASERVICE_MGR_INVALID_OBJECT;
    _u16Dsi = FC_SXM_DATASERVICE_TRAFFIC_DSI_ID;
    _enTunerState = sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_OFF;
    _enOperationMode = sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_INITIAL;
    _bIsSXMTrafficUp = FALSE;
    _bRdbListActive = FALSE;
    _bSetDMI = FALSE;

    /* clear the LTN queue */
    while(TRUE != _qLtn.empty())
    {
        _qLtn.pop();
    }

    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp constructor"));
 
}

/****************************************************************************
*
*FUNCTION:          ~fc_sxm_tclTrafficApp
*
*DESCRIPTION:       Destructor
*
*PARAMETERS:        None 
*
*RETURNVALUES:      None
*                    
******************************************************************************/

fc_sxm_tclTrafficApp::~fc_sxm_tclTrafficApp()
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp destructor"));
    _hTraffic = DATASERVICE_MGR_INVALID_OBJECT;
    _u16Dsi = FC_SXM_DATASERVICE_TRAFFIC_DSI_ID;
    _bIsSXMTrafficUp = FALSE;
    _enTunerState = sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_OFF;
    _enOperationMode = sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_INITIAL;
    _bRdbListActive = FALSE;
    _bSetDMI = FALSE;
    while(TRUE != _qLtn.empty())
    {
        _qLtn.pop();
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp destructor"));
}

/***************************************************************************
*
*FUNCTION:          vInitialize
*
*DESCRIPTION:       Called in cca context by sxmapp-manager
                    Initializes the helper class also
*
*PARAMETERS:        None 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vInitialize"));

    /* call vInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vInitialize();
    /* Initialize all the helper class for sxm traffic */
    fc_sxm_tclTrafficData::instance();

    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vInitialize"));
}

/****************************************************************************
*
*FUNCTION:          vDeInitialize
*
*DESCRIPTION:       Called in cca context by sxmapp-manager
                    DeInitializes the helper class also.
*
*PARAMETERS:        None 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vDeInitialize"));
    /* call vDeInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vDeInitialize();

    /* Deinit of all other stuff starts from here */
    OSAL_DELETE fc_sxm_tclTrafficData::instance();
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vDeInitialize"));
}

/****************************************************************************
*
*FUNCTION:          vDispatchMsgFromQ
*
*DESCRIPTION:       Called by the thread-function.
                    For each message that shall be handled put and entry here.

*PARAMETERS:        *poThreadMsg: msg need to communicate 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vDispatchMsgFromQ(
                                       fc_sxm_tclMessage const *poThreadMsg)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vDispatchMsgFromQ()"));
    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) 
    {
        /* use macro SXM_MSGQ_DISPATCH  for each message 
           specific for traffic */

        /* method-start messages from cca-service */
    	SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgTrafficMStartSetRdbIdWishList,fc_sxm_tclTrafficApp::instance());
        /* message from SMS call back to application thread */
    	SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgDataPayLoad,fc_sxm_tclTrafficApp::instance());

  

        default:
            ETG_TRACE_USR4(("vDispatchMsgFromQ():unknown u32Action=0x%x",
                                                                u32Action));
            break;         
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vDispatchMsgFromQ()"));
}


/*****************************************************************************
*
*FUNCTION:     vOnFiServiceStateChanged
*
*DESCRIPTION:  handles the service state received from frame work

*PARAMETERS:        midw_ext_fi_tcl_SxmDataSrvState const &oFiState 
*
*RETURNVALUES:      None
*                    
******************************************************************************/

tVoid fc_sxm_tclTrafficApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState)
{
        ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vOnFiServiceStateChanged,"
                                              "State = %d",oFiState.Status.enType));
        /* check for the service state and based on the service state notify to the client */
		switch(oFiState.Status.enType)
		{
				case midw_ext_fi_tcl_e8_SxmDataServiceStatus::FI_EN_UNKNOWN:
				{
               _bSetDMI = FALSE;
               _bIsSXMTrafficUp = FALSE;
               _enTunerState = sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_ON;
               _enOperationMode = sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_CHN_UNAVAILABLE;
                  /* update the state and operation mode to CCA */
               vPostStateAndOperationMode();

               bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_UNKNOWN);
				   break;
				}
				case midw_ext_fi_tcl_e8_SxmDataServiceStatus::FI_EN_NOT_AVAILABLE:
				{
					_bSetDMI = FALSE;/* for ticket NIKAI2-888 */
					_bIsSXMTrafficUp = FALSE;
					/* set the tunerstate and operationmode */
               _enTunerState =
                     sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_ON;
               _enOperationMode =
                     sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_CHN_UNAVAILABLE;
               /* update the state and operation mode to CCA */
               vPostStateAndOperationMode();

               bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_NOTAVAILABLE);
					break; 
				}
				case midw_ext_fi_tcl_e8_SxmDataServiceStatus::FI_EN_SUBSCRIBED:
				{
					/* In case of ready state, set the tunerstate
					 operation mode and update to cca */
					_bIsSXMTrafficUp = TRUE;
					_enTunerState = 
               		sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_ON;
					_enOperationMode = 
               		sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_FULLY_OPERABLE;
					/* update the state to tima */
					vPostStateAndOperationMode();
					/* if the Rdbid List is there but DMI is not set,
					then get the LTN and set the DMIs */
					if ( _bRdbListActive && (!_bSetDMI))
					{
						trTrafficRdbidList rTrafficRdbidList;
						fc_sxm_tclTrafficData::instance()->vGetRdbidList(
						                           rTrafficRdbidList);
						/* set the DMIs based on the location id */
						bSetDmi(&rTrafficRdbidList);
					}
               bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_OK);

					break;
				 }
				 case midw_ext_fi_tcl_e8_SxmDataServiceStatus::FI_EN_NOT_SUBSCRIBED:
				 {
					 _bSetDMI = FALSE;/* for ticket NIKAI2-888 */
					 _bIsSXMTrafficUp = FALSE;
                 /* set the tunerstate and operationmode */
                _enTunerState =
                     sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_ON;
                  _enOperationMode =
                     sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_CHN_UNAUTHORIZED;
                 /* update the state and operation mode to CCA */
                  vPostStateAndOperationMode();

                  bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_NOTSUBSCRIBED);
					 break;
             }
				 default:
				 	break;
		}				 
        
        ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vOnFiServiceStateChanged"));
}

/****************************************************************************
*
*FUNCTION:          vClearServiceData
*
*DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.
                    This is a overridden method from the data app.
*PARAMETERS:
*
*RETURNVALUES:      None
*
******************************************************************************/
/*
   Method to be Overriden by each Service
   to clear Service specifi internal data
*/
tVoid fc_sxm_tclTrafficApp::vClearServiceData()
{
	ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vClearServiceData"));
	fc_sxm_tclTrafficData::instance()->vClearTrafficServiceData();
	_u16Dsi = FC_SXM_DATASERVICE_TRAFFIC_DSI_ID;
	_bIsSXMTrafficUp = FALSE;
	_enTunerState = sxm_tcl_e8_TunerState::FI_EN_TUNER_STATE_OFF;
	_enOperationMode = sxm_tcl_e8_OperationMode::FI_EN_TUNER_OP_MODE_INITIAL;
	_bRdbListActive = FALSE;
	_bSetDMI = FALSE;
	while(TRUE != _qLtn.empty())
	{
	    _qLtn.pop();
	}
	ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vClearServiceData"));

}
/****************************************************************************
*
*FUNCTION:          vProcess
*
*DESCRIPTION:       handle the parsing fc_sxm_trMsgDataPayLoad 

*PARAMETERS:        *prMsg: contains the payload received from sms 
*
*RETURNVALUES:      None
*                    
******************************************************************************/
 
tVoid fc_sxm_tclTrafficApp::vProcess(fc_sxm_trMsgDataPayLoad const *prMsg)
{
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficApp::vProcess(fc_sxm_trMsgDataPayLoad)"));
    fc_sxm_trMsgDataPayLoad rMsg = *prMsg;
    
#ifdef FC_SXM_TRAFFIC_PROCESS_ALL_BIT_FLAG
    /* send the raw data for parsing */
    fc_sxm_tclTrafficData::instance()->vParseAndStorePayloadData(rMsg);
#else
    fc_sxm_tclTrafficData::instance()->vParsePayloadData(rMsg);

#endif

    /* free the memory */
    if ( OSAL_NULL != rMsg.u8PayloadData)
    {
       OSAL_DELETE [] (rMsg.u8PayloadData);
       rMsg.u8PayloadData = OSAL_NULL;
    }

    ETG_TRACE_USR4(("<----fc_sxm_tclTrafficApp::vProcess(fc_sxm_trMsgDataPayLoad)"));
    
}

/****************************************************************************
*
*FUNCTION:          bParseServiceSpecificPayloadHeader
*
*DESCRIPTION:       Parse the header of the payload, to identify the correct
                    and extract the 

*PARAMETERS:        None
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tBool fc_sxm_tclTrafficApp::bParseServiceSpecificPayloadHeader( OSAL_BUFFER_HDL pvPayload,
                                      fc_sxm_trMsgDataPayLoad &rTrafficPayloadMsg)
{
    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficApp::vParsePayloadData"));
    /* Local variables to check the payload */
    tU8 u8AllMessagesTXed = FALSE,
    u8ContainsIncident = FALSE,
    u8ContainsSF = FALSE;
    tU8 u8TrafficSyncByte = 0,
    u8AlertCMsgCount = 0;
    size_t tNum = 0;
    tU8 u8Market = 0;
    tBool bIsValidPayLoad = TRUE;
  
    /* check the size of the payload . TBD- removed later */
    tU32 u32PayLoadSize = (tU32)(OSAL.tBufferGetSize(pvPayload));
    ETG_TRACE_USR4(("bParseServiceSpecificPayloadHeader,"
                                          "Size = %d",u32PayLoadSize));
    do
    {
        /* 1st Byte of the SDTP payload is sync byte.Read the SDTP sync byte.
           Value of the sync byte should be 0x81 and the sync byte will 
           indicate the begining of the SDTP packet */
        tNum = OSAL.tBufferReadHead(pvPayload,
                          &u8TrafficSyncByte, SXM_TRAFFIC_SDTP_SYNC_BYTELEN);
        /* check for the correct SDTP packet */
        if (u8TrafficSyncByte != SXM_TRAFFIC_SDTP_SYNC_BYTE)
        {
            ETG_TRACE_USR4(("Sync Byte Error %d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* Read the 1st bit of 2nd byte of the SDTP packet.
           "All Messages Transmitted" flag.To check whether the SDTP packet
           contains the complete message with respect to the
           current market */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, &u8AllMessagesTXed,
                                0, SXM_TRAFFIC_SDTP_ALL_MESSAGES_TXED_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_ALL_MESSAGES_TXED_BITLEN)
        {
            ETG_TRACE_USR4(("TRAFFIC_SDTP_ALL_MESSAGES_TXED_BITLEN Error: %d",
                            tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* check whether the SDTP packet contains the complete msg 
           for the market or not */
        if (u8AllMessagesTXed)
        {
            /* contains the full message for the market */
            rTrafficPayloadMsg.bAllMsgTransmitted = TRUE;
            ETG_TRACE_USR4(("bAllTransmittedFlag True"));
        }
        /* Multiple SDTP packet for the market */
        else
        {
            rTrafficPayloadMsg.bAllMsgTransmitted = FALSE;
            ETG_TRACE_USR4(("bAllTransmittedFlag False"));
        }        
        /* Ignore next 5 bits of 2nd byte.This includes 3bits of RESERVED
           field and the M and W flag fields of 1 bit each */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, NULL, 0,
                              SXM_TRAFFIC_SDTP_RESERVED_PLUS_M_W_FLAGS_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_RESERVED_PLUS_M_W_FLAGS_BITLEN)
        {
            ETG_TRACE_USR4(("RESERVED Byte Error %d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* Read the 7th bit of 2nd byte. it is the S flag means
           to check if this SDTP packet contains Speed&FLow messages */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, &u8ContainsSF, 0,
                                               SXM_TRAFFIC_SDTP_S_FLAG_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_S_FLAG_BITLEN)
        {
            ETG_TRACE_USR4(("SXM_TRAFFIC_SDTP_S_FLAG_BITLEN Error %d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* set the Speed&Flow message flag */
        if(u8ContainsSF)
        {
            ETG_TRACE_USR4(("u8ContainsSF True:%d",u8ContainsSF));
            rTrafficPayloadMsg.bSpeedFlowMsg = TRUE;
        }        
        /* Read the 8th bit of 2nd byte. it is the I flag means
           to check if this SDTP packet contains Incident messages */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, &u8ContainsIncident, 0,
                                               SXM_TRAFFIC_SDTP_I_FLAG_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_I_FLAG_BITLEN)
        {
            ETG_TRACE_USR4(("SXM_TRAFFIC_SDTP_I_FLAG_BITLEN Error %d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* set the IncidentMsg flag */
        if(u8ContainsIncident)
        {
            ETG_TRACE_USR4(("bContainsIncident True"));
            rTrafficPayloadMsg.bIncidentMsg = TRUE;
        }
        /* Read the 3rd byte of SDTP packet. It contains no of Alert-C messages
           in the SDTP packet */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, &u8AlertCMsgCount, 0,
                                            SXM_TRAFFIC_SDTP_MSG_COUNT_BITLEN);
        ETG_TRACE_USR4(("No of Alert-C Messages:%d",u8AlertCMsgCount));

        if (tNum != SXM_TRAFFIC_SDTP_MSG_COUNT_BITLEN)
        {
            ETG_TRACE_USR4(("SXM_TRAFFIC_SDTP_MSG_COUNT_BITLEN Error:%d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* Read 1st 2 bits of 4th byte of SDTP packet.
           this indicates Alert-c flag. ignore this 2
                     bits as it is hard coded to 0b */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, NULL, 0,
                                             SXM_TRAFFIC_SDTP_MSG_TYPE_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_MSG_TYPE_BITLEN)
        {
            ETG_TRACE_USR4(("SXM_TRAFFIC_SDTP_MSG_TYPE_BITLEN Error:%d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* Read next 6 bits of 4th byte and 1st 2 bits of 5th byte.
           It indicates the LTN (Market ID) */
        tNum = OSAL.tBufferReadHead(pvPayload,&u8Market,
                                           SXM_TRAFFIC_SDTP_MARKET_ID_BYTELEN);
        rTrafficPayloadMsg.u8Ltn = u8Market;
        ETG_TRACE_USR4(("Market ID = %d",rTrafficPayloadMsg.u8Ltn));

        if ((tNum != SXM_TRAFFIC_SDTP_MARKET_ID_BYTELEN) ||
                                         (rTrafficPayloadMsg.u8Ltn >= SXM_NUM_TRAFFIC_MARKETS))
        {
            ETG_TRACE_USR4(("SXM_NUM_TRAFFIC_MARKETS Error:%d",tNum));
            bIsValidPayLoad = FALSE;
            break;
        }
        /* Read next 6 bits of 5th byte. these bits contains flags related to
           Regional, urban, national, international */
        tNum = OSAL.tBufferReadHeadBits(pvPayload, NULL, 0,
                                       SXM_TRAFFIC_SDTP_REST_OF_HEADER_BITLEN);
        if (tNum != SXM_TRAFFIC_SDTP_REST_OF_HEADER_BITLEN)
        {
            ETG_TRACE_USR4(("SXM_TRAFFIC_SDTP_REST_OF_HEADER_BITLEN Error:%d",
                                                                       tNum));
            bIsValidPayLoad = FALSE;
            break;
        }           
        
    } while (FALSE);

    /* if valid payload, extract the data */
    if (bIsValidPayLoad)
    {
      tU32 u32AlertCMsgSize = (tU32)(OSAL.tBufferGetSize(pvPayload));
      ETG_TRACE_USR4(("Alert-C MsgSize %d", u32AlertCMsgSize));

      rTrafficPayloadMsg.u32PayLoadDataSize = u32AlertCMsgSize;

      /* allocate the memory to store the data */
      rTrafficPayloadMsg.u8PayloadData = OSAL_NEW tU8[u32AlertCMsgSize];

      /* extract the msg from payload */
      if (rTrafficPayloadMsg.u8PayloadData != OSAL_NULL)
      {
         tNum = OSAL.tBufferReadHead(pvPayload,
         (rTrafficPayloadMsg.u8PayloadData), u32AlertCMsgSize);

         ETG_TRACE_USR4(("Alert-C Msgs Copied from SMS %d", tNum));
      }
      else
      {
         ETG_TRACE_USR4(("Memory Allocation Failed to copy Alert-C Msg"));
         bIsValidPayLoad =  FALSE;
      }
    }    

    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficApp::vParsePayloadData"));
    return bIsValidPayLoad;
}

/****************************************************************************
*
*FUNCTION:          vStartSmsService
*
*DESCRIPTION:       handle command from sxmAppManager to start the service

*PARAMETERS:        None
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vStartSmsService()
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::bStartService"));

    /* Check if the traffic service is already started */
    if (_hTraffic != DATASERVICE_MGR_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Traffic service already started!"));
        return;
    }

    /* Start the traffic service */
    _hTraffic = DATA.hStart(FC_SXM_SRH_DRIVER_NAME,       /* SRH driver name */
                            _u16Dsi,                          /* DSI for traffic service */
                            FALSE,                               /* Disable all the DMIs */
                            DATASERVICE_EVENT_ALL,
                            fc_sxm_tclDataApp::cb_vDataGenericServiceEventCallback, /* Generic service callback */
                            (tVoid*)this);          /* argument to the call back */

    /* check if the sms returns invalid object */
    if (_hTraffic == DATASERVICE_MGR_INVALID_OBJECT)
    {
       ETG_TRACE_USR4(("Could not start Traffic service"));
       return;
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::bStartService"));
   
}


/****************************************************************************
*
*FUNCTION:          vProcess:fc_sxm_trMsgTMCDataTimer
*
*DESCRIPTION:       This is to maintain the timer, to send TMC data

*PARAMETERS:        fc_sxm_trMsgTMCDataTimer *prMsg 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/
tVoid fc_sxm_tclTrafficApp::vProcessTimer(fc_sxm_trMsgTMCDataTimer const *prMsg)
{
    (tVoid)prMsg;
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vProcess"
                                      "(fc_sxm_trMsgTMCDataTimer *prMsg)"));

    /* Stop the timer and notify time out  */
    _mTMCDataUpdateTimer.vStop();
    vTmcdataTimerNotifyTimeOut();

    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vProcess"
                                      "(fc_sxm_trMsgTMCDataTimer *prMsg)"));
}

/****************************************************************************
*
*FUNCTION:          vProcess:fc_sxm_trMsgCmdAppTtfisCmd
*
*DESCRIPTION:       This is to handle the TTFis commands
                    This function is to test the methodstart and
                    method results properties.

*PARAMETERS:        fc_sxm_trMsgCmdAppTtfisCmd *prMsg 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/


tVoid fc_sxm_tclTrafficApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
    fc_sxm_tenTtfisCmdsTraffic enMsgCode=
              (fc_sxm_tenTtfisCmdsTraffic)prMsg->u8MsgCode;
    ETG_TRACE_USR2(("--->fc_sxm_tclTrafficApp::vProcess"
                               "(fc_sxm_trMsgCmdAppTtfisCmd) code=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsTraffic, enMsgCode)));

    /* to handle the commands from TTFis
       to test the method start and method result */
    switch (enMsgCode) {
        case fc_sxm_enTtfisCmdsTraffic_METHODSTART_WISHLIST:
        {
           sxm_tcl_MsgFID_TRAFFIC_S_SetRdbIdMethodStart oMStart;
           sxm_tcl_TrafficRdbidValue oTunerRdbid;
           oTunerRdbid.u8RdbidLTN = prMsg->au8Data[0];
           oTunerRdbid.u16Bsa = (tU16)SXM_GET_U16(&prMsg->au8Data[1]);
           oTunerRdbid.u8RdbidCC = prMsg->au8Data[3];

           oMStart.TunRDBID_Value.push_back(oTunerRdbid);

           fc_sxm_vSendSelfMessage(oMStart);
        }
        break;
        default:
            ETG_TRACE_ERR(("fc_sxm_tclTrafficApp::vProcess"
                           "(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                           ETG_CENUM(fc_sxm_tenTtfisCmdsTraffic, enMsgCode)));
            break;
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vProcess"
                                          "(fc_sxm_trMsgCmdAppTtfisCmd)"));
                                                    
}

/****************************************************************************
*
*FUNCTION:          vProcess:fc_sxm_trMsgCmdDataAppSignalState
*
*DESCRIPTION:       This is to handle the signalstate

*PARAMETERS:        *prMsg 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/
tVoid fc_sxm_tclTrafficApp::vProcessGenericMsg(fc_sxm_trMsgCmdDataAppSignalState const *prMsg)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::fc_sxm_trMsgCmdDataAppSignalState eSignalState=%d",
                         ETG_CENUM(SIGNAL_STATE_ENUM, prMsg->eSignalState)));

    if(SIGNAL_STATE_ACQUIRING == prMsg->eSignalState)
    {
       _bIsSXMTrafficUp = FALSE;
       bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_NOSIGNAL);
       ETG_TRACE_USR4(("Clear the Data on No Signal"));
       fc_sxm_tclTrafficData::instance()->vHandleNoSignal();
    }
    else if(SIGNAL_STATE_ANTENNA_ERROR == prMsg->eSignalState)
    {
        _bIsSXMTrafficUp = FALSE;
        bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_NOANTENNA);
        fc_sxm_tclTrafficData::instance()->vHandleNoSignal();
    }
    else if(SIGNAL_STATE_UNKNOWN == prMsg->eSignalState)
    {
        _bIsSXMTrafficUp = FALSE;
        bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_UNKNOWN);
        fc_sxm_tclTrafficData::instance()->vHandleNoSignal();
    }
    else
    {
       _bIsSXMTrafficUp = TRUE;
       bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_OK);
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::fc_sxm_trMsgCmdDataAppSignalState"));
    
}

/****************************************************************************
*
*FUNCTION:          fc_sxm_trMsgCmdDataAppAntennaState
*
*DESCRIPTION:       This is to handle the antenna state

*PARAMETERS:        *prMsg
*
*RETURNVALUES:      None
*
*****************************************************************************/
tVoid fc_sxm_tclTrafficApp::vProcessGenericMsg(fc_sxm_trMsgCmdDataAppAntennaState const *prMsg)
{
   ETG_TRACE_USR4(("fc_sxm_tclTrafficApp::vProcessGenericMsg(fc_sxm_trMsgCmdDataAppAntennaState), AntennaState=%d",
            ETG_CENUM(ANTENNA_STATE_ENUM, prMsg->rAntennaState)));
   if((ANTENNA_STATE_NOT_DETECTED == prMsg->rAntennaState) ||
      (ANTENNA_STATE_SHORTED == prMsg->rAntennaState))
    {
       _bIsSXMTrafficUp = FALSE;
       bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_NOANTENNA);
    }
    else if((ANTENNA_STATE_UNKNOWN == prMsg->rAntennaState) ||
            (ANTENNA_STATE_UNSUPPORTED == prMsg->rAntennaState))
    {
       _bIsSXMTrafficUp = FALSE;
       bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_UNKNOWN);
    }
    else
    {
       _bIsSXMTrafficUp = TRUE;
       bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_OK);
    }
}

/*****************************************************************************
*
*FUNCTION:          vProcess method start for setrdbidwishlist
*
*DESCRIPTION:       handle command from sxmAppManager to set the rdbidiwshlist
*                   requested from fc_tima

*PARAMETERS:        fc_sxm_trMsgTrafficMStartSetRdbIdWishList *prMsg 
*
*RETURNVALUES:      None
*                    
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vProcess(
                       fc_sxm_trMsgTrafficMStartSetRdbIdWishList const *prMsg)
{

    ETG_TRACE_USR4(("--> fc_sxm_tclTrafficApp::fc_sxm_trMsgTrafficMStartSetRdbIdWishList"));

    if (OSAL_NULL != prMsg)
    {
        /* request stored */
    	fc_sxm_trMsgTrafficMStartSetRdbIdWishList oSetRdbidWishList = *prMsg;
        tU32 u32ListSize = 0;
        trTrafficRdbidList rTrafficRdbidList;

        u32ListSize = (tU32)(oSetRdbidWishList.oFiMsg.TunRDBID_Value.size());
        ETG_TRACE_USR4(("oSetRdbidWishList Size=%d ",
                                                      u32ListSize));      
        /* size of the rdbid list */
        rTrafficRdbidList.u8ListSize = (tU8)u32ListSize;
        for(tU32 u32Index = 0; u32Index < u32ListSize; ++u32Index)
        {
            ETG_TRACE_USR4(("TMCCountryCode=%u TableNo=%u "
              "BSA=%d ServiceID=%u Encrypted=%d ",
				  		oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u8RdbidCC, 
				  		oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u8RdbidLTN, 
				  		oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u16Bsa, 
				  		oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u8RdbidSID, 
				  		oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].bEncFlag));

            /* fill the rdbid values to rdbid list */
            rTrafficRdbidList.aoTrafficRdbid[u32Index].u8LTN =
                  oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u8RdbidLTN;
            rTrafficRdbidList.aoTrafficRdbid[u32Index].u16BSA =
                  oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u16Bsa;
            rTrafficRdbidList.aoTrafficRdbid[u32Index].u8CountryCode =
                  oSetRdbidWishList.oFiMsg.TunRDBID_Value[u32Index].u8RdbidCC;
            
         }/* end of for loop */    
   
            ETG_TRACE_USR4(("Rdbidwishlist MethodResult"));

            sxm_tcl_MsgFID_TRAFFIC_S_SetRdbIdMethodResult oRdbIdWishListMethodResult;

            /* Set the RdbidList to post the RdbidactualList */
            fc_sxm_tclTrafficData::instance()->vSetRdbid(&rTrafficRdbidList);
            /* flag to trac the Rdbid List set */
            _bRdbListActive = TRUE;
             /* if traffic is in ready state set the DMIs
            based on the LTN present in Rdbid List*/
            if (TRUE == _bIsSXMTrafficUp)
            {
                /* set the DMIs based on the LTN */
                if(bSetDmi(&rTrafficRdbidList))
                {
                	/* once DMI is set send the actual list to TIMA */
                	bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::FI_EN_OK);
                }
            } 

            /* fill the Rdbid data to send in method result */
            for(tU32 u32Index = 0; u32Index < u32ListSize; ++u32Index)
            {
                sxm_tcl_TrafficRdbidValue oTunerRdbid;
                oTunerRdbid.u8RdbidLTN = 
                          rTrafficRdbidList.aoTrafficRdbid[u32Index].u8LTN;
                oTunerRdbid.u16Bsa = 
                          rTrafficRdbidList.aoTrafficRdbid[u32Index].u16BSA;
                oTunerRdbid.u8RdbidCC = 
                          rTrafficRdbidList.aoTrafficRdbid[u32Index].u8CountryCode;
                oTunerRdbid.u8RdbidSID = SXM_TRAFFIC_SERVICE_ID;
                /* set the encripted field as FALSE as the data is not encripted */
                oTunerRdbid.bEncFlag = FALSE;
                oRdbIdWishListMethodResult.TunRDBID_Value.push_back(oTunerRdbid);

             }/* end of for loop */

            /* send the answer to the client using addressing-infomation: */
            if (AIL_EN_N_NO_ERROR == 
                           fc_sxm_tclTrafficService::instance()->enSendFiMessage(
                                                      prMsg->rAdressing, 
                                                      oRdbIdWishListMethodResult))
            {
                /* post the rdbidactuallist */
            	//send the actual list once the DMI is set
                //bPostRdbIdActualList();
                ETG_TRACE_COMP(("enSendFiMessage: RDB_ID WishList")); 
            }
            else
            {
                ETG_TRACE_ERR(("Failed to send method result for Wishlist"));
            }
    }

    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::fc_sxm_trMsgTrafficMStartSetRdbIdWishList()"));
    
}

/***************************************************************************
*
*FUNCTION:          bSetDmi
*
*DESCRIPTION:       set the DMIs in SMS based on the LTN received from FC_TPEG

*PARAMETERS:        trTrafficRdbidList const *rTrafficRdbidList 
*
*RETURNVALUES:      True or False
*                    
***************************************************************************/
tBool fc_sxm_tclTrafficApp::bSetDmi(trTrafficRdbidList const  *rTrafficRdbidList)
{

    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::bSetDmi"));
    SXM_DMI tIncidentDMIToSet, tSpeedFlowDMIToSet;
    TRAFFIC_MARKET u8Market;
    //SMSAPI_RETURN_CODE_ENUM eReturnCode;
    BOOLEAN bAdd = TRUE;
    tBool bExistingDMI = FALSE;
    tBool bReturnValue = FALSE;
    trLTN rLtn;

    if (OSAL_NULL != rTrafficRdbidList &&
                             DATASERVICE_MGR_INVALID_OBJECT != _hTraffic)
    {
        trLTN rLtnNew;
        /* extract the market id send by FC_TPEG */
        tU32 u32ListSize = rTrafficRdbidList->u8ListSize;
        /* this Ltn queue is required to keep track of setting DMIs
           and disabling DMIs */
        if (TRUE != _qLtn.empty())
        {
            ETG_TRACE_USR4(("vSetDmi LTN queue AVL"));
            rLtn = _qLtn.front();
            _qLtn.pop();
        }
        else /* for new wishlist */
        {
            rLtn.u16Size = 0;
        }
        
        /* store the no of LTN in the queue */
        rLtnNew.u16Size = (tU16)u32ListSize;
        for(tU32 u32Index = 0; u32Index < u32ListSize; u32Index++)
        {
            u8Market = rTrafficRdbidList->aoTrafficRdbid[u32Index].u8LTN;
            
            /* check for the market id. If market id is valid process it,
                else process the next market id */
            if ( TRAFFIC_INVALID_MARKET != u8Market)
            {

                for(tU32 u32LtnIndex = 0; u32LtnIndex < rLtn.u16Size;
                                                       u32LtnIndex++)
                {
                    /* if the Ltn is already present in the queue and 
                       the DMI is already set then set it 0 for internal check */
                    if (u8Market == (rLtn.u8Ltn[u32LtnIndex]) ) 
                    {
                        rLtn.u8Ltn[u32LtnIndex] = 0;
                        bExistingDMI = TRUE;
                        break;
                    }                 
                }/* end of for loop */

                /* if the DMI is not set,Set the DMI */
                if (!bExistingDMI)
                {
                    /* calculate the DMI for Incident and Speed & Flow message */
                    tIncidentDMIToSet = 
                                  (tU16)(SXM_TRAFFIC_BASE_INCIDENT_PSI + u8Market);
                    tSpeedFlowDMIToSet = 
                    		(tU16)(SXM_TRAFFIC_BASE_SPEED_AND_FLOW_PSI + u8Market);

                    ETG_TRACE_USR4(("bSetDmi IncidentDMI=%d SpeedFlowDMI=%d",
                                       tIncidentDMIToSet,tSpeedFlowDMIToSet));
                    /* set the DMI for Incident Message */
                    if (bModifyFilter(bAdd, tIncidentDMIToSet))
                    {
                        ETG_TRACE_USR4(("vSetDmi tIncidentDMIToSet"
						               "SMSAPI_RETURN_CODE_SUCCESS"));
                        /* set the DMI for Speed&Flow Message */
                        bModifyFilter(bAdd, tSpeedFlowDMIToSet);
                        /* store in the structure to add in the queue */
                        rLtnNew.u8Ltn[u32Index] = u8Market;

                    }
                    else
                    {
                       // ETG_TRACE_USR4(("vSetDmi"
                         //              "tIncidentDMIToSet Failed:%d",eReturnCode));
                        /* decrease the size of the queue by 1,
                           as it is not going to add in the actuallist */
                        --(rLtnNew.u16Size);
                    }
                    bExistingDMI = FALSE;
                }
                else /* already exist , store in the structure */
                {
                    bExistingDMI = FALSE;
                    rLtnNew.u8Ltn[u32Index] = u8Market;
                }
            }
            else /* in case of invalid market id */
            {
                /* check the next market id */
            	--(rLtnNew.u16Size);
                ETG_TRACE_USR4(("TRAFFIC_INVALID_MARKET"));
            }
        }/* end of for loop */

        /* store the new LTN */
        _qLtn.push(rLtnNew);
        //if DMI is set then set the return value to true.
        if ( rLtnNew.u16Size )
		  {
			 bReturnValue = TRUE;
		  }

        /* if the DMI is already set, and
           it is not there in new wishlist,then disable the DMI */
        for(tU32 u32Index = 0; u32Index < (rLtn.u16Size); u32Index++)
        {
            ETG_TRACE_USR4(("Disable DMI"));
            /* not there in the new list */
            if (0 != rLtn.u8Ltn[u32Index])
            {
                 u8Market = rLtn.u8Ltn[u32Index];
                 /* calculate the DMI for Incident and Speed & Flow message */
                 tIncidentDMIToSet =
                		 (tU16)(SXM_TRAFFIC_BASE_INCIDENT_PSI + u8Market);
                 tSpeedFlowDMIToSet = 
                		 (tU16)(SXM_TRAFFIC_BASE_SPEED_AND_FLOW_PSI + u8Market);
                 /* disable the DMI for Incident Message */

                 if (bModifyFilter(FALSE, tIncidentDMIToSet))
                 {
                     ETG_TRACE_USR4(("tIncidentDMIToDisable SMSAPI_RETURN_CODE_SUCCESS:%d",
                                                                   u8Market));
                     /* disable the DMI for Speed&Flow Message */
                     if (bModifyFilter(FALSE, tSpeedFlowDMIToSet))
                     {
                         ETG_TRACE_USR4(("tS&FDMIToDisable SMSAPI_RETURN_CODE_SUCCESS:%d",
                                                                  u8Market));
                     }
                 }
                 else
                 {
                     ETG_TRACE_ERR(("Failed to set DMI"));
                 }
                 /* erase the stored payload data from the disable market id */
                 fc_sxm_tclTrafficData::instance()->vClearOldData(u8Market);
             }
            else
            {
                ETG_TRACE_ERR(("Check for next LTN"));
            }
        }/* end of for loop */
        _bSetDMI = TRUE;
    }
    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::bSetDmi"));
    return bReturnValue;

}

/***************************************************************************
 * FUNCTION:          bPostRdbIdActualList
 *
 * DESCRIPTION:       Post actual rdbid list to TPEG.

* PARAMETER:         tVoid.

* RETURNVALUE:       tBool:returns true or false
*****************************************************************************/

tBool fc_sxm_tclTrafficApp::bPostRdbIdActualList(midw_fi_tcl_e8_TunerReceptionState::tenType enReceptionState)
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::bPostRdbIdActualList "));

    trTrafficRdbidList rTrafficRdbidList;
    rTrafficRdbidList.u8ListSize = 0;
    sxm_tclMsgFID_TRAFFIC_G_RdbIdActualListStatus oRdbidActualListStatus;
	 oRdbidActualListStatus.ListOfRdbIds.clear();
    /* if sxm traffic is up , update wish list as actual list to client */
    if (TRUE == _bIsSXMTrafficUp)
    {
      /* Send wish list as actual list
      get RDBid from traffic data class */
      //vSetRdbListStatus(TRUE); TBD
      /* Get actual list from traffic filter. */
      fc_sxm_tclTrafficData::instance()->vGetRdbidList(rTrafficRdbidList);
        
		/* update the actuallist to send to FC_TPEG */
		trLTN rLtn;
		tU32 u32ListSize = rTrafficRdbidList.u8ListSize;
		/* check the LTN present in the queue */
		if (TRUE != _qLtn.empty())
		{
			rLtn = _qLtn.front();
		}
		else /* send actual list as the emptylist */
		{
			rLtn.u16Size = 0;
		}
		/* when all LTNs set successfully in SXM */
		if (rLtn.u16Size == u32ListSize)
		{
			ETG_TRACE_USR4(("Stored LTNs == Set DMI LTNs"));

			for(tU32 u32Index = 0; u32Index < u32ListSize; ++u32Index)
			{
				sxm_tcl_TrafficRdbidValue oTunerRdbid;
				oTunerRdbid.u8RdbidLTN =
						  rTrafficRdbidList.aoTrafficRdbid[u32Index].u8LTN;
				oTunerRdbid.u16Bsa =
						  rTrafficRdbidList.aoTrafficRdbid[u32Index].u16BSA;
				oTunerRdbid.u8RdbidCC =
						  rTrafficRdbidList.aoTrafficRdbid[u32Index].u8CountryCode;
				oTunerRdbid.u8RdbidSID = SXM_TRAFFIC_SERVICE_ID;

				/* set the encripted field as FALSE as the data is not encripted */
				oTunerRdbid.bEncFlag = FALSE;

				oRdbidActualListStatus.ListOfRdbIds.push_back(oTunerRdbid);

			 }/* end of for loop */
		}
		else /* stored LTN is not same with set DMI LTNs */
		{
			ETG_TRACE_USR4(("Stored LTNs != Set DMI LTNs"));

			for(tU32 u32Index = 0; u32Index < u32ListSize; ++u32Index)
			{
				sxm_tcl_TrafficRdbidValue oTunerRdbid;
				tU8 u8Ltn = rLtn.u8Ltn[u32Index];
				oTunerRdbid.u8RdbidLTN =
							 rLtn.u8Ltn[u32Index];
				oTunerRdbid.u16Bsa =
							 rTrafficRdbidList.aoTrafficRdbid[u32Index].u16BSA;
				oTunerRdbid.u8RdbidCC =
					fc_sxm_tclTrafficData::instance()->u8GetCountryCode(u8Ltn);
				oTunerRdbid.u8RdbidSID = SXM_TRAFFIC_SERVICE_ID;
				/* set the encripted field as FALSE as the data is not encripted */
				oTunerRdbid.bEncFlag = FALSE;
				oRdbidActualListStatus.ListOfRdbIds.push_back(oTunerRdbid);
			}/* end of for loop */
		}

    }
    else
    {
        rTrafficRdbidList.u8ListSize = 0;
    }

    oRdbidActualListStatus.ReceptionState.enType = enReceptionState;

    /* post the actuallist to CCA */
	 fc_sxm_tcl_trTrafficPropertyRdbidActualList oRdbidActualList;
	 oRdbidActualList.oFiMsg = oRdbidActualListStatus;
	 fc_sxm_tclTrafficProperties::instance()->
            oUpdateRdbidActualList.vSetAndNotify(oRdbidActualList);

    ETG_TRACE_USR4(("<---- fc_sxm_tclTrafficApp::bPostRdbIdActualList "));
    return TRUE;
}

/*****************************************************************************
* FUNCTION:          vPostStateAndOperationMode
* 
* DESCRIPTION:       Post state and operation mode.

* PARAMETER:         tVoid.

* RETURNVALUE:       tVoid
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vPostStateAndOperationMode(tVoid) const
{

    ETG_TRACE_USR4(("---> fc_sxm_tclTrafficApp::vPostStateAndOperationMode"));

    fc_sxm_tcl_trTrafficPropertyTunerStateAndMode oTunerStateandMode;
    oTunerStateandMode.oFiMsg.TunerState.enType = _enTunerState; 
    oTunerStateandMode.oFiMsg.OperationMode.enType = _enOperationMode;
    /* send the status to FC_TPEG */
    fc_sxm_tclTrafficProperties::instance()->
                oUpdateTunerStateAndMode.vSetAndNotify(oTunerStateandMode);
    
    ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficApp::vPostStateAndOperationMode"));
}

/*****************************************************************************
* FUNCTION:          vPostTmcData
* 
* DESCRIPTION:       This function will send the TMC data to FC_TIMA.

* PARAMETER:         tVoid.

* RETURNVALUE:       tVoid
*****************************************************************************/

tVoid fc_sxm_tclTrafficApp::vPostTmcData(tVoid)
{
   ETG_TRACE_USR4(("---> fc_sxm_tclTrafficApp::vPostTmcData"));

   fc_sxm_tcl_trTrafficPropertyTMCData oTmcData;
   tBool bDataForRdbIdComplete = FALSE;

   /* Get the TMC data */
   fc_sxm_tclTrafficData::instance()->vGetTMCData(oTmcData.oFiMsg,bDataForRdbIdComplete);

   ETG_TRACE_USR4(("DataForRdbIdComplete = %d", bDataForRdbIdComplete));

   /* Set the data complete flag */
   fc_sxm_tclTrafficData::instance()->vSetTmcDataCompleted(bDataForRdbIdComplete);

   /* set the TMC data for TPEG */
   fc_sxm_tclTrafficProperties::instance()->
                                oUpdateTmcData.vSet(oTmcData);
   /* notify the TMC data to FC_TPEG */
   fc_sxm_tclTrafficProperties::instance()->
                                oUpdateTmcData.vNotify();
   vStartTMCDataTimer();
              
   ETG_TRACE_USR4(("<--- fc_sxm_tclTrafficApp::vPostTmcData")); 
}

/*****************************************************************************
* FUNCTION:          vStartTMCDataTimer: 
* 
* DESCRIPTION:       To check the timer status

* PARAMETER:         tVoid.

* RETURNVALUE:       tVoid
*****************************************************************************/
tVoid fc_sxm_tclTrafficApp::vStartTMCDataTimer(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclTrafficApp::vStartTMCDataTimer()"));

    _mTMCDataUpdateTimer.vStart(FC_SXM_TRAFFIC_DEFAULT_TMCDATA_UPDATE_PERIOD);
}

/*****************************************************************************
* FUNCTION:          bIsTmcDataTimerRunning: 
* 
* DESCRIPTION:       To check the timer status

* PARAMETER:         tVoid.

* RETURNVALUE:       tBool
*****************************************************************************/ 
tBool fc_sxm_tclTrafficApp::bIsTmcDataTimerRunning(tVoid) const
{
    ETG_TRACE_USR4(("fc_sxm_tclTrafficApp::bIsTmcDataTimerRunning()"));

    return(_mTMCDataUpdateTimer.bIsRunning());
}

/*****************************************************************************
* FUNCTION:          vTmcdataTimerNotifyTimeOut: 
* 
* DESCRIPTION:       to update the data when timed out

* PARAMETER:         tVoid.

* RETURNVALUE:       tBool
*****************************************************************************/ 
tVoid fc_sxm_tclTrafficApp::vTmcdataTimerNotifyTimeOut(tVoid) 
{
    ETG_TRACE_USR4(("--->fc_sxm_tclTrafficApp::vTmcdataTimerNotifyTimeOut()"));

    /* If there are any incomplete data for TPEG, then post it */
    if (!fc_sxm_tclTrafficData::instance()->bIsTmcDataCompleted())
    {
        ETG_TRACE_USR4(("vTmcdataTimerNotifyTimeOut posttmcdata"));
        vPostTmcData();
    }
    /* if data is completely sent to TPEG */
    else
    {
         ETG_TRACE_USR4(("vTmcdataTimerNotifyTimeOut IstmcdataReady"));
         fc_sxm_tclTrafficData::instance()->vReadyToSendTMC();
    }

    ETG_TRACE_USR4(("<---fc_sxm_tclTrafficApp::vTmcdataTimerNotifyTimeOut()"));
}


