/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_tabweather_app.cpp
* @brief       Implementation of handling HMI requests and functionalities of the
*              TabWeather service
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_tabweather.h"
#include "fc_sxm_tcl_tabweather_app.h"
#include "fc_sxm_tcl_tabweather_dsrl.h"
#include "fc_sxm_tcl_tabweather_list.h"

// Fractional bits used to represent the loc coordinates
#define WEATHER_LOC_COORD_FRACTIONAL_BITS (22)
#define FC_SXM_TAB_WEATHER_COORD (1 << WEATHER_LOC_COORD_FRACTIONAL_BITS)

//From _location_obj.h
#define FCSXM_TABW_MIN_LATITUDE (0 * 0xFFFF)
#define FCSXM_TABW_MAX_LATITUDE (90 * 0xFFFF)
#define FCSXM_TABW_MIN_LONGITUDE (-178 * 0xFFFF)
#define FCSXM_TABW_MAX_LONGITUDE (-30 * 0xFFFF)
typedef enum {
    fc_sxm_enTtfisCmdsTabweather_MStartGetTabweatherInfoList=1,
   fc_sxm_enTtfisCmdsTabweather_MStartAddFavorite=2,
   fc_sxm_enTtfisCmdsTabweather_MStartRemoveFavorite=3,
   fc_sxm_enTtfisCmdsTabweather_MStartGetFavoriteInfo=4,
   fc_sxm_enTtfisCmdsTabweather_MStartSetListsConfig=5,
   fc_sxm_enTtfisCmdsTabweather_SetListMode=6,
    fc_sxm_enTtfisCmdsTabweather_LAST
} fc_sxm_tenTtfisCmdsTabweather;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_TABWEATHER_APP
#include "trcGenProj/Header/fc_sxm_tcl_tabweather_app.cpp.trc.h"
#endif


/*********************************************************************
 *FUNCTION:     fc_sxm_tclTabweatherApp
 *DESCRIPTION:  Constructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
/*
   peha: constructor: give it a name and provice the used sxmServiceId (not the ccaServiceId!!)
 */
fc_sxm_tclTabweatherApp::fc_sxm_tclTabweatherApp():
	fc_sxm_tclDataApp("sxmTabweatherApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_TABULAR_WEATHER, "weather", TR_CLASS_FC_SXM_TABWEATHER_APP),
       _hTabweather (WEATHER_SERVICE_INVALID_OBJECT),
       _poNearByDsrl (OSAL_NULL),
       _poNearDestDsrl (OSAL_NULL),
       _poAnotherLocationDsrl (OSAL_NULL),
       _poFavoritesDsrl (OSAL_NULL),
       _poNearByList (OSAL_NULL),
       _poNearDestList (OSAL_NULL),
       _poAnotherLocationList (OSAL_NULL),
       _poFavoritesList (OSAL_NULL),
		 _tLocIdForFav (0),
       _bSetAsFavFlag (FALSE),
		 _bGetFavInfoFlag (FALSE)

{
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp constructor"));
}

/*********************************************************************
 *FUNCTION:     fc_sxm_tclTabweatherApp
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclTabweatherApp::~fc_sxm_tclTabweatherApp()
{
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp destructor Entered"));

   /* reset pointers, cleanup will be done by framework */
   _poNearByDsrl=OSAL_NULL;
   _poNearDestDsrl=OSAL_NULL;
   _poAnotherLocationDsrl=OSAL_NULL;
   _poFavoritesDsrl=OSAL_NULL;
   _poNearByList=OSAL_NULL;
   _poNearDestList=OSAL_NULL;
   _poAnotherLocationList=OSAL_NULL;
   _poFavoritesList = OSAL_NULL;
   _hTabweather = WEATHER_SERVICE_INVALID_OBJECT;

	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp destructor Exited"));
}

/*********************************************************************
 *FUNCTION:     vInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vInitialize"));

    //Initialize the class members
   _hTabweather = WEATHER_SERVICE_INVALID_OBJECT;
   _poNearByDsrl=OSAL_NULL;
   _poNearDestDsrl=OSAL_NULL;
   _poAnotherLocationDsrl=OSAL_NULL;
   _poFavoritesDsrl=OSAL_NULL;
   _poNearByList=OSAL_NULL;
   _poNearDestList=OSAL_NULL;
   _poAnotherLocationList=OSAL_NULL;
   _poFavoritesList = OSAL_NULL;

   _tLocIdForFav = 0;
   _bSetAsFavFlag = FALSE;
   _bGetFavInfoFlag = FALSE;

    //Create the worker thread

    /* call vInitialize of base-class to start the worker thread */
    fc_sxm_tclBaseApp::vInitialize();

    fc_sxm_trTabweatherDSRLCfg rDsrlCfg;
    rDsrlCfg.enSortMethod=fc_sxm_enTabweatherSortMethod_NONE;
    rDsrlCfg.rFilterCfg.enWeatherMsgObjectType=(WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

    /*DSRL id - 1*/
	/* Get Capacity & Radius defined for TabWeather Nearby type */
    rDsrlCfg.u32Capacity= FC_SXM_TABWEATHER_DEVICE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_TABWEATHER_DEVICE_DSRL_RADIUS;
	 ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:: call vCreateDsrlAndList: NearBy Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Device, fc_sxm_tenListType_NearBy, &_poNearByDsrl, &_poNearByList, rDsrlCfg);

    /*DSRL id - 2*/
	/* Get Capacity & Radius defined for TabWeather NearDest type */
    rDsrlCfg.u32Capacity= FC_SXM_TABWEATHER_DESTINATION_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_TABWEATHER_DESTINATION_DSRL_RADIUS;
	 ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:: call vCreateDsrlAndList: NearDest Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Dest, fc_sxm_tenListType_NearDest, &_poNearDestDsrl, &_poNearDestList, rDsrlCfg);

    /*DSRL id - 3*/
	/* Get Capacity & Radius defined for TabWeather Favorites type */
    rDsrlCfg.u32Capacity= FC_SXM_TABWEATHER_FAVORITE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_TABWEATHER_FAVORITE_DSRL_RADIUS;
	 ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:: call vCreateDsrlAndList: Favorites Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Favorites, fc_sxm_tenListType_Favorites, &_poFavoritesDsrl, &_poFavoritesList, rDsrlCfg);

    /*DSRL id - 4*/
	/* Get Capacity & Radius defined for TabWeather Standard type */
    rDsrlCfg.u32Capacity= FC_SXM_TABWEATHER_STANDARD_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS;
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:: call vCreateDsrlAndList: Standard Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Standard, fc_sxm_tenListType_AnotherLocation, &_poAnotherLocationDsrl, &_poAnotherLocationList, rDsrlCfg);



	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:Nearby=%08x,Neardest=%08x,Favorites=%08x,Anotherlocation=%08x",
                     _poNearByList,
                     _poNearDestList,
                     _poFavoritesList,
		_poAnotherLocationList));

}

/*********************************************************************
 *FUNCTION:     vDeInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vDeInitialize"));
    /* call vDeInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vDeInitialize();

	vector<fc_sxm_trTabweatherDatabase,allocator<fc_sxm_trTabweatherDatabase> >::iterator weather_list_iter;
    /* Deinit of all other stuff starts from here */
   _poNearByDsrl=OSAL_NULL;
   _poNearDestDsrl=OSAL_NULL;
   _poAnotherLocationDsrl=OSAL_NULL;
   _poFavoritesDsrl = OSAL_NULL;

   _poNearByList=OSAL_NULL;
   _poNearDestList=OSAL_NULL;
   _poAnotherLocationList=OSAL_NULL;
   _poFavoritesList = OSAL_NULL;

   _tLocIdForFav = 0;
   _bSetAsFavFlag = FALSE;
   _bGetFavInfoFlag = FALSE;

	for(weather_list_iter=_TabweatherDBlist.begin();weather_list_iter!=_TabweatherDBlist.end();weather_list_iter++)
	{
		fc_sxm_trTabweatherDatabase const subEntry=*weather_list_iter;

		// Destroy the Weather Duplicate LOCATION_OBJECT
		if (subEntry.hLocation != LOCATION_INVALID_OBJECT)
		{
			LOCATION.vDestroy(subEntry.hLocation);
		}
		else
		{
			ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::Deinitialize - WeatherDB.hLocation invalid "));
		}
	}
}


/*********************************************************************
 *FUNCTION:     vDispatchMsgFromQ
 *DESCRIPTION:  this method is called by the thread-function.
   For each message that shall be handled put and entry here
 *PARAMETER:    fc_sxm_tclMessage
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {
ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vDispatchMsgFromQ: SID=%x action=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()),
                    poThreadMsg->u16GetActionOnly()));
    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) {
        // use macro SXM_MSGQ_DISPATCH  for each message specific for Tabweather
		//Previously used, For getting dataservice ready
		/*SXM_MSGQ_DISPATCH(fc_sxm_trMsgCmdDataAppStartService);*/
        /* method-start messages from cca-service */
    	//SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgDataSmsEvtServiceUp,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH(fc_sxm_trMsgDataSmsEvtServiceUp);
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTabweatherMStartAddFavorite,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTabweatherMStartRemoveFavorite,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTabweatherMStartGetFavoriteInfo,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgTabweatherMStartSetListsConfig,fc_sxm_tclTabweatherApp::instance());
    	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trPropertyTabweatherPSetListMode,fc_sxm_tclTabweatherApp::instance());

        default:
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                            u32Action));
            break;
    }
}

/*****************************************************************************
*FUNCTION:     bWeatherLocationsIterator
*DESCRIPTION:  To read all the locations from weather_ref.db to fc_sxm_trTabweatherDB list
*PARAMETER:    hLocation - A Valid LOCATION_OBJECT , pvIterateArg - Pointer to the caller function
*RETURNVALUE:  TRUE/FALSE
*****************************************************************************/
tBool fc_sxm_tclTabweatherApp::bWeatherLocationsIterator (LOCATION_OBJECT hLocation, tVoid* /*pvIterateArg*/)
{
	LOCID_OBJECT hLocID;
	OSAL_FIXED_OBJECT hLat,hLon;

	if(hLocation != LOCATION_INVALID_OBJECT)
	{
		fc_sxm_tclTabweatherApp::instance()->_trWeatherDB.hLocation = LOCATION.hDuplicate (hLocation);
	}
	else
	{
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::bWeatherLocationsIterator - Weather Location invalid" ));
		return FALSE;
	}
	// tID - DB LocID
	hLocID = LOCATION.hLocID(hLocation);
	if( hLocID != LOCID_INVALID_OBJECT )
	{
		fc_sxm_tclTabweatherApp::instance()->_trWeatherDB.un32locid = LOCID.tID(hLocID);
	}
	else
	{
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::bWeatherLocationsIterator - LOCID Object invalid"));
		return FALSE;
	}
   /* Calculate Lat and Lon for the provided Location Object */
	hLat = LOCATION.hLat(hLocation);
	fc_sxm_tclTabweatherApp::instance()->_trWeatherDB.f32latitue =  (tF32)OSAL_FIXED.n32Value(hLat) / (tF32)FC_SXM_TAB_WEATHER_COORD ; // Destination latitude in flat from DB format

	hLon = LOCATION.hLon(hLocation);
	fc_sxm_tclTabweatherApp::instance()->_trWeatherDB.f32longitude =  (tF32)OSAL_FIXED.n32Value(hLon) / (tF32)FC_SXM_TAB_WEATHER_COORD ; // Destination latitude in flat from DB format

	fc_sxm_tclTabweatherApp::instance()->_TabweatherDBlist.push_back(fc_sxm_tclTabweatherApp::instance()->_trWeatherDB);
	ETG_TRACE_USR4(("WEATHER SIZE: %d", fc_sxm_tclTabweatherApp::instance()->_TabweatherDBlist.size()));
	return TRUE;
}

/*********************************************************************
 *FUNCTION:     vCreateDsrlAndList
 *DESCRIPTION:  Create DSRL and List
 *PARAMETER:    fc_sxm_tenDSRLType, fc_sxm_tenListType, fc_sxm_tclTabweatherDSRL
 *              fc_sxm_trTabweatherDSRLCfg
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vCreateDsrlAndList(fc_sxm_tenDSRLType enDsrlType,
                                                  fc_sxm_tenListType enListType,
                                                  fc_sxm_tclTabweatherDSRL **ppoDsrl,
                                                  fc_sxm_trTabweatherList **ppoList,
																  fc_sxm_trTabweatherDSRLCfg rDsrlCfg)
{
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vCreateDsrlAndList: enDsrlType=%d enListType=%d",
                    ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType),
                    ETG_CENUM(fc_sxm_tenListType, enListType)
                    ));
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::construct dsrl"));
    /* create DSRL, dsrl will remain inactive while not configured*/
    fc_sxm_tclTabweatherDSRL *poDsrl;
    poDsrl=OSAL_NEW fc_sxm_tclTabweatherDSRL(enDsrlType);
    if (OSAL_NULL == poDsrl)
    {
       ETG_TRACE_ERR(("RETURNING from METHOD as MEMORY IS NOT ALLOTED"));
       return ;
    }

    /* register the dsrl */
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::poRegisterDsrl"));
    *ppoDsrl=poDsrl;
    poRegisterDsrl(poDsrl);

    /* create list*/
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::poTabweatherList Create"));
    fc_sxm_trTabweatherList *poTabweatherList=OSAL_NEW fc_sxm_trTabweatherList((tU32)enListType, enListType, this);
    if (OSAL_NULL == poTabweatherList)
    {
       ETG_TRACE_ERR(("RETURNING from METHOD as MEMORY IS NOT ALLOTED"));
       return;
    }

    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::poTabweatherList setDsrl"));
    poTabweatherList->vSetDsrl(poDsrl);

    /* configure the list */
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::poTabweatherList setCfg"));
    fc_sxm_trTabweatherListConfig rListCfg;
    rListCfg.rSortCfg.enSortMethod=fc_sxm_enTabweatherListSortMethod_NONE;
    rListCfg.rFilterCfg.enWeatherMsgObjectType=(WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;
    poTabweatherList->vSetCfg(rListCfg);

    *ppoList = poTabweatherList;

    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::poDsrl  vSetNextConfig"));
    vPatchDsrlConfig(enDsrlType, rDsrlCfg);
    poDsrl->vSetNextConfig(rDsrlCfg);

    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vCreateDsrlAndList: DONE"));
}


/*********************************************************************
 *FUNCTION:     vStartSmsService
 *DESCRIPTION:  handle command from sxmAppManager to start service (Mandatory)
 *PARAMETER:
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vStartSmsService()
{
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vStartSmsService"));

    /*Check if the Tabweather service is already started*/
    if (_hTabweather != WEATHER_SERVICE_INVALID_OBJECT)
    {
        ETG_TRACE_ERR(("Tabweather service already started!"));
        return;
    }

    /*Start the Tabweather service */
    _hTabweather = WEATHER.hStart(FC_SXM_SRH_DRIVER_NAME,
                                  DATASERVICE_EVENT_ALL,
                                  cb_vDataServiceEventCallback,
                                  (tVoid*)this, 
                                  DATASERVICE_OPTION_NONE);

   if (_hTabweather == WEATHER_SERVICE_INVALID_OBJECT)
   {
      ETG_TRACE_ERR(("Could not start Tabweather service"));
   }
   else
   {
      ETG_TRACE_USR4(("Start Tabweather service success"));
   }
}

/* all messages are handled via vProcess(tMsgType const *prMsg) */

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd (Mandatory)
 *PARAMETER:    fc_sxm_trMsgCmdAppTtfisCmd
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) {
    fc_sxm_tenTtfisCmdsTabweather enMsgCode=(fc_sxm_tenTtfisCmdsTabweather)prMsg->u8MsgCode;
    ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) code=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsTabweather, enMsgCode)));
    tU8 const *au8Data=prMsg->au8Data;

    switch (enMsgCode) {
        case fc_sxm_enTtfisCmdsTabweather_MStartGetTabweatherInfoList:
        {
			  /* This case handled to Get Tab weather info the selected list id*/
            midw_ext_sxm_tabweatherfi_tclMsgGetTabWeatherInfoListMethodStart oMStart;
            oMStart.ListId=au8Data[0];

            if (oMStart.ListId == (tU32)fc_sxm_tenListType_AnotherLocation)/*Another City usecase*/
            {
               oMStart.Location.Lat = (tS32)SXM_GET_U32(&au8Data[1]);
               oMStart.Location.Lon = (tS32)SXM_GET_U32(&au8Data[5]);
               oMStart.Location.Radius = FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS;
            }

            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd MStartGetTabweatherInfoList listId=%d",
                            oMStart.ListId
                            ));
            if ((tU32)fc_sxm_tenListType_Invalid != oMStart.ListId)
            {
               fc_sxm_vSendSelfMessage(oMStart);
            }
        }
        break;

        case fc_sxm_enTtfisCmdsTabweather_MStartAddFavorite:
        {
			  /* This case handled to Add Favorite for the selected location*/
            midw_ext_sxm_tabweatherfi_tclMsgAddFavoriteMethodStart oMStart;
            oMStart.Location.Lat = (tS32)SXM_GET_U32(&au8Data[0]);
            oMStart.Location.Lon = (tS32)SXM_GET_U32(&au8Data[4]);
			   oMStart.Location.Radius = FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS;

            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd MStartAddFavorite - END"));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        case fc_sxm_enTtfisCmdsTabweather_MStartRemoveFavorite:
        {
			   /* This case handled to Remove Favorite from the list*/
            midw_ext_sxm_tabweatherfi_tclMsgRemoveFavoriteMethodStart oMStart;
            oMStart.TabWeather_LOCID = (tS32)SXM_GET_U32(&au8Data[0]);

			  ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd MStartRemoveFavorite - END"));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        case fc_sxm_enTtfisCmdsTabweather_MStartGetFavoriteInfo:
        {
			   /* This case handled to Get Favorite for the selected locid */
           midw_ext_sxm_tabweatherfi_tclMsgGetFavoriteInfoMethodStart oMStart;
           oMStart.TabWeather_LOCID = (tS32)SXM_GET_U32(&au8Data[0]);
			  ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd MStartGetFavoriteInfo - END"));
           fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

	    case fc_sxm_enTtfisCmdsTabweather_MStartSetListsConfig:
        {
           /* This case handled to do Method start for the set list capacity */
            midw_ext_sxm_tabweatherfi_tclMsgSetListsConfigMethodStart oMStart;
            oMStart.Config.Capacity=prMsg->au8Data[0];
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::fc_sxm_enTtfisCmdsTabweather_MStartSetListsConfig Config.Capacity=%d",
                            oMStart.Config.Capacity));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

	case fc_sxm_enTtfisCmdsTabweather_SetListMode:
        {
            midw_ext_sxm_tabweatherfi_tclMsgSxmListModeSet oPSet;
            /* Copy ListId param from TTFis*/
            oPSet.ListMode.ListID =(tU32)au8Data[0];
            oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType)au8Data[1];
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd PSetListId ListID=%d",oPSet.ListMode.ListID));
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::ttfis-cmd PSetListMode ListMode=%d",oPSet.ListMode.Mode.enType));
            fc_sxm_vSendSelfMessage(oPSet);

        }
        break;
#if 0
        case (fc_sxm_tenTtfisCmdsTabweather)fc_sxm_enGenricTtfisCmd_DsrlData:
        {
            // peha todo: parameter rGenericCfg.u32Misc can be used as indicator what to to (create, deleter, change dsrl ...)
            fc_sxm_trDSRLGenericCfg rGenericCfg(au8Data);
           // rGenericCfg.vPrint(TR_CLASS_FC_SXM_TABWEATHER_APP, "fc_sxm_tclTabweatherApp");
            rGenericCfg.vPrintGenericCfg(TR_CLASS_FC_SXM_TABWEATHER_APP, "fc_sxm_tclTabweatherApp");
            fc_sxm_trTabweatherDSRLCfg rDsrlCfg(rGenericCfg);
            // apply configuration
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vSetNextConfig START"));
            if (OSAL_NULL == _poNearDestDsrl) {
                ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vSetNextConfig END (_poNearDestDsrl=NULL)"));
                return;
            }
            rDsrlCfg.vPrint();
            _poNearDestDsrl->vSetNextConfig(rDsrlCfg);
            ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vSetNextConfig END"));
        }
        break;
#endif

        case fc_sxm_enTtfisCmdsTabweather_LAST:/*For lint*/
        default:
            ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                           ETG_CENUM(fc_sxm_tenTtfisCmdsTabweather, enMsgCode)));
            break;
    }
}
/*********************************************************************
 *FUNCTION:     bCheckLocationBoundary
 *DESCRIPTION:  Function to check the input location boundary is within the defined boundaries
 *PARAMETER:   s32Latitude - Latitude point;  s32Longitude - Longitude Point
 *RETURNVALUE:  FALSE - If location lies within the boundary or else TRUE
 ********************************************************************/
tBool fc_sxm_tclTabweatherApp::bCheckLocationBoundary(tS32 s32Latitude, tS32 s32Longitude) const
{
	return((s32Latitude < (FCSXM_TABW_MIN_LATITUDE )) ||
	   (s32Latitude > (FCSXM_TABW_MAX_LATITUDE )) ||
	   (s32Longitude < (FCSXM_TABW_MIN_LONGITUDE )) ||
	   (s32Longitude > (FCSXM_TABW_MAX_LONGITUDE)));
}
/*
  Handle method-starts
*/
/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList
 *PARAMETER:    1) Pointer to GetTabweatherInfoList message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList const *prMsg)
{

	midw_ext_sxm_tabweatherfi_tclMsgGetTabWeatherInfoListMethodStart const &oMStart=prMsg->oFiMsg;
    tU32 u32ListId=oMStart.ListId;
	fc_sxm_trTabweatherListConfig rListCfg;
	rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenTabweatherListSortMethod)fc_sxm_enTabweatherListSortMethod_NONE;
	rListCfg.rFilterCfg.enWeatherMsgObjectType=(WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

    if (oMStart.ListId == (tU32)fc_sxm_tenListType_AnotherLocation)
    {
       fc_sxm_trTabweatherDSRLCfg rDsrlCfg;
       tS32 s32SmsValueLat = fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oMStart.Location.Lat);
       tS32 s32SmsValueLon = fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oMStart.Location.Lon);

       ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList lat=%d,lon=%d",s32SmsValueLat,s32SmsValueLon));

	   if (TRUE == bCheckLocationBoundary(s32SmsValueLat,s32SmsValueLon))
	   {
			ETG_TRACE_ERR(("fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList : Lat/Lon value is invalid"));
			fc_sxm_trTabweatherList::vSetCfg(u32ListId, rListCfg);
			vSendList(u32ListId, prMsg->rAdressing);
			return;
	   }
       /* Configure the dsrl for another city location */
	   rDsrlCfg.rLocation = fc_sxm_trDsrlLocation(s32SmsValueLat, s32SmsValueLon, FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS);
      //rDsrlCfg.rLocation = fc_sxm_trDsrlLocation(oMStart.Location.Lat, oMStart.Location.Lon, oMStart.Location.Radius);
	   rDsrlCfg.u32Radius = FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS;
	   rDsrlCfg.u32Capacity = FC_SXM_TABWEATHER_STANDARD_DSRL_CAPACITY;
       rDsrlCfg.enSortMethod = fc_sxm_enTabweatherSortMethod_NONE;
       rDsrlCfg.rFilterCfg.enWeatherMsgObjectType = (WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

       if (OSAL_NULL != _poAnotherLocationList)
       {
          _poAnotherLocationList->vSetLocation(rDsrlCfg.rLocation);
       }

       if (OSAL_NULL != _poAnotherLocationDsrl)
       {
          _poAnotherLocationDsrl->vSetNextConfig(rDsrlCfg);
       }
    }

    /*
      send the answer to the client using adressing-infomation:
    */
    //fc_sxm_tclTabweatherService::instance()->enSendFiMessage(prMsg->rAdressing, ofiTxObj);
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList) vSetCfg: listId=%d enSortMethod=%d",
                   u32ListId,
                   ETG_CENUM(fc_sxm_tenTabweatherListSortMethod, rListCfg.rSortCfg.enSortMethod)
                   ));

   fc_sxm_trTabweatherList::vSetCfg(u32ListId, rListCfg);
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetTabweatherInfoList) vSendList"));
   vSendList(u32ListId, prMsg->rAdressing);
}
/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgTabweatherMStartAddFavorite
 *PARAMETER:    1) Pointer to AddFavorite message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite const *prMsg)
{
	midw_ext_sxm_tabweatherfi_tclMsgAddFavoriteMethodStart const &oMStart=prMsg->oFiMsg;

	vector <fc_sxm_trDistanceDatabase> listDistanceDB;
	vector <fc_sxm_trTabweatherDatabase,allocator<fc_sxm_trTabweatherDatabase> >::iterator vector_list_iter;
	fc_sxm_trDistanceDatabase DistanceDB;

	tBool bSetLocid = FALSE;
	_rSendFavAddressing = prMsg->rAdressing;

    /* Add favorite for the set lat and lon values  */
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite) : START"));
	tS32 s32SmsValueLat = fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oMStart.Location.Lat);
	tS32 s32SmsValueLon = fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oMStart.Location.Lon);

	if (TRUE == bCheckLocationBoundary(s32SmsValueLat,s32SmsValueLon))
	{
		 /* If lat and lon values are not within limits Send the empty list */
		ETG_TRACE_ERR(("fc_sxm_trMsgTabweatherMStartAddFavorite : Lat/Lon value is invalid"));

		vSendFavList(0,FALSE);

		return;
	}


	DISTANCE_OBJECT hRadius = fc_sxm_tclDsrlTypeAdapter::hCreateSMSRadius(FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS);
	if (hRadius == DISTANCE_INVALID_OBJECT)
	{
		ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite)=%p  FAILED",
			hRadius));
		fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
							(tInt)midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_FAVORITES_SETTING_FAILED);
		return;
	}

	LOCATION_OBJECT hTargetLocation = fc_sxm_tclDsrlTypeAdapter::hCreateSMSLocation(s32SmsValueLat, s32SmsValueLon, hRadius);

	if (hTargetLocation == LOCATION_INVALID_OBJECT)
	{
		ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite)=%p  FAILED",
			hTargetLocation));
		fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
							(tInt)midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_FAVORITES_SETTING_FAILED);
		return;
	}
	listDistanceDB.clear();
	 /* If lat and lon values are valid , Read the Tab weather database vector and calculate distance for the set location and db location */
	for(vector_list_iter = _TabweatherDBlist.begin();vector_list_iter != _TabweatherDBlist.end();vector_list_iter++)
	{
		fc_sxm_trTabweatherDatabase const iter=*vector_list_iter;
		DistanceDB.f32latitue = iter.f32latitue;
		DistanceDB.f32longitude = iter.f32longitude;
		DistanceDB.un32locid = iter.un32locid;
		DistanceDB.f32distance =fc_sxm_fGetLocationDistKm(hTargetLocation,iter.hLocation);
		DistanceDB.f32distance =(tF32) fc_sxm_fKm2Miles(DistanceDB.f32distance);

		if (DistanceDB.f32distance <= FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS)
		{
			/* To verify the Distance is within the given radius */
			listDistanceDB.push_back(DistanceDB);
	      ETG_TRACE_USR4(("Distance in miles: %f",DistanceDB.f32distance));
			bSetLocid = TRUE;
		}
	}

   /* Destroy object to avoid memory leak*/
	LOCATION.vDestroy(hTargetLocation);
    DISTANCE.vDestroy(hRadius);

	if (bSetLocid)
	{
		/* If a locid is identified for the given radius */
		std::sort(listDistanceDB.begin(),listDistanceDB.end(), CompareByMember<fc_sxm_trDistanceDatabase,tF32>(&fc_sxm_trDistanceDatabase::f32distance));
		LOC_ID un32Loc_id = listDistanceDB.front().un32locid;
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite) loc_id=%u",un32Loc_id));

		fc_sxm_trTabweatherListEntry rListEntry;
		fc_sxm_trTabweatherList *poList=dynamic_cast<fc_sxm_trTabweatherList*>(poGetActiveListbyDSRL(_poFavoritesDsrl));


		if (OSAL_NULL != poList)
		{
			if (poList->bGetListEntry(un32Loc_id, rListEntry))
			{
				/* If Tab weather list is valid and if loc id is already a favorite, send the list */
				ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartAddFavorite: LOCID is already a Favorite"));

				vSendFavList(un32Loc_id,FALSE);

			}
			else
			{
				/* If the loc id is not existing in the favorite list, add as favorite */
			   LOCID_OBJECT hLocID = LOCID.hCreate(un32Loc_id);
				if(LOCID_INVALID_OBJECT != hLocID)
				{
				   SMSAPI_RETURN_CODE_ENUM eWeatherResult = FAVORITES.eSetFavoriteByLocID(hLocID,
						_hTabweather,TRUE);

					if (SMSAPI_RETURN_CODE_SUCCESS != eWeatherResult)
					{
						ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite), Error, error code=%u",eWeatherResult));
						fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
							(tInt)midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_FAVORITES_SETTING_FAILED);
						return;
					}
					else
					{
						ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartAddFavorite - eSetFavoriteByLocID Success"));
						vSetbSetAsFavFlag(TRUE);
						vSetLocidForFav(un32Loc_id);
					}
					LOCID.vDestroy(hLocID);
				}
				else
				{
					ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartAddFavorite - LOCID Object invalid"));
				}
			}
		}
		else
		{
			ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartAddFavorite - List is NULL"));
		}
	}
	else
	{
		/* If there is no locid identified for the given radius */
		ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartAddFavorite - No weather station"));
		vSendFavList(0,FALSE);

	}


   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite) lat=%d,lon=%d",s32SmsValueLat,s32SmsValueLon));

}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgTabweatherMStartRemoveFavorite
 *PARAMETER:    1) Pointer to RemoveFavorite message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite const *prMsg)
{
    // prepare answer-messgae
   midw_ext_sxm_tabweatherfi_tclMsgRemoveFavoriteMethodStart const &oMStart=prMsg->oFiMsg;
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite) LOCID=%d",
                  oMStart.TabWeather_LOCID));
   fc_sxm_trTabweatherListEntry rListEntry;
   rListEntry.u32SxmId = oMStart.TabWeather_LOCID;
   fc_sxm_trTabweatherList *poList=dynamic_cast<fc_sxm_trTabweatherList*>(poGetActiveListbyDSRL(_poFavoritesDsrl));
   tBool bSetFavoriteError = FALSE;

   if ((OSAL_NULL == poList) || (!poList->bGetListEntry(oMStart.TabWeather_LOCID, rListEntry)) )
   {
		/* If a locid is not identified in the favorite list */
      ETG_TRACE_ERR(("fc_sxm_trMsgTabweatherMStartRemoveFavorite - List is NULL"));
	   fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
              (tInt) midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_FAVORITES_SETTING_FAILED);
      return;
   }
   /* A valid locid is present */
   LOCID_OBJECT hLocId = LOCID.hCreate(oMStart.TabWeather_LOCID);
   midw_ext_sxm_tabweatherfi_tclMsgRemoveFavoriteMethodResult oMRes;
   oMRes.FavStatus = TRUE;

   if (LOCID_INVALID_OBJECT != hLocId)
   {
      SMSAPI_RETURN_CODE_ENUM bFavStatus = FAVORITES.eSetFavoriteByLocID(hLocId,
                                                     _hTabweather,
                                                     FALSE);
      /*_hTabweather is Weather service object, also DSM object*/

      if (SMSAPI_RETURN_CODE_SUCCESS == bFavStatus)
      {
         oMRes.FavStatus = FALSE;//Not favorite
         fc_sxm_trTabweatherList::vUpdateEntry(DSRL_ENTRY_STATUS_REMOVED, rListEntry);
      }
      else
      {
         oMRes.FavStatus = bSetFavoriteError = TRUE;//Favorite
         ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite) Failed, Return code=%d",bFavStatus));
      }
      LOCID.vDestroy (hLocId);
   }
   else // hLocId == INVALID
   {
      bSetFavoriteError = TRUE;
      ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite) Failed ---> INVALID LOCID"));
   }

   if (bSetFavoriteError)
   {
      /* An error has occured.... send method error*/
      fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
              (tInt) midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_FAVORITES_SETTING_FAILED);
      ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite) Failed"));
   }
   else
   {
      fc_sxm_tclTabweatherService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
   }

   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartRemoveFavorite) END, Status=%u",oMRes.FavStatus));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgTabweatherMStartGetFavoriteInfo
 *PARAMETER:    1) Pointer to GetFavoriteInfo message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetFavoriteInfo const *prMsg)
{
	midw_ext_sxm_tabweatherfi_tclMsgGetFavoriteInfoMethodStart const &oMStart=prMsg->oFiMsg;
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetFavoriteInfo) LOCID=%d",
                     oMStart.TabWeather_LOCID));
    tU32 u32Id=oMStart.TabWeather_LOCID;
	fc_sxm_trTabweatherListConfig rListCfg;
	rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenTabweatherListSortMethod)fc_sxm_enTabweatherListSortMethod_NONE;
	rListCfg.rFilterCfg.enWeatherMsgObjectType=(WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

    fc_sxm_trTabweatherListEntry rListEntry;
    rListEntry.u32SxmId = u32Id;

	fc_sxm_trTabweatherList *poList=dynamic_cast<fc_sxm_trTabweatherList*>(poGetActiveListbyDSRL(_poFavoritesDsrl));
	if (OSAL_NULL != poList)
	{
		/* If a locid is not identified in the Favourite list */
		if (!poList->bGetListEntry(u32Id, rListEntry))
		{
         ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetFavoriteInfo) Entry not found"));
			fc_sxm_tclTabweatherService::vSendError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_TabWeather_ErrorTypes::FI_EN_WEATHER_INFO_UNAVAILABLE);
		 return;
        }
        vSetLocidForFav(u32Id);
		vSetbGetFavInfoFlag(TRUE);
		fc_sxm_trTabweatherList::vSetCfg((tU32)fc_sxm_tenListType_Favorites, rListCfg);
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartGetFavoriteInfo) vSendList"));
		vSendList((tU32)fc_sxm_tenListType_Favorites, prMsg->rAdressing);
    }
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgTabweatherMStartSetListsConfig
 *PARAMETER:    1) Pointer to SetListsConfig message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartSetListsConfig const *prMsg)
{
    midw_ext_sxm_tabweatherfi_tclMsgSetListsConfigMethodStart const &oFiMsg = prMsg->oFiMsg;
    tU32 u32Capacity=oFiMsg.Config.Capacity;

    /* configure dsrl with new capacity */
    fc_sxm_trTabweatherDSRLCfg rDsrlCfg;
    rDsrlCfg.u32Capacity = u32Capacity;

    /* configure the radius & Sort Types*/
    rDsrlCfg.enSortMethod = fc_sxm_enTabweatherSortMethod_NONE;
	rDsrlCfg.u32Radius = FC_SXM_TABWEATHER_STANDARD_DSRL_RADIUS;
    rDsrlCfg.rFilterCfg.enWeatherMsgObjectType = (WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

    tBool bNearByStatus=FALSE;
    tBool bDestStatus=FALSE;
    tBool bAnotherLocationStatus=FALSE;

    ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartSetListsConfig) call Capacity=%d",
                     u32Capacity));

    /* Prapare emthod result with success Status*/
    midw_ext_sxm_tabweatherfi_tclMsgSetListsConfigMethodResult oMRes;

	 if( OSAL_NULL != _poNearByDsrl){
		/* So Set new capacity to NearBy DSRL*/
		_poNearByDsrl->vSetNextConfig(rDsrlCfg);
		/* Set the Method reult to true*/
		/*oMRes.Status = TRUE;*/
      bNearByStatus = TRUE;
      ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartSetListsConfig vSetNextConfig Nearby"));
	 }
    if( OSAL_NULL != _poNearDestDsrl){
		/* So Set new capacity to NearDest DSRL*/
		_poNearDestDsrl->vSetNextConfig(rDsrlCfg);
      bDestStatus = TRUE;
      ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartSetListsConfig vSetNextConfig NearDest"));
	 }
    if( OSAL_NULL != _poAnotherLocationDsrl){
		/* So Set new capacity to NearDest DSRL*/
		_poAnotherLocationDsrl->vSetNextConfig(rDsrlCfg);
      bAnotherLocationStatus = TRUE;
      ETG_TRACE_USR4(("fc_sxm_trMsgTabweatherMStartSetListsConfig vSetNextConfig AnotherLocation"));
	 }
    oMRes.Status = (bNearByStatus && bDestStatus && bAnotherLocationStatus);

    ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartSetListsConfig) call enSendFiMessage"));

	 /* Send the method result to our service*/
	 fc_sxm_tclTabweatherService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    ETG_TRACE_ERR(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartSetListsConfig) END"));

}

/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:
*PARAMETER:  Pointer to fc_sxm_trMsgDataSmsEvtServiceUp message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg)
{
	(tVoid) prMsg ;
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgDataSmsEvtServiceUp)"));

	tBool bWeatherSkiFlag = FALSE;
	fc_sxm_tclTabweatherApp::instance()->_TabweatherDBlist.clear();

	/* Read Weather_ref.db to vector*/
	SMSAPI_RETURN_CODE_ENUM eReturnCode = WEATHER.eIterateLocations(_hTabweather,
																	(BOOLEAN)bWeatherSkiFlag,
																	STRING_INVALID_OBJECT,// hCityName
																	STRING_INVALID_OBJECT,//hStateAbbrv
																	fc_sxm_tclTabweatherApp::bWeatherLocationsIterator,
																	(tVoid*)NULL);
	if(eReturnCode == SMSAPI_RETURN_CODE_NO_OBJECTS)
	{
		ETG_TRACE_ERR(("No Weather locations Found"));
	}
	else if (eReturnCode == SMSAPI_RETURN_CODE_SUCCESS)
	{
		ETG_TRACE_USR4(("WEATHER.eIterateLocations Success"));
	}
	else
	{
		ETG_TRACE_ERR(("Error in retrieving Weather locations, Error code = %d",eReturnCode));
	}

}


/*********************************************************************
*FUNCTION:     vProcess
*DESCRIPTION:  Set property to Handle the List Mode
*PARAMETER:  Pointer to fc_sxm_trPropertyTabweatherPSetListMode message
*RETURNVALUE:  None
********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trPropertyTabweatherPSetListMode  const *prMsg)
{
   ETG_TRACE_USR4(( "vProcess fc_sxm_trPropertyTabweatherPSetListMode" ));

   midw_ext_sxm_tabweatherfi_tclMsgSxmListModeSet const &oPSet=prMsg->oFiMsg;

   /* Initialize default values*/
   tU32 u32ListId = oPSet.ListMode.ListID;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyTabweatherPSetListMode: ListID %d",u32ListId));

   fc_sxm_tenListMode enListMode = (fc_sxm_tenListMode)oPSet.ListMode.Mode.enType;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyTabweatherPSetListMode: enListMode %d",enListMode));
   vSetListMode(u32ListId, prMsg->rAdressing,enListMode);

}
/*********************************************************************
 *FUNCTION:     vOnListStateChanged
 *DESCRIPTION:  Method to set and notify TabweatherStatus property to HMI
 *PARAMETER:
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vOnListStateChanged(fc_sxm_tclDataListBase *poDataList)
{
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vOnListStateChanged, List state=%d, List type=%d",
                                                         poDataList->rGetStatusInfo().enListState,
                                                         poDataList->rGetStatusInfo().enListType));

   if (fc_sxm_tenListType_Invalid != (poDataList->rGetStatusInfo().enListType))
   {
      fc_sxm_tcl_trTabweatherPropertyNewUpdateTabweatherInfoListStatus oProperty;
      midw_ext_sxm_tabweatherfi_tclMsgTabWeatherInfoListStatusStatus &oFiMsg=oProperty.oFiMsg;
		//oFiMsg.TabweatherListStatus.GenericConfig.Capacity=TABWEATHER_CAPACITY; // todo:take dsrl-capacity
      oGetDataListsInfo().vCopyListInfosToFi(oFiMsg.TabweatherListStatus); // use conversion-operator
      fc_sxm_tclTabweatherProperties::instance()->oNewUpdateTabweatherInfoListStatus.vSetAndNotify(oProperty);
   }
}

/*********************************************************************
*FUNCTION:     GetFavDsrl
*DESCRIPTION:  Return the Fav. DSRL object
 *PARAMETER:
 *RETURNVALUE:  DSRL Object
 ********************************************************************/
DSRL_OBJECT fc_sxm_tclTabweatherApp::GetFavDsrl() const
{
   DSRL_OBJECT hDsrlObj = DSRL_INVALID_OBJECT;
   if (_poFavoritesDsrl)
   {
      hDsrlObj = _poFavoritesDsrl->hGetSMSDSRL();
   }
   return hDsrlObj;
}

/*********************************************************************
 *FUNCTION:     vOnFiServiceStateChanged
 *DESCRIPTION:  Application get notified about the new fi-service-state that shall be sent to clients
 *PARAMETER:     oFiState
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState) {
    fc_sxm_tcl_trTabweatherPropertySxmServiceStatus oProperty;
    oProperty.oFiMsg.DataServiceState=oFiState;
    fc_sxm_tclTabweatherProperties::instance()->oSxmServiceStatus.vSetAndNotify(oProperty);
}

/*********************************************************************
*FUNCTION:     vSendFavList
*DESCRIPTION:  Send Method result for AddFavorite
*PARAMETER:  un32Loc_id - Locid id ; bStatus - Favorite Status flag set to true if method result is form
                          GetTabweatherInfoList
*RETURNVALUE:  None
********************************************************************/


tVoid fc_sxm_tclTabweatherApp::vSendFavList(LOC_ID un32Loc_id,tBool bStatus)
{
   /* Configure the DSRL */
   fc_sxm_trTabweatherListConfig rListCfg;
   rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenTabweatherListSortMethod)fc_sxm_enTabweatherListSortMethod_NONE;
   rListCfg.rFilterCfg.enWeatherMsgObjectType=(WEATHER_MSG_OBJECT_TYPE_ENUM)fc_sxm_enTabweatherFilterMethod_WEATHER;

   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite) vSetCfg: listId=%d enSortMethod=%d",
                   (tU32)fc_sxm_tenListType_Favorites,
                   ETG_CENUM(fc_sxm_tenTabweatherListSortMethod, rListCfg.rSortCfg.enSortMethod)
                   ));
   vSetLocidForFav(un32Loc_id);
    /*Set Configuration for the new DSRL */
   fc_sxm_trTabweatherList::vSetCfg((tU32)fc_sxm_tenListType_Favorites, rListCfg);
   vSetbGetFavInfoFlag(bStatus);
    /*Fav. Method result to be sent*/
   ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgTabweatherMStartAddFavorite) vSendList, Favorite"));
   vSendList((tU32)fc_sxm_tenListType_Favorites, _rSendFavAddressing);
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgCmdAppClearUserData)
 *DESCRIPTION:  Clear Favorites
*PARAMETER:  fc_sxm_trMsgCmdAppClearUserData const *prMsg
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclTabweatherApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
{
	(void)prMsg; /* TO REMOVE LINT WARNINGS */
	ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp::vProcess(fc_sxm_trMsgCmdAppClearUserData(%s)",
		szGetName()));
	/*Clear favorite With user request NIKAI-2577*/

	if(WEATHER_SERVICE_INVALID_OBJECT != _hTabweather)  {
		ETG_TRACE_USR4(("fc_sxm_tclTabweatherApp:: Clear All Favorite"));
		/* Call SMS API to Clear All Favorites */
		FAVORITES.vClearAllFavorites(_hTabweather);

		/* Say Result is OK */
		vSetUsrDataClearResult(fc_sxm_enDiagResult_Ok);

	}
	else
	{
		ETG_TRACE_ERR(( " WEATHER_SERVICE_INVALID_OBJECT " ));
		/* Say Result is not OK */
		vSetUsrDataClearResult(fc_sxm_enDiagResult_Failed);
	}
}
/*=============================================================================
=======                   END OF FILE                                   =======
=============================================================================*/

