/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_stocks_list.cpp
* @brief       Stocks sort and filter list handler implementation.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_stocks_types.h"
#include "fc_sxm_tcl_stocks_list.h"
#include "fc_sxm_stocks_fi.h"
#include "fc_sxm_service_sxm_stocks.h"
#include "fc_sxm_tcl_stocks_app.h"
#include "fc_sxm_sms_util.h"

#include "fc_sxm_time_profile.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_STOCKS_APP
#include "trcGenProj/Header/fc_sxm_tcl_stocks_list.cpp.trc.h"
#endif

#define FC_SXM_MIDW_STOCKS_MAX_SYMBOL_LENGTH 10
#define FC_SXM_STOCKS_LOADING_CUTOFF_DURATION (15000 * 60) //15 minutes
#define FC_SXM_STOCKS_LOADING_CUTOFF_GRACE_PERIOD 15000 //15 seconds

/*********************************************************************
 *FUNCTION:     vPrintStockListEntry
 *DESCRIPTION:  Method used for printing formatted output in TTFI's
 *PARAMETER:	The entry count value used for formatted output. This
 *	value would be 0 if invoked from fc_sxm_tclStocksDSRL. It would have
 *	a value of > 1 if invoked from fc_sxm_trStocksList::vEmit()
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_trStockQuoteListEntry::vPrintStockListEntry ( tU8 u8EntriesCount, tU32 msecDuration) const
{
	// The only utility of this method is to make a formatted output in TTFI's. Its a utility method
	// which can be invoked from both fc_sxm_tclStocksDSRL::bHandleIterateCallback() or fc_sxm_trStocksList::vEmit().
	if ( 0 == u8EntriesCount)
	{
		ETG_TRACE_USR4(("---------> PRINT NEW STOCK ENTRY FROM fc_sxm_tclStocksDSRL::bHandleIterateCallback <-------------"));
	}
	else
	{
		ETG_TRACE_USR4(("--------->  %u) START PRINTING LIST FROM fc_sxm_trStocksList::vEmit  <-------------", u8EntriesCount));
	}

	ETG_TRACE_USR4(("a) STOCK SYMBOL = %s", strStockSymbol.c_str()));
	ETG_TRACE_USR4(("b) PRICE = %.2f\t c)DELTA = %.2f",  f32MarketPrice, f32Delta));
	ETG_TRACE_USR4(("d) DIRECTION = %d\t e)tEntryID = %u",
			ETG_CENUM(fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM, (fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM) enStockDirection),
			u32SxmId));
	// Only print time left to change stock status if current stockDirection = Invalid
	if ((0 != u8EntriesCount) && ( fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_INVALID == enStockDirection))
		ETG_TRACE_USR4 (("f) Time Since adding the stock as DSRL entry = %u min", (( msecDuration - msecReportTime)/(60*1000))));
	ETG_TRACE_USR4(("---------> FORMATTED OUTPUT ENDS FOR THIS ENTRY <-------------"));
}

/*********************************************************************
 *FUNCTION:     bCompareItem
 *DESCRIPTION:  Comparison for fc_sxm_trStockQuoteListEntry structures
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structures which has to be compared for sorting
 *RETURNVALUE:  bool; True if the compared item is smaller than comparing item; FALSE otherwise
 ********************************************************************/
bool fc_sxm_trStocksList::bCompareItem(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const
{
    ETG_TRACE_USR4(("fc_sxm_trStocksList::bCompareItem IN SortMethod=%d",_rListCfg.rSortCfg.enSortMethod));
    //based on sorted method compare
    switch (_rListCfg.rSortCfg.enSortMethod)
    {
       case fc_sxm_enStocksDSRLSortMethod_NONE:
          {
             return TRUE; //incase of no sorted method
          }
       case fc_sxm_enStocksDSRLSortMethod_ALPHABET:
          {
             return bCompareStockSymbol(prLeft, prRight);
          }
       case fc_sxm_enStocksDSRLSortMethod_FIRST_SAVED_FIRST:
          {
             return bCompareFavoriteOrder(prLeft, prRight);
          }
       default:
          return bCompareListEntryId(prLeft, prRight); //default comparision
    }
}
/*********************************************************************
 *FUNCTION:     bCompareStockSymbol
 *DESCRIPTION:  Function Compare 2 stock symbold by it's name
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structures which has to be compared for sorting
 *RETURNVALUE:  bool; True if the compared item is smaller than comparing item; FALSE otherwise
 ********************************************************************/
bool fc_sxm_trStocksList::bCompareStockSymbol(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const
{
    return ((prLeft->strStockSymbol.compare(prRight->strStockSymbol)) < 0 );
}
/*********************************************************************
 *FUNCTION:     bCompareFavoriteOrder
 *DESCRIPTION:  Function to Compare 2 stock symbold by it's order what user have saved in
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structures which has to be compared for sorting
 *RETURNVALUE:  bool; True if the compared item is smaller than comparing item; FALSE otherwise
 ********************************************************************/
bool fc_sxm_trStocksList::bCompareFavoriteOrder(fc_sxm_trStockQuoteListEntry const *prLeft, fc_sxm_trStockQuoteListEntry const *prRight) const
{
    ETG_TRACE_USR4(("fc_sxm_trStocksList:bCompareFavoriteOrder StockSymbol(L)=%s",prLeft->strStockSymbol.c_str()));
    ETG_TRACE_USR4(("fc_sxm_trStocksList:bCompareFavoriteOrder StockSymbol(R)=%s",prRight->strStockSymbol.c_str()));
    return ((fc_sxm_tclStocksApp::instance()->u32GetFavoriteIndex(prLeft->strStockSymbol)) < (fc_sxm_tclStocksApp::instance()->u32GetFavoriteIndex(prRight->strStockSymbol)));
}

tVoid fc_sxm_trStocksList::vEmit(fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_trStocksList::vEmit START"));
	midw_ext_sxm_stocksfi_tclMsgGetFavoriteStockSymbolsMethodResult oMRes;
	oMRes.ListID=u32GetId();
	ETG_TRACE_USR4(("fc_sxm_trStocksList::vEmit ListID = %u", oMRes.ListID));
	/*
Todo: Variable u8EntriesCount is added only for formatted priting while testing using TTFis
Must be removed when testing is over
	 */
	tU8 u8EntriesCount = 0;
	const OSAL_tMSecond msecCurTime = OSAL_ClockGetElapsedTime();
	OSAL_tMSecond msecNextChange = 0u;

	// Each entry present in data list is retrieved and appended to methodResult
	SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries())
	{
		fc_sxm_trStockQuoteListEntry const *poEntry=*iter;

		midw_ext_fi_tcl_StockQuote oFiEntry;
		fc_sxm_vString2Fi(((poEntry->strStockSymbol).c_str()),oFiEntry.Symbol);
		oFiEntry.MarketPrice=poEntry->f32MarketPrice;
		oFiEntry.Delta=poEntry->f32Delta;
		if ( fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_INVALID == poEntry->enStockDirection )
		{
			// These checks are required only for stocks whose status is "Loading".
			// Based on "Loading" status, an evaluation is performed to see if the status has to be changed to "Unknown"
			if (bIsCutOffTimeReached(msecCurTime, poEntry->msecReportTime))
			{
				/* Carousel based cutoff time reached.... change stock status to "Unknown"*/
				ETG_TRACE_USR4 (("Entry Changed from LOADING ---> UNKNOWN"));
				poEntry->enStockDirection = fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNKNOWN;
			}
			else if ( 0u != msecNextChange )
			{
				// Cutoff time not reached yet.... continue to Display stock status as "Loading"
				//  Check the stock symbol whose status has to be updated first from "Loading" to "Unknown"
				msecNextChange = (msecNextChange <= (poEntry->msecReportTime))? msecNextChange : poEntry->msecReportTime;
			}
			else
			{
				// Cutoff time not reached yet.... continue to Display stock status as "Loading"
				// Implies ( 0u == msecNextChange )
				msecNextChange = poEntry->msecReportTime;
			}
		}
		oFiEntry.Direction.enType = (midw_ext_fi_tcl_e8_STOCK_PRICE_DIRECTION::tenType) poEntry->enStockDirection;
		oMRes.StocksQuoteList.StockQuoteList.push_back(oFiEntry);
		poEntry->vPrintStockListEntry( ++u8EntriesCount, msecCurTime );
	}

	// Send FI message
	fc_sxm_tclStocksService::instance()->enSendFiMessage(rAdressing, oMRes);
	// Recalculate time required for next change based on current time and grace period
	if (0u != msecNextChange)
	{
		/* 15 sec grace period is given so that the status for all INVALID STOCK SYMBOLS are
		 * updated at one shot after system restart instead of changing Invalid stock status one by one.*/
		/* For newly added Invalid stock symbols, the 15 sec grace period won't have any impact apart
		 * from changing stock status from Loading to Unknown after 15 min 15 sec*/
		msecNextChange = FC_SXM_STOCKS_LOADING_CUTOFF_DURATION  + FC_SXM_STOCKS_LOADING_CUTOFF_GRACE_PERIOD - ( msecCurTime - msecNextChange ) ;
		ETG_TRACE_USR4(("Next change Time = %u mSec", msecNextChange));
		fc_sxm_tclStocksApp::instance()->vStartStocksTimer(msecNextChange);
	}
	ETG_TRACE_USR4(("fc_sxm_trStocksList::vEmit END"));
}

/*********************************************************************
 *FUNCTION:     bIsCutOffTimeReached
 *DESCRIPTION:  Return TRUE if stock symbol's cut-off time is exceeded, FALSE otherwise.
 *                If stock price direction is STOCK_PRICE_DIRECTION_INVALID and
 *               If cut-off time < 15 min... HMI should display "Loading" for stock price
 *               If cut-off time >= 15 min... HMI should display "Unknown" for stock price
 *PARAMETER:    Current time value and Time stamp when the entry was reported first
 *RETURNVALUE:  Boolean
 ********************************************************************/

bool fc_sxm_trStocksList::bIsCutOffTimeReached( const OSAL_tMSecond &msecCurTime,
                                               const OSAL_tMSecond &msecEntryTime) const
{
	ETG_TRACE_USR4(("fc_sxm_trStocksList::bIsCutOffTimeReached msecCurTime = %u and msecEntryTime = %u", msecCurTime, msecEntryTime));
	// Check if cutoff based carousel time is reached for the given stock symbol or not.
	// Returns TRUE is carousel based cutoff time is reached, FALSE otherwise
	return (FC_SXM_STOCKS_LOADING_CUTOFF_DURATION < (msecCurTime - msecEntryTime));
}

/*********************************************************************
 *FUNCTION:     bIsFavoriteStockSymbol
 *DESCRIPTION:  Return TRUE if stock symbol is present in internal list, FALSE otherwise
 *PARAMETER:    string whose value has to be compared if its already present in list
 *RETURNVALUE:  Boolean
 ********************************************************************/

tBool fc_sxm_trStocksList::bIsFavoriteStockSymbol( const string &strCompareSymbol)
{
	ETG_TRACE_USR4(("fc_sxm_trStocksList::bIsFavoriteStockSymbol Entered: stockSymbol = %s", strCompareSymbol.c_str()));
	// Return value from the method
	tBool bRetVal = FALSE;
	SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries())
	{
		fc_sxm_trStockQuoteListEntry const *poEntry=*iter;
		// Do a validate entry entry check
		if (( OSAL_NULL == poEntry) || (FALSE == bValidateEntry(poEntry)))
		{
			ETG_TRACE_ERR(("poEntry is NULL/Invalid Entry ---> not Validating for this particular Entry"));
			continue;
		}
		if ( poEntry->strStockSymbol.compare(strCompareSymbol) == 0)
		{
			ETG_TRACE_USR4(("Breaking for stockSymbol = %s", poEntry->strStockSymbol.c_str()));
			bRetVal = TRUE;
			break;
		}
	}
	ETG_TRACE_USR4(("fc_sxm_trStocksList::bIsFavoriteStockSymbol Exited --- bRetVal = %u", bRetVal));
	// Return TRUE if stock symbol is present in internal list, FALSE otherwise
	return bRetVal;
}

/*********************************************************************
 *FUNCTION:     bValidateEntry
 *DESCRIPTION:  Check if the given List Entry is valid or not based on
 the values stored
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structure
 *RETURNVALUE:  Return TRUE if stock symbol is valid, FALSE otherwise
 ********************************************************************/
bool fc_sxm_trStocksList::bValidateEntry (fc_sxm_trStockQuoteListEntry const *poStockQuoteListEntry) const
{
	// Validate the data list entry for any corruptions
	// Return TRUE if stock symbol is not corrupted, FALSE otherwise
	return ((poStockQuoteListEntry->enStockDirection <= fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNKNOWN)
			&& (poStockQuoteListEntry->strStockSymbol.length() < FC_SXM_MIDW_STOCKS_MAX_SYMBOL_LENGTH));
}


/*=============================================================================
  =======                   END OF FILE                                   =======
  =============================================================================*/
