/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_stocks_dsrl.cpp
* @brief       Implementation of Stocks DSRL for favorites.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_stocks_app.h"
#include "fc_sxm_tcl_stocks_dsrl.h"
#include "fc_sxm_tcl_stocks_list.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_STOCKS_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_stocks_dsrl.cpp.trc.h"
#endif

/*********************************************************************
 *FUNCTION:     fc_sxm_trStocksDSRLCfg::vPrintSpecific
 *DESCRIPTION:  Prefix
 *PARAMETER:    NONE
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_trStocksDSRLCfg::vPrintSpecific() const
{
	// Prints the default sort method used to the FC_SXM_MGR_REPORT
	ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenSortMethod=%u", ETG_CENUM(fc_sxm_tenStocksDSRLSortMethod, enSortMethod)));
}

/*********************************************************************
 *FUNCTION:     ~fc_sxm_tclStocksDSRL
 *DESCRIPTION:  Constructor
 *PARAMETER:    1) DSRL TYPE (FAVORITES in case of Stocks)
 2) Stock Ticker service handle
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclStocksDSRL::fc_sxm_tclStocksDSRL(fc_sxm_tenDSRLType enDsrlType):
	fc_sxm_tclConfigDSRL<fc_sxm_trStocksDSRLCfg>(enDsrlType,
                                                 fc_sxm_tclStocksApp::instance(),
                                                 static_cast <tU16> (TR_CLASS_FC_SXM_STOCKS_DSRL))
{ /* Default Constructor*/}


/*********************************************************************
 *FUNCTION:     ~fc_sxm_tclStocksDSRL
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclStocksDSRL::~fc_sxm_tclStocksDSRL()
{
	//Destructor for fc_sxm_tclStocksDSRL class
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL destructor"));
}


/*********************************************************************
 *FUNCTION:     vOnDSRLUpdate
 *DESCRIPTION:  Called when DSRL is updated
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksDSRL::vOnDSRLUpdate(tVoid)
{
	/* Update the property to indicate clients
	 * Currently not handled for stocks and left void
	 */
}

/*********************************************************************
 *FUNCTION:     cb_bDSRLFilter
 *DESCRIPTION:  Stocks service DSRL filter callback. Can be ignored for Stocks
 *PARAMETER:    1) Specific DSRL , 2) DSRL entry object
 *RETURNVALUE:  Returns TRUE on success, FALSE otherwise.
 ********************************************************************/
tBool fc_sxm_tclStocksDSRL::bHandleFilterCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_OBJECT hEntry)
{
	// Filter mechanism is not utilised in stocks feature. By Default returns TRUE
	(tVoid) hDSRL;
	(tVoid) hEntry;
	return TRUE;
}

/*********************************************************************
 *FUNCTION:     cb_n16DSRLSort
 *DESCRIPTION:  Stocks service DSRL sort callback. Sorts 2 DSRL entries
 and returns sorted results. Can be ignored for Stocks
 *PARAMETER:    1) DSRL Handle 2) DSRL entry 1 and 3) DSRL entry 2
 *RETURNVALUE:  tS16
 ********************************************************************/
tS16 fc_sxm_tclStocksDSRL::s16HandleSortCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_OBJECT hEntry1,
		DSRL_ENTRY_OBJECT hEntry2)
{
	// Sort mechanism is not utilised in stocks feature. By Default returns 1
	(tVoid) hDSRL;
	(tVoid) hEntry1;
	(tVoid) hEntry2;

	return (tS16) 1 ;
}

/*********************************************************************
 *FUNCTION:     vHandleDsrlEntryInvalid
 *DESCRIPTION:  Overriden method to handle invalid entries
 *PARAMETER:    1) DSRL entry ID
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksDSRL::vHandleDsrlEntryInvalid(DSRL_ENTRY_ID tEntryID)
{
	// Same entry deleted in Favorites DSRL is also deleted in dataLists Entry
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL::vHandleDsrlEntryInvalid Entered/Exited: tEntryId = %u", tEntryID ));
	fc_sxm_trStocksList::vRemoveEntry(tEntryID);
}

/*********************************************************************
 *FUNCTION:     cb_bDSRLIterate
 *DESCRIPTION:  Stocks service DSRL iterate callback
 *PARAMETER:    None
 *RETURNVALUE:  TRUE if the bHandleIterateCallback() was sucess, FALSE otherwise
 ********************************************************************/
tBool fc_sxm_tclStocksDSRL::bHandleIterateCallback(DSRL_OBJECT hDSRL,
		DSRL_ENTRY_ID tEntryID,
		DSRL_ENTRY_STATUS_ENUM eStatus,
		DSRL_ENTRY_OBJECT hEntryObject)
{
	(tVoid) hDSRL;
	// Method invoked from MIDW-FW's BASE DSRL on receiving status updated from SMS
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL::bHandleIterateCallback Entered, \t tEntryID = %u\t Status = %u",
			tEntryID, ETG_CENUM(DSRL_ENTRY_STATUS_ENUM, eStatus)));
	// Local variables initialisation
	STOCK_MSG_PRICE_DIRECTION_ENUM eDirection = STOCK_MSG_PRICE_DIRECTION_INVALID;
	tUInt uPrice = 0, uPriceDelta = 0;
	tU8 u8PriceFrac = 0, u8PriceDeltaFrac = 0;

	// Code for packing the contents into fc_sxm_trStockQuoteListEntry
	fc_sxm_trStockQuoteListEntry rListEntry(tEntryID);

	// Handle for DSRL_ENTRY_STATUS_CHANGED and DSRL_ENTRY_STATUS_NEW
	// Though DSRL_ENTRY_STATUS_UNCHANGED is handled in Base DSRL and never reported to the
	// respective service, its added to avoid changes to this method in future if Base DSRL changes logic
	if ((DSRL_ENTRY_STATUS_CHANGED == eStatus) ||
			(DSRL_ENTRY_STATUS_NEW == eStatus) ||
			(DSRL_ENTRY_STATUS_UNCHANGED == eStatus))
	{
		/*Get STOCK_MSG object handle from the DSRL_ENTRY*/
		STOCK_MSG_OBJECT hStockMsg = DSRL_ENTRY.hStockMsg(hEntryObject);
		if ( STOCK_MSG_INVALID_OBJECT == hStockMsg )
		{
			ETG_TRACE_ERR(("Stock Invalid! --> Returning from bHandleIterateCallback without processing; tEntryID = %u", tEntryID));
			return TRUE;
		}
		// Todo: Send error values through MethodResult
		// Get stock symbol name
		STRING_OBJECT hName = STOCK_MSG.hSymbolName(hStockMsg);
		if ( STRING_INVALID_OBJECT == hName )
		{
			ETG_TRACE_ERR(("Stock Name Invalid! --> Returning from bHandleIterateCallback without processing tEntryID = %u", tEntryID));
			return TRUE;
		}
		// Get stock price and delta
		SMSAPI_RETURN_CODE_ENUM enApiReturnValue = STOCK_MSG.eValues(hStockMsg,
				&uPrice, &u8PriceFrac,
				&uPriceDelta, &u8PriceDeltaFrac,
				&eDirection);
		ETG_TRACE_USR4(("STOCK_MSG.eValues status = %u",
		                               ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enApiReturnValue)));
		if ( SMSAPI_RETURN_CODE_ERROR == enApiReturnValue )
		{
			ETG_TRACE_ERR(("SMSAPI_RETURN_CODE_ERROR == enApiReturnValue --> Returning from bHandleIterateCallback without processing for tEntryID = %u", tEntryID));
			return FALSE;
		}

		// Step 1: get stock symbol name
		fc_sxm_vCopySmsString2Stl(hName,rListEntry.strStockSymbol);
		// Step 2, validate the Stock message values
		if ( SMSAPI_RETURN_CODE_SUCCESS == enApiReturnValue)
		{
			if (u8PriceFrac <= 99u) //(u8PriceFrac <= 99u)
			{
				rListEntry.msecReportTime = OSAL_ClockGetElapsedTime() ;
				// Step 3: get current market price
				rListEntry.f32MarketPrice = (tF32)uPrice + (tF32)u8PriceFrac / 100.0f;
				// Step 4: get delta value
				rListEntry.f32Delta = (tF32)uPriceDelta + (tF32)u8PriceDeltaFrac / 100.0f;
				// Step 5: get price movement direction
				rListEntry.enStockDirection = ( fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM ) eDirection;
			}
			else  //  (u8PriceFrac > 99u)
			{
				/*
				  If the Fractional value is greater that 99, then the stock values obtained is invalid.
				  As per UIRR for stocks,If the Trade Value fractional value =
				  255, Trade value is received but is reported as NA
				  254, Trade value is received but is reported as Delisted
				  100-253, RFU in protocol for stocks data service
				 */

				/*
				  If the symbol index, value pair does not match, SMS returns SMSAPI_RETURN_CODE_NOT_FOUND
				  for the call STOCK_MSG.eValues(). At such scenarios, Invalid entry values must be displayed.
				 */
				ETG_TRACE_USR4(("INVALID FRACTIONAL VALUE = %u, Return Code = SMSAPI_RETURN_CODE_SUCCESS, tEntryID = %u",u8PriceFrac, tEntryID ));
				vAssignInvalidStockValues (&rListEntry, (tU8)fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNKNOWN);
			}
		}//end of ( SMSAPI_RETURN_CODE_SUCCESS == enApiReturnValue)
		else if ( SMSAPI_RETURN_CODE_NOT_FOUND == enApiReturnValue )
		{
			ETG_TRACE_USR4(("FRACTIONAL VALUE = %u, Return Code = SMSAPI_RETURN_CODE_NOT_FOUND, tEntryID = %u, eStatus = %u",u8PriceFrac, tEntryID, eStatus ));
			vAssignInvalidStockValues (&rListEntry, (tU8)fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_INVALID);
		} //end of ( SMSAPI_RETURN_CODE_NOT_FOUND == enApiReturnValue)
		else
		{
			//enApiReturnValue != SMSAPI_RETURN_CODE_NOT_FOUND or SMSAPI_RETURN_CODE_SUCCESS or SMSAPI_RETURN_CODE_ERROR
			/* Though return code's other than SMSAPI_RETURN_CODE_NOT_FOUND, SMSAPI_RETURN_CODE_SUCCESS, SMSAPI_RETURN_CODE_ERROR
			  are not expected as per SMS API documentation, this part is added as a safety check.Not doing any further processing
			  for this error code to avoid unexpected behavior.
			 */
			ETG_TRACE_ERR(("NOT EXPECTED ERROR; errorCode = %u for tEntryID = %u", enApiReturnValue ,tEntryID));
			return FALSE;
		} // end of else part
	} //end of (if part eStatus = Changed, New, Unchanged)
	else
	{
		//if (DSRL_ENTRY_STATUS_REMOVED == eStatus) || (DSRL_ENTRY_STATUS_UNKNOWN == eStatus)
		vAssignInvalidStockValues (&rListEntry, (tU8)fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_UNKNOWN);
	}
	// Print the contents of Stocks List Entry
	rListEntry.vPrintStockListEntry((tU8)0, (tU32)0);

	// Update the list state entry with the recently received values
	fc_sxm_trStocksList::vUpdateEntry(eStatus, rListEntry);
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL::bHandleIterateCallback Exit"));
	return TRUE;
}

/*********************************************************************
 *FUNCTION:     vAssignInvalidStockValues
 *DESCRIPTION:  Assigns Invalid values to fc_sxm_trStockQuoteListEntry structure.
 This is a utility function.
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structure
 *RETURNVALUE:  NONE
 ********************************************************************/
tVoid fc_sxm_tclStocksDSRL::vAssignInvalidStockValues (fc_sxm_trStockQuoteListEntry *poStockQuoteListEntry,
                                                       tU8 enDirection) const
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL::vAssignInvalidStockValues Entered, enDirection = %u",
            ETG_CENUM(fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM, (fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM) enDirection)));
	/* This method assigns default values to the given fc_sxm_trStockQuoteListEntry structure
	 * The enStockDirection member is updated based on the time elapsed from the entry being added
	 * and the current time */
	poStockQuoteListEntry->msecReportTime = OSAL_ClockGetElapsedTime() ;
	poStockQuoteListEntry->f32MarketPrice = 0.0f;
	poStockQuoteListEntry->f32Delta = 0.0f;
	poStockQuoteListEntry->enStockDirection = (fc_sxm_stocks_enSTOCK_PRICE_DIRECTION_ENUM) enDirection;
	ETG_TRACE_USR4(("fc_sxm_tclStocksDSRL::vAssignInvalidStockValues Exit"));
}


/*=============================================================================
  =======                   END OF FILE                                   =======
  =============================================================================*/

