/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_stocks_app.cpp
* @brief       Application handler implementation for Stocks service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*=============================================================================
  =======                            INCLUDES                             =======
  =============================================================================*/
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_stocks.h"
#include "fc_sxm_tcl_stocks_app.h"
#include "fc_sxm_tcl_stocks_dsrl.h"
#include "fc_sxm_tcl_stocks_list.h"
#include "fc_sxm_tcl_pers_app_data.h"

#ifdef FC_SXM_TIME_PROFILE
#include "fc_sxm_time_profile.h"
#endif //FC_SXM_TIME_PROFILE


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_STOCKS_APP
#include "trcGenProj/Header/fc_sxm_tcl_stocks_app.cpp.trc.h"
#endif

/*=============================================================================
  =======                     PREPROCESSOR MACROS                         =======
  =============================================================================*/
/* FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS MACRO reduces LOC in method vGetStocksByFilter().
 * The same steps were required to be implemented at 2 places within the same method for
 * realizing 2 different use cases.*/

/* FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS MACRO implements the following code,
 * 	setEnableChar.clear();
 * 	vectListBuffer.clear();
 * 	iter = _setAllStockSymbols.lower_bound(strSearchFilter);
 * 	chPrevious = ' ';
 * 	u32StrLen = strSearchFilter.length();
 * 	u32ListCount = 0;
 */
#define FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS( SETCHAR, VECBUFFER, SEARCHFILTER, ITER, ALLSTOCKSYMBOLS, CHPREV, LEN, COUNT) \
{ \
	SETCHAR.clear(); \
	VECBUFFER.clear(); \
	ITER = ALLSTOCKSYMBOLS.lower_bound(SEARCHFILTER); \
	CHPREV = ' '; \
	LEN = (tU32)(SEARCHFILTER.length());  \
	COUNT = 0;  \
}

/*=============================================================================
  =======                             TYPES                               =======
  =============================================================================*/
typedef enum {
	fc_sxm_enTtfisCmdsStocks_MStartGetFavoriteStockSymbols=1,
	fc_sxm_enTtfisCmdsStocks_MStartAddFavorite=2,
	fc_sxm_enTtfisCmdsStocks_MStartRemoveFavorite=3,
	fc_sxm_enTtfisCmdsStocks_MStartGetStockSymbolsByFilter=4,
	fc_sxm_enTtfisCmdsStocks_MStartGetDataProviderInfo=5,
	fc_sxm_enTtfisCmdsStocks_MStartClearDeltaStockSymbolTable=6,
	fc_sxm_enTtfisCmdsStocks_MStartSetListsConfig=7,
	fc_sxm_enTtfisCmdsStocks_MStartGetStockSymbolsByFilterPS = 8,
	fc_sxm_enTtfisCmdsStocks_SetListMode = 9,
	// The enums from values 10-14 are used only for internal testing while working with TTFIs
	fc_sxm_enTtfisCmdsStocks_MStartClearAllFavorites=10,
	fc_sxm_enTtfisCmdsStocks_MStartDSRLIterate=11,
	fc_sxm_enTtfisCmdsStocks_MStartGetStocksServiceAndDsrlStates=12,
	fc_sxm_enTtfisCmdsStocks_MStartGetServiceStateError=13,
	fc_sxm_enTtfisCmdsStocks_MStartAdd5StockSymbols=14
} fc_sxm_tenTtfisCmdsStocks;

/*=============================================================================
  =======                  STATIC FUNCTIONS                               =======
  =============================================================================*/

/*********************************************************************
 *FUNCTION:     cb_bStocksIterateSymbolCallback
 *DESCRIPTION:  This method would be invoked for every know stock symbol.
 *              By default SMS provides stocks in alphabetically sorted order
 *PARAMETER:    1) DATASERVICE_MGR_OBJECT; 2) DATASERVICE_EVENT_MASK;
 *   3) Argument passed at the time of starting Stock ticker service
 *RETURNVALUE:  Boolean value indicating that further values are needed are not
 As per the current desing, this method must always return True
 ********************************************************************/
static tBool cb_bStocksIterateSymbolCallback ( STOCK_TICKER_SERVICE_OBJECT hStockService,
		tCString cPacSymbolName, tVoid* vArg)
{
	/* Method to handle AllStocksymbols Callback.
	 * Callback directly invoked by SMS Library */
	(tVoid) hStockService;

	fc_sxm_tclStocksApp* poStocksApp = static_cast < fc_sxm_tclStocksApp* > (vArg);
	if ( OSAL_NULL != poStocksApp )
	{
		string strNewStockSymbol (cPacSymbolName);
		poStocksApp->vAppendStockSymbol ( strNewStockSymbol);

	}
	return TRUE;
}



/*********************************************************************
 *FUNCTION:     cb_bStocksDSRLIterateCallback
 *DESCRIPTION:  Method to handle AllStocksymbols Callback. This method
 * would be invoked for every know stock symbol. By default SMS provides
 * stocks in alphabetically sorted order
 *PARAMETER:    1) DATASERVICE_MGR_OBJECT; 2) DATASERVICE_EVENT_MASK;
 *   3) Argument passed at the time of starting Stock ticker service
 *RETURNVALUE:  Boolean value indicating that further values are needed are not
 As per the current desing, this method must always return True
 ********************************************************************/
static tBool cb_bStocksDSRLIterateCallback(DSRL_OBJECT hDSRL,
         DSRL_ENTRY_ID tEntryID,
         DSRL_ENTRY_STATUS_ENUM eStatus,
         DSRL_ENTRY_OBJECT hEntryObject,
         void *pvIteratorArg
         )
{
   ETG_TRACE_USR4(("fc_sxm_tclStocksApp::cb_bStocksDSRLIterateCallback Entered"));
   (tVoid) pvIteratorArg;
   (tVoid) hDSRL;
   /* Method to handle DSRL ENTRY Callback. This method is used only for internal testing
    * to validate if the entries stored in SMS and DataList Entries maintained by application
    * are correct.
    * Callback directly invoked by SMS Library */

   tUInt uPrice = 0, uPriceDelta = 0;
   tU8 u8PriceFrac = 0, u8PriceDeltaFrac = 0;
   tCString pacDirection = "";

   STOCK_MSG_PRICE_DIRECTION_ENUM eDirection = STOCK_MSG_PRICE_DIRECTION_INVALID;
   // Extract the Stock Message Object
   STOCK_MSG_OBJECT hStockMsg = DSRL_ENTRY.hStockMsg(hEntryObject);
   if (hStockMsg == STOCK_MSG_INVALID_OBJECT)
   {
      ETG_TRACE_USR4(("fc_sxm_tclStocksApp::cb_bStocksDSRLIterateCallback STOCK INVALID"));
   }
   else
   {
      // Get Stock symbol Name
      STRING_OBJECT hName = STOCK_MSG.hSymbolName(hStockMsg);
      SMSAPI_RETURN_CODE_ENUM eApiReturnValue = STOCK_MSG.eValues(hStockMsg,
               &uPrice, &u8PriceFrac,
               &uPriceDelta, &u8PriceDeltaFrac,
               &eDirection);

      string stockSymbol = "";
      fc_sxm_vCopySmsString2Stl(hName, stockSymbol);
      //		tF32 tf32Price = (tF32)uPrice + (tF32)u8PriceFrac / 100.0f;
      //		tF32 tf32Delta = (tF32)uPriceDelta + (tF32)u8PriceDeltaFrac / 100.0f;

      /* Todo: handle u8PriceFrac values above 99. In case if the Fractional value is above 100, then
       make the stocks value as invalid */

      // Get state "sign"
      tCString pacValueState = (eApiReturnValue == SMSAPI_RETURN_CODE_SUCCESS) ? "OK" : "N/A";
      // Get direction name
      switch (eDirection)
      {
         case STOCK_MSG_PRICE_DIRECTION_UP:
            {
            pacDirection = "+ve";
            break;
         }
         case STOCK_MSG_PRICE_DIRECTION_UNCHANGED:
            {
            pacDirection = "EQU";
            break;
         }
         case STOCK_MSG_PRICE_DIRECTION_DOWN:
            {
            pacDirection = "-ve";
            break;
         }

         case STOCK_MSG_PRICE_DIRECTION_INVALID:
            default:
            {
            pacDirection = "NA";
            break;
         }
      }
      // Trace messages added to print the values
      ETG_TRACE_USR4(("****START Printing formatted output from fc_sxm_tclStocksApp************"));
      ETG_TRACE_USR4(("1) STOCK SYMBOL = %s", stockSymbol.c_str()));
      //ETG_TRACE_USR4(("2) Price = %0.2f", tf32Price));
      //ETG_TRACE_USR4(("3) DELTA = %0.2f", tf32Delta));
      ETG_TRACE_USR4(("4) DIRECTION = %s", pacDirection));
      ETG_TRACE_USR4(("5) eStatus = %d", eStatus));
      ETG_TRACE_USR4(("6) pacValueState = %s", pacValueState));
      ETG_TRACE_USR4(("7) tEntryID = %u", tEntryID));
      ETG_TRACE_USR4(("****END Printing formatted output from fc_sxm_tclStocksApp**************"));
   }
   ETG_TRACE_USR4(("fc_sxm_tclStocksApp::cb_bStocksDSRLIterateCallback Exited"));
   return TRUE;
}

/*=============================================================================
  =======                   METHODS  DEFINITIONS                          =======
  =============================================================================*/
/*********************************************************************
 *FUNCTION:     fc_sxm_tclStocksApp
 *DESCRIPTION:  Constructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclStocksApp::fc_sxm_tclStocksApp():
	 fc_sxm_tclDataApp("sxmStocksApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_STOCKS, "stock_ticker", static_cast <tU16> (TR_CLASS_FC_SXM_STOCKS_APP)),
	_hStockticker(STOCK_TICKER_SERVICE_INVALID_OBJECT),
	_poFavoritesDSRL(OSAL_NULL),
	_poStocksList(OSAL_NULL),
	_u32DSSTVersion(0)
{
	// fc_sxm_tclStocksApp class Constructor
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp constructor"));
}

/*********************************************************************
 *FUNCTION:     fc_sxm_tclStocksApp
 *DESCRIPTION:  Destructor
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
fc_sxm_tclStocksApp::~fc_sxm_tclStocksApp()
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp destructor Entered"));
	// fc_sxm_tclStocksApp class Destructor
	//todo: peha: usually no try-catch is used in middleware
	_poFavoritesDSRL=OSAL_NULL;
	_poStocksList=OSAL_NULL;
	_hStockticker = STOCK_TICKER_SERVICE_INVALID_OBJECT;
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp destructor Exited"));
}

/*********************************************************************
 *FUNCTION:     vInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vInitialize(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vInitialize"));

	// Method invoked by sxm_stocks_service_app
	//Initialize the class member variables
	_hStockticker = STOCK_TICKER_SERVICE_INVALID_OBJECT;
	_poFavoritesDSRL = OSAL_NULL;
	_poStocksList =  OSAL_NULL;
	_u32DSSTVersion = 0 ;
	_setAllStockSymbols.clear();
	_strCharsToEnable.clear();
	_vecFavStockList.clear();

	//Create the stock service worker thread
	/* call vInitialize of base-class to start the worker thread */
	fc_sxm_tclBaseApp::vInitialize();
	vCreateFavoritesDsrl();
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vInitialize Exited"));
}

/*********************************************************************
 *FUNCTION:     vDeInitialize
 *DESCRIPTION:  called in cca-context by sxmapp-manager
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vDeInitialize(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDeInitialize Entered"));
	// call vDeInitialize() of base-class to stop the worker thread
	fc_sxm_tclBaseApp::vDeInitialize();

	// Deinit of all other stuff starts from here

	// _poFavoritesDSRL will be deleted by framework
	_poFavoritesDSRL=OSAL_NULL;
	FC_SXM_MEMORY_CLEAR(_poStocksList);
	_poStocksList=OSAL_NULL;

	_hStockticker = STOCK_TICKER_SERVICE_INVALID_OBJECT;
	_u32DSSTVersion = 0 ;
	_setAllStockSymbols.clear();
	_strCharsToEnable.clear();

	_vecFavStockList.clear();
	// Stop stocks timer if its already running
	vStopStocksTimer();

	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDeInitialize Exited"));
}

/*********************************************************************
 *FUNCTION:     vDispatchMsgFromQ
 *DESCRIPTION:  this method is called by the thread-function.
 For each message that shall be handled put and entry here
 *PARAMETER:    fc_sxm_tclMessage
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDispatchMsgFromQ: SID=%x action=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()),
                    poThreadMsg->u16GetActionOnly()));
	tU32 u32Action=(tU32) poThreadMsg->u32GetAction();
	// Dispatch messages basaed on action values reported
	switch (u32Action)
	{
		// use macro SXM_MSGQ_DISPATCH  for each message specific for stocks
		// method-start messages from cca-service
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartAddFavorite,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartRemoveFavorite,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilter,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartGetDataProviderInfo,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartSetListsConfig,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilterPS,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgStocksMStartClearAllFavorites,fc_sxm_tclStocksApp::instance());
	SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trPropertyStockPSetListMode,fc_sxm_tclStocksApp::instance());


		default:
		ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
					u32Action));
		break;
	}
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  handle command from sxmAppManager to start service (Mandatory)
 *PARAMETER:    fc_sxm_trMsgCmdDataAppStartService
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vStartSmsService()
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::StartService -- STOCK TICKER"));
	//Check if the stocks service is already started
	if (_hStockticker != STOCK_TICKER_SERVICE_INVALID_OBJECT)
	{
		ETG_TRACE_USR4(("Stock ticker service already started!"));
		return;
	}

	//Start the stock ticker service
	_hStockticker = STOCK_TICKER.hStart(FC_SXM_SRH_DRIVER_NAME,
			DATASERVICE_EVENT_ALL,
			cb_vDataServiceEventCallback,
			(tVoid*)this,
			DATASERVICE_OPTION_NONE );

	/*_hStockticker == STOCK_TICKER_SERVICE_INVALID_OBJECT if the call to
	 * STOCK_TICKER.hStart() is a failure**/
	if (_hStockticker == STOCK_TICKER_SERVICE_INVALID_OBJECT)
	{
		ETG_TRACE_ERR(("Could not start Stock Ticker Service service"));
	}
}
/*
  handle command from sxmAppManager to Load favorites on Service start
*/
tVoid fc_sxm_tclStocksApp::vRestoreFavorites()
{
    //call vLoadFavorites of base-class to fetch Favorite list form Persistent
    vLoadFavoriteStockSymbols();
}
/*
  method will be called after Service Stop,to Save Favorite list to Persistent
*/
tVoid fc_sxm_tclStocksApp::vPersistFavorites()
{
    //call base-class to update Favorite list to Persistent
    vSaveFavoriteStockSymbols();
}
/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd (Mandatory)
 *PARAMETER:    fc_sxm_trMsgCmdAppTtfisCmd
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
	// Method to handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd
	fc_sxm_tenTtfisCmdsStocks enMsgCode = (fc_sxm_tenTtfisCmdsStocks) prMsg->u8MsgCode;
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Entered code = %u",
			ETG_CENUM(fc_sxm_tenTtfisCmdsStocks, enMsgCode)));
	tU8 const *au8Data=prMsg->au8Data;
	switch (enMsgCode)
	{
		case fc_sxm_enTtfisCmdsStocks_MStartGetFavoriteStockSymbols:
			{
				// Handle TTFI's cmd GetFavoriteStockSymbols
				const tU32 u32StocksListID = 3;
				midw_ext_sxm_stocksfi_tclMsgGetFavoriteStockSymbolsMethodStart oMStart;
				oMStart.ListID = u32StocksListID; // this corresponds to T_e8_SxmPredefinedListIds::FAVORITES
				ETG_TRACE_USR4(("fc_sxm_tclStocksApp::ttfis-cmd GetFavoriteStockSymbols ListId = %u",oMStart.ListID ));
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartAddFavorite:
			{
				// Handle TTFI's cmd AddFavorite
				midw_ext_sxm_stocksfi_tclMsgAddFavoriteMethodStart oMStart;
				vConvertTTFisStringtoFIString((tCString)&prMsg->au8Data[0] , oMStart.StockSymbol);
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartRemoveFavorite:
			{
				// Handle TTFI's cmd RemoveFavorite
				midw_ext_sxm_stocksfi_tclMsgRemoveFavoriteMethodStart oMStart;
				vConvertTTFisStringtoFIString((tCString)&prMsg->au8Data[0] , oMStart.StockSymbol);
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartGetStockSymbolsByFilter:
			{
				// Handle TTFI's cmd GetStockSymbolsByFilter --- for normal speller support
				midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterMethodStart oMStart;
				vConvertTTFisStringtoFIString((tCString)&prMsg->au8Data[0] , oMStart.StockFilter);
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartGetStockSymbolsByFilterPS:
			{
				// Handle TTFI's cmd GetStockSymbolsByFilterPS --- for predictive speller support
				midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodStart oMStart;
				oMStart.IsBackSpace = prMsg->au8Data[0];
				vConvertTTFisStringtoFIString((tCString)&prMsg->au8Data[1] , oMStart.StockFilter);
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartGetDataProviderInfo:
			{
				// Handle TTFI's cmd GetDataProviderInfo
				midw_ext_sxm_stocksfi_tclMsgGetDataProviderInfoMethodStart oMStart;
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartSetListsConfig:
			{
				// Handle TTFI's cmd SetListsConfig
				midw_ext_sxm_stocksfi_tclMsgSetListsConfigMethodStart oMStart;
				fc_sxm_trStocksDSRLCfg rDsrlCfg;
				rDsrlCfg.u32Capacity=FC_SXM_STOCKS_FAVORITE_DSRL_CAPACITY;
				vPatchDsrlConfig(fc_sxm_enDSRLType_Favorites, rDsrlCfg);
				ETG_TRACE_USR4(("fc_sxm_tclStocksApp::fc_sxm_enTtfisCmdsStocks_MStartSetListsConfig Config.Capacity=%d",
							oMStart.Config.Capacity));
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_SetListMode:
			{
				// Handle TTFI's cmd SetListMode
				midw_ext_sxm_stocksfi_tclMsgSxmListModeSet oPSet;
				// Copy ListId param from TTFis
				oPSet.ListMode.ListID =(tU32)au8Data[0];
				oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType)au8Data[1];
				ETG_TRACE_USR4(("fc_sxm_tclStocksApp::ttfis-cmd PSetListId ListID=%d \tListMode=%d",
							oPSet.ListMode.ListID, oPSet.ListMode.Mode.enType ));
				fc_sxm_vSendSelfMessage(oPSet);
				break;
			}
			// All these methods are used only for internal testing
		case fc_sxm_enTtfisCmdsStocks_MStartClearAllFavorites:
			{
				// Handle TTFI's cmd ClearAllFavorites
				//vClearAllFavorites();
				// Avoids lint warning for not referenced enum entry
				(tVoid) fc_sxm_enTtfisCmdsStocks_MStartClearDeltaStockSymbolTable;
				midw_ext_sxm_stocksfi_tclMsgClearAllFavoritesMethodStart oMStart;
				fc_sxm_vSendSelfMessage(oMStart);
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartDSRLIterate:
			{
				// Handle TTFI's cmd DSRLeIterate
				vDsrlIterate ();
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartGetStocksServiceAndDsrlStates:
			{
				// Handle TTFI's cmd GetService and DSRL states
				vPrintStocksServiceAndDsrlStates();
				break;
			}
		case fc_sxm_enTtfisCmdsStocks_MStartAdd5StockSymbols:
			{
				// Handle TTFI's cmd Add5StocksSymbols
				vAdd5StockSymbols();
				break;
			}
		default:
			{
				// Handle TTFI's cmd Deafult case currently set to error Code Encountered within app
				// Specifically, this handles case fc_sxm_enTtfisCmdsStocks_MStartGetServiceStateError
				ETG_TRACE_USR4(("STOCK_TICKER.eErrorCode = %u",
										ETG_CENUM(DATASERVICE_ERROR_CODE_ENUM, enGetSmsServiceErrorCode())));
				break ;
			}
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Exited"));
}

/*********************************************************************
 *FUNCTION:     vPrintStocksServiceAndDsrlStates
 *DESCRIPTION:  Utility method to print Stocks service state, DSRL state and SMS version
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vPrintStocksServiceAndDsrlStates( tVoid )
{
	// Handle TTFI's cmd GetService and DSRL states
	ETG_TRACE_USR4(("--- Stocks Service State = %u",
	            ETG_CENUM(DATASERVICE_STATE_ENUM, u8getStockTickerServiceState())));
	(tVoid) fc_sxm_enTtfisCmdsStocks_MStartGetServiceStateError;
	if (OSAL_NULL != _poFavoritesDSRL)
	{
		DSRL_STATE_ENUM enDsrlState = DSRL.eState((DSRL_OBJECT)_poFavoritesDSRL->hGetSMSDSRL());
		ETG_TRACE_USR4(("--- Stocks Favorites, DSRL.eState = %u",
				ETG_CENUM(DSRL_STATE_ENUM, enDsrlState)));
	}
	else
	{
		ETG_TRACE_ERR(("_poFavoritesDSRL is NULL"));
	}

	// The following 3 lines of code is used to get only the SMS version being used;
	tU8 u8MajorVersion = 0, u8MinorVersion = 0, u8SubVersion = 0 ;
	tCString cstrSmsVersion = SMS.pacGetVersion(&u8MajorVersion, &u8MinorVersion, &u8SubVersion);
	ETG_TRACE_USR4(("/////// SMS Version = %s\\\\\\\\", cstrSmsVersion));
	ETG_TRACE_USR4(("++++++ Current DSST version  = %u \tCount of stock symbols = %u++++++++",
			_u32DSSTVersion, _setAllStockSymbols.size()));

	// Code added to test the condition NULL string returned from SMS
	string strNullStringCheck ("");
	vAppendStockSymbol ( strNullStringCheck );
}

/*********************************************************************
 *FUNCTION:     vConvertTTFisStringtoFIString
 *DESCRIPTION:  Utility method reads TTFIs string input and converts it into
 *DESCRIPTION:   FI string object.
 *PARAMETER:    Pointer to the TTFIs string object, Handle to FI string object
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vConvertTTFisStringtoFIString( tCString chInString, midw_ext_fi_tclString& rffiObj) const
{
	//Method reads TTFIs string object and converts it into FI string
	//This method cannot be invoked directly from HMI and gets executed only when user enters
	// string inputs from TTFI through cmds AddFavorite, RemoveFavorite and GetStockSymbolsByFilter
	tChar chLeadingChar[12] = { '\0' };
	string strSymbol = OSALUTIL_szSaveStringNCopy (chLeadingChar, chInString, 10);
	ETG_TRACE_USR4(("vConvertTTFisStringtoFIString stock symbol/leading char = %s", strSymbol.c_str()));
	fc_sxm_vString2Fi(strSymbol.c_str() ,rffiObj);
}

/*********************************************************************
 *FUNCTION:     u8getStockTickerServiceState
 *DESCRIPTION:  get the Current state of the Stock Ticker Service
 *PARAMETER:    None
 *RETURNVALUE:  DATASERVICE_STATE_ENUM
 ********************************************************************/
DATASERVICE_STATE_ENUM fc_sxm_tclStocksApp::u8getStockTickerServiceState ( tVoid )
{
	// Returns current state of the Stock Ticker Service
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::u8getStockTickerServiceState Entered/Exited"));
	return ( STOCK_TICKER.eState(_hStockticker) );
}

/*********************************************************************
 *FUNCTION:     bIsStockSymbolsUpdated
 *DESCRIPTION:  Method to check if the stock symbols are updated and store
 *  the new contentVersion for future reference.
 *PARAMETER:    None
 *RETURNVALUE:  TRUE if updated stock symbols are available;
 *    FALSE otherwise
 ********************************************************************/
tBool fc_sxm_tclStocksApp::bIsStockSymbolsUpdated ( tVoid )
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::bIsStocksSymbolUpdated Entered"));
	tUInt u32Version = 0;
	tBool bSymbolUpdate = STOCK_TICKER.bStockSymbolsUpdated( _hStockticker, &u32Version );
	/* If _setAllStockSymbols.size() is 0, populate it even if
	 * u32Version == _u32DSSTVersion */
	if (((tU32) u32Version == _u32DSSTVersion) && (_setAllStockSymbols.size() != 0))
	{
		ETG_TRACE_USR4(("Reference Content Version for Stocks is Same; Version = %u, bSymbolUpdate = %u", u32Version, bSymbolUpdate));
	}
	else
	{
		bSymbolUpdate = TRUE;
		ETG_TRACE_USR4(("Reference Content Version for Stocks is Different; New Version = %u, Old version = %u, bSymbolUpdate = %u", u32Version, _u32DSSTVersion, bSymbolUpdate));
		_u32DSSTVersion = (tU32) u32Version;
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::bIsStocksSymbolUpdated Exited"));
	// TRUE if updated stock symbols are available; FALSE otherwise
	return bSymbolUpdate;
}

/*********************************************************************
 *FUNCTION:     u8getAllKnownSymbols
 *DESCRIPTION:  Method to get all Known stock symbols by the Stock ticker
 *  service.
 *PARAMETER:    None
 *RETURNVALUE:  The return/error code from SMS calls
 ********************************************************************/
tU8 fc_sxm_tclStocksApp::u8getAllKnownSymbols ( tVoid )
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::u8getAllKnownSymbols Entered"));
	// Clear all stock symbols set as new entries will be appended
	SMSAPI_RETURN_CODE_ENUM u8RetCode = STOCK_TICKER.eIterateStockSymbols ( _hStockticker,
			cb_bStocksIterateSymbolCallback,
			(tVoid*)this ) ;
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::u8getAllKnownSymbols Exited, Ret Code (STOCK_TICKER.eIterateStockSymbols) = %u",
			ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, u8RetCode)));
	// Method returns SMSAPI_RETURN_CODE_SUCCESS on sucess and appropriate error code on Failure
	return (tU8) u8RetCode;
}

/****************************************************************************
*
*FUNCTION:          vClearServiceData
*
*DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.
                    This is a overridden method from the data app.
*PARAMETERS:
*
*RETURNVALUES:      None
*
******************************************************************************/
/*
   Method to be Overriden by each Service
   to clear Service specifi internal data
*/
tVoid fc_sxm_tclStocksApp::vClearServiceData()
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vClearServiceData entered"));
	_setAllStockSymbols.clear();
	_strCharsToEnable.clear();

	//First Save Current Favoriste Stock List
	vSaveFavoriteStockSymbols();

	//Then Clear Internal Data
	//_vecFavStockList.clear();


	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vClearServiceData exited"));

}
/*********************************************************************
 *FUNCTION:     vClearAllFavorites
 *DESCRIPTION:  Method to clear all Favorites stored;
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vClearAllFavorites( tVoid )
{
	// Method clears all Favorite Stock Symbols stored internally
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vClearAllFavorites Entered"));
	FAVORITES.vClearAllFavorites(_hStockticker);
	//Also Clear favorites stored persistently
	vClearFavoriteStockSymbols();
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vClearAllFavorites Exited"));
}

/*********************************************************************
 *FUNCTION:     vCreateFavoritesDsrl
 *DESCRIPTION:  Method to instantiate FavoritesDSRL object for stocks
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vCreateFavoritesDsrl ( tVoid )
{
	// Stocks feature uses only Favorites DSRL. Other DSRL types are not required for Stocks
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vCreateFavoritesDsrl Entered"));
	// Hardcoding of values as in Stocks we would handle only Favorites DSRL
	fc_sxm_tenDSRLType enDsrlType = fc_sxm_enDSRLType_Favorites;
	fc_sxm_tenListType enListType =  fc_sxm_tenListType_Favorites;
	fc_sxm_trDsrlLocation rDsrlLocation (0, 0, 0);
	fc_sxm_trStocksDSRLCfg rDsrlCfg;

	rDsrlCfg.rLocation = rDsrlLocation;
	//rDsrlCfg.u32Radius = 0;
	rDsrlCfg.enSortMethod = fc_sxm_enStocksDSRLSortMethod_FIRST_SAVED_FIRST;
	rDsrlCfg.rFilterCfg.enFilterCfg = fc_sxm_enStocksDSRLFilterConfig_NONE;
	// Hardcoding ends here

	// Get Capacity & Radius defined for Stocks Favorite type
    rDsrlCfg.u32Capacity = FC_SXM_STOCKS_FAVORITE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius = FC_SXM_STOCKS_FAVORITE_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vCreateFavoritesDsrl() Favorites Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));

	SXM_ASSERT_RETURN(OSAL_NULL==_poFavoritesDSRL);
	SXM_ASSERT_RETURN(OSAL_NULL==_poStocksList);

	// Allocate memory to hold DSRL object
	_poFavoritesDSRL = OSAL_NEW fc_sxm_tclStocksDSRL(enDsrlType);
	if ( OSAL_NULL == _poFavoritesDSRL )
	{
		ETG_TRACE_ERR(("OSAL_NULL == _poFavoritesDSRL"));
		return ;
	}
	// Register DSRL
	poRegisterDsrl ( _poFavoritesDSRL );
	// Create List
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::_poStocksList Create"));
	_poStocksList=OSAL_NEW fc_sxm_trStocksList((tU32)enListType, enListType, this);
	if ( OSAL_NULL == _poStocksList )
	{
		ETG_TRACE_ERR(("OSAL_NULL == _poStocksList"));
		return ;
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::_poStocksList setDsrl"));
	_poStocksList->vSetDsrl(_poFavoritesDSRL);
	// configure the list
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::_poStocksList setCfg"));
	fc_sxm_trStocksListConfig rListCfg;
	/* Todo: Needs to be changed if sort method has to be applied*/
	rListCfg.rSortCfg.enSortMethod = fc_sxm_enStocksDSRLSortMethod_FIRST_SAVED_FIRST;
	rListCfg.rFilterCfg.enFilterCfg = fc_sxm_enStocksDSRLFilterConfig_NONE;
	_poStocksList->vSetCfg(rListCfg);
	// activate all by configuring the dsrl
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::poDsrl  vSetNextConfig"));
	vPatchDsrlConfig(enDsrlType, rDsrlCfg);

	_poFavoritesDSRL->vSetNextConfig(rDsrlCfg);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vCreateFavoritesDsrl Exited"));
}

/*********************************************************************
 *FUNCTION:     vAppendStockSymbol
 *DESCRIPTION:  Method to append allknown stock symbols to local set
 *PARAMETER:    1) The stock symbol that has to be appended to local set
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vAppendStockSymbol( string &strStockSymbol )
{
	if ((strStockSymbol.length() > 0) && (strStockSymbol.length() <= 8)) //Fix for NIKAI-5262
	{
		/* As per UIRR, Stock_Entry_002, allowing only those stock symbols which contain characters
		 * ABCDEFGHIJKLMNOPQRSTUVWXYZ*+-.:@ to be added to the allKnownStockSymbols set.*/
		tU32 u32Found = (tU32)(strStockSymbol.find_first_not_of("ABCDEFGHIJKLMNOPQRSTUVWXYZ*+-.:@"));
		if ( string::npos == u32Found ) //Fix for NIKAI-5414
		{
			_setAllStockSymbols.insert (strStockSymbol);
			/* _strCharsToEnable will contain all keypad characters to be enabled when there is no
			 * character entered in Keypad.*/
			u32Found = (tU32)(_strCharsToEnable.find(strStockSymbol[0]));
			if ( string::npos == u32Found )
				_strCharsToEnable.push_back(strStockSymbol[0]);
		}
	}
}

/*********************************************************************
 *FUNCTION:     getStockSymbolObject
 *DESCRIPTION:  Construct a valid STOCK_SYMBOL_OBJECT for stocks with the given
 character input
 *PARAMETER:    1) The stock symbol for which the object has to be constructed
 *RETURNVALUE:  STOCK_SYMBOL_OBJECT
 ********************************************************************/
STOCK_SYMBOL_OBJECT fc_sxm_tclStocksApp::getStockSymbolObject (tCString chStockSymbol) const
{
	// Method creates a Stock symbol Object for the given stock symbol
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::getStockSymbolObject(tCString chStockSymbol) Entered"));
	size_t tLen =  OSAL_u32StringLength(chStockSymbol) + 1;
	STRING_OBJECT hStockName = STRING.hCreate(chStockSymbol, tLen);
	if ( STRING_INVALID_OBJECT == hStockName )
	{
		ETG_TRACE_ERR(("fc_sxm_tclStocksApp::getStockSymbolObject ERROR: STRING_INVALID_OBJECT"));
		return STOCK_SYMBOL_INVALID_OBJECT;
	}
	STOCK_SYMBOL_OBJECT hStockTarget = STOCK_SYMBOL.hCreate(hStockName);
	// Destroy the string object created
	STRING.vDestroy(hStockName);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::getStockSymbolObject Exited"));
	/* The ownership of hStockTarget is transferred. Its upto the calling method to clear
	 * memory alloted for hStockTarget*/
	return hStockTarget;
}

/*********************************************************************
 *FUNCTION:     enSetFavoriteStockSymbol
 *DESCRIPTION:  Set/Unset a stock symbol from Favorites DSRL
 *PARAMETER:    1) The stock symbol to be included in Favorites DSRL
				2) TRUE/FALSE to indicate if the symbol is to be
				set/unset from the Favorites list
 *RETURNVALUE:  SMSAPI_RETURN_CODE_ENUM
 ********************************************************************/
SMSAPI_RETURN_CODE_ENUM fc_sxm_tclStocksApp::enSetFavoriteStockSymbol (STOCK_SYMBOL_OBJECT hStockSymbol, tBool bSetFavorite)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::enSetFavoriteStockSymbol Entered/Exited"));
	/* A request to SMS is sent for setting/unsetting the
	 * stock symbol. SMS returns True if the action is successful*/
	return (FAVORITES.eSetFavoriteByStockSymbol(hStockSymbol,
				_hStockticker, bSetFavorite));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_enTtfisCmdsStocks_MStartGetFavoriteStockSymbols
 *PARAMETER:    1) Pointer to fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols) Entered"));
	// Handle for HMI request GetFavoriteStocksSymbols
	vStopStocksTimer();
	midw_ext_sxm_stocksfi_tclMsgGetFavoriteStockSymbolsMethodStart const &oMStart=prMsg->oFiMsg;
	fc_sxm_trStocksListConfig rListCfg;
	tU32 u32ListId=oMStart.ListID;
	rListCfg.rSortCfg.enSortMethod = fc_sxm_enStocksDSRLSortMethod_FIRST_SAVED_FIRST;
	rListCfg.rFilterCfg.enFilterCfg = fc_sxm_enStocksDSRLFilterConfig_NONE;
	fc_sxm_trStocksList::vSetCfg(u32ListId, rListCfg);

	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols) vSendList; ListID = %d", oMStart.ListID));
	vSendList(u32ListId, prMsg->rAdressing);
	// methodResult sent through vEmit
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetFavoriteStockSymbols) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartAddFavorite
 *PARAMETER:    1) Pointer to AddFavorite message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite) Entered"));
#ifdef FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("Allocating memory for Class TimeProfile for AddFavorite"));
	fc_sxmTimeProfile *poTimeProfile = OSAL_NEW  fc_sxmTimeProfile("Stocks::AddFavorite");
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Memory succesfully alloted for Class TimeProfile  for AddFavorite"));
		poTimeProfile->vStart();
	}
	ETG_TRACE_USR4(("TimeProfile->vStart() complete  for AddFavorite"));
#endif //FC_SXM_TIME_PROFILE

	// Handle for HMI request AddFavorite
	vStopStocksTimer();
	midw_ext_sxm_stocksfi_tclMsgAddFavoriteMethodStart const &oMStart=prMsg->oFiMsg;
	tString chStockSymbol = oMStart.StockSymbol.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite)  chStockSymbol = %s", chStockSymbol));

	STOCK_SYMBOL_OBJECT hStockSymbol = getStockSymbolObject ( chStockSymbol );

	if ( (STOCK_SYMBOL_INVALID_OBJECT != hStockSymbol) && (OSAL_NULL != _poStocksList))
	{
		midw_ext_sxm_stocksfi_tclMsgAddFavoriteMethodResult oMRes;

		/* The call _poStocksList->bIsFavoriteStockSymbol is not added inside
		 * if (enFavRetCode == SMSAPI_RETURN_CODE_SUCCESS)
		 * to prevent time taken after setting favorite symbol and receving DSRL callback */

		string strCompareVal (chStockSymbol);
		tBool bAlreadyFavorite = FALSE;


			bAlreadyFavorite = _poStocksList->bIsFavoriteStockSymbol (strCompareVal);

		SMSAPI_RETURN_CODE_ENUM enFavRetCode = enSetFavoriteStockSymbol ( hStockSymbol, TRUE);
		ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite)  enFavRetCode = %u",
				ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enFavRetCode)));
		if ( SMSAPI_RETURN_CODE_SUCCESS == enFavRetCode )
		{
			oMRes.Status = TRUE;
            //If Save is success from SMS
            //Also Add this entry to Persistent data
			vAddFavorite(strCompareVal);
			if (FALSE == bAlreadyFavorite)
			{
				// Do this only if the stock symbol is not already a favorite
				//Set the DSRL state, until it becomes READY
				_poStocksList->vOnSmsDsrlState(fc_sxm_enDsrlState_Updating);
			}
		}
		else
		{
			oMRes.Status = FALSE;
		}
		STOCK_SYMBOL.vDestroy(hStockSymbol);
		ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite)  oMRes.Status = %d", oMRes.Status));
		// Send the method result to Stocks service
		fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

	}
	else
	{
		// Todo: error handling and return value thru MethodError
		ETG_TRACE_ERR(("STOCK_SYMBOL_INVALID_OBJECT --> FAVORITES.eSetFavoriteByStockSymbol() Not processed"));
		fc_sxm_tclStocksService::vSendError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_Stock_ErrorTypes::FI_EN_STOCK_ERRORTYPE_ADDFAVORITE_ERROR);
	}
	// Delete memory alloted for chStockSymbol
	OSAL_DELETE [] chStockSymbol;
	// methodResult would contain the success/failure of Adding a stock symbol to Favorites List
#ifdef FC_SXM_TIME_PROFILE
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Taking Profile Info for AddFavorite"));
		poTimeProfile->vStop();
		// Clear memory alloted for Time Profiling
		OSAL_DELETE poTimeProfile;
	}
	ETG_TRACE_USR4(("TimeProfile->vStop() complete for AddFavorite"));
#endif //FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartAddFavorite) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartRemoveFavorite
 *PARAMETER:    1) Pointer to RemoveFavorite message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite) Entered"));
#ifdef FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("Allocating memory for Class TimeProfile  for RemoveFavorite"));
	fc_sxmTimeProfile *poTimeProfile = OSAL_NEW  fc_sxmTimeProfile("Stocks::RemoveFavorite");
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Memory succesfully alloted for Class TimeProfile  for RemoveFavorite"));
		poTimeProfile->vStart();
	}
	ETG_TRACE_USR4(("TimeProfile->vStart() complete  for RemoveFavorite"));
#endif //FC_SXM_TIME_PROFILE

	// Handle for HMI request RemoveFavorite
	vStopStocksTimer();
	midw_ext_sxm_stocksfi_tclMsgRemoveFavoriteMethodStart const &oMStart=prMsg->oFiMsg;
	tString chStockSymbol = oMStart.StockSymbol.szGet(midw_ext_fi_tclString::FI_EN_UTF8);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite)  chStockSymbol = %s", chStockSymbol));
	STOCK_SYMBOL_OBJECT hStockSymbol = getStockSymbolObject ( chStockSymbol );

	if ( (STOCK_SYMBOL_INVALID_OBJECT != hStockSymbol) && (OSAL_NULL != _poStocksList))
	{
		midw_ext_sxm_stocksfi_tclMsgRemoveFavoriteMethodResult oMRes;
		oMRes.Status = FALSE;

		SMSAPI_RETURN_CODE_ENUM enFavRetCode = enSetFavoriteStockSymbol ( hStockSymbol, FALSE);
		ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite)  enFavRetCode = %u",
						ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enFavRetCode)));

		if ( SMSAPI_RETURN_CODE_SUCCESS == enFavRetCode )
		{
			oMRes.Status = TRUE;
			//If Removal success from SMS
			//Also Remove this entry from Persistent data
			vRemoveFavorite(chStockSymbol);
			_poStocksList->vOnSmsDsrlState(fc_sxm_enDsrlState_Updating);
		}
		STOCK_SYMBOL.vDestroy(hStockSymbol);
		ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite)  oMRes.Status = %d", oMRes.Status));

		// Send the method result to Stocks service
		fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	}
	else
	{
		// Todo: error handling and return value thru MethodError
		ETG_TRACE_ERR(("STOCK_SYMBOL_INVALID_OBJECT --> FAVORITES.eSetFavoriteByStockSymbol() Not processed"));
		fc_sxm_tclStocksService::vSendError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_Stock_ErrorTypes::FI_EN_STOCK_ERRORTYPE_REMOVEFAVORITE_ERROR);
	}
	// Delete memory alloted for chStockSymbol
	OSAL_DELETE [] chStockSymbol;

	// methodResult would contain the success/failure of Removing a stock symbol to Favorites List
#ifdef FC_SXM_TIME_PROFILE
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Taking Profile Info for RemoveFavorite"));
		poTimeProfile->vStop();
		// Clear memory alloted for Time Profiling
		OSAL_DELETE poTimeProfile;
	}
	ETG_TRACE_USR4(("TimeProfile->vStop() complete for RemoveFavorite"));
#endif //FC_SXM_TIME_PROFILE

	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartRemoveFavorite) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartGetStockSymbolsByFilterPS
 *PARAMETER:    1) Pointer to GetStockSymbolsByFilter message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilterPS const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilterPS) Entered"));
#ifdef FC_SXM_TIME_PROFILE
	fc_sxmTimeProfile *poTimeProfile = OSAL_NEW  fc_sxmTimeProfile("Stocks::GetStockSymbolsByFilterPS");
	if (OSAL_NULL != poTimeProfile)
	{
		poTimeProfile->vStart();
	}
#endif //FC_SXM_TIME_PROFILE
	// Method Handles HMI request for Getting Stock Symbols BY Filter for predictive speller
	vStopStocksTimer();
	midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodStart const &oMStart=prMsg->oFiMsg;
	tString chStockFilter = oMStart.StockFilter.szGet(midw_ext_fi_tclString::FI_EN_UTF8);

	midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult oMRes;
	string strSearchFilter(chStockFilter);
	const tU32 u32Length = (tU32)(strSearchFilter.length());
	ETG_TRACE_USR4(("u32Length = %u", u32Length));
	if ((!u32Length) && (TRUE == bIsStockSymbolsUpdated()))
	{
		/* DSST version updated check is performed only when speller screen transition happens
		 * from Stocks screen to speller screen*/
		_setAllStockSymbols.clear();
		_strCharsToEnable.clear();
		tU8 u8ErrorCode = u8getAllKnownSymbols();
		ETG_TRACE_USR4(("SPELLER LAUNCHED --- Checking if DSST is updated === u8getAllKnownSymbols ErrorCode = %u", u8ErrorCode));
	}

   if ((!u32Length) || ((1 == u32Length) && (TRUE == oMStart.IsBackSpace)))
   {
	   /* If there are no characters to be displayed in speller screen's text edit box,
	    * Execute vGetStocksByFilterForZeroChars for faster processing*/
	   vGetStocksByFilterForZeroChars(oMRes);
   }
   else
   {
	   /* If there are >= 1 characters to be displayed in speller screen's text edit box,
	    * these lines of code are executed*/
      ETG_TRACE_USR4(("Starting to iterate through a list of %u Stock Symbols", _setAllStockSymbols.size()));
      ETG_TRACE_USR4(("Trying for stock symbol %s", strSearchFilter.c_str()));
      vGetStocksByFilter(strSearchFilter, oMRes, oMStart.IsBackSpace);
      while ( (0 == oMRes.Count) && (strSearchFilter.length() > 0) )
      {
    	  // Defensive code added to make sure that a stock entry with 0 size is never sent back to HMI
    	  ETG_TRACE_USR4(("Defensive code to make sure that stock entry with 0 list size is never sent back to HMI ... Invoking vGetStocksByFilter allover again"));
    	  strSearchFilter.erase(strSearchFilter.end() - 1 );
    	  vGetStocksByFilter(strSearchFilter, oMRes, FALSE);
      }
   }

	// Send the method result to Stocks service
	fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	OSAL_DELETE [] chStockFilter;
#ifdef FC_SXM_TIME_PROFILE
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Taking Profile Info for GetStockSymbolsByFilter"));
		poTimeProfile->vStop();
		OSAL_DELETE poTimeProfile;
	}
#endif //FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilterPS) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartClearAllFavorites
 *PARAMETER:    1) Pointer to ClearAllFavorites message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartClearAllFavorites const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartClearAllFavorites) Entered"));
	// Method Handles HMI request for Getting Stock Symbols BY Filter for predictive speller
	vStopStocksTimer();
	vClearAllFavorites();
	midw_ext_sxm_stocksfi_tclMsgClearAllFavoritesMethodResult oMRes;
	// Send the method result to Stocks service
	fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartClearAllFavorites) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartGetStockSymbolsByFilter
 *PARAMETER:    1) Pointer to GetStockSymbolsByFilter message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilter const *prMsg)
{
#ifdef FC_SXM_TIME_PROFILE
	fc_sxmTimeProfile *poTimeProfile = OSAL_NEW  fc_sxmTimeProfile("Stocks::GetStockSymbolsByFilter");
	if (OSAL_NULL != poTimeProfile)
	{
		poTimeProfile->vStart();
	}
#endif //FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilter) Entered"));

	/* Method Handles HMI request for Getting Stock Symbols BY Filter for normal speller or when
	 * "List"/"OK" Button is pressed in Predictive Speller*/
	midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterMethodStart const &oMStart=prMsg->oFiMsg;
	tString chStockFilter = oMStart.StockFilter.szGet(midw_ext_fi_tclString::FI_EN_UTF8);

	midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterMethodResult oMRes;
	string strSearchFilter(chStockFilter);

	ETG_TRACE_USR4(("Starting to iterate through a list of %d Stock Symbols", _setAllStockSymbols.size()));
	ETG_TRACE_USR4(("Trying for stock symbol %s", strSearchFilter.c_str()));

	vGetStocksByFilter(strSearchFilter, oMRes.StockFilterList.StockFilterList);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess() oMRes.StockFilterList Count = %u", oMRes.StockFilterList.StockFilterList.size()));

	// Send the method result to Stocks service
	fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	OSAL_DELETE [] chStockFilter;

#ifdef FC_SXM_TIME_PROFILE
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Taking Profile Info for GetStockSymbolsByFilter"));
		poTimeProfile->vStop();
		OSAL_DELETE poTimeProfile;
	}
#endif //FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetStockSymbolsByFilter) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartGetDataProviderInfo
 *PARAMETER:    1) Pointer to GetDataProviderInfo message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetDataProviderInfo const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetDataProviderInfo) Entered"));
#ifdef FC_SXM_TIME_PROFILE
	fc_sxmTimeProfile *poTimeProfile = OSAL_NEW  fc_sxmTimeProfile("Stocks::GetDataProviderInfo");
	if (OSAL_NULL != poTimeProfile)
	{
		poTimeProfile->vStart();
	}
#endif //FC_SXM_TIME_PROFILE

	// Method Handles HMI request for GetDataProviderInfo
	midw_ext_sxm_stocksfi_tclMsgGetDataProviderInfoMethodResult oMRes;
	STRING_OBJECT hDataProviderInfo = STOCK_TICKER.hDataProviderInfo(_hStockticker);
	/* STRING_INVALID_OBJECT != hDataProviderInfo check is required because at times even though
	 * DSST version > 0, DataProvider returns invalid object*/
	if ( STRING_INVALID_OBJECT != hDataProviderInfo)
	{
	    string strDataProvider = "";
        fc_sxm_vCopySmsString2Stl(hDataProviderInfo, strDataProvider);
		ETG_TRACE_USR4(("------- DataProviderInfo =  %s ----------", strDataProvider.c_str()));
		fc_sxm_vString2Fi(strDataProvider.c_str() ,oMRes.DataProviderInfo);
		fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	}
	else
	{
		/* Todo: Error Handling*/
		ETG_TRACE_ERR(("STRING_INVALID_OBJECT --> STOCK_TICKER.hDataProviderInfo() Not processed"));
		fc_sxm_tclStocksService::vSendError(prMsg->rAdressing,
				(tInt) midw_ext_fi_tcl_e8_Stock_ErrorTypes::FI_EN_STOCK_ERRORTYPE_GETDATAPROVIDERINFO_ERROR);
	}
#ifdef FC_SXM_TIME_PROFILE
	if (OSAL_NULL != poTimeProfile)
	{
		ETG_TRACE_USR4(("Taking Profile Info for GetDataProviderInfo"));
		poTimeProfile->vStop();
		// Clear memory alloted for Time Profiling
		OSAL_DELETE poTimeProfile;
	}
#endif //FC_SXM_TIME_PROFILE
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartGetDataProviderInfo) Exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgStocksMStartSetListsConfig
 *PARAMETER:    1) Pointer to SetListsConfig message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartSetListsConfig const  *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartSetListsConfig) Entered"));
	// Method Handles HMI request for SetListsConfig
	midw_ext_sxm_stocksfi_tclMsgSetListsConfigMethodStart const &oFiMsg = prMsg->oFiMsg;
	tU32 u32Capacity=oFiMsg.Config.Capacity;
	// configure dsrl with new capacity
	fc_sxm_trStocksDSRLCfg rDsrlCfg;
	fc_sxm_trDsrlLocation rDsrlLocation (0,0,0);
	rDsrlCfg.u32Capacity = u32Capacity;
	/* Setting lat,long, radius for the dsrl all to be 0 as it wont be needed for stocks application.
	 * Code is added only to maintain uniformity across all SXM MW dataservices*/
	rDsrlCfg.rLocation = rDsrlLocation ;
	// configure the radius & Sort Types
	rDsrlCfg.enSortMethod = fc_sxm_enStocksDSRLSortMethod_FIRST_SAVED_FIRST;
	rDsrlCfg.u32Radius=0;

	// Prepare method result with success Status
	midw_ext_sxm_stocksfi_tclMsgSetListsConfigMethodResult oMRes;

	// We are going to change the capacity only for Near by dsrl
	// And Check if _poFavoritesDSRL is created
	if( OSAL_NULL != _poFavoritesDSRL)
	{
		// So Set new capacity to NearBy DSRL
		_poFavoritesDSRL->vSetNextConfig(rDsrlCfg);
		// Set the Method reult to TRUE
		oMRes.Status = TRUE;
	}
	else
	{
		// Set the Method reult to False such that Capacity not been set
		oMRes.Status = FALSE;
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartSetListsConfig) call enSendFiMessage"));
	// Send the method result to our service
	fc_sxm_tclStocksService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgStocksMStartSetListsConfig) exit"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Set property to Handle the List Mode
 *PARAMETER:  Pointer to fc_sxm_trPropertyStockPSetListMode message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcess(fc_sxm_trPropertyStockPSetListMode  const *prMsg)
{
	ETG_TRACE_USR4(( "vProcess fc_sxm_trPropertyStockPSetListMode" ));
	/* Method Handles HMI request for SetListMode
	 * For Stocks Feature only   fc_sxm_enListMode_Release and fc_sxm_enListMode_Activate
	 * can be expected from HMI*/
	midw_ext_sxm_stocksfi_tclMsgSxmListModeSet const &oPSet=prMsg->oFiMsg;
	// Initialize default values
	tU32 u32ListId = oPSet.ListMode.ListID;
	ETG_TRACE_USR4(( "fc_sxm_trPropertyStockPSetListMode: ListID %u",u32ListId));
	fc_sxm_tenListMode enListMode = (fc_sxm_tenListMode)oPSet.ListMode.Mode.enType;
	ETG_TRACE_USR4(( "fc_sxm_trPropertyStockPSetListMode: enListMode %u",
			ETG_CENUM(fc_sxm_tenListMode, enListMode)));
	if ( fc_sxm_enListMode_Release == enListMode )
		vStopStocksTimer();
	vSetListMode(u32ListId, prMsg->rAdressing,enListMode);
}

/*********************************************************************
 *FUNCTION:     vOnListStateChanged
 *DESCRIPTION:  Method to set and notify StockStatus property to HMI
 *PARAMETER:    1) Invoked on DSRL state changes
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vOnListStateChanged(fc_sxm_tclDataListBase *poDataList)
{
	(tVoid)poDataList;
	//Property update that would be sent to HMI on List Status making a transition to Ready State
	ETG_TRACE_USR4(("^^^^^^fc_sxm_tclStocksApp::vOnListStateChanged() Entered^^^^^^^"));
	fc_sxm_tcl_trStocksPropertyStockStatus oProperty;
	midw_ext_sxm_stocksfi_tclMsgStockStatusStatus &oFiMsg=oProperty.oFiMsg;

	oGetDataListsInfo().vCopyListInfosToFi(oFiMsg.StockStatus); // use conversion-operator
	fc_sxm_tclStocksProperties::instance()->oStockStatus.vSetAndNotify(oProperty);
	ETG_TRACE_USR4(("^^^^^^^^^ fc_sxm_tclStocksApp::vOnListStateChanged() Exited^^^^^^^^^^^^"));
}

/*********************************************************************
 *FUNCTION:     vDsrlIterate
 *DESCRIPTION:  Method that invokes DSRL iterate. Used only for internal testing
 of stocks feature
 *PARAMETER:    None
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vDsrlIterate(tVoid) const
{
	/* THIS METHOD IS USED ONLY FOR INTERNAL TESTING.
	 * The values printed from method cb_bStocksDSRLIterateCallback in TTFIs can be
	 * checked with the actual values stored by Data Lists Entry*/
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDsrlIterate() Entered"));
	SMSAPI_RETURN_CODE_ENUM enReturnCode = SMSAPI_RETURN_CODE_ERROR;
	if (OSAL_NULL != _poFavoritesDSRL)
	{
		enReturnCode = DSRL.eIterate(_poFavoritesDSRL->hGetSMSDSRL(),
				cb_bStocksDSRLIterateCallback,
				OSAL_NULL);
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vDsrlIterate() Exited, DSRL.eIterate retCode = %u",
			ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, enReturnCode)));
}

/*********************************************************************
 *FUNCTION:     vGetStocksByFilterForZeroChars
 *DESCRIPTION:  Method to retrieve a list of stock symbols by Filter
 *                for zero character input in speller
 *PARAMETER:    Handle to the MethodResult of GetStocksSymbolsByFilter Method
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vGetStocksByFilterForZeroChars(
   midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult &oMRes ) const
{
   ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilterForZeroChars() Entered ---> CharsToEnable = %s",
      _strCharsToEnable.c_str()));
   // Method handles requests for zero characters entered in predictive speller's text entry box
   fc_sxm_vString2Fi( _strCharsToEnable.c_str(), oMRes.AllCharsToEnable);
   fc_sxm_vString2Fi( "" , oMRes.DisplayString);
   oMRes.Count = (tU32) _setAllStockSymbols.size();
   ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilterForZeroChars() Exit oMRes.Count = %u", oMRes.Count));
}

/*********************************************************************
 *FUNCTION:     vGetStocksByFilter
 *DESCRIPTION:  Method fills stocks symbols if the stocks count <= 6
 *PARAMETER:    1) Handle to the MethodResult of GetStocksSymbolsByFilter Method
 *              2) vectListBuffer which contains filled up values
 *              3) The listcount of the stocks symbol set for the given leading character
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vSpellerFillStocksList(midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult &oMRes,
		   vector < string >& vectListBuffer, const tU32 u32ListCount) const
{
	ETG_TRACE_USR4(("S P E L L E R -->fc_sxm_tclStocksApp::vSpellerFillStocksList() Entered"));
	const tU32 u32MaxListSize = 6;
	oMRes.StockFilterList.StockFilterList.clear();
	/* Fill in the  oMRes.StockFilterList.StockFilterList only if the size is <= 6 as speller would make an
	 * automatic transition from speller screen to stocks list screen if the number of stock symbols is less
	 * than 6. */
	if (u32ListCount <= u32MaxListSize)
	{
		ETG_TRACE_USR4(("ListSize found to be less than or equal to 6; Actual LisSize = %u",  u32ListCount));
		for (vector < string >::const_iterator vect_iterator = vectListBuffer.begin();
				vect_iterator != vectListBuffer.end();
				++vect_iterator)
		{
			ETG_TRACE_USR4(("Adding Symbol to Filter List %s", (*vect_iterator).c_str()));
			midw_ext_fi_tclString fiStrSymbol;
			fc_sxm_vString2Fi( (*vect_iterator).c_str() ,fiStrSymbol);
			oMRes.StockFilterList.StockFilterList.push_back(fiStrSymbol);
		}
	}
	vectListBuffer.clear();
	ETG_TRACE_USR4(("S P E L L E R <---fc_sxm_tclStocksApp::vSpellerFillStocksList() Exit: StockFilterList Count = %u",
			oMRes.StockFilterList.StockFilterList.size()));
}

/*********************************************************************
 *FUNCTION:     vSpellerFillCharstoEnable
 *DESCRIPTION:  Method fills characters to be enabled in keypad
 *PARAMETER:    1) Handle to the MethodResult of GetStocksSymbolsByFilter Method
 *              2) set of all characters to be enabled
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vSpellerFillCharstoEnable(midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult &oMRes,
		   set < tChar >& setEnableChar) const
{
	//Fill characters to be enabled in Keypad
	string strChars("");
	for (set <tChar>::const_iterator charsIter=setEnableChar.begin(); charsIter != setEnableChar.end(); ++charsIter)
		strChars += *charsIter;
	fc_sxm_vString2Fi( strChars.c_str() , oMRes.AllCharsToEnable);
	setEnableChar.clear();
	ETG_TRACE_USR4(("List of Characters to be displayed in Keypad: %s", strChars.c_str()));
}

/*********************************************************************
 *FUNCTION:     vSpellerFillDispStringandCount
 *DESCRIPTION:  Method fills speller display string and count of stock symbols
 *PARAMETER:    1) Handle to the MethodResult of GetStocksSymbolsByFilter Method
 *              2) Modified string search filter
 *              3) The listcount of the stocks symbol set for the given leading character
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vSpellerFillDispStringandCount(midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult &oMRes,
		   const string &strSearchFilter, const tU32 u32ListCount ) const
{
	//Fill speller display string and count of stock symbols
	fc_sxm_vString2Fi( strSearchFilter.c_str() , oMRes.DisplayString);
	ETG_TRACE_USR4(("Actual display string: %s", strSearchFilter.c_str()));
	oMRes.Count = u32ListCount;
	ETG_TRACE_USR4(("oMRes.Count = %u", u32ListCount));
}

/*********************************************************************
 *FUNCTION:     vGetStocksByFilter
 *DESCRIPTION:  Method to retrieve a list of stock symbols by Filter
 *PARAMETER:    1) Stock symbol starting charater[s]
 *              2) Handle to the MethodResult of GetStocksSymbolsByFilter Method
 *              3) True if backspace is entered by user, False otherwise
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vGetStocksByFilter(string &strSearchFilter,
		midw_ext_sxm_stocksfi_tclMsgGetStockSymbolsByFilterPSMethodResult &oMRes,
		tBool bIsBackSpace) const
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilter() Entered: strSearchFilter = %s", strSearchFilter.c_str()));
	// Local variables declaration
	vector < string > vectListBuffer;
	set < tChar > setEnableChar;
	tBool bLoopOn = FALSE;
	tU32 u32ListCount = 0;
	tChar chPrevious = ' ';
	const tU32 u32MaxListSize = 6;

	// Trim the last character in case if Backspace is entered
	if (bIsBackSpace)
	{
		strSearchFilter.erase(strSearchFilter.end() - 1 );
		ETG_TRACE_USR4(("INIT As Backspace is entered, trimming the searchFilter.: Updated strSearchFilter = %s", strSearchFilter.c_str()));
	}
	set < string >::iterator iter = _setAllStockSymbols.lower_bound(strSearchFilter);
	tU32 u32StrLen = (tU32)(strSearchFilter.length());
	// Step 1: Get list of all stock symbols beginning with the leading character
	/* First do loop is added to avoid a defect when the last item from _setAllStockSymbols is currently
	 * displayed and the user presses backspace. ZZZ and ZZZ.TEST are the last 2 items in the AllStockSymbols.
	 * If ZZZ.TEST is currently displayed and user presses backspace then ZZZ must be displayed in keypad with
	 * 2 items in list*/
	do {
		while (iter != _setAllStockSymbols.end())
		{
			if ((*iter).compare(0, u32StrLen, strSearchFilter) == 0)
			{
				//ETG_TRACE_USR4(("Adding Symbol to Filter List %s", (*iter).c_str()));
				if (u32ListCount < u32MaxListSize)
				{
					// Add only 6 items to this list
					vectListBuffer.push_back(*iter);
				}
				++u32ListCount ;

				//This if condition is required to avoid out_of_range exception
				if ( ((*iter).length() > u32StrLen) && (chPrevious != (*iter)[u32StrLen]) )
				{
					chPrevious = (*iter)[u32StrLen] ;
					setEnableChar.insert(chPrevious);
				}
				*iter++;
			}
			else if ( (bIsBackSpace) && (u32ListCount <= 1) && (u32StrLen) )
			{
				strSearchFilter.erase(strSearchFilter.end() - 1 );
				FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS( setEnableChar, vectListBuffer, strSearchFilter,
						iter, _setAllStockSymbols, chPrevious, u32StrLen, u32ListCount);
				ETG_TRACE_USR4(("Handling for BACKSPACE  from LOOP1, the searchFilter is now reduced to: %s", strSearchFilter.c_str()));
				ETG_TRACE_USR4(("Handling for BACKSPACE  from LOOP1, Setting iterator to %s", (*iter).c_str()));
			}
			else
			{
				ETG_TRACE_USR4(("No more matches for the given searchFilter; Exiting for value: %s", (*iter).c_str()));
				break ;
			}
		} // end of while (iter != _setAllStockSymbols.end())

		if ( (bIsBackSpace) && (u32ListCount <= 1) && (u32StrLen) )
		{
			strSearchFilter.erase(strSearchFilter.end() - 1 );
			FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS( setEnableChar, vectListBuffer, strSearchFilter,
					iter, _setAllStockSymbols, chPrevious, u32StrLen, u32ListCount);
			bLoopOn = TRUE;
			ETG_TRACE_USR4(("Handling for BACKSPACE from LOOP2, the searchFilter is now reduced to: %s", strSearchFilter.c_str()));
			ETG_TRACE_USR4(("Handling for BACKSPACE  from LOOP2, Setting iterator to %s", (*iter).c_str()));
		}
		else if ( (!bIsBackSpace) && (setEnableChar.size() == 1) && (vectListBuffer.size() != 0)
				&& ((*vectListBuffer.begin()).length() != u32StrLen))
		{
			// This part of code would be executed if the user enters "ZZZ." in speller and "ZZZ.TEST" is the
			// next matching character in stocks list.
			// If-else part added as a fix for NIKAI-5918
			strSearchFilter += *setEnableChar.begin();
			FC_SXM_STOCKS_CLEARTEMPSTOCKSYMBOLS( setEnableChar, vectListBuffer, strSearchFilter,
								iter, _setAllStockSymbols, chPrevious, u32StrLen, u32ListCount);
			bLoopOn = TRUE;
			ETG_TRACE_USR4(("Handling for setEnableChar.size() == 1  from LOOP3, Setting iterator to %s", (*iter).c_str()));
		}
		else
		{
			bLoopOn = FALSE;
		}
	} while (bLoopOn); // end of do ... while ()
	vSpellerFillStocksList(oMRes, vectListBuffer, u32ListCount);
	// Step 2: Get all characters set that have to be enabled in GUI
	vSpellerFillCharstoEnable(oMRes, setEnableChar);
	// Step 3: Get the appropriate display string and Count
	vSpellerFillDispStringandCount(oMRes, strSearchFilter, u32ListCount);
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilter() Exited, vecStocksSymbol Count = %u\toMRes.StockFilterList.StockFilterList.size = %u",
			oMRes.Count, oMRes.StockFilterList.StockFilterList.size()));
}

/*********************************************************************
 *FUNCTION:     vGetStocksByFilter
 *DESCRIPTION:  Method to retrieve a list of stock symbols by Filter
 *PARAMETER:    1) Stock symbol starting charater[s]
 *              2) The Vector that needs to be populated.
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vGetStocksByFilter(const string &strSearchFilter, fc_sxm_trVectorSymbolFilter& vecStocksSymbol) const
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilter() Entered: strSearchFilter = %s", strSearchFilter.c_str()));
	/* Method handles HMI requests for
	 * 1) "LIST"/"OK" Button press in predictive speller
	 * 2) Provide stock symbol set for normal speller operation*/
	set < string >::iterator iter = _setAllStockSymbols.lower_bound(strSearchFilter);
	while (iter != _setAllStockSymbols.end())
	{
		if ((*iter).compare(0, strSearchFilter.length(), strSearchFilter) == 0)
		{
			//ETG_TRACE_USR4(("Adding Symbol to Filter List %s", (*iter).c_str()));
			midw_ext_fi_tclString fiStrSymbol;
			fc_sxm_vString2Fi( (*iter).c_str() ,fiStrSymbol);
			vecStocksSymbol.push_back (fiStrSymbol);
			*iter++;
		}
		else
		{
			ETG_TRACE_USR4(("No more matches for the given searchFilter; Exiting for value: %s", (*iter).c_str()));
			break ;
		}
	}
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vGetStocksByFilter() Exited, vecStocksSymbol Count = %u", vecStocksSymbol.size()));
}

// Application get notified about the new fi-service-state that shall be sent to clients
tVoid fc_sxm_tclStocksApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState)
{
	// Property update on ServiceState Changed status to HMI
	fc_sxm_tcl_trStocksPropertySxmServiceStatus oProperty;
	oProperty.oFiMsg.DataServiceState=oFiState;
	fc_sxm_tclStocksProperties::instance()->oSxmServiceStatus.vSetAndNotify(oProperty);
}

/*********************************************************************
 *FUNCTION:     vAdd5StockSymbols
 *DESCRIPTION:  Add 5 stock symbols at one shot. Used only for internal
 *              MIDW testing. To add 20 stock symbols, invoke this method
 *              thru TTFIs cmd, FC_SXM_STOCKS_MSTART_ADD5STOCKSYMBOLS 4 times
 *PARAMETER:    None
 *RETURNVALUE:  None
 *NOTE		 :  This is a test function only. Would not be made available
 for Main Stocks Feature Release.
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vAdd5StockSymbols( tVoid )
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vAdd5StockSymbols Entered"));
	// Method used for internal MIDW testing
	tCString local_Test_Values[20] = {"AAPL", "FB", "GOOG", "MANH", "TWX", "BBBY", "PEP", "ACN", "WMT", "GM",
			"LU", "RIMM", "AQZ.TEST", "KLAC", "YHOO", "XLNX", "SBUX", "LIFE", "COST", "AMZN"};
	static tU8 u8TestIndex = 0u;
	if (u8TestIndex > 19u)
	{
		u8TestIndex = 0u;
	}
	tU8 u8Count = 0u;
	/* Method adds 5 stock symbols at a single shot thereby reducing Developer effort.
	 * This method would be effective if its used between 1 - 4 times after start of an application*/
	do {
		tCString chStockSymbol(local_Test_Values[u8TestIndex]);
		STOCK_SYMBOL_OBJECT hStockSymbol = getStockSymbolObject ( chStockSymbol );
		if ( STOCK_SYMBOL_INVALID_OBJECT != hStockSymbol )
		{
			SMSAPI_RETURN_CODE_ENUM enFavRetCode = enSetFavoriteStockSymbol ( hStockSymbol, TRUE);
			ETG_TRACE_USR4(("Added stock symbol: %s:", local_Test_Values[u8TestIndex++]));
			ETG_TRACE_USR4(("Added stock symbol Error code: %d:", enFavRetCode));
		}
		else
		{
			ETG_TRACE_ERR(("STOCK_SYMBOL_INVALID_OBJECT --> FAVORITES.eSetFavoriteByStockSymbol() Not processed"));
		}
	} while (++u8Count < 5u ) ;

	/* Though there is a known memory leak in not clearing hStockSymbol, it is not done to avoid crash happening
	 * The method enSetFavoriteStockSymbol can be called only from HMI. As this is a test function, memory clearing is not performed*/
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vAdd5StockSymbols Exited"));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgDataSmsEvtServiceUp
 *PARAMETER:    1) Pointer to DataSmsEvtServiceUp message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(trMsgDataSmsEvtServiceUp) ENTERED"));
	(tVoid) prMsg ;
	// Method invoked on stocks service making a transition from INITIAL to READY state
	/* Todo: do error handling for SMS API's */
	tBool bSymbolUpdate = bIsStockSymbolsUpdated();
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcessServiceState --> DATASERVICE_STATE_READY STOCK_TICKER.bStockSymbolsUpdated = %d", bSymbolUpdate));

	tU8 u8ErrorCode = u8getAllKnownSymbols();
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(trMsgDataSmsEvtServiceUp) EXITED --> DATASERVICE_STATE_READY SMSAPI_RETURN_CODE_ENUM STOCK_TICKER.eIterateStockSymbols = %u",
			ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, u8ErrorCode)));
}

/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  Method to send FI message for fc_sxm_trMsgCmdAppClearUserData
 *PARAMETER:    1) Pointer to AppClearUserData message
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclStocksApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgCmdAppClearUserData) Entered"));
	// Method used for Diagnostics. Clears all the favorites stored
	(tVoid) prMsg ;
	fc_sxm_tenDiagResult enDiagResult = fc_sxm_enDiagResult_Failed;
	if (STOCK_TICKER_SERVICE_INVALID_OBJECT != _hStockticker )
	{
		vClearAllFavorites();
		enDiagResult = fc_sxm_enDiagResult_Ok;
	}
	vSetUsrDataClearResult(enDiagResult);
	// Method sends fc_sxm_enDiagResult_Ok on sucess and fc_sxm_enDiagResult_Failed on failure
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vProcess(fc_sxm_trMsgCmdAppClearUserData) Exited; DiagResult Status = %u",
			ETG_CENUM(fc_sxm_tenDiagResult, enDiagResult)));
}

/****************************************************************************
 *FUNCTION:          vProcess:fc_sxm_trStocksTimer
 *DESCRIPTION:       On expiry of Timer, this method is invoked by Framework
 *PARAMETERS:        fc_sxm_trStocksTimer *prMsg
 *RETURNVALUES:      None
 *****************************************************************************/
tVoid fc_sxm_tclStocksApp::vProcessTimer(fc_sxm_trStocksTimer const *prMsg)
{
	(tVoid)prMsg;
	ETG_TRACE_USR4(("--->fc_sxm_tclStocksApp::vProcess"
				"(fc_sxm_trStocksTimer *prMsg)"));
	// This Method is invoked on expiry of timer
	// Stop the timer and notify time out
	vStopStocksTimer();
	vStocksTimerNotifyTimeOut();
	ETG_TRACE_USR4(("<---fc_sxm_tclStocksApp::vProcess"
				"(fc_sxm_trStocksTimer *prMsg)"));
}

/*****************************************************************************
 * FUNCTION:          vStopStocksTimer:
 * DESCRIPTION:       To stop the timer
 * PARAMETER:         tVoid
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclStocksApp::vStopStocksTimer(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vStopStocksTimer() Entered/Exit"));
	// Stop the stocks timer if its already Running
	if (bIsStocksTimerRunning())
		_mStocksTimer.vStop();
}
/*****************************************************************************
 * FUNCTION:          bIsStocksTimerRunning
 * DESCRIPTION:       To check the timer status if its running or not
 * PARAMETER:         tVoid
 * RETURNVALUE:       tBool
 *****************************************************************************/
tBool fc_sxm_tclStocksApp::bIsStocksTimerRunning(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclStocksApp::bIsStocksTimerRunning() Entered/Exit"));
	// Returns true if stocks timer is already Running
	return(_mStocksTimer.bIsRunning());
}

/*****************************************************************************
 * FUNCTION:          vStocksTimerNotifyTimeOut
 * DESCRIPTION:       On timeout, 1) a check is performed if data Provider is received
 *                     OTA or not. 2) The same Timer is reused to notify Loading to Unknown state
 * PARAMETER:         tVoid.
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclStocksApp::vStocksTimerNotifyTimeOut(tVoid)
{
	ETG_TRACE_USR4(("--->fc_sxm_tclStocksApp::vStocksTimerNotifyTimeOut()"));
	/* Calling _poStocksList->vOnSmsDsrlState(fc_sxm_enDsrlState_Changed) to change stock status
	 * from "Loading" to "Unknown" automatically*/
	if ( OSAL_NULL != _poStocksList)
		_poStocksList->vOnSmsDsrlState(fc_sxm_enDsrlState_Changed);
	ETG_TRACE_USR4(("<---fc_sxm_tclStocksApp::vStocksTimerNotifyTimeOut()"));
}

/*****************************************************************************
 * FUNCTION:          vStartStocksTimer
 * DESCRIPTION:       Start stocks timer. After expiry of the timer, a List Status
 *                     Changed would be trigger to HMI
 * PARAMETER:         Time in mSec after which the changed status has to be triggered.
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclStocksApp::vStartStocksTimer(tU32 &msecDuration)
{
	ETG_TRACE_USR4(("--->fc_sxm_tclStocksApp::vStartStocksTimer() Entered msecDuration = %u mSec", msecDuration));
	// If stocks timer is already active, stop and restart it
	vStopStocksTimer();
	// Starts the stocks timer for the given duration
	_mStocksTimer.vStart(msecDuration);
	ETG_TRACE_USR4(("<---fc_sxm_tclStocksApp::vStartStocksTimer() Exit"));
}
/*
 * method to initialize Favorite List from Persistent
 */
tVoid fc_sxm_tclStocksApp::vLoadFavoriteStockSymbols()
{
    //Clear If any old Data
    _vecFavStockList.clear();

    //Get Favorite List from Persistent to internal copy
    trAppUserData::instance()->vGetFavoriteList(_vecFavStockList);

    ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vLoadFavoriteStockSymbols size=%d",_vecFavStockList.size()));
}
/*
 * method to Add particular Stock symbol to internal list
 */
tVoid fc_sxm_tclStocksApp::vAddFavorite(const string& strStockSymbol)
{
    ETG_TRACE_USR4(( "fc_sxm_tclStocksApp::vAddFavorite StockSymbol=%s",strStockSymbol.c_str()));

    //Add entry to internal list
    _vecFavStockList.push_back(strStockSymbol);

    //Update the Same to Persistent pool
    trAppUserData::instance()->vSetFavoriteList(_vecFavStockList);
}
/*
 * method to remove particular Stock Symbol from internal list
 */
tVoid fc_sxm_tclStocksApp::vRemoveFavorite(const string& strStockSymbol)
{
    ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vRemoveFavorite StockSymbol=%s",strStockSymbol.c_str()));

    //Remove entry form internal list
    SXM_FOREACH(vector<string>, itr, _vecFavStockList)
    {
        if((*itr).compare(strStockSymbol) == 0)
        {
            _vecFavStockList.erase(itr);
            break;
        }
    }

    //update the same to persistent pool
    trAppUserData::instance()->vSetFavoriteList(_vecFavStockList);
}
/*
 * method to Clear all Favorites from Persistent
 */
tVoid fc_sxm_tclStocksApp::vClearFavoriteStockSymbols()
{
    ETG_TRACE_USR4(("fc_sxm_tclStocksApp::vClearFavoriteStockSymbols "));

    //First Clear All entries form internal list
    _vecFavStockList.clear();

    //Then Set Empty Data to Datapool
    trAppUserData::instance()->vSetFavoriteList(_vecFavStockList);
}
/*
 * method to initialize Favorite List to Persistent
 */
tVoid fc_sxm_tclStocksApp::vSaveFavoriteStockSymbols()
{
    //Set Current List
    trAppUserData::instance()->vSetFavoriteList(_vecFavStockList);

    ETG_TRACE_USR4(( "fc_sxm_tclStocksApp::vSaveFavoriteStockSymbols size=%d",_vecFavStockList.size()));

}
/*
 * method to find & return Favorite Index value per a entry
 */
tU32 fc_sxm_tclStocksApp::u32GetFavoriteIndex(const string& strStockSymbol) const
{
    tU32 u32FavoriteIndex = 0;
    //loop through our internal list
    //find for entry:STOCK SYMBOL
    for(; u32FavoriteIndex < _vecFavStockList.size(); ++u32FavoriteIndex)
    {
        if(strStockSymbol.compare(_vecFavStockList[u32FavoriteIndex]) == 0)
        {
            break;
        }
    }

    ETG_TRACE_USR4(( "fc_sxm_tclStocksApp::u32GetFavoriteIndex u32Index=%d",u32FavoriteIndex));

    //return Favorite Index
    return u32FavoriteIndex;
}


/*=============================================================================
  =======                   END OF FILE                                   =======
  =============================================================================*/

