/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_utils.cpp
* @brief       class implements all common util functions across sports service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_sports_common.h"
#include "fc_sxm_smse.h"
#include "fc_sxm_tcl_sports_utils.h"

// just take any data-fi
#include "fc_sxm_sports_fi.h"

// Defines hash formula multiplier
#define FC_SXM_STRING_HASH_MULTIPLIER ((tU32)(3))
#define SXM_SPORTS_SEASON_INFO 5
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
//#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_BASE
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_sports_utils.cpp.trc.h"
#endif

/******************************************
 * Type Definitions
 *******************************************/
//The number of header values are obtained from table for SXM_SPORTS_LEAGUES and SXM_SPORTS_LIST
#define FC_SXM_SPORTS_LIST_TOTAL_HEADERS 2
#define FC_SXM_SPORTS_LEAGUES_TOTAL_HEADERS 8
#define FC_SXM_SPORTS_TEAMS_TOTAL_HEADERS 4
#define FC_SXM_SPORTS_H2H_TOTAL_HEADERS 20
#define FC_SXM_SPORTS_GOLF_SCHEDULE_HEADER 6
#define FC_SXM_SPORTS_GOLF_RANK_LIST_HEADER  3
#define FC_SXM_SPORTS_MOTORRACE_SCHEDULE_HEADER 8
#define FC_SXM_SPORTS_MOTORRACE_RANK_LIST_HEADER  5

#define FC_SXM_SPORTS_GMT_OFFSET_IN_SEC_EDT 14400
#define FC_SXM_SPORTS_GMT_OFFSET_IN_SEC_EST 18000


/*
 * Default constructor.... TODO: check if any further initialisation would be required
 */
fc_sxm_tclSportsUtils::fc_sxm_tclSportsUtils()
{
	// Default constructor for fc_sxm_tclSportsUtils class
}
/*
 * Method convert time in seconds to OSAL Date
 */
tVoid fc_sxm_tclSportsUtils::vGetOSALDate(tU32 u32tSecs,tChar* chDate,const tChar* chTimeZone,tBool bDSTSetting) const
{
	struct tm sStartTime;

	const TIME_T tSecs = (const TIME_T)u32GetLocalTime(u32tSecs,chTimeZone,bDSTSetting);
	OSAL.gmtime_r(&tSecs , &sStartTime);

	OSAL_s32PrintFormat(chDate ,"%d%s%d%s%d", sStartTime.tm_mday,"-",sStartTime.tm_mon+1,"-",sStartTime.tm_year + 1900);

}
/*
 * Method convert time in seconds to OSAL Time
 */
tVoid fc_sxm_tclSportsUtils::vGetOSALTime(tU32 u32tSecs,tChar* chTIme,const tChar* chTimeZone,tBool bDSTSetting) const
{
	struct tm sStartTime;

	const TIME_T tSecs = (const TIME_T)u32GetLocalTime(u32tSecs,chTimeZone,bDSTSetting);
	OSAL.gmtime_r(&tSecs , &sStartTime);

	OSAL_s32PrintFormat(chTIme ,"%d%s%d%s%d",sStartTime.tm_hour,":",sStartTime.tm_min,":",sStartTime.tm_sec);
}
/*
 * Method convert time in seconds to OSAL Time
 */
tU32 fc_sxm_tclSportsUtils::u32GetLocalTime(tU32 u32tSecs,const tChar* chTimeZone,tBool bDSTSetting) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::u32GetLocalTime() Entered, DSTStatus %d",bDSTSetting));
	//first convert ET time from SMSe to GMT
	if(bDSTSetting)
	{
		u32tSecs += FC_SXM_SPORTS_GMT_OFFSET_IN_SEC_EDT;
	}
	else
	{
		u32tSecs += FC_SXM_SPORTS_GMT_OFFSET_IN_SEC_EST;
	}
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::u32GetLocalTime() Entered %u",u32tSecs));

	tChar chSign[2];
	intptr_t u32Sec;

	//Read Time ZOne Values form Clock
	OSAL_s32ScanFormat(chTimeZone,"%c%lu",chSign,&u32Sec);

	//Finally End the string with null char
	chSign[1] = '\0';

	ETG_TRACE_USR4(("chSign = %s",chSign));

	//Check if Sign is Negative
	if(OSAL_s32StringCompare(chSign,"-") == 0)
	{
		ETG_TRACE_USR4(("negative....."));
		//Sub Tract TimeZone from SMSe Time
		u32tSecs -= u32Sec;
	}
	else
	{
		//Add TimeZone to SMSe Time
		u32tSecs += u32Sec;
	}

	ETG_TRACE_USR4(("u32tSecs(final) = %d",u32tSecs));

	//Finally return the value
	return u32tSecs;
}

/*
 * Method converts string to unique hash code
 */
tU32 fc_sxm_tclSportsUtils::u32ComputeHashCode(string* strInput) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::u32ComputeHashCode() Entered"));

	tU32 u32Result = 0;

	const char *tCharInput = strInput->c_str();

	for (size_t tIndex = 0; tIndex < strInput->size(); ++tIndex , ++tCharInput)
	{
		u32Result = FC_SXM_STRING_HASH_MULTIPLIER * u32Result + *tCharInput;
	}

	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::u32ComputeHashCode() exit  Result Code : %u",u32Result));
	return u32Result;
}

/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_LIST
 */
tBool fc_sxm_tclSportsUtils::bExtractSportsRow(SXMSportsRow const &row, fc_sxm_trSportsList &sportConfig) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::vExtractSportsRow() Entered"));
	tBool bSuccess = FALSE;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.2, Indices and row values for extraction of SXM_SPORTS_LIST.
	tVoid (fc_sxm_trSportsList::*funcPtrArray[FC_SXM_SPORTS_LIST_TOTAL_HEADERS]) (tU32, string&) = {
			&fc_sxm_trSportsList::vSetSportID,
			&fc_sxm_trSportsList::vSetSportName };

	for (tU8 u8Header = 0u; u8Header < FC_SXM_SPORTS_LIST_TOTAL_HEADERS ; ++u8Header )
	{
		tU32 u32SPID = 0u;
		string strName("");
		if ( row.present & (1u << u8Header))
		{
			if ( row.types & (1u << u8Header))
			{
				strName = row.buffer+row.value[u8Header] ;
				bSuccess = TRUE;
			}
			else
			{
				u32SPID = (tU32) row.value[u8Header] ;
			}
		} // end if ( row.present & (1u << u8Header))

		//Store the values read for each index to the appropriate field in fc_sxm_trSportsList structure
		(sportConfig.*funcPtrArray[u8Header])(u32SPID, strName);
	} // end for loop

	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::vExtractSportsRow() Exit \tsportID = %u\tsportName = %s",
			sportConfig.u8SportID, sportConfig.strSportName.c_str()));
	return bSuccess;
}

/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_LEAGUES
 */
tBool fc_sxm_tclSportsUtils::bExtractLeagueRow(SXMSportsRow const &row, fc_sxm_trLeagueListData &league) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractLeagueRow() Entered"));
	tBool bSuccess = FALSE;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.1, Indices and row values for extraction of SXM_SPORTS_LEAGUES.
	tVoid (fc_sxm_trLeagueListData::*funcPtrArray[FC_SXM_SPORTS_LEAGUES_TOTAL_HEADERS]) (tU32, string&) = {
			&fc_sxm_trLeagueListData::vSetSportID,
			&fc_sxm_trLeagueListData::vSetSportName,
			&fc_sxm_trLeagueListData::vSetAffiliateID,
			&fc_sxm_trLeagueListData::vSetAffiliateName,
			&fc_sxm_trLeagueListData::vSetGDRef,
			&fc_sxm_trLeagueListData::vSetInSeason,
			&fc_sxm_trLeagueListData::vSetAffiliateLongName,
			&fc_sxm_trLeagueListData::vSetAffiliateShortName  };

	for (tU8 u8Header = 0u; u8Header < FC_SXM_SPORTS_LEAGUES_TOTAL_HEADERS; ++u8Header )
	{
		tU32 u32Value = 0u;
		string strValue("");

		if ( row.present & (1u << u8Header))
		{
			if ( row.types & (1u << u8Header))
			{
				strValue = row.buffer+row.value[u8Header] ;
				// Return true if at least one string value is successfully extracted
				bSuccess = TRUE;
			}
			else
			{
				u32Value = (tU32) row.value[u8Header] ;
			}
		} // end if ( row.present & u16Index)
		else // Added to check if the row is present and the corresponding bit in the header is missing
		{
			ETG_TRACE_USR4(("bExtractLeagueRow % d bit is not available to extract",u8Header));
			if (SXM_SPORTS_SEASON_INFO == u8Header)
				u32Value = 1u;// Default value for season Info is represented as 0
		}

		//Store the values read for each index to the appropriate field in fc_sxm_trLeagueListData structure
		(league.*funcPtrArray[u8Header])(u32Value, strValue);
	} //end for loop

	//Set Level and Extract League's ShortName
	vSetLeagueShortNameAndLevel(league);
	ETG_TRACE_USR4(("League ---> u8SportID = %u\t u32AfID = %u\t u32GDRef = %u\t _bInSeason = %u",
			league.u8SportID, league.u32AfID, league.u32GDRef, league.bInSeason));
	ETG_TRACE_USR4(("League --->  strSportName = %s", league.strSportName.c_str()));
	ETG_TRACE_USR4(("League --->  strAfName = %s", league.strAfName.c_str()));
	ETG_TRACE_USR4(("League --->  _strAflLName = %s", league.strAflLName.c_str()));
	ETG_TRACE_USR4(("League --->  _strAflSName = %s", league.strAflSName.c_str()));
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractLeagueRow() Exit"));
	return bSuccess;
}

/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_TEAM
 */
tBool fc_sxm_tclSportsUtils::bExtractTeamRow(SXMSportsRow const &row, fc_sxm_trTeamInfo &newTeam) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractTeamRow() Entered"));
	tBool bSuccess = FALSE;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.5, Indices and row values for extraction of SXM_SPORTS_TEAM.
	tVoid (fc_sxm_trTeamInfo::*funcPtrArray[FC_SXM_SPORTS_TEAMS_TOTAL_HEADERS]) (tU32, string&) = {
			&fc_sxm_trTeamInfo::vSetTeamID,
			&fc_sxm_trTeamInfo::vSetAbbreviatedName,
			&fc_sxm_trTeamInfo::vSetTeamName,
			&fc_sxm_trTeamInfo::vSetNickName  };

	for (tU8 u8Header = 0u; u8Header < FC_SXM_SPORTS_TEAMS_TOTAL_HEADERS; ++u8Header )
	{
		tU32 u32Value = 0u;
		string strValue("");

		if ( row.present & (1u << u8Header))
		{
			if ( row.types & (1u << u8Header))
			{
				strValue = row.buffer+row.value[u8Header] ;
				// Return true if atleast one string value is succesfully extracted
				bSuccess = TRUE;
			}
			else
			{
				u32Value = (tU32) row.value[u8Header] ;
			}
		} // end if ( row.present & u16Index)

		//Store the values read for each index to the appropriate field in fc_sxm_trTeamInfo structure
		(newTeam.*funcPtrArray[u8Header])(u32Value, strValue);
	}//end for loop

	ETG_TRACE_USR4(("Team --->  TeamID = %u", newTeam.u32TeamID));
	ETG_TRACE_USR4(("Team --->  strAbbr = %s", newTeam.strAbbr.c_str()));
	ETG_TRACE_USR4(("Team --->  strTeamName = %s", newTeam.strTeamName.c_str()));
	ETG_TRACE_USR4(("Team --->  strNickName = %s", newTeam.strNickName.c_str()));
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractTeamRow() Exit"));
	return bSuccess;
}

/*
 * Method to extract news item in a row
 */
tBool fc_sxm_tclSportsUtils::bExtractNewsRow(SXMSportsRow const &row, string &strNews) const
{
	tBool bSuccess = FALSE;
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractNewsRow() Enter"));

	if ( row.present & 1u )
	{
		if ( row.types & 1u )
		{
			strNews = row.buffer+row.value[0u] ;
			// Return true if atleast one string value is succesfully extracted
			bSuccess = TRUE;
			ETG_TRACE_USR4(("bExtractNewsRow --->  News = %s", strNews.c_str()));
		}
	} // end if ( row.present & u16Index)
	return bSuccess;
}
/*
 * From the League Data received from SMSe, this method can extract the level of a given league and also
 * set the affiliate's Long Name and Short Name
 */
tVoid fc_sxm_tclSportsUtils::vSetLeagueShortNameAndLevel(fc_sxm_trLeagueListData &league) const
{
	tU8 u8Level = 0u;
	string strName = league.strAfName;
	size_t found = strName.find_first_of(":");
	while ( found != string::npos )
	{
		//increase u8Level by 1 for every : found
		++u8Level;
		// Reassign name with New strings which succeeds the delimiter :
		strName = strName.substr(found+1);
		//Again run the same query to traverse to the next string
		found = strName.find_first_of(":");
	}
	league.u8Level = u8Level;
	league.strAflSName = strName;
	//League request can be made initial only for Root Affiliate as SMSe does not allow
	//to create requests for Child leagues
	league.enRequestState = (u8Level != 0u) ?
			fc_sxm_tenLeagueReqState_Unavailable : fc_sxm_tenLeagueReqState_Initial;
}

/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_H2H
 */
tBool fc_sxm_tclSportsUtils::bExtractH2HScheduleRow(SXMSportsRow const &row, fc_sxm_trH2HTable &rH2HTableData)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractH2HScheduleRow() Entered"));

	tBool bSuccess = FALSE;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.7, Indices and row values for extraction of SXM_SPORTS_H2H.
	tVoid (fc_sxm_trH2HTable::*funcPtrArray[FC_SXM_SPORTS_H2H_TOTAL_HEADERS]) (tU32, string&) = {

			//Home Team Details
			&fc_sxm_trH2HTable::vSetHomeTeamID,
			&fc_sxm_trH2HTable::vSetHomeTeamAbr,
			&fc_sxm_trH2HTable::vSetHomeTeamNick,
			&fc_sxm_trH2HTable::vSetHomeTeamFull,

			//Visit Team Details
			&fc_sxm_trH2HTable::vSetVisitTeamID,
			&fc_sxm_trH2HTable::vSetVisitTeamAbr,
			&fc_sxm_trH2HTable::vSetVisitTeamNick,
			&fc_sxm_trH2HTable::vSetVisitTeamFull,

			//Game Details
			&fc_sxm_trH2HTable::vSetGameStatus,
			&fc_sxm_trH2HTable::vSetGameDivision,
			&fc_sxm_trH2HTable::vSetGameStartTime,

			//Score Details
			&fc_sxm_trH2HTable::vSetHomeTeamScore,
			&fc_sxm_trH2HTable::vSetVisitTeamScore,
			&fc_sxm_trH2HTable::vSetWinnerTeam,

			//XM Audio or Channel Information
			&fc_sxm_trH2HTable::vSetHomeChan,
			&fc_sxm_trH2HTable::vSetVisitChan,
			&fc_sxm_trH2HTable::vSetNatChan,

			//Others
			&fc_sxm_trH2HTable::vSetGameClock,
			&fc_sxm_trH2HTable::vSetTableINdex,
			&fc_sxm_trH2HTable::vSetTableAge

			};

	//Macro : Traverse all the Bits present and Copy the information to Data
	SXM_SPORTS_PARSE_ROW_RETURN_VAL(row,rH2HTableData , FC_SXM_SPORTS_H2H_TOTAL_HEADERS , bSuccess)

	//Traces
	ETG_TRACE_USR4(("HomeTeamID = %u , VisitTeamID = %u",
			rH2HTableData.u32HomeTeamID,rH2HTableData.u32VisitTeamID ));
	ETG_TRACE_USR4(("HomeTeamName = %s ",rH2HTableData.strHomeTeamFull.c_str() ));
	ETG_TRACE_USR4(("u32TableINdex = %u , GameStartTime = %u ", rH2HTableData.u32TableINdex,rH2HTableData.u32GameStartTime));
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractLeagueRow() Exit"));

	return bSuccess;
}


/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_GOLF
 */
tBool fc_sxm_tclSportsUtils::bExtractGolfScheduleRow(SXMSportsRow const &row, fc_sxm_trGolfScheduleTable &rGolfTableData)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractGolfScheduleRow() Entered"));

	tBool bSuccess = FALSE;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.9, Indices and row values for extraction of SXM_SPORTS_GOLF.
	tVoid (fc_sxm_trGolfScheduleTable::*funcPtrArray[FC_SXM_SPORTS_GOLF_SCHEDULE_HEADER]) (tU32, string&) = {


			&fc_sxm_trGolfScheduleTable::vSetEventState,
			&fc_sxm_trGolfScheduleTable::vSetRankTableRef,
			&fc_sxm_trGolfScheduleTable::vSetTourneyTitle,
			&fc_sxm_trGolfScheduleTable::vSetCourseName,
			&fc_sxm_trGolfScheduleTable::vSetYardage,
			&fc_sxm_trGolfScheduleTable::vSetPurse

			};

	//Macro : Traverse all the Bits present and Copy the information to Data
	SXM_SPORTS_PARSE_ROW_RETURN_VAL(row,rGolfTableData , FC_SXM_SPORTS_GOLF_SCHEDULE_HEADER , bSuccess)

	//Traces
	ETG_TRACE_USR4(("GolfEventState = %u , u32RankListTableRef = %u, CourseYardage= %u , Purse = %u",
			rGolfTableData.u32EventState,rGolfTableData.u32RankListTableRef,rGolfTableData.u32CourseYardage,rGolfTableData.u32Purse ));
	ETG_TRACE_USR4(("CourseName = %s ",rGolfTableData.strCourseName.c_str() ));
	ETG_TRACE_USR4(("TourneyTitle = %s ",rGolfTableData.strTourneyTitle.c_str() ));
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractGolfScheduleRow() Exit"));

	return bSuccess;
}


/*
 * Method interprets/extracts sports row values from the sxm_sports_begin_nested
 *  with ext_type = SXM_SPORTS_GOLF_RANK
 */
tBool fc_sxm_tclSportsUtils::bExtractGolfScheduleRankListRow(SXMSportsRow const &row, fc_sxm_trGolfRankList &rSchedukeRankList)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractScheduleRankListRow() Entered"));

		tBool bSuccess = FALSE;

		// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
		// section 4.11.10, Indices and row values for extraction of SXM_SPORTS_GOLF.
		tVoid (fc_sxm_trGolfRankList::*funcPtrArray[FC_SXM_SPORTS_GOLF_RANK_LIST_HEADER]) (tU32, string&) = {

				&fc_sxm_trGolfRankList::vSetGolferRank,
				&fc_sxm_trGolfRankList::vSetGolferName,
				&fc_sxm_trGolfRankList::vSetGolferScore

				};

		//Macro : Traverse all the Bits present and Copy the information to Data
		SXM_SPORTS_PARSE_ROW_RETURN_VAL(row,rSchedukeRankList , FC_SXM_SPORTS_GOLF_RANK_LIST_HEADER , bSuccess)

		//Traces
		ETG_TRACE_USR4(("GolfRank = %s ",rSchedukeRankList.strRank.c_str() ));
		ETG_TRACE_USR4(("GolferName = %s ",rSchedukeRankList.strGolfer.c_str() ));
		ETG_TRACE_USR4(("GolferScore = %s ",rSchedukeRankList.strScore.c_str() ));
		ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractScheduleRankListRow() Exit"));

		return bSuccess;
}


/*
 * Method interprets/extracts sports row values from the sxm_sports_begin
 *  with ext_type = SXM_SPORTS_AUTO
 */
tBool fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRow(SXMSportsRow const &row, fc_sxm_trMotorRaceScheduleTable &rMotorRaceTableData)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRow() Entered"));

	tBool bSuccess = FALSE;

	//Create local Object to hold parsed Table Data
	//fc_sxm_trGolfRankList trGolfRankList;

	// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
	// section 4.11.11, Indices and row values for extraction of SXM_SPORTS_GOLF.
	tVoid (fc_sxm_trMotorRaceScheduleTable::*funcPtrArray[FC_SXM_SPORTS_MOTORRACE_SCHEDULE_HEADER]) (tU32, string&) = {


			&fc_sxm_trMotorRaceScheduleTable::vSetEventState,
			&fc_sxm_trMotorRaceScheduleTable::vSetGameStartTime,
			&fc_sxm_trMotorRaceScheduleTable::vSetRankTableRef,
			&fc_sxm_trMotorRaceScheduleTable::vSetRaceName,
			&fc_sxm_trMotorRaceScheduleTable::vSetTrackName,
			&fc_sxm_trMotorRaceScheduleTable::vSetCompleteRaceLaps,
			&fc_sxm_trMotorRaceScheduleTable::vSetWinnerName,
			&fc_sxm_trMotorRaceScheduleTable::vSetNatChan

			};

	//Macro : Traverse all the Bits present and Copy the information to Data
	SXM_SPORTS_PARSE_ROW_RETURN_VAL(row,rMotorRaceTableData , FC_SXM_SPORTS_MOTORRACE_SCHEDULE_HEADER , bSuccess)

	//Traces
	ETG_TRACE_USR4(("GolfEventState = %u , u32GameStartTime = %u, RankListRef= %u , RaceLaps = %u, 2NatChan = %u",
			rMotorRaceTableData.u32EventState,rMotorRaceTableData.u32GameStartTime,rMotorRaceTableData.u32RankListTableRef,rMotorRaceTableData.u32RaceLaps,rMotorRaceTableData.u32NatChan ));
	ETG_TRACE_USR4(("RaceName = %s ",rMotorRaceTableData.strRaceName.c_str() ));
	ETG_TRACE_USR4(("TrackName = %s ",rMotorRaceTableData.strTrackName.c_str() ));
	ETG_TRACE_USR4(("WinnerName = %s ",rMotorRaceTableData.strWinnerName.c_str() ));

	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRow() Exit"));

	return bSuccess;
}


/*
 * Method interprets/extracts sports row values from the sxm_sports_begin_nested
 *  with ext_type = SXM_SPORTS_AUTO_RANK
 */
tBool fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRankListRow(SXMSportsRow const &row, fc_sxm_trMotorRaceRankList &rSchedukeRankList)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRankListRow() Entered"));

		tBool bSuccess = FALSE;

		// Values are extracted as per the table given in SX-9845-0312 Sports Design Guide for the SiriusXM Embedded (SXe) SDK.pdf,
		// section 4.11.12, Indices and row values for extraction of SXM_SPORTS_GOLF.
		tVoid (fc_sxm_trMotorRaceRankList::*funcPtrArray[FC_SXM_SPORTS_MOTORRACE_RANK_LIST_HEADER]) (tU32, string&) = {

				&fc_sxm_trMotorRaceRankList::vSetRank,
				&fc_sxm_trMotorRaceRankList::vSetName,
				&fc_sxm_trMotorRaceRankList::vSetLeaderLapsCount,
				&fc_sxm_trMotorRaceRankList::vSetCarNumber,
				&fc_sxm_trMotorRaceRankList::vSetCarMake

				};

		//Macro : Traverse all the Bits present and Copy the information to Data
		SXM_SPORTS_PARSE_ROW_RETURN_VAL(row,rSchedukeRankList , FC_SXM_SPORTS_MOTORRACE_RANK_LIST_HEADER , bSuccess)

		//Traces
		ETG_TRACE_USR4(("MotorRaceRank = %s ",rSchedukeRankList.strRank.c_str() ));
		ETG_TRACE_USR4(("RacerName = %s ",rSchedukeRankList.strDriverName.c_str() ));
		ETG_TRACE_USR4(("RacerLaps = %d ",rSchedukeRankList.tu32LeaderLaps ));
		ETG_TRACE_USR4(("CarMake = %s ",rSchedukeRankList.strCarMake.c_str() ));
		ETG_TRACE_USR4(("CarNumber = %s ",rSchedukeRankList.strCarNumber.c_str() ));

		ETG_TRACE_USR4(("fc_sxm_tclSportsUtils::bExtractMotorRaceScheduleRankListRow() Exit"));

		return bSuccess;
}

#if 0
/*
 * Method returns day type based on current system time
 */
tU8 fc_sxm_tclSportsUtils::u8GetDayMapIndex(tU32 u32TableParam) const
{
    SXMTime t;
    tU8 u8DayType = SPORTS_TABLE_IX(u32TableParam);

    sxm_time_decode(sxm_time_now(), &t);

    /* need to correct for table offset between 10pm and midnight */
    if((t.hour >= 22) && (u8DayType < 16))
        u8DayType = u8DayType + 1;

    return u8DayType;
}
#endif
/*
 * Method copies a string values into FI Type string
 */
tVoid fc_sxm_tclSportsUtils::fc_sxm_vString2Fi(tCString sSrcString,  midw_ext_fi_tclString& rffiObj) const
{
	if(OSAL_NULL != sSrcString)
	{
		// Convert the string to ISO-8859-1 format Call the function to convert the string to ISO-8859-1 format
		(tVoid) rffiObj.bSet(sSrcString, midw_ext_fi_tclString::FI_EN_ISO8859_1M);
	}
}

