/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_team.cpp
* @brief       base class implementation for Team based sport types.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sports_app_interface.h"
#include "fc_sxm_tcl_sports_team.h"
#ifndef FC_SXM_SPORTS_UTEST
#include "fc_sxm_tcl_sports_utils.h"
#include "fc_sxm_tcl_pers_sports_data.h"
#else
#include "sports/fc_sxm_tcl_sports_utils.h"
#include "sports/fc_sxm_tcl_pers_sports_data.h"
#endif // end of FC_SXM_SPORTS_UTEST
#include "fc_sxm_smse.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_TEAMSPORTS
#include "trcGenProj/Header/fc_sxm_tcl_sports_team.cpp.trc.h"
#endif

fc_sxm_tclTeamSports::fc_sxm_tclTeamSports() :_u8TeamScheduleAvl(0u)
{
	// Default constructor
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports CTOR"));
}
/*
 * Method checks if All Teams Data is available or not
 */

tBool fc_sxm_tclTeamSports::bIsAllTeamAvailable(tU32 u32LeagueID) const
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bIsAllTeamAvailable Entered"));
	tBool bAvailable = FALSE;
	//Only All Teams are available only for Root Affiliates
	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague, u32LeagueID)
	{
		bAvailable = static_cast <tBool> (cMapIt->second.mapTeams.size() != 0) ;
	}
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bIsAllTeamAvailable Exit\t bAvailable = %u", bAvailable));
	return bAvailable;
}

/*
 * Method adds the AllTeams information provided by SMSe to methodResult for GetAllTeams
 */
tVoid fc_sxm_tclTeamSports::vGetAllTeams(tU32 u32LeagueID, fc_sxm_trAdressing const &rAdressing) const
{
	//Warning: LeagueRequest for the selected league must have been issued to SMSe first
	// Only after receiving "UPDATED" status for league request, AllTeams data can be gathered by HMI
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vGetAllTeams Entered"));
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

	midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodResult oMRes;
	oMRes.AllTeamList.SPORT_ID = _u8SPID;
	oMRes.AllTeamList.AFFILIATE_ID = u32LeagueID;
	oMRes.AllTeamList.TeamDetails.clear();

	SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cLeagueIter, _mapLeague, u32LeagueID)
	{
		vCopyTeamsToFI(cLeagueIter->second.mapTeams, oMRes.AllTeamList.TeamDetails);
	} // end if loop

	//Send FI msg for GetAllTeams
	_poApp->vSendFiMsg(rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vGetAllTeams Exit"));
}
/*
 * Method to Store Particular Team into Favorite List
 */
tBool fc_sxm_tclTeamSports::bSetFavoriteTeam(tU32 u32LeagueId,tU32 u32TeamId) const
{
	ETG_TRACE_USR1(("fc_sxm_tclTeamSports::bSetFavoriteTeam Entered u32LeagueID=%d,u32TeamID=%d",u32LeagueId, u32TeamId));

	tU32 u32RootLeagueID;
	tBool bExists = FALSE;

	if(!(bIsFavorite(u32TeamId)))
	{
		//Get Root Affiliate for requested League from HMI
		if(bIsRootAffiliate(u32LeagueId))
		{
			u32RootLeagueID = u32LeagueId;
		}
		else
		{
			u32RootLeagueID = u32GetRootAffiliate(u32LeagueId);
		}

		//Get Root League Details
		SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cIter, _mapLeague, u32RootLeagueID)
		{
			//for found League,Check for the Team
			// NCG3D-53 fix
			//SXM_IF_FIND_MAP(tU32, fc_sxm_trTeamInfo, cit, cIter->second.mapTeams,u32TeamId)
			SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trTeamInfo, cit, cIter->second.mapTeams,u32TeamId)
				{
				//OK,Found
				bExists = TRUE;
				//Prepare Team Name to be saved persistently
				string strTeamName;
				strTeamName = cit->second.strTeamName + " " + cit->second.strNickName;
				/* Insert data into favorite list */
				trSportsUserData::instance()->vSetFavoriteList(cIter->second.u8SportID, u32TeamId,
						u32LeagueId, cIter->second.u32GDRef, strTeamName, cit->second.strAbbr,
						cit->second.strTeamName, cit->second.strNickName, cIter->second.strSportName);
				}
		}
	}
	//Finaly Return status
	return bExists;
}
/*
 * Method to get Schedules for a particular Team
 */
tVoid fc_sxm_tclTeamSports::vGetTeamSchedules(tU32 u32LeagueID,tU32 u32TeamID,tU8 u8GameType, tU32 u32GameID, fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vGetTeamSchedules Entered u32LeagueID=%d,u32TeamID=%d,u8GameType=%u",u32LeagueID, u32TeamID,u8GameType));

	//Set Current Active LEague
	fc_sxm_tclSportsBase::_u32ActiveLeague = u32LeagueID;

	//Set Schedule Type for Ordering
	fc_sxm_tclSportsBase::_u8ScheduleType = u8GameType;

	if(fc_sxm_tenLeagueReqState_Updated == enGetRequestState(u32LeagueID))
	{
		//Before we get the new Game/Event List From SMS
		//Clear the Old List
		vClearSchedulesList();

		//Make Ignore updates as FALSE
		vIgnoreUpdates(FALSE);

		//Reset Team ID to invalid
		_u32ActiveTeamID = FC_SXM_TEAM_INVALID_ID;

		tU32 u32RootLeagueID;

		//Check if it is a Root League
		if(bIsRootAffiliate(u32LeagueID))
		{
			ETG_TRACE_USR4(("u32LeagueID=%d is a Root League",u32LeagueID));

			u32RootLeagueID = u32LeagueID;

		}
		//Sub League Case
		else
		{
			u32RootLeagueID = u32GetRootAffiliate(u32LeagueID);
		}
		//Check if the league exists
		SXM_IF_FIND_MAP(tU32, fc_sxm_trLeagueData, cit, _mapLeagueDetails,u32RootLeagueID)
		{

			vRetrieveTeamSchedules(cit->second.poLeagueHandle,u32TeamID);
		}

		//Finally Send Method Result to HMI
		vEmitTeamSchedules(u32TeamID,u32GameID,rAdressing);
	}
	else
	{
		//Set TeamID
		_u32ActiveTeamID = u32TeamID;
		_u32FocussedGameID = u32GameID;
	}
}
/*
 * Method to Retrieve Schedules for a particular Team from SMS
 */
tVoid fc_sxm_tclTeamSports::vRetrieveTeamSchedules(tVoid* pLeagueHandle,tU32 u32TeamID)
{
	ETG_TRACE_USR4(("vRetrieveSchedules pLeagueHandle = %x, u32TeamID = %d",pLeagueHandle,u32TeamID));
	_u8TeamScheduleAvl = 0u;

	ptr ext = OSAL_NULL;
	SXMSportsRow row,header;

	sxm_result_code enRetCode = (sxm_result_code) sxm_sports_begin(pLeagueHandle, SXM_SPORTS_TEAM_EVENTS, u32TeamID , &ext, &header);

	if (enRetCode == SXM_E_OK)
	{
		// Iterate until all rows are read.
		while((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			//The sxm_sports_extract_row routine returns the data for leagues from all sports as a row in the table SXM_SPORTS_LEAGUES.
			//Each league would be reported in one row.
			//Extract Schedule Row From SMSe
			(tVoid) bExtractScheduleRow(ext, row, 0, 0, TRUE);
		} //end for loop

		(tVoid) sxm_sports_end(ext);

	} //end if loop
	else
	{
		ETG_TRACE_ERR(("sxm_sports_being failed at fc_sxm_tclSportsBase::vRetrieveSchedules\t enRetCode = %u",
				enRetCode));
	}
}
/*
 * Method to Extract And Fill Schedule Row Data to Method Result
 */
tBool fc_sxm_tclTeamSports::bExtractScheduleRow(tVoid* /*pHandle*/, SXMSportsRow &TableRow,tS32 /*s32TableParam*/, tU32 /*u32EpochValue*/, tBool bIsTeamBased)
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

	//Create local Object to hold parsed Table Data
	fc_sxm_trH2HTable trH2HTableData;

	tBool bDSTStatus = FALSE;

	//Extract the SMS Row Data to our internal object provided
	tBool bAvailable = poUtils->bExtractH2HScheduleRow(TableRow,trH2HTableData);
	if (bAvailable)
	{
		//Game Date
		tChar chDate[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		//Game Time
		tChar chTime[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		const tChar* chGMTTimeZone = "+0000";
		if (trH2HTableData.u32GameStartTime)
		{
			ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow No Conversion"));
			poUtils->vGetOSALDate(trH2HTableData.u32GameStartTime, chDate, chGMTTimeZone, bDSTStatus);
			poUtils->vGetOSALTime(trH2HTableData.u32GameStartTime, chTime, chGMTTimeZone, bDSTStatus);
		}
		string strStartDate(chDate);

		string strStartTime(chTime);

		ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow strStartDate = %s", strStartDate.c_str()));

		ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow strStartTime = %s", strStartTime.c_str()));

		//string strGameString = trH2HTableData.strHomeTeamFull + trH2HTableData.strVisitTeamFull + strStartDate + strStartTime;
		string strGameString = trH2HTableData.strHomeTeamAbr + trH2HTableData.strVisitTeamAbr + strStartDate + strStartTime;

		ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow Unique Game String = %s", strGameString.c_str()));
		// Generate Unique_ID for the input String
		trH2HTableData.u32GameID = poUtils->u32ComputeHashCode(&strGameString);
		ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow Unique Game ID = %u", trH2HTableData.u32GameID ));

		if (bIsGameIDAvailable(trH2HTableData.u32GameID))
		{
			//For Team based Schedules, u32TableINdex contains ScheduleDay's value...from 0 - 16
			if(bIsTeamBased)
			{
				bAvailable = bFilterTeamSchedules(trH2HTableData.u32TableINdex);
				vSetTeamSchedAvl(trH2HTableData.u32TableINdex);
			}
			//If successfully parsed,  copy locally
			if (bAvailable)
				_setSortedH2HSchedules.insert(trH2HTableData);
		}
	}
	else
	{
		ETG_TRACE_ERR(("Error Parsing Table Data..."));
	}

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bExtractAndFillScheduleRow Exited.."));

	//return the status
	return bAvailable;
}
/*
 * Method to Filter Team Schedules as per the requested League
 */
tBool fc_sxm_tclTeamSports::bFilterTeamSchedules(tU32 u32TableIndex) const
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bFilterTeamSchedules Entered u32TableIndex=%d\tActiveLeague = %u",u32TableIndex, fc_sxm_tclSportsBase::_u32ActiveLeague));

	tU32 u32LeagueID = fc_sxm_tclSportsBase::_u32ActiveLeague;
	tU32 u32TableLeague = SPORTS_TABLE_AF(u32TableIndex);
    tBool bAvail = FALSE;

    SXM_ASSERT_RETURN_VAL( OSAL_NULL != _poApp,FALSE)

	tU8 u8DayType = _poApp->u8GetDayMapIndex(u32TableIndex);


	//Check if it is a Root League
	if(bIsRootAffiliate(u32LeagueID))
	{
		ETG_TRACE_USR4(("u32LeagueID=%d is a Root League",u32LeagueID));

		bAvail = bFindSchedule((tU8)fc_sxm_tclSportsBase::_u8ScheduleType,u8DayType);

	}
	//Sub League Case
	else
	{
		tU32 u32RootLeagueID = u32GetRootAffiliate(u32LeagueID);

		if(u32RootLeagueID != AFFILIATE_INVALID_ID)
		{
			ETG_TRACE_USR4(("u32LeagueID=%d has a Root League=%d", u32LeagueID, u32RootLeagueID));

			fc_sxm_trLeagueListData trLeagueListData;
			//Check if it has further sub leagues
			if(bHasSubLeagues(u32LeagueID,trLeagueListData))
			{
				//Get Sub League ID for the intermediate League
				SXM_FOREACH_MAP_CONST (tU32, fc_sxm_trLeagueListData, cMapIt, trLeagueListData.mapSubLeagues)
				{
					//Check to see any of Sub League is matching to our league
					if(u32TableLeague == cMapIt->second.u32AfID)
					{
						bAvail = bFindSchedule((tU8)fc_sxm_tclSportsBase::_u8ScheduleType,u8DayType);
					}
				}
			}
			else		/* Last Sub League */
			{
				//Check if the last sub league is the one what we are refering
				if(u32TableLeague == u32LeagueID)
				{
					bAvail = bFindSchedule((tU8)fc_sxm_tclSportsBase::_u8ScheduleType,u8DayType);
				}
			}
		}
		else
		{
			ETG_TRACE_ERR(("u32LeagueID=%d not belongs to any Root League",u32LeagueID));
		}
	}
	return bAvail;
}
/*
 * Method to Fill Schedule Row Data to Method Result
 */
tVoid fc_sxm_tclTeamSports::vCopySchedulesToFI(midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult &oMRes,tU32 /*u32TeamID*/) const
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vCopySchedulesToFI Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	ETG_TRACE_USR4(("_setSortedH2HSchedules size=%d",_setSortedH2HSchedules.size()));
	tBool bDSTStatus = _poApp->bGetClockDST();
	string strTimeZone = _poApp->pGetTimeZone();

	for(set<fc_sxm_trH2HTable,trCompareSchedules>::const_iterator cit = _setSortedH2HSchedules.begin();
			cit != _setSortedH2HSchedules.end(); ++cit)
	{
		midw_ext_fi_tcl_GameInfo oGameInfo;

		//Copy actual Data recieved from SMS
		//Home team Info
		string strHomeTeam = cit->strHomeTeamFull + " " + cit->strHomeTeamNick;
		oGameInfo.Home_Team_Info.Team_ID = cit->u32HomeTeamID;
		poUtils->fc_sxm_vString2Fi(strHomeTeam.c_str(),oGameInfo.Home_Team_Info.Team_Name);
		poUtils->fc_sxm_vString2Fi(cit->strHomeTeamAbr.c_str(),oGameInfo.Home_Team_Info.Team_Abbr_Name);
		poUtils->fc_sxm_vString2Fi(cit->strHomeTeamFull.c_str(),oGameInfo.Home_Team_Info.Team_Known_Name);
		poUtils->fc_sxm_vString2Fi(cit->strHomeTeamNick.c_str(),oGameInfo.Home_Team_Info.Team_Nick_Name);
		//oGameInfo.Home_Team_Info.Team_Name = strHomeTeam.c_str();
		oGameInfo.Home_Team_Info.Team_Score = cit->u32HomeTeamScore;

		//Visiting Team Info
		oGameInfo.Visiting_Team_Info.Team_ID = cit->u32VisitTeamID;
		string strVisitingTeam = cit->strVisitTeamFull + " " + cit->strVisitTeamNick;
		//oGameInfo.Visiting_Team_Info.Team_Name = strVisitingTeam.c_str();
		poUtils->fc_sxm_vString2Fi(strVisitingTeam.c_str(),oGameInfo.Visiting_Team_Info.Team_Name);
		poUtils->fc_sxm_vString2Fi(cit->strVisitTeamAbr.c_str(),oGameInfo.Visiting_Team_Info.Team_Abbr_Name);
		poUtils->fc_sxm_vString2Fi(cit->strVisitTeamFull.c_str(),oGameInfo.Visiting_Team_Info.Team_Known_Name);
		poUtils->fc_sxm_vString2Fi(cit->strVisitTeamNick.c_str(),oGameInfo.Visiting_Team_Info.Team_Nick_Name);
		oGameInfo.Visiting_Team_Info.Team_Score = cit->u32VisitTeamScore;

		//[[--->Game Info
		//Game Date
		tChar chDate[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };

		if (cit->u32GameStartTime)
		poUtils->vGetOSALDate(cit->u32GameStartTime, chDate, strTimeZone.c_str(), bDSTStatus);
		poUtils->fc_sxm_vString2Fi(chDate,oGameInfo.Game_Time.Date);

		//Game Time
		tChar chTime[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		if (cit->u32GameStartTime)
		poUtils->vGetOSALTime(cit->u32GameStartTime, chTime, strTimeZone.c_str(), bDSTStatus);
		poUtils->fc_sxm_vString2Fi(chTime,oGameInfo.Game_Time.Time);

		//Game Status
		oGameInfo.EventState.enType = (midw_ext_fi_tcl_e8_Game_Status::tenType)cit->u32GameStatus;
		poUtils->fc_sxm_vString2Fi(cit->strGameDivision.c_str(),oGameInfo.Game_Status);

		//Game Winner
		oGameInfo.Winning_Team_ID = cit->u32WinnerTeam;

		//Game Division
		poUtils->fc_sxm_vString2Fi(cit->strGameDivision.c_str(),oGameInfo.Game_Status);
		//Game Clock
		oGameInfo.Game_Clock = cit->u32GameClock;

		//Add Channel Info's
		midw_ext_fi_tcl_XMChannel oXMChannel;
		if(cit->u32HomeChan != 0u){
			oXMChannel.ChannelNumber = cit->u32HomeChan;
			oXMChannel.ChannelType.enType = midw_ext_fi_tcl_XMChannelType::FI_EN_HOME;

			oGameInfo.XMChannelList.push_back(oXMChannel);
		}
		if(cit->u32VisitChan != 0u){
			oXMChannel.ChannelNumber = cit->u32VisitChan;
			oXMChannel.ChannelType.enType = midw_ext_fi_tcl_XMChannelType::FI_EN_AWAY;

			oGameInfo.XMChannelList.push_back(oXMChannel);
		}
		if(cit->u32NatChan != 0u){
			oXMChannel.ChannelNumber = cit->u32NatChan;
			oXMChannel.ChannelType.enType = midw_ext_fi_tcl_XMChannelType::FI_EN_NATIONAL;

			oGameInfo.XMChannelList.push_back(oXMChannel);
		}
		oGameInfo.Game_ID = cit->u32GameID;

		//Prepare Method result with actual SMS data
		oMRes.GameList.GameInfo.push_back(oGameInfo);
	}

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vCopySchedulesToFI exited %u", oMRes.GameList.GameInfo.size()));

}
/*
 * Method to Clear prvious Game List Data
 */
tVoid fc_sxm_tclTeamSports::vClearSchedulesList()
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vResetSchedulesList Entered"));

	_setSortedH2HSchedules.clear();

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vResetSchedulesList exited"));
}
/*
 * Method to send MEthod Result to HMI
 */
tVoid fc_sxm_tclTeamSports::vEmit(tU32 u32GameID,fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vEmit Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult oMResSchedule;
	vFillDefaults(oMResSchedule,u32GameID);
	//Copy Internal Schedules Data to FI
	vCopySchedulesToFI(oMResSchedule);

	//Print Schedules
	vPrintSchedules(oMResSchedule);

	// Send FI message
	_poApp->vSendFiMsg(rAdressing, oMResSchedule);

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vEmit Exited"));
}
/*
 * Method to send MEthod Result to HMI
 */
tVoid fc_sxm_tclTeamSports::vEmitTeamSchedules(tU32 u32TeamID,tU32 u32GameID,fc_sxm_trAdressing const &rAdressing)
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vEmitTeamSchedules Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
	midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult oMResSchedule;
	vFillDefaults(oMResSchedule, u32GameID,u32TeamID );

	//Copy Internal Schedules Data to FI
	vCopySchedulesToFI(oMResSchedule,u32TeamID);

	//Print Schedules
	vPrintSchedules(oMResSchedule);

	// Send FI message
	_poApp->vSendFiMsg(rAdressing, oMResSchedule);

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vEmitTeamSchedules Exited"));
}
/*
 * Method to Find a Team is a Favorite or not
 */
tBool fc_sxm_tclTeamSports::bIsFavorite(tU32 u32TeamID) const
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bisFavorite Entered "));

	tBool bAvailable = FALSE;

	/* Get favorite list size from persistency */
	tU32 u32FavoriteListSize = trSportsUserData::instance()->u32GetFavoriteListSize();

	/*iterate through the favorite list*/
	for(tU32 u32Count=0; u32Count<u32FavoriteListSize; u32Count++) {

		trFavoriteListElement tListElement;
		trSportsUserData::instance()->vGetFavoriteList(u32Count, tListElement);   /* Get data at each list index */

		if((tListElement.u32TeamID == u32TeamID) &&
			(tListElement.u32LeagueID == fc_sxm_tclSportsBase::_u32ActiveLeague) &&
			(tListElement.u8SportID == _u8SPID))
		{
			bAvailable = TRUE;
			break;
		}
	}

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::bisFavorite exited"));

	return bAvailable;
}
/*
 * Method to Clear prvious Game List Data
 */
tVoid fc_sxm_tclTeamSports::vPrintSchedules(midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult &oMResSchedule) const
{
	//Print Data
	for(tU32 u32index=0;u32index<oMResSchedule.GameList.GameInfo.size();u32index++)
	{
		//All details from the passed object are printed on TTFIs
		ETG_TRACE_USR4(("Home Team_ID=%d",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_ID));
		ETG_TRACE_USR4(("Home Team_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_Name.szValue));
		ETG_TRACE_USR4(("Home Team_Abbr=%s",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_Abbr_Name.szValue));
		ETG_TRACE_USR4(("Home Team_Known_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_Known_Name.szValue));
		ETG_TRACE_USR4(("Home Team_Nick_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_Nick_Name.szValue));
		ETG_TRACE_USR4(("Home Team_Score=%d",oMResSchedule.GameList.GameInfo[u32index].Home_Team_Info.Team_Score));

		ETG_TRACE_USR4(("Visiting Team_ID=%d",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_ID));
		ETG_TRACE_USR4(("Visiting Team_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_Name.szValue));
		ETG_TRACE_USR4(("Visiting Team_Abbr=%s",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_Abbr_Name.szValue));
		ETG_TRACE_USR4(("Visiting Team_Known_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_Known_Name.szValue));
		ETG_TRACE_USR4(("Visiting Team_Nick_Name=%s",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_Nick_Name.szValue));
		ETG_TRACE_USR4(("Visiting Team_Score=%d",oMResSchedule.GameList.GameInfo[u32index].Visiting_Team_Info.Team_Score));

		ETG_TRACE_USR4(("Game Date = %s",oMResSchedule.GameList.GameInfo[u32index].Game_Time.Date.szValue));
		ETG_TRACE_USR4(("Game Time = %s",oMResSchedule.GameList.GameInfo[u32index].Game_Time.Time.szValue ));

		ETG_TRACE_USR4(("Game Status=%s",oMResSchedule.GameList.GameInfo[u32index].Game_Status.szValue));
		ETG_TRACE_USR4(("Event State=%u",oMResSchedule.GameList.GameInfo[u32index].EventState.enType));

		ETG_TRACE_USR4(("Game ID =%u",oMResSchedule.GameList.GameInfo[u32index].Game_ID));
	}
}

/*
 * Method fills the default values to be set in the schedules MethodResult to be sent to HMI
 */
tVoid fc_sxm_tclTeamSports::vFillDefaults(midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult &oMResSchedule, tU32 u32GameID,tU32 u32TeamID) const
{
	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	//Fill all the default values irrespective of whether the MethodStart is for Team Schedules or
	// for League Based Schedules
	oMResSchedule.GameList.SPORT_ID =  _u8SPID ;
	//oMResSchedule.GameList.AFFILIATE_ID = fc_sxm_tclSportsBase::_u32ActiveLeague;

	// Fill the Affiliate Info

	// Check whether the active League is the Root Affiliate ID or Sub Affiliate ID and based on that fetch the InSeason Info
	if(bIsRootAffiliate(fc_sxm_tclSportsBase::_u32ActiveLeague)) // For Root Affiliate
	{
		SXM_IF_FIND_MAP_CONST(tU32, fc_sxm_trLeagueListData, cMapIt, _mapLeague, fc_sxm_tclSportsBase::_u32ActiveLeague)
		{
			oMResSchedule.GameList.AFFILIATE_INFO.AFFILIATE_ID = cMapIt->second.u32AfID;
			// Affiliate Name is filled here but this info is not required by HMI
			poUtils->fc_sxm_vString2Fi(((cMapIt->second.strAflSName).c_str()),oMResSchedule.GameList.AFFILIATE_INFO.AffiliateName);
			oMResSchedule.GameList.AFFILIATE_INFO.InSeason = cMapIt->second.bInSeason;
		}
	}

	else // For Sub Root Affiliate
	{
		fc_sxm_trLeagueListData trLeagueListData;
		vGetLeagueData(fc_sxm_tclSportsBase::_u32ActiveLeague,trLeagueListData);

		oMResSchedule.GameList.AFFILIATE_INFO.AFFILIATE_ID = trLeagueListData.u32AfID;
		// Affiliate Name is filled here but this info is not required by HMI
		poUtils->fc_sxm_vString2Fi(((trLeagueListData.strAflSName).c_str()),oMResSchedule.GameList.AFFILIATE_INFO.AffiliateName);
		oMResSchedule.GameList.AFFILIATE_INFO.InSeason = trLeagueListData.bInSeason;
	}

	//Print Affiliate Info
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vFillDefaults: League Name %s",oMResSchedule.GameList.AFFILIATE_INFO.AffiliateName.szValue));
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vFillDefaults: League ID %d",oMResSchedule.GameList.AFFILIATE_INFO.AFFILIATE_ID));
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vFillDefaults: League Season %d",oMResSchedule.GameList.AFFILIATE_INFO.InSeason));

	oMResSchedule.GameList.ListIndex = u32GetGameListIndex(u32GameID); // zero based List Index Passed to HMI
	//Check if particular Team is Favorite
	//for League based it will be Invalid ID
	if (u32TeamID != FC_SXM_TEAM_INVALID_ID)
	{
		// Initialise methodResult for Team based schedules
		oMResSchedule.GameList.bIsRecentGamesAvl = (tBool) (( _u8TeamScheduleAvl & (1u << FC_SXM_SPORTS_SCHEDULE_RECENT)) != 0);
		oMResSchedule.GameList.bIsCurrentGamesAvl = (tBool) (( _u8TeamScheduleAvl & (1u << FC_SXM_SPORTS_SCHEDULE_TODAY)) != 0);
		oMResSchedule.GameList.bIsFutureGamesAvl = (tBool) (( _u8TeamScheduleAvl & (1u << FC_SXM_SPORTS_SCHEDULE_FUTURE)) != 0);
		oMResSchedule.GameList.bIsFavorite = bIsFavorite(u32TeamID);
	}
	else
	{
		// Initialise methodResult for League based schedules
		oMResSchedule.GameList.bIsRecentGamesAvl = bIsScheduleAvilable(fc_sxm_tclSportsBase::_u32ActiveLeague, FC_SXM_SPORTS_SCHEDULE_RECENT);
		oMResSchedule.GameList.bIsCurrentGamesAvl = bIsScheduleAvilable(fc_sxm_tclSportsBase::_u32ActiveLeague, FC_SXM_SPORTS_SCHEDULE_TODAY);
		oMResSchedule.GameList.bIsFutureGamesAvl = bIsScheduleAvilable(fc_sxm_tclSportsBase::_u32ActiveLeague, FC_SXM_SPORTS_SCHEDULE_FUTURE);
		oMResSchedule.GameList.bIsFavorite = FALSE;
	}

	oMResSchedule.GameList.GameType.enType = (midw_ext_fi_tcl_GameType::tenType)fc_sxm_tclSportsBase::_u8ScheduleType;
	ETG_TRACE_USR4(("Sport Team_ID=%u\t isFavorite = %u\t ListIndex =%u \t GameType = %u ",
			oMResSchedule.GameList.SPORT_ID,  oMResSchedule.GameList.bIsFavorite,oMResSchedule.GameList.ListIndex,oMResSchedule.GameList.GameType.enType));
	ETG_TRACE_USR4(("bIsRecentGamesAvl = %u\t bIsCurrentGamesAvl = %u\t bIsFutureGamesAvl = %u",
				oMResSchedule.GameList.bIsRecentGamesAvl, oMResSchedule.GameList.bIsCurrentGamesAvl, oMResSchedule.GameList.bIsFutureGamesAvl));
}

tU32 fc_sxm_tclTeamSports::u32GetGameListIndex(tU32 u32GameID) const
{
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::u32GetEventListIndex entry\t _setSortedH2HSchedules size = %u",_setSortedH2HSchedules.size()));

	tU32 u32ListIndex = 0;

	tBool bfound = FALSE;

	for(set<fc_sxm_trH2HTable,trCompareSchedules>::const_iterator cit = _setSortedH2HSchedules.begin();
			cit != _setSortedH2HSchedules.end(); ++cit,++u32ListIndex)
	{
		if (cit->u32GameID == u32GameID)
		{
			bfound = TRUE;
			break;
		}
	}

	if ( FALSE == bfound ) // Condition For Both Invalid Game ID and Missing Game ID
	{
		u32ListIndex = DEFAULT_GAME_EVENT_ID;
	}

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::u32GetEventListIndex exited\t u32ListIndex Value = %u", u32ListIndex));
	return u32ListIndex;
}

/*
 * Method sets the schedules availability bit
 */
tVoid fc_sxm_tclTeamSports::vSetTeamSchedAvl(tU32 u32TableIndex)
{
	//Set the schedules availability bit for Team Based schedules
	for (tU8 u8ScheduleType = FC_SXM_SPORTS_SCHEDULE_TODAY ;
			u8ScheduleType <= FC_SXM_SPORTS_SCHEDULE_FUTURE;
			++u8ScheduleType)
	{
		//If the schedule type's status bit is set already, don't do any further processing
		if (!( _u8TeamScheduleAvl & (1u << u8ScheduleType)) &&
				(bFindSchedule(u8ScheduleType, (tU8) u32TableIndex)))
		{
			_u8TeamScheduleAvl |= (1u << u8ScheduleType);
			break;
		}
	}
}

/*
 * Method returns TRUE if the Unique Game ID provided is not part of the games list sent to HMI
 */
tBool fc_sxm_tclTeamSports::bIsGameIDAvailable(tU32 u32GameID) const
{
	tBool bInsertFlag = TRUE;
	for (set<fc_sxm_trH2HTable,trCompareSchedules>::const_iterator cIt = _setSortedH2HSchedules.begin(); cIt != _setSortedH2HSchedules.end(); ++cIt)
	{
		if ((cIt->u32GameID) == u32GameID)
		{
			// Same game is identified in the existing list
			ETG_TRACE_USR4(("Identified a Duplicate Game ID = %u",u32GameID));
			bInsertFlag = FALSE;
			break;
		}
	}
	return bInsertFlag;
}

/*
 * Sort key for Team Names
 */
struct trCompareTeams
{
	bool operator()(const midw_ext_fi_tcl_TeamInfo &trLeftTeam, const midw_ext_fi_tcl_TeamInfo &trRightTeam) const
	{
		string strLTeam(trLeftTeam.Team_Name.szValue);
		string strRTeam(trRightTeam.Team_Name.szValue);
		return (strLTeam.compare(strRTeam) < 0);
	}
} ;

/*
 * Method sorts teams based on Team Name (Team Name + " " + Team Nick Name)
 */
tVoid fc_sxm_tclTeamSports::vSortTeams(vector< midw_ext_fi_tcl_TeamInfo >& vectTeams) const
{
	// Method uses std::sort algorithm to sort elements of vector using key "trCompareAllTeams"
	sort(vectTeams.begin(),vectTeams.end(),trCompareTeams());
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vSortTeams Entered & Exit size() = %u", vectTeams.size()));
}

/*
 * Method copies all teams information from mapTeams which is a member variable to
 * vectTeams which is a local variable.
 * This copying is done so that vector can be later sorted based on Team Names
 */

tVoid fc_sxm_tclTeamSports::vCopyTeamsToFI( const map <tU32, fc_sxm_trTeamInfo>& mapTeams,
		vector< midw_ext_fi_tcl_TeamInfo >& vectTeams) const
{
	vectTeams.clear();
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)
	set<tU32> setFavoriteTeams;
	// Collect favorite teams belonging to the league and sport
	vGetFavoritesInLeague(setFavoriteTeams);
	// Method iterates thru all elements of mapTeams and copies them to the given vector
	SXM_FOREACH_MAP_CONST(tU32, fc_sxm_trTeamInfo, cTeamIter, mapTeams)
	{
		midw_ext_fi_tcl_TeamInfo teamInfo;
		teamInfo.Team_ID = cTeamIter->second.u32TeamID ;
		string strFullName = cTeamIter->second.strTeamName + " " + cTeamIter->second.strNickName;
		poUtils->fc_sxm_vString2Fi(strFullName.c_str(), teamInfo.Team_Name);
		teamInfo.Team_Score = 0u;
		poUtils->fc_sxm_vString2Fi(cTeamIter->second.strAbbr.c_str(), teamInfo.Team_Abbr_Name);
		poUtils->fc_sxm_vString2Fi(cTeamIter->second.strTeamName.c_str(), teamInfo.Team_Known_Name);
		poUtils->fc_sxm_vString2Fi(cTeamIter->second.strNickName.c_str(), teamInfo.Team_Nick_Name);
		teamInfo.bIsFavorite = static_cast <tBool> ((setFavoriteTeams.count(cTeamIter->second.u32TeamID) != 0));
		vectTeams.push_back(teamInfo);
		ETG_TRACE_USR4(("Team ID = %u\t Team Name = %s", teamInfo.Team_ID, strFullName.c_str()));
	}
	vSortTeams(vectTeams);
	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vCopyTeamsToFI Exit; \t mapTeams.size() = %u\t vectTeams.size() = %u",
			mapTeams.size(), vectTeams.size()));
}

/*
 * Method collects all favorite teams ids from the given league and given sport
 * from Persistent memory
 */
tVoid fc_sxm_tclTeamSports::vGetFavoritesInLeague(set<tU32>& setFavoriteTeams) const
{
	// Get favorite list size from persistency
	const tU32 u32FavoriteListSize = trSportsUserData::instance()->u32GetFavoriteListSize();
	setFavoriteTeams.clear();

	// iterate through the favorite list
	for(tU32 u32Count=0; u32Count<u32FavoriteListSize; u32Count++) {
		trFavoriteListElement trListElement;
		// Get data at each list index
		trSportsUserData::instance()->vGetFavoriteList(u32Count, trListElement);
		// Add team to the set of Favorite Teams only if
		// sport id and league id matches
		if( (trListElement.u32LeagueID == fc_sxm_tclSportsBase::_u32ActiveLeague) &&
			(trListElement.u8SportID == _u8SPID))
		{
			setFavoriteTeams.insert(trListElement.u32TeamID);
		}
	}
	ETG_TRACE_USR3(("fc_sxm_tclTeamSports::vGetFavoritesInLeague Exit; setFavoriteTeams.size() = %u",
			setFavoriteTeams.size()));
}

