/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_golf.cpp
* @brief       Class implementation for sport type 'Golf'
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_sports_app_interface.h"
#include "fc_sxm_tcl_sports_golf.h"
#ifndef FC_SXM_SPORTS_UTEST
#include "fc_sxm_tcl_sports_utils.h"
#else
#include "sports/fc_sxm_tcl_sports_utils.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_NONTEAMSPORTS
#include "trcGenProj/Header/fc_sxm_tcl_sports_golf.cpp.trc.h"
#endif

#define SXM_SPORTS_SMSE_GOLF_RANK_INDEX 1

fc_sxm_tclGolf::fc_sxm_tclGolf()
{
	// Default constructor
	ETG_TRACE_USR4(("fc_sxm_tclGolf CTOR"));
}

tBool fc_sxm_tclGolf::bExtractNestedScheduleRow(ptr pHandle, tU32 u32RankListTableRef,fc_sxm_trGolfScheduleTable &trGolfScheduleTableData) const
{
 	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractNestedScheduleRow Entered"));
 	ptr ext;
 	SXMSportsRow row2;
 	sxm_result_code enRetCode;
 	tBool bAvailable = FALSE;

 	if ((enRetCode = (sxm_result_code)sxm_sports_begin_nested(pHandle, SXM_SPORTS_TABLE, u32RankListTableRef, &ext, &row2)) == SXM_E_OK)
 	{
 		//The sxm_sports_extract_row routine returns the data for the Leader/Winner event rank list as a row in the table SXM_SPORTS_TABLE.
 		//Each league would be reported in one row.

 		if (((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row2)) == SXM_E_OK) )
 		{
 			//Get instance for Utils
 			fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
 			SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

 			if (((bAvailable = poUtils->bExtractGolfScheduleRankListRow(row2, trGolfScheduleTableData.trGolfRankList)) == TRUE))
 			{
 				ETG_TRACE_USR4((" Name =%s",trGolfScheduleTableData.trGolfRankList.strGolfer.c_str()));
 				ETG_TRACE_USR4((" Score =%s",trGolfScheduleTableData.trGolfRankList.strScore.c_str()));
 				//ETG_TRACE_USR4((" Rank =%s",trGolfScheduleTableData.trGolfRankList.strRank.c_str()));
 			}
 		}

 		else
 			ETG_TRACE_ERR(("Error in Extracting Golf Nested Schedule Row Table Data...errorCode = %u", enRetCode));
 		(tVoid) sxm_sports_end(ext);
 	}
 	else
 	{
 		ETG_TRACE_ERR(("sxm_sports_begin_nested failed at ffc_sxm_tclGolf::bExtractNestedScheduleRow\t enRetCode = %u",enRetCode));
 	}

 	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractNestedScheduleRow Exit"));
 	return bAvailable;
}

/*
 * Method to Extract Golf Schedule Row and Nested Row Data to Method Result
 */
tBool fc_sxm_tclGolf::bExtractScheduleRow(tVoid* pHandle, SXMSportsRow &TableRow,tS32 s32TableParam,tU32 u32EpochValue, tBool /*bIsTeamBased*/)
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractScheduleRow Entered\ts32TableParam = %d", s32TableParam));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

	//Create local Object to hold parsed Table Data
	fc_sxm_trGolfScheduleTable trGolfScheduleTableData;

	//Extract the SMS Row Data to our internal object provided
	tBool bAvailable = poUtils->bExtractGolfScheduleRow(TableRow,trGolfScheduleTableData);
	if (bAvailable)
	{
		string strEventString = trGolfScheduleTableData.strCourseName + trGolfScheduleTableData.strTourneyTitle;
		ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractAndFillScheduleRow Unique Event String = %s", strEventString.c_str()));

		// Generate Unique_ID for the input String
		trGolfScheduleTableData.u32EventID = poUtils->u32ComputeHashCode(&strEventString);
		trGolfScheduleTableData.bIsRankAvailable = TableRow.present & (1 << SXM_SPORTS_SMSE_GOLF_RANK_INDEX);
		trGolfScheduleTableData.u32Epoch = u32EpochValue;

		ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractAndFillScheduleRow Unique Event ID = %u \t Calculated u32Epoch =%u\t GOLF bIsRankAvailable = %u",
				trGolfScheduleTableData.u32EventID,
				trGolfScheduleTableData.u32Epoch,
				trGolfScheduleTableData.bIsRankAvailable));

		// Winner Data Extraction from SXM_SPORTS_SMSE_GOLF_RANK_INDEX Table

		// Data from sxm_sports_begin_nested
		if ((trGolfScheduleTableData.bIsRankAvailable) &&
				(!bExtractNestedScheduleRow(pHandle,trGolfScheduleTableData.u32RankListTableRef,trGolfScheduleTableData)))
		{
			ETG_TRACE_ERR(("Error Parsing Motor Sports Nested Schedule Table Data..."));
			trGolfScheduleTableData.bIsRankAvailable = FALSE;
		}
		// Other data
		trGolfScheduleTableData.s32TableParam = s32TableParam;

		//Check for if this Event already available(To Avoid Duplicate entry)
		if(bIsEventIDAvailable(trGolfScheduleTableData.u32EventID))
		{
			//If successfully parsed,  copy locally
			_setSortedGolfSchedules.insert(trGolfScheduleTableData);
		}
	}
	else
	{
		ETG_TRACE_ERR(("Error Parsing Golf Schedule Table Data..."));
	}

	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractScheduleRow Exited.."));

	//return the status
	return bAvailable;
}


/*
 * Method to Extract Golf Schedule Rank List Data to Method Result
 */
tBool fc_sxm_tclGolf::bExtractEventRankListRow(SXMSportsRow &TableRow)
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractEventRankListRow Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN_VAL( OSAL_NULL!= poUtils , FALSE)

	//Create local Object to hold parsed Table Data
	fc_sxm_trGolfRankList trGolfScheduleRankListData;

	//Extract the SMS Row Data to our internal object provided
	tBool bAvailable = poUtils->bExtractGolfScheduleRankListRow(TableRow,trGolfScheduleRankListData);

	if(bAvailable)
	{
		//If successfully parsed,  copy the data to FI and Send through Method Result to HMI
		_vectorGolfRankList.push_back(trGolfScheduleRankListData);
	}
	else
	{
		ETG_TRACE_ERR(("Error Parsing Golf Schedule Table Data..."));
	}

	ETG_TRACE_USR4(("fc_sxm_tclGolf::bExtractEventRankListRow Exited.."));

	//return the status
	return bAvailable;
}


/*
 * Method to Clear previous Game List Data
 */
tVoid fc_sxm_tclGolf::vClearSchedulesList()
{
	_setSortedGolfSchedules.clear();
}

/*
 * Method to Clear previous Rank List Data
 */
tVoid fc_sxm_tclGolf::vClearEventRankList()
{
	_vectorGolfRankList.clear();
}



tU32 fc_sxm_tclGolf::u32GetEventListIndex(tU32 u32EventID)const
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::u32GetEventListIndex entry\t_setSortedGolfSchedules size=%d",_setSortedGolfSchedules.size()));
	tU32 u32ListIndex = 0;

	tBool bfound = FALSE;

	for(set<fc_sxm_trGolfScheduleTable,trCompareGolfSchedules>::const_iterator cit = _setSortedGolfSchedules.begin();
			cit != _setSortedGolfSchedules.end(); ++cit,++u32ListIndex)
	{
		if (cit->u32EventID == u32EventID)
		{
			bfound = TRUE;
			break;
		}
	}

	if ( FALSE == bfound )
	{
		u32ListIndex = DEFAULT_GAME_EVENT_ID;
	}

	ETG_TRACE_USR4(("fc_sxm_tclGolf::u32GetEventListIndex exited\tu32ListIndex size=%d",u32ListIndex));

	return u32ListIndex;
}

/*
 * Method to Fill Schedule Row Data to Method Result
 */
tVoid fc_sxm_tclGolf::vCopySchedulesToFI(midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::vCopySchedulesToFI Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	ETG_TRACE_USR4(("_setSortedGolfSchedules size=%d",_setSortedGolfSchedules.size()));
	tBool bDSTStatus = _poApp->bGetClockDST();
	string strTimeZone = _poApp->pGetTimeZone();


	for(set<fc_sxm_trGolfScheduleTable,trCompareGolfSchedules>::const_iterator cit = _setSortedGolfSchedules.begin();
			cit != _setSortedGolfSchedules.end(); ++cit)
	{
		midw_ext_fi_tcl_EventSchedule oScheduleInfo;

		//Copy actual Data recieved from SMS
		//Golf Schedule Info
		poUtils->fc_sxm_vString2Fi("", oScheduleInfo.CarNumber);// Not applicable for Golf
		//oScheduleInfo.XMChannelList; No information for Golf
		poUtils->fc_sxm_vString2Fi(cit->strTourneyTitle.c_str(), oScheduleInfo.Tourney_Race);
		poUtils->fc_sxm_vString2Fi(cit->strCourseName.c_str(), oScheduleInfo.Course_Track);
		oScheduleInfo.EventState.enType = (midw_ext_fi_tcl_e8_Game_Status::tenType)cit->u32EventState;

		// Epoch Date
		tChar chDate[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };		
		// Epoch Time
		tChar chTime[SPORTS_MAX_DATE_FORMAT_LEN] = { '\0' };
		
		if(cit->u32Epoch)
		{
			poUtils->vGetOSALDate(cit->u32Epoch, chDate, strTimeZone.c_str(), bDSTStatus);
			poUtils->vGetOSALTime(cit->u32Epoch, chTime, strTimeZone.c_str(), bDSTStatus);
		}
		poUtils->fc_sxm_vString2Fi(chDate,oScheduleInfo.Game_Time.Date);
		poUtils->fc_sxm_vString2Fi(chTime,oScheduleInfo.Game_Time.Time);


		oScheduleInfo.IsRankListAvl = cit->bIsRankAvailable;

		oScheduleInfo.Purse_Laps = cit->u32Purse;
		oScheduleInfo.Yardage_Lap = cit->u32CourseYardage;
		oScheduleInfo.TableParam = cit->s32TableParam;
		oScheduleInfo.TableRef = cit->u32RankListTableRef;

		poUtils->fc_sxm_vString2Fi(cit->trGolfRankList.strGolfer.c_str(), oScheduleInfo.Winner);
		oScheduleInfo.Event_ID = cit->u32EventID;

		//Prepare Method result with actual SMS data
		oMRes.ScheduledEvents.Eventlist.push_back(oScheduleInfo);
	}

	ETG_TRACE_USR4(("fc_sxm_tclTeamSports::vCopySchedulesToFI exited %u", oMRes.ScheduledEvents.Eventlist.size()));
}


/*
 * Method to Print Golf Schedules
 */
tVoid fc_sxm_tclGolf::vPrintSchedules(midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult &oMResSchedule) const
{
	//Print Data
	for(tU32 u32index=0;u32index<oMResSchedule.ScheduledEvents.Eventlist.size();++u32index)
	{
		//All details from the passed object are printed on TTFIs

		ETG_TRACE_USR4((" Tourney Name =%s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Tourney_Race.szValue));
		ETG_TRACE_USR4((" Course Name=%s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Course_Track.szValue));

		ETG_TRACE_USR4(("Event State=%d\tRank Availability =%d\tYardage =%d\tPurse =%d",
				oMResSchedule.ScheduledEvents.Eventlist[u32index].EventState.enType,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].IsRankListAvl,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].Yardage_Lap,
				oMResSchedule. ScheduledEvents.Eventlist[u32index].Purse_Laps));

		ETG_TRACE_USR4(("Winner Name = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Winner.szValue));

		ETG_TRACE_USR4(("TableParam =%d\tTableRef =%d\tEvent_ID = %u",oMResSchedule.ScheduledEvents.Eventlist[u32index].TableParam,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].TableRef,
				oMResSchedule.ScheduledEvents.Eventlist[u32index].Event_ID));
		ETG_TRACE_USR4(("Game Date = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Game_Time.Date.szValue));
		ETG_TRACE_USR4(("Game Time = %s",oMResSchedule.ScheduledEvents.Eventlist[u32index].Game_Time.Time.szValue));
	}
}

/*
 * Method to Fill Schedule Row Data to Method Result
 */
tVoid fc_sxm_tclGolf::vCopyRankListToFI(midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult &oMRes) const
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::vCopyRankListToFI Entered"));

	//Get instance for Utils
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)


	for(vector<fc_sxm_trGolfRankList>::const_iterator cit = _vectorGolfRankList.begin();
			cit != _vectorGolfRankList.end(); ++cit)
	{
		midw_ext_fi_tcl_EventRank oRankInfo;

		//Copy actual Data recieved from SMS
		//Golf Schedule Info
		poUtils->fc_sxm_vString2Fi(cit->strRank.c_str(), oRankInfo.Rank);
		oRankInfo.Laps = 0; // Not Applicable for Golf
		poUtils->fc_sxm_vString2Fi(cit->strGolfer.c_str(), oRankInfo.Name);
		poUtils->fc_sxm_vString2Fi("", oRankInfo.Number);  // Not Applicable for Golf
		poUtils->fc_sxm_vString2Fi(cit->strScore.c_str(), oRankInfo.Score_Make);

		//Prepare Method result with actual SMS data
		oMRes.EventInfo.RankList.push_back(oRankInfo);
	}

	ETG_TRACE_USR4(("fc_sxm_tclGolf::vCopyRankListToFI exited %u", oMRes.EventInfo.RankList.size()));
}


/*
 * Method to Print Golf Schedules
 */
tVoid fc_sxm_tclGolf::vPrintRankList(midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult &oMResSchedule) const
{
	//Print Data
	for(tU32 u32index=0;u32index<oMResSchedule.EventInfo.RankList.size();u32index++)
	{
		//All details from the passed object are printed on TTFIs

		ETG_TRACE_USR4((" Name =%s",oMResSchedule.EventInfo.RankList[u32index].Name.szValue));
		ETG_TRACE_USR4((" Score =%s",oMResSchedule.EventInfo.RankList[u32index].Score_Make.szValue));
		ETG_TRACE_USR4((" Rank =%s",oMResSchedule.EventInfo.RankList[u32index].Rank.szValue));
	}
}

/*
 * Method returns the Rank List Index.
 */

tU8 fc_sxm_tclGolf::u8GetRankedListIndex() const
{
	ETG_TRACE_USR4(("fc_sxm_tclGolf::u8GetRankedListIndex Entered/Exited"));
	//Rank List Index for Golf is 1
	return SXM_SPORTS_SMSE_GOLF_RANK_INDEX;
}

/*
 * Method returns TRUE if the Unique Event ID provided is not part of the events list sent to HMI
 */
tBool fc_sxm_tclGolf::bIsEventIDAvailable(tU32 u32EventID) const
{
	tBool bInsertFlag = TRUE;
	for (set<fc_sxm_trGolfScheduleTable,trCompareGolfSchedules>::const_iterator cIt = _setSortedGolfSchedules.begin(); cIt != _setSortedGolfSchedules.end(); ++cIt)
	{
		if ((cIt->u32EventID) == u32EventID)
		{
			// Same game is identified in the existing list
			ETG_TRACE_USR4(("Identified a Duplicate Event ID = %u",u32EventID));
			bInsertFlag = FALSE;
			break;
		}
	}
	return bInsertFlag;
}
