/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sports_app.cpp
* @brief       Sports application handler
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_main.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_tcl_sms_init.h"
#include "fc_sxm_tcl_sports_app.h"
#include "fc_sxm_tcl_sports_manager.h"
#include "fc_sxm_tcl_sports_utils.h"
#include "fc_sxm_tcl_sports_fi_msgs.h"
#include "fc_sxm_tcl_pers_sports_data.h"
#include "fc_sxm_tcl_sports_properties.h"
#include "fc_sxm_service_sxm_sports.h"

#include "sxm_sxi_internal.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SPORTS_APP
#include "trcGenProj/Header/fc_sxm_tcl_sports_app.cpp.trc.h"
#endif

#define FC_SXM_TIME_IN_MIN 60

/*=============================================================================
  =======                             TYPES                               =======
  =============================================================================*/
/* 
   constructor of the class
 */
fc_sxm_tclSportsApp::fc_sxm_tclSportsApp():
    fc_sxm_tclDataApp("sxmSportsApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_SPORTS, "sports", static_cast <tU16> (TR_CLASS_FC_SXM_SPORTS_APP)),
    _hSports(SPORTS_SERVICE_INVALID_OBJECT),
	_bIsDSTon(FALSE),
    _enSportState(fc_sxm_enSportState_Initial),
    _poSportsManager(OSAL_NULL)
{
    ETG_TRACE_USR4(("fc_sxm_tclSportsApp constructor"));
    /* Initialize the class members */
    _strTimeZone.assign("+0000");	//by Default GMT time
}

/*
  Destructor
*/

fc_sxm_tclSportsApp::~fc_sxm_tclSportsApp()
{
   ETG_TRACE_USR4(("fc_sxm_tclSportsApp destructor"));
   _hSports = SPORTS_SERVICE_INVALID_OBJECT;
    _poSportsManager = OSAL_NULL;
   _strTimeZone.clear();
}

/* 
	Attention: called in cca-context by sxmapp-manager 
*/
tVoid fc_sxm_tclSportsApp::vInitialize(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vInitialize"));
   //Create the sports worker thread

   /* call vInitialize of base-class to start the worker thread */
   fc_sxm_tclBaseApp::vInitialize();
   _poSportsManager = fc_sxm_tclSportsManager::instance();
}

/* 
	Attention: called in cca-context by sxmapp-manager 
*/

tVoid fc_sxm_tclSportsApp::vDeInitialize(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vDeInitialize"));

	/* call vDeInitialize() of base-class to stop the worker thread */
   fc_sxm_tclBaseApp::vDeInitialize();

   /* Deinit of all other stuff starts from here */
   FC_SXM_MEMORY_CLEAR(_poSportsManager)
   fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
   FC_SXM_MEMORY_CLEAR(poUtils)
}

/*
   this method is called by the thread-function.
   For each message that shall be handled put and entry here.
*/
tVoid fc_sxm_tclSportsApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {

	// Get Action Value
   tU32 u32Action=poThreadMsg->u32GetAction();

   switch (u32Action) {

      // dispatch Sports Service State mesaage to CCA
      // Also Do this for every method exposed thru FI
   SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgDataSportsState,fc_sxm_tclSportsApp::instance());

      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetSportList,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetRootAffiliateList,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetAffiliateDetails,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetAllTeams,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetGameList,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetScheduledEvents,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetEventRankList,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsSMSeLeagueCallback,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetNewsDetails,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartGetFavoriteList,fc_sxm_tclSportsApp::instance());
      SXM_MSGQ_DISPATCH_TO_OBJ (fc_sxm_trMsgSportsMStartSetAsFavorite,fc_sxm_tclSportsApp::instance());

	  default:
      ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                          u32Action));
      break;
   }
}

/*
   Method to start SMSe service
*/
tVoid fc_sxm_tclSportsApp::vStartSmsService()
{
	tS32 s32Debuglevel = 0;
	tS32 s32SportsStart ;
	tU8 u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_INITIAL);

	//Transition to INITIAL state by default
	fc_sxm_trMsgDataSmsEvtServiceState rMsg;
	rMsg.eState = DATASERVICE_STATE_INITIAL;
	rMsg.eProductState = DATA_PRODUCT_STATE_READY;
	rMsg.enErrorCode = DATASERVICE_ERROR_CODE_NONE;

	fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);

    ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vStartSmsService"));
    //Method invokes SMSe to start sports service. The DebugLevel for SMS app is currently set to 0
    s32SportsStart = sxm_sports_start(cb_vDataSportsServiceCallback,s32Debuglevel);

    if (s32SportsStart != (tS32) SXM_E_OK )
    {
		ETG_TRACE_USR4(("fc_sxm_tclSportsApp::sxm_sports starts failed:%d",s32SportsStart));
		_hSports = SPORTS_SERVICE_INVALID_OBJECT;

		//update our internal f/w about Data Service Error.
    	fc_sxm_trMsgDataSmsEvtServiceState rMsg;
        rMsg.eState = DATASERVICE_STATE_ERROR;
        //Check if it because of baseline database.
        if (s32SportsStart == (tS32) SXM_E_PIPE ||
        		s32SportsStart == (tS32) SXM_E_ERROR )
        {
            rMsg.enErrorCode = DATASERVICE_ERROR_CODE_DATABASE_CORRUPT;
        }
        //finally call base function to handle new state
    	fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);

		_enSportState = fc_sxm_enSportState_Error;
		u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_ERROR);
	}
    else
    {
		ETG_TRACE_USR4(("fc_sxm_tclSportsApp::sxm_sports starts success"));
		_hSports = (SPORTS_SERVICE_OBJECT)SXM_E_STATE;
		SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)

		tU32 u32SportCount = _poSportsManager->u32GetSportsObjectCount();
		ETG_TRACE_USR4(("u32SportCount = %u\t_enSportState = %u", u32SportCount, _enSportState));

		if ((_enSportState != fc_sxm_enSportState_Stopped) ||
				(u32SportCount == 0) )
		{
			//Make sure that the Sports Object Factory is initialised
			_poSportsManager->vInitSportsManager();

			// Start copying sport list data from persistency class to sports class
			tU32 u32SportsListSize = _poSportsManager->u32LoadSportsList();
			tBool bSportsSize = static_cast <tBool> ((u32SportsListSize > 0u));
			ETG_TRACE_USR4(("u32SportsListSize (%u)\t bSportsSize(%u)", u32SportsListSize, bSportsSize));
			if ( TRUE == bSportsSize )
			{
				// Start copying root list data from persistency class to sports class
				_poSportsManager->vLoadLeagueList();
				// TODO: Start copying favorite list data from persistency class to sports class
				u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_UPDATING);
			}
			_poSportsManager->vSetSportsListStatus(bSportsSize);
		}
		else if (u32SportCount > 0)
		{
			u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_UPDATING);
			_poSportsManager->vSetSportsListStatus(TRUE);
		}
		_enSportState = fc_sxm_enSportState_Initial;
	}

	//Send property update to HMI
    vSendSportsDataStatus(u8Status);
}

/*
   Method to stop SMSe service
*/
tVoid fc_sxm_tclSportsApp:: vStopSmsService()
{
	 ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vStopSmsService"));
     _bSmsServiceStopping=TRUE;
     tU8 u8Status;

    //Clear internal Data
    _bSmsServiceStopping=TRUE;
    vClearServiceData();
    vAllowService(FALSE);

     //Method invokes SMSe to stop sports service.
     tS32 s32SxmSportsStopService =  sxm_sports_stop();
     ETG_TRACE_USR4(("Sports service stopped: %d",s32SxmSportsStopService));
     if(s32SxmSportsStopService == (tS32) SXM_E_OK)
     {
    	 _enSportState = fc_sxm_enSportState_Stopped;
    	 //Send property update to HMI that the service is stopped
    	 u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_STOPPED);
     }
     else
     {
    	 // Report that an error has happened
    	 _enSportState = fc_sxm_enSportState_Error;
    	 u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_ERROR);
     }
	 //The updated state is passed back to DataApp
	 fc_sxm_trMsgDataSmsEvtServiceState rMsg;
	 rMsg.eState = DATASERVICE_STATE_STOPPED;
	 rMsg.eProductState = DATA_PRODUCT_STATE_READY;
	 rMsg.enErrorCode = DATASERVICE_ERROR_CODE_NONE;
	 fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);

     vSendSportsDataStatus(u8Status);
}

/****************************************************************************
*
*FUNCTION:          vClearServiceData
*
*DESCRIPTION:       To clear all the stored data during reinitialisation of SMS.
                    This is a overridden method from the data app.
*PARAMETERS:
*
*RETURNVALUES:      None
*
******************************************************************************/
/*
   Method to be Overriden by each Service
   to clear Service specific internal data
   - Clear User Data - SportObjects, Sportlist, Root List , Fav List , remove league Request
*/
tVoid fc_sxm_tclSportsApp::vClearServiceData()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vClearServiceData(%s)",
		szGetName()));
	// TODO: Clear the internal sport structure
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vClearServiceData() Exit"));
}
/*
   Method Handles the service state change from UPDATING to UPDATED.
*/
tVoid fc_sxm_tclSportsApp::vHandleUpdatedState()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vHandleUpdatedState\t SportsState == SXM_CALLBACK_SPORTS_UPDATED \t Entered"));
	_enSportState = fc_sxm_enSportState_Updated;
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)

	// If the sports list and league list are not yet created,
	if (FALSE == _poSportsManager->bGetSportsListStatus())
	{
		// Sports list is not yet constructed.
		ETG_TRACE_USR4(("Requesting sports manager to create sports list"));

		//Make sure that the Sports Objects read from Persistent memory is cleared
		_poSportsManager->vClearSportsObjects();

		//Extract the list of available sports and let manager create individual classes for every available sport
		vCreateSportsList();
	}
	else
	{
		ETG_TRACE_USR4(("Sports List already available; Clearing only League data"));
		//Retain the sports objects created
		_poSportsManager->vClearLeagueData();
	}
	//Build available leagues specific for individual sports
	vCreateLeagueList();
	// Read Favorite team names from Global Metadata and reset FavoritesList
	vUpdateFavoriteList();
	_poSportsManager->vSetSportsListStatus(TRUE);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vHandleUpdatedState Exit"));
}

/*
   Method Invoked from BaseApp on every state change reported by SMSe for sports service
*/
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgDataSportsState const *prMsg)
{
  	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgDataSportsState)"));
    fc_sxm_trMsgDataSportsState rMsg = *prMsg;

    //Take action based on the state change reported by SMSe
    if (rMsg.s32SportsState == SXM_CALLBACK_SPORTS_UPDATED)
    {
    	vHandleUpdatedState();
		/* Start copying sport list data from sports class to persistency class */
		fc_sxm_tclSportsManager::instance()->vStoreSportsList();
		/* Start copying root list data from sports class to persistency class */
		fc_sxm_tclSportsManager::instance()->vStoreLeagueList();
		/* TODO: Start copying favorite list data from sports class to persistency class */
		tU8 u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_UPDATED);
		vSendSportsDataStatus(u8Status);
		vNotifyMetaDataUpdate(FALSE);
    }
    else if (rMsg.s32SportsState == SXM_CALLBACK_SPORTS_UPDATING)
    {
    	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgDataSportsState)\tSportsState == SXM_CALLBACK_SPORTS_UPDATING "));
    	if (_enSportState == fc_sxm_enSportState_Updated)
    	{
    		// If the state goes from Updated --> Updating, then its a broadscope change
    		vHandleBroadScopeChanges();
    	}
    	_enSportState = fc_sxm_enSportState_Updating;
    }
    // TODO: identify and check if the other states reported by SMSe needs to be handled
    ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgDataSportsState) "));
}
/*
   Method Invoked for every Time Zone Change from Clock
*/
tVoid fc_sxm_tclSportsApp::vSetClockParams(fc_sxm_trMsgDataAppSetClockParams const *prMsg){

	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vSetClockParams(fc_sxm_trMsgDataAppSetClockParams)"));
	ETG_TRACE_USR4((" DSTStatus = %d " , prMsg->bDSTStatus));

	//Copy to locally
	_strTimeZone.assign(prMsg->strTimeZone.szValue);
	_bIsDSTon = prMsg->bDSTStatus;

	//Where there is TImeZone Change
	//We should also GMT offset value in SMS
	vSetSMSGMTOffSet();
	ETG_TRACE_USR4((" TimeZone = %s " , prMsg->strTimeZone.szValue));  

	//Finally inform client about new timezone change
	fc_sxm_tclSportsManager::instance()->vSendTimeChange();

}
/*
 * Method handles the broadscope changes reported by SMSe
 */
tVoid fc_sxm_tclSportsApp::vHandleBroadScopeChanges()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vHandleBroadScopeChanges Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	/*
	 * When a broadscope change is reported, applications must
	 * 1) clear all cached data including sports list and affiliate list.
	 * 2) Remove all requests for league data
	 * 3) Notify App Mgr that BroadScope change is in progress so that it can be propogated
	 *    to other clients.
	 */
	ETG_TRACE_USR4(("Act upon Broadscope changes reported by SMSe"));
	_poSportsManager->vSetSportsListStatus(FALSE);
	ETG_TRACE_USR4(("Local cached data cleared -- Notify HMI with Property update"));
	tU8 u8Status = static_cast<tU8> (midw_ext_fi_tcl_e8SportsServiceDataStatus::FI_EN_SPORTS_SERVICE_DATA_RELOAD);
	vSendSportsDataStatus(u8Status);
	vNotifyMetaDataUpdate(TRUE);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vHandleBroadScopeChanges Exit"));
}

/*
 * Method to Extract the list of available sports and let manager create individual classes for every available sport
 */
tVoid fc_sxm_tclSportsApp::vCreateSportsList()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vCreateSportsList() Entered"));
	ptr ext = OSAL_NULL;
	sxm_result_code enRetCode;
	SXMSportsRow row, header;

	//Query SMSe to send the list of available sports.
	if ((enRetCode = (sxm_result_code) sxm_sports_begin(NULL, SXM_SPORTS_LIST, 0, &ext, &header)) != SXM_E_OK)
	{
		ETG_TRACE_ERR(("fc_sxm_tclSportsApp Return Value\t vCreateSportsList enRetCode = %d", enRetCode));
		ETG_TRACE_ERR(("fc_sxm_tclSportsApp::vCreateSportsList() SXM SPORTS BEGIN ERROR FOR MAKING REQUEST SXM_SPORTS_LIST"));
	}
	else
	{
		fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
		SXM_ASSERT_RETURN( OSAL_NULL!= poUtils )
		SXM_ASSERT_RETURN( OSAL_NULL!=_poSportsManager)

		// Iterate until all rows are read.
		while ((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			//The sxm_sports_extract_row routine returns the data for all sports as a row in the table SXM_SPORTS_LIST.
			//Each sport would be reported in one row.
			//After receiving row data, extract or interpret the values from row so that it can be stored as per HMI requirements
			fc_sxm_trSportsList sportConfig;
			if (TRUE == poUtils->bExtractSportsRow(row, sportConfig))
			{
				//Create Sports data object only if extraction of row data is successful
				_poSportsManager->vCreateSportsObject(sportConfig);
			}
		}	//end for loop

		if( enRetCode != SXM_E_NOENT)
		{
			//Upon reading all available rows; enRetCode should be set to SXM_E_NOENT by SXM.
			//This error condition check is applicable both if the number of sports list item = 0 or end of row has arrived
			ETG_TRACE_ERR(("fc_sxm_tclSportsApp ERROR msg\t enRetCode = %d\text= %x", enRetCode, ext));
		}

		// End routine terminates the extraction unlocks the data for further updates.
		enRetCode = (sxm_result_code) sxm_sports_end(ext);
		ETG_TRACE_USR4(("fc_sxm_tclSportsApp Return Value\t enRetCode = %d", enRetCode));
	} // end else
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vGetSportsList() Exit"));
}


/*
 * Extract the list of available sports and let manager create individual classes for every available sport
 */
tVoid fc_sxm_tclSportsApp::vCreateLeagueList()
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vCreateLeagueList() Entered"));
	ptr ext = OSAL_NULL;
	sxm_result_code enRetCode;
	SXMSportsRow row, header;

	//Query SMSe to send the list of all available leagues for all sports. Each league would be reported in one row.
	if ((enRetCode = (sxm_result_code) sxm_sports_begin(NULL, SXM_SPORTS_LEAGUES, 0, &ext, &header)) != SXM_E_OK)
	{
		ETG_TRACE_ERR(("fc_sxm_tclSportsApp::vCreateLeagueList(): enRetCode = %d", enRetCode));
		ETG_TRACE_ERR(("fc_sxm_tclSportsApp::vCreateLeagueList() SXM SPORTS BEGIN ERROR FOR MAKING REQUEST SXM_SPORTS_LEAGUES"));
	}
	else
	{
		fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
		SXM_ASSERT_RETURN( OSAL_NULL!= poUtils )
		SXM_ASSERT_RETURN( OSAL_NULL!=_poSportsManager)

		// Iterate until all rows are read.
		while ((enRetCode = (sxm_result_code) sxm_sports_extract_row(ext, &row)) == SXM_E_OK)
		{
			//The sxm_sports_extract_row routine returns the data for leagues from all sports as a row in the table SXM_SPORTS_LEAGUES.
			//Each league would be reported in one row.
			fc_sxm_trLeagueListData league;
			if (TRUE == poUtils->bExtractLeagueRow(row, league))
			{
				league.setFamily.insert(league.u32AfID);
				league.u8Level = 0u;
				//Create Sports data object only if extraction of row data is successful
				_poSportsManager->vBuildLeagueList(league);
			}
		} //end for loop

		if( enRetCode != SXM_E_NOENT)
		{
			//Upon reading all available rows; enRetCode should be set to SXM_E_NOENT by SXM.
			//This error condition check is applicable both if the number of sports list item = 0 or end of row has arrived
			ETG_TRACE_ERR(("fc_sxm_tclSportsApp ERROR msg\t enRetCode = %d\text= %x", enRetCode, ext));
		}
		enRetCode = (sxm_result_code) sxm_sports_end(ext);
		ETG_TRACE_USR4(("fc_sxm_tclSportsApp Return Value\t enRetCode = %d", enRetCode));
	} //end else
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vCreateLeagueList() Exit"));
}

/*
 * Method reads Team Names from Global metadata and resets Favorites List from Persistent memory
 */
tVoid fc_sxm_tclSportsApp::vUpdateFavoriteList() const
{
	ETG_TRACE_USR1(("fc_sxm_tclSportsApp::vUpdateFavoriteList Entered"));
	tS32 s32RetCode = 0;
	tU32 u32FavCount = trSportsUserData::instance()->u32GetFavoriteListSize();
	// Copy data and pass it to the requested client
	for(tU32 u32Index = 0u; u32Index < u32FavCount; ++u32Index) {
		// Get data at each list index
		trFavoriteListElement tListElement;
		trSportsUserData::instance()->vGetFavoriteList(u32Index, tListElement);
		ETG_TRACE_USR1(("Team Name to be validated = %s", tListElement.sTeamName.c_str()));

		SXMTeam team;
		if ((s32RetCode = sxm_sxi_global_team( tListElement.u32TeamID, tListElement.u32GDRef, &team)) == SXM_E_OK)
		{
			string strKnwonName = team.ln;
			string strNickName = team.mn;
			string strTeamName = strKnwonName + " " + strNickName;
			string strAbrName = team.sn;

			// Reset team name only if there is a change
			if (strTeamName.compare(tListElement.sTeamName) != 0)
			{
				ETG_TRACE_USR1(("Team Names differ; New Team Name = %s", strTeamName.c_str()));
				(tVoid) trSportsUserData::instance()->bUpdateTeamName(tListElement.u8SportID,
						tListElement.u32TeamID, tListElement.u32LeagueID, strTeamName, strAbrName, strKnwonName, strNickName);
			}
		}
	}

	ETG_TRACE_USR1(("fc_sxm_tclSportsApp::vUpdateFavoriteList Exit \ts32RetCode = %d", s32RetCode));
}

/*
 * Method sends property updates about Sports Service State to HMI
 */
tVoid fc_sxm_tclSportsApp::vSendSportsDataStatus(tU8 u8ServiceStatus) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vSendSportsDataStatus Entered\tStatus = %u", u8ServiceStatus));
	// Property update on ServiceState Changed status to HMI
	fc_sxm_tcl_trSportsPropertyServiceDataStatus oProperty;
	oProperty.oFiMsg.SportsStatus.enType =
			static_cast <midw_ext_fi_tcl_e8SportsServiceDataStatus::tenType> (u8ServiceStatus);

	fc_sxm_tclSportsProperties::instance()->oSportsServiceDataStatus.vSetAndNotify(oProperty);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vSendSportsDataStatus EXit"));
}

/*
 * Method sends property updates about Sports Data Update to HMI
 */
tVoid fc_sxm_tclSportsApp::vSendSportsDataUpdateStatus(tU8 u8SportID, tU32 u32LeagueID , tU8 u8UpdateType ,tU8 u8UpdateStatus) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vSendSportsDataUpdateStatus Entered\t Sport ID = %u , LeagueID = %u, UpdateType = %u , UpdateStatus = %u", u8SportID,u32LeagueID,u8UpdateType,u8UpdateStatus ));
	// Property update on data Update status to HMI
	fc_sxm_tcl_trSportsPropertyDataUpdate oProperty;

	oProperty.oFiMsg.SportID = u8SportID; // Sport ID

	oProperty.oFiMsg.LeagueID = u32LeagueID; //LeagueID

	oProperty.oFiMsg.UpdateType.enType =
				static_cast <midw_ext_fi_tcl_SportsUpdateType::tenType> (u8UpdateType); // UpdateType

	oProperty.oFiMsg.UpdateStatus.enType =
					static_cast <midw_ext_fi_tcl_e8CategoryAvailability::tenType> (u8UpdateStatus); // UpdateStatus

	fc_sxm_tclSportsProperties::instance()->oSportsDataUpdateStatus.vSet(oProperty);
	fc_sxm_tclSportsProperties::instance()->oSportsDataUpdateStatus.vNotify();

	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vSendSportsDataUpdateStatus EXit"));
}


/*
 * SMSe Callback message - League Updates
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsSMSeLeagueCallback const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsSMSeLeagueCallback) Entered"));

	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager);
	_poSportsManager->vRouteMsgToSport(prMsg);

	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsSMSeLeagueCallback) Exit"));
}

/******************************************************************
 * fi-messages
 ******************************************************************/

/*
 * FI message --> methodStart for GetSportList
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetSportList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetSportList) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager);
	_poSportsManager->vSendSportsList(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetSportList) Exit"));
}

/*
 * FI message --> methodStart for GetRootAffiliateList
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetRootAffiliateList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetRootAffiliateList) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager);
	_poSportsManager->vSendLeagueList(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetRootAffiliateList) Exit"));
}

/*
 * FI message --> methodStart for GetAffiliateDetails
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAffiliateDetails const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAffiliateDetails) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendAffiliateDetails(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAffiliateDetails) Exit"));
}

/*
 * FI message --> methodStart for GetAllTeams
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAllTeams const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAllTeams) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendAllTeams(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetAllTeams) Exit"));
}
/*
 * FI message --> methodStart for GetGameList
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetGameList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetGameList) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendGameList(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetGameList) Exit"));
}

/*
 * FI message --> methodStart for GetScheduledEvents for Non Team Sports ( Golf and Motor Race)
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetScheduledEvents const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetScheduledEvents) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendScheduledEvents(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetScheduledEvents) Exit"));
}
/*
 * FI message --> methodStart for GetEventRankList for Non Team Sports ( Golf and Motor Race)
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetEventRankList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetEventRankList) Entered"));
	//Method forwards the CCA message to manager for appropriate action.
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendEventRankList(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetEventRankList) Exit"));
}
/*
 * FI message --> methodStart for GetNewsDetails
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetNewsDetails const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetNewsDetails) Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
	_poSportsManager->vSendNewsDetails(prMsg);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetNewsDetails) Exit"));
}

/*
 * FI message --> methodStart for GetFavoriteList
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetFavoriteList const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetFavoriteList) Entered"));
	fc_sxm_tclSportsUtils* poUtils = fc_sxm_tclSportsUtils::instance();
	SXM_ASSERT_RETURN( OSAL_NULL!= poUtils)

	/* Method result data */
	midw_ext_sxm_sportsfi_tclMsgGetFavoritesMethodResult oMRes;
	/* Get favorite list size from persistency */
	tU32 u32FavoriteListSize = trSportsUserData::instance()->u32GetFavoriteListSize();
	/* Copy data and pass it to the requested client */
	for(tU32 u32Count=0; u32Count<u32FavoriteListSize; u32Count++) {
		midw_ext_fi_tcl_FavoriteInfo oFiEntry;  /* fav list element data */
		trFavoriteListElement tListElement;
		trSportsUserData::instance()->vGetFavoriteList(u32Count, tListElement);   /* Get data at each list index */
		/* Copy data to method result */
		oFiEntry.SPORT_ID = tListElement.u8SportID;
		oFiEntry.Team_ID = tListElement.u32TeamID;
		oFiEntry.AFFILIATE_ID = tListElement.u32LeagueID;
		poUtils->fc_sxm_vString2Fi(tListElement.sTeamName.c_str(),oFiEntry.Team_Name);     /* convert to midw fi string type */
		poUtils->fc_sxm_vString2Fi(tListElement.sTeamAbrName.c_str(),oFiEntry.Team_Abbr_Name);     /* convert to midw fi string type */
		poUtils->fc_sxm_vString2Fi(tListElement.sTeamKnownName.c_str(),oFiEntry.Team_Known_Name);     /* convert to midw fi string type */
		poUtils->fc_sxm_vString2Fi(tListElement.sTeamNickName.c_str(),oFiEntry.Team_Nick_Name);     /* convert to midw fi string type */
		poUtils->fc_sxm_vString2Fi(tListElement.sSportName.c_str(),oFiEntry.Sport_Name);   /* convert to midw fi string type */
		oMRes.FavoriteList.FavoriteInfo.push_back(oFiEntry);         /* insert data to list element */
	}
	// Send FI message
	fc_sxm_tclSportsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartGetFavoriteList) Exit"));
}

/*
 * FI message --> methodStart to SetAsFavorite
 */
tVoid fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartSetAsFavorite const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartSetAsFavorite) Entered"));
	/* Method result data */
	midw_ext_sxm_sportsfi_tclMsgSetAsFavoriteMethodResult oMRes;
	tBool bSuccess = TRUE;
	if((prMsg->oFiMsg).FavoriteControl) {

		bSuccess = fc_sxm_tclSportsManager::instance()->bSetFavoriteTeam((prMsg->oFiMsg).SportID,(prMsg->oFiMsg).AFFILIATE_ID,(prMsg->oFiMsg).TEAM_ID);

	} else {
		trSportsUserData::instance()->vClearFavoriteListElement((prMsg->oFiMsg).SportID, (prMsg->oFiMsg).TEAM_ID, (prMsg->oFiMsg).AFFILIATE_ID);
		/* TODO: In case of restore factory default remember to clear the entire list */
	}
	/* Send method result status as success */
	oMRes.Result = bSuccess;
	/* Send Favorite control method result status */
	oMRes.FavoriteControl = (prMsg->oFiMsg).FavoriteControl;
	// Send FI message
	fc_sxm_tclSportsService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgSportsMStartSetAsFavorite) Exit"));
}

/*********************************************************************
*FUNCTION:     vProcess(fc_sxm_trMsgCmdAppClearUserData)
*DESCRIPTION:  Clear Favorites
*PARAMETER:  fc_sxm_trMsgCmdAppClearUserData const *prMsg
*RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclSportsApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
{
	(void)prMsg; /* TO REMOVE LINT WARNINGS */
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgCmdAppClearUserData(%s)",
		szGetName()));

	// check for valid sports object
	if(SPORTS_SERVICE_INVALID_OBJECT != _hSports)
	{
		ETG_TRACE_USR4(("fc_sxm_tclSportsApp:: Clear All Favorite"));

		/* Call Persistant data  Methods  to Clear All Favorites */
		trSportsUserData::instance()->vClearSportList();

		trSportsUserData::instance()->vClearRootList();

		trSportsUserData::instance()->vClearFavoriteList();

		/* Call SMS API to Remove all request */

		fc_sxm_tclSportsManager::instance()->vRemoveLeagueRequests();

		/* Say Result is OK */
		vSetUsrDataClearResult(fc_sxm_enDiagResult_Ok);

	}
	else
	{
		ETG_TRACE_ERR(( " SPORTS_SERVICE_INVALID_OBJECT " ));
		/* Say Result is not OK */
		vSetUsrDataClearResult(fc_sxm_enDiagResult_Failed);
	}
}
/*********************************************************************
 *FUNCTION:     vProcess
 *DESCRIPTION:  handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd (Mandatory)
 *PARAMETER:    fc_sxm_trMsgCmdAppTtfisCmd
 *RETURNVALUE:  None
 ********************************************************************/
tVoid fc_sxm_tclSportsApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
{
	// Method to handle different TTFI's cmds fc_sxm_trMsgCmdAppTtfisCmd
	//fc_sxm_tenTtfisCmdsSports enMsgCode = (fc_sxm_tenTtfisCmdsSports) prMsg->u8MsgCode;
	//ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Entered code = %u",
		//	ETG_CENUM(fc_sxm_tenTtfisCmdsSports, enMsgCode)));
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) Entered code = %u",
			prMsg->u8MsgCode));

	tVoid (fc_sxm_tclSportsApp::*funcPtrArray[]) (fc_sxm_trMsgCmdAppTtfisCmd const *) const = {
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdSportList,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdSportList,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdRootAffiliateList,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdAffiliateDetails,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdAllTeams,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdSchedules,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdNewsDetails,
    		&fc_sxm_tclSportsApp::vHandleTTFIsCmdEventSchedules,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdEventRankList,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdFavoriteList,
			&fc_sxm_tclSportsApp::vHandleTTFIsCmdSetAsFavorite};

	//Store the values read for each index to the appropriate field in fc_sxm_trSportsList structure
	(this->*funcPtrArray[prMsg->u8MsgCode])(prMsg);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETSPORTLIST issued from TTFIs
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdSportList(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	// Handle TTFI's cmd GetSportList
	(tVoid) prMsg;
	midw_ext_sxm_sportsfi_tclMsgGetSportListMethodStart oMStart;
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETROOTAFFILIATELIST %u()  issued from TTFIs
 * where %u is the sportsID
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdRootAffiliateList(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	// Handle TTFI's cmd GetRootAffiliateList
	midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodStart oMStart;
	oMStart.SportID = prMsg->au8Data[0];
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETAFFILIATEDETAILS %u() %u() issued from TTFIs
 * where first param is the AffiliateID and the second param is the sportsID
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdAffiliateDetails(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetAffiliateDetails
	midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodStart oMStart;
	oMStart.AFFILIATE_ID = prMsg->au8Data[0];
	oMStart.SportID = prMsg->au8Data[1];
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETALLTEAMS %u() %u() issued from TTFIs
 * where first param is the AffiliateID and the second param is the sportsID
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdAllTeams(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetAllTeams
	midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodStart oMStart;
	oMStart.AFFILIATE_ID = prMsg->au8Data[0];
	oMStart.SportID = prMsg->au8Data[1];
	fc_sxm_vSendSelfMessage(oMStart);
}
/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETSCHEDULES %u() %d() %u issued from TTFIs
 * where first param is the sportsID and the second param is the LeagueID and 3rd GameType
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdSchedules(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetAllTeams
	midw_ext_sxm_sportsfi_tclMsgGetGameListMethodStart oMStart;
	oMStart.SportID = prMsg->au8Data[0];
	oMStart.AFFILIATE_ID = SXM_GET_U32(&prMsg->au8Data[1]);
	oMStart.GameType.enType = (midw_ext_fi_tcl_GameType::tenType)prMsg->au8Data[5];
	oMStart.TEAM_ID = SXM_GET_U32(&prMsg->au8Data[6]);
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETSCHEDULES %u() %d() %u issued from TTFIs
 * where first param is the sportsID and the second param is the LeagueID and 3rd GameType
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdEventSchedules(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetEventSchedules
	midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodStart oMStart;
	oMStart.SportID = prMsg->au8Data[0];
	oMStart.AFFILIATE_ID = SXM_GET_U32(&prMsg->au8Data[1]);
	oMStart.EventType.enType = (midw_ext_fi_tcl_GameType::tenType)prMsg->au8Data[5];
	fc_sxm_vSendSelfMessage(oMStart);
}


/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETEVENTRANKLIST %u() %u() %u  %d issued from TTFIs
 * where first param is the sportsID and the second param is the LeagueID and 3rd GameType and 4 th is TableParam
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdEventRankList(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetEventRankList
	midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodStart oMStart;
	oMStart.SportID = prMsg->au8Data[0];
	oMStart.AFFILIATE_ID = SXM_GET_U32(&prMsg->au8Data[1]);
	oMStart.EventType.enType = (midw_ext_fi_tcl_GameType::tenType)prMsg->au8Data[5];
	oMStart.TableParam = (tS32)SXM_GET_U32(&prMsg->au8Data[6]);
	fc_sxm_vSendSelfMessage(oMStart);
}
/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETNEWSDETAILS %u() %u() issued from TTFIs
 * where first param is the AffiliateID and the second param is the sportsID
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdNewsDetails(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	//Handle for TTFI's cmd GetAffiliateDetails
	midw_ext_sxm_sportsfi_tclMsgGetNewsDetailsMethodStart oMStart;
	oMStart.AFFILIATE_ID = prMsg->au8Data[0];
	oMStart.SportID = prMsg->au8Data[1];
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_GETFAVORITELIST issued from TTFIs
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdFavoriteList(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	// Handle TTFI's cmd GetFavoriteList
	(tVoid) prMsg;
	midw_ext_sxm_sportsfi_tclMsgGetFavoritesMethodStart oMStart;
	fc_sxm_vSendSelfMessage(oMStart);
}
/*
 * TTFIs cmd --> Handle FC_SXM_SPORTS_MSTART_SETASFAVORITE issued from TTFIs
 */
tVoid fc_sxm_tclSportsApp::vHandleTTFIsCmdSetAsFavorite(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) const
{
	// Handle TTFI's cmd SetAsFavorite
	(tVoid) prMsg;
	midw_ext_sxm_sportsfi_tclMsgSetAsFavoriteMethodStart oMStart;
	oMStart.SportID = prMsg->au8Data[0];
	oMStart.AFFILIATE_ID = prMsg->au8Data[1];
	oMStart.TEAM_ID = prMsg->au8Data[2];
	oMStart.FavoriteControl = prMsg->au8Data[3];
	fc_sxm_vSendSelfMessage(oMStart);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetAffiliateDetailsMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetRootAffiliateListMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetAllTeamsMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetNewsDetailsMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetNewsDetailsMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetNewsDetailsMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetGameListMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetScheduledEventsMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendFI msg --> Method sends FI msg, midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult
 */
tVoid fc_sxm_tclSportsApp::vSendFiMsg(fc_sxm_trAdressing const  &rAdressing, const midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult& oMRes)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendFiMsg for midw_ext_sxm_sportsfi_tclMsgGetEventRankListMethodResult"));
	fc_sxm_tclSportsService::instance()->enSendFiMessage(rAdressing, oMRes);
}

/*
 * SendError msg --> Method sends Error msg
 */
tVoid fc_sxm_tclSportsApp::vSendErrorMsg(fc_sxm_trAdressing const  &rAdressing, tInt errorMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vSendErrorMsg \tError = %d", errorMsg));
	fc_sxm_tclSportsService::vSendError(rAdressing, errorMsg);
}

/*
 * Method post msg to internal message queue. The msg would be acted upon from SMSe sports thread context
 */
tVoid fc_sxm_tclSportsApp::vPostInternalMsg(const fc_sxm_trMsgSportsSMSeLeagueCallback& leagueCB)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp vPostInternalMsg"));
	vPostMsgNew(leagueCB);
}

/*
 * Method prints the status/values stored for sports service
 */
tVoid fc_sxm_tclSportsApp::vPrintReportSpecific()
{
    // for debugging purpose
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t Sports specific data:"));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tSports Data Service Status  =%u", ETG_CENUM(fc_sxm_tenSportState,_enSportState)));
    SXM_ASSERT_RETURN(OSAL_NULL!=_poSportsManager)
    _poSportsManager->vPrintSportsList();
    _poSportsManager->vPrintLeagueList();
}


/****************************************************************************
 *FUNCTION:          vProcess:fc_sxm_trSportsTimer
 *DESCRIPTION:       On expiry of Timer, this method is invoked by Framework
 *PARAMETERS:        fc_sxm_trStocksTimer *prMsg
 *RETURNVALUES:      None
 *****************************************************************************/
tVoid fc_sxm_tclSportsApp::vProcessTimer(fc_sxm_trSportsTimer const *prMsg)
{
	(tVoid)prMsg;
	ETG_TRACE_USR4(("--->fc_sxm_tclSportsApp::vProcess"
				"(fc_sxm_trSportsTimer *prMsg)"));
	// Stop the timer and notify time out
	vStopSportsTimer();
	vSportsTimerNotifyTimeOut();
	ETG_TRACE_USR4(("<---fc_sxm_tclSportsApp::vProcess"
				"(fc_sxm_trSportsTimer *prMsg)"));
}

/*****************************************************************************
 * FUNCTION:          vStopSportsTimer:
 * DESCRIPTION:       To stop the timer
 * PARAMETER:         tVoid
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclSportsApp::vStopSportsTimer(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsApp::vStopSportsTimer() Entered/Exit"));
	// Stop the sports timer if its already Running
	if (bIsSportsTimerRunning())
		_mSportsTimer.vStop();
}
/*****************************************************************************
 * FUNCTION:          bIsSportsTimerRunning
 * DESCRIPTION:       To check the timer status if its running or not
 * PARAMETER:         tVoid
 * RETURNVALUE:       tBool
 *****************************************************************************/
tBool fc_sxm_tclSportsApp::bIsSportsTimerRunning(tVoid) const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::bIsSportsTimerRunning() Entered/Exit"));
	// Returns true if sports timer is already Running
	tBool bTimerRunningState = _mSportsTimer.bIsRunning();
	ETG_TRACE_USR4(("--->fc_sxm_tclSportsBase::bIsSportsTimerRunning() bTimerRunningState  = %d mSec", bTimerRunningState));
	return(bTimerRunningState);
}

/*****************************************************************************
 * FUNCTION:          vSportsTimerNotifyTimeOut
 * DESCRIPTION:       On timeout, notify the property Update to HMI
 * PARAMETER:         tVoid.
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclSportsApp::vSportsTimerNotifyTimeOut(tVoid) const
{
	ETG_TRACE_USR4(("--->fc_sxm_tclSportsBase::vSportsTimerNotifyTimeOut() Entered"));
	SXM_ASSERT_RETURN(OSAL_NULL != _poSportsManager)
	/* Notify the collected property Updates to HMI - MIDW will collect all the property Update happened in between 0 to 500 ms and send the collected property Updates on timer expiry */
	_poSportsManager->vNotifyLeagueUpdates();
	ETG_TRACE_USR4(("<---fc_sxm_tclSportsBase::vSportsTimerNotifyTimeOut() Exit"));
}

/*****************************************************************************
 * FUNCTION:          vStartSportsTimer
 * DESCRIPTION:       Start sports timer. After expiry of the timer, a List Status
 *                     Changed would be trigger to HMI
 * PARAMETER:         Time in mSec after which the changed status has to be triggered.
 * RETURNVALUE:       tVoid
 *****************************************************************************/
tVoid fc_sxm_tclSportsApp::vStartSportsTimer(tU32 msecDuration)
{
	ETG_TRACE_USR4(("--->fc_sxm_tclSportsBase::vStartSportsTimer() Entered msecDuration = %u mSec", msecDuration));
	// If sports timer is already active, stop and restart it
	vStopSportsTimer();
	// Starts the sports timer for the given duration
	_mSportsTimer.vStart(msecDuration);
	ETG_TRACE_USR4(("<---fc_sxm_tclSportsBase::vStartSportsTimer() Exit"));
}

tVoid fc_sxm_tclSportsApp::vSetSMSGMTOffSet() const
{
	ETG_TRACE_USR4(("fc_sxm_tclSportsBase::vSetSMSGMTOffSet()"));
	tChar chSign[2];
	uintptr_t u32Min;
	//Read Time ZOne Values form Clock
	OSAL_s32ScanFormat(_strTimeZone.c_str(), "%c%lu", &chSign, &u32Min);

	//Finally End the string with null char
	chSign[1] = '\0';

	//Check if Sign is Negative
	if(OSAL_s32StringCompare(chSign,"-") == 0)
	{
		//Finally set GMT offset in SMS
		OSAL.eTimeSetGMToffset(-((tS32)(u32Min/FC_SXM_TIME_IN_MIN)));
	}
	else
	{
		//Finally set GMT offset in SMS
		OSAL.eTimeSetGMToffset(u32Min/FC_SXM_TIME_IN_MIN);
	}
}

/*
 * Method returns day type based on current system time
 */
tU8 fc_sxm_tclSportsApp::u8GetDayMapIndex(tU32 u32TableParam) const
{
    SXMTime t;
    tU8 u8DayType = SPORTS_TABLE_IX(u32TableParam);

    sxm_time_decode(sxm_time_now(), &t);

    /* need to correct for table offset between 10pm and midnight */
    if((t.hour >= 22) && (u8DayType < 16))
        u8DayType = u8DayType + 1;

    return u8DayType;
}

/*
 * Method notifies App Manager about the Broadscope change status
 */
tVoid fc_sxm_tclSportsApp::vNotifyMetaDataUpdate(tBool bUpdateInProgress) const
{
	ETG_TRACE_USR4(("Notifying Meta Data update to other threads"));
	fc_sxm_trMsgAppMgrSportsMetadataUpdate rMsg;
	rMsg.bMetadataUpdate = bUpdateInProgress;
	fc_sxm_tclAppManager::instance()->vPostMsgNew(rMsg);
}
