/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_pers_sports_data.cpp
* @brief       File to maintain persistency for Sports application
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_pers_data.h"
#include "fc_sxm_tcl_pers_sports_data.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CONFIG
#include "trcGenProj/Header/fc_sxm_tcl_pers_sports_data.cpp.trc.h"
#endif

/* Sports user data class constructor */
trSportsUserData::trSportsUserData() 
{
	ETG_TRACE_USR4(("trSportsUserData() constructor"));

	/* Clear all list data */
	vClearListData();
}

/* Sports user data class destructor */
trSportsUserData::~trSportsUserData()
{
	ETG_TRACE_USR4(("trSportsUserData() destructor"));

	/* Clear all list data */
	vClearListData();
}

/* Clears All Member data */
tVoid trSportsUserData::vClearListData()
{
	tSXMAudio_ProtectedSportList trSportsData = _mProtectedSportList.rGet();
	trSportsData.vSportList.clear();
	_mProtectedSportList.vSet(trSportsData);
	
	tSXMAudio_ProtectedRootList trRootData = _mProtectedRootList.rGet();
	trRootData.vRootList.clear();
	_mProtectedRootList.vSet(trRootData);

	tSXMAudio_ProtectedFavoriteList trFavData = _mProtectedFavoriteList.rGet();
	trFavData.vFavoriteList.clear();
	_mProtectedFavoriteList.vSet(trFavData);
}
/* Show sport list data */
tVoid trSportsUserData::vPrintSportList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vPrintSportList()"));

	vector<trSportListElement>::const_iterator iter;
	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	for(iter=trData.vSportList.begin(); iter!=trData.vSportList.end(); ++iter) {
		ETG_TRACE_USR2(("Sport list: Sport ID - %d, Sport Name - %s", (*iter).u8SportID, (*iter).sSportName.c_str()));
	}
}

/* Clear sport list */
tVoid trSportsUserData::vClearSportList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vClearSportList()"));

	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	trData.vSportList.clear();
	_mProtectedSportList.vSet(trData);
}

/* Update or add new element into sport list */
tVoid trSportsUserData::vSetSportList(tU8 u8SportID, string sSportName)
{
	ETG_TRACE_USR4(("trSportsUserData::vUpdateSportList()"));

	/* Frame data */
	trSportListElement tListElement;
	tListElement.u8SportID  = u8SportID;
	tListElement.sSportName = sSportName;
	/* Get list */
	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	/* Update list */
	trData.vSportList.push_back(tListElement);
	/* Store list */
	_mProtectedSportList.vSet(trData);
}

/* Get sport list size */
tU32 trSportsUserData::u32GetSportListSize(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::u32GetSportListSize()"));

	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	return((tU32)(trData.vSportList.size()));
}

/* Get sport list element for the given index */
tVoid trSportsUserData::vGetSportList(tU32 u32Index, tU8 &u8SportID, string &sSportName)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetSportList()"));
	u8SportID = 255; /* TODO: assign SPORT_INVALID_ID macro here */
	sSportName.clear();
	/* Note: index must start with 0 */
	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	if(u32Index < trData.vSportList.size()) {
		trSportListElement tElementData = trData.vSportList.at(u32Index);
		u8SportID = tElementData.u8SportID;
		sSportName = tElementData.sSportName;
	}
	ETG_TRACE_USR3(("Index = %d : u8SportID = %d, sSportName = %s", u32Index, u8SportID, sSportName.c_str()));
}

/* Show root list data */
tVoid trSportsUserData::vPrintRootList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vPrintRootList()"));
	//Utility method to print the available root affiliates
	vector<trRootListElement>::const_iterator iter;
	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	for(iter=trData.vRootList.begin(); iter!=trData.vRootList.end(); ++iter) {
		ETG_TRACE_USR2(("Root List: Sport ID - %d, Affiliate ID - %d, Affiliate Season - %d,  Affiliate Name - %s", 
			(*iter).u8SportID, (*iter).u32AffiliateID,  (*iter).u8Season, (*iter).sAffiliateName.c_str()));
	}
}

/* Clear root list */
tVoid trSportsUserData::vClearRootList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vClearRootList()"));

	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	trData.vRootList.clear();
	_mProtectedRootList.vSet(trData);
}

/* Update or add new element into root list */
tVoid trSportsUserData::vSetRootList(tU8 u8SportID, tU8 u8Season, tU32 u32AffiliateID, string sAffiliateName)
{
	ETG_TRACE_USR4(("trSportsUserData::vUpdateRootList()"));

	/* Frame data */
	trRootListElement tListElement;
	tListElement.u8SportID = u8SportID;
	tListElement.u32AffiliateID = u32AffiliateID;
	tListElement.sAffiliateName = sAffiliateName;
	tListElement.u8Season = u8Season;
	/* Get list */
	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	/* Update list */
	trData.vRootList.push_back(tListElement);
	/* Set list */
	_mProtectedRootList.vSet(trData);
}

/* Get root list size */
tU32  trSportsUserData::u32GetRootListSize(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::u32GetRootListSize()"));

	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	return((tU32)(trData.vRootList.size()));
}

/* Get root list element for given index */
tVoid trSportsUserData::vGetRootList(tU32 u32Index, tU8 &u8SportID, tU8 &u8Season, tU32 &u32AffiliateID, string &sAffiliateName)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetRootList()"));
	u8SportID = 255;          /* TODO: assign SPORT_INVALID_ID macro here */
	u8Season = 255;           /* TODO: assign SEASON_INVALID_ID */
	u32AffiliateID = 65535;   /* TODO: assign AFFILIATE_INVALID_ID */
	sAffiliateName.clear();
	/* Note: index must start with 0 */
	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	if(u32Index < trData.vRootList.size()) {
		trRootListElement tElementData = trData.vRootList.at(u32Index);
		u8SportID = tElementData.u8SportID;
		u8Season = tElementData.u8Season;
		u32AffiliateID = tElementData.u32AffiliateID;
		sAffiliateName = tElementData.sAffiliateName;
	}
	ETG_TRACE_USR3(("Index = %d : u8SportID = %d, u8Season = %d, u32AffiliateID = %d, sAffiliateName = %s", 
		u32Index, u8SportID, u8Season, u32AffiliateID, sAffiliateName.c_str()));
}

/* Show favorite list data */
tVoid trSportsUserData::vPrintFavoriteList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vPrintFavoriteList()"));

	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	vector<trFavoriteListElement>::const_iterator iter = trData.vFavoriteList.begin();
	for(; iter!=trData.vFavoriteList.end(); ++iter) {
		ETG_TRACE_USR2(("Favorite List: Sport ID - %u, Team ID - %u, League ID - %u, GDRef - %u", iter->u8SportID, iter->u32TeamID, iter->u32LeagueID, iter->u32GDRef));
	}
}

/* Clear favorite list */
tVoid trSportsUserData::vClearFavoriteList(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vClearFavoriteList()"));
	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	trData.vFavoriteList.clear();
	_mProtectedFavoriteList.vSet(trData);
	vSetData();
}

/* Method updates the Team name of a FavoriteListElement.
 * Return value is
 * 		TRUE if the updates is successful; FALSE otherwise
 * */
tBool trSportsUserData::bUpdateTeamName(tU8 u8SportID, tU32 u32TeamID, tU32 u32LeagueID, string& strTeamName,
		string& strTeamAbrName,string& strTeamKnownName,string& strTeamNickName )
{
	ETG_TRACE_USR4(("trSportsUserData::bUpdateTeamName() Entered"));
	tBool bSuccess = FALSE;
	// Get Favorites List
	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	vector<trFavoriteListElement>::iterator iter = trData.vFavoriteList.begin();
	for(; iter!=trData.vFavoriteList.end(); ++iter) {
		if ( (iter->u8SportID == u8SportID) && (iter->u32TeamID == u32TeamID) && (iter->u32LeagueID == u32LeagueID) ) {
			iter->sTeamName.assign(strTeamName);
			iter->sTeamAbrName.assign(strTeamAbrName);
			iter->sTeamKnownName.assign(strTeamKnownName);
			iter->sTeamNickName.assign(strTeamNickName);
			bSuccess = TRUE;
			// Save list
			_mProtectedFavoriteList.vSet(trData);
		    vSetData();
			break;
		}
	}
	ETG_TRACE_USR4(("trSportsUserData::bUpdateTeamName() Exit\tReset Success(%u)", bSuccess));
	return bSuccess;
}

/* Clear favorite list element */
tVoid trSportsUserData::vClearFavoriteListElement(tU8 u8SportID, tU32 u32TeamID, tU32 u32LeagueID)
{
	ETG_TRACE_USR4(("trSportsUserData::vClearFavoriteListElement()"));
	/* Get list */
	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	vector<trFavoriteListElement>::iterator iter;
	for(iter=trData.vFavoriteList.begin(); iter!=trData.vFavoriteList.end(); ++iter) {
		if( ((*iter).u8SportID == u8SportID) && ((*iter).u32TeamID == u32TeamID) && ((*iter).u32LeagueID == u32LeagueID) ) {
			trData.vFavoriteList.erase(iter);
			// Save list 
			_mProtectedFavoriteList.vSet(trData);
		    vSetData();
			break;
		}
	}
}

/* Update or add new element into favorite list */

//tVoid trSportsUserData::vSetFavoriteList(tU8 u8SportID, tU32 u32TeamID, tU32 u32LeagueID, tU32 u32GDRef, string& sTeamName, string& sSportName)
tVoid trSportsUserData::vSetFavoriteList(tU8 u8SportID, tU32 u32TeamID, tU32 u32LeagueID, tU32 u32GDRef, const string& sTeamName,
		const string& sTeamAbrName,const string& sTeamKnownName,const string& sTeamNickName,const string& sSportName)
{
	ETG_TRACE_USR4(("trSportsUserData::vUpdateFavoriteList()"));
	/* Frame data */
	trFavoriteListElement tListElement(u8SportID, u32TeamID, u32LeagueID, u32GDRef, sTeamName,
			sTeamAbrName, sTeamKnownName, sTeamNickName, sSportName);
	/* Get list */
	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	/* Update list */
	trData.vFavoriteList.push_back(tListElement);
	/* Save list */
	_mProtectedFavoriteList.vSet(trData);
    vSetData();
}

/* Get favorite list size */
tU32  trSportsUserData::u32GetFavoriteListSize(tVoid)
{
    tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	ETG_TRACE_USR4(("trSportsUserData::u32GetFavoriteListSize() : %d",(tU32)(trData.vFavoriteList.size())));

	return((tU32)(trData.vFavoriteList.size()));
}

/* Get favorite list element for given index */
tVoid trSportsUserData::vGetFavoriteList(tU32 u32Index, trFavoriteListElement &FavoriteElement)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetFavoriteList()"));
	/* NOTE: index must always start with 0 */
	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	if(u32Index < trData.vFavoriteList.size()) {
		FavoriteElement = trData.vFavoriteList[u32Index];
	}
	ETG_TRACE_USR3(("Index = %d : u8SportID = %d, u32TeamID = %d, u32LeagueID = %d, u32GDRef = %d", u32Index,
					            FavoriteElement.u8SportID, FavoriteElement.u32TeamID, FavoriteElement.u32LeagueID, FavoriteElement.u32GDRef));
	ETG_TRACE_USR3(("sTeamName - %s", FavoriteElement.sTeamName.c_str()));
	ETG_TRACE_USR3(("sTeamAbrName - %s", FavoriteElement.sTeamAbrName.c_str()));
	ETG_TRACE_USR3(("sTeamKnownName - %s", FavoriteElement.sTeamKnownName.c_str()));
	ETG_TRACE_USR3(("sTeamNickName - %s", FavoriteElement.sTeamNickName.c_str()));
	ETG_TRACE_USR3(("sSportName - %s", FavoriteElement.sSportName.c_str()));
}

/* Get data */
tVoid trSportsUserData::vGetData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetData()"));

	vGetSportListData();     /* Sport list */
	vGetRootListData();      /* Root list */
	vGetFavoriteListData();  /* Favorite list */
}

tVoid trSportsUserData::vClearSportData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vClearSportData()"));

	vClearSportList();
	vClearRootList();
	vClearFavoriteList();
}

/* Set data */
tVoid trSportsUserData::vSetData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vSetData()"));

	vSetSportListData();     /* Sport list */
	vSetRootListData();      /* Root list */
	vSetFavoriteListData();  /* Favorite list */

}


/* Get data from persistence memory and store it in sport list */
tVoid trSportsUserData::vGetSportListData()
{
	ETG_TRACE_USR4(("trSportsUserData::vGetSportListData()"));

	tU32 u32SportListSize = 0;
	ETG_TRACE_USR4(("trSportsUserData::vGetSportListData() u32SportListSize=%d",u32SportListSize));
	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	//Before writing,Clear of our old internal data if any
	trData.vSportList.clear();

	fc_sxm_tclPersData::instance()->vGetDPSportsList(trData);

    _mProtectedSportList.vSet(trData);
}

/* Set sport list data into binary data to persist across power cycle */
tVoid trSportsUserData::vSetSportListData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vSetSportListData()"));

	tSXMAudio_ProtectedSportList trData = _mProtectedSportList.rGet();
	tU32 u32SportListSize = (tU32)(trData.vSportList.size());
	ETG_TRACE_USR4(("trSportsUserData::vSetSportListData() u32SportListSize=%d",u32SportListSize));

	fc_sxm_tclPersData::instance()->vSetDPSportsList(trData);

	//Finally Clear of our old internal data
	trData.vSportList.clear();
}

/* Get data from persistence memory and store it in root list */
tVoid trSportsUserData::vGetRootListData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetRootListData()"));

	tU32 u32RootListSize = 0;
	ETG_TRACE_USR4(("trSportsUserData::vGetRootListData() u32RootListSize=%d",u32RootListSize));
	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	//Before writing,Clear of our old internal data if any
	trData.vRootList.clear();

	fc_sxm_tclPersData::instance()->vGetDPRootList(trData);

   _mProtectedRootList.vSet(trData);
}

/* Set root list data into binary data to persist across power cycle */
tVoid trSportsUserData::vSetRootListData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vSetRootListData()"));

	tSXMAudio_ProtectedRootList trData = _mProtectedRootList.rGet();
	tU32 u32RootListSize = (tU32)(trData.vRootList.size());
	ETG_TRACE_USR4(("trSportsUserData::vSetRootListData() u32RootListSize=%d",u32RootListSize));

	 for(tU32 u32Count=0; u32Count<u32RootListSize; u32Count++) {
         ETG_TRACE_USR4(("fc_sxm_tclPersData::vSetRootListData() trData.vRootList[%d](u8SportID, u32AffiliateID, u8Season, sAffiliateName) = (%d, %d, %d, %s)",
                 u32Count,
                 trData.vRootList[u32Count].u8SportID,
                 trData.vRootList[u32Count].u32AffiliateID,
                 trData.vRootList[u32Count].u8Season,
                 trData.vRootList[u32Count].sAffiliateName.c_str()));
	 }

	fc_sxm_tclPersData::instance()->vSetDPRootList(trData);

	//Finally Clear of our old internal data
	trData.vRootList.clear();
}

/* Get data from persistence memory and store it in favorite list */
tVoid trSportsUserData::vGetFavoriteListData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vGetFavoriteListData()"));

	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	//Before writing,Clear of our old internal data if any
	trData.vFavoriteList.clear();

	fc_sxm_tclPersData::instance()->vGetDPFavoriteList(trData);

    _mProtectedFavoriteList.vSet(trData);
}

/* Set favorite list data into binary data to persist across power cycle */
tVoid trSportsUserData::vSetFavoriteListData(tVoid)
{
	ETG_TRACE_USR4(("trSportsUserData::vSetFavoriteListData()"));

	tSXMAudio_ProtectedFavoriteList trData = _mProtectedFavoriteList.rGet();
	tU32 u32FavoriteListSize = (tU32)(trData.vFavoriteList.size());
	ETG_TRACE_USR4(("trSportsUserData::vSetFavoriteListData() u32FavoriteListSize=%d",u32FavoriteListSize));

	fc_sxm_tclPersData::instance()->vSetDPFavoriteList(trData);

	//Finally Clear of our old internal data
	trData.vFavoriteList.clear();
}
