/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_phonetics_channels.cpp
* @brief       Implementation for functionalities related to handling Phonetics Data.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_phonetics_channels.h"
#include "fc_sxm_tcl_phonetics_proxy.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PHONETICS_CHANNELS
#include "trcGenProj/Header/fc_sxm_tcl_phonetics_channels.cpp.trc.h"
#endif

// Constructor
fc_sxm_tcl_phonetics_channels::fc_sxm_tcl_phonetics_channels(Ifc_sxm_tclPhoneticsProxy *pIPhoneticsProxy):fc_sxm_tclPhoneticsBase(pIPhoneticsProxy)
{
    ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channel_data constructor"));
}

fc_sxm_tcl_phonetics_channels::~fc_sxm_tcl_phonetics_channels()
{
    ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channel_data destructor"));
    _poPhoneticsProxy = OSAL_NULL;
}


/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vGetPhoneticsData                                             *
 *                                                                                 *
 *DESCRIPTION:       get Phonetics data for the specified file type                *
 *                                                                                 *
 *PARAMETERS:        midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult   *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tcl_phonetics_channels::vGetPhoneticsData(midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult &ofiTxObj)
{
    ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vGetPhoneticsData"));

    tU8 nPhonetype = (ofiTxObj).PhoneticsType;

    if(nPhonetype & midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_CHANNELS)
    {
        ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vGetPhoneticsData:Channels Phonetics type"));
        vGetChannelsPhoneticsData((ofiTxObj).LanguageType, (ofiTxObj).ChannelList);
    }
    else if(nPhonetype == midw_ext_fi_tcl_e8PhoneticsType::FI_EN_PHONETIC_TYPE_UNKNOWN)
    {
        ETG_TRACE_ERR(("fc_sxm_tcl_phonetics_channels::vGetPhoneticsData:Unknown Phonetics type"));
    }
    else
    {
        ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vGetPhoneticsData:Phonetics type not supported"));
    }
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vGetChannelsPhoneticsData                                     *
 *                                                                                 *
 *DESCRIPTION:       Set path to read Phonetics data for the specified language    *
 *                                                                                 *
 *PARAMETERS:        midw_ext_sxm_phoneticsfi_tclMsgGetPhoneticsDataMethodResult   *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tVoid fc_sxm_tcl_phonetics_channels::vGetChannelsPhoneticsData(tU8 nLangtype,
        vector<midw_ext_fi_tcl_ChannelPhoneticsInfo> &vChannelList)
{
    ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vGetChannelsPhoneticsData"));

    //clearing the list
    vChannelList.clear();

    if(nLangtype & midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_ENGLISH)
    {
        ETG_TRACE_USR4(("vGetChannelsPhoneticsData:English Language type"));
        bReadChannelsphoneticsData(FC_SXM_DEFAULT_SMS_PHONETICS_REC_ENG_PATH, vChannelList);
    }
    if(nLangtype & midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_SPANISH)
    {
        ETG_TRACE_USR4(("vGetChannelsPhoneticsData:Spanish Language type"));
        bReadChannelsphoneticsData(FC_SXM_DEFAULT_SMS_PHONETICS_REC_ESP_PATH, vChannelList);
    }
    if(nLangtype & midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_FRENCH)
    {
        ETG_TRACE_USR4(("vGetChannelsPhoneticsData:French Language type"));
        bReadChannelsphoneticsData(FC_SXM_DEFAULT_SMS_PHONETICS_REC_FR_PATH, vChannelList);
    }
    if(nLangtype == midw_ext_fi_tcl_e8LanguageType::FI_EN_PHONETIC_LANG_TYPE_UNKNOWN)
    {
        ETG_TRACE_ERR(("vGetChannelsPhoneticsData:Unknown Language type"));
    }
}

/***********************************************************************************
 *                                                                                 *
 *FUNCTION:          vReadChannelphoneticsData                                     *
 *                                                                                 *
 *DESCRIPTION:       Reading the Channel Phonetics data for the specified language *
 *                                                                                 *
 *PARAMETERS:        Char Pinter & vector<midw_ext_fi_tcl_ChannelPhoneticsInfo>    *
 *                                                                                 *
 *RETURNVALUES:      None                                                          *
 *                                                                                 *
 **********************************************************************************/
tBool fc_sxm_tcl_phonetics_channels::bReadChannelsphoneticsData(const char *pacDictionaryPath,
        vector<midw_ext_fi_tcl_ChannelPhoneticsInfo> &vPhoneticsData)
{
    ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vReadChannelphoneticsData"));

    if(_poPhoneticsProxy != NULL)
    {
        SERVICE_ID tServiceId = SERVICE_INVALID_ID;
        tBool bFileComplete = FALSE;
        tBool bOk = FALSE;

        OSAL_FILE_STRUCT *psDictionaryFile = _poPhoneticsProxy->psFopen(pacDictionaryPath, PHONETICS_FILE_MODE);
        if (psDictionaryFile == NULL)
        {
            ETG_TRACE_ERR(("fc_sxm_tcl_phonetics_channels::vReadChannelphoneticsData Invalid psDictionaryFile"));
            // Not a problem
            return FALSE;
        }
        while (bFileComplete == FALSE)
        {

            // Read the id for this line
            tServiceId = _poPhoneticsProxy->hReadServiceIdFromDictionary(psDictionaryFile);
            if (tServiceId == SERVICE_INVALID_ID)
            {
                ETG_TRACE_USR4(("fc_sxm_tcl_phonetics_channels::vReadChannelphoneticsData EOF/Invalid SID"));
                // End of file or error occurred
                break;
            }
            midw_ext_fi_tcl_ChannelPhoneticsInfo  lChannelListEntry;

            //set service Id
            lChannelListEntry.ID = tServiceId;

            // Read phonetics
            bOk = bReadPhoneticEntry(psDictionaryFile, lChannelListEntry.PhoneticsData);

            if (FALSE == bOk)
            {
                ETG_TRACE_ERR(("bReadPhoneticEntry failed"));
                bFileComplete = TRUE;
            }
            else
            {
                vPhoneticsData.push_back(lChannelListEntry);
            }
        }
        _poPhoneticsProxy->iFclose(psDictionaryFile);
        return TRUE;
    }
    else
    {
        ETG_TRACE_ERR(("fc_sxm_tcl_phonetics_channels::vReadChannelphoneticsData Invalid _poPhoneticsProxy"));
        return FALSE;
    }
}

