/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_neardest.cpp
* @brief       Class to handle Near Destination Parking Data
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_parking_neardest.h"
#include "fc_sxm_tcl_parking_sdk_proxy_if.h"
#include "fc_sxm_tcl_parking_app_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_NEARDEST
#include "trcGenProj/Header/fc_sxm_tcl_parking_neardest.cpp.trc.h"
#endif

/****************************************************************************
 *
 *FUNCTION:          fc_sxm_tclParkingNearDest
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingNearDest::fc_sxm_tclParkingNearDest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearDest constructor"));

    _u32ListID =(tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION;
    _u16FilterMask = FC_SXM_PARKING_DEST_FILTER_MASK;
}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_tclParkingNearDest
 *
 *DESCRIPTION:       Destructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingNearDest::~fc_sxm_tclParkingNearDest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearDest destructor"));
}
/****************************************************************************
 *
 *FUNCTION:          s32RefreshPOIList
 *
 *DESCRIPTION:       To get POI list for new MBR and refresh internal list
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      error code
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingNearDest::s32RefreshPOIList()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearDest::s32RefreshPOIList"));

    //first clear internal POI list
    _mapPOIList.clear();

    tS32 s32ParkingStatus = SXM_E_OK;
    SXMParkingLocation rParkingLocation;
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)

    //begin parking request to fetch parking location
    //fitermask would be 0
    if ((s32ParkingStatus = _poProxy->s32POIBeginExtraction(&_rSXMMbr,0)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-begin: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,(tS32)SXM_E_ERROR)

    //create a set to sort POI list based on POI distance
    set<trDestPOIs,trComparedata> setSortedNearDestList;

    //iterate through parking locations
    while((s32ParkingStatus = _poProxy->s32POIExtractParkingLocation(&rParkingLocation)) == SXM_E_OK)
    {
        ETG_TRACE_USR4(("name:%s",rParkingLocation.name));
        ETG_TRACE_USR4(("R-user flag:%d,",rParkingLocation.user_flag));

        trDestPOIs rDestPOI;

        //prepare POI details we are interested in
        trPOIData rPOIData;
        rPOIData.u16Userflag = (rParkingLocation.user_flag | FC_SXM_PARKING_DEST_FILTER_MASK);
        rDestPOI.rTempPOIData = rPOIData;
        rDestPOI.u32ParkingID = (tU32)rParkingLocation.id;

        //Calculate distance from parking location to destination position
      	rDestPOI.f32Distance = _poApp->f32GetDistanceFromDestination(rParkingLocation.lat,rParkingLocation.lon);

      	ETG_TRACE_USR4(("s32RefreshPOIList1 u32ParkingID=%d,f32Distance=%f",rDestPOI.u32ParkingID,rDestPOI.f32Distance));

      	//Sort the complete POI list based on POI distance
        setSortedNearDestList.insert(rDestPOI);
    }

    //Preapre iterator
    set<trDestPOIs,trComparedata>::const_iterator iter = setSortedNearDestList.begin();

    //Iterate Sorted POI list
    for(tU32 u32Index=0; iter != setSortedNearDestList.end(); ++iter,++u32Index)
    {
    	//We are not interted in all POIs found for Destination MBR
    	//interested only till Max. Capacity closer to destination
    	if(u32Index <= _poApp->u32GetCapacity(u32GetListID()))
    	{
    		//finally copy to internal list
    		_mapPOIList.insert(pair<tU32,trPOIData>(iter->u32ParkingID,iter->rTempPOIData));
    	}
    	//if we are reached capacity,just break from loop
    	else
    		break;
    }

    //Check if no locations available
    if(s32ParkingStatus == SXM_E_NOENT)
        ETG_TRACE_USR4(("No more locations in the list"));

    //finally end the parking request.
    if ((s32ParkingStatus = _poProxy->s32POIEndExtraction()) != 0) {
        ETG_TRACE_ERR(("***parking-end: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    return s32ParkingStatus;
}
/****************************************************************************
 *
 *FUNCTION:          vClearPPOIUserFlag
 *
 *DESCRIPTION:       to clear Parking user flag for all destination PPOI's
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingNearDest::vClearPPOIUserFlag()
{
    vector<PPOISpec> vecPOIList;

    SXM_FOREACH_MAP(tU32, trPOIData, Iter, _mapPOIList)
    {
        PPOISpec rPOISpec;
        rPOISpec.ppoi_id = Iter->first;

        //OK,just clear user flag
        ETG_TRACE_USR4(("Clearing PPOI bit for ParkingID=%d,old flag=%d",Iter->first,Iter->second.u16Userflag));
        rPOISpec.type_flags = (byte)(Iter->second.u16Userflag & FC_SXM_PARKING_NO_FILTER);
        ETG_TRACE_USR4(("Cleared PPOI bit for ParkingID=%d,new flag=%d",Iter->first,rPOISpec.type_flags));

        vecPOIList.push_back(rPOISpec);

    }
    s32SetPPOIList(vecPOIList);
}
