/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_nearby.cpp
* @brief       Class to handle Near By Parking Data
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_parking_nearby.h"
#include "fc_sxm_tcl_parking_fi_msgs.h"
#include "fc_sxm_tcl_parking_sdk_proxy_if.h"
#include "fc_sxm_tcl_parking_app_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_NEARBY
#include "trcGenProj/Header/fc_sxm_tcl_parking_nearby.cpp.trc.h"
#endif

tVoid cb_vParkingRequestCallback(ptr handle,ptr pArg)
{
    ETG_TRACE_USR4(("cb_vParkingRequestCallback handle=%x",handle));

    fc_sxm_tclIParkingApp* poApp = static_cast<fc_sxm_tclIParkingApp*>(pArg);

    fc_sxm_trMsgParkingSMSeRequestCallback rMsg;
    if(poApp != OSAL_NULL)
    {
        poApp->vPostInternalMsg(rMsg);
    }

}
/****************************************************************************
 *
 *FUNCTION:          fc_sxm_tclParkingNearBy
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingNearBy::fc_sxm_tclParkingNearBy()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearBy constructor"));

    _u32ListID = (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY;

}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_tclParkingNearBy
 *
 *DESCRIPTION:       Destructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingNearBy::~fc_sxm_tclParkingNearBy()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearBy destructor"));
}
/****************************************************************************
 *
 *FUNCTION:          vStartRequest
 *
 *DESCRIPTION:       Method to start SMSe service
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingNearBy::vStartRequest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearBy::vStartRequest"));

    SXM_ASSERT_RETURN(OSAL_NULL != _poProxy)

    //Method invokes SMSe to start Request for current position
    tS32 s32ParkingStatus = _poProxy->s32StartRequest(
            &_rSXMMbr,
            &_rSXMPoint,
            cb_vParkingRequestCallback,
            _poApp,
            &_pReqHandle
    );

    //Check If request is already active
    if(s32ParkingStatus == (tS32)SXM_E_BUSY && _pReqHandle != OSAL_NULL)
    {
        ETG_TRACE_USR4(("Request already Active,Modifying existing request..."));

        //Modify the existing request to new position
        s32ParkingStatus = _poProxy->s32ModifyRequest(
                _pReqHandle,
                &_rSXMMbr,
                &_rSXMPoint
        );
    }
    else if (s32ParkingStatus != (tS32) SXM_E_OK )
    {
        ETG_TRACE_ERR(("Error=%d in creating Parking Request",s32ParkingStatus));
    }

}
/****************************************************************************
 *
 *FUNCTION:          vRemoveRequest
 *
 *DESCRIPTION:       Method to Remove SMSe request created already
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingNearBy::vRemoveRequest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingNearBy::vRemoveRequest"));

    SXM_ASSERT_RETURN(OSAL_NULL != _poProxy)

    if (OSAL_NULL != _pReqHandle)
    {
    	//remove request
		tS32 s32ParkingStatus =  _poProxy->s32RemoveCollection(_pReqHandle);

		//Check if it is success
		if(s32ParkingStatus == (tS32) SXM_E_OK)
		{
			ETG_TRACE_USR4(("Active request removed..."));
			_pReqHandle = OSAL_NULL;
		}
		else
		{
			ETG_TRACE_ERR(("request remove failed:%d",s32ParkingStatus));
		}
    }
}
/****************************************************************************
 *
 *FUNCTION:          s32GetParkingList
 *
 *DESCRIPTION:       To get full parking List
 *
 *PARAMETERS:       midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingNearBy::s32GetParkingList(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult &oMRes)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::s32GetParkingList"));

    tS32 s32ParkingStatus = SXM_E_OK;
    SXMParkingLocation rParkingLocation;
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,(tS32)SXM_E_ERROR)

    //begin parking request to fetch parking location
    if ((s32ParkingStatus = _poProxy->s32BeginExtraction(_pReqHandle, OSAL_NULL)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-begin: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    tU32 u32Count =0;
    //iterate till all the parking locations are fetched
    while((s32ParkingStatus = _poProxy->s32ExtractParkingLocation(_pReqHandle, &rParkingLocation)) == SXM_E_OK)
    {
    	ETG_TRACE_USR4(("name:%s",rParkingLocation.name));

    	//Ignoring empty name.
    	if(rParkingLocation.name[0] != '\0')
    	{
    		//Check if this location to be Send
    		if(bIsFilterLocation(rParkingLocation))
    		{
    			//We are interested only till Max Capacity
    			if(u32Count < _poApp->u32GetCapacity(u32GetListID()))
    			{
    				midw_ext_fi_tcl_ParkingInfo rFIParkingInfo;

    				//copy this location to method result
    				vCopyLocationToFI(rParkingLocation,rFIParkingInfo);

    				//finally copy to method result
    				oMRes.ParkingInfoList.push_back(rFIParkingInfo);

    				++u32Count;
    			}
    			//don't continue further
    			else
    				break;
    		}
    	}
    }

    //Check if no locations available
    if(s32ParkingStatus == SXM_E_NOENT)
        ETG_TRACE_USR4(("No more locations in the list"));

    //finally end the parking request.
    if ((s32ParkingStatus = _poProxy->s32EndExtraction(_pReqHandle)) != 0) {
        ETG_TRACE_ERR(("***parking-end: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    return s32ParkingStatus;
}
/****************************************************************************
 *
 *FUNCTION:          s32GetParkingInfo
 *
 *DESCRIPTION:       to get particular parking location
 *
 *PARAMETERS:       u32ParkingID,midw_ext_fi_tcl_ParkingInfo
 *
 *RETURNVALUES:      tS32
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingNearBy::s32GetParkingInfo(tU32 u32ParkingID,midw_ext_fi_tcl_ParkingInfo &rFIParkingInfo)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::s32GetParkingInfo u32ParkingID=%d",u32ParkingID));

    tS32 s32ParkingStatus = SXM_E_OK;
    SXMParkingLocation rParkingLocation;
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)

    //begin parking request to fetch parking location
    if ((s32ParkingStatus = _poProxy->s32BeginExtraction(_pReqHandle, OSAL_NULL)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-begin: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    //iterate till all the parking locations are fetched
    while((s32ParkingStatus = _poProxy->s32ExtractParkingLocation(_pReqHandle, &rParkingLocation)) == SXM_E_OK)
    {
        //Check if ParkingID matches
        if(u32ParkingID == static_cast<tU32>(rParkingLocation.id))
        {
            //copy this location to method result
            vCopyLocationToFI(rParkingLocation,rFIParkingInfo);
        }
    }

    //Check if no locations available
    if(s32ParkingStatus == SXM_E_NOENT)
        ETG_TRACE_USR4(("No more locations in the list"));

    //finally end the parking request.
    if ((s32ParkingStatus = _poProxy->s32EndExtraction(_pReqHandle)) != 0) {
        ETG_TRACE_ERR(("***parking-end: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    return s32ParkingStatus;
}
