/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_parking_base.cpp
* @brief       base class for parking list(nearby,NearDest & Favorites)
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_parking_fi.h"
#include "fc_sxm_tcl_parking_base.h"
#include "fc_sxm_tcl_parking_sdk_proxy_if.h"
#include "fc_sxm_tcl_parking_app_if.h"
#include "fc_sxm_generic_utils.h"
#include "fc_sxm_tcl_parking_fi_msgs.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"
#include "fc_sxm_trace_macros.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_PARKING_BASE
#include "trcGenProj/Header/fc_sxm_tcl_parking_base.cpp.trc.h"
#endif

const tU16 arFullnessValue[] =
{
        0, 10, 20, 30, 40, 50, 55, 60, 65, 70, 75, 80, 85, 90, 95, 100
};

#define FC_SXM_PARKNG_PRICING_MAX_HOURS 24
#define FC_SXM_PARKING_CENTS_PER_DOLLAR 100.00
#define FC_SXM_PARKING_NO_OF_DAYS_IN_WEEK 7


tVoid cb_vParkingPOICallback(ptr usercx,byte filtermask)
{
    ETG_TRACE_USR4(("cb_vParkingPOICallback usercx=%x filtermask=%d",usercx,filtermask));

    fc_sxm_tclIParkingApp* poApp = static_cast<fc_sxm_tclIParkingApp*>(usercx);

    fc_sxm_trMsgParkingSMSePOICallback rMsg;
    rMsg.u32FilterMask = static_cast<tU32>(filtermask);

    if(poApp != OSAL_NULL)
    {
        poApp->vPostInternalMsg(rMsg);
    }

}

/****************************************************************************
 *
 *FUNCTION:          fc_sxm_tclParkingBase
 *
 *DESCRIPTION:       Constructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingBase::fc_sxm_tclParkingBase()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase constructor"));

    //Clear class member data
    vClearMemberData();

}
/****************************************************************************
 *
 *FUNCTION:          ~fc_sxm_tclParkingBase
 *
 *DESCRIPTION:       Destructor
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
fc_sxm_tclParkingBase::~fc_sxm_tclParkingBase()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase destructor"));

    //Clear class member data
    vClearMemberData();
}
/****************************************************************************
 *
 *FUNCTION:          vClearMemberData
 *
 *DESCRIPTION:       to clear member variable
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vClearMemberData()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase vClearMemberData"));

    //default SXM point
    _rSXMPoint.lon = 0;
    _rSXMPoint.lat = 0;

    //default MBR
    _rSXMMbr.ll.lon = 0;
    _rSXMMbr.ll.lat = 0;
    _rSXMMbr.ur.lon = 0;
    _rSXMMbr.ur.lat = 0;

    //initialize handle
    _pReqHandle = OSAL_NULL;

    _poApp = OSAL_NULL;
    _poProxy = OSAL_NULL;

    _u16FilterMask = FC_SXM_PARKING_NO_FILTER;

    _u32ListID = (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_CUSTOM;

    _trListMode.enType = midw_ext_fi_tcl_e8_SxmListMode::FI_EN_RELEASE;

}
/*
 * Method sets app object to be the observer for parking objects
 */
tVoid fc_sxm_tclParkingBase::vSetApp(fc_sxm_tclIParkingApp *poParkingObserver,fc_sxm_tclIParkingSdkProxy *poParkingSdkProxy)
{
    _poApp = poParkingObserver;
    _poProxy = poParkingSdkProxy;
}
/****************************************************************************
 *
 *FUNCTION:          vSetPosition
 *
 *DESCRIPTION:       function to set position
 *
 *PARAMETERS:		lattitude,longitude
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vSetPosition(tS32 s32Lon,tS32 s32Lat)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase vSetPosition"));

    //convert Navi Position to float values
    _rSXMPoint.lon = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(s32Lon);
    _rSXMPoint.lat = fc_sxm_tclGenericUtils::f32DegreeNavToFloat(s32Lat);

    ETG_TRACE_USR4(("_rSXMPoint.lon=%f,_rSXMPoint.lat=%f",_rSXMPoint.lon,_rSXMPoint.lat));

    //Set or move MBR as per new location
    vPrepareMBR();

}
/****************************************************************************
 *
 *FUNCTION:          vPrepareMBR
 *
 *DESCRIPTION:       function to set MBR around new position
 *
 *PARAMETERS:		NOne
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vPrepareMBR()
{
    SXM_ASSERT_RETURN(OSAL_NULL != _poApp)
    tF32 f32RadiusDelta = _poApp->f32GetRadius(u32GetListID());

    _rSXMMbr.ll.lon = static_cast<tF32>(_rSXMPoint.lon - f32RadiusDelta);
    _rSXMMbr.ll.lat = static_cast<tF32>(_rSXMPoint.lat - f32RadiusDelta);
    _rSXMMbr.ur.lon = static_cast<tF32>(_rSXMPoint.lon + f32RadiusDelta);
    _rSXMMbr.ur.lat = static_cast<tF32>(_rSXMPoint.lat + f32RadiusDelta);

}
/****************************************************************************
 *
 *FUNCTION:          vSetListMode
 *
 *DESCRIPTION:       function to set list mode
 *
 *PARAMETERS:		 Mode
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vSetListMode(const midw_ext_fi_tcl_e8_SxmListMode& trListMode)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase vSetListMode=%d",trListMode.enType));

    SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

    if( _trListMode == trListMode )
    {
    	ETG_TRACE_USR4(("List Mode is Same..."));

    	return;
    }

    _trListMode = trListMode;

	tU32 u32ListID = u32GetListID();

    if(trListMode.enType == (midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE))
    {
        //and send the property update
    	_poApp->vSendUpdate(u32ListID);

    	//Start the Timer
    	_poApp->vStartListTimer(u32ListID);

    	//Save Current Device position
    	//which can be used for destination list to sort
    	_poApp->vFreezeDevicePosition();
    }
    else if(trListMode.enType == (midw_ext_fi_tcl_e8_SxmListMode::FI_EN_RELEASE))
    {
    	_poApp->vStopListTimer();

    	//for Nearby Case Resume sending position updates to SDK
    	if(u32GetListID() == static_cast<tU32>(midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY))
    	{
    		_poApp->vReleaseDevicePosition();
    	}
    	//for Dest & Favorites, Update List state
    	else
    	{
            //and send the property update
        	_poApp->vSendUpdate(u32ListID);
    	}
    }
}
/****************************************************************************
 *
 *FUNCTION:          vStartRequest
 *
 *DESCRIPTION:       Method to start SMSe service
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vStartRequest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::vStartRequest"));

}
/****************************************************************************
 *
 *FUNCTION:          vRemoveRequest
 *
 *DESCRIPTION:       Method to Remove SMSe request created already
 *
 *PARAMETERS:		None
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vRemoveRequest()
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::vRemoveRequest"));

}
/****************************************************************************
 *
 *FUNCTION:          s32GetParkingList
 *
 *DESCRIPTION:       To get full parking List
 *
 *PARAMETERS:       midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingBase::s32GetParkingList(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult &oMRes)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::s32GetParkingList FilterMask=%d",_u16FilterMask));

    tS32 s32ParkingStatus = SXM_E_OK;
    SXMParkingLocation rParkingLocation;
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,(tS32)SXM_E_ERROR)

    //begin parking request to fetch parking location
    if ((s32ParkingStatus = _poProxy->s32POIBeginExtraction(OSAL_NULL,_u16FilterMask)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-begin: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    tU32 u32Count =0;
    tBool bFilterLocation = FALSE;
    //iterate till all the parking locations are fetched
    while((s32ParkingStatus = _poProxy->s32POIExtractParkingLocation(&rParkingLocation)) == SXM_E_OK)
    {
    	ETG_TRACE_USR4(("name:%s,",rParkingLocation.name));

    	//Ignoring empty name.
    	if(rParkingLocation.name[0] != '\0') 
    	{
    		//dont apply filter settings for favorite use case
    		bFilterLocation = (u32GetListID() == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE)? TRUE : bIsFilterLocation(rParkingLocation);

    		//Check if this location to be Send
    		if(bFilterLocation)
    		{
    			//We are interested only till Max Capacity
    			if(u32Count < _poApp->u32GetCapacity(u32GetListID()))
    			{
    				midw_ext_fi_tcl_ParkingInfo rFIParkingInfo;

    				//copy this location to method result
    				vCopyLocationToFI(rParkingLocation,rFIParkingInfo);

    				//finally copy to method result
    				oMRes.ParkingInfoList.push_back(rFIParkingInfo);

    				++u32Count;
    			}
    			//don't continue further.
    			else
    				break;
    		}
    	}
    }

    //Check if no locations available
    if(s32ParkingStatus == SXM_E_NOENT)
        ETG_TRACE_USR4(("No more locations in the list"));

    //finally end the parking request.
    if ((s32ParkingStatus = _poProxy->s32POIEndExtraction()) != 0) {
        ETG_TRACE_ERR(("***parking-end: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    return s32ParkingStatus;

}
/****************************************************************************
 *
 *FUNCTION:          s32GetParkingInfo
 *
 *DESCRIPTION:       to get particular parking location
 *
 *PARAMETERS:       u32ParkingID,midw_ext_fi_tcl_ParkingInfo
 *
 *RETURNVALUES:      tS32
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingBase::s32GetParkingInfo(tU32 u32ParkingID,midw_ext_fi_tcl_ParkingInfo &rFIParkingInfo)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::s32GetParkingInfo u32ParkingID=%d",u32ParkingID));

    tS32 s32ParkingStatus = SXM_E_OK;
    SXMParkingLocation rParkingLocation;
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)

    //begin parking request to fetch parking location
    if ((s32ParkingStatus = _poProxy->s32POIBeginExtraction(OSAL_NULL,_u16FilterMask)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-begin: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    //iterate till all the parking locations are fetched
    while((s32ParkingStatus = _poProxy->s32POIExtractParkingLocation(&rParkingLocation)) == SXM_E_OK)
    {
        //Check if ParkingID matches
        if(u32ParkingID == static_cast<tU32>(rParkingLocation.id))
        {
            //copy this location to method result
            vCopyLocationToFI(rParkingLocation,rFIParkingInfo);
        }
    }

    //Check if no locations available
    if(s32ParkingStatus == SXM_E_NOENT)
        ETG_TRACE_USR4(("No more locations in the list"));

    //finally end the parking request.
    if ((s32ParkingStatus = _poProxy->s32POIEndExtraction()) != 0) {
        ETG_TRACE_ERR(("***parking-end: error code %d", s32ParkingStatus));
        return s32ParkingStatus;
    }

    return s32ParkingStatus;
}


/****************************************************************************
 *
 *FUNCTION:          s32SetPPOIList()
 *
 *DESCRIPTION:       to set new POI list as PPOI's
 *
 *PARAMETERS:        None
 *
 *RETURNVALUES:      error code
 *
 ******************************************************************************/
tS32 fc_sxm_tclParkingBase::s32SetPPOIList(vector<PPOISpec>& vecCombinedPOIList)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::s32SetPPOIList POI list size=%d",vecCombinedPOIList.size()));

    tS32 s32ParkingStatus = SXM_E_OK;

    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poProxy,(tS32)SXM_E_ERROR)
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,(tS32)SXM_E_ERROR)

    //Set desired POI list as PPOI list
    if ((s32ParkingStatus = _poProxy->s32SetPPOIList((tU32)(vecCombinedPOIList.size()),vecCombinedPOIList.data(),cb_vParkingPOICallback,_poApp)) != SXM_E_OK)
    {
        ETG_TRACE_ERR(("***parking-s32SetPPOIList: error code %d", s32ParkingStatus));
    }

    return s32ParkingStatus;
}

/****************************************************************************
 *
 *FUNCTION:          bCheckFilterConfig
 *
 *DESCRIPTION:       Check particular Parking location is to be filtered as per client spec
 *
 *PARAMETERS:        SXMParkingLocation - particular parking location
 *
 *RETURNVALUES:      TRUE - Location should not be filtered
 *RETURNVALUES:      FALSE - Location Should be filtered
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingBase::bIsFilterLocation(SXMParkingLocation &rLocation)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::bIsFilterLocation"));

    midw_ext_fi_tcl_AmenityInfo Amen;
    midw_ext_fi_tcl_ParkingInfo rParkingInfo;

    //Copy Amenities to FI type
    vCopyAmenities(rLocation.amen,Amen);

    //Get Primary Height info from SMS
    tU32 u32Height = static_cast<tU32>(rLocation.height);

    //Copy Opening Times(used for Open Status)
    vCopyOpeningTimes(rLocation,rParkingInfo);

    //Some times,Open Status is available from broardcast
    if(rLocation.valid & SXM_PARKING_VALID_STATUS)
    {
        //If So,Just Copy the value from broadcast
        rParkingInfo.OpenStatus.enType = static_cast<midw_ext_fi_tcl_e8OpenStatus::tenType>(rLocation.status);
    }
    //If not available in broadcast
    //We should calculate open status form Opening Times
    else
    {
        //Copy the Open status based on Opening Times
        rParkingInfo.OpenStatus.enType = enGetOpenStatus(rParkingInfo);
    }

/*    //Set Confi for Parking location from SMS
    rSMSParkingLocation.vSetFilterConfig(Amen.Cash,Amen.Card,Amen.SelfParking,Amen.ParkingType,
            enOpenStatus,Amen.SUVCharges,u32Height);*/

    return _FilterConfig.bIsParkingStationAllowed(Amen,rParkingInfo.OpenStatus,u32Height);

}
/****************************************************************************
 *
 *FUNCTION:          u8GetOpenStatus
 *
 *DESCRIPTION:       Util function to prepare Parking Location Current opening status
 *
 *PARAMETERS:        rLocation - Parking Location from SMS
 *
 *RETURNVALUES:      midw_ext_fi_tcl_e8OpenStatus - Current Opening status
 *
 ******************************************************************************/
midw_ext_fi_tcl_e8OpenStatus::tenType fc_sxm_tclParkingBase::enGetOpenStatus(midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::bGetOpenStatus"));

    //First Check if Opening Times are Empty
    if(rParkingInfo.OpeningTimes.size() == 0)
    {
        //Then we don't have any info from broadcast
        return midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_UNKNOWN;
    }

    tBool bIsMorethanStartTime = FALSE;
    tBool bIsLessThanEndTime = FALSE;
    tBool bIsOpen = FALSE;
    tBool bIsClosed = FALSE;

    //Check If poApp is NULL
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_UNKNOWN)

    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bGetOpenStatus current Day=%d",_poApp->u8GetCurentWeekDay()));

    //loop through for each day of the week.
    for(tU8 u8DayType = 0; u8DayType < rParkingInfo.OpeningTimes.size(); ++u8DayType)
    {
        //We are interested only for current day
        if(_poApp->u8GetCurentWeekDay() == rParkingInfo.OpeningTimes[u8DayType].DayType.enType)
        {
            //per a Day ,we have set of opening slots
            for(tU8 u8PerDayTime = 0; u8PerDayTime < rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes.size(); ++u8PerDayTime)
            {

                //Check if Parking is Open for 24 Hours
                bIsOpen = bCheckOpen24Hours(
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Minutes,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Minutes
                        );

                //Check If Parking location is Closed for full Day
                bIsClosed= bCheckClosedStatus(
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Minutes,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Minutes
                        );

                //Check if Current User Hour is more than start time
                bIsMorethanStartTime = bCheckStartTime(
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Minutes
                        );

                //Check if Current User Hour is less than End time
                bIsLessThanEndTime = bCheckEndTime(
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].StartTime.Minutes,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Hours,
                        rParkingInfo.OpeningTimes[u8DayType].PerDayOpeningTimes[u8PerDayTime].EndTime.Minutes
                        );

            }
        }
    }

    //finally Calaculate & Return Open Status
    return enFindOpenStatus(bIsClosed,bIsOpen,bIsMorethanStartTime,bIsLessThanEndTime);

}
/****************************************************************************
 *
 *FUNCTION:          enFindOpenStatus
 *
 *DESCRIPTION:       Util function to find Open Status
 *
 *PARAMETERS:        Open , Closed, Start & End time  Staus
 *
 *RETURNVALUES:      Open Status
 *
 ******************************************************************************/
midw_ext_fi_tcl_e8OpenStatus::tenType fc_sxm_tclParkingBase::enFindOpenStatus(
        tBool bIsClosed,tBool bIsOpen,tBool bIsMorethanStartTime,tBool bIsLessThanEndTime) const
{
    //first Check if Location closed
    if(bIsClosed)
    {
        return midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_CLOSED;
    }

    //If not,Then Check if is Open
    //1. If Open for Full Day.
    //2. If Oepn for particular Time,Check if Current time within Range
    if( (bIsOpen) || (bIsMorethanStartTime && bIsLessThanEndTime))
    {
        return midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_OPEN;
    }
    //User Current Time is not in the Range
    else
    {
        //return Location is Closed.
        return midw_ext_fi_tcl_e8OpenStatus::FI_EN_PARKING_STATUS_CLOSED;
    }

}
/****************************************************************************
 *
 *FUNCTION:          bCheckOpenStatus
 *
 *DESCRIPTION:       Util function to Check if Parking Location Time is Open from SMS
 *
 *PARAMETERS:        Start & End time in Hrs & Minutes
 *
 *RETURNVALUES:      Status
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingBase::bCheckOpen24Hours(
        tU32 u32StartTimeinHrs,tU32 u32StartTimeinMts,tU32 u32EndTimeinHrs,tU32 u32EndTimeinMts) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckOpen24Hours u32StartTimeinHrs=%d,u32StartTimeinMts=%d,"
            "u32EndTimeinHrs=%d,u32EndTimeinMts=%d",u32StartTimeinHrs,u32StartTimeinMts,u32EndTimeinHrs,u32EndTimeinMts));

    tBool bStatus = FALSE;
    //When parking Location is Open,
    //StartTime is encoded as 24 Hours
    if(u32StartTimeinHrs == 24 && u32EndTimeinHrs == 24)
    {
        bStatus = TRUE;
    }
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckOpen24Hours bStatus = %d",bStatus));
    return bStatus;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckClosedStatus
 *
 *DESCRIPTION:       Util function to Check if Parking Location Time is Open from SMS
 *
 *PARAMETERS:        Start & End time in Hrs & Minutes
 *
 *RETURNVALUES:      Status
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingBase::bCheckClosedStatus(
        tU32 u32StartTimeinHrs,tU32 u32StartTimeinMts,tU32 u32EndTimeinHrs,tU32 u32EndTimeinMts) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckClosedStatus u32StartTimeinHrs=%d,u32StartTimeinMts=%d,"
            "u32EndTimeinHrs=%d,u32EndTimeinMts=%d",u32StartTimeinHrs,u32StartTimeinMts,u32EndTimeinHrs,u32EndTimeinMts));

    tBool bStatus = FALSE;
    //When parking Location is Closed,
    //StartTime & End Time are encoded as 00 Hours
    if( u32StartTimeinHrs == 0 && u32StartTimeinMts == 0 &&
         u32EndTimeinHrs == 0 && u32EndTimeinMts ==0 )
    {
        bStatus = TRUE;
    }

    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckClosedStatus bStatus = %d",bStatus));
    return bStatus;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckStartTime
 *
 *DESCRIPTION:       Util function to comapre current time with Parking Location Start time
 *
 *PARAMETERS:        Start time in Hrs & Minutes
 *
 *RETURNVALUES:      Status
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingBase::bCheckStartTime(tU32 u32StartTimeinHrs,tU32 u32StartTimeinMts) const
{
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,FALSE)

    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckStartTime u32StartTimeinHrs=%d,u32StartTimeinMts=%d,"
            "CurentHour=%d,CurentMinutes=%d",u32StartTimeinHrs,u32StartTimeinMts,_poApp->u8GetCurentHour(),_poApp->u8GetCurentMinutes()));

    tBool bStatus = FALSE;
    if(static_cast<tU32>(_poApp->u8GetCurentHour()) > u32StartTimeinHrs)
    {
        bStatus = TRUE;
    }
    else if(static_cast<tU32>(_poApp->u8GetCurentHour()) == u32StartTimeinHrs)
    {
        if(static_cast<tU32>(_poApp->u8GetCurentMinutes()) >= u32StartTimeinMts)
        {
            bStatus = TRUE;
        }
    }
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckStartTime bStatus = %d",bStatus));

    return bStatus;
}
/****************************************************************************
 *
 *FUNCTION:          bCheckEndTime
 *
 *DESCRIPTION:       Util function to comapre current time with Parking Location End time
 *
 *PARAMETERS:        Start & End time in Hrs & Minutes
 *
 *RETURNVALUES:      Status
 *
 ******************************************************************************/
tBool fc_sxm_tclParkingBase::bCheckEndTime(tU32 u32StartTimeinHrs,tU32 /*u32StartTimeinMts*/,tU32 u32EndTimeinHrs,tU32 u32EndTimeinMts) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckEndTime u32EndTimeinHrs=%d,u32EndTimeinMts=%d",u32EndTimeinHrs,u32EndTimeinMts));

    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _poApp,FALSE)

    //First Check if End time of next day
    //i.e. End time is earlier than Start time, eg. Opened at 06:00 and closed at 01:00
    if(u32EndTimeinHrs < u32StartTimeinHrs)
    {
    	u32EndTimeinHrs = u32EndTimeinHrs + 24;
    }

    tBool bStatus = FALSE;
    if(static_cast<tU32>(_poApp->u8GetCurentHour()) < u32EndTimeinHrs)
    {
        bStatus = TRUE;
    }
    else if(static_cast<tU32>(_poApp->u8GetCurentHour()) == u32EndTimeinHrs)
    {
        if(static_cast<tU32>(_poApp->u8GetCurentMinutes()) < u32EndTimeinMts)
        {
            bStatus = TRUE;
        }
    }
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::bCheckEndTime bStatus = %d",bStatus));
    return bStatus;
}
/****************************************************************************
 *
 *FUNCTION:          vCopyLocationToFI
 *
 *DESCRIPTION:       copy particular location to FI
 *
 *PARAMETERS:		SXMParkingLocation,midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vCopyLocationToFI(SXMParkingLocation &rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::vCopyLocationToFI"));

    //parking location details
    rParkingInfo.ID = static_cast<tU32>(rLocation.id);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.name,rParkingInfo.Name);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.address,rParkingInfo.Address);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.address_sec,rParkingInfo.SecondaryAddress);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.city,rParkingInfo.City);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.zip,rParkingInfo.Zip);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.phone,rParkingInfo.Phone);
    fc_sxm_tclGenericUtils::fc_sxm_vString2Fi(rLocation.state,rParkingInfo.State);

    rParkingInfo.Capacity = static_cast<tU32>(rLocation.capacity);

    //lat & lon details
    rParkingInfo.PrimaryLocation.Latitude.s32Value = fc_sxm_tclGenericUtils::s32SmsDegreeFloatToNav(rLocation.lat);
    rParkingInfo.PrimaryLocation.Longitude.s32Value = fc_sxm_tclGenericUtils::s32SmsDegreeFloatToNav(rLocation.lon);
    rParkingInfo.SecondaryLocation.Latitude.s32Value = fc_sxm_tclGenericUtils::s32SmsDegreeFloatToNav(rLocation.lat_sec);
    rParkingInfo.SecondaryLocation.Longitude.s32Value = fc_sxm_tclGenericUtils::s32SmsDegreeFloatToNav(rLocation.lon_sec);

    SXM_ASSERT_RETURN(OSAL_NULL != _poApp)

    //distance
    if(u32GetListID() == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_BY)
    {
        //for near by,we are directly getting distance from sdk
        rParkingInfo.Distance = static_cast<tF32>(rLocation.distance);
    }
    //for remaining ListID's
    //we should calculate distance ourself.
    else if(u32GetListID() == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_NEAR_DESTINATION)
    {
		if(_trListMode.enType == (midw_ext_fi_tcl_e8_SxmListMode::FI_EN_FREEZE))
		{
			//calculate distance from device position when destination list is frozen
			//this is required for sorting list based on distance
			rParkingInfo.Distance = _poApp->f32GetDistanceFromFrozenDevicePos(rLocation.lat,rLocation.lon);
		}
		else
		{
			//when list is not frozen we should sort the list based on actual current position
			rParkingInfo.Distance = _poApp->f32GetCurrentDistance(rParkingInfo.PrimaryLocation.Latitude.s32Value
													,rParkingInfo.PrimaryLocation.Longitude.s32Value);
		}
    }
    else if(u32GetListID() == (tU32)midw_ext_fi_tcl_e8_SxmListType::FI_EN_FAVORITE)
    {
    	rParkingInfo.Distance = _poApp->f32GetCurrentDistance(rParkingInfo.PrimaryLocation.Latitude.s32Value
    														,rParkingInfo.PrimaryLocation.Longitude.s32Value);
    }

    //Height details
    rParkingInfo.PrimaryHeight = static_cast<tU32>(rLocation.height);
    rParkingInfo.SecondaryHeight = static_cast<tU32>(rLocation.height_sec);
    rParkingInfo.AvailableTime = static_cast<tU32>(rLocation.avail_time);

    //relative pricing
    rParkingInfo.RelativePricingDaily = static_cast<tU32>(rLocation.price_day);
    rParkingInfo.RelativePricingHourly = static_cast<tU32>(rLocation.price_hour);

    //Is Favorite or not
    rParkingInfo.IsFavorite = _poApp->bIsFavorite(rParkingInfo.ID);

    //Copy Opening Times
    vCopyOpeningTimes(rLocation,rParkingInfo);

    //Some times,Open Status is available from broardcast
    if(rLocation.valid & SXM_PARKING_VALID_STATUS)
    {
        //If So,Just Copy the value from broadcast
        rParkingInfo.OpenStatus.enType = static_cast<midw_ext_fi_tcl_e8OpenStatus::tenType>(rLocation.status);
    }
    //If not available in broadcast
    //We should calculate open status form Opening Times
    else
    {
        //Copy the Open status based on Opening Times
        rParkingInfo.OpenStatus.enType = enGetOpenStatus(rParkingInfo);
    }

    //Copy Fullness data
    vCopyFullness(rLocation,rParkingInfo);

    //Copy Amenities
    vCopyAmenities(rLocation.amen,rParkingInfo.Amenities);

    //Copy Pricing details
    vCopyPricing(rLocation,rParkingInfo.PricingInfo);

}
/****************************************************************************
 *
 *FUNCTION:          vCopyFullness
 *
 *DESCRIPTION:       Method to copy fullness details to FI
 *
 *PARAMETERS:		rLocation -location details from SXM
 *PARAMETERS:		rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vCopyFullness(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::vCopyFullness"));

    tBool bExists = FALSE;

    //Check for valid fileds
    bExists = bCopyFullnessNow(rLocation,rParkingInfo);
    bExists = bCopyFullness15Min(rLocation,rParkingInfo);
    bExists = bCopyFullness30Min(rLocation,rParkingInfo);
    bExists = bCopyFullness45Min(rLocation,rParkingInfo);
    bExists = bCopyFullness60Min(rLocation,rParkingInfo);

    //if no valid filed is present
    if(FALSE == bExists)
    {
        //copy default values
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_UNKNOWN;
        rFullness.Value = arFullnessValue[0];
        rParkingInfo.Fullness.push_back(rFullness);
    }
}
/****************************************************************************
 *
 *FUNCTION:          bCopyFullnessNow
 *
 *DESCRIPTION:       Method to copy fullness for Now
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingBase::bCopyFullnessNow(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    //for now
    if  (rLocation.valid & SXM_PARKING_VALID_FULL0)
    {
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_NOW;
        rFullness.Value = arFullnessValue[rLocation.fullness[0]];
        rParkingInfo.Fullness.push_back(rFullness);
        return TRUE;
    }

    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          bCopyFullness15Min
 *
 *DESCRIPTION:       Method to copy fullness for Next 15 min
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingBase::bCopyFullness15Min(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    //for next 15Min
    if  (rLocation.valid & SXM_PARKING_VALID_FULL1)
    {
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_15MIN;
        rFullness.Value = arFullnessValue[rLocation.fullness[1]];
        rParkingInfo.Fullness.push_back(rFullness);
        return TRUE;
    }

    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          bCopyFullness30Min
 *
 *DESCRIPTION:       Method to copy fullness ffor Next 30 min
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingBase::bCopyFullness30Min(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    //next 30 min
    if  (rLocation.valid & SXM_PARKING_VALID_FULL2)
    {
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_30MIN;
        rFullness.Value = arFullnessValue[rLocation.fullness[2]];
        rParkingInfo.Fullness.push_back(rFullness);
        return TRUE;
    }

    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          bCopyFullness45Min
 *
 *DESCRIPTION:       Method to copy fullness for Next 45 min
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingBase::bCopyFullness45Min(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    //for next 45 min
    if  (rLocation.valid & SXM_PARKING_VALID_FULL3)
    {
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_45MIN;
        rFullness.Value = arFullnessValue[rLocation.fullness[3]];
        rParkingInfo.Fullness.push_back(rFullness);
        return TRUE;
    }

    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          bCopyFullnessNow
 *
 *DESCRIPTION:       Method to copy fullness for Next 60 min
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tBool fc_sxm_tclParkingBase::bCopyFullness60Min(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{
    //for next 60 min
    if  (rLocation.valid & SXM_PARKING_VALID_FULL4)
    {
        midw_ext_fi_tcl_Fullness rFullness;
        rFullness.Type.enType = midw_ext_fi_tcl_e8FullType::FI_EN_PARKING_FULLNESS_60MIN;
        rFullness.Value = arFullnessValue[rLocation.fullness[4]];
        rParkingInfo.Fullness.push_back(rFullness);
        return TRUE;
    }

    return FALSE;
}
/****************************************************************************
 *
 *FUNCTION:          vCopyAmenities
 *
 *DESCRIPTION:       Method to copy amenities details to FI
 *
 *PARAMETERS:		u32SDKAmen -Amenities bit field from SXM
 *PARAMETERS:		rAmenInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vCopyAmenities(tU32 u32SDKAmen,midw_ext_fi_tcl_AmenityInfo& rAmenInfo) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::vCopyAmenities"));

    //copy amenities from MSB to LSB by 2 bits
    rAmenInfo.Open24Hours.enType = static_cast<midw_ext_fi_tcl_e8Open24Hours::tenType>(u32GetAmenValue(u32SDKAmen,30));
    rAmenInfo.Cash.enType = static_cast<midw_ext_fi_tcl_e8CashOrCard::tenType>(u32GetAmenValue(u32SDKAmen,28));
    rAmenInfo.Card.enType = static_cast<midw_ext_fi_tcl_e8CashOrCard::tenType>(u32GetAmenValue(u32SDKAmen,26));
    rAmenInfo.ParkingType.enType = static_cast<midw_ext_fi_tcl_e8ParkingType::tenType>(u32GetAmenValue(u32SDKAmen,24));
    rAmenInfo.MetroLocation.enType = static_cast<midw_ext_fi_tcl_e8MetroLocation::tenType>(u32GetAmenValue(u32SDKAmen,22));
    rAmenInfo.AirPort.enType = static_cast<midw_ext_fi_tcl_e8AirPort::tenType>(u32GetAmenValue(u32SDKAmen,20));
    rAmenInfo.OvernightParking.enType = static_cast<midw_ext_fi_tcl_e8OvernightParking::tenType>(u32GetAmenValue(u32SDKAmen,18));
    rAmenInfo.Reservations.enType = static_cast<midw_ext_fi_tcl_e8Reservations::tenType>(u32GetAmenValue(u32SDKAmen,16));
    rAmenInfo.SelfParking.enType = static_cast<midw_ext_fi_tcl_e8SelfParking::tenType>(u32GetAmenValue(u32SDKAmen,14));
    rAmenInfo.Security.enType = static_cast<midw_ext_fi_tcl_e8Security::tenType>(u32GetAmenValue(u32SDKAmen,12));
    rAmenInfo.TollTag.enType = static_cast<midw_ext_fi_tcl_e8TollTag::tenType>(u32GetAmenValue(u32SDKAmen,10));
    rAmenInfo.SUVCharges.enType = static_cast<midw_ext_fi_tcl_e8SUVCharges::tenType>(u32GetAmenValue(u32SDKAmen,8));
    rAmenInfo.RV.enType = static_cast<midw_ext_fi_tcl_e8RV::tenType>(u32GetAmenValue(u32SDKAmen,6));

    //last 6 bits are reserved for future use.
}
/****************************************************************************
 *
 *FUNCTION:          u32GetAmenValue
 *
 *DESCRIPTION:       Method to Get Amenity value for a position
 *
 *PARAMETERS:       u32Val - Amenity value
 *PARAMETERS:       u8Pos - no. of positions to be shifted
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tU32 fc_sxm_tclParkingBase::u32GetAmenValue(tU32 u32Val,tU8 u8Pos) const
{
    return ((u32Val >> u8Pos) & 3);
}
/****************************************************************************
 *
 *FUNCTION:          vCopyPricing
 *
 *DESCRIPTION:       Method to copy price details to FI
 *
 *PARAMETERS:		SXMParkingLocation,midw_ext_fi_tcl_PriceInfo
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vCopyPricing(SXMParkingLocation& rLocation,midw_ext_fi_tcl_PriceInfo& rPricingInfo) const
{
    ETG_TRACE_USR2(("fc_sxm_tclParkingBase::vCopyPricing"));

    tU32 u32WeekDayhr = 0;
    tU32 u32SatDayhr = 0;
    tU32 u32SunDayhr = 0;

    //fill Daily pricing
    rPricingInfo.WeekdayDaily = static_cast<tF32>(rLocation.weekday_day/FC_SXM_PARKING_CENTS_PER_DOLLAR);
    rPricingInfo.SaturdayDaily = static_cast<tF32>(rLocation.saturday_day/FC_SXM_PARKING_CENTS_PER_DOLLAR);
    rPricingInfo.SundayDaily = static_cast<tF32>(rLocation.sunday_day/FC_SXM_PARKING_CENTS_PER_DOLLAR);
    rPricingInfo.EarlyBird = static_cast<tF32>(rLocation.ebird/FC_SXM_PARKING_CENTS_PER_DOLLAR);
    rPricingInfo.Event = static_cast<tF32>(rLocation.event/FC_SXM_PARKING_CENTS_PER_DOLLAR);
    rPricingInfo.Other = static_cast<tF32>(rLocation.other/FC_SXM_PARKING_CENTS_PER_DOLLAR);

    //for Hourly pricing,loop through for each hour.
    for(tU32 u32index=0;u32index<FC_SXM_PARKNG_PRICING_MAX_HOURS;u32index++)
    {
        //fill FI for each hour
        u32WeekDayhr += rLocation.weekday_hour[u32index];
        u32SatDayhr += rLocation.saturday_hour[u32index];
        u32SunDayhr += rLocation.sunday_hour[u32index];
    }

    //fill FI for each hour
    if( (rLocation.valid & SXM_PARKING_VALID_WK_HR) && u32WeekDayhr)
    {
        for(tU32 u32index=0;u32index<FC_SXM_PARKNG_PRICING_MAX_HOURS;u32index++)
        {
            rPricingInfo.WeekdayHourly.push_back(static_cast<tF32>(rLocation.weekday_hour[u32index]/FC_SXM_PARKING_CENTS_PER_DOLLAR));
        }
    }
    if((rLocation.valid & SXM_PARKING_VALID_SA_HR) && u32SatDayhr)
    {
        for(tU32 u32index=0;u32index<FC_SXM_PARKNG_PRICING_MAX_HOURS;u32index++)
        {
            rPricingInfo.SaturdayHourly.push_back(static_cast<tF32>(rLocation.saturday_hour[u32index]/FC_SXM_PARKING_CENTS_PER_DOLLAR));
        }
    }
    if((rLocation.valid & SXM_PARKING_VALID_SU_HR) && u32SunDayhr)
    {
        for(tU32 u32index=0;u32index<FC_SXM_PARKNG_PRICING_MAX_HOURS;u32index++)
        {
            rPricingInfo.SundayHourly.push_back(static_cast<tF32>(rLocation.sunday_hour[u32index]/FC_SXM_PARKING_CENTS_PER_DOLLAR));
        }
    }
}
/****************************************************************************
 *
 *FUNCTION:          vCopyOpeningTimes
 *
 *DESCRIPTION:       Method to copy Opening time details to FI
 *
 *PARAMETERS:       rLocation -location details from SXM
 *PARAMETERS:       rParkingInfo - details to be pushed in
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vCopyOpeningTimes(SXMParkingLocation& rLocation,midw_ext_fi_tcl_ParkingInfo& rParkingInfo) const
{

    for(tU8 u8DayType = 0;u8DayType < FC_SXM_PARKING_NO_OF_DAYS_IN_WEEK;u8DayType++)
    {
        midw_ext_fi_tcl_OpeningTime rOpeningTimes;
        //first copy the day type
        rOpeningTimes.DayType.enType = static_cast<midw_ext_fi_tcl_e8DayType::tenType>(u8DayType);

        midw_ext_fi_tcl_PerDayOpeningTimes rPerDayOpeningTimes;
        //prepare actual times of particular day
        string strOpeningTime(rLocation.days[u8DayType].times);

        //first check if specified hours are present
        string::size_type  position = strOpeningTime.find("-");
        if(position != string::npos)
        {
            string strTimeinHrs,strTimeinMts;

            //Prepare Start Time
            string strStartTime = strOpeningTime.substr(0,position);
            vGetOpeningTime(strStartTime,strTimeinHrs,strTimeinMts);
            rPerDayOpeningTimes.StartTime.Hours = static_cast<tU32>(std::atoi(strTimeinHrs.c_str()));
            rPerDayOpeningTimes.StartTime.Minutes = static_cast<tU32>(std::atoi(strTimeinMts.c_str()));

            //Preapre End time
            string strEndTime = strOpeningTime.substr(position+1,strOpeningTime.size());
            vGetOpeningTime(strEndTime,strTimeinHrs,strTimeinMts);
            rPerDayOpeningTimes.EndTime.Hours = static_cast<tU32>(std::atoi(strTimeinHrs.c_str()));
            rPerDayOpeningTimes.EndTime.Minutes = static_cast<tU32>(std::atoi(strTimeinMts.c_str()));

        }
        //check if it is open for 24 hours
        else if(strOpeningTime.find("O") != string::npos)
        {
            rPerDayOpeningTimes.StartTime.Hours = 24;
            rPerDayOpeningTimes.StartTime.Minutes = 00;

            rPerDayOpeningTimes.EndTime.Hours = 24;
            rPerDayOpeningTimes.EndTime.Minutes = 00;

        }
        //check if it is closed for a day
        else if(strOpeningTime.find("C") != string::npos)
        {
            rPerDayOpeningTimes.StartTime.Hours = 00;
            rPerDayOpeningTimes.StartTime.Minutes = 00;

            rPerDayOpeningTimes.EndTime.Hours = 00;
            rPerDayOpeningTimes.EndTime.Minutes = 00;

        }
        //if none of above,no data is available.
        else
        {
            //dont push any values
            return;
        }
        //finally push values
        rOpeningTimes.PerDayOpeningTimes.push_back(rPerDayOpeningTimes);

        //Finally push into FI
        rParkingInfo.OpeningTimes.push_back(rOpeningTimes);
    }
}
/****************************************************************************
 *
 *FUNCTION:          vGetOpeningTime
 *
 *DESCRIPTION:       Method to Parse Parking timings to Hours & minutes
 *
 *PARAMETERS:
 *PARAMETERS:
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vGetOpeningTime(string& strParkingTime,string& strTimeHrs,string &strTimeMts) const
{
    string::size_type  position = strParkingTime.find(":");
    if(position != string::npos)
    {
        strTimeHrs = strParkingTime.substr(0,position);
        strTimeMts = strParkingTime.substr(position+1,strParkingTime.size());
    }
}
/****************************************************************************
 *
 *FUNCTION:          bSetFilterConfig
 *
 *DESCRIPTION:       Method to set filter config as per client spec
 *
 *PARAMETERS:       midw_ext_sxm_parkingfi_tclMsgSetFilterConfigMethodStart
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vSetFilterConfig(const midw_ext_sxm_parkingfi_tclMsgSetFilterConfigMethodStart &oMStart)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::vSetFilterConfig"));

    const midw_ext_fi_tcl_AmenityInfo &Amen = oMStart.FilterConfig.Amenities;

    _FilterConfig.vSetFilterConfig(Amen.Cash,Amen.Card,Amen.SelfParking,Amen.ParkingType,
            oMStart.FilterConfig.ClosedLocations,Amen.SUVCharges,oMStart.FilterConfig.EntranceHeight);

}
/****************************************************************************
 *
 *FUNCTION:          bSetFilterConfig
 *
 *DESCRIPTION:       Method to set filter config as per client spec
 *
 *PARAMETERS:       midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodStart
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/

tVoid fc_sxm_tclParkingBase::vSetFilterConfig(const midw_ext_sxm_parkingfi_tclMsgSetFilterConfigurationMethodStart &oMStart)
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::vSetFilterConfiguration"));

    _FilterConfig.vSetFilterConfig(oMStart);

}
/****************************************************************************
 *
 *FUNCTION:          vPrintParkingList
 *
 *DESCRIPTION:       Method to print all parking locations(debug purpose)
 *
 *PARAMETERS:		midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vPrintParkingList(midw_ext_sxm_parkingfi_tclMsgGetParkingInfoListMethodResult &oMRes) const
{
    ETG_TRACE_USR4(("fc_sxm_tclParkingBase::vPrintParkingList"));

    for(tU32 u32index=0;u32index<oMRes.ParkingInfoList.size();u32index++)
    {
        vPrintParkingInfo(oMRes.ParkingInfoList[u32index]);
    }
}
/****************************************************************************
 *
 *FUNCTION:          vPrintParkingInfo
 *
 *DESCRIPTION:       Method to print particular parking location(debug purpose)
 *
 *PARAMETERS:       midw_ext_fi_tcl_ParkingInfo
 *
 *RETURNVALUES:      None
 *
 ******************************************************************************/
tVoid fc_sxm_tclParkingBase::vPrintParkingInfo(midw_ext_fi_tcl_ParkingInfo &rParkingInfo) const
{
    ETG_TRACE_USR4(("ID=%d",rParkingInfo.ID));
    ETG_TRACE_USR4(("Name=%s",rParkingInfo.Name.szValue));
    ETG_TRACE_USR2(("Address=%s",rParkingInfo.Address.szValue));
    ETG_TRACE_USR2(("Secondary Address=%s",rParkingInfo.SecondaryAddress.szValue));
    ETG_TRACE_USR2(("City=%s",rParkingInfo.City.szValue));
    ETG_TRACE_USR2(("Zip=%s",rParkingInfo.Zip.szValue));
    ETG_TRACE_USR2(("Phone=%s",rParkingInfo.Phone.szValue));
    ETG_TRACE_USR2(("State=%s",rParkingInfo.State.szValue));

    ETG_TRACE_USR2(("capacity=%d",rParkingInfo.Capacity));
    ETG_TRACE_USR2(("Distance=%f",rParkingInfo.Distance));

    ETG_TRACE_USR4(("Primary_Lat=%d",rParkingInfo.PrimaryLocation.Latitude.s32Value));
    ETG_TRACE_USR4(("Primary_Lon=%d",rParkingInfo.PrimaryLocation.Longitude.s32Value));
    ETG_TRACE_USR2(("Seconday_Lat=%d",rParkingInfo.SecondaryLocation.Latitude.s32Value));
    ETG_TRACE_USR2(("Seconday_Lon=%d",rParkingInfo.SecondaryLocation.Longitude.s32Value));

    ETG_TRACE_USR4(("Primary_Height=%d",rParkingInfo.PrimaryHeight));
    ETG_TRACE_USR2(("Secondary_Height=%d",rParkingInfo.SecondaryHeight));

    ETG_TRACE_USR2(("Avail Time=%d",rParkingInfo.AvailableTime));
    ETG_TRACE_USR2(("Relative Price Daily=%d",rParkingInfo.RelativePricingDaily));
    ETG_TRACE_USR2(("Relative Price Hourly=%d",rParkingInfo.RelativePricingHourly));

    ETG_TRACE_USR2(("Open Status=%d",rParkingInfo.OpenStatus.enType));
    ETG_TRACE_USR2(("IsFavorite=%d",rParkingInfo.IsFavorite));

    //fullness
    for(tU32 u32FullnessIndex=0;u32FullnessIndex<rParkingInfo.Fullness.size();u32FullnessIndex++)
    {
        ETG_TRACE_USR2(("Fullness Type=%d",rParkingInfo.Fullness[u32FullnessIndex].Type.enType));
        ETG_TRACE_USR2(("Fullness Val=%d",rParkingInfo.Fullness[u32FullnessIndex].Value));
    }

    ETG_TRACE_USR2(("Amen-Open24Hrs=%d",rParkingInfo.Amenities.Open24Hours.enType));
    ETG_TRACE_USR2(("Amen-Cash=%d",rParkingInfo.Amenities.Cash.enType));
    ETG_TRACE_USR2(("Amen-Card=%d",rParkingInfo.Amenities.Card.enType));
    ETG_TRACE_USR2(("Amen-ParkingType=%d",rParkingInfo.Amenities.ParkingType.enType));
    ETG_TRACE_USR2(("Amen-MetroLocation=%d",rParkingInfo.Amenities.MetroLocation.enType));
    ETG_TRACE_USR2(("Amen-AirPort=%d",rParkingInfo.Amenities.AirPort.enType));
    ETG_TRACE_USR2(("Amen-OverNightParking=%d",rParkingInfo.Amenities.OvernightParking.enType));
    ETG_TRACE_USR2(("Amen-Reservations=%d",rParkingInfo.Amenities.Reservations.enType));
    ETG_TRACE_USR2(("Amen-SelfParking=%d",rParkingInfo.Amenities.SelfParking.enType));
    ETG_TRACE_USR2(("Amen-Security=%d",rParkingInfo.Amenities.Security.enType));
    ETG_TRACE_USR2(("Amen-TollTag=%d",rParkingInfo.Amenities.TollTag.enType));
    ETG_TRACE_USR2(("Amen-SUV=%d",rParkingInfo.Amenities.SUVCharges.enType));
    ETG_TRACE_USR2(("Amen-RV=%d",rParkingInfo.Amenities.RV.enType));

    //Opening times
    for(tU32 u32DayType=0;u32DayType<rParkingInfo.OpeningTimes.size();u32DayType++)
    {
        ETG_TRACE_USR2(("Day Type=%d",rParkingInfo.OpeningTimes[u32DayType].DayType.enType));

        for(tU32 u32Index=0;u32Index<rParkingInfo.OpeningTimes[u32DayType].PerDayOpeningTimes.size();u32Index++)
        {
            midw_ext_fi_tcl_PerDayOpeningTimes &rPerDayOpeningTimes = rParkingInfo.OpeningTimes[u32DayType].PerDayOpeningTimes[u32Index];
            ETG_TRACE_USR2(("StartTime=>Hours=%d,Minutes=%d",rPerDayOpeningTimes.StartTime.Hours,rPerDayOpeningTimes.StartTime.Minutes));
            ETG_TRACE_USR2(("EndTime=>Hours=%d,Minutes=%d",rPerDayOpeningTimes.EndTime.Hours,rPerDayOpeningTimes.EndTime.Minutes));
        }

    }

    //Pricing INfo
    ETG_TRACE_USR2(("Pricing-WeekDay_Daily =%f",rParkingInfo.PricingInfo.WeekdayDaily));
    ETG_TRACE_USR2(("Pricing-SaturDay_Daily =%f",rParkingInfo.PricingInfo.SaturdayDaily));
    ETG_TRACE_USR2(("Pricing-SunDay_Daily =%f",rParkingInfo.PricingInfo.SundayDaily));
    ETG_TRACE_USR2(("Pricing-Early Bird =%f",rParkingInfo.PricingInfo.EarlyBird));
    ETG_TRACE_USR2(("Pricing-Event =%f",rParkingInfo.PricingInfo.Event));
    ETG_TRACE_USR2(("Pricing-Other =%f",rParkingInfo.PricingInfo.Other));

    ETG_TRACE_USR2(("Pricing-WeekDay_hourly size =%d",rParkingInfo.PricingInfo.WeekdayHourly.size()));
    ETG_TRACE_USR2(("Pricing-SaturDay_hourly size =%d",rParkingInfo.PricingInfo.SaturdayHourly.size()));
    ETG_TRACE_USR2(("Pricing-SunDay_hourly size =%d",rParkingInfo.PricingInfo.SundayHourly.size()));
}
#undef FC_SXM_PARKING_CENTS_PER_DOLLAR
