/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_theater_list.cpp
* @brief       Theater List Handler implementation.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_theater_list.h"
#include "fc_sxm_tcl_movies_app.h"
#include "fc_sxm_service_sxm_movies.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_generic_utils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_MOVIES_APP
#include "trcGenProj/Header/fc_sxm_tcl_theater_list.cpp.trc.h"
#endif

static map<MOVIE_ID, fc_sxm_trMovieDetails> _mUSMoviesList;

tVoid fc_sxm_trTheatreListConfig::vPrintSpecific() const {
   //print the sorted method
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenSortMethod=%d", ETG_CENUM(fc_sxm_tenMoviesSortMethod, rSortCfg.enSortMethod)));
}

/*
   Method to Retrieve Movies List playing in a Theater
*/

tU32 fc_sxm_trTheatreList::u32GetMoviesList(fc_sxm_trTheatreListEntry const *prEntry,midw_ext_fi_tcl_MoviesList &oMovieList)  const
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetMoviesList"));
   tU32 u32MoviesCount=0;

   /* Iterate through the Movies List of particular theater entry*/
   SXM_FOREACH_MAP_CONST(MOVIE_ID,fc_sxm_trMovieDetails,iter,prEntry->MoviesList)
   {
      /* Local Movie Info copy to hold Movies details*/
      midw_ext_fi_tcl_MovieInfo oMovieInfo;

      /* Retrieve required movie details from our theater entry*/
      oMovieInfo.MOVIE_ID = iter->second.movie_id;
      fc_sxm_vString2Fi( iter->second.movieName.c_str() , oMovieInfo.MovieName );
      fc_sxm_vString2Fi( iter->second.movieRating.sMovieRatingTesxt.c_str() , oMovieInfo.MovieRating );
#if 1
      ETG_TRACE_USR4(( "fc_sxm_trTheatreList::u32GetMoviesList:: LOC ID: %d",prEntry->u32SxmId ));
      ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetMoviesList::ADD: MOVIE_ID=%d MovieName=%30s MovieRating=%15s MovieRatingenum = %d",
            (tU16)(oMovieInfo.MOVIE_ID),
            (tString)(oMovieInfo.MovieName),
            (tString)oMovieInfo.MovieRating,
            iter->second.movieRating.enMovieRating
           ));
#endif

      /* Before Adding into List,Check whether this Movie Entry(ID) is already exists in the List*/
      if(bIsMovieFound(oMovieList,(tU16)(oMovieInfo.MOVIE_ID)) == FALSE)
      {
         ETG_TRACE_USR4(( "fc_sxm_trTheatreList::u32GetMoviesList:: movie not Found So Pushing MOVIE_ID = %d",oMovieInfo.MOVIE_ID));
         /* OK,Movie is not exists in List,So Add it now*/
         oMovieList.MoviesList.push_back(oMovieInfo);
      }

      /* increment movies count*/
      u32MoviesCount++;
   }

   return u32MoviesCount;
}
/*
   Method to find MOVIE_ID exists in MoviesList
*/
bool fc_sxm_trTheatreList::bIsMovieFound(midw_ext_fi_tcl_MoviesList const &oMoviesList, MOVIE_ID const n16MovieID) const {

   bool bMovieFound = false;
   /* Iterate through the Movies List*/
   SXM_FOREACH_CONST(std::vector<midw_ext_fi_tcl_MovieInfo> , pMoviesIter , oMoviesList.MoviesList)
   {
      /* Check whehter the this Movie ID is already exists*/
      if(n16MovieID == (tU16)((*pMoviesIter).MOVIE_ID))
      {
         /* Yes,Movie is already exists*/
         bMovieFound = true;

         ETG_TRACE_USR4(( "fc_sxm_trTheatreList::bIsbMovieFound:: bMovieFound ,So IGNORING MOVIE_ID = %d",(*pMoviesIter).MOVIE_ID));

         /* then don't continue*/
         break;
      }

   }
   /* return the check result*/
   return bMovieFound;
}
/*
  will be called by base-class for outside DSRL sorting
 */
bool fc_sxm_trTheatreList::bCompareItem(fc_sxm_trTheatreListEntry const *prLeft, fc_sxm_trTheatreListEntry const *prRight) const {
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::bCompareItem IN sortMethod = %u",
		   ETG_CENUM(fc_sxm_tenMoviesSortMethod, _rListCfg.rSortCfg.enSortMethod)));
   //based on sorted method compare
   switch (_rListCfg.rSortCfg.enSortMethod) {

      case fc_sxm_enMoviesSortMethod_NONE:
         {
            return TRUE; //incase of no sorted method
         }
      case fc_sxm_enMoviesSortMethod_DISTANCE:
         {
            return bCompareSmsDistance(prLeft, prRight);
         }
      case fc_sxm_enMoviesSortMethod_NAME:
         {
            return bCompareTheaterName(prLeft, prRight);
         }
      case fc_sxm_enMoviesSortMethod_FIRST_SAVED_FIRST:
         {
            return bCompareFavoriteIndex(prLeft, prRight);
         }
      default:
         return bCompareListEntryId(prLeft, prRight); //default comparision

   }
}
bool fc_sxm_trTheatreList::bCompareTheaterName(fc_sxm_trTheatreListEntry const *prLeft, fc_sxm_trTheatreListEntry const *prRight) const {

   string sLeftTheater = prLeft->TheaterDetails.TheatreName.szValue;
   string sRightTheater = prRight->TheaterDetails.TheatreName.szValue;

   /*First Convert both the strings to UpperCase */
   ConvertToUpper(sLeftTheater);
   ConvertToUpper(sRightTheater);

   /* Compare both the names using OSAL compare func */
   if( OSAL_s32StringCompare( sLeftTheater.c_str(),sRightTheater.c_str() ) < 0)
   {
      return true;
   }
   return false;
}
bool fc_sxm_trTheatreList::bFilterPassed(fc_sxm_trTheatreListEntry const &rItem) const {
   //based on filtered method do the action  
   switch (_rListCfg.rFilterCfg.enFilterMethod) 
   {
      case fc_sxm_enMoviesFilterMethod_NAME: //based on name
         {
            ETG_TRACE_USR4(("fc_sxm_trTheatreList::bFilterPassed fc_sxm_enMoviesFilterMethod_NAME "));
            return bSearchFound(_rListCfg.rFilterCfg.SearchName,rItem.TheaterDetails.TheatreName);
         }
      case fc_sxm_enMoviesFilterMethod_MOVIEID: //based on movie id
         {
            ETG_TRACE_USR4(("fc_sxm_trTheatreList::bFilterPassed fc_sxm_enMoviesFilterMethod_MOVIEID"));
            return bFilterMovie(rItem, _rListCfg.rFilterCfg.u32MovieID);
         }
      case fc_sxm_enMoviesFilterMethod_NONE:
      default:
         {
            ETG_TRACE_USR4(("fc_sxm_trTheatreList::bFilterPassed fc_sxm_enMoviesFilterMethod_NONE"));
            return true;
         }
    }
}
/*  
   function that shall filter theater playing particular movie 
*/
bool fc_sxm_trTheatreList::bFilterMovie(fc_sxm_trTheatreListEntry const &rEntry,tU32 u32MoviesID) const
   {
     midw_ext_fi_tcl_MoviesList oMovieList; 

     /* Get Movies playing in a theater */
     if( u32GetMoviesList(&rEntry,oMovieList) > 0 )
        {
            /* Check for our MovieID */
            return bIsMovieFound(oMovieList,(N16)u32MoviesID);
        }
     return false;
}
/*
  will be called by base-class when data are ready and the list shall be sent to hmi
 */
tVoid fc_sxm_trTheatreList::vEmit(fc_sxm_trAdressing const &rAdressing) {

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vEmit START"));

   /* Define local Method Result for Movies List*/
   midw_ext_sxm_moviesfi_tclMsgGetTheatreListMethodResult oMRes;

   /* Get the current List ID*/
   oMRes.ListID = u32GetId();

   const trSortedSetType &SortedEntries = rGetSortedEntries();

   /* Iterate through each sorted eantry and assign values to our result*/
   SXM_FOREACH_CONST(trSortedSetType, iter, SortedEntries ) {

   fc_sxm_trTheatreListEntry const *poEntry=*iter;

   /* Make a Local TheaterInfo Entry to hold our Method Result values*/
   midw_ext_fi_tcl_TheatreInfo oFiEntry;

   /* Retrieve all necessary values required for our Method Result from the Sorted Entries*/
   oFiEntry.Theater_LOCID=poEntry->u32SxmId;
   oFiEntry.TheatreName = poEntry->TheaterDetails.TheatreName;
   oFiEntry.TheaterCity = poEntry->TheaterDetails.TheatreCity;
   oFiEntry.Distance =  fGetFIDistance(poEntry);
   
   /* Get Theater Location */
   oFiEntry.TheaterLocation = fc_sxm_tclMoviesApp::instance()->rGetTheaterLocation(poEntry->hLocation);

   /* copy the local TheaterInfo entry to our Method result */

   oMRes.TheatreInfoList.TheatreInfolist.push_back(oFiEntry);

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::ADD: SMS_ID=%d TheaterName=%90s City=%15s TheaterLocLat=%d TheaterLocLon=%d distance=%f",
            oFiEntry.Theater_LOCID,
            oFiEntry.TheatreName.szValue,
            oFiEntry.TheaterCity.szValue,
            oFiEntry.TheaterLocation.Lat,
            oFiEntry.TheaterLocation.Lon,
            oFiEntry.Distance));
   }

   ETG_TRACE_USR4(( "fc_sxm_trTheatreList::vEmit Sending List to HMI" ));
   
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING 
   // write the list-operation to the method-result
   vListOperation2Fi(oMRes.Operation);
   ETG_TRACE_USR4(("fc_sxm_tclMoviesList::vEmit(%d) enListState=%d enListMode=%d numEntries=%u fiOp=%d START",
                   oMRes.ListID, 
                   ETG_CENUM(fc_sxm_tenListState, enGetListState()), 
                   ETG_CENUM(fc_sxm_tenListMode, enGetListMode()), 
                   rGetSortedEntries().size(),
                   oMRes.Operation.enType));
#endif
   // write current list-info to the method-result
   oMRes.ListInfo=rGetStatusInfo();
 
   /* Finally Send the local Method Result to our Service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(rAdressing, oMRes);

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vEmit END"));

}

#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING 
/*
    will be called by base-class when data are ready and the list shall be sent to hmi
 */
tVoid fc_sxm_trTheatreList::vStartTimer() {
    ETG_TRACE_USR4(("fc_sxm_trTheatreList::vStartTimer START"));
    // start the list timer implemented in Movies app
    fc_sxm_tclMoviesApp::instance()->vStartListTimer();
}

/*
  will be called by base-class to check the timer status
 */
tBool fc_sxm_trTheatreList::bIsListModeTimerRunning() {
    ETG_TRACE_USR4(("fc_sxm_trTheatreList::vIsListModeTimerRunning START"));

    return(fc_sxm_tclMoviesApp::instance()->bIsListUpdateTimerRunning());
}
/*
  will be called by base-class when list is released
 */
tVoid fc_sxm_trTheatreList::vStopTimer() {
    ETG_TRACE_USR4(("fc_sxm_trTheatreList::vStopTimer STOP"));

    fc_sxm_tclMoviesApp::instance()->vStopListTimer();
}
#endif
/*
  Method to Send Movies List to FI
*/
tVoid fc_sxm_trTheatreList::vSendMoviesList(tU32 u32ListID,fc_sxm_trAdressing const &rAdressing) {

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendMoviesList START"));

   /* Define local Method Result for Theater List*/
   midw_ext_sxm_moviesfi_tclMsgGetMoviesListMethodResult oMRes;

   /* Get the current List ID*/
   oMRes.ListID = u32ListID;
   
   /* Get Movies Playing in complete US*/
   u32GetUSMoviesList(oMRes.MoviesInfoList);

   ETG_TRACE_USR4(( "fc_sxm_trTheatreList::vSendMoviesList Sending List to HMI" ));
   /* OK,We have movies list to send to FI*/
   fc_sxm_tclMoviesApp::instance()->vSortList(oMRes.MoviesInfoList);  

   /* Before Sending ,Print Movies List once(this is for debugging only)*/
   fc_sxm_tclMoviesApp::instance()->vPrintMovies(oMRes.MoviesInfoList);

   /* Finally Send the Movies List to our Service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(rAdressing, oMRes);

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendMoviesList END"));

}
/*
   Method to retrieve complete US Movies from internal list
*/
tU32 fc_sxm_trTheatreList::u32GetUSMoviesList(midw_ext_fi_tcl_MoviesList &oMovieList)
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetUSMoviesList"));
   tU32 u32MoviesCount=0;

   /* Check for 0 elements*/
   if(0 == _mUSMoviesList.size() )
   {
      return u32MoviesCount;
   }

   /* Iterate through the Movies List of particular theater entry*/
   SXM_FOREACH_MAP(MOVIE_ID,fc_sxm_trMovieDetails,iter,_mUSMoviesList)
   {
      /* Local Movie Info copy to hold Movies details*/
      midw_ext_fi_tcl_MovieInfo oMovieInfo;

      /* Retrieve required movie details from our theater entry*/
      oMovieInfo.MOVIE_ID = iter->second.movie_id;
      fc_sxm_vString2Fi( iter->second.movieName.c_str() , oMovieInfo.MovieName);
      fc_sxm_vString2Fi( iter->second.movieRating.sMovieRatingTesxt.c_str() , oMovieInfo.MovieRating );
      #if 1
         ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetUSMoviesList::ADD: MOVIE_ID=%d MovieName=%30s MovieRating=%15s MovieRatingenum = %d",
               (tU16)oMovieInfo.MOVIE_ID,
			   (tString)(oMovieInfo.MovieName),
			   (tString)(oMovieInfo.MovieRating),
               iter->second.movieRating.enMovieRating
              ));
      #endif

      oMovieList.MoviesList.push_back(oMovieInfo);

      /* increment movies count*/
      u32MoviesCount++;
   }

   return u32MoviesCount;
}
/*
   Method to Send Movie Details to FI
 */
tVoid fc_sxm_trTheatreList::vSendUSMovieInfo(tU32 u32ListID,tU32 u32MovieID,fc_sxm_trAdressing const &rAdressing) 
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendUSMovieInfo START"));

   /* ethod Result for Movie Information*/
   midw_ext_sxm_moviesfi_tclMsgGetMovieInfoMethodResult oMRes;

   /* Var to hold Movie finding status*/
   tBool bMovieFound=FALSE;;

   /* Get the current List ID*/
   oMRes.ListID = u32ListID;
   
   /*  Check for empty case */
   if(0 == _mUSMoviesList.size() )
   {
      return;
   }

   /* Iterate through existing US movies list */
   SXM_FOREACH_MAP(MOVIE_ID,fc_sxm_trMovieDetails,iter,_mUSMoviesList)
   {
      /* Check for our MovieID */
      if(u32MovieID == (tU32)(iter->first))
      {
         /* OK,Copy the required Movie Details to our FI */
         oMRes.MovieInfo.MOVIE_ID =  iter->first;
         /* Retrieve Other Movie Details */
         vRetrieveMovieDetails(iter->second,oMRes.MovieInfo);
         /* Say We have found our Movie in the list */
         bMovieFound = TRUE;
         /* break here,No need to continue further*/
         break;
      }
   }

   /* Check If MovieID found */
   if(FALSE == bMovieFound)
   {
      ETG_TRACE_ERR(( "fc_sxm_trTheatreList::vSendUSMovieInfo Movied_ID=%d not Found",u32MovieID ));
      /* Particular MovieId is not found in current list ,So Send a Error report to HMI*/
       fc_sxm_tclMoviesService::vSendError(rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
   }
   else
   {
      ETG_TRACE_USR4(( "fc_sxm_trTheatreList::vSendUSMovieInfo Sending Movie Info to HMI" ));

      /* OK,Send the Found Movie details to HMi */
      /* Before Sending ,Print Movies Details once(this is for debugging only)*/
      ETG_TRACE_USR4(( "\n<<<<<<<<<<<<<< Movie Info >>>>>>>>>>>>" ));
      

      ETG_TRACE_USR4(("MOVIE ID: %d", oMRes.MovieInfo.MOVIE_ID));
      ETG_TRACE_USR4(("Movie Name: %s", oMRes.MovieInfo.MovieName.szValue ));
      ETG_TRACE_USR4(("Movie Actors: %s", oMRes.MovieInfo.MovieActors.szValue ));
      ETG_TRACE_USR4(("Movies Summury: %s", oMRes.MovieInfo.MovieSummury.szValue ));
      ETG_TRACE_USR4(("Movie Rating: %s",oMRes.MovieInfo.MovieRating.szValue ));
      ETG_TRACE_USR4(("Movie Length(String): %s", oMRes.MovieInfo.sMovieLength.szValue ));

      /* Finally Send the Movies details to our Service*/
      fc_sxm_tclMoviesService::instance()->enSendFiMessage(rAdressing, oMRes);
   }

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendMovieInfo END"));

}
/*
   Method to Search a Movie From Movies List
 */
tVoid fc_sxm_trTheatreList::vSearchMovie(midw_ext_fi_tclString trSearchMovieName,fc_sxm_trAdressing const &rAdressing) {

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendMoviesList START"));

   /* Define local Method Result for Theater List*/
   midw_ext_sxm_moviesfi_tclMsgSearchMovieMethodResult oMRes;
   
   /* Get Movies Playing in this Theater Entry*/
   u32SearchMoviesList(trSearchMovieName,oMRes.MoviesList);

   ETG_TRACE_USR4(( "fc_sxm_trTheatreList::vSendMoviesList Sending List to HMI" ));
   
   /* OK,We have movies list to send to FI*/
   fc_sxm_tclMoviesApp::instance()->vSortList(oMRes.MoviesList);  //TODO:currently Sorting is not done for movies list

   /* Before Sending ,Print Movies List once(this is for debugging only)*/
   fc_sxm_tclMoviesApp::instance()->vPrintMovies(oMRes.MoviesList);

   /* Finally Send the Movies List to our Service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(rAdressing, oMRes);

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vSendMoviesList END"));

}
/*
   to get the Particular Movie Information playing in a theater
*/

tBool fc_sxm_trTheatreList::bGetMovieInfo(tU32 u32MovieID,fc_sxm_trTheatreListEntry const *prEntry,midw_ext_fi_tcl_MovieDetails &oMovieDetails) const
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::bGetMovieInfo IN"));
   tBool bMovieFound=FALSE;

   /* Iterate through the Movies List of particular theater entry*/
   SXM_FOREACH_MAP_CONST(MOVIE_ID,fc_sxm_trMovieDetails,iter,prEntry->MoviesList)
   {
    
      /* Check if our MovieID exists in the Movies List*/
      if(u32MovieID == (tU32)(iter->first))
      {
         ETG_TRACE_USR4(("fc_sxm_trTheatreList::bGetMovieInfo bMovieFound MovieID = %d",iter->first ));

         /*If ofund copy the required details */
         oMovieDetails.MOVIE_ID =  iter->first;
         /* Retrieve Other Movie Details */
         fc_sxm_trTheatreList::vRetrieveMovieDetails(iter->second,oMovieDetails);
         /* Retrieve Movie Showtimes */
         vGetMovieShowTimesList(iter->second.movieShowTimes,oMovieDetails.MovieTimesList);
         /* Say We have found our Movie in the list */
		   bMovieFound=TRUE;
         /* No need to look further,So break down here*/
         break;

      }
   }

   return bMovieFound;
}
/*
   helper to Retireve Movie details from our internal info.
*/
tVoid fc_sxm_trTheatreList::vRetrieveMovieDetails(fc_sxm_trMovieDetails MovieDetails,midw_ext_fi_tcl_MovieDetails &rMovieDetails)
{
   /* Retrieve Movie Name */
   fc_sxm_vString2Fi( MovieDetails.movieName.c_str() , rMovieDetails.MovieName );
   /* Retrieve Movie Rating and Filter Invalid string */
   fc_sxm_vString2Fi( MovieDetails.movieRating.sMovieRatingTesxt.c_str() , rMovieDetails.MovieRating );
   fc_sxm_tclMoviesApp::instance()->vCheckForInvalidString(rMovieDetails.MovieRating);
   /* Retrieve Movie Length and Filter Invalid string */
   fc_sxm_vString2Fi( fc_sxm_trTheatreList::sGetMovieRunTime(MovieDetails.movieLength).c_str() , rMovieDetails.sMovieLength );
   fc_sxm_tclMoviesApp::instance()->vCheckForInvalidString(rMovieDetails.sMovieLength);
   /* Retrieve Movie Actors and Filter Invalid string */
   fc_sxm_vString2Fi( MovieDetails.movieActors.c_str() , rMovieDetails.MovieActors );
   fc_sxm_tclMoviesApp::instance()->vCheckForInvalidString(rMovieDetails.MovieActors);
   /* Retrieve Movie Summury and Filter Invalid string */
   fc_sxm_vString2Fi( MovieDetails.movieSummury.c_str() , rMovieDetails.MovieSummury );
   fc_sxm_tclMoviesApp::instance()->vCheckForInvalidString(rMovieDetails.MovieSummury);
}
/*
   helper to Search a String in another string
*/
tBool fc_sxm_trTheatreList::bSearchFound(string trSearchString,midw_ext_fi_tclString trOriginalString)
{
   string strOriginal = trOriginalString.szValue;

   /* First Convert both the strings to Upper case*/
   ConvertToUpper(trSearchString);
   ConvertToUpper(strOriginal);

   /* searches a substring in a Original string*/
   const char* pSubString = strstr(strOriginal.c_str(),trSearchString.c_str());

   /* A match is found if the pointer returned by the strstr function is not NULL */
   if( pSubString != NULL)
   {
      /* OK, pSubString is not NULL,
         We have Found Sub-String in Original String*/
      return TRUE;
   }

   /* We Have not found String*/
   return FALSE;
}
/*
   helper to Convert a String Characters to Uppercase
*/
tVoid fc_sxm_trTheatreList::ConvertToUpper(string &strSource)
{
   tU32 u32Val = 0;
   string strFinal;
   tU32 u32HexSize = 0;			//UTF-8 character size in bytes
   vector <tU32> u32UniCode;
   tU32 u32Index = 0;
   /*Loop through the size of the string */
   while( u32Index < strSource.length() )
   {
      u32HexSize = fc_sxm_tclGenericUtils::vGetUnicode((unsigned char*)&(strSource[u32Index]),u32Val);
      /*vector to store Unicode Values */
      u32UniCode.push_back(u32Val);
      ETG_TRACE_USR4(("fc_sxm_trTheatreList::ConvertToUpper string:vGetUnicode strSource[%d] ,hex value = 0x%x,Unicode = 0x%x ,u32HexSize = %d",u32Index,strSource[u32Index],u32Val,u32HexSize));
        u32Index+=u32HexSize;
    }

   /*Loop through the Unicode Values of the string */
   for(vector<tU32>::iterator it = u32UniCode.begin(); it != u32UniCode.end(); ++it)
   {
   string chString="";
          fc_sxm_tclGenericUtils::toUpper(*it);
          fc_sxm_tclGenericUtils::vUnicodeToString(*it,chString);
          ETG_TRACE_USR4(("fc_sxm_trTheatreList::ConvertToUpper string:vUnicodeToString u32val = 0x%x , chString = %s", *it, chString.c_str()));
          strFinal.append(chString);
    strSource = strFinal;

   }
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::ConvertToUpper string = %s",strSource.c_str()));
}
/*
   Method to Retrieve Movies List containing MovieName
*/

tU32 fc_sxm_trTheatreList::u32SearchMoviesList(midw_ext_fi_tclString trSearchMovieName,midw_ext_fi_tcl_MoviesList &oMovieList)
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetMoviesList"));
   tU32 u32MoviesCount=0;

   if(0 == _mUSMoviesList.size() )
   {
      return u32MoviesCount;
   }
   /* Iterate through the Movies List of particular theater entry*/
   SXM_FOREACH_MAP(MOVIE_ID,fc_sxm_trMovieDetails,iter,_mUSMoviesList)
   {
      /* Local Movie Info copy to hold Movies details*/
      midw_ext_fi_tcl_MovieInfo oMovieInfo;

      /* Retrieve required movie details from our theater entry*/
      oMovieInfo.MOVIE_ID = iter->second.movie_id;
      fc_sxm_vString2Fi( iter->second.movieName.c_str() , oMovieInfo.MovieName );
      fc_sxm_vString2Fi( iter->second.movieRating.sMovieRatingTesxt.c_str()  , oMovieInfo.MovieRating );

      /* Check this theater's movies contains our movie name */
      if(TRUE == bSearchFound(trSearchMovieName.szValue,oMovieInfo.MovieName))
      {
         /* OK,We found that this movie contains our movie name*/
         ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32SearchMoviesList::ADD: MOVIE_ID=%d MovieName=%30s MovieRating=%15s MovieRatingenum = %d",
            (tU32)(oMovieInfo.MOVIE_ID),
			(tString)(oMovieInfo.MovieName),
			(tString)(oMovieInfo.MovieRating),
            iter->second.movieRating.enMovieRating
           ));

         /* OK,Add it to Method Result*/
         oMovieList.MoviesList.push_back(oMovieInfo);
         
         /* increment movies count*/
         u32MoviesCount++;

      }
   }

   return u32MoviesCount;
}
/*
   this method will return distance b/w two locations.
*/

tF32 fc_sxm_trTheatreList::fGetFIDistance(fc_sxm_trTheatreListEntry const *prEntry) const
{
   /* Asign Distance value between target to this theater */
    /* peha:todo: we should either use km or miles in fc_sxm */
    return fc_sxm_fKm2Miles(fc_sxm_fGetLocationDistKm(_hLocation,prEntry->hLocation));
}
/*
   this method will retrieve Movie Show Times from our internal list to FI
*/

tVoid fc_sxm_trTheatreList::vGetMovieShowTimesList(vector<UN32,allocator<UN32> > internalMovieTimes,
                                              midw_ext_fi_tcl_MovieTimesList &FIMovieTimesList) const
{
   /* Retrieve Movie Times info */
   struct tm sLocalTime; 
   TIME_T tSecs;
   tS32 s32Today;
   tBool bOneDay = FALSE;
   midw_ext_fi_tcl_MovieTimes trMovieTimes;
   midw_ext_fi_tcl_ShowTimes trRedundantTimes;

   if(0 == internalMovieTimes.size())
   {
      /* There are no Movie Times */
      ETG_TRACE_ERR(("fc_sxm_trTheatreList::vGetMovieShowTimesList No Show Times"));
      return;
   }
 
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::vGetMovieShowTimesList 1st Time Info=%d",internalMovieTimes.front() ));
   /* Get First Value */
   tSecs = internalMovieTimes.front(); 
   OSAL.gmtime_r(&tSecs, &sLocalTime);
   s32Today = sLocalTime.tm_mday;

   vector<UN32>::iterator pMovieTimesIte = internalMovieTimes.begin(); 

   /* Check this First Value with the list elements*/
   //SXM_FOREACH(vector<UN32> , pMovieTimesIte , internalMovieTimes)
   // {
   while(pMovieTimesIte != internalMovieTimes.end())
   {
      tSecs = *pMovieTimesIte; 
      OSAL.gmtime_r(&tSecs, &sLocalTime);

      ETG_TRACE_USR4(("fc_sxm_trTheatreList::vGetMovieShowTimesList Day=%d,Hour=%d,Min=%d",sLocalTime.tm_mday,sLocalTime.tm_hour,sLocalTime.tm_min ));
      ETG_TRACE_USR4(("fc_sxm_trTheatreList::vGetMovieShowTimesList s32Today=%d",s32Today ));

      if(sLocalTime.tm_mday == s32Today)
      {
         /* OK,Retrieve the required values*/
         trMovieTimes.ShowDate.Date = (tU16)(sLocalTime.tm_mday);
         trMovieTimes.ShowDate.Month = (tU16)(sLocalTime.tm_mon+1u);
         trMovieTimes.ShowDate.Year = sLocalTime.tm_year+1900;

         midw_ext_fi_tcl_ShowTimes trShowTimes;
         trShowTimes.Hour = (tU16)(sLocalTime.tm_hour);
         trShowTimes.Minutes = (tU16)(sLocalTime.tm_min);
         
         if(trRedundantTimes == trShowTimes)
            {
               ETG_TRACE_ERR(("fc_sxm_trTheatreList::vGetMovieShowTimesList Ignoring Redundant ShowTimes" ));
            }
         else
            {
               trRedundantTimes = trShowTimes;
               trMovieTimes.ShowTimes.push_back(trShowTimes);
            }

         bOneDay = TRUE; 
         pMovieTimesIte++;

      }
      else
      {
         /* OK,Show Times are of More than One day*/
         bOneDay = FALSE;
         
         /* Reset Redundant Times for next day */
         trRedundantTimes.Hour = 0;
         trRedundantTimes.Minutes = 0;

         ETG_TRACE_USR4(("fc_sxm_trTheatreList::vGetMovieShowTimesList Pushing to MovieTimes" ));
         /* Copy this into FI*/
         FIMovieTimesList.MovieTimes.push_back(trMovieTimes);
         /* Change Today to next value*/
         s32Today = sLocalTime.tm_mday;
         /* Clear the Show Times for the Next Day*/
         trMovieTimes.ShowTimes.clear();

         ETG_TRACE_USR4(("fc_sxm_trTheatreList::vGetMovieShowTimesList Show Times Cleared" ));
      }
   }

   if(bOneDay == TRUE)
   {
      FIMovieTimesList.MovieTimes.push_back(trMovieTimes);
   }

   /* Print Movie Show TImes */
   for(tU32 i=0;i<FIMovieTimesList.MovieTimes.size();i++){

      for(tU32 j=0;j<FIMovieTimesList.MovieTimes.at(i).ShowTimes.size();j++){

         ETG_TRACE_USR4(("Movie ShowTimes: YEar=%d,Month=%d,Day=%d - %d::%d TimesCount=%d", 
                     FIMovieTimesList.MovieTimes.at(i).ShowDate.Year,
                     FIMovieTimesList.MovieTimes.at(i).ShowDate.Month,
                     FIMovieTimesList.MovieTimes.at(i).ShowDate.Date,
                     FIMovieTimesList.MovieTimes.at(i).ShowTimes.at(j).Hour,
                     FIMovieTimesList.MovieTimes.at(i).ShowTimes.at(j).Minutes,
                     FIMovieTimesList.MovieTimes.at(i).ShowTimes.size() ));
      }
   }

}
/*
   Method to update a movie into our internal list
*/
tBool fc_sxm_trTheatreList::bUpdateMovieEntry(fc_sxm_trMovieDetails const &MovieDetails)
{

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::bUpdateMovieEntry IN"));

   MOVIE_ID n16MovieID = MovieDetails.movie_id;
   
   /* check if val exist */
   map<MOVIE_ID, fc_sxm_trMovieDetails>::iterator iter = _mUSMoviesList.find(n16MovieID);

   tBool bExists=FALSE;

   /* Check if val matches to end of list*/
   if (iter!=_mUSMoviesList.end()) {
      bExists=TRUE;
   }

   /* If val exists */
   if (bExists) 
   {
      ETG_TRACE_USR4(("fc_sxm_trTheatreList:bUpdateMovieEntry updated"));

      /* write new value */
      iter->second=MovieDetails;
   }
   else 
   {
      ETG_TRACE_USR4(("fc_sxm_trTheatreList:bUpdateMovieEntry new"));

      /* insert new value */
      _mUSMoviesList.insert(make_pair(n16MovieID,MovieDetails)).first;
   }

   return bExists;
}
/*
   Method to Get US movies list Count 
*/
tU32 fc_sxm_trTheatreList::u32GetUSMoviesListCount()
{
   ETG_TRACE_USR4(("fc_sxm_trTheatreList::u32GetUSMoviesListCount"));
   return (tU32)(_mUSMoviesList.size());
}
/*
   Method to delete/clear complete US movies list.
*/
tVoid fc_sxm_trTheatreList::vClearMovieList()
{
   ETG_TRACE_USR4(( "fc_sxm_trTheatreList::vClearMovieList Size=%d",_mUSMoviesList.size() ));
   /* Check for empty case*/
   if( 0 == _mUSMoviesList.size())
   {
      return;
   }

   /* Clear US movies Map list*/
   _mUSMoviesList.clear();
}
/*
   this method will convert Time from int type value to string type.
*/
string fc_sxm_trTheatreList::sGetMovieRunTime(UN8 const un8MovieDuration)
{
   string sMovieDuration;
   unsigned int iTemp;

   if(un8MovieDuration == 0)
   {
      sMovieDuration.append("0 h 0 min");
      return sMovieDuration;
   }
   /* Convert to Hours*/
   iTemp = un8MovieDuration/60;
   if(iTemp > 0)
   {
      sMovieDuration.append(sConvertInt2Str(iTemp));
      sMovieDuration.append(" h ");
   }

   /* Convert to Minutes*/
   iTemp = un8MovieDuration % 60;
   if(iTemp > 0)
   {
      sMovieDuration.append(sConvertInt2Str(iTemp));
      sMovieDuration.append(" min");
   }

   ETG_TRACE_USR4(("fc_sxm_trTheatreList::sGetMovieRunTime MovieDuration=%s",sMovieDuration.c_str() ));

   return sMovieDuration;
}
/*
helper method to convert a int value to string value
*/
string fc_sxm_trTheatreList::sConvertInt2Str(unsigned int number)
{
    string temp="";
    string returnvalue="";

    /* Take last value & convert */
    while (number>0)
    {
        temp = temp + (tChar)((number%10u)+48u);   /* add '0' ASCII Value(48) */
        number/=10;
    }

    /* Reverse the String */
    for (unsigned int i=0;i<temp.length();i++)
        returnvalue += temp[(temp.length()-i)-1];

    /* Return the final string*/
    return returnvalue;
}
