/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_movies_app.cpp
* @brief       Application handler implementation for Movies service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_movies.h"
#include "fc_sxm_tcl_movies_app.h"
#include "fc_sxm_tcl_movies_dsrl.h"
#include "fc_sxm_sms_util.h"

typedef enum {
   fc_sxm_enTtfisCmdsMovies_MSetListsConfig=1,
   fc_sxm_enTtfisCmdsMovies_MStartGetTheatreList=2,
   fc_sxm_enTtfisCmdsMovies_MStartGetTheatreInfo=3,
   fc_sxm_enTtfisCmdsMovies_MStartGetMoviesList=4,
   fc_sxm_enTtfisCmdsMovies_MStartGetMovieInfo=5,
   fc_sxm_enTtfisCmdsMovies_MStartAddFavorite=6,
   fc_sxm_enTtfisCmdsMovies_MStartRemoveFavorite=7,
   fc_sxm_enTtfisCmdsMovies_MStartSearchTheater=8,
   fc_sxm_enTtfisCmdsMovies_MStartSearchMovie=9,
   fc_sxm_enTtfisCmdsMovies_PSetMode=10

} fc_sxm_tenTtfisCmdsMovies;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_MOVIES_APP
#include "trcGenProj/Header/fc_sxm_tcl_movies_app.cpp.trc.h"
#endif

/*
  Implementation of all callback-functions for the sms needed by movies.
  Only communication to audio-app:
  1.) Posting messages using fc_sxm_tclMoviesApp::instance()->vPostMsgNew(rMsg);
  2.) writing to semaphore-protected data (fc_sxm_tclMoviesProperties)
*/


/* 
   peha: constructor: give it a name and provice the used sxmServiceId (not the ccaServiceId!!)
 */
fc_sxm_tclMoviesApp::fc_sxm_tclMoviesApp():
    fc_sxm_tclDataApp("sxmMoviesApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_MOVIES, "movies", TR_CLASS_FC_SXM_MOVIES_APP)
{
    ETG_TRACE_USR4(("fc_sxm_tclMoviesApp constructor"));

    /* Initialize the class members */
   _hMovies = MOVIES_SERVICE_INVALID_OBJECT;
   _poNearByDsrl=OSAL_NULL;
   _poFavoriteDsrl=OSAL_NULL;
   _poNearbyTheatreList = OSAL_NULL;
   _poFavoriteTheatreList = OSAL_NULL;
   _u32NearbyCapacity = 0;
   _u32NearbyRadius = 0;

}

/*
  Destructor
*/

fc_sxm_tclMoviesApp::~fc_sxm_tclMoviesApp()
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp destructor"));

      FC_SXM_MEMORY_CLEAR(_poNearByDsrl);
      FC_SXM_MEMORY_CLEAR(_poFavoriteDsrl);
      FC_SXM_MEMORY_CLEAR(_poNearbyTheatreList);
      FC_SXM_MEMORY_CLEAR(_poFavoriteTheatreList);

      /* Clear the US movies List if any*/
      fc_sxm_trTheatreList::vClearMovieList();


   _hMovies = MOVIES_SERVICE_INVALID_OBJECT;

}

/* 
	Attention: called in cca-context by sxmapp-manager 
*/
tVoid fc_sxm_tclMoviesApp::vInitialize(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vInitialize"));
   //Create the movies worker thread

   /* call vInitialize of base-class to start the worker thread */
   fc_sxm_tclBaseApp::vInitialize();

   fc_sxm_trMoviesDSRLCfg rDsrlCfg;
   fc_sxm_trTheatreListConfig rListCfg;


   /* configure FAVORITES*/
   // DSRL
   rDsrlCfg.enSortMethod = fc_sxm_enMoviesSortMethod_NONE;
   rDsrlCfg.u32Capacity= FC_SXM_MOVIES_FAVORITE_DSRL_CAPACITY;
   rDsrlCfg.u32Radius  = FC_SXM_MOVIES_FAVORITE_DSRL_RADIUS;
   // LIST
   rListCfg.rSortCfg.enSortMethod=fc_sxm_enMoviesSortMethod_FIRST_SAVED_FIRST;
   rListCfg.rFilterCfg.enFilterMethod = fc_sxm_enMoviesFilterMethod_NONE;
   
   /* Create the DSRL and List for Favorites*/
   vCreateDsrlAndList(fc_sxm_enDSRLType_Favorites, fc_sxm_tenListType_Favorites, 
                       &_poFavoriteDsrl,rDsrlCfg,
                       &_poFavoriteTheatreList,rListCfg);


   /* configure NEARBY */
   // DSRL
   rDsrlCfg.enSortMethod = fc_sxm_enMoviesSortMethod_NONE;
   rDsrlCfg.u32Capacity= FC_SXM_MOVIES_DEVICE_DSRL_CAPACITY;
   rDsrlCfg.u32Radius  = FC_SXM_MOVIES_DEVICE_DSRL_RADIUS;
   // LIST
   rListCfg.rSortCfg.enSortMethod=fc_sxm_enMoviesSortMethod_DISTANCE;
   rListCfg.rFilterCfg.enFilterMethod = fc_sxm_enMoviesFilterMethod_NONE;

   /* Create the DSRL and List for Nearby*/
   vCreateDsrlAndList(fc_sxm_enDSRLType_Device, fc_sxm_tenListType_NearBy, 
                      &_poNearByDsrl,rDsrlCfg,
                      &_poNearbyTheatreList,rListCfg);

   // todo: _u32NearbyCapacity ... still needed?
   _u32NearbyCapacity = rDsrlCfg.u32Capacity;     //MOVIE_CAPACITY;
   _u32NearbyRadius =rDsrlCfg.u32Radius;         //MOVIE_RADIUS;
}

/* 
	Attention: called in cca-context by sxmapp-manager 
*/

tVoid fc_sxm_tclMoviesApp::vDeInitialize(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vDeInitialize"));

	/* call vDeInitialize() of base-class to stop the worker thread */
   fc_sxm_tclBaseApp::vDeInitialize();

   /* Deinit of all other stuff starts from here */
}


/* 
   this method is called by the thread-function.
   For each message that shall be handled put and entry here.
*/
tVoid fc_sxm_tclMoviesApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {

	/* Get Action Value */
   tU32 u32Action=poThreadMsg->u32GetAction();

   switch (u32Action) {

      /* dispatch Movies Service State mesaage to CCA*/
   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMStartSetListsConfig,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMStartGetTheatreList,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMStartGetTheaterInfo,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMStartGetMoviesList,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMStartGetMovieInfo,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMAddFavorite,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMRemoveFavorite,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgMoviesMSearchMovie,fc_sxm_tclMoviesApp::instance());

   SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trPropertyMoviesPSetListMode,fc_sxm_tclMoviesApp::instance());

		default:
         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                            u32Action));
         break;         
   }
}

tVoid fc_sxm_tclMoviesApp::vCreateDsrlAndList(fc_sxm_tenDSRLType enDsrlType, 
                                              fc_sxm_tenListType enListType, 
                                              fc_sxm_tclMoviesDSRL **ppoDsrl,fc_sxm_trMoviesDSRLCfg rDsrlCfg,
                                              fc_sxm_trTheatreList **ppoTheaterList,fc_sxm_trTheatreListConfig const &rListCfg) {


   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vCreateDsrlAndList: enDsrlType=%d enListType=%d",
                   ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType),
                   ETG_CENUM(fc_sxm_tenListType, enListType)
                   ));
   
   fc_sxm_tclMoviesDSRL *poDsrl;
   /* create DSRL, dsrl will remain inactive while not configured */
	poDsrl=OSAL_NEW fc_sxm_tclMoviesDSRL(enDsrlType);

   /* register the dsrl */
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::poRegisterDsrl"));
   *ppoDsrl = poDsrl;
   poRegisterDsrl(poDsrl);

   fc_sxm_trTheatreList *poTheatreList;
   /* create list */
   poTheatreList=OSAL_NEW fc_sxm_trTheatreList((tU32)enListType, enListType, this);
   *ppoTheaterList = poTheatreList;

   if(OSAL_NULL!=poDsrl){
      /* Then Create DSRL*/
       vPatchDsrlConfig(enDsrlType, rDsrlCfg);
       poDsrl->vSetNextConfig(rDsrlCfg);
   }
   if(OSAL_NULL!=poTheatreList){
   /* Set the DSRL to List*/
      poTheatreList->vSetDsrl(poDsrl);
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
        fc_sxm_trTheatreList::vListOperation(poTheatreList->u32GetId(), fc_sxm_enListOperation_Configure, fc_sxm_trAdressing(), rListCfg);
#else
        // old method will also work for new list handling:
        poTheatreList->vSetCfg(rListCfg);
#endif
        
   }
   /* Then Create DSRL*/

	ETG_TRACE_USR4(("Please Set Device Location...."));
}

/* all messages are handled via vProcess(tMsgType *prMsg) */


/*
  handle command from sxmAppManager to start service (Mandatory)
*/
tVoid fc_sxm_tclMoviesApp::vStartSmsService()
{
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::bStartService"));

   /* Check if the movies service is already started */
   if (_hMovies != MOVIES_SERVICE_INVALID_OBJECT)
   {
     ETG_TRACE_USR4(("Movies service already started!"));
     return;
   }


   /* Start the movies service */
   _hMovies = MOVIES.hStart(FC_SXM_SRH_DRIVER_NAME,
                            DATASERVICE_EVENT_ALL,
                            cb_vDataServiceEventCallback,	//CallBack Function
                            (tVoid*)this, 
                            DATASERVICE_OPTION_NONE);

   /* Check for Valid Movies Service Object */
   if (_hMovies == MOVIES_SERVICE_INVALID_OBJECT)
   {
     ETG_TRACE_USR4(("Could not start Movies service"));
   }

}
/*
  handle command from sxmAppManager to Load favorites on Service start
*/
tVoid fc_sxm_tclMoviesApp::vRestoreFavorites()
{
    //call vLoadFavorites of base-class to fetch Favorite list form Persistent
    vLoadFavorites();
}
/*
  method will be called after Service Stop,to Save Favorite list to Persistent
*/
tVoid fc_sxm_tclMoviesApp::vPersistFavorites()
{
    ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vPersistFavorites"));

    //call base-class to update Favorite list to Persistent
    fc_sxm_tclDataApp::vSaveFavorites();
}
/*
   Message to handle TTFIs commands for movies app
*/
tVoid fc_sxm_tclMoviesApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) {

   fc_sxm_tenTtfisCmdsMovies enMsgCode=(fc_sxm_tenTtfisCmdsMovies)prMsg->u8MsgCode;
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) code=%d",
                 ETG_CENUM(fc_sxm_tenTtfisCmdsMovies, enMsgCode)));
   tU8 const *au8Data=prMsg->au8Data;

   switch (enMsgCode) 
   {
      /* Lists Config Method */
      case fc_sxm_enTtfisCmdsMovies_MSetListsConfig:
      {
         midw_ext_sxm_moviesfi_tclMsgSetListsConfigMethodStart oMStart;

         //oMStart.Config.Capacity=au8Data[0]; TODO : Currently We are not going to configure Capacity through FI.
         //oMStart.SortType.enType=(midw_ext_fi_tcl_e8_SortType::tenType)au8Data[1];

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd ListsConfig capacity=%d ",
                         oMStart.Config.Capacity
                         ));
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;
      /* Get TheaterList Method */
      case fc_sxm_enTtfisCmdsMovies_MStartGetTheatreList:
      {
         midw_ext_sxm_moviesfi_tclMsgGetTheatreListMethodStart oMStart;

         /* Retrive HMI inputs for ListID & MoviesID*/
         oMStart.ListID = au8Data[0];
         oMStart.MovieID = au8Data[1];

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd GetTheatreLis listId=%d ",
                         oMStart.ListID
                         ));
         
         /* Send Self Message to Handle GetTheatreListMethodStart*/
         fc_sxm_vSendSelfMessage(oMStart);
      }
      break;
      /* Get TheaterInfo Method */
      case fc_sxm_enTtfisCmdsMovies_MStartGetTheatreInfo:
      {
         midw_ext_sxm_moviesfi_tclMsgGetTheatreInfoMethodStart oMStart;
         
         /* Copy ListID & TheaterID params from TTFis*/
         oMStart.ListID= au8Data[0];
         oMStart.THEATRE_LOCID=(tS32)SXM_GET_U32(&au8Data[1]);

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetTheatreInfo listId=%d itemId=%d",
                         oMStart.ListID,
                         oMStart.THEATRE_LOCID
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      /* Get MoviesList Method */
      case fc_sxm_enTtfisCmdsMovies_MStartGetMoviesList:
      {
         midw_ext_sxm_moviesfi_tclMsgGetMoviesListMethodStart oMStart;
         
         /* Copy ListID & TheaterID params from TTFis*/
         oMStart.ListID=au8Data[0];
         oMStart.Theater_LOCID=(tS32)SXM_GET_U32(&au8Data[1]);

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetMoviesList listId=%d itemId=%d",
                         oMStart.ListID,
                         oMStart.Theater_LOCID
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      /* Get MovieInfo Method */
      case fc_sxm_enTtfisCmdsMovies_MStartGetMovieInfo:
      {
         midw_ext_sxm_moviesfi_tclMsgGetMovieInfoMethodStart oMStart;
         
         /* Copy ListID & TheaterID params from TTFis*/
         oMStart.ListID=au8Data[0];
         oMStart.MOVIE_ID=(tS32)SXM_GET_U32(&au8Data[1]);
         oMStart.Theater_LOCID=(tS32)SXM_GET_U32(&au8Data[5]);

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetMovieInfo MovieID=%d listId=%d LocId=%d",
                         oMStart.MOVIE_ID,
                         oMStart.ListID,
                         oMStart.Theater_LOCID
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      /* Add Favorite Method */
	   case fc_sxm_enTtfisCmdsMovies_MStartAddFavorite:
      {
         midw_ext_sxm_moviesfi_tclMsgAddFavoriteMethodStart oMStart;
         
         /* Copy TheaterID param from TTFis*/
         oMStart.Theater_LOCID=(tS32)SXM_GET_U32(&au8Data[0]);

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetMovieInfo LocId=%d",
                         oMStart.Theater_LOCID
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      /* Remove favorite Method */
      case fc_sxm_enTtfisCmdsMovies_MStartRemoveFavorite:
      {
         midw_ext_sxm_moviesfi_tclMsgRemoveFavoriteMethodStart oMStart;
         
         /* Copy TheaterID param from TTFis*/
         oMStart.Theater_LOCID=(tS32)SXM_GET_U32(&au8Data[0]);

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetMovieInfo LocId=%d",
                         oMStart.Theater_LOCID
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      case (fc_sxm_tenTtfisCmdsMovies)fc_sxm_enGenricTtfisCmd_StartDataService:
      {
         tBool bStart=au8Data[0];
         ETG_TRACE_USR4(("fc_sxm_tenTtfisCmdsMovies::fc_sxm_enGenricTtfisCmd_StartDataService bStart=%d",
                         bStart));
         if (bStart) {
             fc_sxm_trMsgCmdDataAppStartService rMsg;
            
             fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);
         }
         else {
             fc_sxm_trMsgCmdDataAppStopService rMsg;
             
             fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);
         }
      }
      break;
      case fc_sxm_enTtfisCmdsMovies_MStartSearchTheater:
      {
         midw_ext_sxm_moviesfi_tclMsgGetTheatreListMethodStart oMStart;
         oMStart.ListID = 1;
         oMStart.MovieID = 0;
         
         /* Copy TheaterID param from TTFis*/
         //fc_sxm_vString2Fi(au8Data[0],oMStart.TheatreName);
         tU8 SearchOption = (tU8)au8Data[0];

         switch(SearchOption)
         {
            case 1:
               fc_sxm_vString2Fi("IMAX",oMStart.SearchString);
               break;
            case 2:
               fc_sxm_vString2Fi(" City ",oMStart.SearchString);
               break;
            case 3:
               fc_sxm_vString2Fi("14",oMStart.SearchString);
               break;
            case 4:
               fc_sxm_vString2Fi("bir",oMStart.SearchString);
               break;
            case 5:
               fc_sxm_vString2Fi("Theater",oMStart.SearchString);
               break;
            case 6:
               fc_sxm_vString2Fi("",oMStart.SearchString);
               break;
            case 7:
               fc_sxm_vString2Fi("T",oMStart.SearchString);
               break;
            case 8:
               fc_sxm_vString2Fi("h",oMStart.SearchString);
               break;
            case 9:
               fc_sxm_vString2Fi("t",oMStart.SearchString);
               break;
            default:
               fc_sxm_vString2Fi("Theater",oMStart.SearchString);
               break;
         }

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartSearchTheater TheaterName=%s",
                         oMStart.SearchString.szValue
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;
      case fc_sxm_enTtfisCmdsMovies_MStartSearchMovie:
      {
         midw_ext_sxm_moviesfi_tclMsgSearchMovieMethodStart oMStart;
         
         /* Copy TheaterID param from TTFis*/
         //fc_sxm_vString2Fi(au8Data[0],oMStart.TheatreName);
         tU8 SearchOption = (tU8)au8Data[0];

         switch(SearchOption)
         {
            case 1:
               fc_sxm_vString2Fi("3D",oMStart.MovieName);
               break;
            case 2:
               fc_sxm_vString2Fi("Evil",oMStart.MovieName);
               break;
            case 3:
               fc_sxm_vString2Fi("te",oMStart.MovieName);
               break;
            case 4:
               fc_sxm_vString2Fi("The Karate Kid",oMStart.MovieName);
               break;
            case 5:
               fc_sxm_vString2Fi("Lottery",oMStart.MovieName);
               break;
            default:
               fc_sxm_vString2Fi(" ",oMStart.MovieName);
               break;
         }

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd MStartGetMovieInfo TheaterName=%s",
                         oMStart.MovieName.szValue
                         ));
         
         fc_sxm_vSendSelfMessage(oMStart);

      }
      break;

      /* for list mode */
      case fc_sxm_enTtfisCmdsMovies_PSetMode:
      {
         midw_ext_sxm_moviesfi_tclMsgSxmListModeSet oPSet;
         
         /* Copy TheaterID param from TTFis*/
         oPSet.ListMode.ListID =(tU32)au8Data[0];
         //int iMode = au8Data[1];
         oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType)au8Data[1]; 

         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd PSetListId ListID=%d",
                         oPSet.ListMode.ListID
                         ));
         ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::ttfis-cmd PSetListMode ListMode=%d",
                         oPSet.ListMode.Mode.enType
                         ));
         
         fc_sxm_vSendSelfMessage(oPSet);
      }
      break;

      default:
         ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                        ETG_CENUM(fc_sxm_tenTtfisCmdsMovies, enMsgCode)));
      break;
   }
}
/*
  Method To Get complete movies list playing in US from SMS
*/
tVoid fc_sxm_tclMoviesApp::vGetUSMoviesList()
{
   ETG_TRACE_USR4((" vProcess GetUSMoviesList IN"));

   /* Local Var to hold Return Code val*/
   SMSAPI_RETURN_CODE_ENUM eReturnCode;

   /* Get US Movies List from SMS */
   eReturnCode = MOVIES.eIterate(_hMovies,bIterateUSMovies,
                    NULL);

   /* Check return code for No entries */
   if (SMSAPI_RETURN_CODE_NO_OBJECTS == eReturnCode)
   {
      ETG_TRACE_ERR(("No MOvies"));
      return;
   }
   /* Check return code for Error*/
   else if (SMSAPI_RETURN_CODE_SUCCESS != eReturnCode)
   {
      ETG_TRACE_ERR(("Error in Retrieving Movies"));
      return;
   }

   ETG_TRACE_USR4((" vProcess GetUSMoviesList OUT"));
}
/*
  Handle method-starts
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartSetListsConfig  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgSetListsConfigMethodStart const &oMStart=prMsg->oFiMsg;
   (tVoid)oMStart; /* This is added to avoid Lint Warning,must be removed if you are using it later */

   //_u32Capacity = oMStart.Config.Capacity;


   /* Prapare emthod result with success Status*/
   midw_ext_sxm_moviesfi_tclMsgSetListsConfigMethodResult oMRes;

   /* We are going to change the capacity only for Near by dsrl*/
   /* And Check if _poNearByDsrl is created */
   if( OSAL_NULL != _poNearByDsrl){

      fc_sxm_trMoviesDSRLCfg rDsrlCfg=_poNearByDsrl->rGetNextConfig();
      rDsrlCfg.u32Capacity=oMStart.Config.Capacity;
      ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartSetListsConfig) call vSetNextConfig: capacity=%d " ,
                        rDsrlCfg.u32Capacity
                        ));
      /* So Set new capacity to NearBy DSRL*/
      _poNearByDsrl->vSetNextConfig(rDsrlCfg);
      /* Set the Method reult to true*/
      oMRes.Status = TRUE;
   }else{
      /* Set the Method reult to False such that Capacity not been set*/
      oMRes.Status = FALSE;
   }

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartSetListsConfig) call enSendFiMessage"));

   /* Send the method result to our service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartSetListsConfig) END"));

}
/*
  Method to Handle GetTheatreList
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheatreList  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgGetTheatreListMethodStart const &oMStart=prMsg->oFiMsg;

   /* Retrieve the available Params*/
   tU32 u32ListId = oMStart.ListID;
   tU32 u32MovieID = oMStart.MovieID;
   string SearchName = oMStart.SearchString.szValue;
   
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheatreList) ListId=%d,MovieID=%d,SearchName=%s",u32ListId,u32MovieID,SearchName.c_str()));

   fc_sxm_trTheatreListConfig rListCfg;
   fc_sxm_tenListType enListType = (fc_sxm_tenListType)u32ListId;
   switch(enListType)
      {
      case fc_sxm_tenListType_NearBy:
         rListCfg.rSortCfg.enSortMethod = fc_sxm_enMoviesSortMethod_DISTANCE;
         break;
      case fc_sxm_tenListType_Favorites:
         rListCfg.rSortCfg.enSortMethod = fc_sxm_enMoviesSortMethod_FIRST_SAVED_FIRST;
         break;
      default:
         rListCfg.rSortCfg.enSortMethod = fc_sxm_enMoviesSortMethod_NONE;
         break;
      }
   //save the configuration value sent by HMI
   rListCfg.rFilterCfg.u32MovieID = u32MovieID;
   if(0 == u32MovieID)
   {
      /* Initiate Call to Send the List to FI*/
      rListCfg.rFilterCfg.enFilterMethod = fc_sxm_enMoviesFilterMethod_NONE;
   }
   else
   {
      rListCfg.rFilterCfg.enFilterMethod = fc_sxm_enMoviesFilterMethod_MOVIEID;
   }
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheatreList) SearchName.u32GetSize()=%d",SearchName.size()));

   if(SearchName.size() > 0)     
   {
       ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheatreList) SearchName Changed"));
       rListCfg.rFilterCfg.SearchName = SearchName;
       rListCfg.rFilterCfg.enFilterMethod = fc_sxm_enMoviesFilterMethod_NAME;
   }
   /* find the correct list by ListID */
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(u32ListId);
   
   /* Check poList for NULL type*/

#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
   fc_sxm_trTheatreList::vListOperation(u32ListId, fc_sxm_tclDataListBase::enListOperationFromFi(oMStart.Operation), prMsg->rAdressing, rListCfg);
#else     
   if (OSAL_NULL != poList) {
       poList->vSetCfg(rListCfg);
       vSendList(u32ListId, prMsg->rAdressing);
   }
#endif


   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheatreList) END"));

}
/*
  Method to Handle GetTheaterInfo
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheaterInfo  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgGetTheatreInfoMethodStart const &oMStart=prMsg->oFiMsg;
   tU32 u32LocId=oMStart.THEATRE_LOCID;
   /* 
    todo: add entry to mark result as invalid when loc-id can not be found.
    this should be easier to handle than an error-message
   */
   midw_ext_sxm_moviesfi_tclMsgGetTheatreInfoMethodResult oMRes;

   oMRes.ListID=oMStart.ListID;
   
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheaterInfo) ListId=%d,LocID=%d",oMRes.ListID,u32LocId));

   /* Theater Info to be filled from our internal list*/
   midw_ext_fi_tcl_TheatreDetails &rTheatreInfo=oMRes.TheatreInfo;

   /* copy ListID to FI ListID type*/
   tU32 u32ListId=(tU32)oMStart.ListID;

   /* entry to be filled from list: */
   fc_sxm_trTheatreListEntry rEntry;

   /* find the correct list by ListID */
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(u32ListId);

   /* Check poList for NULL type*/
   if (OSAL_NULL != poList) {
     /* OK, we found the list */
     /* Check List for the LOCID*/
     if (!poList->bGetListEntry(u32LocId, rEntry)) {
                  
         ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheaterInfo) Entry not found"));

         /* Not Found LOCIND in List,So Send an Error to FI*/
         fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);

         return;

     }
   }
   else
   {
      ETG_TRACE_ERR(( "fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMStartGetTheaterInfo List NOT FOUND" ));

      /* List is not found,So Send an Error to FI*/
      fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
   }

   /* OK,found LOCID,Copy to CCA Type*/
   rTheatreInfo = rEntry.TheaterDetails;
   vCheckForInvalidString(rTheatreInfo.TheatreName);
   vCheckForInvalidString(rTheatreInfo.TheatreStreetNum);
   vCheckForInvalidString(rTheatreInfo.TheatreStreetName);
   vCheckForInvalidString(rTheatreInfo.TheatreCity);
   vCheckForInvalidString(rTheatreInfo.TheatreState);
   vCheckForInvalidString(rTheatreInfo.TheatreZip);
   vCheckForInvalidString(rTheatreInfo.TheatrePhone);
   /* Extract SMS Theater Location to FI Location*/
   rTheatreInfo.TheaterLocation = rGetTheaterLocation(rEntry.hLocation);

   /* Check for other than Favorite case*/   
   //if(u32ListId != (tU32)fc_sxm_tenListType_Favorites )
  // {
      /*Check if this theater is favorite type */
      if(TRUE == bIsFavorite(u32LocId))
      {
         ETG_TRACE_USR4(( "Theater in Fav list" ));
         rTheatreInfo.isTheaterFavorite = TRUE ;
      }
      else
      {
         ETG_TRACE_USR4(( "Theater Not in Fav list" ));
         rTheatreInfo.isTheaterFavorite = FALSE ;
      }
  // }

#if 1

   ETG_TRACE_USR4(("\n"));
   ETG_TRACE_USR4(( "<<<<<<<<<<<<<< Theater Info >>>>>>>>>>>>" ));
	ETG_TRACE_USR4(("LOC ID: %d", oMRes.TheatreInfo.Theater_LOCID));
	ETG_TRACE_USR4(("Theater Name: %s", oMRes.TheatreInfo.TheatreName.szValue));
	ETG_TRACE_USR4(("Theater Street Name: %s", oMRes.TheatreInfo.TheatreStreetName.szValue));
	ETG_TRACE_USR4(("Theater City: %s", oMRes.TheatreInfo.TheatreCity.szValue));
	ETG_TRACE_USR4(("Theater State: %s", oMRes.TheatreInfo.TheatreState.szValue));
	ETG_TRACE_USR4(("Theater Zip: %s", oMRes.TheatreInfo.TheatreZip.szValue));
	ETG_TRACE_USR4(("Theater Phone: %s", oMRes.TheatreInfo.TheatrePhone.szValue));
    ETG_TRACE_USR4(("Theater Location Lat: %d", oMRes.TheatreInfo.TheaterLocation.Lat));
    ETG_TRACE_USR4(("Theater Location Lon: %d", oMRes.TheatreInfo.TheaterLocation.Lon));

	if(oMRes.TheatreInfo.isTheaterFavorite == TRUE)
    {
        ETG_TRACE_USR4(("Entry is in Favorite List!"));
	} else {
		ETG_TRACE_USR4(("Entry is not in Favorite List!"));
	}
	rEntry.vPrintTheatreAmenities();
#endif

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetTheaterInfo)- Found LOCID - Sending MethodResult to CCA"));
   
   /* Send the Method Result to CCA*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}
/*
  Method to Handle Get Movies List
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetMoviesList  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgGetMoviesListMethodStart const &oMStart=prMsg->oFiMsg;

   tU32 u32LocId=oMStart.Theater_LOCID;
   /* 
   todo: add entry to mark result as invalid when loc-id can not be found.
   this should be easier to handle than an error-message
   */
   tU32 u32ListId = oMStart.ListID;

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetMoviesList) ListId=%d,LocID=%d",u32ListId,u32LocId));

   /* Check if this request for "List Movies" */
   if(0 == u32LocId)
   {
      /* OK,user requested for "List Movies" */
      vGetListMovies(u32ListId,prMsg->rAdressing);
   }
   else
   {
      /* OK,User Requested for Movies list playing in particular theater */
      vGetMoviesListForSelectedTheater(u32ListId,u32LocId,prMsg->rAdressing);
   }
}
/*
  Method to Handle Get Movie info
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetMovieInfo  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgGetMovieInfoMethodStart const &oMStart=prMsg->oFiMsg;
   
   /* Retrieve LocID & MovieID from message */
   tU32 u32LocId=oMStart.Theater_LOCID;
   tU32 u32MovieId=oMStart.MOVIE_ID;

   /* prepare a local method result to send to FI*/
   midw_ext_sxm_moviesfi_tclMsgGetMovieInfoMethodResult oMRes;

   tU32 u32ListId = oMRes.ListID = oMStart.ListID;

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetMovieInfo) ListId=%d,LocID=%d,MOiveID=%d",u32ListId,u32LocId,u32MovieId));

   /* entry to be filled from list: */
   fc_sxm_trTheatreListEntry rEntry;

   /* find the correct list by ListID */
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(u32ListId);

   /* Check poList for NULL type*/
   if (OSAL_NULL != poList) {
   
      if(0 == u32LocId)
      {
         /* Details of a Movie playing in any theater */
         fc_sxm_trTheatreList::vSendUSMovieInfo(u32ListId,u32MovieId,prMsg->rAdressing);
      }
      else
      {
         /* 
            OK,User requested for Movie details playing in particular theater
            Then retrieve that theater entry from our internal list
         */
         if (poList->bGetListEntry(u32LocId, rEntry)) 
         {
            /* Get Movie Info Playing in that theater*/
            if(TRUE == poList->bGetMovieInfo(u32MovieId,&rEntry,oMRes.MovieInfo) )
            {
               ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMStartGetMovieInfo Sending List to HMI" ));

               /* OK,We have found user requested movie playing in this theater*/
               /* Before Sending ,Print Movies Details once(this is for debugging only)*/
               ETG_TRACE_USR4(( "\n<<<<<<<<<<<<<< Movie Info >>>>>>>>>>>>" ));
               
               ETG_TRACE_USR4(("MOVIE ID: %d", oMRes.MovieInfo.MOVIE_ID));
               ETG_TRACE_USR4(("Movie Name: %s", oMRes.MovieInfo.MovieName.szValue ));
               ETG_TRACE_USR4(("Movie Actors: %s", oMRes.MovieInfo.MovieActors.szValue ));
               ETG_TRACE_USR4(("Movies Summury: %s", oMRes.MovieInfo.MovieSummury.szValue ));
               ETG_TRACE_USR4(("Movie Rating: %s",oMRes.MovieInfo.MovieRating.szValue ));
               ETG_TRACE_USR4(("Movie Length(String): %s", oMRes.MovieInfo.sMovieLength.szValue ));
               
               /* Finally Send the local Method Result to our Service*/
               fc_sxm_tclMoviesService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
            }
            else
            {
               ETG_TRACE_ERR(( "fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMStartGetMovieInfo MovieID=%d is not available in this theater",u32MovieId ));
               /* Not OK,Movie Not found in current theater,So Send a Error report to HMI*/
               fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing, (tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
            }

         }
         else
         {
            ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMStartGetMovieInfo) Entry not found"));

            /* Not Found theater in our internal List,So Send an Error to FI*/
            fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(( "fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMStartGetMovieInfo List NOT FOUND" ));

      /* List is not found,So Send an Error to FI*/
      fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
   }
}
/*
  Method to Handle Adding a theater to Favorites
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMAddFavorite  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgAddFavoriteMethodStart const &oMStart=prMsg->oFiMsg;
   
   /* Retrieve LocID from message */
   tU32 u32LocId = oMStart.Theater_LOCID;
   LOC_ID tLocID = (LOC_ID)u32LocId;
   
   ETG_TRACE_USR4(("vProcess fc_sxm_trMsgMoviesMAddFavorite LocID=%d",u32LocId));
   
   midw_ext_sxm_moviesfi_tclMsgAddFavoriteMethodResult oMRes;
   oMRes.Status = FALSE;

   LOCID_OBJECT hLocID;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;

   hLocID = LOCID.hCreate(tLocID);
   /* Check for Valid LocID & Movies Service Objects*/
   if(LOCID_INVALID_OBJECT != hLocID || MOVIES_SERVICE_INVALID_OBJECT != _hMovies)
   {
      /* OK,Set this LOCID as Favorite*/
      eReturnCode = FAVORITES.eSetFavoriteByLocID(hLocID,
                                             _hMovies,
                                             TRUE);
      LOCID.vDestroy(hLocID);

      /* Check return code for success*/
      if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode)
      {
         /* Intial Values to Add a Favorite Entry into our Internal list*/
        // tU32 u32NearbyListId = (tU32)fc_sxm_tenListType_NearBy;
         DSRL_ENTRY_STATUS_ENUM eStatus = DSRL_ENTRY_STATUS_NEW;

         /* entry to be filled from list: */
         fc_sxm_trTheatreListEntry rEntry;

         /* find the Nearby List*/
         fc_sxm_trTheatreList *poNearbyList=dynamic_cast<fc_sxm_trTheatreList*>(poGetActiveListbyDSRL(_poNearByDsrl));

         /* Check poList for NULL type*/
         if (OSAL_NULL != poNearbyList) {

            /* Retrieve the Theater Entry from Nearby List*/
            if(TRUE == poNearbyList->bGetListEntry(u32LocId, rEntry)){

               ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgMoviesMAddFavorite Found LocID=%d in our List",u32LocId ));
   			
	              /* Update Favorite type */
                  rEntry.TheaterDetails.isTheaterFavorite = TRUE;
				  /* Activate the Favorite List by it's DSRL */
                  poGetActiveListbyDSRL(_poFavoriteDsrl);
				  /* And Add the Entry to Favorite list */
	               fc_sxm_trTheatreList::vUpdateEntry(eStatus,rEntry);

	               //Update internal Favorite List for Sort Order
	               fc_sxm_tclDataApp::vAddFavorite(u32LocId);
         			
		            /* Finaly Say Operation is Success*/
                  oMRes.Status = TRUE;

                  ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgMoviesMAddFavorite TRUE" ));
            }
            else
            {
               ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMAddFavorite Not Found LocID=%d in our List",u32LocId ));
            }

         }
         else
         {
            ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMAddFavorite SMS TRUE but Nearby List Not exists" ));
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMAddFavorite FALSE" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMAddFavorite Invalid LOdID or Invalid Movie Service" ));
   }
   
   /* Finally Send the local Method Result to our Service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}
/*
  Method to Handle Removing a theater to Favorites
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMRemoveFavorite  const *prMsg) {

   midw_ext_sxm_moviesfi_tclMsgRemoveFavoriteMethodStart const &oMStart=prMsg->oFiMsg;

   /* Extract LocID from Method Start*/
   tU32 u32LocId = oMStart.Theater_LOCID;
   ETG_TRACE_USR4(("vProcess fc_sxm_trMsgMoviesMRemoveFavorite LocID=%d",u32LocId));
   
   /* Prepared Method Result*/
   midw_ext_sxm_moviesfi_tclMsgRemoveFavoriteMethodResult oMRes;
   oMRes.Status = FALSE;

   /* Retrieve LocID from message */
   LOC_ID tLocID = (LOC_ID)u32LocId;

   LOCID_OBJECT hLocID ;
   SMSAPI_RETURN_CODE_ENUM eReturnCode;

   /* Create LocID Object*/
   hLocID = LOCID.hCreate(tLocID);

   /* Check for Valid LocID & Movies Service Objects*/
   if(LOCID_INVALID_OBJECT != hLocID || MOVIES_SERVICE_INVALID_OBJECT != _hMovies)
   {
      /* OK,ReSet this LOCID as Favorite*/
      eReturnCode = FAVORITES.eSetFavoriteByLocID(hLocID,
                                             _hMovies,
                                             FALSE);
      LOCID.vDestroy(hLocID);

      /* Check return code for success*/
      if (SMSAPI_RETURN_CODE_SUCCESS == eReturnCode)
      {
         /* OK,hLocID is Removed from Favorites,So Now We have to Remove from our internal List*/

         /* Intial Values to Remove a Favorite Entry into our Internal list*/
         //tU32 u32ListId = (tU32)fc_sxm_tenListType_Favorites;
         DSRL_ENTRY_STATUS_ENUM eStatus = DSRL_ENTRY_STATUS_REMOVED;

         /* entry to be filled from list: */
         fc_sxm_trTheatreListEntry rEntry;

         /* find the correct list by ListID */
         fc_sxm_trTheatreList *poFavList=dynamic_cast<fc_sxm_trTheatreList*>(poGetActiveListbyDSRL(_poFavoriteDsrl));

         /* Check poList for NULL type*/
         if (OSAL_NULL != poFavList) {

            /* Retrieve the Theater Entry for corresponding LocID*/
            poFavList->bGetListEntry(u32LocId, rEntry);

            ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgMoviesMRemoveFavorite Found LocID=%d in our List",u32LocId ));

            /* OK We found the LOCID,Remove the retrieved entry from our internal List*/
            fc_sxm_trTheatreList::vUpdateEntry(eStatus,rEntry);

            //Update internal Favorite List for Sort Order
            fc_sxm_tclDataApp::vRemoveFavorite(u32LocId);

            /* Finaly Say Operation is Success*/
            oMRes.Status = TRUE;

            ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgMoviesMRemoveFavorite SMS TRUE" ));

         }
         else
         {
            ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMRemoveFavorite SMS TRUE but List Not exists" ));
         }

      }
      else
      {
         ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMRemoveFavorite FALSE" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vProcess fc_sxm_trMsgMoviesMRemoveFavorite Invalid LOdID or Invalid Movie Service" ));
   }
   
   /* Finally Send the local Method Result to our Service*/
   fc_sxm_tclMoviesService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}
/*
  Method to Handle Searching a Movie from available nearby theater list
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgMoviesMSearchMovie  const *prMsg) {

   ETG_TRACE_USR4(( "vProcess fc_sxm_trMsgMoviesMSearchMovie" ));

   midw_ext_sxm_moviesfi_tclMsgSearchMovieMethodStart const &oMStart=prMsg->oFiMsg;
   
   /* First clear our internal list */
   fc_sxm_trTheatreList::vClearMovieList();

   /* 
      And Get US movies list from SMS
   */
   vGetUSMoviesList();

	/* Check the Size Now */
   tU32 u32CurrentMovieSize = fc_sxm_trTheatreList::u32GetUSMoviesListCount();

   ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMSearchMovie US Movies List is available(%d)",u32CurrentMovieSize ));

   if(u32CurrentMovieSize)
   {   
      /* OK,we have movies list ready,now search & Send it to HMI */
      fc_sxm_trTheatreList::vSearchMovie(oMStart.MovieName,prMsg->rAdressing);
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::fc_sxm_trMsgMoviesMSearchMovie Movies not available yet"));

      /* Not Found LOCIND in List,So Send an Error to FI*/
      fc_sxm_tclMoviesService::vSendError(prMsg->rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
   }
}

/*
  Set property to Handle the List Mode
*/
tVoid fc_sxm_tclMoviesApp::vProcess(fc_sxm_trPropertyMoviesPSetListMode  const *prMsg) 
{
   ETG_TRACE_USR4(( "vProcess fc_sxm_trPropertyMoviesPSetListMode" ));

   midw_ext_sxm_moviesfi_tclMsgSxmListModeSet const &oPSet=prMsg->oFiMsg;

   /* Initialize default values*/
   tU32 u32ListId = oPSet.ListMode.ListID;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyMoviesPSetListMode: ListID %d",u32ListId));

   fc_sxm_tenListMode enListMode = (fc_sxm_tenListMode)oPSet.ListMode.Mode.enType;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyMoviesPSetListMode: enListMode %d",enListMode));
   vSetListMode(u32ListId, prMsg->rAdressing,enListMode);
   
}

#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
tVoid fc_sxm_tclMoviesApp::vProcessTimer(fc_sxm_trMsgListUpdateTimer const *prMsg) {
   (tVoid)prMsg;
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess"
                    "(fc_sxm_trMsgListUpdateTimer)"));
   vStopListTimer();
   /* find the correct list by ListID */
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vProcess _u32ListID %d",_u32ListID ));
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(_u32ListID);
   poList->vOnListTimerExpire();
}

/*
   Method to start the Listupdate Timer for movies app
*/
tVoid fc_sxm_tclMoviesApp::vStartListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vStartListTimer()"));

    _mListUpdateTimer.vStart(FC_SXM_LIST_DATA_UPDATE_PERIOD);
}
/*
   Method to stop the Listupdate Timer for movies app
*/
tVoid fc_sxm_tclMoviesApp::vStopListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vStopListTimer()"));

    _mListUpdateTimer.vStop();
}
/*
   Method to check the Listupdate Timer for movies app
*/
tBool fc_sxm_tclMoviesApp::bIsListUpdateTimerRunning(tVoid) const {
    return(_mListUpdateTimer.bIsRunning());
}

#endif


tBool fc_sxm_tclMoviesApp::bIsFavorite(tU32 u32LocID) const
{
   tU32 u32ListId = (tU32)fc_sxm_tenListType_Favorites;

   /* entry to be filled from list: */
   fc_sxm_trTheatreListEntry rEntry;

   /* find the correct list by ListID */
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(u32ListId);

   /* Check poList for NULL type*/
   if (OSAL_NULL != poList) {
     /* OK, we found the list */
     /* Check List for the LOCID*/
     return poList->bGetListEntry(u32LocID, rEntry);
   }
   else
   {
      return FALSE;
   }
}
/* 
	This Method Will be called whenever any changes happend to our List
*/
tVoid fc_sxm_tclMoviesApp::vOnListStateChanged(fc_sxm_tclDataListBase *poDataList) {

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vOnListStateChanged List State = %d",poDataList->rGetStatusInfo().enListState ));

   /* Define a Property to send to CCA*/
   fc_sxm_tcl_trMoviesPropertyListsStatus oProperty;

   /* Make local reference of the Property */
   midw_ext_sxm_moviesfi_tclMsgTheatreListsStatusStatus &oFiMsg=oProperty.oFiMsg;

   /* Assign List Capacity set in SetListsConfig Method Start*/
  // oFiMsg.TheatreListsStatus.GenericConfig.Capacity = MOVIE_CAPACITY; 

   /* Assign other List Info*/
   oGetDataListsInfo().vCopyListInfosToFi(oFiMsg.TheatreListsStatus); 

   /* Finally Send the Notification for the Property update*/
   fc_sxm_tclMoviesProperties::instance()->oListsStatus.vSet(oProperty);
   fc_sxm_tclMoviesProperties::instance()->oListsStatus.vNotify();

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vOnListStateChanged OUT"));

}
/* 
	SMS CallBack method to iterate Movie entries
*/
BOOLEAN fc_sxm_tclMoviesApp::bIterateUSMovies(
   MOVIE_OBJECT hMovie,
   void *pvEventCallbackArg
)
{
   (tVoid)pvEventCallbackArg;    // To Avoid LInt warning
   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::bIterateUSMovies IN"));

   /* Check for Invalid Movie Object */
   if(MOVIE_INVALID_OBJECT == hMovie)
   {
      ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::bIterateUSMovies Invalid Movie Object"));
      return FALSE;
   }

   fc_sxm_trMovieDetails moviedetails;

   /* Copy the movie's information */
   MOVIE_ID movieId = MOVIE.tID(hMovie);

   /* Extract Movie details from MOVIE Object*/
   moviedetails.movie_id = movieId;
   fc_sxm_vCopySmsString2Stl(MOVIE.hName(hMovie),moviedetails.movieName);
   fc_sxm_vCopySmsString2Stl(MOVIE.hActors(hMovie),moviedetails.movieActors);
   fc_sxm_vCopySmsString2Stl(MOVIE.hSynopsis(hMovie),moviedetails.movieSummury);

   MOVIE_RATING_ENUM MovieRating = MOVIE.eRating(hMovie);
   moviedetails.movieRating.enMovieRating = MovieRating;
   fc_sxm_vCopySmsString2Stl(MOVIE.hRatingText(hMovie),moviedetails.movieRating.sMovieRatingTesxt);

   moviedetails.movieLength = MOVIE.un8RunTime(hMovie);

   /* And Copy/update to our internal list*/
   fc_sxm_trTheatreList::bUpdateMovieEntry(moviedetails);

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::bIterateUSMovies OUT"));

   return TRUE;
}
/* 
   Method used to Check Theater information for Invalid strings(e.g n/a)
*/
tVoid fc_sxm_tclMoviesApp::vCheckForInvalidString(midw_ext_fi_tclString &rString) const
{
   if(OSAL_s32StringCompare((tString)rString,"n/a") == 0){
      rString = "";
   }
}
/* 
	Method to Extract SMS Location Object to FI Location type
*/
midw_ext_fi_tcl_SxmLocation fc_sxm_tclMoviesApp::rGetTheaterLocation(LOCATION_OBJECT hLocObj) const
{
   OSAL_FIXED_OBJECT oLat = LOCATION.hLat(hLocObj);
   OSAL_FIXED_OBJECT oLon = LOCATION.hLon(hLocObj);

    midw_ext_fi_tcl_SxmLocation trTheaterLoc;

    //int iLat = (int)fc_sxm_fFixed2Float(oLat);
    //int iLon = (int)fc_sxm_fFixed2Float(oLon);

    trTheaterLoc.Lat = (tS32)(fc_sxm_fFixed2Float(oLat)*1000000);    //10^6 is used to take 6 values after decimal point
    trTheaterLoc.Lon = (tS32)(fc_sxm_fFixed2Float(oLon)*1000000);

    return trTheaterLoc;
}
/* 
   Method to handle "List Movies" request
*/
tVoid fc_sxm_tclMoviesApp::vGetListMovies(tU32 u32ListId,fc_sxm_trAdressing const &rAdressing)
{
   /* First clear our internal list */
   fc_sxm_trTheatreList::vClearMovieList();

   /* Then request SMS for new Movies List */
   vGetUSMoviesList();

   /* Check the Size Now */
   tU32 u32CurrentMovieSize = fc_sxm_trTheatreList::u32GetUSMoviesListCount();
   ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::vGetListMovies US Movies List(%d)",u32CurrentMovieSize));
   
   if(u32CurrentMovieSize == 0)
   {
      ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::vGetListMovies Movies list is not available yet" ));
      /* List is not found,So Send an Error to FI*/
      fc_sxm_tclMoviesService::vSendError(rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
      return;
   }
   /* We have Movies List ready to Send to HMI*/
   fc_sxm_trTheatreList::vSendMoviesList(u32ListId,rAdressing);
}
/* 
   Method to handle Movies list playing in selected theater
*/
tVoid fc_sxm_tclMoviesApp::vGetMoviesListForSelectedTheater(tU32 u32ListId,tU32 u32LocId,fc_sxm_trAdressing const &rAdressing)
{
   midw_ext_sxm_moviesfi_tclMsgGetMoviesListMethodResult oMRes;

   //first check if US movies list already available
   if(fc_sxm_trTheatreList::u32GetUSMoviesListCount() == 0)
   {
      //No, then fetch US movies list
      vGetUSMoviesList();

      //Now,check if we got some movie entries
      if(fc_sxm_trTheatreList::u32GetUSMoviesListCount() == 0)
      {
         ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::vGetMoviesListForSelectedTheater Movies list is not available yet" ));
         /* List is not found,So Send an Error to FI*/
         fc_sxm_tclMoviesService::vSendError(rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
         return;
      }
   }

   /* Assign ListID to Method Result */
   oMRes.ListID = u32ListId;

   /* entry to be filled from list: */
   fc_sxm_trTheatreListEntry rEntry;

   /* find the correct list by ListID */
   fc_sxm_trTheatreList *poList=fc_sxm_trTheatreList::poGetTypedListById(u32ListId);

   /* Check poList for NULL type*/
   if (OSAL_NULL != poList) 
   {
      /* Get theater Entry of particular LOCID*/
      if (poList->bGetListEntry(u32LocId, rEntry)) 
      {
         /* Get Movies List Playing in that theater*/
         poList->u32GetMoviesList(&rEntry,oMRes.MoviesInfoList);


         ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::vGetMoviesListForSelectedTheater Sending List to HMI" ));

         /* Sort the retrieved movies list by their name */
         vSortList(oMRes.MoviesInfoList); 
         
         /* Print Movies List That we sending to FI (debugging purpose) */
         vPrintMovies(oMRes.MoviesInfoList);
         
         /* Finally Send the local Method Result to our Service*/
         fc_sxm_tclMoviesService::instance()->enSendFiMessage(rAdressing, oMRes);

      }
      else
      {
         ETG_TRACE_ERR(("fc_sxm_tclMoviesApp::vGetMoviesListForSelectedTheater Entry not found"));

         /* Not Found LOCIND in List,So Send an Error to FI*/
         fc_sxm_tclMoviesService::vSendError(rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
      }
   }
   else
   {
      ETG_TRACE_ERR(( "fc_sxm_tclMoviesApp::vGetMoviesListForSelectedTheater List NOT FOUND" ));

      /* List is not found,So Send an Error to FI*/
       fc_sxm_tclMoviesService::vSendError(rAdressing,(tU16)midw_ext_fi_tcl_e8MovieErrorTypes::FI_EN_UNAVAILABLE);
   }
}

/* 
	Method to Print Movies List
*/
tVoid fc_sxm_tclMoviesApp::vPrintMovies(midw_ext_fi_tcl_MoviesList const &oMoviesList) const{

   ETG_TRACE_USR4(( "\n<<<<<<<<<<<<<< Movie List >>>>>>>>>>>>" ));
   

   /* Check for 0 entries*/
   if( 0 == oMoviesList.MoviesList.size())
   {
      ETG_TRACE_ERR(("fc_sxm_tclMoviesApp Theere are no Movies to Print"));
      return;
   }

   /* Iterate through each elemnt in the list */
   SXM_FOREACH_CONST(std::vector<midw_ext_fi_tcl_MovieInfo> , pMoviesIter , oMoviesList.MoviesList)
   {
      /* Print each elemnt to Concole*/
      ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vPrintMovies:: MOVIE_ID=%d MovieName=%50s MovieRating=%20s",
            (*pMoviesIter).MOVIE_ID,
            (*pMoviesIter).MovieName,
            (*pMoviesIter).MovieRating
            ));
   }
}
/* 
	Method to Sort the List using selection Sort
*/
tVoid fc_sxm_tclMoviesApp::vSortList(midw_ext_fi_tcl_MoviesList &oMoviesList) const {

   ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vSortList IN"));

   /* Get the size of the list*/
   unsigned int iSize = (unsigned int)oMoviesList.MoviesList.size();
   
   /* Check for 0 entries*/
   if(0 == iSize){
      ETG_TRACE_ERR(("fc_sxm_tclMoviesApp Theere are no Movies to Sort"));
      return;
   }

   unsigned int i,j;

   /* use this loop to select a element to compare with remaining elements in list*/
   for (i = 0; i < (iSize-1); i++)
   {
      /* Loop will check 'i' elemtent with remining elements in list */
      for (j=i+1; j<iSize; j++)
	   {
         /* check if 'i'th elemtn is greater than 'j'th element*/
         if( OSAL_s32StringCompare(oMoviesList.MoviesList.at(i).MovieName.szValue ,oMoviesList.MoviesList.at(j).MovieName.szValue) > 0 )
         {
            /* OK, so swap both the element*/
            midw_ext_fi_tcl_MovieInfo trTemp;
            trTemp = oMoviesList.MoviesList.at(i);
            oMoviesList.MoviesList.at(i) = oMoviesList.MoviesList.at(j);
            oMoviesList.MoviesList.at(j) = trTemp;
          }

      }
   }
}
/* 
	Method to Print Local Time to Console
*/
tVoid fc_sxm_tclMoviesApp::vPrintLocalTime() const{

   struct tm sLocalTime; 
   TIME_T tSecs;
   UN32 un32OSALTimeInSecs;
   OSAL_RETURN_CODE_ENUM eReturnCode;

   /* Get current time from OSAL */
   eReturnCode = OSAL.eTimeGet(&un32OSALTimeInSecs);
   if (eReturnCode != OSAL_SUCCESS)
   {
      return;
   }
   /* Get our UTC time as a tm struct(broken down time) */
   tSecs = (TIME_T)un32OSALTimeInSecs;
   OSAL.localtime_r(&tSecs, &sLocalTime);

   ETG_TRACE_USR4(("Current Time HH::MM::SS=%2d:%2d:%2d",sLocalTime.tm_hour,sLocalTime.tm_min,sLocalTime.tm_sec ));

}
/* 
	Application get notified about the new fi-service-state that shall be sent to clients
*/
tVoid fc_sxm_tclMoviesApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState) {
    fc_sxm_tcl_trMoviesPropertySxmServiceStatus oProperty;

    ETG_TRACE_USR4(("fc_sxm_tclMoviesApp::vOnFiServiceStateChanged Service State = %d",
                    ETG_CENUM(midw_ext_fi_tcl_e8_SxmDataServiceStatus::tenType, oFiState.Status.enType) ));

    oProperty.oFiMsg.DataServiceState=oFiState;
    fc_sxm_tclMoviesProperties::instance()->oSxmServiceStatus.vSetAndNotify(oProperty);
}
/*
   Message containing clear user data
*/
tVoid fc_sxm_tclMoviesApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
{
   (void)prMsg; /* TO REMOVE LINT WARNINGS */
   ETG_TRACE_USR4(( "fc_sxm_tclMoviesApp::vProcess(fc_sxm_trMsgCmdAppClearUserData" ));

   if ( MOVIES_SERVICE_INVALID_OBJECT == _hMovies )
   {
      ETG_TRACE_ERR(( " MOVIES_SERVICE_INVALID_OBJECT " ));
      vSetUsrDataClearResult(fc_sxm_enDiagResult_Failed);
      return;
   }
   FAVORITES.vClearAllFavorites(_hMovies);
   //clear Persistent list as well.
   fc_sxm_tclDataApp::vClearFavorites();
   vSetUsrDataClearResult(fc_sxm_enDiagResult_Ok);
}
tVoid fc_sxm_tclMoviesApp::vPrintReportSpecific() {
    // for debugging purpose
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tspecific data:"));

    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tDevice DSRL Capacity  =%u", _u32NearbyCapacity));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tDevice DSRL Radius    =%u", _u32NearbyRadius));
}
