/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_fuel_list.cpp
* @brief       List implementation for fuel service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_fuel_types.h"
#include "fc_sxm_tcl_fuel_list.h"
#include "fc_sxm_fuel_fi.h"
#include "fc_sxm_service_sxm_fuel.h"
#include "fc_sxm_tcl_fuel_app.h"
#include "fc_sxm_sms_util.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_FUEL_APP
#include "trcGenProj/Header/fc_sxm_tcl_fuel_list.cpp.trc.h"
#endif

/*********************************************************************
 *
 *FUNCTION:    vPrintSpecific
 *
 *DESCRIPTION:  Print configuration
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tVoid fc_sxm_trFuelListConfig::vPrintSpecific() const {
    /* for debuging purpose*/
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenSortMethod=%d", ETG_CENUM(fc_sxm_tenFuelListSortMethod, rSortCfg.enSortMethod)));
    if (rSortCfg.enSortMethod==fc_sxm_enFuelSortMethod_PRICE) {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tenSortFuelType=%d", ETG_CENUM(FUEL_TYPE_ENUM, rSortCfg.enFuelType)));
    }
    /* print filetr by fueltype*/
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tFilter FuelType=%d", ETG_CENUM(FUEL_TYPE_ENUM, rFilterCfg.enFuelType)));
    if (rFilterCfg.sBrand!="") {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tFilter Brand=%s", rFilterCfg.sBrand.c_str()));
    }
}

/*********************************************************************
 *
 *FUNCTION:    u32GetFuelPrice
 *
 *DESCRIPTION:  Get price info
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
tU32 fc_sxm_trFuelList::u32GetFuelPrice(fc_sxm_trFuelListEntry const *prEntry) const {
      /* for NULL Check*/ 
      if (FALSE == bValidateEntry(prEntry))
      {
         ETG_TRACE_ERR(("fc_sxm_trFuelList::u32GetFuelPrice"));
         return 0;
      }

    SXM_FOREACH_MMAP_CONST(FUEL_TYPE_ENUM, fc_sxm_trFuelTypeListEntry, iter, prEntry->listFuelTypes) {
        if (iter->second.enFuelType==_rListCfg.rSortCfg.enFuelType) {
#if 0
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::u32GetFuelPrice(%u): %u",
                            _rListCfg.rSortCfg.enFuelType,
                            iter->second.u32Price));
#endif
            /*Search fuel list for standard type fuel name*/
            if(FUEL_TYPE_UNKNOWN != _rListCfg.rSortCfg.enFuelType)   {
               return iter->second.u32Price;
            }
            else
            {
               /*Search fuel list for unknown type fuel name*/
               if (0 == OSAL_s32StringCompare(iter->second.oFuelName.c_str(),
                                  _rListCfg.rFilterCfg.sFuelName.c_str())) 
               {
                  return iter->second.u32Price;
               }
            }
        }
    }

   /* todo: define marker for unknown price in fi*/
    return 0x0FFFFFFF;
}

/*********************************************************************
 *
 *FUNCTION:    bCompareItem
 *
 *DESCRIPTION:  Compare function
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if left is greater than right
 *
 ********************************************************************/
bool fc_sxm_trFuelList::bCompareItem(fc_sxm_trFuelListEntry const *prLeft, fc_sxm_trFuelListEntry const *prRight) const {
    
      /* for NULL Check*/ 
      if ((FALSE == bValidateEntry(prLeft)) ||(FALSE == bValidateEntry(prRight)))
      {
         ETG_TRACE_ERR(("fc_sxm_trFuelList::bCompareItem"));
         return false;
      }
    /* based on the sort method do the action */
    switch (_rListCfg.rSortCfg.enSortMethod) {
        case fc_sxm_enFuelSortMethod_PRICE:  /* based on price*/
        {
            tU32 u32Left=u32GetFuelPrice(prLeft);
            tU32 u32Right=u32GetFuelPrice(prRight);
            ETG_TRACE_USR4((" Fuel Sort Method PRICE"));
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::bCompareItem left:0x%08x=%u right:0x%08x=%u",prLeft->u32SxmId, u32Left, prRight->u32SxmId, u32Right));
            if (u32Left != u32Right) {
                return u32GetFuelPrice(prLeft)<u32GetFuelPrice(prRight);
            } else {
                return bCompareSmsDistance(prLeft, prRight);
            }
        }
        case fc_sxm_enFuelSortMethod_DISTANCE: /* based on Distance*/
            ETG_TRACE_USR4((" Fuel Sort Method DISTANCE"));
            return bCompareSmsDistance(prLeft, prRight);
        case fc_sxm_enFuelSortMethod_BRANDNAME:
            /*Sort by FUEL Brand Name*/ 
            ETG_TRACE_USR4((" Fuel Sort Method BRAND NAME"));
            return bCompareBrandName(prLeft, prRight);
        case fc_sxm_enFuelSortMethod_FIRST_SAVED_FIRST:
           {
              return bCompareFavoriteIndex(prLeft, prRight);
           }
        default: // default compare based on List entry id
            ETG_TRACE_USR4((" Fuel No Sort"));
            return bCompareListEntryId(prLeft, prRight);
    }
}
/*********************************************************************
 *
 *FUNCTION:    bFilterPassed
 *
 *DESCRIPTION:  Filter function
 *PARAMETER:    None
 *
 *RETURNVALUE: TRUE if item found in the list
 *
 ********************************************************************/
bool fc_sxm_trFuelList::bFilterPassed(fc_sxm_trFuelListEntry const &rItem) const {
    tBool bFuelType=FALSE;
    tBool bBrand=FALSE;
    
    ETG_TRACE_USR4(("fc_sxm_trFuelList:: fc_sxm_trFuelList::bFilterPassed Fuel Name: %d - %s",
                     _rListCfg.rFilterCfg.enFuelType,
                     _rListCfg.rFilterCfg.sFuelName.c_str()));
    /* for NULL Check*/
    if (FALSE == bValidateEntry(&rItem))
    {
         ETG_TRACE_ERR(("fc_sxm_trFuelList::bFilterPassed"));
         return false;
    }
    /*Filter all types and brands for favorite list*/
    tU32  u32GetListId = const_cast<fc_sxm_trFuelList*>(this)->u32GetId();
    if(fc_sxm_tenListType_Favorites == (fc_sxm_tenListType)u32GetListId)	{
    	ETG_TRACE_USR4(("favorite config %d ",_rListCfg.rFilterCfg.enFuelType));
    	return true;
    }
    /* based on Fuel type do the action */
    switch (_rListCfg.rFilterCfg.enFuelType)
    {
      case FUEL_TYPE_UNKNOWN: /* fuel tyep is unknown*/
          {
            SXM_FOREACH_MMAP_CONST(FUEL_TYPE_ENUM, fc_sxm_trFuelTypeListEntry, iter, rItem.listFuelTypes)
            {
               if ((0 == OSAL_s32StringCompare(iter->second.oFuelName.c_str(),
                                            _rListCfg.rFilterCfg.sFuelName.c_str())) &&
                    (TRUE==iter->second.bAvailable))
               {
                  ETG_TRACE_USR4(("fc_sxm_trFuelList:: Filter Type found %s",iter->second.oFuelName.c_str()));
                  bFuelType=TRUE;
                  break;
               }
            }        
          }
          break;
      default:
        {
            /*Get Station Info for types*/
          SXM_FOREACH_MMAP_CONST(FUEL_TYPE_ENUM, fc_sxm_trFuelTypeListEntry, iter, rItem.listFuelTypes) {
            if ((iter->second.enFuelType==_rListCfg.rFilterCfg.enFuelType)
            		&& (TRUE==iter->second.bAvailable))  {
                    ETG_TRACE_USR4(("fc_sxm_trFuelList:: Filter Type %d",iter->second.enFuelType));
                    bFuelType=TRUE;
                    break;
            }
          }
        }
        break;
    }  
    /*Filter by brand name*/
    if(_rListCfg.rFilterCfg.sBrand=="All Brands")   
    {
        bBrand=TRUE; /*Defalut brand name is All Brands*/
    }
    else  if(_rListCfg.rFilterCfg.sBrand==rItem.oBrandName)
    {
        bBrand=TRUE;
    }
    ETG_TRACE_USR4(("fc_sxm_trFuelList:: Filter passed %d",(bFuelType && bBrand)));
    return (bFuelType && bBrand);
}

tBool fc_sxm_trFuelList::bIsListDataValid()
{
   ETG_TRACE_USR4(("fc_sxm_trFuelList::bIsListDataValid"));

   tU32  u32GetListId = const_cast<fc_sxm_trFuelList*>(this)->u32GetId();
   if(fc_sxm_tenListType_Favorites == (fc_sxm_tenListType)u32GetListId)	{
    	return TRUE;
   }  
   
   //fc_sxm_trFuelListEntry trListEntry;

   SXM_FOREACH_MAP_CONST(tU32, fc_sxm_trFuelListEntry, iter, _mapEntries) {
      if (iter->second.u32NumPrices > 0) {
       return TRUE;
      }
   }
   return FALSE;
}
/*********************************************************************
 *
 *FUNCTION:    vFillListEntry
 *
 *DESCRIPTION:  Fill fuel list info
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/

tVoid fc_sxm_trFuelList::vFillListEntry(fc_sxm_trFuelListEntry const &roEntry, midw_ext_fi_tcl_FuelInfo &oFiEntry) const {
    /* for NULL Check*/
	if (FALSE == bValidateEntry(&roEntry))
    {
         ETG_TRACE_ERR(("fc_sxm_trFuelList::bFilterPassed"));
         return;
    }

    /*Get Sxm ID*/    
    oFiEntry.Fuel_LOCID=roEntry.u32SxmId;      
    /*Get Lat Lon */
    oFiEntry.FuelLoc.Lat=roEntry.oLat;
    oFiEntry.FuelLoc.Lon=roEntry.oLon;
    /*Get Distance*/
    oFiEntry.FuelDistKm=fc_sxm_fGetLocationDistKm(_hLocation,roEntry.hLocation);
    oFiEntry.FuelDistMile=fc_sxm_fKm2Miles(oFiEntry.FuelDistKm);
    /*Station Name*/                 
    fc_sxm_vString2Fi(roEntry.oStationName.c_str(),oFiEntry.StationName);       
    /*Brand Name*/
    fc_sxm_vString2Fi(roEntry.oBrandName.c_str(),oFiEntry.Brand);
	/* Send number of fuel type available for current station*/
    oFiEntry.NumPrices=roEntry.u32NumPrices;
    /*Get Station Info for types*/
    SXM_FOREACH_MMAP_CONST(FUEL_TYPE_ENUM, fc_sxm_trFuelTypeListEntry, typeiter, roEntry.listFuelTypes)
    {
        const fc_sxm_trFuelTypeListEntry &roTypeEntry=typeiter->second;
        if (roTypeEntry.enFuelType==_rListCfg.rFilterCfg.enFuelType) 
        {
            /*Get unkown fuel type and its price info*/
            if(FUEL_TYPE_UNKNOWN != _rListCfg.rFilterCfg.enFuelType)  {
               oFiEntry.FuelType.enType=(midw_ext_fi_tcl_e8_FuelTypeEnum::tenType)roTypeEntry.enFuelType;
               oFiEntry.FuelAge= (tU8)(roTypeEntry.u32AgeInDays);
               fc_sxm_vString2Fi(roTypeEntry.oFuelName.c_str(),oFiEntry.FuelName);           
               /*Get Fuel Price for the Fuel type*/
               tU32 u32Price=roTypeEntry.u32Price / 10;
               /*Round up cents */
               tU32 u32InCents=roTypeEntry.u32Price % 10;
               if((0!=u32InCents)&&(u32InCents>=5))   {
                  u32Price=u32Price+1;
               }
               char szPrice[PRINT_SIZE];
               OSALUTIL_s32SaveNPrintFormat(szPrice, PRINT_SIZE, "$%u.%.2u", u32Price / 100, u32Price % 100);        
               fc_sxm_vString2Fi((tString)szPrice,oFiEntry.FuelPrice);
               break;
            }
            else  {
             /*Search fuel list for unknown type fuel name*/
               if (0 == OSAL_s32StringCompare(roTypeEntry.oFuelName.c_str(),
                                            _rListCfg.rFilterCfg.sFuelName.c_str())) 
               {  
                  oFiEntry.FuelType.enType=(midw_ext_fi_tcl_e8_FuelTypeEnum::tenType)roTypeEntry.enFuelType;
                  oFiEntry.FuelAge= (tU8)(roTypeEntry.u32AgeInDays);
                  fc_sxm_vString2Fi(roTypeEntry.oFuelName.c_str(),oFiEntry.FuelName);           
                  /*Get Fuel Price for the Fuel type*/
                  tU32 u32Price=roTypeEntry.u32Price / 10;
                  /*Round up cents */
                  tU32 u32InCents=roTypeEntry.u32Price % 10;
                  if((0!=u32InCents)&&(u32InCents>=5))   {
                     u32Price=u32Price+1;
                  }
                  char szPrice[PRINT_SIZE];
                  OSALUTIL_s32SaveNPrintFormat(szPrice, PRINT_SIZE, "$%u.%.2u", u32Price / 100, u32Price % 100);        
                  fc_sxm_vString2Fi((tString)szPrice,oFiEntry.FuelPrice);
                  break;
               }
            }
        }
    }
    /*If Filter type is not available for a fuel station*/
    if((0!=oFiEntry.NumPrices)&&(oFiEntry.FuelPrice==OSAL_NULL))	{
    	ETG_TRACE_ERR(("fc_sxm_trFuelList::Condition fuel price not filter"));
    	oFiEntry.NumPrices=0;
    }
}
/*********************************************************************
 *
 *FUNCTION:    vFillListEntry
 *
 *DESCRIPTION:   For debuging purpose
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/

tVoid fc_sxm_trFuelList::vTraceFiEntry(midw_ext_fi_tcl_FuelInfo const &oFiEntry) const {
    /*Trace For Fuel Info List*/
    if (etg_bIsTraceActive(TR_CLASS_FC_SXM_FUEL_APP, (tU16)TR_LEVEL_USER_4)) {
        ETG_TRACE_USR4(("\tSMS_ID=%u",oFiEntry.Fuel_LOCID));

        ETG_TRACE_USR4(("\tLAT=%d LON=%d",
                        oFiEntry.FuelLoc.Lat,
                        oFiEntry.FuelLoc.Lon));
        ETG_TRACE_USR4(("\tStation Name=%s",oFiEntry.StationName.szValue));
        ETG_TRACE_USR4(("\tBrand Name=%s",oFiEntry.Brand.szValue));
        ETG_TRACE_USR4(("\tDistance Km=%7f",oFiEntry.FuelDistKm));
        ETG_TRACE_USR4(("\tDistance Miles=%7f",oFiEntry.FuelDistMile));
        ETG_TRACE_USR4(("\tNumber of price=%d",oFiEntry.NumPrices));
        /*Trace For Fuel Info List for fuel type*/
        ETG_TRACE_USR4(("\tFuel Name=%s",oFiEntry.FuelName.szValue));
        ETG_TRACE_USR4(("\tFuel Price=%s",oFiEntry.FuelPrice.szValue)); 
        ETG_TRACE_USR4(("\tFuel type=%d Fuel Age = %d",
                        oFiEntry.FuelType.enType,
                        oFiEntry.FuelAge
                        ));            
    }
}

/*********************************************************************
 *
 *FUNCTION:    vEmit
 *
 *DESCRIPTION:   Send result to HMI
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
#ifndef FC_SXM_FUEL_OPTIMIZE_FROZEN_LIST /*START FC_SXM_FUEL_OPTIMIZE_FROZEN_LIST*/
tVoid fc_sxm_trFuelList::vEmit(fc_sxm_trAdressing const &rAdressing) { 
    /* to send method result for fuel list info*/
    midw_ext_sxm_fuelfi_tclMsgGetFuelInfoListMethodResult oMRes;
    /* store the listid*/
    oMRes.ListID=u32GetId();
#  ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING

		ETG_TRACE_USR4(("fc_sxm_tclFuelList::vEmit(%d) enListState=%d enListMode=%d numEntries=%u START",
						oMRes.ListID,
						ETG_CENUM(fc_sxm_tenListState, enGetListState()),
						ETG_CENUM(fc_sxm_tenListMode, enGetListMode()),
						rGetSortedEntries().size()));
	#  else
		ETG_TRACE_USR4(("fc_sxm_tclFuelList::vEmit(%d) bIsFrozen=%d freezeState=%u numEntries=%u START",
						oMRes.ListID,
						bIsFrozen(),
						ETG_CENUM(fc_sxm_tenListFreezeState, enGetListFreezeState()),
						rGetSortedEntries().size()));
	#  endif
		/*Fill all available fuel info from internal list*/
		SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries()) {
			midw_ext_fi_tcl_FuelInfo oFiEntry;
			oMRes.FuelInfoList.FuelInfolist.push_back(oFiEntry);
			midw_ext_fi_tcl_FuelInfo &roFiEntry=oMRes.FuelInfoList.FuelInfolist.back();
			vFillListEntry(**iter, roFiEntry);
			ETG_TRACE_USR3(("fc_sxm_tclFuelApp::ADD: SMS_ID=0x%08x",roFiEntry.Fuel_LOCID));
			vTraceFiEntry(roFiEntry);
		}
		/* send the method result*/
	#  ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
		// write the list-operation to the method-result
		vListOperation2Fi(oMRes.Operation);
	#  endif
		// write current list-info to the method-result
		oMRes.ListInfo=rGetStatusInfo();
    fc_sxm_tclFuelService::instance()->enSendFiMessage(rAdressing, oMRes);
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vEmit END"));
}
#else
tVoid fc_sxm_trFuelList::vEmit(fc_sxm_trAdressing const &rAdressing) {    
    midw_ext_sxm_fuelfi_tclMsgGetFuelInfoListMethodResult &oMRes=_oMResFuelInfoList;
    /* get list id*/
    oMRes.ListID=u32GetId();
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vEmit(%d) bIsFrozen=%d freezeState=%u START",
                    oMRes.ListID, bIsFrozen(), ETG_CENUM(fc_sxm_tenListFreezeState, enGetListFreezeState())));
    
    tBool bAdd=FALSE;
    if (enGetListFreezeState() != fc_sxm_enListFreezeState_Frozen) {
        oMRes.FuelInfoList.FuelInfolist.clear(); /* clear fuel list info*/
        ETG_TRACE_USR4(("fc_sxm_tclFuelApp:bAdd=TRUE"));
        bAdd=TRUE;
    }

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp:rGetSortedEntries=%u",rGetSortedEntries().size()));
    
    SXM_FOREACH_CONST(trSortedSetType, iter, rGetSortedEntries()) {
        fc_sxm_trFuelListEntry const &roEntry=**iter;
        tBool bUpdate=(roEntry.enState==fc_sxm_enListEntryState_Updated);
        ETG_TRACE_USR4(("fc_sxm_tclFuelApp:bAdd=%d  bUpdate=%d", bAdd, bUpdate));
        
        if (bAdd || bUpdate) {
            midw_ext_fi_tcl_FuelInfo oFiEntry;
            vFillListEntry(roEntry, oFiEntry);

            if (bAdd) {
                ETG_TRACE_USR3(("fc_sxm_tclFuelApp::ADD: SMS_ID=0x%08x",oFiEntry.Fuel_LOCID));
                oMRes.FuelInfoList.FuelInfolist.push_back(oFiEntry);
            } else {
                SXM_FOREACH(vector<midw_ext_fi_tcl_FuelInfo>, iter2,oMRes.FuelInfoList.FuelInfolist) {
                    ETG_TRACE_USR3(("fc_sxm_tclFuelApp::UPDATE: SMS_ID=0x%08x",roEntry.u32SxmId));
                    if (roEntry.u32SxmId==iter2->Fuel_LOCID) {
                        (*iter2)=oFiEntry;
                        break;
                    }
                }           
            }

            vTraceFiEntry(oFiEntry);
        }
    }

    /* send method result for fuel list info*/
    fc_sxm_tclFuelService::instance()->enSendFiMessage(rAdressing, oMRes);

    if (enGetListFreezeState() == fc_sxm_enListFreezeState_Idle) {
        oMRes.FuelInfoList.FuelInfolist.clear(); /* clear list info*/
    }

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vEmit END"));   
}
#endif /*END FC_SXM_FUEL_OPTIMIZE_FROZEN_LIST*/


#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING 
/*
  will be called by base-class when data are ready and the list shall be sent to hmi
 */
tVoid fc_sxm_trFuelList::vStartTimer() {
    ETG_TRACE_USR4(("fc_sxm_trFuelList::vStartTimer START"));

    fc_sxm_tclFuelApp::instance()->vStartListTimer();
}


/*
  will be called by base-class to check the timer status
 */
tBool fc_sxm_trFuelList::bIsListModeTimerRunning() {
    tBool bRes=fc_sxm_tclFuelApp::instance()->bIsListUpdateTimerRunning();
    ETG_TRACE_USR4(("fc_sxm_trFuelList::vIsListModeTimerRunning: %d", bRes));

    return bRes;
}
/*
  will be called by base-class when list is released
 */
tVoid fc_sxm_trFuelList::vStopTimer() {
    ETG_TRACE_USR4(("fc_sxm_trFuelList::vStopTimer STOP"));

    fc_sxm_tclFuelApp::instance()->vStopListTimer();
}
#endif


/*********************************************************************
 *FUNCTION:     bValidateEntry
 *DESCRIPTION:  Check if the given List Entry is valid or not based on 
                  the values stored
 *PARAMETER:    Pointer to fc_sxm_trStockQuoteListEntry structure 
 *RETURNVALUE:  Return TRUE if stock symbol is valid, FALSE otherwise
 ********************************************************************/
tBool fc_sxm_trFuelList::bValidateEntry (fc_sxm_trFuelListEntry const *poFuelListEntry) const
{
       /*Error check for invalid entries*/
	   tBool bRes=((poFuelListEntry != OSAL_NULL) && (!poFuelListEntry->oBrandName.empty()));
	   if(FALSE==bRes){
	     ETG_TRACE_ERR(("Entry Object is not valid"));
	   }
	   /*NULL Check for remove lint warning*/
	   if(poFuelListEntry != OSAL_NULL){
		   /*Error check for invalid location object
			* for nearby and station location*/
		   bRes=(_hLocation!=LOCATION_INVALID_OBJECT)&&
				(poFuelListEntry->hLocation!=LOCATION_INVALID_OBJECT);
		   if(FALSE==bRes){
			 ETG_TRACE_ERR(("Location Object is not valid"));
		   }
	   }
	   return bRes;
}

/*********************************************************************
 *
 *FUNCTION:    bCompareBrandName
 *
 *DESCRIPTION:   Compaire two brand name
 *PARAMETER:    None
 *
 *RETURNVALUE: None
 *
 ********************************************************************/
bool fc_sxm_trFuelList::bCompareBrandName(fc_sxm_trFuelListEntry const *prLeft, fc_sxm_trFuelListEntry const *prRight) const {

   string sLeftBand = prLeft->oBrandName;
   string sRightBrand = prRight->oBrandName;

   /*First Convert both the strings to UpperCase */
   ConvertToUpper(sLeftBand);
   ConvertToUpper(sRightBrand);

   /* Compare both the names using OSAL compare func */
   if( OSAL_s32StringCompare( sLeftBand.c_str(),sRightBrand.c_str() ) == 0)
   {
      return bCompareSmsDistance(prLeft, prRight);      
   }
   else if(OSAL_s32StringCompare( sLeftBand.c_str(),sRightBrand.c_str() ) < 0){
      return true;
   }
   
   return false;
}

/*
   helper to Convert a String Characters to Uppercase
*/
tVoid fc_sxm_trFuelList::ConvertToUpper(string &str)
{
   /*Loop through the size of the string */
   for(unsigned int i=0;i < str.length();i++)
   {
      /* Reset the value of the array position to the newupper case letter */
      str[i] = (tChar)toupper(str[i]);
   }
}
