/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_fuel_app.cpp
* @brief       Application handler for fuel service.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_tcl_sms_init.h"
#include "fc_sxm_service_sxm_fuel.h"
#include "fc_sxm_tcl_fuel_app.h"
#include "fc_sxm_tcl_fuel_dsrl.h"
#include "fc_sxm_tcl_fuel_list.h"
#include "fc_sxm_sms_util.h"

typedef enum {
    fc_sxm_enTtfisCmdsFuel_MStartGetList=1,
    fc_sxm_enTtfisCmdsFuel_MStartGetStationInfo=2,
    fc_sxm_enTtfisCmdsFuel_MStartSetListsConfig=3,
    fc_sxm_enTtfisCmdsFuel_MStartAddFavorite=4,
    fc_sxm_enTtfisCmdsFuel_MStartRemoveFavorite=5,
    fc_sxm_enTtfisCmdsFuel_ModifyDsrlEnty=6,
    fc_sxm_enTtfisCmdsFuel_SetListMode=7,
    fc_sxm_enTtfisCmdsFuel_ListOperation=8,
    fc_sxm_enTtfisCmdsFuel_LAST=20
} fc_sxm_tenTtfisCmdsFuel;

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_FUEL_APP
#include "trcGenProj/Header/fc_sxm_tcl_fuel_app.cpp.trc.h"
#endif

/*Define static variable*/
vector<fc_sxm_trFuelType> fc_sxm_tclFuelApp::_vFuelTypeList;
vector<string> fc_sxm_tclFuelApp::_vUnknownFuelName;

/*
  Implementation of all callback-functions for the sms needed by fuel.
  Only communication to audio-app:
  1.) Posting messages using fc_sxm_tclFuelApp::instance()->vPostMsgNew(rMsg);
  2.) writing to semaphore-protected data (fc_sxm_tclFuelProperties)
*/



/*
  peha: constructor: give it a name and provice the used sxmServiceId (not the ccaServiceId!!)
*/
/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclFuelApp
 *
 *DESCRIPTION:  constructor
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclFuelApp::fc_sxm_tclFuelApp():
    fc_sxm_tclDataApp("sxmFuelApp", _enStaticServiceId, CCA_C_U16_SRV_SXM_FUEL, "fuel", TR_CLASS_FC_SXM_FUEL_APP)
{
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp constructor"));
    /*Initialize the class members*/
    _hFuel = FUEL_SERVICE_INVALID_OBJECT;
    _enSortMethod = FUEL_PRICE_SORT_METHOD_PRICE;
    _hLocation=LOCATION_INVALID_OBJECT;
    _hDSRL=DSRL_INVALID_OBJECT;
    _poNearByDsrl=OSAL_NULL;
    _poNearDestDsrl=OSAL_NULL;
    _poFavDsrl=OSAL_NULL;
    _poNearByList=OSAL_NULL;
    _poDestList=OSAL_NULL;
    _poFavList=OSAL_NULL; 
    _u32Capacity=0;
    _u32Radius=0;
    _enTestListMode=fc_sxm_enListMode_Invalid;
    _bDataProductsEnabled = FALSE;

}

/*********************************************************************
 *
 *FUNCTION:     ~fc_sxm_tclFuelApp
 *
 *DESCRIPTION:  Destructor
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/

fc_sxm_tclFuelApp::~fc_sxm_tclFuelApp()
{
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp destructor"));
    _hFuel = FUEL_SERVICE_INVALID_OBJECT;
    _enSortMethod = FUEL_PRICE_SORT_METHOD_PRICE;
    _hLocation=LOCATION_INVALID_OBJECT;
    _hDSRL=DSRL_INVALID_OBJECT;

    /** Delete the instance of FavoritesDSRL and Fuel List**/
    FC_SXM_MEMORY_CLEAR(_poNearByDsrl);
    FC_SXM_MEMORY_CLEAR(_poNearDestDsrl);
    FC_SXM_MEMORY_CLEAR(_poFavDsrl);
    FC_SXM_MEMORY_CLEAR(_poNearByList);
    FC_SXM_MEMORY_CLEAR(_poDestList);
    FC_SXM_MEMORY_CLEAR(_poFavList);	
}

/*********************************************************************
 *
 *FUNCTION:     vInitialize
 *
 *DESCRIPTION:  Initialize
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vInitialize"));

    /*Initialize the class members*/
    _hFuel = FUEL_SERVICE_INVALID_OBJECT;
    _enSortMethod = FUEL_PRICE_SORT_METHOD_PRICE;
    _poNearByDsrl=OSAL_NULL;
    _poNearDestDsrl=OSAL_NULL;
    _poFavDsrl=OSAL_NULL;


    //_u32Capacity=FUEL_CAPACITY;
    //_u32Radius=FUEL_RADIUS;
    /*Create the fuel worker thread*/

    /* call vInitialize of base-class to start the worker thread */
    fc_sxm_tclBaseApp::vInitialize();

    fc_sxm_trFuelDSRLCfg rDsrlCfg;

    rDsrlCfg.enSortMethod=fc_sxm_enFuelDSRLSortMethod_FIRST_SAVED_FIRST;
    rDsrlCfg.rFilterCfg.enFuelType=FUEL_TYPE_UNKNOWN;
    rDsrlCfg.rFilterCfg.sBrand="All Brands";
    
    /* Get Capacity & Radius defined for Fuel Favorites type */
    rDsrlCfg.u32Capacity= FC_SXM_FUEL_FAVORITE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_FUEL_FAVORITE_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp:: call vCreateDsrlAndList: Favorites Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Favorites, fc_sxm_tenListType_Favorites, &_poFavDsrl, rDsrlCfg, &_poFavList);

    rDsrlCfg.enSortMethod=fc_sxm_enFuelDSRLSortMethod_DISTANCE;
    /* Get Capacity & Radius defined for Fuel Nearby type */
    rDsrlCfg.u32Capacity= FC_SXM_FUEL_DEVICE_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_FUEL_DEVICE_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp:: call vCreateDsrlAndList: NearBy Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Device, fc_sxm_tenListType_NearBy, &_poNearByDsrl, rDsrlCfg, &_poNearByList);

    // todo: remouve _u32Capacity ...
    _u32Capacity = rDsrlCfg.u32Capacity;       
    _u32Radius = rDsrlCfg.u32Radius;         
    
    /* Get Capacity & Radius defined for Fuel NearDest type */
    rDsrlCfg.u32Capacity= FC_SXM_FUEL_DESTINATION_DSRL_CAPACITY;
    rDsrlCfg.u32Radius  = FC_SXM_FUEL_DESTINATION_DSRL_RADIUS;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp:: call vCreateDsrlAndList: NearDest Capacity =%d ,Radius = %d ",rDsrlCfg.u32Capacity,rDsrlCfg.u32Radius ));
    vCreateDsrlAndList(fc_sxm_enDSRLType_Dest, fc_sxm_tenListType_NearDest, &_poNearDestDsrl, rDsrlCfg, &_poDestList);
    
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::trMsgDataSmsEvtServiceUp END"));
}

/*********************************************************************
 *
 *FUNCTION:     vDeInitialize
 *
 *DESCRIPTION:  Attention: called in cca-context by sxmapp-manager 
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vDeInitialize"));
    /* call vDeInitialize() of base-class to stop the worker thread */
    fc_sxm_tclBaseApp::vDeInitialize();

    /* Deinit of all other stuff starts from here */
}

/*********************************************************************
 *
 *FUNCTION:     vDispatchMsgFromQ
 *
 *DESCRIPTION:  this method is called by the thread-function.
 *              For each message that shall be handled 
 *              put and entry here.
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vDispatchMsgFromQ: SID=%x action=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()),
                    poThreadMsg->u16GetActionOnly()));

    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) {
        /* use macro SXM_MSGQ_DISPATCH  for each message specific for fuel*/
        /* next line to call vProcess(fc_sxm_trMsgFuelMStartGetList *) */

        /* method-start messages from cca-service */


        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFuelMStartGetList,fc_sxm_tclFuelApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFuelMStartGetStationInfo, fc_sxm_tclFuelApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFuelMStartSetListsConfig,fc_sxm_tclFuelApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFuelMStartAddFavorite,fc_sxm_tclFuelApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgFuelMStartRemoveFavorite,fc_sxm_tclFuelApp::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trPropertyFuelPSetListMode,fc_sxm_tclFuelApp::instance());

        default:
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                            u32Action));
            break;
    }
}
/*********************************************************************
 *
 *FUNCTION:     vCreateDsrlAndList
 *
 *DESCRIPTION:  this method is called to create DSRL for Nearby,
 *              Near destination and favorite fuel info list
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vCreateDsrlAndList(fc_sxm_tenDSRLType enDsrlType, fc_sxm_tenListType enListType, fc_sxm_tclFuelDSRL **ppoDsrl, 
                                            fc_sxm_trFuelDSRLCfg rDsrlCfg, fc_sxm_trFuelList **ppoList) {
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vCreateDsrlAndList: enDsrlType=%d enListType=%d",
                    ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType),
                    ETG_CENUM(fc_sxm_tenListType, enListType)
                    ));
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::construct dsrl"));
    /* create DSRL, dsrl will remain inactive while not configured*/
    fc_sxm_tclFuelDSRL *poDsrl;
    poDsrl=OSAL_NEW fc_sxm_tclFuelDSRL(enDsrlType);

    /* register the dsrl*/
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::poRegisterDsrl"));
    *ppoDsrl=poDsrl;
    poRegisterDsrl(poDsrl);

    /* create list*/
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::poFuelList Create"));
    fc_sxm_trFuelList *poFuelList=OSAL_NEW fc_sxm_trFuelList((tU32)enListType, enListType, this);
    *ppoList=poFuelList;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::poFuelList setDsrl"));
    if(OSAL_NULL!=poFuelList){
      poFuelList->vSetDsrl(poDsrl);
    }

    /* configure the list*/
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::poFuelList setCfg"));
    fc_sxm_trFuelListConfig rListCfg;
    rListCfg.rSortCfg.enSortMethod= static_cast<fc_sxm_tenFuelListSortMethod>(rDsrlCfg.enSortMethod);
    rListCfg.rFilterCfg.enFuelType=FUEL_TYPE_UNKNOWN;
    if(OSAL_NULL!=poFuelList){
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
        fc_sxm_trFuelList::vListOperation(poFuelList->u32GetId(), fc_sxm_enListOperation_Configure, fc_sxm_trAdressing(), rListCfg);
#else
        poFuelList->vSetCfg(rListCfg);
#endif
    }

    /* activate all by configuring the dsrl*/
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::poDsrl  vSetNextConfig"));
    if(OSAL_NULL!=poDsrl){
        vPatchDsrlConfig(enDsrlType, rDsrlCfg);
        poDsrl->vSetNextConfig(rDsrlCfg);
    }
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vCreateDsrlAndList: DONE"));


}

/* all messages are handled via vProcess(tMsgType const *prMsg) */

/*
  handle command from sxmAppManager to start service (Mandatory)
*/
/*********************************************************************
 *
 *FUNCTION:     vStartSmsService
 *
 *DESCRIPTION:  Start Fuel service
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vStartSmsService()
{
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::bStartService"));

    /*Check if the fuel service is already started*/
    if (_hFuel != FUEL_SERVICE_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Fuel service already started!"));
        return;
    }

    /*Start the fuel service*/
    _hFuel = FUEL.hStart(FC_SXM_SRH_DRIVER_NAME,
                         _enSortMethod,
                         DATASERVICE_EVENT_ALL,
                         cb_vDataServiceEventCallback,
                         (tVoid*)this, 
                         DATASERVICE_OPTION_NONE);

    
    
    if (_hFuel == FUEL_SERVICE_INVALID_OBJECT)
    {
        ETG_TRACE_USR4(("Could not start Fuel service"));
        return;
    }
    else
    {
       // vGetFueltypes();
    }
}
/*
  handle command from sxmAppManager to Load favorites on Service start
*/
tVoid fc_sxm_tclFuelApp::vRestoreFavorites()
{
    //call vLoadFavorites of base-class to fetch Favorite list form Persistent
    fc_sxm_tclDataApp::vLoadFavorites();
}
/*
  method will be called after Service Stop,to Save Favorite list to Persistent
*/
tVoid fc_sxm_tclFuelApp::vPersistFavorites()
{
    //call base-class to update Favorite list to Persistent
    fc_sxm_tclDataApp::vSaveFavorites();
}
 /*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg)
 *
 *DESCRIPTION:  Contains messages to handle TTFis commands
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) {
    fc_sxm_tenTtfisCmdsFuel enMsgCode=(fc_sxm_tenTtfisCmdsFuel)prMsg->u8MsgCode;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) code=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsFuel, enMsgCode)));
    tU8 const *au8Data=prMsg->au8Data;

    switch (enMsgCode) {
       /* Get List method */
        case fc_sxm_enTtfisCmdsFuel_MStartGetList:
        {
            midw_ext_sxm_fuelfi_tclMsgGetFuelInfoListMethodStart oMStart;
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::Get Parameter"));
            /*Get Methor start parameters
              1. List ID
              2. Sort Type
              3. Fuel Type
              4. Brand Name
              5. Fuel Name
            */
            oMStart.ListID=au8Data[0];
            oMStart.SortType.enType=(midw_ext_fi_tcl_e8_SortType::tenType)au8Data[1];
            oMStart.FuelType.FuelType.enType=(midw_ext_fi_tcl_e8_FuelTypeEnum::tenType)au8Data[2];
            fc_sxm_vString2Fi("",oMStart.Brand);
            fc_sxm_vString2Fi("regular",oMStart.FuelType.FuelName);

            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ttfis-cmd MStartGetList listId=%d sortMethod=%d fuelType=%d  brandName=%s",
                            oMStart.ListID,
                            ETG_CENUM(fc_sxm_tenFuelListSortMethod, oMStart.SortType.enType),
                            oMStart.FuelType.FuelType.enType,
                            oMStart.Brand.szValue
                            ));
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::Fuel Name:%s",oMStart.FuelType.FuelName.szValue));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;
        /* Get station info methoc */
        case fc_sxm_enTtfisCmdsFuel_MStartGetStationInfo:
        {
            midw_ext_sxm_fuelfi_tclMsgGetFuelStationInfoMethodStart oMStart;

            oMStart.ListID=au8Data[0];
            oMStart.FUEL_LOCID=(tU32)SXM_GET_U32(&au8Data[1]);
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ttfis-cmd MStartGetStationInfo listId=%d itemId=%d",
                            oMStart.ListID,
                            oMStart.FUEL_LOCID
                            ));
            fc_sxm_vSendSelfMessage(oMStart);

        }
        break;
        /* List Config method */
	    case fc_sxm_enTtfisCmdsFuel_MStartSetListsConfig:
        {
            midw_ext_sxm_fuelfi_tclMsgSetListsConfigMethodStart oMStart;

            oMStart.Config.Capacity=prMsg->au8Data[0];
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::fc_sxm_enTtfisCmdsFuel_MStartSetListsConfig Config.Capacity=%d",
                            oMStart.Config.Capacity));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;
        /* Add Favorite method */
        case fc_sxm_enTtfisCmdsFuel_MStartAddFavorite:
        {
            midw_ext_sxm_fuelfi_tclMsgAddFavoriteMethodStart oMStart;

            oMStart.FUEL_LOCID=(tU32)SXM_GET_U32(&au8Data[1]);
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::fc_sxm_enTtfisCmdsFuel_MStartAddFavorite FUEL_LOCID=%d",
                            oMStart.FUEL_LOCID));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;
        case fc_sxm_enTtfisCmdsFuel_MStartRemoveFavorite:
        {
            midw_ext_sxm_fuelfi_tclMsgRemoveFavoriteMethodStart oMStart;

            oMStart.FUEL_LOCID=(tU32)SXM_GET_U32(&au8Data[1]);
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::fc_sxm_enTtfisCmdsFuel_MStartRemoveFavorite FUEL_LOCID=%d",
                            oMStart.FUEL_LOCID));
            fc_sxm_vSendSelfMessage(oMStart);
        }
        break;

        case fc_sxm_enTtfisCmdsFuel_ModifyDsrlEnty:
        {
            tU8 u8DsrlId=au8Data[0];
            tU32 u32EntryId=(tU32)SXM_GET_U32(&au8Data[1]);
            fc_sxm_trDsrlTestModification::tenType enType=(fc_sxm_trDsrlTestModification::tenType)au8Data[5];
            tU32 u32Val=(tU32)SXM_GET_U32(&au8Data[6]);

            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ModifyDsrlEnty dsrlId=%u u32EntryId=0x%08x enType=%d u32Val=%u",
                            u8DsrlId,
                            u32EntryId,
                            ETG_CENUM(fc_sxm_trDsrlTestModification::tenType, enType),
                            u32Val));
            fc_sxm_tclBaseDSRL  *poDsrl=poGetDsrl(u8DsrlId);
            if (OSAL_NULL == poDsrl) {
                ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ModifyDsrlEnty: Dsrl not found!"));
                return;
            }
            fc_sxm_trDsrlTestModification &rModification=poDsrl->_rTestModification;
            rModification.enType=enType;
            rModification.bActive=(enType!=fc_sxm_trDsrlTestModification::enType_Idle);
            rModification.u32Id=u32EntryId;
            rModification.u32Val=u32Val;
            poDsrl->vForceUpdate();
        }
        break;

        case fc_sxm_enTtfisCmdsFuel_SetListMode:
        {
            midw_ext_sxm_fuelfi_tclMsgSxmListModeSet oPSet;
            /* Copy ListId param from TTFis*/
            oPSet.ListMode.ListID =(tU32)au8Data[0];
            oPSet.ListMode.Mode.enType = (midw_ext_fi_tcl_e8_SxmListMode::tenType)au8Data[1];
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ttfis-cmd PSetListId ListID=%d",
                         oPSet.ListMode.ListID
                         ));
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::ttfis-cmd PSetListMode ListMode=%d",
                         oPSet.ListMode.Mode.enType
                         ));
            fc_sxm_vSendSelfMessage(oPSet);

        }
        break;

        case fc_sxm_enTtfisCmdsFuel_ListOperation:
        {
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
            fc_sxm_tenListOperation enListOp = (fc_sxm_tenListOperation)au8Data[0];
            tS32 s32ChangeLat=(tS32)SXM_GET_U32(&au8Data[1]);
            tBool bChangeSort = au8Data[5];

            // todo: use c_enum for list-op 
            ETG_TRACE_ERR(("fc_sxm_enTtfisCmdsFuel_ListOperation enListOp=%d s32ChangeLat=%d bChangeSort=%d",
                           enListOp,
                           s32ChangeLat,
                           bChangeSort));
            fc_sxm_trFuelListConfig rCurrentConfig;

            if (s32ChangeLat) {
                fc_sxm_trDsrlLocation rLocation = rGetDestLocation();
                rLocation.s32Longitude +=s32ChangeLat;
                _poNearDestDsrl->vSetLocation(rLocation);
                _poDestList->vSetLocation(rLocation);
            }
            if (bChangeSort) {
                rCurrentConfig.rSortCfg.enSortMethod == fc_sxm_enFuelSortMethod_PRICE ?
                    rCurrentConfig.rSortCfg.enSortMethod = fc_sxm_enFuelSortMethod_DISTANCE : 
                    rCurrentConfig.rSortCfg.enSortMethod = fc_sxm_enFuelSortMethod_PRICE;
            }

            if (enListOp != fc_sxm_enListOperation_Idle) {
                fc_sxm_trAdressing rAdressing;
                _poDestList->vListOperation(enListOp, rAdressing, rCurrentConfig);

            }

#endif
        }
        break;
        case fc_sxm_enTtfisCmdsFuel_LAST:
            ETG_TRACE_USR4(("fc_sxm_tclFuelApp::fc_sxm_enTtfisCmdsFuel_LAST"));
            break;

      case (fc_sxm_tenTtfisCmdsFuel)fc_sxm_enGenricTtfisCmd_StartDataService:
        {
            tBool bStart=au8Data[0];
            ETG_TRACE_USR4(("fc_sxm_tenTtfisCmdsFuel::fc_sxm_enGenricTtfisCmd_StartDataService bStart=%d",
                            bStart));
            if (bStart) {
                fc_sxm_trMsgCmdDataAppStartService rMsg;
               
                fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);
            }
            else {
                fc_sxm_trMsgCmdDataAppStopService rMsg;
               
                fc_sxm_tclDataApp::vProcessGenericMsg(&rMsg);
            }
        }
           break;
        
       default:
            ETG_TRACE_ERR(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgCmdTtfisCmd) unknown code=%d",
                           ETG_CENUM(fc_sxm_tenTtfisCmdsFuel, enMsgCode)));
            break;
        
    }
}

/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgFuelMStartGetList 
 *               const *prMsg)
 *
 *DESCRIPTION:  Process method for fuel info list
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetList const *prMsg) {

    midw_ext_sxm_fuelfi_tclMsgGetFuelInfoListMethodStart const &oMStart=prMsg->oFiMsg;
    tU32 u32ListId=oMStart.ListID;

    /* configure list*
       Get Info from FI parameters*/
    fc_sxm_trFuelListConfig rListCfg;
    /*configure sort method*/
    if(u32ListId == fc_sxm_tenListType_Favorites)
    {
        rListCfg.rSortCfg.enSortMethod=(fc_sxm_tenFuelListSortMethod)fc_sxm_enFuelSortMethod_FIRST_SAVED_FIRST;
    }
    else
    {
    rListCfg.rSortCfg.enSortMethod=static_cast<fc_sxm_tenFuelListSortMethod>(oMStart.SortType.enType);
    }
    /*Sort configure Fuel Type */
    rListCfg.rSortCfg.enFuelType=(FUEL_TYPE_ENUM)(oMStart.FuelType.FuelType.enType);
    /*Filter configure Fuel Type*/
    rListCfg.rFilterCfg.enFuelType=(FUEL_TYPE_ENUM)(oMStart.FuelType.FuelType.enType);
    /*Filter configure for Brand Name*/
    rListCfg.rFilterCfg.sBrand.assign(oMStart.Brand.szValue);
    rListCfg.rFilterCfg.sFuelName.assign(oMStart.FuelType.FuelName.szValue);

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetList) call vSetCfg: listId=%d enSortMethod=%d enFuelType=%d sBrand=%s",
                    u32ListId,
                    ETG_CENUM(fc_sxm_tenFuelListSortMethod, rListCfg.rSortCfg.enSortMethod),
                    rListCfg.rFilterCfg.enFuelType,
                    rListCfg.rFilterCfg.sBrand.c_str()
                    ));
    ETG_TRACE_USR4((" FuelName=%s",rListCfg.rFilterCfg.sFuelName.c_str()));
    /*Send Configured info*/
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
    ETG_TRACE_USR4(("fc_sxm_trFuelList::vListOperation() Called"));
    fc_sxm_trFuelList::vListOperation(u32ListId, fc_sxm_tclDataListBase::enListOperationFromFi(oMStart.Operation), prMsg->rAdressing, rListCfg);
#else
    ETG_TRACE_USR4(("fc_sxm_trFuelList::vSetCfg(u32ListId, rListCfg) Called"));
    fc_sxm_trFuelList::vSetCfg(u32ListId, rListCfg);
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetList) call vSendList"));
    vSendList(u32ListId, prMsg->rAdressing );
#endif
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetList) END"));

}
/*********************************************************************
 *
 *FUNCTION:     vFillStationInfo
 *
 *DESCRIPTION:  Fill station info
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vFillStationInfo(midw_ext_fi_tcl_FuelStationInfo &rStationInfo, fc_sxm_trFuelListEntry const &rEntry) const {

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vFillStationInfo (u32Id=%d) StationName=%15s Brand=%15s",
                    rEntry.u32SxmId,
                    rEntry.oStationName.c_str(),
                    rEntry.oBrandName.c_str()));


    /*Station Name*/
    fc_sxm_vString2Fi(rEntry.oStationName.c_str(),rStationInfo.StationName);
    /*Brand Name*/
    fc_sxm_vString2Fi(rEntry.oBrandName.c_str(),rStationInfo.Brand);
    /*Street Name*/
    fc_sxm_vString2Fi(rEntry.oStreetName.c_str(),rStationInfo.StreetName);
    /*City Name*/
    fc_sxm_vString2Fi(rEntry.oCity.c_str(),rStationInfo.City);
    /*State Name*/
    fc_sxm_vString2Fi(rEntry.oState.c_str(),rStationInfo.State);
    /*ZipCode*/
    fc_sxm_vString2Fi(rEntry.oZipCode.c_str(),rStationInfo.ZipCode);
    /*PhoneNo*/
    fc_sxm_vString2Fi(rEntry.oPhone.c_str(),rStationInfo.Phone);
    /*Amenity*/
    fc_sxm_vString2Fi(rEntry.oAmenity.c_str(),rStationInfo.Amenity);
    /*Sxm Fuel ID*/
    rStationInfo.Fuel_LOCID=rEntry.u32SxmId;
    rStationInfo.FuelFavInfo=rEntry.bFavorite;
	/* Send number of fuel type available for current station*/
    rStationInfo.NumPrices=rEntry.u32NumPrices;
    SXM_FOREACH_CONST(vector<midw_ext_fi_tcl_e8_FuelStationAmenity>,iter,rEntry.ListAmenity)
    {
        rStationInfo.StationAmenitylist.push_back(*iter);
    }
    
    SXM_FOREACH_MMAP_CONST(FUEL_TYPE_ENUM, fc_sxm_trFuelTypeListEntry, iter, rEntry.listFuelTypes) {
        char szPrice[PRINT_SIZE];
        fc_sxm_trFuelTypeListEntry rGetEntry=iter->second;
        midw_ext_fi_tcl_FuelTypeInfolist oFiListEntry;

        oFiListEntry.FuelAge= (tU8)(rGetEntry.u32AgeInDays);
        oFiListEntry.FuelAvail=rGetEntry.bAvailable;
        fc_sxm_vString2Fi(rGetEntry.oFuelName.c_str(),oFiListEntry.FuelName);
        /*Get Fuel Price for the Fuel type*/
        tU32 u32Price= rGetEntry.u32Price/ 10;
        /*Round up cents */
        tU32 u32InCents=rGetEntry.u32Price % 10;
        if((0!=u32InCents)&&(u32InCents>=5))   {
           u32Price=u32Price+1;
        }
        /*NIKAI-1531 [Change Print statement]*/
        OSALUTIL_s32SaveNPrintFormat(szPrice, PRINT_SIZE, "$%u.%.2u", u32Price / 100, u32Price % 100);
        fc_sxm_vString2Fi(szPrice,oFiListEntry.FuelPrice);

        oFiListEntry.FuelType.enType = static_cast<midw_ext_fi_tcl_e8_FuelTypeEnum::tenType>(rGetEntry.enFuelType);
        ETG_TRACE_USR4(("  FuelType=%d Price=%15s Name=%15s age=%d bAvailable=%d",
                        ETG_CENUM(midw_ext_fi_tcl_e8_FuelTypeEnum::tenType,oFiListEntry.FuelType.enType),
                        oFiListEntry.FuelPrice.szValue,
                        oFiListEntry.FuelName.szValue,
                        oFiListEntry.FuelAge,
                        oFiListEntry.FuelAvail));
        rStationInfo.FuelTypelist.push_back(oFiListEntry);

#if 0
        midw_ext_fi_tcl_FuelTypeInfolist oFiListEntry;
        fc_sxm_trFuelTypeListEntry rEntry=iter->second;
        char szPrice[PRINT_SIZE];
        tU32 u32Price=rEntry.u32Price / 10;
        OSALUTIL_s32SaveNPrintFormat(szPrice, PRINT_SIZE, "%u.%u", u32Price / 100, u32Price % 100);
        oFiListEntry.FuelPrice=szPrice;
        oFiListEntry.FuelName=rEntry.oFuelName.c_str();
        oFiListEntry.FuelAge=rEntry.u32AgeInDays;
        oFiListEntry.FuelAvail=rEntry.bAvailable;
        ETG_TRACE_USR4(("  FuelType=%d Price=%d (%15s) Name=%15s age=%d bAvailable=%d",
                        rEntry.enFuelType,
                        rEntry.u32Price,
                        szPrice,
                        rEntry.oFuelName.c_str(),
                        rEntry.u32AgeInDays,
                        rEntry.bAvailable));

        rStationInfo.FuelTypelist.push_back();
#endif
    }

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vFillStationInfo END"));
}
/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgFuelMStartGetStationInfo const *prMsg)
 *
 *DESCRIPTION:  Process method for fuel station info list
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetStationInfo const *prMsg) {

    midw_ext_sxm_fuelfi_tclMsgGetFuelStationInfoMethodStart const &oMStart=prMsg->oFiMsg;
    tU32 u32Id=oMStart.FUEL_LOCID;
    tBool bFuelFavInfo=FALSE;
    /*
      todo: add entry to mark result as invalid when loc-id can not be found.
      this should be easier to handle than an error-message
    */
    midw_ext_sxm_fuelfi_tclMsgGetFuelStationInfoMethodResult oMRes;
    oMRes.ListID=oMStart.ListID;
    midw_ext_fi_tcl_FuelStationInfo &rStationInfo=oMRes.FuelStationInfo;
    tU32 u32ListId=(tU32)oMStart.ListID;
    /*entry to be filled from list:*/
    fc_sxm_trFuelListEntry rEntry;

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess::fc_sxm_trMsgFuelMStartGetStationInfo %d - %u",
                    oMRes.ListID,oMStart.FUEL_LOCID));
    /*find the correct list:*/
    fc_sxm_trFuelList *poList = fc_sxm_trFuelList::poGetTypedListById(u32ListId);
    if (OSAL_NULL != poList) {
        /* OK, we found the list*/
        if (!poList->bGetListEntry(u32Id, rEntry)) {
            /*Send Error message if invlaid sxmid is input or sxmid is not available for the current
              list
            */
            ETG_TRACE_ERR(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartGetStationInfo) Entry not found"));
            fc_sxm_tclFuelService::instance()->vSendError(prMsg->rAdressing,midw_ext_fi_tcl_e8FuelErrorTypes::FI_EN_FUEL_STATION_UNAVAILABLE);
            return;
        }
    }
    /* set id in case we have not found the list or the entry*/
    rEntry.u32SxmId=u32Id;
    /* Extract SMS Fuel Location to FI Location*/
    rStationInfo.FuelLocation = rGetFuelLocation(rEntry.hLocation);
    /* convert to fi-type*/
    vFillStationInfo(rStationInfo,rEntry);   
    /*find the correct list From Favorite:*/
    /*Check sxm is a enrty of favorite list
      TRUE: Station is in favorite 
    */
    poList=fc_sxm_trFuelList::poGetTypedListById(fc_sxm_tenListType_Favorites);
    if (OSAL_NULL != poList) {
        /* OK, we found the list*/
        if (TRUE == poList->bGetListEntry(u32Id, rEntry)) {         
            ETG_TRACE_ERR(("Entry Is on Favotite"));
            bFuelFavInfo=TRUE;
        }
    }
    oMRes.FuelStationInfo.FuelFavInfo=bFuelFavInfo;
    /*Trace To Send Station Info*/
    ETG_TRACE_USR4(("StationName:%s",oMRes.FuelStationInfo.StationName.szValue));
    ETG_TRACE_USR4(("Brand:%s",oMRes.FuelStationInfo.Brand.szValue));
    ETG_TRACE_USR4(("StreetName:%s",oMRes.FuelStationInfo.StreetName.szValue));
    ETG_TRACE_USR4(("City:%s",oMRes.FuelStationInfo.City.szValue));
    ETG_TRACE_USR4(("State:%s",oMRes.FuelStationInfo.State.szValue));
    ETG_TRACE_USR4(("ZipCode:%s",oMRes.FuelStationInfo.ZipCode.szValue));
    ETG_TRACE_USR4(("Phone:%s",oMRes.FuelStationInfo.Phone.szValue));
    ETG_TRACE_USR4(("Fuel_LOCID:%d",oMRes.FuelStationInfo.Fuel_LOCID));
    ETG_TRACE_USR4(("FuelFavInfo:%d",oMRes.FuelStationInfo.FuelFavInfo));
    ETG_TRACE_USR4(("FuelAmenity:%s",oMRes.FuelStationInfo.Amenity.szValue));
    ETG_TRACE_USR4(("Fuel Location Lat: %d", oMRes.FuelStationInfo.FuelLocation.Lat));
    ETG_TRACE_USR4(("Fuel Location Lon: %d", oMRes.FuelStationInfo.FuelLocation.Lon));
    ETG_TRACE_USR4(("Fuel type Number : %d", oMRes.FuelStationInfo.NumPrices));
    SXM_FOREACH(vector<midw_ext_fi_tcl_e8_FuelStationAmenity>,iter,oMRes.FuelStationInfo.StationAmenitylist)
    {
      ETG_TRACE_USR4(("Amenity Enum::%d",
           ETG_CENUM(midw_ext_fi_tcl_e8_FuelStationAmenity::tenType,(*iter).enType)));
    }
    /*Send station info Method Reasult*/
    fc_sxm_tclFuelService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
}
/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgFuelMStartSetListsConfig const *prMsg)
 *
 *DESCRIPTION:  Process method set list configuration
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartSetListsConfig const *prMsg)
{
    midw_ext_sxm_fuelfi_tclMsgSetListsConfigMethodStart const &oFiMsg = prMsg->oFiMsg;
    bool bNearByStatus = false;
    bool bDestStatus = false;
    /* configure dsrl with new capacity */
    fc_sxm_trFuelDSRLCfg rDsrlCfg;
    _u32Capacity=oFiMsg.Config.Capacity;
    rDsrlCfg.u32Capacity = _u32Capacity;

    /* configure the radius & Sort Types*/
    rDsrlCfg.enSortMethod = fc_sxm_enFuelDSRLSortMethod_DISTANCE;
    rDsrlCfg.u32Radius=_u32Radius;

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartSetListsConfig) call Capacity=%d",
                    _u32Capacity));

    /* Prapare emthod result with success Status*/
    midw_ext_sxm_fuelfi_tclMsgSetListsConfigMethodResult oMRes;

    /* We are going to change the capacity only for Near by dsrl*/
    /* And Check if _poNearByDsrl is created */

    if( OSAL_NULL != _poNearByDsrl){
        ETG_TRACE_USR4(("fc_sxm_tclFuelApp::Set config for NearBy"));
		/* So Set new capacity to NearBy DSRL*/
		_poNearByDsrl->vSetNextConfig(rDsrlCfg);
		/* Set the Method reult to true*/
		bNearByStatus = true;
    }

    /* We are going to change the capacity only for Destination  dsrl*/
    /* And Check if _poNearDestDsrl is created */

    if( OSAL_NULL != _poNearDestDsrl){
        ETG_TRACE_USR4(("fc_sxm_tclFuelApp::Set config for Destination"));
		/* So Set new capacity to Destination DSRL*/
		_poNearDestDsrl->vSetNextConfig(rDsrlCfg);
		/* Set the Method reult to true*/
		bDestStatus = true;
    }
    oMRes.Status = (bNearByStatus && bDestStatus);

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartSetListsConfig) call enSendFiMessage"));

    /* Send the method result to our service*/
    fc_sxm_tclFuelService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartSetListsConfig) END"));

}
/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgFuelMStartAddFavorite const *prMsg)
 *
 *DESCRIPTION:  Add Favorite for LOC ID
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/

tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartAddFavorite const *prMsg)
{
    midw_ext_sxm_fuelfi_tclMsgAddFavoriteMethodStart const &oFiMsg = prMsg->oFiMsg;
    tU32 u32SxmID = oFiMsg.FUEL_LOCID;
    tBool bStatus=FALSE;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;

    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartAddFavorite) call SxmFuelID=%d",
                    u32SxmID));
    /* Prapare emthod result with success Status*/
    midw_ext_sxm_fuelfi_tclMsgAddFavoriteMethodResult oMRes;

    /*find the correct list From Favorite:*/
    LOCID_OBJECT hLocID = LOCID.hCreate(u32SxmID);
    if(LOCID_INVALID_OBJECT!=hLocID)
    {
       /*Check for Invalid Location Object*/
       if(FUEL_SERVICE_INVALID_OBJECT!=_hFuel)
      {
           eReturnCode = FAVORITES.eSetFavoriteByLocID(hLocID,
                                                      _hFuel,
                                                      TRUE);
         if(SMSAPI_RETURN_CODE_SUCCESS!=eReturnCode)
         {
                    ETG_TRACE_ERR(("Favorite Unsucessfull Error :%d",eReturnCode));
         }
         else
         {
            if(TRUE == bIsUpdateEntry(u32SxmID))
            {
                //Update internal Favorite List for Sort Order
                fc_sxm_tclDataApp::vAddFavorite(u32SxmID);

               /* Return True for successful adition */
               bStatus=TRUE;
                  ETG_TRACE_USR4(("Favorite Add sucessfull "));
             }
          }
      }
      else
      {
           ETG_TRACE_ERR(("Invalid Fuel Service Object"));
       }
        /*Destroy Location Object*/
        LOCID.vDestroy(hLocID);
    }
    else
    {
      ETG_TRACE_ERR(("Invalid LOCID_OBJECT for Add Favorite")); 
    }
   

    /*Get TRUE : Sucess and FALSE: Failure and send result*/
    oMRes.Status = bStatus;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartAddFavorite) call enSendFiMessage :%d",bStatus));
    /* Send the method result to our service*/
    fc_sxm_tclFuelService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartAddFavorite) END"));
}

/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trMsgFuelMStartRemoveFavorite const *prMsg)
 *
 *DESCRIPTION:  Remove Favorite for LOC ID
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartRemoveFavorite const *prMsg)
{
    midw_ext_sxm_fuelfi_tclMsgRemoveFavoriteMethodStart const &oFiMsg = prMsg->oFiMsg;
    tU32 u32SxmID = oFiMsg.FUEL_LOCID;
    tBool bStatus=FALSE;
    LOCID_OBJECT hLocID=LOCID_INVALID_OBJECT;
    SMSAPI_RETURN_CODE_ENUM eReturnCode;
    fc_sxm_trFuelListEntry rEntry;
    
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartRemoveFavorite) call SxmFuelID=%d",
                    u32SxmID));
    /* Prepare method result with success Status*/
    midw_ext_sxm_fuelfi_tclMsgRemoveFavoriteMethodResult oMRes;
        /*Check sxmId is already in favorite list*/
        /*find the correct list From Favorite:*/
    fc_sxm_trFuelList *poList=dynamic_cast<fc_sxm_trFuelList*>(poGetActiveListbyDSRL(_poFavDsrl));
    ETG_TRACE_USR4(("poList=0x%08x :::: _poFavList=0x%08x",poList,_poFavList));
    if(OSAL_NULL != poList)   {
      if (TRUE == poList->bGetListEntry(u32SxmID, rEntry)) 
	   {
         /*Check for Invalid Location Object*/
         hLocID = LOCID.hCreate(u32SxmID);
         if(LOCID_INVALID_OBJECT!=hLocID)
         {
            if(FUEL_SERVICE_INVALID_OBJECT!=_hFuel)
            {
               /* set id in case we have not found the list or the entry*/
               rEntry.u32SxmId=u32SxmID;
               /*Remove Entry From The List*/
               eReturnCode = FAVORITES.eSetFavoriteByLocID(hLocID,
                                                          _hFuel,
                                                          FALSE);
               if(SMSAPI_RETURN_CODE_SUCCESS!=eReturnCode)
               {
                  ETG_TRACE_ERR(("Favorite Unsuccessful Error :%d",eReturnCode));
               }
               else
               {
                  /* OK We found the LOCID,Remove the retrieved entry from our internal List*/
                  rEntry.bFavorite=FALSE;
                  poList->vUpdateEntry(DSRL_ENTRY_STATUS_REMOVED,rEntry);

                  //Update internal Favorite List for Sort Order
                  fc_sxm_tclDataApp::vRemoveFavorite(u32SxmID);

                  /*Send TRUE to HMI*/
                  bStatus=TRUE;
                  ETG_TRACE_USR4(("Favorite Remove successful "));
                }
            }
            else
            {
               ETG_TRACE_ERR(("Invalid Fuel Service Object"));
            }
                /*Destroy Location Object*/
                LOCID.vDestroy(hLocID);
         }
         else
         {
            ETG_TRACE_ERR(("Invalid LOCID_OBJECT for Favorite")); 
         }
      }
    }
    /*Get TRUE : Success and FALSE: Failure and send result*/
    oMRes.Status = bStatus;
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartRemoveFavorite) call enSendFiMessage:%d",bStatus));
    /* Send the method result to our service*/
    fc_sxm_tclFuelService::instance()->enSendFiMessage(prMsg->rAdressing, oMRes);
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgFuelMStartRemoveFavorite) END"));
}



/*********************************************************************
 *
 *FUNCTION:     vProcess(fc_sxm_trPropertyFuelPSetListMode const *prMsg)
 *
 *DESCRIPTION:  Set property to Handle the List Mode
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcess(fc_sxm_trPropertyFuelPSetListMode  const *prMsg) 
{
   ETG_TRACE_USR4(( "vProcess fc_sxm_trPropertyFuelPSetListMode" ));

   midw_ext_sxm_fuelfi_tclMsgSxmListModeSet const &oPSet=prMsg->oFiMsg;   

   /* Initialize default values*/
   tU32 u32ListId = oPSet.ListMode.ListID;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyFuelPSetListMode: ListID %d",u32ListId));

   fc_sxm_tenListMode enListMode = (fc_sxm_tenListMode)oPSet.ListMode.Mode.enType;
   ETG_TRACE_USR4(( "fc_sxm_trPropertyFuelPSetListMode: enListMode %d",enListMode));
   vSetListMode(u32ListId, prMsg->rAdressing,enListMode);

   
}

/*********************************************************************
 *
 *FUNCTION:     bIsUpdateEntry
 *
 *DESCRIPTION:  Check the SxmID in the Device List Or in 
 *              Destination List any one of the list return success
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclFuelApp::bIsUpdateEntry(tU32 u32SxmID) 
{
    fc_sxm_trFuelListEntry rEntry;
    tBool bIsNearBy=FALSE;
    tBool bIsDest=FALSE;
   fc_sxm_trFuelList *poFavList=OSAL_NULL;
   DSRL_ENTRY_STATUS_ENUM eStatus = DSRL_ENTRY_STATUS_NEW;
   

   /*First Check in Destination List */
   fc_sxm_trFuelList *poList=dynamic_cast<fc_sxm_trFuelList*>(poGetActiveListbyDSRL(_poNearDestDsrl));
   if (OSAL_NULL != poList) {
      /* OK, we found the list*/
      if (poList->bGetListEntry(u32SxmID, rEntry)) {        
         ETG_TRACE_USR4(("Entry Is on fc_sxm_enDSRLType_Dest"));
         bIsDest=TRUE;
      }
   }
   /*Check weather list is already add by near Destination */
   if(TRUE!=bIsDest) {
      /*Check For Device List*/
      poList=dynamic_cast<fc_sxm_trFuelList*>(poGetActiveListbyDSRL(_poNearByDsrl));
      if (OSAL_NULL != poList) {
         /* OK, we found the list*/
         if (poList->bGetListEntry(u32SxmID, rEntry)) {        
            ETG_TRACE_USR4(("Entry Is on fc_sxm_enDSRLType_Device"));
            bIsNearBy=TRUE;
         }
      }
   }
    if(bIsNearBy||bIsDest)
    {
      /*find the correct list From Favorite:*/
      poFavList = dynamic_cast<fc_sxm_trFuelList*>(poGetActiveListbyDSRL(_poFavDsrl));
      if(OSAL_NULL!=poFavList)  
      {
         rEntry.bFavorite=TRUE;
         poFavList->vUpdateEntry(eStatus,rEntry);
         return TRUE;
      }
    }

    return FALSE;  
}

/*********************************************************************
 *
 *FUNCTION:     vOnListStateChanged
 *
 *DESCRIPTION:  Send the property of updated Lists Status
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vOnListStateChanged(fc_sxm_tclDataListBase *poDataList) {
    /* This line is added to avoid LINT warnings. Please
       remove as soon as the variables are used.*/
    (tVoid) poDataList; 
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vOnListStateChanged : Capacity:%d", _u32Capacity));
    fc_sxm_tcl_trFuelPropertyListsStatus oProperty;
    midw_ext_sxm_fuelfi_tclMsgFuelInfoListsStatusStatus &oFiMsg=oProperty.oFiMsg;
   
    oGetDataListsInfo().vCopyListInfosToFi(oFiMsg.InfoList);
    /*Send FI Status*/ 
    fc_sxm_tclFuelProperties::instance()->oListsStatus.vSet(oProperty);
    fc_sxm_tclFuelProperties::instance()->oListsStatus.vNotify();

}

/*********************************************************************
 *
 *FUNCTION:    vOnFiServiceStateChanged
 *
 *DESCRIPTION:   Application get notified about the new 
 *               fi-service-state that shall be sent to clients
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const &oFiState) {
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vOnFiServiceStateChanged : State:%20s (%d)",
                    fc_sxm_szSmsDataServiceStateFiToString(oFiState.Status), 
                    oFiState.Status.enType));
    fc_sxm_tcl_trFuelPropertySxmServiceStatus oProperty;
    oProperty.oFiMsg.DataServiceState=oFiState;
    fc_sxm_tclFuelProperties::instance()->oSxmServiceStatus.vSetAndNotify(oProperty);
}
/*********************************************************************
 *
 *FUNCTION:    vPrintReportSpecific
 *
 *DESCRIPTION:  for debuging
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vPrintReportSpecific() {
   /* for debuging */
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\tspecific data:"));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\ttestListMode=%u", ETG_CENUM(fc_sxm_tenListMode, _enTestListMode))); 
}

/*********************************************************************
 *
 *FUNCTION:    rGetFuelLocation
 *
 *DESCRIPTION:  Method to Extract SMS Location Object to FI Location type
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
midw_ext_fi_tcl_SxmLocation fc_sxm_tclFuelApp::rGetFuelLocation(LOCATION_OBJECT hLocObj)const
{
   OSAL_FIXED_OBJECT oLat = LOCATION.hLat(hLocObj);
   OSAL_FIXED_OBJECT oLon = LOCATION.hLon(hLocObj);

   midw_ext_fi_tcl_SxmLocation trFuelLoc;

   trFuelLoc.Lat = (tS32)((fc_sxm_fFixed2Float(oLat))*(1000000));    //10^6 is used to take 6 values after decimal point
   trFuelLoc.Lon = (tS32)((fc_sxm_fFixed2Float(oLon))*(1000000));

    return trFuelLoc;
}
/*********************************************************************
 *
 *FUNCTION:    vSendFueltypes
 *
 *DESCRIPTION:  Send fuel type to HMI collecting from SMS lib
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vSendFuelTypes(tVoid)  {
   SMSAPI_RETURN_CODE_ENUM eReturnCode;
   fc_sxm_tcl_trFuelPropertyTypeListStatus oFuelTypeProperty;
   midw_ext_sxm_fuelfi_tclMsgFuelTypeListStatus &oFiTypeMsg=oFuelTypeProperty.oFiMsg;
   
   ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vSendFueltypes"))
   /*Clear List*/
   _vFuelTypeList.clear();
   _vUnknownFuelName.clear();
   /*SMS call Back to get fuel type list*/
   if(FUEL_SERVICE_INVALID_OBJECT!=_hFuel)   {
      eReturnCode = FUEL.eFuelTypes(_hFuel,
         fc_sxm_tclFuelApp::bHandleFuelTypeCallback,
         OSAL_NULL);
      if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)  {
            ETG_TRACE_USR4(("FUEL.eFuelTypes Error: %d!",eReturnCode));
      }
   }
   
   /* Send fuel list of diffult type first time
      System Startup
      and if any unkown fuel type found 
   */
      midw_ext_fi_tcl_FuelTypeInfo GetFuelType;
      /*Update Available new Fuel Type Name From SMS*/
      SXM_FOREACH(vector<fc_sxm_trFuelType>,iter,_vFuelTypeList)
      {
         GetFuelType.FuelType.enType=static_cast<midw_ext_fi_tcl_e8_FuelTypeEnum::tenType>((*iter).eFuelType);
         fc_sxm_vString2Fi((*iter).sFuelYpe.c_str(),GetFuelType.FuelName);
         oFiTypeMsg.TypeName.TypeNameList.push_back(GetFuelType);
      }
      /*Update Available Unknown fuel Name From SMS*/
      SXM_FOREACH(vector<string>,iter,_vUnknownFuelName)
      {
         GetFuelType.FuelType.enType=midw_ext_fi_tcl_e8_FuelTypeEnum::FI_EN_FUEL_TYPE_UNKNOWN;
         fc_sxm_vString2Fi((*iter).c_str(),GetFuelType.FuelName);
         oFiTypeMsg.TypeName.TypeNameList.push_back(GetFuelType);
      }
      /*Trace : Fuel Type Name List*/
      for (tU32 i=0; i<oFuelTypeProperty.oFiMsg.TypeName.TypeNameList.size(); i++)
      {
         ETG_TRACE_USR4(("Fuel Type - %d : %s",
         oFuelTypeProperty.oFiMsg.TypeName.TypeNameList[i].FuelType.enType,
         oFuelTypeProperty.oFiMsg.TypeName.TypeNameList[i].FuelName.szValue));
      }
      /*Notify to Fi for new Fuel type list*/
      fc_sxm_tclFuelProperties::instance()->oTypeNameStatus.vSetAndNotify(oFuelTypeProperty);    
}
/*********************************************************************
 *
 *FUNCTION:    vProcess(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
 *
 *DESCRIPTION:  message to clear User data
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg)
{
    (void)prMsg; /* TO REMOVE LINT WARNINGS */
     ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess(fc_sxm_trMsgCmdAppClearUserData(%s)",
                     szGetName()));
     if(FUEL_SERVICE_INVALID_OBJECT != _hFuel)  {        
        ETG_TRACE_USR4(("fc_sxm_tclFuelApp:: Clear All Favorite"));
        FAVORITES.vClearAllFavorites(_hFuel);
        //clear Persistent list as well.
        fc_sxm_tclDataApp::vClearFavorites();
        vSetUsrDataClearResult(fc_sxm_enDiagResult_Ok);
     }
     else     
     {
        ETG_TRACE_ERR(( " FUEL_SERVICE_INVALID_OBJECT " ));
        vSetUsrDataClearResult(fc_sxm_enDiagResult_Failed);
     }                          
}
/*********************************************************************
 *
 *FUNCTION:    vProcess(fc_sxm_trMsgListUpdateFuelTimer const *prMsg)
 *
 *DESCRIPTION:  message containing List update timer for fuel
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
#ifndef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
tVoid fc_sxm_tclFuelApp::vProcessTimer(fc_sxm_trMsgListUpdateFuelTimer const *prMsg) {
   (tVoid)prMsg; /* To Remove Lint warning*/
   ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess"
                    "(fc_sxm_trMsgListUpdateTimer)"));
   vStopListTimer();
   /* find the correct list by ListID */
   ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vProcess _u32ListID %d",_u32ListID ));
   fc_sxm_trFuelList *poList=fc_sxm_trFuelList::poGetTypedListById(_u32ListID);
   if(OSAL_NULL!=poList)  {
      poList->vOnListTimerExpire();
   }
}
/*********************************************************************
 *
 *FUNCTION:    vStartListTimer
 *
 *DESCRIPTION:  start list timer for fuel app
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vStartListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vStartListTimer()"));

    _mListUpdateTimer.vStart(FC_SXM_LIST_DATA_UPDATE_PERIOD);
}
/*********************************************************************
 *
 *FUNCTION:    vStopListTimer
 *
 *DESCRIPTION: stop the list timer for fuel app
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vStopListTimer(tVoid) {
    ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vStopListTimer()"));

    _mListUpdateTimer.vStop();
}
/*********************************************************************
 *
 *FUNCTION:    bIsListUpdateTimerRunning
 *
 *DESCRIPTION:  check the timer is running or not
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclFuelApp::bIsListUpdateTimerRunning(tVoid) const {
    return(_mListUpdateTimer.bIsRunning());
}
#endif
/*********************************************************************
 *
 *FUNCTION:    vProcess(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg)
 *
 *DESCRIPTION:  check the timer is running or not
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclFuelApp::vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const *prMsg) {
 	
	(tVoid) prMsg ;

    /*Send fuel type list property to HMI*/
    vSendFuelTypes();
}
/*********************************************************************
 *
 *FUNCTION:    bEnableDataProducts
 *
 *DESCRIPTION:  To Enable Fuel Products
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  BOOLEAN
 *
 ********************************************************************/
tBool fc_sxm_tclFuelApp::bEnableDataProducts()  const {

	 ETG_TRACE_USR4(("fc_sxm_tclFuelApp::bEnableDataProducts()"));

    SMSAPI_RETURN_CODE_ENUM eReturnCode; 

    // Enable the DB product 
    eReturnCode = DATA.eEnableProduct( _hFuel, DATA_PRODUCT_TYPE_DB_UPDATES, DATA_PRODUCT_MASK_NONE); 
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) {
       ETG_TRACE_USR4(("failed to enable DATA_PRODUCT_TYPE_DB_UPDATES "));
       return FALSE;
    }

    // Enable the Price product 
    eReturnCode = DATA.eEnableProduct( _hFuel, DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES,DATA_PRODUCT_MASK_NONE); 
    if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS) {
       ETG_TRACE_USR4(("failed to enable DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES "));
       return FALSE;
    }

    return TRUE;

}

DATA_PRODUCT_STATE_ENUM fc_sxm_tclFuelApp::enGetDataProductState(DATASERVICE_MGR_OBJECT hDataService) const {

	ETG_TRACE_USR4(("fc_sxm_tclFuelApp::enGetDataProductState()"));

     //Fuel uses DB & Fuel Prices products
     //so check if both products are enabled
	 DATA_PRODUCT_STATE_ENUM enDBProdState = enGetDBProductState(hDataService);
	 DATA_PRODUCT_STATE_ENUM enFuelPriceProdState = enGetFuelPriceProductState(hDataService);

     if ( ( DATA_PRODUCT_STATE_READY == enDBProdState) &&
    		 ( DATA_PRODUCT_STATE_READY == enFuelPriceProdState ) )
     {
         return DATA_PRODUCT_STATE_READY;
     }
	 //fix for NCG3D-2230
     else if (( DATA_PRODUCT_STATE_UNSUBSCRIBED == enDBProdState) ||
    		 (DATA_PRODUCT_STATE_UNSUBSCRIBED == enFuelPriceProdState ))
     {
    	 return DATA_PRODUCT_STATE_UNSUBSCRIBED;
     }
     else if (( DATA_PRODUCT_STATE_DISABLED == enDBProdState) ||
    		 ( DATA_PRODUCT_STATE_DISABLED  == enFuelPriceProdState ))
     {
    	 return DATA_PRODUCT_STATE_DISABLED;
     }
     else
     {
         return DATA_PRODUCT_STATE_ERROR;
     }
}
DATA_PRODUCT_STATE_ENUM fc_sxm_tclFuelApp::enGetDBProductState(DATASERVICE_MGR_OBJECT hDataService) const{

    DATA_PRODUCT_STATE_ENUM eState = DATA_PRODUCT_STATE_DISABLED;
    DATA_PRODUCT_MASK eMask;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Get the state of the Fuel DB updates
    eReturn = DATA.eProductState(
    	   hDataService,
           DATA_PRODUCT_TYPE_DB_UPDATES, &eMask,
           &eState );

    ETG_TRACE_USR4(("DATA_PRODUCT_TYPE_DB_UPDATES State=%d",
                            ETG_CENUM(DATA_PRODUCT_STATE_ENUM, eState)
                         ));

    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
    {
       ETG_TRACE_USR4(("failed to get state of DATA_PRODUCT_TYPE_DB_UPDATES ReturnCode=%d,State=%d",
                        ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturn),
                        ETG_CENUM(DATA_PRODUCT_STATE_ENUM, eState)
                     ));
    }

    return eState;
}
DATA_PRODUCT_STATE_ENUM fc_sxm_tclFuelApp::enGetFuelPriceProductState(DATASERVICE_MGR_OBJECT hDataService) const{

    DATA_PRODUCT_STATE_ENUM eState = DATA_PRODUCT_STATE_DISABLED;
    DATA_PRODUCT_MASK eMask;
    SMSAPI_RETURN_CODE_ENUM eReturn;

    // Get the state of the Fuel Price product
    eReturn = DATA.eProductState(
    	   hDataService,
           DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES, &eMask,
           &eState );

    ETG_TRACE_USR4(("DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES State=%d",
                            ETG_CENUM(DATA_PRODUCT_STATE_ENUM, eState)
                         ));

    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
    {
       ETG_TRACE_USR4(("failed to get state of DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES ReturnCode=%d,State=%d",
                        ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturn),
                        ETG_CENUM(DATA_PRODUCT_STATE_ENUM, eState)
                     ));    
    }

    return eState;
}
tVoid fc_sxm_tclFuelApp::vDisableDataProducts() {

	ETG_TRACE_USR4(("fc_sxm_tclFuelApp::vDisableDataProducts" ));

     SMSAPI_RETURN_CODE_ENUM eReturn;

    // Disable Fuel DB products
    eReturn = DATA.eDisableProduct(
          _hFuel,
           DATA_PRODUCT_TYPE_DB_UPDATES );

    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
    {
       ETG_TRACE_USR4(("failed to disable DATA_PRODUCT_TYPE_DB_UPDATES ReturnCode=%d",
                        ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturn)
                     ));    
    }

    // Disable Fuel Price products
    eReturn = DATA.eDisableProduct(
          _hFuel,
           DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES );

    if ( SMSAPI_RETURN_CODE_SUCCESS != eReturn )
    {
       ETG_TRACE_USR4(("failed to disable DATA_PRODUCT_TYPE_FUEL_PRICE_UPDATES ReturnCode=%d",
                        ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturn)
                     ));    
    }
}
/*********************************************************************
 *
 *FUNCTION:     bHandleFuelTypeCallback
 *
 *DESCRIPTION:  Iterator For Get all Fuel Type
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
 tBool fc_sxm_tclFuelApp::bHandleFuelTypeCallback(STRING_OBJECT hFuelType,
                              FUEL_TYPE_ENUM eFuelType,
                              void * /*pvIteratorArg*/)
{
   string strFuelType = "";
   fc_sxm_vCopySmsString2Stl(hFuelType,strFuelType);

   if(FUEL_TYPE_UNKNOWN==eFuelType) {
      _vUnknownFuelName.push_back(strFuelType);
   }
   else  {
      fc_sxm_trFuelType rListType;
      rListType.sFuelYpe = strFuelType;
      rListType.eFuelType=eFuelType;
      _vFuelTypeList.push_back(rListType);
   }
   ETG_TRACE_USR4(("FuelApp Fuel Name %d : %s ",eFuelType,strFuelType.c_str()));

   return TRUE;
}

