/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_agw_products.cpp
* @brief       Implementation of handling different AGW Products
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_common.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_agw_products.h"
#include "fc_sxm_tcl_agw_shapes.h"
#include "fc_sxm_tcl_agw_tiles.h"
#include "fc_sxm_tcl_agw_storm_track_data.h"
#include "fc_sxm_tcl_agw_surface_data.h"
#include "fc_sxm_tcl_agw_storm_attr_data.h"
#include "fc_sxm_tcl_agw_now_rad_data.h"
#include "fc_sxm_tcl_agw_wind_magnitude_data.h"
#include "fc_sxm_tcl_agw_wind_direction_data.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_AGW_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_agw_products.cpp.trc.h"
#endif


//######### implementation fc_sxm_tclAgwProductBase #########
fc_sxm_tclAgwProductBase::fc_sxm_tclAgwProductBase(AGW_PRODUCT_TYPE_ENUM enProductType_):
    enProductType(enProductType_),
    bChanged(FALSE),
    u32UpdateCounter(0),
    bPendingChange(FALSE)
{
			ETG_TRACE_USR4(("fc_sxm_tclAgwProductBase:: CTOR prod=%d",
                            ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    
    fc_sxm_tclAgwProducts::instance()->vAddProduct(this);
}

fc_sxm_tclAgwProductBase::~fc_sxm_tclAgwProductBase() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProductBase:: DCTOR prod=%d",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    fc_sxm_tclAgwProducts::instance()->vRemoveProduct(this);
}

tVoid fc_sxm_tclAgwProductBase::vSetChanged() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProductBase::vSetChanged() prod=%d",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    vClearOutput();
    if (bIsLocked()) {
        // only used for tiles
        // shared memory is blocked, we can not write to it
        bChanged=FALSE;
        bPendingChange=TRUE;
    }
    else {

        bPendingChange=FALSE;
        bChanged=TRUE;
    }
}



//######### implementation fc_sxm_tclAgwProducts ###########
fc_sxm_tclAgwProducts::fc_sxm_tclAgwProducts(){
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::CTOR"));

}

fc_sxm_tclAgwProducts::~fc_sxm_tclAgwProducts() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::DTOR"));

    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::DTOR: delete prod %d",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        fc_sxm_tclAgwProducts::instance()->vRemoveProduct(poProduct);
        OSAL_DELETE poProduct;

    }
}

tVoid fc_sxm_tclAgwProducts::vInit() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit()"));
    // add supported products here. better init later only for products requested by tima?
#if FC_SXM_AGW_ENABLE_STORM_ATTRIBUTES 
	vAddProduct<fc_sxm_tclShapeProductStormAttrData>();
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled storm attributes product"));
#endif
#if FC_SXM_AGW_ENABLE_SURFACE 
	vAddProduct<fc_sxm_tclShapeProductSurfaceData>();   
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled surface product"));	
#endif
#if FC_SXM_AGW_ENABLE_NOWRAD 
	vAddProduct<fc_sxm_tclTileProductNowRadData>();
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled nowrad product"));
#endif
#if FC_SXM_AGW_ENABLE_STORM_TRACK 
	vAddProduct<fc_sxm_tclShapeProductStormTrackData>();
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled storm track product"));
#endif
#if FC_SXM_AGW_ENABLE_WIND_MAG
	vAddProduct<fc_sxm_tclTileProductWindMagnitudeData>();
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled wind mag product"));
#endif
#if FC_SXM_AGW_ENABLE_WIND_DIR 
	vAddProduct<fc_sxm_tclTileProductWindDirectionData>();
	ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vInit- enabled wind dir product"));
#endif
}

tVoid fc_sxm_tclAgwProducts::vAddProduct(fc_sxm_tclAgwProductBase *poProduct) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vAddProduct(%d)",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        _mapProducts[poProduct->enProductType]=poProduct;
}

tVoid fc_sxm_tclAgwProducts::vRemoveProduct(fc_sxm_tclAgwProductBase *poProduct) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vRemoveProduct(%d)",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
    SXM_IF_FIND_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts, poProduct->enProductType) {
        _mapProducts.erase(iter);
    }
}


fc_sxm_tclAgwProductBase *fc_sxm_tclAgwProducts::poGetProduct(AGW_PRODUCT_TYPE_ENUM enProductType) {
    SXM_IF_FIND_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts, enProductType) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::poGetProduct(%d)  FOUND",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
        return iter->second;
    }
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::poGetProduct(%d) FAILED",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    return OSAL_NULL;
}


tVoid fc_sxm_tclAgwProducts::vNotify() {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vNotify"));
    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        if (poProduct->bChanged) {
            ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vNotify(%d) CHANGED",
                            ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));

            poProduct->u32UpdateCounter++;
            poProduct->vNotify();
            poProduct->bChanged=FALSE;
        }
    }
}

tVoid fc_sxm_tclAgwProducts::vClearCounters() {
    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vClearCounters(%d)",
                        ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        poProduct->bChanged=FALSE;
        poProduct->u32UpdateCounter=0;
        poProduct->bPendingChange=FALSE;
    }
}

tBool fc_sxm_tclAgwProducts::bIsChanged() {
    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        if (poProduct->bChanged==TRUE) {
            ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::bIsChanged() TRUE"));
			if(!poProduct->bGetPendingChangeFlag())
			{
			ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::bIsChanged()& poProduct->bGetPendingChangeFlag() FALSE "));
            return TRUE;
			}
			else
			ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::bIsChanged()& poProduct->bGetPendingChangeFlag() TRUE "));
        }
    }
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::bIsChanged() FALSE"));
    return FALSE;
}

tVoid fc_sxm_tclAgwProducts::vSetChanged(AGW_PRODUCT_TYPE_ENUM enProductType) {
    ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vSetChanged(%d)",
                    ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, enProductType)));
    SXM_IF_FIND_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts, enProductType) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        if (OSAL_NULL == poProduct) {
            return;
        }
        poProduct->vSetChanged();
    }
}

tVoid fc_sxm_tclAgwProducts::vSetFilters(AGW_SERVICE_OBJECT hAgw) {
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::vSetFilters()"));
    SXM_FOREACH_MAP(AGW_PRODUCT_TYPE_ENUM, fc_sxm_tclAgwProductBase*, iter, _mapProducts) {
        fc_sxm_tclAgwProductBase *poProduct=iter->second;
        ETG_TRACE_USR4(("fc_sxm_tclAgwProducts::eAddFilter for Product %d",
                       ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
        
		if(AGW.eAddFilter (hAgw, poProduct->enProductType) == SMSAPI_RETURN_CODE_ERROR)
		{
			ETG_TRACE_ERR(("fc_sxm_tclAgwProducts::eAddFilter for Product %d failed",
                           ETG_CENUM(AGW_PRODUCT_TYPE_ENUM, poProduct->enProductType)));
			//error state -- log the details
		}
    }
}
