/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sxmapp_manager.h
* @brief       Provides the declaration for sxm application manager. Also contains the
*              generic message required for all services like decoder , antenna, signal
*              state etc.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_SXMAPP_MANAGER_H_
#define _FC_SXM_TCL_SXMAPP_MANAGER_H_

#include "fc_sxm_tcl_config.h"
#include "fc_sxm_tcl_base_service.h"
#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_tcl_base_dsrl.h"
#include "fc_sxm_client_spm_fi.h"
#include "fc_sxm_client_spm.h"
#include "fc_sxm_audio_fi.h"
#include "fc_sxm_clienthandler_profilemgr_fi.h"
//201103L: To check for the C++11 standard
# if __cplusplus < 201103L
   #define FC_SXM_CONST_STATIC_FLOAT_DECLARATION const
#else
   #define FC_SXM_CONST_STATIC_FLOAT_DECLARATION constexpr
#endif

//Forward declarations
class fc_sxm_tclAudioApp;
class fc_sxm_tclFuelApp;
class fc_sxm_tclCanadianFuelApp;
class fc_sxm_tclAgwApp;
class fc_sxm_tclSafeviewApp;
class fc_sxm_tclStocksApp;
class fc_sxm_tclAgwApp;
class fc_sxm_tclTabweatherApp;
class fc_sxm_tclWsAlertsApp;

typedef enum {
    fc_sxm_enAction_Invalid,
    fc_sxm_enAction_AppMgrStopCcaServices,
    fc_sxm_enAction_AppMgrStopData,
    fc_sxm_enAction_AppMgrStartData,
    fc_sxm_enAction_AudioStopDecoder
}fc_sxm_tenAction;


class fc_sxm_tclHelpers {
 public:
    static tS32 fc_sxm_s32DegreeNavToSms(tS32 s32NavCoordinate) {
        return (tS32)(fFactorNavPosToFloat*(tFloat)s32NavCoordinate);
    }

    static tS32 fc_sxm_s32DegreeSmsToNav(tS32 s32SmsCoordinate) {
        return (tS32)((tFloat)s32SmsCoordinate / fFactorNavPosToFloat);
    }

    static tS32 fc_sxm_s32SmsDegreeFloatToNav(tFloat fFloatSmsCoordinate) {
        return (tS32)(fFloatSmsCoordinate / fFactorNavPos);
    }

    static tS32 fc_sxm_s32DegreeFloatToSms(tFloat fFloatCoordinate) {
        return (tS32)((tFloat)fFloatCoordinate *  0xFFFF);
    }

    static FC_SXM_CONST_STATIC_FLOAT_DECLARATION tFloat fFactorNavPosToFloat=(360.0f/4294967295.0f) * 0xFFFF;
    static FC_SXM_CONST_STATIC_FLOAT_DECLARATION tFloat fFactorNavPos=(360.0f/4294967295.0f);
};



/*
  messgage from audio-app that sms is ready to start data-services
*/
struct fc_sxm_trMsgLoopBack_DecoderState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 1);
    fc_sxm_trMsgLoopBack_DecoderState(tBool bReady_ = FALSE):
        bReady(bReady_)
    { }
    tBool bReady;
};

// Message from audio
struct fc_sxm_trMsgAppMgrDestination: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 2);
    fc_sxm_trDsrlLocation rLocation;
};

// Message for antenna state
struct fc_sxm_trMsgLoopBack_AntennaState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 3);
    ANTENNA_STATE_ENUM eAntennaState;
};

// Message for signal state
struct fc_sxm_trMsgLoopBack_SignalState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 4);
    SIGNAL_STATE_ENUM eSignalState;
};

/*
  messgage from states to reset the watchdog timer when the sv_event is 0
*/
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
struct fc_sxm_trSXMWatchdogReset: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 5);
    tU32 u32SXMWatchdogCycles;
};
#endif

// Messages from Spm
struct fc_sxm_trMsgAudioCvmEvent: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 8);
    fc_sxm_trMsgAudioCvmEvent() {
       vSetMsgPrioHigh();
    }
#ifdef VARIANT_S_FTR_ENABLE_SPM_CORE_FI
	spm_corefi_tclMsgCvmEventStatus oFiMsg;
#else
    cfc_spmfi_tclMsgCvmEventStatus oFiMsg;
#endif
};


struct fc_sxm_trMsgCvmErrorRecoveryDone: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 9);
    fc_sxm_trMsgCvmErrorRecoveryDone(tBool bSet) {
       vSetMsgPrioHigh();
       bStart = bSet;
    }
    tBool bStart;

};
// Message from audio
struct fc_sxm_trMsgAppMgrClockParams: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 6);
    midw_ext_fi_tclString strTimeZone;
    tBool bDSTStatus;
    midw_ext_fi_tcl_LocalTimeDate oLocalTime;
};

// Message from audio
struct fc_sxm_trMsgAppMgrSportsMetadataUpdate: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 7);
    tBool bMetadataUpdate;
};
struct fc_sxm_trMsgAppMgrClockParameters: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 8);
    midw_ext_fi_tclString strTimeZone;
    tBool bDSTStatus;
};
// Messages from DATA-App to CCA-Thread to indicate the service-state
struct fc_sxm_trMsgAppMgrDataAppServiceStatus: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 100);
    fc_sxm_tenServiceID enServiceID;
    DATASERVICE_STATE_ENUM eState;
};

/* MethodStart for set Destination Position from CCA client */
struct fc_sxm_trMsgAudioMStartSetDevicePostion: public fc_sxm_tclMessage
{
	/* Service and service ID */
	SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 202);
	operator fc_sxm_trDsrlLocation() const {
		fc_sxm_trDsrlLocation rRet;
		rRet.s32Latitude=fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oFiMsg.Latitude);
		rRet.s32Longitude=fc_sxm_tclHelpers::fc_sxm_s32DegreeNavToSms(oFiMsg.Longitude);
		return rRet;
	}
	tBool bValid() const {
		return (oFiMsg.Latitude != 0 && oFiMsg.Longitude != 0);
	}
	/* Re-addressing info */
	fc_sxm_trAdressing rAdressing;
	midw_ext_sxm_audiofi_tclMsgSetDevicePositionMethodStart oFiMsg;
};

struct fc_sxm_trMsgAppMgrSmsDbError: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 310);
    fc_sxm_trMsgAppMgrSmsDbError(fc_sxm_tenServiceID enServiceId_=fc_sxm_enServiceID_AppManager):
        enServiceId(enServiceId_)
    {};    
    fc_sxm_tenServiceID enServiceId;

};


struct fc_sxm_trMsgAppMgrRestoreSmsDb: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 311);
    fc_sxm_trMsgAppMgrRestoreSmsDb(fc_sxm_tenServiceID enServiceId_=fc_sxm_enServiceID_AppManager):
        enServiceId(enServiceId_)
    {};

    fc_sxm_tenServiceID enServiceId;
};

struct fc_sxm_tcl_trAppMgrPropertySxmServiceStatus {
    fc_sxm_tcl_trAppMgrPropertySxmServiceStatus()
    {}
    bool operator!=(fc_sxm_tcl_trAppMgrPropertySxmServiceStatus const& b) const
    {
        return !(oFiMsg==b.oFiMsg);
    }

    midw_ext_sxm_audiofi_tclMsgSxmDataServiceStatusStatus oFiMsg;
 
};

// Messages from Profile Manager
struct fc_sxm_trMsgAppMgrProfileDataChangeUpdate: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 400);

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    profile_mainfi_tclMsgdataChangedStatus oFiMsg;
#endif
};
struct fc_sxm_trMsgAppMgrProfileActiveProfileUpdate: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 401);

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    profile_mainfi_tclMsgactiveProfileStatus oFiMsg;
#endif
};
struct fc_sxm_trMsgAppMgrProfileCopyProfileUpdate: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 402);

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    profile_mainfi_tclMsgcopyProfileStatus oFiMsg;
#endif
};
struct fc_sxm_trMsgAppMgrProfileDeleteProfileUpdate: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AppManager, 403);

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    profile_mainfi_tclMsgdeleteProfileStatus oFiMsg;
#endif
};


#define FC_SXM_APP_MANAGER_TIMER_ID_SV_TIMER 2
#define FC_SXM_APP_MANAGER_TIMER_ID_DEV_POS_TIMER 3

#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
#define FC_SXM_APP_MANAGER_TIMER_ID_WATCHDOG_TIMER 4
#endif

class fc_sxm_tclAppManager:
public fc_sxm_tclSingleton<fc_sxm_tclAppManager>
{
    friend class fc_sxm_tclSingleton<fc_sxm_tclAppManager>;
    class trSvTimer;
    class trDevicePosTimer;
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    class trSXMWatchdogTimer;
#endif
    class trTimerBase;
    public:

    //Constructor
    fc_sxm_tclAppManager();

    //Destructor
    virtual ~fc_sxm_tclAppManager();

    //Initialize the application manager
    tVoid vInitialize(tVoid);

    //DeInitialize the application manager
    tVoid vDeInitialize(tVoid);

    tVoid vProcess(fc_sxm_trMsgAudioMStartSetDevicePostion const *prMsg);

    //Start the data services
    tBool bStartSXMDataServices(tVoid);
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    tVoid vSXMWatchdogReset(tVoid);
    tVoid vStartSXMWatchdog(tVoid);
    tVoid vStopSXMWatchdog(tVoid);
#endif

    tBool bIsDecoderReady() const {
      	return _bDecoderReady;
      }
#ifndef FC_SXM_DISABLE_SMS
   tVoid vRegisterDataApp(tVoid);
#endif
    /*
      Allocate Message of type "M" (action-id = rMsg.u32Action).
      Valid receipents are "fc_sxm_enServiceID_Main" and "fc_sxm_enServiceID_AppManager"
    */
    template<class M>
        tVoid vPostMsgNew(M const &rMsg) {
        if (fc_sxm_enGetServiceId(M::u32Action)!= fc_sxm_enServiceID_AppManager) {
            return;
        }
        M *pMsg=OSAL_NEW M(rMsg);
        vPostMsg(pMsg);
    }

   // Get Pointer to App, if existing
    fc_sxm_tclBaseApp *poGetApp(fc_sxm_tenServiceID enServiceID) {
        map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*>::iterator iter
            = _mapAppRegisterDetails.find(enServiceID);
        if (iter != _mapAppRegisterDetails.end())
        {
            return iter->second;
        }
        return OSAL_NULL;
    }

    /*
      post message "prMsg"  to cca-context as loopback-message.
      "prMsg" has to be allocated by caller, vPostMsg takes custudie
      valid service-ids are "fc_sxm_enServiceID_Main" and "fc_sxm_enServiceID_AppManager"
    */
    tVoid vPostMsg(fc_sxm_tclMessage *prMsg) const;

    //Process messages "prMsg" (cca-conext)
    tVoid vDispatchMsg(fc_sxm_tclMessage *prMsg);

    // handle ttfis-cmds for  appThreads
    tVoid vDistributeTtfisCmd(fc_sxm_trMsgCmdAppTtfisCmd *prMsg) const;

    template<class M>
        tVoid vSendToBaseApp(M const &rMsg) const {
        fc_sxm_tenServiceID enServiceId =rMsg.enServiceId;
        
        // send message to specific sxm-app
        SXM_IF_FIND_MAP_CONST(fc_sxm_tenServiceID, fc_sxm_tclBaseApp*, iter, _mapAppRegisterDetails, enServiceId)
        {
            fc_sxm_tclBaseApp *poBaseApp = (*iter).second;
            SXM_ASSERT_RETURN(poBaseApp != OSAL_NULL);
            poBaseApp->vPostMsgNew(rMsg);
        }        
    }

    // handle ttfis-cmds for appManager
    tVoid vHandleTtfisCmd(tU8 u8MsgCode, tU8 const *pu8Data);

    map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*> const &oGetMapOfApps() {
        return _mapAppRegisterDetails;
    }

    /*
      Distribute message to all threads according to rMsg.u32Action
    */
    template<class M>
        tVoid vDistributeMessage(M const &rMsg)
    {
        fc_sxm_tenServiceID enServiceID=fc_sxm_enGetServiceId(M::u32Action);

        vTraceDistributeStart(&rMsg);
        if (enServiceID==fc_sxm_enServiceID_AppManager) {
            // todo:check if msg comes from our thread
            M rMsgCopy=rMsg;
            vDispatchMsg(&rMsgCopy);
        }
#if 0
        if (enServiceID==fc_sxm_enServiceID_Main) {
            // todo:check if msg comes from our thread
            M rMsgCopy=rMsg;
            fc_sxm_tclApp::instance()->vDispatchMsg(&rMsgCopy);
        }
#endif
        else if (enServiceID==fc_sxm_enServiceID_All || enServiceID==fc_sxm_enServiceID_AllData) {
            // broadcast message:
            //Iterate through the registered apps
            SXM_FOREACH_MAP(fc_sxm_tenServiceID, fc_sxm_tclBaseApp*, iterator, _mapAppRegisterDetails)
            {
                fc_sxm_tclBaseApp *poBaseApp = (*iterator).second;
                SXM_ASSERT_RETURN(poBaseApp != OSAL_NULL);
                if (poBaseApp->bWantsMsg(M::u32Action)) {
                    M *pMsg=OSAL_NEW M(rMsg);
                    poBaseApp->vPostMsg(pMsg);
                }
            }
       
        }
        else {
            // send message to specific sxm-app
            SXM_IF_FIND_MAP(fc_sxm_tenServiceID, fc_sxm_tclBaseApp*, iterator, _mapAppRegisterDetails, enServiceID) 
            {
                fc_sxm_tclBaseApp *poBaseApp = (*iterator).second;
                SXM_ASSERT_RETURN(poBaseApp != OSAL_NULL);
                if (poBaseApp->bWantsMsg(M::u32Action)) {
                    M *pMsg=OSAL_NEW M(rMsg);
                    poBaseApp->vPostMsg(pMsg);
                }
            } 
        }
        vTraceDistributeEnd(&rMsg);



    }



    /* for data-status */
    fc_sxm_tclAutoProperty<fc_sxm_tcl_trAppMgrPropertySxmServiceStatus, 
        CCA_C_U16_SRV_SXM_AUDIO,              /* the service-id of the CCA-Service */
        MIDW_EXT_SXM_AUDIOFI_C_U16_SXMDATASERVICESTATUS      /* give FID if notification to service is needed  */
        > oSxmDataServiceStatusFi;

    // interface for data-app to report their status.
    tVoid vOnDataServiceStatusChanged(fc_sxm_tenServiceID enServiceId, DATASERVICE_STATE_ENUM enNewState, tBool bEmit=TRUE);

    tVoid vHandleTimer(tU16 u16TimerId) const;

 private:
    tVoid vUpdateDataStatusFi(fc_sxm_tenServiceID enServiceId, DATASERVICE_STATE_ENUM enOldState, DATASERVICE_STATE_ENUM enNewState, tBool bEmit);
    tVoid vSetDevicePosition(fc_sxm_trDsrlLocation rNextDevicePosition);

    class trTimerBase {
    public:
        trTimerBase();
        virtual ~trTimerBase();

        virtual tVoid vFire()=0;
        virtual tU16 vGetTimerId()=0;
        tBool bIsRunning() const {
            return _bRunning;
        };
        tVoid vStart(tU32 u32Ms);
        tVoid vStop();

        static tVoid vHandleTimer(tU16 u16TimerId);
        private:
        tBool _bRunning;
        static set<trTimerBase*> _setTimers;

    };

    friend tVoid trTimerBase::vStart(tU32 u32Ms);
    static tVoid vStartTimer(tU16 u16TimerID, tU32 u32Ms);

    friend tVoid trTimerBase::vStop();
    static tVoid vStopTimer(tU16 u16TimerID);






    fc_sxm_tenAction enGetRunningActionType() const {
        return _enRunningAction;
    }
    
    tVoid vHandleDataAppStopped();
    tVoid vHandleDataAppStarted(fc_sxm_tenServiceID enServiceId);


 tVoid vSetAction(fc_sxm_tenAction enActionType, tU32 u32Ms) {
     _enRunningAction=enActionType;
     _oSvTimer.vStart(u32Ms);
}

 tVoid vStopAction(fc_sxm_tenAction enActionType);





    // message that audio has succesfully started sms
    tVoid vProcess(fc_sxm_trMsgLoopBack_DecoderState const *prMsg);

#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    tVoid vProcess(fc_sxm_trSXMWatchdogReset const *prMsg);
#endif

    // message from audio containing the current destination
    tVoid vProcess(fc_sxm_trMsgAppMgrDestination const *prMsg);

    // message containing antenna status
    tVoid vProcess(fc_sxm_trMsgLoopBack_AntennaState const *prMsg);

    // message containing signal state
    tVoid vProcess(fc_sxm_trMsgLoopBack_SignalState const *prMsg);

    // message containing data-status of a data-app
    tVoid vProcess(fc_sxm_trMsgAppMgrDataAppServiceStatus const *prMsg);

    tVoid vProcess(fc_sxm_trMsgAppMgrSmsDbError const *prMsg) const;
    
    tVoid vProcess(fc_sxm_trMsgAppMgrRestoreSmsDb const *prMsg) const;
    
    tVoid vProcess(fc_sxm_trMsgAppMgrClockParams const *prMsg);

    tVoid vProcess(fc_sxm_trMsgAppMgrSportsMetadataUpdate const *prMsg);
	
	tVoid vProcess(fc_sxm_trMsgAppMgrClockParameters const *prMsg);


    tVoid vProcess(trSvTimer *prMsg);
    tVoid vProcess(trDevicePosTimer *prMsg);
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    tVoid vProcess(trSXMWatchdogTimer *prMsg);
#endif
    //Register the application
    tVoid vRegisterApp(fc_sxm_tclBaseApp *poBaseApp);

    //Unregister the application
    tVoid vUnRegisterApp(fc_sxm_tenServiceID enServiceID);

    tVoid vTraceDistributeStart(fc_sxm_tclMessage const *poMsg) const;
    tVoid vTraceDistributeEnd(fc_sxm_tclMessage const *poMsg) const;

    //Map to store registration details for SXM Services
    map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*> _mapAppRegisterDetails;
    map<fc_sxm_tenServiceID, DATASERVICE_STATE_ENUM> _mapDataServiceStates;

    template<class CHILD, int ID>
        class trTimer:
    public trTimerBase
    {
    public:
        virtual tVoid vFire() {
            fc_sxm_tclAppManager::instance()->vProcess((CHILD *)this);
        }
        virtual tU16 vGetTimerId() {
            return (tU16)ID;
        }

    };

    class fc_sxm_tclSmsDecouple:
    public fc_sxm_tclBaseThread
    {
    public:
        fc_sxm_tclSmsDecouple(); 
        
        // called by app-manager
        tVoid vSetSmsDevicePosition(fc_sxm_trDsrlLocation rDevicePosition);
        
    protected:
        // called by my thread-fn
        virtual tVoid vHandleUserEvent(tU32 u32Mask);
        
    private:
        fc_sxm_tclProtectedData<fc_sxm_trDsrlLocation> _rLocation;
        
    };

    tBool _bDecoderReady;
    fc_sxm_trDsrlLocation _rNextDevicePosition;
    fc_sxm_trDsrlLocation _rActiveDevicePosition;
    fc_sxm_tclSmsDecouple _oSmsDecouple;
    class trSvTimer:
    public trTimer<trSvTimer, FC_SXM_APP_MANAGER_TIMER_ID_SV_TIMER>
    {};
    class trDevicePosTimer:
    public trTimer<trDevicePosTimer, FC_SXM_APP_MANAGER_TIMER_ID_DEV_POS_TIMER>
    {};
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    class trSXMWatchdogTimer:
    public trTimer<trSXMWatchdogTimer, FC_SXM_APP_MANAGER_TIMER_ID_WATCHDOG_TIMER>
    {};
#endif
    trSvTimer _oSvTimer;
    trDevicePosTimer _oPosTimer;
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    trSXMWatchdogTimer _oSXMWatchdogTimer;
#endif
    // currently running action
    fc_sxm_tenAction _enRunningAction;
    fc_sxm_tenServiceID _enCurDataApp;
    tU32 _u32DataStartIntervalMs;
    tU32 _u32DevicePosIntervalMs;
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    SIGNAL_STATE_ENUM _enSignalState;
    tU32 _u32SXMWatchdogCycles;
#endif
    template<class APP> 
    tVoid vAddApp() {
        if (fc_sxm_tclConfig::instance()->bIsAppEnabled(APP::_enStaticServiceId)) {
            vRegisterApp(APP::instance());
            APP::instance()->vInitialize();
        }
    }


};











#endif //_FC_SXM_TCL_SXMAPP_MANAGER_H_
