/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sxmapp_manager.cpp
* @brief       Sxm application manager implementation
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_main.h"
#include "fc_sxm_sms_util.h"

// sms
#include "fc_sxm_tcl_sms_init.h"

// base-classes for all apps
#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_tcl_data_app.h"

// audio
#include "fc_sxm_audio_if.h"
#include "fc_sxm_diag_if.h"

// data-applications
#include "fc_sxm_tcl_fuel_app.h"
#include "fc_sxm_tcl_canadian_fuel_app.h"
#include "fc_sxm_tcl_traffic_app.h"
#include "fc_sxm_tcl_movies_app.h"
#include "fc_sxm_tcl_agw_app.h"
#include "fc_sxm_tcl_channelart_app.h"

#include "fc_sxm_tcl_stocks_app.h"
#include "fc_sxm_tcl_tabweather_app.h"
#include "fc_sxm_tcl_wsalerts_app.h"
#include "fc_sxm_tcl_parking_app.h"
#include "fc_sxm_tcl_phonetics_app.h"

#ifndef FC_SXM_SPORTS_UTEST
#include "fc_sxm_tcl_sports_app.h"
#else
#include "sports/fc_sxm_tcl_sports_app.h"
#endif //end of FC_SXM_SPORTS_UTEST


#include "sxm_common.h"
#include "sxm_sdk.h"
#include "sxm_sports.h"

// own header
#include "fc_sxm_tcl_sxmapp_manager.h"

#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
#include <signal.h>
#include "fc_sxm_audiorouting_lib.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
#include "fc_sxm_tcl_profilemanager.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXMAPP_MANAGER
#include "trcGenProj/Header/fc_sxm_tcl_sxmapp_manager.cpp.trc.h"
#endif


const tC8 *c8ModuleName = "/dev/x65HA";

static tC8 *cb_c8GetSportsSDKFilePath (tC8 *c8FilePath, unsigned int u32FilepathSize, tC8 c8FileType,
		                               const tC8 *c8Module,const tC8 *c8FileName)
{
	(tVoid)u32FilepathSize;
	ETG_TRACE_USR4(("fc_sxm_makefilepath  starts"));
	sprintf(c8FilePath, "/var/opt/bosch/dynamic/ffs/siriusxm/%c/%s/%s",c8FileType,c8Module,c8FileName);
	ETG_TRACE_USR4(("fc_sxm_makefilepath:%s",c8FilePath));
	return c8FilePath;
}

tS32 cb_s32LogInterface (const tC8 *c8Logout)
{
	return OSAL.iPuts(c8Logout);
}

typedef enum {
    fc_sxm_enTtfisCmdsAppManager_SetPosFiDeviceLocation=1,    
    fc_sxm_enTtfisCmdsAppManager_SetDestinationLocation=2,    
    fc_sxm_enTtfisCmdsAppManager_PrintReport=3,
    fc_sxm_enTtfisCmdsAppManager_LAST    
} fc_sxm_tenTtfisCmdsAppManager;


fc_sxm_tclAppManager::fc_sxm_tclAppManager():
    _bDecoderReady(FALSE),
    _enRunningAction(fc_sxm_enAction_Invalid),
    _enCurDataApp(fc_sxm_enServiceID_Audio),
    _u32DataStartIntervalMs(0),
    _u32DevicePosIntervalMs(FC_SXM_DEVICE_POSITION_UPDATE_INTERVAL)
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    ,_enSignalState(SIGNAL_STATE_UNKNOWN)
    ,_u32SXMWatchdogCycles(0)
#endif
{


    ETG_TRACE_USR4(("fc_sxm_tclAppManager constructor"));
}


fc_sxm_tclAppManager::~fc_sxm_tclAppManager()
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager destructor"));
}

#ifndef FC_SXM_DISABLE_SMS
tVoid fc_sxm_tclAppManager::vRegisterDataApp(tVoid)
{

#  ifndef FC_SXM_DISABLE_TRAFFIC
    vAddApp<fc_sxm_tclTrafficApp>();
#  endif

#  ifndef FC_SXM_DISABLE_MOVIES
    vAddApp<fc_sxm_tclMoviesApp>();
#  endif

#  ifndef FC_SXM_DISABLE_AGW
    vAddApp<fc_sxm_tclAgwApp>();
#  endif

#  ifndef FC_SXM_DISABLE_TABWEATHER
    vAddApp<fc_sxm_tclTabweatherApp>();
#  endif

#  ifndef FC_SXM_DISABLE_FUEL
    vAddApp<fc_sxm_tclFuelApp>();
#  endif

#  ifndef FC_SXM_DISABLE_CANADIAN_FUEL
    vAddApp<fc_sxm_tclCanadianFuelApp>();
#  endif

#  ifndef FC_SXM_DISABLE_SPORTS
    vAddApp<fc_sxm_tclSportsApp>();
#  endif

#  ifndef FC_SXM_DISABLE_STOCKS
    vAddApp<fc_sxm_tclStocksApp>();
#  endif

#  ifndef FC_SXM_DISABLE_WSALERTS
    vAddApp<fc_sxm_tclWsAlertsApp>();
#  endif
#  ifndef FC_SXM_DISABLE_PARKING
    vAddApp<fc_sxm_tclParkingApp>();
#  endif
}
#endif
/*
  Initialize audio and data-applications
*/
tVoid fc_sxm_tclAppManager::vInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vInitialize"));
    //Initialize SMS
#if 0
    tBool bSmsInitOk=fc_sxm_tclSMSInit::instance()->InitializeSMS();
    if (!bSmsInitOk) {
        ETG_TRACE_ERR(("fc_sxm_tclAppManager::could not initialize SMS_OSAL"));
        return;
    }
#endif

    // start decouple-thread
    _oSmsDecouple.vStart();
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    //create Profile Manager instance
    fc_sxm_tcl_profilemanager::instance();
#endif

#ifndef FC_SXM_DISABLE_SMS
    //Create the audio application
    vAddApp<fc_sxm_tclAudioApp>();

    //Create the data service apps
#  ifndef FC_SXM_DISABLE_CHANNELART
    vAddApp<fc_sxm_tclChannelartApp>();
#  endif

#  ifndef FC_SXM_DISABLE_PHONETICS
    vAddApp<fc_sxm_tclPhoneticsApp>();
#  endif
    tU8 u8VariantType = fc_sxm_tclApp::instance()->u8GetVariantType();
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vInitialize() Variant type: 0x%x", u8VariantType));
    switch(u8VariantType)
    {
    case FC_SXM_VARIANT_TYPE_DA:
        //Disable Data services for DA variant
        break;
    case FC_SXM_VARIANT_TYPE_NAVI:
        /*FALLTHROUGH*/
    case FC_SXM_VARIANT_TYPE_NOT_CONFIGURED:
        //Enable Data services for NAV variant or if KDS not configured.
#ifndef VARIANT_S_FTR_DISABLE_DATASERVICES
        //Disable data services for INF4CV NAV variant
        ETG_TRACE_USR4(("fc_sxm_tclAppManager::vInitialize() register data app"));
        vRegisterDataApp();
#endif
        break;
     default:
        break;
    }

    oSxmDataServiceStatusFi.vNotify();
   
#endif

#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    /* Start SXM watchdog timer */
    _oSXMWatchdogTimer.vStart(FC_SXM_WATCHDOG_INTERVAL);
    ETG_TRACE_USR4(("SXM Watchdog timer started"));
#endif

    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vInitialize END"));
}


tVoid fc_sxm_tclAppManager::vDeInitialize(tVoid)
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vDeInitialize"));

    // start decouple-thread
    _oSmsDecouple.vStop();
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
    //destroy Profile manager instance
    fc_sxm_tcl_profilemanager *poProfileManager = fc_sxm_tcl_profilemanager::instance();
    FC_SXM_MEMORY_CLEAR(poProfileManager)
#endif

    // Deinitialze all apps in reverse order they where created
    // todo: use reverse iterator
    map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp* >::reverse_iterator iter=_mapAppRegisterDetails.rbegin();
    while (!(iter==_mapAppRegisterDetails.rend())) {
        fc_sxm_tclBaseApp *poBaseApp=iter->second;
        ETG_TRACE_USR4(("fc_sxm_tclAppManager::bUnRegisterApp() ServiceID = %d", poBaseApp->enGetServiceId()));
        _mapAppRegisterDetails.erase(iter->first);
        poBaseApp->vDeInitialize();
        OSAL_DELETE(poBaseApp);
        iter=_mapAppRegisterDetails.rbegin();
    }
}

// register an application with its sxm-service-id
tVoid fc_sxm_tclAppManager::vRegisterApp(fc_sxm_tclBaseApp *poBaseApp)
{


    ETG_TRACE_USR4(("fc_sxm_tclAppManager::bRegisterApp() ServiceID = 0x%x", poBaseApp->enGetServiceId()));
    SXM_ASSERT_RETURN(OSAL_NULL != poBaseApp);
    _mapAppRegisterDetails.insert(pair<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*>
                                  (poBaseApp->enGetServiceId(), poBaseApp));

    if (poBaseApp->enGetServiceId() != fc_sxm_enServiceID_Audio) {
    	_mapDataServiceStates.insert(pair<fc_sxm_tenServiceID, DATASERVICE_STATE_ENUM>
    	    								(poBaseApp->enGetServiceId(), DATASERVICE_STATE_INVALID));
        vOnDataServiceStatusChanged(poBaseApp->enGetServiceId(), DATASERVICE_STATE_STOPPED, FALSE);
    }
}

// deregister an application with its sxm-service-id
tVoid fc_sxm_tclAppManager::vUnRegisterApp(fc_sxm_tenServiceID enServiceID)
{

    ETG_TRACE_USR4(("fc_sxm_tclAppManager::bUnRegisterApp() ServiceID = %d", enServiceID));

    map<fc_sxm_tenServiceID, fc_sxm_tclBaseApp*>::iterator iter
        = _mapAppRegisterDetails.find(enServiceID);
    if (iter != _mapAppRegisterDetails.end())
    {
        _mapAppRegisterDetails.erase(iter);
    }

}

/*
  Send a message to fc_sxm_tclAppManager.
  Data given via pvData have to be allocated by caller, we take custodie.
*/
tVoid fc_sxm_tclAppManager::vPostMsg(fc_sxm_tclMessage *prMsg) const
{
    fc_sxm_tclApp::instance()->vPostMsg(prMsg);
}


tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrDataAppServiceStatus const *prMsg) {
    ETG_TRACE_USR3(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrDataAppServiceStatus) enServiceID=%d eState=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, prMsg->enServiceID),
                    ETG_CENUM(DATASERVICE_STATE_ENUM, prMsg->eState)));
    SXM_ASSERT_RETURN(OSAL_NULL != prMsg);
    vOnDataServiceStatusChanged(prMsg->enServiceID, prMsg->eState, TRUE);
}

// evaluate and emit new fi-status
tVoid fc_sxm_tclAppManager::vUpdateDataStatusFi(fc_sxm_tenServiceID enServiceId, DATASERVICE_STATE_ENUM enOldState, DATASERVICE_STATE_ENUM enNewState, tBool bEmit) {
    midw_ext_fi_tcl_e8_SxmDataServiceStatus enOldFiSrvState=fc_sxm_oSmsDataServiceState2Fi(enOldState);
    midw_ext_fi_tcl_e8_SxmDataServiceStatus enNewFiSrvState=fc_sxm_oSmsDataServiceState2Fi(enNewState);
    if (!(enOldState == enNewState)) {
        ETG_TRACE_USR2(("fc_sxm_tcl_sxmapp_manager::new Fi-ServiceState(%d):%20s-->%20s (%d)",
                        ETG_CENUM(fc_sxm_tenServiceID, enServiceId),
                        fc_sxm_szSmsDataServiceStateFiToString(enOldFiSrvState), 
                        fc_sxm_szSmsDataServiceStateFiToString(enNewFiSrvState), 
                        enNewFiSrvState.enType));
        // updata property of fi-service-states
        midw_ext_fi_tcl_SxmDataSrvState oFiSrvState;
        oFiSrvState.Status=enNewFiSrvState;
        // lock property
        std::vector<midw_ext_fi_tcl_SxmListElemDataSrvState> &rList=oSxmDataServiceStatusFi.rAccess().oFiMsg.List;
        tBool bFound=FALSE;
        // create entry
        midw_ext_fi_tcl_SxmListElemDataSrvState oNewListElem;
        oNewListElem.Type.enType=(midw_ext_fi_tcl_e8_SxmDataServiceType::tenType)((enServiceId>>8) & 0xFF);
        oNewListElem.Status=oFiSrvState;
        SXM_FOREACH(vector<midw_ext_fi_tcl_SxmListElemDataSrvState>, iter, rList) {
            if (oNewListElem.Type==iter->Type) {
                // update entry if entry already exists
                iter->Status=oFiSrvState;
                bFound=TRUE;
            }
        }
        if (!bFound) {
            // create new entry
            rList.push_back(oNewListElem);
        }
        // release property
        oSxmDataServiceStatusFi.vRelease();
    }
    if (bEmit) {
        oSxmDataServiceStatusFi.vNotify();
    }
}

tVoid fc_sxm_tclAppManager::vOnDataServiceStatusChanged(fc_sxm_tenServiceID enServiceId, DATASERVICE_STATE_ENUM enNewState, tBool bEmit) {
    if (enServiceId==fc_sxm_enServiceID_All || enServiceId==fc_sxm_enServiceID_AllData || enServiceId==fc_sxm_enServiceID_Audio) {
        ETG_TRACE_ERR(("fc_sxm_tclAppManager::vOnDataServiceStatusChanged(%d): invalid serviceId",
                       ETG_CENUM(fc_sxm_tenServiceID, enServiceId)));
        return;
    }

    DATASERVICE_STATE_ENUM &enOldStateRef=_mapDataServiceStates[enServiceId];
    DATASERVICE_STATE_ENUM enOldState=enOldStateRef;
    if (enOldState != enNewState) {
        ETG_TRACE_USR2(("fc_sxm_tclAppManager::vProcessServiceState(%d): SmsState:%d->%d",
                        ETG_CENUM(fc_sxm_tenServiceID, enServiceId),
                        ETG_CENUM(DATASERVICE_STATE_ENUM, enOldState),
                        ETG_CENUM(DATASERVICE_STATE_ENUM, enNewState)));
        enOldStateRef = enNewState;
        if (enOldState != DATASERVICE_STATE_INVALID) {
            if (enNewState==DATASERVICE_STATE_STOPPED) {
                if (_enRunningAction==fc_sxm_enAction_AppMgrStopData) {
                    vHandleDataAppStopped();
                }
            }
            if (enNewState==DATASERVICE_STATE_READY) {
                if (_enRunningAction==fc_sxm_enAction_AppMgrStartData) {
                    vHandleDataAppStarted(enServiceId);
                }
            }
        }
    }

    // evaluate and emit new fi-status
    vUpdateDataStatusFi(enServiceId, enOldState, enNewState, bEmit);
}

/* Audio-app indictated that sms is ready to be used by data-services.
   broadcast to all data-applications the start-command.
   broadcast is realized by special sxm-service ids.
*/
tVoid fc_sxm_tclAppManager::vStopAction(fc_sxm_tenAction enActionType) {
    if (_enRunningAction!=enActionType) {
        return;
    }
    ETG_TRACE_USR3(("fc_sxm_tclAppManager::vStopAction(%d)",
                    ETG_CENUM(fc_sxm_tenAction, enActionType)));

    switch (_enRunningAction) {
        case fc_sxm_enAction_AppMgrStopData:
        {
            fc_sxm_trMsgAudioDataStopped rMsg;
            fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);  
        }
        break;
        default:
            break;
         
    }
    _oSvTimer.vStop();
    _enRunningAction=fc_sxm_enAction_Invalid;
}



tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgLoopBack_DecoderState const *prMsg) {

    tBool bReady=prMsg->bReady;
    ETG_TRACE_USR1(("fc_sxm_tclAppManager::fc_sxm_trMsgLoopBack_DecoderState: bReady=%d",
                    bReady));
    if (_enRunningAction!=fc_sxm_enAction_Invalid) {
        // terminate running actions
        vStopAction(_enRunningAction);
    } 

    _bDecoderReady=bReady;
    if (bReady) {
        // provide initial device-positon to sms
        vSetDevicePosition(_rNextDevicePosition);

        _u32DataStartIntervalMs=fc_sxm_tclConfig::instance()->u32GetDataServiceMaxStartTimeMs();
        _u32DevicePosIntervalMs=fc_sxm_tclConfig::instance()->u32GetDevicePosTimeMs();

        if (_u32DataStartIntervalMs) {
            vSetAction(fc_sxm_enAction_AppMgrStartData, _u32DataStartIntervalMs);
            _enCurDataApp=(fc_sxm_tenServiceID)fc_sxm_enServiceID_Audio;
            //start SMSe SDK for SMSe services
        	tS32 s32SxmSdkStart = sxm_sdk_start(c8ModuleName, cb_c8GetSportsSDKFilePath, cb_s32LogInterface);
    	    ETG_TRACE_USR4(("fc_sxm_tclAppManager::sxm_sdk_start:%d", s32SxmSdkStart));
    	    if (s32SxmSdkStart == SXM_E_FAULT)
    	    {
    		    ETG_TRACE_USR4(("fc_sxm_tclAppManager::sxm_sdk_start failed"));
    	    }
            vHandleDataAppStarted(_enCurDataApp);
        }
        else {

            fc_sxm_trMsgCmdDataAppStartService rOutMsg;
            vDistributeMessage(rOutMsg);
        }
    } else {
        vSetAction(fc_sxm_enAction_AppMgrStopData, 0);
        fc_sxm_trMsgCmdDataAppStopService rOutMsg;
        vDistributeMessage(rOutMsg);
        vHandleDataAppStopped();
        _rActiveDevicePosition=fc_sxm_trDsrlLocation();
    }
}

tVoid fc_sxm_tclAppManager::vHandleDataAppStarted(fc_sxm_tenServiceID enServiceId) {
    tBool bDone=FALSE;
    ETG_TRACE_USR1(("fc_sxm_tclAppManager::vHandleDataAppStarted: enServiceId=%d, _enCurDataApp=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId),
                    ETG_CENUM(fc_sxm_tenServiceID, _enCurDataApp)));

    if ((fc_sxm_tclSystemStates::instance()->bIsSMSLIBCvOn())) {
    	return;
    }
    if (enServiceId != _enCurDataApp) {
        return;
    }
    SXM_IF_FIND_MAP(fc_sxm_tenServiceID, DATASERVICE_STATE_ENUM, iter,_mapDataServiceStates, _enCurDataApp) {
        ++iter;
    }
    else {
        // set first element
        iter=_mapDataServiceStates.begin();
    }
    while (iter !=_mapDataServiceStates.end() && iter->second==DATASERVICE_STATE_READY) {
        // to avoid race-conditions, send start-command even if data-service is already ready
        fc_sxm_trMsgCmdDataAppStartService rOutMsg;
        _mapAppRegisterDetails[iter->first]->vPostMsgNew(rOutMsg);
    }
    if (iter!= _mapDataServiceStates.end()) {
        // there is still a dataservice left, that is not ready and we did not send a message yet
        fc_sxm_trMsgCmdDataAppStartService rOutMsg;
        _enCurDataApp=iter->first;
        fc_sxm_tclBaseApp *poBaseApp= _mapAppRegisterDetails[_enCurDataApp];
        if (poBaseApp == OSAL_NULL) {
            return;
        }
        poBaseApp->vPostMsgNew(rOutMsg);
        ETG_TRACE_USR4(("fc_sxm_tclAppManager::vHandleDataAppStarted: request start of %d",
                        ETG_CENUM(fc_sxm_tenServiceID, _enCurDataApp)));
        _oSvTimer.vStart(_u32DataStartIntervalMs);
    } else {
        // all data-services have been notified.
        bDone=TRUE;
        vStopAction(fc_sxm_enAction_AppMgrStartData);
    }
} 


tVoid fc_sxm_tclAppManager::vHandleDataAppStopped() {
    // check if all data-apps are stopped
    tBool bAllStopped =TRUE;
    SXM_FOREACH_MAP(fc_sxm_tenServiceID, DATASERVICE_STATE_ENUM, iter, _mapDataServiceStates) {
        ETG_TRACE_USR4(("fc_sxm_tclAppManager::vHandleDataAppStopped() check %x:%d",
                        ETG_CENUM(fc_sxm_tenServiceID, iter->first),
                        ETG_CENUM(DATASERVICE_STATE_ENUM, iter->second)));
        if (iter->second != DATASERVICE_STATE_STOPPED) {
            bAllStopped = FALSE;
            break;
        }
    }

    if (bAllStopped) {
        ETG_TRACE_USR1(("fc_sxm_tclAppManager::vHandleDataAppStopped: All Stopped now"));
        vStopAction(fc_sxm_enAction_AppMgrStopData);
        sxm_sdk_stop();
    }
}

// message containing antenna status
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgLoopBack_AntennaState const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgLoopBack_AntennaState)"));
    fc_sxm_trMsgCmdDataAppAntennaState rOutMsg;
    rOutMsg.rAntennaState = prMsg->eAntennaState;
    vDistributeMessage(rOutMsg);
}

// message containing signal state
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgLoopBack_SignalState const *prMsg)
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgLoopBack_SignalState)"));
#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
    _enSignalState = prMsg->eSignalState;
#endif
    fc_sxm_trMsgCmdDataAppSignalState rOutMsg;
    rOutMsg.eSignalState = prMsg->eSignalState;
    vDistributeMessage(rOutMsg);
}

tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrDestination const *prMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrDestination) START"));
    /* 
       the following message has the service-di fc_sxm_enServiceID_AllData:
       It will be distributed to all data-apps.
    */
    fc_sxm_trMsgDataAppSetDestLocation rOutMsg;
    rOutMsg.rLocation=prMsg->rLocation;
    vDistributeMessage(rOutMsg);
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrDestination) END"));
}
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParams const *prMsg){
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParams) START"));
    /*
       the following message has the service-di fc_sxm_enServiceID_AllData:
       It will be distributed to all data-apps.
    */
    fc_sxm_trMsgDataAppSetClockParams rOutMsg;
    rOutMsg.strTimeZone = prMsg->strTimeZone;
    rOutMsg.bDSTStatus = prMsg->bDSTStatus;
    rOutMsg.oLocalTime = prMsg->oLocalTime;
    vDistributeMessage(rOutMsg);
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParams) END"));
}

// Message sent from Sports App to Audio App to notify a Broadscope change
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrSportsMetadataUpdate const *prMsg) {
	ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrSportsMetadatUpdate) bMetadataUpdate = %u",
				prMsg->bMetadataUpdate));
	fc_sxm_trMsgAppSportsMetaDataUpdate rOutMsg;
	rOutMsg.bMetaDataUpdate = prMsg->bMetadataUpdate;
	fc_sxm_tclAudioApp::instance()->vPostMsgNew(rOutMsg);
}

tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParameters const *prMsg){
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParameters) START"));
    /*
       the following message has the service-di fc_sxm_enServiceID_AllData:
       It will be distributed to all data-apps.
    */
    fc_sxm_trMsgDataAppSetClockParameters rOutMsg;
    rOutMsg.strTimeZone = prMsg->strTimeZone;
    rOutMsg.bDSTStatus = prMsg->bDSTStatus;
    vDistributeMessage(rOutMsg);
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrClockParameters) END"));
}

tVoid fc_sxm_tclAppManager::vSetDevicePosition(fc_sxm_trDsrlLocation rNextDevicePosition) {
    _rNextDevicePosition=rNextDevicePosition;
    tBool bChanged=(_rNextDevicePosition != _rActiveDevicePosition);
    tBool bTimerRunning= _oPosTimer.bIsRunning();
    tBool bSend = _bDecoderReady && bChanged && (!bTimerRunning);

    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vSetDevicePosition: _bDecoderReady=%d bChanged=%d bTimerRunning=%d -> bSend=%d",
                    _bDecoderReady,
                    bChanged,
                    bTimerRunning,
                    bSend));
    if (bSend)
    {
        fc_sxm_trMsgDataAppSetDeviceLocation rMsgSetDeviceLocation;
        rMsgSetDeviceLocation.rLocation=rNextDevicePosition;
        vDistributeMessage(rMsgSetDeviceLocation);
        _oSmsDecouple.vSetSmsDevicePosition(rMsgSetDeviceLocation.rLocation);
        _oPosTimer.vStart(_u32DevicePosIntervalMs);
    }
}

tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAudioMStartSetDevicePostion const *prMsg)
{
	ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAudioMStartSetDevicePostion):WGS84 lat=%f long=%f" ,
	                    (tF32)(prMsg->oFiMsg.Latitude),
	                    (tF32)(prMsg->oFiMsg.Longitude) ));

	if(prMsg->bValid())
	{
		vSetDevicePosition(*prMsg);

		//Set Device for SDK based services
		fc_sxm_trMsgDataAppNavPosition rAppNavPos;
		rAppNavPos.rNavLocation.s32Latitude = prMsg->oFiMsg.Latitude;
		rAppNavPos.rNavLocation.s32Longitude = prMsg->oFiMsg.Longitude;
		vDistributeMessage(rAppNavPos);
	}

}

tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrSmsDbError const *prMsg) const {
    // app reports db-error, let diag deal with it
    ETG_TRACE_USR1(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrSmsDbError) app:%d",
                    ETG_CENUM(fc_sxm_tenServiceID, prMsg->enServiceId)));

    fc_sxm_trMsgDiagSmsDbError rMsg(prMsg->enServiceId);
    fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
}


// restore sms-db of single app
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrRestoreSmsDb const *prMsg) const {
    ETG_TRACE_USR1(("fc_sxm_tclAppManager::vProcess(fc_sxm_trMsgAppMgrRestoreSmsDb) App=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, prMsg->enServiceId)));
    fc_sxm_trMsgAppRestoreSmsDb rMsg(prMsg->enServiceId);
    vSendToBaseApp(rMsg);
}

// Here we receive the loopback-message for us (cca-context)
tVoid fc_sxm_tclAppManager::vDispatchMsg(fc_sxm_tclMessage *poThreadMsg) {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vDispatchMsg: SID=%x action=%d START", 
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()), 
                    poThreadMsg->u16GetActionOnly()));
    tU32 u32Action=poThreadMsg->u32GetAction();
    switch (u32Action) {
      
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgLoopBack_DecoderState,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrDestination,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgLoopBack_AntennaState,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgLoopBack_SignalState,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrDataAppServiceStatus,this);
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioCvmEvent, fc_sxm_tclSystemStates::instance());
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrSmsDbError,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrRestoreSmsDb,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrClockParams,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAudioMStartSetDevicePostion,this);
        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrSportsMetadataUpdate, this);
		#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrProfileDataChangeUpdate, fc_sxm_tcl_profilemanager::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrProfileActiveProfileUpdate, fc_sxm_tcl_profilemanager::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrProfileCopyProfileUpdate, fc_sxm_tcl_profilemanager::instance());
		SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgAppMgrProfileDeleteProfileUpdate, fc_sxm_tcl_profilemanager::instance());
		#endif
        default:
            ETG_TRACE_USR4(("fc_sxm_tclAppManager::vDispatchMsgFromQ(): unknown u32Action=0x%x",
                            u32Action));
            break;         
    }

}

tVoid fc_sxm_tclAppManager::vHandleTtfisCmd(tU8 u8MsgCode, tU8 const *pu8Data) {
    fc_sxm_tenTtfisCmdsAppManager enMsgCode=(fc_sxm_tenTtfisCmdsAppManager)u8MsgCode;
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vHandleTtfisCmd(): code=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsAppManager, enMsgCode)));

    switch (enMsgCode) {
        case fc_sxm_enTtfisCmdsAppManager_SetPosFiDeviceLocation:
        {
			ETG_TRACE_ERR(("Setting device location through TTFIs cmd not handled anymore ... Refer NAVI TTFIs cmd for setting device location"));
        }
        break;
        case fc_sxm_enTtfisCmdsAppManager_SetDestinationLocation:
        {
			ETG_TRACE_ERR(("Setting device destination through TTFIs cmd not handled anymore  ... Refer NAVI TTFIs cmd for setting device destination"));
        }
        break;
        case fc_sxm_enTtfisCmdsAppManager_PrintReport:
        {
            ETG_TRACE_ERR(("fc_sxm_enTtfisCmdsAppManager_PrintReport"));
            fc_sxm_trMsgCmdDiagPrintReport rMsg;
            fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg); 
        }
        break;
        case fc_sxm_enTtfisCmdsAppManager_LAST:
        default:
            ETG_TRACE_ERR(("fc_sxm_tclAppManager::vHandleTtfisCmd(): Unsupported code=%d",
                           ETG_CENUM(fc_sxm_tenTtfisCmdsAppManager, enMsgCode)));            
            break;
    }   
}


// ttfis-cmds for appManager  and appThreads
tVoid fc_sxm_tclAppManager::vDistributeTtfisCmd(fc_sxm_trMsgCmdAppTtfisCmd *prMsg) const {
    fc_sxm_tenServiceID enServiceId =prMsg->enServiceId;
    ETG_TRACE_USR2(("fc_sxm_tclAppManager::vDistributeTtfisCmd(): enServiceId=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId)));
    // send message to specific sxm-app
    vSendToBaseApp(*prMsg);
}

tVoid fc_sxm_tclAppManager::vTraceDistributeStart(fc_sxm_tclMessage const *poMsg) const {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vTraceDistributeStart: SID=%x action=%d START", 
                    ETG_CENUM(fc_sxm_tenServiceID, poMsg->enGetServiceId()), 
                    poMsg->u16GetActionOnly()));
}
tVoid fc_sxm_tclAppManager::vTraceDistributeEnd(fc_sxm_tclMessage const *poMsg) const {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vTraceDistributeEnd: SID=%x action=%d START", 
                    ETG_CENUM(fc_sxm_tenServiceID, poMsg->enGetServiceId()), 
                    poMsg->u16GetActionOnly()));
}



tVoid fc_sxm_tclAppManager::vHandleTimer(tU16 u16TimerId) const
{
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vHandleTimer(u16TimerId=%d) _enRunningAction=%d",
                    u16TimerId, 
                    ETG_CENUM(fc_sxm_tenAction, _enRunningAction)));
    trTimerBase::vHandleTimer(u16TimerId);
}


tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_tclAppManager::trSvTimer *prMsg) 
{
    (void)prMsg;
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(trSvTimer) _enRunningAction=%d",
                    ETG_CENUM(fc_sxm_tenAction, _enRunningAction)));
    if (_enRunningAction==fc_sxm_enAction_AppMgrStopData) {
        // Check if all the data services are stopped
    	vHandleDataAppStopped();
    }
    else if (_enRunningAction==fc_sxm_enAction_AppMgrStartData) {
        // action might need to continue to notify remaining apps
        vHandleDataAppStarted(_enCurDataApp);
    }
}

tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_tclAppManager::trDevicePosTimer *prMsg)
{
    (void)prMsg;
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::vProcess(trDevicePosTimer)"));
    // set next postion in case it has changed while timer was running
    vSetDevicePosition(_rNextDevicePosition);
}



#if(FC_SXM_ENABLE_WATCHDOG_TIMER)

tVoid fc_sxm_tclAppManager::vSXMWatchdogReset(tVoid)
{
	ETG_TRACE_USR4(("(vSXMWatchdogReset)"));
	_u32SXMWatchdogCycles = 0;
}

tVoid fc_sxm_tclAppManager::vStartSXMWatchdog(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_trSXMWatchdog start timer"));
	_u32SXMWatchdogCycles = 0;
	_oSXMWatchdogTimer.vStart(FC_SXM_WATCHDOG_INTERVAL);
}

tVoid fc_sxm_tclAppManager::vStopSXMWatchdog(tVoid)
{
	ETG_TRACE_USR4(("fc_sxm_trSXMWatchdog stopped"));
	_oSXMWatchdogTimer.vStop();
	_u32SXMWatchdogCycles = 0;
}
#endif


#if(FC_SXM_ENABLE_WATCHDOG_TIMER)
tVoid fc_sxm_tclAppManager::vProcess(fc_sxm_tclAppManager::trSXMWatchdogTimer *prMsg)
{
   (void)prMsg;
   tU32 u32MaxCycles = 0;
   ETG_TRACE_USR4(("vProcess(trSXMWatchDogTimer)"));

   if(fc_sxm_tclAudioProxy::instance()->eGetAudioProxyState() != fc_sxm_enAudioReady )
   {
      /* Restart watchdog timer */
      _oSXMWatchdogTimer.vStart(FC_SXM_WATCHDOG_INTERVAL);

      _u32SXMWatchdogCycles++;

      ETG_TRACE_USR2(("SXM Audio state not ready, watchdog cycle count: %d", _u32SXMWatchdogCycles));

      u32MaxCycles = fc_sxm_tclConfig::instance()->u32GetWatchdogMaxCyclesTimeMs();

      if (_u32SXMWatchdogCycles == (u32MaxCycles - 1))
      {
         // Raise the signal to get the callstack the threads
         raise(SIGUSR2);
      }

      if(ARL_EN_ISRC_ACT_ON == fc_sxm_arl_tclISource::instance()->enGetSXMSourceState()) 
      {
         ETG_TRACE_USR2(("Current source is SXM"));

         if(_u32SXMWatchdogCycles >= u32MaxCycles)
         {
        	 // Raise the ABORT signal on WDT expired (2 secs.)
        	 raise(SIGABRT);
         }
	  }
   }
   else 
   {
      ETG_TRACE_USR2(("SXM Audio CCA state is ready"));

      _u32SXMWatchdogCycles = 0;
   }
}
#endif

tVoid fc_sxm_tclAppManager::vStartTimer(tU16 u16TimerID, tU32 u32Ms)
{
	tBool bResult;
	bResult = fc_sxm_tclApp::instance()->vStartTimer(u16TimerID, u32Ms);
	ETG_TRACE_USR2(("vStartTimer return value - %d", bResult));
}

tVoid fc_sxm_tclAppManager::vStopTimer(tU16 u16TimerID)
{
    fc_sxm_tclApp::instance()->vStopTimer(u16TimerID);
}

fc_sxm_tclAppManager::trTimerBase::trTimerBase():
    _bRunning(FALSE)
{
    _setTimers.insert(this);
}

fc_sxm_tclAppManager::trTimerBase::~trTimerBase() 
{
    _setTimers.erase(this);
}



tVoid fc_sxm_tclAppManager::trTimerBase::vStart(tU32 u32Ms) {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::trTimerBase::vStart(id=%d) u32Ms=%d",
                    vGetTimerId(), u32Ms));
    if (u32Ms) {
        _bRunning=TRUE;
    }
    fc_sxm_tclAppManager::vStartTimer(vGetTimerId(), u32Ms);
}
tVoid fc_sxm_tclAppManager::trTimerBase::vStop() {
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::trTimerBase::vStop(id=%d)", vGetTimerId()));
    fc_sxm_tclAppManager::vStopTimer(vGetTimerId());
    _bRunning=FALSE;
}


tVoid fc_sxm_tclAppManager::trTimerBase::vHandleTimer(tU16 u16TimerId) {
    tBool bFound=FALSE;
    tBool bRunning=FALSE;
    SXM_FOREACH(set<trTimerBase*>, iter, _setTimers) {
        if ((*iter)->vGetTimerId()==u16TimerId) {
            bFound=TRUE;
            if ((*iter)->bIsRunning()) {
                bRunning=TRUE;
                (*iter)->_bRunning=FALSE;
                (*iter)->vFire();
            }
            break;
        }
    }    
    ETG_TRACE_USR1(("fc_sxm_tclAppManager::trTimerBase::vHandleTimer(%u): bFound=%d bRunning=%d", u16TimerId, bFound, bRunning));

}

fc_sxm_tclAppManager::fc_sxm_tclSmsDecouple::fc_sxm_tclSmsDecouple():
    fc_sxm_tclBaseThread(OSAL_NULL, "smsDecouple")
{}

tVoid fc_sxm_tclAppManager::fc_sxm_tclSmsDecouple::vSetSmsDevicePosition(fc_sxm_trDsrlLocation rDevicePosition) {
    // store parameter in sem-protected data
    _rLocation.rAccess()=rDevicePosition;
    _rLocation.vRelease();
    /* send event to thread-fn, that will call us with vHandleUserEvent in 
       thread-context of our class 
    */
    vPostUserEvent();
}

tVoid fc_sxm_tclAppManager::fc_sxm_tclSmsDecouple::vHandleUserEvent(tU32 u32Mask) {
    SXM_ASSERT_RETURN(FC_SXM_FIRST_USER_EVT == u32Mask);
    // read data from sem-protected data
    fc_sxm_trDsrlLocation rDeviceLocation= _rLocation.rGet();
    // todo: trigger sms with new location;
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED.hCreateFromFixed(rDeviceLocation.s32Latitude, 16);
    OSAL_FIXED_OBJECT hLon = OSAL_FIXED.hCreateFromFixed(rDeviceLocation.s32Longitude, 16);
    tBool bRes=DEVICE.bSetPosition (hLat, hLon);
    ETG_TRACE_USR4(("fc_sxm_tclAppManager::fc_sxm_tclSmsDecouple DEVICE.bSetPosition: bRes=%d lat=%d, long=%d",
                    bRes,
                    rDeviceLocation.s32Latitude,
                    rDeviceLocation.s32Longitude));
    //Destroy the object
    OSAL_FIXED.vDestroy(hLat);
    OSAL_FIXED.vDestroy(hLon);
}


set<fc_sxm_tclAppManager::trTimerBase*> fc_sxm_tclAppManager::trTimerBase::_setTimers;


