/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sms_init.cpp
* @brief       Sms initialization implementation
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include <pthread.h>
#include <signal.h>
#include <execinfo.h>
//SMS OSAL include
//#include "osal.h"
#include "fc_sxm_common.h"

//SMS API include
#include "fc_sxm_sms.h"

#if 0
//SMS SRH driver include
#include "srh.h"

//SMS serial io driver include
#include "sio_shim.h"

//SMS posix serial driver
#include "posix_serial.h"
#endif 

#include "fc_sxm_tcl_sms_init.h"
#include "fc_sxm_trace_macros.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SMS_INIT
#include "trcGenProj/Header/fc_sxm_tcl_sms_init.cpp.trc.h"
#endif

#ifdef GEN3X86
#define FC_SXM_GPIO_ID_SHDN 119
#define FC_SXM_GPIO_ID_RSTN 120
#endif
static fc_sxm_tclSem _oInitSem;
static fc_sxm_tclSem _oGpioSem;
static fc_sxm_tclSem _oTaskRegisterSem;
static int SMSLIB_iNumThreads=0;
static tBool _bDecoderCreated=FALSE;

// Global file handle for GPIO
OSAL_FILE_STRUCT* psGPIO = NULL;

// OSAL Start Handlers
static const OSAL_START_HANDLER_STRUCT gsStartHandlers =
{
    // OSAL Driver Registration Handlers
    fc_sxm_tclSMSInit::bRegisterDrivers,
    fc_sxm_tclSMSInit::vUnRegisterDrivers,
    NULL,

    // OSAL Start Handler
    fc_sxm_tclSMSInit::bStartHandler,
    NULL,

    // OSAL Monitor Handlers
    {
        NULL,
        NULL,

        NULL,
        NULL,

        NULL,
        NULL,
        NULL,
        NULL
    }
};


void *pfSmsInitThreadFn(void *)
{
    ETG_TRACE_USR4(("pfSmsInitThreadFn Start"));

#ifndef GEN3X86
    fc_sxm_tclSMSInit::instance()->vStartX65Module();
#endif

    void *pvRes =(void *)(intptr_t )OSAL.bStart(&gsStartHandlers);
    // start failed, we have to deblock from here

    _oInitSem.vPost();
    ETG_TRACE_USR4(("pfSmsInitThreadFn End"));
    return pvRes;
}

 
/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclSMSInit
 *
 *DESCRIPTION:  Constructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclSMSInit::fc_sxm_tclSMSInit():
#ifdef GEN3X86
    _enX65PowerState(fc_sxm_enX65PowerState_Off),
#else
    _enX65PowerState(fc_sxm_enX65PowerState_Off),
#endif
    _bGpioInitialized(FALSE),
    _bInSmsInitPhase(FALSE),
    _u32X65ResetCounter(0)
{
    _oGpioSem.vOpen();
    _oTaskRegisterSem.vOpen();
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit constructor"));
}

/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclSMSInit
 *
 *DESCRIPTION:  Destructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclSMSInit::~fc_sxm_tclSMSInit()
{
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit destructor"));
}


tVoid vInitializeGpio(const tChar* pcGPIOPath) {

    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vInitializeGpio (%s)", pcGPIOPath));

    FILE *fp;
    fp = fopen(pcGPIOPath, "rb+");
    if (OSAL_NULL == fp) {
        ETG_TRACE_ERR(("fc_sxm_tclSMSInit::vInitializeGpio (%s) fopen FAILED", pcGPIOPath));
        return;
    }

    char set_value[4];
    strcpy(set_value,"out");
    rewind(fp);
    fwrite(&set_value, sizeof(char), 3, fp); 
    fclose(fp); 

    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vInitializeGpio DONE"));
}


tVoid fc_sxm_tclSMSInit::vInitializeGpios() {
#ifndef GEN3X86	
#ifdef SMS_64_BIT_SUPPORT
   if (!_bGpioInitialized)
   {
      _bGpioInitialized=TRUE;
      OSAL_tIODescriptor hFd = OSAL_ERROR;
      tS32 errorCode = OSAL_ERROR;
      OSAL_tenAccess enAccess = OSAL_EN_READWRITE;
      OSAL_tGPIODevID resetID = (OSAL_tGPIODevID) OSAL_EN_RST_XM;
      OSAL_trGPIOData resetData = {resetID, 1};
      OSAL_tGPIODevID shdnID = (OSAL_tGPIODevID) OSAL_EN_CPU_SHDN_XM;
      OSAL_trGPIOData shdnData = {shdnID, 1};
      ETG_TRACE_USR4(("vInitializeGpios new style"));

      hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, enAccess);
      if (hFd != OSAL_ERROR)
      {
         errorCode = OSAL_s32IOControl(hFd, OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE , (intptr_t) &resetData);
         if(OSAL_OK != errorCode)
         {
            ETG_TRACE_USR4(("OSAL_s32IOControl failed for reset: %d", OSAL_u32ErrorCode()));
         }
         errorCode = OSAL_s32IOControl(hFd, OSAL_C_32_IOCTRL_GPIO_SET_ACTIVE_STATE , (intptr_t) &shdnData);
         if (OSAL_OK != errorCode)
         {
            ETG_TRACE_USR4(("OSAL_s32IOControl failed for shdn: %d", OSAL_u32ErrorCode()));
         }
         OSAL_s32IOClose(hFd);
      }
      else
      {
         ETG_TRACE_USR4(("OSAL GPIO device open failed with FD: %d errorCode: %d ", (int)hFd, OSAL_u32ErrorCode()));
      }

   }
#else
	if (!_bGpioInitialized) 
	{
        _bGpioInitialized=TRUE;
        // configure GPIOs
        vInitializeGpio("/sys/class/boardcfg/boardcfg/cpu-shdn-xm/direction");
        vInitializeGpio("/sys/class/boardcfg/boardcfg/cpu-rst-xm/direction");
    }
#endif
    
#endif
}

/*********************************************************************
 *
 *FUNCTION:     InitializeSMS
 *
 *DESCRIPTION:  Initialize the SMS library
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclSMSInit::InitializeSMS(tVoid)
{
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::InitializeSMS START"));

   vInitializeGpios();


    pthread_t hSmsInitThread;
    //    return TRUE;
    _oInitSem.vOpen(0);
        ETG_TRACE_USR4(("fc_sxm_tclSMSInit::InitializeSMS call pthread_create"));



    int rc = pthread_create(&hSmsInitThread, NULL, pfSmsInitThreadFn, NULL);

    return (rc==0);
}

tVoid fc_sxm_tclSMSInit::vWaitSmsInitialized() {
    // thread has been created, so can post the semaphore when ready
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::InitializeSMS call wait sem"));
    _oInitSem.vGet();
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::InitializeSMS START"));
}
/*********************************************************************
 *
 *FUNCTION:     bRegisterDrivers
 *
 *DESCRIPTION:  Register the serial drivers
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclSMSInit::bRegisterDrivers(tVoid *pvArg)
{
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bRegisterDrivers"));

   (tVoid)pvArg;
   tS32 s32RetVal;

   /////////////////////////////
   // Register Serial Port(s)...
   /////////////////////////////

   // Register LSIM serial port
   s32RetVal = OSAL.n32RegNode(
      FC_SXM_SERIAL_DEVICE,
      &GsPosixSerialInterface,
      FC_SXM_SERIAL_DEVICE,
      strlen(FC_SXM_SERIAL_DEVICE) + 1);

   if(s32RetVal != DEV_OK)
   {
      ETG_TRACE_ERR(("Error! Cannot Register %s", FC_SXM_SERIAL_DEVICE));
      return FALSE;
   }

   ETG_TRACE_USR4(("Registered device %s", FC_SXM_SERIAL_DEVICE));

   // Register LSIM serial port via a shim i/o driver.
   // This driver allows us to use the serial port's modem
   // signals as independently controlled gpio.
   s32RetVal = OSAL.n32RegNode(
      FC_SXM_RADIO_NAME,
      &GsSioInterface,
      FC_SXM_SERIAL_DEVICE,
      strlen(FC_SXM_SERIAL_DEVICE) + 1);

   if(s32RetVal != DEV_OK)
   {
      ETG_TRACE_ERR(("Error! Cannot Register %s", FC_SXM_RADIO_NAME));
      return FALSE;
   }

   ETG_TRACE_USR4(("Registered device %s", FC_SXM_RADIO_NAME));

   /////////////////////////////
   // Register SRH Driver
   /////////////////////////////
   SRH_vRegisterDriver(const_cast<char *> FC_SXM_SMS_UART_PARAMS);

   ETG_TRACE_USR4(("Registered SRH driver"));

   return TRUE;
}

/*********************************************************************
 *
 *FUNCTION:     vUnRegisterDrivers
 *
 *DESCRIPTION:  Unregister the serial drivers
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclSMSInit::vUnRegisterDrivers(tVoid *pvArg)
{
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vUnRegisterDrivers"));

   (tVoid)pvArg;

   // Unregister SRH Drivers
   SRH_vUnRegisterDriver();

   OSAL.bUnRegNode(FC_SXM_RADIO_NAME);
   OSAL.bUnRegNode(FC_SXM_SERIAL_DEVICE);

   return;
}

/*********************************************************************
 *
 *FUNCTION:     bStartHandler
 *
 *DESCRIPTION:  Start Handler 
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tBool fc_sxm_tclSMSInit::bStartHandler(const tVoid *pvArg)
{
   (tVoid)pvArg;
   tBool bTempDirSet;


   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bStartHandler START"));

   //////////////////////////////////////////
   // Set the Temp. Directory Path
   //////////////////////////////////////////
   bTempDirSet = OSAL.bFileSystemSetTempPath(FC_SXM_SMS_TEMP_DIR_PATH);

   if (bTempDirSet == FALSE)
   {
      ETG_TRACE_ERR(("Error! Cannot set the temp path"));
      return FALSE;
   }
   else
   {
      ETG_TRACE_USR4(("set the temp path successfully"));
   }

   // Open Serial-I/O shim for SXM-Module GPIO. We need to assign
   // our global file handle so that our SRH HW implementation
   // can use the modem signals to drive power and reset of the
   // SXM-module.
#ifdef  GEN3X86
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bStartHandler: call psFopen"));
   psGPIO = OSAL.psFopen(FC_SXM_RADIO_NAME, SIO_SHIM_IO_MODE);
   if(psGPIO == NULL)
   {
      ETG_TRACE_ERR(("Error! Cannot open device %s for SXM-Module GPIO", FC_SXM_RADIO_NAME));
      return FALSE;
   }
   else
   {
      ETG_TRACE_USR4(("Opened device FC_SXM_RADIO_NAME for SXM-Module GPIO"));
   }
#endif

   ETG_TRACE_USR4(("SMS Init success!"));

   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bStartHandler: call _oInitSem.vPost"));
   _oInitSem.vPost();
   ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bStartHandler: DONE"));
   return TRUE;
}



/****************For External Diagnosis mode****************/

//Open the specified port for writing
static FILE* iOpenPort(const tChar* pcGPIOPath)
{
   FILE * fp;
   fp = fopen(pcGPIOPath, "rb+");
   if (OSAL_NULL == fp) {
       ETG_TRACE_USR4(("fp is null"));
   }
   else
   {
      ETG_TRACE_USR4(("fp success"));
   }
   return fp;
}

//Write the specifed value to successfully opened port
static tVoid vSetPort(const tChar* pcGPIOPath, tBool bOn)
{   
   FILE *fp = iOpenPort(pcGPIOPath);

   if (OSAL_NULL != fp)
   {
      char cSigValue = (bOn)? '1' : '0';
      rewind(fp);
      fwrite(&cSigValue, sizeof(char), 1, fp); 
      fclose(fp); 
   }
}

#ifdef GEN3X86
extern "C" void SRH_vSetLineSignal(int iGpio, int iValue);
static tVoid vSetPower(tBool bOn) {
   ETG_TRACE_USR4(("vSetPower = %d",bOn));
   SRH_vSetLineSignal(FC_SXM_GPIO_ID_SHDN, bOn ? 1 : 0);
}

//Set High/low for reset pin of  to X65 module
static tVoid vSetReset(tBool bOn) {
    ETG_TRACE_USR4(("vSetReset = %d",bOn));
   SRH_vSetLineSignal(FC_SXM_GPIO_ID_SHDN, bOn ? 0 : 1);
}

#else
//Switch on/off power to X65 module
static tVoid vSetPower(tBool bOn) {
   ETG_TRACE_USR4(("vSetPower = %d",bOn));
   vSetPort("/sys/class/boardcfg/boardcfg/cpu-shdn-xm/value", bOn);
}

//Set High/low for reset pin of  to X65 module
static tVoid vSetReset(tBool bOn) {
    ETG_TRACE_USR4(("vSetReset = %d",bOn));
    vSetPort("/sys/class/boardcfg/boardcfg/cpu-rst-xm/value", bOn);
}
#endif

/*Two steps are involved in starting the X65 module
  1.)Switch on the power to X65.Wait for 250 ms as mentioned in SXM 
     specification before doing any activity.
  2.)Set the reset pin of X65 to high.It would have been pulled 
     to reset when SMS was shutting down.Again wait for 250 ms before 
     doing any activity
*/
tVoid fc_sxm_tclSMSInit::vStartX65Module() 
{
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vStartX65Module enterd  _enX65PowerState=%d",
                    ETG_CENUM(fc_sxm_tenX65PowerState, _enX65PowerState)));  

    fc_sxm_tclScopedLock lock(_oGpioSem);
    vInitializeGpios();
    _bInSmsInitPhase=TRUE;

    if (_enX65PowerState==fc_sxm_enX65PowerState_Off) {
        _enX65PowerState=fc_sxm_enX65PowerState_On;
        _u32X65ResetCounter=0;
        vSetPower(true);
        OSAL_s32ThreadWait(250);
        vSetReset(true);
        OSAL_s32ThreadWait(100);
    }
}

tVoid fc_sxm_tclSMSInit::vStopX65Module() 
{
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vStopX65Module START"));  
    fc_sxm_tclScopedLock lock(_oGpioSem);
    if (_enX65PowerState==fc_sxm_enX65PowerState_On ||
       _enX65PowerState ==fc_sxm_enX65PowerState_AutoOn) {
        _enX65PowerState=fc_sxm_enX65PowerState_Off;
        _u32X65ResetCounter=0;
        vSetReset(false);
        OSAL_s32ThreadWait(250);
        vSetPower(false);
        OSAL_s32ThreadWait(50);
    }

}

tVoid fc_sxm_tclSMSInit::vResetX65Module() 
{
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vResetX65Module enterd _u32X65ResetCounter=%d", _u32X65ResetCounter));  
    fc_sxm_tclScopedLock lock(_oGpioSem);

    if (_u32X65ResetCounter) {
        _bInSmsInitPhase=TRUE;
        vSetReset(false);
        OSAL_s32ThreadWait(100);
        vSetReset(true);
    }
    _u32X65ResetCounter++;

}





tVoid fc_sxm_tclSMSInit::vPatchPrio(string oThreadName, size_t tThreadId, int &iPrioVal) 
{
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vPatchPrio? tThreadId=0x%08x prio:%d oThreadName=%s", 
                    tThreadId, iPrioVal, oThreadName.c_str()));  
    if (_bInSmsInitPhase) {
        if (iPrioVal==OSAL_TASK_PRIORITY_LOW) {
            int iPatchedPrioVal=OSAL_TASK_PRIORITY_MEDIUM;
            ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vPatchPrio! tThreadId=0x%08x prio:%d-->%d oThreadName=%s", 
                            tThreadId, iPrioVal, iPatchedPrioVal, oThreadName.c_str()));  
            fc_sxm_trSmsThreadPatchInfo rPatchInfo(oThreadName, tThreadId, iPrioVal, iPatchedPrioVal);
            _qThreadPatchInfos.push_back(rPatchInfo);
            iPrioVal=iPatchedPrioVal;
        }
    }
}

extern "C"
{
    OSAL_RETURN_CODE_ENUM OS_eTaskChangePriority (
                                                  OSAL_OBJECT_HDL hTaskObj,
                                                  OSAL_TASK_PRIORITY_ENUM eNewPriority);
}


tVoid fc_sxm_tclSMSInit::vRestoreThreadPrios() {
    _bInSmsInitPhase=FALSE;
    // todo: semaphore protection
    for  (deque<fc_sxm_trSmsThreadPatchInfo>::const_iterator iter=_qThreadPatchInfos.begin();
          iter!=_qThreadPatchInfos.end();++iter) {
        fc_sxm_trSmsThreadPatchInfo const &rThreadInfo=*iter;
        ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vRestoreThreadPrios tThreadId=0x%08x prio:%d-->%d oThreadName=%s", 
                        rThreadInfo._tThreadId, rThreadInfo._iPatchedPrioVal, rThreadInfo._iOrigPrioVal, rThreadInfo._oThreadName.c_str()));  
        OS_eTaskChangePriority((OSAL_OBJECT_HDL )rThreadInfo._tThreadId, (OSAL_TASK_PRIORITY_ENUM)rThreadInfo._iOrigPrioVal);
    }
    _qThreadPatchInfos.clear();
    
}



tVoid fc_sxm_tclSMSInit::vSetX65AutoOn() {
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vSetX65AutoOn _enX65PowerState=%d",
                    ETG_CENUM(fc_sxm_tenX65PowerState, _enX65PowerState)));
    fc_sxm_tclScopedLock lock(_oGpioSem);

#ifndef GEN3X86
    // we get informed after startup or at end of CV that the module has been started automatically, e.g. by HW
    _enX65PowerState=fc_sxm_enX65PowerState_AutoOn;
#endif
    _bInSmsInitPhase=TRUE;

}

tVoid fc_sxm_tclSMSInit::vSmsLibX65PowerOnRequest() {
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vSmsLibX65PowerOnRequest _enX65PowerState=%d",
                    ETG_CENUM(fc_sxm_tenX65PowerState, _enX65PowerState)));
    fc_sxm_tclScopedLock lock(_oGpioSem);
    vStartX65Module();
    _enX65PowerState=fc_sxm_enX65PowerState_On;

}

tVoid fc_sxm_tclSMSInit::vSmsLibX65PowerOffRequest() {
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::vSmsLibX65PowerOffRequest _enX65PowerState=%d",
                    ETG_CENUM(fc_sxm_tenX65PowerState, _enX65PowerState)));

    fc_sxm_tclScopedLock lock(_oGpioSem);
    switch (_enX65PowerState) {
        case fc_sxm_enX65PowerState_On:
            vStopX65Module();
            _enX65PowerState=fc_sxm_enX65PowerState_Off;
            break;
        case fc_sxm_enX65PowerState_AutoOn:
        case fc_sxm_enX65PowerState_Off:
        default:
            break;
    }
}


tBool  fc_sxm_tclSMSInit::bWaitDecoderDown(tU32 u32MaxMs) {
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bWaitDecoderDown u32MaxMs=%u", u32MaxMs));
    tBool bRes=FALSE;
    tU32 u32Cyles=(u32MaxMs / 100) + 1;
    for (tU32 i= 0; i < u32Cyles; i++) {
        ETG_TRACE_USR4(("fc_sxm_tclSMSInit::_bDecoderCreated=%u", _bDecoderCreated));
        if (!_bDecoderCreated) {
            bRes=TRUE;
            break;
        }
        OSAL_s32ThreadWait(100);
    }

    OSAL_s32ThreadWait(100);
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bWaitDecoderDown bRes=%u", bRes));
    return bRes;
}
tBool fc_sxm_tclSMSInit::bWaitSmsDown(tU32 u32MaxMs) {
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bWaitSmsDown u32MaxMs=%u", u32MaxMs));
    tBool bRes=FALSE;
    tU32 u32Cyles=(u32MaxMs / 100) + 1;
    for (tU32 i= 0; i < u32Cyles; i++) {
        ETG_TRACE_USR4(("fc_sxm_tclSMSInit::SMSLIB_iNumThreads=%u", SMSLIB_iNumThreads));
        if (SMSLIB_iNumThreads <= 3) {
            bRes=TRUE;
            break;
        }
        OSAL_s32ThreadWait(100);
    }

    OSAL_s32ThreadWait(100);
    ETG_TRACE_USR4(("fc_sxm_tclSMSInit::bWaitSmsDown bRes=%u", bRes));
    return bRes;
}


extern "C"
{
    void fc_sxm_vSetPatchedPrio(char const *szThreadName,  OSAL_OBJECT_HDL tThreadId, OSAL_TASK_PRIORITY_ENUM *piPrioVal)
    {
        int iPrioCopy=(int)*piPrioVal;
        fc_sxm_tclSMSInit::instance()->vPatchPrio(szThreadName, (size_t)tThreadId, iPrioCopy);
        *piPrioVal=(OSAL_TASK_PRIORITY_ENUM)iPrioCopy;
    }


    // Interface to be used by smslib to control gpios of x65 module
    void fc_sxm_vSwitchOnX65() {
        fc_sxm_tclSMSInit::instance()->vSmsLibX65PowerOnRequest();
    }
    void fc_sxm_vSwitchOffX65() {
        fc_sxm_tclSMSInit::instance()->vSmsLibX65PowerOffRequest();
    }
    void fc_sxm_vResetX65() {
        fc_sxm_tclSMSInit::instance()->vResetX65Module();
    }
    int fc_sxm_bDoSmsLogging() {
        return etg_bIsTraceActive(TR_CLASS_FC_SXM_SMSLIB_SMS_LOGGING, 5);
    }


#define FC_SXM_MAX_BACKTRACE_ENTRIES 10

    tVoid fc_sxm_vPrintStack() {
        if (etg_bIsTraceActive(TR_CLASS_FC_SXM_SMS_INIT, 1)) {
            void *aEntries[FC_SXM_MAX_BACKTRACE_ENTRIES];
            size_t nEntries;
            char **ppFnNames;
            size_t i=0;
            
            nEntries=backtrace(aEntries, FC_SXM_MAX_BACKTRACE_ENTRIES);
            ppFnNames = backtrace_symbols(aEntries, (int)nEntries);
            ETG_TRACE_ERR(("BACKTRACE: %d frames", nEntries));
            for (i=0; i < nEntries; i++) {
                ETG_TRACE_ERR(("%s\n", ppFnNames[i]));
            }
            ETG_TRACE_ERR(("\n"));
            free(ppFnNames);
        }
    }
}



