/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_sem.cpp
* @brief       Implementation of the semaphore functionality
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include <string>

using namespace std;

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"


#include "fc_sxm_trace_macros.h"
#include "fc_sxm_unique_name.h"
#include "fc_sxm_tcl_sem.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_SEM
#include "trcGenProj/Header/fc_sxm_tcl_sem.cpp.trc.h"
#endif

fc_sxm_tclSem::fc_sxm_tclSem(tVoid) {
    _hSem = OSAL_C_INVALID_HANDLE;
    _u32Count = 0;
    _u32Tid=0;
}

fc_sxm_tclSem::~fc_sxm_tclSem(tVoid) {
    vClose();
    _hSem = OSAL_C_INVALID_HANDLE;
    _u32Count = 0;
    _u32Tid=0;
}

tVoid fc_sxm_tclSem::vOpen(tU32 u32InitVal) {
    if (_hSem != OSAL_C_INVALID_HANDLE) {
       
        ETG_TRACE_ERR(("fc_sxm_tclSem:vOpen(0x%p): already opened",(tVoid *) this));
        return;
    }

    _oSemName = fc_sxm_tclUniqueName::instance()->oGetUniqueName();
    if(OSAL_ERROR  == OSAL_s32SemaphoreCreate(/*tCString*/        _oSemName.c_str(),
                                              /*OSAL_tSemHandle*/ &_hSem,
                                              /*uCount*/          (tU32)u32InitVal))
    {
        ETG_TRACE_ERR(("fc_sxm_tclSem:vOpen(0x%p): OSAL_s32SemaphoreCreate (%s) ERROR", (tVoid *)this, _oSemName.c_str()));
        _hSem = OSAL_C_INVALID_HANDLE;	
    }
    ETG_TRACE_USR2(("fc_sxm_tclSem:vOpen(0x%p)", (tVoid *)this));
        
}
tVoid fc_sxm_tclSem::vClose(tVoid) {
    if (_hSem == OSAL_C_INVALID_HANDLE) {
        return;
    }
        
    if(OSAL_ERROR  == OSAL_s32SemaphoreClose(_hSem))
    {
            
        ETG_TRACE_ERR(("fc_sxm_tclSem:vClose:OSAL_s32SemaphoreClose (%s) ERROR", _oSemName.c_str()));
        _hSem = OSAL_C_INVALID_HANDLE;
        return;
    }
    _hSem = OSAL_C_INVALID_HANDLE;

    if(OSAL_ERROR  == OSAL_s32SemaphoreDelete(_oSemName.c_str()))
    {               
        ETG_TRACE_ERR(("fc_sxm_tclSem:vClose:OSAL_s32SemaphoreDelete (%s) ERROR", _oSemName.c_str()));	
            
    }
    _u32Count = 0;
    _u32Tid=0;
    ETG_TRACE_USR2(("fc_sxm_tclSem:vClose(0x%p)", (tVoid *)this));

}

tVoid fc_sxm_tclSem::vGet() {
    bGet(OSAL_C_TIMEOUT_FOREVER);
}

tBool fc_sxm_tclSem::bGet(tU32 u32Timeout) {
    tBool bRes=FALSE;
    if (_hSem == OSAL_C_INVALID_HANDLE) {
        ETG_TRACE_USR2(("fc_sxm_tclSem:vGet(0x%p): invalid handle", (tVoid *)this));
        return bRes;
    }
    ETG_TRACE_USR2(("fc_sxm_tclSem:vGet(0x%p): START count=%u", (tVoid *)this, _u32Count));
    pthread_t u32Tid=pthread_self();
    if (!pthread_equal(u32Tid, _u32Tid)) {
        // if we are not already owner, we really have to take the sem
        tS32 s32Ret = OSAL_s32SemaphoreWait( _hSem, u32Timeout);
        if (OSAL_OK != s32Ret) {
            ETG_TRACE_ERR(("fc_sxm_tclSem:vGet: failed for Sem %s", _oSemName.c_str()));
        }
        else {
            bRes=TRUE;
            _u32Tid = u32Tid;
        }
    } else {
        ETG_TRACE_USR4(("fc_sxm_tclSem:vGet(0x%p): tid equal", (tVoid *)this));

    }
    _u32Count++;
    ETG_TRACE_USR2(("fc_sxm_tclSem:vGet(0x%p): END count=%u bRes=%d", (tVoid *)this, _u32Count, bRes));
    return bRes;
}

tVoid fc_sxm_tclSem::vPost() {
    if (_hSem == OSAL_C_INVALID_HANDLE) {
        ETG_TRACE_USR2(("fc_sxm_tclSem:vPost(0x%p): invalid handle", (tVoid *)this));
        return;
    }
    ETG_TRACE_USR2(("fc_sxm_tclSem:vPost(0x%p): START count=%u", (tVoid *)this, _u32Count));
    pthread_t u32Tid=pthread_self();
    if (pthread_equal(u32Tid, _u32Tid)) {
        ETG_TRACE_USR4(("fc_sxm_tclSem:vPost(0x%p): tid equal", (tVoid *)this));

        _u32Count--;
        if (_u32Count==0) {
            _u32Tid=0;
            tS32 s32Ret = OSAL_s32SemaphorePost( _hSem );
            if (OSAL_OK != s32Ret) {
                ETG_TRACE_ERR(("fc_sxm_tclSem:vPost: failed for Sem %s", _oSemName.c_str()));
            }     
        }
    }
    else if (_u32Tid==0){
        ETG_TRACE_USR4(("fc_sxm_tclSem:vPost(0x%p): tid==0", (tVoid *)this));

        //handle special case of sem with initial value 0
        tS32 s32Ret = OSAL_s32SemaphorePost( _hSem );
        if (OSAL_OK != s32Ret) {
            ETG_TRACE_ERR(("fc_sxm_tclSem:vPost: failed for Sem %s", _oSemName.c_str()));
        }     
    }
    /*
      assigning 0 as invalid thread-id to pthread_id  is not ok.
      correct way might be to create a dead thread and using its id as invalid id.
    */
    ETG_TRACE_USR2(("fc_sxm_tclSem:vPost(0x%p): END count=%u", (tVoid *)this, _u32Count));
}
    

