/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_data_list.cpp
* @brief       Implementation of data list handling
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

/*
  implementation of base-class for all data-apps
*/
#include "fc_sxm_tcl_data_app.h"
#include "fc_sxm_tcl_data_list.hpp"


// just take any data-fi
#include "fc_sxm_fuel_fi.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_BASE_APP
#include "trcGenProj/Header/fc_sxm_tcl_data_list.cpp.trc.h"
#endif

tVoid fc_sxm_trBaseListConfig::vPrint() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tbUsesLocation=%d", bUsesLocation()));

    vPrintSpecific();
}
fc_sxm_tclDataListBase::fc_sxm_tclDataListBase(tU32 u32ListId, fc_sxm_tenListType enListType, fc_sxm_tclDataApp *poDataApp, tU16 u16TraceClass):
    _u32ListId(u32ListId),
    _enListType(enListType),
    _enListState(fc_sxm_enListState_Initial),
    _poBaseDsrl(OSAL_NULL),
    _enListFreezeState(fc_sxm_enListFreezeState_Idle),
    _enListActiveState(fc_sxm_tenListActivateState_Idle),
    _enListMode(fc_sxm_enListMode_Invalid),
    _u32ListSize(0),
    _hLocation(LOCATION_INVALID_OBJECT),
    _poDataApp(poDataApp),
    _u16TraceClass(u16TraceClass)
{
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:CTOR"));
    fc_sxm_listStatusInfo rListInfo;
    rListInfo.u32Id=_u32ListId;
    rListInfo.enListType=_enListType;
    rListInfo.enListState=_enListState;
    rListInfo.bActivated=FALSE;
    rListInfo.bFrozen=FALSE;
    rListInfo.u32Size = _u32ListSize;
    
    _poDataApp->vAddDataList(this, rListInfo);
}

fc_sxm_tclDataListBase::~fc_sxm_tclDataListBase() {
    // de-register this list at our data-app
    if (OSAL_NULL!=_poDataApp) {
        _poDataApp->vRemoveDataList(this);
    }
    _poBaseDsrl=OSAL_NULL;    

    if (_hLocation != LOCATION_INVALID_OBJECT) {
        LOCATION.vDestroy(_hLocation);
    }
    _hLocation = LOCATION_INVALID_OBJECT;
    _poDataApp = OSAL_NULL;
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:DTOR"));
}

tBool fc_sxm_tclDataListBase::bRequestPending() const {
    tBool bRes=_rPendingRequest.u16AppId!=0;
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:bRequestPending=%d", bRes));
    return bRes;
}

char const *fc_sxm_tclDataListBase::szGetAppName() const {
    if (OSAL_NULL!=_poDataApp) {
        return _poDataApp->szGetName();
    }
    return "";
}

tVoid fc_sxm_tclDataListBase::vPrintReport() const {
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tID=%d:", _u32ListId));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tType=%d", ETG_CENUM(fc_sxm_tenListType, _enListType)));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tState=%d", ETG_CENUM(fc_sxm_tenListState, _enListState)));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tCFG:"));
        poGetBaseCfg()->vPrint();
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tbIsActivated=%d", bIsActivated()));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tbFrozen=%d", bIsFrozen()));
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tSTATISTICS:"));
        vPrintStatistics();
        ETG_TRACE_ERR_CLS((  TR_CLASS_FC_SXM_REPORT, "\t\t\t\tpoList=0x%08x", this));
        fc_sxm_tclBaseDSRL const *poDsrl=poGetDsrl();
        ETG_TRACE_ERR_CLS((  TR_CLASS_FC_SXM_REPORT, "\t\t\t\tpoDsrl=0x%08x", poDsrl));
        SXM_ASSERT_RETURN(OSAL_NULL != poDsrl); 
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tDSRL-ID=%d", poDsrl->u32GetId()));
}



tVoid fc_sxm_tclDataListBase::vUpdateListState(fc_sxm_tenListState enNewState) {
    fc_sxm_tenListState enOldState=_enListState;
    _enListState=enNewState;
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase poList=0x%08x (%10s, %d):vUpdateListState %d-->%d",
                    this,
                    szGetAppName(),
                    ETG_CENUM(fc_sxm_tenListType, _enListType),
                    ETG_CENUM(fc_sxm_tenListState, enOldState),
                    ETG_CENUM(fc_sxm_tenListState, enNewState)
                    ));
    

    if (enNewState==fc_sxm_enListState_Initial || enNewState==fc_sxm_enListState_Error) {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "vUpdateListState:Clear"));
        vClearData();
    }
    else if (!bIsActivated() && enNewState==fc_sxm_enListState_Changed) {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vUpdateListState modify changed to ready"));        
        // state changed is not used towards client
        _enListState=fc_sxm_enListState_Ready;
    }
    _u32ListSize = u32GetListSize();
    
    // update property listsState
    if(OSAL_NULL != _poDataApp)
    {
        _poDataApp->vUpdateListInfo(this);
    }
}


tVoid fc_sxm_tclDataListBase::vOnSmsDsrlState(fc_sxm_tenDsrlState enSmsDsrlState,tBool bDataAvailable) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vOnSmsDsrlState=%d ListFreezeState=%d ListActiveState=%d",
                    ETG_CENUM(fc_sxm_tenDsrlState, enSmsDsrlState),
                    ETG_CENUM(fc_sxm_tenListFreezeState, _enListFreezeState),
                    ETG_CENUM(fc_sxm_tenListActivateState, _enListActiveState)));

    switch (enSmsDsrlState) {
        case fc_sxm_enDsrlState_Changed:
        {
            if (bRequestPending()) {
                vRefresh();
                vEmit();
                vUpdateListState(fc_sxm_enListState_Ready);
            } else {
                vUpdateListState(fc_sxm_enListState_Changed);
            }
        }
		break;
        case fc_sxm_enDsrlState_Ready:
        {
            /* Check if our list contains valid data to be send to HMI */
            if(bDataAvailable == TRUE && bIsListDataValid() == FALSE) {
                vUpdateListState(fc_sxm_enListState_Updating);
                return;
            }

            if (bRequestPending()) {
                vRefresh();
                vEmit();
                vUpdateListState(fc_sxm_enListState_Ready);
            }else if (_enListState != fc_sxm_enListState_Changed) {
                vUpdateListState(fc_sxm_enListState_Ready);
            }
            /* Check if Freeze state is initiated and if we have some entries to send to HMI */
            if(u32GetListSize() > 0 && _enListFreezeState == fc_sxm_enListFreezeState_Enter) {
               /* Change to Freezing as we have not sent list to HMI yet */
               _enListFreezeState = fc_sxm_enListFreezeState_Freezing;
               /* Set Activated flag , Freeze internally activate the list as well */
               _enListActiveState = fc_sxm_tenListActivateState_Activated;
               
               /* inform HMI that , List is Frozen that is ready to be fetched */
               vUpdateListState(fc_sxm_enListState_Frozen);
            }
            /* Check if activate state is initiated*/
            else if(_enListActiveState == fc_sxm_tenListActivateState_Enter) {
               /* Set Activated flag , Freeze internally activate the list as well */
               _enListActiveState = fc_sxm_tenListActivateState_Activated;
               
               /* inform HMI that , List is Frozen that is ready to be fetched */
               vUpdateListState(fc_sxm_enListState_Activated);
            }
        }
		break;
        case fc_sxm_enDsrlState_Updating:
        {
             _enListState = fc_sxm_enListState_Updating;
        }
        break;
        case fc_sxm_enDsrlState_Error:
        {
            vUpdateListState(fc_sxm_enListState_Error);
        }
        break;
        case fc_sxm_enDsrlState_Stopped:
        {
            //Reset List Specific data 
            vClearListData();
            vUpdateListState(fc_sxm_enListState_Initial);
        }
        break;
        case fc_sxm_enDsrlState_Initial:
        {
            vUpdateListState(fc_sxm_enListState_Initial);
        }
        break;
        default:
            break;
    }
}

tVoid fc_sxm_tclDataListBase::vOnListTimerExpire(){
    vUpdateListState(fc_sxm_enListState_Changed);
}

tVoid fc_sxm_tclDataListBase::vSetListMode(fc_sxm_trAdressing const &rAdressing, fc_sxm_tenListMode enListMode){
    ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode(enListMode=%d)",enListMode));
	(tVoid)rAdressing;

#ifndef FC_SXM_DISABLE_DYNAMIC_LIST
    switch(enListMode){
        case fc_sxm_enListMode_Activate:
        {
            /* Set our internal mode */
            _enListMode = fc_sxm_enListMode_Activate;

            /* Check if list is already Activated */
            if (!bIsActivated()) {

               _enListActiveState = fc_sxm_tenListActivateState_Enter;

               /* Check if List is Ready */
               if(_enListState == fc_sxm_enListState_Ready){

                  /* Change our internal list state */
                  _enListActiveState = fc_sxm_tenListActivateState_Activated;
                  
                  /* inform HMI that , List is activated that is ready to be fetched */
                  vUpdateListState(fc_sxm_enListState_Activated);
               }
               else {
                  ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode List is not Ready"));
               }
            }
            else {
               ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode List is already activated,Can't Activate again")); 
            }
        }
        break;
        case fc_sxm_enListMode_Freeze:
        {
            _enListMode = fc_sxm_enListMode_Freeze;
            
            /* Check if list is already Frozen */
            if (!bIsFrozen()) {
               /* Enter into the Freeze state,but list is may not ready to fetch*/
               _enListFreezeState=fc_sxm_enListFreezeState_Enter;           
               
               /* Check conditions for List Non-Empty */
               if((_enListState == fc_sxm_enListState_Ready) 
                  && u32GetListSize() > 0) {
                  /* Change state to Freezing as List is not sent to HMI yet */
                  _enListFreezeState=fc_sxm_enListFreezeState_Freezing;

                  /* Set Activated flag , Freeze internally activate the list as well */
                  _enListActiveState = fc_sxm_tenListActivateState_Activated;
                  
                  /* inform HMI that , List is Frozen that is ready to be fetched */
                  vUpdateListState(fc_sxm_enListState_Frozen);
               }
               else {
                  ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp List is not Ready (or) List size is empty ## size=%d",u32GetListSize()));
               }
            }
            else {
               ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode List is already frozen,Can't Freeze again")); 
            }
        }
        break;
        case fc_sxm_enListMode_Release:
        {
            _enListFreezeState=fc_sxm_enListFreezeState_Idle;
            _enListActiveState = fc_sxm_tenListActivateState_Idle;
            _enListMode = fc_sxm_enListMode_Release;
            vClearData(FALSE);
            vStopTimer();
            if(_enListState == fc_sxm_enListState_Initial || 
               _enListState == fc_sxm_enListState_Updating || 
               _enListState == fc_sxm_enListState_Error) {
                  ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode Ignore Updating to HMI"));
            }
            else {
                /* Finally Update HMI it is ready to be fetched again. */
                vUpdateListState(fc_sxm_enListState_Ready);
            }
        }
        break;
		case fc_sxm_enListMode_Invalid:
        default:
		    /* default mode will be invalid */
            ETG_TRACE_ERR_DCL((_u16TraceClass, "fc_sxm_tclDataApp::vSetListMode error state"));
        break;
    }
#endif
}

tVoid fc_sxm_tclDataListBase::vRequest(fc_sxm_trAdressing const &rAdressing) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vRequest _enListState=%d ",
                        ETG_CENUM(fc_sxm_tenListState, _enListState)
                        ));

    switch(_enListState) {
        case fc_sxm_enListState_Changed:
        case fc_sxm_enListState_Frozen:
        case fc_sxm_enListState_Activated:
        {
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_enListState_Changed"));
            vRefresh();
            _rPendingRequest=rAdressing;
            vEmit();
            vUpdateListState(fc_sxm_enListState_Ready);
        }
		  break;
        case fc_sxm_enListState_Ready:
        {
            ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_enListState_Ready"));
            vRefresh();
            _rPendingRequest=rAdressing;
            vEmit();
        }
        break;

        case fc_sxm_enListState_Error:
        {
            vRefresh();
            vEmit(); // todo: send empty list in this case
        }
        break;
        case fc_sxm_enListState_Updating:
        case fc_sxm_enListState_Initial:
        {
            _rPendingRequest=rAdressing;
                
        }
        break;

        default:
            break;
                
    }
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vRequest END"));

}


tVoid fc_sxm_tclDataListBase::vSetDsrl(fc_sxm_tclBaseDSRL *poBaseDsrl) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vSetDsrl poBaseDsrl=0x%08x to this=0x%08x START", poBaseDsrl, this));
    
    if (_poBaseDsrl!=poBaseDsrl) {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vSetDsrl CHANGED"));
        _poBaseDsrl=poBaseDsrl;
        vClearData();
        if (bIsActivated()) {
            vRefresh();
        }
    }
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vSetDsrl END"));
}

tVoid fc_sxm_tclDataListBase::vSetLocation(fc_sxm_trDsrlLocation const &rLocation) {
    if (_rLocation != rLocation) {
        _rLocation = rLocation;
        vSetSmsLocation(rLocation);
         /* For NearDest & Another Loc type,
           SMS is taking time to get new updates (for more than 10 miles distance change)
           So inform HMI that some updation is going on */
        if(fc_sxm_tenListType_AnotherLocation == _enListType){
            vUpdateListState(fc_sxm_enListState_Updating);
        }
        if(((_enListMode == fc_sxm_enListMode_Freeze && bIsFrozen())|| (_enListMode == fc_sxm_enListMode_Activate && bIsActivated())) &&
            (FALSE == bIsListModeTimerRunning())) {
            vStartTimer();
        }

    }
}
/*
   Notify HMI that List is Updating
   - called when Near Destination Location is changed
*/
tVoid fc_sxm_tclDataListBase::vNotifyUpdatingState() {
   ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vNotifyUpdatingState to HMI"));
   vUpdateListState(fc_sxm_enListState_Updating);
}
/* 
   method to clear list specific data
*/
tVoid fc_sxm_tclDataListBase::vClearListData() {
      
   ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:vClearListData"));

   _enListState = fc_sxm_enListState_Initial;
   _enListFreezeState = fc_sxm_enListFreezeState_Idle;
   _enListActiveState = fc_sxm_tenListActivateState_Idle;
   _enListMode = fc_sxm_enListMode_Invalid;
}

fc_sxm_listStatusInfo fc_sxm_tclDataListBase::rGetStatusInfo() const {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:rGetStatusInfo"));

    fc_sxm_listStatusInfo rRes;
    rRes.u32Id=_u32ListId;
    rRes.u32Size = _u32ListSize;
    rRes.enListType=_enListType;
    rRes.enListState=_enListState;
    rRes.bActivated=bIsActivated();
    rRes.bFrozen=bIsFrozen();
    return rRes;
}

fc_sxm_tclDataListBase *fc_sxm_tclDataListBase::poGetDataListBase(tU32 u32Id, fc_sxm_tclDataApp *poDataApp) {
    return poDataApp->poGetDataListById(u32Id);
}

bool fc_sxm_tclDataListBase::bCompareListEntryId(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight) {
    return prLeft->u32SxmId<prRight->u32SxmId;
}


bool fc_sxm_tclDataListBase::bCompareSmsDistance(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight) const {

   /* Check for Invalid Location Obj */
   SXM_ASSERT_RETURN_VAL(LOCATION_INVALID_OBJECT != _hLocation , false);
   SXM_ASSERT_RETURN_VAL(LOCATION_INVALID_OBJECT != prLeft->hLocation, false);
   SXM_ASSERT_RETURN_VAL(LOCATION_INVALID_OBJECT != prRight->hLocation, false);

   double fDist1=fc_sxm_fGetLocationDistKm(_hLocation,prLeft->hLocation);
   double fDist2=fc_sxm_fGetLocationDistKm(_hLocation,prRight->hLocation);
   if (fDist1<fDist2) {
       return true;
   } else if (fDist2<fDist1) {
       return false;
   }
   else {
	  return bCompareListEntryId(prLeft, prRight);
   }

#if 0
   /* calculate Distance between Target & Left Location*/
   DISTANCE_OBJECT hDist1 = LOCATION.hDistance(_hLocation,prLeft->hLocation);
   /* calculate Distance between Target & Right Location*/
   DISTANCE_OBJECT hDist2 = LOCATION.hDistance(_hLocation,prRight->hLocation);

   /* Check for Invalid Distance Object */
   SXM_ASSERT_RETURN_VAL(DISTANCE_INVALID_OBJECT != hDist1, false);
   SXM_ASSERT_RETURN_VAL(DISTANCE_INVALID_OBJECT != hDist2, false);

   /* Compare both distance*/
   N16 distancebw = DISTANCE.n16Compare(hDist1,hDist2);

   DISTANCE.vDestroy(hDist1);
   DISTANCE.vDestroy(hDist2);

   /* Check if they are not equal by distance*/
   switch( distancebw )
   {
	   case -1:
	     return true;
	   case 1:
	     return false;
      default:
	  return bCompareListEntryId(prLeft, prRight);
   }
#endif

}
bool fc_sxm_tclDataListBase::bCompareFavoriteIndex(fc_sxm_trBaseListEntry const *prLeft, fc_sxm_trBaseListEntry const *prRight) const {

    SXM_ASSERT_RETURN_VAL(_poDataApp != OSAL_NULL , false);

    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclDataListBase:bCompareFavoriteIndex SxmID(L)=%d,SxmID(R)=%d",prLeft->u32SxmId,prRight->u32SxmId));
    return (_poDataApp->u32GetFavoriteIndex(prLeft->u32SxmId)) < (_poDataApp->u32GetFavoriteIndex(prRight->u32SxmId));
}

tVoid fc_sxm_tclDataListBase::vSetSmsLocation(fc_sxm_trDsrlLocation const &rLocation){
    LOCATION_OBJECT hNewLocation = fc_sxm_tclDsrlTypeAdapter::hCreateSMSLocation(rLocation.s32Latitude, rLocation.s32Longitude, DISTANCE_INVALID_OBJECT);
    if (_hLocation != LOCATION_INVALID_OBJECT) {
        LOCATION.vDestroy(_hLocation);
    }
    _hLocation=hNewLocation;
}


bool fc_sxm_listStatusInfo::operator!=(fc_sxm_listStatusInfo const &b) const {
    SXM_RET_1_IF_NEQ(u32Id);
    SXM_RET_1_IF_NEQ(u32Size);
    SXM_RET_1_IF_NEQ(enListType);
    SXM_RET_1_IF_NEQ(enListState);
    SXM_RET_1_IF_NEQ(bActivated);
    SXM_RET_1_IF_NEQ(bFrozen);
    return FALSE;        
}

fc_sxm_listStatusInfo::operator midw_ext_fi_tcl_SxmListInfo() const {
    midw_ext_fi_tcl_SxmListInfo oFiRes;
    oFiRes.ListID=u32Id; // todo: make tU32
    oFiRes.Size=u32Size;
    oFiRes.ListType.enType=(midw_ext_fi_tcl_e8_SxmListType::tenType)enListType;  
    oFiRes.Status.enType=(midw_ext_fi_tcl_e8_SxmListStatus::tenType)enListState;
    return oFiRes;
}




