/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_data_app.h
* @brief       Declaration for Data Application which is used to dispatch messages to
*              specific data apps. It is used as base class in other data apps.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_DATA_APP_H_
#define _FC_SXM_TCL_DATA_APP_H_

#include "fc_sxm_types.h"
#include "fc_sxm_sms.h"
#include "fc_sxm_smse.h"

#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_tcl_base_dsrl.h"
#include "fc_sxm_tcl_ahl_object_list.h"
#include "fc_sxm_tcl_data_list.hpp"
#include "fc_sxm_tcl_data_lists.h"
#include "fc_sxm_main.h"
#include "fc_sxm_tcl_config.h"

class midw_ext_fi_tcl_SxmDataSrvState;
// declare messages that have to be handled by all data-apps
// Messages from CCA-Thread
struct fc_sxm_trMsgCmdDataAppStartService: public fc_sxm_tclMessage {
    fc_sxm_trMsgCmdDataAppStartService(){};
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 1);
};

struct fc_sxm_trMsgCmdDataAppStopService: public fc_sxm_tclMessage {
   fc_sxm_trMsgCmdDataAppStopService(){bRestart = FALSE;}
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 2);
    tBool bRestart;

};


struct fc_sxm_trMsgCmdDataAppCreateDsrl: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 3);

};

// as received from HMI
struct fc_sxm_trMsgDataAppSetDestLocation: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 4);
    fc_sxm_trDsrlLocation rLocation;
};

// as received from SMS
struct fc_sxm_trMsgDataAppSetDeviceLocation: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 5);
    fc_sxm_trDsrlLocation rLocation;
};

// As received from SMS
struct fc_sxm_trMsgCmdDataAppAntennaState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 6);
    ANTENNA_STATE_ENUM rAntennaState;
};

// As received from SMS
struct fc_sxm_trMsgCmdDataAppSignalState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 7);
    SIGNAL_STATE_ENUM eSignalState;
};

// Messages from SMS-Callbacks
struct fc_sxm_trMsgDataSmsEvtServiceState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 10);

    fc_sxm_trMsgDataSmsEvtServiceState():
        eState(DATASERVICE_STATE_INVALID),
        eProductState(DATA_PRODUCT_STATE_DISABLED),
        enErrorCode(DATASERVICE_ERROR_CODE_NONE)
    {}
    DATASERVICE_STATE_ENUM eState;
    DATA_PRODUCT_STATE_ENUM eProductState;
    DATASERVICE_ERROR_CODE_ENUM enErrorCode;

};

// Messages from SMS-Callbacks
struct fc_sxm_trMsgDataPayLoad: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 11);

    fc_sxm_trMsgDataPayLoad():bAllMsgTransmitted(FALSE),
       bIncidentMsg(FALSE),
       bSpeedFlowMsg(FALSE),
       u8Ltn(0),
       u32PayLoadDataSize(0),
       u8PayloadData(NULL)
    {};   
    tBool bAllMsgTransmitted;
    tBool bIncidentMsg;
    tBool bSpeedFlowMsg;
    tU8 u8Ltn;

    tU32 u32PayLoadDataSize;
    tU8 *u8PayloadData;
  
};

struct fc_sxm_trMsgDataSmsEvtServiceUp: public fc_sxm_tclMessage {
    fc_sxm_trMsgDataSmsEvtServiceUp(){};
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 12);
};

struct fc_sxm_trMsgDataSmsEvtDsrlState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 13);
    tU32 u32DsrlId;
    DSRL_STATE_ENUM enState;
};

struct fc_sxm_trMsgDataSportsState: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 14);

    fc_sxm_trMsgDataSportsState():s32SportsState(SXM_CALLBACK_SPORTS_UPDATING)
    {}
    tS32 s32SportsState;
};
// as received from HMI
struct fc_sxm_trMsgDataAppSetClockParams: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 15);
    midw_ext_fi_tclString strTimeZone;
    tBool bDSTStatus;
    midw_ext_fi_tcl_LocalTimeDate oLocalTime;

};
struct fc_sxm_trMsgDataAppSetClockParameters: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 17);
    midw_ext_fi_tclString strTimeZone;
    tBool bDSTStatus;

};
struct fc_sxm_trSDKLocation {
	fc_sxm_trSDKLocation( tS32 s32Latitude_=0, tS32 s32Longitude_=0):
        s32Latitude(s32Latitude_),
        s32Longitude(s32Longitude_)
    {}
   tS32 s32Latitude;
   tS32 s32Longitude;
    bool operator!=(fc_sxm_trSDKLocation const &b) const {
        SXM_RET_1_IF_NEQ(s32Latitude);
        SXM_RET_1_IF_NEQ(s32Longitude);
        return FALSE;
    }
    tBool bIsValid() const {
        return (s32Latitude != 0) && (s32Longitude != 0);
    }
};

// Navi direct values to app for device position
struct fc_sxm_trMsgDataAppNavPosition: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_AllData, 16);
    fc_sxm_trSDKLocation rNavLocation;
};



class fc_sxm_tclDsrlList:public fc_sxm_tclAhlObjectList<fc_sxm_tclBaseDSRL> {
 public:
    tVoid vSetLocation(fc_sxm_trDsrlLocation const &rLocation, fc_sxm_tenDSRLType enDsrlType);
    tVoid vSetDestLocation(fc_sxm_trDsrlLocation const &rLocation) {
        vSetLocation(rLocation, fc_sxm_enDSRLType_Dest);
    }
	tVoid vRestart();
    tVoid vOnSmsServiceStateChanged(tBool bSmsServiceStopping);
    tVoid vPrintReport();
};

// base-class for all data-apps (fuel, weather ...)
class fc_sxm_tclDataApp:public fc_sxm_tclBaseApp {
    //    friend class fc_sxm_tclBaseDSRL;

 public:
    /*
      constructor: just call constructor of base class for all apps fc_sxm_tclBaseApp
    */
    fc_sxm_tclDataApp(tCString szAppName, fc_sxm_tenServiceID enServiceID, tU16 u16CcaServiceId, char const *szSmsConfigDir, tU16 u16TraceClass):
        fc_sxm_tclBaseApp(szAppName, enServiceID, u16CcaServiceId, u16TraceClass),
        _enSmsServiceState(DATASERVICE_STATE_INVALID),
        _enSmsServiceErrorCode(DATASERVICE_ERROR_CODE_NONE),
        _bSmsServiceStopping(FALSE),
        _oSmsConfigDir(szSmsConfigDir),
		_u32ListID(0),
        _bStartRequested(FALSE),
        _u32StartCounter(0),
        _bRestoreSmsDb(FALSE),
        _u32NextDsrlId(0),
        _bDataProductsEnabled(TRUE)
		
    {};

        tU16 u16GetTraceClass() const {
            return _u16TraceClass;
        }
        virtual ~fc_sxm_tclDataApp() {
            _oDsrlList.vClear();
            _vecFavoriteList.clear();
        }
        /*
          overwriting bWantsMsg from base-class
          determines from u32Action of a received message, if the message is for us
        */
   virtual tBool bWantsMsg(tU32 u32Action) const {
       fc_sxm_tenServiceID enServiceId=fc_sxm_enGetServiceId(u32Action);
       if (enServiceId==enGetServiceId() ||  // message specific for our app
           enServiceId==fc_sxm_enServiceID_All ||  // message has to be handled by all Apps
           enServiceId==fc_sxm_enServiceID_AllData) // messsage is for all data-apps
       {
           return TRUE;
       }
       return FALSE; // message is not for us
   }

   tU32 u32GetNextDsrlId() {
       // todo: check if lock needed
       return ++_u32NextDsrlId;
   }

   fc_sxm_tclDataListBase* poGetActiveListbyDSRL(fc_sxm_tclBaseDSRL *poBaseDsrl);

   tVoid vOnDsrlState(fc_sxm_tclBaseDSRL *poBaseDsrl, fc_sxm_tenDsrlState enDsrlState,tBool bDataAvailable);

   tVoid vUpdateListInfo(fc_sxm_tclDataListBase *poDataList);

   fc_sxm_trDsrlLocation rGetInitialLocation(fc_sxm_tenListType enListType) {
       switch (enListType) {
           case fc_sxm_tenListType_NearBy:
               return _rDeviceLocation;
           case fc_sxm_tenListType_NearDest:
               return _rDestLocation;
           default:
               return fc_sxm_trDsrlLocation();
       }
   }

 public: 
   virtual tVoid vPrintReport();   
   virtual tVoid vPrintReportSpecific(){};

   tU32 u32GetFavoriteIndex(tU32 u32LOCID) const;

 protected:
   /* 
      Method called by each Application to get the DSRL configuration for particular DSRL type
   */
   tVoid vPatchDsrlConfig(fc_sxm_tenDSRLType enDSRLType, fc_sxm_trDSRLBaseCfg &rCfg)
   {
       fc_sxm_tclConfigDsrls::instance()->vPatchDsrlConfig(enGetServiceId(), enDSRLType, rCfg);
   }
   /* central method that receives all thread-message. Must not be overwritten!
      Generic messages ( "fc_sxm_enServiceID_All" and "fc_sxm_enServiceID_AllData")
      will be dispatched directly, messages specific for this app are forwarded
      to virtual vDispatchMsgFromQ() that has to be defined by the dericed class.
   */
   virtual tVoid vDispatchMsgFromQInternal(fc_sxm_tclMessage const *poThreadMsg);

   fc_sxm_tclBaseDSRL *poGetDsrl(tU32 u32DsrlId);
       // put pointer to _oDsrlList
   fc_sxm_tclBaseDSRL *poRegisterDsrl(fc_sxm_tclBaseDSRL *poDsrl);
       // remove pointer from _oDsrlList
   tVoid vUnregisterDsrl(fc_sxm_tclBaseDSRL *poDsrl);

   tVoid vUpdateDsrls(tBool bSmsServiceStopping) {
       _oDsrlList.vOnSmsServiceStateChanged(bSmsServiceStopping);
   }

   //Favorite Specific methods
   tVoid vLoadFavorites();
   tVoid vAddFavorite(tU32 u32LOCID);
   tVoid vRemoveFavorite(tU32 u32LOCID);
   tVoid vSaveFavorites();
   tVoid vClearFavorites();

 public:
   tVoid vAddDataList(fc_sxm_tclDataListBase *poDataList, fc_sxm_listStatusInfo const &rListInfo);
   tVoid vRemoveDataList(fc_sxm_tclDataListBase *poDataList);

   virtual tVoid vOnListStateChanged(fc_sxm_tclDataListBase *poDataList);
   virtual tVoid vOnFiServiceStateChanged(midw_ext_fi_tcl_SxmDataSrvState const & /* oFiSrvState */) { }
   fc_sxm_tclDataLists const &oGetDataListsInfo() {
       return _oDataLists;
   }
   fc_sxm_tclDataListBase *poGetDataListById(tU32 u32Id) {
       return _oDataLists.poGetById(u32Id);
   }

   fc_sxm_trDsrlLocation const &rGetDestLocation() {
       return _rDestLocation;
   }

   fc_sxm_trDsrlLocation const &rGetDeviceLocation() {
       return _rDeviceLocation;
   }

   fc_sxm_trSDKLocation const &rGetSDKDeviceLocation() {
       return _rNavLocation;
   }
   
   tVoid vSendList(tU32 u32ListId, fc_sxm_trAdressing const &rAdressing);
   tVoid vSetListMode(tU32 u32ListId, fc_sxm_trAdressing const &rAdressing, fc_sxm_tenListMode enListMode);


   // request our stored service-state
   DATASERVICE_STATE_ENUM enGetSmsServiceState() const;
   virtual DATA_PRODUCT_STATE_ENUM enGetDataProductState(DATASERVICE_MGR_OBJECT hDataService) const;

   tBool bIsStartRequested() {
       return _bStartRequested;
   }

   /* this method is thread-safe and can be called from dsrl-callback */
   tVoid vOnSmsDsrlState(tU32 u32DsrlId, DSRL_STATE_ENUM eState) {
    fc_sxm_trMsgDataSmsEvtDsrlState rMsg;
    rMsg.enState=eState;
    rMsg.u32DsrlId=u32DsrlId;

#ifdef FC_SXM_ITERATE_DSRL_BY_DSM
    // todo: increase prio
    if (bAccess()) {
        /*
          we could directly gain access, so we can handle the message in the context of the DSM
        */
        vProcess(&rMsg);
        vRelease();
        // todo: decrease prio
    }
    else {
        // we have to handle it in the context of the data-app.
        vPostMsgNew(rMsg);
    }
#else
        vPostMsgNew(rMsg);
#endif
   }

 private:
      DATASERVICE_STATE_ENUM _enSmsServiceState;
protected:
      // ask the sms-object about the service-state
      //virtual DATASERVICE_STATE_ENUM enRequestSmsServiceState() const = 0;
      virtual DATASERVICE_STATE_ENUM enRequestSmsServiceState(DATASERVICE_MGR_OBJECT hDataService) const = 0;
      DATASERVICE_ERROR_CODE_ENUM _enSmsServiceErrorCode;
      tBool _bSmsServiceStopping;
      string _oSmsConfigDir;
	  tU32 _u32ListID;

 private:
      tBool _bStartRequested;
      tU32 _u32StartCounter;
      tBool _bRestoreSmsDb;
   tU32 _u32NextDsrlId;
   fc_sxm_tclDsrlList _oDsrlList;
   fc_sxm_trDsrlLocation _rDestLocation;
   fc_sxm_trDsrlLocation _rDeviceLocation;
   fc_sxm_trSDKLocation _rNavLocation;
   fc_sxm_tclDataLists _oDataLists;
   vector<tU32> _vecFavoriteList;
protected:
   tBool _bDataProductsEnabled;

   /*
     For each messages that has to be handled by each data-app a pure virtual method "vProcess"
     is declared that has to be implemented by the derived class
   */
   // handle generic commands From CCA-Thread:
   //Start the service
   // todo: all data-services shall rename vProcess(fc_sxm_trMsgCmdDataAppStartService) to vStartSmsService
 protected:

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgCmdDataAppStartService const * /* prMsg */) {
       _bStartRequested=TRUE;
       _bSmsServiceStopping=FALSE;
       vStartSmsService();
       vRestoreFavorites();
       //vAllowService(TRUE);
   }


   //Stop the service
   // todo: all data-services shall rename vProcess(fc_sxm_trMsgCmdDataAppStopService) to vStopSmsService

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgCmdDataAppStopService const *prMsg) {
       if (_bStartRequested) {
           _bStartRequested= prMsg->bRestart;
           vStopSmsService();
       }

   };
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataAppSetClockParams const *prMsg) {
	   vSetClockParams(prMsg);
   }
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataAppSetClockParameters const *prMsg) {
       vSetClockParameters(prMsg);
   }

   // restore sms-data to default
   virtual tVoid vProcessBaseAppMsg(fc_sxm_trMsgAppRestoreSmsDb const *prMsg);

 private:
   //Create the DSRL, deprecated, please remove any usage

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgCmdDataAppCreateDsrl const * /* prMsg */) { }

   //prepares message based on Service & Subscription status
   static tBool bGetSDKServiceStatus(fc_sxm_trMsgDataSmsEvtServiceState &rMsg,tS32 s32ServiceStatus,tS32 s32SubscritionStatus);


 protected:
   /* 
      Method to be Overriden by each Service 
      to clear Service specific internal data
   */
   virtual tVoid vClearServiceData();

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtDsrlState const *prMsg);
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataAppSetDestLocation const *prMsg);

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataAppSetDeviceLocation const *prMsg);
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataAppNavPosition const *prMsg);


   
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgCmdDataAppAntennaState const *prMsg);

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgCmdDataAppSignalState const *prMsg);


   static tVoid cb_vDataServiceEventCallback(DATASERVICE_MGR_OBJECT hDataService,
                                             DATASERVICE_EVENT_MASK tEventMask,
                                             tVoid *pvEventCallback);

   static tVoid cb_vDataGenericServiceEventCallback(
                                                    DATASERVICE_MGR_OBJECT hDataService,
                                                    DATASERVICE_EVENT_MASK tCurrentEvent,
                                                    tVoid *pvEventArg,
                                                    tVoid *pvEventCallbackArg);

   static tVoid cb_vDataSportsServiceCallback(tS32 s32SportsStae, tS32 s32SubscritionStatus);

   //callback function to receive parking module-level event indications
   static tVoid cb_vDataParkingServiceCallback(tS32 s32ParkingState, tS32 s32SubscritionStatus);

   virtual tBool bParseServiceSpecificPayloadHeader( OSAL_BUFFER_HDL /* pvPayload */,
                                fc_sxm_trMsgDataPayLoad& /* rPayloadMsg */) { return FALSE; }

 public:

   // get our stored error-code
   virtual DATASERVICE_ERROR_CODE_ENUM enGetSmsServiceErrorCode() const {
       return _enSmsServiceErrorCode;
   }

   // todo: make pure virtual
   virtual const DATASERVICE_MGR_OBJECT hGetSmsService() const =0;
   
   virtual const DATASERVICE_MGR_OBJECT &hGetSmsServiceRef() =0;

   virtual tVoid vSetSmsServiceRefInvalid() =0;


 protected:
   // ask the sms-object about its error-code
   virtual DATASERVICE_ERROR_CODE_ENUM enRequestSmsServiceErrorCode(DATASERVICE_MGR_OBJECT hDataService) const =0;
   //todo: vStartSmsService() has to be implemented by each data-app by renaming vProcess(fc_sxm_trMsgCmdDataAppStartService)
   virtual tVoid vStartSmsService() = 0;
   virtual tVoid vStopSmsService() = 0;
   virtual tVoid vDisableDataProducts() { }
   tVoid vTraceStopSmsService() const;
   virtual tVoid vOnNewDevicePosition(fc_sxm_trSDKLocation&) {/* to be implemnted by classes who is interested in actual Navi values*/}
   virtual tVoid vRestoreFavorites(){ /*  to be implemented by each Service to Restore favorites from Persistent if any */}
   virtual tVoid vPersistFavorites(){ /*  to be implemented by each Service to Save favorites to Persistent*/}
   //Process service state

   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceState const *prMsg);
   virtual tVoid vProcessGenericMsg(fc_sxm_trMsgDataSmsEvtServiceUp const * /* prMsg */) { }

   //Enable Data Products
   virtual tBool bEnableDataProducts() const { return TRUE; }
   virtual tBool bResetProductState() { return TRUE; }

   virtual tVoid vSetClockParams(fc_sxm_trMsgDataAppSetClockParams const * /*prMsg*/) {
   /* to be implemented by apps who want to use clock params*/
   }
   virtual tVoid vSetClockParameters(fc_sxm_trMsgDataAppSetClockParameters const */*prMsg*/) {
   /* to be implemented by apps who want to use clock params*/
   }

   tVoid vOnDataBaseCorrupted() const;
};

tU32 fc_sxm_u32GetNewDsrlId(fc_sxm_tclDataApp *prDataApp);

#define SXM_MSGQ_DISPATCH_TO_DSRL(TYPE) \
    case TYPE::u32Action:       \
    {                                                       \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const *>(poThreadMsg);   \
        if (OSAL_NULL != poTypedMsg) {                         \
            fc_sxm_tclBaseDSRL *poDsrl=poGetDsrl(poTypedMsg->u32DsrlId); \
            if (OSAL_NULL != poDsrl) {                                  \
                /* OK,  dsrl is still valid */                          \
                poDsrl->vProcess(poTypedMsg);                           \
            }                                                           \
        }                                                               \
    }                                                       \
    break


// macro to define access-methods for app-specific sms-service object
#define SXM_SMS_DATASRV_ACCESS(SMS_OBJ_NAME, MEMBER_NAME)               \
                                                                        \
    public:                                                             \
    virtual DATASERVICE_ERROR_CODE_ENUM enRequestSmsServiceErrorCode(DATASERVICE_MGR_OBJECT hDataService) const { \
        return SMS_OBJ_NAME.eErrorCode(hDataService);                    \
    }                                                                   \
                                                                        \
    /* for internal use */                                              \
    virtual const DATASERVICE_MGR_OBJECT &hGetSmsServiceRef() {               \
        return MEMBER_NAME;                                             \
    }                                                                   \
    /* for internal use */                                              \
    virtual tVoid vSetSmsServiceRefInvalid() {                          \
    	MEMBER_NAME = DATASERVICE_MGR_INVALID_OBJECT;                   \
    }                                                                   \
                                                                        \
    /* for external use */                                              \
    virtual const DATASERVICE_MGR_OBJECT hGetSmsService() const {             \
        if (enGetSmsServiceState() == DATASERVICE_STATE_READY) {        \
            return MEMBER_NAME;                                         \
        } else {                                                        \
            return DATASERVICE_MGR_INVALID_OBJECT;                      \
        }                                                               \
    }                                                                   \
                                                                        \
    protected:                                                          \
    /* for internal use */                                              \
    virtual DATASERVICE_STATE_ENUM enRequestSmsServiceState(DATASERVICE_MGR_OBJECT hDataService) const {   \
		if (hDataService != DATASERVICE_MGR_INVALID_OBJECT) {            \
            return SMS_OBJ_NAME.eState(hDataService);                    \
		} else {                                                        \
			return DATASERVICE_STATE_INVALID;							\
        }                                                               \
    }                                                                   \
                                                                        \
                                                                        \
    virtual tVoid vStopSmsService() {                                   \
        vTraceStopSmsService();                                         \
        _bSmsServiceStopping=TRUE;                                      \
        vUpdateDsrls(_bSmsServiceStopping);                             \
        vClearServiceData();											\
        vAllowService(FALSE);											\
        vPersistFavorites();											\
        SMS_OBJ_NAME.vStop(MEMBER_NAME);                                \
    }                                                                   \

// macro to define access-methods for app-specific sms-service object
#define SXM_SMSE_DATASRV_ACCESS(SMS_OBJ_NAME, MEMBER_NAME)               \
                                                                        \
    public:                                                             \
    virtual DATASERVICE_ERROR_CODE_ENUM enRequestSmsServiceErrorCode(DATASERVICE_MGR_OBJECT hDataService) const { \
    	(tVoid)hDataService;                                          \
        return DATASERVICE_ERROR_CODE_NONE;                    \
    }                                                                   \
                                                                        \
    /* for internal use */                                              \
    virtual const DATASERVICE_MGR_OBJECT &hGetSmsServiceRef() {               \
        return MEMBER_NAME;                                             \
    }                                                                   \
    /* for internal use */                                              \
    virtual tVoid vSetSmsServiceRefInvalid() {                          \
    	MEMBER_NAME = DATASERVICE_MGR_INVALID_OBJECT;                   \
    }                                                                   \
                                                                        \
    /* for external use */                                              \
    virtual const DATASERVICE_MGR_OBJECT hGetSmsService() const {             \
        if (enGetSmsServiceState() == DATASERVICE_STATE_READY) {        \
            return MEMBER_NAME;                                         \
        } else {                                                        \
            return DATASERVICE_MGR_INVALID_OBJECT;                      \
        }                                                               \
    }                                                                   \
                                                                        \
    protected:                                                          \
    /* for internal use */                                              \
    virtual DATASERVICE_STATE_ENUM enRequestSmsServiceState(DATASERVICE_MGR_OBJECT hDataService) const {   \
		if (hDataService != DATASERVICE_MGR_INVALID_OBJECT) {            \
            return DATASERVICE_STATE_INITIAL;                    \
		} else {                                                        \
			return DATASERVICE_STATE_INVALID;							\
        }                                                               \
    }                                                                   \


#endif //_FC_SXM_TCL_DATA_APP_H_
