/**
 * @swcomponent fc_sxm
 * @{
 * @file        fc_sxm_tcl_config.cpp
 * @brief       Implementation for global function to convert service name to service id,
 *              dsrl name to dsrl type. Also providung implementation for preset and dsrls
 *              configurations.
 * @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 * @}
 */

#include "fc_sxm_common.h"
#include "fc_sxm_tcl_config.h"
#include "fc_sxm_sms_util.h"
#include "fc_sxm_tcl_pers_data.h"
#include "fc_sxm_tcl_diag_defset.h"
#include <ctype.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_CONFIG
#include "trcGenProj/Header/fc_sxm_tcl_config.cpp.trc.h"
#endif

#ifdef __cplusplus
extern "C"
{
#endif

//extern int FcSxm_bDoSerialRxLogToFile;//manoj
#ifdef __cplusplus
}
#endif

#ifndef VARIANT_S_FTR_ENABLE_DP_CLIENT_MOCK
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"
#else
#include"diaglib_if.h"
#endif

static fc_sxm_tenServiceID fc_sxm_enString2SxmServiceId(string oName)
{
   fc_sxm_vToLower(oName);
   if(oName == "audio")
      return fc_sxm_enServiceID_Audio;
   if(oName == "channelart")
      return fc_sxm_enServiceID_Channelart;
   if(oName == "traffic")
      return fc_sxm_enServiceID_Traffic;
   if(oName == "movies")
      return fc_sxm_enServiceID_Movies;
   if(oName == "agw")
      return fc_sxm_enServiceID_AGW;
   if(oName == "tabweather")
      return fc_sxm_enServiceID_Tabweather;
   if(oName == "fuel")
      return fc_sxm_enServiceID_Fuel;
   if(oName == "stocks")
      return fc_sxm_enServiceID_Stocks;
   if(oName == "safeview")
      return fc_sxm_enServiceID_Safeview;
   if(oName == "sports")
      return fc_sxm_enServiceID_Sports;
   if(oName == "canadianfuel")
      return fc_sxm_enServiceID_Canadian_Fuel;
   if(oName == "wsalerts")
      return fc_sxm_enServiceID_WsAlerts;
   if(oName == "parking")
      return fc_sxm_enServiceID_Parking;
   if(oName == "phonetics")
      return fc_sxm_enServiceID_Phonetics;
   return fc_sxm_enServiceID_All;
}

static fc_sxm_tenDSRLType fc_sxm_enString2DsrlType(string oName)
{
   fc_sxm_vToLower(oName);
   if(oName == "standard")
      return fc_sxm_enDSRLType_Standard;
   if(oName == "destination")
      return fc_sxm_enDSRLType_Dest;
   if(oName == "device")
      return fc_sxm_enDSRLType_Device;
   if(oName == "favorites")
      return fc_sxm_enDSRLType_Favorites;
   return fc_sxm_enDSRLType_MAX;
}

fc_sxm_tclConfig::fc_sxm_tclConfig()
         : _u32DelaySmsStartMs(FC_SXM_DELAY_STARTUP_SMS_MS),
           _u32DelayDataStartMs(FC_SXM_DELAY_STARTUP_DATA_MS),
           _u32DelayAudioStartMs(FC_SXM_DELAY_STARTUP_AUDIO_MS),
           _u32DataServiceMaxStartTimeMs(FC_SXM_DATA_SERVICE_MAX_START_TIME),
           _u32DelayDevicePosTimeMs(FC_SXM_DEVICE_POSITION_UPDATE_INTERVAL),
           _u32WatchdogMaxCycleTimeMs(FC_SXM_WATCHDOG_TIMER_MAX_CYCLES),
           _bIsMixedPreset(false),
           _u8NumberOfPresets(0)
{
   vInit(FC_SXM_SMS_FFS_BASE_PATH"/debug.cfg");
   
   	  /* Read KDS parameter */
   tS32 s32ReturnCode = 0xFF;
   s32ReturnCode = DP_s32GetConfigItem("GenericTunerParameter", "PresetListType", &_bIsMixedPreset, 1, false);
   if(s32ReturnCode != DP_S32_NO_ERR)
   {
       _bIsMixedPreset = false;
   }

   s32ReturnCode = DP_s32GetConfigItem("GenericTunerParameter", "NumberOfPresetsPerBankForMixedAllBands", &_u8NumberOfPresets, 1, false);
   if(s32ReturnCode != DP_S32_NO_ERR)
   {
       _u8NumberOfPresets = 0;
   }
   ETG_TRACE_USR4(("fc_sxm_tclConfig u8PresetType: %d _u8NumberOfPresets: %d", _bIsMixedPreset, _u8NumberOfPresets));
}

tVoid fc_sxm_tclConfig::vReadEntry(string const &oKey, vector<string> const &oVals)
{
   ETG_TRACE_USR2(("fc_sxm_tclConfig::vReadEntry(%s)", oKey.c_str()));

   string const &oFirstVal = oVals.front();
   tU32 u32Val = 0;
   if(oKey == "disableApps")
   {
      ETG_TRACE_USR2(("disableApps"));
      SXM_FOREACH_CONST(vector<string>, iter, oVals)
      {
         if(bGetHex(*iter, &u32Val))
         {
            ETG_TRACE_COMP(("disableApps: u16Val=0x%x", u32Val));
            _setDisabledApps.insert((fc_sxm_tenServiceID) u32Val);
         }
      }

   }
   else if(oKey == "u32DelaySmsStartMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32DelaySmsStartMs = u32Val;
         ETG_TRACE_COMP(("u32DelaySmsStartMs: _u32DelaySmsStartMs=%u", _u32DelaySmsStartMs));
      }
   }
   else if(oKey == "u32DelayDataStartMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32DelayDataStartMs = u32Val;
         ETG_TRACE_COMP(("u32DelayDataStartMs: _u32DelayDataStartMs=%u", _u32DelayDataStartMs));
      }
   }

   else if(oKey == "u32DataServiceMaxStartTimeMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32DataServiceMaxStartTimeMs = u32Val;
         ETG_TRACE_COMP(("u32DataServiceMaxStartTimeMs: _u32DataServiceMaxStartTimeMs=%u", _u32DataServiceMaxStartTimeMs));
      }
   }

   else if(oKey == "u32DelayDevicePosTimeMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32DelayDevicePosTimeMs = u32Val;
         ETG_TRACE_COMP(("u32DelayDevicePosTimeMs: _u32DelayDevicePosTimeMs=%u", _u32DelayDevicePosTimeMs));
      }
   }
   else if(oKey == "u32WatchdogMaxCyclesTimeMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32WatchdogMaxCycleTimeMs = u32Val;
         ETG_TRACE_COMP(("u32WatchdogMaxCyclesTimeMs: _u32WatchdogMaxCycleTimeMs=%u", _u32WatchdogMaxCycleTimeMs));
      }
   }

   else if(oKey == "u32DelayAudioStartMs")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         _u32DelayAudioStartMs = u32Val;
         ETG_TRACE_COMP(("u32DelayAudioStartMs: _u32DelayAudioStartMs=%u", _u32DelayAudioStartMs));
      }
   }

   else if(oKey == "bLogSerialRxToFile")
   {
      if(bGetDec(oFirstVal, &u32Val))
      {
         //FcSxm_bDoSerialRxLogToFile=u32Val;
         //ETG_TRACE_COMP(("bLogSerialRxToFile=%u", FcSxm_bDoSerialRxLogToFile));
      }
   }
   else
   {
      ETG_TRACE_COMP(("Add Key-Val pair:%20s->%s", oKey.c_str(), oFirstVal.c_str()));
      vStoreToMap(oKey, oVals);
   }
}


//DbVersion
fc_sxm_tclDbVersionConfig::fc_sxm_tclDbVersionConfig()
: _u32SmsCfgDbVersion(0xFFFF),
  _u32FuelDbVersion(0xFFFF),
  _u32TrafficDbVersion(0xFFFF),
  _u32MoviesDbVersion(0xFFFF),
  _u32AgwDbVersion(0xFFFF),
  _u32TabWeatherDbVersion(0xFFFF),
  _u32StocksDbVersion(0xFFFF),
  _u32SportsDbVersion(0xFFFF),
  _u32CanadianFuelDbVersion(0xFFFF),
  _u32WsAlertsDbVersion(0xFFFF),
  _u32ParkingDbVersion(0xFFFF),
  _u32PhoneticsDbVersion(0xFFFF),
  _u32ChannelArtDbVersion(0xFFFF)
{
	ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::Constructor start"));

    vInit(FC_SXM_STATIC_SIRIUSXM_PATH"/DbVersion.cfg");
    _oSem.vOpen();

}

fc_sxm_tclDbVersionConfig::~fc_sxm_tclDbVersionConfig() {
    _oSem.vClose();
}

tVoid fc_sxm_tclDbVersionConfig::vReadEntry(string const &oKey, vector<string> const &oVals)
{
    ETG_TRACE_USR2(("fc_sxm_tclDbVersionConfig::vReadEntry(%s)", oKey.c_str()));

    string const &oFirstVal = oVals.front();
    tU32 u32Val = 0;
    if(oKey == "SmsCfg")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32SmsCfgDbVersion = u32Val;
            ETG_TRACE_COMP(("SmsCfg: _u32SmsCfgDbVersion=%u", _u32SmsCfgDbVersion));
        }
    }
    else if(oKey == "Fuel")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32FuelDbVersion = u32Val;
            ETG_TRACE_COMP(("Fuel: _u32FuelDbVersion=%u", _u32FuelDbVersion));
        }
    }
    else if(oKey == "Traffic")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32TrafficDbVersion = u32Val;
            ETG_TRACE_COMP(("Traffic: _u32TrafficDbVersion=%u", _u32TrafficDbVersion));
        }
    }

    else if(oKey == "Movies")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32MoviesDbVersion = u32Val;
            ETG_TRACE_COMP(("Movies: _u32MoviesDbVersion=%u", _u32MoviesDbVersion));
        }
    }

    else if(oKey == "Agw")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32AgwDbVersion = u32Val;
            ETG_TRACE_COMP(("Agw: _u32AgwDbVersion=%u", _u32AgwDbVersion));
        }
    }
    else if(oKey == "TabWeather")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32TabWeatherDbVersion = u32Val;
            ETG_TRACE_COMP(("TabWeather: _u32TabWeatherDbVersion=%u", _u32TabWeatherDbVersion));
        }
    }

    else if(oKey == "Stocks")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32StocksDbVersion = u32Val;
            ETG_TRACE_COMP(("Stocks: _u32StocksDbVersion=%u", _u32StocksDbVersion));
        }
    }

    else if(oKey == "Sports")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32SportsDbVersion = u32Val;
            ETG_TRACE_COMP(("Sports: _u32SportsDbVersion=%u", _u32SportsDbVersion));
        }
    }

    else if(oKey == "CanadianFuel")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32CanadianFuelDbVersion = u32Val;
            ETG_TRACE_COMP(("CanadianFuel: _u32CanadianFuelDbVersion=%u", _u32CanadianFuelDbVersion));
        }
    }

    else if(oKey == "WsAlerts")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32WsAlertsDbVersion = u32Val;
            ETG_TRACE_COMP(("WsAlerts: _u32WsAlertsDbVersion=%u", _u32WsAlertsDbVersion));
        }
    }

    else if(oKey == "Parking")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32ParkingDbVersion = u32Val;
            ETG_TRACE_COMP(("Parking: _u32ParkingDbVersion=%u", _u32ParkingDbVersion));
        }
    }
    else if(oKey == "Phonetics")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
            _u32PhoneticsDbVersion = u32Val;
            ETG_TRACE_COMP(("Phonetics: _u32PhoneticsDbVersion=%u", _u32PhoneticsDbVersion));
        }
    }
    else if(oKey == "States")
    {
        if(bGetDec(oFirstVal, &u32Val))
        {
        	_u32StatesVersion = u32Val;
            ETG_TRACE_COMP(("Phonetics: _u32StatesVersion=%u", _u32StatesVersion));
        }
    }
    else if(oKey == "ChannelArt")
       {
           if(bGetDec(oFirstVal, &u32Val))
           {
        	   _u32ChannelArtDbVersion = u32Val;
               ETG_TRACE_COMP(("ChannelArt: _u32ChannelArtDbVersion=%u", _u32ChannelArtDbVersion));
           }
       }
    else
    {
        ETG_TRACE_COMP(("Add Key-Val pair:%20s->%s", oKey.c_str(), oFirstVal.c_str()));
        vStoreToMap(oKey, oVals);
    }
}


tBool fc_sxm_tclConfig::bIsAppEnabled(fc_sxm_tenServiceID enServiceId) const
{
   set<fc_sxm_tenServiceID>::const_iterator iterName = _setDisabledApps.find(enServiceId);
   if(iterName != _setDisabledApps.end())
   {
      return FALSE;
   }
   return TRUE;
}

tVoid fc_sxm_tclConfig::vCheckKDSPresetType()
{
    if(bGetPresetType() != fc_sxm_tclPersData::instance()->bGetDPPresetType())
    {
        string oDstDir = FC_SXM_DEFAULT_SMS_CFG_PATH;
        string oSrcDir = FC_SXM_STATIC_SMS_CFG_PATH;
        string oSubDir = "sms.cfg";

        if(bGetPresetType() == 1)
        {
            oSubDir = "sms_mixed_preset.cfg";
        }

        ETG_TRACE_USR4(("fc_sxm_tclApp::vCheckKDSPresetType() Sms.cfg is updated based on KDS preset type: %d!", bGetPresetType()));

       fc_sxm_tclSmsDirRestorer::instance()->vDoRestoreFactoryMixed(oSubDir,oSrcDir,oDstDir);


       fc_sxm_tclPersData::instance()->vSetDPPresetType(bGetPresetType());
    }
}

// Presets Defaults
fc_sxm_tclConfigPresets::fc_sxm_tclConfigPresets()
{
   vInit(FC_SXM_SMS_FFS_BASE_PATH"/presets.cfg");
}

tVoid fc_sxm_tclConfigPresets::vReadEntry(string const &oKey, vector<string> const &oVals)
{
   ETG_TRACE_USR2(("fc_sxm_tclConfigPresets::vReadEntry(numVal=%d key=%s)", oVals.size(), oKey.c_str()));
   if(oKey == "preset")
   {
      if(oVals.size() >= 3)
      {
         fc_sxm_trDefaultPreset rPreset;
         tBool bRes = TRUE;
         bRes = bRes & (tBool)(bGetDec(oVals[0], &rPreset.u32Bank));
         bRes = bRes & (tBool)(bGetDec(oVals[1], &rPreset.u32Preset));
         bRes = bRes & (tBool)(bGetDec(oVals[2], &rPreset.u32Channel));
         if(bRes)
         {
            _vecPresets.push_back(rPreset);
         }
      }
   }
   else
   {
      vStoreToMap(oKey, oVals);
   }

}
////////////////////////////////////////////////////////////////////////////////////
/***************              DSRL Configuration                *******************/
////////////////////////////////////////////////////////////////////////////////////
fc_sxm_tclConfigDsrls::fc_sxm_tclConfigDsrls()
{
   vInit(FC_SXM_SMS_FFS_BASE_PATH"/dsrl.cfg");
}
/* 
 Method overriden from fc_sxm_tclConfigFile class
 */
tVoid fc_sxm_tclConfigDsrls::vReadEntry(const string &oKey, const vector<string> &oVals)
{
   ETG_TRACE_USR4(("fc_sxm_tclConfigDsrls::vReadEntry key = %s ", oKey.c_str()));
   ETG_TRACE_USR4
   (("fc_sxm_tclConfigDsrls::vReadEntry numVal = %d ", oVals.size()));

   if(oVals.size() != 3)
   {
      return;
   }
   fc_sxm_tenServiceID enServiceId = (fc_sxm_tenServiceID) fc_sxm_enString2SxmServiceId(oKey);
   fc_sxm_tenDSRLType enDsrlType = (fc_sxm_tenDSRLType) fc_sxm_enString2DsrlType(oVals.front());
   tU32 u32Key = u32GenKey(enServiceId, enDsrlType);
   fc_sxm_trDSRLBaseCfg rDsrlCfg;
   tBool bRes = TRUE;
   bRes = bRes & (tBool)(bGetDec(oVals[1], &rDsrlCfg.u32Capacity));
   bRes = bRes & (tBool)(bGetDec(oVals[2], &rDsrlCfg.u32Radius));
   ETG_TRACE_USR4
   (("fc_sxm_tclConfigDsrls::enServiceId=%x enDsrlType=%d u32Capacity=%d u32Radius=%d", ETG_CENUM(fc_sxm_tenServiceID,
      enServiceId), ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType), rDsrlCfg.u32Capacity, rDsrlCfg.u32Radius));

   if(enServiceId == fc_sxm_enServiceID_All || enDsrlType == fc_sxm_enDSRLType_MAX || bRes == FALSE)
   {
      ETG_TRACE_USR4(("fc_sxm_tclConfigDsrls::INVALID input"));
      return;
   }
   _mapDsrlConfigs[u32Key] = rDsrlCfg;

}
/* 
 Called by App to Get DSRL Configuration
 */
tVoid fc_sxm_tclConfigDsrls::vPatchDsrlConfig(fc_sxm_tenServiceID enServiceID,
   fc_sxm_tenDSRLType enDSRLType,
   fc_sxm_trDSRLBaseCfg &rCfg)
{
   ETG_TRACE_USR2(("fc_sxm_tclConfigDsrls::vPatchDSRLConfig ServiceID=0x%08x enDSRLType=%d u32Capacity=%d u32Radius=%d", ETG_CENUM(fc_sxm_tenServiceID,
      enServiceID), ETG_CENUM(fc_sxm_tenDSRLType, enDSRLType), rCfg.u32Capacity, rCfg.u32Radius));
   // check if we have a configuration for the requested dsrl
   fc_sxm_tclMapDsrlConfigs::const_iterator iterCfg = _mapDsrlConfigs.find(u32GenKey(enServiceID, enDSRLType));
   if(iterCfg == _mapDsrlConfigs.end())
   {
      ETG_TRACE_USR3(("No patch found"));
      return;
   }
   fc_sxm_trDSRLBaseCfg const &rStoredCfg = iterCfg->second;
   ETG_TRACE_USR2(("fc_sxm_tclConfigDsrls::vPatchDSRLConfig NEW u32Capacity=%d u32Radius=%d", rStoredCfg.u32Capacity, rStoredCfg.u32Radius));
   if(rStoredCfg.u32Capacity)
   {
      // only patch with values != 0
      rCfg.u32Capacity = rStoredCfg.u32Capacity;
   }
   if(rStoredCfg.u32Radius)
   {
      // only patch with values != 0
      rCfg.u32Radius = rStoredCfg.u32Radius;
   }
}

tVoid fc_sxm_tclDbVersionConfig::vRestore(string const &oSubDir) {
    string cmd;
    string oDstDir = FC_SXM_DEFAULT_SMS_CFG_PATH;
    string oSrcDir = FC_SXM_STATIC_SMS_CFG_PATH;

    ETG_TRACE_USR1(("fc_sxm_tclDbVersionConfig::bRestore() dir=%s START",
            oSubDir.c_str()));

    fc_sxm_tclSmsDirRestorer::instance()->vRestoreDefConfig(oSubDir,oSrcDir,oDstDir);

    ETG_TRACE_USR1(("fc_sxm_tclDbVersionConfig(%s) END",
            oSubDir.c_str()));
}


tBool fc_sxm_tclDbVersionConfig::bIsDbValid(tU32 u32CfgVersion, tU32 u32DpVersion)
{
    tBool bResult = FALSE;
    if(u32CfgVersion == 0xFFFF || u32DpVersion == 0xFFFF)
        return FALSE;

    else if(u32DpVersion != u32CfgVersion)
        bResult = TRUE;

    return bResult;
}


tVoid fc_sxm_tclDbVersionConfig::vUpdateDbVersion(tU32 u32ServiceId)
{
    switch(u32ServiceId)
    {
    case fc_sxm_enServiceID_Fuel:
    {
        if(bIsDbValid(u32GetFuelDbVersion(), fc_sxm_tclPersData::instance()->u32GetFuelVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Fuel DB Version is updated!"));
            vRestore("fuel");
            fc_sxm_tclPersData::instance()->vSetFuelVersion(u32GetFuelDbVersion());
        }
    }
        break;

    case fc_sxm_enServiceID_Canadian_Fuel:
    {
        if(bIsDbValid(u32GetCanadianFuelDbVersion(), fc_sxm_tclPersData::instance()->u32GetCanadianFuelVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Canadian Fuel DB Version is updated!"));
            vRestore("canadianfuel");
            fc_sxm_tclPersData::instance()->vSetCanadianFuelVersion(u32GetCanadianFuelDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Traffic:
    {
        if(bIsDbValid(u32GetTrafficDbVersion(), fc_sxm_tclPersData::instance()->u32GetTrafficVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Traffic DB Version is updated!"));
            vRestore("traffic");
            fc_sxm_tclPersData::instance()->vSetTrafficVersion(u32GetTrafficDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Movies:
    {
        if(bIsDbValid(u32GetMoviesDbVersion(), fc_sxm_tclPersData::instance()->u32GetMoviesVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Movies DB Version is updated!"));
            vRestore("movies");
            fc_sxm_tclPersData::instance()->vSetMoviesVersion(u32GetMoviesDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_AGW:
    {
        if(bIsDbValid(u32GetAgwDbVersion(), fc_sxm_tclPersData::instance()->u32GetAgwVersion())) {
        ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Agw DB Version is updated!"));
        vRestore("agw");
        fc_sxm_tclPersData::instance()->vSetAgwVersion(u32GetAgwDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Tabweather:
    {
        if(bIsDbValid(u32GetTabWeatherDbVersion(), fc_sxm_tclPersData::instance()->u32GetTabWeatherVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() TabWeather DB Version is updated!"));
            vRestore("weather");
            fc_sxm_tclPersData::instance()->vSetTabWeatherVersion(u32GetTabWeatherDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Stocks:
    {
        if(bIsDbValid(u32GetStocksDbVersion(), fc_sxm_tclPersData::instance()->u32GetStocksVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Stocks DB Version is updated!"));
            vRestore("stock_ticker");
            fc_sxm_tclPersData::instance()->vSetStocksVersion(u32GetStocksDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Sports:
    {
        if(bIsDbValid(u32GetSportsDbVersion(), fc_sxm_tclPersData::instance()->u32GetSportsVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Sports DB Version is updated!"));
            vRestore("sports");
            fc_sxm_tclPersData::instance()->vSetSportsVersion(u32GetSportsDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_WsAlerts:
    {
        if(bIsDbValid(u32GetWsAlertsDbVersion(), fc_sxm_tclPersData::instance()->u32GetWsAlertsVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() WsAlerts DB Version is updated!"));
            vRestore("ws_alerts");
            fc_sxm_tclPersData::instance()->vSetWsAlertsVersion(u32GetWsAlertsDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Parking:
    {
        if(bIsDbValid(u32GetParkingDbVersion(), fc_sxm_tclPersData::instance()->u32GetParkingVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Parking DB Version is updated!"));
            vRestore("parking");
            fc_sxm_tclPersData::instance()->vSetParkingVersion(u32GetParkingDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Phonetics:
    {
        if(bIsDbValid(u32GetPhoneticsDbVersion(), fc_sxm_tclPersData::instance()->u32GetPhoneticsVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Phonetics DB Version is updated!"));
            vRestore("phonetics");
            fc_sxm_tclPersData::instance()->vSetPhoneticsVersion(u32GetPhoneticsDbVersion());
        }
    }
        break;
    case fc_sxm_enServiceID_Channelart:
    {
        if(bIsDbValid(u32GetChannelArtVersion(), fc_sxm_tclPersData::instance()->u32GetChannelArtVersion())) {
            ETG_TRACE_USR4(("fc_sxm_tclDbVersionConfig::vCheckDBVersion() Channel art DB Version is updated!"));
            vRestore("channelart");
            fc_sxm_tclPersData::instance()->vSetChannelArtVersion(u32GetChannelArtVersion());
        }
    }
    break;
    default:
        break;
    }
}

tVoid fc_sxm_tclDbVersionConfig::vCheckDBVersion()
{

    vUpdateDbVersion(fc_sxm_enServiceID_Fuel);
    vUpdateDbVersion(fc_sxm_enServiceID_Canadian_Fuel);
    vUpdateDbVersion(fc_sxm_enServiceID_Traffic);
    vUpdateDbVersion(fc_sxm_enServiceID_Movies);
    vUpdateDbVersion(fc_sxm_enServiceID_AGW);
    vUpdateDbVersion(fc_sxm_enServiceID_Tabweather);
    vUpdateDbVersion(fc_sxm_enServiceID_Stocks);
    vUpdateDbVersion(fc_sxm_enServiceID_Sports);
    vUpdateDbVersion(fc_sxm_enServiceID_WsAlerts);
    vUpdateDbVersion(fc_sxm_enServiceID_Parking);
    vUpdateDbVersion(fc_sxm_enServiceID_Phonetics);
    vUpdateDbVersion(fc_sxm_enServiceID_Channelart);

    if(bIsDbValid(u32GetSmsCfgDbVersion(), fc_sxm_tclPersData::instance()->u32GetSmsCfgVersion()))
   {
    	string oDstDir = FC_SXM_DEFAULT_SMS_CFG_PATH;
    	string oSrcDir = FC_SXM_STATIC_SMS_CFG_PATH;
        string oSubDir = "sms.cfg";
    	ETG_TRACE_USR4(("fc_sxm_tclApp::vCheckDBVersion() Sms.cfg Version is updated!"));
       //vRestore("sms.cfg");
       fc_sxm_tclSmsDirRestorer::instance()->vDoRestoreFactory(oSubDir,oSrcDir,oDstDir);
       fc_sxm_tclPersData::instance()->vSetSmsCfgVersion(u32GetSmsCfgDbVersion());
   }
   //States file
   if(bIsDbValid(u32GetStatesVersion(), fc_sxm_tclPersData::instance()->u32GetStatesVersion()))
   {
	   string oDstDir = FC_SXM_DEFAULT_SDK_R_PATH;
	   string oSrcDir = FC_SXM_STATIC_SDK_R_PATH;
       string oSubDir = "states";
       ETG_TRACE_USR4(("fc_sxm_tclApp::vCheckDBVersion() States Version is updated!"));
       //vRestore("states");
       fc_sxm_tclSmsDirRestorer::instance()->vDoRestoreStatesFile(oSubDir,oSrcDir,oDstDir);
       fc_sxm_tclPersData::instance()->vSetStatesVersion(u32GetStatesVersion());
   }
}

