/**
 * @swcomponent fc_sxm
 * @{
 * @file        fc_sxm_tcl_base_dsrl.h
 * @brief       Declaration for BaseDSRL, DSRL type, dsrl adapter classes
 * @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
 *              The reproduction, distribution and utilization of this file as
 *              well as the communication of its contents to others without express
 *              authorization is prohibited. Offenders will be held liable for the
 *              payment of damages. All rights reserved in the event of the grant
 *              of a patent, utility model or design.
 * @}
 */

#ifndef _FC_SXM_TCL_BASE_DSRL_H_
#define _FC_SXM_TCL_BASE_DSRL_H_

#include "fc_sxm_types.h"
#include "fc_sxm_sms.h"

class fc_sxm_tclDataApp;
class fc_sxm_tclBaseDSRL;
struct fc_sxm_trMsgDataSmsEvtDsrlState;

struct fc_sxm_trDsrlTestModification
{
   typedef enum
   {
      enType_Idle,
      enType_EntryRemoved,
      enType_Change1,
      enType_Change2
   } tenType;
   tBool bActive;
   tBool bOnce;
   tenType enType;
   tU32 u32Id;
   tU32 u32Val;

   fc_sxm_trDsrlTestModification()
            : bActive(FALSE),
              bOnce(TRUE),
              enType(enType_Idle),
              u32Id(0),
              u32Val(0)
   {
   }
};

struct fc_sxm_trDsrlLocation
{
   fc_sxm_trDsrlLocation(tS32 s32Latitude_ = 0, tS32 s32Longitude_ = 0, tU32 u32Radius_ = 0)
            : s32Latitude(s32Latitude_),
              s32Longitude(s32Longitude_),
              u32Radius(u32Radius_)
   {
   }
   tS32 s32Latitude;
   tS32 s32Longitude;
   tU32 u32Radius;
   bool operator!=(fc_sxm_trDsrlLocation const &b) const
   {
      SXM_RET_1_IF_NEQ(s32Latitude);
      SXM_RET_1_IF_NEQ(s32Longitude);
      SXM_RET_1_IF_NEQ(u32Radius);
      return FALSE;
   }
   tBool bIsValid() const
   {
      return (s32Latitude != 0) && (s32Longitude != 0);
   }
};

typedef enum
{
   fc_sxm_enDsrlState_Error,
   fc_sxm_enDsrlState_Initial,
   fc_sxm_enDsrlState_Updating,
   fc_sxm_enDsrlState_Ready,
   fc_sxm_enDsrlState_Changed,
   fc_sxm_enDsrlState_Stopped,
   fc_sxm_enDsrlState_WaitUpdating,
   fc_sxm_enDsrlState_LAST
} fc_sxm_tenDsrlState;

struct fc_sxm_trDSRLBaseCfg
{

   virtual ~fc_sxm_trDSRLBaseCfg()
   {
   }

   virtual tVoid vPrint() const;
   virtual tVoid vPrintSpecific() const
   {
   }

   tU32 u32Capacity;
   fc_sxm_trDsrlLocation rLocation;
   tU32 u32Radius;
};

typedef enum
{
   fc_sxm_enCfgMode_Ignore,
   fc_sxm_enCfgMode_Const,
   fc_sxm_enCfgMode_Modify
} fc_sxm_tenCfgMode;

typedef enum
{
   fc_sxm_enDSRLType_Standard,
   fc_sxm_enDSRLType_Dest,
   fc_sxm_enDSRLType_Device,
   fc_sxm_enDSRLType_Favorites,
   fc_sxm_enDSRLType_MAX
} fc_sxm_tenDSRLType;

struct fc_sxm_trDSRLGenericCfg :
   public fc_sxm_trDSRLBaseCfg
{
   fc_sxm_trDSRLGenericCfg(tU8 const *pu8Data)
   {
      enDsrlType = (fc_sxm_tenDSRLType) pu8Data[0];
      u32Capacity = SXM_GET_U32(&pu8Data[1]);
      tS32 s32Lat = (tS32) SXM_GET_U32(&pu8Data[5]);
      tS32 s32Long = (tS32) SXM_GET_U32(&pu8Data[9]);
      u32Radius = SXM_GET_U32(&pu8Data[13]);
      rLocation = fc_sxm_trDsrlLocation(s32Lat, s32Long, u32Radius);
      u8SortMethod = pu8Data[17];
      u8FilterMethod = pu8Data[18];
      u32Misc = SXM_GET_U32(&pu8Data[19]);
      oString = (const char *) &pu8Data[23];
   }
   tVoid vPrintGenericCfg(tU16 u16TraceClass, char const *szPrefix) const;
   fc_sxm_tenDSRLType enDsrlType;
   tU8 u8SortMethod;
   tU8 u8FilterMethod;
   tU32 u32Misc;
   string oString;
};

struct fc_sxm_trDSRLConstCfg
{
   fc_sxm_trDSRLConstCfg(fc_sxm_tenDSRLType enDsrlType_,
      tU32 u32DsrlId_,
      fc_sxm_tclDataApp *poDataApp_,
      tU16 u16TraceClass_)
            : enDsrlType(enDsrlType_),
              u32DsrlId(u32DsrlId_),
              poDataApp(poDataApp_),
              u16TraceClass(u16TraceClass_)
   {
   }

   fc_sxm_tenDSRLType enDsrlType;
   tU32 u32DsrlId;
   fc_sxm_tclDataApp *poDataApp;
   tU16 u16TraceClass;
   tVoid vClear()
   {
      u32DsrlId = 0;
      poDataApp = OSAL_NULL;
   }

   tVoid vPrint() const;
};

struct fc_sxm_trDsrlStatistics
{
   fc_sxm_trDsrlStatistics()
            : u32NumFiltered(0),
              u32NumSorted(0),
              u32NumIterated(0),

              u32NumIteratedUnchanged(0),
              u32NumIteratedNew(0),
              u32NumIteratedChanged(0),
              u32NumIteratedRemoved(0),
              u32NumIteratedUnknown(0),
              u32NumIteratedInvalid(0),

              rMSecLastEvt(0),
              rMSecLastUpdateStart(0),
              rMSecLastReady(0),
              rMSecLastIterateEnd(0)
   {
   }
   tU32 u32NumFiltered;
   tU32 u32NumSorted;
   tU32 u32NumIterated;

   tU32 u32NumIteratedUnchanged;
   tU32 u32NumIteratedNew;
   tU32 u32NumIteratedChanged;
   tU32 u32NumIteratedRemoved;
   tU32 u32NumIteratedUnknown;
   tU32 u32NumIteratedInvalid;

   OSAL_tMSecond rMSecLastEvt;
   OSAL_tMSecond rMSecLastUpdateStart;
   OSAL_tMSecond rMSecLastReady;
   OSAL_tMSecond rMSecLastIterateEnd;

   tVoid vPrint() const;
   tVoid vResetIterate()
   {
      u32NumIterated = 0;
      u32NumIteratedUnchanged = 0;
      u32NumIteratedNew = 0;
      u32NumIteratedChanged = 0;
      u32NumIteratedRemoved = 0;
      u32NumIteratedUnknown = 0;
      u32NumIteratedInvalid = 0;
   }
};

class fc_sxm_tclDsrlTypeAdapter
{
public:
   fc_sxm_tclDsrlTypeAdapter(fc_sxm_tclBaseDSRL *prBaseDsrl);

   virtual ~fc_sxm_tclDsrlTypeAdapter()
   {
      //vDestroy();
      vDestroySMSDSRL(&_hSmsDSRL);
      _hSmsDSRL = DSRL_INVALID_OBJECT;
      _prBaseDsrl = NULL;
   }

   // check how DSRL used Location
   virtual fc_sxm_tenCfgMode enGetLocationCfgMode() const
   {
      return fc_sxm_enCfgMode_Ignore;
   }
   // check how DSRL used Radius
   virtual fc_sxm_tenCfgMode enGetRadiusCfgMode() const
   {
      return fc_sxm_enCfgMode_Ignore;
   }

   // default behavior: we can modify capacity
   tBool bModifyCapacity(tU32 u32Capacity);

   tVoid vOnSmsDsrlStopped()
   {
      _hSmsDSRL = DSRL_INVALID_OBJECT;
   }

   // default behavior: we can not modify the location
   virtual tBool bModifyLocation(fc_sxm_trDsrlLocation const &rLocation, tU32 u32Radius)
   {
      (tVoid) rLocation;
      (tVoid) u32Radius;
      return FALSE;
   }

   virtual tVoid vPatchLocationCfg(fc_sxm_trDsrlLocation &rLocation)
   {
      (tVoid) rLocation;
      return;
   }
   // default behavior: we can modify the filter
   tBool bModifyFilter();

   // create dsrl with given config
   virtual tBool bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg)=0;

   // destroy the DSRL
   virtual tVoid vDestroy()
   {
      vDestroySMSDSRL(&_hSmsDSRL);
   }

   // default behavior: we have no location
   virtual LOCATION_OBJECT hGetSMSLocation()
   {
      return LOCATION_INVALID_OBJECT;
   }
   //DSRL object handle
   DSRL_OBJECT hGetSMSDSRL()
   {
      return _hSmsDSRL;
   }
   ;

   // utiliys to create and delete of sms-objects
   static DISTANCE_OBJECT hCreateSMSRadius(tU32 u32Radius);
   static LOCATION_OBJECT hCreateSMSLocation(tS32 s32Lat, tS32 s32Long, DISTANCE_OBJECT hSmsRadius);
   static tVoid vDestroySMSRadius(DISTANCE_OBJECT *phSmsRadius);

   static tVoid vDestroySMSLocation(LOCATION_OBJECT *phSmsLocation);

protected:

   tVoid vDestroySMSDSRL(DSRL_OBJECT *phSmsDSRL) const;

   //DSRL object handle
   DSRL_OBJECT _hSmsDSRL;

   fc_sxm_tclBaseDSRL *_prBaseDsrl;
   fc_sxm_trDSRLConstCfg const &_rConstCfg;
   fc_sxm_tclDataApp *poGetDataApp();

   //DSRL event callback
   DSRL_CALLBACK _pfDSRLEvent;

};

class fc_sxm_tclDsrlTypeStandard :
   public fc_sxm_tclDsrlTypeAdapter
{

public:
   fc_sxm_tclDsrlTypeStandard(fc_sxm_tclBaseDSRL *prBaseDsrl)
            : fc_sxm_tclDsrlTypeAdapter(prBaseDsrl),
              _hSmsLocation(LOCATION_INVALID_OBJECT),
              _hSmsRadius(DISTANCE_INVALID_OBJECT)
   {
   }
   virtual ~fc_sxm_tclDsrlTypeStandard()
   {
      _hSmsLocation = LOCATION_INVALID_OBJECT;
      _hSmsRadius = DISTANCE_INVALID_OBJECT;
   }
   virtual fc_sxm_tenCfgMode enGetLocationCfgMode() const
   {
      return fc_sxm_enCfgMode_Modify;
   }
   virtual fc_sxm_tenCfgMode enGetRadiusCfgMode() const
   {
      return fc_sxm_enCfgMode_Modify;
   }

   virtual tBool bModifyLocation(fc_sxm_trDsrlLocation const &rLocation, tU32 u32Radius);

   virtual tBool bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg);
   virtual tVoid vDestroy()
   {
      vDestroySMSRadius(&_hSmsRadius);
      vDestroySMSLocation(&_hSmsLocation);
      fc_sxm_tclDsrlTypeAdapter::vDestroy();
   }

   virtual LOCATION_OBJECT hGetSMSLocation()
   {
      return _hSmsLocation;
   }
private:
   //Location object
   LOCATION_OBJECT _hSmsLocation;
   //Radius around above location
   DISTANCE_OBJECT _hSmsRadius;

};

class fc_sxm_tclDsrlTypeNearDest :
   public fc_sxm_tclDsrlTypeStandard
{

public:
   fc_sxm_tclDsrlTypeNearDest(fc_sxm_tclBaseDSRL *prBaseDsrl)
            : fc_sxm_tclDsrlTypeStandard(prBaseDsrl)
   {
   }

   virtual tVoid vPatchLocationCfg(fc_sxm_trDsrlLocation &rLocation);
};

class fc_sxm_tclDsrlTypeDevice :
   public fc_sxm_tclDsrlTypeAdapter
{
public:
   fc_sxm_tclDsrlTypeDevice(fc_sxm_tclBaseDSRL *prBaseDsrl)
            : fc_sxm_tclDsrlTypeAdapter(prBaseDsrl),
              _hSmsRadius(DISTANCE_INVALID_OBJECT)
   {
   }

   //Location object
   virtual ~fc_sxm_tclDsrlTypeDevice()
   {
      _hSmsRadius = DISTANCE_INVALID_OBJECT;
   }
   virtual fc_sxm_tenCfgMode enGetLocationCfgMode() const
   {
      return fc_sxm_enCfgMode_Ignore;
   }
   virtual fc_sxm_tenCfgMode enGetRadiusCfgMode() const
   {
      return fc_sxm_enCfgMode_Const;
   }
   virtual tBool bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg);
   virtual tVoid vDestroy()
   {
      fc_sxm_tclDsrlTypeAdapter::vDestroy();
   }
   // only to be used in callback-context
   virtual LOCATION_OBJECT hGetSMSLocation()
   {
      return DSRL.hDeviceLocation(_hSmsDSRL);
      //fc_sxm_hGetSmsDeviceLocation();
   }

   virtual tVoid vPatchLocationCfg(fc_sxm_trDsrlLocation &rLocation)
   {
      rLocation = fc_sxm_trDsrlLocation();
   }

private:
   //Radius around device location
   DISTANCE_OBJECT _hSmsRadius;

};

class fc_sxm_tclDsrlTypeFavorites :
   public fc_sxm_tclDsrlTypeAdapter
{
public:
   fc_sxm_tclDsrlTypeFavorites(fc_sxm_tclBaseDSRL *prBaseDsrl)
            : fc_sxm_tclDsrlTypeAdapter(prBaseDsrl)
   {
   }

   virtual ~fc_sxm_tclDsrlTypeFavorites()
   {
      // so far nothing
   }
   virtual tBool bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg);
   virtual tVoid vDestroy()
   {
      fc_sxm_tclDsrlTypeAdapter::vDestroy();
   }

private:

};

/* common base-class for all DSRLS.
 The type of the DSRL (Standard, Device, Favorites is handled by the member _prAdapter)
 */

class fc_sxm_tclBaseDSRL
{
public:

   //Constructor
   fc_sxm_tclBaseDSRL(fc_sxm_trDSRLConstCfg const &rConstCfg);

   //copy constructor
   fc_sxm_tclBaseDSRL(fc_sxm_tclBaseDSRL const &rBaseDSRL);

   //assignment operator
   fc_sxm_tclBaseDSRL& operator= (fc_sxm_tclBaseDSRL const &rBaseDSRL);

   //Destructor
   virtual ~fc_sxm_tclBaseDSRL();

   //Init the DSRL
   virtual tVoid vInitialize(tVoid)
   {
   }
   ;

   //DeInit the DSRL
   virtual tVoid vDeInitialize(tVoid)
   {
   }
   ;

   tVoid vForceUpdate();
   tVoid vOnSmsServiceStateChanged(tBool bSmsServiceStopping)
   {
      _bSmsServiceStopping = bSmsServiceStopping;
      bApplyNextConfig();
   }

   tVoid vOnSmsDsrlStopped()
   {
      _prAdapter->vOnSmsDsrlStopped();
   }

   //Update notification when the DSRL is ready and iterated
   virtual tVoid vOnDSRLUpdate(tVoid) = 0;

   fc_sxm_tclDataApp *poGetDataApp()
   {
      return _rConstCfg.poDataApp;
   }

   fc_sxm_tenDSRLType enGetDSRLType()
   {
      return _rConstCfg.enDsrlType;
   }
   fc_sxm_trDSRLConstCfg const &rGetConstCfg()
   {
      return _rConstCfg;
   }

   /* no sem-protection
    data will partly be written in dsrl-callbacks, but be read in app-context
    */
   fc_sxm_trDsrlStatistics rStatistics;

   fc_sxm_trDsrlStatistics rGetStatistics()
   {
      return rStatistics;
   }

   struct trDSRLCheckResult
   {
      trDSRLCheckResult()
               : bRecreate(FALSE),
                 bModifyCapacity(FALSE),
                 bModifyLocation(FALSE),
                 bModifyFilter(FALSE)
      {
      }
      tBool bRecreate;
      tBool bModifyCapacity;
      tBool bModifyLocation;
      tBool bModifyFilter;

      tBool bNeedsUpdate()
      {
         return bModifyCapacity || bModifyLocation || bModifyFilter || bRecreate;
      }
   };

   tU32 u32GetId() const
   {
      return _rConstCfg.u32DsrlId;
   }

   //DSRL object handle
   DSRL_OBJECT hGetSMSDSRL() const
   {
      return _prAdapter->hGetSMSDSRL();
   }
   ;

   //private:
   virtual tBool bHandleFilterCallback(DSRL_OBJECT hDSRL, DSRL_ENTRY_OBJECT hEntry)=0;

   virtual tS16 s16HandleSortCallback(DSRL_OBJECT hDSRL, DSRL_ENTRY_OBJECT hEntry1, DSRL_ENTRY_OBJECT hEntry2)=0;

   virtual tBool bHandleIterateCallback(DSRL_OBJECT hDSRL,
      DSRL_ENTRY_ID tEntryID,
      DSRL_ENTRY_STATUS_ENUM eStatus,
      DSRL_ENTRY_OBJECT hEntryObject)=0;

   /*
    special callback needed, if entry became invalid:
    no object is provided to get the unique id, so we use DSRL_ENTRY_ID as key
    todo: make pure virtual if implementation is updated in all apps
    */
   virtual tVoid vHandleDsrlEntryInvalid(DSRL_ENTRY_ID tEntryID)
   {
      (tVoid) tEntryID;
   }
   ;

   static tS32 s32FloatToSmsDist(float floatArg)
   {
      return (tU32) (floatArg * 0xFFFF);
   }

   //Process the DSRL state
   tVoid vOnSmsDsrlState(DSRL_STATE_ENUM enState);

   virtual tBool bUseSMSDefaultSort()=0;

   DSRL_STATE_ENUM enGetSmsDsrlState();

   tVoid vSetLocation(fc_sxm_trDsrlLocation const &rLocation);
   virtual tVoid vSetNextConfig()
   {
      bApplyNextConfig();
   }

   tVoid vSetChangedFlag(tBool bDataChanged);

   /* Method used to indicate the data is available from sms*/
   tVoid vSetDataAvailableFlag(tBool bDataChanged);

   tBool bGetChangedFlag() const;
   tBool bGetDataAvailableFlag() const;

   /* Method to get DSRL ready count */
   tU32 u32GetNumDsrlReadyCount()
   {
      return _u32DsrlReadyCount;
   }

   /* Method to increment the DSRL ready callback count */
   tVoid vIncrementDsrlReadyCount()
   {
      _u32DsrlReadyCount++;
   }

   /* Method to ReSet the DSRL Ready Count*/
   tVoid vResetDsrlReadyCount()
   {
      _u32DsrlReadyCount = 0;
   }

   const DATASERVICE_MGR_OBJECT hGetSmsService();
   DATASERVICE_STATE_ENUM enGetSmsServiceState();
   DATA_PRODUCT_STATE_ENUM enGetDataProductState(DATASERVICE_MGR_OBJECT hDataService);

   fc_sxm_trDsrlTestModification _rTestModification;

   tVoid vPrintReport() const;

   tVoid vPrintStatistics() const
   {
      rStatistics.vPrint();
   }

   tVoid vPrintCfg() const
   {
      _prCurrentCfg->vPrint();
   }

protected:
   virtual trDSRLCheckResult rCheckNextConfig();
   virtual tBool bApplyNextConfig();
   tVoid vTraceNewConfig(fc_sxm_trDSRLBaseCfg const &rNewConfig) const;

   fc_sxm_trDSRLConstCfg _rConstCfg;
   fc_sxm_trDSRLBaseCfg *_prNextCfg;
   fc_sxm_trDSRLBaseCfg const *_prCurrentCfg;
   tBool _bSmsServiceStopping;

   LOCATION_OBJECT hGetSMSLocation()
   {
      return _prAdapter->hGetSMSLocation();
   }

   /* sort and filter depend on application-specific data-types,
    so checking has to be implemented by derived class
    */
   virtual tBool bIsFilterChanged()=0;
   virtual tBool bIsSortChanged()=0;
   virtual tVoid vCopyConfig()=0;

private:
   tVoid vNotifyDsrlState(fc_sxm_tenDsrlState enDsrlState);

   fc_sxm_tclDsrlTypeAdapter *_prAdapter;
   tBool _bDataChanged;
   tBool _bDataAvailable;
   tU32 _u32DsrlReadyCount;
   //Destroy the DSRL
   tVoid vDestroy(tVoid);
   DSRL_STATE_ENUM _enDsrlState;

   fc_sxm_tenDsrlState _enDsrlStateForApp;

};

// external helper-fn to get a new id for our DSRL (we can not include the header of prDataApp from here)
extern tU32 fc_sxm_u32GetNewDsrlId(fc_sxm_tclDataApp *prDataApp);

/*
 Template to create DSRLs with application-specific config-parameters for Sort and Filter
 */
template<class CONFIG>
class fc_sxm_tclConfigDSRL :
   public fc_sxm_tclBaseDSRL
{
public:

   fc_sxm_tclConfigDSRL(fc_sxm_tenDSRLType enDsrlType, fc_sxm_tclDataApp *prDataApp, tU16 u16TraceClass)
            : fc_sxm_tclBaseDSRL(fc_sxm_trDSRLConstCfg(enDsrlType,
               fc_sxm_u32GetNewDsrlId(prDataApp),
               prDataApp,
               u16TraceClass))
   {
      _prCurrentCfg = &_rCurrentConfig;
      _prNextCfg = &_rNextConfig;
   }

   tVoid vSetNextConfig(CONFIG const &rConfig)
   {
      vTraceNewConfig(rConfig);

      _rNextConfig = rConfig;
      bApplyNextConfig();
   }

   CONFIG const &rGetCurrentConfig()
   {
      return _rCurrentConfig;
   }

   CONFIG const &rGetNextConfig()
   {
      return _rNextConfig;
   }

   virtual tVoid vCopyConfig()
   {
      _rCurrentConfig = _rNextConfig;
   }

   virtual tBool bUseSMSDefaultSort()
   {
      return (tU32) _rCurrentConfig.enSortMethod == 0;
   }
protected:
   /*
    sort and filter are defined here.
    rFilterCfg must provide operator!=
    */
   virtual tBool bIsFilterChanged()
   {
      return _rNextConfig.rFilterCfg != _rCurrentConfig.rFilterCfg;
   }

   virtual tBool bIsSortChanged()
   {
      return _rNextConfig.enSortMethod != _rCurrentConfig.enSortMethod;
   }

private:
   CONFIG _rNextConfig;
   CONFIG _rCurrentConfig;

};

#endif //_FC_SXM_TCL_BASE_DSRL_H_
