/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_base_dsrl.cpp
* @brief       Implementation of Base DSRL fucnality
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_tcl_data_app.h"
#include "fc_sxm_tcl_base_dsrl.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_BASE_DSRL
#include "trcGenProj/Header/fc_sxm_tcl_base_dsrl.cpp.trc.h"
#endif



/* generic callback-fn for dsrl-state */
static tVoid cb_vDSRLEventCallback(DSRL_OBJECT hSmsDSRL, 
                                   DSRL_STATE_ENUM eState,
                                   tVoid *pvCallbackArg) {
    (tVoid)hSmsDSRL;
    fc_sxm_tclBaseDSRL *prBaseDrsl=(fc_sxm_tclBaseDSRL *)pvCallbackArg;

    SXM_ASSERT_RETURN(OSAL_NULL !=prBaseDrsl);
    if (DSRL_STATE_STOPPED==eState) {
        prBaseDrsl->vOnSmsDsrlStopped();
    }

    fc_sxm_trDSRLConstCfg const &rConstCfg=prBaseDrsl->rGetConstCfg();
    fc_sxm_tclDataApp *poDataApp=rConstCfg.poDataApp;
    poDataApp->vOnSmsDsrlState(rConstCfg.u32DsrlId, eState);
}

/* 
   generic callback-fn for filter
   forward to baseDsrl given in pvFilterArg
*/
static tBool cb_bDSRLFilter(DSRL_OBJECT hSmsDSRL, 
                            DSRL_ENTRY_OBJECT hEntry,
                            tVoid *pvFilterArg) {
    fc_sxm_tclBaseDSRL *prBaseDrsl=(fc_sxm_tclBaseDSRL *)pvFilterArg;
    prBaseDrsl->rStatistics.u32NumFiltered++;
    return prBaseDrsl->bHandleFilterCallback(hSmsDSRL,  hEntry);
}

/* 
   generic callback-fn for sort
   forward to baseDsrl given in pvSortArg
*/
static tS16 cb_n16DSRLSort(DSRL_OBJECT hSmsDSRL, 
                           DSRL_ENTRY_OBJECT hEntry1,
                           DSRL_ENTRY_OBJECT hEntry2, 
                           tVoid *pvSortArg) {
    fc_sxm_tclBaseDSRL *prBaseDrsl=(fc_sxm_tclBaseDSRL *)pvSortArg;
    prBaseDrsl->rStatistics.u32NumSorted++;
    return prBaseDrsl->s16HandleSortCallback(hSmsDSRL, hEntry1, hEntry2);
}

/* 
   generic callback-fn for iterate
   forward to baseDsrl given in pvIteratorArg
*/
static tBool cb_bDSRLIterate(DSRL_OBJECT hSmsDSRL, 
                             DSRL_ENTRY_ID tEntryID,
                             DSRL_ENTRY_STATUS_ENUM eStatus, 
                             DSRL_ENTRY_OBJECT hEntry, 
                             tVoid *pvIteratorArg) {
    fc_sxm_tclBaseDSRL *prBaseDrsl=(fc_sxm_tclBaseDSRL *)pvIteratorArg;
    prBaseDrsl->rStatistics.u32NumIterated++;
    ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL::cb_bDSRLIterate tEntryID=%u eStatus=%d  START", tEntryID, eStatus));
    fc_sxm_trDsrlTestModification &rTestModification=prBaseDrsl->_rTestModification;
    if (rTestModification.bActive) {
        ETG_TRACE_USR4(("TestModification::bActive=%d u32Id=%u START", rTestModification.bActive, rTestModification.u32Id));
        if (rTestModification.u32Id==tEntryID) {
            if (rTestModification.enType==fc_sxm_trDsrlTestModification::enType_EntryRemoved) {
                ETG_TRACE_USR4(("REMOVED"));
                eStatus=DSRL_ENTRY_STATUS_REMOVED;
            } else if (rTestModification.enType!=fc_sxm_trDsrlTestModification::enType_Idle) {
                if (eStatus==DSRL_ENTRY_STATUS_UNCHANGED) {
                    ETG_TRACE_USR4(("CHANGED"));
                    eStatus=DSRL_ENTRY_STATUS_CHANGED;
                }
            }
        }
    }
 
    //Set and indicate that DSRL data is available
    prBaseDrsl->vSetDataAvailableFlag(TRUE);

    switch (eStatus) {
        case DSRL_ENTRY_STATUS_UNCHANGED:
            prBaseDrsl->rStatistics.u32NumIteratedUnchanged++;   
            return TRUE; 
        case DSRL_ENTRY_STATUS_NEW:
            prBaseDrsl->rStatistics.u32NumIteratedNew++;    
            break;
        case DSRL_ENTRY_STATUS_CHANGED:
            prBaseDrsl->rStatistics.u32NumIteratedChanged++;    
            break;
        case DSRL_ENTRY_STATUS_REMOVED:
            prBaseDrsl->rStatistics.u32NumIteratedRemoved++;    
            break;
        case DSRL_ENTRY_STATUS_UNKNOWN:
            prBaseDrsl->rStatistics.u32NumIteratedUnknown++;    
            break;
    }
 
    if (hEntry==DSRL_ENTRY_INVALID_OBJECT) {
        prBaseDrsl->rStatistics.u32NumIteratedInvalid++;    
        ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL::cb_bDSRLIterate call vHandleDsrlEntryInvalid"));
        prBaseDrsl->vHandleDsrlEntryInvalid(tEntryID);
        return TRUE;
    }

    //Set and indicate DSRL data available flag
    prBaseDrsl->vSetDataAvailableFlag(TRUE);
    prBaseDrsl->bHandleIterateCallback(hSmsDSRL, tEntryID, eStatus, hEntry);
    return TRUE;
}

tVoid fc_sxm_trDsrlStatistics::vPrint() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnumFiltered=%u",    u32NumFiltered));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnumSorted  =%u",    u32NumSorted));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tnumIterated=%u",    u32NumIterated));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tUNCHANGED =%u",   u32NumIteratedUnchanged));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tNEW       =%u",   u32NumIteratedNew));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tCHANGED   =%u",   u32NumIteratedChanged));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tREMOVED   =%u",   u32NumIteratedRemoved));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tUNKNOWN   =%u",   u32NumIteratedUnknown));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t\tINVALID   =%u",   u32NumIteratedInvalid));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tlastEvt    =%u.%u", rMSecLastEvt/1000,         rMSecLastEvt%1000));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tlastUpdateS=%u.%u", rMSecLastUpdateStart/1000, rMSecLastUpdateStart%1000));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tlastReady  =%u.%u", rMSecLastReady/1000,       rMSecLastReady%1000));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\tlastIterEnd=%u.%u", rMSecLastIterateEnd/1000,  rMSecLastIterateEnd%1000));
}

tVoid fc_sxm_trDSRLBaseCfg::vPrint() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t u32Capacity=%u", u32Capacity));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t location   =%u:%u", rLocation.s32Latitude, rLocation.s32Longitude));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\t\t u32Radius  =%u", u32Radius));
    vPrintSpecific();
}

tVoid fc_sxm_trDSRLConstCfg::vPrint() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tID         =%d", u32DsrlId));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tType       =%d", ETG_CENUM(fc_sxm_tenDSRLType, enDsrlType)));
}


tVoid fc_sxm_trDSRLGenericCfg::vPrintGenericCfg(tU16 u16TraceClass, char const *szPrefix) const {
    ETG_TRACE_USR4_DCL((u16TraceClass, "fc_sxm_trDSRLGenericCfg::rGenericCfg: (%s)", szPrefix));
    ETG_TRACE_USR4_DCL((u16TraceClass, "enDsrlType=%d u32Capacity=%u rLocation:%d:%d, radius=%u, u8Sort=%u u8Filter=%u u32Misc=%u string=%s",
                        enDsrlType,
                        u32Capacity,
                        rLocation.s32Latitude,
                        rLocation.s32Longitude,
                        u32Radius,
                        u8SortMethod,
                        u8FilterMethod,
                        u32Misc,
                        oString.c_str()
                        ));
}

/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclBaseDSRL
 *
 *DESCRIPTION:  Constructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclBaseDSRL::fc_sxm_tclBaseDSRL(fc_sxm_trDSRLConstCfg const &rConstCfg):
    _rConstCfg(rConstCfg),
    _prNextCfg(OSAL_NULL),
    _prCurrentCfg(OSAL_NULL),
	_bSmsServiceStopping(FALSE),
    _prAdapter(OSAL_NULL),
    _bDataChanged(FALSE),
    _bDataAvailable(FALSE),
    _u32DsrlReadyCount(0),
    _enDsrlState(DSRL_STATE_STOPPED),
    _enDsrlStateForApp(fc_sxm_enDsrlState_LAST)
{
    // Create sub-object according to DSRL-Type
    switch (rConstCfg.enDsrlType) {
        case fc_sxm_enDSRLType_Standard:
        {
            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeStandard(this);
        }
        break;
        case fc_sxm_enDSRLType_Dest:
        {
            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeNearDest(this);
        }
        break;
        case fc_sxm_enDSRLType_Device:
        {
            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeDevice(this);
        }
        break;
        case fc_sxm_enDSRLType_Favorites:
        {
            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeFavorites(this);
        }
        break;
        default:

            break;
    }
    SXM_ASSERT_RETURN(OSAL_NULL != _prAdapter);
}

//copy constructor
fc_sxm_tclBaseDSRL::fc_sxm_tclBaseDSRL(fc_sxm_tclBaseDSRL const &rBaseDSRL):
    	    _rConstCfg(rBaseDSRL._rConstCfg),
    	    _prNextCfg(rBaseDSRL._prNextCfg),
    	    _prCurrentCfg(rBaseDSRL._prCurrentCfg),
			_prAdapter(OSAL_NULL),
    		_bSmsServiceStopping(rBaseDSRL._bSmsServiceStopping),
    	    _bDataChanged(rBaseDSRL._bDataChanged),
    	    _bDataAvailable(rBaseDSRL._bDataAvailable),
    	    _u32DsrlReadyCount(rBaseDSRL._u32DsrlReadyCount),
    	    _enDsrlState(rBaseDSRL._enDsrlState),
    	    _enDsrlStateForApp(rBaseDSRL._enDsrlStateForApp)
{
	// Create sub-object according to DSRL-Type
	    switch (_rConstCfg.enDsrlType) {
	        case fc_sxm_enDSRLType_Standard:
	        {
	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeStandard(this);
	        }
	        break;
	        case fc_sxm_enDSRLType_Dest:
	        {
	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeNearDest(this);
	        }
	        break;
	        case fc_sxm_enDSRLType_Device:
	        {
	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeDevice(this);
	        }
	        break;
	        case fc_sxm_enDSRLType_Favorites:
	        {
	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeFavorites(this);
	        }
	        break;
	        default:

	            break;
	    }
	    SXM_ASSERT_RETURN(OSAL_NULL != _prAdapter);
	    _prAdapter = rBaseDSRL._prAdapter;

}

//Assignment operator
fc_sxm_tclBaseDSRL& fc_sxm_tclBaseDSRL::operator=(fc_sxm_tclBaseDSRL const &rBaseDSRL)
{
	if (&rBaseDSRL != this) {
		_rConstCfg = rBaseDSRL._rConstCfg;
		_prNextCfg = rBaseDSRL._prNextCfg;
		_prCurrentCfg = rBaseDSRL._prCurrentCfg;
		_bSmsServiceStopping = rBaseDSRL._bSmsServiceStopping;
		_bDataChanged = rBaseDSRL._bDataChanged;
		_bDataAvailable = rBaseDSRL._bDataAvailable;
		_u32DsrlReadyCount = rBaseDSRL._u32DsrlReadyCount;
		_enDsrlState = rBaseDSRL._enDsrlState;
		_enDsrlStateForApp = rBaseDSRL._enDsrlStateForApp;

    // Create sub-object according to DSRL-Type
    	    switch (_rConstCfg.enDsrlType) {
    	        case fc_sxm_enDSRLType_Standard:
    	        {
    	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeStandard(this);
    	        }
    	        break;
    	        case fc_sxm_enDSRLType_Dest:
    	        {
    	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeNearDest(this);
    	        }
    	        break;
    	        case fc_sxm_enDSRLType_Device:
    	        {
    	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeDevice(this);
    	        }
    	        break;
    	        case fc_sxm_enDSRLType_Favorites:
    	        {
    	            _prAdapter= OSAL_NEW fc_sxm_tclDsrlTypeFavorites(this);
    	        }
    	        break;
    	        default:

    	            break;
    	    }

    	    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _prAdapter,*this);
    	    _prAdapter = rBaseDSRL._prAdapter;
			
	}
    	    return *this;

}

/*********************************************************************
 *
 *FUNCTION:     fc_sxm_tclBaseDSRL
 *
 *DESCRIPTION:  Destructor
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
fc_sxm_tclBaseDSRL::~fc_sxm_tclBaseDSRL()
{
    if (_prAdapter != OSAL_NULL) {
        OSAL_DELETE _prAdapter;
        _prAdapter=OSAL_NULL;
    }
    _prCurrentCfg=OSAL_NULL,
    _prNextCfg=OSAL_NULL,
    _rConstCfg.vClear();

}

fc_sxm_tclDataApp *fc_sxm_tclDsrlTypeAdapter::poGetDataApp() {
    
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _prBaseDsrl, OSAL_NULL);
    return _prBaseDsrl->poGetDataApp();
}

DSRL_STATE_ENUM fc_sxm_tclBaseDSRL::enGetSmsDsrlState() {
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::enGetSmsDsrlState=poDsrl=0x%08x _enDsrlState=%d", 
                  this,
                  ETG_CENUM(DSRL_STATE_ENUM, _enDsrlState)));
    return _enDsrlState;
}

tVoid fc_sxm_tclBaseDSRL::vSetLocation(fc_sxm_trDsrlLocation const &rLocation) {
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vSetLocation=lat=%d long=%d", 
                    rLocation.s32Latitude,
                    rLocation.s32Longitude));
    
    SXM_ASSERT_RETURN(OSAL_NULL !=_prNextCfg);
    _prNextCfg->rLocation=rLocation;
    _prNextCfg->rLocation.u32Radius=0;

    /* Reset the DSRL Ready Count as DSRL location is changed */
    vResetDsrlReadyCount();

    bApplyNextConfig();
}

const DATASERVICE_MGR_OBJECT fc_sxm_tclBaseDSRL::hGetSmsService() {
    fc_sxm_tclDataApp *poDataApp=poGetDataApp();
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != poDataApp, DATASERVICE_MGR_INVALID_OBJECT);
    return poDataApp->hGetSmsService();
}

DATASERVICE_STATE_ENUM fc_sxm_tclBaseDSRL::enGetSmsServiceState() {
    fc_sxm_tclDataApp *poDataApp=poGetDataApp();
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != poDataApp, DATASERVICE_STATE_INVALID);
    return poDataApp->enGetSmsServiceState();
}
DATA_PRODUCT_STATE_ENUM fc_sxm_tclBaseDSRL::enGetDataProductState(DATASERVICE_MGR_OBJECT hDataService) {
    fc_sxm_tclDataApp *poDataApp=poGetDataApp();
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != poDataApp, DATA_PRODUCT_STATE_ERROR);
    return poDataApp->enGetDataProductState(hDataService);
}

tVoid fc_sxm_tclBaseDSRL::vSetChangedFlag(tBool bDataChanged) {
    if (_bDataChanged  != bDataChanged) {
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vSetChangedFlag=%d", bDataChanged));
        _bDataChanged=bDataChanged;
    }
}
tVoid fc_sxm_tclBaseDSRL::vSetDataAvailableFlag(tBool bDataAvailable) {
    if (_bDataAvailable  != bDataAvailable) {
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vSetDataAvailableFlag=%d", bDataAvailable));
        _bDataAvailable=bDataAvailable;
    }
}
tBool fc_sxm_tclBaseDSRL::bGetChangedFlag() const {
    return _bDataChanged;
}
tBool fc_sxm_tclBaseDSRL::bGetDataAvailableFlag() const {
    return _bDataAvailable;
}
/*
  Check wihich actions have to be taken with the new configuration.
  bRecreate: DSRL has to be destroyed an recreated to take the new config
  bModifyXXX: TRUE: The according configuration of the DSRL has to be modified
  FALSE: The configuration-item is unchanged
*/
fc_sxm_tclBaseDSRL::trDSRLCheckResult fc_sxm_tclBaseDSRL::rCheckNextConfig() {
    tBool bLocationChanged=FALSE;
    tBool bRadiusChanged=FALSE;
    trDSRLCheckResult rRes;
	if( hGetSMSDSRL() == DSRL_INVALID_OBJECT)
	{
		/* For the First Time,DSRL has to be created */
		rRes.bRecreate = TRUE;
	}

    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prNextCfg, rRes);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prCurrentCfg, rRes);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prAdapter, rRes);

    rRes.bModifyCapacity = _prNextCfg->u32Capacity != _prCurrentCfg->u32Capacity;
    ETG_TRACE_USR3_DCL((_rConstCfg.u16TraceClass, "rCheckNextConfig enDsrlType=%d locationCfgMode=%d radiusCfgmode=%d START",
                    ETG_CENUM(fc_sxm_tenDSRLType, _rConstCfg.enDsrlType),
                    ETG_CENUM(fc_sxm_tenCfgMode, _prAdapter->enGetLocationCfgMode()),
                    ETG_CENUM(fc_sxm_tenCfgMode, _prAdapter->enGetRadiusCfgMode())));
    if (_prNextCfg->u32Radius != _prCurrentCfg->u32Radius) {
        bRadiusChanged=TRUE;
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "Radius: %u-->%u", _prCurrentCfg->u32Radius, _prNextCfg->u32Radius));    
        if (_prAdapter->enGetRadiusCfgMode() == fc_sxm_enCfgMode_Const) {
            rRes.bRecreate=TRUE;
        } else if (_prAdapter->enGetRadiusCfgMode() == fc_sxm_enCfgMode_Modify) {
            rRes.bModifyLocation=TRUE;
        } 
    }



    if (_prNextCfg->rLocation != _prCurrentCfg->rLocation) {
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "rCheckNextConfig location:lat:%d-->%d long:%d-->%d",
                        _prCurrentCfg->rLocation.s32Latitude,
                        _prNextCfg->rLocation.s32Latitude,
                        _prCurrentCfg->rLocation.s32Longitude,
                        _prNextCfg->rLocation.s32Longitude
                        ));
        bLocationChanged=TRUE;
        if (_prAdapter->enGetLocationCfgMode() == fc_sxm_enCfgMode_Modify) {
            rRes.bModifyLocation=TRUE;
        }
        else if (_prAdapter->enGetLocationCfgMode() == fc_sxm_enCfgMode_Const) {
            rRes.bRecreate=TRUE;
        }
    }

    // sorting  is specific to derived class and has to be checked there (virtual)
    if (bIsSortChanged()) {
        rRes.bRecreate=TRUE;
    }


    // filter  is specific to derived class and has to be checked there (virtual)
    rRes.bModifyFilter=bIsFilterChanged();

    ETG_TRACE_USR3_DCL((_rConstCfg.u16TraceClass, "rCheckNextConfig bIsSortChanged=%d bLocationChanged=%d bRadiusChanged=%d bModifyCapacity=%d bModifyFilter=%d bModifyLocation=%d bRecreate=%d END", 
                    bIsSortChanged(), bLocationChanged, bRadiusChanged, 
                    rRes.bModifyCapacity, rRes.bModifyFilter, rRes.bModifyLocation, rRes.bRecreate));

    return rRes;
}

tBool fc_sxm_tclBaseDSRL::bApplyNextConfig() {
    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prAdapter, FALSE);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prNextCfg, FALSE);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL !=_prCurrentCfg, FALSE);

    _prAdapter->vPatchLocationCfg(_prNextCfg->rLocation);
    tBool bSmsDsrlInvalid=(hGetSMSDSRL() == DSRL_INVALID_OBJECT);
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bApplyNextConfig, _enDsrlState=%d bSmsDsrlInvalid=%d smsServiceState=%d",
                    ETG_CENUM(DSRL_STATE_ENUM, _enDsrlState),
                    bSmsDsrlInvalid,
                    ETG_CENUM(DATASERVICE_STATE_ENUM, enGetSmsServiceState())));
    trDSRLCheckResult rCfgState=rCheckNextConfig();

    tBool bRes=TRUE;

    if (_bSmsServiceStopping == TRUE) {
		_bSmsServiceStopping = FALSE;
		_prAdapter->vDestroy();
		vCopyConfig();
		bRes= TRUE;

        ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL:: service stopping, DSRL destroyed"));
    }
    else if (enGetSmsServiceState()!=DATASERVICE_STATE_READY) {
        _prAdapter->vDestroy();
        vCopyConfig();
        bRes= TRUE;

        ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL:: service not ready, DSRL destroyed"));
    }
    else if ((_prAdapter->enGetLocationCfgMode()!=fc_sxm_enCfgMode_Ignore) && (!_prNextCfg->rLocation.bIsValid())) {
        vCopyConfig();
        bRes= TRUE;
    }
    else if (bSmsDsrlInvalid || rCfgState.bRecreate) {
        _prAdapter->vDestroy();
        vCopyConfig();
        bRes=_prAdapter->bCreate(*_prCurrentCfg);
    } 
    else {
        vCopyConfig();
        /*
          Tell Data-app that our dsrl will soon go to state updating
        */
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
        if (rCfgState.bNeedsUpdate() && (_enDsrlState == DSRL_STATE_READY || _enDsrlState == DSRL_STATE_UPDATING)) {
            ETG_TRACE_USR2_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bApplyNextConfig: --> fc_sxm_enDsrlState_WaitUpdating"));
            vNotifyDsrlState(fc_sxm_enDsrlState_WaitUpdating);
        }
#endif
        if (rCfgState.bModifyCapacity) {
            bRes=bRes && _prAdapter->bModifyCapacity(_prNextCfg->u32Capacity);
        }
        if (rCfgState.bModifyLocation) {
            bRes=bRes && _prAdapter->bModifyLocation(_prNextCfg->rLocation, _prNextCfg->u32Radius);
        }
        if (rCfgState.bModifyFilter) {
            bRes=bRes && _prAdapter->bModifyFilter();
        }
      
    }
    return bRes;
}


tVoid fc_sxm_tclBaseDSRL::vPrintReport() const {
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\tID=%d", _rConstCfg.u32DsrlId));

    SXM_ASSERT_RETURN(OSAL_NULL !=_rConstCfg.poDataApp);

    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tAPP        =%s", _rConstCfg.poDataApp->szGetName()));
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tState      =%d", ETG_CENUM(DSRL_STATE_ENUM, _enDsrlState)));
    ETG_TRACE_ERR_CLS((  TR_CLASS_FC_SXM_REPORT, "\t\t\t\tpoDsrl     =0x%08x", this));   
    _rConstCfg.vPrint();
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tCFG:"));
    vPrintCfg();
    ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "\t\t\t\tSTATISTICS:"));
    vPrintStatistics();
}

tVoid fc_sxm_tclBaseDSRL::vTraceNewConfig(fc_sxm_trDSRLBaseCfg const &rNewConfig) const {
    if (etg_bIsTraceActive(TR_CLASS_FC_SXM_BASE_DSRL, (tU16)TR_LEVEL_USER_4)) {
        // todo: add resonable traces
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "vTraceNewConfig of DSRL:"));

        SXM_ASSERT_RETURN(OSAL_NULL !=_rConstCfg.poDataApp);

        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "APP=%s", _rConstCfg.poDataApp->szGetName()));

        _rConstCfg.vPrint();
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "OLD Dsrl-Cfg:"));

        if(OSAL_NULL != _prCurrentCfg)
        {
            _prCurrentCfg->vPrint();
        }
        ETG_TRACE_FATAL_CLS((TR_CLASS_FC_SXM_REPORT, "NEW Dsrl-Cfg:"));
        rNewConfig.vPrint();
        
    }
}

tVoid fc_sxm_tclBaseDSRL::vNotifyDsrlState(fc_sxm_tenDsrlState enDsrlState) {
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
    if (_enDsrlStateForApp != enDsrlState) {
        _enDsrlStateForApp=enDsrlState;
        _rConstCfg.poDataApp->vOnDsrlState(this, _enDsrlStateForApp);
    }
#else
_rConstCfg.poDataApp->vOnDsrlState(this, enDsrlState,bGetDataAvailableFlag());
#endif
}
fc_sxm_tclDsrlTypeAdapter::fc_sxm_tclDsrlTypeAdapter(fc_sxm_tclBaseDSRL *prBaseDsrl):
    _hSmsDSRL(DSRL_INVALID_OBJECT),
    _prBaseDsrl(prBaseDsrl),
    _rConstCfg(_prBaseDsrl->rGetConstCfg()),
    _pfDSRLEvent(cb_vDSRLEventCallback)
{}

tVoid fc_sxm_tclDsrlTypeAdapter::vDestroySMSDSRL(DSRL_OBJECT *phSmsDSRL) const {
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vDestroyDSRL?"));
    if (*phSmsDSRL != DSRL_INVALID_OBJECT) {
        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vDestroyDSRL!"));

        DSRL.vDestroy(*phSmsDSRL);
        *phSmsDSRL = DSRL_INVALID_OBJECT;

        ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "DSRL destroyed"));
    }

}

DISTANCE_OBJECT fc_sxm_tclDsrlTypeAdapter::hCreateSMSRadius(tU32 u32Radius) {
    DISTANCE_OBJECT hSmsRadius;
    // Create the DISTANCE object
    // peha: must distance object be destroyed ???
    // or is it destroyed in destructur of location/dslr
    hSmsRadius = DISTANCE.hCreate(u32Radius, DISTANCE_UNIT_TYPE_MILES);
    if (hSmsRadius == DISTANCE_INVALID_OBJECT) {
        ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL::hCreateSMSRadius:u32Radius=%d  FAILED",
                    u32Radius));
    }
    SXM_ASSERT_RETURN_VAL(hSmsRadius != DISTANCE_INVALID_OBJECT, FALSE);
    return hSmsRadius;
}

tVoid fc_sxm_tclDsrlTypeAdapter::vDestroySMSRadius(DISTANCE_OBJECT *phSmsRadius) {
    if (*phSmsRadius != DISTANCE_INVALID_OBJECT) {
        DISTANCE.vDestroy (*phSmsRadius);
        *phSmsRadius=DISTANCE_INVALID_OBJECT;
    }
}





LOCATION_OBJECT fc_sxm_tclDsrlTypeAdapter::hCreateSMSLocation(tS32 s32Lat, tS32 s32Long, DISTANCE_OBJECT hSmsRadius) {

    ETG_TRACE_USR4(("fc_sxm_tclBaseDSRL::hCreateSMSLocation:u32Lat=%d  u32Long=%d",
                    s32Lat, s32Long));
    //Check if the DISTANCE object for radius is created

    //Create fixed objects for latitude and longitde
    OSAL_FIXED_OBJECT hLat = OSAL_FIXED.hCreateFromFixed(s32Lat, 16);
    OSAL_FIXED_OBJECT hLon = OSAL_FIXED.hCreateFromFixed(s32Long, 16);

    //Create the LOCATION object
    LOCATION_OBJECT hNewSmsLocation = LOCATION.hCreateForRadius(hLat, hLon, hSmsRadius);
   
    // todo: check if custody goes to dsrl
        OSAL_FIXED.vDestroy(hLat);
        OSAL_FIXED.vDestroy(hLon);

    //SXM_ASSERT_RETURN_VAL(hNewSmsLocation != LOCATION_INVALID_OBJECT, hNewSmsLocation);
    return hNewSmsLocation;
}

tVoid fc_sxm_tclDsrlTypeAdapter::vDestroySMSLocation(LOCATION_OBJECT *phSmsLocation) {

    if (*phSmsLocation != LOCATION_INVALID_OBJECT) {
        //Destroy the LOCATION object
        LOCATION.vDestroy(*phSmsLocation);
        *phSmsLocation = LOCATION_INVALID_OBJECT;
    }
}






tBool fc_sxm_tclDsrlTypeAdapter::bModifyFilter() {

    SXM_ASSERT_RETURN_VAL(_hSmsDSRL != DSRL_INVALID_OBJECT, FALSE);
    SMSAPI_RETURN_CODE_ENUM eReturnCode = DSRL.eModifyFilter(_hSmsDSRL,
                                                             cb_bDSRLFilter, 
                                                             (tVoid *)_prBaseDsrl);
    
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        ETG_TRACE_ERR_DCL((_rConstCfg.u16TraceClass, "DSRL.eModifyFilter failed, status = %d", eReturnCode));
        return FALSE;
    }
    return TRUE;
}


tBool fc_sxm_tclDsrlTypeAdapter::bModifyCapacity(tU32 u32Capacity) {

    SXM_ASSERT_RETURN_VAL(_hSmsDSRL != DSRL_INVALID_OBJECT, FALSE);
    SMSAPI_RETURN_CODE_ENUM eReturnCode = DSRL.eModifyCapacity(_hSmsDSRL, (size_t)u32Capacity);
    if(eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
    {
        ETG_TRACE_ERR(("DSRL.eModifyCapacity failed, status = %d", eReturnCode));
        return FALSE;
    } 
    return TRUE;
}


tBool fc_sxm_tclDsrlTypeStandard::bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg) {
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateStandardDSRL lat=%d long=%d radius=%u",
                    _rBaseCfg.rLocation.s32Latitude,
                    _rBaseCfg.rLocation.s32Longitude,
                    _rBaseCfg.u32Radius));
    //Check if the dsrl is already created
    SXM_ASSERT_RETURN_VAL(_hSmsDSRL == DSRL_INVALID_OBJECT, TRUE);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _rConstCfg.poDataApp, FALSE);


    SXM_ASSERT_RETURN_VAL( DATASERVICE_STATE_READY == _rConstCfg.poDataApp->enGetSmsServiceState(), FALSE);
    _hSmsRadius=hCreateSMSRadius(_rBaseCfg.u32Radius);
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateStandardDSRL _hSmsRadius=%d", (_hSmsRadius==DISTANCE_INVALID_OBJECT) ));
    SXM_ASSERT_RETURN_VAL(_hSmsRadius != DISTANCE_INVALID_OBJECT, FALSE);
    _hSmsLocation=hCreateSMSLocation(_rBaseCfg.rLocation.s32Latitude, _rBaseCfg.rLocation.s32Longitude, _hSmsRadius);
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateStandardDSRL _hSmsLocation=%d", (_hSmsLocation==LOCATION_INVALID_OBJECT) ));
        
	if (_hSmsLocation != LOCATION_INVALID_OBJECT) {

		ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateStandardDSRL"));

		//Create the DSRL
		_hSmsDSRL = DSRL.hCreate(
								_rConstCfg.poDataApp->hGetSmsService(), (DSRL_TARGET_OBJECT*)&_hSmsLocation, 
								1, (size_t)_rBaseCfg.u32Capacity, 
								_prBaseDsrl->bUseSMSDefaultSort() ? OSAL_NULL : cb_n16DSRLSort, (tVoid *)_prBaseDsrl, 
								cb_bDSRLFilter, (tVoid *)_prBaseDsrl, 
								_pfDSRLEvent, (tVoid *)_prBaseDsrl);

		ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateStandardDSRL _hSmsDSRL=%d", (_hSmsDSRL==DSRL_INVALID_OBJECT) ));

	    SXM_ASSERT_RETURN_VAL(_hSmsDSRL != DSRL_INVALID_OBJECT, FALSE);
	}
      
    return TRUE;
}

tBool fc_sxm_tclDsrlTypeStandard::bModifyLocation(fc_sxm_trDsrlLocation const &rNewLocation, tU32 u32Radius) {
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclDsrlTypeStandard::bModifyLocation u32Radius=%d s32Latitude=%d  s32Longitude=%d",
                    u32Radius, rNewLocation.s32Latitude, rNewLocation.s32Longitude));

    DISTANCE_OBJECT hNewRadius= hCreateSMSRadius(u32Radius);

    //Check if the DISTANCE object for radius is created
    SXM_ASSERT_RETURN_VAL(hNewRadius != DISTANCE_INVALID_OBJECT, FALSE);

    // todo: check if custody goes to dsrl
    vDestroySMSRadius(&_hSmsRadius);
    _hSmsRadius=hNewRadius;

    LOCATION_OBJECT hNewLocation=hCreateSMSLocation(rNewLocation.s32Latitude,
                                                    rNewLocation.s32Longitude,
                                                    hNewRadius);
    	
	if (hNewLocation != LOCATION_INVALID_OBJECT) {

		//Modify the target location
		SMSAPI_RETURN_CODE_ENUM eReturnCode = DSRL.eModifyTargets(_hSmsDSRL, 
																DSRL_MODIFY_OPERATION_REPLACE,
																(DSRL_TARGET_OBJECT *)&hNewLocation,
																1);

	    SXM_ASSERT_RETURN_VAL((eReturnCode == SMSAPI_RETURN_CODE_SUCCESS ||
	    		eReturnCode == SMSAPI_RETURN_CODE_API_NOT_ALLOWED), FALSE);

		vDestroySMSLocation(&_hSmsLocation);
	    _hSmsLocation=hNewLocation;
	}

    return TRUE;
}



tBool fc_sxm_tclDsrlTypeDevice::bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg) {
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateDeviceDSRL"));
    //Check if the dsrl is already created
    SXM_ASSERT_RETURN_VAL(_hSmsDSRL == DSRL_INVALID_OBJECT, TRUE);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _rConstCfg.poDataApp, FALSE);
    //SXM_ASSERT_RETURN_VAL( DATASERVICE_STATE_READY == _rConstCfg.poDataApp->enGetSmsServiceState(), FALSE);
    SXM_ASSERT_RETURN_VAL( DATASERVICE_MGR_INVALID_OBJECT != _rConstCfg.poDataApp->hGetSmsService(), FALSE);
    _hSmsRadius=hCreateSMSRadius(_rBaseCfg.u32Radius);
    SXM_ASSERT_RETURN_VAL(_hSmsRadius != DISTANCE_INVALID_OBJECT, FALSE);


    //Create the DSRL

    _hSmsDSRL = DSRL.hCreateForDeviceLocation(
                                              _rConstCfg.poDataApp->hGetSmsService(), _hSmsRadius, _rBaseCfg.u32Capacity, 
                                              _prBaseDsrl->bUseSMSDefaultSort() ? OSAL_NULL : cb_n16DSRLSort, (tVoid *)_prBaseDsrl, 
                                               cb_bDSRLFilter, (tVoid *)_prBaseDsrl, 
                                              _pfDSRLEvent, (tVoid *)_prBaseDsrl);

    SXM_ASSERT_RETURN_VAL(_hSmsDSRL != DSRL_INVALID_OBJECT, FALSE);
    return TRUE;

}



tBool fc_sxm_tclDsrlTypeFavorites::bCreate(fc_sxm_trDSRLBaseCfg const &_rBaseCfg) {
    (tVoid)_rBaseCfg;
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::bCreateFavoritesDSRL"));
    //Check if the dsrl is already created
    SXM_ASSERT_RETURN_VAL(_hSmsDSRL == DSRL_INVALID_OBJECT, TRUE);
    SXM_ASSERT_RETURN_VAL(OSAL_NULL != _rConstCfg.poDataApp, FALSE);

    //SXM_ASSERT_RETURN_VAL( DATASERVICE_STATE_READY == _rConstCfg.poDataApp->enGetSmsServiceState(), FALSE);
    SXM_ASSERT_RETURN_VAL( DATASERVICE_MGR_INVALID_OBJECT != _rConstCfg.poDataApp->hGetSmsService(), FALSE);

    //Create the DSRL
    _hSmsDSRL = DSRL.hCreateFavorites(_rConstCfg.poDataApp->hGetSmsService(), 
                                       _prBaseDsrl->bUseSMSDefaultSort() ? OSAL_NULL : cb_n16DSRLSort, (tVoid *)_prBaseDsrl, 
                                      _pfDSRLEvent, (tVoid *)_prBaseDsrl);
	    
    //Check if the dsrl is created now
    SXM_ASSERT_RETURN_VAL(_hSmsDSRL != DSRL_INVALID_OBJECT, FALSE);
    return TRUE;

}

tVoid fc_sxm_tclBaseDSRL::vForceUpdate() {
    vOnSmsDsrlState(enGetSmsDsrlState());
}

/*********************************************************************
 *
 *FUNCTION:     vProcessDSRLState
 *
 *DESCRIPTION:  Process the DSRL state
 *             
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ********************************************************************/
tVoid fc_sxm_tclBaseDSRL::vOnSmsDsrlState(DSRL_STATE_ENUM enState)
{
    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vProcessDSRLState poDsrl=0x%08x %10s enDsrlType=%d, State = %d", 
                    this,
                    _rConstCfg.poDataApp->szGetName(),
                    ETG_CENUM(fc_sxm_tenDSRLType, _rConstCfg.enDsrlType),
                    ETG_CENUM(DSRL_STATE_ENUM, enState)
                    ));
    SXM_ASSERT_RETURN(OSAL_NULL != _rConstCfg.poDataApp);
    _enDsrlState=enState;
    // inform app that we will iterate this dsrl now

    rStatistics.rMSecLastEvt=OSAL_ClockGetElapsedTime();
    fc_sxm_tclDataListBase* poActivaeList = _rConstCfg.poDataApp->poGetActiveListbyDSRL(this);

#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
    fc_sxm_tenDsrlState enDsrlState=_enDsrlStateForApp;
#else
    fc_sxm_tenDsrlState enDsrlState=fc_sxm_enDsrlState_Error;
#endif
    //Process the state transition
    switch(enState)
    {
        case DSRL_STATE_INITIAL:
            enDsrlState=fc_sxm_enDsrlState_Initial;
            vResetDsrlReadyCount();
            break;

        case DSRL_STATE_UPDATING:
            rStatistics.u32NumFiltered=0;
            rStatistics.u32NumSorted=0;
            rStatistics.rMSecLastUpdateStart=OSAL_ClockGetElapsedTime();
            enDsrlState=fc_sxm_enDsrlState_Updating;
            break;

        case DSRL_STATE_READY:
        {
            if  (rCheckNextConfig().bNeedsUpdate()) {
                bApplyNextConfig();
                break;
            }
#ifdef VARIANT_S_FTR_ENABLE_SXM_NEW_LIST_HANDLING
            if (fc_sxm_enDsrlState_WaitUpdating == _enDsrlStateForApp) {
                // we wait for an dsrl-update
                break;
            }
#endif
            /* Increment the Dsrl Ready count */
            vIncrementDsrlReadyCount();

            rStatistics.vResetIterate();
            rStatistics.rMSecLastReady=OSAL_ClockGetElapsedTime();

            ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL DSRL.eIterate  _hSmsDSRL=0x%08x START", hGetSMSDSRL()));

            /* Reset Changed Flag - indicates Data is not changed by default*/
            vSetChangedFlag(FALSE);            

            /* Reset Data Available Flag - indicated that Iterate Callback is not triggered*/
            vSetDataAvailableFlag(FALSE);

            SMSAPI_RETURN_CODE_ENUM eReturnCode = DSRL.eIterate(hGetSMSDSRL(), cb_bDSRLIterate, (tVoid *)this);
            ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL DSRL.eIterate   END"));
            rStatistics.rMSecLastIterateEnd=OSAL_ClockGetElapsedTime();
            if (eReturnCode != SMSAPI_RETURN_CODE_SUCCESS)
            {
                ETG_TRACE_ERR_DCL((_rConstCfg.u16TraceClass, "DSRL.eIterate failed, status = %d", 
                               ETG_CENUM(SMSAPI_RETURN_CODE_ENUM, eReturnCode)));
                return;
            }
            
            // Check broadcast updates
            enDsrlState =  bGetChangedFlag() ? fc_sxm_enDsrlState_Changed : fc_sxm_enDsrlState_Ready;

            // For Fuel Service,we should proceed from 2nd callback only
            if( ((poGetDataApp()->enGetServiceId() == fc_sxm_enServiceID_Fuel) || (poGetDataApp()->enGetServiceId() == fc_sxm_enServiceID_Canadian_Fuel)) &&
                (enGetDSRLType() != fc_sxm_enDSRLType_Favorites) &&
                (u32GetNumDsrlReadyCount() < 2) &&
                (poActivaeList->bIsListDataValid() == FALSE) ) {
                    ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "u32GetNumDsrlReadyCount = %d",
                       u32GetNumDsrlReadyCount()));
                    return;
            }

            //Notify DSRL update
            vOnDSRLUpdate();
        }
        break;

        case DSRL_STATE_STOPPED:
            enDsrlState=fc_sxm_enDsrlState_Stopped;
            break;

        case DSRL_STATE_ERROR:
            enDsrlState=fc_sxm_enDsrlState_Error;
            break;

        case DSRL_STATE_UNKNOWN:
            enDsrlState=fc_sxm_enDsrlState_Error;
            break;

        default:
            enDsrlState=fc_sxm_enDsrlState_Error;
            ETG_TRACE_USR4_DCL((_rConstCfg.u16TraceClass, "fc_sxm_tclBaseDSRL::vProcessDSRLState, Invalid state! (%d)", enState));
            break;
    }

    // inform app that we will iterate this dsrl now
    vNotifyDsrlState(enDsrlState);

}




tVoid fc_sxm_tclDsrlTypeNearDest::vPatchLocationCfg(fc_sxm_trDsrlLocation &rLocation) {
    rLocation=_prBaseDsrl->poGetDataApp()->rGetDestLocation();
}
