/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_base_app.h
* @brief       Declaration for BaseApp which is used as base application handler and
*              used by all Sxm applications. It has worker thread. It provides
*              functionality to post message to the in-queue and to dispatch the messages
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef _FC_SXM_TCL_BASE_APP_H_
#define _FC_SXM_TCL_BASE_APP_H_


#include "fc_sxm_base_thread.h"
#include "fc_sxm_tcl_timer.h"
#include "fc_sxm_diag_types.h"


// Messages from apps
struct fc_sxm_trMsgCmdAppTtfisCmd: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_All, 1);
    fc_sxm_tenServiceID enServiceId;
    tU8 u8MsgCode;
    tU8 u8DataLen;
    tU8 au8Data[0xFF];
};

/* message to be sent to all apps
   default-implementation will be proviced by base-app
*/
struct fc_sxm_trMsgCmdAppClearUserData: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_All, 111);
    fc_sxm_trMsgCmdAppClearUserData() {};
};

struct fc_sxm_trMsgCmdAppPrintReport: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_All, 112);
    fc_sxm_trMsgCmdAppPrintReport() {};
};

struct fc_sxm_trMsgAppRestoreSmsDb: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_All, 113);
    fc_sxm_trMsgAppRestoreSmsDb(fc_sxm_tenServiceID enServiceId_=fc_sxm_enServiceID_All):
        enServiceId(enServiceId_)
    {};

    fc_sxm_tenServiceID enServiceId;
};


// generic implemantation of action timeout
struct fc_sxm_trMsgAppActionTimer: public fc_sxm_tclMessage {
    SXM_MAKE_ACTION(fc_sxm_enServiceID_All, 115);
};

class fc_sxm_tclTimerBase;
class fc_sxm_tclBaseApp
{
    friend class fc_sxm_tclBaseThread;
    friend class fc_sxm_tclTimerBase;
    template<class APP, class M, class C>
        friend class fc_sxm_tclTimer;
public:



   //Constructor (CCA-Context)
    fc_sxm_tclBaseApp(tCString szAppName, fc_sxm_tenServiceID enServiceId, tU16 u16CcaServiceId, tU16 u16TraceClass=TR_CLASS_FC_SXM_BASE_APP);

   //Destructor (CCA-Context)
   virtual ~fc_sxm_tclBaseApp();

   /*Initialize the application (CCA-Context)
     this method may be overwritten by derived class,
     but as LAST action the overwriting method has to call this method.
   */     
   virtual tVoid vInitialize(tVoid);

   /*Deinitialize the application (CCA-Context)
     this method may be overwritten by derived class,
     but as FIRST action the overwriting method has to call this method.
   */
   virtual tVoid vDeInitialize(tVoid);



   // get the sxm-service-id that identifies the app
   fc_sxm_tenServiceID enGetServiceId() const {
       return _enServiceId;
   }

   tU16 u16GetCcaServiceId() {
       return _u16CcaServiceId;
   }

  virtual tU16 u16GetTraceClass() const {
       return _u16TraceClass;
   }
   /*
     determines from u32Action of a received message, if the message is for us.
   */
   virtual tBool bWantsMsg(tU32 u32Action) const{
       fc_sxm_tenServiceID enServiceId=fc_sxm_enGetServiceId(u32Action);
       if (enServiceId==_enServiceId ||  // message specific for our app
           enServiceId==fc_sxm_enServiceID_All) // message has to be handled by all Apps
       {
           return TRUE;
       }
       return FALSE;
   }

   /*
     Shortcut for bWantsMsg(poThreadMsg->u32GetAction());
   */
   virtual tBool bWantsMsg(fc_sxm_tclMessage const *poThreadMsg) const {
       return bWantsMsg(poThreadMsg->u32GetAction());
   }

   /* 
      Allocate and post message to me (any context)
   */
   template<class M>
       tVoid vPostMsgNew(M const &rMsg) {
       _oWorkerThread.vPostMsgNew(rMsg);
   }
   
   /* 
      Post already allocated message "pMsg" to me.
      (any context)
   */
   tVoid vPostMsg(fc_sxm_tclMessage *prMsg) {
       _oWorkerThread.vPostMsg(prMsg);
   }

   tChar const *szGetName() const {
       return _oWorkerThread.szGetName();
   }
   
   virtual tVoid vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg);


   /* called to clear user-data, default implementation:
      vSetUsrDataClearResult(fc_sxm_enDiagResult_NoAction)
      if app has to to something, vSetUsrDataClearResult has to be called
      with appropriate code (fc_sxm_enDiagResult_Ok or fc_sxm_enDiagResult_Failed)
      after action is done, e.g. 
      audio: set default channel, delete presets,
      data: clear favorites
   */
   virtual tVoid vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const *prMsg);

   virtual tVoid vProcessBaseAppMsg(fc_sxm_trMsgCmdAppPrintReport const *prMsg);

   virtual tVoid vProcessBaseAppMsg(fc_sxm_trMsgAppRestoreSmsDb const *prMsg) {
	   (tVoid)prMsg;
   }

   virtual tVoid vPrintReport() { }

 protected:
   /*
     Dispatch thread-message "poThreadMsg" from worker-thread
     May not be overwritten by sxm-app.
     (workerThread context)
   */
   virtual tVoid vDispatchMsgFromQInternal(fc_sxm_tclMessage const *poThreadMsg);

   /*
     
     method to be implemented by app to dispatch messages specific to the app:
     Will be call for received messages with service-id == _enServiceId
     (workerThread context)
   */
   virtual tVoid vDispatchMsgFromQ(fc_sxm_tclMessage const *poThreadMsg) = 0;

   /*
     Interface to be called by app to indicate that a cca-service with id "u16CcaServiceId"
     may be started (bAllow=TRUE) or not.
     note: "u16CcaServiceId" != _enServiceId
     (Any context)
   */
   tVoid vAllowService(tBool bAllow, tU16 u16CcaServiceId=0) const;

   tVoid vSetUsrDataClearResult(fc_sxm_tenDiagResult enRes);

    /* request can be requested here from sms-context.
       access can only be granted when we currently don't handle a message,
       otherwise we might end in a dead-lock.
    */
   tBool bAccess() {
       return _oWorkerThread.bAccess(OSAL_C_TIMEOUT_NOBLOCKING);
   }
    tVoid vRelease() {
        _oWorkerThread.vRelease();    
    }

 private:
   // iterface for timer-handling
   tVoid vRegisterTimer(fc_sxm_tclTimerBase *poTimer);
   tVoid vUnregisterTimer(fc_sxm_tclTimerBase *poTimer);
   // interface called from worker-thread when a timer-event was received
   tVoid vHandleTimerEvent();
    list<fc_sxm_tclTimerBase *> _oTimerList;
   // interface to be used in timer callback context.
   tVoid vPostTimerEvent() {
       _oWorkerThread.vPostTimerEvent();
   }






    // action we have requested from another app or app-manager
    fc_sxm_tenServiceID _enServiceId;
    tU16 _u16CcaServiceId;
 protected:
    const tU16 _u16TraceClass;
 private:
    fc_sxm_tclBaseThread _oWorkerThread;

};




/* 
   macro to be used in vDispatchMsgFromQ() for each action-id specific to the app.
   Call this->vProcess(TYPE const *) for action TYPE::u32Action.
   note: This is the reason that all thread-messages have to implement to member "u32Action"vProcessGenericMsg
*/
#define SXM_MSGQ_DISPATCH(TYPE)                                         \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
           /* vProcess(poTypedMsg);   */                                    \
           vProcessGenericMsg(poTypedMsg);                                   \
        }                                                               \
    }                                                                   \
    break

/* 
   macro to be used in vDispatchMsgFromQ() for each action-id specific to the app.
   Call this->vProcess(TYPE const *) for action TYPE::u32Action.
   note: This is the reason that all thread-messages have to implement to member "u32Action"vProcessGenericMsg
*/
#define SXM_BASEAPP_MSGQ_DISPATCH(TYPE)                                         \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
           vProcessBaseAppMsg(poTypedMsg);                                   \
        }                                                               \
    }                                                                   \
    break
/*
   Like SXM_MSGQ_DISPATCH, but "vProcess()" of another object "OBJ" is called.
   Call OBJ->vProcess(TYPE const *) for action TYPE::u32Action.
*/
#define SXM_MSGQ_DISPATCH_TO_OBJ(TYPE, OBJ)                             \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
            OBJ->vProcess(poTypedMsg);                                  \
        }                                                               \
    }                                                                   \
    break

#define SXM_MSGQ_DISPATCH_TO_TIMEROBJ(TYPE, OBJ)                             \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
            OBJ->vProcessTimer(poTypedMsg);                                  \
        }                                                               \
    }                                                                   \
    break
#define SXM_MSGQ_DISPATCH_TO_2OBJ(TYPE, OBJ1, OBJ2)                     \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
            OBJ1->vProcess(poTypedMsg);                                 \
            OBJ2->vProcess(poTypedMsg);                                 \
        }                                                               \
    }                                                                   \
    break


#define SXM_MSGQ_DISPATCH_TO_3OBJ(TYPE, OBJ1, OBJ2, OBJ3)               \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
            OBJ1->vProcess(poTypedMsg);                                 \
            OBJ2->vProcess(poTypedMsg);                                 \
            OBJ3->vProcess(poTypedMsg);                                 \
        }                                                               \
    }                                                                   \
    break

#define SXM_MSGQ_DISPATCH_TO_4OBJ(TYPE, OBJ1, OBJ2, OBJ3, OBJ4)         \
    case TYPE::u32Action:                                               \
    {                                                                   \
        TYPE const *poTypedMsg= dynamic_cast<TYPE const*>(poThreadMsg); \
        if (OSAL_NULL != poTypedMsg) {                                  \
            OBJ1->vProcess(poTypedMsg);                                 \
            OBJ2->vProcess(poTypedMsg);                                 \
            OBJ3->vProcess(poTypedMsg);                                 \
            OBJ4->vProcess(poTypedMsg);                                 \
        }                                                               \
    }                                                                   \
    break

#endif //_FC_SXM_TCL_BASE_APP_H_
