/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_tcl_base_app.cpp
* @brief       Base Application implementation
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_main.h"
#include "fc_sxm_tcl_base_app.h"
#include "fc_sxm_tcl_timer.h"
#include "fc_sxm_diag_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_BASE_APP
#include "trcGenProj/Header/fc_sxm_tcl_base_app.cpp.trc.h"
#endif



/*
  Constructor for all sxm-apps.
  Create the worker-thread that will receive all messages for the app.
  szAppName: name of the app, used as thread-name for the worker-thread.
  enServiceId: id of the sxm-app
*/
fc_sxm_tclBaseApp::fc_sxm_tclBaseApp(tCString szAppName, fc_sxm_tenServiceID enServiceId, tU16 u16CcaServiceId, tU16 u16TraceClass):
    _enServiceId(enServiceId),
    _u16CcaServiceId(u16CcaServiceId),
    _u16TraceClass(u16TraceClass),
    _oWorkerThread(const_cast<fc_sxm_tclBaseApp *>(this), szAppName)
{
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp CTOR for %s", szGetName()));
}

/*
  Destructor
*/
fc_sxm_tclBaseApp::~fc_sxm_tclBaseApp()
{
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp DTOR for %s", szGetName()));
    _oTimerList.clear();
}

/*
  Start worker thread. 
  See header!!!
*/
tVoid fc_sxm_tclBaseApp::vInitialize() {
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp::vInitialize for %s", szGetName()));

    //Start the worker thread
    _oWorkerThread.vStart();    
}

/*
  Stop worker thread. 
  See header!!!
*/
tVoid fc_sxm_tclBaseApp::vDeInitialize() {
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp::vDeInitialize for %s", szGetName()));
    //Stop the worker thread
    _oWorkerThread.vStop();
}

/*
  Inform sxm-main-class that a cca-Service is allowed to be available or not
*/
tVoid fc_sxm_tclBaseApp::vAllowService(tBool bAllow, tU16 u16CcaServiceId) const
{
    if (!u16CcaServiceId) {
        u16CcaServiceId=_u16CcaServiceId;
    }
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp::vAllowService(%d) CCA-SID=%d for %s", bAllow, u16CcaServiceId, szGetName()));
    if (u16CcaServiceId) {
        fc_sxm_trMsgCmdMainAllowService rMsg;
        rMsg.bAllow=bAllow;
        rMsg.u16CcaServiceId=u16CcaServiceId;
        fc_sxm_tclApp::instance()->vPostMsgNew(rMsg);
    }
}

tVoid fc_sxm_tclBaseApp::vSetUsrDataClearResult(fc_sxm_tenDiagResult enRes) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp(%10s)::vSetUsrDataClearResult: enRes=%d",
                    szGetName(),
                    ETG_CENUM(fc_sxm_tenDiagResult, enRes)));
    // following method is thread-save
    fc_sxm_tclDiagUsrDataClear::instance()->vSetResult(_enServiceId, enRes);
}

tVoid fc_sxm_tclBaseApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppTtfisCmd const *prMsg) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp::vProcess(fc_sxm_trMsgCmdAppTtfisCmd): code=%d not implemented for %s",
                    prMsg->u8MsgCode, szGetName()));
}


tVoid fc_sxm_tclBaseApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppClearUserData const * /* prMsg */)
{
     ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp::vProcess(fc_sxm_trMsgCmdAppClearUserData(%s)",
                     szGetName()));
    vSetUsrDataClearResult(fc_sxm_enDiagResult_NoAction);
}
/*
  generic dispatch-method for all thread-messages. 
  see header!!!
*/
tVoid fc_sxm_tclBaseApp::vDispatchMsgFromQInternal(fc_sxm_tclMessage const *poThreadMsg) {
    
    /* Check for NULL */
    if(OSAL_NULL == poThreadMsg) {
       ETG_TRACE_ERR(("fc_sxm_tclBaseApp::vDispatchMsgFromQInternal message is NULL"));
       return;
    }

    tU32 u32Action=poThreadMsg->u32GetAction();
    ETG_TRACE_USR3_DCL((_u16TraceClass, "fc_sxm_tclBaseApp::vDispatchMsgFromQInternal: SID=%x action=%d bWantsMsg=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, poThreadMsg->enGetServiceId()), 
                    poThreadMsg->u16GetActionOnly(),
                    bWantsMsg(u32Action)));

    if (!bWantsMsg(u32Action)) {
        /* message is not for us */
    } else if (poThreadMsg->enGetServiceId()==fc_sxm_enServiceID_All) {
        switch(u32Action) {

        	SXM_BASEAPP_MSGQ_DISPATCH(fc_sxm_trMsgCmdAppTtfisCmd);
        	SXM_BASEAPP_MSGQ_DISPATCH(fc_sxm_trMsgCmdAppPrintReport);
        	SXM_BASEAPP_MSGQ_DISPATCH(fc_sxm_trMsgAppRestoreSmsDb);
            SXM_BASEAPP_MSGQ_DISPATCH(fc_sxm_trMsgCmdAppClearUserData);
#if 0
            SXM_MSGQ_DISPATCH(fc_sxm_trMsgCmdAppPrintReport);
            SXM_MSGQ_DISPATCH(fc_sxm_trMsgAppRestoreSmsDb);
            SXM_MSGQ_DISPATCH(fc_sxm_trMsgCmdAppClearUserData);



        SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdAppTtfisCmd,);
#endif


            
            default:
                ETG_TRACE_USR4(("fc_sxm_tclBaseApp(%s)::vDispatchMsgFromQInternal: calling vDispatchMsgFromQ on fc_sxm_enServiceID_All", szGetName()));
                vDispatchMsgFromQ(poThreadMsg);
                break;
        }
    } else {
                vDispatchMsgFromQ(poThreadMsg);
    }
}
tVoid fc_sxm_tclBaseApp::vRegisterTimer(fc_sxm_tclTimerBase *poTimer) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp(%s)::vRegisterTimer", szGetName()));
    SXM_ASSERT_RETURN(OSAL_NULL!=poTimer);
    SXM_IF_FIND(list<fc_sxm_tclTimerBase *>, iter, _oTimerList, poTimer) {
    ETG_TRACE_USR1_DCL((_u16TraceClass, "fc_sxm_tclBaseApp::vRegisterTimer: already registered"));
        // do nothing, already registered
    } else {
        _oTimerList.push_back(poTimer);
    }
}

tVoid fc_sxm_tclBaseApp::vUnregisterTimer(fc_sxm_tclTimerBase *poTimer) {
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp(%s)::vUnregisterTimer", szGetName()));
    SXM_ASSERT_RETURN(OSAL_NULL!=poTimer);

    SXM_IF_FIND(list<fc_sxm_tclTimerBase *>, iter, _oTimerList, poTimer) {
        _oTimerList.erase(iter);
    } else {
        ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp::vUnregisterTimer: Not registered"));
        // do nothing, not registered
    }
}

tVoid fc_sxm_tclBaseApp::vHandleTimerEvent() {
    ETG_TRACE_USR4(("fc_sxm_tclBaseApp vHandleTimerEvent %s", szGetName()));

    SXM_FOREACH(list<fc_sxm_tclTimerBase *>, iter, _oTimerList) {
        fc_sxm_tclTimerBase *pTimer=*iter;

        SXM_ASSERT_RETURN(OSAL_NULL!=pTimer);

        if (pTimer->bIsFired()) {
            ETG_TRACE_USR4(("fc_sxm_tclBaseApp vHandleTimerEvent: calling execute of timer 0x%08x", pTimer));
            pTimer->vExecute();
        }
    }
}

tVoid fc_sxm_tclBaseApp::vProcessBaseAppMsg(fc_sxm_trMsgCmdAppPrintReport const * /* prMsg */)
{
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp fc_sxm_trMsgCmdAppPrintReport(%s) START", szGetName()));
    vPrintReport();
    fc_sxm_tclDiagReporter::instance()->vHandleAppAnswer(enGetServiceId());
    ETG_TRACE_USR4_DCL((_u16TraceClass, "fc_sxm_tclBaseApp fc_sxm_trMsgCmdAppPrintReport(%s) END", szGetName()));
}
