/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_property.h
* @brief       Declration and implmentation for Property class which is used to send
*              the notification
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include "fc_sxm_unique_name.h"
#include "fc_sxm_tcl_sem.h"
#ifndef _FC_SXM_PROPERTY_H_
#define _FC_SXM_PROPERTY_H_
extern tVoid fc_sxm_vNotifyPropertyChanged(tU16 enServiceId, tU16 u16FunktionId);

tChar * fc_sxm_szGetUniqueName(tChar *pOut);

/*
  base-class for semaphore-protected data.
*/
struct fc_sxm_tclProtectedBase {
    public:
    /* constructor: open semaphore, close of semaphore will be done automatically in its destructor */
    fc_sxm_tclProtectedBase() 
    {
        _oSem.vOpen( 1);    
    }
    /* lock the data */
    tVoid vLock() {
        _oSem.vGet();
    };
    /* unlock the data */
    tVoid vUnlock() {
        _oSem.vPost();
    };
   virtual ~fc_sxm_tclProtectedBase(){}

    private:
    fc_sxm_tclSem _oSem;
};


/* 
   template to add semaphore-protection to a data-structure of class M
*/
template<class M> 
struct fc_sxm_tclProtectedData:private fc_sxm_tclProtectedBase {
    public: 
    /* write the data, locking is handle automatically */
    tVoid vSet(M const &rMIn) {
        fc_sxm_tclProtectedBase::vLock();
        rM=rMIn;
        fc_sxm_tclProtectedBase::vUnlock();
    }
    /* read the data, locking is handle automatically */
    M rGet() {
        M rMRes;
        fc_sxm_tclProtectedBase::vLock();
        rMRes = rM;
        fc_sxm_tclProtectedBase::vUnlock();
        return rMRes;
    }
    
    /* 
       For larger data a reference can be obtained
       The data will be locked
       note: when reference is no longer needed it has to be release using vRelease()!!!
    */
    //lint -save -e1536
   M  &rAccess() {
        fc_sxm_tclProtectedBase::vLock();
        return rM;
    }
   //lint -restore
    /*
      Release the data that have prior been locked by calling rAccess
    */
    tVoid vRelease() {
        vUnlock();
    }

    protected:
    M rM;
};

/*  
    Protected data with additional notification-mechanism
    note: Class M needs operator!=() defined.

*/
template<class M, int SRVDI, int FNID> 
    struct fc_sxm_tclProperty:public fc_sxm_tclProtectedData<M> {

        /* inform service-handler that property has changed */
        tVoid vNotify() {
                     fc_sxm_vNotifyPropertyChanged((tU16)SRVDI, (tU16)FNID);
        }

    };



/*  
    Protected data with additional auto-notification-mechanism vSetAndNotify()
    note: Class M needs operator!=() defined.

*/
template<class M, int SRVDI, int FNID>
    struct fc_sxm_tclAutoProperty:public fc_sxm_tclProperty<M, SRVDI, FNID> {
        /* if data differ, write the data and notify service-handler  */
        tVoid vSetAndNotify(M const &rMIn) {
            M &rMOld=fc_sxm_tclProtectedData<M>::rAccess();
            if (rMIn != rMOld) {
                rMOld=rMIn;
                fc_sxm_tclProtectedData<M>::vRelease();
                fc_sxm_tclProperty<M, SRVDI, FNID>::vNotify();
                return;
            }
            fc_sxm_tclProtectedData<M>::vRelease();
        }
    };

    /*  
    Protected data with additional notification-mechanism
    note: Class M needs operator!=() defined.
    Special property class with tracing enabled
*/
template<class M, int SRVDI, int FNID> 
    struct fc_sxm_tclPropertyTr:public fc_sxm_tclProtectedData<M> {

        /* inform service-handler that property has changed */
        tVoid vNotify() {
            if ((tU32)SRVDI!= 0) {
                M rM = fc_sxm_tclProtectedData<M>::rGet();
                rM.vTrace();
                fc_sxm_vNotifyPropertyChanged((tU16)SRVDI, (tU16)FNID);
            }
        }
    };

/*  
    Protected data with additional auto-notification-mechanism vSetAndNotify()
    note: Class M needs operator!=() defined.
    Special property class with tracing enabled
*/
template<class M, int SRVDI, int FNID>
    struct fc_sxm_tclAutoPropertyTr:public fc_sxm_tclPropertyTr<M, SRVDI, FNID> {
        /* if data differ, write the data and notify service-handler  */
        tVoid vSetAndNotify(M const &rMIn) {
            M &rMOld=fc_sxm_tclProtectedData<M>::rAccess();
            if (rMIn != rMOld) {
                rMOld=rMIn;
                fc_sxm_tclProtectedData<M>::vRelease();
                fc_sxm_tclPropertyTr<M, SRVDI, FNID>::vNotify();
                return;
            }
            fc_sxm_tclProtectedData<M>::vRelease();
        }
    };


#endif
