/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_assert.h
* @brief       Provides the macros to do common functionality like
*              1. memory clear,
*              2. for_each, for_each_const for vector, map
*              3. conditional find for map, set
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#ifndef DAB_ASSERT_H
#define DAB_ASSERT_H


/*
  Some usefull macros for
  - ASSERTS
  - bit-operations
  - STL-containers
*/


#include "fc_sxm_config.h"


#define SXM_GET_U32(pu8) ((tU32)(((tU32)((pu8)[0])<<24) + ((tU32)((pu8)[1])<<16) + ((tU32)((pu8)[2])<<8) + (pu8)[3]))
#define SXM_GET_U24(pu8) ((tU32)(((tU32)((pu8)[1])<<16) + ((tU32)((pu8)[2])<<8) + (pu8)[3]))
#define SXM_GET_U16(pu8) ((tU16)(((tU16)((pu8)[0])<<8) + (pu8)[1]))

#define SXM_GET_BOOL(a) ((a)!=0 ? TRUE:FALSE)

//lint -emacro(572, SXM_SET_U32) prio2: Excessive shift
//lint -emacro(778, SXM_SET_U32) prio2: Constant expression evaluates to 0 in operation '>>'
#define SXM_SET_U32(pu8, u32)   \
    (pu8)[0]=(tU8)(((tU32)(u32)>>24)& 0xFF);       \
    (pu8)[1]=(tU8)(((tU32)(u32)>>16)& 0xFF);  \
    (pu8)[2]=(tU8)(((tU32)(u32)>>8) & 0xFF);  \
    (pu8)[3]=(tU8)(((tU32)(u32))    & 0xFF);

//lint -emacro(572, SXM_SET_U16) prio2: Excessive shift
//lint -emacro(778, SXM_SET_U16) prio2: Constant expression evaluates to 0 in operation '>>'
#define SXM_SET_U16(pu8, u16)   \
    (pu8)[0]=(tU8)(((tU16)(u16)>>8) & 0xFF);         \
    (pu8)[1]=(tU8)(((tU16)(u16))    & 0xFF);


#define SXM_BOOL_FROM_BIT(val, mask) (((val)&(mask))!=0)

#define SXM_SET_BIT_FROM_TO(val, valBitPos, target, targetBitPos) \
    if ((val) & (1<<(valBitPos))) {                               \
        target &= ~(1<<(targetBitPos));                                \
    } else {                                              \
        target |= (1<<(targetBitPos));                         \
    }

#define SXM_SET_BIT(bSet, target, targetBitPos) \
    if (bSet) {                                          \
        target &= (tU8)(~(1<<(targetBitPos)));                   \
    } else {                                              \
        target |= (tU8)((1<<(targetBitPos)));                          \
    }


#define SXM_ASSERT_RETURN(expr)                 \
    if (!(expr)) {                              \
        NORMAL_M_ASSERT_ALWAYS();               \
        return;                                 \
    }

#define SXM_ASSERT_RETURN_VAL(expr, retval)     \
    if (!(expr)) {                              \
        NORMAL_M_ASSERT_ALWAYS();               \
        return (retval);                        \
    }

#define SXM_RET_1_IF_NEQ(var) \
if (var != b.var) { \
    return TRUE; \
}

#define SXM_RET_0_IF_NEQ(var) \
if (var != b.var) { \
    return FALSE; \
}

#define SXM_RET_1_IF_LWR(var) \
if (var < b.var) { \
    return TRUE; \
}


#define SXM_FOREACH(contType, iterName, contName)                 \
    for (contType::iterator                                      \
     iterName = contName.begin(); iterName != contName.end(); ++iterName)

#define SXM_FOREACH_MAP(firstType, secondType, iterName, contName)          \
    for (map<firstType, secondType>::iterator                               \
     iterName = contName.begin(); iterName != contName.end(); ++iterName)

#define SXM_FOREACH_MAP_CONST(firstType, secondType, iterName, contName)          \
    map<firstType, secondType> const &contRef##iterName=contName;                                   \
    map<firstType, secondType>::const_iterator iterName##End =contRef##iterName.end();\
    for (map<firstType, secondType>::const_iterator                               \
     iterName = contRef##iterName.begin(); iterName != iterName##End; ++iterName)

#define SXM_FOREACH_MMAP(firstType, secondType, iterName, contName)     \
    for (multimap<firstType, secondType>::iterator                      \
             iterName = contName.begin(); iterName != contName.end(); ++iterName)

#define SXM_FOREACH_MMAP_CONST(firstType, secondType, iterName, contName)     \
    for (multimap<firstType, secondType>::const_iterator                      \
             iterName = contName.begin(); iterName != contName.end(); ++iterName)

#define SXM_FOREACH_CONST(contType, iterName, contName)                 \
    contType const &contRef##iterName=contName;                                   \
    contType::const_iterator iterName##End=contRef##iterName.end();               \
    for (contType::const_iterator                                       \
             iterName = contRef##iterName.begin(); iterName != iterName##End; ++iterName)

#define SXM_IF_FIND(contType, iterName, contName, val)                      \
    contType::iterator iterName = find(contName.begin(),contName.end(),val); \
    if  (iterName != contName.end())


#define SXM_IF_FIND_MAP(firstType, secondType, iterName, contName, val)     \
    map<firstType, secondType>::iterator iterName = contName.find(val); \
    if  (iterName != contName.end())

#define SXM_IF_FIND_MAP_CONST(firstType, secondType, iterName, contName, val)     \
    map<firstType, secondType>::const_iterator iterName = contName.find(val); \
    if  (iterName != contName.end())

#define SXM_IF_FIND_SET(keyType, iterName, contName, val)     \
    set<keyType>::iterator iterName = contName.find(val); \
    if  (iterName != contName.end())

#define SXM_IF_FIND_SET_CONST(keyType, iterName, contName, val)     \
    set<keyType>::const_iterator iterName = contName.find(val); \
    if  (iterName != contName.end())

//Note FC_SXM_MEMORY_CLEAR must not be invoked for clearing memory alloted to arrays
#define FC_SXM_MEMORY_CLEAR(MEMORY)     \
	if ( MEMORY)                          \
    {                                     \
        OSAL_DELETE MEMORY;               \
        MEMORY = OSAL_NULL;               \
    } 

#endif
