/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_ttfis_cmd.cpp
* @brief       Implementation of CCA Application for TTFIS command.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/

#include <sys/types.h>
#include <signal.h>

#include "fc_sxm_ttfis_cmd.h"
#include "fc_sxm_audio_fi.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_traffic_fi.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_APPLICATION
#include "trcGenProj/Header/fc_sxm_ttfis_cmd.cpp.trc.h"
#endif

#ifdef printf
#undef printf
#endif

/* TODO:
add service-id to ttfis-cmds and dispatch to data-app accordingly.
Otherwise each app has to edit this file.
*/

/* Class constructor */
typedef enum {
    fc_sxm_enTtfisCmdsMain_checkSms=1,
    fc_sxm_enTtfisCmdsMain_LAST    
} fc_sxm_tenTtfisCmdsMain;

fc_sxm_tclTtfisCmd::fc_sxm_tclTtfisCmd() {

}

/* Class destructor */
fc_sxm_tclTtfisCmd::~fc_sxm_tclTtfisCmd() {

}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vTraceCmdManager()
*
* DESCRIPTION: Callback Function for handling TRACE commands.
*
* PARAMETER: puchData - Input for the TRACE command.
*
* RETURNVALUE: tVoid
*
*******************************************************************************/
tVoid fc_sxm_tclTtfisCmd::vHandleCmdDirect(fc_sxm_trMsgCmdTtfisCmd &rCmd) const {
    switch (rCmd.u8MsgCode) {
        case 0x00:
            ETG_TRACE_ERR(("FC_SXM: writing backtrace off all threads to errmem and console"));
            raise(SIGRTMIN);
            break;
        default:
            ETG_TRACE_ERR(("vHandleCmdDirect: unknown cmd 0x%08x", rCmd.u8MsgCode));
            break;
    }
}


tVoid fc_sxm_tclTtfisCmd::vTraceCmdManager(const tUChar* puchData)
{
   if( puchData == OSAL_NULL)
   {
      // Donot Continue Further processing
      return;
   }
   else if (puchData[0]==0) {
       return;
   }
   else
   {
       // post message to cca-context
       fc_sxm_trMsgCmdTtfisCmd rCmd;
       rCmd.enServiceId=(fc_sxm_tenServiceID)SXM_GET_U16(&puchData[1]);
       rCmd.u8MsgCode=puchData[3];
       rCmd.u8DataLen=(tU8)(puchData[0]-3);
       tU32 u32SrvId=(tU32)rCmd.enServiceId;
       if (u32SrvId==0xFFFF) {
           fc_sxm_tclTtfisCmd::instance()->vHandleCmdDirect(rCmd);
           return;
       }
       OSAL_pvMemorySet(rCmd.au8Data,0,0xFF);
       if (rCmd.u8DataLen) {
           OSAL_pvMemoryCopy(rCmd.au8Data, &puchData[4], rCmd.u8DataLen);
       }
       fc_sxm_vPostMsgToCcaObj(fc_sxm_tclTtfisCmd::instance(), rCmd);
   }
}


tVoid fc_sxm_tclTtfisCmd::vHandleTtfisCmd(tU8 u8MsgCode, tU8 const *pu8Data) const
{
    fc_sxm_tenTtfisCmdsMain enMsgCode=(fc_sxm_tenTtfisCmdsMain)u8MsgCode;
    (void)pu8Data; /* TO REMOVE LINT WARNINGS */

    ETG_TRACE_USR1(("fc_sxm_tclTtfisCmd::vHandleMainTtfisCmd u8MsgCode=%d",
                    ETG_CENUM(fc_sxm_tenTtfisCmdsMain, enMsgCode))); 

    switch (enMsgCode) 
    {
        case fc_sxm_enTtfisCmdsMain_checkSms:
        {
            tBool  bOk;
            UN32 un32CurrentAllocatedBlocks,
                un32CurrentActualAllocatedBlocks,
                un32CurrentUserBytes, un32CurrentActualBytes,
                un32MaxAllocatedBlocks, un32MaxActualAllocatedBlocks,
                un32MaxUserBytes, un32MaxActualBytes,
                un32TotalSystemBytes;
            bOk = OSAL.bMemoryUsage( &un32CurrentAllocatedBlocks,
                                     &un32CurrentActualAllocatedBlocks,
                                     &un32CurrentUserBytes,
                                     &un32CurrentActualBytes,
                                     &un32MaxAllocatedBlocks,
                                     &un32MaxActualAllocatedBlocks,
                                     &un32MaxUserBytes,
                                     &un32MaxActualBytes,
                                     &un32TotalSystemBytes
                                     );
            printf("Memory Usage (bOk=%d):\r\n", bOk);
            printf("\tCurrent Memory Usage:\r\n");
            printf("\t\t%u blocks\r\n", un32CurrentAllocatedBlocks );
            printf("\t\t%u actual blocks\r\n",
                   un32CurrentActualAllocatedBlocks);
            printf("\t\t%u user bytes\r\n", un32CurrentUserBytes);
            printf("\t\t%u actual bytes\r\n", un32CurrentActualBytes);
            printf("\tMaximum Memory Usage:\r\n");
            printf("\t\t%u blocks\r\n", un32MaxAllocatedBlocks );
            printf("\t\t%u actual blocks\r\n",
                   un32MaxActualAllocatedBlocks);
            printf("\t\t%u user bytes\r\n", un32MaxUserBytes);
            printf("\t\t%u actual bytes\r\n", un32MaxActualBytes);
            printf("\tTotal System Memory: %u\r\n",
                   un32TotalSystemBytes );
            printf("\tOS-Overhead: %.1f%%\r\n",
                   un32CurrentUserBytes != 0 ?
                   (((double)(un32CurrentActualBytes -
                              un32CurrentUserBytes))
                    / un32CurrentUserBytes) * 100 : 100);

        }
        break;
        case fc_sxm_enTtfisCmdsMain_LAST:
            break;
        default:
           break;
    }
}
// function where the ttfis-input is actually handled.
tVoid fc_sxm_tclTtfisCmd::vProcess(fc_sxm_trMsgCmdTtfisCmd const *prMsg) const
{
    fc_sxm_tenServiceID enServiceId =prMsg->enServiceId;
    tU8 u8MsgCode=prMsg->u8MsgCode;
    tU8 u8DataLen = prMsg->u8DataLen;

    ETG_TRACE_USR1(("fc_sxm_tclTtfisCmd::vProcess(fc_sxm_trMsgCmdTtfisCmd) enServiceId=%d u8MsgCode=%d u8DataLen=%d",
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId),
                    u8MsgCode,
                    u8DataLen)); 

    switch (enServiceId) {
        case fc_sxm_enServiceID_Main:
            vHandleTtfisCmd(u8MsgCode, prMsg->au8Data);
            break;
        case fc_sxm_enServiceID_AppManager:
            fc_sxm_tclAppManager::instance()->vHandleTtfisCmd(u8MsgCode, prMsg->au8Data);
            break;
        default:
        {
            fc_sxm_trMsgCmdAppTtfisCmd rAppTtfisCmd;
            rAppTtfisCmd.enServiceId=prMsg->enServiceId;
            rAppTtfisCmd.u8MsgCode=prMsg->u8MsgCode;
            rAppTtfisCmd.u8DataLen=prMsg->u8DataLen;
            memcpy(rAppTtfisCmd.au8Data,prMsg->au8Data, 0xFF);
            fc_sxm_tclAppManager::instance()->vDistributeTtfisCmd(&rAppTtfisCmd);
        }
        break;
    }
}


tVoid fc_sxm_tclTtfisCmd::vSendSelfMessage(const fi_tclMessageBase& roSendFiObj) const 
{

    ETG_TRACE_USR1(("bSendSelfMessage: u16ServiceId=%d, fid=0x%x, opCode=0x%x",
                    roSendFiObj.u16GetServiceID(),
                    roSendFiObj.u16GetFunctionID(),
                    roSendFiObj.u8GetOpCode())); 

    fi_tclVisitorMessage oOutVisitorMsg( roSendFiObj.corfoGetTypeBase() );


		oOutVisitorMsg.vInitServiceData (
				fc_sxm_u16GetAppId(),              				// AppID of this application
				 fc_sxm_u16GetAppId(),        				// AppID of the Server
				AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
				0,                    				// StreamCounter
				0,   // RegId
				0,                                  // command counter
				roSendFiObj.u16GetServiceID(),//u16ServiceId,			                // SID of the service
				roSendFiObj.u16GetFunctionID(), //u16FID, //FID
				roSendFiObj.u8GetOpCode() //u8Opcode  // MethodStart
				);
 

    if (AIL_EN_N_NO_ERROR != fc_sxm_tclApp::instance()->enPostMsg(&oOutVisitorMsg)) {
        ETG_TRACE_USR1(("bSendSelfMessage failed!")); 
        
    }
}
