/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_main.cpp
* @brief       Implementation of CCA Application.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/



/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "fc_sxm_main.h"
#include "fc_sxm_tcl_config.h"
#include "fc_sxm_tcl_pers_data.h"
#include "fc_sxm_ttfis_cmd.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_service_sxm_audio.h"
#include "fc_sxm_service_sxm_fuel.h"
#include "fc_sxm_service_sxm_canadian_fuel.h"
#include "fc_sxm_service_sxm_movies.h"
#include "fc_sxm_service_sxm_traffic.h"
#include "fc_sxm_service_sxm_tabweather.h"
#include "fc_sxm_service_sxm_agw.h"
#include "fc_sxm_service_sxm_channelart.h"
#include "fc_sxm_service_sxm_stocks.h"
#include "fc_sxm_service_sxm_sports.h"
#include "fc_sxm_clienthandler_tts.h"
#include "fc_sxm_clienthandler_profilemgr.h"
#include "fc_sxm_client_spm.h"
#include "fc_sxm_audiorouting_lib.h"
#include "fc_sxm_tcl_states.h"
#include "fc_sxm_diaglibhandler.h"
#include "fc_sxm_client_diaglog.h"
#include "fc_sxm_diaglib_iocontrollistener.h"
#include "fc_sxm_tcl_audio_app.h"
#include "fc_sxm_client_clock.h"
#include "fc_sxm_service_sxm_wsalerts.h"
#include "fc_sxm_service_sxm_parking.h"
#include "fc_sxm_service_sxm_phonetics.h"

#if(FC_SXM_ENABLE_LCM_WATCHDOG_TIMER)
#include <signal.h>
#endif

// sms
#include "fc_sxm_tcl_sms_init.h"

#include "fc_sxm_tcl_diag_handler_test.h"


#define DP_S_IMPORT_INTERFACE_FI
#include "dp_generic_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_APPLICATION
#include "trcGenProj/Header/fc_sxm_main.cpp.trc.h"
#endif

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/
#ifdef SMS_64_BIT_SUPPORT
  tU32 fc_sxm_tclApp::u32MessageID = 0;
#endif

/******************************************************************************/
/*                                                                            */
/* METHODS                                                                    */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* FUNCTION: fc_sxm_tclApp::fc_sxm_tclApp()
*
* DESCRIPTION: Constructor.
*
*              Initialize static self reference to be used in function
*              callbacks.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_sxm_tclApp::fc_sxm_tclApp()
: ahl_tclBaseOneThreadApp(CCA_C_U16_APP_FC_SXM),
  _enOffWaitAckState(fc_sxm_enOffWaitAckState_Idle),
  _bIsModuleMounted(FALSE),
  _u16WDTCount(0)
{
  ETG_TRACE_USR1(("fc_sxm_tclApp() entered."));
  // start x65 module as early as possible
  fc_sxm_tclSMSInit::instance()->InitializeSMS();
  //  fc_sxm_tclSMSInit::instance()->vStopX65Module();
  //  fc_sxm_tclSMSInit::instance()->vStartX65Module();
  
}

/*******************************************************************************
*
* FUNCTION: fc_sxm_tclApp::~fc_sxm_tclApp()
*
* DESCRIPTION: Destructor.
*
*              Invalidate static self reference.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_sxm_tclApp::~fc_sxm_tclApp()
{
  ETG_TRACE_USR1(("~fc_sxm_tclApp() entered."));
  OSAL_DELETE fc_sxm_tclConfig::instance();
  OSAL_DELETE fc_sxm_tclConfigPresets::instance();
  _bIsModuleMounted = FALSE;
  _u16WDTCount = 0;

}

/*******************************************************************************
*
* FUNCTION: ahl_fc_sxm_tclApp::theServer()
*
* DESCRIPTION: Create the Radio data application
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
*******************************************************************************/
fc_sxm_tclApp* fc_sxm_tclApp::theServer()
{
   return instance();
}

#ifndef FC_SXM_DISABLE_SMS
#define FC_SXM_ADD_CLIENT(C) \
  _oClientList.poAdd(C::instance()); \
  _oMemberList.insert(C::instance());

#define FC_SXM_ADD_SERVICE(C) \
  _oServiceList.poAdd(C::instance()); \
  _oMemberList.insert(C::instance());

tVoid fc_sxm_tclApp::vEnableDataService()
{
      FC_SXM_ADD_SERVICE(fc_sxm_tclFuelService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclTrafficService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclMoviesService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclAgwService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclTabweatherService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclStocksService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclSportsService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclCanadianFuelService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclWsAlertsService);
      FC_SXM_ADD_SERVICE(fc_sxm_tclParkingService);
}
#endif


/*******************************************************************************
*
* FUNCTION: tBool fc_sxm_tclApp::bOnInit()
*
* DESCRIPTION: This function is called by the CCA framework before starting any
*              CCA message related communication action. As default this
*              function always returns TRUE. The user has to implement all his
*              local application initialization (create client and service
*              handler, create worker threads, allocate application memory,...).
*              If returns TRUE, initialization could be successfully performed
*              and framework will register application, else, a reset of the
*              system is forced immediately.
*
*              This function is the counterpart of vOnApplicationClose().
*
* PARAMETER: None.
*
* RETURNVALUE: TRUE  = Application successfully initialized.
*              FALSE = Application NOT successfully initialized.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::bOnInit().
*******************************************************************************/
tBool fc_sxm_tclApp::bOnInit()
{
  tBool bResult = TRUE;

  ETG_TRACE_USR1(("bOnInit() entered."));

  //Whether module is mounted or not
  _bIsModuleMounted = bIsModuleMounted();
  if(FALSE == _bIsModuleMounted)
  {
	  ETG_TRACE_USR1(("bOnInit() Module is not mounted"));
	  return bResult;
  }
  fc_sxm_tclConfig::instance();
  fc_sxm_tclConfigPresets::instance();
  fc_sxm_tclConfigDsrls::instance();

#ifdef FC_SXM_MAIN_TEST
  ETG_TRACE_USR4(("testSem start."));
    fc_sxm_tclSem oTestSem;
    oTestSem.vOpen();
  ETG_TRACE_USR4(("testSem opened."));
    oTestSem.vGet();
  ETG_TRACE_USR4(("testSem get 1 OK."));
    oTestSem.vGet();
  ETG_TRACE_USR4(("testSem get 2 OK."));
    oTestSem.vPost();
  ETG_TRACE_USR4(("testSem post 1 OK."));
    oTestSem.vPost();
  ETG_TRACE_USR4(("testSem post 2 OK."));


  ETG_TRACE_USR4(("testSem end."));


  string testVal;
  fc_sxm_tclConfig::instance()->bGetVal("val3", testVal);
  ETG_TRACE_USR4(("bOnInit() testVal=%s", testVal.c_str()));
  
  vector<fc_sxm_trDefaultPreset> const &rPresets=fc_sxm_tclConfigPresets::instance()->roGetDefaultPresets();
  SXM_FOREACH_CONST(vector<fc_sxm_trDefaultPreset>, iterPresets, rPresets) {
      ETG_TRACE_ERR(("defaultPreset:bank=%d preset=%d channel=%d",
                     iterPresets->u32Bank, iterPresets->u32Preset, iterPresets->u32Channel));   
  }

#endif
  

#ifndef FC_SXM_DISABLE_SMS
  ETG_TRACE_USR4(("fc_sxm_tclApp::bOnInit() vCheckDBVersion start"));
  fc_sxm_tclDbVersionConfig::instance()->vCheckDBVersion();
  ETG_TRACE_USR4(("fc_sxm_tclApp::bOnInit() vCheckDBVersion end"));

  fc_sxm_tclConfig::instance()->vCheckKDSPresetType();

  //FC_SXM_ADD_CLIENT(fc_sxm_tclPosClient);
  FC_SXM_ADD_CLIENT(fc_sxm_tclSpmClient);
  FC_SXM_ADD_CLIENT(fc_sxm_tclDiagLogClient);
  FC_SXM_ADD_CLIENT(fc_sxm_tclTTSClient);
#ifdef VARIANT_S_FTR_ENABLE_PERSONALIZATION
  FC_SXM_ADD_CLIENT(fc_sxm_tclProfileMgrClient);
#endif
  //FC_SXM_ADD_CLIENT(fc_sxm_tclClockClient);

  FC_SXM_ADD_SERVICE(fc_sxm_tclAudioService);
  FC_SXM_ADD_SERVICE(fc_sxm_tclChannelartService);
  FC_SXM_ADD_SERVICE(fc_sxm_tclPhoneticsService);

  tU8 u8VariantType = u8GetVariantType();
  ETG_TRACE_USR4(("fc_sxm_tclApp::bOnInit() Variant type: 0x%x", u8VariantType));
  switch(u8VariantType)
  {
  case FC_SXM_VARIANT_TYPE_DA:
      //Disable Data services for DA variant
      break;
  case FC_SXM_VARIANT_TYPE_NAVI:
      /*FALLTHROUGH*/
  case FC_SXM_VARIANT_TYPE_NOT_CONFIGURED:
      //Enable Data services for NAV variant or if KDS not configured.
#ifndef VARIANT_S_FTR_DISABLE_DATASERVICES
      //Disable data services for INF4CV NAV variant
      ETG_TRACE_USR4(("fc_sxm_tclApp::bOnInit() enable data services"));
      vEnableDataService();
#endif
      break;
   default:
      break;
  }
  _oMemberList.insert(fc_sxm_arl_tclISource::instance());
  _oMemberList.insert(fc_sxm_diagLibHandler::instance());
  _oMemberList.insert(fc_sxm_tclTtfisCmd::instance());

  //send SXM source availability to arl based on KDS parameter.
  vSendSourceAvailability();

  //Initialise the diaglib
  fc_sxm_diagLibHandler::instance()->vInit();
  
#endif

  //Init the SXMApp Manager
  fc_sxm_tclAppManager::instance()->vInitialize();


  //Plug in the trace channel
  vPlugTraceIn((TR_tenTraceChan)TR_TTFIS_SXM, (OSAL_tpfCallback)fc_sxm_tclTtfisCmd::vTraceCmdManager);

  ETG_TRACE_USR1(("bOnInit() END."));

  return bResult;
}


#if(FC_SXM_ENABLE_LCM_WATCHDOG_TIMER)
tBool fc_sxm_tclApp::bOnWatchdog()
{
	ETG_TRACE_USR1(("bOnWatchdog() entered."));

	if((TRUE == _bIsModuleMounted)
		&& (fc_sxm_tclAudioProxy::instance()->eGetAudioProxyState() != fc_sxm_enAudioReady))
	{
		++_u16WDTCount;
		ETG_TRACE_USR1(("bOnWatchdog() Module not Ready! _u16ResetFlag: %d",_u16WDTCount));

		//Giving additional 12000 ms for the first time
		if(_u16WDTCount == 1)
		{
		    return TRUE;
		}

		ETG_TRACE_FATAL(("bOnWatchdog() Raise SIGUSR2"));
        // Raise the signal to get the callstack the threads
        raise(SIGUSR2);
		return FALSE;
	}
	return TRUE;
}
#endif

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vOnApplicationClose()
*
* DESCRIPTION: This function is called by the CCA framework to indicate the
*              imminent shutdown of the application. The user has to implement
*              the de-initialization of all his local application data (destroy
*              client and service handler, destroy worker threads, free
*              application memory, ... ). After this function call the
*              destructor of this application will be called.
*
*              This function is the counterpart of bOnInit().
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnApplicationClose().
*******************************************************************************/
tVoid fc_sxm_tclApp::vOnApplicationClose()
{
  ETG_TRACE_USR1(("vOnApplicationClose() entered."));
  
  
  try  
  {
      _oClientList.vClear();
      _oServiceList.vClear();

      fc_sxm_tclAppManager::instance()->vDeInitialize();

      _oMemberList.clear();

      OSAL_DELETE fc_sxm_tclDiagLogClient::instance();

      OSAL_DELETE fc_sxm_tclTtfisCmd::instance();
      //delete the diaglib handler
      OSAL_DELETE fc_sxm_diagLibHandler::instance();
      // delete audio-routing-lib
      OSAL_DELETE fc_sxm_arl_tclISource::instance();

      OSAL_DELETE fc_sxm_tclAppManager::instance();

  }
  catch (...) 
  { } 
  

  ahl_tclBaseOneThreadApp::vOnApplicationClose();
}


fc_sxm_tclBaseService *fc_sxm_tclApp::poGetServiceHandler(tU16 u16ServiceId) {
    return _oServiceList.poGetAhlObject(u16ServiceId);
}
fc_sxm_tclBaseClient *fc_sxm_tclApp::poGetClientHandler(tU16 u16ServiceId) {
    return _oClientList.poGetAhlObject(u16ServiceId);
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vOnLoadSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              loading of last mode settings. The user has to load the
*              previously stored last mode settings from persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoadSettings().
*******************************************************************************/
tVoid fc_sxm_tclApp::vOnLoadSettings()
{
   ETG_TRACE_USR4(("vOnLoadSettings() START"));
   fc_sxm_tclPersData::instance()->vLoadData();
   ETG_TRACE_USR4(("vOnLoadSettings() END"));
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vOnSaveSettings()
*
* DESCRIPTION: This function is called by the CCA framework to trigger the 
*              storage of last mode settings. The user has to store the last
*              mode settings to persistent memory.
*
* PARAMETER: None.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnSaveSettings().
*******************************************************************************/
tVoid fc_sxm_tclApp::vOnSaveSettings()
{
    ETG_TRACE_USR4(("vOnSaveSettings() START"));
    fc_sxm_tclPersData::instance()->vSaveData();
    ETG_TRACE_USR4(("vOnSaveSettings() END"));
}

tVoid fc_sxm_tclApp::vOnNewAppStateNotify(tU32 u32OldAppState, tU32 u32AppState) {
	ETG_TRACE_USR4(("vOnNewAppStateNotify(%d) START", u32AppState));
	if(FALSE == _bIsModuleMounted)
	{
	  ETG_TRACE_USR1(("vOnNewAppStateNotify() Module is not mounted"));
	  return;
	}
    (tVoid)u32OldAppState;
    tBool bCcaOn=FALSE;
    if (u32AppState==AMT_C_U32_STATE_NORMAL ||
        u32AppState==AMT_C_U32_STATE_DIAGNOSIS ||
        u32AppState==AMT_C_U32_STATE_PAUSE) {
        _enOffWaitAckState=fc_sxm_enOffWaitAckState_Idle;
        bCcaOn=TRUE;
    }else if (u32AppState==AMT_C_U32_STATE_OFF) {
        _enOffWaitAckState=fc_sxm_enOffWaitAckState_Idle;
        vAppStateChanged(u32AppState, 20000);
        _enOffWaitAckState=fc_sxm_enOffWaitAckState_BlockNext;
        /* later the ahl-framework will also try to call vAppStateChanged()
           but we have overwritten this virtual method and will block it
        */
    }

    fc_sxm_trMsgAudioCmdNewAppState rMsg(bCcaOn);
#ifndef FC_SXM_DISABLE_SMS
    fc_sxm_tclAudioApp::instance()->vPostMsgNew(rMsg);
#endif
    ETG_TRACE_USR4(("vOnNewAppStateNotify() END"));
}


/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vOnTimer(tU16 u16TimerId)
*
* DESCRIPTION: This function is called by the CCA framework on the expiration
*              of a previously via function bStartTimer() started timer. The
*              expired timer is forwarded to the respective service or
*              client-handler via a call of vProcessTimer().
*
*              The function is called from this applications context and
*              therefore no interthread programming rules must be considered
*              and the application methods and/or member variables can be
*              accessed without using the static self reference
*              'instance()'.
*
* PARAMETER: [IN] u16TimerId = Identifier of the expired timer.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnTimer().
*******************************************************************************/
tVoid fc_sxm_tclApp::vOnTimer(tU16 u16TimerId)
{
  ETG_TRACE_USR4(("vOnTimer() entered."));
  fc_sxm_tclAppManager::instance()->vHandleTimer(u16TimerId);
}

/*******************************************************************************
*
* FUNCTION: tVoid fc_sxm_tclApp::vOnLoopback
*                 (tU16 u16ServiceID, amt_tclServiceData* poMessage)
*
* DESCRIPTION: This function is called by the CCA framework if a message was
*              sent from one if this applications services or from a
*              callback-handler of this application (so called self or loopback
*              messages). The message can be forwarded to the respective service
*              via a call of vOnLoopbackService(). If there is no service
*              registered for this message then the user has the option to
*              directly evaluate the message in the default block of this.
*              methods switch-clause.
*
* PARAMETER: [IN] u16ServiceID = Identifier of the service from where the
*                                message was sent.
*
*            [IN] poMessage = Message object.
*
* RETURNVALUE: None.
*
********************************************************************************
* Overrides method ahl_tclBaseOneThreadApp::vOnLoopback().
*******************************************************************************/
/*
tVoid fc_sxm_tclApp::vOnLoopback(tU16 u16ServiceID, amt_tclBaseMessage* poMessage)
{
    ETG_TRACE_USR4(("vOnLoopback() DUMMY u16ServiceID=0x%04x",u16ServiceID));
}
*/

tVoid fc_sxm_tclApp::vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage)
{
    (tVoid)u16ServiceID;
  tU16 u16ServiceId=poMessage->u16GetServiceID();
  ETG_TRACE_USR4(("vOnLoopback() SID=0x%04x START", u16ServiceId));
  
  // extract payload from message by 
  // initializing a visitor message as container copy function
  gm_tclU32Message oU32Msg(poMessage);
  fi_tclVisitorMessage    oVisitorMsg ((amt_tclBaseMessage*) poMessage );
  switch (u16ServiceId) {
      case fc_sxm_enServiceID_Main:
      {
#ifdef SMS_64_BIT_SUPPORT
         tU32 u32MessageID = oU32Msg.u32GetDWord();
         ETG_TRACE_USR4(("vOnLoopback() u32MessageID =%d", u32MessageID));
         map<tU32, fc_sxm_tclMessage *>::iterator _mapLoopbackMessagesIt;
         _mapLoopbackMessagesIt = _mapLoopbackMessages.find(u32MessageID);
         if(_mapLoopbackMessagesIt != _mapLoopbackMessages.end())
         {
            fc_sxm_tclMessage *poThreadMsg= (fc_sxm_tclMessage *)_mapLoopbackMessagesIt->second;
            // dispatch message that is meant for cca-context
            ETG_TRACE_USR4(("vOnLoopback() fc_sxm_enServiceID_Main, Action=%d", poThreadMsg->u16GetActionOnly()));
            vDispatchMsg(poThreadMsg);
            _mapLoopbackMessages.erase(_mapLoopbackMessagesIt);
            /* delete allocated memory */
            OSAL_DELETE poThreadMsg;
         }
#else
         fc_sxm_tclMessage *poThreadMsg= (fc_sxm_tclMessage *)((uintptr_t)(oU32Msg.u32GetDWord()));
         // dispatch message that is meant for cca-context
         ETG_TRACE_USR4(("vOnLoopback() fc_sxm_enServiceID_Main, Action=%d", poThreadMsg->u16GetActionOnly()));
         vDispatchMsg(poThreadMsg);
         /* delete allocated memory */
         OSAL_DELETE poThreadMsg;
#endif

      }
      break;
      case fc_sxm_enServiceID_AppManager:
      {
#ifdef SMS_64_BIT_SUPPORT
         tU32 u32MessageID = oU32Msg.u32GetDWord();
         ETG_TRACE_USR4(("vOnLoopback() u32MessageID =%d", u32MessageID));
         map<tU32, fc_sxm_tclMessage *>::iterator _mapLoopbackMessagesIt;
         _mapLoopbackMessagesIt = _mapLoopbackMessages.find(u32MessageID);
         if(_mapLoopbackMessagesIt != _mapLoopbackMessages.end())
         {
            fc_sxm_tclMessage *poThreadMsg= (fc_sxm_tclMessage *)_mapLoopbackMessagesIt->second;
            ETG_TRACE_USR4(("vOnLoopback() fc_sxm_enServiceID_AppManager, Action=%d", poThreadMsg->u16GetActionOnly()));
            fc_sxm_tclAppManager::instance()->vDispatchMsg(poThreadMsg);
            _mapLoopbackMessages.erase(_mapLoopbackMessagesIt);
            /* delete allocated memory */
            OSAL_DELETE poThreadMsg;
         }
#else
         fc_sxm_tclMessage *poThreadMsg= (fc_sxm_tclMessage *)((uintptr_t)(oU32Msg.u32GetDWord()));
         ETG_TRACE_USR4(("vOnLoopback() fc_sxm_enServiceID_AppManager, Action=%d", poThreadMsg->u16GetActionOnly()));
         fc_sxm_tclAppManager::instance()->vDispatchMsg(poThreadMsg);
         /* delete allocated memory */
         OSAL_DELETE poThreadMsg;
#endif
          
      }
      break;
      default:
      {
          // call vOnLoopback of base-class (see using-directive in declaration)
          ETG_TRACE_COMP(("vOnLoopback() called with serviceId=0x%04x, fnId=%d opCode=%d (IGNORED)", 
                          u16ServiceId,
                          poMessage->u16GetFunctionID(),
                          poMessage->u8GetOpCode()));
          ahl_tclBaseOneThreadApp::vOnLoopback(u16ServiceID, poMessage);
      }
      break;
      }

  ETG_TRACE_USR4(("vOnLoopback() SID=0x%04x END", u16ServiceId));

}
tVoid fc_sxm_tclApp::vDispatchMsg(fc_sxm_tclMessage *poThreadMsg) {
    fc_sxm_tenServiceID enServiceId=poThreadMsg->enGetServiceId();
    ETG_TRACE_USR4(("fc_sxm_tclApp::vDispatchMsg: SID=%x action=%d START", 
                    ETG_CENUM(fc_sxm_tenServiceID, enServiceId), 
                    poThreadMsg->u16GetActionOnly()));

    tU32 u32Action=poThreadMsg->u32GetAction();
    if (enServiceId!=fc_sxm_enServiceID_Main) {
        ETG_TRACE_USR1(("fc_sxm_tclApp::vDispatchMsg: not for me: u32Action=0x%x",
                        u32Action));
        return;
    }
    switch (u32Action) {
            //SXM_MSGQ_DISPATCH(fc_sxm_trMsgCmdMainAllowService);
            //SXM_MSGQ_DISPATCH(fc_sxm_trMsgCcaThreadMsg);

            //SXM_MSGQ_DISPATCH(fc_sxm_trMsgOffReached);

    SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCmdMainAllowService,fc_sxm_tclApp::instance());
    SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgCcaThreadMsg,fc_sxm_tclApp::instance());

    SXM_MSGQ_DISPATCH_TO_OBJ(fc_sxm_trMsgOffReached,fc_sxm_tclApp::instance());

        default:
            ETG_TRACE_USR4(("fc_sxm_tclApp::vDispatchMsg: unknown u32Action=0x%x",
                            u32Action));
            break;  
    }
}

tVoid fc_sxm_tclApp::vProcess(fc_sxm_trMsgCmdMainAllowService const *prMsg) {
    fc_sxm_tclBaseService *poServiceHanlder=poGetServiceHandler(prMsg->u16CcaServiceId);
    if (OSAL_NULL != poServiceHanlder) {
        // todo: call ahl-base-class
       poServiceHanlder->bAllowService(prMsg->bAllow);
    }
}

/*
  dispatch message to one of our sub-objects.
*/
tVoid fc_sxm_tclApp::vProcess(fc_sxm_trMsgCcaThreadMsg const *prMsg) const
{
    ETG_TRACE_USR4(("fc_sxm_tclApp::vProcess(fc_sxm_trMsgCcaThreadMsg) START"));
    SXM_ASSERT_RETURN(OSAL_NULL != prMsg);
    fc_sxm_tclMsgCallerBase *poCaller=prMsg->poCaller;
    SXM_ASSERT_RETURN(OSAL_NULL != poCaller);
    tVoid *pvTarget=poCaller->pvGetTarget();
    SXM_ASSERT_RETURN(OSAL_NULL != pvTarget);
    SXM_IF_FIND_SET_CONST(tVoid *, iter, _oMemberList, pvTarget) {
        // only members of _oMemberList may be accessed by the method
        poCaller->vCall();
    } else {
        ETG_TRACE_USR4(("fc_sxm_tclApp::vProcess(fc_sxm_trMsgCcaThreadMsg): no target 0x%08x not registered",
                        pvTarget));
    }
    OSAL_DELETE poCaller;

    ETG_TRACE_USR4(("fc_sxm_tclApp::vProcess(fc_sxm_trMsgCcaThreadMsg) END"));
}



tVoid fc_sxm_tclApp::vNotifyPropertyChanged(tU16 enServiceId, tU16 u16FunktionId) {
     ETG_TRACE_USR4(("fc_sxm_tclApp::vNotifyPropertyChanged() entered. enServiceId=0x%04x FID = 0x%4x.", 
                     enServiceId, u16FunktionId));
    fc_sxm_tclBaseService *poService=poGetServiceHandler(enServiceId);
    if (OSAL_NULL != poService) {
        poService->vHandlePropertyChanged(u16FunktionId);
    }
}

tVoid fc_sxm_vNotifyPropertyChanged(tU16 enServiceId, tU16 u16FunktionId) {
    fc_sxm_tclApp::instance()->vNotifyPropertyChanged(enServiceId, u16FunktionId);
}


tU16 fc_sxm_u16GetAppId() {
    return fc_sxm_tclApp::instance()->u16GetAppId();
}

ail_tenCommunicationError fc_sxm_tclApp::enPostMsg(amt_tclServiceData* poMessage) {
    ail_tenCommunicationError enCommError = AIL_EN_N_NOT_SEND;

    if(poMessage!=NULL)
    {
       tBool valid = poMessage->bIsValid();
       if((poMessage->u16GetRegisterID()==AMT_C_U16_REGID_INVALID) || (poMessage->u16GetServiceID()==AMT_C_U16_SERVICE_UNDEF))
       {
          poMessage->bDelete();
          ETG_TRACE_ERR(("Result PostMessage:not_sent %d",
                enCommError));
       }
       else
       {
          enCommError=enPostMessage(poMessage);
          if (AIL_EN_N_NO_ERROR != enCommError)
          {
             ETG_TRACE_USR4(("Result PostMessage: %d poMessage->bIsValid() = %d",
                   enCommError,valid));
             poMessage->bDelete();
          }
       }
    }
    return enCommError;
}



/*
  Send a message to fc_sxm_tclAppManager.
  Data given via pvData have to be allocated by caller, we take custodie.
*/
#ifdef SMS_64_BIT_SUPPORT
tVoid fc_sxm_tclApp::vPostMsg(fc_sxm_tclMessage *prMsg)
#else
tVoid fc_sxm_tclApp::vPostMsg(fc_sxm_tclMessage *prMsg) const
#endif
{
    tU32 u32Action=prMsg->u32GetAction();
    ETG_TRACE_USR4(("fc_sxm_tclApp::vPostMsg: SID=%x action=%d START", 
                    ETG_CENUM(fc_sxm_tenServiceID, prMsg->enGetServiceId()), 
                    prMsg->u16GetActionOnly()));
    /* Create a 'generic message' (with tU32 content), intended to be sent 
       to our own application (loopback message)
       we abuse some address-field for our data
    */
    tU16 u16AppId_=fc_sxm_u16GetAppId();
    gm_tclU32Message oLoopbackMsg(
                                  u16AppId_,    // Source AppID 
                                  u16AppId_,    // Target AppID 
                                  0,                            // RegisterID   
                                  0,                            // CmdCounter  
                                  (tU16)(u32Action>>16),                 // ServiceID   
                                  (tU16)(u32Action&0xFFFF),                   // Function ID
                                  AMT_C_U8_CCAMSG_OPCODE_STATUS // Opcode
                                  );
#ifdef SMS_64_BIT_SUPPORT
    fc_sxm_tclApp::u32MessageID++;
    // The 32Bit content of the message are used to store the data-pointer.
    _mapLoopbackMessages.insert(std::pair<tU32, fc_sxm_tclMessage *>(u32MessageID, prMsg));
    ETG_TRACE_USR4(("fc_sxm_tclApp::vPostMsg:Inserted u32MessageID=%d, prMsg=%d", fc_sxm_tclApp::u32MessageID, prMsg));
    oLoopbackMsg.vSetDWord(fc_sxm_tclApp::u32MessageID);
#else
	// The 32Bit content of the message are used to store the data-pointer.
    oLoopbackMsg.vSetDWord((uintptr_t)prMsg); 
#endif
    

    // send the created loopback-message
    if (oLoopbackMsg.bIsValid())
    {
        // Post message
        ail_tenCommunicationError enResult = 
            fc_sxm_tclApp::instance()->enPostMessage(&oLoopbackMsg, TRUE);
         
        if (enResult != AIL_EN_N_NO_ERROR)
        {
            ETG_TRACE_ERR(( "fc_sxm_tclApp::vPostMsg: enPostMessage() failed," 
                              "'ail_tenCommunicationError' = %u", (tU32)enResult));
            NORMAL_M_ASSERT_ALWAYS();
        }
    }
    else
    {
        NORMAL_M_ASSERT_ALWAYS();
    }

}


tVoid fc_sxm_vSendSelfMessage(const fi_tclMessageBase& roSendFiObj) {
    fc_sxm_tclTtfisCmd::instance()->vSendSelfMessage(roSendFiObj);
}

tVoid fc_sxm_vSendError(fc_sxm_trAdressing const &rAdressing, tU16 u16ErrorCode) {
    fc_sxm_tclBaseService::vSendError(rAdressing, u16ErrorCode); 
}


tVoid fc_sxm_tclApp::vProcess(fc_sxm_trMsgOffReached const *prMsg) {
   (tVoid)prMsg; //Solving lint error
    ETG_TRACE_USR4(("fc_sxm_tclApp::vProcess(fc_sxm_trMsgOffReached):_enOffWaitAckState=%d", 
                    ETG_CENUM(fc_sxm_tenOffWaitAckState, _enOffWaitAckState)));

    if (_enOffWaitAckState==fc_sxm_enOffWaitAckState_WaitApp) {
        // now we got ack from app and can send ack to SPM
        _enOffWaitAckState=fc_sxm_enOffWaitAckState_Idle;
        vAppStateChanged(AMT_C_U32_STATE_OFF, 0);
    }
}

tVoid fc_sxm_tclApp::vAppStateChanged (tU32 u32AppState, OSAL_tMSecond tApproximatedChangeTime) {
    // this call comes from ahl-framwork, we don't want to ack the stopp immedialtelly, so we block this call:
    ETG_TRACE_COMP(("fc_sxm_tclApp::vAppStateChanged: u32AppState=%d tApproximatedChangeTime=%d _enOffWaitAckState=%d", 
                    u32AppState, tApproximatedChangeTime, 
                    ETG_CENUM(fc_sxm_tenOffWaitAckState, _enOffWaitAckState)));
    switch (_enOffWaitAckState) {
        case fc_sxm_enOffWaitAckState_Idle:
            // case when ack has been received from app
            ail_tclAppInterfaceRestricted::vAppStateChanged(u32AppState, tApproximatedChangeTime);
            break;
        case fc_sxm_enOffWaitAckState_WaitApp:
            // case when AMT_C_U16_PWR_PROXY_STATE_CHANGE_IMMEDIATELY is received from SPM before APP has answered
            _enOffWaitAckState=fc_sxm_enOffWaitAckState_Idle;
            ail_tclAppInterfaceRestricted::vAppStateChanged(u32AppState, tApproximatedChangeTime);
            break;
        case fc_sxm_enOffWaitAckState_BlockNext:
            // this is the call from ahl-framework that we have to ignore
            _enOffWaitAckState=fc_sxm_enOffWaitAckState_WaitApp;
            break;
        default:
            break;
            
    }
}

#ifndef FC_SXM_DISABLE_SMS
tU8 fc_sxm_tclApp::u8GetVariantType() const
{
   tU8 u8VariantType = FC_SXM_VARIANT_TYPE_NOT_CONFIGURED;
#ifdef DP_U16_KDSADR_AIVIVARIANTCODING
   dp_tclKdsAIVIVariantCoding oKdsElement;
   if ((tU8) DP_U8_ELEM_STATUS_VALID == oKdsElement.u8GetVariantSelection((tU8&) u8VariantType)) {
      ETG_TRACE_USR4(("fc_sxm_tclApp::u8GetVariantType() : 0x%x", u8VariantType));
   }
   else {
      ETG_TRACE_ERR(("fc_sxm_tclApp ::u8GetVariantType(): read failed"));
   }
#endif
   return u8VariantType;
}
#endif

/*
 * Method reads KDS parameter to check whether SXM module is mounted or not. *
 */
tBool fc_sxm_tclApp::bIsModuleMounted() const
{
	tBool bIsModuleMounted = FALSE;
	const tU8 u8ModuleNotMounted = 0x00u;
	const tU8 u8ModuleMounted = 0x01u;
	tU8 u8SourceType = u8ModuleNotMounted;

#ifdef DP_U16_KDSADR_CMVARIANTCODING
	dp_tclKdsCMVariantCoding oSourceType;
	if ((tU8) DP_U8_ELEM_STATUS_VALID == oSourceType.u8GetXMTuner((tU8&) u8SourceType))
	{
	   if (u8SourceType == u8ModuleMounted)
	   {
	      bIsModuleMounted = TRUE;
	   }
	   ETG_TRACE_USR4(("fc_sxm_tclApp::bIsModuleMounted() : 0x%x", u8SourceType));
	}
	else
	{
	   ETG_TRACE_ERR(("fc_sxm_tclApp ::bIsModuleMounted(): read failed"));
	}
#endif
	return bIsModuleMounted;
}
/*
 * Method sends source availability to arl based on module availability
 */
tVoid fc_sxm_tclApp::vSendSourceAvailability()
{
   if (TRUE == _bIsModuleMounted)
   {
      fc_sxm_arl_tclISource::instance()->vSendSourceAvailability(ARL_EN_SRC_PLAYABLE, ARL_EN_REASON_NEWMEDIA);
   }
   else
   {
      fc_sxm_arl_tclISource::instance()->vSendSourceAvailability(ARL_EN_SRC_NOT_AVAILABLE, ARL_EN_REASON_NOMEDIA);
   }
}
