/**
* @swcomponent fc_sxm
* @{
* @file        fc_sxm_clienthandler_tts.cpp
* @brief       Implementation of CCA client-handler for the TTS.
* @copyright   (C) 2016 Robert Bosch Engineering and Business Solutions Private Limited.
*              The reproduction, distribution and utilization of this file as
*              well as the communication of its contents to others without express
*              authorization is prohibited. Offenders will be held liable for the
*              payment of damages. All rights reserved in the event of the grant
*              of a patent, utility model or design.
* @}
*/


#include "fc_sxm_clienthandler_tts.h"
#include "fc_sxm_tcl_sxmapp_manager.h"
#include "fc_sxm_audiorouting_lib.h"
#include "fc_sxm_tcl_wsalerts_app.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_SXM_APPLICATION
#include "trcGenProj/Header/fc_sxm_clienthandler_tts.cpp.trc.h"
#endif

/*
 * CTOR
 */
fc_sxm_tclTTSClient::fc_sxm_tclTTSClient()
         : fc_sxm_tclBaseClient(CCA_C_U16_SRV_SDS_TTS, FC_SXM_C_U16_SERVICE_TTS_FI_MAJOR_VERSION, FC_SXM_C_U16_SERVICE_TTS_FI_MINOR_VERSION, CCA_C_U16_APP_SDS_TTS),
           _sSynthesizeText(""),
           _bIsAbort(FALSE),
           _bIsStop(FALSE)
{
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient() entered."));
}

/*
 * DTOR
 */
fc_sxm_tclTTSClient::~fc_sxm_tclTTSClient(tVoid)
{
   ETG_TRACE_USR4(("~fc_sxm_tclTTSClient() entered."));
}

/**
 * This function is called by the CCA framework when the service this client-handler
 * has registered for has become available.
 * @override ahl_tclBaseOneThreadClientHandler::vOnServiceAvailable()
 */
tVoid fc_sxm_tclTTSClient::vOnServiceAvailable()
{
   ETG_TRACE_USR4(("vOnServiceAvailable() entered. AppID = %u.", u16GetServerAppID()));
}

/**
 * This function is called by the CCA framework when the service this client-handler
 * has registered for has become unavailable.
 * @override ahl_tclBaseOneThreadClientHandler::vOnServiceUnavailable()
 */
tVoid fc_sxm_tclTTSClient::vOnServiceUnavailable()
{
   ETG_TRACE_USR4(("vOnServiceUnavailable() entered. AppID = %u.", u16GetServerAppID()));
}

/**
 * Handle all received status-fi-messages.
 * @brief For each function-id of interest use the macro SXM_PSTATUS_ENTRY to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnStatus(tU16 /*u16FID*/, amt_tclServiceData* /*pServiceDataMessage*/)
{
//    ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnStatus FID=%u", u16FID));
//    switch (u16FID) {
//
//        default:
//            vOnUnknownMessage(pServiceDataMessage);
//            break;
//    }
}


/**
 * Handle all received method-result fi-messages.
 * @brief For each function-id of interest use the macro SXM_MRESULT_ENTRY to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnMethodResult(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnMethodResult FID=%u", u16FID));
   switch (u16FID) {
      /*
        use macro SXM_MRESULT_ENTRY for each suported function-id:
        fill according thread-message for fiObject and adressing
        according to "poMessage" and send message accoring to u32Action of thread-message
      */
      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT,
         sds_ttsfi_tclMsgPrepareVoiceOutputMethodResult,
         fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput);

      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_ABORTPROMPT,
         sds_ttsfi_tclMsgAbortPromptMethodResult,
         fc_sxm_trMsgTTSClientMResultAbortPrompt);

      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_PAUSEPROMPT,
         sds_ttsfi_tclMsgPausePromptMethodResult,
         fc_sxm_trMsgTTSClientMResultPausePrompt);

      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_CONTINUEPROMPT,
         sds_ttsfi_tclMsgContinuePromptMethodResult,
         fc_sxm_trMsgTTSClientMResultContinuePrompt);
   default:
      vOnUnknownMessage(pServiceDataMessage);
   }
}

/**
 * Handle all received method-result first fi-messages.
 * @brief For each function-id of interest use the macro SXM_MRESULT_ENTRY to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnMethodResultFirst(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnMethodResultFirst FID=%u", u16FID));
   switch (u16FID) {
      /*
        use macro SXM_MRESULT_ENTRY for each suported function-id:
        fill according thread-message for fiObject and adressing
        according to "poMessage" and send message accoring to u32Action of thread-message
      */
      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultFirst,
         fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak);
   default:
      vOnUnknownMessage(pServiceDataMessage);
   }
}

/**
 * Handle all received method-result middle fi-messages.
 * @brief For each function-id of interest use the macro SXM_MRESULT_ENTRY to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnMethodResultMiddle(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnMethodResultMiddle FID=%u", u16FID));
   switch (u16FID) {
      /*
        use macro SXM_MRESULT_ENTRY for each suported function-id:
        fill according thread-message for fiObject and adressing
        according to "poMessage" and send message accoring to u32Action of thread-message
      */
      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultMiddle,
         fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak);
   default:
      vOnUnknownMessage(pServiceDataMessage);
   }
}

/**
 * Handle all received method-result last fi-messages.
 * @brief For each function-id of interest use the macro SXM_MRESULT_ENTRY to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnMethodResultLast(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnMethodResultLast FID=%u", u16FID));
   switch (u16FID) {
      /*
        use macro SXM_MRESULT_ENTRY for each suported function-id:
        fill according thread-message for fiObject and adressing
        according to "poMessage" and send message accoring to u32Action of thread-message
      */
      SXM_MRESULT_ENTRY(SDS_TTSFI_C_U16_SYNTHESIZESPEAK,
         sds_ttsfi_tclMsgSynthesizeSpeakMethodResultLast,
         fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak);
   default:
      vOnUnknownMessage(pServiceDataMessage);
   }
}

/**
 * Handle all received method-error last fi-messages.
 * @brief For each function-id of interest use the macro to create the
 *        corresponding thread-message and send its app
 * @param pServiceDataMessage
 */
tVoid fc_sxm_tclTTSClient::vOnError(tU16 u16FID, amt_tclServiceData* pServiceDataMessage)
{
   amt_tclServiceDataError oErrorMsg( pServiceDataMessage );
   tU16 u16ErrorCode = oErrorMsg.u16GetErrorData();

   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vOnError FID=%u, u16ErrorCode=%u", u16FID, u16ErrorCode));

   switch (u16FID) {
      /*
        use macro SXM_MERROR_ENTRY for each suported function-id:
        fill according thread-message for fiObject and adressing
        according to "poMessage" and send message accoring to u32Action of thread-message
      */
      SXM_MERROR_ENTRY(SDS_TTSFI_C_U16_PREPAREVOICEOUTPUT, u16ErrorCode, fc_sxm_trMsgTTSClientError);
      SXM_MERROR_ENTRY(SDS_TTSFI_C_U16_ABORTPROMPT, u16ErrorCode, fc_sxm_trMsgTTSClientError);
      SXM_MERROR_ENTRY(SDS_TTSFI_C_U16_PAUSEPROMPT, u16ErrorCode, fc_sxm_trMsgTTSClientError);
      SXM_MERROR_ENTRY(SDS_TTSFI_C_U16_CONTINUEPROMPT, u16ErrorCode, fc_sxm_trMsgTTSClientError);

      default:
         vOnUnknownMessage(pServiceDataMessage);
      }
}

/**
 * Request for audio source channel
 * @return TRUE when audio source get allocated else FALSE
 */
tBool fc_sxm_tclTTSClient::bRequestAudioChannel(tVoid) const
{
	tBool bResult = false;
	if(fc_sxm_tclWsAlertsApp::instance()->enGetAudioSourceActivity() == FC_SXM_EN_AUDIO_SOURCE_ON)
	{
		ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bRequestAudioChannel,already done"));
	}
	else
	{
		bResult = fc_sxm_arl_tclISource::instance()->bRequestAudioRoute(ARL_SRC_SXM_TTS ,ARL_EN_ISRC_ACT_ON);
	}
	ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bRequestAudioChannel, bResult=%d ", bResult));
   return bResult;
}

/**
 * Release audio source channel
 * @return TRUE when audio source get released else FALSE
 */
tBool fc_sxm_tclTTSClient::bReleaseAudioChannel(tVoid) const
{
   tBool bResult = fc_sxm_arl_tclISource::instance()->bRequestAudioRoute(ARL_SRC_SXM_TTS, ARL_EN_ISRC_ACT_OFF);
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bReleaseAudioChannel, bResult=%d", bResult));
   return bResult;
}

/**
 * Start announcement for text received as input parameter
 * @param sSynthesizedAlertMessage[in]
 * @return TRUE when request send for audio channel or prepare voice output else FALSE
 */
tBool fc_sxm_tclTTSClient::bStartAnnouncement(const string& sSynthesizeAlertMessage)
{

   _sSynthesizeText = sSynthesizeAlertMessage;
   tBool bResult = bRequestAudioChannel();
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vStartWsAlertAnnounce, bResult=%d", bResult));
   return bResult;
}

tBool fc_sxm_tclTTSClient::bClearSynthesizeText(tVoid)
{
	tBool bResult = false;
	_sSynthesizeText.clear();
	if(_sSynthesizeText.empty())
	{
		bResult = true;
	}
	return bResult;
}

/**
 * Abort or stop announcement
 * @return TRUE when request is sent to TTS successfully else FALSE
 */
tBool fc_sxm_tclTTSClient::bAbortAnnouncement(tVoid)
{
   tBool bResult = bSendAbortPrompt();
   _bIsAbort = TRUE;
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bAbortAnnouncement, bResult=%d", bResult));
   return bResult;
}

/**
 * Stop announcement
 * @return TRUE when request is sent to TTS successfully else FALSE
 */
tBool fc_sxm_tclTTSClient::bStopAnnouncement(tVoid)
{
   tBool bResult = bSendAbortPrompt();
   _bIsStop = TRUE;
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bStopAnnouncement, bResult=%d", bResult));
   return bResult;
}

/**
 * Pause announcement
 * @return TRUE when request is sent to TTS successfully else FALSE
 */
tBool fc_sxm_tclTTSClient::bPauseAnnouncement(tVoid) const
{
   tBool bResult = bSendPausePrompt();
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bPauseAnnouncement, bResult=%d", bResult));
   return bResult;
}

/**
 * Resume announcement
 * @return TRUE when request is sent to TTS successfully else FALSE
 */
tBool fc_sxm_tclTTSClient::bResumeAnnouncement(tVoid) const
{
   tBool bResult = bSendContinuePrompt();
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bResumeAnnouncement, bResult=%d", bResult));
   return bResult;
}

/**
 * Handle the Audio source state update
 * @param[in] audio source activity
 */
tVoid fc_sxm_tclTTSClient::vHandleAudioSourceOn() const
{
   fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateReady::instance());
   fc_sxm_tclTTSContext::instance()->vDoAction();
}

tVoid fc_sxm_tclTTSClient::vHandleAudioSourceOff() const
{
   fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateInitial::instance());
}

tVoid fc_sxm_tclTTSClient::vHandleAudioSourcePause() const
{
   fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePaused::instance());
}

/**
 * Send prepare voice output request to TTS
 * @return TRUE when request sends successfully without any error  else FALSE
 */
tBool fc_sxm_tclTTSClient::bSendPrepareVoiceOutput(tVoid) const
{
   sds_ttsfi_tclMsgPrepareVoiceOutputMethodStart oMessage;
   ail_tenCommunicationError enResult = enSendFiMessage(oMessage);
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bSendPrepareVoiceOutput, enResult = %d", enResult));
   return ((AIL_EN_N_NO_ERROR == enResult) ? TRUE : FALSE);
}

/**
 * Send synthesize speak request to TTS
 * @return TRUE when request sends successfully without any error else FALSE
 */
tBool fc_sxm_tclTTSClient::bSendSynthesizeSpeakToTTS(tVoid) const
{
   sds_ttsfi_tclMsgSynthesizeSpeakMethodStart oMessage;
   oMessage.TextAndTags.bSet(_sSynthesizeText.c_str(), sds_fi_tclString::FI_EN_UTF8);
   oMessage.TextStyle.enType = sds_fi_tcl_e8_TextStyleOption::FI_EN_DICTS;
   oMessage.AudioOutputLocation.u8Value = sds_fi_tcl_b8_AudioOutputLocation::FI_C_U8_BIT_STANDARD;
   oMessage.StatusInfo.enType = sds_fi_tcl_e8_StatusInfoOption::FI_EN_NOINFO;
   oMessage.ExpDate = 0;
   oMessage.LastPackage = TRUE;

   ail_tenCommunicationError enResult = enSendFiMessage(oMessage);
   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bSendSynthesizeSpeakToTTS, enResult = %d _sSynthesizeMessage=%s",
            enResult, _sSynthesizeText.c_str()));
   return ((AIL_EN_N_NO_ERROR == enResult) ? TRUE : FALSE);
}

/**
 * Send abort prompt request to TTS
 * @return TRUE when request sends successfully without any error else FALSE
 */
tBool fc_sxm_tclTTSClient::bSendAbortPrompt(tVoid) const
{
   sds_ttsfi_tclMsgAbortPromptMethodStart oMessage;
   oMessage.AbortPromptOption.enType = sds_fi_tcl_e8_PromptAbortOption::FI_EN_IMMEDIATE;

   ail_tenCommunicationError enResult = enSendFiMessage(oMessage);

   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bSendAbortPrompt, enResult = %d", enResult));
   return ((AIL_EN_N_NO_ERROR == enResult) ? TRUE : FALSE);
}

/**
 * Send pause prompt request to TTS
 * @return TRUE when request sends successfully without any error else FALSE
 */
tBool fc_sxm_tclTTSClient::bSendPausePrompt(tVoid) const
{
   sds_ttsfi_tclMsgPausePromptMethodStart oMessage;
   oMessage.PausePromptOption.enType = sds_fi_tcl_e8_PromptAbortOption::FI_EN_IMMEDIATE;

   ail_tenCommunicationError enResult = enSendFiMessage(oMessage);

   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bSendPausePrompt, enResult = %d", enResult));
   return ((AIL_EN_N_NO_ERROR == enResult) ? TRUE : FALSE);
}

/**
 * Send continue prompt request to TTS
 * @return TRUE when request sends successfully without any error else FALSE
 */
tBool fc_sxm_tclTTSClient::bSendContinuePrompt(tVoid) const
{
   sds_ttsfi_tclMsgContinuePromptMethodStart oMessage;
   oMessage.Dummy = 0x00;

   ail_tenCommunicationError enResult = enSendFiMessage(oMessage);

   ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bSendContinuePrompt, enResult = %d", enResult));
   return ((AIL_EN_N_NO_ERROR == enResult) ? TRUE : FALSE);
}

/**
 * Handle the Prepare voice output method result
 * @param poMResult[in]
 */
tVoid fc_sxm_tclTTSClient::vHandlePrepareVoiceOutputResult(fc_sxm_trMsgTTSClientMResultPrepareVoiceOutput const * /*poMResult*/) const
{
   fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePrepared::instance());
   fc_sxm_tclTTSContext::instance()->vDoAction();
}

/**
 * Handle the synthesize speak method result fist
 * @param poMResult[in] contains the first method result for synthesize speak
 */
tVoid fc_sxm_tclTTSClient::vHandleMethodResultFirstSynthesizeSpeak(fc_sxm_trMsgTTSClientMResultFirstSynthesizeSpeak const * poMResult) const
{
   if(poMResult)
   {
      sds_fi_tcl_e8_JobStatus::tenType enJobStatus = poMResult->oFiMsg.JobProgress.JobStatus.enType;
      if((sds_fi_tcl_e8_JobStatus::FI_EN_JOBQUEUED == enJobStatus) ||
         (sds_fi_tcl_e8_JobStatus::FI_EN_JOBSTARTED == enJobStatus))
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateStarted::instance());
         ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultFirst, enJobStatus=Started"));
      }
      else
      {
         ETG_TRACE_ERR(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultFirst, enJobStatus = %d is incorrect",
                  ETG_CENUM(sds_fi_tcl_e8_JobStatus::tenType,enJobStatus)));
         vHandleError();
      }
   }
}

/**
 * Handle the middle method result for synthesize speak
 * @param poMResult[in] contains the middle method result for synthesize speak
 */
tVoid fc_sxm_tclTTSClient::vHandleMethodResultMiddleSynthesizeSpeak(fc_sxm_trMsgTTSClientMResultMiddleSynthesizeSpeak const * poMResult) const
{
   if(poMResult)
   {
      sds_fi_tcl_e8_JobStatus::tenType enJobStatus = poMResult->oFiMsg.JobProgress.JobStatus.enType;
      if(sds_fi_tcl_e8_JobStatus::FI_EN_JOBSTARTED == enJobStatus)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateStarted::instance());
         ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultMiddle, enJobStatus=Started"));
      }
      else if(sds_fi_tcl_e8_JobStatus::FI_EN_JOBCONTINUED == enJobStatus)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateResumed::instance());
         ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultMiddle, enJobStatus=Continued"));
      }
      else if(sds_fi_tcl_e8_JobStatus::FI_EN_JOBPAUSED == enJobStatus)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePaused::instance());
         ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultMiddle, enJobStatus=Paused"));
      }
      else
      {
         ETG_TRACE_ERR(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultMiddle, enJobStatus=%d is incorrect",
                  ETG_CENUM(sds_fi_tcl_e8_JobStatus::tenType, enJobStatus)));
         vHandleError();
      }

   }
}

/**
 * Handle the last method result for synthesize speak
 * @param poMResult[in] contains the last method result for synthesize speak
 */
tVoid fc_sxm_tclTTSClient::vHandleMethodResultLastSynthesizeSpeak(fc_sxm_trMsgTTSClientMResultLastSynthesizeSpeak const * poMResult) const
{
   if(poMResult)
   {
      sds_fi_tcl_e8_JobStatus::tenType enJobStatus = poMResult->oFiMsg.JobProgress.JobStatus.enType;
      if(sds_fi_tcl_e8_JobStatus::FI_EN_JOBFINISHED == enJobStatus)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateStopped::instance());
         ETG_TRACE_USR4(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultLast, enJobStatus=Finished"));
      }
      else
      {
         ETG_TRACE_ERR(("fc_sxm_tclTTSClient::bHandleSendSynthesizeSpeakMResultLast, enJobStatus=%d",
                  ETG_CENUM(sds_fi_tcl_e8_JobStatus::tenType, enJobStatus)));
         vHandleError();
      }
      bReleaseAudioChannel();
   }
}

/**
 * Handle the pause prompt method result
 */
tVoid fc_sxm_tclTTSClient::vHandlePausePromptResult(fc_sxm_trMsgTTSClientMResultPausePrompt const *poMResult) const
{
   if(poMResult)
   {
      ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vHandlePausePromptResult"));
      fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePaused::instance());
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclTTSClient::vHandlePausePromptResult, poMResult=NULL"));
   }
}

/**
 * Handle the continue prompt method result
 */
tVoid fc_sxm_tclTTSClient::vHandleContinuePromptResult(fc_sxm_trMsgTTSClientMResultContinuePrompt const * poMResult) const
{
   if(poMResult)
   {
      ETG_TRACE_USR4(("fc_sxm_tclTTSClient::vHandleContinuePromptResult"));
      fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateResumed::instance());
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclTTSClient::vHandleContinuePromptResult, poMResult=NULL"));
   }
}

/**
 * Handle abort prompt result
 * @param poMResult[in] contain the response for abort prompt request
 * @return TRUE when announcement aborted else FALSE
 */
tVoid fc_sxm_tclTTSClient::vHandleAbortPromptResult(fc_sxm_trMsgTTSClientMResultAbortPrompt const *poMResult)
{
   if(poMResult)
   {
      if(_bIsAbort)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateAborted::instance());
         _bIsAbort = FALSE;
      }
      else if(_bIsStop)
      {
         fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateStopped::instance());
         _bIsStop = FALSE;
      }
   }
   else
   {
      ETG_TRACE_ERR(("fc_sxm_tclTTSClient::vHandleAbortPromptResult, poMResult=NULL"));
   }
}

/**
 * Handle error for method start
 * @param poMessage[in]
 */
tVoid fc_sxm_tclTTSClient::vHandleError() const
{
	ETG_TRACE_USR4(( "fc_sxm_tclTTSClient::vHandleError() "));
	fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStateError::instance());
	bReleaseAudioChannel();
	fc_sxm_tclWsAlertsApp::instance()->vNotifyHmiErrorState();
	fc_sxm_tclTTSContext::instance()->vSetState(fc_sxm_tclTTSStatePending::instance());
	fc_sxm_tclTTSContext::instance()->vDoAction();

}
